package fr.inra.agrosyst.api.utils;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: DaoUtils.java 4008 2014-04-16 07:54:57Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/utils/DaoUtils.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.util.PagerBean;
import org.nuiton.util.PagerBeanUtil;

import com.google.common.collect.Maps;

/**
 * Class containing utilities methods for agrosyst DAO package.
 *
 * @author Eric Chatellier
 * @author Arnaud Thimel (Code Lutin)
 */
public class DaoUtils {

    protected static final String LIKE =
            "TRANSLATE(LOWER( %s ),"
            + "'áàâãäåāăąèééêëēĕėęěìíîïìĩīĭḩóôõöōŏőùúûüũūŭůäàáâãåæçćĉčöòóôõøüùúûßéèêëýñîìíïş',"
            + "'aaaaaaaaaeeeeeeeeeeiiiiiiiihooooooouuuuuuuuaaaaaaeccccoooooouuuuseeeeyniiiis')"
            + "like LOWER( %s )";

    /**
     * Generate sql like operator case and accent insensitive.
     *
     * @param field1 entity field to search into
     * @param field2 value field (must be accent escaped)
     * @return sql string
     */
    public static String getFieldLikeInsensitive(String field1, String field2) {
        String query = String.format(LIKE, field1, field2);
        return query;
    }

    /**
     * Build pager instance from current page,
     *
     * @param page
     * @param itemPerPage
     * @param totalCount
     * @return
     */
    public static PagerBean getPager(int page, int itemPerPage, long totalCount) {
        PagerBean bean = new PagerBean();
        bean.setPageIndex(page);
        bean.setPageSize(itemPerPage);
        bean.setPagesNumber(PagerBeanUtil.getTotalPage((int) totalCount, itemPerPage));
        bean.setRecords(totalCount);
        return bean;
    }

    public static String addQueryAttribute(Map<String, Object> args, String entityAttributeName, Object value) {
        String baseAttributeName = entityAttributeName.replaceAll("[.]", "_");

        int index = 0;
        String queryAttributeName;
        do {
            queryAttributeName = baseAttributeName + index;
            index++;
        } while (args.containsKey(queryAttributeName));

        args.put(queryAttributeName, value);
        return queryAttributeName;
    }

    protected static String getQueryForAttributeLike(String entityAlias, String entityAttributeName, Map<String, Object> args, String likeValue, String operator) {
        // TODO AThimel 12/07/13 Refactor : peut-être qu'il n'est pas nécessaire d'utiliser la méthode "getFieldLikeInsensitive"
        String alias = StringUtils.isBlank(entityAlias) ? "" : entityAlias + ".";
        String queryAttributeName = addQueryAttribute(args, entityAttributeName, StringUtils.stripAccents(likeValue));
        String result = " " + operator + " " + DaoUtils.getFieldLikeInsensitive(alias + entityAttributeName, ":" + queryAttributeName);

        return result;
    }

    public static String andAttributeLike(String entityAlias, String entityAttributeName, Map<String, Object> args, String value) {
        String result = "";
        if (StringUtils.isNotBlank(value)) {
            result = getQueryForAttributeLike(entityAlias, entityAttributeName, args, "%" + value + "%", "AND");
        }
        return result;
    }

    public static String orAttributeLike(String entityAlias, String entityAttributeName, Map<String, Object> args, String value) {
        String result = "";
        if (StringUtils.isNotBlank(value)) {
            result = getQueryForAttributeLike(entityAlias, entityAttributeName, args, "%" + value + "%", "OR");
        }
        return result;
    }
    
    public static String andAttributeStartLike(String entityAlias, String entityAttributeName, Map<String, Object> args, String value) {
        String result = "";
        if (StringUtils.isNotBlank(value)) {
            result = getQueryForAttributeLike(entityAlias, entityAttributeName, args, value + "%", "AND");
        }
        return result;
    }

    public static String orAttributeStartLike(String entityAlias, String entityAttributeName, Map<String, Object> args, String value) {
        String result = "";
        if (StringUtils.isNotBlank(value)) {
            result = getQueryForAttributeLike(entityAlias, entityAttributeName, args, value + "%", "OR");
        }
        return result;
    }

    public static String orAttributeEquals(String entityAlias, String entityAttributeName, Map<String, Object> args, Object value) {
        String result = getQueryForAttributeEquals(entityAlias, entityAttributeName, args, value, "OR");
        return result;
    }

    protected static String getQueryForAttributeEquals(String entityAlias, String entityAttributeName, Map<String, Object> args, Object value, String operator) {
        String result = "";

        if (value != null) {
            String alias = StringUtils.isBlank(entityAlias) ? "" : entityAlias + ".";
            String queryAttributeName = addQueryAttribute(args, entityAttributeName, value);
            result += String.format(" %s %s = :%s", operator, alias + entityAttributeName, queryAttributeName);
        }

        return result;
    }

    protected static String getQueryForAttributeNotEquals(String entityAlias, String entityAttributeName, Map<String, Object> args, Object value, String operator) {
        String result = "";

        if (value != null) {
            String alias = StringUtils.isBlank(entityAlias) ? "" : entityAlias + ".";
            String queryAttributeName = addQueryAttribute(args, entityAttributeName, value);
            result += String.format(" %s %s != :%s", operator, alias + entityAttributeName, queryAttributeName);
        }

        return result;
    }

    public static String andAttributeEquals(String entityAlias, String entityAttributeName, Map<String, Object> args, Object value) {
        String result = getQueryForAttributeEquals(entityAlias, entityAttributeName, args, value, "AND");
        return result;
    }

    public static String andAttributeNotEquals(String entityAlias, String entityAttributeName, Map<String, Object> args, Object value) {
        String result = getQueryForAttributeNotEquals(entityAlias, entityAttributeName, args, value, "AND");
        return result;
    }

    protected static String getQueryForAttributeIn(String entityAlias, String entityAttributeName, Map<String, Object> args, Set<?> values, String operator) {
        String result = "";

        if (values != null) {
            if (values.isEmpty()) {
                // Requesting IN with empty list could never return any result. Motivated by issue http://forge.codelutin.com/issues/4748
                result += String.format(" %s 1 = 0 ", operator);
            } else {
                String alias = StringUtils.isBlank(entityAlias) ? "" : entityAlias + ".";
                String queryAttributeName = addQueryAttribute(args, entityAttributeName, values);
                result += String.format(" %s %s in ( :%s )", operator, alias + entityAttributeName, queryAttributeName);
            }
        }

        return result;
    }

    public static String andAttributeInIfNotEmpty(String entityAlias, String entityAttributeName, Map<String, Object> args, Set<?> values) {
        if (values != null && !values.isEmpty()) {
            String result = getQueryForAttributeIn(entityAlias, entityAttributeName, args, values, "AND");
            return result;
        } else {
            return "";
        }
    }

    public static String andAttributeIn(String entityAlias, String entityAttributeName, Map<String, Object> args, Set<?> values) {
        String result = getQueryForAttributeIn(entityAlias, entityAttributeName, args, values, "AND");
        return result;
    }

    protected static String getQueryForAttributeNotIn(String entityAlias, String entityAttributeName, Map<String, Object> args, Set<?> values, String operator) {
        String result = "";

        if (values != null) {
            if (values.isEmpty()) {
                // Requesting NOT IN with empty list will always return results
                result += String.format(" %s 1 = 1 ", operator);
            } else {
                String alias = StringUtils.isBlank(entityAlias) ? "" : entityAlias + ".";
                String queryAttributeName = addQueryAttribute(args, entityAttributeName, values);
                result += String.format(" %s %s not in ( :%s )", operator, alias + entityAttributeName, queryAttributeName);
            }
        }

        return result;
    }

    public static String andAttributeNotIn(String entityAlias, String entityAttributeName, Map<String, Object> args, Set<?> values) {
        String result = getQueryForAttributeNotIn(entityAlias, entityAttributeName, args, values, "AND");
        return result;
    }

    public static Map<String, Object> asArgsMap() {
        Map<String, Object> result = Maps.newLinkedHashMap();
        return result;
    }

    public static Map<String, Object> asArgsMap(String key1, Object value1) {
        Map<String, Object> result = asArgsMap();
        result.put(key1, value1);
        return result;
    }

    public static Map<String, Object> asArgsMap(String key1, Object value1, String key2, Object value2) {
        Map<String, Object> result = asArgsMap(key1, value1);
        result.put(key2, value2);
        return result;
    }

    public static Map<String, Object> asArgsMap(String key1, Object value1, String key2, Object value2, String key3, Object value3) {
        Map<String, Object> result = asArgsMap(key1, value1, key2, value2);
        result.put(key3, value3);
        return result;
    }

    public static Map<String, Object> asArgsMap(String key1, Object value1, String key2, Object value2, String key3, Object value3,String key4, Object value4) {
        Map<String, Object> result = asArgsMap(key1, value1, key2, value2,key3, value3);
        result.put(key4, value4);
        return result;
    }



    public static Object[] toArgsArray(Map<String, Object> args) {
        Object[] result = new Object[args.size() * 2];
        int index = 0;
        for (Map.Entry<String, Object> entry : args.entrySet()) {
            result[index++] = entry.getKey();
            result[index++] = entry.getValue();
        }
        return result;
    }

    public static LinkedHashMap<Integer, String> toRelatedMap(List<Object[]> input) {
        LinkedHashMap<Integer, String> result = Maps.newLinkedHashMap();
        if (input != null) {
            for (Object[] entry : input) {
                Integer campaign = (Integer) entry[0];
                String topiaId = (String) entry[1];
                result.put(campaign, topiaId);
            }
        }
        return result;
    }

    public static Double median(List<Double> list) {
        Double result = null;
        if (list != null && !list.isEmpty()) {
            int middle = list.size() / 2;
            if (list.size() % 2 == 1) {
                result = list.get(middle);
            } else {
                Double valInf = list.get(middle - 1);
                Double valSup = list.get(middle);
                if (valInf != null && valSup != null) {
                    result = (valInf + valSup) / 2.0;
                }
            }
        }
        return result;
    }

}
