package fr.inra.agrosyst.api.entities.referential;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: RefLocationTopiaDao.java 4329 2014-09-12 15:14:03Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/referential/RefLocationTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.topia.persistence.TopiaException;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.utils.DaoUtils;

public class RefLocationTopiaDao extends AbstractRefLocationTopiaDao<RefLocation> {

    public static final Function<String, String> NORMALIZE_COMMUNE_FOR_SEARCH = new Function<String, String>() {
        @Override
        public String apply(String input) {
            String  result = null;
            if (input != null) {
                result = input.toLowerCase().trim();
                Iterable<String> parts = Splitter.onPattern("[\\s-_,.]").trimResults().omitEmptyStrings().split(result); // C'est surtout le trimResults et omitEmptyStrings qui sont utiles ici
                result = Joiner.on("-").join(parts);
                if (result.startsWith("la-") || result.startsWith("le-") || result.startsWith("les-") || result.startsWith("los-")) {
                    result = result.replaceFirst("-", " ");
                }
                result = result.replaceAll("-d-", "-d'");
            }
            return result;
        }
    };

    /**
     * Search all RefCommuneValues matching user custom additional {@code filter}.
     *
     * @param filter custom filter
     * @return matching domains
     * @throws TopiaException
     */
    public List<RefLocation> findActiveLocations(String filter, int maxResults) throws TopiaException {
        String query = "FROM " + getEntityClass().getName() + " rl";
        query += " WHERE 1 = 1";
        Map<String, Object> args = Maps.newHashMap();

        // apply non null filter
        if (filter != null) {

            String escapedFilter = NORMALIZE_COMMUNE_FOR_SEARCH.apply(filter);
            if (StringUtils.isNotBlank(escapedFilter)) {

                // Post code
                query += DaoUtils.andAttributeStartLike("rl", RefLocation.PROPERTY_CODE_POSTAL, args, escapedFilter);

                // Town name
                query += DaoUtils.orAttributeLike("rl", RefLocation.PROPERTY_COMMUNE, args, escapedFilter);
            }
        }

        // active
        query += DaoUtils.andAttributeEquals("rl", RefLocation.PROPERTY_ACTIVE, args, true);

        query += " ORDER BY rl." + RefLocation.PROPERTY_COMMUNE;

        List<RefLocation> result = find(query, args, 0, maxResults);

        return result;
    }

}
