package fr.inra.agrosyst.api.entities.practiced;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PracticedSystemTopiaDao.java 4613 2014-12-06 18:48:26Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/practiced/PracticedSystemTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.GrowingSystemTopiaDao;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.practiced.PracticedSystemFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.security.SecurityContext;
import fr.inra.agrosyst.services.security.SecurityHelper;

import org.nuiton.util.PagerBean;

/**
 * Practiced system dao custom methods.
 * 
 * @author Eric Chatellier
 */
public class PracticedSystemTopiaDao extends AbstractPracticedSystemTopiaDao<PracticedSystem> {

    protected static final String PROPERTY_GROWING_SYSTEM_ID = PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_TOPIA_ID;
    protected static final String PROPERTY_GROWING_SYSTEM_NAME = PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_NAME;

    protected static final String PROPERTY_GROWING_SYSTEM_GROWING_PLAN = PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_GROWING_PLAN;
    protected static final String PROPERTY_GROWING_SYSTEM_GROWING_PLAN_ID = PROPERTY_GROWING_SYSTEM_GROWING_PLAN + "." + GrowingPlan.PROPERTY_TOPIA_ID;
    protected static final String PROPERTY_GROWING_SYSTEM_GROWING_PLAN_NAME = PROPERTY_GROWING_SYSTEM_GROWING_PLAN + "." + GrowingPlan.PROPERTY_NAME;

    protected static final String PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN = PROPERTY_GROWING_SYSTEM_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN;
    protected static final String PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_ID = PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID;
    protected static final String PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_NAME = PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_NAME;
    protected static final String PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_CAMPAIGN = PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN;

    public ResultList<PracticedSystem> getFilteredPracticedSystems(PracticedSystemFilter filter, SecurityContext securityContext) {
        StringBuilder query = new StringBuilder(" FROM " + getEntityClass().getName() + " ps ");
        query .append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // apply non null filter
        if (filter != null) {

            // PracticedSystem name
            query .append(DaoUtils.andAttributeLike("ps", PracticedSystem.PROPERTY_NAME, args, filter.getPracticedSystemName()));

            // Growing System
            query .append(DaoUtils.andAttributeLike("ps", PROPERTY_GROWING_SYSTEM_NAME, args, filter.getGrowingSystemName()));

            // PracticedSystem campaigns
            query .append(DaoUtils.andAttributeLike("ps", PracticedSystem.PROPERTY_CAMPAIGNS, args, filter.getPracticedSystemCampaign()));

            // PracticeSystem active
            query .append(DaoUtils.andAttributeEquals("ps", PracticedSystem.PROPERTY_ACTIVE, args, filter.getActive()));

            // PracticeSystem validated
            query .append(DaoUtils.andAttributeEquals("ps", PracticedSystem.PROPERTY_VALIDATED, args, filter.getValidated()));

            // Domain name
            query .append(DaoUtils.andAttributeLike("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_NAME, args, filter.getDomainName()));
            
            // GrowingPlan name
            query .append(DaoUtils.andAttributeLike("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_NAME, args, filter.getGrowingPlanName()));

            // Navigation context
            NavigationContext navigationContext = filter.getNavigationContext();
            if (navigationContext != null) {

                // campaigns
                // FIXME query must compare PracticedSystem campaing and not domain campaigns
                query .append(DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_CAMPAIGN, args, navigationContext.getCampaigns()));

                // networks
                if (navigationContext.getNetworksCount() > 0) {
                    Set<String> growingSystemIds = getProjectionHelper().networksToGrowingSystems(navigationContext.getNetworks());
                    query.append(DaoUtils.andAttributeIn("ps", PROPERTY_GROWING_SYSTEM_ID, args, growingSystemIds));
                }

                // domains
                query .append(DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_ID, args, navigationContext.getDomains()));

                // growingPlans
                query .append(DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_ID, args, navigationContext.getGrowingPlans()));

                // growingSystems
                // FIXME query must compare code and not id
                query .append(DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_ID, args, navigationContext.getGrowingSystems()));
            }
        }

        SecurityHelper.addPracticedSystemFilter(query, args, securityContext, "ps");

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;

        List<PracticedSystem> practicedSystems = find(query.toString() + " ORDER BY lower (ps." + PracticedSystem.PROPERTY_NAME +")", args, startIndex, endIndex);
        long totalCount = findUnique("SELECT count(*) " + query, args);

        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);

        ResultList<PracticedSystem> result = ResultList.of(practicedSystems, pager);

        return result;
    }
    
    public List<PracticedSystem> getPracticedSystemsWithoutPlot(NavigationContext navigationContext, SecurityContext securityContext) {
        StringBuilder query = new StringBuilder(" FROM " + getEntityClass().getName() + " ps ");
        query.append(" WHERE ps not in (SELECT pp." + PracticedPlot.PROPERTY_PRACTICED_SYSTEM + " FROM " + PracticedPlot.class.getName() + " pp) ");
        Map<String, Object> args = Maps.newLinkedHashMap();

        if (navigationContext != null) {

            // campaigns
            // FIXME query must compare PracticedSystem campaing and not domain campaigns
            query.append(DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_CAMPAIGN, args, navigationContext.getCampaigns()));

            // networks
            if (navigationContext.getNetworksCount() > 0) {
                Set<String> growingSystemIds = getProjectionHelper().networksToGrowingSystems(navigationContext.getNetworks());
                query.append(DaoUtils.andAttributeIn("ps", PROPERTY_GROWING_SYSTEM_ID, args, growingSystemIds));
            }

            // domains
            query.append(DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_ID, args, navigationContext.getDomains()));

            // growingPlans
            query.append(DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_ID, args, navigationContext.getGrowingPlans()));

            // growingSystems
            // FIXME query must compare code and not id
            query.append(DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_ID, args, navigationContext.getGrowingSystems()));
        }

        SecurityHelper.addPracticedSystemFilter(query, args, securityContext, "ps");

        List<PracticedSystem> practicedSystems = findAll(query + " ORDER BY lower (ps." + PracticedSystem.PROPERTY_NAME +")", args);

        return practicedSystems;
    }

    public List<PracticedSystem> getPracticedSystemsWithoutCropCycle(NavigationContext navigationContext) {
        String query = "FROM " + getEntityClass().getName() + " ps";
        query += " WHERE ps." + PracticedSystem.PROPERTY_TOPIA_ID + " NOT IN (" +
                "SELECT pcc." + PracticedCropCycle.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_TOPIA_ID + "" +
                " FROM " + PracticedCropCycle.class.getName() + " pcc" +
                ") ";

        Map<String, Object> args = Maps.newLinkedHashMap();
        // Navigation context
        if (navigationContext != null) {

            // campaigns
            query += DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_CAMPAIGN, args, navigationContext.getCampaigns());

            // networks
            if (navigationContext.getNetworksCount() > 0) {
                Set<String> growingSystemIds = getProjectionHelper().networksToGrowingSystems(navigationContext.getNetworks());
                query += DaoUtils.andAttributeIn("ps", PROPERTY_GROWING_SYSTEM_ID, args, growingSystemIds);
            }

            // domains
            query += DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_DOMAIN_ID, args, navigationContext.getDomains());

            // growingPlans
            query += DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_GROWING_PLAN_ID, args, navigationContext.getGrowingPlans());

            // growingSystems
            query += DaoUtils.andAttributeInIfNotEmpty("ps", PROPERTY_GROWING_SYSTEM_ID, args, navigationContext.getGrowingSystems());
        }

        List<PracticedSystem> result = findAll(query, args);
        return result;
    }

    public void validatePracticedSystem(String practicedSystemId, Date now) {
        Map<String, Object> args = DaoUtils.asArgsMap("practicedSystemId", practicedSystemId, "now", now);
        topiaJpaSupport.execute("UPDATE " + getEntityClass().getName() + " ps" +
                " SET ps.validated=true, ps.validationDate=:now, ps.updateDate=:now" +
                " WHERE ps." + PracticedSystem.PROPERTY_TOPIA_ID + "=:practicedSystemId", args);

        // TODO AThimel 10/12/13 Validate all underlying entities
    }

} //PracticedSystemTopiaDao<E extends PracticedSystem>
