package fr.inra.agrosyst.api.entities.performance;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PerformanceTopiaDao.java 3394 2014-02-21 13:07:41Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/performance/PerformanceTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.util.PagerBean;

import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.Plot;
import fr.inra.agrosyst.api.entities.Zone;
import fr.inra.agrosyst.api.entities.security.AgrosystUser;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.performance.PerformanceFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.security.SecurityContext;

public class PerformanceTopiaDao extends AbstractPerformanceTopiaDao<Performance> {

    public ResultList<Performance> getFilteredPerformances(PerformanceFilter filter, SecurityContext securityContext) throws TopiaException {
        Map<String, Object> args = Maps.newLinkedHashMap();
        StringBuilder query = new StringBuilder("FROM " + Performance.class.getName() + " P");
        query.append(" WHERE 1 = 1");

        // apply non null filter
        if (filter != null) {
            // practiced
            query.append(DaoUtils.andAttributeEquals("P", Performance.PROPERTY_PRACTICED, args, filter.isPracticed()));

            // name
            query.append(DaoUtils.andAttributeLike("P", Performance.PROPERTY_NAME, args, filter.getPerformanceName()));
            
            // domain name
            if (StringUtils.isNotEmpty(filter.getDomainName())) {
                if (StringUtils.isNumeric(filter.getDomainName())) {
                    query.append(" AND EXISTS (FROM " + Domain.class.getName() + " D" +
                        " WHERE D in elements(P." + Performance.PROPERTY_DOMAINS + ")" +
                        " AND D." + Domain.PROPERTY_CAMPAIGN + " = :domainCampaign" +
                        ")");
                    args.put("domainCampaign", Integer.valueOf(filter.getDomainName()));
                } else {
                    query.append(" AND EXISTS (FROM " + Domain.class.getName() + " D" +
                        " WHERE D in elements(P." + Performance.PROPERTY_DOMAINS + ")" +
                        " AND " + DaoUtils.getFieldLikeInsensitive("D." + Domain.PROPERTY_NAME, ":domainName") +
                        ")");
                    args.put("domainName", "%" + filter.getDomainName() + "%");
                }
            }

            // growing system name
            if (StringUtils.isNotEmpty(filter.getGrowingSystemName())) {
                query.append(" AND EXISTS (FROM " + GrowingSystem.class.getName() + " G" +
                    " WHERE G in elements(P." + Performance.PROPERTY_GROWING_SYSTEMS + ")" +
                    " AND " + DaoUtils.getFieldLikeInsensitive("G." + GrowingSystem.PROPERTY_NAME, ":growingSystemName") +
                    ")");
                args.put("growingSystemName", "%" + filter.getGrowingSystemName() + "%");
            }
            
            // plot name
            if (StringUtils.isNotEmpty(filter.getPlotName())) {
                query.append(" AND EXISTS (FROM " + Plot.class.getName() + " PL" +
                    " WHERE PL in elements(P." + Performance.PROPERTY_PLOTS + ")" +
                    " AND " + DaoUtils.getFieldLikeInsensitive("PL." + Plot.PROPERTY_NAME, ":plotName") +
                    ")");
                args.put("plotName", "%" + filter.getPlotName() + "%");
            }
            
            // zone name
            if (StringUtils.isNotEmpty(filter.getZoneName())) {
                query.append(" AND EXISTS (FROM " + Zone.class.getName() + " Z" +
                    " WHERE Z in elements(P." + Performance.PROPERTY_ZONES + ")" +
                    " AND " + DaoUtils.getFieldLikeInsensitive("Z." + Zone.PROPERTY_NAME, ":zoneName") +
                    ")");
                args.put("zoneName", "%" + filter.getZoneName() + "%");
            }
        }

        // security (author)
        if (!securityContext.isAdmin()) {
            String userId = securityContext.getUserId();
            query.append(DaoUtils.andAttributeEquals("P", Performance.PROPERTY_AUTHOR + "." + AgrosystUser.PROPERTY_TOPIA_ID, args, userId));
        }

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;

        String queryString = query.toString();
        String queryAndOrder = queryString + " ORDER BY " +
                "P." + Performance.PROPERTY_TOPIA_CREATE_DATE;
        List<Performance> performances = find(queryAndOrder, args, startIndex, endIndex);
        long totalCount = findUnique("SELECT count(*) " + queryString, args);

        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);
        ResultList<Performance> result = ResultList.of(performances, pager);
        return result;
    }

} //PerformanceTopiaDao
