package fr.inra.agrosyst.api.entities.migration;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: V1_5_0_3__7078_RefFertiMinPriceFix.java 4940 2015-05-11 14:47:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/migration/V1_5_0_3__7078_RefFertiMinPriceFix.java $
 * %%
 * Copyright (C) 2013 - 2015 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.entities.Price;
import fr.inra.agrosyst.api.entities.PriceImpl;
import fr.inra.agrosyst.api.exceptions.AgrosystTechnicalException;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.flyway.TopiaFlywayJdbcMigration;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Created by davidcosse on 07/05/15.
 */
public class V1_5_0_3__7078_RefFertiMinPriceFix implements TopiaFlywayJdbcMigration {

    private static final Log log = LogFactory.getLog(V1_5_0_3__7078_RefFertiMinPriceFix.class);

    @Override
    public void migrate(Connection connection) {
        Statement findAllPricesStmt = null;
        ResultSet results = null;

        try {
            String getPricesQuery = "SELECT topiaid, objectid, sourceunit, domain, practicedsystem, type, priceunit FROM price p WHERE type LIKE 'INPUT' AND objectid LIKE 'categ%'";

            Map<String, Price> priceByUniqueKey = Maps.newHashMap();
            List<String> duplicatePrices = Lists.newArrayList();

            findAllPricesStmt = connection.createStatement();
            findAllPricesStmt.execute(getPricesQuery);
            results = findAllPricesStmt.getResultSet();
            while(results.next()){
                String topiaId = results.getString(Price.PROPERTY_TOPIA_ID);
                String objectId = results.getString(Price.PROPERTY_OBJECT_ID);
                String sourceUnit = results.getString(Price.PROPERTY_SOURCE_UNIT);
                String domain = results.getString(Price.PROPERTY_DOMAIN);
                String practicedSystem = results.getString(Price.PROPERTY_PRACTICED_SYSTEM);
                String type = results.getString(Price.PROPERTY_TYPE);
                String priceUnit = results.getString(Price.PROPERTY_PRICE_UNIT);

                String fixedObjectId = fixObjectid(objectId);

                checkUniqueKey(priceByUniqueKey, duplicatePrices, topiaId, objectId, sourceUnit, domain, practicedSystem, type, priceUnit, fixedObjectId);
            }

            deleteDuplicatedPrices(connection, duplicatePrices);

            updatedFixedPrices(connection, priceByUniqueKey);

            connection.commit();

        } catch (SQLException e) {
            throw new AgrosystTechnicalException("Exception lors de la mise à jour des prix", e);
        } finally {
            if (results != null) {
                try {
                    results.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
            if (findAllPricesStmt != null) {
                try {
                    findAllPricesStmt.close();
                } catch (SQLException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    protected void updatedFixedPrices(Connection connection, Map<String, Price> priceByUniqueKey) {
        PreparedStatement updatePriceObjectIdStmt = null;
        if (!priceByUniqueKey.isEmpty()) {
            String updatePriceObjectIdQuery = "UPDATE price SET objectid = ? WHERE topiaid = ?";
            try {
                updatePriceObjectIdStmt = connection.prepareStatement(updatePriceObjectIdQuery);
                Collection<Price> pricesToUpdate = priceByUniqueKey.values();
                for (Price price : pricesToUpdate) {
                    String topiaId = price.getTopiaId();
                    String objectId = price.getObjectId();
                    updatePriceObjectIdStmt.setString(1, objectId);
                    updatePriceObjectIdStmt.setString(2, topiaId);
                    updatePriceObjectIdStmt.execute();
                }
            } catch (SQLException e) {
                throw new AgrosystTechnicalException("Exception lors de la mise à jour des prix", e);
            } finally {
                if (updatePriceObjectIdStmt != null) {
                    try {
                        updatePriceObjectIdStmt.close();
                    } catch (SQLException e) {
                        e.printStackTrace();
                    }
                }
            }

        }
    }

    protected void deleteDuplicatedPrices(Connection connection, List<String> duplicatePrices) {
        PreparedStatement deletePriceStmt = null;
        if (!duplicatePrices.isEmpty()) {
            String deletePricesQuery = "DELETE FROM price WHERE topiaid = ?";
            try {
                deletePriceStmt = connection.prepareStatement(deletePricesQuery);
                for (String duplicatePrice : duplicatePrices) {
                    deletePriceStmt.setString(1, duplicatePrice);
                    deletePriceStmt.execute();
                }
            } catch (SQLException e) {
                throw new AgrosystTechnicalException("Exception lors de la mise à jour des prix", e);
            } finally {
                if (deletePriceStmt != null) {
                    try {
                        deletePriceStmt.close();
                    } catch (SQLException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
    }

    protected void checkUniqueKey(Map<String, Price> priceByUniqueKey, List<String> duplicatePrices, String topiaId, String objectId, String sourceUnit, String domain, String practicedSystem, String type, String priceUnit, String fixedObjectId) {
        Price p = new PriceImpl();
        p.setTopiaId(topiaId);
        p.setObjectId(fixedObjectId);

        // "price_uniqueness" UNIQUE, btree (
        // (COALESCE(objectid, '###'::text)),
        // (COALESCE(sourceunit, '###'::text)),
        // (COALESCE(domain, 'd'::character varying)),
        // (COALESCE(practicedsystem, 'ps'::character varying)),
        // type,
        // priceunit)
        String uniqueKey = fixedObjectId + sourceUnit + domain + practicedSystem + type + priceUnit;
        if (priceByUniqueKey.containsKey(uniqueKey)) {
            duplicatePrices.add(topiaId);
        } else {
            priceByUniqueKey.put(uniqueKey, p);
            if (log.isInfoEnabled()) {
                if (!objectId.contentEquals(fixedObjectId)) {
                    log.info(String.format("Prix changé de %s -> %s", objectId, fixedObjectId));
                }
            }
        }
    }


    public String fixObjectid(String objID) {
        objID = String.format(" %s ", objID);

        String objectId =  "categ=%s forme=%s n=%s p2O5=%s k2O=%s bore=%s fer=%s calcium=%s manganese=%s molybdene=%s mgO=%s oxyde_de_sodium=%s sO3=%s cuivre=%s zinc=%s ";
        String[] keys = objectId.split("=%s ");

        String[] values = new String[keys.length];
        int nbElement = keys.length;
        for (int element = 0; element <= (nbElement - 1) ;element ++) {
            // comportement is a bit different for last one
            boolean isNextElementNotLastOne = element + 1 < keys.length;

            String keyFrom = String.format(" %s=", keys[element]);
            String keyNext = isNextElementNotLastOne ? String.format(" %s=", keys[element + 1]) : " ";

            String[] valAndNK = objID.split(keyFrom);
            if (valAndNK.length > 0) {
                String nextKey;
                if (StringUtils.contains(valAndNK[1], keyNext)) {
                    nextKey = keyNext;
                } else {
                    nextKey = "null";
                }

                String[] firstElementIsValue =  valAndNK[1].split(nextKey);
                String value = firstElementIsValue [0];
                values[element] = StringUtils.isBlank(value) ? "null" : value;
                int splitIndex = (keyFrom.length() + values[element].length() + (nextKey.equals("null") ? 0 : keyNext.length()));
                objID = objID.substring(splitIndex);

                objID = keyNext + objID;
            }
        }

        objectId = String.format(objectId, values[0], values[1], values[2], values[3], values[4], values[5], values[6], values[7], values[8], values[9], values[10], values[11], values[12], values[13], values[14]);
        objectId = objectId.substring(0, objectId.length() - 1);
        objectId = StringUtils.stripAccents(objectId);
        return objectId;
    }
}
