package fr.inra.agrosyst.api.entities.managementmode;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: ManagementModeTopiaDao.java 4076 2014-05-22 07:47:04Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/managementmode/ManagementModeTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.CroppingPlanEntry;
import fr.inra.agrosyst.api.entities.Domain;
import fr.inra.agrosyst.api.entities.GrowingPlan;
import fr.inra.agrosyst.api.entities.GrowingSystem;
import fr.inra.agrosyst.api.entities.security.ComputedUserPermission;
import fr.inra.agrosyst.api.entities.security.PermissionObjectType;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeDto;
import fr.inra.agrosyst.api.services.managementmode.ManagementModeFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.security.SecurityContext;

import org.apache.commons.lang3.tuple.Pair;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.util.PagerBean;

import java.util.List;
import java.util.Map;
import java.util.Set;

public class ManagementModeTopiaDao extends AbstractManagementModeTopiaDao<ManagementMode> {
    
    protected static final String PROPERTY_GROWING_SYSTEM_NAME = ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_NAME;
    protected static final String PROPERTY_DOMAIN_CAMPAIGN = ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN;
    protected static final String PROPERTY_DOMAIN_ID = ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID;
    protected static final String PROPERTY_DOMAIN_NAME = ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_NAME;
    protected static final String PROPERTY_GROWING_PLAN_ID = ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_TOPIA_ID;
    protected static final String PROPERTY_GROWING_PLAN_NAME = ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_NAME;
    protected static final String PROPERTY_GROWING_SYSTEM_ID = ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_TOPIA_ID;

    public Pair<Map<GrowingSystem, ManagementModeDto>, PagerBean> getFilteredManagementModeDtos(
            ManagementModeFilter filter, SecurityContext securityContext) throws TopiaException {

        String growingSystemManagementModeQuery = " FROM " + GrowingSystem.class.getName() + " gs ";
        growingSystemManagementModeQuery += " WHERE gs IN (";
        growingSystemManagementModeQuery += " SELECT mm." + ManagementMode.PROPERTY_GROWING_SYSTEM + " FROM " + ManagementMode.class.getName() + " mm";
        growingSystemManagementModeQuery += " WHERE 1 = 1 ";

        Map<String, Object> args = Maps.newLinkedHashMap();
        growingSystemManagementModeQuery = filtratedQuery(growingSystemManagementModeQuery, args, filter, securityContext);
        growingSystemManagementModeQuery += ")";

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;
        List<GrowingSystem> growingSystems = find("SELECT DISTINCT gs " +  growingSystemManagementModeQuery + " ORDER BY gs." + GrowingSystem.PROPERTY_NAME, args, startIndex, endIndex);
        long totalCount = findUnique("SELECT count(*) " + growingSystemManagementModeQuery, args);

        args = Maps.newLinkedHashMap();
        String query = "SELECT " +
                "mm." + ManagementMode.PROPERTY_GROWING_SYSTEM + ", " +
                "mm." + ManagementMode.PROPERTY_TOPIA_ID + ", " +
                "mm." + ManagementMode.PROPERTY_CATEGORY + " FROM " + ManagementMode.class.getName() + " mm";

        query += " WHERE 1 = 1 ";

        query = filtratedQuery(query, args, filter, securityContext);

        Set<GrowingSystem> growingSystemSet = Sets.newHashSet(growingSystems);
        query += DaoUtils.andAttributeIn("mm", ManagementMode.PROPERTY_GROWING_SYSTEM, args, growingSystemSet);

        List<Object[]> results = findAll(query + " ORDER BY mm." + PROPERTY_GROWING_SYSTEM_NAME, args);

        Map<GrowingSystem, ManagementModeDto> gsManagmentModeDtos = Maps.newLinkedHashMap();

        if (results != null) {
            for (Object[] result: results) {

                GrowingSystem gs = (GrowingSystem) result[0];
                ManagementModeDto managementModeDto = gsManagmentModeDtos.get(gs);
                if (managementModeDto == null) {
                    managementModeDto= new ManagementModeDto();
                    gsManagmentModeDtos.put(gs,managementModeDto);
                }
                String mmId = (String) result[1];
                ManagementModeCategory category = (ManagementModeCategory) result[2];
                if (category == ManagementModeCategory.PLANNED) {
                    managementModeDto.setPlannedManagementModeId(mmId);
                } else {
                    managementModeDto.setObservedManagementModeId(mmId);
                }
            }
        }

        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);

        Pair<Map<GrowingSystem, ManagementModeDto>, PagerBean> result = Pair.of(gsManagmentModeDtos, pager);
        return result;
    }

    protected String filtratedQuery(String query, Map<String, Object> args, ManagementModeFilter filter, SecurityContext securityContext) {
        if (filter != null) {

            // decision rule name
            query += DaoUtils.andAttributeEquals("mm", ManagementMode.PROPERTY_VERSION_NUMBER, args, filter.getVersion());

            // growing system name
            query += DaoUtils.andAttributeLike("mm", PROPERTY_GROWING_SYSTEM_NAME, args, filter.getGrowingSystemName());

            // campaign
            query += DaoUtils.andAttributeEquals("mm", PROPERTY_DOMAIN_CAMPAIGN, args, filter.getCampaign());

            // category
            query += DaoUtils.andAttributeEquals("mm", ManagementMode.PROPERTY_CATEGORY, args, filter.getManagementModeCategory());

            // growing plan
            query += DaoUtils.andAttributeLike("mm", PROPERTY_GROWING_PLAN_NAME, args, filter.getGrowingPlanName());

            // domain
            query += DaoUtils.andAttributeLike("mm", PROPERTY_DOMAIN_NAME, args, filter.getDomainName());


            // Navigation context
            NavigationContext navigationContext = filter.getNavigationContext();
            if (navigationContext != null) {

                // campaigns
                query += DaoUtils.andAttributeInIfNotEmpty("mm", PROPERTY_DOMAIN_CAMPAIGN, args, navigationContext.getCampaigns());

                // networks
                if (navigationContext.getNetworksCount() > 0) {
                    Set<String> growingPlanIds = networksToGrowingSystems(navigationContext.getNetworks());
                    query += DaoUtils.andAttributeIn("mm", ManagementMode.PROPERTY_TOPIA_ID, args, growingPlanIds);
                }

                // domains
                query += DaoUtils.andAttributeInIfNotEmpty("mm", PROPERTY_DOMAIN_ID, args, navigationContext.getDomains());

                // growingPlans
                query += DaoUtils.andAttributeInIfNotEmpty("mm", PROPERTY_GROWING_PLAN_ID, args, navigationContext.getGrowingPlans());

                // growingSystems
                query += DaoUtils.andAttributeInIfNotEmpty("mm", ManagementMode.PROPERTY_TOPIA_ID, args, navigationContext.getGrowingSystems());

            }
        }

        if (!securityContext.isAdmin()) {
            query += " AND ( mm." + ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_CODE + " IN (" +
                    "SELECT DISTINCT cup." + ComputedUserPermission.PROPERTY_OBJECT +
                    " FROM " + ComputedUserPermission.class.getName() + " cup" +
                    " WHERE cup." + ComputedUserPermission.PROPERTY_USER_ID + " = :cup_userId" +
                    " AND cup." + ComputedUserPermission.PROPERTY_TYPE + " = :cup_type_code" +
                    ")" +
                    " OR mm." + ManagementMode.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_TOPIA_ID + " IN (" +
                    "SELECT DISTINCT cup." + ComputedUserPermission.PROPERTY_OBJECT +
                    " FROM " + ComputedUserPermission.class.getName() + " cup" +
                    " WHERE cup." + ComputedUserPermission.PROPERTY_USER_ID + " = :cup_userId" +
                    " AND cup." + ComputedUserPermission.PROPERTY_TYPE + " = :cup_type_id" +
                    ")" +
                    " )";
            args.put("cup_userId", securityContext.getUserId());
            args.put("cup_type_code", PermissionObjectType.GROWING_SYSTEM_CODE);
            args.put("cup_type_id", PermissionObjectType.GROWING_SYSTEM_ID);
        }

        return query;
    }

    protected Set<String> networksToGrowingSystems(Set<String> networksIds) {
        Set<String> result = getProjectionHelper().networksToGrowingSystems(networksIds);
        return result;
    }

    public List<ManagementModeCategory> findManagementModeCategories(String growingSystemId) {
        List<ManagementModeCategory> managementModeCategories = Lists.newArrayList();
        String query = "SELECT DISTINCT " +
                "mm." + ManagementMode.PROPERTY_CATEGORY + " FROM " + ManagementMode.class.getName() + " mm";
        query += " WHERE 1 = 1";
        Map<String, Object> args = Maps.newLinkedHashMap();
        query += DaoUtils.andAttributeLike("mm", PROPERTY_GROWING_SYSTEM_ID, args, growingSystemId);
        List<ManagementModeCategory> results = findAll(query + " ORDER BY mm." + ManagementMode.PROPERTY_CATEGORY, args);

        if (results != null) {
            for (ManagementModeCategory result: results) {
                managementModeCategories.add(result);
            }
        }

        return managementModeCategories;
    }

    public List<CroppingPlanEntry> getCroppingPlanEntryWithDomainCode(String domainsCode) {
        String query = " FROM " + CroppingPlanEntry.class.getName() + " cpe " +
            " WHERE cpe." + CroppingPlanEntry.PROPERTY_DOMAIN + " IN (" +
            " SELECT d FROM " + Domain.class.getName() + " d " +
            " WHERE d." + Domain.PROPERTY_CODE +" = :domainCode )";
        Map<String, Object> args = Maps.newLinkedHashMap();
        args.put("domainCode", domainsCode);
        List<CroppingPlanEntry> allCroppingPlanEntries = findAll(query + " ORDER BY cpe." + CroppingPlanEntry.PROPERTY_CODE, args);
        Map<String, CroppingPlanEntry> uniqueCroppingPlanEntryCodes = Maps.newHashMap();
        for(CroppingPlanEntry croppingPlanEntry : allCroppingPlanEntries) {
            if (uniqueCroppingPlanEntryCodes.get(croppingPlanEntry.getCode())== null) {
                uniqueCroppingPlanEntryCodes.put(croppingPlanEntry.getCode(), croppingPlanEntry);
            }
        }
        List<CroppingPlanEntry> result = Lists.newArrayList(uniqueCroppingPlanEntryCodes.values());
        return result;
    }

}
