package fr.inra.agrosyst.api.entities.managementmode;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: DecisionRuleTopiaDao.java 4653 2014-12-15 14:00:17Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/managementmode/DecisionRuleTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.util.PagerBean;

import com.google.common.collect.Maps;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.managementmode.DecisionRuleFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.security.SecurityContext;
import fr.inra.agrosyst.services.security.SecurityHelper;

public class DecisionRuleTopiaDao extends AbstractDecisionRuleTopiaDao<DecisionRule> {


    public ResultList<DecisionRule> getFilteredDecisionRules(DecisionRuleFilter filter, SecurityContext securityContext) throws TopiaException {
        StringBuilder query = new StringBuilder(" FROM " + DecisionRule.class.getName() + " dr ");
        query.append(" WHERE 1 = 1 ");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // FIXME echatellier 20140218 : this is a huge hack
        query.append(DaoUtils.andAttributeEquals("dr", DecisionRule.PROPERTY_VERSION_NUMBER, args, 1));

        // apply non null filter
        if (filter != null) {

            query.append(DaoUtils.andAttributeEquals("dr", DecisionRule.PROPERTY_ACTIVE, args, filter.getActive()));

            // decision rule name
            query.append(DaoUtils.andAttributeLike("dr", DecisionRule.PROPERTY_NAME, args, filter.getDecisionRuleName()));

            // intervention type
            query.append(DaoUtils.andAttributeEquals("dr", DecisionRule.PROPERTY_INTERVENTION_TYPE, args, filter.getInterventionType()));

            // Navigation context
            NavigationContext navigationContext = filter.getNavigationContext();
            if (navigationContext != null) {

                if (navigationContext.getCampaignsCount() == 0 &&
                        navigationContext.getNetworksCount() == 0 &&
                        navigationContext.getDomainsCount() == 0 &&
                        navigationContext.getGrowingPlansCount() == 0 &&
                        navigationContext.getGrowingSystemsCount() == 0){

                    // we load all active domain's code
                    Set< String > domainCodes = getProjectionHelper().domainsToDomainCodes(null, true);
                    query.append(DaoUtils.andAttributeIn("dr", DecisionRule.PROPERTY_DOMAIN_CODE, args, domainCodes));
                } else {
                    // campaigns
                    if(navigationContext.getCampaignsCount() > 0 &&
                            navigationContext.getNetworksCount() == 0 &&
                            navigationContext.getDomainsCount() == 0 &&
                            navigationContext.getGrowingPlansCount() == 0 &&
                            navigationContext.getGrowingSystemsCount() == 0) {
                        Set< String > domainCodes = getProjectionHelper().campaignsToDomainCodes(navigationContext.getCampaigns(), true);
                        query.append(DaoUtils.andAttributeIn("dr", DecisionRule.PROPERTY_DOMAIN_CODE, args, domainCodes));
                    }

                    // networks
                    if(navigationContext.getNetworksCount() > 0) {
                        Set<String> domainCodes = getProjectionHelper().networksToDomainsCode(navigationContext.getNetworks(), true);
                        query.append(DaoUtils.andAttributeIn("dr", DecisionRule.PROPERTY_DOMAIN_CODE, args, domainCodes));
                    }

                    // domains
                    if(navigationContext.getDomainsCount() > 0) {
                        Set< String > domainCodes = getProjectionHelper().domainsToDomainCodes(navigationContext.getDomains(), true);
                        query.append(DaoUtils.andAttributeIn("dr", DecisionRule.PROPERTY_DOMAIN_CODE, args, domainCodes));
                    }

                    // growingPlan
                    if(navigationContext.getGrowingPlansCount() > 0) {
                        Set< String > domainCodes = getProjectionHelper().growingPlansToDomainCodes(navigationContext.getGrowingPlans(), true);
                        query.append(DaoUtils.andAttributeIn("dr", DecisionRule.PROPERTY_DOMAIN_CODE, args, domainCodes));
                    }

                    // growingSystem
                    if(navigationContext.getGrowingSystemsCount() > 0) {
                        Set< String > domainCodes = getProjectionHelper().growingSystemsToDomainCodes(navigationContext.getGrowingSystems(), true);
                        query.append(DaoUtils.andAttributeIn("dr", DecisionRule.PROPERTY_DOMAIN_CODE, args, domainCodes));
                    }
                }
            } else {
                // we load all active domain's code
                Set< String > domainCodes = getProjectionHelper().domainsToDomainCodes(null, true);
                query.append(DaoUtils.andAttributeIn("dr", DecisionRule.PROPERTY_DOMAIN_CODE, args, domainCodes));
            }
        }

        SecurityHelper.addDecisionRuleFilter(query, args, securityContext, "dr");

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;
        List<DecisionRule> decisionRules = find(query + " ORDER BY lower (dr." + DecisionRule.PROPERTY_NAME + "), dr." + DecisionRule.PROPERTY_TOPIA_ID, args, startIndex, endIndex);
        long totalCount = findUnique("SELECT count(" + DecisionRule.PROPERTY_TOPIA_ID + ") " + query, args);

        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);
        ResultList<DecisionRule> result = ResultList.of(decisionRules, pager);

        return result;
    }

    protected Set<String> networksToGrowingSystems(Set<String> networksIds) {
        Set<String> result = getProjectionHelper().networksToGrowingSystems(networksIds);
        return result;
    }

    public List<DecisionRule> findAllRelatedDecisionRules(String code) {
        String query = " FROM " + getEntityClass().getName()
                + " WHERE " + DecisionRule.PROPERTY_CODE + " = :code"
                + " ORDER BY " + DecisionRule.PROPERTY_VERSION_NUMBER + " ASC";
        List<DecisionRule> result = findAll(query, DaoUtils.asArgsMap("code", code));
        return result;
    }

    public DecisionRule getLastRuleVersion(String code) {
        String query = "FROM " + DecisionRule.class.getName() + " dr";
        query += " WHERE 1 = 1";
        query += " AND dr." + DecisionRule.PROPERTY_CODE + " = :code ";

        String hql = query + " ORDER BY dr." + DecisionRule.PROPERTY_VERSION_NUMBER + " DESC";
        Map<String, Object> args = DaoUtils.asArgsMap("code", code);
        DecisionRule result = findFirstOrNull(hql, args);

        return result;
    }
}
