package fr.inra.agrosyst.api.entities;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: ZoneTopiaDao.java 4731 2015-01-20 16:10:53Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/ZoneTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.util.PagerBean;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.effective.EffectivePerennialCropCycle;
import fr.inra.agrosyst.api.entities.effective.EffectiveSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.measure.MeasurementSession;
import fr.inra.agrosyst.api.entities.performance.Performance;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.effective.EffectiveZoneFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.security.SecurityContext;
import fr.inra.agrosyst.services.security.SecurityHelper;

public class ZoneTopiaDao extends AbstractZoneTopiaDao<Zone> {

    protected static final String PROPERTY_PLOT_NAME = Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_NAME;

    /**
     * query from plot LEFT OUTER JOIN pl
     */
    protected static final String PROPERTY_DOMAIN = Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_DOMAIN;
    protected static final String PROPERTY_DOMAIN_NAME = PROPERTY_DOMAIN + "." + Domain.PROPERTY_NAME;
    protected static final String PROPERTY_DOMAIN_CAMPAIGN = PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN;
    protected static final String PROPERTY_GROWING_SYSTEM = Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_GROWING_SYSTEM;


    private static final Log log = LogFactory.getLog(ZoneTopiaDao.class);

    public ResultList<Zone> getFilteredZones(EffectiveZoneFilter filter, SecurityContext securityContext) throws TopiaException {

        // used of OUTER JOIN to be able to use ORDER BY on growing system that can be null
        StringBuilder query = new StringBuilder(
                // setup Select because of
                "FROM " + Zone.class.getName() +
                        " z LEFT OUTER JOIN z." + Zone.PROPERTY_PLOT + " AS pl " +
                        " LEFT OUTER JOIN pl." + Plot.PROPERTY_GROWING_SYSTEM + " gs " +
                        " LEFT OUTER JOIN pl." + Plot.PROPERTY_DOMAIN + " domain " +
                        " LEFT OUTER JOIN gs." + GrowingSystem.PROPERTY_GROWING_PLAN + " gp "
        );
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // apply non null filter
        if (filter != null) {

            // zone name
            query.append(DaoUtils.andAttributeLike("z", Zone.PROPERTY_NAME, args, filter.getZoneName()));
            
            // plot name
            query.append(DaoUtils.andAttributeLike("pl", Plot.PROPERTY_NAME, args, filter.getPlotName()));

            // active plot
            query.append(DaoUtils.andAttributeEquals("pl", Plot.PROPERTY_ACTIVE, args, filter.getActive()));

            // domain name
            query.append(DaoUtils.andAttributeLike("domain", Domain.PROPERTY_NAME, args, filter.getDomainName()));
            
            // growing plan name
            query.append(DaoUtils.andAttributeLike("gp", GrowingPlan.PROPERTY_NAME, args, filter.getGrowingPlanName()));
            
            // growing system name
            query.append(DaoUtils.andAttributeLike("gs", GrowingSystem.PROPERTY_NAME, args, filter.getGrowingSystemName()));

            // campaign
            query.append(DaoUtils.andAttributeEquals("domain", Domain.PROPERTY_CAMPAIGN, args, filter.getCampaign()));

            // Navigation context
            NavigationContext navigationContext = filter.getNavigationContext();
            if (navigationContext != null) {

                // campaigns
                query.append(DaoUtils.andAttributeInIfNotEmpty("domain", Domain.PROPERTY_CAMPAIGN, args, navigationContext.getCampaigns()));

                // networks
                if (navigationContext.getNetworksCount() > 0) {

                    Set<String> domainIds = networksToDomains(navigationContext.getNetworks());
                    query.append(DaoUtils.andAttributeIn("domain", Domain.PROPERTY_TOPIA_ID, args, domainIds));
                }

                // domains
                Set<String> domains = Sets.newHashSet(navigationContext.getDomains());
                if (!navigationContext.getGrowingSystems().isEmpty()) {
                    String hql = "SELECT" +
                            " gs." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID +
                            " FROM " + GrowingSystem.class.getName() + " gs" +
                            " WHERE gs." + GrowingSystem.PROPERTY_TOPIA_ID + " IN :gsIds";
                    List<String> domainsIds = findAll(hql, DaoUtils.asArgsMap("gsIds", navigationContext.getGrowingSystems()));
                    domains.addAll(domainsIds);
                }
                if (!navigationContext.getGrowingPlans().isEmpty()) {
                    String hql = "SELECT" +
                            " gp." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID +
                            " FROM " + GrowingPlan.class.getName() + " gp" +
                            " WHERE gp." + GrowingPlan.PROPERTY_TOPIA_ID + " IN :gpIds";
                    List<String> domainsIds = findAll(hql, DaoUtils.asArgsMap("gpIds", navigationContext.getGrowingPlans()));
                    domains.addAll(domainsIds);
                }

                query.append(DaoUtils.andAttributeInIfNotEmpty("domain", Domain.PROPERTY_TOPIA_ID, args, domains));
            }
        }

        SecurityHelper.addZoneFilter(query, args, securityContext, "z");

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;

        // add SELECT to the query otherwise a List of object [] is returned
        List<Zone> zones = find("SELECT z " + query +
                " ORDER BY domain." + Domain.PROPERTY_CAMPAIGN + " DESC, " +
                " lower(domain." + Domain.PROPERTY_NAME + ")," +
                " lower(gp." + GrowingPlan.PROPERTY_NAME +") NULLS LAST," +
                " lower(gs." + GrowingSystem.PROPERTY_NAME + ") NULLS LAST," +
                " lower(pl." + Plot.PROPERTY_NAME + ")," +
                " lower(z." + Zone.PROPERTY_NAME + ")",
                args, startIndex, endIndex);

        long totalCount = findUnique("SELECT count(*) " + query, args);
        
        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);
        ResultList<Zone> result = ResultList.of(zones, pager);
        return result;
    }
    
    protected Set<String> networksToDomains(Set<String> networksIds) {
        Set<String> result = getProjectionHelper().networksToDomains(networksIds);
        return result;
    }

    /**
     * Find all zone using same zone's duplication code.
     *
     * @return related zones
     */
    public LinkedHashMap<Integer, String> findAllRelatedZones(String code) {
        String query = "SELECT " + PROPERTY_DOMAIN_CAMPAIGN + ", " + Zone.PROPERTY_TOPIA_ID
                + " FROM " + Zone.class.getName()
                + " WHERE " + Zone.PROPERTY_CODE + " = :code"
                + " ORDER BY " + PROPERTY_DOMAIN_CAMPAIGN + " DESC";
        List<Object[]> zones = findAll(query, DaoUtils.asArgsMap("code", code));
        LinkedHashMap<Integer, String> result = DaoUtils.toRelatedMap(zones);
        return result;
    }

    public List<Zone> findZonesWithoutCycle(String zoneId, String domainId) {
        String query = "FROM " + Zone.class.getName() + " z "
                     + "WHERE z." + PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID + " =:domainId "
                     + "AND z." + Zone.PROPERTY_ACTIVE + " = true "
                     + "AND z NOT IN "
                     + "(SELECT epcc." + EffectivePerennialCropCycle.PROPERTY_ZONE + " FROM " + EffectivePerennialCropCycle.class.getName() + " epcc "
                     + " WHERE epcc." + EffectivePerennialCropCycle.PROPERTY_ZONE + "." + PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID + " =:domainId "
                     + ")"
                     + "AND z NOT IN "
                     + "(SELECT escc." + EffectiveSeasonalCropCycle.PROPERTY_ZONE + " FROM " + EffectiveSeasonalCropCycle.class.getName() + " escc "
                     + " WHERE escc." + EffectiveSeasonalCropCycle.PROPERTY_ZONE + "." + PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID + " =:domainId)"
                     + " AND z." + Zone.PROPERTY_TOPIA_ID + " != :zoneId";
        Map<String, Object> args = Maps.newLinkedHashMap();
        args.put("domainId", domainId);
        args.put("zoneId", zoneId);
        List<Zone> res = findAll(query, args);

        return res;
    }

    public Map<String, Long> forQuery(Iterable<String> zonesIds, String subQuery) {
        Map<String, Long> result = Maps.newHashMap();

        Map<String, Object> args = Maps.newLinkedHashMap();
        Set<String> zIds = Sets.newHashSet(zonesIds);

        Preconditions.checkArgument(zIds != null && !zIds.isEmpty());

        String query = " SELECT z." + Zone.PROPERTY_TOPIA_ID +", ";

        query += "(" + subQuery + ")";

        query += " FROM " + Zone.class.getName() + " z  WHERE 1 = 1 " ;

        query += DaoUtils.andAttributeIn("z", Zone.PROPERTY_TOPIA_ID, args, zIds);

        List<Object[]> zonesTuples = findAll(query, args);
        for (Object[] zoneTuple : zonesTuples) {
            String zoneId = (String) zoneTuple[0];
            Long count = (Long)zoneTuple[1];
//            if (log.isDebugEnabled()) {
//                log.debug(
//                        "Pour zone:" + zoneId + " nbUsed:" + count);
//            }
            result.put(zoneId, count);
        }

        return result;
    }

    public Map<String, Long> getZonesUsedForPerformances(Iterable<String> zonesIds) {
        String query =
            "SELECT COUNT(*) FROM " + Performance.class.getName() + " p " +
            " WHERE z IN ELEMENTS (p." + Performance.PROPERTY_ZONES + ")";
        Map<String, Long> result = forQuery(zonesIds, query);
        return result;
    }

    public Map<String, Long> getZonesUsedForMesurementSessions(Iterable<String> zonesIds) {
        String query =
            "SELECT COUNT(*) FROM " + MeasurementSession.class.getName() + " ms " +
            " WHERE ms." + MeasurementSession.PROPERTY_ZONE + " = z";
        Map<String, Long> result = forQuery(zonesIds, query);
        return result;
    }

    public Map<String, Long> getZonesUsedForEffectivePerennialCropCycles(Iterable<String> zonesIds) {
        String query =
            "SELECT COUNT(*) FROM " + EffectivePerennialCropCycle.class.getName() + " epcc " +
            " WHERE epcc." + EffectivePerennialCropCycle.PROPERTY_ZONE + "= z";
        Map<String, Long> result = forQuery(zonesIds, query);
        return result;
    }

    public Map<String, Long> getZonesUsedForEffectiveSeasonalCropCycles(Iterable<String> zonesIds) {
        String query =
            "SELECT COUNT(*) FROM " + EffectiveSeasonalCropCycle.class.getName() + " escc " +
            " WHERE escc." + EffectiveSeasonalCropCycle.PROPERTY_ZONE + " = z";
        Map<String, Long> result = forQuery(zonesIds, query);
        return result;
    }

    public List<Zone> findZonesFromDomainCode(String domainCode, SecurityContext securityContext) {
        Map<String, Object> args = Maps.newLinkedHashMap();

        StringBuilder query = new StringBuilder(" FROM " + Zone.class.getName() + " z ");
        query.append(" WHERE z." + PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE + " = :domainCode ");
        args.put("domainCode", domainCode);
        query.append(" AND z." + PROPERTY_DOMAIN + "." + Domain.PROPERTY_ACTIVE + " = TRUE");
        query.append(" AND z." + Zone.PROPERTY_PLOT + "." + Plot.PROPERTY_ACTIVE + " = TRUE ");
        query.append(" AND z." + Zone.PROPERTY_ACTIVE + " = TRUE ");

        query.append(" AND ( z." + Zone.PROPERTY_TOPIA_ID);
        query.append("       IN (");
        query.append("            SELECT z1." + Zone.PROPERTY_TOPIA_ID + " FROM " + Zone.class.getName() + " z1 ");
        query.append("            WHERE z1." + PROPERTY_GROWING_SYSTEM + " IS NULL ");
        query.append("           )");
        query.append("       OR z." + Zone.PROPERTY_TOPIA_ID);
        query.append("       IN (");
        query.append("            SELECT z2." + Zone.PROPERTY_TOPIA_ID + " FROM " + Zone.class.getName() + " z2 ");
        query.append("            WHERE z2." + PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_ACTIVE + " = TRUE");
        query.append("           )");
        query.append("     )");

        SecurityHelper.addZoneFilter(query, args, securityContext, "z");
        List<Zone> zones = findAll(query + " ORDER BY lower(z." + Zone.PROPERTY_NAME + "), lower (z." + PROPERTY_PLOT_NAME + "), lower(z." + PROPERTY_DOMAIN_NAME + "), z." + PROPERTY_DOMAIN_CAMPAIGN, args);
        return zones;
    }
}
