package fr.inra.agrosyst.api.entities;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PriceTopiaDao.java 4612 2014-12-06 18:43:48Z echatellier $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/PriceTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.services.common.ProductPriceSummary;
import fr.inra.agrosyst.api.services.common.ProductPrices;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.common.CommonService;

public class PriceTopiaDao extends AbstractPriceTopiaDao<Price> {

    public ProductPrices computePriceIndication(Price filter,
                                                Set<Integer> campaigns,
                                                String excludeDomainId,
                                                String excludePracticedSystemId) {
        Preconditions.checkArgument(campaigns != null && !campaigns.isEmpty());

        Map<String, Object> args = Maps.newHashMap();

        // projection for price count, average, minimum, and maximum
        String queryProjectionStats = "SELECT " +
                " count(p." + Price.PROPERTY_TOPIA_ID + "), " +
                " avg  (p." + Price.PROPERTY_PRICE + "), " +
                " min  (p." + Price.PROPERTY_PRICE + "), " +
                " max  (p." + Price.PROPERTY_PRICE + ")  ";

        // projection for price median
        String queryProjectionAll = "SELECT p." + Price.PROPERTY_PRICE;

        // Query filter shared between all possible queries
        String queryFilter = " FROM  " + getEntityClass().getName() + " p " +
                " WHERE p." + Price.PROPERTY_PRICE + " IS NOT NULL ";

        // price type
        queryFilter += DaoUtils.andAttributeEquals("p", Price.PROPERTY_TYPE, args, filter.getType());
        // price objectId
        queryFilter += DaoUtils.andAttributeEquals("p", Price.PROPERTY_OBJECT_ID, args, filter.getObjectId());
        // price sourceUnit
        queryFilter += DaoUtils.andAttributeEquals("p", Price.PROPERTY_PRICE_UNIT, args, filter.getPriceUnit());

        if (!Strings.isNullOrEmpty(excludeDomainId)) {
            queryFilter += " AND p." + Price.PROPERTY_DOMAIN + " IS NOT NULL ";
            queryFilter += DaoUtils.andAttributeNotEquals("p", Price.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID, args, excludeDomainId);
        } else if (!Strings.isNullOrEmpty(excludePracticedSystemId)) {
            queryFilter += " AND p." + Price.PROPERTY_PRACTICED_SYSTEM + " IS NOT NULL ";
            queryFilter += DaoUtils.andAttributeNotEquals("p", Price.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_TOPIA_ID, args, excludePracticedSystemId);
        }

        Object[] allCampaignsObjects = findAnyOrNull(queryProjectionStats + queryFilter, args);
        ProductPriceSummary allCampaignsPricesSummary = toPriceSummary(allCampaignsObjects);

        String orderedResult = " ORDER BY p." + Price.PROPERTY_PRICE + " ASC ";

        if (allCampaignsPricesSummary.getCountedPrices() > 0L) {
            List<Double> allCampaignsPrices = findAll(queryProjectionAll + queryFilter + orderedResult, args);
            Double median = DaoUtils.median(allCampaignsPrices);
            allCampaignsPricesSummary.setMedianPrice(median);
        }

        if (!Strings.isNullOrEmpty(excludeDomainId)) {
            queryFilter += DaoUtils.andAttributeIn("p", Price.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN, args, campaigns);
        } else if (!Strings.isNullOrEmpty(excludePracticedSystemId)) {
            queryFilter += DaoUtils.andAttributeEquals(
                    "p",
                    Price.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_CAMPAIGNS,
                    args,
                    CommonService.ARRANGE_CAMPAIGNS_SET.apply(campaigns)); // TODO AThimel 11/11/13 Pas top, ça ne prend que l'égalité parfaite sur les années
        }

        Object[] currentCampaignsObjects = findAnyOrNull(queryProjectionStats + queryFilter, args);
        ProductPriceSummary currentCampaignsPricesSummary = toPriceSummary(currentCampaignsObjects);

        if (currentCampaignsPricesSummary.getCountedPrices() > 0L) {
            List<Double> currentCampaignsCampaignsPrices = findAll(queryProjectionAll + queryFilter + orderedResult, args);
            Double median = DaoUtils.median(currentCampaignsCampaignsPrices);
            currentCampaignsPricesSummary.setMedianPrice(median);
        }

        ProductPrices result = new ProductPrices();
        result.setAllCampaignsProductPriceSummary(allCampaignsPricesSummary);
        result.setCurrentCampaignsProductPriceSummary(currentCampaignsPricesSummary);
        result.setCurrentCampaigns(CommonService.ARRANGE_CAMPAIGNS_SET.apply(campaigns));

        return result;
    }

    protected ProductPriceSummary toPriceSummary(Object[] objects) {

        ProductPriceSummary result = new ProductPriceSummary();
        Long count = (Long) objects[0];
        result.setCountedPrices(count);
        if (count > 0l) {
            result.setAveragePrice((Double) objects[1]);
            result.setLowerPrice((Double) objects[2]);
            result.setHigherPrice((Double) objects[3]);
        }
        return result;
    }


    public List<Price> getPrices0(String domainTopiaId, String practicedSystemId, List<String> objectIds) {
        List<Price> result;

        StringBuilder query = new StringBuilder("FROM " + getEntityClass().getName() + " p ");
        query.append(" WHERE 1 = 1");

        Map<String, Object> args = Maps.newHashMap();

        if (!Strings.isNullOrEmpty(domainTopiaId)) {

            query.append(DaoUtils.andAttributeLike("p", Price.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID, args, domainTopiaId));

            if (objectIds != null) {
                Set<String> objectIds2 = Sets.newHashSet(objectIds);
                query.append(DaoUtils.andAttributeInIfNotEmpty("p", Price.PROPERTY_OBJECT_ID, args, objectIds2));
                // TODO david find a way to get fuel prices.
            }
        } else if (!Strings.isNullOrEmpty(practicedSystemId)) {
            query.append(DaoUtils.andAttributeEquals("p", Price.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_TOPIA_ID, args, practicedSystemId));
        }

        result = findAll(query.toString(), args);


        return result;
    }

    public List<Price> getPricesForObjectIdLike(String objectId) {
        List<Price> result;
        Map<String, Object> args = Maps.newHashMap();
        StringBuilder query = new StringBuilder("FROM " + getEntityClass().getName() + " p ");
        query.append(" WHERE 1 = 1");
        query.append(DaoUtils.andAttributeLike("p", Price.PROPERTY_OBJECT_ID ,args,  objectId));
        result = findAll(query.toString(), args);
        return result;
    }

} //PriceTopiaDao
