package fr.inra.agrosyst.api.entities;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: PlotTopiaDao.java 3840 2014-03-21 10:48:23Z athimel $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/PlotTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import fr.inra.agrosyst.api.utils.DaoUtils;

/**
 * @author cosse
 */
public class PlotTopiaDao extends AbstractPlotTopiaDao<Plot> {

    protected static final String PROPERTY_DOMAIN_ID = Plot.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID;
    protected static final String PROPERTY_DOMAIN_CAMPAIGN = Plot.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN;

    public List<Plot> findAllByDomainId(String domainId) {
        List<Plot> result = forProperties(PROPERTY_DOMAIN_ID, domainId).findAll();
        return result;
    }
    
    public List<Plot> findAllByDomainOrderByName(Domain domain) {
        StringBuilder query = new StringBuilder(" FROM " + Plot.class.getName() + " p ");
        query.append(" WHERE p." + Plot.PROPERTY_DOMAIN + " = :domain ");

        Map<String, Object> args = DaoUtils.asArgsMap("domain", domain);

        query.append(" ORDER BY p." + Plot.PROPERTY_NAME);


        List<Plot> result = findAll(query.toString(), args);
        return result;
    }

    public List<Plot> findAllFreePlotInDomain(String domainId) {
        String query = "FROM " + Plot.class.getName() + " P";
        query += " WHERE " + PROPERTY_DOMAIN_ID + " = :domainId";
        query += " AND " + Plot.PROPERTY_GROWING_SYSTEM + " is null";
        List<Plot> result = findAll(query, DaoUtils.asArgsMap("domainId", domainId));
        return result;
    }

    /**
     * Find all plot using same plot's duplication code.
     *
     * @return related plots
     */
    public LinkedHashMap<Integer, String> findAllRelatedPlots(String code) {
        String query = "SELECT " + PROPERTY_DOMAIN_CAMPAIGN + ", " + Plot.PROPERTY_TOPIA_ID
                + " FROM " + getEntityClass().getName()
                + " WHERE " + Plot.PROPERTY_CODE + " = :code"
                + " ORDER BY " + PROPERTY_DOMAIN_CAMPAIGN + " DESC";
        List<Object[]> plots = findAll(query, DaoUtils.asArgsMap("code", code));
        LinkedHashMap<Integer, String> result = DaoUtils.toRelatedMap(plots);
        return result;
    }

    /**
     * Get domain'plot total area.
     * 
     * @param domainId domain id
     * @return total area
     */
    public double getDomainPlotTotalArea(String domainId) {
        String query = "SELECT sum(" + Plot.PROPERTY_AREA + ")"
                + " FROM " + Plot.class.getName()
                + " WHERE " + PROPERTY_DOMAIN_ID + " = :domainId"
                + " AND " + Plot.PROPERTY_ACTIVE + " = true";
        Double totalArea = findUniqueOrNull(query, DaoUtils.asArgsMap("domainId", domainId));
        Double result = totalArea == null ? 0.0 : totalArea;
        return result;
    }

    public Long countAllGrowingSystemsForPlots(Set<String> plotTopiaIds) {
        Preconditions.checkArgument(plotTopiaIds != null && !plotTopiaIds.isEmpty());
        StringBuilder query = new StringBuilder(
                "SELECT count(DISTINCT p." + Plot.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_TOPIA_ID + ") FROM " + Plot.class.getName() + " p ");
        query.append(" WHERE 1 = 1");

        Map<String, Object> args = Maps.newLinkedHashMap();

        query.append(DaoUtils.andAttributeIn("p", Plot.PROPERTY_TOPIA_ID, args, plotTopiaIds));

        long result = findUnique(query.toString(), args);

        return result;
    }
}
