package fr.inra.agrosyst.api.entities;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: NetworkTopiaDao.java 4210 2014-07-21 12:06:31Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/NetworkTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.network.NetworkFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import org.nuiton.util.PagerBean;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author cosse
 *
 */
public class NetworkTopiaDao extends AbstractNetworkTopiaDao<Network> {
    
    protected static final String PROPERTY_CAMPAIGN = Network.PROPERTY_NAME;

    public ResultList<Network> getFilteredNetworks(NetworkFilter filter, List<NetworkManager> managers) {
        StringBuilder query = new StringBuilder("FROM " + getEntityClass().getName() + " n ");
        query.append(" WHERE 1 = 1");

        Map<String, Object> args = Maps.newLinkedHashMap();

        // apply non null filter
        if (filter != null) {
            // network name
            query.append(DaoUtils.andAttributeLike("n", Network.PROPERTY_NAME, args, filter.getNetworkName()));

            // active
            query.append(DaoUtils.andAttributeEquals("n", Network.PROPERTY_ACTIVE, args, filter.getActive()));

            // network managers
            if (managers != null) { // TODO AThimel 08/08/13 Find a better solution
                query.append(" AND ( 1=0 ");
                int index = 0;
                for (NetworkManager manager : managers) {
                    String key = "manager" + index++;
                    query.append(String.format(" OR :%s in elements ( %s )", key, "n." + Network.PROPERTY_MANAGERS));
                    args.put(key, manager);
                }
                query.append(" ) ");
            }

        }

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;
        List<Network> networks = find(query + " ORDER BY lower (n." + Network.PROPERTY_NAME + ")", args, startIndex, endIndex);
        long totalCount = findUnique("SELECT count(*) " + query, args);

        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);
        ResultList<Network> result = ResultList.of(networks, pager);
        return result;
    }

    protected Set<String> buildFullParentSet(Set<String> networkIds) {
        Set<String> result = Sets.newHashSet();
        if (networkIds != null) {
            for (String networkId : networkIds) {
                buildParentSet(result, networkId);
            }
        }
        return result;
    }

    protected void buildParentSet(Set<String> result, String networkId) {
        if (!Strings.isNullOrEmpty(networkId) && !result.contains(networkId)) {
            Network network = forTopiaIdEquals(networkId).findUnique();
            result.add(networkId);
            if (network.getParents() != null) {
                for (Network parent : network.getParents()) {
                    buildParentSet(result, parent.getTopiaId());
                }
            }
        }
    }

    public LinkedHashMap<String, String> getNameFilteredNetworks(String name, Integer size, Set<String> exclusions,
                                                                 String selfNetworkId, boolean onNetworkEdition) {
        String query = "SELECT n."+ Network.PROPERTY_TOPIA_ID +", n." + Network.PROPERTY_NAME
                + " FROM " + getEntityClass().getName() + " n"
                + " WHERE n." + Network.PROPERTY_ACTIVE + " = true ";

        Map<String, Object> args = Maps.newLinkedHashMap();
        query += DaoUtils.andAttributeLike("n", Network.PROPERTY_NAME, args, name);

        // Exclude parents
        Set<String> fullExclusionSet = buildFullParentSet(exclusions);

        // Exclude self and children networks
        if (!Strings.isNullOrEmpty(selfNetworkId)) {
            fullExclusionSet.add(selfNetworkId);
            // Add all its children to avoid loops
            Set<Network> children = loadNetworksWithDescendantHierarchy(Sets.newHashSet(selfNetworkId));
            Iterables.addAll(fullExclusionSet, Iterables.transform(children, Entities.GET_TOPIA_ID));
        }

        if (onNetworkEdition) {
            // Ignore all networks having some SdCs has children
            GrowingSystemTopiaDao growingSystemDAO = topiaDaoSupplier.getDao(GrowingSystem.class, GrowingSystemTopiaDao.class);
            Iterable<String> usedByGrowingSystemsNetworks = growingSystemDAO.getAllNetworksUsedByGrowingSystems();
            Iterables.addAll(fullExclusionSet, usedByGrowingSystemsNetworks);
        } else {
            // Ignore all networks having some networks has children
            Iterable<String> usedByNetworks = getAllNetworksUsedAsParents();
            Iterables.addAll(fullExclusionSet, usedByNetworks);
        }

        query += DaoUtils.andAttributeNotIn("n", Network.PROPERTY_TOPIA_ID, args, fullExclusionSet);

        int nbResult = size != null ? size : 10;
        String order = " ORDER BY lower (n." + Network.PROPERTY_NAME + ")";
        List<Object[]> networks = find(query + order, args, 0, nbResult);
        LinkedHashMap<String, String> result = Maps.newLinkedHashMap();
        for (Object[] network : networks) {
            String topiaId = (String) network[0];
            String networkName = (String) network[1];
            result.put(topiaId, networkName);
        }
        return result;
    }

    public Set<Network> loadNetworksWithDescendantHierarchy(Set<String> networkIds) {

        Set<Network> result  = Sets.newHashSet();
        for (String networkId : networkIds) { // TODO AThimel 19/09/13 All can be loaded with only 1 query
            Network network = forTopiaIdEquals(networkId).findUnique();
            result.add(network);
        }
        Set<Network> children = findNetworkDescendantHierarchyExcluding(result, new HashSet<String>());
        result.addAll(children);

        return result;
    }

    protected Set<Network> findNetworkDescendantHierarchyExcluding(Collection<Network> networks, Set<String> networksToExclude) {
        Set<Network> children = Sets.newHashSet();
        if (networks != null && !networks.isEmpty()) {
            Preconditions.checkArgument(networksToExclude != null);
            Set<String> networksExcluded = Sets.newHashSet(networksToExclude);
            StringBuilder query = new StringBuilder("FROM " + getEntityClass().getName() + " n");
            query.append(" WHERE 1=1");
            Map<String, Object> args = Maps.newLinkedHashMap();

            query.append(" AND ( 1=0 ");
            int index = 0;
            for (Network parent : networks) {
                networksExcluded.add(parent.getTopiaId());
                String key = "parent" + index++;
                query.append(String.format(" OR :%s in elements ( %s )", key, "n." + Network.PROPERTY_PARENTS));
                args.put(key, parent);
            }
            query.append(" ) ");

            // networksToExclude
            query.append(DaoUtils.andAttributeNotIn("n", Network.PROPERTY_TOPIA_ID, args, networksExcluded));

            List<Network> networkList = findAll(query.toString(), args);
            if (!networkList.isEmpty()) {
                children.addAll(networkList);
                children.addAll(findNetworkDescendantHierarchyExcluding(networkList, networksExcluded));
            }
        }

        return children;
    }

    public Iterable<String> getAllNetworksUsedAsParents() {
        String hql = " SELECT DISTINCT n." + Network.PROPERTY_PARENTS +
                " FROM " + getEntityClass().getName() + " n " ;
        List<Network> networks = findAll(hql, DaoUtils.asArgsMap());

        Iterable<String> result = Iterables.transform(networks, Entities.GET_TOPIA_ID);
        return result;
    }

    public Iterable<Network> findAllByNameWithoutCase(String name) {
        String hql = newFromClause() + " WHERE UPPER(" + Network.PROPERTY_NAME + ") = :name";
        List<Network> networks = findAll(hql, DaoUtils.asArgsMap("name", name.toUpperCase()));
        return networks;
    }

}
