package fr.inra.agrosyst.api.entities;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: GrowingSystemTopiaDao.java 4948 2015-05-21 08:43:01Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/GrowingSystemTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Objects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.managementmode.ManagementMode;
import fr.inra.agrosyst.api.entities.managementmode.ManagementModeCategory;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.common.GrowingSystemsIndicator;
import fr.inra.agrosyst.api.services.growingsystem.GrowingSystemFilter;
import fr.inra.agrosyst.api.services.network.NetworkFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.common.ProjectionHelper;
import fr.inra.agrosyst.services.security.SecurityContext;
import fr.inra.agrosyst.services.security.SecurityHelper;
import org.apache.commons.lang3.StringUtils;
import org.nuiton.util.PagerBean;

import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;

/**
 * @author Arnaud Thimel : thimel@codelutin.com
 */
public class GrowingSystemTopiaDao extends AbstractGrowingSystemTopiaDao<GrowingSystem> {

    public static final String PROPERTY_GROWING_PLAN_DOMAIN = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN;
    public static final String PROPERTY_GROWING_PLAN_DOMAIN_ID = PROPERTY_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID;
    public static final String PROPERTY_GROWING_PLAN_ID = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_TOPIA_ID;
    protected static final String PROPERTY_GROWING_PLAN_TYPE = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_TYPE;
    protected static final String PROPERTY_GROWING_PLAN_NAME = GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_NAME;
    protected static final String PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN = PROPERTY_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN;
    protected static final String PROPERTY_GROWING_PLAN_DOMAIN_NAME = PROPERTY_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_NAME;

    public ResultList<GrowingSystem> getFilteredGrowingSystems(GrowingSystemFilter filter, SecurityContext securityContext) {
        StringBuilder query = new StringBuilder("FROM " + getEntityClass().getName() + " gs");
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // apply non null filter
        if (filter != null) {
            addFilerQuery(filter, query, args);
        }

        SecurityHelper.addGrowingSystemFilter(query, args, securityContext, "gs");

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;
        String queryString = query.toString();
        String queryAndOrder = queryString + " ORDER BY " +
                "lower ( gs." + GrowingSystem.PROPERTY_NAME + " )" +
                ", gs." + PROPERTY_GROWING_PLAN_NAME +
                ", gs." + PROPERTY_GROWING_PLAN_DOMAIN_NAME +
                ", gs." + PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN;
        List<GrowingSystem> growingSystems = find(queryAndOrder, args, startIndex, endIndex);
        long totalCount = findUnique("SELECT count(*) " + queryString, args);

        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);
        ResultList<GrowingSystem> result = ResultList.of(growingSystems, pager);
        return result;
    }

    protected void addFilerQuery(GrowingSystemFilter filter, StringBuilder query, Map<String, Object> args) {
        // SDC name
        query.append(DaoUtils.andAttributeLike("gs", GrowingSystem.PROPERTY_NAME, args, filter.getGrowingSystemName()));

        // SDC numéro DEPHY
        query.append(DaoUtils.andAttributeLike("gs", GrowingSystem.PROPERTY_DEPHY_NUMBER, args, filter.getDephyNumber()));

        // SDC filière
        query.append(DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_SECTOR, args, filter.getSector()));

        // Domain name
        query.append(DaoUtils.andAttributeLike("gs", PROPERTY_GROWING_PLAN_DOMAIN_NAME, args, filter.getDomainName()));

        // Growing plan name
        query.append(DaoUtils.andAttributeLike("gs", PROPERTY_GROWING_PLAN_NAME, args, filter.getGrowingPlanName()));

        // active
        query.append(DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_ACTIVE, args, filter.getActive()));

        // validated
        query.append(DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_VALIDATED, args, filter.getValidated()));

        // campaign
        query.append(DaoUtils.andAttributeEquals("gs", PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN, args, filter.getCampaign()));

        // network
        if (StringUtils.isNotEmpty(filter.getNetwork())) {
            NetworkTopiaDao networkDAO = topiaDaoSupplier.getDao(Network.class, NetworkTopiaDao.class);
            NetworkFilter networkFilter = new NetworkFilter();
            networkFilter.setNetworkName(filter.getNetwork());
            ResultList<Network> networks = networkDAO.getFilteredNetworks(networkFilter, null);

            Set<String> networkIds = Sets.newHashSet(Collections2.transform(networks.getElements(), Entities.GET_TOPIA_ID));
            Set<String> growingSystemIds = getProjectionHelper().networksToGrowingSystems(networkIds);
            query.append(DaoUtils.andAttributeIn("gs", GrowingSystem.PROPERTY_TOPIA_ID, args, growingSystemIds));
        }

        // Navigation context
        NavigationContext navigationContext = filter.getNavigationContext();
        if (navigationContext != null) {
            addNavigationContextQuery(query, args, navigationContext);
        }
    }

    protected void addNavigationContextQuery(StringBuilder query, Map<String, Object> args, NavigationContext navigationContext) {
        // campaigns
        query.append(DaoUtils.andAttributeInIfNotEmpty("gs", PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN, args, navigationContext.getCampaigns()));

        // networks
        if (navigationContext.getNetworksCount() > 0) {
            Set<String> growingSystemIds = getProjectionHelper().networksToGrowingSystems(navigationContext.getNetworks());
            query.append(DaoUtils.andAttributeIn("gs", GrowingSystem.PROPERTY_TOPIA_ID, args, growingSystemIds));
        }

        // domains
        query.append(DaoUtils.andAttributeInIfNotEmpty("gs", PROPERTY_GROWING_PLAN_DOMAIN_ID, args, navigationContext.getDomains()));

        // growingPlans
        query.append(DaoUtils.andAttributeInIfNotEmpty("gs", PROPERTY_GROWING_PLAN_ID, args, navigationContext.getGrowingPlans()));

        // growingSystems
        query.append(DaoUtils.andAttributeInIfNotEmpty("gs", GrowingSystem.PROPERTY_TOPIA_ID, args, navigationContext.getGrowingSystems()));
    }

    protected Set<String> networksProjection(Set<String> networkIds, String projection) {
        return networksProjection(networkIds, projection, null, null);
    }

    protected Set<String> networksProjection(final Set<String> networkIds, final String projection,
                                             final String filterProperty, final Object filterValue) {

        Callable<LinkedHashSet<String>> loader = new Callable<LinkedHashSet<String>>() {
            @Override
            public LinkedHashSet<String> call() throws Exception {
                StringBuilder query = new StringBuilder(String.format("SELECT DISTINCT gs.%s FROM %s gs", projection, getEntityClass().getName()));
                query.append(" WHERE ( 1 = 0");
                Map<String, Object> args = Maps.newLinkedHashMap();

                NetworkTopiaDao networkDAO = topiaDaoSupplier.getDao(Network.class, NetworkTopiaDao.class);
                Set<Network> networks = networkDAO.loadNetworksWithDescendantHierarchy(networkIds);
                ProjectionHelper.GET_NETWORK_SUB_QUERY(query, args, networks);

                query.append(")");

                query.append(DaoUtils.andAttributeEquals("gs", filterProperty, args, filterValue));

                List<String> projectedList = findAll(query.toString(), args);
                LinkedHashSet<String> result = Sets.newLinkedHashSet(projectedList);

                return result;
            }
        };

        String key = String.format("%s_%s_%s_%s", networkIds, projection, filterProperty, filterValue);
        Set<String> result = getCacheService().get("networksProjection", key, loader);
        return result;
    }

    /**
     * Find all growingSystems using same growingSystem's duplication code.
     *
     * @return related growingSystem
     */
    public LinkedHashMap<Integer, String> findAllRelatedGrowingSystemsByCampaigns(String code) {
        String query = "SELECT " + PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN + ", " + GrowingSystem.PROPERTY_TOPIA_ID
                + " FROM " + getEntityClass().getName()
                + " WHERE " + GrowingSystem.PROPERTY_CODE + " = :code"
                + " ORDER BY " + PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN + " DESC";
        List<Object[]> growingSystems = findAll(query, DaoUtils.asArgsMap("code", code));
        LinkedHashMap<Integer, String> result = DaoUtils.toRelatedMap(growingSystems);
        return result;
    }

    /**
     * return all growing system codes, growing plan codes, domains code matching the same dephy id as the given one, if dephy id is null, null is return.
     * @param dephy dephy id
     * @return list of codes by campaigns,
     *         code[0]: Growing system code
     *         code[1]: Growing plan code
     *         code[2]: Domain code
     */
    public LinkedHashMap<Integer, List<String>> findAllDephyRelatedGSCodesGPCodesDomainCodesByCampaigns(String dephy) {
        String query = "SELECT " + PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN + ", "
                + GrowingSystem.PROPERTY_CODE + ", "
                + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_CODE + ", "
                + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE + " "
                + " FROM " + getEntityClass().getName()
                + " WHERE " + GrowingSystem.PROPERTY_DEPHY_NUMBER + " = :dephy"
                + " ORDER BY " + PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN;
        List<Object[]> growingSystems = findAll(query, DaoUtils.asArgsMap("dephy", dephy));
        LinkedHashMap<Integer, List<String>> result = getGS_GP_D_CodesByCampaigns(growingSystems);
        return result;
    }

    protected static LinkedHashMap<Integer, List<String>> getGS_GP_D_CodesByCampaigns(List<Object[]> input) {
        LinkedHashMap<Integer, List<String>> result = null;
        if (input != null) {
            result = Maps.newLinkedHashMap();
            for (Object[] entry : input) {
                Integer campaign = (Integer) entry[0];
                String growingSystemCode = (String) entry[1];
                String growingPlanCode = (String) entry[2];
                String domainCode = (String) entry[3];
                List<String> codes = Lists.newArrayList(growingSystemCode, growingPlanCode, domainCode);
                result.put(campaign, codes);
            }
        }
        return result;
    }

    public List<GrowingSystem> findAllByCodeAndCampaign(String code, Set<Integer> campaigns) {
        Preconditions.checkArgument(campaigns != null && !campaigns.isEmpty());
        List<GrowingSystem> result;
        String query = "FROM " + getEntityClass().getName() + " gs WHERE 1 = 1";
        Map<String, Object> args = Maps.newLinkedHashMap();
        query += DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_CODE, args, code);
        query += DaoUtils.andAttributeIn("gs", PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN, args, campaigns);
        result = findAll(query+" ORDER BY gs." + PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN + " ASC ", args);
        return result;
    }

    public List<GrowingSystem> findAllActiveWritableByDomain(Domain domain, SecurityContext securityContext) {

        StringBuilder query = new StringBuilder(" FROM " + getEntityClass().getName() + " gs WHERE 1 = 1 ");
        Map<String, Object> args = Maps.newLinkedHashMap();

        query.append(DaoUtils.andAttributeEquals("gs", PROPERTY_GROWING_PLAN_DOMAIN, args, domain));
        query.append(DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_ACTIVE, args, true));

        SecurityHelper.addWritableGrowingSystemFilter(query, args, securityContext, "gs");

        query.append(" ORDER BY LOWER ( gs." + GrowingSystem.PROPERTY_NAME + " ) ");

        List<GrowingSystem> result = findAll(query.toString(), args);

        return result;
    }

    public List<GrowingSystem> findAllAvailableForManagementModeDuplication(List<GrowingSystem> growingSystems, GrowingSystem managementModeGrowingSystem) {

        String query = "FROM " + getEntityClass().getName() + " gs WHERE 1 = 1";
        Map<String, Object> args = Maps.newLinkedHashMap();

        // Ne sont disponibles que les systèmes de culture qui n'ont pas de mode de gestion portant dessus.
        query += DaoUtils.andAttributeEquals("gs", PROPERTY_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_CODE, args, managementModeGrowingSystem.getGrowingPlan().getDomain().getCode());
        query += DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_ACTIVE, args, true);
        query += (" AND gs NOT IN (" +
                "   SELECT " +
                "   mm." + ManagementMode.PROPERTY_GROWING_SYSTEM +
                "   FROM " + ManagementMode.class.getName() + " mm" +
                " ) AND gs IN (:growingSystems)");

        args.put("growingSystems", growingSystems);
        List<GrowingSystem> result = findAll(query, args);

        return result;
    }

    public List<GrowingSystem> findAllAvailableFoPracticedSystemDuplication(GrowingSystemFilter growingSystemFilter, GrowingSystem growingSystem) {
        Map<String, Object> args = Maps.newLinkedHashMap();
        StringBuilder query = new StringBuilder("FROM " + getEntityClass().getName() + " gs");
        query.append(" WHERE 1 = 1");

        addFilerQuery(growingSystemFilter, query, args);
        query.append(DaoUtils.andAttributeEquals("gs", PROPERTY_GROWING_PLAN_DOMAIN + "." + Domain.PROPERTY_CODE, args, growingSystem.getGrowingPlan().getDomain().getCode()));

        List<GrowingSystem> result = findAll(query.toString(), args);

        return result;
    }


    public Set<String> getAllGrowingSystemCodes() {
        String query = "SELECT DISTINCT " + GrowingSystem.PROPERTY_CODE +
                " FROM " + getEntityClass().getName();
        List<String> list = findAll(query, DaoUtils.asArgsMap());
        Set<String> result = Sets.newHashSet(list);
        return result;
    }

    public void validateGrowingSystem(String growingSystemId, Date now) {

        Map<String, Object> args = DaoUtils.asArgsMap("growingSystemId", growingSystemId, "now", now);
        topiaJpaSupport.execute("UPDATE " + getEntityClass().getName() + " gs" +
                " SET gs.validated=true, gs.validationDate=:now, gs.updateDate=:now" +
                " WHERE gs." + GrowingSystem.PROPERTY_TOPIA_ID + "=:growingSystemId", args);

        // TODO AThimel 10/12/13 Validate all underlying entities

    }

    public Iterable<String> getAllNetworksUsedByGrowingSystems() {
        String hql = " SELECT DISTINCT gs. " + GrowingSystem.PROPERTY_NETWORKS +
                " FROM " + getEntityClass().getName() + " gs ";
        List<Network> networks = findAll(hql, DaoUtils.asArgsMap());

        Iterable<String> result = Iterables.transform(networks, Entities.GET_TOPIA_ID);
        return result;
    }

    public List<GrowingSystem> getGrowingSystemsForManagementMode(NavigationContext navigationContext, SecurityContext securityContext) {
        Map<String, Object> args = Maps.newLinkedHashMap();

        StringBuilder query = new StringBuilder(" FROM " + GrowingSystem.class.getName() + " gs ");
        query.append(" WHERE gs NOT IN (" +
                "   SELECT " +
                "   mm1." + ManagementMode.PROPERTY_GROWING_SYSTEM +
                "   FROM " + ManagementMode.class.getName() + " mm1," + ManagementMode.class.getName() + " mm2 " +
                "   WHERE mm1." + ManagementMode.PROPERTY_CATEGORY + " = :prevu " +
                "   AND mm2." + ManagementMode.PROPERTY_CATEGORY + " = :constate " +
                "   AND mm1." + ManagementMode.PROPERTY_GROWING_SYSTEM + " = mm2." + ManagementMode.PROPERTY_GROWING_SYSTEM +
                " ) AND gs." + GrowingSystem.PROPERTY_ACTIVE + " = true ");

        args.put("prevu", ManagementModeCategory.PLANNED);
        args.put("constate", ManagementModeCategory.OBSERVED);

        addNavigationContextQuery(query, args, navigationContext);

        SecurityHelper.addWritableGrowingSystemFilter(query, args, securityContext, "gs");

        List<GrowingSystem> result = findAll(query + " ORDER BY gs." + GrowingSystem.PROPERTY_NAME, args);
        return result;
    }



    protected Map<Object, Long> networksProjection(final Set<String> networkIds, final String projection,
                                                   final String filterProperty, final Object filterValue,
                                                   final String groupByProperty) {

        Callable<LinkedHashMap<Object, Long>> loader = new Callable<LinkedHashMap<Object, Long>>() {
            @Override
            public LinkedHashMap<Object, Long> call() throws Exception {
                StringBuilder query = new StringBuilder(String.format("SELECT gs.%s, COUNT (gs.%s) FROM %s gs", groupByProperty, projection, getEntityClass().getName()));
                query.append(" WHERE ( 1 = 0");
                Map<String, Object> args = Maps.newLinkedHashMap();

                int index = 0;
                NetworkTopiaDao networkDAO = topiaDaoSupplier.getDao(Network.class, NetworkTopiaDao.class);
                Set<Network> networks = networkDAO.loadNetworksWithDescendantHierarchy(networkIds);
                ProjectionHelper.GET_NETWORK_SUB_QUERY(query, args, networks);

                query.append(")");

                query.append(DaoUtils.andAttributeEquals("gs", filterProperty, args, filterValue));

                query.append(String.format(" GROUP BY gs.%s ", groupByProperty));

                List<Object[]> projectedList = findAll(query.toString(), args);
                LinkedHashMap<Object, Long> result = Maps.newLinkedHashMap();
                for (Object[] objects : projectedList) {
                    result.put(objects[0], (Long)objects[1]);
                }
                return result;
            }
        };

        String key = String.format("%s_%s_%s_%s_%s", networkIds, projection, filterProperty, filterValue, groupByProperty);
        Map<Object, Long> result = getCacheService().get("networksProjection", key, loader);
        return result;
    }

    protected GrowingSystemsIndicator getIndicator(List<GrowingSystemsIndicator> indicators, Object o) {
        TypeDEPHY expectedDephy = null;
        if (o instanceof TypeDEPHY) {
            expectedDephy = (TypeDEPHY)o;
        }
        Sector expectedSector = null;
        if (o instanceof Sector) {
            expectedSector = (Sector)o;
        }
        for (GrowingSystemsIndicator indicator : indicators) {
            if (Objects.equal(indicator.getSector(), expectedSector)
                    && Objects.equal(indicator.getTypeDephy(), expectedDephy)) {
                return indicator;
            }
        }
        GrowingSystemsIndicator result;
        if (expectedDephy != null) {
            result = new GrowingSystemsIndicator(expectedDephy);
        } else if (expectedSector != null) {
            result = new GrowingSystemsIndicator(expectedSector);
        } else {
            result = new GrowingSystemsIndicator();
        }
        indicators.add(result);
        return result;
    }

    public List<GrowingSystemsIndicator> networksToGrowingSystemIndicators(Set<String> networkIds) {

        List<GrowingSystemsIndicator> result = Lists.newArrayList();

        GrowingSystemsIndicator gsIndicator = new GrowingSystemsIndicator();
        gsIndicator.setCount(getProjectionHelper().networksToGrowingSystems(networkIds).size());
        gsIndicator.setActive(networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID, GrowingSystem.PROPERTY_ACTIVE, true).size());
        gsIndicator.setValidated(networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID, GrowingSystem.PROPERTY_VALIDATED, true).size());
        result.add(gsIndicator);

        {
            Map<Object, Long> map = networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID, null, null, GrowingSystem.PROPERTY_SECTOR);
            for (Map.Entry<Object, Long> entry : map.entrySet()) {
                Object key = entry.getKey();
                GrowingSystemsIndicator indicator = getIndicator(result, key);
                indicator.setCount(entry.getValue());
            }
        }
        {
            Map<Object, Long> map = networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID, GrowingSystem.PROPERTY_ACTIVE, true, GrowingSystem.PROPERTY_SECTOR);
            for (Map.Entry<Object, Long> entry : map.entrySet()) {
                Object key = entry.getKey();
                GrowingSystemsIndicator indicator = getIndicator(result, key);
                indicator.setActive(entry.getValue());
            }
        }
        {
            Map<Object, Long> map = networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID, GrowingSystem.PROPERTY_VALIDATED, true, GrowingSystem.PROPERTY_SECTOR);
            for (Map.Entry<Object, Long> entry : map.entrySet()) {
                Object key = entry.getKey();
                GrowingSystemsIndicator indicator = getIndicator(result, key);
                indicator.setValidated(entry.getValue());
            }
        }


        {
            Map<Object, Long> map = networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID, null, null, PROPERTY_GROWING_PLAN_TYPE);
            for (Map.Entry<Object, Long> entry : map.entrySet()) {
                Object key = entry.getKey();
                GrowingSystemsIndicator indicator = getIndicator(result, key);
                indicator.setCount(entry.getValue());
            }
        }
        {
            Map<Object, Long> map = networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID, GrowingSystem.PROPERTY_ACTIVE, true, PROPERTY_GROWING_PLAN_TYPE);
            for (Map.Entry<Object, Long> entry : map.entrySet()) {
                Object key = entry.getKey();
                GrowingSystemsIndicator indicator = getIndicator(result, key);
                indicator.setActive(entry.getValue());
            }
        }
        {
            Map<Object, Long> map = networksProjection(networkIds, GrowingSystem.PROPERTY_TOPIA_ID, GrowingSystem.PROPERTY_VALIDATED, true, PROPERTY_GROWING_PLAN_TYPE);
            for (Map.Entry<Object, Long> entry : map.entrySet()) {
                Object key = entry.getKey();
                GrowingSystemsIndicator indicator = getIndicator(result, key);
                indicator.setValidated(entry.getValue());
            }
        }

        return result;
    }

    public List<Integer> findCampaigns(String growingSystemCode) {

        Map<String, Object> args = DaoUtils.asArgsMap();
        String hql = " SELECT DISTINCT gs." + PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN +
                " FROM " + getEntityClass().getName() + " gs "+
                " WHERE 1=1 ";
        hql += DaoUtils.andAttributeEquals("gs", GrowingSystem.PROPERTY_CODE, args, growingSystemCode);

        hql += " ORDER BY gs." + PROPERTY_GROWING_PLAN_DOMAIN_CAMPAIGN + " ASC ";
        List<Integer> result = findAll(hql, args);
        return result;
    }
}
