package fr.inra.agrosyst.api.entities;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: GrowingPlanTopiaDao.java 4069 2014-05-21 13:39:21Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/GrowingPlanTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.util.PagerBean;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;

import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.growingplan.GrowingPlanFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.security.SecurityContext;
import fr.inra.agrosyst.services.security.SecurityHelper;

/**
 * Custom dao methods on growingPlan.
 *
 * @author Eric Chatellier
 */
public class GrowingPlanTopiaDao extends AbstractGrowingPlanTopiaDao<GrowingPlan> {

    protected static final String PROPERTY_DOMAIN_NAME = GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_NAME;
    protected static final String PROPERTY_DOMAIN_CAMPAIGN = GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN;
    protected static final String PROPERTY_DOMAIN_ID = GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID;

    /**
     * Search growingPlan matching user navigation context and custom additional {@code filter}.
     *
     * @param filter          custom filter
     * @param securityContext
     * @return matching domains
     * @throws TopiaException
     */
    public ResultList<GrowingPlan> getFilteredGrowingPlans(GrowingPlanFilter filter, SecurityContext securityContext) throws TopiaException {
        StringBuilder query = new StringBuilder("FROM " + GrowingPlan.class.getName() + " gp ");
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // apply non null filter
        if (filter != null) {

            // growing plan name
            query.append(DaoUtils.andAttributeLike("gp", GrowingPlan.PROPERTY_NAME, args, filter.getGrowingPlanName()));

            // active
            query.append(DaoUtils.andAttributeEquals("gp", GrowingPlan.PROPERTY_ACTIVE, args, filter.getActive()));

            // type
            query.append(DaoUtils.andAttributeEquals("gp", GrowingPlan.PROPERTY_TYPE, args, filter.getTypeDephy()));

            // domain name
            query.append(DaoUtils.andAttributeLike("gp", PROPERTY_DOMAIN_NAME, args, filter.getDomainName()));

            // campaigns
            query.append(DaoUtils.andAttributeEquals("gp", PROPERTY_DOMAIN_CAMPAIGN, args, filter.getCampaign()));

            // Navigation context
            NavigationContext navigationContext = filter.getNavigationContext();
            if (navigationContext != null) {

                // campaigns
                Set<Integer> ncCampaigns = navigationContext.getCampaigns();
                query.append(DaoUtils.andAttributeInIfNotEmpty("gp", PROPERTY_DOMAIN_CAMPAIGN, args, ncCampaigns));

                // networks
                if (navigationContext.getNetworksCount() > 0) {
                    Set<String> growingPlanIds = networksToGrowingPlans(navigationContext.getNetworks());
                    query.append(DaoUtils.andAttributeIn("gp", GrowingPlan.PROPERTY_TOPIA_ID, args, growingPlanIds));
                }

                // domains
                Set<String> ncDomains = navigationContext.getDomains();
                query.append(DaoUtils.andAttributeInIfNotEmpty("gp", PROPERTY_DOMAIN_ID, args, ncDomains));

                // growingPlans
                Set<String> growingPlans = Sets.newHashSet(navigationContext.getGrowingPlans());
                if (!navigationContext.getGrowingSystems().isEmpty()) {
                    String hql = "SELECT" +
                            " gs." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_TOPIA_ID +
                            " FROM " + GrowingSystem.class.getName() + " gs" +
                            " WHERE gs." + GrowingSystem.PROPERTY_TOPIA_ID + " IN :gsIds";
                    List<String> growingPlansIds = findAll(hql, DaoUtils.asArgsMap("gsIds", navigationContext.getGrowingSystems()));
                    growingPlans.addAll(growingPlansIds);
                }
                query.append(DaoUtils.andAttributeInIfNotEmpty("gp", GrowingPlan.PROPERTY_TOPIA_ID, args, growingPlans));

            }
        }

        SecurityHelper.addGrowingPlanFilter(query, args, securityContext, "gp");

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;

        String queryString = query.toString();
        String queryAndOrder = queryString + " ORDER BY " +
                "lower (gp." + GrowingPlan.PROPERTY_NAME + ")" +
                ", gp." + PROPERTY_DOMAIN_NAME + "" +
                ", gp." + PROPERTY_DOMAIN_CAMPAIGN;
        List<GrowingPlan> growingPlans = find(queryAndOrder, args, startIndex, endIndex);
        long totalCount = findUnique("SELECT count(*) " + queryString, args);

        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);
        ResultList<GrowingPlan> result = ResultList.of(growingPlans, pager);
        return result;
    }

    protected Set<String> networksToGrowingPlans(Set<String> networksIds) {
        Set<String> result = getProjectionHelper().networksToGrowingPlans(networksIds);
        return result;
    }

    public LinkedHashSet<String> domainsToGrowingPlansCode(Set<String> domainCodes) {
        Preconditions.checkArgument(domainCodes != null && !domainCodes.isEmpty());

        String domainCodeProperty = GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE;

        String query = String.format("SELECT DISTINCT gp.%s FROM %s gp", GrowingPlan.PROPERTY_CODE, getEntityClass().getName());
        query += " WHERE 1 = 1";
        Map<String, Object> args = Maps.newLinkedHashMap();

        // domains
        if (domainCodes.size() == 1) {
            query += DaoUtils.andAttributeEquals("gp", domainCodeProperty, args, domainCodes.iterator().next());
        } else {
            query += DaoUtils.andAttributeIn("gp", domainCodeProperty, args, domainCodes);
        }

        List<String> growingPlanCodes = findAll(query, args);
        LinkedHashSet<String> result = Sets.newLinkedHashSet(growingPlanCodes);

        return result;
    }

    /**
     * Find all growingPlans using same growingPlan's duplication code.
     *
     * @return related growingPlan
     */
    public LinkedHashMap<Integer, String> findAllRelatedGrowingPlans(String code) {

        String query = "SELECT " + PROPERTY_DOMAIN_CAMPAIGN + ", " + GrowingPlan.PROPERTY_TOPIA_ID
                + " FROM " + getEntityClass().getName()
                + " WHERE " + GrowingPlan.PROPERTY_CODE + " = :code"
                + " ORDER BY " + PROPERTY_DOMAIN_CAMPAIGN + " DESC";

        List<Object[]> growingPlans = findAll(query, DaoUtils.asArgsMap("code", code));
        LinkedHashMap<Integer, String> result = DaoUtils.toRelatedMap(growingPlans);
        return result;
    }

    public Set<String> getAllGrowingPlanCodes() {
        String query = "SELECT DISTINCT " + GrowingPlan.PROPERTY_CODE +
                " FROM " + getEntityClass().getName();
        List<String> list = findAll(query, DaoUtils.asArgsMap());
        Set<String> result = Sets.newHashSet(list);
        return result;
    }

    public void validateGrowingPlan(String growingPlanId, Date now) {

        Map<String, Object> args = DaoUtils.asArgsMap("growingPlanId", growingPlanId, "now", now);
        topiaJpaSupport.execute("UPDATE " + getEntityClass().getName() + " gp" +
                " SET gp.validated=true, gp.validationDate=:now, gp.updateDate=:now" +
                " WHERE gp." + GrowingPlan.PROPERTY_TOPIA_ID + "=:growingPlanId", args);

        // TODO AThimel 10/12/13 Validate all underlying entities

    }

} //GrowingPlanTopiaDao<E extends GrowingPlan>
