package fr.inra.agrosyst.api.entities;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: DomainTopiaDao.java 4977 2015-06-11 04:13:34Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/DomainTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.collect.Collections2;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.NavigationContext;
import fr.inra.agrosyst.api.entities.effective.EffectiveIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleConnection;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleNode;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.entities.referential.RefLocation;
import fr.inra.agrosyst.api.services.ResultList;
import fr.inra.agrosyst.api.services.domain.DomainFilter;
import fr.inra.agrosyst.api.utils.DaoUtils;
import fr.inra.agrosyst.services.security.SecurityContext;
import fr.inra.agrosyst.services.security.SecurityHelper;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.persistence.TopiaException;
import org.nuiton.util.PagerBean;

import java.util.Collection;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Custom dao methods on domains.
 *
 * @author Eric Chatellier
 */
public class DomainTopiaDao extends AbstractDomainTopiaDao<Domain> {

    protected static final String PROPERTY_LOCATION_DEPARTEMENT = Domain.PROPERTY_LOCATION + "." + RefLocation.PROPERTY_DEPARTEMENT;
    protected static final String PRACTICED_NODE_SEASONAL_CYCLE_PRACTICED_SYSTEM_CAMPAIGNS = PracticedCropCycleNode.PROPERTY_PRACTICED_SEASONAL_CROP_CYCLE + "." + PracticedSeasonalCropCycle.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_CAMPAIGNS;

    private static final Log log = LogFactory.getLog(DomainTopiaDao.class);

    /**
     * Search domain matching user context filter and custom additional {@code filter}.
     *
     * @param filter custom filter
     * @return matching domains
     * @throws TopiaException
     */
    public ResultList<Domain> getFilteredDomains(DomainFilter filter, SecurityContext securityContext) throws TopiaException {
        StringBuilder query = new StringBuilder("FROM " + getEntityClass().getName() + " D");
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // apply non null filter
        if (filter != null) {

            // name
            query.append(DaoUtils.andAttributeLike("D", Domain.PROPERTY_NAME, args, filter.getDomainName()));

            // active
            query.append(DaoUtils.andAttributeEquals("D", Domain.PROPERTY_ACTIVE, args, filter.getActive()));

            // type
            query.append(DaoUtils.andAttributeEquals("D", Domain.PROPERTY_TYPE, args, filter.getType()));

            // main contact
            query.append(DaoUtils.andAttributeLike("D", Domain.PROPERTY_MAIN_CONTACT, args, filter.getMainContact()));

            // departement
            query.append(DaoUtils.andAttributeLike("D", PROPERTY_LOCATION_DEPARTEMENT, args, filter.getDepartement()));

            // campaign
            query.append(DaoUtils.andAttributeEquals("D", Domain.PROPERTY_CAMPAIGN, args, filter.getCampaign()));

            // Navigation context
            NavigationContext navigationContext = filter.getNavigationContext();
            if (navigationContext != null) {

                // campaigns
                Set<Integer> ncCampaigns = navigationContext.getCampaigns();
                query.append(DaoUtils.andAttributeInIfNotEmpty("D", Domain.PROPERTY_CAMPAIGN, args, ncCampaigns));

                // networks
                if (navigationContext.getNetworksCount() > 0) {
                    Set<String> domainIds = networksToDomains(navigationContext.getNetworks());
                    query.append(DaoUtils.andAttributeIn("D", Domain.PROPERTY_TOPIA_ID, args, domainIds));
                }

                // domains
                Set<String> domains = Sets.newHashSet(navigationContext.getDomains());
                if (!navigationContext.getGrowingSystems().isEmpty()) {
                    String hql = "SELECT" +
                            " gs." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID +
                            " FROM " + GrowingSystem.class.getName() + " gs" +
                            " WHERE gs." + GrowingSystem.PROPERTY_TOPIA_ID + " IN :gsIds";
                    List<String> domainsIds = findAll(hql, DaoUtils.asArgsMap("gsIds", navigationContext.getGrowingSystems()));
                    domains.addAll(domainsIds);
                }
                if (!navigationContext.getGrowingPlans().isEmpty()) {
                    String hql = "SELECT" +
                            " gp." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_TOPIA_ID +
                            " FROM " + GrowingPlan.class.getName() + " gp" +
                            " WHERE gp." + GrowingPlan.PROPERTY_TOPIA_ID + " IN :gpIds";
                    List<String> domainsIds = findAll(hql, DaoUtils.asArgsMap("gpIds", navigationContext.getGrowingPlans()));
                    domains.addAll(domainsIds);
                }

                query.append(DaoUtils.andAttributeInIfNotEmpty("D", Domain.PROPERTY_TOPIA_ID, args, domains));
            }
        }

        SecurityHelper.addDomainFilter(query, args, securityContext, "D");

        int page = filter != null ? filter.getPage() : 0;
        int count = filter != null ? filter.getPageSize() : 10;
        int startIndex = page * count;
        int endIndex = page * count + count - 1;

        String queryString = query.toString();
        String queryAndOrder = queryString + " ORDER BY " +
                "lower (D." + Domain.PROPERTY_NAME + ")" +
                ", D." + Domain.PROPERTY_CAMPAIGN;
        List<Domain> domains = find(queryAndOrder, args, startIndex, endIndex);
        long totalCount = findUnique("SELECT count(*) " + queryString, args);

        // build result bean
        PagerBean pager = DaoUtils.getPager(page, count, totalCount);
        ResultList<Domain> result = ResultList.of(domains, pager);
        return result;
    }

    protected Set<String> networksToDomains(Set<String> networksIds) {
        Set<String> result = getProjectionHelper().networksToDomains(networksIds);
        return result;
    }

    /**
     * Find all domains using same domain's duplication code.
     *
     * @return related domain (ordered by campaign desc)
     */
    public LinkedHashMap<Integer, String> findAllRelatedDomains(String code) {
        String query = "SELECT " + Domain.PROPERTY_CAMPAIGN + ", " + Domain.PROPERTY_TOPIA_ID
                + " FROM " + getEntityClass().getName()
                + " WHERE " + Domain.PROPERTY_CODE + " = :code"
                + " ORDER BY " + Domain.PROPERTY_CAMPAIGN + " DESC";
        List<Object[]> domains = findAll(query, DaoUtils.asArgsMap("code", code));
        LinkedHashMap<Integer, String> result = DaoUtils.toRelatedMap(domains);
        return result;
    }

    /**
     * Retourne le nombre de domaine dont le nom est egal à celui demandé (sans tenir compte
     * de la case.
     *
     * @param domainName domain name
     * @return domain count
     */
    public long countDomainWithName(String domainName) {
        String query = "SELECT count(*) FROM " + getEntityClass().getName()
                + " WHERE lower(" + Domain.PROPERTY_NAME + ") = lower(:" + Domain.PROPERTY_NAME + ")";
        Number result = findUnique(query, DaoUtils.asArgsMap(Domain.PROPERTY_NAME, domainName));
        return result.longValue();
    }

    public List<Integer> getAllCampaigns() {
        String query = "SELECT DISTINCT " + Domain.PROPERTY_CAMPAIGN +
                " FROM " + getEntityClass().getName() +
                " ORDER BY " + Domain.PROPERTY_CAMPAIGN + " ASC";
        List<Integer> result = findAll(query, DaoUtils.asArgsMap());
        return result;
    }

    public Set<String> getAllDomainCodes() {
        String query = "SELECT DISTINCT " + Domain.PROPERTY_CODE +
                " FROM " + getEntityClass().getName();
        List<String> list = findAll(query, DaoUtils.asArgsMap());
        Set<String> result = Sets.newHashSet(list);
        return result;
    }

    public void validateDomain(String domainId, Date now) {

        Map<String, Object> args = DaoUtils.asArgsMap("domainId", domainId, "now", now);
        topiaJpaSupport.execute("UPDATE " + getEntityClass().getName() + " d" +
                " SET d.validated=true, d.validationDate=:now, d.updateDate=:now" +
                " WHERE d." + Domain.PROPERTY_TOPIA_ID + "=:domainId", args);

        Domain domain = forTopiaIdEquals(domainId).findUnique();

        {
            args = DaoUtils.asArgsMap("domain", domain);
            topiaJpaSupport.execute("UPDATE " + CroppingPlanEntry.class.getName() + " cpe" +
                    " SET cpe." + CroppingPlanEntry.PROPERTY_VALIDATED + "=true" +
                    " WHERE cpe." + CroppingPlanEntry.PROPERTY_TOPIA_ID + " IN (" +
                    "   SELECT sub." + CroppingPlanEntry.PROPERTY_TOPIA_ID + " FROM " + CroppingPlanEntry.class.getName() + " sub" +
                    "   WHERE sub." + CroppingPlanEntry.PROPERTY_DOMAIN + "=:domain" +
                    " )", args);
        }

        {
            args = DaoUtils.asArgsMap("domain", domain);
            topiaJpaSupport.execute("UPDATE " + CroppingPlanSpecies.class.getName() + " cps" +
                    " SET cps." + CroppingPlanSpecies.PROPERTY_VALIDATED + "=true" +
                    " WHERE cps." + CroppingPlanSpecies.PROPERTY_TOPIA_ID + " IN (" +
                    "   SELECT sub." + CroppingPlanSpecies.PROPERTY_TOPIA_ID + " FROM " + CroppingPlanSpecies.class.getName() + " sub" +
                    "   WHERE sub." + CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY + "." + CroppingPlanEntry.PROPERTY_DOMAIN + "=:domain" +
                    " )", args);
        }

        {
            args = DaoUtils.asArgsMap("domain", domain);
            topiaJpaSupport.execute("UPDATE " + Ground.class.getName() + " s" +
                    " SET s." + Ground.PROPERTY_VALIDATED + "=true" +
                    " WHERE s." + Ground.PROPERTY_TOPIA_ID + " IN ( " +
                    "   SELECT sub." + Ground.PROPERTY_TOPIA_ID + " FROM " + Ground.class.getName() + " sub" +
                    "   WHERE sub." + Ground.PROPERTY_DOMAIN + "=:domain" +
                    " )", args);
        }

        {
            args = DaoUtils.asArgsMap("domain", domain);
            topiaJpaSupport.execute("UPDATE " + GeoPoint.class.getName() + " gd" +
                    " SET gd." + GeoPoint.PROPERTY_VALIDATED + "=true" +
                    " WHERE gd." + GeoPoint.PROPERTY_TOPIA_ID + " IN ( " +
                    "   SELECT sub." + GeoPoint.PROPERTY_TOPIA_ID + " FROM " + GeoPoint.class.getName() + " sub" +
                    "   WHERE sub." + GeoPoint.PROPERTY_DOMAIN + "=:domain" +
                    " )", args);
        }

        {
            Collection<WeatherStation> weatherStations = domain.getWeatherStations();
            if (CollectionUtils.isNotEmpty(weatherStations)) {
                args = DaoUtils.asArgsMap("weatherStationIds", Collections2.transform(weatherStations, Entities.GET_TOPIA_ID));
                topiaJpaSupport.execute("UPDATE " + WeatherStation.class.getName() + " ws" +
                        " SET ws." + WeatherStation.PROPERTY_VALIDATED + "=true" +
                        " WHERE ws." + WeatherStation.PROPERTY_TOPIA_ID + " IN ( :weatherStationIds )", args);
            }
        }

        {
            args = DaoUtils.asArgsMap("domain", domain);
            topiaJpaSupport.execute("UPDATE " + Equipment.class.getName() + " m" +
                    " SET m." + Equipment.PROPERTY_VALIDATED + "=true" +
                    " WHERE m." + Equipment.PROPERTY_TOPIA_ID + " IN ( " +
                    "   SELECT sub." + Equipment.PROPERTY_TOPIA_ID + " FROM " + Equipment.class.getName() + " sub" +
                    "   WHERE sub." + Equipment.PROPERTY_DOMAIN + "=:domain" +
                    " )", args);
        }

        {
            args = DaoUtils.asArgsMap("domain", domain);
            topiaJpaSupport.execute("UPDATE " + ToolsCoupling.class.getName() + " tc" +
                    " SET tc." + ToolsCoupling.PROPERTY_VALIDATED + "=true" +
                    " WHERE tc." + ToolsCoupling.PROPERTY_TOPIA_ID + " IN ( " +
                    "   SELECT sub." + ToolsCoupling.PROPERTY_TOPIA_ID + " FROM " + ToolsCoupling.class.getName() + " sub" +
                    "   WHERE sub." + ToolsCoupling.PROPERTY_DOMAIN + "=:domain" +
                    " )", args);
        }

    }

    public String getDomainCodeForGrowingSystem(String gsTopiaId) {
        String query = "SELECT gs." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." +Domain.PROPERTY_CODE +
                " FROM " + GrowingSystem.class.getName() + " gs " +
                " WHERE gs." + GrowingSystem.PROPERTY_TOPIA_ID + " = :gsTopiaId" ;

        String domainCode = findUnique(query, DaoUtils.asArgsMap("gsTopiaId", gsTopiaId));
        return domainCode;
    }

    /*
     * WARNING AThimel 18/02/2014 : Do not use this method to get writable domains, this is only usable on DecisionRule creation !
     * See https://forge.codelutin.com/issues/4439#note-6
     */
    public List<Domain> getActiveWritableDomainsForDecisionRuleCreation(SecurityContext securityContext) {
        StringBuilder query = new StringBuilder("FROM " + getEntityClass().getName() + " D");
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // active
        query.append(DaoUtils.andAttributeEquals("D", Domain.PROPERTY_ACTIVE, args, true));

        SecurityHelper.addWritableDomainFilterForDecisionRuleCreation(query, args, securityContext, "D");

        String queryString = query.toString();
        String queryAndOrder = queryString + " ORDER BY " +
                "lower (D." + Domain.PROPERTY_NAME + ")";
        List<Domain> domains = findAll(queryAndOrder, args);

        // Filter to retain only one domain per code
        // TODO AThimel 18/02/14 Find a better way
        final Set<String> presentCodes = Sets.newHashSet();
        Iterables.removeIf(domains, new Predicate<Domain>() {
            @Override
            public boolean apply(Domain input) {
                // If the code is already present in Set, 'added' will be false
                boolean added = presentCodes.add(input.getCode());
                return !added;
            }
        });

        return domains;
    }

    /**
     * return all domain topiaIds related the the domains codes given has parameters
     * and for campaigns matching the ones given has parameters
     * @param code domain Code
     * @param campaigns campaigns to look for domains
     * @return domain topiaIds
     */
    public List<String> findDomainIdsForCampaigns(String code, Set<Integer> campaigns, boolean includeCropsFromInactiveDomains){
        Preconditions.checkArgument(campaigns != null && !campaigns.isEmpty());
        StringBuilder query = new StringBuilder(
                "SELECT DISTINCT " + Domain.PROPERTY_TOPIA_ID + " FROM " + getEntityClass().getName() + " D");
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        // active
        if (!includeCropsFromInactiveDomains) {
            query.append(DaoUtils.andAttributeEquals("D", Domain.PROPERTY_ACTIVE, args, true));
        }

        //code
        query.append(DaoUtils.andAttributeEquals("D", Domain.PROPERTY_CODE, args, code));

        // campaigns
        query.append(DaoUtils.andAttributeIn("D", Domain.PROPERTY_CAMPAIGN, args, campaigns));

        List<String> domainIds = findAll(query.toString(), args);

        return domainIds;
    }

    public List<String> findCroppingPlanCodeForDomainsAndCampaigns(String code, Set<Integer> campaigns){
        Preconditions.checkArgument(campaigns != null && !campaigns.isEmpty());
        StringBuilder query = new StringBuilder(
                "SELECT DISTINCT cp." + CroppingPlanEntry.PROPERTY_CODE + " FROM " + CroppingPlanEntry.class.getName() + " cp ");
        query.append(" WHERE 1 = 1");

        Map<String, Object> args = Maps.newLinkedHashMap();
        // domains
        query.append(DaoUtils.andAttributeEquals("cp", CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE, args, code));
        query.append(DaoUtils.andAttributeEquals("cp", CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_ACTIVE, args, true));

        // campaigns
        query.append(DaoUtils.andAttributeIn("cp", CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN, args, campaigns));

        List<String> result = findAll(query.toString(), args);

        return result;
    }

    public List<String> findCroppingPlanSpeciesCodeForCropCodeAndCampaigns(String code, Set<Integer> campaigns){
        Preconditions.checkArgument(campaigns != null && !campaigns.isEmpty());
        StringBuilder query = new StringBuilder(
                "SELECT DISTINCT cps." + CroppingPlanSpecies.PROPERTY_CODE + " FROM " + CroppingPlanSpecies.class.getName() + " cps ");
        query.append(" WHERE 1 = 1");

        Map<String, Object> args = Maps.newLinkedHashMap();
        // domains
        query.append(DaoUtils.andAttributeEquals("cps", CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY + "." + CroppingPlanEntry.PROPERTY_CODE , args, code));
        query.append(DaoUtils.andAttributeEquals("cps", CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY + "." + CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_ACTIVE, args, true));

        // campaigns
        query.append(DaoUtils.andAttributeIn("cps", CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY + "." + CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN, args, campaigns));

        List<String> result = findAll(query.toString(), args);

        return result;
    }

    public List<String> findToolsCouplingCodeForDomainsAndCampaigns(String code, Set<Integer> campaigns){
        Preconditions.checkArgument(campaigns != null && !campaigns.isEmpty());
        StringBuilder query = new StringBuilder(
                "SELECT DISTINCT tc." + ToolsCoupling.PROPERTY_CODE + " FROM " + ToolsCoupling.class.getName() + " tc ");
        query.append(" WHERE 1 = 1");

        Map<String, Object> args = Maps.newLinkedHashMap();
        // domains
        query.append(DaoUtils.andAttributeEquals("tc", ToolsCoupling.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE, args, code));
        query.append(DaoUtils.andAttributeEquals("tc", ToolsCoupling.PROPERTY_DOMAIN + "." + Domain.PROPERTY_ACTIVE, args, true));

        // campaigns
        query.append(DaoUtils.andAttributeIn("tc", ToolsCoupling.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN, args, campaigns));

        List<String> toolsCouplingCodes = findAll(query.toString(), args);

        return toolsCouplingCodes;
    }

    public List<ToolsCoupling> findToolsCouplingsForDomainCodeAndCampaigns(String domainCode, Set<Integer> campaigns){
        Preconditions.checkArgument(campaigns != null && !campaigns.isEmpty());
        StringBuilder query = new StringBuilder(
                "SELECT DISTINCT tc FROM " + ToolsCoupling.class.getName() + " tc ");
        query.append(" WHERE 1 = 1 ");

        Map<String, Object> args = Maps.newLinkedHashMap();
        // domains
        query.append(DaoUtils.andAttributeEquals("tc", ToolsCoupling.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE, args, domainCode));
        query.append(DaoUtils.andAttributeEquals("tc", ToolsCoupling.PROPERTY_DOMAIN + "." + Domain.PROPERTY_ACTIVE, args, true));

        // campaigns
        query.append(DaoUtils.andAttributeIn("tc", ToolsCoupling.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CAMPAIGN, args, campaigns));

        List<ToolsCoupling> toolsCouplings = findAll(query.toString(), args);

        return toolsCouplings;
    }

    protected Map<String, Long> queryBody(Iterable<String> toolsCouplingIds, String campaign, String subQuery) {
        Map<String, Long> result = Maps.newHashMap();

        Map<String, Object> args = Maps.newLinkedHashMap();
        Set<String> tcIds = Sets.newHashSet(toolsCouplingIds);
        Preconditions.checkArgument(tcIds != null && !tcIds.isEmpty());

        if (campaign != null) {
            args.put("campaign" , campaign);
        }

        String query =
                " SELECT tc." + ToolsCoupling.PROPERTY_TOPIA_ID + " ,";

        query += "(" + subQuery + ")";

        query +=" FROM " + ToolsCoupling.class.getName() + " tc  WHERE 1 = 1 " ;

        query += DaoUtils.andAttributeIn("tc", ToolsCoupling.PROPERTY_TOPIA_ID, args, tcIds);

        List<Object[]> tcs = findAll(query + " GROUP BY tc." + ToolsCoupling.PROPERTY_TOPIA_ID , args);

        for (Object[] tcTuple : tcs) {
            String tcId = (String) tcTuple[0];
            Long count = (Long)tcTuple[1];
//            if (log.isDebugEnabled()) {
//                log.debug(
//                        "Pour toulsCoupling:" + tcId +
//                                " nbUsed:" + count);
//            }
            result.put(tcId, count);
        }

        return result;
    }

    public Map<String, Long> getTCUsedForEffectiveInterventions(Iterable<String> toolsCouplingIds) {
        String query =
            "SELECT COUNT(*) FROM " + EffectiveIntervention.class.getName() + " ei " +
            " WHERE tc IN ELEMENTS (ei." + EffectiveIntervention.PROPERTY_TOOL_COUPLINGS + ")";
        Map<String, Long> result = queryBody(toolsCouplingIds, null, query);
        return result;
    }

    public Map<String, Long> getTCUsedForPracticedInterventionNodesAndConnections(Iterable<String> toolsCouplingIds, String campaign) {
        String query =
            "SELECT COUNT(*) FROM " + PracticedCropCycleNode.class.getName() + " pccn " +
            " WHERE pccn." + PRACTICED_NODE_SEASONAL_CYCLE_PRACTICED_SYSTEM_CAMPAIGNS + " LIKE :campaign " +
            " AND pccn." + PracticedCropCycleNode.PROPERTY_PRACTICED_SEASONAL_CROP_CYCLE + "." + PracticedSeasonalCropCycle.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN +"."+ Domain.PROPERTY_CODE + "=" + "tc." + ToolsCoupling.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE +
            " AND pccn IN " +
            "   (" +
            "     SELECT pi." + PracticedIntervention.PROPERTY_PRACTICED_CROP_CYCLE_CONNECTION + "." + PracticedCropCycleConnection.PROPERTY_SOURCE +
            "     FROM " + PracticedIntervention.class.getName() + " pi " +
            "     WHERE tc." + ToolsCoupling.PROPERTY_CODE + " IN ELEMENTS (pi." + PracticedIntervention.PROPERTY_TOOLS_COUPLING_CODES + ")" +
            "    )";
        Map<String, Long> result = queryBody(toolsCouplingIds, campaign, query);
        return result;
    }

    public Map<String, Long> getTCUsedForPracticedInterventionPhases(Iterable<String> toolsCouplingIds, String campaign) {
        String query =
            "SELECT COUNT(ppcc) FROM " + PracticedPerennialCropCycle.class.getName() + " ppcc, " + PracticedIntervention.class.getName() + " pi " +
            " WHERE ppcc." + PracticedPerennialCropCycle.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_CAMPAIGNS + " LIKE :campaign " +
            " AND pi." + PracticedIntervention.PROPERTY_PRACTICED_CROP_CYCLE_PHASE + " IN ELEMENTS (ppcc." + PracticedPerennialCropCycle.PROPERTY_CROP_CYCLE_PHASES +")" +
            " AND tc." + ToolsCoupling.PROPERTY_CODE + " IN ELEMENTS (pi." + PracticedIntervention.PROPERTY_TOOLS_COUPLING_CODES + ")";
        Map<String, Long> result = queryBody(toolsCouplingIds, campaign, query);
        return result;
    }

} //DomainTopiaDao<E extends Domain>
