package fr.inra.agrosyst.api.entities;

/*
 * #%L
 * Agrosyst :: Services
 * $Id: CroppingPlanSpeciesTopiaDao.java 4878 2015-04-02 20:20:30Z dcosse $
 * $HeadURL: https://svn.codelutin.com/agrosyst/tags/agrosyst-1.5.3/agrosyst-services/src/main/java/fr/inra/agrosyst/api/entities/CroppingPlanSpeciesTopiaDao.java $
 * %%
 * Copyright (C) 2013 - 2014 INRA
 * %%
 * INRA - Tous droits réservés
 * #L%
 */

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import fr.inra.agrosyst.api.entities.effective.EffectiveCropCycleSpecies;
import fr.inra.agrosyst.api.entities.effective.EffectiveSpeciesStade;
import fr.inra.agrosyst.api.entities.measure.Measure;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleConnection;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleNode;
import fr.inra.agrosyst.api.entities.practiced.PracticedCropCycleSpecies;
import fr.inra.agrosyst.api.entities.practiced.PracticedIntervention;
import fr.inra.agrosyst.api.entities.practiced.PracticedPerennialCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSeasonalCropCycle;
import fr.inra.agrosyst.api.entities.practiced.PracticedSpeciesStade;
import fr.inra.agrosyst.api.entities.practiced.PracticedSystem;
import fr.inra.agrosyst.api.utils.DaoUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * @author David Cossé
 */
public class CroppingPlanSpeciesTopiaDao extends AbstractCroppingPlanSpeciesTopiaDao<CroppingPlanSpecies> {

    private static final Log log = LogFactory.getLog(CroppingPlanSpeciesTopiaDao.class);

    public static String FROM_NODE_TO_DOMAIN_QUERY = PracticedCropCycleNode.PROPERTY_PRACTICED_SEASONAL_CROP_CYCLE+"."+ PracticedSeasonalCropCycle.PROPERTY_PRACTICED_SYSTEM+"."+ PracticedSystem.PROPERTY_GROWING_SYSTEM+"."+ GrowingSystem.PROPERTY_GROWING_PLAN+"."+ GrowingPlan.PROPERTY_DOMAIN;


    public List<CroppingPlanSpecies> getCroppingPlanSpeciesForCodeAndDomainId (Set<String> speciesCodes, String domainId) {
        Map<String, Object> args = Maps.newLinkedHashMap();
        StringBuilder query = new StringBuilder("FROM " + CroppingPlanSpecies.class.getName() + " cps ");
        query.append("  WHERE 1 = 1 ");
        query.append(DaoUtils.andAttributeIn("cps", CroppingPlanSpecies.PROPERTY_CODE, args, speciesCodes));
        query.append(DaoUtils.andAttributeEquals("cps", CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY + "." + CroppingPlanEntry.PROPERTY_DOMAIN + '.' + Domain.PROPERTY_TOPIA_ID, args, domainId));
        List<CroppingPlanSpecies> result = findAll(query.toString(), args);

        return result;
    }

    protected Map<String, Long> queryBody(Iterable<String> croppingPlanSpeciesId, String campaign, String subQuery) {

        Map<String, Long> result = Maps.newHashMap();

        Map<String, Object> args = Maps.newLinkedHashMap();
        Set<String> cpeIds = Sets.newHashSet(croppingPlanSpeciesId);

        if (campaign != null) {
            args.put("campaign" , campaign);
        }

        StringBuilder query = new StringBuilder("SELECT cps."+ CroppingPlanSpecies.PROPERTY_TOPIA_ID +", ");

        query.append("(" + subQuery + ")");

        query.append(" FROM " + CroppingPlanSpecies.class.getName() + " cps  WHERE 1 = 1 ");
        query.append(DaoUtils.andAttributeIn("cps", CroppingPlanSpecies.PROPERTY_TOPIA_ID, args, cpeIds));

        List<Object[]> cpss = findAll(query.toString(), args);
        for (Object[] cpsTuple : cpss) {
            String cpsId = (String) cpsTuple[0];
            Long count = (Long)cpsTuple[1];
//            if (log.isDebugEnabled()) {
//                log.debug("Pour culture:" + cpsId +" nbUsed:" + count);
//            }
            result.put(cpsId, count);
        }

        return result;

    }

    public Map<String, Long> getCPSpeciesUsedForMeasures(
            Iterable<String> croppingPlanSpeciesIds) {

        // count nb species into Measure
        String query =
            "SELECT COUNT(*) FROM " + Measure.class.getName() + " m " +
            "  WHERE m." + Measure.PROPERTY_CROPPING_PLAN_SPECIES + "."+ CroppingPlanSpecies.PROPERTY_TOPIA_ID + " = cps." + CroppingPlanSpecies.PROPERTY_TOPIA_ID;

        Map<String, Long> result = queryBody(croppingPlanSpeciesIds, null, query);
        return result;
    }

    public Map<String, Long> getCPSpeciesUsedForEffectiveCropCycleSpecies(
            Iterable<String> croppingPlanSpeciesIds) {

        // count nb species use into EffectiveCropCycleSpecies
        String query =
            "SELECT COUNT(*) FROM " + EffectiveCropCycleSpecies.class.getName() + " eccs " +
            "   WHERE eccs."+ EffectiveCropCycleSpecies.PROPERTY_CROPPING_PLAN_SPECIES + "." + CroppingPlanSpecies.PROPERTY_TOPIA_ID + " = cps." + CroppingPlanSpecies.PROPERTY_TOPIA_ID;

        Map<String, Long> result = queryBody(croppingPlanSpeciesIds, null, query);
        return result;
    }

    public Map<String, Long> getCPSpeciesUsedForEffectiveSpeciesStades(
            Iterable<String> croppingPlanSpeciesIds) {

        // count nb species used into EffectiveSpeciesStade
        String query =
            "SELECT COUNT(*) FROM " + EffectiveSpeciesStade.class.getName() + " ess " +
            "   WHERE ess."+ EffectiveSpeciesStade.PROPERTY_CROPPING_PLAN_SPECIES+"."+ CroppingPlanSpecies.PROPERTY_TOPIA_ID + " = cps." + CroppingPlanSpecies.PROPERTY_TOPIA_ID;

        Map<String, Long> result = queryBody(croppingPlanSpeciesIds, null, query);
        return result;
    }

    public Map<String, Long> getCPSpeciesUsedForPracticedCropCycleSpecies(
            Iterable<String> croppingPlanSpeciesIds,
            String campaign) {
         // count nb species used for practicedPerennialCropCycle into PracticedCropCycleSpecies
        String query =
            "SELECT COUNT(*) FROM " + PracticedCropCycleSpecies.class.getName() + " pccs " +
            "   WHERE pccs." + PracticedCropCycleSpecies.PROPERTY_CROPPING_PLAN_SPECIES_CODE + " =cps."+ CroppingPlanSpecies.PROPERTY_CODE +
            "   AND pccs." + PracticedCropCycleSpecies.PROPERTY_CYCLE + "." + PracticedPerennialCropCycle.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_CAMPAIGNS + " LIKE :campaign ";

        Map<String, Long> result = queryBody(croppingPlanSpeciesIds, campaign, query);
        return result;
    }

    public List<PracticedCropCycleNode> getPracticedCropCycleNodeForCampaignAndDomainCode(String domainCode, String campaign) {

        String query = "FROM " + PracticedCropCycleNode.class.getName() + " pccn " +
            "  WHERE " +
            "  pccn." + DomainTopiaDao.PRACTICED_NODE_SEASONAL_CYCLE_PRACTICED_SYSTEM_CAMPAIGNS + " LIKE :campaign " +
            "  AND pccn." + FROM_NODE_TO_DOMAIN_QUERY + "." + Domain.PROPERTY_CODE + " = :domainCode";

        Map<String, Object> args = Maps.newLinkedHashMap();
        args.put("campaign", campaign);
        args.put("domainCode", domainCode);

        List<PracticedCropCycleNode> result = findAll(query, args);
        return result;
    }

    public Map<String, Long> getCPSpeciesUsedForPracticedPerennialCropCycleStades(
            Iterable<String> croppingPlanSpeciesIds,
            String campaign) {
         // count nb species used for practicedPerennialCropCycle into stades (found from Phases)
        String query =
            "SELECT COUNT(ppcc) FROM " + PracticedPerennialCropCycle.class.getName() + " ppcc, " + PracticedIntervention.class.getName() + " pi," + PracticedSpeciesStade.class.getName() + " pss " +
            "  WHERE pss." + PracticedSpeciesStade.PROPERTY_SPECIES_CODE + "= cps." + CroppingPlanSpecies.PROPERTY_CODE +
            "  AND ppcc." + PracticedPerennialCropCycle.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_CAMPAIGNS + " LIKE :campaign " +
            //"  AND ppcc." + practicedPerennialCropCycle.PROPERTY_PRACTICED_SYSTEM + "." + PracticedSystem.PROPERTY_GROWING_SYSTEM + "." + GrowingSystem.PROPERTY_GROWING_PLAN + "." + GrowingPlan.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE + "=" + "cps." + CroppingPlanSpecies.PROPERTY_CROPPING_PLAN_ENTRY +"." + CroppingPlanEntry.PROPERTY_DOMAIN + "." + Domain.PROPERTY_CODE +
            "  AND pi." + PracticedIntervention.PROPERTY_PRACTICED_CROP_CYCLE_PHASE + " IN ELEMENTS (" +
            "      ppcc." + PracticedPerennialCropCycle.PROPERTY_CROP_CYCLE_PHASES +")" +
            "  AND pss IN ELEMENTS(pi." + PracticedIntervention.PROPERTY_SPECIES_STADES + ")";

        Map<String, Long> result = queryBody(croppingPlanSpeciesIds, campaign, query);
        return result;
    }

    public List<PracticedIntervention> practicedInterventionsForSources(List<PracticedCropCycleNode> nodes) {
        StringBuilder query = new StringBuilder("FROM " + PracticedIntervention.class.getName() + " pi");
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        String piSource = PracticedIntervention.PROPERTY_PRACTICED_CROP_CYCLE_CONNECTION + "."+ PracticedCropCycleConnection.PROPERTY_SOURCE;
        Set<PracticedCropCycleNode> nodesSet = Sets.newHashSet(nodes);
        query.append(DaoUtils.andAttributeInIfNotEmpty("pi", piSource, args, nodesSet));

        List<PracticedIntervention> result = findAll(query.toString(), args);
        return result;
    }

    public List<PracticedIntervention> practicedInterventionsForTargets(List<PracticedCropCycleNode> nodes) {
        StringBuilder query = new StringBuilder("FROM " + PracticedIntervention.class.getName() + " pi");
        query.append(" WHERE 1 = 1");
        Map<String, Object> args = Maps.newLinkedHashMap();

        String piSource = PracticedIntervention.PROPERTY_PRACTICED_CROP_CYCLE_CONNECTION + "."+ PracticedCropCycleConnection.PROPERTY_TARGET;
        Set<PracticedCropCycleNode> nodesSet = Sets.newHashSet(nodes);
        query.append(DaoUtils.andAttributeInIfNotEmpty("pi", piSource, args, nodesSet));

        List<PracticedIntervention> result = findAll(query.toString(), args);
        return result;
    }

    /**
     * get a map of speciesStade and it's related Species TopiaId.
     * @param croppingPlanSpeciesIds species Id
     * @return a map of speciesStade and it's related Species TopiaId.
     */
    public Map<PracticedSpeciesStade, String> getSpeciesPracticedSpeciesStades(
            Iterable<String> croppingPlanSpeciesIds) {

        Set<String> cpeSpeciesIds = Sets.newHashSet(croppingPlanSpeciesIds);
        Map<String, Object> args = Maps.newLinkedHashMap();
        String query =
                "SELECT pss, cps."+ CroppingPlanSpecies.PROPERTY_TOPIA_ID + " FROM " + PracticedSpeciesStade.class.getName() + " pss, " + CroppingPlanSpecies.class.getName() + " cps " +
                        "  WHERE pss." + PracticedSpeciesStade.PROPERTY_SPECIES_CODE + "= cps." + CroppingPlanSpecies.PROPERTY_CODE;

        query += DaoUtils.andAttributeIn("cps", CroppingPlanSpecies.PROPERTY_TOPIA_ID, args, cpeSpeciesIds);

        Map<PracticedSpeciesStade, String> result = Maps.newHashMap();
        List<Object[]> queryResults = findAll(query, args);
        for (Object[] res : queryResults) {
            PracticedSpeciesStade speciesStade = (PracticedSpeciesStade) res[0];
            String speciesId = (String)res[1];
            result.put(speciesStade, speciesId);
        }
        return result;
    }


}
