package fr.ifremer.tutti.ui.swing.content.protocol;

/*
 * #%L
 * Tutti :: UI
 * %%
 * Copyright (C) 2012 - 2014 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Multimap;
import fr.ifremer.tutti.persistence.entities.TuttiEntities;
import fr.ifremer.tutti.persistence.entities.protocol.CaracteristicMappingRow;
import fr.ifremer.tutti.persistence.entities.protocol.OperationFieldMappingRow;
import fr.ifremer.tutti.persistence.entities.protocol.SpeciesProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocol;
import fr.ifremer.tutti.persistence.entities.protocol.TuttiProtocols;
import fr.ifremer.tutti.persistence.entities.referential.Caracteristic;
import fr.ifremer.tutti.persistence.entities.referential.Species;
import fr.ifremer.tutti.ui.swing.util.AbstractTuttiBeanUIModel;
import org.apache.commons.lang3.mutable.MutableInt;
import org.nuiton.util.beans.Binder;
import org.nuiton.util.beans.BinderFactory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author Tony Chemit - chemit@codelutin.com
 * @since 0.3
 */
public class EditProtocolUIModel extends AbstractTuttiBeanUIModel<TuttiProtocol, EditProtocolUIModel> implements TuttiProtocol {

    private static final long serialVersionUID = 1L;

    public static final String PROPERTY_CARACTERISTIC_MAPPING_ROWS = "caracteristicMappingRows";

    public static final String PROPERTY_OPERATION_FIELD_MAPPING_ROWS = "operationFieldMappingRows";

    public static final String PROPERTY_SPECIES_ROW = "speciesRow";

    public static final String PROPERTY_BENTHOS_ROW = "benthosRow";

    public static final String PROPERTY_REMOVE_SPECIES_ENABLED = "removeSpeciesEnabled";

    public static final String PROPERTY_REMOVE_BENTHOS_ENABLED = "removeBenthosEnabled";

    public static final String PROPERTY_REMOVE_CARACTERISTIC_MAPPING_ENABLED = "removeCaracteristicMappingEnabled";

    public static final String PROPERTY_MOVE_UP_CARACTERISTIC_MAPPING_ENABLED = "moveUpCaracteristicMappingEnabled";

    public static final String PROPERTY_MOVE_DOWN_CARACTERISTIC_MAPPING_ENABLED = "moveDownCaracteristicMappingEnabled";

    public static final String PROPERTY_IMPORTED = "imported";

    public static final String PROPERTY_CLONED = "cloned";

    public static final String PROPERTY_CLEANED = "cleaned";

    /**
     * Delegate edit object.
     *
     * @since 1.3
     */
    protected final TuttiProtocol editObject = TuttiProtocols.newTuttiProtocol();

    /**
     * Flag when a incoming protocol is imported.
     *
     * @since 1.0
     */
    protected boolean imported;

    /**
     * Flag when a incoming protocol is cloned.
     *
     * @since 1.0
     */
    protected boolean cloned;

    /**
     * Flag when a incoming protocol is cleaned.
     *
     * @since 2.4
     */
    protected boolean cleaned;

    /**
     * Can user remove a selected species?
     *
     * @since 0.3
     */
    protected boolean removeSpeciesEnabled;

    /**
     * Can user remove a selected benthos?
     *
     * @since 0.3
     */
    protected boolean removeBenthosEnabled;

    protected boolean removeCaracteristicMappingEnabled;

    protected boolean moveUpCaracteristicMappingEnabled;

    protected boolean moveDownCaracteristicMappingEnabled;

    protected List<Species> allSpecies;

    protected List<Species> allSynonyms;

    protected Multimap<String, Species> allSpeciesByTaxonId;

    protected Map<String, Species> allReferentSpeciesByTaxonId;

    protected List<Caracteristic> caracteristics;

    protected Map<String, Caracteristic> allCaracteristic;

    protected Map<Caracteristic, EditProtocolCaracteristicsRowModel> caracteristicMappingRows =
            new HashMap<Caracteristic, EditProtocolCaracteristicsRowModel>();

    protected List<EditProtocolOperationFieldsRowModel> operationFieldMappingRows =
            new ArrayList<EditProtocolOperationFieldsRowModel>();

    /**
     * Number of rows for each column
     *
     * @since 3.10
     */
    protected Map<String, MutableInt> numberOfRowsByColumn = new HashMap<String, MutableInt>();

    protected List<EditProtocolSpeciesRowModel> speciesRow;

    protected List<EditProtocolSpeciesRowModel> benthosRow;

    protected static Binder<EditProtocolUIModel, TuttiProtocol> toBeanBinder =
            BinderFactory.newBinder(EditProtocolUIModel.class,
                                    TuttiProtocol.class);

    protected static Binder<TuttiProtocol, EditProtocolUIModel> fromBeanBinder =
            BinderFactory.newBinder(TuttiProtocol.class, EditProtocolUIModel.class);

    public EditProtocolUIModel() {
        super(fromBeanBinder, toBeanBinder);
    }

    @Override
    protected TuttiProtocol newEntity() {
        return TuttiProtocols.newTuttiProtocol();
    }

    public void setLengthClassesPmfm(List<Caracteristic> lengthClassesPmfm) {
        List<String> ids = Lists.newArrayList(TuttiEntities.collecIds(lengthClassesPmfm));
        setLengthClassesPmfmId(ids);
    }

    public void setIndividualObservationPmfm(List<Caracteristic> individualObservationPmfm) {
        List<String> ids = TuttiEntities.collecIds(individualObservationPmfm);
        setIndividualObservationPmfmId(ids);
    }

    public Collection<EditProtocolCaracteristicsRowModel> getCaracteristicMappingRows() {
        return caracteristicMappingRows.values();
    }

    public void addCaracteristicMappingRow(EditProtocolCaracteristicsRowModel newRow) {
        caracteristicMappingRows.put(newRow.getPsfm(), newRow);
        firePropertyChange(PROPERTY_CARACTERISTIC_MAPPING_ROWS, null, getCaracteristicMappingRows());
    }

    public void removeCaracteristicMappingRows(Collection<EditProtocolCaracteristicsRowModel> rowsToRemove) {
        for (EditProtocolCaracteristicsRowModel row : rowsToRemove) {
            caracteristicMappingRows.remove(row.getPsfm());
        }
        firePropertyChange(PROPERTY_CARACTERISTIC_MAPPING_ROWS, null, getCaracteristicMappingRows());
    }

    public void setCaracteristicMappingRows(List<EditProtocolCaracteristicsRowModel> caracteristicMappingRows) {
        this.caracteristicMappingRows = new HashMap<Caracteristic, EditProtocolCaracteristicsRowModel>(
                Maps.uniqueIndex(caracteristicMappingRows,
                                 new Function<EditProtocolCaracteristicsRowModel, Caracteristic>() {

                                     @Override
                                     public Caracteristic apply(EditProtocolCaracteristicsRowModel editProtocolCaracteristicsRowModel) {
                                         return editProtocolCaracteristicsRowModel.getPsfm();
                                     }
                                 })
        );
        firePropertyChange(PROPERTY_CARACTERISTIC_MAPPING_ROWS, null, caracteristicMappingRows);
    }

    public boolean isCaracteristicUsedInMapping(Caracteristic caracteristic) {
        return caracteristicMappingRows.containsKey(caracteristic);
    }

    public Collection<Caracteristic> getUsedCaracteristics() {
        return caracteristicMappingRows.keySet();
    }

    public List<EditProtocolOperationFieldsRowModel> getOperationFieldMappingRows() {
        return operationFieldMappingRows;
    }

    public void setOperationFieldMappingRows(List<EditProtocolOperationFieldsRowModel> operationFieldMappingRows) {
        Object oldValue = getOperationFieldMappingRows();
        this.operationFieldMappingRows = operationFieldMappingRows;
        firePropertyChange(PROPERTY_OPERATION_FIELD_MAPPING_ROWS, oldValue, operationFieldMappingRows);
    }

    public List<EditProtocolSpeciesRowModel> getSpeciesRow() {
        return speciesRow;
    }

    public void setSpeciesRow(List<EditProtocolSpeciesRowModel> speciesRow) {
        Object oldValue = getSpeciesRow();
        this.speciesRow = speciesRow;
        firePropertyChange(PROPERTY_SPECIES_ROW, oldValue, speciesRow);
    }

    public List<EditProtocolSpeciesRowModel> getBenthosRow() {
        return benthosRow;
    }

    public void setBenthosRow(List<EditProtocolSpeciesRowModel> benthosRow) {
        Object oldValue = getBenthosRow();
        this.benthosRow = benthosRow;
        firePropertyChange(PROPERTY_BENTHOS_ROW, oldValue, benthosRow);
    }

    public boolean isRemoveSpeciesEnabled() {
        return removeSpeciesEnabled;
    }

    public void setRemoveSpeciesEnabled(boolean removeSpeciesEnabled) {
        this.removeSpeciesEnabled = removeSpeciesEnabled;
        firePropertyChange(PROPERTY_REMOVE_SPECIES_ENABLED, null, removeSpeciesEnabled);
    }

    public boolean isRemoveBenthosEnabled() {
        return removeBenthosEnabled;
    }

    public void setRemoveBenthosEnabled(boolean removeBenthosEnabled) {
        this.removeBenthosEnabled = removeBenthosEnabled;
        firePropertyChange(PROPERTY_REMOVE_BENTHOS_ENABLED, null, removeBenthosEnabled);
    }

    public boolean isRemoveCaracteristicMappingEnabled() {
        return removeCaracteristicMappingEnabled;
    }

    public void setRemoveCaracteristicMappingEnabled(boolean removeCaracteristicMappingEnabled) {
        this.removeCaracteristicMappingEnabled = removeCaracteristicMappingEnabled;
        firePropertyChange(PROPERTY_REMOVE_CARACTERISTIC_MAPPING_ENABLED, null, removeCaracteristicMappingEnabled);
    }

    public boolean isMoveUpCaracteristicMappingEnabled() {
        return moveUpCaracteristicMappingEnabled;
    }

    public void setMoveUpCaracteristicMappingEnabled(boolean moveUpCaracteristicMappingEnabled) {
        this.moveUpCaracteristicMappingEnabled = moveUpCaracteristicMappingEnabled;
        firePropertyChange(PROPERTY_MOVE_UP_CARACTERISTIC_MAPPING_ENABLED, null, moveUpCaracteristicMappingEnabled);
    }

    public boolean isMoveDownCaracteristicMappingEnabled() {
        return moveDownCaracteristicMappingEnabled;
    }

    public void setMoveDownCaracteristicMappingEnabled(boolean moveDownCaracteristicMappingEnabled) {
        this.moveDownCaracteristicMappingEnabled = moveDownCaracteristicMappingEnabled;
        firePropertyChange(PROPERTY_MOVE_DOWN_CARACTERISTIC_MAPPING_ENABLED, null, moveDownCaracteristicMappingEnabled);
    }

    public boolean isImported() {
        return imported;
    }

    public void setImported(boolean imported) {
        Object oldValue = isImported();
        this.imported = imported;
        firePropertyChange(PROPERTY_IMPORTED, oldValue, imported);
    }

    public boolean isCloned() {
        return cloned;
    }

    public void setCloned(boolean cloned) {
        Object oldValue = isCloned();
        this.cloned = cloned;
        firePropertyChange(PROPERTY_CLONED, oldValue, cloned);
    }

    public boolean isCleaned() {
        return cleaned;
    }

    public void setCleaned(boolean cleaned) {
        Object oldValue = isCleaned();
        this.cleaned = cleaned;
        firePropertyChange(PROPERTY_CLEANED, oldValue, cleaned);
    }

    public List<Species> getAllSpecies() {
        return allSpecies;
    }

    public void setAllSpecies(List<Species> allSpecies) {
        this.allSpecies = allSpecies;
        if (allReferentSpeciesByTaxonId != null && allSpecies != null) {
            allSynonyms = Lists.newArrayList(allSpecies);
            allSynonyms.removeAll(allReferentSpeciesByTaxonId.values());
        }
    }

    public Multimap<String, Species> getAllSpeciesByTaxonId() {
        return allSpeciesByTaxonId;
    }

    public Collection<Species> getAllSynonyms(String taxonId) {
        return allSpeciesByTaxonId.get(taxonId);
    }

    public void setAllSpeciesByTaxonId(Multimap<String, Species> allSpeciesByTaxonId) {
        this.allSpeciesByTaxonId = allSpeciesByTaxonId;
    }

    public List<Species> getAllSynonyms() {
        return allSynonyms;
    }

    public Map<String, Species> getAllReferentSpeciesByTaxonId() {
        return allReferentSpeciesByTaxonId;
    }

    public void setAllReferentSpeciesByTaxonId(Map<String, Species> allReferentSpeciesByTaxonId) {
        this.allReferentSpeciesByTaxonId = allReferentSpeciesByTaxonId;
        if (allReferentSpeciesByTaxonId != null && allSpecies != null) {
            allSynonyms = Lists.newArrayList(allSpecies);
            // tchemit-2013-10-04 Do not do a removeAll (bad performance)
            for (Species species : allReferentSpeciesByTaxonId.values()) {
                allSynonyms.remove(species);
            }
        }
    }

    public List<Caracteristic> getCaracteristics() {
        return caracteristics;
    }

    public void setCaracteristics(List<Caracteristic> caracteristics) {
        this.caracteristics = caracteristics;
    }

    public Map<String, Caracteristic> getAllCaracteristic() {
        return allCaracteristic;
    }

    public void setAllCaracteristic(Map<String, Caracteristic> allCaracteristic) {
        this.allCaracteristic = allCaracteristic;
    }

    //------------------------------------------------------------------------//
    //-- TuttiProtocol methods                                              --//
    //------------------------------------------------------------------------//

    @Override
    public String getName() {
        return editObject.getName();
    }

    @Override
    public void setName(String name) {
        Object oldValue = getName();
        editObject.setName(name);
        firePropertyChange(PROPERTY_NAME, oldValue, name);
    }

    @Override
    public String getComment() {
        return editObject.getComment();
    }

    @Override
    public void setComment(String comment) {
        Object oldValue = getComment();
        editObject.setComment(comment);
        firePropertyChange(PROPERTY_COMMENT, oldValue, comment);
    }

    @Override
    public List<String> getLengthClassesPmfmId() {
        return editObject.getLengthClassesPmfmId();
    }

    @Override
    public void setLengthClassesPmfmId(List<String> lengthClassesPmfmId) {
        editObject.setLengthClassesPmfmId(lengthClassesPmfmId);
        // force to always propagates (need to recompte data of combobox in species table)
        firePropertyChange(PROPERTY_LENGTH_CLASSES_PMFM_ID, null, lengthClassesPmfmId);
    }

    @Override
    public List<String> getIndividualObservationPmfmId() {
        return editObject.getIndividualObservationPmfmId();
    }

    @Override
    public void setIndividualObservationPmfmId(List<String> individualObservationPmfmId) {
        editObject.setIndividualObservationPmfmId(individualObservationPmfmId);
        firePropertyChange(PROPERTY_INDIVIDUAL_OBSERVATION_PMFM_ID, null, individualObservationPmfmId);
    }

    @Override
    public List<CaracteristicMappingRow> getCaracteristicMapping() {
        return editObject.getCaracteristicMapping();
    }

    @Override
    public void setCaracteristicMapping(List<CaracteristicMappingRow> caracteristicMapping) {
        editObject.setCaracteristicMapping(caracteristicMapping);
        firePropertyChange(PROPERTY_CARACTERISTIC_MAPPING, null, caracteristicMapping);
    }

    @Override
    public CaracteristicMappingRow getCaracteristicMapping(int index) {
        return editObject.getCaracteristicMapping(index);
    }

    @Override
    public boolean isCaracteristicMappingEmpty() {
        return editObject.isCaracteristicMappingEmpty();
    }

    @Override
    public int sizeCaracteristicMapping() {
        return editObject.sizeCaracteristicMapping();
    }

    @Override
    public void addCaracteristicMapping(CaracteristicMappingRow caracteristicMapping) {
        editObject.addCaracteristicMapping(caracteristicMapping);
    }

    @Override
    public void addAllCaracteristicMapping(Collection<CaracteristicMappingRow> caracteristicMapping) {
        editObject.addAllCaracteristicMapping(caracteristicMapping);
    }

    @Override
    public boolean removeCaracteristicMapping(CaracteristicMappingRow caracteristicMapping) {
        return editObject.removeCaracteristicMapping(caracteristicMapping);
    }

    @Override
    public boolean removeAllCaracteristicMapping(Collection<CaracteristicMappingRow> caracteristicMapping) {
        return editObject.removeAllCaracteristicMapping(caracteristicMapping);
    }

    @Override
    public boolean containsCaracteristicMapping(CaracteristicMappingRow caracteristicMapping) {
        return editObject.containsCaracteristicMapping(caracteristicMapping);
    }

    @Override
    public boolean containsAllCaracteristicMapping(Collection<CaracteristicMappingRow> caracteristicMapping) {
        return editObject.containsAllCaracteristicMapping(caracteristicMapping);
    }

    @Override
    public String getIndividualObservationPmfmId(int index) {
        return editObject.getIndividualObservationPmfmId(index);
    }

    @Override
    public boolean isIndividualObservationPmfmIdEmpty() {
        return editObject.isIndividualObservationPmfmIdEmpty();
    }

    @Override
    public int sizeIndividualObservationPmfmId() {
        return editObject.sizeIndividualObservationPmfmId();
    }

    @Override
    public void addIndividualObservationPmfmId(String individualObservationPmfmId) {
        editObject.addIndividualObservationPmfmId(individualObservationPmfmId);
    }

    @Override
    public void addAllIndividualObservationPmfmId(Collection<String> individualObservationPmfmId) {
        editObject.addAllIndividualObservationPmfmId(individualObservationPmfmId);
    }

    @Override
    public boolean removeIndividualObservationPmfmId(String individualObservationPmfmId) {
        return editObject.removeIndividualObservationPmfmId(individualObservationPmfmId);
    }

    @Override
    public boolean removeAllIndividualObservationPmfmId(Collection<String> individualObservationPmfmId) {
        return editObject.removeAllIndividualObservationPmfmId(individualObservationPmfmId);
    }

    @Override
    public boolean containsIndividualObservationPmfmId(String individualObservationPmfmId) {
        return editObject.containsIndividualObservationPmfmId(individualObservationPmfmId);
    }

    @Override
    public boolean containsAllIndividualObservationPmfmId(Collection<String> individualObservationPmfmId) {
        return editObject.containsAllIndividualObservationPmfmId(individualObservationPmfmId);
    }

    @Override
    public String getLengthClassesPmfmId(int index) {
        return editObject.getLengthClassesPmfmId(index);
    }

    @Override
    public boolean isLengthClassesPmfmIdEmpty() {
        return editObject.isLengthClassesPmfmIdEmpty();
    }

    @Override
    public int sizeLengthClassesPmfmId() {
        return editObject.sizeLengthClassesPmfmId();
    }

    @Override
    public void addLengthClassesPmfmId(String lengthClassesPmfmId) {
        editObject.addLengthClassesPmfmId(lengthClassesPmfmId);
        // force to always propagates (need to recompte data of combobox in species table)
        firePropertyChange(PROPERTY_LENGTH_CLASSES_PMFM_ID, null, getLengthClassesPmfmId());
    }

    @Override
    public void addAllLengthClassesPmfmId(Collection<String> lengthClassesPmfmId) {
        editObject.addAllLengthClassesPmfmId(lengthClassesPmfmId);
        // force to always propagates (need to recompte data of combobox in species table)
        firePropertyChange(PROPERTY_LENGTH_CLASSES_PMFM_ID, null, getLengthClassesPmfmId());
    }

    @Override
    public boolean removeLengthClassesPmfmId(String lengthClassesPmfmId) {
        boolean result = editObject.removeLengthClassesPmfmId(lengthClassesPmfmId);
        // force to always propagates (need to recompte data of combobox in species table)
        firePropertyChange(PROPERTY_LENGTH_CLASSES_PMFM_ID, null, getLengthClassesPmfmId());
        return result;
    }

    @Override
    public boolean removeAllLengthClassesPmfmId(Collection<String> lengthClassesPmfmId) {
        boolean result = editObject.removeAllLengthClassesPmfmId(lengthClassesPmfmId);
        // force to always propagates (need to recompte data of combobox in species table)
        firePropertyChange(PROPERTY_LENGTH_CLASSES_PMFM_ID, null, getLengthClassesPmfmId());
        return result;
    }

    @Override
    public boolean containsLengthClassesPmfmId(String lengthClassesPmfmId) {
        return editObject.containsLengthClassesPmfmId(lengthClassesPmfmId);
    }

    @Override
    public boolean containsAllLengthClassesPmfmId(Collection<String> lengthClassesPmfmId) {
        return editObject.containsAllLengthClassesPmfmId(lengthClassesPmfmId);
    }

    @Override
    public SpeciesProtocol getSpecies(int index) {
        return editObject.getSpecies(index);
    }

    @Override
    public boolean isSpeciesEmpty() {
        return editObject.isSpeciesEmpty();
    }

    @Override
    public int sizeSpecies() {
        return editObject.sizeSpecies();
    }

    @Override
    public void addSpecies(SpeciesProtocol species) {
        editObject.addSpecies(species);
    }

    @Override
    public void addAllSpecies(Collection<SpeciesProtocol> species) {
        editObject.addAllSpecies(species);
    }

    @Override
    public boolean removeSpecies(SpeciesProtocol species) {
        return editObject.removeSpecies(species);
    }

    @Override
    public boolean removeAllSpecies(Collection<SpeciesProtocol> species) {
        return editObject.removeAllSpecies(species);
    }

    @Override
    public boolean containsSpecies(SpeciesProtocol species) {
        return editObject.containsSpecies(species);
    }

    @Override
    public boolean containsAllSpecies(Collection<SpeciesProtocol> species) {
        return editObject.containsAllSpecies(species);
    }

    @Override
    public List<SpeciesProtocol> getSpecies() {
        return editObject.getSpecies();
    }

    @Override
    public void setSpecies(List<SpeciesProtocol> species) {
        editObject.setSpecies(species);
    }

    @Override
    public SpeciesProtocol getBenthos(int index) {
        return editObject.getBenthos(index);
    }

    @Override
    public boolean isBenthosEmpty() {
        return editObject.isBenthosEmpty();
    }

    @Override
    public int sizeBenthos() {
        return editObject.sizeBenthos();
    }

    @Override
    public void addBenthos(SpeciesProtocol benthos) {
        editObject.addBenthos(benthos);
    }

    @Override
    public void addAllBenthos(Collection<SpeciesProtocol> benthos) {
        editObject.addAllBenthos(benthos);
    }

    @Override
    public boolean removeBenthos(SpeciesProtocol benthos) {
        return editObject.removeBenthos(benthos);
    }

    @Override
    public boolean removeAllBenthos(Collection<SpeciesProtocol> benthos) {
        return editObject.removeAllBenthos(benthos);
    }

    @Override
    public boolean containsBenthos(SpeciesProtocol benthos) {
        return editObject.containsBenthos(benthos);
    }

    @Override
    public boolean containsAllBenthos(Collection<SpeciesProtocol> benthos) {
        return editObject.containsAllBenthos(benthos);
    }

    @Override
    public List<SpeciesProtocol> getBenthos() {
        return editObject.getBenthos();
    }

    @Override
    public void setBenthos(List<SpeciesProtocol> benthos) {
        editObject.setBenthos(benthos);
    }

    @Override
    public Integer getVersion() {
        return editObject.getVersion();
    }

    @Override
    public void setVersion(Integer version) {
        editObject.setVersion(version);
    }

    @Override
    public String getImportColumns(int index) {
        return editObject.getImportColumns(index);
    }

    @Override
    public boolean isImportColumnsEmpty() {
        return editObject.isImportColumnsEmpty();
    }

    @Override
    public int sizeImportColumns() {
        return editObject.sizeImportColumns();
    }

    @Override
    public void addImportColumns(String importColumns) {
        editObject.addImportColumns(importColumns);
        firePropertyChanged(TuttiProtocol.PROPERTY_IMPORT_COLUMNS, null, getImportColumns());
    }

    @Override
    public void addAllImportColumns(Collection<String> importColumns) {
        editObject.addAllImportColumns(importColumns);
        firePropertyChanged(TuttiProtocol.PROPERTY_IMPORT_COLUMNS, null, getImportColumns());
    }

    @Override
    public boolean removeImportColumns(String importColumns) {
        boolean removeImportColumns = editObject.removeImportColumns(importColumns);
        firePropertyChanged(TuttiProtocol.PROPERTY_IMPORT_COLUMNS, null, getImportColumns());
        return removeImportColumns;
    }

    @Override
    public boolean removeAllImportColumns(Collection<String> importColumns) {
        boolean removeAllImportColumns = editObject.removeAllImportColumns(importColumns);
        firePropertyChanged(TuttiProtocol.PROPERTY_IMPORT_COLUMNS, null, getImportColumns());
        return removeAllImportColumns;
    }

    @Override
    public boolean containsImportColumns(String importColumns) {
        return editObject.containsImportColumns(importColumns);
    }

    @Override
    public boolean containsAllImportColumns(Collection<String> importColumns) {
        return editObject.containsAllImportColumns(importColumns);
    }

    @Override
    public Collection<String> getImportColumns() {
        return editObject.getImportColumns();
    }

    @Override
    public void setImportColumns(Collection<String> importColumns) {
        editObject.setImportColumns(importColumns);
        firePropertyChanged(TuttiProtocol.PROPERTY_IMPORT_COLUMNS, null, getImportColumns());
    }

    @Override
    public void setOperationFieldMapping(Collection<OperationFieldMappingRow> operationFieldMapping) {
        editObject.setOperationFieldMapping(operationFieldMapping);
        firePropertyChanged(TuttiProtocol.PROPERTY_OPERATION_FIELD_MAPPING, null, getOperationFieldMapping());
    }

    @Override
    public Collection<OperationFieldMappingRow> getOperationFieldMapping() {
        return editObject.getOperationFieldMapping();
    }

    @Override
    public boolean containsAllOperationFieldMapping(Collection<OperationFieldMappingRow> operationFieldMapping) {
        return editObject.containsAllOperationFieldMapping(operationFieldMapping);
    }

    @Override
    public boolean containsOperationFieldMapping(OperationFieldMappingRow operationFieldMapping) {
        return editObject.containsOperationFieldMapping(operationFieldMapping);
    }

    @Override
    public boolean removeAllOperationFieldMapping(Collection<OperationFieldMappingRow> operationFieldMapping) {
        boolean removeAllOperationFieldMapping = editObject.removeAllOperationFieldMapping(operationFieldMapping);
        firePropertyChanged(TuttiProtocol.PROPERTY_OPERATION_FIELD_MAPPING, null, getOperationFieldMapping());
        return removeAllOperationFieldMapping;
    }

    @Override
    public boolean removeOperationFieldMapping(OperationFieldMappingRow operationFieldMapping) {
        boolean removeOperationFieldMapping = editObject.removeOperationFieldMapping(operationFieldMapping);
        firePropertyChanged(TuttiProtocol.PROPERTY_OPERATION_FIELD_MAPPING, null, getOperationFieldMapping());
        return removeOperationFieldMapping;
    }

    @Override
    public void addAllOperationFieldMapping(Collection<OperationFieldMappingRow> operationFieldMapping) {
        editObject.addAllOperationFieldMapping(operationFieldMapping);
        firePropertyChanged(TuttiProtocol.PROPERTY_OPERATION_FIELD_MAPPING, null, getOperationFieldMapping());
    }

    @Override
    public void addOperationFieldMapping(OperationFieldMappingRow operationFieldMapping) {
        editObject.addOperationFieldMapping(operationFieldMapping);
        firePropertyChanged(TuttiProtocol.PROPERTY_OPERATION_FIELD_MAPPING, null, getOperationFieldMapping());
    }

    @Override
    public int sizeOperationFieldMapping() {
        return editObject.sizeOperationFieldMapping();
    }

    @Override
    public boolean isOperationFieldMappingEmpty() {
        return editObject.isOperationFieldMappingEmpty();
    }

    @Override
    public OperationFieldMappingRow getOperationFieldMapping(int index) {
        return editObject.getOperationFieldMapping(index);
    }

    public int numberOfRows(String column) {
        int result = 0;
        MutableInt mutableInt = numberOfRowsByColumn.get(column);
        if (mutableInt != null) {
            result = mutableInt.intValue();
        }
        return result;
    }

    public void resetNumbersOfRows() {
        numberOfRowsByColumn = new HashMap<String, MutableInt>();
    }

    public int incNumberOfRows(String column) {
        MutableInt mutableInt = numberOfRowsByColumn.get(column);
        if (mutableInt == null) {
            mutableInt = new MutableInt(1);
            numberOfRowsByColumn.put(column, mutableInt);
        } else {
            mutableInt.increment();
        }
        return mutableInt.intValue();

    }

    public int decNumberOfRows(String column) {
        MutableInt mutableInt = numberOfRowsByColumn.get(column);
        Preconditions.checkNotNull(mutableInt);
        Preconditions.checkArgument(mutableInt.intValue() > 0);
        mutableInt.decrement();
        return mutableInt.intValue();
    }

}
