package fr.ifremer.tutti.ui.swing.launcher;

/*
 * #%L
 * Tutti :: Launcher
 * %%
 * Copyright (C) 2015 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

/**
 * Created on 1/3/15.
 *
 * @author Tony Chemit - chemit@codelutin.com
 * @since 1.0
 */
public class LauncherProperties {

    public static final String LAUNCHER_PROPERTIES_FILENAME = "launcher.properties";

    public static final String PROPERTY_UPDATER_JAR = "updater.jar";

    public static final String PROPERTY_UPDATER_LOGFILE = "updater.logFile";

    public static final String PROPERTY_APPLICATION_JAR = "application.jar";

    public static final String PROPERTY_APPLICATION_LOGFILE = "application.logFile";

    public static final String PROPERTY_APPLICATION_JVM_OPTIONS = "application.jvmOptions";

    public static final String PROPERTY_APPLICATION_OPTIONS = "application.options";

    private final Path basedir;

    Properties launcherProperties;

    public LauncherProperties(Path basedir) {
        this.basedir = basedir;
    }

    public void load() throws IOException {

        // application launcher properties
        Path applicationFile = basedir.resolve(Launcher.APPLICATION_DIR).resolve(LAUNCHER_PROPERTIES_FILENAME);
        // local launcher properties
        Path localFile = basedir.resolve(LAUNCHER_PROPERTIES_FILENAME);

        if (!(Files.exists(localFile) || Files.exists(applicationFile))) {

            throw new IOException("Could not find " + LAUNCHER_PROPERTIES_FILENAME);

        }

        Properties p1 = new Properties();
        if (Files.isReadable(applicationFile)) {
            try (FileReader reader = new FileReader(applicationFile.toFile())) {
                p1.load(reader);
            }
        }

        launcherProperties = new Properties(p1);
        if (Files.isReadable(localFile)) {
            try (FileReader reader = new FileReader(localFile.toFile())) {
                launcherProperties.load(reader);
            }
        }

    }

    public Path getUpdaterJarFile() {

        String jarFileName = launcherProperties.getProperty(PROPERTY_UPDATER_JAR);
        Path jarFile = basedir.resolve(jarFileName);
        return jarFile;

    }

    public Path getUpdaterLogFile() {

        String logFileName = launcherProperties.getProperty(PROPERTY_UPDATER_LOGFILE);
        Path logFile = basedir.resolve(logFileName);
        return logFile;

    }

    public Path getApplicationJarFile() {

        String jarFileName = launcherProperties.getProperty(PROPERTY_APPLICATION_JAR);
        Path jarFile = basedir.resolve(jarFileName);
        return jarFile;

    }

    public Path getApplicationLogFile() {

        String logFileName = launcherProperties.getProperty(PROPERTY_APPLICATION_LOGFILE);
        Path logFile = basedir.resolve(logFileName);
        return logFile;

    }

    public List<String> getApplicationJvmOptions() {

        List<String> options = getOptions(PROPERTY_APPLICATION_JVM_OPTIONS);
        return options;

    }

    public List<String> getApplicationOtherOptions() {

        List<String> options = getOptions(PROPERTY_APPLICATION_OPTIONS);
        return options;

    }

    protected List<String> getOptions(String key) {

        List<String> result = new ArrayList<>();
        String options = launcherProperties.getProperty(key);
        if (!(options == null || options.trim().isEmpty())) {

            String[] javaOptsArray = options.replaceAll("\"", "").split(" ");
            for (String opt : javaOptsArray) {
                if (!(opt == null || opt.trim().isEmpty())) {
                    result.add(opt);
                }
            }
        }

        return result;

    }

}
