/*
 * #%L
 * IsisFish
 * 
 * $Id: FilterModel.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2010 Ifremer, CodeLutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.widget.filter;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * This class represents a generic filter model.
 * 
 * It contains all data to be used for filter process.
 *
 * @author chemit
 */
public abstract class FilterModel<M, R> {

    private static final Log log = LogFactory.getLog(FilterModel.class);

    /** list of original items */
    protected List<M> original;

    /** current filter result list of items */
    protected List<M> result;

    /** list of param keys to be used */
    protected List<String> paramKeys;

    /** map -&gt; paramKey to param models */
    protected Map<String, FilterParamModel<M, ?>> paramModels;
    /**
     * flag to says inclusive mode (need one param accept)
     * or exclusive mode (need all params to accept)
     */
    protected boolean exclusiveMode = false;

    /**
     * @param item item to convert to result
     * @return converted value from item
     */
    public abstract R convertToResult(M item);

    protected FilterModel(List<M> original, Map<String, FilterParamModel<M, ?>> paramModels) {
        this.original = original;
        this.paramModels = paramModels;
        this.result = new ArrayList<M>();
    }

    public FilterParamModel<M, ?> getParamModel(String paramKey) {
        return paramModels.get(paramKey);
    }

    public void reset() {
        for (FilterParamModel<M, ?> paramModel : paramModels.values()) {
            paramModel.reset();
        }
        result.clear();
    }

    public List<M> doFilter() {
        
        // obtain list of param to use
        List<FilterParamModel<M, ?>> paramToUse = getParamsToUse();
        // filter original list with params found
        List<M> list;
        if (exclusiveMode) {
            list = exclusiveAccept(paramToUse);
        } else {
            list = inclusiveAccept(paramToUse);
        }
        log.debug("<-- " + this);
        return list;
    }

    public void selectAll() {
        reset();
        result.addAll(original);
    }

    public List<M> getFiltered() {
        return result;
    }

    public List<R> getFilteredResult() {
        List<R> result = new ArrayList<R>();
        for (M m : this.result) {
            result.add(convertToResult(m));
        }

        return result;
    }

    public List<M> getOriginal() {
        return original;
    }

    public int getNbResult() {
        return result.size();
    }

    public int getNbOriginal() {
        return original.size();
    }

    private List<FilterParamModel<M, ?>> getParamsToUse() {
        List<FilterParamModel<M, ?>> paramToUse = new ArrayList<FilterParamModel<M, ?>>();
        for (FilterParamModel<M, ?> paramModel : paramModels.values()) {
            if (paramModel.state == FilterParamModel.State.USE) {
                paramToUse.add(paramModel);
            }
        }
        return paramToUse;
    }

    private List<M> inclusiveAccept(List<FilterParamModel<M, ?>> paramToUse) {
        for (M m : original) {
            boolean accept = inclusiveAccept(paramToUse, m);
            if (accept) {
                // keep this item
                result.add(m);
            }
        }
        setParamResultState(paramToUse);
        return result;
    }

    private List<M> exclusiveAccept(List<FilterParamModel<M, ?>> paramToUse) {
        for (M m : original) {
            boolean accept = exclusiveAccept(paramToUse, m);
            if (accept) {
                // keep this item
                result.add(m);
            }
        }
        setParamResultState(paramToUse);
        return result;
    }

    private void setParamResultState(List<FilterParamModel<M, ?>> paramToUse) {
        for (FilterParamModel<M, ?> paramModel : paramToUse) {
            paramModel.resultState = paramModel.hits == 0 ?
                    FilterParamModel.ResultState.KO :
                    FilterParamModel.ResultState.OK;
        }
    }

    private boolean inclusiveAccept(List<FilterParamModel<M, ?>> paramToUse, M m) {
        boolean result = false;
        for (FilterParamModel<M, ?> paramModel : paramToUse) {
            if (paramModel.accept(m)) {
                result = true;
            }
        }
        return result;
    }

    private boolean exclusiveAccept(List<FilterParamModel<M, ?>> paramToUse, M m) {
        boolean result = true;
        for (FilterParamModel<M, ?> paramModel : paramToUse) {
            if (!paramModel.accept(m)) {
                result = false;
            }
        }
        return result;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(super.toString());
        sb.append(" (nbOriginal ").append(getNbOriginal()).append(')');
        sb.append(" (nbFiltered ").append(getNbResult()).append(')');
        for (FilterParamModel<M, ?> paramModel : getParamsToUse()) {
            sb.append('\n').append(paramModel);
        }
        return sb.toString();
    }

    public void setFiltered(List<M> filtered) {
        this.result.clear();
        this.result.addAll(filtered);
    }
}
