/*
 * #%L
 * IsisFish
 * 
 * $Id: SimulationParameterImpl.java 4072 2014-07-29 11:25:57Z bpoussin $
 * $HeadURL$
 * %%
 * Copyright (C) 2006 - 2012 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator;

import static org.nuiton.i18n.I18n.t;

import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixFactory;
import org.nuiton.math.matrix.MatrixHelper;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.SortedProperties;
import org.nuiton.util.StringUtil;
import org.nuiton.util.VersionUtil;

import fr.ifremer.isisfish.IsisConfig;
import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.datastore.ObjectiveStorage;
import fr.ifremer.isisfish.datastore.OptimizationStorage;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.datastore.RuleStorage;
import fr.ifremer.isisfish.datastore.SensitivityAnalysisStorage;
import fr.ifremer.isisfish.datastore.SensitivityExportStorage;
import fr.ifremer.isisfish.datastore.SimulationPlanStorage;
import fr.ifremer.isisfish.datastore.StorageHelper;
import fr.ifremer.isisfish.entities.Observation;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.entities.PopulationDAO;
import fr.ifremer.isisfish.entities.Strategy;
import fr.ifremer.isisfish.entities.StrategyDAO;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.export.ExportHelper;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.rule.RuleHelper;
import fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis;
import fr.ifremer.isisfish.util.ConverterUtil;
import java.util.Collections;
import org.apache.commons.beanutils.ConvertUtilsBean;

import static org.nuiton.i18n.I18n.n;

import static fr.ifremer.isisfish.simulator.SimulationParameterPropertiesHelper.DOT;
import static fr.ifremer.isisfish.simulator.SimulationParameterPropertiesHelper.LIST_SEPARATOR;

/**
 * Real {@link SimulationParameter} implementation.
 * 
 * @author chatellier
 * @version $Revision: 4072 $
 * 
 * Last update : $Date: 2014-07-29 13:25:57 +0200 (Tue, 29 Jul 2014) $
 * By : $Author: bpoussin $
 */
public class SimulationParameterImpl implements SimulationParameter {

    /** Logger for this class. */
    final private static Log log = LogFactory.getLog(SimulationParameterImpl.class);

    /** Remember last read properties. */
    protected Properties propertiesParameters;

    /**
     * Isis-fish version that permit to do the simulation.
     * Must be set just before simulation by simulator.
     */
    protected String isisFishVersion;

    /** Description de la simulation. */
    protected String description;

    /** Nom de la region sur lequel on simule. */
    protected String regionName;

    /** Nombre d'année sur lequel on souhaite simuler. */
    protected Integer numberOfYear;

    /** Le nom de l'objet script a utiliser pour faire la simulation. */
    protected String simulatorName;

    /** Utilisation ou non du cache. */
    protected Boolean useCache;

    /** Utilisation ou non des statistiques. */
    protected Boolean useStatistic;

    /** La liste des strategies a utiliser pour la simulation. */
    protected List<Strategy> strategies;

    /** La liste des populations a utiliser pour la simulation. */
    protected List<Population> populations;

    /** Les effectifs initiaux des différentes population. */
    protected Map<Population, MatrixND> numbers;

    /** La liste des regles de gestions a utiliser pour la simulation. */
    protected List<Rule> rules;

    /** La liste des regles potentiellement ajoutée par les plans de simulation. */
    protected List<String> extraRules;

    /** La liste des plans a utiliser pour la simulation. */
    protected List<SimulationPlan> simulationPlans;

    /** Ma liste des exports automatique a faire en fin de simulation. */
    protected List<String> exportNames;

    /** Utilisation du script de pre simulation. */
    protected Boolean usePreScript;

    /** Le script de pre simulation a utiliser. */
    protected String preScript;

    /** Le script de pre simulation defini par Isis */
    protected String generatedPreScript;

    /** Utilisation du plan de simulation. */
    protected Boolean useSimulationPlan;

    /**
     * Le numero de sequence de la liste des plans de simulation. Le premier
     * élement du plan doit etre 0. Si la simulation ne fait pas partie d'un
     * plan la valeur est -1. */
    protected Integer simulationPlanNumber;

    /** Export utilisés pour les analyses de sensibilités. */
    protected List<SensitivityExport> sensitivityExports;

    /**
     * Nombre de simulation constituant l'analyse de sensibilité.
     * 
     * FIXME ce parametre ne devrait pas ce trouver à cet endroit.
     * Ce n'est pas un parametres, mais une infos de simulation.
     * Cela permet de savoir combien de simulation il y a au total
     * pour savoir quand elle sont terminées.
     */
    protected Integer numberOfSensitivitySimulation;

    /**
     * Dans une analyze de sensibilite, conserver les résultats de toutes
     * les simulations est inutile, seule les résultats de la premières
     * sont nécéssaires. Les resultats peuvent être supprimés après les
     * export de données.
     */
    protected Boolean sensitivityAnalysisOnlyKeepFirst;

    /** Script utilisé pour les analyses de sensibilités. */
    protected SensitivityAnalysis sensitivityAnalysis;

    /** Utilisation d'un script de simulation. */
    protected Boolean useOptimization;

    /** Script d'optimisation. */
    protected Optimization optimization;

    /** Fonction d'objectif. */
    protected Objective objective;

    /** Exports et observations (optimisation). */
    protected Map<Export, Observation> optimizationExportsObservations;

    /** La liste des resultats qui nous interesse. */
    protected Collection<String> resultEnabled;

    /** Autre parametre defini par l'utilisateur. */
    protected Map<String, String> tagValue;

    /**
     * Si vrai seul les exports automatiques ne sont souhaiter, la simulation
     * pourrait etre supprimée a la fin.
     */
    protected Boolean onlyExport;

    /** Le niveau de log du simulateur à utiliser pendant la simulation. Par defaut info. */
    protected String simulLogLevel;
    /** Le niveau de log des scripts à utiliser pendant la simulation. Par defaut info. */
    protected String scriptLogLevel;
    /** Le niveau de log des librairies à utiliser pendant la simulation. Par defaut error. */
    protected String libLogLevel;

    /** La region corespondant au parametre {@link #regionName}. */
    protected transient RegionStorage region;

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#check()
     */
    @Override
    public List<String> check() {
        List<String> result = new ArrayList<String>();
        // TODO faire la verif et pour chaque erreur mettre un message
        // dans la liste result (par exemple si pas d'effectif
        // pour une pop
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getIsisFishVersion()
     */
    @Override
    public String getIsisFishVersion() {
        // if not already set
        if (isisFishVersion == null) {
            if (propertiesParameters != null) {
                isisFishVersion = SimulationParameterPropertiesHelper.getIsisFishVersion(propertiesParameters);
            } else {
                // default value
                isisFishVersion = IsisConfig.getVersion();
            }
        }
        return isisFishVersion;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setIsisFishVersion(java.lang.String)
     */
    @Override
    public void setIsisFishVersion(String isisFishVersion) {
        this.isisFishVersion = isisFishVersion;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getDescription()
     */
    @Override
    public String getDescription() {

        if (description == null) {
            if (propertiesParameters != null) {
                description = SimulationParameterPropertiesHelper
                        .getDescription(propertiesParameters);
            } else {
                // default value
                description = "";
            }
        }
        return this.description;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setDescription(java.lang.String)
     */
    @Override
    public void setDescription(String description) {
        this.description = description;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRegion()
     */
    @Override
    public RegionStorage getRegion() {

        if (region == null || !region.getName().equals(getRegionName())) {
            // bien faire attention ici que la bonne region est ouverte
            // - isis-database-3 dans le cas d'une region normale
            // - simulation dans le cas d'une simulation
            region = RegionStorage.getRegion(getRegionName());
        } else {
            if (log.isDebugEnabled()) {
                log.debug(t("Region %s already inited", getRegionName()));
            }
        }
        return this.region;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOf(fr.ifremer.isisfish.entities.Population)
     */
    @Override
    public MatrixND getNumberOf(Population pop) {
        if (numbers == null) {
            numbers = new LinkedHashMap<Population, MatrixND>();
        }
        MatrixND result = numbers.get(pop);
        if (result == null) {
            List groups = pop.getPopulationGroup();
            List zones = pop.getPopulationZone();

            if (groups.isEmpty()) {
                groups.add("No group");
            }
            if (zones.isEmpty()) {
                zones.add("No zone");
            }
            List[] semantics = new List[] { groups, zones };
            result = MatrixFactory.getInstance().create(
                    n("matrixAbundance"),
                    semantics,
                    new String[] { n("Groups"), n("Zones") });

            numbers.put(pop, result);
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getExtraRules()
     */
    @Override
    public List<String> getExtraRules() {

        if (extraRules == null) {
            // no properties in config ?
            extraRules = new ArrayList<String>();
        }
        return extraRules;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addExtraRules(java.lang.String)
     */
    @Override
    public void addExtraRules(String... extraRules) {
        getExtraRules().addAll(Arrays.asList(extraRules));
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getPopulations()
     */
    @Override
    public List<Population> getPopulations() {
        if (populations == null) {
            populations = new ArrayList<Population>();

            if (propertiesParameters != null) {

                try {
                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y es, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDB();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    PopulationDAO populationDAO = IsisFishDAOHelper
                            .getPopulationDAO(tx);
                    String[] populationList = SimulationParameterPropertiesHelper
                            .getPopulationNames(propertiesParameters);
                    for (String name : populationList) {
                        if (StringUtils.isNotEmpty(name)) {
                            try {
                                Population population = populationDAO.findByName(name);
                                populations.add(population);

                                List number = SimulationParameterPropertiesHelper.getPopulationNumbers(propertiesParameters, name);
                                MatrixND mat = getNumberOf(population);
                                mat.fromList(number);
                            } catch (TopiaException eee) {
                                if (log.isWarnEnabled()) {
                                    log.warn("Can't find population: " + name, eee);
                                }
                            }
                        }
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee1) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get PopulationDAO", eee1);
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip population reloading");
                }
            }
        }
        return populations;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setPopulations(java.util.List)
     */
    @Override
    public void setPopulations(List<Population> populations) {
        this.populations = populations;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getStrategies()
     */
    @Override
    public List<Strategy> getStrategies() {

        if (strategies == null) {

            strategies = new ArrayList<Strategy>();

            if (propertiesParameters != null) {
                strategies = new ArrayList<Strategy>();

                try {

                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y es, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDB();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    StrategyDAO strategyDAO = IsisFishDAOHelper.getStrategyDAO(tx);
                    String[] strategyList = SimulationParameterPropertiesHelper
                            .getStrategieNames(propertiesParameters);
                    for (String name : strategyList) {
                        if (StringUtils.isNotEmpty(name)) {
                            try {
                                Strategy str = strategyDAO.findByName(name);
                                strategies.add(str);
                            } catch (TopiaException eee) {
                                if (log.isWarnEnabled()) {
                                    log.warn("Can't find strategy: " + name, eee);
                                }
                            }
                        }
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee1) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get StrategyDAO", eee1);
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip strategies reloading");
                }
            }
        }
        return strategies;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setStrategies(java.util.List)
     */
    @Override
    public void setStrategies(List<Strategy> strategies) {
        this.strategies = strategies;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addSimulationPlan(fr.ifremer.isisfish.simulator.SimulationPlan)
     */
    @Override
    public void addSimulationPlan(SimulationPlan plan) {
        getSimulationPlans().add(plan);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#removeSimulationPlan(fr.ifremer.isisfish.simulator.SimulationPlan)
     */
    @Override
    public boolean removeSimulationPlan(SimulationPlan plan) {
        return getSimulationPlans().remove(plan);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulationPlans()
     */
    @Override
    public List<SimulationPlan> getSimulationPlans() {
        if (simulationPlans == null) {

            simulationPlans = new ArrayList<SimulationPlan>();

            if (propertiesParameters != null) {
                // simulation plan
                String[] planList = SimulationParameterPropertiesHelper
                        .getSimulationPlanNames(propertiesParameters);
                int planIndex = 0;
                for (String name : planList) {
                    if (StringUtils.isNotEmpty(name)) {
                        try {
                            SimulationPlanStorage planStorage = SimulationPlanStorage.getSimulationPlan(name);
                            if (planStorage != null) { // since 4.0.0.3 can return null
                                SimulationPlan plan = planStorage.getNewInstance();
                                StorageHelper.populateStorageParams(planIndex++,
                                        getRegion().getStorage(), plan, propertiesParameters,
                                        SimulationParameterPropertiesHelper.PLAN_KEY);
                                simulationPlans.add(plan);
                            }
                        } catch (Exception eee) {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't find plan: " + name, eee);
                            }
                        }
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip plans reloading");
                }
            }
        }
        return this.simulationPlans;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulationPlans(java.util.List)
     */
    @Override
    public void setSimulationPlans(List<SimulationPlan> plans) {
        this.simulationPlans = plans;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isIndependentPlan()
     */
    @Override
    public boolean isIndependentPlan() {
        boolean result = true;
        for (SimulationPlan plan : getSimulationPlans()) {
            if (!(plan instanceof SimulationPlanIndependent)) {
                result = false;
                break;
            }
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#clearPlans()
     */
    @Override
    public void clearPlans() {
        if (simulationPlans != null) {
            simulationPlans.clear();
        }
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#addRule(fr.ifremer.isisfish.rule.Rule)
     */
    @Override
    public void addRule(Rule rule) {
        getRules().add(rule);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#removeRule(fr.ifremer.isisfish.rule.Rule)
     */
    @Override
    public boolean removeRule(Rule rule) {
        return getRules().remove(rule);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRules()
     */
    @Override
    public List<Rule> getRules() {
        if (rules == null) {
            rules = new ArrayList<Rule>();

            if (propertiesParameters != null) {
                try {
                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y es, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDB();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    // rules
                    String[] ruleList = SimulationParameterPropertiesHelper
                            .getRuleNames(propertiesParameters);
                    int ruleIndex = 0;
                    for (String name : ruleList) {
                        if (StringUtils.isNotEmpty(name)) {
                            try {
                                Rule rule = RuleStorage.getRule(name).getNewInstance();
                                RuleHelper.populateRule(ruleIndex++, tx, rule, propertiesParameters);
                                rules.add(rule);
                            } catch (Exception eee) {
                                if (log.isWarnEnabled()) {
                                    log.warn("Can't find rule: " + name, eee);
                                }
                            }
                        }
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee1) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get TopiaContext", eee1);
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip rule reloading");
                }
            }
        }
        return rules;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setRules(java.util.List)
     */
    @Override
    public void setRules(List<Rule> rules) {
        this.rules = rules;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#clearRules()
     */
    @Override
    public void clearRules() {
        if (rules != null) {
            rules.clear();
        }
    }

    @Override
    public boolean getUseCache() {
        if (useCache == null) {
            if (propertiesParameters != null) {
                // en version < 4.3, le parametre était nommé 'useOptimization'
                // qualifiait le cache, il été renommé ensuite
                if (VersionUtil.smallerThan(getIsisFishVersion(), "4.3.0.0")) {
                    useCache = Boolean.valueOf(propertiesParameters.getProperty(
                            SimulationParameterPropertiesHelper.USE_OPTIMIZATION_KEY, "true"));
                } else {
                    useCache = Boolean.valueOf(propertiesParameters.getProperty(
                            SimulationParameterPropertiesHelper.USE_CACHE_KEY, "true"));
                }
            } else {
                useCache = Boolean.TRUE;
            }
        }
        return useCache;
    }

    @Override
    public void setUseCache(boolean useCache) {
        this.useCache = useCache;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUseStatistic()
     */
    @Override
    public boolean getUseStatistic() {

        if (useStatistic == null) {
            if (propertiesParameters != null) {
                useStatistic = SimulationParameterPropertiesHelper.getUseStatistic(propertiesParameters);
            } else {
                useStatistic = Boolean.FALSE;
            }
        }
        return this.useStatistic;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUseStatistic(boolean)
     */
    @Override
    public void setUseStatistic(boolean useStatistic) {
        this.useStatistic = useStatistic;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getExportNames()
     */
    @Override
    public List<String> getExportNames() {
        if (exportNames == null) {
            exportNames = new ArrayList<String>();

            if (propertiesParameters != null) {
                // exports
                String[] exportList = SimulationParameterPropertiesHelper.getExportNames(propertiesParameters);
                for (String name : exportList) {
                    if (StringUtils.isNotEmpty(name)) {
                        exportNames.add(name);
                    }
                }
            }
        }
        return this.exportNames;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setExportNames(java.util.List)
     */
    @Override
    public void setExportNames(List<String> exportNames) {
        this.exportNames = exportNames;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOfSensitivitySimulation()
     */
    @Override
    public int getNumberOfSensitivitySimulation() {

        if (numberOfSensitivitySimulation == null) {

            if (propertiesParameters != null) {
                numberOfSensitivitySimulation =
                        SimulationParameterPropertiesHelper.getNumberOfSensitivitySimulation(propertiesParameters);
            } else {
                numberOfSensitivitySimulation = -1;
            }
        }

        return numberOfSensitivitySimulation;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setNumberOfSensitivitySimulation(int)
     */
    @Override
    public void setNumberOfSensitivitySimulation(
            int numberOfSensitivitySimulation) {
        this.numberOfSensitivitySimulation = numberOfSensitivitySimulation;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSensitivityAnalysis()
     */
    @Override
    public SensitivityAnalysis getSensitivityAnalysis() {

        if (sensitivityAnalysis == null) {
            if (propertiesParameters != null) {
                String analysisName = SimulationParameterPropertiesHelper
                        .getSensitivityAnalysis(propertiesParameters);
                if (StringUtils.isNotEmpty(analysisName)) {
                    try {
                        SensitivityAnalysisStorage sensitivityStorage = SensitivityAnalysisStorage.getSensitivityAnalysis(analysisName);
                        sensitivityAnalysis = sensitivityStorage.getNewInstance();
                        // 0 = only single sensitivity
                        StorageHelper.populateStorageParams(0, getRegion().getStorage(),
                                sensitivityAnalysis, propertiesParameters,
                                SimulationParameterPropertiesHelper.SENSITIVITY_KEY);
                    } catch (Exception eee) {
                        sensitivityAnalysis = null;
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find sensitivity: " + sensitivityAnalysis, eee);
                        }
                    }
                }
            }
        }

        return sensitivityAnalysis;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSensitivityAnalysis(fr.ifremer.isisfish.simulator.sensitivity.SensitivityAnalysis)
     */
    @Override
    public void setSensitivityAnalysis(
            SensitivityAnalysis sensitivityAnalysis) {
        this.sensitivityAnalysis = sensitivityAnalysis;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSensitivityExport()
     */
    @Override
    public List<SensitivityExport> getSensitivityExport() {
        if (sensitivityExports == null) {
            sensitivityExports = new ArrayList<SensitivityExport>();

            if (propertiesParameters != null) {
                try {
                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y es, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDbResult();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    // sensitivity export
                    String[] sensitivityExportList = SimulationParameterPropertiesHelper
                            .getSensitivityExportNames(propertiesParameters);
                    int sensitivityExportIndex = 0;
                    for (String name : sensitivityExportList) {
                        try {
                            if (!StringUtils.isEmpty(name)) {
                                SensitivityExport sensitivityExport = SensitivityExportStorage
                                        .getSensitivityExport(name)
                                        .getNewInstance();
                                ExportHelper.populateSensitivityExport(
                                                sensitivityExportIndex++, tx,
                                                sensitivityExport,
                                                propertiesParameters);
                                sensitivityExports.add(sensitivityExport);
                            }
                        } catch (Exception eee) {
                            if (log.isWarnEnabled()) {
                                log.warn("Can't find SensitivityExport: " + name, eee);
                            }
                        }
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee1) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get TopiaContext", eee1);
                    }
                }
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("No last read properties, skip sensitivity exports reloading");
                }
            }
        }
        return sensitivityExports;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSensitivityExport(java.util.List)
     */
    @Override
    public void setSensitivityExport(List<SensitivityExport> sensitivityExport) {
        this.sensitivityExports = sensitivityExport;
    }

    @Override
    public Optimization getOptimization() {
        if (optimization == null) {
            if (propertiesParameters != null) {
                String optimizationName = SimulationParameterPropertiesHelper
                        .getOptimizationName(propertiesParameters);
                if (!StringUtils.isEmpty(optimizationName)) {
                    try {
                        OptimizationStorage optimizationStorage =
                                OptimizationStorage.getOptimization(optimizationName);
                        optimization = optimizationStorage.getNewInstance();
                        // 0 = only single optimization
                        StorageHelper.populateStorageParams(0, getRegion().getStorage(),
                                optimization, propertiesParameters,
                                SimulationParameterPropertiesHelper.OPTIMIZATION_KEY);
                    } catch (Exception eee) {
                        optimization = null;
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find optimization: " + optimization, eee);
                        }
                    }
                }
            }
        }

        return optimization;
    }

    @Override
    public boolean getUseOptimization() {
        if (useOptimization == null) {
            if (propertiesParameters != null) {
                // en version < 4.3, le parametre était nommé 'useOptimization'
                // qualifiant le cache, il a été renommé ensuite
                if (VersionUtil.smallerThan(getIsisFishVersion(), "4.3.0.0")) {
                    useOptimization = Boolean.FALSE;
                } else {
                    useOptimization = Boolean.valueOf(propertiesParameters.getProperty(
                            SimulationParameterPropertiesHelper.USE_OPTIMIZATION_KEY, "false"));
                }
            } else {
                useOptimization = Boolean.FALSE;
            }
        }
        return useOptimization;
    }

    @Override
    public void setUseOptimization(boolean useOptimization) {
        this.useOptimization = useOptimization;
    }

    @Override
    public void setOptimization(Optimization optimization) {
        this.optimization = optimization;
    }

    @Override
    public Objective getObjective() {
        if (objective == null) {
            if (propertiesParameters != null) {
                String objectiveName = SimulationParameterPropertiesHelper
                        .getObjectiveName(propertiesParameters);
                if (!StringUtils.isEmpty(objectiveName)) {
                    try {
                        ObjectiveStorage objectiveStorage =
                                ObjectiveStorage.getObjective(objectiveName);
                        objective = objectiveStorage.getNewInstance();
                        // 0 = only single objective
                        StorageHelper.populateStorageParams(0, getRegion().getStorage(),
                                objective, propertiesParameters,
                                SimulationParameterPropertiesHelper.OBJECTIVE_KEY);
                    } catch (Exception eee) {
                        objective = null;
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find objective: " + objective, eee);
                        }
                    }
                }
            }
        }
        return objective;
    }

    @Override
    public void setObjective(Objective objective) {
        this.objective = objective;
    }

    @Override
    public Map<Export, Observation> getOptimizationExportsObservations() {
        if (optimizationExportsObservations == null) {
            // must be sorted for ui (LinkedHashMap)
            optimizationExportsObservations = new LinkedHashMap<>();

            if (propertiesParameters != null) {
                try {
                    // On verifie tout d'abord que l'on ai pas dans une simulation
                    // si on y est, on utilise le context static non null du thread local
                    // Resoud les lazy exceptions des parametres des regles
                    boolean mustClose = false;
                    TopiaContext tx = SimulationContext.get().getDB();

                    if (tx == null) {
                        // not in simulation, create transaction
                        tx = getRegion().getStorage().beginTransaction();
                        mustClose = true;
                    }

                    // rules
                    String[] optimizationExportList = SimulationParameterPropertiesHelper
                            .getOptimizationExportNames(propertiesParameters);
                    ConvertUtilsBean beanUtils = ConverterUtil.getConverter(tx);
                    int optimizationExportIndex = 0;
                    for (String name : optimizationExportList) {
                        if (StringUtils.isNotBlank(name)) {
                            try {
                                Export export = ExportStorage.getExport(name).getNewInstance();
                                Observation observation = null;
                                String observationId = propertiesParameters.getProperty(
                                        SimulationParameterPropertiesHelper.
                                                OPTIMIZATION_OBSERVATION_KEY
                                                + DOT + optimizationExportIndex);
                                if (StringUtils.isNotBlank(observationId)) {
                                    observation = (Observation)beanUtils.convert(observationId, TopiaEntity.class);
                                }
                                optimizationExportsObservations.put(export, observation);
                            } catch (Exception eee) {
                                if (log.isWarnEnabled()) {
                                    log.warn("Can't find rule: " + name, eee);
                                }
                            }
                        }
                        optimizationExportIndex++;
                    }

                    // si la transaction a été ouverte (pas dans une simulation)
                    // on la referme
                    if (mustClose) {
                        tx.closeContext();
                    }
                } catch (TopiaException eee) {
                    if (log.isWarnEnabled()) {
                        log.warn("Can't get TopiaContext", eee);
                    }
                }
            }
        }

        return optimizationExportsObservations;
    }

    @Override
    public void setOptimizationExportsObservations(Map<Export, Observation> optimizationExportsObservations) {
        this.optimizationExportsObservations = optimizationExportsObservations;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUseAnalysePlan()
     */
    @Override
    public boolean getUseSimulationPlan() {

        if (useSimulationPlan == null) {

            if (propertiesParameters != null) {
                useSimulationPlan = SimulationParameterPropertiesHelper.getUseSimulationPlan(propertiesParameters);
            } else {
                useSimulationPlan = Boolean.FALSE;
            }
        }
        return useSimulationPlan;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUseAnalysePlan(boolean)
     */
    @Override
    public void setUseSimulationPlan(boolean useSimulationPlan) {
        this.useSimulationPlan = useSimulationPlan;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulationPlanNumber()
     */
    @Override
    public int getSimulationPlanNumber() {

        if (simulationPlanNumber == null) {

            if (propertiesParameters != null) {
                simulationPlanNumber = SimulationParameterPropertiesHelper.getSimulationPlanNumber(propertiesParameters);
            } else {
                simulationPlanNumber = -1;
            }
        }

        return simulationPlanNumber;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulationPlanNumber(int)
     */
    @Override
    public void setSimulationPlanNumber(int simulationPlanNumber) {
        this.simulationPlanNumber = simulationPlanNumber;
    }

    @Override
    public boolean isSensitivityAnalysisOnlyKeepFirst() {
        if (sensitivityAnalysisOnlyKeepFirst == null) {

            if (propertiesParameters != null) {
                sensitivityAnalysisOnlyKeepFirst = SimulationParameterPropertiesHelper.isSensitivityAnalysisOnlyKeepFirst(propertiesParameters);
            } else {
                sensitivityAnalysisOnlyKeepFirst = Boolean.FALSE;
            }
        }

        return sensitivityAnalysisOnlyKeepFirst;
    }

    @Override
    public void setSensitivityAnalysisOnlyKeepFirst(boolean onlyKeepFirst) {
        sensitivityAnalysisOnlyKeepFirst = onlyKeepFirst;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getNumberOfYear()
     */
    @Override
    public int getNumberOfYear() {

        if (numberOfYear == null) {

            if (propertiesParameters != null) {
                numberOfYear = SimulationParameterPropertiesHelper
                        .getNumberOfYear(propertiesParameters);
            } else {
                numberOfYear = 1;
            }
        }
        return numberOfYear;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setNumberOfYear(int)
     */
    @Override
    public void setNumberOfYear(int numberOfYear) {
        this.numberOfYear = numberOfYear;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getUsePreScript()
     */
    @Override
    public boolean getUsePreScript() {

        if (usePreScript == null) {

            if (propertiesParameters != null) {
                usePreScript = SimulationParameterPropertiesHelper.getUsePreScript(propertiesParameters);
            } else {
                usePreScript = Boolean.FALSE;
            }
        }

        return usePreScript;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setUsePreScript(boolean)
     */
    @Override
    public void setUsePreScript(boolean usePreScript) {
        this.usePreScript = usePreScript;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getPreScript()
     */
    @Override
    public String getPreScript() {

        if (preScript == null) {

            if (propertiesParameters != null) {
                preScript = SimulationParameterPropertiesHelper.getPreScript(propertiesParameters);
            } else {
                preScript = "";
            }
        }

        return preScript;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setPreScript(java.lang.String)
     */
    @Override
    public void setPreScript(String preScript) {
        this.preScript = preScript;
    }

    @Override
    public String getGeneratedPreScript() {

        if (generatedPreScript == null) {

            if (propertiesParameters != null) {
                generatedPreScript = SimulationParameterPropertiesHelper
                        .getGeneratedPreScript(propertiesParameters);
            } else {
                generatedPreScript = "";
            }
        }

        return generatedPreScript;
    }

    @Override
    public void setGeneratedPreScript(String preScript) {
        this.generatedPreScript = preScript;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getRegionName()
     */
    @Override
    public String getRegionName() {

        if (regionName == null) {
            if (propertiesParameters != null) {
                regionName = SimulationParameterPropertiesHelper
                        .getRegionName(propertiesParameters);
            } else {
                // defaut value
                regionName = "";
            }
        }

        return regionName;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setRegionName(java.lang.String)
     */
    @Override
    public void setRegionName(String regionName) {
        this.regionName = regionName;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulatorName()
     */
    @Override
    public String getSimulatorName() {

        if (simulatorName == null) {
            if (propertiesParameters != null) {
                simulatorName = SimulationParameterPropertiesHelper
                        .getSimulatorName(propertiesParameters);
            } else {
                simulatorName = "DefaultSimulator.java";
            }
        }
        return simulatorName;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulatorName(java.lang.String)
     */
    @Override
    public void setSimulatorName(String simulatorName) {
        this.simulatorName = simulatorName;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getResultEnabled()
     */
    @Override
    public Collection<String> getResultEnabled() {

        if (resultEnabled == null) {

            resultEnabled = new LinkedList<String>();

            if (propertiesParameters != null) {
                String[] resultList = 
                        SimulationParameterPropertiesHelper.getResultNames(propertiesParameters);
                Collections.addAll(resultEnabled, resultList);
            }
        }
        return resultEnabled;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setResultEnabled(java.util.Collection)
     */
    @Override
    public void setResultEnabled(Collection<String> resultEnabled) {
        this.resultEnabled = resultEnabled;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getTagValue()
     */
    @Override
    public Map<String, String> getTagValue() {

        if (tagValue == null) {
            if (propertiesParameters != null) {
                tagValue = SimulationParameterPropertiesHelper.getTagValue(propertiesParameters);
            } else {
                tagValue = new LinkedHashMap<String, String>();
            }
        }

        return tagValue;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setTagValue(java.util.Map)
     */
    @Override
    public void setTagValue(Map<String, String> tagValue) {
        this.tagValue = tagValue;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getSimulLogLevel()
     */
    @Override
    public String getSimulLogLevel() {

        if (simulLogLevel == null) {

            if (propertiesParameters != null) {
                simulLogLevel = SimulationParameterPropertiesHelper
                        .getSimulLogLevel(propertiesParameters);
            } else {
                simulLogLevel = "info";
            }
        }

        return simulLogLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setSimulLogLevel(java.lang.String)
     */
    @Override
    public void setSimulLogLevel(String logLevel) {
        if (log.isDebugEnabled()) {
            log.debug(t("isisfish.params.changeLogLev", simulLogLevel,
                            logLevel));
        }
        this.simulLogLevel = logLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getScriptLogLevel()
     */
    @Override
    public String getScriptLogLevel() {

        if (scriptLogLevel == null) {
            if (propertiesParameters != null) {
                scriptLogLevel = SimulationParameterPropertiesHelper
                        .getScriptLogLevel(propertiesParameters);
            } else {
                scriptLogLevel = "info";
            }
        }

        return scriptLogLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setScriptLogLevel(java.lang.String)
     */
    @Override
    public void setScriptLogLevel(String logLevel) {
        if (log.isDebugEnabled()) {
            log.debug(t("isisfish.params.changeLogLev", scriptLogLevel,
                    logLevel));
        }
        this.scriptLogLevel = logLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#getLibLogLevel()
     */
    @Override
    public String getLibLogLevel() {
        if (libLogLevel == null) {
            if (propertiesParameters != null) {
                libLogLevel = SimulationParameterPropertiesHelper
                        .getLibLogLevel(propertiesParameters);
            } else {
                libLogLevel = "error";
            }
        }

        return libLogLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#setLibLogLevel(java.lang.String)
     */
    @Override
    public void setLibLogLevel(String logLevel) {
        if (log.isDebugEnabled()) {
            log.debug(t("isisfish.params.changeLogLev", libLogLevel, logLevel));
        }
        this.libLogLevel = logLevel;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulErrorLevel()
     */
    @Override
    public boolean isSimulErrorLevel() {
        return "error".equals(getSimulLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulWarnLevel()
     */
    @Override
    public boolean isSimulWarnLevel() {
        return "warn".equals(getSimulLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulInfoLevel()
     */
    @Override
    public boolean isSimulInfoLevel() {
        return "info".equals(getSimulLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isSimulDebugLevel()
     */
    @Override
    public boolean isSimulDebugLevel() {
        return "debug".equals(getSimulLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptErrorLevel()
     */
    @Override
    public boolean isScriptErrorLevel() {
        return "error".equals(getScriptLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptWarnLevel()
     */
    @Override
    public boolean isScriptWarnLevel() {
        return "warn".equals(getScriptLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptInfoLevel()
     */
    @Override
    public boolean isScriptInfoLevel() {
        return "info".equals(getScriptLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isScriptDebugLevel()
     */
    @Override
    public boolean isScriptDebugLevel() {
        return "debug".equals(getScriptLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibErrorLevel()
     */
    @Override
    public boolean isLibErrorLevel() {
        return "error".equals(getLibLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibWarnLevel()
     */
    @Override
    public boolean isLibWarnLevel() {
        return "warn".equals(getLibLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibInfoLevel()
     */
    @Override
    public boolean isLibInfoLevel() {
        return "info".equals(getLibLogLevel());
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#isLibDebugLevel()
     */
    @Override
    public boolean isLibDebugLevel() {
        return "debug".equals(getLibLogLevel());
    }

    /**
     * Permet d'ajouter des parametres directement à partir de leur
     * representation chaine.
     * 
     * A ne pas utiliser normalement, sert uniquement dans les prescripts des
     * simulation des AS.
     * 
     * @param key key
     * @param value value
     * @since 3.4.0.0
     */
    @Override
    public void setProperty(String key, String value) {
        propertiesParameters.setProperty(key, value);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#copy()
     */
    @Override
    public SimulationParameter copy() {
        SimulationParameterImpl result = new SimulationParameterImpl();

        if (propertiesParameters != null) {
            result.propertiesParameters = new SortedProperties();
            // I think that iteration work better than parameters in contructor
            result.propertiesParameters.putAll(propertiesParameters);
        }

        // still needed to be copied, if there is no propertiesParameters
        result.isisFishVersion = getIsisFishVersion();
        result.description = getDescription();
        result.regionName = getRegionName();
        result.numberOfYear = getNumberOfYear();
        result.simulatorName = getSimulatorName();
        result.useCache = getUseCache();
        result.useStatistic = getUseStatistic();
        result.usePreScript = getUsePreScript();
        result.preScript = getPreScript();
        result.generatedPreScript = getGeneratedPreScript();
        result.useSimulationPlan = getUseSimulationPlan();
        result.simulationPlanNumber = getSimulationPlanNumber();
        if (exportNames != null) {
            result.exportNames = new LinkedList<String>(exportNames);
        }
        if (strategies != null) {
            result.strategies = new LinkedList<Strategy>(strategies);
        }
        if (populations != null) {
            result.populations = new LinkedList<Population>(populations);
        }
        if (rules != null) {
            result.rules = new LinkedList<Rule>(rules);
        }
        if (simulationPlans != null) {
            result.simulationPlans = new LinkedList<SimulationPlan>(simulationPlans);
        }
        if (objective != null) {
            result.objective = objective;
        }
        result.useOptimization = getUseOptimization();
        if (optimization != null) {
            result.optimization = optimization;
        }
        if (optimizationExportsObservations != null) {
            result.optimizationExportsObservations = new LinkedHashMap<Export, Observation>(optimizationExportsObservations);
        }
        if (resultEnabled != null) {
            result.resultEnabled = new LinkedList<String>(resultEnabled);
        }
        if (numbers != null) {
            result.numbers = new HashMap<Population, MatrixND>(numbers);
        }
        if (tagValue != null) {
            result.tagValue = new HashMap<String, String>(tagValue);
        }
        result.simulLogLevel = getSimulLogLevel();
        result.scriptLogLevel = getScriptLogLevel();
        result.libLogLevel = getLibLogLevel();
        result.numberOfSensitivitySimulation = getNumberOfSensitivitySimulation();
        if (sensitivityExports != null) {
            result.sensitivityExports = new LinkedList<SensitivityExport>(sensitivityExports);
        }
        result.sensitivityAnalysis = sensitivityAnalysis;

        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#deepCopy()
     */
    @Override
    public SimulationParameter deepCopy() {
        Properties props = toProperties();
        SimulationParameter newInstance = new SimulationParameterImpl();
        newInstance.fromProperties(props);
        return newInstance;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#toString()
     */
    @Override
    public String toString() {
        Properties prop = toProperties();
        return SimulationParameterPropertiesHelper.toString(prop);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#toProperties()
     */
    @Override
    public Properties toProperties() {
        Properties result = new SortedProperties();

        result.setProperty(SimulationParameterPropertiesHelper.ISIS_FISH_VERSION_KEY, getIsisFishVersion());
        result.setProperty(SimulationParameterPropertiesHelper.DESCRIPTION_KEY, getDescription());
        result.setProperty(SimulationParameterPropertiesHelper.REGION_NAME_KEY, getRegionName());
        result.setProperty(SimulationParameterPropertiesHelper.NUMBER_OF_YEAR_KEY, String.valueOf(getNumberOfYear()));
        result.setProperty(SimulationParameterPropertiesHelper.SIMULATOR_NAME_KEY, getSimulatorName());
        result.setProperty(SimulationParameterPropertiesHelper.USE_CACHE_KEY, String.valueOf(getUseCache()));
        result.setProperty(SimulationParameterPropertiesHelper.USE_STATISTIC_KEY, String.valueOf(getUseStatistic()));

        // strategies
        if (strategies != null) {
            String strategyList = "";
            for (Strategy str : getStrategies()) {
                strategyList += str.getName() + SimulationParameterPropertiesHelper.LIST_SEPARATOR;
            }
            result.setProperty(SimulationParameterPropertiesHelper.STRATEGIES_KEY,
                    StringUtil.substring(strategyList, 0, -1));
        } else {
            if (propertiesParameters != null) {
                String strs = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.STRATEGIES_KEY);
                if (strs != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.STRATEGIES_KEY, strs);
                }
            }
        }

        // populations
        if (populations != null) {
            String populationList = "";
            for (Population pop : getPopulations()) {
                populationList += pop.getName() + LIST_SEPARATOR;
                MatrixND number = getNumberOf(pop);
                String numberAsString = String.valueOf(number.toList());
                result.setProperty(
                        SimulationParameterPropertiesHelper.POPULATION_KEY+ DOT + pop.getName()
                                + DOT + SimulationParameterPropertiesHelper.NUMBER_KEY, numberAsString);
            }
            result.setProperty(SimulationParameterPropertiesHelper.POPULATIONS_KEY,
                    StringUtil.substring(populationList, 0, -1));
        } else {
            if (propertiesParameters != null) {
                String pops = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.POPULATIONS_KEY);
                if (pops != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.POPULATIONS_KEY, pops);
                    SimulationParameterPropertiesHelper.copy(propertiesParameters, result, 
                            SimulationParameterPropertiesHelper.POPULATION_KEY + DOT);
                }
            }
        }

        // rules
        if (rules != null) {
            String ruleList = "";
            int ruleIndex = 0;
            for (Rule rule : getRules()) {
                ruleList += RuleStorage.getName(rule) + LIST_SEPARATOR;
                Properties ruleProp = RuleHelper.getRuleAsProperties(
                        ruleIndex++, getRegion().getStorage(), rule);
                result.putAll(ruleProp);
            }
            result.setProperty(SimulationParameterPropertiesHelper.RULES_KEY, ruleList);
        } else {
            if (propertiesParameters != null) {
                String rules = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.RULES_KEY);
                if (rules != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.RULES_KEY, rules);
                    SimulationParameterPropertiesHelper.copy(propertiesParameters, result,
                            SimulationParameterPropertiesHelper.RULE_KEY + DOT);
                }
            }
        }

        // anaylyse plans
        if (simulationPlans != null) {
            String planList = "";
            int planIndex = 0;
            for (SimulationPlan plan : getSimulationPlans()) {
                planList += SimulationPlanStorage.getName(plan) + LIST_SEPARATOR;
                Properties planProp = StorageHelper.getParamsAsProperties(planIndex++,
                        getRegion().getStorage(), plan, SimulationParameterPropertiesHelper.PLAN_KEY);
                result.putAll(planProp);
            }
            result.setProperty(SimulationParameterPropertiesHelper.PLANS_KEY, planList);
        } else {
            if (propertiesParameters != null) {
                String plans = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.PLANS_KEY);
                if (plans != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.PLANS_KEY, plans);
                    SimulationParameterPropertiesHelper.copy(propertiesParameters, result,
                            SimulationParameterPropertiesHelper.PLAN_KEY + DOT);
                }
            }
        }

        // objective
        if (objective != null) {
            String objectiveName = ObjectiveStorage.getName(objective);
            Properties objectiveProp = StorageHelper.getParamsAsProperties(0,
                    getRegion().getStorage(), objective,
                    SimulationParameterPropertiesHelper.OBJECTIVE_KEY);
            result.putAll(objectiveProp);
            result.setProperty(SimulationParameterPropertiesHelper.OBJECTIVE_KEY,
                    objectiveName);
        } else {
            if (propertiesParameters != null) {
                String objective = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.OBJECTIVE_KEY);
                if (objective != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.OBJECTIVE_KEY, objective);
                    SimulationParameterPropertiesHelper.copy(propertiesParameters, result,
                            SimulationParameterPropertiesHelper.OBJECTIVE_KEY + DOT);
                }                    for (String key : propertiesParameters.stringPropertyNames()) {
                        if (key.startsWith(SimulationParameterPropertiesHelper.OBJECTIVE_KEY + DOT)) {
                            result.setProperty(key, propertiesParameters.getProperty(key));
                        }
                    }

            }
        }

        // optimization
        if (optimization != null) {
            String optimizationName = ObjectiveStorage.getName(optimization);
            Properties optimizationProp = StorageHelper.getParamsAsProperties(0,
                    getRegion().getStorage(), optimization,
                    SimulationParameterPropertiesHelper.OPTIMIZATION_KEY);
            result.putAll(optimizationProp);
            result.setProperty(SimulationParameterPropertiesHelper.OPTIMIZATION_KEY,
                    optimizationName);
        } else {
            if (propertiesParameters != null) {
                String optimization = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.OPTIMIZATION_KEY);
                if (optimization != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.OPTIMIZATION_KEY, optimization);
                    SimulationParameterPropertiesHelper.copy(propertiesParameters, result,
                            SimulationParameterPropertiesHelper.OPTIMIZATION_KEY + DOT);
                }
            }
        }

        // optimization export and observations
        if (optimizationExportsObservations != null) {
            String optimizationExportsList = "";
            int optimizationExportIndex = 0;
            ConvertUtilsBean beanUtils = ConverterUtil.getConverter(null);
            for (Map.Entry<Export, Observation> exportObservationEntry : optimizationExportsObservations.entrySet()) {
                Export export = exportObservationEntry.getKey();
                optimizationExportsList += ExportStorage.getName(export) + LIST_SEPARATOR;

                // add in props observation id export index
                Observation observation = exportObservationEntry.getValue();
                if (observation != null) {
                    result.setProperty(
                            SimulationParameterPropertiesHelper.OPTIMIZATION_OBSERVATION_KEY + DOT
                                    + optimizationExportIndex,
                            beanUtils.convert(observation));
                }
                optimizationExportIndex++;
            }
            result.setProperty(SimulationParameterPropertiesHelper.OPTIMIZATION_EXPORTS_KEY,
                    optimizationExportsList);
        } else {
            if (propertiesParameters != null) {
                String optimizationExports = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.OPTIMIZATION_EXPORTS_KEY);
                if (optimizationExports != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.OPTIMIZATION_EXPORTS_KEY,
                            optimizationExports);
                }
                SimulationParameterPropertiesHelper.copy(propertiesParameters, result,
                        SimulationParameterPropertiesHelper.OPTIMIZATION_OBSERVATION_KEY + DOT);
            }
        }
        
        // export names
        String exportList = StringUtils.join(getExportNames(), LIST_SEPARATOR);
        result.setProperty(SimulationParameterPropertiesHelper.EXPORTS_KEY, exportList);

        // number of sensitivity simulation
        result.setProperty(SimulationParameterPropertiesHelper.NUMBER_OF_SENSITIVITY_SIMULATION_KEY,
                String.valueOf(getNumberOfSensitivitySimulation()));

        // analysis name
        if (sensitivityAnalysis != null) {
            String analysisName = SensitivityAnalysisStorage
                    .getName(getSensitivityAnalysis());
            result.setProperty(SimulationParameterPropertiesHelper.SENSITIVITY_ANALYSIS_KEY,
                    analysisName);

            // analysis parameter
            Properties analysisParams = StorageHelper.getParamsAsProperties(0,
                    getRegion().getStorage(), getSensitivityAnalysis(),
                    SimulationParameterPropertiesHelper.SENSITIVITY_KEY);
            result.putAll(analysisParams);
        } else {
            if (propertiesParameters != null) {
                String sensitivityanalysis = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.SENSITIVITY_ANALYSIS_KEY);
                if (sensitivityanalysis != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.SENSITIVITY_ANALYSIS_KEY,
                            sensitivityanalysis);
                    SimulationParameterPropertiesHelper.copy(propertiesParameters, result,
                            SimulationParameterPropertiesHelper.SENSITIVITY_KEY + DOT);
                }
            }
        }

        // sensitivity exports
        if (sensitivityExports != null) {
            String sensitivityExportList = "";
            int sensitivityExportIndex = 0;
            for (SensitivityExport sensitivityExport : getSensitivityExport()) {
                sensitivityExportList += SensitivityExportStorage
                        .getName(sensitivityExport) + LIST_SEPARATOR;
                Properties exportProp = ExportHelper.getSensitivityExportAsProperties(
                        sensitivityExportIndex++, getRegion().getStorage(), sensitivityExport);
                result.putAll(exportProp);
            }
            result.setProperty(SimulationParameterPropertiesHelper.SENSITIVITY_EXPORTS_KEY,
                    sensitivityExportList);
        } else {
            if (propertiesParameters != null) {
                String sensitivityexports = propertiesParameters.getProperty(
                        SimulationParameterPropertiesHelper.SENSITIVITY_EXPORTS_KEY);
                if (sensitivityexports != null) {
                    result.setProperty(SimulationParameterPropertiesHelper.SENSITIVITY_EXPORTS_KEY,
                            sensitivityexports);
                    SimulationParameterPropertiesHelper.copy(propertiesParameters, result,
                            SimulationParameterPropertiesHelper.SENSITIVITY_EXPORTS_KEY + DOT);
                }
            }
        }

        // sensitivity params
        result.setProperty(SimulationParameterPropertiesHelper.SENSITIVITY_ANALYSIS_ONLY_KEEP_FIRST_KEY,
                String.valueOf(isSensitivityAnalysisOnlyKeepFirst()));

        result.setProperty(SimulationParameterPropertiesHelper.GENERATED_PRE_SCRIPT_KEY,
                getGeneratedPreScript());
        result.setProperty(SimulationParameterPropertiesHelper.USE_PRE_SCRIPT_KEY,
                String.valueOf(getUsePreScript()));
        result.setProperty(SimulationParameterPropertiesHelper.PRE_SCRIPT_KEY,
                getPreScript());
        result.setProperty(SimulationParameterPropertiesHelper.USE_SIMULATION_PLAN_KEY,
                String.valueOf(getUseSimulationPlan()));
        result.setProperty(SimulationParameterPropertiesHelper.SIMULATION_PLAN_NUMBER_KEY,
                String.valueOf(getSimulationPlanNumber()));
        result.setProperty(SimulationParameterPropertiesHelper.USE_OPTIMIZATION_KEY,
                String.valueOf(getUseOptimization()));

        String resultList = StringUtils.join(getResultEnabled(),
                SimulationParameterPropertiesHelper.LIST_SEPARATOR);
        result.setProperty(SimulationParameterPropertiesHelper.RESULT_NAMES_KEY, resultList);

        for (Map.Entry<String, String> e : getTagValue().entrySet()) {
            result.setProperty(SimulationParameterPropertiesHelper.TAG_VALUE_KEY+ DOT
                    + e.getKey(), e.getValue());
        }

        result.setProperty(SimulationParameterPropertiesHelper.SIMUL_LOG_LEVEL_KEY,
                getSimulLogLevel());
        result.setProperty(SimulationParameterPropertiesHelper.SCRIPT_LOG_LEVEL_KEY,
                getScriptLogLevel());
        result.setProperty(SimulationParameterPropertiesHelper.LIB_LOG_LEVEL_KEY,
                getLibLogLevel());
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#fromProperties(java.util.Properties)
     */
    @Override
    public void fromProperties(Properties props) {

        // save properties (use full to read again parameter)
        // for exports, or rules....
        this.propertiesParameters = props;

    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationParameter#reloadContextParameters()
     */
    @Override
    public void reloadContextParameters() throws TopiaException {

        // On verifie tout d'abord que l'on ai pas dans une simulation
        // si on y es, on utilise le context static non null du thread local
        // Resoud les lazy exceptions des parametres des regles
        boolean mustClose = false;
        TopiaContext tx = SimulationContext.get().getDB();

        if (tx == null) {
            // not in simulation, create transaction
            tx = getRegion().getStorage().beginTransaction();
            mustClose = true;
        }

        // reload rules
        if (log.isDebugEnabled()) {
            log.debug("Reloading rules");
        }
        for (Rule rule : getRules()) {
            try {
                for (Field field : rule.getClass().getFields()) {
                    // le champ ne doit pas être privé
                    if (Modifier.isPublic(field.getModifiers())) {
                        // si le type est topia entity (ou sous classe)
                        if (TopiaEntity.class.isAssignableFrom(field.getType())) {
                            TopiaEntity entity = (TopiaEntity) field.get(rule);
                            // il est possible que les parametres soient null
                            // dans ce cas, il ne faut pas essayer de les recharger
                            if (entity != null) {
                                // reloading
                                TopiaEntity newEntity = tx.findByTopiaId(entity
                                        .getTopiaId());
                                field.set(rule, newEntity);
                            }
                        }
                    }
                }
            } catch (IllegalArgumentException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't refresh rule field", e);
                }
            } catch (IllegalAccessException e) {
                if (log.isErrorEnabled()) {
                    log.error("Can't access rule field", e);
                }
            }
        }

        // si la transaction a été ouverte (pas dans une simulation)
        // on la referme
        if (mustClose) {
            tx.commitTransaction();
            tx.closeContext();
        }
    }
}
