/*
 * #%L
 * IsisFish
 * 
 * $Id: ResultManager.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2007 - 2010 Ifremer, Code Lutin, Cédric Pineau, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator;

import static org.nuiton.i18n.I18n.t;

import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;

import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.datastore.ExportStorage;
import fr.ifremer.isisfish.entities.Observation;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.types.TimeStep;

/**
 * Used to store and retrieve result during simulation. This class manage
 * result listener
 * 
 * Created: 13 nov. 07 11:54:47
 *
 * @author poussin
 * @version $Revision: 3969 $
 *
 * Last update: $Date: 2014-04-17 18:48:13 +0200 (Thu, 17 Apr 2014) $
 * by : $Author: echatellier $
 */
public class ResultManager {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(ResultManager.class);

    protected SimulationContext context = null;
    protected Set<SimulationResultListener> listeners = new LinkedHashSet<SimulationResultListener>();
    protected SimulationResultGetter getter = null;
    
    /** result enabled */
    transient protected Set<String> enabledResult = null;

    /**
     * 
     */
    public ResultManager(SimulationContext context) {
        this.context = context;
    }
    
    /**
     * Don't use this method to add listener, use
     * {@link SimulationContext#addSimulationListener(SimulationListener)}
     * @param l
     */
    protected void addSimulationResultListener(SimulationResultListener l) {
        listeners.add(l);
        if (l instanceof SimulationResultGetter) {
            getter = (SimulationResultGetter) l;
        }
    }
    
    /**
     * Don't use this method to remove listener, use
     * {@link SimulationContext#removeSimulationListener(SimulationListener)}
     * @param l
     */
    protected void removeSimulationResultListener(SimulationResultListener l) {
        listeners.remove(l);
        if (l == getter) {
            getter = null;
            // search new getter
            for (SimulationResultListener i : listeners) {
                if (i instanceof SimulationResultGetter) {
                    getter = (SimulationResultGetter) i;
                    break;
                }
            }
        }
    }
    
    /**
     * Permet de savoir si lorsque l'on ajoutera ce resultat, il sera
     * sauvé ou non.
     */
    public boolean isEnabled(String name) {
         name = name.trim();
         if (enabledResult == null) {
             SimulationParameter param = context.getSimulationStorage().getParameter();
             enabledResult = new HashSet<String>();
             
             Collection<String> resultEnabled = param.getResultEnabled();
             enabledResult.addAll(resultEnabled);
             List<String> exportNames = param.getExportNames();
             if (exportNames != null) {
                 for (String exportName : exportNames) {
                     ExportStorage storage = ExportStorage.getExport(exportName);
                     try {
                         Export export = storage.getNewInstance();
                         for (String resultName : export.getNecessaryResult()) {
                             enabledResult.add(resultName);
                         }
                     } catch (IsisFishException eee) {
                         if (log.isWarnEnabled()) {
                             log.warn(t("Can't instanciate export %1$s", exportName), eee);
                         }
                     }
                 }
             }
             List<Rule> rules = param.getRules();
             if (rules != null) {
                 for (Rule rule : rules) {
                     for (String resultName : rule.getNecessaryResult()) {
                         enabledResult.add(resultName);
                     }
                 }
             }
             List<SimulationPlan> plans = param.getSimulationPlans();
             if (plans != null) {
                 for (SimulationPlan plan : plans) {
                     for (String resultName : plan.getNecessaryResult()) {
                         enabledResult.add(resultName);
                     }
                 }
             }

            // on objective and optimization
            Objective objective = param.getObjective();
            if (objective != null) {
                for (String resultName : objective.getNecessaryResult()) {
                    enabledResult.add(resultName);
                }
            }

            Optimization optimization = param.getOptimization();
            if (optimization != null) {
                for (String resultName : optimization.getNecessaryResult()) {
                    enabledResult.add(resultName);
                }
            }

            // optimization export
            Map<Export, Observation> optimizationExportsObservations = param.getOptimizationExportsObservations();
            if (optimizationExportsObservations != null) {
                for (Export export : optimizationExportsObservations.keySet()) {
                    for (String resultName : export.getNecessaryResult()) {
                        enabledResult.add(resultName);
                    }
                }
            }

            log.info("Enabled result: " + enabledResult);
         }
         boolean result = enabledResult.contains(name);
         return result;
     }
     public void addActiveRule(TimeStep step, Rule rule) throws IsisFishException {
         log.warn("addActiveRule not implemented");
         // FIXME: addActiveRule not implemented
     }
     
     public void addResult(TimeStep step, MatrixND mat) throws IsisFishException{
         addResult(false, step, mat.getName(), mat);
     }

     public void addResult(TimeStep step, Population pop, MatrixND mat) throws IsisFishException{
         addResult(false, step, mat.getName(), pop, mat);
     }

     public void addResult(boolean force, TimeStep step, MatrixND mat) throws IsisFishException{
         addResult(force, step, mat.getName(), mat);
     }

     public void addResult(boolean force, TimeStep step, Population pop, MatrixND mat) throws IsisFishException{
         addResult(force, step, mat.getName(), pop, mat);
     }

     public void addResult(TimeStep step, String name, Population pop, MatrixND mat) throws IsisFishException{
         addResult(false, step, name, pop, mat);
     }

     public void addResult(TimeStep step, String name, MatrixND mat) throws IsisFishException{
         addResult(false, step, name, mat);
     }
     
     public void addResult(boolean force, TimeStep step, String name, Population pop, MatrixND mat) throws IsisFishException{
         // don't call generic addResult, because pop name is added
         if (force || isEnabled(name)) {
             for (SimulationResultListener l : listeners) {
                 l.addResult(context, step, name + " " + pop, mat);
             }
         }
     }

     public void addResult(boolean force, TimeStep step, String name, MatrixND mat) throws IsisFishException{
         if (force || isEnabled(name)) {
             for (SimulationResultListener l : listeners) {
                 l.addResult(context, step, name, mat);
             }
         }
     }

     /**
      * Retourne la matrice stocke pour un pas de temps
      * @param step le pas de temps que l'on souhaite
      * @param pop la population pour lequelle on souhaite le resultat
      * @param name le nom des resultats dont on veut la matrice
      * @return La matrice demandée ou null si aucune matrice ne correspond a
      * la demande.
      */
      public MatrixND getMatrix(TimeStep step, Population pop, String name){
          name += " " + pop;
          return getMatrix(step, name);
      }

      public MatrixND getMatrix(TimeStep step, String name){
          MatrixND result = null;
         if (getter != null) {
             result = getter.getMatrix(context, step, name);
          }
         return result;
      }
      
      /**
       * Retourne une matrice contenant tous les pas de temps.
       * @param pop la population pour lequel on souhaite la matrice
       * @param name le nom des resultats dont on veut une matrice globale.
       */
      public MatrixND getMatrix(Population pop, String name){
          name += " " + pop;
          return getMatrix(name);
      }

      /**
       * Retourne une matrice contenant tous les pas de temps.
       * @param name le nom des resultats dont on veut une matrice globale.
       */
      public MatrixND getMatrix(String name){
          MatrixND result = null;
          if (getter != null) {
              result = getter.getMatrix(context, name);
          }
          return result;
      }
   
}


