/*
 * #%L
 * IsisFish
 * 
 * $Id: LogConsole.java 3969 2014-04-17 16:48:13Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, Code Lutin, Benjamin Poussin, Tony Chemit
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.logging.console;

import static org.nuiton.i18n.I18n.t;

import java.awt.EventQueue;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.io.File;
import java.io.IOException;

import javax.swing.JCheckBox;
import javax.swing.JScrollBar;
import javax.swing.JScrollPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.widget.IconFactory;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.ui.logging.console.LogConsoleUI;

/**
 * this is the console of log
 *
 * @author chemit
 */
public class LogConsole extends LogConsoleUI implements ChangeListener {

    /** serialVersionUID. */
    private static final long serialVersionUID = 3534645407010584861L;

    static private Log log = LogFactory.getLog(LogConsole.class);

    public static final String EDITOR_SIZE_CHANGED_PROPERTY = "editorSize";

    public static final String DISPOSE_CHANGED_PROPERTY = "dispose";

    public static final String RESET_CHANGED_PROPERTY = "reset";

    public static final String TEXT_CHANGED_PROPERTY = "text";

    public static final String LEVEL_FATAL_CHANGED_PROPERTY = "FATAL";

    public static final String LEVEL_ERROR_CHANGED_PROPERTY = "ERROR";

    public static final String LEVEL_WARN_CHANGED_PROPERTY = "WARN";

    public static final String LEVEL_INFO_CHANGED_PROPERTY = "INFO";

    public static final String LEVEL_DEBUG_CHANGED_PROPERTY = "DEBUG";

    public static final String LEVEL_TRACE_CHANGED_PROPERTY = "TRACE";

    /**
     * Create a new log console for the given logFile.
     *
     * @param logFile     the log logFile to display
     * @param smtpServer  server use to send mails
     * @param defaultFrom default from email
     * @param defaultTo   default to email
     * @param title       title of the frame
     * @return the new frame created
     * @throws java.io.IOException      if any problem with reader
     * @throws IllegalArgumentException if any problem before init
     */
    public static LogConsole newConsole(File logFile, String smtpServer, String
            defaultFrom, String defaultTo, String title) throws IOException, IllegalArgumentException {

        if (!logFile.exists()) {
            String message1 = t("could not found log file %1$s", logFile);
            log.warn(message1);
            throw new IllegalArgumentException(message1);
        }

        LogConsoleHandler handler;
        LogConsoleModel model;
        LogConsole ui;

        // create model        
        model = new LogConsoleModel(logFile, 30, smtpServer, defaultFrom, defaultTo);

        // create handler
        handler = new LogConsoleHandler(model);

        long t0 = System.currentTimeMillis();

        // start handler
        handler.start();

        log.info("handler start in " + (System.currentTimeMillis() - t0));

        // create ui
        ui = new LogConsole(handler, title);

        // launchUI ui
        ui.setVisible(true);

        // notify the model to adjust the nbLinesInEditor
        ui.firePropertyChange(EDITOR_SIZE_CHANGED_PROPERTY, null, ui.getContent());

        // obtain the tail of stream
        handler.read(model.nbLines - model.nbLinesInEditor);

        return ui;
    }

    /** ui model */
    protected LogConsoleModel model;

    /** ui updater */
    private UpdateUI uiUpdater;

    protected LogConsole(LogConsoleHandler handler, String title) {

        super();

        this.model = handler.model;

        this.setTitle(title);

        content.setAutoscrolls(false);
        // we do not use scrollPane scrollbars, they are linked to editor conten
        // we add a customized one not linked with the size of the editor content
        // but with the nbLines
        scroll.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        scroll.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_NEVER);
        reset.setIcon(IconFactory.getIcon(IconFactory.RESET));
        sendMail.setIcon(IconFactory.getIcon(IconFactory.MAIL));
        if (sendMail.getIcon() != null) {
            sendMail.setText("");
        }
        if (reset.getIcon() != null) {
            reset.setText("");
        }
        initEvents(handler);

        uiUpdater = new UpdateUI(this);

        handler.setStatusBar(getStatusBar());

        log.info(this);
    }

    @Override
    public void dispose() {
        super.dispose();
        firePropertyChange(DISPOSE_CHANGED_PROPERTY, false, true);
    }

    protected boolean disableScroll = true;

    /**
     * the console listen the change on the model.
     * 
     * TODO Deal with ui data update : use a Timer (or a Worker)
     *
     * @param e change event
     */
    public void stateChanged(ChangeEvent e) {

        // model declares a change

        // refresh ui

        EventQueue.invokeLater(uiUpdater);

    }

    protected void initEvents(final LogConsoleHandler handler) {

        setDefaultCloseOperation(DISPOSE_ON_CLOSE);

        // the handler listens the console properties' changes (dispose,reset,filter,...)
        this.addPropertyChangeListener(handler);

        // the dialog listens changes of model
        model.addChangeListener(this);

        // handler listens mouse wheel actions fro editor
        content.addMouseWheelListener(handler);

        // handler listens scrollbar mouse wheel events
        scrollbar.addMouseWheelListener(handler);

        // handler listens scrollbar adjustments events
        scrollbar.addAdjustmentListener(handler);

        // handler listens scroll components events (only resize event)
        scroll.addComponentListener(new ComponentListener() {
            public void componentResized(ComponentEvent e) {
                // notify model
                firePropertyChange(EDITOR_SIZE_CHANGED_PROPERTY, null, content);
            }

            public void componentMoved(ComponentEvent e) {
                // nothing to do
            }

            public void componentShown(ComponentEvent e) {
                // nothing to do
            }

            public void componentHidden(ComponentEvent e) {
                // nothing to do
            }
        });
        setLevelActionListener(levelFatal, LEVEL_FATAL_CHANGED_PROPERTY);
        setLevelActionListener(levelError, LEVEL_ERROR_CHANGED_PROPERTY);
        setLevelActionListener(levelWarn, LEVEL_WARN_CHANGED_PROPERTY);
        setLevelActionListener(levelInfo, LEVEL_INFO_CHANGED_PROPERTY);
        setLevelActionListener(levelDebug, LEVEL_DEBUG_CHANGED_PROPERTY);
        setLevelActionListener(levelTrace, LEVEL_TRACE_CHANGED_PROPERTY);

        apply.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                firePropertyChange(TEXT_CHANGED_PROPERTY, null, ".*"+message.getText()+".*");
                reset.setEnabled(model.canReset());
            }
        });

        reset.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {

                // reset ui
                message.setText("");
                levelFatal.setSelected(false);
                levelError.setSelected(false);
                levelWarn.setSelected(false);
                levelInfo.setSelected(false);
                levelDebug.setSelected(false);
                levelTrace.setSelected(false);

                firePropertyChange(RESET_CHANGED_PROPERTY, null, true);
                reset.setEnabled(false);
            }
        });

        sendMail.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                handler.openLogMail();
            }
        });
    }

    protected void setLevelActionListener(final JCheckBox button, final String property) {
        button.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                firePropertyChange(property, null, button.isSelected());
                reset.setEnabled(model.canReset());
            }
        });
    }

    protected class UpdateUI implements Runnable {
        private JScrollBar scrollbar;
        protected LogConsole console;
        protected long nbLines;
        protected int increment;

        public UpdateUI(LogConsole logConsole) {
            this.console = logConsole;
            this.scrollbar = logConsole.getScrollbar();

        }

        public void run() {

            content.setText(console.model.getText());

            long nbLines1 = console.model.getNbLines();
            if (nbLines != nbLines1) {
                // the model changes his reader
                //TODO make it works with reader bigger than MAX_INT...                
                // we must adjust scrollbar max value, and uiIncrement
                scrollbar.setMaximum((int) nbLines1 - scrollbar.getModel().getExtent());
                // keep the new value
                nbLines = nbLines1;
            }
            EventQueue.invokeLater(new Runnable() {
                public void run() {
                    if (console.model.isBOF()) {
                        console.getContent().setCaretPosition(0);
                        scrollbar.setValue(0);
                        return;
                    }

                    scrollbar.setValue((int) console.model.getFirstLinePosition());

                }
            });
        }
    }

    public static void IsisStart() {

        try {
            IsisFish.init();
        } catch (Exception e) {
            if (log.isErrorEnabled()) {
                log.error("Can't init isis", e);
            }
        }

    }
}
