/*
 * #%L
 * $Id: FilterableComboBox.java 3632 2012-03-05 10:26:31Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/isis-fish/tags/isis-fish-4.1.1.1/src/main/java/fr/ifremer/isisfish/ui/widget/FilterableComboBox.java $
 * %%
 * Copyright (C) 2012 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.widget;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractListModel;
import javax.swing.ComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.ListDataEvent;
import javax.swing.event.ListDataListener;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jdesktop.swingx.JXTextField;
import org.nuiton.util.Resource;

/**
 * A filterable combobox using an additional text field and reset button.
 * 
 * @author chatellier
 * @version $Revision: 3632 $
 * 
 * Last update : $Date: 2012-03-05 11:26:31 +0100 (Mon, 05 Mar 2012) $
 * By : $Author: echatellier $
 */
public class FilterableComboBox extends JPanel implements ListDataListener {

    private static Log log = LogFactory.getLog(FilterableComboBox.class);

    /** serialVersionUID. */
    private static final long serialVersionUID = -6669320998140538372L;

    protected JComboBox dataBox;
    protected JTextField filterField;
    protected JButton resetButton;

    protected FilterComboBoxModel filterModel;

    protected List<ActionListener> actionListeners;

    protected boolean filtering = false;

    public FilterableComboBox() {
        actionListeners = new ArrayList<ActionListener>();
        buildLayout();
    }

    public FilterableComboBox(ComboBoxModel model) {
        this();
        setModel(model);
    }

    class FilterComboBoxModel extends AbstractListModel implements ComboBoxModel {

        /** serialVersionUID. */
        private static final long serialVersionUID = 6256289874593024153L;

        protected ComboBoxModel model;

        protected List<Integer> realIndexes = new ArrayList<Integer>();

        protected Object selectedItem;

        public FilterComboBoxModel(ComboBoxModel model) {
            this.model = model;
            fireDataChanged();
        }

        public ComboBoxModel getModel() {
            return model;
        }

        @Override
        public int getSize() {
            int size = realIndexes.size();
            return size;
        }

        @Override
        public Object getElementAt(int index) {
            int realIndex = realIndexes.get(index);
            Object element = model.getElementAt(realIndex);
            return element;
        }

        public void fireDataChanged() {
            filtering = true;

            if (log.isDebugEnabled()) {
                log.debug("Refreshing real indexes list");
            }

            // filter list
            realIndexes.clear();
            for (int i = 0; i < model.getSize(); ++i) {
                Object element = model.getElementAt(i);
                if (element.toString().contains(filterField.getText())) {
                    realIndexes.add(i);
                }
            }

            fireIntervalRemoved(this, 0, model.getSize());
            fireIntervalAdded(this, 0, realIndexes.size());
            filtering = false;
        }

        @Override
        public void setSelectedItem(Object anItem) {
            this.selectedItem = anItem;
        }

        @Override
        public Object getSelectedItem() {
            return selectedItem;
        }
    }

    class FilterDocumentListener implements DocumentListener {
        @Override
        public void insertUpdate(DocumentEvent e) {
            updateFilter();
        }

        @Override
        public void removeUpdate(DocumentEvent e) {
            updateFilter();
        }

        @Override
        public void changedUpdate(DocumentEvent e) {
            updateFilter();
        }

        protected void updateFilter() {
            ((FilterComboBoxModel) dataBox.getModel()).fireDataChanged();
            resetButton.setEnabled(!filterField.getText().isEmpty());
        }
    }

    class FilterActionListener implements ActionListener {
        /*
         * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            filterField.setText("");
        }
    }

    class ComboBoxActionListener implements ActionListener {
        /*
         * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
         */
        @Override
        public void actionPerformed(ActionEvent e) {
            if (!filtering) {
                for (ActionListener a : actionListeners) {
                    a.actionPerformed(e);
                }
            }
        }
    }

    /**
     * Build layout.
     */
    private void buildLayout() {
        dataBox = new JComboBox();
        dataBox.addActionListener(new ComboBoxActionListener());
        filterField = new JXTextField(_("isisfish.common.filter"));
        // fix size
        filterField.setPreferredSize(new Dimension(200, 0));
        resetButton = new JButton(Resource.getIcon("/icons/cancel.png"));
        resetButton.addActionListener(new FilterActionListener());
        resetButton.setEnabled(false);
        filterField.getDocument().addDocumentListener(new FilterDocumentListener());

        setLayout(new BorderLayout());
        add(dataBox, BorderLayout.CENTER);

        JPanel filterPanel = new JPanel(new BorderLayout());
        filterPanel.add(filterField, BorderLayout.CENTER);
        filterPanel.add(resetButton, BorderLayout.EAST);

        add(filterPanel, BorderLayout.EAST);
    }

    /**
     * Change model.
     * 
     * @param model model to set
     */
    public void setModel(ComboBoxModel model) {
        model.addListDataListener(this);
        dataBox.setModel(new FilterComboBoxModel(model));
    }

    public ComboBoxModel getModel() {
        return ((FilterComboBoxModel)dataBox.getModel()).getModel();
    }

    public Object getSelectedItem() {
        return dataBox.getSelectedItem();
    }

    public void setSelectedItem(Object object) {
        dataBox.setSelectedItem(object);
    }

    public boolean addActionListener(ActionListener e) {
        return actionListeners.add(e);
    }

    public boolean removeActionListener(ActionListener o) {
        return actionListeners.remove(o);
    }

    /*
     * @see javax.swing.event.ListDataListener#intervalAdded(javax.swing.event.ListDataEvent)
     */
    @Override
    public void intervalAdded(ListDataEvent e) {
        if (!filtering) {
            if (log.isDebugEnabled()) {
                log.debug("intervalAdded : fireDataChanged");
            }
            ((FilterComboBoxModel) dataBox.getModel()).fireDataChanged();
        }
    }

    /*
     * @see javax.swing.event.ListDataListener#intervalRemoved(javax.swing.event.ListDataEvent)
     */
    @Override
    public void intervalRemoved(ListDataEvent e) {
        if (!filtering) {
            if (log.isDebugEnabled()) {
                log.debug("intervalRemoved : fireDataChanged");
            }
            ((FilterComboBoxModel) dataBox.getModel()).fireDataChanged();
        }
    }

    /*
     * @see javax.swing.event.ListDataListener#contentsChanged(javax.swing.event.ListDataEvent)
     */
    @Override
    public void contentsChanged(ListDataEvent e) {
        if (!filtering) {
            if (log.isDebugEnabled()) {
                log.debug("contentsChanged : fireDataChanged");
            }
            ((FilterComboBoxModel) dataBox.getModel()).fireDataChanged();
        }
    }
}
