/*
 * #%L
 * IsisFish
 * 
 * $Id: InProcessSimulatorLauncher.java 3748 2012-08-30 14:07:12Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2010 Ifremer, Code Lutin, Benjamin Poussin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator.launcher;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.rmi.RemoteException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.event.TopiaTransactionEvent;
import org.nuiton.topia.event.TopiaTransactionListener;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.FileUtil;
import org.nuiton.util.ObjectUtil;
import org.nuiton.util.StringUtil;

import fr.ifremer.isisfish.IsisConfig;
import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.aspect.AspectClassLoader;
import fr.ifremer.isisfish.aspect.CacheAspect;
import fr.ifremer.isisfish.aspect.RuleAspect;
import fr.ifremer.isisfish.aspect.TraceAspect;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.datastore.SimulatorStorage;
import fr.ifremer.isisfish.simulator.SimulationContext;
import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.simulator.SimulationException;
import fr.ifremer.isisfish.simulator.SimulationExportResultWrapper;
import fr.ifremer.isisfish.simulator.SimulationListener;
import fr.ifremer.isisfish.simulator.SimulationParameter;
import fr.ifremer.isisfish.simulator.SimulationPreScript;
import fr.ifremer.isisfish.simulator.Simulator;
import fr.ifremer.isisfish.types.TimeStep;
import fr.ifremer.isisfish.types.Month;

/**
 * Fait une simulation dans la meme jvm.
 * 
 * @author poussin
 * @version $Revision: 3748 $
 * 
 * Last update : $Date: 2012-08-30 16:07:12 +0200 (Thu, 30 Aug 2012) $
 * By : $Author: echatellier $
 */
public class InProcessSimulatorLauncher implements SimulatorLauncher {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    private static Log log = LogFactory
            .getLog(InProcessSimulatorLauncher.class);

    protected SimulationStorage simulation;

    /**
     * {@inheritDoc}
     */
    public void simulate(SimulationService simulationService,
            SimulationItem simulationItem) throws RemoteException {

        // get real simulation informations for this launcher
        SimulationControl control = simulationItem.getControl();
        File simulationZip = simulationItem.getSimulationZip();
        String simulationPrescript = simulationItem.getSimulationPrescriptContent();
        
        String id = control.getId();
        if (log.isInfoEnabled()) {
            log.info(_("simulate %s with file %s", id, simulationZip));
        }

        try {

            // remove simulation if already exists
            if (SimulationStorage.localyExists(id)) {
                if (log.isWarnEnabled()) {
                    log.warn("Warning , simulation " + id + " aleady exists");
                    log.warn("Deleting it before doing simulation");
                }
                // storage can be opened (result UI)
                SimulationStorage storage = SimulationStorage.getSimulation(id);
                storage.closeStorage();
                
                FileUtil.deleteRecursively(storage.getFile());
            }

            if (log.isDebugEnabled()) {
                log.debug("Timing : before import zip : " + new java.util.Date());
            }
            simulation = SimulationStorage
                    .importAndRenameZip(simulationZip, id);
            if (log.isDebugEnabled()) {
                log.debug("Timing : after import zip : " + new java.util.Date());
            }
            
            // WARNING: make sure to not open Region before being in simulation context
            // add missing control informations
            SimulationParameter param = simulation.getParameter();
            control.setStep(new TimeStep());
            control.setProgress(0);
            control.setStarted(true);
            
            // replace prescript if specified on method
            if (StringUtils.isNotBlank(simulationPrescript)) {
                param.setUsePreScript(true);
                param.setPreScript(simulationPrescript);
            }

            int lastYear = param.getNumberOfYear();
            int lastDate = lastYear * Month.NUMBER_OF_MONTH;
            control.setProgressMax(lastDate);

            simulation = localSimulate(control, simulation);
        } catch (Exception eee) {
            log.error(_("Can't do simulation %s", id), eee);
            if (simulation != null) {
                simulation.getInformation().setException(eee);
            }
        }
    }

    @Override
    public SimulationStorage getSimulationStorage(
            SimulationService simulationService, SimulationControl control)
            throws RemoteException {

        return simulation;

    }

    @Override
    public void updateControl(SimulationService simulationService,
            SimulationControl control) throws RemoteException {
        // in this case, control is set directly by main thread
    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher#maxSimulationThread()
     */
    public int maxSimulationThread() {
        return IsisFish.config.getSimulatorInMaxThreads();
    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher#getCheckProgressionInterval()
     */
    @Override
    public int getCheckProgressionInterval() {

        // par defaut, pour les in process 5 secondes
        int interval = 1;
        return interval;
    }

    @Override
    public String toString() {
        return _("isisfish.simulator.launcher.inprocess");
    }

    /**
     * Display message both in commons-logging and control text progress.
     * 
     * @param control control
     * @param message message to display
     */
    protected void message(SimulationControl control, String message) {
        // log
        if (log.isInfoEnabled()) {
            log.info(message);
        }
        // control
        if (control != null) {
            control.setText(message);
        }
    }

    /**
     * fait la simulation en local dans un nouveau thread, cela permet
     * pour chaque simulation d'avoir les bons scripts dans le classloader
     * et non pas d'utiliser les scripts d'une autre simulation
     * 
     * @param control le controleur de simulation, peut-etre null si on ne 
     * souhaite pas controler la simulation
     * @param simulation la simulation a faire
     * @return le storage après simulation en locale
     */
    protected SimulationStorage localSimulate(SimulationControl control,
            SimulationStorage simulation) {
        SimThread simThread = new SimThread(control, simulation);
        // add simulation logger, we can't make it before since we need thread name
        // anyway since if accept only log from simThread, no need to init it before
        String simulLogLevel = simulation.getParameter().getSimulLogLevel();
        String scriptLogLevel = simulation.getParameter().getScriptLogLevel();
        String libLogLevel = simulation.getParameter().getLibLogLevel();
        try {
            simulation.addSimulationLogger(simulLogLevel, scriptLogLevel,
                    libLogLevel, simThread.getName());
        } catch (Exception e) {
            if (log.isWarnEnabled()) {
                log.warn(_("isisfish.error.add.logger.simulation", e));
            }
        }
        try {
            simThread.start();
            try {
                simThread.join();
            } catch (InterruptedException eee) {
                if (log.isWarnEnabled()) {
                    log.warn(_("isisfish.error.wait.simThread"), eee);
                }
            }
        } finally {
            // remove simulation logger (no more need since thread is dead)
            simulation.removeSimulationLogger();
        }
        return simulation;
    }

    protected class SimThread extends Thread {
        protected SimulationControl control;
        protected SimulationStorage simulation;

        public SimThread(SimulationControl control, SimulationStorage simulation) {
            super("SimThread " + (control != null ? control.getId() : ""));
            this.control = control;
            this.simulation = simulation;
        }

        @Override
        public void run() {
            simulation = localSimulateSameThread(control, simulation);
        }
    }

    /**
     * Modifie le classloader du thread passé en paramètre. 
     * <p>
     * Sert pour les simulations pour qu'elles puissent trouver les 
     * script, rule et export
     *  
     * @param thread le thread dont on souhaite modifier le classloader ou null
     * @param directory le répertoire qui servira pour le classloader
     * @return le classe loader modifié
     */
    protected AspectClassLoader changeClassLoader(Thread thread, File directory) {
        try {
            URL[] classpath = new URL[] { directory.toURI().toURL(),
            // poussin 20080821 : il semble ne plus trouve les formules,
                    // est-ce mieux avec le compile dir ?
                    IsisFish.config.getCompileDirectory().toURI().toURL() };
            //URL [] classpath = new URL[]{directory.toURL()};
            AspectClassLoader loader = new AspectClassLoader(classpath,
                    IsisFish.class.getClassLoader()); //new URLClassLoader(classpath);
            //AspectClassLoader loader =  new AspectClassLoader(classpath, ClassLoader.getSystemClassLoader());
            //new URLClassLoader(classpath);
            thread.setContextClassLoader(loader);
            log.info("Classloader used for simulation: " + loader + " "
                    + Arrays.toString(loader.getURLs()));
            return loader;
        } catch (MalformedURLException eee) {
            // on leve un runtime, car normalement cette erreur est pratiquement
            // impossible car on creer l'url a partir d'un File ce qui ne pose
            // noralement pas de probleme
            throw new IsisFishRuntimeException(_("isisfish.error.change.classloader", directory), eee);
        }
    }

    /**
     * Cree le simulation context, creer le ClassLoader, met en place les AOP,
     * met a jour des informations sur la simulation et lance la simulation en
     * local
     * 
     * @param control le controleur de simulation, peut-etre null si on ne 
     * souhaite pas controler la simulation
     * @param simulation la simulation a faire
     * @return le storage après simulation en locale
     */
    protected SimulationStorage localSimulateSameThread(
            SimulationControl control, SimulationStorage simulation) {
        SimulationStorage result = null;
        String jvmVersion = System.getProperty("java.runtime.version");
        log.info(SimpleDateFormat.getInstance().format(new java.util.Date())
                + " Java version: " + jvmVersion + " Isis-fish version: "
                + IsisConfig.getVersion());
        long start = System.nanoTime();
        simulation.getInformation().setSimulationStart(new java.util.Date());
        AspectClassLoader classLoader;
        try {
            File rootDirectory = simulation.getDirectory();

            //
            // Creation et initialisation du context de simulation
            //
            SimulationContext context = SimulationContext.get();
            context.setSimulationControl(control != null ? control
                    : new SimulationControl(simulation.getName()));

            // changement de classloader
            // IMPORTANT : must be set AFTER :
            //  - SimulationContext.get();
            //  - context.setSimulationControl()
            classLoader = changeClassLoader(Thread.currentThread(), rootDirectory);
            context.setClassLoader(classLoader);
            
            // this directory is used to change isis-database root directory
            // is simulation context
            context.setScriptDirectory(rootDirectory);
            context.setSimulationStorage(simulation);
            
            // Warning : Rule have to be instanciated after aspect definition
            classLoader.deploy(RuleAspect.class);

            SimulationParameter parameters = simulation.getParameter();
            parameters.setIsisFishVersion(IsisConfig.getVersion());
            // forceReload, save all modification in parameter and reread it
            parameters = simulation.getForceReloadParameter();

            //
            // Activation de l'OAP demandée
            //
            if (parameters.getUseStatistic() || parameters.getUseOptimization()) {
                                
                if (parameters.getUseStatistic()) {
                    message(control, _("isisfish.message.setting.trace.aspects"));
                    classLoader.deploy(TraceAspect.class);
                }
                if (parameters.getUseOptimization()) {
                    message(control, _("isisfish.message.setting.cache.aspects"));
                    classLoader.deploy(CacheAspect.class);
                }
            }

            // recherche du simulateur a utiliser
            String simulatorName = parameters.getSimulatorName();
            SimulatorStorage simulator = SimulatorStorage
                    .getSimulator(simulatorName);
            Simulator simulatorObject = simulator.getNewSimulatorInstance();

            // on se met listener sur tc pour connaitre tous les nouveaux objets
            ObjectCreationListener objectCreationListener = new ObjectCreationListener();
            context.getDB().addTopiaTransactionListener(objectCreationListener);

            // 
            // Ajout des listeners pour la simulation
            // 
            initSimulationListener(context);

            //
            // Lancement du script de simulation selectionné
            //
            message(control, _("isisfish.message.simulation.execution"));

            //
            // Call listener simulation (used per example for prescript)
            //
            context.fireBeforeSimulation();
            simulatorObject.simulate(context);

            //
            // Ajout des nouveaux objets créés durant la simulation
            //            
            message(control, _("isisfish.message.add.objets.simulation"));
            // on ajoute sur le DBResult car pour les exports peut-etre auront
            // nous besoin de ces nouveaux objets, et durant la simulation
            // les resultats sont recuperer dans le DBResult
            TopiaContext add = context.getDbResult();
            for (TopiaEntity e : objectCreationListener.getNewObjects()) {
                log.debug("Add new object: " + e + "(" + e.getClass().getName()
                        + ")");
                add.add(e);
            }
            add.commitTransaction();

            //
            // Call listener simulation (used per example for export)
            //
            context.fireAfterSimulation();

            message(control, _("isisfish.message.simulation.ended"));

            simulation.getInformation().setSimulationEnd(new java.util.Date());

            // la simulation est termine on avance la progress au dernier cran
            // attention on utilise ca aussi pour detecter la fin d'une simulation
            // quand date =progressMax
            control.setProgress(control.getProgress() + 1);

            // suppression des résultats si l'utilisateur a demande à ne conserver
            // que les resultats de seulement la première simulation d'une AS
            if (parameters.isSensitivityAnalysisOnlyKeepFirst() && !control.getId().endsWith("_0")) {
                context.getDbResult().clear(true);
            }

        } catch (OutOfMemoryError eee) {
            log.error(_("isisfish.error.during.simulation"), eee);
            simulation.getInformation().setException(eee);
            throw new SimulationException(_("isisfish.error.out.memory"), eee);
        } catch (Exception eee) {
            log.error(_("isisfish.error.during.simulation"), eee);
            simulation.getInformation().setException(eee);
            throw new SimulationException(
                    _("isisfish.error.during.simulation"), eee);
        } finally {
            //
            // Finaly close all TopiaContext used during simulation
            //

            SimulationContext context = SimulationContext.get();

            context.closeDB();
            context.closeDBResult();

            try {
                // close all transaction 
                if (context.getSimulationStorage() != null) {
                    context.getSimulationStorage().closeMemStorage();
                    context.getSimulationStorage().closeStorage();
                }
            } catch (TopiaException eee) {
                if (log.isWarnEnabled()) {
                    log.warn("Can't close all transaction", eee);
                }
            }

            //
            // Affichage des statistiques
            //
            long end = System.nanoTime();
            log.info("Simulation time: "
                    + DurationFormatUtils.formatDuration(
                            (end - start) / 1000000, "s'.'S"));
            SimulationParameter param = simulation.getParameter();
            if (param.getUseStatistic()) {
                String trace = context.getTrace().printStatisticAndClear();
                simulation.getInformation().setStatistic(trace);
            }
            if (param.getUseOptimization()) {
                String cache = context.getCache().printStatistiqueAndClear();
                simulation.getInformation().setOptimizationUsage(cache);
            }

            // cleanup specific context build directory
            File simulationBuildDirectory = IsisFish.config.getCompileDirectory();
            if (log.isDebugEnabled()) {
                log.debug("Delete simulation build directory : " + simulationBuildDirectory.getAbsolutePath());
            }
            FileUtil.deleteRecursively(simulationBuildDirectory);

            // context is used in TraceAspect.printStatistiqueAndClear()
            SimulationContext.remove();
        }
        return result;
    }

    protected void initSimulationListener(SimulationContext context)
            throws Exception {
        SimulationStorage simulation = context.getSimulationStorage();
        // enregistrement des listeners de resultat
        // - ResultStorage
        context.addSimulationListener(simulation.getResultStorage());

        // - TODO: mexico xml result
        // - TODO: vle result
        String simListener = context.getSimulationStorage().getParameter()
                .getTagValue().get("SimulationListener");
        if (simListener != null) {
            String[] simListeners = StringUtil.split(simListener, ",");
            for (String l : simListeners) {
                context.addSimulationListener((SimulationListener) ObjectUtil
                        .create(l));
            }
        }

        // enregistrement des listeners de simulation
        // - prescript (before simulation)
        // - export (after simulation)
        context.addSimulationListener(new SimulationPreScript());
        // si le simulateur est de type SimulationStep il faut ajouter les regles 
        context.addSimulationListener(new SimulationExportResultWrapper());
    }

    protected class ObjectCreationListener implements TopiaTransactionListener {

        /** List qui contient tous les objets creer durant la simulation */
        protected List<TopiaEntity> newObjects = new ArrayList<TopiaEntity>();

        /**
         * @return Returns the newObjects.
         */
        public List<TopiaEntity> getNewObjects() {
            return this.newObjects;
        }

        /*
         * @see org.nuiton.topia.event.TopiaTransactionListener#commit(org.nuiton.topia.event.TopiaTransactionEvent)
         */
        public void commit(TopiaTransactionEvent event) {
            // rien a faire, car normalement toujours rollback en fin de mois

        }

        /* (non-Javadoc)
         * @see org.nuiton.topia.event.TopiaTransactionListener#rollback(org.nuiton.topia.event.TopiaTransactionEvent)
         */
        public void rollback(TopiaTransactionEvent event) {
            log.debug("Transaction rollback " + event.getEntities().size()
                    + " object(s)");
            // FIXME le jour ou on aura l'isolation on pourra directement
            // ajouter dans un autre TopiaContext les objets ajouté durant la
            // simulation de cette maniere les objets creer au pas de temps
            // N seront dispo pour etre utilisé au pas de temps N+1
            // Sinon une autre methode est de faire cette ajout
            // dans l'event rollback qui est leve a la fin de chaque pas de temps
            for (TopiaEntity entity : event.getEntities()) {
                if (event.isCreate(entity)) {
                    log.debug("New object detected during simulation: "
                            + entity + "(" + entity.getClass().getName() + ")");
                    newObjects.add(entity);
                }
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * Do nothing (no restriction on inprocess launcher).
     */
    @Override
    public void simulationStopRequest(SimulationJob job) {

    }

}
