/*
 * #%L
 * IsisFish
 * 
 * $Id$
 * $HeadURL$
 * %%
 * Copyright (C) 2009 - 2011 Ifremer, CodeLutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.mexico.xml;

import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.StringEscapeUtils;
import org.dom4j.Element;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;

import fr.ifremer.isisfish.mexico.MexicoHelper;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.sensitivity.DesignPlan;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.simulator.sensitivity.FactorGroup;
import fr.ifremer.isisfish.simulator.sensitivity.domain.ContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.DiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationDiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.MatrixContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.RuleDiscreteDomain;

/**
 * Parse xml using dom.
 * 
 * @see Element
 *
 * @author chatellier
 * @version $Revision: 1.0 $
 * 
 * Last update : $Date: 13 mars 2009 $
 * By : $Author: chatellier $
 */
public class DomXMLParser {

    public static final String DISCRETE = "discrete";
    public static final String ENUMERATION = "enumeration";
    public static final String DOMAIN = "domain";
    public static final String FIXED = "fixed";
    public static final String VALUE = "value";
    public static final String RULE = "rule";
    public static final String RULES = "rules";
    public static final String VERSION = "version";
    public static final String NAME = "name";
    public static final String TYPE = "type";
    public static final String PROPERTY = "property";
    public static final String TARGET = "target";
    public static final String CONTINUOUS = "continuous";
    public static final String MATRIXCONTINUOUS = "matrixcontinuous";
    public static final String COEFFICIENT = "coefficient";
    public static final String EQUATIONCONTINUOUS = "equationcontinuous";
    public static final String EQUATION = "equation";
    public static final String VARIABLE = "variable";
    public static final String PERCENTAGE = "percentage";
    public static final String REFERENCE = "reference";
    public static final String RANGE = "range";
    public static final String MIN = "min";
    public static final String MAX = "max";
    public static final String CARDINALITY = "cardinality";
    public static final String INTEGER = "integer";
    public static final String MX = "mx";

    /**
     * Parse element root node as Design plan.
     * 
     * @param rootElement root dom element
     * @param topiaContext database context
     * @return a {@link DesignPlan}
     */
    public static DesignPlan parseDesignPlan(Element rootElement, TopiaContext topiaContext) {

        DesignPlan plan = new DesignPlan();

        // could return "2" or null
        String version = rootElement.attributeValue(VERSION);

        List<Element> factorGroupElements = rootElement.selectNodes("child::factors");
        FactorGroup factorGroup = null;
        if (!factorGroupElements.isEmpty()) {
            if ("2".equals(version)) {
                factorGroup = parseFactorGroupV2(factorGroupElements.get(0), topiaContext);
            }
            else {
                factorGroup = parseFactorGroup(factorGroupElements.get(0), topiaContext);
            }
        }
        plan.setFactorGroup(factorGroup);
        
        return plan;
    }

    /**
     * Recursive parse of factor group elements (&gt;factors&lt;).
     * 
     * @param fgElement factor group element (&gt;factors&lt;)
     * @param topiaContext context
     * @return factor group
     *
     * @deprecated since 4.0.0.0, this parsing method parse experimentalDesign
     *      in version "0" or "null" version, don't remove for
     *      data reading purpose, but could be removed in a future version
     */
    @Deprecated
    protected static FactorGroup parseFactorGroup(Element fgElement, TopiaContext topiaContext) {

        String factorGroupName = fgElement.attributeValue(NAME);
        String factorGroupType = fgElement.attributeValue("type");
        boolean continuous = "continuous".equalsIgnoreCase(factorGroupType);
        FactorGroup factorGroup = new FactorGroup(factorGroupName, continuous);

        // sub factor group
        List<Element> factorGroupElements = fgElement.selectNodes("child::factors");
        for (Element factorGroupElement : factorGroupElements) {
            FactorGroup subFactorGroup = parseFactorGroup(factorGroupElement, topiaContext);
            factorGroup.addFactor(subFactorGroup);
        }
        
        // normal factors
        List<Element> factorElements = fgElement.selectNodes("child::factor");

        for (Element factorElement : factorElements) {
            String type = factorElement.attributeValue(TYPE);
            String name = factorElement.attributeValue(NAME);
            String property = factorElement.attributeValue(PROPERTY);
            String path = factorElement.element(TARGET).getText().trim();

            // double
            if ("real".equals(type)) {
                Factor factor = new Factor(name);
                factor.setPath(path);
                Element fixedElement = factorElement.element(DOMAIN).element(FIXED);
                // tous les domaines continues
                if (property.endsWith(CONTINUOUS)) {

                    ContinuousDomain domain;
                    if (property.equals(MATRIXCONTINUOUS)) {
                        // matrix specific
                        // <coefficient operator="-" value="0.799"/>
                        MatrixContinuousDomain mdomain = new MatrixContinuousDomain(true);
                        Element coefficientElement = fixedElement.element(COEFFICIENT);
                        mdomain.setCoefficient(Double.valueOf(coefficientElement.attributeValue(VALUE)));

                        // <mx name="test1" step="0">...
                        Element matrixElement = fixedElement.element(MX);
                        MatrixND matrix = MexicoHelper.getMatrixFromXml(matrixElement, topiaContext);
                        mdomain.setReferenceValue(matrix);

                        domain = mdomain;
                    }
                    else if (property.equals(EQUATIONCONTINUOUS)) {
                        // equation specific
                        EquationContinuousDomain edomain = new EquationContinuousDomain(true);
                        Element coefficientElement = fixedElement.element(COEFFICIENT);
                        edomain.setCoefficient(Double.valueOf(coefficientElement.attributeValue(VALUE)));

                        Element equationElement = fixedElement.element(EQUATION);
                        edomain.setReferenceValue(Double.valueOf(equationElement.attributeValue(REFERENCE)));
                        edomain.setVariableName(equationElement.attributeValue(VARIABLE));

                        domain = edomain;
                    }
                    else {
                        // continous domain
                        domain = new ContinuousDomain();

                        Element percentageElement = fixedElement.element(PERCENTAGE);
                        if (percentageElement != null) {
                            domain.setCoefficient(Double.valueOf(percentageElement.attributeValue(COEFFICIENT)));
                            domain.setReferenceValue(Double.valueOf(percentageElement.attributeValue(REFERENCE)));
                            domain.setPercentageType(true);
                        }
                        else {
                            // <range max="1.0" min="0.0"/>
                            Element rangeElement = fixedElement.element(RANGE);
                            domain.setMinBound(Double.valueOf(rangeElement.attributeValue(MIN)));
                            domain.setMaxBound(Double.valueOf(rangeElement.attributeValue(MAX)));
                        }
                    }

                    factor.setCardinality(Integer.valueOf(fixedElement.attributeValue(CARDINALITY)));

                    factor.setDomain(domain);
                } else if (DISCRETE.equals(property)) {
                    DiscreteDomain domain = new DiscreteDomain();
                    List<Element> valueElements = fixedElement.element(
                            ENUMERATION).elements(VALUE);
                    int label = 0;
                    for (Element valueElement : valueElements) {
                        domain.getValues().put(label,
                                Double.valueOf(valueElement.getTextTrim()));
                        ++label;
                    }
                    factor.setDomain(domain);
                }
                factorGroup.addFactor(factor);
            } else if (INTEGER.equals(type)) {
                Factor factor = new Factor(name);
                factor.setPath(path);
                Element fixedElement = factorElement.element(DOMAIN).element(FIXED);
                if (CONTINUOUS.equals(property)) {
                    ContinuousDomain domain;

                    if(property.equals(MATRIXCONTINUOUS)) {
                        // matrix specific
                        MatrixContinuousDomain mdomain = new MatrixContinuousDomain(true);

                        Element coefficientElement = fixedElement.element(COEFFICIENT);
                        mdomain.setCoefficient(Double.valueOf(coefficientElement.attributeValue(VALUE)));

                        // matrix specific
                        Element matrixElement = fixedElement.element(MX);
                        MatrixND matrix = MexicoHelper.getMatrixFromXml(matrixElement, topiaContext);
                        mdomain.setReferenceValue(matrix);

                        domain = mdomain;
                    }
                    else if (property.equals(EQUATIONCONTINUOUS)) {
                        // equation specific
                        
                        EquationContinuousDomain edomain = new EquationContinuousDomain(true);
                        
                        Element coefficientElement = fixedElement.element(COEFFICIENT);
                        edomain.setCoefficient(Double.valueOf(coefficientElement.attributeValue(VALUE)));
                        
                        Element equationElement = fixedElement.element(EQUATION);
                        edomain.setReferenceValue(Double.valueOf(equationElement.attributeValue(REFERENCE)));
                        edomain.setVariableName(equationElement.attributeValue(VARIABLE));

                        domain = edomain;
                    }
                    else {
                        // continous domain
                        domain = new ContinuousDomain();
                        
                        Element percentageElement = fixedElement.element(PERCENTAGE);
                        if (percentageElement != null) {
                            domain.setCoefficient(Double.valueOf(percentageElement.attributeValue(COEFFICIENT)));
                            domain.setReferenceValue(Double.valueOf(percentageElement.attributeValue(REFERENCE)));
                            domain.setPercentageType(true);
                        }
                        else {
                            // <range max="1" min="3"/>
                            Element rangeElement = fixedElement.element(RANGE);
                            domain.setMinBound(Integer.valueOf(rangeElement.attributeValue(MIN)));
                            domain.setMaxBound(Integer.valueOf(rangeElement.attributeValue(MAX)));
                        }
                        
                    }

                    factor.setCardinality(Integer.valueOf(fixedElement.attributeValue(CARDINALITY)));

                    factor.setDomain(domain);
                } else if (DISCRETE.equals(property)) {
                    DiscreteDomain domain = new DiscreteDomain();
                    List<Element> valueElements = fixedElement.element(
                            ENUMERATION).elements(VALUE);
                    int label = 0;
                    for (Element valueElement : valueElements) {
                        domain.getValues().put(label,
                                Integer.valueOf(valueElement.getTextTrim()));
                        ++label;
                    }
                    factor.setDomain(domain);
                }
                factorGroup.addFactor(factor);
            } else if (RULE.equals(type)) {
                Factor factor = new Factor(name);
                factor.setPath(path);
                Element fixedElement = factorElement.element(DOMAIN).element(FIXED);
                if (DISCRETE.equals(property)) {
                    RuleDiscreteDomain domain = new RuleDiscreteDomain();
                    List<Element> valueElements = fixedElement.element(ENUMERATION).elements(VALUE);
                    int label = 0;
                    for (Element valueElement : valueElements) {
                        Element rulesElement = valueElement.element(RULES);
                        List<Rule> rulesValue = MexicoHelper.getRulesFromXml(rulesElement, topiaContext);
                        domain.getValues().put(label, rulesValue);
                        ++label;
                    }
                    factor.setDomain(domain);
                }
                factorGroup.addFactor(factor);
            }
        }

        return factorGroup;
    }
    
    /**
     * Recursive parse of factor group elements (&gt;factors&lt;).
     * 
     * This version handle xml file with min/max and percentage factor
     * in each continuous factors.
     * 
     * @param fgElement factor group element (&gt;factors&lt;)
     * @param topiaContext context
     * @return factor group
     */
    protected static FactorGroup parseFactorGroupV2(Element fgElement, TopiaContext topiaContext) {

        String factorGroupName = fgElement.attributeValue(NAME);
        String factorGroupType = fgElement.attributeValue("type");
        boolean continuous = "continuous".equalsIgnoreCase(factorGroupType);
        FactorGroup factorGroup = new FactorGroup(factorGroupName, continuous);

        // sub factor group
        List<Element> factorGroupElements = fgElement.selectNodes("child::factors");
        for (Element factorGroupElement : factorGroupElements) {
            FactorGroup subFactorGroup = parseFactorGroupV2(factorGroupElement, topiaContext);
            factorGroup.addFactor(subFactorGroup);
        }
        
        // normal factors
        List<Element> factorElements = fgElement.selectNodes("child::factor");

        for (Element factorElement : factorElements) {
            String type = factorElement.attributeValue(TYPE);
            String name = factorElement.attributeValue(NAME);
            String property = factorElement.attributeValue(PROPERTY);
            String path = factorElement.element(TARGET).getText().trim();
            String cardinalityString = factorElement.attributeValue(CARDINALITY);
            Integer cardinality = 0;
            if (StringUtils.isNotEmpty(cardinalityString)) {
                cardinality = Integer.valueOf(cardinalityString);
            }

            // double
            if ("real".equals(type)) {
                Factor factor = new Factor(name);
                factor.setPath(path);
                Element fixedElement = factorElement.element(DOMAIN).element(FIXED);
                // tous les domaines continues
                if (property.endsWith(CONTINUOUS)) {

                    ContinuousDomain domain;
                    if(property.equals(MATRIXCONTINUOUS)) {
                        
                        MatrixContinuousDomain mdomain = new MatrixContinuousDomain();
                        Element referenceElement = fixedElement.element(REFERENCE);
                        if (referenceElement != null) {
                            // matrix specific
                            // <coefficient operator="-" value="0.799"/>
                            mdomain.setCoefficient(Double.valueOf(referenceElement.attributeValue(COEFFICIENT)));
                            // <mx name="test1" step="0">...
                            Element matrixElement = referenceElement.element(MX);
                            MatrixND matrix = MexicoHelper.getMatrixFromXml(matrixElement, topiaContext);
                            mdomain.setReferenceValue(matrix);
                            mdomain.setPercentageType(true);
                        }
                        else {
                            Element rangeElement = fixedElement.element(RANGE);
                            Element minElement = rangeElement.element(MIN);
                            Element maxElement = rangeElement.element(MAX);
                            MatrixND minMatrix = MexicoHelper.getMatrixFromXml(minElement.element(MX), topiaContext);
                            MatrixND maxMatrix = MexicoHelper.getMatrixFromXml(maxElement.element(MX), topiaContext);
                            mdomain.setMinBound(minMatrix);
                            mdomain.setMaxBound(maxMatrix);
                        }

                        domain = mdomain;
                    }
                    else if (property.equals(EQUATIONCONTINUOUS)) {
                        // equation specific
                        EquationContinuousDomain edomain = new EquationContinuousDomain();
                        edomain.setVariableName(fixedElement.attributeValue(VARIABLE));
                        Element referenceElement = fixedElement.element(REFERENCE);
                        if (referenceElement != null) {
                            edomain.setCoefficient(Double.valueOf(referenceElement.attributeValue(COEFFICIENT)));
                            edomain.setReferenceValue(Double.valueOf(referenceElement.getTextTrim()));
                            edomain.setPercentageType(true);
                        }
                        else {
                            Element rangeElement = fixedElement.element(RANGE);
                            Element minElement = rangeElement.element(MIN);
                            Element maxElement = rangeElement.element(MAX);
                            edomain.setMinBound(Double.valueOf(minElement.getTextTrim()));
                            edomain.setMaxBound(Double.valueOf(maxElement.getTextTrim()));
                        }

                        domain = edomain;
                    }
                    else {
                        // continous domain
                        domain = new ContinuousDomain();

                        Element referenceElement = fixedElement.element(REFERENCE);
                        if (referenceElement != null) {
                            domain.setCoefficient(Double.valueOf(referenceElement.attributeValue(COEFFICIENT)));
                            domain.setReferenceValue(Double.valueOf(referenceElement.getTextTrim()));
                            domain.setPercentageType(true);
                        }
                        else {
                            // <range max="1.0" min="0.0"/>
                            Element rangeElement = fixedElement.element(RANGE);
                            Element minElement = rangeElement.element(MIN);
                            Element maxElement = rangeElement.element(MAX);
                            domain.setMinBound(Double.valueOf(minElement.getTextTrim()));
                            domain.setMaxBound(Double.valueOf(maxElement.getTextTrim()));
                        }
                    }

                    factor.setDomain(domain);
                } else if (DISCRETE.equals(property)) {
                    DiscreteDomain domain = new DiscreteDomain();
                    List<Element> valueElements = fixedElement.element(
                            ENUMERATION).elements(VALUE);
                    int label = 0;
                    for (Element valueElement : valueElements) {
                        domain.getValues().put(label,
                                Double.valueOf(valueElement.getTextTrim()));
                        ++label;
                    }
                    factor.setDomain(domain);
                }
                factor.setCardinality(cardinality);
                factorGroup.addFactor(factor);
            } else if (INTEGER.equals(type)) {
                Factor factor = new Factor(name);
                factor.setPath(path);
                Element fixedElement = factorElement.element(DOMAIN).element(FIXED);
                if (CONTINUOUS.equals(property)) {
                    ContinuousDomain domain;

                    if(property.equals(MATRIXCONTINUOUS)) {
                        
                        MatrixContinuousDomain mdomain = new MatrixContinuousDomain();
                        Element referenceElement = fixedElement.element(REFERENCE);
                        if (referenceElement != null) {
                            // matrix specific
                            // <coefficient operator="-" value="0.799"/>
                            mdomain.setCoefficient(Double.valueOf(referenceElement.attributeValue(COEFFICIENT)));
                            // <mx name="test1" step="0">...
                            Element matrixElement = referenceElement.element(MX);
                            MatrixND matrix = MexicoHelper.getMatrixFromXml(matrixElement, topiaContext);
                            mdomain.setReferenceValue(matrix);
                            mdomain.setPercentageType(true);
                        }
                        else {
                            Element rangeElement = fixedElement.element(RANGE);
                            Element minElement = rangeElement.element(MIN);
                            Element maxElement = rangeElement.element(MAX);
                            MatrixND minMatrix = MexicoHelper.getMatrixFromXml(minElement.element(MX), topiaContext);
                            MatrixND maxMatrix = MexicoHelper.getMatrixFromXml(maxElement.element(MX), topiaContext);
                            mdomain.setMinBound(minMatrix);
                            mdomain.setMaxBound(maxMatrix);
                        }

                        domain = mdomain;
                    }
                    else if (property.equals(EQUATIONCONTINUOUS)) {
                        // equation specific
                        EquationContinuousDomain edomain = new EquationContinuousDomain();
                        edomain.setVariableName(fixedElement.attributeValue(VARIABLE));
                        Element referenceElement = fixedElement.element(REFERENCE);
                        if (referenceElement != null) {
                            edomain.setCoefficient(Double.valueOf(referenceElement.attributeValue(COEFFICIENT)));
                            edomain.setReferenceValue(Double.valueOf(referenceElement.getTextTrim()));
                            edomain.setPercentageType(true);
                        }
                        else {
                            Element rangeElement = fixedElement.element(RANGE);
                            Element minElement = rangeElement.element(MIN);
                            Element maxElement = rangeElement.element(MAX);
                            edomain.setMinBound(Double.valueOf(minElement.getTextTrim()));
                            edomain.setMaxBound(Double.valueOf(maxElement.getTextTrim()));
                        }

                        domain = edomain;
                    }
                    else {
                        // continous domain
                        domain = new ContinuousDomain();

                        Element referenceElement = fixedElement.element(REFERENCE);
                        if (referenceElement != null) {
                            domain.setCoefficient(Double.valueOf(referenceElement.attributeValue(COEFFICIENT)));
                            domain.setReferenceValue(Double.valueOf(referenceElement.getTextTrim()));
                            domain.setPercentageType(true);
                        }
                        else {
                            // <range max="1.0" min="0.0"/>
                            Element rangeElement = fixedElement.element(RANGE);
                            Element minElement = rangeElement.element(MIN);
                            Element maxElement = rangeElement.element(MAX);
                            domain.setMinBound(Integer.valueOf(minElement.getTextTrim()));
                            domain.setMaxBound(Integer.valueOf(maxElement.getTextTrim()));
                        }
                    }

                    factor.setDomain(domain);
                } else if (DISCRETE.equals(property)) {
                    DiscreteDomain domain = new DiscreteDomain();
                    List<Element> valueElements = fixedElement.element(
                            ENUMERATION).elements(VALUE);
                    int label = 0;
                    for (Element valueElement : valueElements) {
                        domain.getValues().put(label,
                                Integer.valueOf(valueElement.getTextTrim()));
                        ++label;
                    }
                    factor.setDomain(domain);
                }
                factor.setCardinality(cardinality);
                factorGroup.addFactor(factor);
            } else if (RULE.equals(type)) {
                Factor factor = new Factor(name);
                factor.setPath(path);
                Element fixedElement = factorElement.element(DOMAIN).element(FIXED);
                if (DISCRETE.equals(property)) {
                    RuleDiscreteDomain domain = new RuleDiscreteDomain();
                    List<Element> valueElements = fixedElement.element(ENUMERATION).elements(VALUE);
                    int label = 0;
                    for (Element valueElement : valueElements) {
                        Element rulesElement = valueElement.element(RULES);
                        List<Rule> rulesValue = MexicoHelper.getRulesFromXml(rulesElement, topiaContext);
                        domain.getValues().put(label, rulesValue);
                        ++label;
                    }
                    factor.setDomain(domain);
                }
                factor.setCardinality(cardinality);
                factorGroup.addFactor(factor);
            } else if (EQUATION.equals(type)) {
                Factor factor = new Factor(name);
                factor.setPath(path);
                Element fixedElement = factorElement.element(DOMAIN).element(FIXED);
                if (DISCRETE.equals(property)) {
                    EquationDiscreteDomain domain = new EquationDiscreteDomain();
                    List<Element> valueElements = fixedElement.element(ENUMERATION).elements(VALUE);
                    int label = 0;
                    for (Element valueElement : valueElements) {
                        String content = StringEscapeUtils.unescapeXml(valueElement.getText());
                        domain.getValues().put(label, content);
                        ++label;
                    }
                    factor.setDomain(domain);
                }
                factor.setCardinality(cardinality);
                factorGroup.addFactor(factor);
            } else if ("string".equals(type)) {
                Factor factor = new Factor(name);
                factor.setPath(path);
                Element fixedElement = factorElement.element(DOMAIN).element(FIXED);
                if (DISCRETE.equals(property)) {
                    DiscreteDomain domain = new DiscreteDomain();
                    List<Element> valueElements = fixedElement.element(ENUMERATION).elements(VALUE);
                    int label = 0;
                    for (Element valueElement : valueElements) {
                        String content = valueElement.getText();
                        Object object = MexicoHelper.getObjectFromString(content, topiaContext);
                        domain.getValues().put(label, object);
                        ++label;
                    }
                    factor.setDomain(domain);
                }
                factor.setCardinality(cardinality);
                factorGroup.addFactor(factor);
            }
        }

        return factorGroup;
    }
}
