/*
 * #%L
 * IsisFish
 * 
 * $Id: ResultMappedStorage.java 3756 2012-09-04 08:18:20Z echatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2012 Ifremer, Code Lutin, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.datastore;

import static org.nuiton.i18n.I18n._;

import java.io.File;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.apache.commons.collections.BidiMap;
import org.apache.commons.collections.bidimap.DualHashBidiMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixIterator;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.util.ArrayUtil;
import org.nuiton.util.HashList;

import fr.ifremer.isisfish.IsisFishDAOHelper;
import fr.ifremer.isisfish.IsisFishException;
import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.entities.ActiveRule;
import fr.ifremer.isisfish.entities.ActiveRuleDAO;
import fr.ifremer.isisfish.entities.Population;
import fr.ifremer.isisfish.export.Export;
import fr.ifremer.isisfish.export.SensitivityExport;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.SimulationContext;
import fr.ifremer.isisfish.simulator.SimulationException;
import fr.ifremer.isisfish.simulator.SimulationPlan;
import fr.ifremer.isisfish.simulator.SimulationResultGetter;
import fr.ifremer.isisfish.simulator.SimulationResultListener;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.types.TimeStep;

/**
 * Cette classe permet de conserver des résultats de simulation. Elle permet
 * ensuite de les récupérer.
 * 
 * Created: 31 aout 2012
 *
 * @author Benjamin Poussin <poussin@codelutin.com>
 * @version $Revision: 3756 $
 *
 * Mise a jour: $Date: 2012-09-04 10:18:20 +0200 (Tue, 04 Sep 2012) $
 * par : $Author: echatellier $
 */
public class ResultMappedStorage implements SimulationResultListener,
        SimulationResultGetter, ResultStorage { // ResultStorage

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(ResultMappedStorage.class);

    //static protected MatrixFactory matrixFactory = MatrixFactory.getInstance(DoubleBigMappedVector.class);

    protected SimulationStorage simulation = null;
    protected RandomAccessFile raf;
    protected long offset;
    protected Map<TimeStep, Map<String, ResultMapped>> stepNameResults =
            new TreeMap<TimeStep, Map<String, ResultMapped>>();
    protected Map<String, Map<TimeStep, ResultMapped>> nameStepResults =
            new TreeMap<String, Map<TimeStep, ResultMapped>>();

    /** result enabled */
    transient protected Set<String> enabledResult = null;

    /**
     * Convertie une entite, month, timestep en string et inversement.
     * Entity = "TopiaId:Entity.toString"
     */
    static protected class EntitySemanticsDecorator /*implements SemanticsDecorator*/ {
        static final private String SEP = ":";

        protected TopiaContext tx;
        /** en cle les representation interne (get) en valeur les valeurs decoree (getKey) */
        protected  BidiMap cache = new DualHashBidiMap();

        public EntitySemanticsDecorator() {
        }

        public EntitySemanticsDecorator(TopiaContext tx) {
            this.tx = tx;
        }

        public Object decorate(Object internalValue) {
            Object result = cache.get(internalValue);
            if (result == null && internalValue != null) {
                result = internalValue;
                if (internalValue instanceof String) {
                    if (StringUtils.startsWith((String)internalValue, Month.class.getName())) {
                        String val = StringUtils.substringAfter((String)internalValue, SEP);
                        int monthNumber = Integer.parseInt(val);
                        result = new Month(monthNumber);
                    } else if (StringUtils.startsWith((String)internalValue, TimeStep.class.getName())) {
                        String val = StringUtils.substringAfter((String)internalValue, SEP);
                        int stepNumber = Integer.parseInt(val);
                        result = new TimeStep(stepNumber);
                    } else if (StringUtils.startsWith((String)internalValue, "fr.ifremer.isisfish.entities.")) {
                        if (tx == null) {
                            result = StringUtils.substringAfter((String)internalValue, SEP);
                        } else {
                            try {
                                String id = StringUtils.substringBefore((String)internalValue, SEP);
                                result = tx.findByTopiaId(id);
                            } catch (TopiaException eee) {
                                log.info("Fallback use string representation because"
                                        + " i can't decorate (String->Entity): "
                                        + internalValue, eee);
                                // si on arrive pas a convertir cette fois-ci, on
                                // renvoi internalValue pour que le undecorate est toutes les infos
                                // et donc que la prochaine fois on y arrive peut etre
                            }
                        }
                    }
                }
                cache.put(internalValue, result);
            }
            return result;
        }

        public Object undecorate(Object decoratedValue) {
            Object result = cache.getKey(decoratedValue);
            if (result == null && decoratedValue != null) {
                if (decoratedValue instanceof Month) {
                    result = Month.class.getName() + SEP + ((Month)decoratedValue).getMonthNumber();
                } else if (decoratedValue instanceof TimeStep) {
                    result = TimeStep.class.getName() + SEP + ((TimeStep)decoratedValue).getStep();
                } else if (decoratedValue instanceof TopiaEntity) {
                    result = ((TopiaEntity)decoratedValue).getTopiaId() + SEP + decoratedValue;
                } else {
                    result = String.valueOf(decoratedValue);
                }
                cache.put(result, decoratedValue);
            }
            return result;
        }

    }

    /**
     * Represente un resultat
     * Lors de la construction, si on ne passe que le RandomAccessFile et l'offset, on relie un resultat
     * si on passe toutes les infos, on ecrit le resultat.
     * Les deux operations doivent etre symetrique on lit et ecrit les meme informations
     */
    static protected class ResultMapped {
        protected RandomAccessFile raf;
        protected long offset;
        protected long size;
        protected TimeStep step;
        protected String name;
        protected MatrixND matrix;

        /** read data from file */
        public ResultMapped(RandomAccessFile raf, long offset) throws IOException {
            this.raf = raf;
            this.offset = offset;

            int stepValue = raf.readInt();
            step = new TimeStep(stepValue);

            name = raf.readUTF();

            int dimSize = raf.readInt();

            String[] dimNames = new String[dimSize];
            for (int i=0; i<dimSize; i++) {
                dimNames[i] = raf.readUTF();
            }

            int[] dims = new int[dimSize];
            for (int i=0; i<dimSize; i++) {
                dims[i] = raf.readInt();
            }

            List[] sems = new List[dimSize];
            for (int i=0; i<dimSize; i++) {
                sems[i] = new ArrayList();
                for (int j=0; j<dims[i]; j++) {
                    String s = raf.readUTF();
                    sems[i].add(s);
                }
            }

            long size = raf.getFilePointer();
            long dataOffset = offset + size;

            int dataSize = raf.readInt(); // en mettant un int on est limite a 2Go x 8 (double) = 16Go par matrice
            //DoubleBigMappedVector data = new DoubleBigMappedVector(raf, dataOffset, dataSize);
            //this.matrix = matrixFactory.create(name, sems, dimNames, data);

            size += dataSize * 8; /* un double est sur 8 bytes*/
            this.size = size;
        }

        /** write date to file */
        public ResultMapped(RandomAccessFile raf, long offset,
                TimeStep step, String name, MatrixND matrix) throws IOException {
            this.raf = raf;
            this.offset = offset;
            this.step = step;
            this.name = name;

            String[] dimNames = matrix.getDimensionNames();
            int[] dims = matrix.getDim();
            List[] sems = matrix.getSemantics();
            int dataSize = 0; //MatrixHelper.getVectorSize(dims);  // en mettant un int on est limite a 2Go x 8 (double) = 16Go par matrice

            raf.write(step.getStep()); // ecriture du pas de temps
            raf.writeUTF(name);        // ecriture du nom du resultat
            raf.write(dims.length);    // ecriture de la taille du tableau de dimension et semantique

            for (String s : dimNames) {
                raf.writeUTF(s); // ecriture du nom de chaque dimensions
            }
            for (int d : dims) {
                raf.write(d); // ecriture de chaque dimension
            }

            // conversion et enregistrement des semantiques
            //SemanticsDecorator deco = new EntitySemanticsDecorator();
            for (int i=0; i<sems.length; i ++) {
                List l = sems[i];
                List undecorate = new ArrayList(l.size());
                sems[i] = undecorate;
                for (Object o : l) {
                    //o = deco.undecorate(o);
                    undecorate.add(o);
                    raf.writeUTF(o.toString()); // ecriture de chaque dimension
                }
            }

            raf.writeInt(dataSize);

            // on prend la position apres l'ecriture de l'entete
            long size = raf.getFilePointer();

            long dataOffset = offset + size;
            // on cree la nouvelle matrice comme il faut (semantique non decore)
            //DoubleBigMappedVector data = new DoubleBigMappedVector(raf, dataOffset, dataSize);
            //this.matrix = MatrixFactory.getInstance().create(name, sems, dimNames, data);
            // et on met les valeurs de l'ancienne dans la nouvelle
            this.matrix.paste(matrix);

            size += dataSize * 8; /* un double est sur 8 bytes*/
            this.size = size;
        }

        /**
         * retourne la taille en nombre de byte
         * @return
         */
        public long size() {
            return size;
        }

        public TimeStep getStep() {
            return step;
        }

        public String getName() {
            return name;
        }

        public MatrixND getMatrix() {
            return matrix;
        }
        
        public MatrixND getMatrix(TopiaContext tx) {
            // on met la matrice dans un decorateur pour convertir automatiquement les semantiques
            //MatrixND result = new MatrixSemanticsDecorator(matrix, new EntitySemanticsDecorator(tx));
            return null;
        }
    }

    /**
     * Les ResultStorage ne doivent pas etre instancier directement, mais
     * recuperer a partir d'un
     * {@link fr.ifremer.isisfish.datastore.SimulationStorage#getResultStorage()}
     *
     * @param simulation storage to get result
     */
    public ResultMappedStorage(SimulationStorage simulation) throws IOException {
        this.simulation = simulation;
        File file = SimulationStorage.getResultFile(simulation.getDirectory());

        // le fichier semble se corrompre à la relecture (ouverture
        // en lecture seule s'il existe déjà)
        if (file.isFile()) {
            raf = new RandomAccessFile(file, "r");
        } else {
            raf = new RandomAccessFile(file, "rw");
        }

        // on lit les donnees deja presente
        offset = 0;
        while (offset < raf.length()) {
            ResultMapped r = new ResultMapped(raf, offset);
            storeResult(r);
            offset += r.size();
        }
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        raf.close();
    }

    /**
     * Methode interne pour que les deux Map soit mise a jour en meme temps
     * C'est la seul methode qui permet d'ajouter des noms de result dans les Map
     * @param r 
     */
    protected void storeResult(ResultMapped r) {
        TimeStep step = r.getStep();
        String name = r.getName();

        getResult(step).put(name, r);
        Map<TimeStep, ResultMapped> stepResult = nameStepResults.get(name);
        if (stepResult == null) {
            stepResult = new TreeMap<TimeStep, ResultMapped>();
            nameStepResults.put(name, stepResult);
        }
        stepResult.put(step, r);
    }

    protected Map<String, ResultMapped> getResult(TimeStep step) {
        Map<String, ResultMapped> result = stepNameResults.get(step);
        if (result == null) {
            result = new TreeMap<String, ResultMapped>();
            stepNameResults.put(step, result);
        }
        return result;
    }

    protected ResultMapped getResult(TimeStep step, String name) {
        ResultMapped result = getResult(step).get(name);
        return result;
    }

    protected ResultMapped getResult(TimeStep step, String name, Population pop) {
        name += " " + pop;
        ResultMapped result = getResult(step, name);
        return result;
    }

    protected Map<TimeStep, ResultMapped> getResult(String name, Population pop) {
        name += " " + pop;
        Map<TimeStep, ResultMapped> result = getResult(name);
        return result;
    }

    protected Map<TimeStep, ResultMapped> getResult(String name) {
        Map<TimeStep, ResultMapped> result = nameStepResults.get(name);
        if (result == null) {
            // on retourne une map vide, pour ne jamais retourne null
            result = Collections.EMPTY_MAP;
        }
        return result;
    }


    public void addResult(TimeStep step, MatrixND mat) throws IsisFishException {
        addResult(false, step, mat.getName(), mat);
    }

    public void addResult(TimeStep step, Population pop, MatrixND mat) throws IsisFishException {
        addResult(false, step, mat.getName(), pop, mat);
    }

    public void addResult(boolean force, TimeStep step, MatrixND mat) throws IsisFishException {
        addResult(force, step, mat.getName(), mat);
    }

    public void addResult(boolean force, TimeStep step, Population pop, MatrixND mat) throws IsisFishException {
        addResult(force, step, mat.getName(), pop, mat);
    }

    public void addResult(TimeStep step, String name, Population pop, MatrixND mat) throws IsisFishException {
        addResult(false, step, name, pop, mat);
    }

    public void addResult(TimeStep step, String name, MatrixND mat) throws IsisFishException {
        addResult(false, step, name, mat);
    }

    public void addResult(boolean force, TimeStep step, String name, Population pop, MatrixND mat) throws IsisFishException {
        if (force || isEnabled(name)) {
            doAddResult(step, name + " " + pop, mat);
        }
    }

    public void addResult(boolean force, TimeStep step, String name, MatrixND mat) throws IsisFishException {
        if (force || isEnabled(name)) {
            doAddResult(step, name, mat);
        }
    }

    protected void doAddResult(TimeStep step, String name, MatrixND mat) throws IsisFishException {
        // si la matrice n'a pas de semantique on refuse
        for (int i = 0; i < mat.getDimCount(); i++) {
            // la semantique n'est pas bonne des qu'il y a un null dedans
            if (mat.getSemantic(i).contains(null)) {
                throw new SimulationException(
                        "Erreur le résultat que vous souhaitez enregistrer n'a pas d'information convenable pour la dimension: "
                                + i + " " + mat.getDimensionName(i));
            }
        }

        try {
            ResultMapped r = new ResultMapped(raf, offset, step, name, mat);
            storeResult(r);
            offset += r.size();
        } catch (IOException eee) {
            log.warn("Can't add result '" + name + "' at step " + step, eee);
        }
    }







    /**
     * Permet de savoir si lorsque l'on ajoutera ce resultat, il sera
     * sauvé ou non.
     * 
     * Check for result name returned by :
     * <ul>
     *  <li>{@link Export#getNecessaryResult()}</li>
     *  <li>{@link SensitivityExport#getNecessaryResult()}</li>
     *  <li>{@link Rule#getNecessaryResult()}</li>
     *  <li>{@link SimulationPlan#getNecessaryResult()}</li>
     * </ul>
     * 
     * @param name result name
     * @return {@code true} if result is enabled
     */
    public boolean isEnabled(String name) {
        name = name.trim();
        if (enabledResult == null) {
            enabledResult = new HashSet<String>();

            Collection<String> resultEnabled = simulation.getParameter()
                    .getResultEnabled();
            enabledResult.addAll(resultEnabled);

            // test on export
            List<String> exportNames = simulation.getParameter().getExportNames();
            if (exportNames != null) {
                for (String exportName : exportNames) {
                    ExportStorage storage = ExportStorage.getExport(exportName);
                    try {
                        Export export = storage.getNewExportInstance();
                        for (String resultName : export.getNecessaryResult()) {
                            enabledResult.add(resultName);
                        }
                    } catch (IsisFishException eee) {
                        if (log.isWarnEnabled()) {
                            log.warn(_("isisfish.error.instanciate.export",
                                            exportName), eee);
                        }
                    }
                }
            }

            // test on sensitivity export
            List<SensitivityExport> sensitivityExports = simulation
                    .getParameter().getSensitivityExport();
            if (sensitivityExports != null) {
                for (SensitivityExport sensitivityExport : sensitivityExports) {
                    for (String resultName : sensitivityExport.getNecessaryResult()) {
                        enabledResult.add(resultName);
                    }
                }
            }

            // test on rules
            List<Rule> rules = simulation.getParameter().getRules();
            if (rules != null) {
                for (Rule rule : rules) {
                    for (String resultName : rule.getNecessaryResult()) {
                        enabledResult.add(resultName);
                    }
                }
            }

            // test on plans
            List<SimulationPlan> plans = simulation.getParameter().getSimulationPlans();
            if (plans != null) {
                for (SimulationPlan plan : plans) {
                    for (String resultName : plan.getNecessaryResult()) {
                        enabledResult.add(resultName);
                    }
                }
            }
            log.info("Enabled result: " + enabledResult);
        }
        boolean result = enabledResult.contains(name);
        return result;
    }


    public void addActiveRule(TimeStep step, Rule rule) throws IsisFishException {
        try {
            TopiaContext tx = null;
            boolean mustClose = false;

            if (simulation == SimulationContext.get().getSimulationStorage()) {
                tx = SimulationContext.get().getDbResult();
            }
            if (tx == null) {
                // not in simulation, create transaction
                tx = simulation.getStorage().beginTransaction();
                mustClose = true;
            }
            ActiveRuleDAO ps = IsisFishDAOHelper.getActiveRuleDAO(tx);
            ActiveRule result = ps.create();
            result.setActiveRuleStep(step);
            result.setName(RuleStorage.getName(rule));
            result.setParam(RuleStorage.getParamAsString(rule));
            ps.update(result);
            if (mustClose) {
                tx.commitTransaction();
                tx.closeContext();
            }
        } catch (TopiaException eee) {
            throw new IsisFishException("Can't add result", eee);
        }
    }

    /**
     * Retourne la liste de tous les résultats. Si le résultat est categorisé
     * par une population alors le nom de la population est automatiquement
     * ajouté au nom du résultat.
     */
    public List<String> getResultName() {
        List<String> result = new ArrayList<String>(nameStepResults.keySet());
        return result;
    }

    /**
     * Retourne la matrice stocke pour un pas de temps
     * @param step le pas de temps que l'on souhaite
     * @param pop la population pour lequelle on souhaite le resultat
     * @param name le nom des resultats dont on veut la matrice
     * @return La matrice demandée ou null si aucune matrice ne correspond a
     * la demande.
     */
    public MatrixND getMatrix(TimeStep step, Population pop, String name) {
        String newName = name + " " + pop;
        return getMatrix(step, newName, null);
    }

    /**
     * Retourne la matrice stocke pour un pas de temps
     * @param step le pas de temps que l'on souhaite
     * @param pop la population pour lequelle on souhaite le resultat
     * @param name le nom des resultats dont on veut la matrice
     * @return La matrice demandée ou null si aucune matrice ne correspond a
     * la demande.
     */
    public MatrixND getMatrix(TimeStep step, Population pop, String name, TopiaContext tx) {
        String newName = name + " " + pop;
        return getMatrix(step, newName, tx);
    }

    public MatrixND getMatrix(TimeStep step, String name) {
        return getMatrix(step, name, null);
    }

    /**
     * Retourne une matrice contenant tous les pas de temps.
     *
     * @param name le nom des resultats dont on veut une matrice globale.
     */
    public MatrixND getMatrix(TimeStep step, String name, TopiaContext tx) {
        ResultMapped r = getResult(step, name);
        MatrixND mat = null;
        if (r != null) {
            tx = getTx(tx);
            mat = r.getMatrix(tx);
        }
        return mat;
    }

    /**
     * Retourne une matrice contenant tous les pas de temps.
     * @param pop la population pour lequel on souhaite la matrice
     * @param name le nom des resultats dont on veut une matrice globale.
     */
    public MatrixND getMatrix(Population pop, String name) {
        String newName = name + " " + pop;
        return getMatrix(newName, null);
    }
    
    /* (non-Javadoc)
     * @see fr.ifremer.isisfish.datastore.ResultStorage#getMatrix(fr.ifremer.isisfish.entities.Population, java.lang.String, org.nuiton.topia.TopiaContext)
     */
    @Override
    public MatrixND getMatrix(Population pop, String name, TopiaContext tx) {
        String newName = name + " " + pop;
        return getMatrix(newName, tx);
    }

    /**
     * Retourne une matrice contenant tous les pas de temps.
     * 
     * @param name le nom des resultats dont on veut une matrice globale.
     */
    public MatrixND getMatrix(String name) {
        return getMatrix(name, null);
    }

    /**
     * Retourne une matrice contenant tous les pas de temps.
     * @param name le nom des resultats dont on veut une matrice globale.
     */
    public MatrixND getMatrix(String name, TopiaContext tx) {
        log.debug("Get result: " + name);

        MatrixND resultMat = null;
        Map<TimeStep, ResultMapped> results = getResult(name);
        // recuperation des resultats qui nous interesse
        if (!results.isEmpty()) {
            // creation de la liste de date
            TimeStep lastStep = getLastStep();
            List<TimeStep> steps = new ArrayList<TimeStep>();
            TimeStep step = new TimeStep(0);
            steps.add(step);
            while (step.before(lastStep)) {
                step = step.next();
                steps.add(step);
            }

            if (log.isTraceEnabled()) {
                log.trace("Steps list : " + steps);
            }


            MatrixND mat = results.values().iterator().next().getMatrix();

            // recuperation des noms des dimensions
            String[] dimNames = new String[1 + mat.getDimCount()];
            dimNames[0] = _("isisfish.common.date");
            for (int i = 1; i < dimNames.length; i++) {
                dimNames[i] = mat.getDimensionName(i - 1);
            }

            // creation de la semantique pour la matrice resultat. +1 pour les dates
            List[] sem = new List[1 + mat.getDimCount()];
            sem[0] = steps;

            for (int i = 1; i < sem.length; i++) {
                sem[i] = new HashList();
            }

            for (ResultMapped result : results.values()) {
                MatrixND mattmp = result.getMatrix();
                if (log.isTraceEnabled()) {
                    log.trace("Ajout de la semantics: "
                            + Arrays.asList(mattmp.getSemantics()));
                }

                for (int s = 0; s < mattmp.getDimCount(); s++) {
                    sem[s + 1].addAll(mattmp.getSemantic(s));
                }
            }

            if (log.isTraceEnabled()) {
                log.trace("La semantique final est: " + Arrays.asList(sem));
            }

            // creation de la matrice resultat
            //resultMat = matrixFactory.create(name, sem, dimNames);

            // recuperation du resultat pour chaque date de la simulation, de Date(0) à lastDate
            for (ResultMapped result : results.values()) {
                TimeStep d = result.getStep();
                mat = result.getMatrix();
                // on met ce resultat dans la matrice result si besoin
                if (mat != null) {
                    // on recupere dans la matrice resultat l'endroit on il faut
                    // mettre la matrice
                    MatrixND submat = resultMat.getSubMatrix(0, d, 1);
                    // on met les valeur de mat dans la sous matrice extraite
                    for (MatrixIterator mi = mat.iterator(); mi.next();) {
                        submat.setValue(
                                ArrayUtil.concat(new Object[] { d },
                                mi.getSemanticsCoordinates()), mi.getValue());
                    }
                }
            }
        }
        if (resultMat != null) {
            // on decore la matrice resultat au dernier moment, tous les calcules
            // ce font avec les strings
            tx = getTx(tx);
            //resultMat = new MatrixSemanticsDecorator(resultMat, new EntitySemanticsDecorator(tx));
        }
        return resultMat;
    }

    /**
     * Get last simulation date.
     * 
     * @return last simulation date
     */
    public TimeStep getLastStep() {
        int monthNumber = simulation.getParameter().getNumberOfYear()
                * Month.NUMBER_OF_MONTH;
        TimeStep result = new TimeStep(monthNumber - 1); // -1 because date begin at 0
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationResultListener#addResult(fr.ifremer.isisfish.simulator.SimulationContext, fr.ifremer.isisfish.types.TimeStep, java.lang.String, org.nuiton.math.matrix.MatrixND)
     */
    @Override
    public void addResult(SimulationContext context, TimeStep step, String name,
            MatrixND mat) throws IsisFishException {
        doAddResult(step, name, mat);
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationResultGetter#getMatrix(fr.ifremer.isisfish.simulator.SimulationContext, fr.ifremer.isisfish.types.TimeStep, java.lang.String)
     */
    @Override
    public MatrixND getMatrix(SimulationContext context, TimeStep step, String name) {
        MatrixND result = null;
        try {
            result = getMatrix(step, name, context.getDB());
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn(String.format("Can't get result: %1$s", name), eee);
            }
        }
        return result;

    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationResultGetter#getMatrix(fr.ifremer.isisfish.simulator.SimulationContext, java.lang.String)
     */
    @Override
    public MatrixND getMatrix(SimulationContext context, String name) {
        MatrixND result = null;
        try {
            result = getMatrix(name, context.getDB());
        } catch (TopiaException eee) {
            if (log.isWarnEnabled()) {
                log.warn(String.format("Can't get result: %1$s", name), eee);
            }
        }
        return result;
    }

    /**
     * Try to find better tx. If argument is not null, return it. otherwize
     * try to get tx in SimulationContext.
     *
     * @param tx
     * @return
     */
    protected TopiaContext getTx(TopiaContext tx) {
        TopiaContext result = tx;
        if (tx == null) {
            // si on a pas de tx, on recherche si on est dans une simulation
            // pour recuperer la tx de la simulation
            if (simulation == SimulationContext.get().getSimulationStorage()) {
                try {
                    result = SimulationContext.get().getDB();
                } catch (TopiaException eee) {
                    throw new IsisFishRuntimeException("Can't get database from SimulationContext", eee);
                }
            }
        }
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationListener#afterSimulation(fr.ifremer.isisfish.simulator.SimulationContext)
     */
    @Override
    public void afterSimulation(SimulationContext context) {
    }

    /*
     * @see fr.ifremer.isisfish.simulator.SimulationListener#beforeSimulation(fr.ifremer.isisfish.simulator.SimulationContext)
     */
    @Override
    public void beforeSimulation(SimulationContext context) {
    }

    // public void addActivatedRule(ResultStorage self, Date date, RegleParam rule){
    //     List rules = (List)activatedRules.get(date);
    //     if(rules == null){
    //         activatedRules.put(date, rules = new LinkedList());
    //     }
    //     rules.add(rule);
    // }

    // /**
    // * Retourne pour une date données tous les RegleParam qui ont été activé
    // * a la date demandé.
    // * @return une list de {@link fr.ifremer.nodb.RegleParam}
    // */
    // public List getActivatedRule(ResultStorage self, Date date){
    //     List rules = (List)activatedRules.get(date);
    //     if(rules == null){
    //         activatedRules.put(date, rules = new LinkedList());
    //     }
    //     return rules;
    // }

} 