/*
 * #%L
 * IsisFish
 * 
 * $Id: SubProcessSimulationLauncher.java 3460 2011-10-06 20:08:09Z chatellier $
 * $HeadURL$
 * %%
 * Copyright (C) 2002 - 2011 Ifremer, Code Lutin, Benjamin Poussin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.simulator.launcher;

import static org.nuiton.i18n.I18n._;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLClassLoader;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.util.FileUtil;

import fr.ifremer.isisfish.IsisFish;
import fr.ifremer.isisfish.datastore.SimulationStorage;
import fr.ifremer.isisfish.simulator.SimulationControl;
import fr.ifremer.isisfish.util.CompileHelper;

/**
 * Lanceur de simulation dans un sous processus.
 * 
 * Usefull article about sub process management :
 * http://www.javaworld.com/javaworld/jw-12-2000/jw-1229-traps.html?page=4
 * 
 * @see ProcessBuilder
 * @see Process
 * 
 * @author poussin
 * @version $Revision: 3460 $
 * 
 * Last update : $Date: 2011-10-06 22:08:09 +0200 (Thu, 06 Oct 2011) $
 * By : $Author: chatellier $
 */
public class SubProcessSimulationLauncher implements SimulatorLauncher {

    /** Class logger (protected for inner classes) */
    protected static Log log = LogFactory
            .getLog(SubProcessSimulationLauncher.class);

    //protected boolean finished = false;

    @Override
    public String toString() {
        return _("isisfish.simulator.launcher.subprocess");
    }

    @Override
    public void simulate(SimulationService simulationService,
        SimulationItem simulationItem) throws RemoteException {

        // get simulation information for this launcher
        SimulationControl control = simulationItem.getControl();
        File simulationZip = simulationItem.getSimulationZip();
        String simulationPrescript = simulationItem.getSimulationPrescriptContent();
            
        String simulationId = control.getId();
        SimulationStorage simulation = null;
        try {
            // simulation = SimulationStorage.importAndRenameZip(
            //     simulationZip, simulationId);
            simulation = subProcessSimulate(control, simulationZip, simulationPrescript);
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error(_("Can't do simulation %s", simulationId), eee);
            }
            // FIXME simulation is always null if exception
            if (simulation != null) {
                simulation.getInformation().setException(eee);
            }
        }

        // FIXME comment not valid
        // on retourne directement le simulation storage passe en argument
        // car la simulation a ete faite avec
        //return simulation;

    }

    @Override
    public int maxSimulationThread() {
        // TODO rendre configurable le nombre de processeurs a utiliser
        int result = Runtime.getRuntime().availableProcessors();
        return result;
    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher#getCheckProgressionInterval()
     */
    @Override
    public int getCheckProgressionInterval() {

        // par defaut, pour les subprocess 5 secondes
        // FIXME rendre configurable

        int interval = 5;
        return interval;
    }

    /**
     * Display both message on UI (listeners and log).
     * 
     * @param control
     * @param message
     */
    protected void message(SimulationControl control, String message) {
        log.info(message);
        if (control != null) {
            control.setText(message);
        }
    }

    public SimulationStorage subProcessSimulate(SimulationControl control,
            File simulationZip, String simulationPrescript) throws Exception {

        message(control, _("isisfish.message.simulation.prepare"));

        String simulationId = control.getId();
        // on ferme le SimulationStorage pour ne pas interferer avec le process
        //simulation.closeStorage();

        // write prescript in a temporary file
        File tempPrescriptFile = null;
        if (!StringUtils.isEmpty(simulationPrescript)) {
            tempPrescriptFile = File.createTempFile("isis", ".prescript");
            tempPrescriptFile.deleteOnExit();
            FileUtil.writeString(tempPrescriptFile, simulationPrescript);
        }
        
        String java = System.getProperty("java.home") + File.separator + "bin"
                + File.separator + "java";
        //String classpath = System.getProperty("java.class.path");

        // TODO make a method to get runtime classpath
        String classpath = CompileHelper.getClassPathAsString(new ArrayList<File>());
        
        if (classpath == null) {
            // could not find the jvm property (jnlp context for example)
            // rebuild the classpath as in {@link AspectClassLoader}
            String pathSep = File.pathSeparator;
            ClassLoader loader = IsisFish.class.getClassLoader();
            if (loader instanceof URLClassLoader) {
                URLClassLoader urlLoader = (URLClassLoader) loader;
                StringBuilder buffer = new StringBuilder();
                for (URL url : urlLoader.getURLs()) {
                    buffer.append(pathSep).append(url.getPath());
                }
                classpath = buffer.substring(1);
            } else {
                //FIXME, should do as in {@link AspectClassLoader}
            }
        }
        
        if (log.isDebugEnabled()) {
            log.debug("classpath to use : " + classpath);
        }

        // common args
        List<String> command = new ArrayList<String>();
        command.add(java);
        command.add("-Xmx1024M");

        // jri args
        String libraryPath = System.getProperty("java.library.path");
        if (StringUtils.isNotBlank(libraryPath)) {
            command.add("-Djava.library.path=\"" + libraryPath + "\"");
        }
        String rType = System.getProperty("R.type");
        if (StringUtils.isNotBlank(rType)) {
            command.add("-DR.type=\"" + rType + "\"");
        }

        // common args
        command.add("-classpath");
        command.add(classpath);
        command.add(IsisFish.class.getName());
        command.add("--option");
        command.add("launch.ui");
        command.add("false");
        
        // prepare le process
        ProcessBuilder processBuilder = null;
        if (tempPrescriptFile != null) {
            command.add("--simulateWithSimulationAndScript");
            command.add(simulationId);
            command.add(simulationZip.getAbsolutePath());
            command.add(tempPrescriptFile.getAbsolutePath());
        }
        else {
            command.add("--simulateWithSimulation");
            command.add(simulationId);
            command.add(simulationZip.getAbsolutePath());
        }

        processBuilder = new ProcessBuilder(command);
        processBuilder.redirectErrorStream(true);

        // demarrage du process
        Process process = processBuilder.start();
        if (log.isInfoEnabled()) {
            log.info(_("SubProcess start: %s %s", process, processBuilder.command()));
        }

        // prepare de thread de surveillance du process si control n'est pas null
        Thread monitor = new SimulationCheckpointExternalProcessThread(control,
                simulationId, process);
        monitor.start();

        // on attend que la simulation soit fini
        int status = process.waitFor();

        if (log.isInfoEnabled()) {
            log.info("SubProcess finished (status = " + status + ")");
        }

        //finished = true;
        SimulationStorage result = SimulationStorage
                .getSimulation(simulationId);
        return result;
    }

    /**
     * This thread is responsible to synchronized SimulationControl used locally with
     * remote simulation control for remote simulation.
     *
     * This thread dead when {@link SimulationControl#isRunning()} is false
     * 
     * @author poussin
     */
    protected class SimulationCheckpointExternalProcessThread extends Thread { // SimulationCheckpointThread

        protected SimulationControl control = null;
        protected String simulationId = null;
        protected Process process = null;
        // on l'appel plutot out que in, car c le output du process lance
        protected InputStream out = null;

        public SimulationCheckpointExternalProcessThread(
                SimulationControl control, String simulationId, Process process) {
            if (log.isInfoEnabled()) {
                log.info("Lancement du thread de surveillance des simulations externes");
            }
            this.control = control;
            this.simulationId = simulationId;
            this.process = process;
            this.out = process.getInputStream();
        }

        @Override
        public void run() {
            InputStreamReader osr = new InputStreamReader(out);
            BufferedReader br = new BufferedReader(osr);

            //int sleepTime = 2000;

            // use tip available at
            // http://www.javaworld.com/javaworld/jw-12-2000/jw-1229-traps.html?page=4
            // for reading subprocess output
            String line = null;
            try {
                while ((line = br.readLine()) != null) {
                    // add reading line to logging output
                    if (log.isInfoEnabled()) {
                        log.info(SubProcessSimulationLauncher.this.toString()
                                + ">" + line);
                    }

                    //Thread.sleep(sleepTime);

                    // on ne lit pas le stop, car le stop ne peut-etre appeler
                    // que par l'utilisateur qui est de ce cote de la machine
                    //SimulationStorage.readControl(simulationId, control, "stop");

                    if (control.isStopSimulationRequest()) {
                        // FIXME, un destroy du process est peut-etre un peu violent ?
                        process.destroy();
                        // passe artificiellement le control a fini
                        control.stopSimulation();
                    }

                    if (!control.isRunning()) {
                        return;
                    }
                }
            } catch (IOException e) {
                if (log.isErrorEnabled()) {
                    log.error( _("isisfish.simulator.subprocess.readoutput.error"),
                                    e);
                }
            }
        }
    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher#getSimulationStorage(fr.ifremer.isisfish.simulator.launcher.SimulationService, fr.ifremer.isisfish.simulator.SimulationControl)
     */
    @Override
    public SimulationStorage getSimulationStorage(
            SimulationService simulationService, SimulationControl control)
            throws RemoteException {

        String simulationId = control.getId();

        // In sub process, simulation is locally stored
        SimulationStorage result = SimulationStorage
                .getSimulation(simulationId);
        return result;

    }

    /*
     * @see fr.ifremer.isisfish.simulator.launcher.SimulatorLauncher#updateControl(fr.ifremer.isisfish.simulator.launcher.SimulationService, fr.ifremer.isisfish.simulator.SimulationControl)
     */
    @Override
    public void updateControl(SimulationService simulationService,
            SimulationControl control) throws RemoteException {

        String simulationId = control.getId();
        // on ne lit pas le stop, car le stop ne peut-etre appeler
        // que par l'utilisateur qui est de ce cote de la machine
        SimulationStorage.readControl(simulationId, control, "stop");

    }

    /**
     * {@inheritDoc}
     * 
     * Do nothing (no restriction on subprocess launcher).
     */
    @Override
    public void simulationStopRequest(SimulationJob job) {
        // TODO kill -9 sub process
    }
}
