/*
 * #%L
 * $Id: SensitivityInputHandler.java 3595 2012-01-19 10:24:08Z echatellier $
 * $HeadURL: http://svn.forge.codelutin.com/svn/isis-fish/tags/isis-fish-4.1.0.0/src/main/java/fr/ifremer/isisfish/ui/sensitivity/SensitivityInputHandler.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin, Chatellier Eric
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 2 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-2.0.html>.
 * #L%
 */

package fr.ifremer.isisfish.ui.sensitivity;

import static org.nuiton.i18n.I18n._;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.SortedMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.text.JTextComponent;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import jaxx.runtime.context.JAXXInitialContext;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.math.matrix.MatrixND;
import org.nuiton.math.matrix.gui.MatrixPanelEditor;
import org.nuiton.topia.TopiaContext;
import org.nuiton.topia.TopiaException;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityContextable;

import fr.ifremer.isisfish.IsisFishRuntimeException;
import fr.ifremer.isisfish.datastore.RegionStorage;
import fr.ifremer.isisfish.entities.Equation;
import fr.ifremer.isisfish.entities.FisheryRegion;
import fr.ifremer.isisfish.rule.Rule;
import fr.ifremer.isisfish.simulator.sensitivity.Domain;
import fr.ifremer.isisfish.simulator.sensitivity.Factor;
import fr.ifremer.isisfish.simulator.sensitivity.FactorGroup;
import fr.ifremer.isisfish.simulator.sensitivity.domain.ContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.DiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.EquationDiscreteDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.MatrixContinuousDomain;
import fr.ifremer.isisfish.simulator.sensitivity.domain.RuleDiscreteDomain;
import fr.ifremer.isisfish.types.Month;
import fr.ifremer.isisfish.types.RangeOfValues;
import fr.ifremer.isisfish.types.TimeStep;
import fr.ifremer.isisfish.types.TimeUnit;
import fr.ifremer.isisfish.ui.SimulationUI;
import fr.ifremer.isisfish.ui.input.InputAction;
import fr.ifremer.isisfish.ui.input.InputContentUI;
import fr.ifremer.isisfish.ui.input.InputHandler;
import fr.ifremer.isisfish.ui.input.InputOneEquationUI;
import fr.ifremer.isisfish.ui.input.tree.FisheryDataProvider;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeHelper;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeNode;
import fr.ifremer.isisfish.ui.input.tree.FisheryTreeRenderer;
import fr.ifremer.isisfish.ui.simulator.RuleChooser;
import fr.ifremer.isisfish.ui.simulator.SimulAction;
import fr.ifremer.isisfish.ui.widget.editor.StepComponent;
import fr.ifremer.isisfish.ui.widget.editor.MonthComponent;

/**
 * Handler for sensitivity tab ui (fishery region factors).
 * 
 * @author chatellier
 * @version $Revision: 3595 $
 * 
 * Last update : $Date: 2012-01-19 11:24:08 +0100 (Thu, 19 Jan 2012) $
 * By : $Author: echatellier $
 */
public class SensitivityInputHandler extends InputHandler {

    /** Class logger. */
    private static Log log = LogFactory.getLog(SensitivityInputHandler.class);

    /**
     * 
     * @param sensitivityTabUI
     */
    public void loadFisheryRegionTree(SensitivityTabUI sensitivityTabUI) {
        FisheryRegion fisheryRegion = sensitivityTabUI.getFisheryRegion();

        if (fisheryRegion == null) {
            // show empty region ui
            sensitivityTabUI.getCardlayoutPrincipal().show(sensitivityTabUI.getInputPanePrincipal(),"none");
            TreeModel model = new DefaultTreeModel(null);
            sensitivityTabUI.getFisheryRegionTree().setModel(model);
        }
        else {
            // init tree model loader with fishery region
            FisheryTreeHelper treeHelper = new FisheryTreeHelper();
            FisheryDataProvider dataProvider = new FisheryDataProvider(fisheryRegion);
            treeHelper.setDataProvider(dataProvider);
            TreeModel model = treeHelper.createTreeModel(fisheryRegion);
            sensitivityTabUI.getFisheryRegionTree().setModel(model);
            sensitivityTabUI.getFisheryRegionTree().setCellRenderer(new FisheryTreeRenderer(dataProvider));
            treeHelper.setUI(sensitivityTabUI.getFisheryRegionTree(), true, false, null);

            // global context value : fisheryRegion, regionStorage, treeHelper
            sensitivityTabUI.setContextValue(fisheryRegion);
            sensitivityTabUI.setContextValue(treeHelper);
            sensitivityTabUI.setContextValue(model);
            sensitivityTabUI.setContextValue(fisheryRegion.getTopiaContext());

            sensitivityTabUI.getCardlayoutPrincipal().show(sensitivityTabUI.getInputPanePrincipal(),"normale");
        }
    }

    /**
     * Changement de selection dans l'arbre de la pecherie.
     * 
     * @param sensitivityTabUI
     * @param event
     */
    public void nodeSelectionChanged(SensitivityTabUI sensitivityTabUI, TreeSelectionEvent event) {

        TreePath newTreePath = event.getNewLeadSelectionPath();

        if (newTreePath != null) {
            Object lastTreePath = newTreePath.getLastPathComponent();
            if (lastTreePath instanceof FisheryTreeNode) {
                FisheryTreeNode isisTreeNode = (FisheryTreeNode)lastTreePath;

                Class<?> internalClass = isisTreeNode.getInternalClass();

                // noeud qui n'en charge pas d'autres (= un bean)
                TopiaEntityContextable topiaEntity = null;
                String topiaId = isisTreeNode.getId();

                try {
                    if (isisTreeNode.isStaticNode()) {
                        FisheryRegion fisheryRegion = sensitivityTabUI.getContextValue(FisheryRegion.class);
                        TopiaContext topiaContext = fisheryRegion.getTopiaContext();
                        topiaEntity = (TopiaEntityContextable)topiaContext.findByTopiaId(topiaId);
                    }

                    InputContentUI inputContentUI = getUIInstanceForBeanClass(internalClass, sensitivityTabUI);

                    // mandatory set
                    inputContentUI.getSaveVerifier().reset(); // before set bean !!!
                    if (topiaEntity != null) {
                        inputContentUI.getSaveVerifier().addCurrentEntity(topiaEntity);
                        inputContentUI.getSaveVerifier().setInputContentUI(inputContentUI);
                    }

                    inputContentUI.setBean((TopiaEntityContextable)topiaEntity);
                    inputContentUI.setActive(topiaEntity != null);
                    inputContentUI.setLayer(true);
                    inputContentUI.setSensitivity(true);

                    // add initialized ui to panel
                    sensitivityTabUI.getCardlayoutPrincipal().show(sensitivityTabUI.getInputPanePrincipal(), "normale");
                    sensitivityTabUI.getInputPane().removeAll();
                    sensitivityTabUI.getInputPane().add(inputContentUI, BorderLayout.CENTER);
                    sensitivityTabUI.getInputPane().repaint();
                    sensitivityTabUI.getInputPane().validate();
                } catch (Exception ex) {
                    throw new IsisFishRuntimeException("Can't display bean " + topiaId, ex);
                }
            }
        }
    }

    /**
     * Add new continuous factor group in factor tree.
     * 
     * @param sensitivityTabUI
     * @param continuous continuous
     */
    public void addNewFactorGroup(SensitivityTabUI sensitivityTabUI, boolean continuous) {
        String factorName = JOptionPane.showInputDialog(sensitivityTabUI, _("isisfish.sensitivity.newfactorname"),
                _("isisfish.sensitivity.title"), JOptionPane.QUESTION_MESSAGE);
        
        if (StringUtils.isNotBlank(factorName)) {
            FactorGroup rootFactorGroup = sensitivityTabUI.getSimulAction().getFactorGroup();
            FactorGroup factorGroup = new FactorGroup(factorName, continuous);
            rootFactorGroup.addFactor(factorGroup);
            sensitivityTabUI.setFactorModel();
        }
    }

    /**
     * Move factors to another factorgroup.
     * 
     * @param sensitivityTabUI
     * @param selectedFactorGroup
     * @param movedFactors
     */
    public void moveFactor(SensitivityTabUI sensitivityTabUI, FactorGroup selectedFactorGroup, List<Factor> movedFactors) {
        try {
            // add all factors, to do first, throw
            // exception if can't be done
            selectedFactorGroup.addAllFactors(movedFactors);

            // remove duplicated from factor group
            FactorGroup rootFactorGroup = sensitivityTabUI.getSimulAction().getFactorGroup();
            if (!rootFactorGroup.equals(selectedFactorGroup)) {
                rootFactorGroup.removeAll(movedFactors);
            }
            for (int index = 0 ; index < rootFactorGroup.size(); ++index) {
                Factor factor = rootFactorGroup.get(index);
                if (factor instanceof FactorGroup) {
                    FactorGroup factorGroup = (FactorGroup)factor;
                    if (!factorGroup.equals(selectedFactorGroup)) {
                        factorGroup.removeAll(movedFactors);
                    }
                }
            }
            sensitivityTabUI.setFactorModel();
        } catch (IllegalArgumentException ex) {
            JOptionPane.showMessageDialog(sensitivityTabUI, _("isisfish.sensitivity.moveillegal"),
                    _("isisfish.sensitivity.title"), JOptionPane.ERROR_MESSAGE);
        }
    }
    
    /**
     * Mouse click on factors tree.
     * 
     * <ul>
     *  <li>normal click : factor edit</li>
     *  <li>right click : popup menu</li>
     * </ul>
     * 
     * @param sensitivityTabUI ui
     * @param e mouse event
     */
    public void factorsTreeMouseClicked(final SensitivityTabUI sensitivityTabUI, MouseEvent e) {
        // clic droit
        if (e.getButton() == MouseEvent.BUTTON3) {
            JPopupMenu menu = new JPopupMenu();
            JMenuItem menuItemDelete = new JMenuItem(_("isisfish.common.delete"));
            menuItemDelete.addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    deleteSelectedFactors(sensitivityTabUI);
                }
            });
            menu.add(menuItemDelete);
            menu.show(e.getComponent(), e.getX(), e.getY());
        }
        else if (e.getButton() == MouseEvent.BUTTON1 && e.getClickCount() == 2) {
            // autre double clic
            factorSelected(sensitivityTabUI);
        }
    }

    /**
     * Factor selection, display modification wizard.
     * 
     * @param sensitivityTabUI
     */
    protected void factorSelected(SensitivityTabUI sensitivityTabUI) {
        // get selected factor
        TreePath selectedPath = sensitivityTabUI.getFactorsTree().getSelectionPath();
        
        // method appelee au clic, donc pas forcement de selection
        if (selectedPath != null) {
            Object[] pathWay = selectedPath.getPath();
            Object selectedObject = pathWay[pathWay.length - 1];
            if (selectedObject != null) {
                if (!(selectedObject instanceof FactorGroup)) {
                    Factor selectedFactor = (Factor)selectedObject;
                    FactorWizardUI factorWizardUI = new FactorWizardUI(sensitivityTabUI);
                    SensitivityInputHandler handler = factorWizardUI.getHandler();
                    handler.initExistingFactor(factorWizardUI, selectedFactor);
                    factorWizardUI.pack();
                    factorWizardUI.setLocationRelativeTo(sensitivityTabUI);
                    factorWizardUI.setVisible(true);
                }
            }
        }
    }

    /**
     * Delete selection factors.
     */
    protected void deleteSelectedFactors(SensitivityTabUI sensitivityTabUI) {
        // get selected factor
        TreePath[] selectedPaths = sensitivityTabUI.getFactorsTree().getSelectionPaths();
        if (!ArrayUtils.isEmpty(selectedPaths)) { // can happen
            for (TreePath selectedPath : selectedPaths) {
                Object[] pathWay = selectedPath.getPath();
                // > 2 : can't delete root
                if (pathWay.length >= 2) {
                    Object selectedObject = pathWay[pathWay.length - 1];
                    if (selectedObject != null) {
                        if (selectedObject instanceof Factor) {
                            Factor selectedFactor = (Factor)selectedObject;
                            FactorGroup selectedFactorGroup = (FactorGroup)pathWay[pathWay.length - 2];
                            if (log.isDebugEnabled()) {
                                log.debug("Deleting factor " + selectedFactor.getName());
                            }
                            selectedFactorGroup.remove(selectedFactor);
                            sensitivityTabUI.setFactorModel();
                        }
                    }
                }
            }
        }
    }

    /**
     * Un init new se fait toujours sur une entité.
     * 
     * On recupere des info sur le type correspondant à la proprieté a mettre
     * en facteur pour savoir le composant d'edition et s'il peut etre continue
     * ou pas.
     * 
     * @param factorWizardUI factorWizardUI
     * @param bean bean in current ui
     * @param property bean property to edit
     */
    public void initNewFactor(FactorWizardUI factorWizardUI, TopiaEntityContextable bean, String property) {

        // path is topiaId#property
        // ex : fwn#fsd#0.3425345#name
        // for JAXX : cOrigine start with upper case
        // for commons beanutils : must be lower case
        String beanProperty = StringUtils.uncapitalize(property);
        String path = bean.getTopiaId() + "#" + beanProperty;
        factorWizardUI.setFactorPath(path);
        factorWizardUI.getFactorNameField().setText(bean.toString() + "." + beanProperty);

        // get value for pointed path
        //TopiaContext topiaContext = factorWizardUI.getContextValue(TopiaContext.class);
        //Class<?> classForPath = getPropertyClass(path, topiaContext);
        // peut etre pas une bonne idée que ce soit basé sur les valeurs
        // au lieu des types (mais pour RangeOfValues, pas evident)
        Object valueForPath = getPropertyValue(bean, beanProperty);
        boolean continuePossible = canBeContinue(valueForPath);
        boolean continueSelected = isContinue(valueForPath);

        // init panel
        if (continuePossible) {
            factorWizardUI.continuousPanel = getContinuousPanel(valueForPath, bean, property);
            factorWizardUI.getContinuousPanelContainer().add(factorWizardUI.continuousPanel, BorderLayout.CENTER);
        }
        
        // after, for binding on continuePossible, continueSelected to work
        factorWizardUI.setContinuePossible(continuePossible);
        factorWizardUI.setContinueSelected(continueSelected);
        factorWizardUI.getContinueRadio().setSelected(continueSelected);
    }

    /**
     * Permet d'intialiser le wizard avec une valeur independante de la
     * provenance (utile pour les facteurs sur les parametres de regles).
     * 
     * TODO chatellier : ne doit pas fonctionner pour des equations.
     * 
     * @param factorWizardUI factorWizardUI
     * @param value value
     */
    public void initNewFactorWithValue(FactorWizardUI factorWizardUI, Object value) {
        boolean continuePossible = canBeContinue(value);
        boolean continueSelected = isContinue(value);

        // init panel
        if (continuePossible) {
            factorWizardUI.continuousPanel = getContinuousPanel(value, null, null);
            factorWizardUI.getContinuousPanelContainer().add(factorWizardUI.continuousPanel, BorderLayout.CENTER);
        }

        // after, for binding on continuePossible, continueSelected to work
        factorWizardUI.setContinuePossible(continuePossible);
        factorWizardUI.setContinueSelected(continueSelected);
        factorWizardUI.getContinueRadio().setSelected(continueSelected);
    }
    
    /**
     * Reaffiche un facteur existant.
     * 
     * @param factorWizardUI
     * @param factor factor
     */
    public void initExistingFactor(FactorWizardUI factorWizardUI, Factor factor) {

        factorWizardUI.setExistingValue(true);
        String factorPath = factor.getPath();
        Domain domain = factor.getDomain();
        String factorName = factor.getName();
        factorWizardUI.setFactorPath(factorPath);
        factorWizardUI.getComment().setText(factor.getComment());

        // dans le cas d'un facteur equation
        // il faut enlever la varible du nom
        // sinon, à la sauvegarde elle sera reajoutée
        if (domain instanceof EquationContinuousDomain) {
            EquationContinuousDomain equationDomain = (EquationContinuousDomain)domain;
            String suffix = equationDomain.getVariableName();
            factorName = StringUtils.removeEnd(factorName, "." + suffix);
            // can be called _xxx after char remplacement for R
            factorName = StringUtils.removeEnd(factorName, "_" + suffix);
        }

        if (domain instanceof ContinuousDomain) {
            
            ContinuousDomain cDomain = (ContinuousDomain) domain;

            TopiaContext topiaContext = factorWizardUI.getContextValue(TopiaContext.class);
            factorWizardUI.continuousPanel = getContinuousPanelWithValue(factor, cDomain, topiaContext);
            factorWizardUI.getContinuousPanelContainer().add(factorWizardUI.continuousPanel, BorderLayout.CENTER);

            factorWizardUI.getContinueRadio().setSelected(true);
            factorWizardUI.setContinueSelected(true);
            factorWizardUI.setContinuePossible(true);
        } else {

            // restaure discrete domain
            DiscreteDomain dDomain = (DiscreteDomain)domain;

            int nb = dDomain.getValues().size();
            factorWizardUI.getDiscretNumber().setText(String.valueOf(nb));
            factorWizardUI.discretComponents.clear();
            factorWizardUI.getTabPane().removeAll();

            SortedMap<Object, Object> values = dDomain.getValues();
            int i = 0;
            for (Object o : values.values()) {
                i++;
                JComponent c = null;
                if (o != null) {
                    c = getEditorWithValue(factorWizardUI, factor, dDomain, o);
                }
                else {
                    if (log.isWarnEnabled()) {
                        log.warn("Null value in factor");
                    }
                }
                factorWizardUI.discretComponents.add(c);
                JScrollPane js = new JScrollPane(c);
                String tabName = _("isisfish.sensitivity.discretevaluelabel", i);
                factorWizardUI.getTabPane().addTab(tabName, js);
            }
            
            // init non selected continous panel
            if (factorPath.indexOf('#') != -1) {
                try {
                    // get initial value in database
                    String topiaId = factorPath.substring(0, factorPath.lastIndexOf('#'));
                    String property = factorPath.substring(factorPath.lastIndexOf('#') + 1);
                    TopiaContext topiaContext = factorWizardUI.getContextValue(TopiaContext.class);
                    TopiaEntityContextable entity = (TopiaEntityContextable)topiaContext.findByTopiaId(topiaId);
                    String getter = "get" + StringUtils.capitalize(property);
                    Method m = entity.getClass().getMethod(getter);
                    Object valueForPath = m.invoke(entity);
                    
                    boolean continuePossible = canBeContinue(valueForPath);
                    factorWizardUI.setContinuePossible(continuePossible);
                    if (continuePossible) {
                        factorWizardUI.continuousPanel = getContinuousPanel(valueForPath, entity, property);
                        factorWizardUI.getContinuousPanelContainer().add(factorWizardUI.continuousPanel, BorderLayout.CENTER);
                    }
                } catch (Exception ex) {
                    throw new IsisFishRuntimeException("Can't get factor path database value", ex);
                }
            }
        }

        factorWizardUI.getFactorNameField().setText(factorName);
    }

    /**
     * Refresh tab for discrete factor numbers.
     * 
     * @param factorWizardUI factorWizardUI
     */
    public void addTabs(FactorWizardUI factorWizardUI) {
        String discreteNumber = factorWizardUI.getDiscretNumber().getText();
        int nbTab = Integer.parseInt(discreteNumber);
        factorWizardUI.discretComponents.clear();
        factorWizardUI.getTabPane().removeAll();
        for (int i = 1; i <= nbTab ; i++) {
            JComponent c = getNewDiscreteComponent(factorWizardUI);
            factorWizardUI.discretComponents.add(c);
            String tabName = _("isisfish.sensitivity.discretevaluelabel", i);
            factorWizardUI.getTabPane().addTab(tabName, c);
        }
        factorWizardUI.pack();
    }

    /**
     * Retourne le type de valeur pointé par le path du facteur.
     * 
     * Il vaut mieux recuperer le type de la propriété a mettre en facteur
     * que la valeur elle meme; elle peut être null et on ne peut pas en
     * deduire sont type.
     * 
     * @param factorPath factor path
     * @param context context
     * @return class
     */
    protected Class<?> getPropertyClass(String factorPath, TopiaContext context) {
        Class<?> result = null;
        if (factorPath.indexOf('#') != -1) {
            String topiaId = factorPath.substring(0, factorPath.lastIndexOf('#'));
            String property = factorPath.substring(factorPath.lastIndexOf('#') + 1);

            try {
                TopiaEntity entity = context.findByTopiaId(topiaId);
                
                if (log.isDebugEnabled()) {
                    log.debug("Factor path " + factorPath + " denoted entity " + entity);
                }
                
                String getter = "get" + StringUtils.capitalize(property);
                Method method = entity.getClass().getMethod(getter);
                result = method.getReturnType();
                
            } catch (Exception ex) {
                if (log.isErrorEnabled()) {
                    log.error("Can't find entity for " + topiaId, ex);
                }
            }
        }
        return result;
    }
    
    /**
     * Get value for fieldName in entity.
     * 
     * @param entity
     * @param fieldName
     * @return method return value
     */
    protected Object getPropertyValue(Object entity, String fieldName) {
        Object result = null;
        try {
            // fieldName maybe be sometime lower case
            String getMethod = "get" + StringUtils.capitalize(fieldName);

            Method m = entity.getClass().getMethod(getMethod);
            result = m.invoke(entity);
        } catch (Exception ex) {
            if (log.isErrorEnabled()) {
                log.error("Can't get entity value", ex);
            }
        }
        return result;
    }

    /**
     * Return if type can be defined as a factor.
     * 
     * @param type type
     * @return {@code true} if type can be defined as a factor
     */
    public boolean canBeFactor(Class type) {
        boolean result = false;
        
        if (TopiaEntity.class.isAssignableFrom(type)) {
            result = true;
        } else if (double.class.isAssignableFrom(type)) {
            result = true;
        } else if (Number.class.isAssignableFrom(type)) {
            result = true;
        } else if (TimeStep.class.isAssignableFrom(type)) {
            result = true;
        } else if (Month.class.isAssignableFrom(type)) {
            result = true;
        }
        
        return result;
    }

    /**
     * Return true if value can be defined in continuous factor.
     * 
     * Il serait plus interessant de le faire sur les types et non sur les
     * valeur mais pour {@link RangeOfValues} par exemple, seule la valeur
     * a un type Float... donc pas evident.
     * 
     * @param value value
     * @return continuous enabled
     */
    public boolean canBeContinue(Object value) {
        boolean result = false;

        if (value instanceof Double) {
            result = true;
        } else if (value instanceof Long) {
            result = true;
        } else if (value instanceof Equation) {
            result = true;
        } else if (value instanceof MatrixND) {
            result = true;
        } else if (value instanceof RangeOfValues) {
            RangeOfValues rangeOfValues = (RangeOfValues)value;
            if (rangeOfValues.getType().equals("Float")) {
                result = true;
            }
        } else if (value instanceof TimeUnit) {
            result = true;
        } else if (value instanceof String) {
            // todo fix string value :(
            result = true;
        }

        return result;
    }

    /**
     * Return if value is is continue factor enable.
     * 
     * @param value value to test
     * @return {@code true} if value is is continue factor enable
     */
    public boolean isContinue(Object value) {
        boolean result = false;
        if (value instanceof RangeOfValues) {
            RangeOfValues range = (RangeOfValues)value;
            if (range.getType().equals(RangeOfValues.TYPE_FLOAT)) {
                String textValue = range.getValues();
                // TODO need comment !!!
                if (textValue.matches("^\\ *[0-9]*\\ *\\-\\ *[0-9]*\\ *$")) {
                    result = true;
                }
            }
        }
        return result;
    }
    
    /**
     * Get copy of component with original entity value as default value.
     * 
     * @param value value
     * @param bean bean
     * @param property property
     * @return component copy
     */
    public ContinuousPanelContainerUI getContinuousPanel(Object value, TopiaEntityContextable bean, String property) {
        ContinuousPanelContainerUI result;

        if (value instanceof RangeOfValues) {
            RangeOfValues rangeOfValues = (RangeOfValues)value;
            DefaultContinuousPanelUI ui = new DefaultContinuousPanelUI();
            String values = rangeOfValues.getValues();
            String min = "0";
            String max = "0";
            if (values.matches("^\\ *[0-9]*\\ *\\-\\ *[0-9]*\\ *$")) {
                int first = values.indexOf("-");
                if (first != -1) {
                    min = values.substring(0, first);
                    max = values.substring(first + 1);
                }
            }
            ui.init(min, max, min, null);
            result = ui;
        }
        else if (value instanceof Equation) {
            Equation equation = (Equation)value;
            EquationContinuousPanelUI ui = new EquationContinuousPanelUI(new JAXXInitialContext().add(new InputAction()));
            ui.setSelectedEquation(equation);
            ui.setText(_("isisfish.common.equation")); // can't get real name
            ui.setFormuleCategory(equation.getCategory());
            ui.setClazz(equation.getClass());
            ui.setBeanProperty(property);
            ui.setBean(bean);
            result = ui;
        } else if (value instanceof MatrixND) {
            MatrixND matrix = (MatrixND)value;
            MatrixContinuousPanelUI matrixPanel = new MatrixContinuousPanelUI();
            matrixPanel.init(matrix.clone(), matrix.clone(), matrix.clone(), null);
            result = matrixPanel;
        } else if (value instanceof TimeUnit) {
            TimeUnit timeUnit = (TimeUnit)value;
            DefaultContinuousPanelUI ui = new DefaultContinuousPanelUI();
            ui.init(String.valueOf(timeUnit.getTime()), String.valueOf(timeUnit.getTime()),
                    String.valueOf(timeUnit.getTime()), null);
            result = ui;
        } else {
            DefaultContinuousPanelUI ui = new DefaultContinuousPanelUI();
            ui.init(String.valueOf(value), String.valueOf(value), String.valueOf(value), null);
            result = ui;
        }

        if (log.isDebugEnabled()) {
            log.debug("Component for " + value + " (" + bean + ", " + property + ")");
            log.debug(" > " + result);
        }

        return result;
    }
    
    /**
     * Get continuous editor for component with given value.
     * 
     * @param domain domain
     * @param factor factor
     * @param topiaContext context used to get database value in case of equation factors
     * @return component copy
     */
    public ContinuousPanelContainerUI getContinuousPanelWithValue(Factor factor, ContinuousDomain domain, TopiaContext topiaContext) {
        ContinuousPanelContainerUI result = null;

        if (domain instanceof EquationContinuousDomain) {
            String factorPath = factor.getPath();
            if (factor.getPath().indexOf('#') != -1) {
                String topiaId = factorPath.substring(0, factorPath.lastIndexOf('#'));
                String property = factorPath.substring(factorPath.lastIndexOf('#') + 1);

                // get bean in database
                try {
                    TopiaEntityContextable entity = (TopiaEntityContextable)topiaContext.findByTopiaId(topiaId);
                    
                    String getter = "get" + StringUtils.capitalize(property);
                    Method m = entity.getClass().getMethod(getter);
                    Equation value = (Equation)m.invoke(entity);
                    
                    // fill component
                    EquationContinuousPanelUI ui = new EquationContinuousPanelUI(
                            new JAXXInitialContext().add(new InputAction()));
                    ui.setText(value.getContent());
                    ui.setFormuleCategory(value.getCategory());
                    ui.setText(_("isisfish.common.equation")); // can't get real name
                    ui.setClazz(value.getClass());
                    ui.setBeanProperty(property);
                    ui.setBean(entity);
        
                    EquationContinuousDomain equationDomain = (EquationContinuousDomain) domain;
                    ui.addDomain(equationDomain);
        
                    result = ui;
                } catch (Exception ex) {
                    throw new IsisFishRuntimeException("Can't restore initial factor database property", ex);
                }
            }
            // TODO path with no # (normalement pas possible pour les equations)
        } else if (domain instanceof MatrixContinuousDomain) {
            MatrixContinuousPanelUI continuousPanel = new MatrixContinuousPanelUI();
            // factor numerique continue (percentage)
            if (domain.isPercentageType()) {
                MatrixND matrix = (MatrixND)domain.getReferenceValue();
                // il y a bien 3 fois domain.getReferenceValue() pas d'erreur
                continuousPanel.initExisting(
                    matrix.clone(),
                    matrix.clone(),
                    matrix.clone(),
                    String.valueOf(domain.getCoefficient() * 100.0));
            }
            else {
                // factor numerique continue (min/max)
                // pas d'erreur sur 2 fois domain.getMinBound()
                MatrixND matrixMin = (MatrixND)domain.getMinBound();
                MatrixND matrixMax = (MatrixND)domain.getMaxBound();
                continuousPanel.initExisting(
                    matrixMin.clone(),
                    matrixMax.clone(),
                    matrixMin.clone(),
                    "");
            }
            result = continuousPanel;
        } else {
            DefaultContinuousPanelUI continuousPanel = new DefaultContinuousPanelUI();
            
            // factor numerique continue (percentage)
            if (domain.isPercentageType()) {
                // il y a bien 3 fois domain.getReferenceValue() pas d'erreur
                continuousPanel.initExisting(
                    String.valueOf(domain.getReferenceValue()),
                    String.valueOf(domain.getReferenceValue()),
                    String.valueOf(domain.getReferenceValue()),
                    String.valueOf(domain.getCoefficient() * 100.0 ));
            }
            else {
                // factor numerique continue (min/max)
                // pas d'erreur sur 2 fois domain.getMinBound()
                // on initialise avec minBound comme valeur de référence.
                continuousPanel.initExisting(
                    String.valueOf(domain.getMinBound()),
                    String.valueOf(domain.getMaxBound()),
                    String.valueOf(domain.getMinBound()),
                    "");
            }
            result = continuousPanel;
        }

        return result;
    }
    
    /**
     * Get editor for value (discrete).
     * 
     * @param factorWizardUI context for context value (RegionStorage)
     * @param value type to get editor
     * @param factor
     * @param domain
     * @return component
     */
    protected JComponent getEditorWithValue(FactorWizardUI factorWizardUI, Factor factor, Domain domain, Object value) {

        JComponent result = null;
        
        if (Double.class.isAssignableFrom(value.getClass())) {
            result = new JTextField();
            ((JTextField)result).setText(String.valueOf(value));
        }
        else if (double.class.isAssignableFrom(value.getClass())) {
            result = new JTextField();
            ((JTextField)result).setText(String.valueOf(value));
        }
        else if (MatrixND.class.isAssignableFrom(value.getClass())) {
            result = new MatrixPanelEditor();
            ((MatrixPanelEditor)result).setMatrix((MatrixND)value);
        }
        else if (domain instanceof RuleDiscreteDomain) {
            result = new RuleChooser(factorWizardUI);
            ((RuleChooser)result).setRulesList((List<Rule>)value);
        }
        else if (domain instanceof EquationDiscreteDomain) {
            String factorPath = factor.getPath();
            if (factor.getPath().indexOf('#') != -1) {
                String topiaId = factorPath.substring(0, factorPath.lastIndexOf('#'));
                String property = factorPath.substring(factorPath.lastIndexOf('#') + 1);

                // get bean in database
                try {
                    TopiaContext topiaContext = factorWizardUI.getContextValue(TopiaContext.class);
                    TopiaEntityContextable entity = (TopiaEntityContextable)topiaContext.findByTopiaId(topiaId);

                    String getter = "get" + StringUtils.capitalize(property);
                    Method m = entity.getClass().getMethod(getter);
                    Equation equation = (Equation)m.invoke(entity);

                    // fill component
                    InputOneEquationUI ui = new InputOneEquationUI(factorWizardUI);
                    ui.setAutoSaveModification(false);
                    ui.setFormuleCategory(equation.getCategory());
                    ui.setText(_("isisfish.common.equation")); // can't get real name
                    ui.setClazz(value.getClass());
                    ui.setBeanProperty(property);
                    ui.setBean(entity); // set bean fire content modification event
                    ui.getEditor().setText((String)value);
                    ui.setActive(true);

                    result = ui;
                } catch (Exception ex) {
                    throw new IsisFishRuntimeException("Can't restore initial factor database property", ex);
                }
            }
            // TODO path with no # (normalement pas possible pour les equations)
        }
        else if (value instanceof TimeUnit) {
            result = new JTextField();
            ((JTextField)result).setText(String.valueOf(((TimeUnit)value).getTime()));
        } else if (value instanceof TopiaEntity) {
            RegionStorage regionStorage = factorWizardUI.getContextValue(RegionStorage.class);
            TopiaContext context = null;
            try {
                context = regionStorage.getStorage().beginTransaction();
                List list = context.find("from " + value.getClass().getName());
                JComboBox c = new JComboBox(list.toArray());
                c.setSelectedItem(value);
                result = c;
            } catch (TopiaException ex) {
                throw new IsisFishRuntimeException("Can't get entity list", ex);
            } finally {
                if (context != null) {
                    try {
                        context.closeContext();
                    } catch (TopiaException ex) {
                        throw new IsisFishRuntimeException("Can't get entity list", ex);
                    }
                }
            }
        } else if (value instanceof TimeStep) {
            TimeStep timeStep = (TimeStep)value;
            result = new StepComponent(timeStep.getMonth().getMonthNumber(), timeStep.getYear());
        } else if (value instanceof Month) {
            Month month = (Month)value;
            result = MonthComponent.createMounthCombo(month.getMonthNumber());
        } else if (value instanceof String) {
            // valeur non typées ???
            result = new JTextField();
            ((JTextField)result).setText(String.valueOf(value));
        }
        
        if (log.isDebugEnabled()) {
            log.debug("Editor for value " + value + " is " + result);
        }

        return result;
    }
    
    /**
     * Get copy of component with original entity value as default value.
     * Used to add new tab to a new or existing factor.
     * 
     * Le composant retourné est inclut dans un jscrollpane (sauf pour les
     * matrices qui contient deja un jscrollpane)
     * 
     * @param factorWizardUI factorWizardUI
     * @return component copy
     */
    public JComponent getNewDiscreteComponent(FactorWizardUI factorWizardUI) {

        JComponent result = null;
        String factorPath = factorWizardUI.getFactorPath();

        if (factorPath.indexOf('#') != -1) {
            String topiaId = factorPath.substring(0, factorPath.lastIndexOf('#'));
            String property = factorPath.substring(factorPath.lastIndexOf('#') + 1);
            TopiaContext topiaContext = factorWizardUI.getContextValue(TopiaContext.class);

            try {
                // get bean in database
                TopiaEntityContextable entity = (TopiaEntityContextable)topiaContext.findByTopiaId(topiaId);
                String getter = "get" + StringUtils.capitalize(property);
                Method m = entity.getClass().getMethod(getter);
                Object value = m.invoke(entity);

                // init new jcomponent for value
                if (value instanceof Number) {
                    result = new JTextField(String.valueOf(value));
                } else if (value instanceof MatrixND) {
                    result = new MatrixPanelEditor();
                    MatrixND matrix = ((MatrixND)value).copy();
                    ((MatrixPanelEditor)result).setMatrix(matrix);
                } else if (value instanceof RangeOfValues) {
                    RangeOfValues rangeOfValues = (RangeOfValues)value;
                    result = new JTextField(rangeOfValues.getValues());
                } else if (value instanceof TimeUnit) {
                    TimeUnit timeUnit = (TimeUnit)value;
                    result = new JTextField(String.valueOf(timeUnit.getTime()));
                } else if (value instanceof Equation) {
                    Equation equation = (Equation)value;
                    // fill component
                    InputOneEquationUI ui = new InputOneEquationUI(factorWizardUI);
                    ui.setAutoSaveModification(false);
                    ui.setText(equation.getContent());
                    ui.setFormuleCategory(equation.getCategory());
                    ui.setText(_("isisfish.common.equation")); // can't get real name
                    ui.setClazz(value.getClass());
                    ui.setBeanProperty(property);
                    ui.setBean(entity);
                    ui.setActive(true);
                    result = ui;
                }

            } catch (Exception ex) {
                throw new IsisFishRuntimeException("Can't restore intial factor database property", ex);
            }
        }
        else {
            // dans ce cas c'est des regles, pop de départ ou parametres de regles
            // c'est un peu galere car le code n'a rien a voir avec le reste
            // donc, c'est du cas par cas
            if (factorPath.equals("parameters.rules")) {
                result = new RuleChooser(factorWizardUI);
            } else if (factorPath.startsWith("parameters.population.")) {
                MatrixContinuousPanelUI currentPanel = (MatrixContinuousPanelUI)factorWizardUI.continuousPanel;
                // on copie une des matrices du composant
                MatrixND matrix = currentPanel.getReferenceValuePanel().getMatrix().copy();
                result = new MatrixPanelEditor(); // bug in 2.2.x with matrix in constructor
                ((MatrixPanelEditor)result).setMatrix(matrix);
            } else if (factorPath.startsWith("parameters.rule.")) {
                Pattern pattern = Pattern.compile("^parameters\\.rule\\.\\d+\\.parameter\\.\\w+\\.(.+)$");
                Matcher matcher = pattern.matcher(factorPath);
                if (matcher.matches()) {
                    String className = matcher.group(1);
                    try {
                        if (log.isDebugEnabled()) {
                            log.debug("Looking for a component for class : " + className);
                        }
                        Class type = Class.forName(className);
                        result = getRuleDiscreteComponent(factorWizardUI, type);
                    } catch (ClassNotFoundException ex) {
                        if (log.isWarnEnabled()) {
                            log.warn("Can't find class for rule parameter", ex);
                        }
                    }
                } else {
                    // double...
                    result = new JTextField();
                }
            } else {
                if (log.isWarnEnabled()) {
                    log.warn("Can't find component for path " + factorPath);
                }
            }
        }

        if (log.isDebugEnabled()) {
            log.debug("Component for path " + factorPath + " is " + result);
        }

        // hack : si on met 2 fois un jscrollpane, rien ne s'affiche
        if (!(result instanceof MatrixPanelEditor)) {
            result = new JScrollPane(result);
        }

        return result;
    }

    /**
     * Get new special 
     * @param clazz
     * @return
     */
    protected JComponent getRuleDiscreteComponent(FactorWizardUI factorWizardUI, Class type) {

        JComponent result = null;

        if (TopiaEntity.class.isAssignableFrom(type)) {
            RegionStorage regionStorage = factorWizardUI.getContextValue(RegionStorage.class);
            TopiaContext context = null;
            try {
                context = regionStorage.getStorage().beginTransaction();
                List list = context.find("from " + type.getName());
                JComboBox c = new JComboBox(list.toArray());
                result = c;
            } catch (TopiaException ex) {
                throw new IsisFishRuntimeException("Can't get entity list", ex);
            } finally {
                if (context != null) {
                    try {
                        context.closeContext();
                    } catch (TopiaException ex) {
                        throw new IsisFishRuntimeException("Can't get entity list", ex);
                    }
                }
            }
        } else if (TimeStep.class.isAssignableFrom(type)) {
            result = new StepComponent(0, 0);
        } else if (Month.class.isAssignableFrom(type)) {
            result = MonthComponent.createMounthCombo(0);
        } else {
            result = new JTextField();
        }

        return result;
    }

    /**
     * Save current factor.
     * 
     * @param factorWizardUI factorWizardUI
     */
    public void save(FactorWizardUI factorWizardUI) {

        // first check is factor is valid
        boolean factorValid = true;
        ContinuousPanelContainerUI continuousPanel = factorWizardUI.continuousPanel;
        if (continuousPanel != null) {
            factorValid = continuousPanel.isFactorValid();
        }
        if (!factorValid) {
            JOptionPane.showMessageDialog(factorWizardUI, _("isisfish.sensitivity.factor.notvalid"),
                    _("isisfish.sensitivity.title"), JOptionPane.ERROR_MESSAGE);
            return;
        }

        // call specific method depending on continuous/discrete
        if (factorWizardUI.getContinueRadio().isSelected()) {
            saveContinue(factorWizardUI.getFactorNameField().getText(),
                factorWizardUI.getComment().getText(), factorWizardUI.getFactorPath(), factorWizardUI.continuousPanel,
                factorWizardUI.getSimulAction(), factorWizardUI.isExistingValue());
        } else {
            saveDiscret(factorWizardUI.getFactorNameField().getText(),
                    factorWizardUI.getComment().getText(), factorWizardUI.getFactorPath(), factorWizardUI.discretComponents,
                factorWizardUI.getSimulAction(), factorWizardUI.isExistingValue());
        }

        // refresh factor list
        factorWizardUI.getContextValue(SimulationUI.class, "SimulationUI").refreshFactorTree();

        // close window
        factorWizardUI.dispose();
    }

    /**
     * Save a continous factor.
     * 
     * @param name factor name
     * @param comment comment
     * @param path factor path
     * @param panel panel
     * @param action action
     * @param exist exist
     */
    protected void saveContinue(String name,
            String comment, String path, ContinuousPanelContainerUI panel,
            SimulAction action, boolean exist) {
        if (panel instanceof MatrixContinuousPanelUI) {
            MatrixContinuousPanelUI matrixPanel = (MatrixContinuousPanelUI) panel;
            if (matrixPanel.isPercentageTypeFactor()) {
                MatrixND referenceValue = matrixPanel.getReferenceValuePanel().getMatrix();
                Double coefficient = Double.valueOf(matrixPanel.getCoefficientField().getText()) / 100;
                action.addContinuousMatrixFactor(name, comment, path,
                        referenceValue, coefficient, exist);
            }
            else {
                MatrixND minBound = matrixPanel.getMinValuePanel().getMatrix();
                MatrixND maxBound = matrixPanel.getMaxValuePanel().getMatrix();
                action.addContinuousMatrixFactor(name, comment, path,
                    minBound, maxBound, exist);
            }

        } else if (panel instanceof EquationContinuousPanelUI) {
            try {
                EquationContinuousPanelUI equationPanel = (EquationContinuousPanelUI) panel;

                TopiaEntityContextable bean = equationPanel.getBean();
                TopiaContext topiaContext = bean.getTopiaContext();

                String property = equationPanel.getBeanProperty();
                property = StringUtils.capitalize(property) + "Content";
                Method m = bean.getClass().getMethod("set" + property, String.class);
                m.invoke(bean, equationPanel.getEditor().getEditor().getText());

                // Save equation
                bean.update();
                topiaContext.commitTransaction();
                
                List<EquationContinuousDomain> domains = equationPanel.getDomains();
                for (EquationContinuousDomain domain : domains) {
                    action.addContinuousEquationFactor(name, comment, path, domain, exist);
                }
            } catch (Exception ex) {
                if (log.isErrorEnabled()) {
                    log.error("Can't call method : ", ex);
                }
            }
        } else if (panel instanceof DefaultContinuousPanelUI) {
            DefaultContinuousPanelUI defaultPanel = (DefaultContinuousPanelUI) panel;
            
            if (defaultPanel.isPercentageTypeFactor()) {
                Double referenceValue = Double.valueOf(defaultPanel.getReferenceValueField().getText());
                Double coefficient = Double.valueOf(defaultPanel.getCoefficientField().getText()) / 100;
                action.addContinuousPercentageFactor(name, comment, path,
                        referenceValue, coefficient, exist);
            }
            else {
                double minBound = Double.parseDouble(defaultPanel.getContinueMin().getText());
                double maxBound = Double.parseDouble(defaultPanel.getContinueMax().getText());
                action.addContinuousFactor(name, comment, path,
                    minBound, maxBound, exist);
            }
        }
    }

    /**
     * Save a discret factor.
     * 
     * @param name
     * @param comment
     * @param path
     * @param components
     * @param action
     * @param exist
     */
    protected void saveDiscret(String name,
            String comment, String path, List<JComponent> components,
            SimulAction action, boolean exist) {
        List<Object> values = new ArrayList<Object>();
        
        boolean ruleFactor = false;
        boolean equationFactor = false;
        for (JComponent component : components) {

            // get internat component value
            Object result = null;
            if (component instanceof JTextComponent) {
                result = ((JTextComponent) component).getText();
            } else if (component instanceof MatrixPanelEditor) {
                result = ((MatrixPanelEditor) component).getMatrix();
            } else if (component instanceof InputOneEquationUI) {
                result = ((InputOneEquationUI) component).getEditor().getText();
                equationFactor = true;
            } else if (component instanceof RuleChooser) {
                result = ((RuleChooser)component).getRulesList();
                ruleFactor = true;
            } else if (component instanceof StepComponent) {
                result = new TimeStep(((StepComponent)component).getSelectedValue());
            } else if (component instanceof MonthComponent) {
                result = new Month(((MonthComponent)component).getSelectedValue());
            } else if (component instanceof JComboBox) {
                // on suppose qu'il y a dedans des TopiaEntity
                result = ((JComboBox)component).getSelectedItem();
            }

            values.add(result);
        }
        
        if (ruleFactor) {
            action.addDiscreteRuleFactor(name, comment, path, values, exist);
        } else if (equationFactor) {
            action.addDiscreteEquationFactor(name, comment, path, values, exist);
        } else {
            action.addDiscreteFactor(name, comment, path, values, exist);
        }
    }

    /**
     * Return value in swing component that could be next used into factor
     * for discrete factor values.
     * 
     * @param component component
     * @return factor value
     */
    protected Object getComponentValue(JComponent component) {
        Object result = null;
        if (component instanceof JTextComponent) {
            result = ((JTextComponent) component).getText();
        } else if (component instanceof MatrixPanelEditor) {
            result = ((MatrixPanelEditor) component).getMatrix();
        } else if (component instanceof InputOneEquationUI) {
            result = ((InputOneEquationUI) component).getEditor().getText();
        } else if (component instanceof RuleChooser) {
            result = ((RuleChooser)component).getRulesList();
        }
        if (log.isDebugEnabled()) {
            log.debug("Value for component : " + component.getClass().getSimpleName() + " is " + result);
        }

        return result;
    }

    /**
     * Remove current factor.
     * 
     * @param factorWizardUI factorWizardUI
     */
    public void remove(FactorWizardUI factorWizardUI) {
        factorWizardUI.getSimulAction().removeFactor(factorWizardUI.getFactorPath());
        factorWizardUI.getContextValue(SimulationUI.class, "SimulationUI").refreshFactorTree();
        factorWizardUI.dispose();
    }
}
