/*
 * #%L
 * EchoBase :: UI
 * 
 * $Id: EchoBaseApplicationContext.java 842 2013-08-15 10:53:16Z tchemit $
 * $HeadURL: https://forge.codelutin.com/svn/echobase/tags/echobase-2.2/echobase-ui/src/main/java/fr/ifremer/echobase/ui/EchoBaseApplicationContext.java $
 * %%
 * Copyright (C) 2011 Ifremer, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * #L%
 */
package fr.ifremer.echobase.ui;

import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import com.opensymphony.xwork2.ActionContext;
import fr.ifremer.echobase.config.EchoBaseConfiguration;
import fr.ifremer.echobase.entities.TopiaEchoBaseInternalPersistenceContext;
import fr.ifremer.echobase.entities.TopiaEchoBasePersistenceContext;
import fr.ifremer.echobase.entities.spatial.SpatialDataCache;
import fr.ifremer.echobase.persistence.EchoBaseDbMeta;
import fr.ifremer.echobase.persistence.EchoBaseEntityHelper;
import fr.ifremer.echobase.persistence.EchobaseTopiaContexts;
import fr.ifremer.echobase.services.DefaultEchoBaseServiceContext;
import fr.ifremer.echobase.services.EchoBaseServiceContext;
import fr.ird.converter.FloatConverter;
import org.apache.commons.beanutils.ConvertUtils;
import org.apache.commons.beanutils.Converter;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.i18n.init.DefaultI18nInitializer;
import org.nuiton.topia.TopiaContext;
import org.nuiton.util.converter.ConverterUtil;

import javax.servlet.ServletContext;
import java.beans.Introspector;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

/**
 * @author tchemit <chemit@codelutin.com>
 * @since 0.1
 */
public class EchoBaseApplicationContext {

    /** Logger. */
    private static final Log log =
            LogFactory.getLog(EchoBaseApplicationContext.class);

    /** Key to store the single instance of the application context */
    private static final String APPLICATION_CONTEXT_PARAMETER =
            "echobaseApplicationContext";

    /** Set of all loggued user sessions to be close at shutdown time. */
    protected Set<EchoBaseSession> sessions;

    public static EchoBaseApplicationContext getApplicationContext(ActionContext actionContext) {
        Map<String, Object> application = actionContext.getApplication();
        EchoBaseApplicationContext result =
                (EchoBaseApplicationContext) application.get(
                        APPLICATION_CONTEXT_PARAMETER);
        return result;
    }

    public static EchoBaseApplicationContext getApplicationContext(ServletContext servletContext) {
        EchoBaseApplicationContext result =
                (EchoBaseApplicationContext) servletContext.getAttribute(
                        APPLICATION_CONTEXT_PARAMETER);
        return result;
    }

    public static void setApplicationContext(ServletContext servletContext,
                                             EchoBaseApplicationContext applicationContext) {
        servletContext.setAttribute(APPLICATION_CONTEXT_PARAMETER,
                                    applicationContext);
    }

    public static void removeApplicationContext(ServletContext servletContext) {
        servletContext.removeAttribute(APPLICATION_CONTEXT_PARAMETER);
    }

    protected EchoBaseConfiguration configuration;

    protected EchoBaseDbMeta dbMeta;

    /** Root context for the internal database. */
    protected TopiaContext internalRootContext;

    /**
     * Flag setted to true when internal db was just created (should then
     * display in ui created user password).
     *
     * @since 1.1
     */
    protected boolean defaultUsersCreated;

    /**
     * A simple cache of spatial data.
     *
     * @since 2.2
     */
    protected final SpatialDataCache spatialDataCache;

    public EchoBaseApplicationContext() {

        spatialDataCache = new SpatialDataCache();
    }

    public Set<EchoBaseSession> getEchoBaseSessions() {
        return sessions;
    }

    public synchronized void registerEchoBaseSession(EchoBaseSession session) {
        Preconditions.checkNotNull(session);
        Preconditions.checkNotNull(session.getUser());
        if (sessions == null) {
            sessions = Sets.newHashSet();
        }
        if (log.isInfoEnabled()) {
            log.info("Register user session for [" +
                     session.getUser().getEmail() + "]");
        }
        sessions.add(session);
    }

    public synchronized void destroyEchoBaseSession(EchoBaseSession session) {
        Preconditions.checkNotNull(session);
        Preconditions.checkNotNull(session.getUser());
        Preconditions.checkNotNull(sessions);
        if (log.isInfoEnabled()) {
            log.info("Destroy user session for [" +
                     session.getUser().getEmail() + "]");
        }
        // remove session from active ones
        sessions.remove(session);
        // close session
        session.close();
    }

    public void init() {

        // init I18n
        DefaultI18nInitializer i18nInitializer =
                new DefaultI18nInitializer("echobase-i18n");
        i18nInitializer.setMissingKeyReturnNull(true);
        I18n.init(i18nInitializer, Locale.getDefault());

        // init converters
        Converter converter = ConverterUtil.getConverter(Float.class);
        if (converter != null) {
            ConvertUtils.deregister(Float.class);
        }
        ConvertUtils.register(new FloatConverter(), Float.class);

        // initialize configuration
        EchoBaseConfiguration configuration = new EchoBaseConfiguration();

        // initialize internal root context
        TopiaContext internalRootContext =
                EchobaseTopiaContexts.newInternalDb(
                        configuration.getInternalDbDirectory());

        setConfiguration(configuration);
        setDbMeta(EchoBaseDbMeta.newDbMeta());
        setInternalRootContext(internalRootContext);
    }

    public EchoBaseConfiguration getConfiguration() {
        return configuration;
    }

    public EchoBaseDbMeta getDbMeta() {
        return dbMeta;
    }

    public TopiaContext getInternalRootContext() {
        return internalRootContext;
    }

    public boolean isDefaultUsersCreated() {
        return defaultUsersCreated;
    }

    public SpatialDataCache getSpatialDataCache() {
        return spatialDataCache;
    }

    public EchoBaseServiceContext newServiceContext(Locale locale,
                                                    TopiaContext topiaInternalContext,
                                                    TopiaContext topiaContext) {

        EchoBaseServiceContext newServiceContext =
                DefaultEchoBaseServiceContext.newContext(
                        locale,
                        configuration,
                        dbMeta,
                        spatialDataCache);

        TopiaEchoBaseInternalPersistenceContext internalPersistenceContext =
                new TopiaEchoBaseInternalPersistenceContext(topiaInternalContext);

        newServiceContext.setEchoBaseInternalPersistenceContext(
                internalPersistenceContext);

        TopiaEchoBasePersistenceContext persistenceContext =
                new TopiaEchoBasePersistenceContext(topiaContext);

        newServiceContext.setEchoBasePersistenceContext(persistenceContext);

        return newServiceContext;
    }

    public void close() {

        try {
            spatialDataCache.clear();

            if (internalRootContext != null) {
                // release internal db
                EchoBaseEntityHelper.releaseRootContext(internalRootContext);
            }
        } finally {

            try {
                // release all user sessions
                if (CollectionUtils.isNotEmpty(sessions)) {
                    for (EchoBaseSession session : sessions) {
                        destroyEchoBaseSession(session);
                    }
                }
            } finally {
                // see http://wiki.apache.org/commons/Logging/FrequentlyAskedQuestions#A_memory_leak_occurs_when_undeploying.2Fredeploying_a_webapp_that_uses_Commons_Logging._How_do_I_fix_this.3F
                ClassLoader contextClassLoader = Thread.currentThread().getContextClassLoader();
                LogFactory.release(contextClassLoader);

                Introspector.flushCaches();
            }
        }
    }

    public void setDbMeta(EchoBaseDbMeta dbMeta) {
        this.dbMeta = dbMeta;
    }

    public void setInternalRootContext(TopiaContext internalRootContext) {
        this.internalRootContext = internalRootContext;
    }

    public void setDefaultUsersCreated(boolean defaultUsersCreated) {
        this.defaultUsersCreated = defaultUsersCreated;
    }

    public void setConfiguration(EchoBaseConfiguration configuration) {
        this.configuration = configuration;
    }

//    protected final EventListenerList listenerList = new EventListenerList();
//
//    public void addWorkingDbChangeListener(WorkingDbChangeListener listener) {
//        listenerList.add(WorkingDbChangeListener.class, listener);
//    }
//
//    public void removeWorkingDbChangeListener(WorkingDbChangeListener listener) {
//        listenerList.remove(WorkingDbChangeListener.class, listener);
//    }
//
//    public void fireVoyageChangeEvent(String jdbcUrl, String voyageId) {
//        VoyageChangedEvent event = new VoyageChangedEvent(jdbcUrl, voyageId);
//        for (WorkingDbChangeListener listener : listenerList.getListeners(WorkingDbChangeListener.class)) {
//            listener.onVoyageChanged(event);
//        }
//    }
//
//    public void fireDbChangeEvent(String jdbcUrl) {
//        DbChangedEvent event = new DbChangedEvent(jdbcUrl);
//        for (WorkingDbChangeListener listener : listenerList.getListeners(WorkingDbChangeListener.class)) {
//            listener.onDbChanged(event);
//        }
//    }
}
