/*
 * #%L
 * Wao :: Web Interface
 * 
 * $Id: Administration.java 583 2010-06-30 15:34:55Z fdesbois $
 * $HeadURL: svn+ssh://fdesbois@labs.libre-entreprise.org/svnroot/suiviobsmer/tags/wao-1.5.1/wao-ui/src/main/java/fr/ifremer/wao/ui/pages/Administration.java $
 * %%
 * Copyright (C) 2009 - 2010 Ifremer
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

package fr.ifremer.wao.ui.pages;

import fr.ifremer.wao.WaoBusinessException;
import fr.ifremer.wao.WaoBusinessException.Type;
import fr.ifremer.wao.WaoException;
import fr.ifremer.wao.WaoProperty;
import fr.ifremer.wao.bean.ConnectedUser;
import fr.ifremer.wao.io.ImportResults;
import fr.ifremer.wao.io.ImportResultsImpl;
import fr.ifremer.wao.bean.UserRole;
import fr.ifremer.wao.entity.Company;
import fr.ifremer.wao.entity.CompanyImpl;
import fr.ifremer.wao.entity.WaoUser;
import fr.ifremer.wao.service.ServiceBoat;
import fr.ifremer.wao.service.ServiceCartography;
import fr.ifremer.wao.service.ServiceReferential;
import fr.ifremer.wao.service.ServiceSampling;
import fr.ifremer.wao.service.ServiceUser;
import fr.ifremer.wao.ui.data.ErrorReport;
import fr.ifremer.wao.ui.data.GenericSelectModel;
import fr.ifremer.wao.ui.components.Layout;
import fr.ifremer.wao.ui.data.ImportEngine;
import fr.ifremer.wao.ui.data.RequiresAuthentication;
import fr.ifremer.wao.ui.services.WaoManager;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.OptionModel;
import org.apache.tapestry5.RenderSupport;
import org.apache.tapestry5.SelectModel;
import org.apache.tapestry5.annotations.Environmental;
import org.apache.tapestry5.annotations.IncludeJavaScriptLibrary;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.beaneditor.BeanModel;
import org.apache.tapestry5.corelib.components.BeanEditForm;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.internal.OptionModelImpl;
import org.apache.tapestry5.internal.SelectModelImpl;
import org.apache.tapestry5.ioc.Messages;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.ioc.services.TypeCoercer;
import org.apache.tapestry5.services.BeanModelSource;
import org.slf4j.Logger;

/**
 * Administration
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois <fdesbois@codelutin.com>
 */
@RequiresAuthentication(value = {UserRole.ADMIN, UserRole.COORDINATOR}, readOnlyAllowed = false)
@IncludeStylesheet("context:css/administration.css")
@IncludeJavaScriptLibrary("context:js/AdminUserSelect.js")
public class Administration {

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private ConnectedUser currentUser;

    @Inject
    private Logger logger;
                                  
    @Inject
    private WaoManager manager;

    @Inject
    private ServiceUser serviceUser;

    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private ServiceCartography serviceCartography;

    @Inject
    private TypeCoercer typeCoercer;

    @Inject
    private Messages messages;

    @Environmental
    private RenderSupport renderSupport;

    @Property
    private String companyId;

    @Property
    private String userId;

    @Persist
    private List<Company> companies;

    @Inject
    private PropertyAccess propertyAccess;

    private GenericSelectModel<Company> companiesSelectModel;

    private Company company;

    private GenericSelectModel<WaoUser> usersSelectModel;

    @InjectComponent
    private BeanEditForm userForm;

    private boolean addNewCompanySelected;
    private boolean addNewUserSelected;
    private boolean deleteUserSelected;

    void setupRender() throws WaoException {
        companies = null;
        getCompanies();
        userEdited = null;
        userModel = null;
    }

    void onActivate(EventContext ec) {
        if (ec.getCount() > 0) {
            companyId = ec.get(String.class, 0);
            if (ec.getCount() > 1) {
                userId = ec.get(String.class, 1);
            }
        } else if (currentUser.isCoordinator()) {
            companyId = currentUser.getCompany().getId();
        }
    }

    Object[] onPassivate() {
        return new String[] {companyId, userId};
    }

    void afterRender() {
        // Prototype class for UserSelect (manual usage because of specific style on options)
        renderSupport.addScript("new AdminUserSelect();");
    }

   /**************************** IMPORTS (ADMIN) *******************************/

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceBoat serviceBoat;

    public ImportEngine getFishingZoneImportEngine() {
        return new ImportEngine() {
            
            @Override
            public ImportResults execute(InputStream input)
                    throws WaoException, WaoBusinessException {
                int nbImported = serviceReferential.importFishingZoneCsv(input);
                ImportResults results = new ImportResultsImpl();
                results.setNbRowsImported(nbImported);
                return results;
            }
        };
    }

    public ImportEngine getSamplingPlanImportEngine() {
        return new ImportEngine() {

            @Override
            public ImportResults execute(InputStream input)
                    throws WaoException, WaoBusinessException {
                ImportResults results = serviceSampling.importSamplingPlanCsv(input);
                return results;
            }
        };
    }

    @Log
    public ImportEngine getBoatImportEngine() {
        return new ImportEngine() {

            @Override
            public ImportResults execute(InputStream input)
                    throws WaoException, WaoBusinessException {
                int[] result = serviceBoat.importBoatCsv(input);
                ImportResults results = new ImportResultsImpl();
                results.setNbRowsImported(result[0]);
                results.setNbRowsImportedNew(result[1]);
                return results;
            }
        };
    }

    public ImportEngine getActivityCalendarImportEngine() {
        return new ImportEngine() {

            @Override
            public ImportResults execute(InputStream input)
                    throws WaoException, WaoBusinessException {
                // Will execute a thread to import activityCalendars
                serviceBoat.importActivityCalendarCsv(input);
                // Errors will be written in an activityCalendarLogFile
                return null;
            }
        };
    }

    public InputStream getActivityCalendarLogFile() {
        return getActivityCalendarLogFile(WaoProperty.FILENAME_LOG_ACTIVITY_IMPORT);
    }

    public InputStream getActivityCalendarAccessLogFile() {
        return getActivityCalendarLogFile(WaoProperty.FILENAME_LOG_ACTIVITY_ACCESS);
    }
    
    private InputStream getActivityCalendarLogFile(WaoProperty property) {
        InputStream result = null;
        try {
            switch (property) {
                case FILENAME_LOG_ACTIVITY_ACCESS:
                    result = serviceBoat.getActivityCalendarLogAccessFile(); break;
                case FILENAME_LOG_ACTIVITY_IMPORT:
                    result = serviceBoat.getActivityCalendarLogFile();
            }
        } catch (FileNotFoundException eee) {
            if (logger.isInfoEnabled()) {
                logger.info("Not existing log file for " + property);
            }
        }
        return result;
    }

    public boolean isActivityCalendarImportRun() {
        return manager.isActivityCalendarImportRun();
    }

    public ImportEngine getBoatDistrictImportEngine() {
        return new ImportEngine() {

            @Override
            public ImportResults execute(InputStream input)
                    throws WaoException, WaoBusinessException {
                return serviceCartography.importBoatDistrictKml(input);
            }
        };
    }

    @Log
    void onImported(ErrorReport report) {
        for (String info : report.getInfos()) {
            layout.addInfo(info);
        }
        for (String error : report.getErrors()) {
            layout.addError(error);
        }
    }

    /**************************** FORMS ****************************************/

    public List<Company> getCompanies() throws WaoException {
        if (companies == null) {
            companies = serviceUser.getCompanies(false);
        }
        return companies;
    }

    public GenericSelectModel<Company> getCompaniesSelectModel() throws WaoException {
        if (companiesSelectModel == null) {
            companiesSelectModel = new GenericSelectModel<Company>(getCompanies(),
                    Company.class, Company.NAME, "id", propertyAccess);
        }
        return companiesSelectModel;
    }

    public Company getCompany() throws WaoException {
        if (company == null) {
            if (companyId != null) {
                company = getCompaniesSelectModel().findObject(companyId);
            } else {
                company = new CompanyImpl();
            }
        }
        return company;
    }

    public boolean isUsersAvailable() throws WaoException {
        return !StringUtils.isEmpty(getCompany().getId()) && getCompany().getActive();
    }

    void onSelectedFromAddNewCompany() {
        addNewCompanySelected = true;
    }

    void onSuccessFromActionsForm() {
        if (addNewCompanySelected) {
            companyId = null;            
        }
        userId = null;
        userEdited = null;
    }

    void onSuccessFromCompany() throws WaoException {
        serviceUser.createUpdateCompany(company);
        companyId = company.getId();
    }

    /******************** USER ACTIONS FORM ***********************************/

    private List<WaoUser> users;

    @Property
    private WaoUser user;

    @Log
    public List<WaoUser> getUsers() {
        if (users == null) {
            users = serviceUser.getUsersByCompany(getCompany());
        }
        return users;
    }

//    @Log
//    public GenericSelectModel<WaoUser> getUsersSelectModel() throws WaoException {
//        if (usersSelectModel == null) {
//            List<WaoUser> users = serviceUser.getUsersByCompany(getCompany());
//            if (logger.isDebugEnabled()) {
//                logger.debug("Nb users : " + users.size());
//            }
//            usersSelectModel = new GenericSelectModel<WaoUser>(users, WaoUser.class,
//                    "fullName", "id", propertyAccess);
//        }
//        return usersSelectModel;
//    }

    private WaoUser findUser(String id) {
        for (WaoUser user : getUsers()) {
            if (user.getId().equals(id)) {
                return user;
            }
        }
        return null;
    }

    public String getUserOptionStyle() {
        String result = "";
        // Inactive user
        if (!user.getActive()) {
            result = "line-through";
        }
        return result;
    }

    @Log
    void onSelectedFromAddNewUser() {
        addNewUserSelected = true;
    }

    @Log
    void onSelectedFromDeleteUser() {
        deleteUserSelected = true;
    }

    @Log
    void onSuccessFromUserActionsForm() throws WaoException {
        if (addNewUserSelected) {
            userId = null;
        } else if (deleteUserSelected && userId != null) {
            userEdited = findUser(userId);
            try {
                serviceUser.deleteUser(userEdited);
                userEdited = null;
                userId = null;
                layout.addInfo("Utilisateur supprimé avec succès !");
            } catch (WaoBusinessException eee) {
                layout.addError(eee.getMessage());
            }
        }
        if (logger.isDebugEnabled()) {
            logger.debug("'User id : " + userId);
        }
    }

     /******************** USER FORM ******************************************/

    @InjectComponent
    private Zone userFormZone;

    @Inject
    private BeanModelSource beanModelSource;

    /**
     * User is persistant cause of autobuild problem ?!?
     */
    @Persist
    private WaoUser userEdited;

    @Persist
    private BeanModel<WaoUser> userModel;

    @Property
    private boolean generatePassword;

    @Property
    private String password;

    public WaoUser getUserEdited() throws WaoException {
        if (userEdited == null) {
            if (userId != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("User exist in selectModel : " + userId);
                }
                userEdited = findUser(userId);
            } else {
                userEdited = serviceUser.getNewUser(getCompany());
                generatePassword = true;
            }
        }
        return userEdited;
    }

    public BeanModel<WaoUser> getUserModel() {
        if (userModel == null) {
            userModel = beanModelSource.createEditModel(WaoUser.class, messages);
            userModel.add("userRole", null);
            userModel.include(
                    WaoUser.FIRST_NAME,
                    WaoUser.LAST_NAME,
                    WaoUser.LOGIN,
                    WaoUser.PASSWORD,
                    WaoUser.PHONE_NUMBER,
                    "userRole",
                    WaoUser.ACTIVE
            );
        }
        return userModel;
    }

    @Log
    void onSelectedFromUserForm() {
        refreshUserRoleZone = false;
    }

    @Log
    void onValidateFormFromUserForm() {
        userForm.clearErrors();
        if (userEdited.getUserRoles().isEmpty()) {
            userForm.recordError("L'utilisateur doit au minimum avoir un rôle.");
        }
    }

    @Log
    Object onSuccessFromUserForm() throws WaoException {
        if (logger.isDebugEnabled()) {
            logger.debug("User : " + userEdited);
        }

        if (!StringUtils.isEmpty(password)) {
            userEdited.setPassword(password);
            userEdited.setPasswordChanged(true);
        }
        try {
            serviceUser.createUpdateUser(userEdited, generatePassword);
            userId = userEdited.getId();
        } catch (WaoBusinessException eee) {
            if (eee.getType().equals(Type.SMTP_NOT_FOUND)) {
                layout.addInfo(eee.getMessage());
                userId = userEdited.getId();
            } else {
                layout.addError(eee.getMessage());
            }
        }
        return this;
    }

    @Log
    Object onFailureFromUserForm() {
        return userForm;
    }

    /****************************** UserRole **********************************/

    @InjectComponent
    private Zone userRoleZone;

    private SelectModel userRoleSelectModel;

    @Persist
    @Property
    private UserRole userRole;

    @Persist
    @Property
    private boolean readOnly;

    @Property
    private UserRole role;

    @Property
    private int roleIndex;

    @Property
    private boolean refreshUserRoleZone;

    public SelectModel getUserRoleSelectModel() {
        if (userRoleSelectModel == null) {
            List<OptionModel> options = new ArrayList<OptionModel>();
            options.add(newUserRoleOption(UserRole.OBSERVER));
            options.add(newUserRoleOption(UserRole.COORDINATOR));
            // Only admin can set ADMIN and GUEST role
            if (currentUser.isAdmin()) {
                options.add(newUserRoleOption(UserRole.ADMIN));
                options.add(newUserRoleOption(UserRole.GUEST));
            }
            userRoleSelectModel = new SelectModelImpl(null, options);
        }
        return userRoleSelectModel;
    }
    
    private OptionModel newUserRoleOption(UserRole role) {
        return new OptionModelImpl(role.getLabel(), role);
    }

    public String getRoleReadOnly() {
        String text = "";
        if (userEdited.isReadOnly(role)) {
            text = "lecture seule";
        }
        return text;
    }

    @Log
    void onChangeFromUserRole(String value) {
        if (StringUtils.isNotEmpty(value)) {
            userRole = UserRole.valueOf(value);
        } else {
            userRole = null;
        }
    }

    @Log
    void onChangeFromReadOnly(boolean value) {
        readOnly = value;
    }

    @Log
    Object onActionFromAddRole() {
        if (logger.isDebugEnabled()) {
            logger.debug("Add userRole : " + userRole +
                    " (readOnly=" + readOnly + ")");
        }
        if (userRole != null) {
            userEdited.addUserRole(userRole, readOnly);
        }
        // Reset fields
        userRole = null;
        readOnly = false;
        // Refresh the zone
        refreshUserRoleZone = true;
        return userRoleZone;
    }

    @Log
    Object onActionFromRemoveRole(int roleIndex) {
        UserRole roleToRemove = userEdited.getUserRoles().get(roleIndex);
        userEdited.removeUserRole(roleToRemove);
        // Refresh the zone
        refreshUserRoleZone = true;
        return userRoleZone;
    }
        
}
