/*
 * *##% 
 * SuiviObsmer :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.suiviobsmer.ui.pages;


import fr.ifremer.suiviobsmer.SuiviObsmerBusinessException;
import fr.ifremer.suiviobsmer.SuiviObsmerContext;
import fr.ifremer.suiviobsmer.SuiviObsmerException;
import fr.ifremer.suiviobsmer.bean.FacadeRow;
import fr.ifremer.suiviobsmer.bean.ImportResults;
import fr.ifremer.suiviobsmer.entity.Company;
import fr.ifremer.suiviobsmer.entity.FishingZone;
import fr.ifremer.suiviobsmer.entity.SampleMonth;
import fr.ifremer.suiviobsmer.entity.SampleRow;
import fr.ifremer.suiviobsmer.entity.WaoUser;
import fr.ifremer.suiviobsmer.services.ServiceSampling;
import fr.ifremer.suiviobsmer.ui.base.SuiviObsmerPage;
import fr.ifremer.suiviobsmer.ui.components.Layout;
import java.text.DateFormat;
import java.text.NumberFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.List;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.nuiton.util.PeriodDates;
import org.slf4j.Logger;

/**
 * SampingPlan
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois
 * @version $Revision: 305 $
 *
 * Mise a jour: $Date: 2010-02-01 19:16:04 +0100 (lun., 01 févr. 2010) $
 * par : $Author: fdesbois $
 */
@IncludeStylesheet("context:css/sampling.css")
public class SamplingPlan implements SuiviObsmerPage {

    @Override
    public boolean isOnlyForAdmin() {
        return false;
    }

    @Inject
    private Logger log;

    @Inject
    private ServiceSampling serviceSampling;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private WaoUser user;

    /**
     * Page initialization
     */
    void setupRender() {
        periodBegin = getPeriod().getFromDate();
        periodEnd = getPeriod().getThruDate();
    }

    /**
     * Get current user from Layout component
     * @return current user
     */
//    public User getUser() {
//        return layout.getCurrentUser();
//    }

    /**************************** IMPORT (ADMIN) *******************************/

    /**
     * Fichier CSV contenant un plan d'échantillonnage
     */
    @Property
    private UploadedFile samplingPlanCsvFile;

    @Log
    void onSuccessFromImportSamplingPlan() throws SuiviObsmerException {
        try {
            ImportResults result = serviceSampling.importSamplingPlanCsv(samplingPlanCsvFile.getStream());
            layout.getFeedBack().addInfo(result.getNbRowsImported() + " lignes du plan importés,  " +
                    result.getNbRowsRefused() + " refusés.");
            for (String error : result.getErrors()) {
                layout.getFeedBack().addInfo(error);
            }
        } catch (SuiviObsmerBusinessException eee) {
            layout.getFeedBack().addError(eee.getMessage());
        }
    }

    /**************************** PERIOD SELECTION *****************************/

    @Persist
    private PeriodDates period;

    @Property
    private Date periodBegin;

    @Property
    private Date periodEnd;

    public PeriodDates getPeriod() {
        if (period == null) {
            period = PeriodDates.createMonthsPeriodFromToday(11);
        }
        return period;
    }

    void onSuccessFromPeriodForm() {
        if (periodBegin != null && periodEnd != null) {
            period.setFromDate(periodBegin);
            period.setThruDate(periodEnd);
        }
    }

    /**************************** MAIN TABLE ***********************************/

    /** ------------------------- DATA -------------------------------------- **/

    /**
     * Main data for samplingPlan : List of SampleRow ordered by FishingZone.
     */
    private List<SampleRow> data;

    private List<Date> months;

    @Property
    private Date month;

    /**
     * Current SampleRow from loop
     */
    @Property
    private SampleRow row;

    @Property
    private String currentFacadeName;

    @Property
    private String currentSectorName;

    /**
     * Return List of SampleRow from suiviobsmer-business
     * @return List of SampleRow
     * @throws SuiviObsmerException
     */
    public List<SampleRow> getData() throws SuiviObsmerException {
        if (data == null) {            
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getSampleRowsOrderedByFishingZone]");
            }
            Company company = null;
            if (!user.getAdmin()) {
                company = user.getCompany();
            }
            FacadeRow facade = serviceSampling.getSampleRowsOrderedByFishingZone(getPeriod(), company);
            data = facade.getValues();
        }
        return data;
    }

    public List<Date> getMonths() {
        if (months == null) {
            months = getPeriod().getMonths();
        }
        return months;
    }

    public boolean isFacadeChanged() {
        if (!row.getFacade().equals(currentFacadeName)) {
            currentFacadeName = row.getFacade();
            return true;
        }
        return false;
    }

    public boolean isSectorChanged() {
        if (!row.getSectors().equals(currentSectorName)) {
            currentSectorName = row.getSectors();
            return true;
        }
        return false;
    }

    public String getCompanyName() {
        return row.getCompany() != null ? row.getCompany().getName() : "";
    }

    public boolean isCurrentMonth() {
        String currentStr = getDateFormat().format(new Date());
        String monthStr = getDateFormat().format(month);
        return currentStr.equals(monthStr);
    }

    public Integer getNbTidesExpected() {
        SampleMonth sampleMonth = row.getSampleMonth(month);
        if (sampleMonth != null) {
            return sampleMonth.getExpectedTidesValue();
        }
        return null;
    }

    public String getFishingZones() {
        String result = "";
        for (FishingZone zone : row.getFishingZone()) {
            result += zone.getDistrictCode() + ", ";
        }
        return result.substring(0, result.length()-2);
    }

    public Integer getNbTidesReal() {
        SampleMonth sampleMonth = row.getSampleMonth(month);
        if (sampleMonth != null) {
            return sampleMonth.getRealTidesValue();
        }
        return null;
    }

    public boolean hasNbTidesReal() {
        Date current = new Date();
        boolean validMonth = month.before(current) || isCurrentMonth();
        return validMonth && getNbTidesReal() != null;
    }

    public int getTotalTidesExpected() {
        int total = 0;
        for (Date currentMonth : getMonths()) {
            SampleMonth sampleMonth = row.getSampleMonth(currentMonth);
            if (sampleMonth != null) {
                total += sampleMonth.getExpectedTidesValue();
            }
        }
        return total;
    }

    public int getTotalTidesReal() {
        int total = 0;
        for (Date currentMonth : getMonths()) {
            SampleMonth sampleMonth = row.getSampleMonth(currentMonth);
            if (sampleMonth != null) {
                total += sampleMonth.getRealTidesValue();
            }
        }
        return total;
    }

    public String getTotalPercentage() {
        double result = 0;
        if (getTotalTidesExpected() > 0) {
            result = (double)getTotalTidesReal() / (double)getTotalTidesExpected();
        }
        return NumberFormat.getPercentInstance().format(result);
    }

    public NumberFormat getNumberFormat() {
        return NumberFormat.getNumberInstance();
    }

    /** ------------------------- HTML & STYLE ------------------------------ **/

    @Property
    private int rowIndex;

    public int getNbColumnsForProfession() {
        // code, program.name, fishingZone.districts, profession.code, profession.libelle
        int fixed = 5;
        if (user.getAdmin()) {
            // company, program.periodBegin, program.periodEnd, fishingZonesInfos
            // profession.meshSize, profession.size, profession.other, profession.species
            fixed += 8;
        }
        return fixed;
    }

    public int getNbColumnsForMonths() {
        return getMonths().size() + 1;
    }

    public int getNbColumnsForOther() {
        return 3;
    }

    public int getNbColumnsTotal() {
        return getNbColumnsForProfession() +
                getNbColumnsForMonths() + getNbColumnsForOther();
    }

    public String getMainClass() {
        return user.getAdmin() ? "admin" : "user";
    }

    public String getParityClass() {
        return rowIndex % 2 == 0 ? "even" : "odd";
    }

    public String getActionsClass() {
        return user.getAdmin() ? "width100" : "width30";
    }

    public String getRealTidesClass() {
        String result = "real-warning";
        if (getNbTidesReal() < getNbTidesExpected()) {
            result += "-inf";
        } else if (getNbTidesReal() > getNbTidesExpected()) {
            result += "-sup";
        }
        return result;
    }

    public String getMonthCurrentClass() {
        return isCurrentMonth() ? "selected" : "";
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("MM/yyyy");
    }

    /** ------------------------- ACTIONS ----------------------------------- **/

    /**
     * Used to filter period using dates from the program selected in table.
     * @param rowIndex index of the row in the table
     * @throws SuiviObsmerException for a data problem
     */
    void onActionFromFilterPeriodDates(int rowIndex) throws SuiviObsmerException {
        row = getData().get(rowIndex);
        period.setFromDate(row.getPeriodBegin());
        period.setThruDate(row.getPeriodEnd());
    }

    void onActionFromDeleteSampleRow(int rowIndex) throws SuiviObsmerException {
        row = getData().get(rowIndex);
        serviceSampling.deleteSampleRow(row);
    }

    public boolean canAccessBoats() {
        // Test if the row isFinished with a gap of 1 month (today - 1 month)
        return !row.isFinished(-1);
    }
        
}
