/*
 * *##% 
 * SuiviObsmer :: Web Interface
 * Copyright (C) 2009 - 2010 Ifremer
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * ##%*
 */

package fr.ifremer.suiviobsmer.ui.pages;

import fr.ifremer.suiviobsmer.SuiviObsmerContext;
import fr.ifremer.suiviobsmer.SuiviObsmerException;
import fr.ifremer.suiviobsmer.bean.BoatFilterImpl;
import fr.ifremer.suiviobsmer.entity.Boat;
import fr.ifremer.suiviobsmer.bean.BoatFilter;
import fr.ifremer.suiviobsmer.bean.CompanyBoatInfos;
import fr.ifremer.suiviobsmer.entity.BoatInfos;
import fr.ifremer.suiviobsmer.entity.Company;
import fr.ifremer.suiviobsmer.entity.Contact;
import fr.ifremer.suiviobsmer.entity.ElligibleBoat;
import fr.ifremer.suiviobsmer.entity.WaoUser;
import fr.ifremer.suiviobsmer.entity.Profession;
import fr.ifremer.suiviobsmer.entity.SampleRow;
import fr.ifremer.suiviobsmer.services.ServiceBoat;
import fr.ifremer.suiviobsmer.services.ServiceReferential;
import fr.ifremer.suiviobsmer.services.ServiceSampling;
import fr.ifremer.suiviobsmer.services.ServiceUser;
import fr.ifremer.suiviobsmer.ui.base.AbstractFilteredPage;
import fr.ifremer.suiviobsmer.ui.data.BusinessUtils;
import fr.ifremer.suiviobsmer.ui.base.GenericSelectModel;
import fr.ifremer.suiviobsmer.ui.base.SuiviObsmerPage;
import fr.ifremer.suiviobsmer.ui.components.FeedBack;
import fr.ifremer.suiviobsmer.ui.components.Layout;
import fr.ifremer.suiviobsmer.ui.data.BoatDataSource;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.List;
import java.util.zip.GZIPInputStream;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.tapestry5.Block;
import org.apache.tapestry5.EventContext;
import org.apache.tapestry5.PersistenceConstants;
import org.apache.tapestry5.StreamResponse;
import org.apache.tapestry5.annotations.IncludeStylesheet;
import org.apache.tapestry5.annotations.InjectComponent;
import org.apache.tapestry5.annotations.InjectPage;
import org.apache.tapestry5.annotations.Log;
import org.apache.tapestry5.annotations.Persist;
import org.apache.tapestry5.annotations.Property;
import org.apache.tapestry5.annotations.SessionState;
import org.apache.tapestry5.corelib.components.Zone;
import org.apache.tapestry5.ioc.annotations.Inject;
import org.apache.tapestry5.ioc.services.PropertyAccess;
import org.apache.tapestry5.services.Response;
import org.apache.tapestry5.upload.services.UploadedFile;
import org.slf4j.Logger;

/**
 * Boats
 *
 * Created: 9 nov. 2009
 *
 * @author fdesbois
 * @version $Revision: 294 $
 *
 * Mise a jour: $Date: 2010-01-29 18:21:13 +0100 (ven., 29 janv. 2010) $
 * par : $Author: fdesbois $
 */
@IncludeStylesheet("context:css/boats.css")
public class Boats extends AbstractFilteredPage implements SuiviObsmerPage {

    @Override
    public boolean isOnlyForAdmin() {
        return false;
    }

    @Inject
    private Logger log;

    @InjectComponent
    private Layout layout;

    @SessionState
    @Property
    private WaoUser user;

    @Inject
    private ServiceReferential serviceReferential;

    @Inject
    private ServiceSampling serviceSampling;

    @Inject
    private ServiceBoat serviceBoat;

    @Inject
    private PropertyAccess propertyAccess;

    @Property
    private String sampleRowContextId;

    /** Selected boat immatriculation for showing boatInfos */
    @Property
    private Integer boatSelectedImmatriculation;

    @InjectComponent
    private FeedBack filterFeedback;

    void setupRender() throws SuiviObsmerException {
        sampleRow = null;
        getSampleRow();
        // Profession select reset
        professionSelectModel = null;
        getProfessionSelectModel();

        // Manage sampleRow from context
        if (isSampleRowExists()) {
            // Filters will be set and displayed
            getFilter().setSampleRow(sampleRow);
            filtersHidden = false;
            boats = null;
            filterFeedback.addInfo("Vous êtes en cours de sélection d'un navire pour la ligne du plan " + sampleRow.getCode());
            professionId = getSampleRow().getProfession().getTopiaId();
        }

        // Initialize filters using AbstractFilteredPage superclass
        initSelectFilters(false, false, false);

        companyBoatInfos = null;
        //boatSelectedImmatriculation = null;
    }

    void onActivate(EventContext ec) {
        if (ec.getCount() > 0) {
            sampleRowContextId = ec.get(String.class, 0);
            if (ec.getCount() > 1) {
                boatSelectedImmatriculation = ec.get(Integer.class, 1);
            }
        }
    }

    Object[] onPassivate() {
        return new Object[] { sampleRowContextId, boatSelectedImmatriculation };
    }

    /**************************** Import Forms ********************************/

    /** Csv file for boats import */
    @Property
    private UploadedFile boatsCsvFile;

    @Log
    void onSuccessFromImportBoatsForm() {
        //importBoatsForm.clearErrors();
        try {
            int[] result = serviceBoat.importBoatCsv(boatsCsvFile.getStream());
            // Suppress persitant list of boats
            boats = null;
            layout.getFeedBack().addInfo(result[0] + " navires importés dont " + result[1] + " nouveaux");
        } catch (SuiviObsmerException eee) {
            layout.getFeedBack().addError(eee.getMessage());
        }
    }

    /** Csv file for activityCalendar import, can be a Gzip file */
    @Property
    private UploadedFile activityCalendarsCsvFile;

    private InputStream activityCalendarLogFile;

    protected static final String GZIP_MIMETYPE = "application/x-gzip";

    InputStream getActivityCalendarLogFile() throws FileNotFoundException {
        if (activityCalendarLogFile == null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getActivityCalendarLogFile]");
            }
            activityCalendarLogFile = serviceBoat.getActivityCalendarLogFile();
        }
        return activityCalendarLogFile;
    }

    public boolean isLogFileExists() {
        try {
            getActivityCalendarLogFile();
            return true;
        } catch (FileNotFoundException eee) {
            if (log.isInfoEnabled()) {
                log.info("Aucun fichier de log existant pour les calendriers d'activité");
            }
            return false;
        }
    }

    @Log
    void onSuccessFromImportActivityCalendarsForm() throws SuiviObsmerException, IOException {
        InputStream input = activityCalendarsCsvFile.getStream();
        if (log.isDebugEnabled()) {
            log.debug("Content type : " + activityCalendarsCsvFile.getContentType());
        }
        if (activityCalendarsCsvFile.getContentType().equals(GZIP_MIMETYPE)) {
            if (log.isDebugEnabled()) {
                log.debug("Gzip file");
            }
            input = new GZIPInputStream(input);
        }
        serviceBoat.importActivityCalendarCsv(input);
        companyBoatInfos = null;
    }
    
    public StreamResponse onActionFromShowActivityCalendarLogFile() {
        return new StreamResponse() {

            @Override
            public String getContentType() {
                return "text/plain;charset=utf-8";
            }

            @Override
            public InputStream getStream() throws IOException {
                return getActivityCalendarLogFile();
            }

            @Override
            public void prepareResponse(Response response) {
            }
        };
    }

    /**************************** Filters Form *****************************************/

    /** Filters to apply on boats list */
    @Persist
    private BoatFilter boatFilter;

    /** Current sampleRow from context */
    @Persist
    private SampleRow sampleRow;

    /** Profession select */
    @Persist
    private GenericSelectModel<Profession> professionSelectModel;

    @Property
    private String professionId;

    /** Used to manage view of filters zone */
    @Persist
    private boolean filtersHidden;

    @InjectComponent
    private Zone filtersZone;

    /**
     * Filters to apply on boats list. Method used by AbstractFilteredPage superclass. 
     *
     * @return the current BoatFilter to apply
     * @throws SuiviObsmerException
     * @see AbstractFilteredPage
     */
    @Override
    public BoatFilter getFilter() throws SuiviObsmerException {
        if (boatFilter == null) {
            if (log.isDebugEnabled()) {
                log.debug("Create filter");
            }
            boatFilter = new BoatFilterImpl();
            // Only rows which are not finished nearly one month
            boatFilter.setNbMonthFinishedFromToday(-1);
        }
        return boatFilter;
    }

    @Override
    protected boolean isAvailableDataForFiltersOnly() {
        return true;
    }

    /**
     * Retrieve sampleRow from business with id from url context. The sampleRowId from context is set by SamplingPlan page.
     * When the sampleRow is not null, the filters are disabled to optimize searching boats for this row.
     *
     * @return the sampleRow from business
     * @throws SuiviObsmerException
     */
    public SampleRow getSampleRow() throws SuiviObsmerException {
        if (sampleRow == null && !StringUtils.isEmpty(sampleRowContextId)) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getSampleRow]");
            }
            sampleRow = serviceSampling.getSampleRow(sampleRowContextId);
        }
        return sampleRow;
    }

    public boolean isSampleRowExists() throws SuiviObsmerException {
        return getSampleRow() != null;
    }

    public GenericSelectModel<Profession> getProfessionSelectModel() throws SuiviObsmerException {
        if (professionSelectModel == null) {           
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getProfessions]");
            }
            List<Profession> professions = serviceReferential.getProfessions();
            professionSelectModel = new GenericSelectModel<Profession>(professions, Profession.class,
                "code", "topiaId", propertyAccess);
        }
        return professionSelectModel;
    }

    public boolean isFiltersHidden() {
        return filtersHidden;
    }

    public String getHiddenStyle() {
        return isFiltersHidden() ? "hidden" : "";
    }

    Block onActionFromShowFilters() {
        filtersHidden = false;
        return filtersZone.getBody();
    }

    Block onActionFromHideFilters() {
        filtersHidden = true;
        return filtersZone.getBody();
    }

    /**
     * EVENT_SELECTED :: reset submit button for filtersForm. Reset filters and sampleRow from context.
     */
    void onSelectedFromReset() {
        // Suppress persistant boat filter and sampleRowId in url context.
        boatFilter = null;
        sampleRowContextId = null;
    }

    /**
     * EVENT_SELECTED :: search sumit button for filtersForm. Get profession from selectModel to add it in filters.
     * Other filters are managed by AbstractFilteredPage superclass.
     *
     * @throws SuiviObsmerException for business errors
     * @see AbstractFilteredPage#onSelectedFromSearch() 
     */
    @Override
    public void onSelectedFromSearch() throws SuiviObsmerException {
        super.onSelectedFromSearch();
        if (log.isDebugEnabled()) {
            log.debug("FILTER : facadeName=" + boatFilter.getFacadeName());
            log.debug("FILTER : sectorName=" + boatFilter.getSectorName());
            log.debug("FILTER : sampleRow=" + boatFilter.getSampleRow());
            log.debug("FILTER : profession=" + boatFilter.getProfession());
            log.debug("FILTER : boatName=" + boatFilter.getBoatName());
            log.debug("FILTER : boatImmatriculation=" + boatFilter.getBoatImmatriculation());
            log.debug("FILTER : boatDistrictCode=" + boatFilter.getBoatDistrictCode());
            log.debug("FILTER : company=" + boatFilter.getCompany());
        }
        Profession profession = getProfessionSelectModel().findObject(professionId);
        boatFilter.setProfession(profession);
    }

    /**
     * EVENT_SUCCESS :: for filtersForm. Reset persitant data to have new data using filters selected.
     *
     * @return the filtersZone if it's a refresh data for select, or the page otherwise
     * @throws SuiviObsmerException for business errors
     */
    Object onSuccessFromFiltersForm() throws SuiviObsmerException {
        if (isEdited()) {
            return filtersZone.getBody();
        }
        // Suppress persistant boats list to get new one with filter
        boats = null;
        // Suppress persistant immatriculation for boat selected
        boatSelectedImmatriculation = null;
        // Suppress persistant boat informations for boat selected
        companyBoatInfos = null;
        return this;
    }

    /**************************** Boats List *****************************************/

    /** DataSource containing the boats */
    @Persist
    private BoatDataSource boats;

    /** Selected boat immatriculation for showing boatInfos */
//    @Property
//    @Persist
//    private Integer boatSelectedImmatriculation;

    /** Current boat in the Grid */
    @Property
    private Boat boat;

    /** Used to change style a line on two */
    private boolean even = true;

    /** Used to refresh boats Grid */
    @InjectComponent
    private Zone boatsZone;

    /**
     * Get dataSource which contains current boats to show on the page. Depends on filters sets by user.
     * BoatDataSource uses serviceBoat to retrieve boats with filters.
     *
     * @return the BoatDataSource which contains the boats map
     * @throws SuiviObsmerException
     * @see BoatDataSource
     */
    public BoatDataSource getBoats() throws SuiviObsmerException {
        if (boats == null) {
            boats = new BoatDataSource(getFilter(), serviceBoat);
        }
        return boats;
    }

    /**
     * Initialized row CSS style depends on selection in the list. The style is also changed a line on two.
     *
     * @return the CSS style class to used for the current row in the Grid
     */
    public String getRowStyle() {
        String result = "";
        even = !even;        
        if (boatSelectedImmatriculation != null &&
                boat.getImmatriculation() == boatSelectedImmatriculation) {
            result = "selected";
        } else if (!boat.getActive()) {
            result = "refused";
        } else {
            result = even ? "even" : "odd";
        }
        return result;
    }

    public DateFormat getDateFormat() {
        return new SimpleDateFormat("dd/MM/yyyy");
    }
    
    public boolean canCreateNewContactFromList() throws SuiviObsmerException {
        if (!user.getAdmin() && isSampleRowExists()) {
            return boat.canCreateContact(user.getCompany());
        }
        return false;
    }

    Block onActionFromShowBoatInfos(Integer boatImma) throws SuiviObsmerException {
        boatSelectedImmatriculation = boatImma;
        // Suppress persistant boat informations
        companyBoatInfos = null;

        if (user.getAdmin() && getCompany() != null) {
            companySelectedId = getCompany().getTopiaId();
        }
        return boatsZone.getBody();
    }

    /**************************** Boat selected Infos *************************/

    @Persist
    private CompanyBoatInfos companyBoatInfos;

    @InjectComponent
    private Zone boatInfosZone;

    @Inject
    private Block displayBoatInfos;

    @Inject
    private Block editBoatInfos;

    @Property
    private ElligibleBoat elligibleBoat;

    @Property
    private String boatInfosSampleRowId;

    private boolean boatInfosEditable;

    private Date boardingFromDate;

    private GenericSelectModel<Company> companies;

    @Inject
    private ServiceUser serviceUser;

    @Property
    @Persist(PersistenceConstants.FLASH)
    private String companySelectedId;

    private Company company;

    /*@Persist
    private SuiviObsmerPropertyChangeListener propertyChange;*/

    /** /////////////////////// GLOBAL TO THE BLOCK /////////////////////// **/

    public boolean isActivityCalendarImportRun() {
        return SuiviObsmerContext.isActivityCalendarImportRun();
    }

    public Block getActiveBoatInfosBlock() {
        if (boatInfosEditable) {
            return editBoatInfos;
        }
        return displayBoatInfos;
    }

    public CompanyBoatInfos getCompanyBoatInfos() throws SuiviObsmerException {
        if (companyBoatInfos == null && boatSelectedImmatriculation != null && getCompany() != null) {
            if (log.isInfoEnabled()) {
                log.info("BUSINESS REQUEST [getElligibleBoats]");
            }            
            companyBoatInfos = serviceBoat.getCompanyBoatInfos(boatSelectedImmatriculation, getCompany());
        }
        return companyBoatInfos;
    }

    /**
     * Retrieve companies select model for admin user. Only active companies
     * will be used for this model.
     *
     * @return the GenericSelectModel<Company>
     * @throws SuiviObsmerException
     */
    public GenericSelectModel<Company> getCompanies() throws SuiviObsmerException {
        if (companies == null) {
            if (log.isDebugEnabled()) {
                log.debug("BUSINESS REQUEST [getCompanies]");
            }
            List<Company> results = serviceUser.getCompanies(true);
            companies = new GenericSelectModel<Company>(results, Company.class, "name","topiaId",propertyAccess);
        }
        return companies;
    }

    /**
     * Get the current company. The user one if it's not an admin or the
     * selected one otherwise.
     *
     * @return the current Company
     * @throws SuiviObsmerException
     */
    public Company getCompany() throws SuiviObsmerException {
        if (company == null) {
            if (StringUtils.isNotEmpty(companySelectedId)) {
                company = getCompanies().findObject(companySelectedId);
            } else if (!user.getAdmin()) {
                company = user.getCompany();
            }
        }
        return company;
    }

    /**
     * Used to get the selected boat.
     *
     * @return the selected boat from the grid
     * @throws SuiviObsmerException
     */
    public Boat getBoatSelected() throws SuiviObsmerException {
        return getBoats().get(boatSelectedImmatriculation);
    }

    /**
     * Date for calculate boardings done for the selected boat. All boardings (number of contacts) will be calculated from this date.
     * Initialized as 12 months from today, but can be set by user.
     *
     * @return the Date used to calculate boardings done on the selected boat
     */
    public Date getBoardingFromDate() {
        if (boardingFromDate == null) {
            Calendar calendar = new GregorianCalendar();
            calendar.add(Calendar.MONTH, -12);
            boardingFromDate = calendar.getTime();
        }
        return boardingFromDate;
    }

    /**
     * Used by Tapestry to set the boardingFromDate from calculateBoardings form.
     *
     * @param boardingFromDate used to calculate boardings done on the selected boat
     */
    public void setBoardingFromDate(Date boardingFromDate) {
        this.boardingFromDate = boardingFromDate;
    }

    public BoatInfos getBoatInfos() throws SuiviObsmerException {
        if (getCompanyBoatInfos() == null) {
            return null;
        }
        return getCompanyBoatInfos().getBoatInfos();
    }

    /**
     * Get CSS style class to apply on lastContact depends on its state.
     *
     * @return the CSS style to apply on contact block
     * @throws SuiviObsmerException
     * @see BusinessUtils#getContactStyle(Contact, boolean)
     */
    public String getlastContactStyle() throws SuiviObsmerException {
        Contact contact = getCompanyBoatInfos().getLastContact();
        return BusinessUtils.getContactStyle(contact, user.getAdmin());
    }

    /** /////////////////////// FOR EACH ELLIGIBLEBOAT ROW ///////////////// **/

    public String getElligibleRowInfos() {
        return BusinessUtils.getTooltipSampleRow(elligibleBoat.getSampleRow());
    }

    public boolean isElligibleBoatCompanyActiveFalse() {
        return BooleanUtils.isFalse(elligibleBoat.getCompanyActive());
    }

    public boolean canCreateNewContactFromElligibleBoat() throws SuiviObsmerException {
        if (!user.getAdmin() && !isElligibleBoatCompanyActiveFalse()) {  
            Boat currentBoat = getBoatInfos().getBoat();
            return currentBoat.canCreateContact(user.getCompany());
        }
        return false;
    }

    public String getElligibleRowStyle() {
        boolean condition1 = elligibleBoat.getCompanyActive() == null && !elligibleBoat.getGlobalActive();
        return condition1 || isElligibleBoatCompanyActiveFalse() ? "line-through" : "";
    }

    /** /////////////////////// ACTIONS //////////////////////////////////// **/

    Block onSuccessFromCompanySelectForm() throws SuiviObsmerException {
        companyBoatInfos = null;
        getCompanyBoatInfos();   
        return boatInfosZone.getBody();
    }

    Block onActionFromEditBoatInfos() throws SuiviObsmerException {
        boatInfosEditable = true;
        // WARNING :: Reset sampleRow select for edition mode, no limit for
        // showing sampleRows instead of filters
        getFilter().setNbMonthFinishedFromToday(null);
        resetSampleRowSelect();
        getFilter().setNbMonthFinishedFromToday(-1);
        return boatInfosZone.getBody();
    }

    Block onActionFromCancelEditBoatInfos() throws SuiviObsmerException {
        /*if (propertyChange.isChanged()) {
            if (log.isDebugEnabled()) {
                log.debug("Change property");
            }
        }*/
        
        // Set boatSelectedImmatriculation for boatInfos to reinitialize it (next getBoatInfos() call)
        //boatSelectedImmatriculation = getBoatInfos().getBoat().getImmatriculation();
        // Suppress boatInfos to get the one from serviceBoat (not updated yet)
        companyBoatInfos = null;
        return boatInfosZone.getBody();
    }

    void onSelectedFromAddBoatInfosSampleRow() throws SuiviObsmerException {
        // We stay in edition mode
        boatInfosEditable = true;
        if (StringUtils.isNotEmpty(boatInfosSampleRowId)) {
            SampleRow row = getSampleRowSelectModel().findObject(boatInfosSampleRowId);
            companyBoatInfos.setNewElligibleBoat(row);
        }
    }

    void onSelectedFromRemoveBoatInfosSampleRow(String sampleRowCode) throws SuiviObsmerException {
        boatInfosEditable = true;
        companyBoatInfos.removeElligibleBoat(sampleRowCode);
    }

    void onSelectedFromActiveBoatInfosSampleRow(String sampleRowCode) throws SuiviObsmerException {
        boatInfosEditable = true;
        companyBoatInfos.activeElligibleBoat(sampleRowCode);
    }

    Block onSuccessFromBoatInfosForm() throws SuiviObsmerException {
        if (!boatInfosEditable) {
            // Save data
            serviceBoat.createUpdateCompanyBoatInfos(companyBoatInfos);
        }
        return boatInfosZone.getBody();
    }

    Block onSuccessFromCalculateBoardings() throws SuiviObsmerException {
        return boatInfosZone.getBody();
    }

    /**************************** Create new contact **************************/

    @InjectPage
    private Contacts contacts;

    Object onActionFromAddNewContactFromBoat(int boatImmatriculation) throws SuiviObsmerException {
        // Get selected sampleRow
        sampleRow = getSampleRow();
        // Get boat from list
        boat = getBoats().get(boatImmatriculation);
        contacts.createNewContact(boat, sampleRow);
        return contacts;
    }

    Object onActionFromAddNewContactFromSampleRow(String sampleRowCode) throws SuiviObsmerException {
        // Get sampleRow from elligibleBoat list
        ElligibleBoat elligible = getCompanyBoatInfos().getElligibleBoat(sampleRowCode);
        sampleRow = elligible.getSampleRow();
        contacts.createNewContact(getBoatSelected(), sampleRow);
        return contacts;
    }

    /**************************** Go to last ActivityCalendar *****************/

    @InjectPage
    private BoatActivityCalendar calendarPage;

    public boolean hasActivityCalendar() throws SuiviObsmerException {
        return getBoatSelected().sizeActivityCalendar() != 0;
    }

    Object onActionFromShowLastActivityCalendar() throws SuiviObsmerException {
        calendarPage.setBoat(getBoatSelected());
        return calendarPage;
    }
}
