/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: VradiTask.java 1756 2010-11-10 13:58:25Z sletellier $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.5.1/vradi-swing/src/main/java/com/jurismarches/vradi/ui/task/VradiTask.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.task;

import static org.nuiton.i18n.I18n._;

import java.awt.Component;
import java.awt.Cursor;

import com.jurismarches.vradi.ui.helpers.UIHelper;
import javax.swing.JFrame;
import javax.swing.SwingWorker;

import jaxx.runtime.JAXXContext;
import jaxx.runtime.SwingUtil;
import jaxx.runtime.swing.ErrorDialogUI;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * @author sletellier
 */
public abstract class VradiTask<T> extends SwingWorker <T,Void> {

    static private final Log log = LogFactory.getLog(VradiTask.class);

    protected JAXXContext context;
    protected String i18nKey;
    protected AbstractProgressBarUI pb = null;
    protected boolean showErrorDialog = true;

    public VradiTask(JAXXContext context){
        this.context = context;
        this.i18nKey = null;
        this.pb = new ProgressBarUI(context);
    }

    public VradiTask(JAXXContext context, boolean showProgressBar){
        this.context = context;
        this.i18nKey = null;
        if (showProgressBar){
            this.pb = new ProgressBarUI(context);
        }
    }

    public VradiTask(JAXXContext context, String i18nKey){
        this.context = context;
        this.i18nKey = i18nKey;
        this.pb = new ProgressBarUI(context);
    }

    public VradiTask(JAXXContext context, boolean showProgressBar, String i18nKey){
        this.context = context;
        this.i18nKey = i18nKey;
        if (showProgressBar){
            this.pb = new ProgressBarUI(context);
        }
    }

    public VradiTask(JAXXContext context, AbstractProgressBarUI pb, String i18nKey){
        this.context = context;
        this.i18nKey = i18nKey;
        this.pb = pb;
    }

    public VradiTask(JAXXContext context, AbstractProgressBarUI pb, String i18nKey, boolean showErrorDialog){
        this.context = context;
        this.i18nKey = i18nKey;
        this.pb = pb;
        this.showErrorDialog = showErrorDialog;
    }

    /*
         * Main task. Executed in background thread.
         */
    @Override
    protected T doInBackground() {
        Component component = (Component)context;
        component.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        if (pb != null) {
            if (i18nKey != null){
                pb.setTitle(getTitleMsg());
                pb.setMsg(getMsg());
            }
            JFrame frame = context.getContextValue(JFrame.class, "parent");
            if (frame != null) {
                SwingUtil.center(frame, pb);
            } else {
                SwingUtil.center(component, pb);
            }
            pb.pack();
            pb.setVisible(true);
            UIHelper.registerComponentToSaveDispositionConfig(pb);
        }
        try {
            return doAction();
        } catch (Exception eee) {
            if (log.isErrorEnabled()) {
                log.error(getErrorMsg(), eee);
                if (showErrorDialog) {
                    ErrorDialogUI.showError(eee);
                }
                doWhenError(eee);
            }
        }
        return null;
    }
    /*
         * Executed in event dispatching thread
         */
    @Override
    protected void done() {
        try {
            doWhenDone();
        } catch (Exception eee) {
            log.error(getErrorMsg(), eee);
            if (showErrorDialog) {
                ErrorDialogUI.showError(eee);
            }
        } finally {
            if (pb != null) {
                pb.dispose();
            }
            ((Component)context).setCursor(Cursor.getPredefinedCursor(Cursor.DEFAULT_CURSOR));
        }
    }

    protected String getTitleMsg(){
        // FIXME EC20100910 ce n'est pas du i18n correct (pas parseable)
        return _(i18nKey + ".title");
    }

    protected String getMsg(){
        // FIXME EC20100910 ce n'est pas du i18n correct (pas parseable)
        return _(i18nKey + ".msg");
    }

    protected String getErrorMsg(){
        if (i18nKey != null) {
            
            // FIXME EC20100910 ce n'est pas du i18n correct (pas parseable)
            return _(i18nKey + ".error");
        }
        return _("vradi.error.executeTask");
    }

    public abstract T doAction() throws Exception;

    public void doWhenDone() throws Exception{
        // Do nothing
    }

    public void doWhenError(Exception eee){
        // Do nothing
    }

}
