/*
 * #%L
 * Vradi :: Swing
 * 
 * $Id: OfferListTableModel.java 1715 2010-10-27 19:21:28Z tchemit $
 * $HeadURL: svn+ssh://sletellier@labs.libre-entreprise.org/svnroot/vradi/vradi/tags/vradi-0.3.1/vradi-swing/src/main/java/com/jurismarches/vradi/ui/offer/models/OfferListTableModel.java $
 * %%
 * Copyright (C) 2009 - 2010 JurisMarches, Codelutin
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the 
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public 
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */
package com.jurismarches.vradi.ui.offer.models;

import java.awt.Component;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.DefaultTableCellRenderer;

import com.jurismarches.vradi.entities.XmlStream;
import jaxx.runtime.JAXXObject;
import jaxx.runtime.binding.DefaultJAXXBinding;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.i18n.I18n;
import org.nuiton.wikitty.FieldType;
import org.nuiton.wikitty.WikittyExtension;

import com.jurismarches.vradi.beans.FormPagedResult;
import com.jurismarches.vradi.entities.Form;
import com.jurismarches.vradi.entities.FormImpl;
import com.jurismarches.vradi.entities.Infogene;
import com.jurismarches.vradi.entities.InfogeneImpl;
import com.jurismarches.vradi.entities.Status;
import com.jurismarches.vradi.services.VradiService;
import com.jurismarches.vradi.ui.helpers.ToolTipHelper;
import org.nuiton.wikitty.WikittyProxy;
import org.nuiton.wikitty.WikittyService;
import org.nuiton.wikitty.WikittyServiceEvent;
import org.nuiton.wikitty.WikittyServiceListener;

import static org.nuiton.i18n.I18n._;

/**
 * OfferListTableModel is the data model for search results table.
 * Its column model is based on the Infogene fields.
 * 
 * @author letellier
 * @version $Revision: 1715 $ $Date: 2010-10-27 21:21:28 +0200 (mer., 27 oct. 2010) $
 */
public class OfferListTableModel extends AbstractTableModel implements WikittyServiceListener {

    static private final Log log = LogFactory.getLog(OfferListTableModel.class);

    private static final long serialVersionUID = 1L;
    
    public static final String PROPERTY_PAGE_TO_SHOW = "pageToShow";
    public static final String PROPERTY_NB_FORMS_PER_PAGE = "nbFormsPerPage";
    /**
     * serves as a unique binding for either PROPERTY_PAGE_TO_SHOW or
     * PROPERTY_NB_FORMS_PER_PAGE or the two.
     */
    public static final String PROPERTY_BINDING_CHANGE = "bindings";
    
    public static final String PROPERTY_TOTAL_FORMS = "totalFoundFormNb";
    public static final String PROPERTY_NB_PAGES = "nbPagesAsText";
    public static final String PROPERTY_LAST_PAGE = "lastPage";
    
    protected final PropertyChangeSupport propertyChangeSupport =
            new PropertyChangeSupport(this);
    protected final List<Column> columns = new ArrayList<Column>();
    
    protected FormPagedResult formPagedResult = new FormPagedResult();

    // Add cache to refresh table
    Map<String, Status> statusesCached = new HashMap<String, Status>();
    Map<String, XmlStream> streamsCached = new HashMap<String, XmlStream>();
    Map<String, Form> formsCached = new HashMap<String, Form>();
    Map<String, String> toolTipCached = new HashMap<String, String>();

    // Show thesaurus tooltip
    protected boolean showThesaurusToolTip = false;
    
    public OfferListTableModel() {
        this(false);
    }

    public OfferListTableModel(boolean showThesaurusToolTip) {
        // register each model on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, WikittyService.ServiceListenerType.ALL);

        initColumns();
        formPagedResult.setPageToShow(1);
        formPagedResult.setNbFormsToShow(10);
        this.showThesaurusToolTip = showThesaurusToolTip;
    }
    
    public OfferListTableModel(FormPagedResult formPage) {
        this(formPage, false);
    }

    public OfferListTableModel(FormPagedResult formPagedResult, boolean showThesaurusToolTip) {
        // register each model on wikitty service
        VradiService.getWikittyService().addWikittyServiceListener(this, WikittyService.ServiceListenerType.ALL);

        initColumns();
        formPagedResult.setPageToShow(formPagedResult.getPageToShow());
        formPagedResult.setNbFormsToShow(formPagedResult.getNbFormsToShow());
        setFormPagedResult(formPagedResult);
        this.showThesaurusToolTip = showThesaurusToolTip;
    }

    /**
     * Raises property change event for PROPERTY_TOTAL_FORMS,
     * PROPERTY_NB_PAGES, PROPERTY_LAST_PAGE.
     *
     * Plus, the method <code>AbstractTableModel.fireTableDataChanged()</code> is invoked.
     *
     * @param formPagedResult
     */
    public void setFormPagedResult(FormPagedResult formPagedResult) {
        if (formPagedResult == null) {
            throw new IllegalArgumentException("formPage is null");
        }

        this.formPagedResult = formPagedResult;

        // Put in cache
        formsCached.clear();
        WikittyProxy proxy = VradiService.getWikittyProxy();
        List<Form> forms = proxy.restore(Form.class, formPagedResult.getFormsIdsToShow());
        for (Form form : forms) {
            formsCached.put(form.getWikittyId(), form);
        }
        
        propertyChangeSupport.firePropertyChange(PROPERTY_TOTAL_FORMS, null,
                formPagedResult.getTotalFoundFormNb());
        
        propertyChangeSupport.firePropertyChange(PROPERTY_NB_PAGES, null,
                getNbPages());
        
        propertyChangeSupport.firePropertyChange(PROPERTY_LAST_PAGE, null,
                isLastPage());
        
        fireTableDataChanged();
    }
    
    public FormPagedResult getFormPagedResult() {
        return formPagedResult;
    }
    
    /**
     * Column.
     */
    static class Column {
        private static final String I18N_COLUMN_PREFIX = "vradi.offer.";
        
        final String i18name;
        final Class<?> columnClass;
        final String fqName;

        Column(String extension, String name, Class<?> columnClass) {
            this.fqName = extension + "." + name;
            this.columnClass = columnClass;
            this.i18name = I18n._(I18N_COLUMN_PREFIX + fqName);
        }
    }
    
    protected void initColumns() {
        // FIXME: retrieve extension from service

        // la liste des colonnes contient tout les champs de l'infogene
        List<String> fieldNames = new ArrayList<String>();
        for(String fieldName : InfogeneImpl.extensionInfogene.getFieldNames()) {
            fieldNames.add(Infogene.EXT_INFOGENE + "." + fieldName);
        }

        // plus datePeremption, datePub, xmlStreamURL
        fieldNames.add(Form.FQ_FIELD_FORM_DATEPUB);
        fieldNames.add(Form.FQ_FIELD_FORM_DATEPEREMPTION);
        fieldNames.add(Form.FQ_FIELD_FORM_XMLSTREAM);

        for (String fqFieldName : fieldNames) {
            int dot = fqFieldName.lastIndexOf(".");
            String extensionName = fqFieldName.substring(0, dot);
            String fieldName = fqFieldName.substring(dot + 1);
            WikittyExtension extension =
                    extensionName.equals(Infogene.EXT_INFOGENE) ?
                            InfogeneImpl.extensionInfogene :
                            FormImpl.extensionForm;

            FieldType fieldType = extension.getFieldType(fieldName);

            Class<?> columnClass;
            if (fieldType.getType() == FieldType.TYPE.BOOLEAN) {
                columnClass = Boolean.class;

            } else if (fieldType.getType() == FieldType.TYPE.DATE) {
                columnClass = Date.class;

            } else if (fieldType.getType() == FieldType.TYPE.NUMERIC) {
                columnClass = Double.class;

            } else if (fieldName.equals(Infogene.FIELD_INFOGENE_STATUS)) {
                columnClass = Status.class;

            } else if (fieldName.equals(Form.FIELD_FORM_XMLSTREAM)) {
                columnClass = XmlStream.class;

            } else {
                columnClass = String.class;
            }

            Column column = new Column(extensionName, fieldName,
                    columnClass);
            columns.add(column);
        }
    }
    
    public Integer getNbFormsPerPage() {
        return formPagedResult.getNbFormsToShow();
    }

    /**
     * Raises property change event for PROPERTY_NB_FORMS_PER_PAGE and
     * PROPERTY_PAGE_TO_SHOW, PROPERTY_BINDING_CHANGE.
     * 
     * As PROPERTY_BINDING_CHANGE property is changed, a new search is executed.
     * 
     * @param nbFormsPerPage
     */
    public void setNbFormsPerPage(Integer nbFormsPerPage) {
        Integer nbFormsToShow = formPagedResult.getNbFormsToShow();
        
        int topRow = nbFormsToShow * (formPagedResult.getPageToShow() - 1);
        topRow = topRow + 1;
        
        Integer pageToShow = (int) Math.ceil(topRow / nbFormsPerPage.doubleValue());
        if (pageToShow < 1) {
            pageToShow = 1;
        }
        
        formPagedResult.setNbFormsToShow(nbFormsPerPage);
        propertyChangeSupport.firePropertyChange(PROPERTY_NB_FORMS_PER_PAGE, nbFormsToShow,
                nbFormsPerPage);
        
        setPageToShow(pageToShow);
    }

    public int getPageToShow() {
        return formPagedResult.getPageToShow();
    }
    
    /**
     * Raises property change event for PROPERTY_PAGE_TO_SHOW, PROPERTY_BINDING_CHANGE.
     * 
     * As PROPERTY_BINDING_CHANGE property is changed, a new search is executed.
     * 
     * @param pageToShow
     */
    public void setPageToShow(int pageToShow) {
        int oldValue = formPagedResult.getPageToShow();
        formPagedResult.setPageToShow(pageToShow);
        
        propertyChangeSupport.firePropertyChange(PROPERTY_PAGE_TO_SHOW, oldValue,
                pageToShow);
        
        propertyChangeSupport.firePropertyChange(PROPERTY_BINDING_CHANGE, Boolean.FALSE,
                Boolean.TRUE);
    }
    
    public boolean isLastPage() {
        int nbPages = getNbPages();
        int pageToShow = formPagedResult.getPageToShow();
        
        boolean result = (nbPages == pageToShow);
        return result;
    }

    public String getNbPagesAsText() {
        int nbPages = getNbPages();
        return " / " + String.valueOf(nbPages);
    }

    protected int getNbPages() {
        int totalFoundFormNb = formPagedResult.getTotalFoundFormNb();
        int nbFormsPerPage = formPagedResult.getNbFormsToShow();
        
        int nbPages = (int) Math.ceil(totalFoundFormNb / (double) nbFormsPerPage);
        
        // can be negative if all (-1) result per page is selected
        if (nbPages <= 0) {
            nbPages = 1;
        }
        
        return nbPages;
    }
    
    public String getFieldToSort() {
        return formPagedResult.getFieldToSort();
    }

    public void setFieldToSort(String fieldToSort) {
        if (fieldToSort != null) {
            formPagedResult.setFieldToSort(fieldToSort);
        }
    }

    public boolean isAscending() {
        return formPagedResult.isAscending();
    }

    public void setAscending(boolean ascending) {
        formPagedResult.setAscending(ascending);
    }

    public int getTotalFoundFormNb() {
        return formPagedResult.getTotalFoundFormNb();
    }

    @Override
    public boolean isCellEditable(int rowIndex, int columnIndex) {
        return false;
    }

    @Override
    public String getColumnName(int col) {
        Column column = columns.get(col);
        return column.i18name;
    }

    @Override
    public int getRowCount() {
        return formPagedResult.getFormsIdsToShow() != null ?
                formPagedResult.getFormsIdsToShow().size() :
                0;
    }

    @Override
    public int getColumnCount() {
        return columns.size();
    }

    public String getFormIdAt(int row) {
        List<String> formsToShow = formPagedResult.getFormsIdsToShow();
        if (formsToShow == null) {
            return null;
        }
        if (row >= formsToShow.size()) {
            return null;
        }

        return formsToShow.get(row);
    }

    @Override
    public Object getValueAt(int row, int col) {
        List<String> formsToShow = formPagedResult.getFormsIdsToShow();
        if (formsToShow == null) {
            return null;
        }
        if (row >= formsToShow.size()) {
            return null;
        }
        
        String formId = formsToShow.get(row);
        if (formId == null) {
            return null;
        }

        // Get from cache
        Form form = formsCached.get(formId);

        if (form == null) {
            form = VradiService.getWikittyProxy().restore(Form.class, formId);
            formsCached.put(formId, form);
        }

        // Except for status
        String fqColumnName = getColumnFqName(col);

        int indexOf = fqColumnName.indexOf('.');

        String extension = fqColumnName.substring(0, indexOf);
        String fieldName = fqColumnName.substring(indexOf + 1);

        // Get proxy
        WikittyProxy proxy = VradiService.getWikittyProxy();

        if(fieldName.equals(Infogene.FIELD_INFOGENE_STATUS)) {
            String statusId = form.getStatus();

            Status statusCached = statusesCached.get(statusId);
            if (statusCached == null) {

                // If not found restor it
                statusCached = proxy.restore(Status.class, statusId);

                // Put in cache
                statusesCached.put(statusId, statusCached);
            }
            return statusCached;
        }

        // Except for stream
        if(fieldName.equals(Form.FIELD_FORM_XMLSTREAM)) {
            String xmlSreamId = form.getXmlStream();

            // Get in cache
            XmlStream streamCached = streamsCached.get(xmlSreamId);
            if (streamCached == null) {

                // If not found restor it
                streamCached = proxy.restore(XmlStream.class, xmlSreamId);

                // Put in cache
                streamsCached.put(xmlSreamId, streamCached);
            }
            return streamCached;
        }

        Object field = form.getField(extension, fieldName);
        FieldType fieldType = form.getFieldType(extension, fieldName);
        Object result = fieldType.getValidValue(field);
        return result;
    }

    @Override
    public Class<?> getColumnClass(int col) {
        Column column = columns.get(col);
        return column.columnClass;
    }

    public String getColumnFqName(int col) {
        Column column = columns.get(col);
        return column.fqName;
    }

    public String getToolTip(int row) {
        List<String> formsIdsToShow = formPagedResult.getFormsIdsToShow();
        if (formsIdsToShow == null) {
            return null;
        }
        if (row >= formsIdsToShow.size()) {
            return null;
        }

        String formId = formsIdsToShow.get(row);

        String toolTip = toolTipCached.get(formId);
        if (toolTip == null) {
            // Get in cache
            Form form = formsCached.get(formId);
            if (showThesaurusToolTip) {
                toolTip = ToolTipHelper.getEmailThesaurusToolTip(form);
            } else {
                toolTip = ToolTipHelper.getToolTip(form);
            }

            // Set in cache
            toolTipCached.put(formId, toolTip);
        }
        return toolTip;
    }

    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport.addPropertyChangeListener(listener);
    }

    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertyChangeSupport.removePropertyChangeListener(listener);
    }

    public void addPropertyChangeListener(String propertyName,
                                          PropertyChangeListener listener) {
        propertyChangeSupport.addPropertyChangeListener(propertyName, listener);
    }

    public void removePropertyChangeListener(String propertyName,
                                             PropertyChangeListener listener) {
        propertyChangeSupport.removePropertyChangeListener(propertyName, listener);
    }

    static public abstract class OfferListTableModelBinding
            extends DefaultJAXXBinding {

        protected OfferListTableModel model;

        public OfferListTableModelBinding(JAXXObject source, String id,
                                          OfferListTableModel model) {
            super(source, id, false);
            this.model = model;
        }

        @Override
        public void applyDataBinding() {
            if (model != null) {
                model.addPropertyChangeListener(PROPERTY_BINDING_CHANGE, this);
            }
        }

        @Override
        public void removeDataBinding() {
            if (model != null) {
                model.removePropertyChangeListener(PROPERTY_BINDING_CHANGE, this);
            }
        }
    }

    static public class OfferListTableCellRenderer extends
            DefaultTableCellRenderer {
        private static final long serialVersionUID = 1L;

        @Override
        public Component getTableCellRendererComponent(JTable table,
                Object value, boolean isSelected, boolean hasFocus, int row,
                int column) {

            String stringValue;

            if (value == null) {
                stringValue = null;
            } else if (value instanceof Date) {
                stringValue = DateFormat.getDateTimeInstance(DateFormat.SHORT,
                        DateFormat.SHORT, Locale.FRANCE).format((Date) value);
                
            } else if (value instanceof Status) {
                stringValue = _(((Status)value).getName());
            } else if (value instanceof XmlStream) {
                stringValue = _(((XmlStream)value).getName());
            } else {
                stringValue = String.valueOf(value);
            }

            OfferListTableModel model = (OfferListTableModel) table.getModel();
            setToolTipText(model.getToolTip(table.convertRowIndexToModel(row)));

            return super.getTableCellRendererComponent(table, stringValue,
                    isSelected, hasFocus, row, column);
        }
    }

    @Override
    public void putWikitty(WikittyServiceEvent event) {

        // map between id and extensions "name" (not extension ids)
        Map<String, Set<String>> idAndExtensions = event.getIdExtensions();
        for (String wikittyId : event.getIds()) {
            Set<String> wikittyExtensions = idAndExtensions.get(wikittyId);
            if (log.isDebugEnabled()) {
                log.debug("Receive wikitty service put event : " + wikittyId);
            }

            // Update cache
            WikittyProxy proxy = VradiService.getWikittyProxy();
            if (wikittyExtensions.contains(Status.EXT_STATUS)) {
                if (statusesCached.containsKey(wikittyId)) {
                    Status status = proxy.restore(Status.class, wikittyId);
                    statusesCached.put(wikittyId, status);
                }
            } else if (wikittyExtensions.contains(XmlStream.EXT_XMLSTREAM)) {
                if (streamsCached.containsKey(wikittyId)) {
                    XmlStream stream = proxy.restore(XmlStream.class, wikittyId);
                    streamsCached.put(wikittyId, stream);
                }
            } else if (wikittyExtensions.contains(Form.EXT_FORM)) {
                if (formsCached.containsKey(wikittyId)) {
                    Form form = proxy.restore(Form.class, wikittyId);
                    formsCached.put(wikittyId, form);

                    // Remove toolTip cached
                    toolTipCached.remove(wikittyId);
                }
            }
        }
    }

    @Override
    public void removeWikitty(WikittyServiceEvent event) {

        for (String wikittyId : event.getIds()) {

            if (statusesCached.containsKey(wikittyId)) {
                statusesCached.remove(wikittyId);

            } else if (streamsCached.containsKey(wikittyId)) {
                statusesCached.remove(wikittyId);
            }
        }
    }

    @Override
    public void clearWikitty(WikittyServiceEvent wikittyServiceEvent) {
    }

    @Override
    public void putExtension(WikittyServiceEvent wikittyServiceEvent) {
    }

    @Override
    public void removeExtension(WikittyServiceEvent wikittyServiceEvent) {
    }

    @Override
    public void clearExtension(WikittyServiceEvent wikittyServiceEvent) {
    }
}
