package org.sharengo.wikitty.search;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.Criteria;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyUtil;

/**
 * Helper to create a criteria with a restriction
 *
 * Element :
 * <extensionName>.<fieldName>[.<fieldType>] : search on an extension and field with specific type (optionnal)
 * Criteria.ALL_EXTENSIONS.<fieldName>.<fieldType> : search on all extension and field name with specific type
 *
 * <fieldType> specify seach on field as NUMERIC, STRING, WIKITTY, BOOLEAN, TEXT
 */
public class Search {

    static private Log log = LogFactory.getLog(Search.class);

    public enum KIND {
        AND, OR, NOT
    }

    KIND kind = KIND.AND;
    List<Restriction> restrictions;
    List<Search> subSearchs;

    public Search() {
        restrictions = new ArrayList<Restriction>();
        subSearchs = new ArrayList<Search>();
    }

    /**
     * Create Search query with field in wikitty argument
     *
     * @param wikitty example use to create query
     * @return
     */
    static public Search query(Wikitty wikitty) {
        Search result = Search.query();
        result.kind = KIND.AND;
        // result object must have same extension that wikitty example
        for (String extName : wikitty.getExtensionNames()) {
            result.eq(Element.ELT_EXTENSION, extName);
        }

        for (String fqfieldName : wikitty.fieldNames()) {
            Object value = wikitty.getFqField(fqfieldName);
            if (value != null) { // FIXME poussin 20090830 how to search for null value in field ?
                FieldType type = wikitty.getFieldType(fqfieldName);
                if (type.isCollection()) {
                    Collection collection = (Collection) value;
                    for (Object o : collection) {
                        String strValue = WikittyUtil.toString(type, o);
                        result.eq(fqfieldName, strValue);
                    }
                } else {
                    String strValue = WikittyUtil.toString(type, value);
                    result.eq(fqfieldName, strValue);
                }
            }
        }

        return result;
    }

    public static Search query() {
        Search search = new Search();
        search.kind = KIND.AND;
        return search;
    }

    public static Search query(KIND kind) {
        Search search = new Search();
        search.kind = kind;
        return search;
    }

    /**
     * Create new query on existant criteria to add new constraint to existant
     * criteria
     * @param criteria
     * @return
     */
    public static Search query(Criteria criteria) {
        Search search = query();
        search.restrictions.add(criteria.getRestriction());
        return search;
    }

    private static Element elt(String element) {
        Element elm = new Element();
        elm.setName(element);
        return elm;
    }

    protected Search handle( Restriction dto ) {
        restrictions.add( dto );
        return this;
    }

    public Search contains(String element, Collection<String> values) {
        return handle( RestrictionHelper.contains(elt(element),
                new ArrayList<String>(values)) );
    }

    public Search eq(String element, String value) {
        return handle( RestrictionHelper.eq(elt(element), value) );
    }

    public Search neq(String element, String value) {
        return handle( RestrictionHelper.neq(elt(element), value) );
    }

    public Search gt(String element, String value) {
        return handle( RestrictionHelper.great(elt(element), value) );
    }

    public Search ge(String element, String value) {
        return handle( RestrictionHelper.greatEq(elt(element), value) );
    }

    public Search lt(String element, String value) {
        return handle( RestrictionHelper.less(elt(element), value) );
    }

    public Search le(String element, String value) {
        return handle( RestrictionHelper.lessEq(elt(element), value) );
    }

    public Search bw(String element, String lowerValue, String upperValue) {
        return handle( RestrictionHelper.between(elt(element), lowerValue, upperValue) );
    }

    public Search sw(String element, String value) {
        return handle( RestrictionHelper.start(elt(element), value) );
    }

    public Search nsw(String element, String value) {
        return handle( RestrictionHelper.not(
                        RestrictionHelper.start(elt(element), value)) );
    }

    public Search ew(String element, String value) {
        return handle( RestrictionHelper.end(elt(element), value) );
    }

    public Search notew(String element, String value) {
        return handle( RestrictionHelper.not(
                        RestrictionHelper.end(elt(element), value)) );
    }

    public Search keyword(String value) {
        return handle(RestrictionHelper.keyword(value));
    }

    public Search not() {
        Search not = Search.query(KIND.NOT);
        subSearchs.add(not);

        Search search = Search.query(kind);
        not.subSearchs.add(search);

        return search;
    }

    public Search or() {
        Search search = Search.query(KIND.OR);
        subSearchs.add(search);
        return search;
    }

    public Search and() {
        Search search = Search.query(KIND.AND);
        subSearchs.add(search);
        return search;
    }

    public Search associated(String foreignFieldName) {
        Search search = new SubSearch( foreignFieldName, this );
        return search;
    }

    /**
     * return unnamed criteria
     * @return new criteria
     */
    public Criteria criteria(String name) {
        Criteria criteria = new Criteria(name);
        Restriction result = getRestrictions();
        criteria.setRestriction(result);
        return criteria;
    }

    protected Restriction getRestrictions() throws UnsupportedOperationException {
        Restriction result;
        if (restrictions.isEmpty() && subSearchs.isEmpty()) {
            result = RestrictionHelper.rFalse();

        } else if (restrictions.size() == 1 && subSearchs.isEmpty()) {
            result = restrictions.remove(0);

        } else if (subSearchs.size() == 1 && restrictions.isEmpty()) {
            Search subSearch = subSearchs.get(0);
            result = subSearch.getRestrictions();

            if(kind == KIND.NOT) {
                result = RestrictionHelper.not(result);
            }

        } else {
            List<Restriction> allRestrictions = new ArrayList<Restriction>(restrictions);
            for (Search subSearch : subSearchs) {
                Restriction restriction = subSearch.getRestrictions();
                allRestrictions.add(restriction);
            }
            switch (kind) {
                case OR:
                    result = RestrictionHelper.or(allRestrictions);
                    break;
                case AND:
                    result = RestrictionHelper.and(allRestrictions);
                    break;
                default:
                    throw new UnsupportedOperationException("Can't handle restriction kind " + kind.name());
            }
        }
        return result;
    }

    /**
     * return criteria with name
     * @return new criteria
     */
    public Criteria criteria() {
        return criteria(null);
    }
}