/* -*- Mode: Java; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set shiftwidth=2 tabstop=2 autoindent cindent expandtab: */
/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*jshint globalstrict: false */

// Initializing PDFJS global object (if still undefined)
if (typeof PDFJS === 'undefined') {
  (typeof window !== 'undefined' ? window : this).PDFJS = {};
}

PDFJS.version = '1.0.277';
PDFJS.build = '250d394';

(function pdfjsWrapper() {
  // Use strict in our context only - users might not want it
  'use strict';

/* -*- Mode: Java; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set shiftwidth=2 tabstop=2 autoindent cindent expandtab: */
/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/* globals Cmd, ColorSpace, Dict, MozBlobBuilder, Name, PDFJS, Ref, URL,
           Promise */

'use strict';

var globalScope = (typeof window === 'undefined') ? this : window;

var isWorker = (typeof window == 'undefined');

var FONT_IDENTITY_MATRIX = [0.001, 0, 0, 0.001, 0, 0];

var TextRenderingMode = {
  FILL: 0,
  STROKE: 1,
  FILL_STROKE: 2,
  INVISIBLE: 3,
  FILL_ADD_TO_PATH: 4,
  STROKE_ADD_TO_PATH: 5,
  FILL_STROKE_ADD_TO_PATH: 6,
  ADD_TO_PATH: 7,
  FILL_STROKE_MASK: 3,
  ADD_TO_PATH_FLAG: 4
};

var ImageKind = {
  GRAYSCALE_1BPP: 1,
  RGB_24BPP: 2,
  RGBA_32BPP: 3
};

// The global PDFJS object exposes the API
// In production, it will be declared outside a global wrapper
// In development, it will be declared here
if (!globalScope.PDFJS) {
  globalScope.PDFJS = {};
}

globalScope.PDFJS.pdfBug = false;

PDFJS.VERBOSITY_LEVELS = {
  errors: 0,
  warnings: 1,
  infos: 5
};

// All the possible operations for an operator list.
var OPS = PDFJS.OPS = {
  // Intentionally start from 1 so it is easy to spot bad operators that will be
  // 0's.
  dependency: 1,
  setLineWidth: 2,
  setLineCap: 3,
  setLineJoin: 4,
  setMiterLimit: 5,
  setDash: 6,
  setRenderingIntent: 7,
  setFlatness: 8,
  setGState: 9,
  save: 10,
  restore: 11,
  transform: 12,
  moveTo: 13,
  lineTo: 14,
  curveTo: 15,
  curveTo2: 16,
  curveTo3: 17,
  closePath: 18,
  rectangle: 19,
  stroke: 20,
  closeStroke: 21,
  fill: 22,
  eoFill: 23,
  fillStroke: 24,
  eoFillStroke: 25,
  closeFillStroke: 26,
  closeEOFillStroke: 27,
  endPath: 28,
  clip: 29,
  eoClip: 30,
  beginText: 31,
  endText: 32,
  setCharSpacing: 33,
  setWordSpacing: 34,
  setHScale: 35,
  setLeading: 36,
  setFont: 37,
  setTextRenderingMode: 38,
  setTextRise: 39,
  moveText: 40,
  setLeadingMoveText: 41,
  setTextMatrix: 42,
  nextLine: 43,
  showText: 44,
  showSpacedText: 45,
  nextLineShowText: 46,
  nextLineSetSpacingShowText: 47,
  setCharWidth: 48,
  setCharWidthAndBounds: 49,
  setStrokeColorSpace: 50,
  setFillColorSpace: 51,
  setStrokeColor: 52,
  setStrokeColorN: 53,
  setFillColor: 54,
  setFillColorN: 55,
  setStrokeGray: 56,
  setFillGray: 57,
  setStrokeRGBColor: 58,
  setFillRGBColor: 59,
  setStrokeCMYKColor: 60,
  setFillCMYKColor: 61,
  shadingFill: 62,
  beginInlineImage: 63,
  beginImageData: 64,
  endInlineImage: 65,
  paintXObject: 66,
  markPoint: 67,
  markPointProps: 68,
  beginMarkedContent: 69,
  beginMarkedContentProps: 70,
  endMarkedContent: 71,
  beginCompat: 72,
  endCompat: 73,
  paintFormXObjectBegin: 74,
  paintFormXObjectEnd: 75,
  beginGroup: 76,
  endGroup: 77,
  beginAnnotations: 78,
  endAnnotations: 79,
  beginAnnotation: 80,
  endAnnotation: 81,
  paintJpegXObject: 82,
  paintImageMaskXObject: 83,
  paintImageMaskXObjectGroup: 84,
  paintImageXObject: 85,
  paintInlineImageXObject: 86,
  paintInlineImageXObjectGroup: 87,
  paintImageXObjectRepeat: 88,
  paintImageMaskXObjectRepeat: 89,
  paintSolidColorImageMask: 90,
  constructPath: 91
};

// A notice for devs. These are good for things that are helpful to devs, such
// as warning that Workers were disabled, which is important to devs but not
// end users.
function info(msg) {
  if (PDFJS.verbosity >= PDFJS.VERBOSITY_LEVELS.infos) {
    console.log('Info: ' + msg);
  }
}

// Non-fatal warnings.
function warn(msg) {
  if (PDFJS.verbosity >= PDFJS.VERBOSITY_LEVELS.warnings) {
    console.log('Warning: ' + msg);
  }
}

// Fatal errors that should trigger the fallback UI and halt execution by
// throwing an exception.
function error(msg) {
  // If multiple arguments were passed, pass them all to the log function.
  if (arguments.length > 1) {
    var logArguments = ['Error:'];
    logArguments.push.apply(logArguments, arguments);
    console.log.apply(console, logArguments);
    // Join the arguments into a single string for the lines below.
    msg = [].join.call(arguments, ' ');
  } else {
    console.log('Error: ' + msg);
  }
  console.log(backtrace());
  UnsupportedManager.notify(UNSUPPORTED_FEATURES.unknown);
  throw new Error(msg);
}

function backtrace() {
  try {
    throw new Error();
  } catch (e) {
    return e.stack ? e.stack.split('\n').slice(2).join('\n') : '';
  }
}

function assert(cond, msg) {
  if (!cond) {
    error(msg);
  }
}

var UNSUPPORTED_FEATURES = PDFJS.UNSUPPORTED_FEATURES = {
  unknown: 'unknown',
  forms: 'forms',
  javaScript: 'javaScript',
  smask: 'smask',
  shadingPattern: 'shadingPattern',
  font: 'font'
};

var UnsupportedManager = PDFJS.UnsupportedManager =
  (function UnsupportedManagerClosure() {
  var listeners = [];
  return {
    listen: function (cb) {
      listeners.push(cb);
    },
    notify: function (featureId) {
      warn('Unsupported feature "' + featureId + '"');
      for (var i = 0, ii = listeners.length; i < ii; i++) {
        listeners[i](featureId);
      }
    }
  };
})();

// Combines two URLs. The baseUrl shall be absolute URL. If the url is an
// absolute URL, it will be returned as is.
function combineUrl(baseUrl, url) {
  if (!url) {
    return baseUrl;
  }
  if (/^[a-z][a-z0-9+\-.]*:/i.test(url)) {
    return url;
  }
  var i;
  if (url.charAt(0) == '/') {
    // absolute path
    i = baseUrl.indexOf('://');
    if (url.charAt(1) === '/') {
      ++i;
    } else {
      i = baseUrl.indexOf('/', i + 3);
    }
    return baseUrl.substring(0, i) + url;
  } else {
    // relative path
    var pathLength = baseUrl.length;
    i = baseUrl.lastIndexOf('#');
    pathLength = i >= 0 ? i : pathLength;
    i = baseUrl.lastIndexOf('?', pathLength);
    pathLength = i >= 0 ? i : pathLength;
    var prefixLength = baseUrl.lastIndexOf('/', pathLength);
    return baseUrl.substring(0, prefixLength + 1) + url;
  }
}

// Validates if URL is safe and allowed, e.g. to avoid XSS.
function isValidUrl(url, allowRelative) {
  if (!url) {
    return false;
  }
  // RFC 3986 (http://tools.ietf.org/html/rfc3986#section-3.1)
  // scheme = ALPHA *( ALPHA / DIGIT / "+" / "-" / "." )
  var protocol = /^[a-z][a-z0-9+\-.]*(?=:)/i.exec(url);
  if (!protocol) {
    return allowRelative;
  }
  protocol = protocol[0].toLowerCase();
  switch (protocol) {
    case 'http':
    case 'https':
    case 'ftp':
    case 'mailto':
      return true;
    default:
      return false;
  }
}
PDFJS.isValidUrl = isValidUrl;

function shadow(obj, prop, value) {
  Object.defineProperty(obj, prop, { value: value,
                                     enumerable: true,
                                     configurable: true,
                                     writable: false });
  return value;
}

var PasswordResponses = PDFJS.PasswordResponses = {
  NEED_PASSWORD: 1,
  INCORRECT_PASSWORD: 2
};

var PasswordException = (function PasswordExceptionClosure() {
  function PasswordException(msg, code) {
    this.name = 'PasswordException';
    this.message = msg;
    this.code = code;
  }

  PasswordException.prototype = new Error();
  PasswordException.constructor = PasswordException;

  return PasswordException;
})();

var UnknownErrorException = (function UnknownErrorExceptionClosure() {
  function UnknownErrorException(msg, details) {
    this.name = 'UnknownErrorException';
    this.message = msg;
    this.details = details;
  }

  UnknownErrorException.prototype = new Error();
  UnknownErrorException.constructor = UnknownErrorException;

  return UnknownErrorException;
})();

var InvalidPDFException = (function InvalidPDFExceptionClosure() {
  function InvalidPDFException(msg) {
    this.name = 'InvalidPDFException';
    this.message = msg;
  }

  InvalidPDFException.prototype = new Error();
  InvalidPDFException.constructor = InvalidPDFException;

  return InvalidPDFException;
})();

var MissingPDFException = (function MissingPDFExceptionClosure() {
  function MissingPDFException(msg) {
    this.name = 'MissingPDFException';
    this.message = msg;
  }

  MissingPDFException.prototype = new Error();
  MissingPDFException.constructor = MissingPDFException;

  return MissingPDFException;
})();

var NotImplementedException = (function NotImplementedExceptionClosure() {
  function NotImplementedException(msg) {
    this.message = msg;
  }

  NotImplementedException.prototype = new Error();
  NotImplementedException.prototype.name = 'NotImplementedException';
  NotImplementedException.constructor = NotImplementedException;

  return NotImplementedException;
})();

var MissingDataException = (function MissingDataExceptionClosure() {
  function MissingDataException(begin, end) {
    this.begin = begin;
    this.end = end;
    this.message = 'Missing data [' + begin + ', ' + end + ')';
  }

  MissingDataException.prototype = new Error();
  MissingDataException.prototype.name = 'MissingDataException';
  MissingDataException.constructor = MissingDataException;

  return MissingDataException;
})();

var XRefParseException = (function XRefParseExceptionClosure() {
  function XRefParseException(msg) {
    this.message = msg;
  }

  XRefParseException.prototype = new Error();
  XRefParseException.prototype.name = 'XRefParseException';
  XRefParseException.constructor = XRefParseException;

  return XRefParseException;
})();


function bytesToString(bytes) {
  var length = bytes.length;
  var MAX_ARGUMENT_COUNT = 8192;
  if (length < MAX_ARGUMENT_COUNT) {
    return String.fromCharCode.apply(null, bytes);
  }
  var strBuf = [];
  for (var i = 0; i < length; i += MAX_ARGUMENT_COUNT) {
    var chunkEnd = Math.min(i + MAX_ARGUMENT_COUNT, length);
    var chunk = bytes.subarray(i, chunkEnd);
    strBuf.push(String.fromCharCode.apply(null, chunk));
  }
  return strBuf.join('');
}

function stringToArray(str) {
  var length = str.length;
  var array = [];
  for (var i = 0; i < length; ++i) {
    array[i] = str.charCodeAt(i);
  }
  return array;
}

function stringToBytes(str) {
  var length = str.length;
  var bytes = new Uint8Array(length);
  for (var i = 0; i < length; ++i) {
    bytes[i] = str.charCodeAt(i) & 0xFF;
  }
  return bytes;
}

function string32(value) {
  return String.fromCharCode((value >> 24) & 0xff, (value >> 16) & 0xff,
                             (value >> 8) & 0xff, value & 0xff);
}

function log2(x) {
  var n = 1, i = 0;
  while (x > n) {
    n <<= 1;
    i++;
  }
  return i;
}

function readInt8(data, start) {
  return (data[start] << 24) >> 24;
}

function readUint16(data, offset) {
  return (data[offset] << 8) | data[offset + 1];
}

function readUint32(data, offset) {
  return ((data[offset] << 24) | (data[offset + 1] << 16) |
         (data[offset + 2] << 8) | data[offset + 3]) >>> 0;
}

// Lazy test the endianness of the platform
// NOTE: This will be 'true' for simulated TypedArrays
function isLittleEndian() {
  var buffer8 = new Uint8Array(2);
  buffer8[0] = 1;
  var buffer16 = new Uint16Array(buffer8.buffer);
  return (buffer16[0] === 1);
}

Object.defineProperty(PDFJS, 'isLittleEndian', {
  configurable: true,
  get: function PDFJS_isLittleEndian() {
    return shadow(PDFJS, 'isLittleEndian', isLittleEndian());
  }
});

  // Lazy test if the userAgant support CanvasTypedArrays
function hasCanvasTypedArrays() {
  var canvas = document.createElement('canvas');
  canvas.width = canvas.height = 1;
  var ctx = canvas.getContext('2d');
  var imageData = ctx.createImageData(1, 1);
  return (typeof imageData.data.buffer !== 'undefined');
}

Object.defineProperty(PDFJS, 'hasCanvasTypedArrays', {
  configurable: true,
  get: function PDFJS_hasCanvasTypedArrays() {
    return shadow(PDFJS, 'hasCanvasTypedArrays', hasCanvasTypedArrays());
  }
});

var Uint32ArrayView = (function Uint32ArrayViewClosure() {

  function Uint32ArrayView(buffer, length) {
    this.buffer = buffer;
    this.byteLength = buffer.length;
    this.length = length === undefined ? (this.byteLength >> 2) : length;
    ensureUint32ArrayViewProps(this.length);
  }
  Uint32ArrayView.prototype = Object.create(null);

  var uint32ArrayViewSetters = 0;
  function createUint32ArrayProp(index) {
    return {
      get: function () {
        var buffer = this.buffer, offset = index << 2;
        return (buffer[offset] | (buffer[offset + 1] << 8) |
          (buffer[offset + 2] << 16) | (buffer[offset + 3] << 24)) >>> 0;
      },
      set: function (value) {
        var buffer = this.buffer, offset = index << 2;
        buffer[offset] = value & 255;
        buffer[offset + 1] = (value >> 8) & 255;
        buffer[offset + 2] = (value >> 16) & 255;
        buffer[offset + 3] = (value >>> 24) & 255;
      }
    };
  }

  function ensureUint32ArrayViewProps(length) {
    while (uint32ArrayViewSetters < length) {
      Object.defineProperty(Uint32ArrayView.prototype,
        uint32ArrayViewSetters,
        createUint32ArrayProp(uint32ArrayViewSetters));
      uint32ArrayViewSetters++;
    }
  }

  return Uint32ArrayView;
})();

var IDENTITY_MATRIX = [1, 0, 0, 1, 0, 0];

var Util = PDFJS.Util = (function UtilClosure() {
  function Util() {}

  Util.makeCssRgb = function Util_makeCssRgb(rgb) {
    return 'rgb(' + rgb[0] + ',' + rgb[1] + ',' + rgb[2] + ')';
  };

  // Concatenates two transformation matrices together and returns the result.
  Util.transform = function Util_transform(m1, m2) {
    return [
      m1[0] * m2[0] + m1[2] * m2[1],
      m1[1] * m2[0] + m1[3] * m2[1],
      m1[0] * m2[2] + m1[2] * m2[3],
      m1[1] * m2[2] + m1[3] * m2[3],
      m1[0] * m2[4] + m1[2] * m2[5] + m1[4],
      m1[1] * m2[4] + m1[3] * m2[5] + m1[5]
    ];
  };

  // For 2d affine transforms
  Util.applyTransform = function Util_applyTransform(p, m) {
    var xt = p[0] * m[0] + p[1] * m[2] + m[4];
    var yt = p[0] * m[1] + p[1] * m[3] + m[5];
    return [xt, yt];
  };

  Util.applyInverseTransform = function Util_applyInverseTransform(p, m) {
    var d = m[0] * m[3] - m[1] * m[2];
    var xt = (p[0] * m[3] - p[1] * m[2] + m[2] * m[5] - m[4] * m[3]) / d;
    var yt = (-p[0] * m[1] + p[1] * m[0] + m[4] * m[1] - m[5] * m[0]) / d;
    return [xt, yt];
  };

  // Applies the transform to the rectangle and finds the minimum axially
  // aligned bounding box.
  Util.getAxialAlignedBoundingBox =
    function Util_getAxialAlignedBoundingBox(r, m) {

    var p1 = Util.applyTransform(r, m);
    var p2 = Util.applyTransform(r.slice(2, 4), m);
    var p3 = Util.applyTransform([r[0], r[3]], m);
    var p4 = Util.applyTransform([r[2], r[1]], m);
    return [
      Math.min(p1[0], p2[0], p3[0], p4[0]),
      Math.min(p1[1], p2[1], p3[1], p4[1]),
      Math.max(p1[0], p2[0], p3[0], p4[0]),
      Math.max(p1[1], p2[1], p3[1], p4[1])
    ];
  };

  Util.inverseTransform = function Util_inverseTransform(m) {
    var d = m[0] * m[3] - m[1] * m[2];
    return [m[3] / d, -m[1] / d, -m[2] / d, m[0] / d,
      (m[2] * m[5] - m[4] * m[3]) / d, (m[4] * m[1] - m[5] * m[0]) / d];
  };

  // Apply a generic 3d matrix M on a 3-vector v:
  //   | a b c |   | X |
  //   | d e f | x | Y |
  //   | g h i |   | Z |
  // M is assumed to be serialized as [a,b,c,d,e,f,g,h,i],
  // with v as [X,Y,Z]
  Util.apply3dTransform = function Util_apply3dTransform(m, v) {
    return [
      m[0] * v[0] + m[1] * v[1] + m[2] * v[2],
      m[3] * v[0] + m[4] * v[1] + m[5] * v[2],
      m[6] * v[0] + m[7] * v[1] + m[8] * v[2]
    ];
  };

  // This calculation uses Singular Value Decomposition.
  // The SVD can be represented with formula A = USV. We are interested in the
  // matrix S here because it represents the scale values.
  Util.singularValueDecompose2dScale =
    function Util_singularValueDecompose2dScale(m) {

    var transpose = [m[0], m[2], m[1], m[3]];

    // Multiply matrix m with its transpose.
    var a = m[0] * transpose[0] + m[1] * transpose[2];
    var b = m[0] * transpose[1] + m[1] * transpose[3];
    var c = m[2] * transpose[0] + m[3] * transpose[2];
    var d = m[2] * transpose[1] + m[3] * transpose[3];

    // Solve the second degree polynomial to get roots.
    var first = (a + d) / 2;
    var second = Math.sqrt((a + d) * (a + d) - 4 * (a * d - c * b)) / 2;
    var sx = first + second || 1;
    var sy = first - second || 1;

    // Scale values are the square roots of the eigenvalues.
    return [Math.sqrt(sx), Math.sqrt(sy)];
  };

  // Normalize rectangle rect=[x1, y1, x2, y2] so that (x1,y1) < (x2,y2)
  // For coordinate systems whose origin lies in the bottom-left, this
  // means normalization to (BL,TR) ordering. For systems with origin in the
  // top-left, this means (TL,BR) ordering.
  Util.normalizeRect = function Util_normalizeRect(rect) {
    var r = rect.slice(0); // clone rect
    if (rect[0] > rect[2]) {
      r[0] = rect[2];
      r[2] = rect[0];
    }
    if (rect[1] > rect[3]) {
      r[1] = rect[3];
      r[3] = rect[1];
    }
    return r;
  };

  // Returns a rectangle [x1, y1, x2, y2] corresponding to the
  // intersection of rect1 and rect2. If no intersection, returns 'false'
  // The rectangle coordinates of rect1, rect2 should be [x1, y1, x2, y2]
  Util.intersect = function Util_intersect(rect1, rect2) {
    function compare(a, b) {
      return a - b;
    }

    // Order points along the axes
    var orderedX = [rect1[0], rect1[2], rect2[0], rect2[2]].sort(compare),
        orderedY = [rect1[1], rect1[3], rect2[1], rect2[3]].sort(compare),
        result = [];

    rect1 = Util.normalizeRect(rect1);
    rect2 = Util.normalizeRect(rect2);

    // X: first and second points belong to different rectangles?
    if ((orderedX[0] === rect1[0] && orderedX[1] === rect2[0]) ||
        (orderedX[0] === rect2[0] && orderedX[1] === rect1[0])) {
      // Intersection must be between second and third points
      result[0] = orderedX[1];
      result[2] = orderedX[2];
    } else {
      return false;
    }

    // Y: first and second points belong to different rectangles?
    if ((orderedY[0] === rect1[1] && orderedY[1] === rect2[1]) ||
        (orderedY[0] === rect2[1] && orderedY[1] === rect1[1])) {
      // Intersection must be between second and third points
      result[1] = orderedY[1];
      result[3] = orderedY[2];
    } else {
      return false;
    }

    return result;
  };

  Util.sign = function Util_sign(num) {
    return num < 0 ? -1 : 1;
  };

  // TODO(mack): Rename appendToArray
  Util.concatenateToArray = function concatenateToArray(arr1, arr2) {
    Array.prototype.push.apply(arr1, arr2);
  };

  Util.prependToArray = function concatenateToArray(arr1, arr2) {
    Array.prototype.unshift.apply(arr1, arr2);
  };

  Util.extendObj = function extendObj(obj1, obj2) {
    for (var key in obj2) {
      obj1[key] = obj2[key];
    }
  };

  Util.getInheritableProperty = function Util_getInheritableProperty(dict,
                                                                     name) {
    while (dict && !dict.has(name)) {
      dict = dict.get('Parent');
    }
    if (!dict) {
      return null;
    }
    return dict.get(name);
  };

  Util.inherit = function Util_inherit(sub, base, prototype) {
    sub.prototype = Object.create(base.prototype);
    sub.prototype.constructor = sub;
    for (var prop in prototype) {
      sub.prototype[prop] = prototype[prop];
    }
  };

  Util.loadScript = function Util_loadScript(src, callback) {
    var script = document.createElement('script');
    var loaded = false;
    script.setAttribute('src', src);
    if (callback) {
      script.onload = function() {
        if (!loaded) {
          callback();
        }
        loaded = true;
      };
    }
    document.getElementsByTagName('head')[0].appendChild(script);
  };

  return Util;
})();

/**
 * PDF page viewport created based on scale, rotation and offset.
 * @class
 * @alias PDFJS.PageViewport
 */
var PageViewport = PDFJS.PageViewport = (function PageViewportClosure() {
  /**
   * @constructor
   * @private
   * @param viewBox {Array} xMin, yMin, xMax and yMax coordinates.
   * @param scale {number} scale of the viewport.
   * @param rotation {number} rotations of the viewport in degrees.
   * @param offsetX {number} offset X
   * @param offsetY {number} offset Y
   * @param dontFlip {boolean} if true, axis Y will not be flipped.
   */
  function PageViewport(viewBox, scale, rotation, offsetX, offsetY, dontFlip) {
    this.viewBox = viewBox;
    this.scale = scale;
    this.rotation = rotation;
    this.offsetX = offsetX;
    this.offsetY = offsetY;

    // creating transform to convert pdf coordinate system to the normal
    // canvas like coordinates taking in account scale and rotation
    var centerX = (viewBox[2] + viewBox[0]) / 2;
    var centerY = (viewBox[3] + viewBox[1]) / 2;
    var rotateA, rotateB, rotateC, rotateD;
    rotation = rotation % 360;
    rotation = rotation < 0 ? rotation + 360 : rotation;
    switch (rotation) {
      case 180:
        rotateA = -1; rotateB = 0; rotateC = 0; rotateD = 1;
        break;
      case 90:
        rotateA = 0; rotateB = 1; rotateC = 1; rotateD = 0;
        break;
      case 270:
        rotateA = 0; rotateB = -1; rotateC = -1; rotateD = 0;
        break;
      //case 0:
      default:
        rotateA = 1; rotateB = 0; rotateC = 0; rotateD = -1;
        break;
    }

    if (dontFlip) {
      rotateC = -rotateC; rotateD = -rotateD;
    }

    var offsetCanvasX, offsetCanvasY;
    var width, height;
    if (rotateA === 0) {
      offsetCanvasX = Math.abs(centerY - viewBox[1]) * scale + offsetX;
      offsetCanvasY = Math.abs(centerX - viewBox[0]) * scale + offsetY;
      width = Math.abs(viewBox[3] - viewBox[1]) * scale;
      height = Math.abs(viewBox[2] - viewBox[0]) * scale;
    } else {
      offsetCanvasX = Math.abs(centerX - viewBox[0]) * scale + offsetX;
      offsetCanvasY = Math.abs(centerY - viewBox[1]) * scale + offsetY;
      width = Math.abs(viewBox[2] - viewBox[0]) * scale;
      height = Math.abs(viewBox[3] - viewBox[1]) * scale;
    }
    // creating transform for the following operations:
    // translate(-centerX, -centerY), rotate and flip vertically,
    // scale, and translate(offsetCanvasX, offsetCanvasY)
    this.transform = [
      rotateA * scale,
      rotateB * scale,
      rotateC * scale,
      rotateD * scale,
      offsetCanvasX - rotateA * scale * centerX - rotateC * scale * centerY,
      offsetCanvasY - rotateB * scale * centerX - rotateD * scale * centerY
    ];

    this.width = width;
    this.height = height;
    this.fontScale = scale;
  }
  PageViewport.prototype = /** @lends PDFJS.PageViewport.prototype */ {
    /**
     * Clones viewport with additional properties.
     * @param args {Object} (optional) If specified, may contain the 'scale' or
     * 'rotation' properties to override the corresponding properties in
     * the cloned viewport.
     * @returns {PDFJS.PageViewport} Cloned viewport.
     */
    clone: function PageViewPort_clone(args) {
      args = args || {};
      var scale = 'scale' in args ? args.scale : this.scale;
      var rotation = 'rotation' in args ? args.rotation : this.rotation;
      return new PageViewport(this.viewBox.slice(), scale, rotation,
                              this.offsetX, this.offsetY, args.dontFlip);
    },
    /**
     * Converts PDF point to the viewport coordinates. For examples, useful for
     * converting PDF location into canvas pixel coordinates.
     * @param x {number} X coordinate.
     * @param y {number} Y coordinate.
     * @returns {Object} Object that contains 'x' and 'y' properties of the
     * point in the viewport coordinate space.
     * @see {@link convertToPdfPoint}
     * @see {@link convertToViewportRectangle}
     */
    convertToViewportPoint: function PageViewport_convertToViewportPoint(x, y) {
      return Util.applyTransform([x, y], this.transform);
    },
    /**
     * Converts PDF rectangle to the viewport coordinates.
     * @param rect {Array} xMin, yMin, xMax and yMax coordinates.
     * @returns {Array} Contains corresponding coordinates of the rectangle
     * in the viewport coordinate space.
     * @see {@link convertToViewportPoint}
     */
    convertToViewportRectangle:
      function PageViewport_convertToViewportRectangle(rect) {
      var tl = Util.applyTransform([rect[0], rect[1]], this.transform);
      var br = Util.applyTransform([rect[2], rect[3]], this.transform);
      return [tl[0], tl[1], br[0], br[1]];
    },
    /**
     * Converts viewport coordinates to the PDF location. For examples, useful
     * for converting canvas pixel location into PDF one.
     * @param x {number} X coordinate.
     * @param y {number} Y coordinate.
     * @returns {Object} Object that contains 'x' and 'y' properties of the
     * point in the PDF coordinate space.
     * @see {@link convertToViewportPoint}
     */
    convertToPdfPoint: function PageViewport_convertToPdfPoint(x, y) {
      return Util.applyInverseTransform([x, y], this.transform);
    }
  };
  return PageViewport;
})();

var PDFStringTranslateTable = [
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0x2D8, 0x2C7, 0x2C6, 0x2D9, 0x2DD, 0x2DB, 0x2DA, 0x2DC, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0x2022, 0x2020, 0x2021, 0x2026, 0x2014,
  0x2013, 0x192, 0x2044, 0x2039, 0x203A, 0x2212, 0x2030, 0x201E, 0x201C,
  0x201D, 0x2018, 0x2019, 0x201A, 0x2122, 0xFB01, 0xFB02, 0x141, 0x152, 0x160,
  0x178, 0x17D, 0x131, 0x142, 0x153, 0x161, 0x17E, 0, 0x20AC
];

function stringToPDFString(str) {
  var i, n = str.length, strBuf = [];
  if (str[0] === '\xFE' && str[1] === '\xFF') {
    // UTF16BE BOM
    for (i = 2; i < n; i += 2) {
      strBuf.push(String.fromCharCode(
        (str.charCodeAt(i) << 8) | str.charCodeAt(i + 1)));
    }
  } else {
    for (i = 0; i < n; ++i) {
      var code = PDFStringTranslateTable[str.charCodeAt(i)];
      strBuf.push(code ? String.fromCharCode(code) : str.charAt(i));
    }
  }
  return strBuf.join('');
}

function stringToUTF8String(str) {
  return decodeURIComponent(escape(str));
}

function isEmptyObj(obj) {
  for (var key in obj) {
    return false;
  }
  return true;
}

function isBool(v) {
  return typeof v == 'boolean';
}

function isInt(v) {
  return typeof v == 'number' && ((v | 0) == v);
}

function isNum(v) {
  return typeof v == 'number';
}

function isString(v) {
  return typeof v == 'string';
}

function isNull(v) {
  return v === null;
}

function isName(v) {
  return v instanceof Name;
}

function isCmd(v, cmd) {
  return v instanceof Cmd && (!cmd || v.cmd == cmd);
}

function isDict(v, type) {
  if (!(v instanceof Dict)) {
    return false;
  }
  if (!type) {
    return true;
  }
  var dictType = v.get('Type');
  return isName(dictType) && dictType.name == type;
}

function isArray(v) {
  return v instanceof Array;
}

function isStream(v) {
  return typeof v == 'object' && v !== null && v !== undefined &&
         ('getBytes' in v);
}

function isArrayBuffer(v) {
  return typeof v == 'object' && v !== null && v !== undefined &&
         ('byteLength' in v);
}

function isRef(v) {
  return v instanceof Ref;
}

/**
 * Promise Capability object.
 *
 * @typedef {Object} PromiseCapability
 * @property {Promise} promise - A promise object.
 * @property {function} resolve - Fullfills the promise.
 * @property {function} reject - Rejects the promise.
 */

/**
 * Creates a promise capability object.
 * @alias PDFJS.createPromiseCapability
 *
 * @return {PromiseCapability} A capability object contains:
 * - a Promise, resolve and reject methods.
 */
function createPromiseCapability() {
  var capability = {};
  capability.promise = new Promise(function (resolve, reject) {
    capability.resolve = resolve;
    capability.reject = reject;
  });
  return capability;
}

PDFJS.createPromiseCapability = createPromiseCapability;

/**
 * Polyfill for Promises:
 * The following promise implementation tries to generally implement the
 * Promise/A+ spec. Some notable differences from other promise libaries are:
 * - There currently isn't a seperate deferred and promise object.
 * - Unhandled rejections eventually show an error if they aren't handled.
 *
 * Based off of the work in:
 * https://bugzilla.mozilla.org/show_bug.cgi?id=810490
 */
(function PromiseClosure() {
  if (globalScope.Promise) {
    // Promises existing in the DOM/Worker, checking presence of all/resolve
    if (typeof globalScope.Promise.all !== 'function') {
      globalScope.Promise.all = function (iterable) {
        var count = 0, results = [], resolve, reject;
        var promise = new globalScope.Promise(function (resolve_, reject_) {
          resolve = resolve_;
          reject = reject_;
        });
        iterable.forEach(function (p, i) {
          count++;
          p.then(function (result) {
            results[i] = result;
            count--;
            if (count === 0) {
              resolve(results);
            }
          }, reject);
        });
        if (count === 0) {
          resolve(results);
        }
        return promise;
      };
    }
    if (typeof globalScope.Promise.resolve !== 'function') {
      globalScope.Promise.resolve = function (value) {
        return new globalScope.Promise(function (resolve) { resolve(value); });
      };
    }
    if (typeof globalScope.Promise.reject !== 'function') {
      globalScope.Promise.reject = function (reason) {
        return new globalScope.Promise(function (resolve, reject) {
          reject(reason);
        });
      };
    }
    if (typeof globalScope.Promise.prototype.catch !== 'function') {
      globalScope.Promise.prototype.catch = function (onReject) {
        return globalScope.Promise.prototype.then(undefined, onReject);
      };
    }
    return;
  }
  var STATUS_PENDING = 0;
  var STATUS_RESOLVED = 1;
  var STATUS_REJECTED = 2;

  // In an attempt to avoid silent exceptions, unhandled rejections are
  // tracked and if they aren't handled in a certain amount of time an
  // error is logged.
  var REJECTION_TIMEOUT = 500;

  var HandlerManager = {
    handlers: [],
    running: false,
    unhandledRejections: [],
    pendingRejectionCheck: false,

    scheduleHandlers: function scheduleHandlers(promise) {
      if (promise._status == STATUS_PENDING) {
        return;
      }

      this.handlers = this.handlers.concat(promise._handlers);
      promise._handlers = [];

      if (this.running) {
        return;
      }
      this.running = true;

      setTimeout(this.runHandlers.bind(this), 0);
    },

    runHandlers: function runHandlers() {
      var RUN_TIMEOUT = 1; // ms
      var timeoutAt = Date.now() + RUN_TIMEOUT;
      while (this.handlers.length > 0) {
        var handler = this.handlers.shift();

        var nextStatus = handler.thisPromise._status;
        var nextValue = handler.thisPromise._value;

        try {
          if (nextStatus === STATUS_RESOLVED) {
            if (typeof(handler.onResolve) == 'function') {
              nextValue = handler.onResolve(nextValue);
            }
          } else if (typeof(handler.onReject) === 'function') {
              nextValue = handler.onReject(nextValue);
              nextStatus = STATUS_RESOLVED;

              if (handler.thisPromise._unhandledRejection) {
                this.removeUnhandeledRejection(handler.thisPromise);
              }
          }
        } catch (ex) {
          nextStatus = STATUS_REJECTED;
          nextValue = ex;
        }

        handler.nextPromise._updateStatus(nextStatus, nextValue);
        if (Date.now() >= timeoutAt) {
          break;
        }
      }

      if (this.handlers.length > 0) {
        setTimeout(this.runHandlers.bind(this), 0);
        return;
      }

      this.running = false;
    },

    addUnhandledRejection: function addUnhandledRejection(promise) {
      this.unhandledRejections.push({
        promise: promise,
        time: Date.now()
      });
      this.scheduleRejectionCheck();
    },

    removeUnhandeledRejection: function removeUnhandeledRejection(promise) {
      promise._unhandledRejection = false;
      for (var i = 0; i < this.unhandledRejections.length; i++) {
        if (this.unhandledRejections[i].promise === promise) {
          this.unhandledRejections.splice(i);
          i--;
        }
      }
    },

    scheduleRejectionCheck: function scheduleRejectionCheck() {
      if (this.pendingRejectionCheck) {
        return;
      }
      this.pendingRejectionCheck = true;
      setTimeout(function rejectionCheck() {
        this.pendingRejectionCheck = false;
        var now = Date.now();
        for (var i = 0; i < this.unhandledRejections.length; i++) {
          if (now - this.unhandledRejections[i].time > REJECTION_TIMEOUT) {
            var unhandled = this.unhandledRejections[i].promise._value;
            var msg = 'Unhandled rejection: ' + unhandled;
            if (unhandled.stack) {
              msg += '\n' + unhandled.stack;
            }
            warn(msg);
            this.unhandledRejections.splice(i);
            i--;
          }
        }
        if (this.unhandledRejections.length) {
          this.scheduleRejectionCheck();
        }
      }.bind(this), REJECTION_TIMEOUT);
    }
  };

  function Promise(resolver) {
    this._status = STATUS_PENDING;
    this._handlers = [];
    try {
      resolver.call(this, this._resolve.bind(this), this._reject.bind(this));
    } catch (e) {
      this._reject(e);
    }
  }
  /**
   * Builds a promise that is resolved when all the passed in promises are
   * resolved.
   * @param {array} array of data and/or promises to wait for.
   * @return {Promise} New dependant promise.
   */
  Promise.all = function Promise_all(promises) {
    var resolveAll, rejectAll;
    var deferred = new Promise(function (resolve, reject) {
      resolveAll = resolve;
      rejectAll = reject;
    });
    var unresolved = promises.length;
    var results = [];
    if (unresolved === 0) {
      resolveAll(results);
      return deferred;
    }
    function reject(reason) {
      if (deferred._status === STATUS_REJECTED) {
        return;
      }
      results = [];
      rejectAll(reason);
    }
    for (var i = 0, ii = promises.length; i < ii; ++i) {
      var promise = promises[i];
      var resolve = (function(i) {
        return function(value) {
          if (deferred._status === STATUS_REJECTED) {
            return;
          }
          results[i] = value;
          unresolved--;
          if (unresolved === 0) {
            resolveAll(results);
          }
        };
      })(i);
      if (Promise.isPromise(promise)) {
        promise.then(resolve, reject);
      } else {
        resolve(promise);
      }
    }
    return deferred;
  };

  /**
   * Checks if the value is likely a promise (has a 'then' function).
   * @return {boolean} true if value is thenable
   */
  Promise.isPromise = function Promise_isPromise(value) {
    return value && typeof value.then === 'function';
  };

  /**
   * Creates resolved promise
   * @param value resolve value
   * @returns {Promise}
   */
  Promise.resolve = function Promise_resolve(value) {
    return new Promise(function (resolve) { resolve(value); });
  };

  /**
   * Creates rejected promise
   * @param reason rejection value
   * @returns {Promise}
   */
  Promise.reject = function Promise_reject(reason) {
    return new Promise(function (resolve, reject) { reject(reason); });
  };

  Promise.prototype = {
    _status: null,
    _value: null,
    _handlers: null,
    _unhandledRejection: null,

    _updateStatus: function Promise__updateStatus(status, value) {
      if (this._status === STATUS_RESOLVED ||
          this._status === STATUS_REJECTED) {
        return;
      }

      if (status == STATUS_RESOLVED &&
          Promise.isPromise(value)) {
        value.then(this._updateStatus.bind(this, STATUS_RESOLVED),
                   this._updateStatus.bind(this, STATUS_REJECTED));
        return;
      }

      this._status = status;
      this._value = value;

      if (status === STATUS_REJECTED && this._handlers.length === 0) {
        this._unhandledRejection = true;
        HandlerManager.addUnhandledRejection(this);
      }

      HandlerManager.scheduleHandlers(this);
    },

    _resolve: function Promise_resolve(value) {
      this._updateStatus(STATUS_RESOLVED, value);
    },

    _reject: function Promise_reject(reason) {
      this._updateStatus(STATUS_REJECTED, reason);
    },

    then: function Promise_then(onResolve, onReject) {
      var nextPromise = new Promise(function (resolve, reject) {
        this.resolve = reject;
        this.reject = reject;
      });
      this._handlers.push({
        thisPromise: this,
        onResolve: onResolve,
        onReject: onReject,
        nextPromise: nextPromise
      });
      HandlerManager.scheduleHandlers(this);
      return nextPromise;
    },
    
    catch: function Promise_catch(onReject) {
      return this.then(undefined, onReject);
    }
  };

  globalScope.Promise = Promise;
})();

var StatTimer = (function StatTimerClosure() {
  function rpad(str, pad, length) {
    while (str.length < length) {
      str += pad;
    }
    return str;
  }
  function StatTimer() {
    this.started = {};
    this.times = [];
    this.enabled = true;
  }
  StatTimer.prototype = {
    time: function StatTimer_time(name) {
      if (!this.enabled) {
        return;
      }
      if (name in this.started) {
        warn('Timer is already running for ' + name);
      }
      this.started[name] = Date.now();
    },
    timeEnd: function StatTimer_timeEnd(name) {
      if (!this.enabled) {
        return;
      }
      if (!(name in this.started)) {
        warn('Timer has not been started for ' + name);
      }
      this.times.push({
        'name': name,
        'start': this.started[name],
        'end': Date.now()
      });
      // Remove timer from started so it can be called again.
      delete this.started[name];
    },
    toString: function StatTimer_toString() {
      var i, ii;
      var times = this.times;
      var out = '';
      // Find the longest name for padding purposes.
      var longest = 0;
      for (i = 0, ii = times.length; i < ii; ++i) {
        var name = times[i]['name'];
        if (name.length > longest) {
          longest = name.length;
        }
      }
      for (i = 0, ii = times.length; i < ii; ++i) {
        var span = times[i];
        var duration = span.end - span.start;
        out += rpad(span['name'], ' ', longest) + ' ' + duration + 'ms\n';
      }
      return out;
    }
  };
  return StatTimer;
})();

PDFJS.createBlob = function createBlob(data, contentType) {
  if (typeof Blob !== 'undefined') {
    return new Blob([data], { type: contentType });
  }
  // Blob builder is deprecated in FF14 and removed in FF18.
  var bb = new MozBlobBuilder();
  bb.append(data);
  return bb.getBlob(contentType);
};

PDFJS.createObjectURL = (function createObjectURLClosure() {
  // Blob/createObjectURL is not available, falling back to data schema.
  var digits =
    'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=';

  return function createObjectURL(data, contentType) {
    if (!PDFJS.disableCreateObjectURL &&
        typeof URL !== 'undefined' && URL.createObjectURL) {
      var blob = PDFJS.createBlob(data, contentType);
      return URL.createObjectURL(blob);
    }

    var buffer = 'data:' + contentType + ';base64,';
    for (var i = 0, ii = data.length; i < ii; i += 3) {
      var b1 = data[i] & 0xFF;
      var b2 = data[i + 1] & 0xFF;
      var b3 = data[i + 2] & 0xFF;
      var d1 = b1 >> 2, d2 = ((b1 & 3) << 4) | (b2 >> 4);
      var d3 = i + 1 < ii ? ((b2 & 0xF) << 2) | (b3 >> 6) : 64;
      var d4 = i + 2 < ii ? (b3 & 0x3F) : 64;
      buffer += digits[d1] + digits[d2] + digits[d3] + digits[d4];
    }
    return buffer;
  };
})();

function MessageHandler(name, comObj) {
  this.name = name;
  this.comObj = comObj;
  this.callbackIndex = 1;
  this.postMessageTransfers = true;
  var callbacksCapabilities = this.callbacksCapabilities = {};
  var ah = this.actionHandler = {};

  ah['console_log'] = [function ahConsoleLog(data) {
    console.log.apply(console, data);
  }];
  ah['console_error'] = [function ahConsoleError(data) {
    console.error.apply(console, data);
  }];
  ah['_unsupported_feature'] = [function ah_unsupportedFeature(data) {
    UnsupportedManager.notify(data);
  }];

  comObj.onmessage = function messageHandlerComObjOnMessage(event) {
    var data = event.data;
    if (data.isReply) {
      var callbackId = data.callbackId;
      if (data.callbackId in callbacksCapabilities) {
        var callback = callbacksCapabilities[callbackId];
        delete callbacksCapabilities[callbackId];
        if ('error' in data) {
          callback.reject(data.error);
        } else {
          callback.resolve(data.data);
        }
      } else {
        error('Cannot resolve callback ' + callbackId);
      }
    } else if (data.action in ah) {
      var action = ah[data.action];
      if (data.callbackId) {
        Promise.resolve().then(function () {
          return action[0].call(action[1], data.data);
        }).then(function (result) {
          comObj.postMessage({
            isReply: true,
            callbackId: data.callbackId,
            data: result
          });
        }, function (reason) {
          comObj.postMessage({
            isReply: true,
            callbackId: data.callbackId,
            error: reason
          });
        });
      } else {
        action[0].call(action[1], data.data);
      }
    } else {
      error('Unknown action from worker: ' + data.action);
    }
  };
}

MessageHandler.prototype = {
  on: function messageHandlerOn(actionName, handler, scope) {
    var ah = this.actionHandler;
    if (ah[actionName]) {
      error('There is already an actionName called "' + actionName + '"');
    }
    ah[actionName] = [handler, scope];
  },
  /**
   * Sends a message to the comObj to invoke the action with the supplied data.
   * @param {String} actionName Action to call.
   * @param {JSON} data JSON data to send.
   * @param {Array} [transfers] Optional list of transfers/ArrayBuffers
   */
  send: function messageHandlerSend(actionName, data, transfers) {
    var message = {
      action: actionName,
      data: data
    };
    this.postMessage(message, transfers);
  },
  /**
   * Sends a message to the comObj to invoke the action with the supplied data.
   * Expects that other side will callback with the response.
   * @param {String} actionName Action to call.
   * @param {JSON} data JSON data to send.
   * @param {Array} [transfers] Optional list of transfers/ArrayBuffers.
   * @returns {Promise} Promise to be resolved with response data.
   */
  sendWithPromise:
    function messageHandlerSendWithPromise(actionName, data, transfers) {
    var callbackId = this.callbackIndex++;
    var message = {
      action: actionName,
      data: data,
      callbackId: callbackId
    };
    var capability = createPromiseCapability();
    this.callbacksCapabilities[callbackId] = capability;
    try {
      this.postMessage(message, transfers);
    } catch (e) {
      capability.reject(e);
    }
    return capability.promise;
  },
  /**
   * Sends raw message to the comObj.
   * @private
   * @param message {Object} Raw message.
   * @param transfers List of transfers/ArrayBuffers, or undefined.
   */
  postMessage: function (message, transfers) {
    if (transfers && this.postMessageTransfers) {
      this.comObj.postMessage(message, transfers);
    } else {
      this.comObj.postMessage(message);
    }
  }
};

function loadJpegStream(id, imageUrl, objs) {
  var img = new Image();
  img.onload = (function loadJpegStream_onloadClosure() {
    objs.resolve(id, img);
  });
  img.src = imageUrl;
}


var DEFAULT_ICON_SIZE = 22; // px
var HIGHLIGHT_OFFSET = 4; // px
var SUPPORTED_TYPES = ['Link', 'Text', 'Widget'];

var Annotation = (function AnnotationClosure() {
  // 12.5.5: Algorithm: Appearance streams
  function getTransformMatrix(rect, bbox, matrix) {
    var bounds = Util.getAxialAlignedBoundingBox(bbox, matrix);
    var minX = bounds[0];
    var minY = bounds[1];
    var maxX = bounds[2];
    var maxY = bounds[3];

    if (minX === maxX || minY === maxY) {
      // From real-life file, bbox was [0, 0, 0, 0]. In this case,
      // just apply the transform for rect
      return [1, 0, 0, 1, rect[0], rect[1]];
    }

    var xRatio = (rect[2] - rect[0]) / (maxX - minX);
    var yRatio = (rect[3] - rect[1]) / (maxY - minY);
    return [
      xRatio,
      0,
      0,
      yRatio,
      rect[0] - minX * xRatio,
      rect[1] - minY * yRatio
    ];
  }

  function getDefaultAppearance(dict) {
    var appearanceState = dict.get('AP');
    if (!isDict(appearanceState)) {
      return;
    }

    var appearance;
    var appearances = appearanceState.get('N');
    if (isDict(appearances)) {
      var as = dict.get('AS');
      if (as && appearances.has(as.name)) {
        appearance = appearances.get(as.name);
      }
    } else {
      appearance = appearances;
    }
    return appearance;
  }

  function Annotation(params) {
    if (params.data) {
      this.data = params.data;
      return;
    }

    var dict = params.dict;
    var data = this.data = {};

    data.subtype = dict.get('Subtype').name;
    var rect = dict.get('Rect') || [0, 0, 0, 0];
    data.rect = Util.normalizeRect(rect);
    data.annotationFlags = dict.get('F');

    var color = dict.get('C');
    if (isArray(color) && color.length === 3) {
      // TODO(mack): currently only supporting rgb; need support different
      // colorspaces
      data.color = color;
    } else {
      data.color = [0, 0, 0];
    }

    // Some types of annotations have border style dict which has more
    // info than the border array
    if (dict.has('BS')) {
      var borderStyle = dict.get('BS');
      data.borderWidth = borderStyle.has('W') ? borderStyle.get('W') : 1;
    } else {
      var borderArray = dict.get('Border') || [0, 0, 1];
      data.borderWidth = borderArray[2] || 0;

      // TODO: implement proper support for annotations with line dash patterns.
      var dashArray = borderArray[3];
      if (data.borderWidth > 0 && dashArray) {
        if (!isArray(dashArray)) {
          // Ignore the border if dashArray is not actually an array,
          // this is consistent with the behaviour in Adobe Reader. 
          data.borderWidth = 0;
        } else {
          var dashArrayLength = dashArray.length;
          if (dashArrayLength > 0) {
            // According to the PDF specification: the elements in a dashArray
            // shall be numbers that are nonnegative and not all equal to zero.
            var isInvalid = false;
            var numPositive = 0;
            for (var i = 0; i < dashArrayLength; i++) {
              var validNumber = (+dashArray[i] >= 0);
              if (!validNumber) {
                isInvalid = true;
                break;
              } else if (dashArray[i] > 0) {
                numPositive++;
              }
            }
            if (isInvalid || numPositive === 0) {
              data.borderWidth = 0;
            }
          }
        }
      }
    }

    this.appearance = getDefaultAppearance(dict);
    data.hasAppearance = !!this.appearance;
    data.id = params.ref.num;
  }

  Annotation.prototype = {

    getData: function Annotation_getData() {
      return this.data;
    },

    hasHtml: function Annotation_hasHtml() {
      return false;
    },

    getHtmlElement: function Annotation_getHtmlElement(commonObjs) {
      throw new NotImplementedException(
        'getHtmlElement() should be implemented in subclass');
    },

    // TODO(mack): Remove this, it's not really that helpful.
    getEmptyContainer: function Annotation_getEmptyContainer(tagName, rect,
                                                             borderWidth) {
      assert(!isWorker,
        'getEmptyContainer() should be called from main thread');

      var bWidth = borderWidth || 0;

      rect = rect || this.data.rect;
      var element = document.createElement(tagName);
      element.style.borderWidth = bWidth + 'px';
      var width = rect[2] - rect[0] - 2 * bWidth;
      var height = rect[3] - rect[1] - 2 * bWidth;
      element.style.width = width + 'px';
      element.style.height = height + 'px';
      return element;
    },

    isInvisible: function Annotation_isInvisible() {
      var data = this.data;
      if (data && SUPPORTED_TYPES.indexOf(data.subtype) !== -1) {
        return false;
      } else {
        return !!(data &&
                  data.annotationFlags &&            // Default: not invisible
                  data.annotationFlags & 0x1);       // Invisible
      }
    },

    isViewable: function Annotation_isViewable() {
      var data = this.data;
      return !!(!this.isInvisible() &&
                data &&
                (!data.annotationFlags ||
                 !(data.annotationFlags & 0x22)) &&  // Hidden or NoView
                data.rect);                          // rectangle is nessessary
    },

    isPrintable: function Annotation_isPrintable() {
      var data = this.data;
      return !!(!this.isInvisible() &&
                data &&
                data.annotationFlags &&              // Default: not printable
                data.annotationFlags & 0x4 &&        // Print
                data.rect);                          // rectangle is nessessary
    },

    loadResources: function Annotation_loadResources(keys) {
      return new Promise(function (resolve, reject) {
        this.appearance.dict.getAsync('Resources').then(function (resources) {
          if (!resources) {
            resolve();
            return;
          }
          var objectLoader = new ObjectLoader(resources.map,
                                              keys,
                                              resources.xref);
          objectLoader.load().then(function() {
            resolve(resources);
          }, reject);
        }, reject);
      }.bind(this));
    },

    getOperatorList: function Annotation_getOperatorList(evaluator) {

      if (!this.appearance) {
        return Promise.resolve(new OperatorList());
      }

      var data = this.data;

      var appearanceDict = this.appearance.dict;
      var resourcesPromise = this.loadResources([
        'ExtGState',
        'ColorSpace',
        'Pattern',
        'Shading',
        'XObject',
        'Font'
        // ProcSet
        // Properties
      ]);
      var bbox = appearanceDict.get('BBox') || [0, 0, 1, 1];
      var matrix = appearanceDict.get('Matrix') || [1, 0, 0, 1, 0 ,0];
      var transform = getTransformMatrix(data.rect, bbox, matrix);
      var self = this;

      return resourcesPromise.then(function(resources) {
          var opList = new OperatorList();
          opList.addOp(OPS.beginAnnotation, [data.rect, transform, matrix]);
          return evaluator.getOperatorList(self.appearance, resources, opList).
            then(function () {
              opList.addOp(OPS.endAnnotation, []);
              self.appearance.reset();
              return opList;
            });
        });
    }
  };

  Annotation.getConstructor =
      function Annotation_getConstructor(subtype, fieldType) {

    if (!subtype) {
      return;
    }

    // TODO(mack): Implement FreeText annotations
    if (subtype === 'Link') {
      return LinkAnnotation;
    } else if (subtype === 'Text') {
      return TextAnnotation;
    } else if (subtype === 'Widget') {
      if (!fieldType) {
        return;
      }

      if (fieldType === 'Tx') {
        return TextWidgetAnnotation;
      } else {
        return WidgetAnnotation;
      }
    } else {
      return Annotation;
    }
  };

  // TODO(mack): Support loading annotation from data
  Annotation.fromData = function Annotation_fromData(data) {
    var subtype = data.subtype;
    var fieldType = data.fieldType;
    var Constructor = Annotation.getConstructor(subtype, fieldType);
    if (Constructor) {
      return new Constructor({ data: data });
    }
  };

  Annotation.fromRef = function Annotation_fromRef(xref, ref) {

    var dict = xref.fetchIfRef(ref);
    if (!isDict(dict)) {
      return;
    }

    var subtype = dict.get('Subtype');
    subtype = isName(subtype) ? subtype.name : '';
    if (!subtype) {
      return;
    }

    var fieldType = Util.getInheritableProperty(dict, 'FT');
    fieldType = isName(fieldType) ? fieldType.name : '';

    var Constructor = Annotation.getConstructor(subtype, fieldType);
    if (!Constructor) {
      return;
    }

    var params = {
      dict: dict,
      ref: ref,
    };

    var annotation = new Constructor(params);

    if (annotation.isViewable() || annotation.isPrintable()) {
      return annotation;
    } else {
      warn('unimplemented annotation type: ' + subtype);
    }
  };

  Annotation.appendToOperatorList = function Annotation_appendToOperatorList(
      annotations, opList, pdfManager, partialEvaluator, intent) {

    function reject(e) {
      annotationsReadyCapability.reject(e);
    }

    var annotationsReadyCapability = createPromiseCapability();

    var annotationPromises = [];
    for (var i = 0, n = annotations.length; i < n; ++i) {
      if (intent === 'display' && annotations[i].isViewable() ||
          intent === 'print' && annotations[i].isPrintable()) {
        annotationPromises.push(
          annotations[i].getOperatorList(partialEvaluator));
      }
    }
    Promise.all(annotationPromises).then(function(datas) {
      opList.addOp(OPS.beginAnnotations, []);
      for (var i = 0, n = datas.length; i < n; ++i) {
        var annotOpList = datas[i];
        opList.addOpList(annotOpList);
      }
      opList.addOp(OPS.endAnnotations, []);
      annotationsReadyCapability.resolve();
    }, reject);

    return annotationsReadyCapability.promise;
  };

  return Annotation;
})();
PDFJS.Annotation = Annotation;


var WidgetAnnotation = (function WidgetAnnotationClosure() {

  function WidgetAnnotation(params) {
    Annotation.call(this, params);

    if (params.data) {
      return;
    }

    var dict = params.dict;
    var data = this.data;

    data.fieldValue = stringToPDFString(
      Util.getInheritableProperty(dict, 'V') || '');
    data.alternativeText = stringToPDFString(dict.get('TU') || '');
    data.defaultAppearance = Util.getInheritableProperty(dict, 'DA') || '';
    var fieldType = Util.getInheritableProperty(dict, 'FT');
    data.fieldType = isName(fieldType) ? fieldType.name : '';
    data.fieldFlags = Util.getInheritableProperty(dict, 'Ff') || 0;
    this.fieldResources = Util.getInheritableProperty(dict, 'DR') || Dict.empty;

    // Building the full field name by collecting the field and
    // its ancestors 'T' data and joining them using '.'.
    var fieldName = [];
    var namedItem = dict;
    var ref = params.ref;
    while (namedItem) {
      var parent = namedItem.get('Parent');
      var parentRef = namedItem.getRaw('Parent');
      var name = namedItem.get('T');
      if (name) {
        fieldName.unshift(stringToPDFString(name));
      } else {
        // The field name is absent, that means more than one field
        // with the same name may exist. Replacing the empty name
        // with the '`' plus index in the parent's 'Kids' array.
        // This is not in the PDF spec but necessary to id the
        // the input controls.
        var kids = parent.get('Kids');
        var j, jj;
        for (j = 0, jj = kids.length; j < jj; j++) {
          var kidRef = kids[j];
          if (kidRef.num == ref.num && kidRef.gen == ref.gen) {
            break;
          }
        }
        fieldName.unshift('`' + j);
      }
      namedItem = parent;
      ref = parentRef;
    }
    data.fullName = fieldName.join('.');
  }

  var parent = Annotation.prototype;
  Util.inherit(WidgetAnnotation, Annotation, {
    isViewable: function WidgetAnnotation_isViewable() {
      if (this.data.fieldType === 'Sig') {
        warn('unimplemented annotation type: Widget signature');
        return false;
      }

      return parent.isViewable.call(this);
    }
  });

  return WidgetAnnotation;
})();

var TextWidgetAnnotation = (function TextWidgetAnnotationClosure() {
  function TextWidgetAnnotation(params) {
    WidgetAnnotation.call(this, params);

    if (params.data) {
      return;
    }

    this.data.textAlignment = Util.getInheritableProperty(params.dict, 'Q');
  }

  // TODO(mack): This dupes some of the logic in CanvasGraphics.setFont()
  function setTextStyles(element, item, fontObj) {

    var style = element.style;
    style.fontSize = item.fontSize + 'px';
    style.direction = item.fontDirection < 0 ? 'rtl': 'ltr';

    if (!fontObj) {
      return;
    }

    style.fontWeight = fontObj.black ?
                            (fontObj.bold ? 'bolder' : 'bold') :
                            (fontObj.bold ? 'bold' : 'normal');
    style.fontStyle = fontObj.italic ? 'italic' : 'normal';

    var fontName = fontObj.loadedName;
    var fontFamily = fontName ? '"' + fontName + '", ' : '';
    // Use a reasonable default font if the font doesn't specify a fallback
    var fallbackName = fontObj.fallbackName || 'Helvetica, sans-serif';
    style.fontFamily = fontFamily + fallbackName;
  }


  Util.inherit(TextWidgetAnnotation, WidgetAnnotation, {
    hasHtml: function TextWidgetAnnotation_hasHtml() {
      return !this.data.hasAppearance && !!this.data.fieldValue;
    },

    getHtmlElement: function TextWidgetAnnotation_getHtmlElement(commonObjs) {
      assert(!isWorker, 'getHtmlElement() shall be called from main thread');

      var item = this.data;

      var element = this.getEmptyContainer('div');
      element.style.display = 'table';

      var content = document.createElement('div');
      content.textContent = item.fieldValue;
      var textAlignment = item.textAlignment;
      content.style.textAlign = ['left', 'center', 'right'][textAlignment];
      content.style.verticalAlign = 'middle';
      content.style.display = 'table-cell';

      var fontObj = item.fontRefName ?
                    commonObjs.getData(item.fontRefName) : null;
      setTextStyles(content, item, fontObj);

      element.appendChild(content);

      return element;
    },

    getOperatorList: function TextWidgetAnnotation_getOperatorList(evaluator) {
      if (this.appearance) {
        return Annotation.prototype.getOperatorList.call(this, evaluator);
      }

      var opList = new OperatorList();
      var data = this.data;

      // Even if there is an appearance stream, ignore it. This is the
      // behaviour used by Adobe Reader.
      if (!data.defaultAppearance) {
        return Promise.resolve(opList);
      }

      var stream = new Stream(stringToBytes(data.defaultAppearance));
      return evaluator.getOperatorList(stream, this.fieldResources, opList).
        then(function () {
          return opList;
        });
    }
  });

  return TextWidgetAnnotation;
})();

var InteractiveAnnotation = (function InteractiveAnnotationClosure() {
  function InteractiveAnnotation(params) {
    Annotation.call(this, params);
  }

  Util.inherit(InteractiveAnnotation, Annotation, {
    hasHtml: function InteractiveAnnotation_hasHtml() {
      return true;
    },

    highlight: function InteractiveAnnotation_highlight() {
      if (this.highlightElement &&
         this.highlightElement.hasAttribute('hidden')) {
        this.highlightElement.removeAttribute('hidden');
      }
    },

    unhighlight: function InteractiveAnnotation_unhighlight() {
      if (this.highlightElement &&
         !this.highlightElement.hasAttribute('hidden')) {
        this.highlightElement.setAttribute('hidden', true);
      }
    },

    initContainer: function InteractiveAnnotation_initContainer() {

      var item = this.data;
      var rect = item.rect;

      var container = this.getEmptyContainer('section', rect, item.borderWidth);
      container.style.backgroundColor = item.color;

      var color = item.color;
      var rgb = [];
      for (var i = 0; i < 3; ++i) {
        rgb[i] = Math.round(color[i] * 255);
      }
      item.colorCssRgb = Util.makeCssRgb(rgb);

      var highlight = document.createElement('div');
      highlight.className = 'annotationHighlight';
      highlight.style.left = highlight.style.top = -HIGHLIGHT_OFFSET + 'px';
      highlight.style.right = highlight.style.bottom = -HIGHLIGHT_OFFSET + 'px';
      highlight.setAttribute('hidden', true);

      this.highlightElement = highlight;
      container.appendChild(this.highlightElement);

      return container;
    }
  });

  return InteractiveAnnotation;
})();

var TextAnnotation = (function TextAnnotationClosure() {
  function TextAnnotation(params) {
    InteractiveAnnotation.call(this, params);

    if (params.data) {
      return;
    }

    var dict = params.dict;
    var data = this.data;

    var content = dict.get('Contents');
    var title = dict.get('T');
    data.content = stringToPDFString(content || '');
    data.title = stringToPDFString(title || '');

    if (data.hasAppearance) {
      data.name = 'NoIcon';
    } else {
      data.rect[1] = data.rect[3] - DEFAULT_ICON_SIZE;
      data.rect[2] = data.rect[0] + DEFAULT_ICON_SIZE;
      data.name = dict.has('Name') ? dict.get('Name').name : 'Note';
    }

    if (dict.has('C')) {
      data.hasBgColor = true;
    }
  }

  var ANNOT_MIN_SIZE = 10;

  Util.inherit(TextAnnotation, InteractiveAnnotation, {

    getHtmlElement: function TextAnnotation_getHtmlElement(commonObjs) {
      assert(!isWorker, 'getHtmlElement() shall be called from main thread');

      var item = this.data;
      var rect = item.rect;

      // sanity check because of OOo-generated PDFs
      if ((rect[3] - rect[1]) < ANNOT_MIN_SIZE) {
        rect[3] = rect[1] + ANNOT_MIN_SIZE;
      }
      if ((rect[2] - rect[0]) < ANNOT_MIN_SIZE) {
        rect[2] = rect[0] + (rect[3] - rect[1]); // make it square
      }

      var container = this.initContainer();
      container.className = 'annotText';

      var image  = document.createElement('img');
      image.style.height = container.style.height;
      image.style.width = container.style.width;
      var iconName = item.name;
      image.src = PDFJS.imageResourcesPath + 'annotation-' +
        iconName.toLowerCase() + '.svg';
      image.alt = '[{{type}} Annotation]';
      image.dataset.l10nId = 'text_annotation_type';
      image.dataset.l10nArgs = JSON.stringify({type: iconName});

      var contentWrapper = document.createElement('div');
      contentWrapper.className = 'annotTextContentWrapper';
      contentWrapper.style.left = Math.floor(rect[2] - rect[0] + 5) + 'px';
      contentWrapper.style.top = '-10px';

      var content = document.createElement('div');
      content.className = 'annotTextContent';
      content.setAttribute('hidden', true);

      var i, ii;
      if (item.hasBgColor) {
        var color = item.color;
        var rgb = [];
        for (i = 0; i < 3; ++i) {
          // Enlighten the color (70%)
          var c = Math.round(color[i] * 255);
          rgb[i] = Math.round((255 - c) * 0.7) + c;
        }
        content.style.backgroundColor = Util.makeCssRgb(rgb);
      }

      var title = document.createElement('h1');
      var text = document.createElement('p');
      title.textContent = item.title;

      if (!item.content && !item.title) {
        content.setAttribute('hidden', true);
      } else {
        var e = document.createElement('span');
        var lines = item.content.split(/(?:\r\n?|\n)/);
        for (i = 0, ii = lines.length; i < ii; ++i) {
          var line = lines[i];
          e.appendChild(document.createTextNode(line));
          if (i < (ii - 1)) {
            e.appendChild(document.createElement('br'));
          }
        }
        text.appendChild(e);

        var pinned = false;

        var showAnnotation = function showAnnotation(pin) {
          if (pin) {
            pinned = true;
          }
          if (content.hasAttribute('hidden')) {
            container.style.zIndex += 1;
            content.removeAttribute('hidden');
          }
        };

        var hideAnnotation = function hideAnnotation(unpin) {
          if (unpin) {
            pinned = false;
          }
          if (!content.hasAttribute('hidden') && !pinned) {
            container.style.zIndex -= 1;
            content.setAttribute('hidden', true);
          }
        };

        var toggleAnnotation = function toggleAnnotation() {
          if (pinned) {
            hideAnnotation(true);
          } else {
            showAnnotation(true);
          }
        };

        image.addEventListener('click', function image_clickHandler() {
          toggleAnnotation();
        }, false);
        image.addEventListener('mouseover', function image_mouseOverHandler() {
          showAnnotation();
        }, false);
        image.addEventListener('mouseout', function image_mouseOutHandler() {
          hideAnnotation();
        }, false);

        content.addEventListener('click', function content_clickHandler() {
          hideAnnotation(true);
        }, false);
      }

      content.appendChild(title);
      content.appendChild(text);
      contentWrapper.appendChild(content);
      container.appendChild(image);
      container.appendChild(contentWrapper);

      return container;
    }
  });

  return TextAnnotation;
})();

var LinkAnnotation = (function LinkAnnotationClosure() {
  function LinkAnnotation(params) {
    InteractiveAnnotation.call(this, params);

    if (params.data) {
      return;
    }

    var dict = params.dict;
    var data = this.data;

    var action = dict.get('A');
    if (action) {
      var linkType = action.get('S').name;
      if (linkType === 'URI') {
        var url = action.get('URI');
        if (isName(url)) {
          // Some bad PDFs do not put parentheses around relative URLs.
          url = '/' + url.name;
        } else if (url) {
          url = addDefaultProtocolToUrl(url);
        }
        // TODO: pdf spec mentions urls can be relative to a Base
        // entry in the dictionary.
        if (!isValidUrl(url, false)) {
          url = '';
        }
        data.url = url;
      } else if (linkType === 'GoTo') {
        data.dest = action.get('D');
      } else if (linkType === 'GoToR') {
        var urlDict = action.get('F');
        if (isDict(urlDict)) {
          // We assume that the 'url' is a Filspec dictionary
          // and fetch the url without checking any further
          url = urlDict.get('F') || '';
        }

        // TODO: pdf reference says that GoToR
        // can also have 'NewWindow' attribute
        if (!isValidUrl(url, false)) {
          url = '';
        }
        data.url = url;
        data.dest = action.get('D');
      } else if (linkType === 'Named') {
        data.action = action.get('N').name;
      } else {
        warn('unrecognized link type: ' + linkType);
      }
    } else if (dict.has('Dest')) {
      // simple destination link
      var dest = dict.get('Dest');
      data.dest = isName(dest) ? dest.name : dest;
    }
  }

  // Lets URLs beginning with 'www.' default to using the 'http://' protocol.
  function addDefaultProtocolToUrl(url) {
    if (url && url.indexOf('www.') === 0) {
      return ('http://' + url);
    }
    return url;
  }

  Util.inherit(LinkAnnotation, InteractiveAnnotation, {
    hasOperatorList: function LinkAnnotation_hasOperatorList() {
      return false;
    },

    getHtmlElement: function LinkAnnotation_getHtmlElement(commonObjs) {

      var container = this.initContainer();
      container.className = 'annotLink';

      var item = this.data;

      container.style.borderColor = item.colorCssRgb;
      container.style.borderStyle = 'solid';

      var link = document.createElement('a');
      link.href = link.title = this.data.url || '';

      container.appendChild(link);

      return container;
    }
  });

  return LinkAnnotation;
})();




var NetworkManager = (function NetworkManagerClosure() {

  var OK_RESPONSE = 200;
  var PARTIAL_CONTENT_RESPONSE = 206;

  function NetworkManager(url, args) {
    this.url = url;
    args = args || {};
    this.isHttp = /^https?:/i.test(url);
    this.httpHeaders = (this.isHttp && args.httpHeaders) || {};
    this.withCredentials = args.withCredentials || false;
    this.getXhr = args.getXhr ||
      function NetworkManager_getXhr() {
        return new XMLHttpRequest();
      };

    this.currXhrId = 0;
    this.pendingRequests = {};
    this.loadedRequests = {};
  }

  function getArrayBuffer(xhr) {
    var data = (xhr.mozResponseArrayBuffer || xhr.mozResponse ||
                xhr.responseArrayBuffer || xhr.response);
    if (typeof data !== 'string') {
      return data;
    }
    var length = data.length;
    var buffer = new Uint8Array(length);
    for (var i = 0; i < length; i++) {
      buffer[i] = data.charCodeAt(i) & 0xFF;
    }
    return buffer;
  }

  NetworkManager.prototype = {
    requestRange: function NetworkManager_requestRange(begin, end, listeners) {
      var args = {
        begin: begin,
        end: end
      };
      for (var prop in listeners) {
        args[prop] = listeners[prop];
      }
      return this.request(args);
    },

    requestFull: function NetworkManager_requestRange(listeners) {
      return this.request(listeners);
    },

    request: function NetworkManager_requestRange(args) {
      var xhr = this.getXhr();
      var xhrId = this.currXhrId++;
      var pendingRequest = this.pendingRequests[xhrId] = {
        xhr: xhr
      };

      xhr.open('GET', this.url);
      xhr.withCredentials = this.withCredentials;
      for (var property in this.httpHeaders) {
        var value = this.httpHeaders[property];
        if (typeof value === 'undefined') {
          continue;
        }
        xhr.setRequestHeader(property, value);
      }
      if (this.isHttp && 'begin' in args && 'end' in args) {
        var rangeStr = args.begin + '-' + (args.end - 1);
        xhr.setRequestHeader('Range', 'bytes=' + rangeStr);
        pendingRequest.expectedStatus = 206;
      } else {
        pendingRequest.expectedStatus = 200;
      }

      xhr.mozResponseType = xhr.responseType = 'arraybuffer';

      if (args.onProgress) {
        xhr.onprogress = args.onProgress;
      }
      if (args.onError) {
        xhr.onerror = function(evt) {
          args.onError(xhr.status);
        };
      }
      xhr.onreadystatechange = this.onStateChange.bind(this, xhrId);

      pendingRequest.onHeadersReceived = args.onHeadersReceived;
      pendingRequest.onDone = args.onDone;
      pendingRequest.onError = args.onError;

      xhr.send(null);

      return xhrId;
    },

    onStateChange: function NetworkManager_onStateChange(xhrId, evt) {
      var pendingRequest = this.pendingRequests[xhrId];
      if (!pendingRequest) {
        // Maybe abortRequest was called...
        return;
      }

      var xhr = pendingRequest.xhr;
      if (xhr.readyState >= 2 && pendingRequest.onHeadersReceived) {
        pendingRequest.onHeadersReceived();
        delete pendingRequest.onHeadersReceived;
      }

      if (xhr.readyState !== 4) {
        return;
      }

      if (!(xhrId in this.pendingRequests)) {
        // The XHR request might have been aborted in onHeadersReceived()
        // callback, in which case we should abort request
        return;
      }

      delete this.pendingRequests[xhrId];

      // success status == 0 can be on ftp, file and other protocols
      if (xhr.status === 0 && this.isHttp) {
        if (pendingRequest.onError) {
          pendingRequest.onError(xhr.status);
        }
        return;
      }
      var xhrStatus = xhr.status || OK_RESPONSE;

      // From http://www.w3.org/Protocols/rfc2616/rfc2616-sec14.html#sec14.35.2:
      // "A server MAY ignore the Range header". This means it's possible to
      // get a 200 rather than a 206 response from a range request.
      var ok_response_on_range_request =
          xhrStatus === OK_RESPONSE &&
          pendingRequest.expectedStatus === PARTIAL_CONTENT_RESPONSE;

      if (!ok_response_on_range_request &&
          xhrStatus !== pendingRequest.expectedStatus) {
        if (pendingRequest.onError) {
          pendingRequest.onError(xhr.status);
        }
        return;
      }

      this.loadedRequests[xhrId] = true;

      var chunk = getArrayBuffer(xhr);
      if (xhrStatus === PARTIAL_CONTENT_RESPONSE) {
        var rangeHeader = xhr.getResponseHeader('Content-Range');
        var matches = /bytes (\d+)-(\d+)\/(\d+)/.exec(rangeHeader);
        var begin = parseInt(matches[1], 10);
        pendingRequest.onDone({
          begin: begin,
          chunk: chunk
        });
      } else {
        pendingRequest.onDone({
          begin: 0,
          chunk: chunk
        });
      }
    },

    hasPendingRequests: function NetworkManager_hasPendingRequests() {
      for (var xhrId in this.pendingRequests) {
        return true;
      }
      return false;
    },

    getRequestXhr: function NetworkManager_getXhr(xhrId) {
      return this.pendingRequests[xhrId].xhr;
    },

    isPendingRequest: function NetworkManager_isPendingRequest(xhrId) {
      return xhrId in this.pendingRequests;
    },

    isLoadedRequest: function NetworkManager_isLoadedRequest(xhrId) {
      return xhrId in this.loadedRequests;
    },

    abortAllRequests: function NetworkManager_abortAllRequests() {
      for (var xhrId in this.pendingRequests) {
        this.abortRequest(xhrId | 0);
      }
    },

    abortRequest: function NetworkManager_abortRequest(xhrId) {
      var xhr = this.pendingRequests[xhrId].xhr;
      delete this.pendingRequests[xhrId];
      xhr.abort();
    }
  };

  return NetworkManager;
})();



var ChunkedStream = (function ChunkedStreamClosure() {
  function ChunkedStream(length, chunkSize, manager) {
    this.bytes = new Uint8Array(length);
    this.start = 0;
    this.pos = 0;
    this.end = length;
    this.chunkSize = chunkSize;
    this.loadedChunks = [];
    this.numChunksLoaded = 0;
    this.numChunks = Math.ceil(length / chunkSize);
    this.manager = manager;
    this.initialDataLength = 0;
  }

  // required methods for a stream. if a particular stream does not
  // implement these, an error should be thrown
  ChunkedStream.prototype = {

    getMissingChunks: function ChunkedStream_getMissingChunks() {
      var chunks = [];
      for (var chunk = 0, n = this.numChunks; chunk < n; ++chunk) {
        if (!(chunk in this.loadedChunks)) {
          chunks.push(chunk);
        }
      }
      return chunks;
    },

    getBaseStreams: function ChunkedStream_getBaseStreams() {
      return [this];
    },

    allChunksLoaded: function ChunkedStream_allChunksLoaded() {
      return this.numChunksLoaded === this.numChunks;
    },

    onReceiveData: function ChunkedStream_onReceiveData(begin, chunk) {
      var end = begin + chunk.byteLength;

      assert(begin % this.chunkSize === 0, 'Bad begin offset: ' + begin);
      // Using this.length is inaccurate here since this.start can be moved
      // See ChunkedStream.moveStart()
      var length = this.bytes.length;
      assert(end % this.chunkSize === 0 || end === length,
             'Bad end offset: ' + end);

      this.bytes.set(new Uint8Array(chunk), begin);
      var chunkSize = this.chunkSize;
      var beginChunk = Math.floor(begin / chunkSize);
      var endChunk = Math.floor((end - 1) / chunkSize) + 1;
      var curChunk;

      for (curChunk = beginChunk; curChunk < endChunk; ++curChunk) {
        if (!(curChunk in this.loadedChunks)) {
          this.loadedChunks[curChunk] = true;
          ++this.numChunksLoaded;
        }
      }
    },

    onReceiveInitialData: function ChunkedStream_onReceiveInitialData(data) {
      this.bytes.set(data);
      this.initialDataLength = data.length;
      var endChunk = (this.end === data.length ?
        this.numChunks : Math.floor(data.length / this.chunkSize));
      for (var i = 0; i < endChunk; i++) {
        this.loadedChunks[i] = true;
        ++this.numChunksLoaded;
      }
    },

    ensureRange: function ChunkedStream_ensureRange(begin, end) {
      if (begin >= end) {
        return;
      }

      if (end <= this.initialDataLength) {
        return;
      }

      var chunkSize = this.chunkSize;
      var beginChunk = Math.floor(begin / chunkSize);
      var endChunk = Math.floor((end - 1) / chunkSize) + 1;
      for (var chunk = beginChunk; chunk < endChunk; ++chunk) {
        if (!(chunk in this.loadedChunks)) {
          throw new MissingDataException(begin, end);
        }
      }
    },

    nextEmptyChunk: function ChunkedStream_nextEmptyChunk(beginChunk) {
      var chunk, n;
      for (chunk = beginChunk, n = this.numChunks; chunk < n; ++chunk) {
        if (!(chunk in this.loadedChunks)) {
          return chunk;
        }
      }
      // Wrap around to beginning
      for (chunk = 0; chunk < beginChunk; ++chunk) {
        if (!(chunk in this.loadedChunks)) {
          return chunk;
        }
      }
      return null;
    },

    hasChunk: function ChunkedStream_hasChunk(chunk) {
      return chunk in this.loadedChunks;
    },

    get length() {
      return this.end - this.start;
    },

    get isEmpty() {
      return this.length === 0;
    },

    getByte: function ChunkedStream_getByte() {
      var pos = this.pos;
      if (pos >= this.end) {
        return -1;
      }
      this.ensureRange(pos, pos + 1);
      return this.bytes[this.pos++];
    },

    getUint16: function ChunkedStream_getUint16() {
      var b0 = this.getByte();
      var b1 = this.getByte();
      return (b0 << 8) + b1;
    },

    getInt32: function ChunkedStream_getInt32() {
      var b0 = this.getByte();
      var b1 = this.getByte();
      var b2 = this.getByte();
      var b3 = this.getByte();
      return (b0 << 24) + (b1 << 16) + (b2 << 8) + b3;
    },

    // returns subarray of original buffer
    // should only be read
    getBytes: function ChunkedStream_getBytes(length) {
      var bytes = this.bytes;
      var pos = this.pos;
      var strEnd = this.end;

      if (!length) {
        this.ensureRange(pos, strEnd);
        return bytes.subarray(pos, strEnd);
      }

      var end = pos + length;
      if (end > strEnd) {
        end = strEnd;
      }
      this.ensureRange(pos, end);

      this.pos = end;
      return bytes.subarray(pos, end);
    },

    peekBytes: function ChunkedStream_peekBytes(length) {
      var bytes = this.getBytes(length);
      this.pos -= bytes.length;
      return bytes;
    },

    getByteRange: function ChunkedStream_getBytes(begin, end) {
      this.ensureRange(begin, end);
      return this.bytes.subarray(begin, end);
    },

    skip: function ChunkedStream_skip(n) {
      if (!n) {
        n = 1;
      }
      this.pos += n;
    },

    reset: function ChunkedStream_reset() {
      this.pos = this.start;
    },

    moveStart: function ChunkedStream_moveStart() {
      this.start = this.pos;
    },

    makeSubStream: function ChunkedStream_makeSubStream(start, length, dict) {
      this.ensureRange(start, start + length);

      function ChunkedStreamSubstream() {}
      ChunkedStreamSubstream.prototype = Object.create(this);
      ChunkedStreamSubstream.prototype.getMissingChunks = function() {
        var chunkSize = this.chunkSize;
        var beginChunk = Math.floor(this.start / chunkSize);
        var endChunk = Math.floor((this.end - 1) / chunkSize) + 1;
        var missingChunks = [];
        for (var chunk = beginChunk; chunk < endChunk; ++chunk) {
          if (!(chunk in this.loadedChunks)) {
            missingChunks.push(chunk);
          }
        }
        return missingChunks;
      };
      var subStream = new ChunkedStreamSubstream();
      subStream.pos = subStream.start = start;
      subStream.end = start + length || this.end;
      subStream.dict = dict;
      return subStream;
    },

    isStream: true
  };

  return ChunkedStream;
})();

var ChunkedStreamManager = (function ChunkedStreamManagerClosure() {

  function ChunkedStreamManager(length, chunkSize, url, args) {
    this.stream = new ChunkedStream(length, chunkSize, this);
    this.length = length;
    this.chunkSize = chunkSize;
    this.url = url;
    this.disableAutoFetch = args.disableAutoFetch;
    var msgHandler = this.msgHandler = args.msgHandler;

    if (args.chunkedViewerLoading) {
      msgHandler.on('OnDataRange', this.onReceiveData.bind(this));
      msgHandler.on('OnDataProgress', this.onProgress.bind(this));
      this.sendRequest = function ChunkedStreamManager_sendRequest(begin, end) {
        msgHandler.send('RequestDataRange', { begin: begin, end: end });
      };
    } else {

      var getXhr = function getXhr() {
        return new XMLHttpRequest();
      };
      this.networkManager = new NetworkManager(this.url, {
        getXhr: getXhr,
        httpHeaders: args.httpHeaders,
        withCredentials: args.withCredentials
      });
      this.sendRequest = function ChunkedStreamManager_sendRequest(begin, end) {
        this.networkManager.requestRange(begin, end, {
          onDone: this.onReceiveData.bind(this),
          onProgress: this.onProgress.bind(this)
        });
      };
    }

    this.currRequestId = 0;

    this.chunksNeededByRequest = {};
    this.requestsByChunk = {};
    this.callbacksByRequest = {};

    this._loadedStreamCapability = createPromiseCapability();

    if (args.initialData) {
      this.setInitialData(args.initialData);
    }
  }

  ChunkedStreamManager.prototype = {

    setInitialData: function ChunkedStreamManager_setInitialData(data) {
      this.stream.onReceiveInitialData(data);
      if (this.stream.allChunksLoaded()) {
        this._loadedStreamCapability.resolve(this.stream);
      } else if (this.msgHandler) {
        this.msgHandler.send('DocProgress', {
          loaded: data.length,
          total: this.length
        });
      }
    },

    onLoadedStream: function ChunkedStreamManager_getLoadedStream() {
      return this._loadedStreamCapability.promise;
    },

    // Get all the chunks that are not yet loaded and groups them into
    // contiguous ranges to load in as few requests as possible
    requestAllChunks: function ChunkedStreamManager_requestAllChunks() {
      var missingChunks = this.stream.getMissingChunks();
      this.requestChunks(missingChunks);
      return this._loadedStreamCapability.promise;
    },

    requestChunks: function ChunkedStreamManager_requestChunks(chunks,
                                                               callback) {
      var requestId = this.currRequestId++;

      var chunksNeeded;
      var i, ii;
      this.chunksNeededByRequest[requestId] = chunksNeeded = {};
      for (i = 0, ii = chunks.length; i < ii; i++) {
        if (!this.stream.hasChunk(chunks[i])) {
          chunksNeeded[chunks[i]] = true;
        }
      }

      if (isEmptyObj(chunksNeeded)) {
        if (callback) {
          callback();
        }
        return;
      }

      this.callbacksByRequest[requestId] = callback;

      var chunksToRequest = [];
      for (var chunk in chunksNeeded) {
        chunk = chunk | 0;
        if (!(chunk in this.requestsByChunk)) {
          this.requestsByChunk[chunk] = [];
          chunksToRequest.push(chunk);
        }
        this.requestsByChunk[chunk].push(requestId);
      }

      if (!chunksToRequest.length) {
        return;
      }

      var groupedChunksToRequest = this.groupChunks(chunksToRequest);

      for (i = 0; i < groupedChunksToRequest.length; ++i) {
        var groupedChunk = groupedChunksToRequest[i];
        var begin = groupedChunk.beginChunk * this.chunkSize;
        var end = Math.min(groupedChunk.endChunk * this.chunkSize, this.length);
        this.sendRequest(begin, end);
      }
    },

    getStream: function ChunkedStreamManager_getStream() {
      return this.stream;
    },

    // Loads any chunks in the requested range that are not yet loaded
    requestRange: function ChunkedStreamManager_requestRange(
                      begin, end, callback) {

      end = Math.min(end, this.length);

      var beginChunk = this.getBeginChunk(begin);
      var endChunk = this.getEndChunk(end);

      var chunks = [];
      for (var chunk = beginChunk; chunk < endChunk; ++chunk) {
        chunks.push(chunk);
      }

      this.requestChunks(chunks, callback);
    },

    requestRanges: function ChunkedStreamManager_requestRanges(ranges,
                                                               callback) {
      ranges = ranges || [];
      var chunksToRequest = [];

      for (var i = 0; i < ranges.length; i++) {
        var beginChunk = this.getBeginChunk(ranges[i].begin);
        var endChunk = this.getEndChunk(ranges[i].end);
        for (var chunk = beginChunk; chunk < endChunk; ++chunk) {
          if (chunksToRequest.indexOf(chunk) < 0) {
            chunksToRequest.push(chunk);
          }
        }
      }

      chunksToRequest.sort(function(a, b) { return a - b; });
      this.requestChunks(chunksToRequest, callback);
    },

    // Groups a sorted array of chunks into as few continguous larger
    // chunks as possible
    groupChunks: function ChunkedStreamManager_groupChunks(chunks) {
      var groupedChunks = [];
      var beginChunk = -1;
      var prevChunk = -1;
      for (var i = 0; i < chunks.length; ++i) {
        var chunk = chunks[i];

        if (beginChunk < 0) {
          beginChunk = chunk;
        }

        if (prevChunk >= 0 && prevChunk + 1 !== chunk) {
          groupedChunks.push({ beginChunk: beginChunk,
                               endChunk: prevChunk + 1 });
          beginChunk = chunk;
        }
        if (i + 1 === chunks.length) {
          groupedChunks.push({ beginChunk: beginChunk,
                               endChunk: chunk + 1 });
        }

        prevChunk = chunk;
      }
      return groupedChunks;
    },

    onProgress: function ChunkedStreamManager_onProgress(args) {
      var bytesLoaded = (this.stream.numChunksLoaded * this.chunkSize +
                         args.loaded);
      this.msgHandler.send('DocProgress', {
        loaded: bytesLoaded,
        total: this.length
      });
    },

    onReceiveData: function ChunkedStreamManager_onReceiveData(args) {
      var chunk = args.chunk;
      var begin = args.begin;
      var end = begin + chunk.byteLength;

      var beginChunk = this.getBeginChunk(begin);
      var endChunk = this.getEndChunk(end);

      this.stream.onReceiveData(begin, chunk);
      if (this.stream.allChunksLoaded()) {
        this._loadedStreamCapability.resolve(this.stream);
      }

      var loadedRequests = [];
      var i, requestId;
      for (chunk = beginChunk; chunk < endChunk; ++chunk) {
        // The server might return more chunks than requested
        var requestIds = this.requestsByChunk[chunk] || [];
        delete this.requestsByChunk[chunk];

        for (i = 0; i < requestIds.length; ++i) {
          requestId = requestIds[i];
          var chunksNeeded = this.chunksNeededByRequest[requestId];
          if (chunk in chunksNeeded) {
            delete chunksNeeded[chunk];
          }

          if (!isEmptyObj(chunksNeeded)) {
            continue;
          }

          loadedRequests.push(requestId);
        }
      }

      // If there are no pending requests, automatically fetch the next
      // unfetched chunk of the PDF
      if (!this.disableAutoFetch && isEmptyObj(this.requestsByChunk)) {
        var nextEmptyChunk;
        if (this.stream.numChunksLoaded === 1) {
          // This is a special optimization so that after fetching the first
          // chunk, rather than fetching the second chunk, we fetch the last
          // chunk.
          var lastChunk = this.stream.numChunks - 1;
          if (!this.stream.hasChunk(lastChunk)) {
            nextEmptyChunk = lastChunk;
          }
        } else {
          nextEmptyChunk = this.stream.nextEmptyChunk(endChunk);
        }
        if (isInt(nextEmptyChunk)) {
          this.requestChunks([nextEmptyChunk]);
        }
      }

      for (i = 0; i < loadedRequests.length; ++i) {
        requestId = loadedRequests[i];
        var callback = this.callbacksByRequest[requestId];
        delete this.callbacksByRequest[requestId];
        if (callback) {
          callback();
        }
      }

      this.msgHandler.send('DocProgress', {
        loaded: this.stream.numChunksLoaded * this.chunkSize,
        total: this.length
      });
    },

    onError: function ChunkedStreamManager_onError(err) {
      this._loadedStreamCapability.reject(err);
    },

    getBeginChunk: function ChunkedStreamManager_getBeginChunk(begin) {
      var chunk = Math.floor(begin / this.chunkSize);
      return chunk;
    },

    getEndChunk: function ChunkedStreamManager_getEndChunk(end) {
      if (end % this.chunkSize === 0) {
        return end / this.chunkSize;
      }

      // 0 -> 0
      // 1 -> 1
      // 99 -> 1
      // 100 -> 1
      // 101 -> 2
      var chunk = Math.floor((end - 1) / this.chunkSize) + 1;
      return chunk;
    }
  };

  return ChunkedStreamManager;
})();



// The maximum number of bytes fetched per range request
var RANGE_CHUNK_SIZE = 65536;

// TODO(mack): Make use of PDFJS.Util.inherit() when it becomes available
var BasePdfManager = (function BasePdfManagerClosure() {
  function BasePdfManager() {
    throw new Error('Cannot initialize BaseManagerManager');
  }

  BasePdfManager.prototype = {
    onLoadedStream: function BasePdfManager_onLoadedStream() {
      throw new NotImplementedException();
    },

    ensureDoc: function BasePdfManager_ensureDoc(prop, args) {
      return this.ensure(this.pdfDocument, prop, args);
    },

    ensureXRef: function BasePdfManager_ensureXRef(prop, args) {
      return this.ensure(this.pdfDocument.xref, prop, args);
    },

    ensureCatalog: function BasePdfManager_ensureCatalog(prop, args) {
      return this.ensure(this.pdfDocument.catalog, prop, args);
    },

    getPage: function BasePdfManager_pagePage(pageIndex) {
      return this.pdfDocument.getPage(pageIndex);
    },

    cleanup: function BasePdfManager_cleanup() {
      return this.pdfDocument.cleanup();
    },

    ensure: function BasePdfManager_ensure(obj, prop, args) {
      return new NotImplementedException();
    },

    requestRange: function BasePdfManager_ensure(begin, end) {
      return new NotImplementedException();
    },

    requestLoadedStream: function BasePdfManager_requestLoadedStream() {
      return new NotImplementedException();
    },

    updatePassword: function BasePdfManager_updatePassword(password) {
      this.pdfDocument.xref.password = this.password = password;
      if (this._passwordChangedCapability) {
        this._passwordChangedCapability.resolve();
      }
    },

    passwordChanged: function BasePdfManager_passwordChanged() {
      this._passwordChangedCapability = createPromiseCapability();
      return this._passwordChangedCapability.promise;
    },

    terminate: function BasePdfManager_terminate() {
      return new NotImplementedException();
    }
  };

  return BasePdfManager;
})();

var LocalPdfManager = (function LocalPdfManagerClosure() {
  function LocalPdfManager(data, password) {
    var stream = new Stream(data);
    this.pdfDocument = new PDFDocument(this, stream, password);
    this._loadedStreamCapability = createPromiseCapability();
    this._loadedStreamCapability.resolve(stream);
  }

  LocalPdfManager.prototype = Object.create(BasePdfManager.prototype);
  LocalPdfManager.prototype.constructor = LocalPdfManager;

  LocalPdfManager.prototype.ensure =
      function LocalPdfManager_ensure(obj, prop, args) {
    return new Promise(function (resolve, reject) {
      try {
        var value = obj[prop];
        var result;
        if (typeof value === 'function') {
          result = value.apply(obj, args);
        } else {
          result = value;
        }
        resolve(result);
      } catch (e) {
        reject(e);
      }
    });
  };

  LocalPdfManager.prototype.requestRange =
      function LocalPdfManager_requestRange(begin, end) {
    return Promise.resolve();
  };

  LocalPdfManager.prototype.requestLoadedStream =
      function LocalPdfManager_requestLoadedStream() {
  };

  LocalPdfManager.prototype.onLoadedStream =
      function LocalPdfManager_getLoadedStream() {
    return this._loadedStreamCapability.promise;
  };

  LocalPdfManager.prototype.terminate =
      function LocalPdfManager_terminate() {
    return;
  };

  return LocalPdfManager;
})();

var NetworkPdfManager = (function NetworkPdfManagerClosure() {
  function NetworkPdfManager(args, msgHandler) {

    this.msgHandler = msgHandler;

    var params = {
      msgHandler: msgHandler,
      httpHeaders: args.httpHeaders,
      withCredentials: args.withCredentials,
      chunkedViewerLoading: args.chunkedViewerLoading,
      disableAutoFetch: args.disableAutoFetch,
      initialData: args.initialData
    };
    this.streamManager = new ChunkedStreamManager(args.length, RANGE_CHUNK_SIZE,
                                                  args.url, params);

    this.pdfDocument = new PDFDocument(this, this.streamManager.getStream(),
                                    args.password);
  }

  NetworkPdfManager.prototype = Object.create(BasePdfManager.prototype);
  NetworkPdfManager.prototype.constructor = NetworkPdfManager;

  NetworkPdfManager.prototype.ensure =
      function NetworkPdfManager_ensure(obj, prop, args) {
    var pdfManager = this;

    return new Promise(function (resolve, reject) {
      function ensureHelper() {
        try {
          var result;
          var value = obj[prop];
          if (typeof value === 'function') {
            result = value.apply(obj, args);
          } else {
            result = value;
          }
          resolve(result);
        } catch(e) {
          if (!(e instanceof MissingDataException)) {
            reject(e);
            return;
          }
          pdfManager.streamManager.requestRange(e.begin, e.end, ensureHelper);
        }
      }

      ensureHelper();
    });
  };

  NetworkPdfManager.prototype.requestRange =
      function NetworkPdfManager_requestRange(begin, end) {
    return new Promise(function (resolve) {
      this.streamManager.requestRange(begin, end, function() {
        resolve();
      });
    }.bind(this));
  };

  NetworkPdfManager.prototype.requestLoadedStream =
      function NetworkPdfManager_requestLoadedStream() {
    this.streamManager.requestAllChunks();
  };

  NetworkPdfManager.prototype.onLoadedStream =
      function NetworkPdfManager_getLoadedStream() {
    return this.streamManager.onLoadedStream();
  };

  NetworkPdfManager.prototype.terminate =
      function NetworkPdfManager_terminate() {
    this.streamManager.networkManager.abortAllRequests();
  };

  return NetworkPdfManager;
})();



var Page = (function PageClosure() {

  var LETTER_SIZE_MEDIABOX = [0, 0, 612, 792];

  function Page(pdfManager, xref, pageIndex, pageDict, ref, fontCache) {
    this.pdfManager = pdfManager;
    this.pageIndex = pageIndex;
    this.pageDict = pageDict;
    this.xref = xref;
    this.ref = ref;
    this.fontCache = fontCache;
    this.idCounters = {
      obj: 0
    };
    this.resourcesPromise = null;
  }

  Page.prototype = {
    getPageProp: function Page_getPageProp(key) {
      return this.pageDict.get(key);
    },

    getInheritedPageProp: function Page_inheritPageProp(key) {
      var dict = this.pageDict;
      var value = dict.get(key);
      while (value === undefined) {
        dict = dict.get('Parent');
        if (!dict) {
          break;
        }
        value = dict.get(key);
      }
      return value;
    },

    get content() {
      return this.getPageProp('Contents');
    },

    get resources() {
      var value = this.getInheritedPageProp('Resources');
      // For robustness: The spec states that a \Resources entry has to be
      // present, but can be empty. Some document omit it still. In this case
      // return an empty dictionary:
      if (value === undefined) {
        value = Dict.empty;
      }
      return shadow(this, 'resources', value);
    },

    get mediaBox() {
      var obj = this.getInheritedPageProp('MediaBox');
      // Reset invalid media box to letter size.
      if (!isArray(obj) || obj.length !== 4) {
        obj = LETTER_SIZE_MEDIABOX;
      }
      return shadow(this, 'mediaBox', obj);
    },

    get view() {
      var mediaBox = this.mediaBox;
      var cropBox = this.getInheritedPageProp('CropBox');
      if (!isArray(cropBox) || cropBox.length !== 4) {
        return shadow(this, 'view', mediaBox);
      }

      // From the spec, 6th ed., p.963:
      // "The crop, bleed, trim, and art boxes should not ordinarily
      // extend beyond the boundaries of the media box. If they do, they are
      // effectively reduced to their intersection with the media box."
      cropBox = Util.intersect(cropBox, mediaBox);
      if (!cropBox) {
        return shadow(this, 'view', mediaBox);
      }
      return shadow(this, 'view', cropBox);
    },

    get annotationRefs() {
      return shadow(this, 'annotationRefs',
                    this.getInheritedPageProp('Annots'));
    },

    get rotate() {
      var rotate = this.getInheritedPageProp('Rotate') || 0;
      // Normalize rotation so it's a multiple of 90 and between 0 and 270
      if (rotate % 90 !== 0) {
        rotate = 0;
      } else if (rotate >= 360) {
        rotate = rotate % 360;
      } else if (rotate < 0) {
        // The spec doesn't cover negatives, assume its counterclockwise
        // rotation. The following is the other implementation of modulo.
        rotate = ((rotate % 360) + 360) % 360;
      }
      return shadow(this, 'rotate', rotate);
    },

    getContentStream: function Page_getContentStream() {
      var content = this.content;
      var stream;
      if (isArray(content)) {
        // fetching items
        var xref = this.xref;
        var i, n = content.length;
        var streams = [];
        for (i = 0; i < n; ++i) {
          streams.push(xref.fetchIfRef(content[i]));
        }
        stream = new StreamsSequenceStream(streams);
      } else if (isStream(content)) {
        stream = content;
      } else {
        // replacing non-existent page content with empty one
        stream = new NullStream();
      }
      return stream;
    },

    loadResources: function Page_loadResources(keys) {
      if (!this.resourcesPromise) {
        // TODO: add async getInheritedPageProp and remove this.
        this.resourcesPromise = this.pdfManager.ensure(this, 'resources');
      }
      return this.resourcesPromise.then(function resourceSuccess() {
        var objectLoader = new ObjectLoader(this.resources.map,
                                            keys,
                                            this.xref);
        return objectLoader.load();
      }.bind(this));
    },

    getOperatorList: function Page_getOperatorList(handler, intent) {
      var self = this;

      var pdfManager = this.pdfManager;
      var contentStreamPromise = pdfManager.ensure(this, 'getContentStream',
                                                   []);
      var resourcesPromise = this.loadResources([
        'ExtGState',
        'ColorSpace',
        'Pattern',
        'Shading',
        'XObject',
        'Font'
        // ProcSet
        // Properties
      ]);

      var partialEvaluator = new PartialEvaluator(pdfManager, this.xref,
                                                  handler, this.pageIndex,
                                                  'p' + this.pageIndex + '_',
                                                  this.idCounters,
                                                  this.fontCache);

      var dataPromises = Promise.all([contentStreamPromise, resourcesPromise]);
      var pageListPromise = dataPromises.then(function(data) {
        var contentStream = data[0];
        var opList = new OperatorList(intent, handler, self.pageIndex);

        handler.send('StartRenderPage', {
          transparency: partialEvaluator.hasBlendModes(self.resources),
          pageIndex: self.pageIndex,
          intent: intent
        });
        return partialEvaluator.getOperatorList(contentStream, self.resources,
          opList).then(function () {
            return opList;
          });
      });

      var annotationsPromise = pdfManager.ensure(this, 'annotations');
      return Promise.all([pageListPromise, annotationsPromise]).then(
          function(datas) {
        var pageOpList = datas[0];
        var annotations = datas[1];

        if (annotations.length === 0) {
          pageOpList.flush(true);
          return pageOpList;
        }

        var annotationsReadyPromise = Annotation.appendToOperatorList(
          annotations, pageOpList, pdfManager, partialEvaluator, intent);
        return annotationsReadyPromise.then(function () {
          pageOpList.flush(true);
          return pageOpList;
        });
      });
    },

    extractTextContent: function Page_extractTextContent() {
      var handler = {
        on: function nullHandlerOn() {},
        send: function nullHandlerSend() {}
      };

      var self = this;

      var pdfManager = this.pdfManager;
      var contentStreamPromise = pdfManager.ensure(this, 'getContentStream',
                                                   []);

      var resourcesPromise = this.loadResources([
        'ExtGState',
        'XObject',
        'Font'
      ]);

      var dataPromises = Promise.all([contentStreamPromise,
                                      resourcesPromise]);
      return dataPromises.then(function(data) {
        var contentStream = data[0];
        var partialEvaluator = new PartialEvaluator(pdfManager, self.xref,
                                                    handler, self.pageIndex,
                                                    'p' + self.pageIndex + '_',
                                                    self.idCounters,
                                                    self.fontCache);

        return partialEvaluator.getTextContent(contentStream,
                                               self.resources);
      });
    },

    getAnnotationsData: function Page_getAnnotationsData() {
      var annotations = this.annotations;
      var annotationsData = [];
      for (var i = 0, n = annotations.length; i < n; ++i) {
        annotationsData.push(annotations[i].getData());
      }
      return annotationsData;
    },

    get annotations() {
      var annotations = [];
      var annotationRefs = (this.annotationRefs || []);
      for (var i = 0, n = annotationRefs.length; i < n; ++i) {
        var annotationRef = annotationRefs[i];
        var annotation = Annotation.fromRef(this.xref, annotationRef);
        if (annotation) {
          annotations.push(annotation);
        }
      }
      return shadow(this, 'annotations', annotations);
    }
  };

  return Page;
})();

/**
 * The `PDFDocument` holds all the data of the PDF file. Compared to the
 * `PDFDoc`, this one doesn't have any job management code.
 * Right now there exists one PDFDocument on the main thread + one object
 * for each worker. If there is no worker support enabled, there are two
 * `PDFDocument` objects on the main thread created.
 */
var PDFDocument = (function PDFDocumentClosure() {
  function PDFDocument(pdfManager, arg, password) {
    if (isStream(arg)) {
      init.call(this, pdfManager, arg, password);
    } else if (isArrayBuffer(arg)) {
      init.call(this, pdfManager, new Stream(arg), password);
    } else {
      error('PDFDocument: Unknown argument type');
    }
  }

  function init(pdfManager, stream, password) {
    assert(stream.length > 0, 'stream must have data');
    this.pdfManager = pdfManager;
    this.stream = stream;
    var xref = new XRef(this.stream, password, pdfManager);
    this.xref = xref;
  }

  function find(stream, needle, limit, backwards) {
    var pos = stream.pos;
    var end = stream.end;
    var strBuf = [];
    if (pos + limit > end) {
      limit = end - pos;
    }
    for (var n = 0; n < limit; ++n) {
      strBuf.push(String.fromCharCode(stream.getByte()));
    }
    var str = strBuf.join('');
    stream.pos = pos;
    var index = backwards ? str.lastIndexOf(needle) : str.indexOf(needle);
    if (index == -1) {
      return false; /* not found */
    }
    stream.pos += index;
    return true; /* found */
  }

  var DocumentInfoValidators = {
    get entries() {
      // Lazily build this since all the validation functions below are not
      // defined until after this file loads.
      return shadow(this, 'entries', {
        Title: isString,
        Author: isString,
        Subject: isString,
        Keywords: isString,
        Creator: isString,
        Producer: isString,
        CreationDate: isString,
        ModDate: isString,
        Trapped: isName
      });
    }
  };

  PDFDocument.prototype = {
    parse: function PDFDocument_parse(recoveryMode) {
      this.setup(recoveryMode);
      try {
        // checking if AcroForm is present
        this.acroForm = this.catalog.catDict.get('AcroForm');
        if (this.acroForm) {
          this.xfa = this.acroForm.get('XFA');
          var fields = this.acroForm.get('Fields');
          if ((!fields || !isArray(fields) || fields.length === 0) &&
              !this.xfa) {
            // no fields and no XFA -- not a form (?)
            this.acroForm = null;
          }
        }
      } catch (ex) {
        info('Something wrong with AcroForm entry');
        this.acroForm = null;
      }
    },

    get linearization() {
      var length = this.stream.length;
      var linearization = false;
      if (length) {
        try {
          linearization = new Linearization(this.stream);
          if (linearization.length != length) {
            linearization = false;
          }
        } catch (err) {
          if (err instanceof MissingDataException) {
            throw err;
          }

          info('The linearization data is not available ' +
               'or unreadable PDF data is found');
          linearization = false;
        }
      }
      // shadow the prototype getter with a data property
      return shadow(this, 'linearization', linearization);
    },
    get startXRef() {
      var stream = this.stream;
      var startXRef = 0;
      var linearization = this.linearization;
      if (linearization) {
        // Find end of first obj.
        stream.reset();
        if (find(stream, 'endobj', 1024)) {
          startXRef = stream.pos + 6;
        }
      } else {
        // Find startxref by jumping backward from the end of the file.
        var step = 1024;
        var found = false, pos = stream.end;
        while (!found && pos > 0) {
          pos -= step - 'startxref'.length;
          if (pos < 0) {
            pos = 0;
          }
          stream.pos = pos;
          found = find(stream, 'startxref', step, true);
        }
        if (found) {
          stream.skip(9);
          var ch;
          do {
            ch = stream.getByte();
          } while (Lexer.isSpace(ch));
          var str = '';
          while (ch >= 0x20 && ch <= 0x39) { // < '9'
            str += String.fromCharCode(ch);
            ch = stream.getByte();
          }
          startXRef = parseInt(str, 10);
          if (isNaN(startXRef)) {
            startXRef = 0;
          }
        }
      }
      // shadow the prototype getter with a data property
      return shadow(this, 'startXRef', startXRef);
    },
    get mainXRefEntriesOffset() {
      var mainXRefEntriesOffset = 0;
      var linearization = this.linearization;
      if (linearization) {
        mainXRefEntriesOffset = linearization.mainXRefEntriesOffset;
      }
      // shadow the prototype getter with a data property
      return shadow(this, 'mainXRefEntriesOffset', mainXRefEntriesOffset);
    },
    // Find the header, remove leading garbage and setup the stream
    // starting from the header.
    checkHeader: function PDFDocument_checkHeader() {
      var stream = this.stream;
      stream.reset();
      if (find(stream, '%PDF-', 1024)) {
        // Found the header, trim off any garbage before it.
        stream.moveStart();
        // Reading file format version
        var MAX_VERSION_LENGTH = 12;
        var version = '', ch;
        while ((ch = stream.getByte()) > 0x20) { // SPACE
          if (version.length >= MAX_VERSION_LENGTH) {
            break;
          }
          version += String.fromCharCode(ch);
        }
        // removing "%PDF-"-prefix
        this.pdfFormatVersion = version.substring(5);
        return;
      }
      // May not be a PDF file, continue anyway.
    },
    parseStartXRef: function PDFDocument_parseStartXRef() {
      var startXRef = this.startXRef;
      this.xref.setStartXRef(startXRef);
    },
    setup: function PDFDocument_setup(recoveryMode) {
      this.xref.parse(recoveryMode);
      this.catalog = new Catalog(this.pdfManager, this.xref);
    },
    get numPages() {
      var linearization = this.linearization;
      var num = linearization ? linearization.numPages : this.catalog.numPages;
      // shadow the prototype getter
      return shadow(this, 'numPages', num);
    },
    get documentInfo() {
      var docInfo = {
        PDFFormatVersion: this.pdfFormatVersion,
        IsAcroFormPresent: !!this.acroForm,
        IsXFAPresent: !!this.xfa
      };
      var infoDict;
      try {
        infoDict = this.xref.trailer.get('Info');
      } catch (err) {
        info('The document information dictionary is invalid.');
      }
      if (infoDict) {
        var validEntries = DocumentInfoValidators.entries;
        // Only fill the document info with valid entries from the spec.
        for (var key in validEntries) {
          if (infoDict.has(key)) {
            var value = infoDict.get(key);
            // Make sure the value conforms to the spec.
            if (validEntries[key](value)) {
              docInfo[key] = (typeof value !== 'string' ?
                              value : stringToPDFString(value));
            } else {
              info('Bad value in document info for "' + key + '"');
            }
          }
        }
      }
      return shadow(this, 'documentInfo', docInfo);
    },
    get fingerprint() {
      var xref = this.xref, hash, fileID = '';

      if (xref.trailer.has('ID')) {
        hash = stringToBytes(xref.trailer.get('ID')[0]);
      } else {
        hash = calculateMD5(this.stream.bytes.subarray(0, 100), 0, 100);
      }

      for (var i = 0, n = hash.length; i < n; i++) {
        fileID += hash[i].toString(16);
      }

      return shadow(this, 'fingerprint', fileID);
    },

    getPage: function PDFDocument_getPage(pageIndex) {
      return this.catalog.getPage(pageIndex);
    },

    cleanup: function PDFDocument_cleanup() {
      return this.catalog.cleanup();
    }
  };

  return PDFDocument;
})();



var Name = (function NameClosure() {
  function Name(name) {
    this.name = name;
  }

  Name.prototype = {};

  var nameCache = {};

  Name.get = function Name_get(name) {
    var nameValue = nameCache[name];
    return (nameValue ? nameValue : (nameCache[name] = new Name(name)));
  };

  return Name;
})();

var Cmd = (function CmdClosure() {
  function Cmd(cmd) {
    this.cmd = cmd;
  }

  Cmd.prototype = {};

  var cmdCache = {};

  Cmd.get = function Cmd_get(cmd) {
    var cmdValue = cmdCache[cmd];
    return (cmdValue ? cmdValue : (cmdCache[cmd] = new Cmd(cmd)));
  };

  return Cmd;
})();

var Dict = (function DictClosure() {
  var nonSerializable = function nonSerializableClosure() {
    return nonSerializable; // creating closure on some variable
  };

  var GETALL_DICTIONARY_TYPES_WHITELIST = {
    'Background': true,
    'ExtGState': true,
    'Halftone': true,
    'Layout': true,
    'Mask': true,
    'Pagination': true,
    'Printing': true
  };

  function isRecursionAllowedFor(dict) {
    if (!isName(dict.Type)) {
      return true;
    }
    var dictType = dict.Type.name;
    return GETALL_DICTIONARY_TYPES_WHITELIST[dictType] === true;
  }

  // xref is optional
  function Dict(xref) {
    // Map should only be used internally, use functions below to access.
    this.map = Object.create(null);
    this.xref = xref;
    this.objId = null;
    this.__nonSerializable__ = nonSerializable; // disable cloning of the Dict
  }

  Dict.prototype = {
    assignXref: function Dict_assignXref(newXref) {
      this.xref = newXref;
    },

    // automatically dereferences Ref objects
    get: function Dict_get(key1, key2, key3) {
      var value;
      var xref = this.xref;
      if (typeof (value = this.map[key1]) != 'undefined' || key1 in this.map ||
          typeof key2 == 'undefined') {
        return xref ? xref.fetchIfRef(value) : value;
      }
      if (typeof (value = this.map[key2]) != 'undefined' || key2 in this.map ||
          typeof key3 == 'undefined') {
        return xref ? xref.fetchIfRef(value) : value;
      }
      value = this.map[key3] || null;
      return xref ? xref.fetchIfRef(value) : value;
    },

    // Same as get(), but returns a promise and uses fetchIfRefAsync().
    getAsync: function Dict_getAsync(key1, key2, key3) {
      var value;
      var xref = this.xref;
      if (typeof (value = this.map[key1]) !== undefined || key1 in this.map ||
          typeof key2 === undefined) {
        if (xref) {
          return xref.fetchIfRefAsync(value);
        }
        return Promise.resolve(value);
      }
      if (typeof (value = this.map[key2]) !== undefined || key2 in this.map ||
          typeof key3 === undefined) {
        if (xref) {
          return xref.fetchIfRefAsync(value);
        }
        return Promise.resolve(value);
      }
      value = this.map[key3] || null;
      if (xref) {
        return xref.fetchIfRefAsync(value);
      }
      return Promise.resolve(value);
    },

    // no dereferencing
    getRaw: function Dict_getRaw(key) {
      return this.map[key];
    },

    // creates new map and dereferences all Refs
    getAll: function Dict_getAll() {
      var all = Object.create(null);
      var queue = null;
      var key, obj;
      for (key in this.map) {
        obj = this.get(key);
        if (obj instanceof Dict) {
          if (isRecursionAllowedFor(obj)) {
            (queue || (queue = [])).push({target: all, key: key, obj: obj});
          } else {
            all[key] = this.getRaw(key);
          }
        } else {
          all[key] = obj;
        }
      }
      if (!queue) {
        return all;
      }

      // trying to take cyclic references into the account
      var processed = Object.create(null);
      while (queue.length > 0) {
        var item = queue.shift();
        var itemObj = item.obj;
        var objId = itemObj.objId;
        if (objId && objId in processed) {
          item.target[item.key] = processed[objId];
          continue;
        }
        var dereferenced = Object.create(null);
        for (key in itemObj.map) {
          obj = itemObj.get(key);
          if (obj instanceof Dict) {
            if (isRecursionAllowedFor(obj)) {
              queue.push({target: dereferenced, key: key, obj: obj});
            } else {
              dereferenced[key] = itemObj.getRaw(key);
            }
          } else {
            dereferenced[key] = obj;
          }
        }
        if (objId) {
          processed[objId] = dereferenced;
        }
        item.target[item.key] = dereferenced;
      }
      return all;
    },

    set: function Dict_set(key, value) {
      this.map[key] = value;
    },

    has: function Dict_has(key) {
      return key in this.map;
    },

    forEach: function Dict_forEach(callback) {
      for (var key in this.map) {
        callback(key, this.get(key));
      }
    }
  };

  Dict.empty = new Dict(null);

  return Dict;
})();

var Ref = (function RefClosure() {
  function Ref(num, gen) {
    this.num = num;
    this.gen = gen;
  }

  Ref.prototype = {};

  return Ref;
})();

// The reference is identified by number and generation.
// This structure stores only one instance of the reference.
var RefSet = (function RefSetClosure() {
  function RefSet() {
    this.dict = {};
  }

  RefSet.prototype = {
    has: function RefSet_has(ref) {
      return ('R' + ref.num + '.' + ref.gen) in this.dict;
    },

    put: function RefSet_put(ref) {
      this.dict['R' + ref.num + '.' + ref.gen] = true;
    },

    remove: function RefSet_remove(ref) {
      delete this.dict['R' + ref.num + '.' + ref.gen];
    }
  };

  return RefSet;
})();

var RefSetCache = (function RefSetCacheClosure() {
  function RefSetCache() {
    this.dict = Object.create(null);
  }

  RefSetCache.prototype = {
    get: function RefSetCache_get(ref) {
      return this.dict['R' + ref.num + '.' + ref.gen];
    },

    has: function RefSetCache_has(ref) {
      return ('R' + ref.num + '.' + ref.gen) in this.dict;
    },

    put: function RefSetCache_put(ref, obj) {
      this.dict['R' + ref.num + '.' + ref.gen] = obj;
    },

    putAlias: function RefSetCache_putAlias(ref, aliasRef) {
      this.dict['R' + ref.num + '.' + ref.gen] = this.get(aliasRef);
    },

    forEach: function RefSetCache_forEach(fn, thisArg) {
      for (var i in this.dict) {
        fn.call(thisArg, this.dict[i]);
      }
    },

    clear: function RefSetCache_clear() {
      this.dict = Object.create(null);
    }
  };

  return RefSetCache;
})();

var Catalog = (function CatalogClosure() {
  function Catalog(pdfManager, xref) {
    this.pdfManager = pdfManager;
    this.xref = xref;
    this.catDict = xref.getCatalogObj();
    this.fontCache = new RefSetCache();
    assert(isDict(this.catDict),
      'catalog object is not a dictionary');

    this.pagePromises = [];
  }

  Catalog.prototype = {
    get metadata() {
      var streamRef = this.catDict.getRaw('Metadata');
      if (!isRef(streamRef)) {
        return shadow(this, 'metadata', null);
      }

      var encryptMetadata = (!this.xref.encrypt ? false :
                             this.xref.encrypt.encryptMetadata);

      var stream = this.xref.fetch(streamRef, !encryptMetadata);
      var metadata;
      if (stream && isDict(stream.dict)) {
        var type = stream.dict.get('Type');
        var subtype = stream.dict.get('Subtype');

        if (isName(type) && isName(subtype) &&
            type.name === 'Metadata' && subtype.name === 'XML') {
          // XXX: This should examine the charset the XML document defines,
          // however since there are currently no real means to decode
          // arbitrary charsets, let's just hope that the author of the PDF
          // was reasonable enough to stick with the XML default charset,
          // which is UTF-8.
          try {
            metadata = stringToUTF8String(bytesToString(stream.getBytes()));
          } catch (e) {
            info('Skipping invalid metadata.');
          }
        }
      }

      return shadow(this, 'metadata', metadata);
    },
    get toplevelPagesDict() {
      var pagesObj = this.catDict.get('Pages');
      assert(isDict(pagesObj), 'invalid top-level pages dictionary');
      // shadow the prototype getter
      return shadow(this, 'toplevelPagesDict', pagesObj);
    },
    get documentOutline() {
      var obj = null;
      try {
        obj = this.readDocumentOutline();
      } catch (ex) {
        if (ex instanceof MissingDataException) {
          throw ex;
        }
        warn('Unable to read document outline');
      }
      return shadow(this, 'documentOutline', obj);
    },
    readDocumentOutline: function Catalog_readDocumentOutline() {
      var xref = this.xref;
      var obj = this.catDict.get('Outlines');
      var root = { items: [] };
      if (isDict(obj)) {
        obj = obj.getRaw('First');
        var processed = new RefSet();
        if (isRef(obj)) {
          var queue = [{obj: obj, parent: root}];
          // to avoid recursion keeping track of the items
          // in the processed dictionary
          processed.put(obj);
          while (queue.length > 0) {
            var i = queue.shift();
            var outlineDict = xref.fetchIfRef(i.obj);
            if (outlineDict === null) {
              continue;
            }
            if (!outlineDict.has('Title')) {
              error('Invalid outline item');
            }
            var dest = outlineDict.get('A');
            if (dest) {
              dest = dest.get('D');
            } else if (outlineDict.has('Dest')) {
              dest = outlineDict.getRaw('Dest');
              if (isName(dest)) {
                dest = dest.name;
              }
            }
            var title = outlineDict.get('Title');
            var outlineItem = {
              dest: dest,
              title: stringToPDFString(title),
              color: outlineDict.get('C') || [0, 0, 0],
              count: outlineDict.get('Count'),
              bold: !!(outlineDict.get('F') & 2),
              italic: !!(outlineDict.get('F') & 1),
              items: []
            };
            i.parent.items.push(outlineItem);
            obj = outlineDict.getRaw('First');
            if (isRef(obj) && !processed.has(obj)) {
              queue.push({obj: obj, parent: outlineItem});
              processed.put(obj);
            }
            obj = outlineDict.getRaw('Next');
            if (isRef(obj) && !processed.has(obj)) {
              queue.push({obj: obj, parent: i.parent});
              processed.put(obj);
            }
          }
        }
      }
      return (root.items.length > 0 ? root.items : null);
    },
    get numPages() {
      var obj = this.toplevelPagesDict.get('Count');
      assert(
        isInt(obj),
        'page count in top level pages object is not an integer'
      );
      // shadow the prototype getter
      return shadow(this, 'num', obj);
    },
    get destinations() {
      function fetchDestination(dest) {
        return isDict(dest) ? dest.get('D') : dest;
      }

      var xref = this.xref;
      var dests = {}, nameTreeRef, nameDictionaryRef;
      var obj = this.catDict.get('Names');
      if (obj && obj.has('Dests')) {
        nameTreeRef = obj.getRaw('Dests');
      } else if (this.catDict.has('Dests')) {
        nameDictionaryRef = this.catDict.get('Dests');
      }

      if (nameDictionaryRef) {
        // reading simple destination dictionary
        obj = nameDictionaryRef;
        obj.forEach(function catalogForEach(key, value) {
          if (!value) {
            return;
          }
          dests[key] = fetchDestination(value);
        });
      }
      if (nameTreeRef) {
        var nameTree = new NameTree(nameTreeRef, xref);
        var names = nameTree.getAll();
        for (var name in names) {
          if (!names.hasOwnProperty(name)) {
            continue;
          }
          dests[name] = fetchDestination(names[name]);
        }
      }
      return shadow(this, 'destinations', dests);
    },
    get attachments() {
      var xref = this.xref;
      var attachments = null, nameTreeRef;
      var obj = this.catDict.get('Names');
      if (obj) {
        nameTreeRef = obj.getRaw('EmbeddedFiles');
      }

      if (nameTreeRef) {
        var nameTree = new NameTree(nameTreeRef, xref);
        var names = nameTree.getAll();
        for (var name in names) {
          if (!names.hasOwnProperty(name)) {
            continue;
          }
          var fs = new FileSpec(names[name], xref);
          if (!attachments) {
            attachments = {};
          }
          attachments[stringToPDFString(name)] = fs.serializable;
        }
      }
      return shadow(this, 'attachments', attachments);
    },
    get javaScript() {
      var xref = this.xref;
      var obj = this.catDict.get('Names');

      var javaScript = [];
      if (obj && obj.has('JavaScript')) {
        var nameTree = new NameTree(obj.getRaw('JavaScript'), xref);
        var names = nameTree.getAll();
        for (var name in names) {
          if (!names.hasOwnProperty(name)) {
            continue;
          }
          // We don't really use the JavaScript right now. This code is
          // defensive so we don't cause errors on document load.
          var jsDict = names[name];
          if (!isDict(jsDict)) {
            continue;
          }
          var type = jsDict.get('S');
          if (!isName(type) || type.name !== 'JavaScript') {
            continue;
          }
          var js = jsDict.get('JS');
          if (!isString(js) && !isStream(js)) {
            continue;
          }
          if (isStream(js)) {
            js = bytesToString(js.getBytes());
          }
          javaScript.push(stringToPDFString(js));
        }
      }

      // Append OpenAction actions to javaScript array
      var openactionDict = this.catDict.get('OpenAction');
      if (isDict(openactionDict)) {
        var objType = openactionDict.get('Type');
        var actionType = openactionDict.get('S');
        var action = openactionDict.get('N');
        var isPrintAction = (isName(objType) && objType.name === 'Action' &&
                            isName(actionType) && actionType.name === 'Named' &&
                            isName(action) && action.name === 'Print');
        
        if (isPrintAction) {
          javaScript.push('print(true);');
        }
      }

      return shadow(this, 'javaScript', javaScript);
    },

    cleanup: function Catalog_cleanup() {
      var promises = [];
      this.fontCache.forEach(function (promise) {
        promises.push(promise);
      });
      return Promise.all(promises).then(function (translatedFonts) {
        for (var i = 0, ii = translatedFonts.length; i < ii; i++) {
          var font = translatedFonts[i].dict;
          delete font.translated;
        }
        this.fontCache.clear();
      }.bind(this));
    },

    getPage: function Catalog_getPage(pageIndex) {
      if (!(pageIndex in this.pagePromises)) {
        this.pagePromises[pageIndex] = this.getPageDict(pageIndex).then(
          function (a) {
            var dict = a[0];
            var ref = a[1];
            return new Page(this.pdfManager, this.xref, pageIndex, dict, ref,
                            this.fontCache);
          }.bind(this)
        );
      }
      return this.pagePromises[pageIndex];
    },

    getPageDict: function Catalog_getPageDict(pageIndex) {
      var capability = createPromiseCapability();
      var nodesToVisit = [this.catDict.getRaw('Pages')];
      var currentPageIndex = 0;
      var xref = this.xref;

      function next() {
        while (nodesToVisit.length) {
          var currentNode = nodesToVisit.pop();

          if (isRef(currentNode)) {
            xref.fetchAsync(currentNode).then(function (obj) {
              if ((isDict(obj, 'Page') || (isDict(obj) && !obj.has('Kids')))) {
                if (pageIndex === currentPageIndex) {
                  capability.resolve([obj, currentNode]);
                } else {
                  currentPageIndex++;
                  next();
                }
                return;
              }
              nodesToVisit.push(obj);
              next();
            }.bind(this), capability.reject.bind(capability));
            return;
          }

          // must be a child page dictionary
          assert(
            isDict(currentNode),
            'page dictionary kid reference points to wrong type of object'
          );
          var count = currentNode.get('Count');
          // Skip nodes where the page can't be.
          if (currentPageIndex + count <= pageIndex) {
            currentPageIndex += count;
            continue;
          }

          var kids = currentNode.get('Kids');
          assert(isArray(kids), 'page dictionary kids object is not an array');
          if (count === kids.length) {
            // Nodes that don't have the page have been skipped and this is the
            // bottom of the tree which means the page requested must be a
            // descendant of this pages node. Ideally we would just resolve the
            // promise with the page ref here, but there is the case where more
            // pages nodes could link to single a page (see issue 3666 pdf). To
            // handle this push it back on the queue so if it is a pages node it
            // will be descended into.
            nodesToVisit = [kids[pageIndex - currentPageIndex]];
            currentPageIndex = pageIndex;
            continue;
          } else {
            for (var last = kids.length - 1; last >= 0; last--) {
              nodesToVisit.push(kids[last]);
            }
          }
        }
        capability.reject('Page index ' + pageIndex + ' not found.');
      }
      next();
      return capability.promise;
    },

    getPageIndex: function Catalog_getPageIndex(ref) {
      // The page tree nodes have the count of all the leaves below them. To get
      // how many pages are before we just have to walk up the tree and keep
      // adding the count of siblings to the left of the node.
      var xref = this.xref;
      function pagesBeforeRef(kidRef) {
        var total = 0;
        var parentRef;
        return xref.fetchAsync(kidRef).then(function (node) {
          if (!node) {
            return null;
          }
          parentRef = node.getRaw('Parent');
          return node.getAsync('Parent');
        }).then(function (parent) {
          if (!parent) {
            return null;
          }
          return parent.getAsync('Kids');
        }).then(function (kids) {
          if (!kids) {
            return null;
          }
          var kidPromises = [];
          var found = false;
          for (var i = 0; i < kids.length; i++) {
            var kid = kids[i];
            assert(isRef(kid), 'kids must be a ref');
            if (kid.num == kidRef.num) {
              found = true;
              break;
            }
            kidPromises.push(xref.fetchAsync(kid).then(function (kid) {
              if (kid.has('Count')) {
                var count = kid.get('Count');
                total += count;
              } else { // page leaf node
                total++;
              }
            }));
          }
          if (!found) {
            error('kid ref not found in parents kids');
          }
          return Promise.all(kidPromises).then(function () {
            return [total, parentRef];
          });
        });
      }

      var total = 0;
      function next(ref) {
        return pagesBeforeRef(ref).then(function (args) {
          if (!args) {
            return total;
          }
          var count = args[0];
          var parentRef = args[1];
          total += count;
          return next(parentRef);
        });
      }

      return next(ref);
    }
  };

  return Catalog;
})();

var XRef = (function XRefClosure() {
  function XRef(stream, password) {
    this.stream = stream;
    this.entries = [];
    this.xrefstms = {};
    // prepare the XRef cache
    this.cache = [];
    this.password = password;
  }

  XRef.prototype = {
    setStartXRef: function XRef_setStartXRef(startXRef) {
      // Store the starting positions of xref tables as we process them
      // so we can recover from missing data errors
      this.startXRefQueue = [startXRef];
    },

    parse: function XRef_parse(recoveryMode) {
      var trailerDict;
      if (!recoveryMode) {
        trailerDict = this.readXRef();
      } else {
        warn('Indexing all PDF objects');
        trailerDict = this.indexObjects();
      }
      trailerDict.assignXref(this);
      this.trailer = trailerDict;
      var encrypt = trailerDict.get('Encrypt');
      if (encrypt) {
        var ids = trailerDict.get('ID');
        var fileId = (ids && ids.length) ? ids[0] : '';
        this.encrypt = new CipherTransformFactory(encrypt, fileId,
                                                  this.password);
      }

      // get the root dictionary (catalog) object
      if (!(this.root = trailerDict.get('Root'))) {
        error('Invalid root reference');
      }
    },

    processXRefTable: function XRef_processXRefTable(parser) {
      if (!('tableState' in this)) {
        // Stores state of the table as we process it so we can resume
        // from middle of table in case of missing data error
        this.tableState = {
          entryNum: 0,
          streamPos: parser.lexer.stream.pos,
          parserBuf1: parser.buf1,
          parserBuf2: parser.buf2
        };
      }

      var obj = this.readXRefTable(parser);

      // Sanity check
      if (!isCmd(obj, 'trailer')) {
        error('Invalid XRef table: could not find trailer dictionary');
      }
      // Read trailer dictionary, e.g.
      // trailer
      //    << /Size 22
      //      /Root 20R
      //      /Info 10R
      //      /ID [ <81b14aafa313db63dbd6f981e49f94f4> ]
      //    >>
      // The parser goes through the entire stream << ... >> and provides
      // a getter interface for the key-value table
      var dict = parser.getObj();

      // The pdflib PDF generator can generate a nested trailer dictionary
      if (!isDict(dict) && dict.dict) {
        dict = dict.dict;
      }
      if (!isDict(dict)) {
        error('Invalid XRef table: could not parse trailer dictionary');
      }
      delete this.tableState;

      return dict;
    },

    readXRefTable: function XRef_readXRefTable(parser) {
      // Example of cross-reference table:
      // xref
      // 0 1                    <-- subsection header (first obj #, obj count)
      // 0000000000 65535 f     <-- actual object (offset, generation #, f/n)
      // 23 2                   <-- subsection header ... and so on ...
      // 0000025518 00002 n
      // 0000025635 00000 n
      // trailer
      // ...

      var stream = parser.lexer.stream;
      var tableState = this.tableState;
      stream.pos = tableState.streamPos;
      parser.buf1 = tableState.parserBuf1;
      parser.buf2 = tableState.parserBuf2;

      // Outer loop is over subsection headers
      var obj;

      while (true) {
        if (!('firstEntryNum' in tableState) || !('entryCount' in tableState)) {
          if (isCmd(obj = parser.getObj(), 'trailer')) {
            break;
          }
          tableState.firstEntryNum = obj;
          tableState.entryCount = parser.getObj();
        }

        var first = tableState.firstEntryNum;
        var count = tableState.entryCount;
        if (!isInt(first) || !isInt(count)) {
          error('Invalid XRef table: wrong types in subsection header');
        }
        // Inner loop is over objects themselves
        for (var i = tableState.entryNum; i < count; i++) {
          tableState.streamPos = stream.pos;
          tableState.entryNum = i;
          tableState.parserBuf1 = parser.buf1;
          tableState.parserBuf2 = parser.buf2;

          var entry = {};
          entry.offset = parser.getObj();
          entry.gen = parser.getObj();
          var type = parser.getObj();

          if (isCmd(type, 'f')) {
            entry.free = true;
          } else if (isCmd(type, 'n')) {
            entry.uncompressed = true;
          }

          // Validate entry obj
          if (!isInt(entry.offset) || !isInt(entry.gen) ||
              !(entry.free || entry.uncompressed)) {
            console.log(entry.offset, entry.gen, entry.free,
                        entry.uncompressed);
            error('Invalid entry in XRef subsection: ' + first + ', ' + count);
          }

          if (!this.entries[i + first]) {
            this.entries[i + first] = entry;
          }
        }

        tableState.entryNum = 0;
        tableState.streamPos = stream.pos;
        tableState.parserBuf1 = parser.buf1;
        tableState.parserBuf2 = parser.buf2;
        delete tableState.firstEntryNum;
        delete tableState.entryCount;
      }

      // Per issue 3248: hp scanners generate bad XRef
      if (first === 1 && this.entries[1] && this.entries[1].free) {
        // shifting the entries
        this.entries.shift();
      }

      // Sanity check: as per spec, first object must be free
      if (this.entries[0] && !this.entries[0].free) {
        error('Invalid XRef table: unexpected first object');
      }
      return obj;
    },

    processXRefStream: function XRef_processXRefStream(stream) {
      if (!('streamState' in this)) {
        // Stores state of the stream as we process it so we can resume
        // from middle of stream in case of missing data error
        var streamParameters = stream.dict;
        var byteWidths = streamParameters.get('W');
        var range = streamParameters.get('Index');
        if (!range) {
          range = [0, streamParameters.get('Size')];
        }

        this.streamState = {
          entryRanges: range,
          byteWidths: byteWidths,
          entryNum: 0,
          streamPos: stream.pos
        };
      }
      this.readXRefStream(stream);
      delete this.streamState;

      return stream.dict;
    },

    readXRefStream: function XRef_readXRefStream(stream) {
      var i, j;
      var streamState = this.streamState;
      stream.pos = streamState.streamPos;

      var byteWidths = streamState.byteWidths;
      var typeFieldWidth = byteWidths[0];
      var offsetFieldWidth = byteWidths[1];
      var generationFieldWidth = byteWidths[2];

      var entryRanges = streamState.entryRanges;
      while (entryRanges.length > 0) {
        var first = entryRanges[0];
        var n = entryRanges[1];

        if (!isInt(first) || !isInt(n)) {
          error('Invalid XRef range fields: ' + first + ', ' + n);
        }
        if (!isInt(typeFieldWidth) || !isInt(offsetFieldWidth) ||
            !isInt(generationFieldWidth)) {
          error('Invalid XRef entry fields length: ' + first + ', ' + n);
        }
        for (i = streamState.entryNum; i < n; ++i) {
          streamState.entryNum = i;
          streamState.streamPos = stream.pos;

          var type = 0, offset = 0, generation = 0;
          for (j = 0; j < typeFieldWidth; ++j) {
            type = (type << 8) | stream.getByte();
          }
          // if type field is absent, its default value is 1
          if (typeFieldWidth === 0) {
            type = 1;
          }
          for (j = 0; j < offsetFieldWidth; ++j) {
            offset = (offset << 8) | stream.getByte();
          }
          for (j = 0; j < generationFieldWidth; ++j) {
            generation = (generation << 8) | stream.getByte();
          }
          var entry = {};
          entry.offset = offset;
          entry.gen = generation;
          switch (type) {
            case 0:
              entry.free = true;
              break;
            case 1:
              entry.uncompressed = true;
              break;
            case 2:
              break;
            default:
              error('Invalid XRef entry type: ' + type);
          }
          if (!this.entries[first + i]) {
            this.entries[first + i] = entry;
          }
        }

        streamState.entryNum = 0;
        streamState.streamPos = stream.pos;
        entryRanges.splice(0, 2);
      }
    },

    indexObjects: function XRef_indexObjects() {
      // Simple scan through the PDF content to find objects,
      // trailers and XRef streams.
      function readToken(data, offset) {
        var token = '', ch = data[offset];
        while (ch !== 13 && ch !== 10) {
          if (++offset >= data.length) {
            break;
          }
          token += String.fromCharCode(ch);
          ch = data[offset];
        }
        return token;
      }
      function skipUntil(data, offset, what) {
        var length = what.length, dataLength = data.length;
        var skipped = 0;
        // finding byte sequence
        while (offset < dataLength) {
          var i = 0;
          while (i < length && data[offset + i] == what[i]) {
            ++i;
          }
          if (i >= length) {
            break; // sequence found
          }
          offset++;
          skipped++;
        }
        return skipped;
      }
      var trailerBytes = new Uint8Array([116, 114, 97, 105, 108, 101, 114]);
      var startxrefBytes = new Uint8Array([115, 116, 97, 114, 116, 120, 114,
                                          101, 102]);
      var endobjBytes = new Uint8Array([101, 110, 100, 111, 98, 106]);
      var xrefBytes = new Uint8Array([47, 88, 82, 101, 102]);

      var stream = this.stream;
      stream.pos = 0;
      var buffer = stream.getBytes();
      var position = stream.start, length = buffer.length;
      var trailers = [], xrefStms = [];
      while (position < length) {
        var ch = buffer[position];
        if (ch === 32 || ch === 9 || ch === 13 || ch === 10) {
          ++position;
          continue;
        }
        if (ch === 37) { // %-comment
          do {
            ++position;
            if (position >= length) {
              break;
            }
            ch = buffer[position];
          } while (ch !== 13 && ch !== 10);
          continue;
        }
        var token = readToken(buffer, position);
        var m;
        if (token === 'xref') {
          position += skipUntil(buffer, position, trailerBytes);
          trailers.push(position);
          position += skipUntil(buffer, position, startxrefBytes);
        } else if ((m = /^(\d+)\s+(\d+)\s+obj\b/.exec(token))) {
          this.entries[m[1]] = {
            offset: position,
            gen: m[2] | 0,
            uncompressed: true
          };

          var contentLength = skipUntil(buffer, position, endobjBytes) + 7;
          var content = buffer.subarray(position, position + contentLength);

          // checking XRef stream suspect
          // (it shall have '/XRef' and next char is not a letter)
          var xrefTagOffset = skipUntil(content, 0, xrefBytes);
          if (xrefTagOffset < contentLength &&
              content[xrefTagOffset + 5] < 64) {
            xrefStms.push(position);
            this.xrefstms[position] = 1; // don't read it recursively
          }

          position += contentLength;
        } else {
          position += token.length + 1;
        }
      }
      // reading XRef streams
      var i, ii;
      for (i = 0, ii = xrefStms.length; i < ii; ++i) {
        this.startXRefQueue.push(xrefStms[i]);
        this.readXRef(/* recoveryMode */ true);
      }
      // finding main trailer
      var dict;
      for (i = 0, ii = trailers.length; i < ii; ++i) {
        stream.pos = trailers[i];
        var parser = new Parser(new Lexer(stream), true, null);
        var obj = parser.getObj();
        if (!isCmd(obj, 'trailer')) {
          continue;
        }
        // read the trailer dictionary
        if (!isDict(dict = parser.getObj())) {
          continue;
        }
        // taking the first one with 'ID'
        if (dict.has('ID')) {
          return dict;
        }
      }
      // no tailer with 'ID', taking last one (if exists)
      if (dict) {
        return dict;
      }
      // nothing helps
      // calling error() would reject worker with an UnknownErrorException.
      throw new InvalidPDFException('Invalid PDF structure');
    },

    readXRef: function XRef_readXRef(recoveryMode) {
      var stream = this.stream;

      try {
        while (this.startXRefQueue.length) {
          var startXRef = this.startXRefQueue[0];

          stream.pos = startXRef + stream.start;

          var parser = new Parser(new Lexer(stream), true, null);
          var obj = parser.getObj();
          var dict;

          // Get dictionary
          if (isCmd(obj, 'xref')) {
            // Parse end-of-file XRef
            dict = this.processXRefTable(parser);
            if (!this.topDict) {
              this.topDict = dict;
            }

            // Recursively get other XRefs 'XRefStm', if any
            obj = dict.get('XRefStm');
            if (isInt(obj)) {
              var pos = obj;
              // ignore previously loaded xref streams
              // (possible infinite recursion)
              if (!(pos in this.xrefstms)) {
                this.xrefstms[pos] = 1;
                this.startXRefQueue.push(pos);
              }
            }
          } else if (isInt(obj)) {
            // Parse in-stream XRef
            if (!isInt(parser.getObj()) ||
                !isCmd(parser.getObj(), 'obj') ||
                !isStream(obj = parser.getObj())) {
              error('Invalid XRef stream');
            }
            dict = this.processXRefStream(obj);
            if (!this.topDict) {
              this.topDict = dict;
            }
            if (!dict) {
              error('Failed to read XRef stream');
            }
          } else {
            error('Invalid XRef stream header');
          }

          // Recursively get previous dictionary, if any
          obj = dict.get('Prev');
          if (isInt(obj)) {
            this.startXRefQueue.push(obj);
          } else if (isRef(obj)) {
            // The spec says Prev must not be a reference, i.e. "/Prev NNN"
            // This is a fallback for non-compliant PDFs, i.e. "/Prev NNN 0 R"
            this.startXRefQueue.push(obj.num);
          }

          this.startXRefQueue.shift();
        }

        return this.topDict;
      } catch (e) {
        if (e instanceof MissingDataException) {
          throw e;
        }
        info('(while reading XRef): ' + e);
      }

      if (recoveryMode) {
        return;
      }
      throw new XRefParseException();
    },

    getEntry: function XRef_getEntry(i) {
      var xrefEntry = this.entries[i];
      if (xrefEntry && !xrefEntry.free && xrefEntry.offset) {
        return xrefEntry;
      }
      return null;
    },

    fetchIfRef: function XRef_fetchIfRef(obj) {
      if (!isRef(obj)) {
        return obj;
      }
      return this.fetch(obj);
    },

    fetch: function XRef_fetch(ref, suppressEncryption) {
      assert(isRef(ref), 'ref object is not a reference');
      var num = ref.num;
      if (num in this.cache) {
        var cacheEntry = this.cache[num];
        return cacheEntry;
      }

      var xrefEntry = this.getEntry(num);

      // the referenced entry can be free
      if (xrefEntry === null) {
        return (this.cache[num] = null);
      }

      if (xrefEntry.uncompressed) {
        xrefEntry = this.fetchUncompressed(ref, xrefEntry, suppressEncryption);
      } else {
        xrefEntry = this.fetchCompressed(xrefEntry, suppressEncryption);
      }

      if (isDict(xrefEntry)) {
        xrefEntry.objId = 'R' + ref.num + '.' + ref.gen;
      }
      return xrefEntry;
    },

    fetchUncompressed: function XRef_fetchUncompressed(ref, xrefEntry,
                                                       suppressEncryption) {
      var gen = ref.gen;
      var num = ref.num;
      if (xrefEntry.gen !== gen) {
        error('inconsistent generation in XRef');
      }
      var stream = this.stream.makeSubStream(xrefEntry.offset +
                                             this.stream.start);
      var parser = new Parser(new Lexer(stream), true, this);
      var obj1 = parser.getObj();
      var obj2 = parser.getObj();
      var obj3 = parser.getObj();
      if (!isInt(obj1) || parseInt(obj1, 10) !== num ||
          !isInt(obj2) || parseInt(obj2, 10) !== gen ||
          !isCmd(obj3)) {
        error('bad XRef entry');
      }
      if (!isCmd(obj3, 'obj')) {
        // some bad PDFs use "obj1234" and really mean 1234
        if (obj3.cmd.indexOf('obj') === 0) {
          num = parseInt(obj3.cmd.substring(3), 10);
          if (!isNaN(num)) {
            return num;
          }
        }
        error('bad XRef entry');
      }
      if (this.encrypt && !suppressEncryption) {
        try {
          xrefEntry = parser.getObj(this.encrypt.createCipherTransform(num,
                                                                       gen));
        } catch (ex) {
          // Almost all streams must be encrypted, but sometimes
          // they are not, probably due to some broken generators.
          // Retrying without encryption...
          return this.fetch(ref, true);
        }
      } else {
        xrefEntry = parser.getObj();
      }
      if (!isStream(xrefEntry)) {
        this.cache[num] = xrefEntry;
      }
      return xrefEntry;
    },

    fetchCompressed: function XRef_fetchCompressed(xrefEntry,
                                                   suppressEncryption) {
      var tableOffset = xrefEntry.offset;
      var stream = this.fetch(new Ref(tableOffset, 0));
      if (!isStream(stream)) {
        error('bad ObjStm stream');
      }
      var first = stream.dict.get('First');
      var n = stream.dict.get('N');
      if (!isInt(first) || !isInt(n)) {
        error('invalid first and n parameters for ObjStm stream');
      }
      var parser = new Parser(new Lexer(stream), false, this);
      parser.allowStreams = true;
      var i, entries = [], num, nums = [];
      // read the object numbers to populate cache
      for (i = 0; i < n; ++i) {
        num = parser.getObj();
        if (!isInt(num)) {
          error('invalid object number in the ObjStm stream: ' + num);
        }
        nums.push(num);
        var offset = parser.getObj();
        if (!isInt(offset)) {
          error('invalid object offset in the ObjStm stream: ' + offset);
        }
      }
      // read stream objects for cache
      for (i = 0; i < n; ++i) {
        entries.push(parser.getObj());
        num = nums[i];
        var entry = this.entries[num];
        if (entry && entry.offset === tableOffset && entry.gen === i) {
          this.cache[num] = entries[i];
        }
      }
      xrefEntry = entries[xrefEntry.gen];
      if (xrefEntry === undefined) {
        error('bad XRef entry for compressed object');
      }
      return xrefEntry;
    },

    fetchIfRefAsync: function XRef_fetchIfRefAsync(obj) {
      if (!isRef(obj)) {
        return Promise.resolve(obj);
      }
      return this.fetchAsync(obj);
    },

    fetchAsync: function XRef_fetchAsync(ref, suppressEncryption) {
      return new Promise(function (resolve, reject) {
          var tryFetch = function () {
            try {
              resolve(this.fetch(ref, suppressEncryption));
            } catch (e) {
              if (e instanceof MissingDataException) {
                this.stream.manager.requestRange(e.begin, e.end, tryFetch);
                return;
              }
              reject(e);
            }
          }.bind(this);
          tryFetch();
        }.bind(this));
      },

    getCatalogObj: function XRef_getCatalogObj() {
      return this.root;
    }
  };

  return XRef;
})();

/**
 * A NameTree is like a Dict but has some advantageous properties, see the
 * spec (7.9.6) for more details.
 * TODO: implement all the Dict functions and make this more efficent.
 */
var NameTree = (function NameTreeClosure() {
  function NameTree(root, xref) {
    this.root = root;
    this.xref = xref;
  }

  NameTree.prototype = {
    getAll: function NameTree_getAll() {
      var dict = {};
      if (!this.root) {
        return dict;
      }
      var xref = this.xref;
      // reading name tree
      var processed = new RefSet();
      processed.put(this.root);
      var queue = [this.root];
      while (queue.length > 0) {
        var i, n;
        var obj = xref.fetchIfRef(queue.shift());
        if (!isDict(obj)) {
          continue;
        }
        if (obj.has('Kids')) {
          var kids = obj.get('Kids');
          for (i = 0, n = kids.length; i < n; i++) {
            var kid = kids[i];
            if (processed.has(kid)) {
              error('invalid destinations');
            }
            queue.push(kid);
            processed.put(kid);
          }
          continue;
        }
        var names = obj.get('Names');
        if (names) {
          for (i = 0, n = names.length; i < n; i += 2) {
            dict[names[i]] = xref.fetchIfRef(names[i + 1]);
          }
        }
      }
      return dict;
    }
  };
  return NameTree;
})();

/**
 * "A PDF file can refer to the contents of another file by using a File 
 * Specification (PDF 1.1)", see the spec (7.11) for more details.
 * NOTE: Only embedded files are supported (as part of the attachments support)
 * TODO: support the 'URL' file system (with caching if !/V), portable 
 * collections attributes and related files (/RF)
 */
var FileSpec = (function FileSpecClosure() {
  function FileSpec(root, xref) {
    if (!root || !isDict(root)) {
      return;
    }
    this.xref = xref;
    this.root = root;
    if (root.has('FS')) {
      this.fs = root.get('FS');
    }
    this.description = root.has('Desc') ?
                         stringToPDFString(root.get('Desc')) :
                         '';
    if (root.has('RF')) {
      warn('Related file specifications are not supported');
    }
    this.contentAvailable = true;
    if (!root.has('EF')) {
      this.contentAvailable = false;
      warn('Non-embedded file specifications are not supported');
    }
  }

  function pickPlatformItem(dict) {
    // Look for the filename in this order:
    // UF, F, Unix, Mac, DOS
    if (dict.has('UF')) {
      return dict.get('UF');
    } else if (dict.has('F')) {
      return dict.get('F');
    } else if (dict.has('Unix')) {
      return dict.get('Unix');
    } else if (dict.has('Mac')) {
      return dict.get('Mac');
    } else if (dict.has('DOS')) {
      return dict.get('DOS');
    } else {
      return null;
    }
  }

  FileSpec.prototype = {
    get filename() {
      if (!this._filename && this.root) {
        var filename = pickPlatformItem(this.root) || 'unnamed';
        this._filename = stringToPDFString(filename).
          replace(/\\\\/g, '\\').
          replace(/\\\//g, '/').
          replace(/\\/g, '/');
      }
      return this._filename;
    },
    get content() {
      if (!this.contentAvailable) {
        return null;
      }
      if (!this.contentRef && this.root) {
        this.contentRef = pickPlatformItem(this.root.get('EF'));
      }
      var content = null;
      if (this.contentRef) {
        var xref = this.xref;
        var fileObj = xref.fetchIfRef(this.contentRef);
        if (fileObj && isStream(fileObj)) {
          content = fileObj.getBytes();
        } else {
          warn('Embedded file specification points to non-existing/invalid ' +
            'content');
        }
      } else {
        warn('Embedded file specification does not have a content');
      }
      return content;
    },
    get serializable() {
      return {
        filename: this.filename,
        content: this.content
      };
    }
  };
  return FileSpec;
})();

/**
 * A helper for loading missing data in object graphs. It traverses the graph
 * depth first and queues up any objects that have missing data. Once it has
 * has traversed as many objects that are available it attempts to bundle the
 * missing data requests and then resume from the nodes that weren't ready.
 *
 * NOTE: It provides protection from circular references by keeping track of
 * of loaded references. However, you must be careful not to load any graphs
 * that have references to the catalog or other pages since that will cause the
 * entire PDF document object graph to be traversed.
 */
var ObjectLoader = (function() {
  function mayHaveChildren(value) {
    return isRef(value) || isDict(value) || isArray(value) || isStream(value);
  }

  function addChildren(node, nodesToVisit) {
    var value;
    if (isDict(node) || isStream(node)) {
      var map;
      if (isDict(node)) {
        map = node.map;
      } else {
        map = node.dict.map;
      }
      for (var key in map) {
        value = map[key];
        if (mayHaveChildren(value)) {
          nodesToVisit.push(value);
        }
      }
    } else if (isArray(node)) {
      for (var i = 0, ii = node.length; i < ii; i++) {
        value = node[i];
        if (mayHaveChildren(value)) {
          nodesToVisit.push(value);
        }
      }
    }
  }

  function ObjectLoader(obj, keys, xref) {
    this.obj = obj;
    this.keys = keys;
    this.xref = xref;
    this.refSet = null;
  }

  ObjectLoader.prototype = {
    load: function ObjectLoader_load() {
      var keys = this.keys;
      this.capability = createPromiseCapability();
      // Don't walk the graph if all the data is already loaded.
      if (!(this.xref.stream instanceof ChunkedStream) ||
          this.xref.stream.getMissingChunks().length === 0) {
        this.capability.resolve();
        return this.capability.promise;
      }

      this.refSet = new RefSet();
      // Setup the initial nodes to visit.
      var nodesToVisit = [];
      for (var i = 0; i < keys.length; i++) {
        nodesToVisit.push(this.obj[keys[i]]);
      }

      this.walk(nodesToVisit);
      return this.capability.promise;
    },

    walk: function ObjectLoader_walk(nodesToVisit) {
      var nodesToRevisit = [];
      var pendingRequests = [];
      // DFS walk of the object graph.
      while (nodesToVisit.length) {
        var currentNode = nodesToVisit.pop();

        // Only references or chunked streams can cause missing data exceptions.
        if (isRef(currentNode)) {
          // Skip nodes that have already been visited.
          if (this.refSet.has(currentNode)) {
            continue;
          }
          try {
            var ref = currentNode;
            this.refSet.put(ref);
            currentNode = this.xref.fetch(currentNode);
          } catch (e) {
            if (!(e instanceof MissingDataException)) {
              throw e;
            }
            nodesToRevisit.push(currentNode);
            pendingRequests.push({ begin: e.begin, end: e.end });
          }
        }
        if (currentNode && currentNode.getBaseStreams) {
          var baseStreams = currentNode.getBaseStreams();
          var foundMissingData = false;
          for (var i = 0; i < baseStreams.length; i++) {
            var stream = baseStreams[i];
            if (stream.getMissingChunks && stream.getMissingChunks().length) {
              foundMissingData = true;
              pendingRequests.push({
                begin: stream.start,
                end: stream.end
              });
            }
          }
          if (foundMissingData) {
            nodesToRevisit.push(currentNode);
          }
        }

        addChildren(currentNode, nodesToVisit);
      }

      if (pendingRequests.length) {
        this.xref.stream.manager.requestRanges(pendingRequests,
            function pendingRequestCallback() {
          nodesToVisit = nodesToRevisit;
          for (var i = 0; i < nodesToRevisit.length; i++) {
            var node = nodesToRevisit[i];
            // Remove any reference nodes from the currrent refset so they
            // aren't skipped when we revist them.
            if (isRef(node)) {
              this.refSet.remove(node);
            }
          }
          this.walk(nodesToVisit);
        }.bind(this));
        return;
      }
      // Everything is loaded.
      this.refSet = null;
      this.capability.resolve();
    }
  };

  return ObjectLoader;
})();


var ISOAdobeCharset = [
  '.notdef', 'space', 'exclam', 'quotedbl', 'numbersign', 'dollar',
  'percent', 'ampersand', 'quoteright', 'parenleft', 'parenright',
  'asterisk', 'plus', 'comma', 'hyphen', 'period', 'slash', 'zero',
  'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight',
  'nine', 'colon', 'semicolon', 'less', 'equal', 'greater', 'question',
  'at', 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M',
  'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z',
  'bracketleft', 'backslash', 'bracketright', 'asciicircum', 'underscore',
  'quoteleft', 'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l',
  'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z',
  'braceleft', 'bar', 'braceright', 'asciitilde', 'exclamdown', 'cent',
  'sterling', 'fraction', 'yen', 'florin', 'section', 'currency',
  'quotesingle', 'quotedblleft', 'guillemotleft', 'guilsinglleft',
  'guilsinglright', 'fi', 'fl', 'endash', 'dagger', 'daggerdbl',
  'periodcentered', 'paragraph', 'bullet', 'quotesinglbase',
  'quotedblbase', 'quotedblright', 'guillemotright', 'ellipsis',
  'perthousand', 'questiondown', 'grave', 'acute', 'circumflex', 'tilde',
  'macron', 'breve', 'dotaccent', 'dieresis', 'ring', 'cedilla',
  'hungarumlaut', 'ogonek', 'caron', 'emdash', 'AE', 'ordfeminine',
  'Lslash', 'Oslash', 'OE', 'ordmasculine', 'ae', 'dotlessi', 'lslash',
  'oslash', 'oe', 'germandbls', 'onesuperior', 'logicalnot', 'mu',
  'trademark', 'Eth', 'onehalf', 'plusminus', 'Thorn', 'onequarter',
  'divide', 'brokenbar', 'degree', 'thorn', 'threequarters', 'twosuperior',
  'registered', 'minus', 'eth', 'multiply', 'threesuperior', 'copyright',
  'Aacute', 'Acircumflex', 'Adieresis', 'Agrave', 'Aring', 'Atilde',
  'Ccedilla', 'Eacute', 'Ecircumflex', 'Edieresis', 'Egrave', 'Iacute',
  'Icircumflex', 'Idieresis', 'Igrave', 'Ntilde', 'Oacute', 'Ocircumflex',
  'Odieresis', 'Ograve', 'Otilde', 'Scaron', 'Uacute', 'Ucircumflex',
  'Udieresis', 'Ugrave', 'Yacute', 'Ydieresis', 'Zcaron', 'aacute',
  'acircumflex', 'adieresis', 'agrave', 'aring', 'atilde', 'ccedilla',
  'eacute', 'ecircumflex', 'edieresis', 'egrave', 'iacute', 'icircumflex',
  'idieresis', 'igrave', 'ntilde', 'oacute', 'ocircumflex', 'odieresis',
  'ograve', 'otilde', 'scaron', 'uacute', 'ucircumflex', 'udieresis',
  'ugrave', 'yacute', 'ydieresis', 'zcaron'
];

var ExpertCharset = [
  '.notdef', 'space', 'exclamsmall', 'Hungarumlautsmall', 'dollaroldstyle',
  'dollarsuperior', 'ampersandsmall', 'Acutesmall', 'parenleftsuperior',
  'parenrightsuperior', 'twodotenleader', 'onedotenleader', 'comma',
  'hyphen', 'period', 'fraction', 'zerooldstyle', 'oneoldstyle',
  'twooldstyle', 'threeoldstyle', 'fouroldstyle', 'fiveoldstyle',
  'sixoldstyle', 'sevenoldstyle', 'eightoldstyle', 'nineoldstyle',
  'colon', 'semicolon', 'commasuperior', 'threequartersemdash',
  'periodsuperior', 'questionsmall', 'asuperior', 'bsuperior',
  'centsuperior', 'dsuperior', 'esuperior', 'isuperior', 'lsuperior',
  'msuperior', 'nsuperior', 'osuperior', 'rsuperior', 'ssuperior',
  'tsuperior', 'ff', 'fi', 'fl', 'ffi', 'ffl', 'parenleftinferior',
  'parenrightinferior', 'Circumflexsmall', 'hyphensuperior', 'Gravesmall',
  'Asmall', 'Bsmall', 'Csmall', 'Dsmall', 'Esmall', 'Fsmall', 'Gsmall',
  'Hsmall', 'Ismall', 'Jsmall', 'Ksmall', 'Lsmall', 'Msmall', 'Nsmall',
  'Osmall', 'Psmall', 'Qsmall', 'Rsmall', 'Ssmall', 'Tsmall', 'Usmall',
  'Vsmall', 'Wsmall', 'Xsmall', 'Ysmall', 'Zsmall', 'colonmonetary',
  'onefitted', 'rupiah', 'Tildesmall', 'exclamdownsmall', 'centoldstyle',
  'Lslashsmall', 'Scaronsmall', 'Zcaronsmall', 'Dieresissmall',
  'Brevesmall', 'Caronsmall', 'Dotaccentsmall', 'Macronsmall',
  'figuredash', 'hypheninferior', 'Ogoneksmall', 'Ringsmall',
  'Cedillasmall', 'onequarter', 'onehalf', 'threequarters',
  'questiondownsmall', 'oneeighth', 'threeeighths', 'fiveeighths',
  'seveneighths', 'onethird', 'twothirds', 'zerosuperior', 'onesuperior',
  'twosuperior', 'threesuperior', 'foursuperior', 'fivesuperior',
  'sixsuperior', 'sevensuperior', 'eightsuperior', 'ninesuperior',
  'zeroinferior', 'oneinferior', 'twoinferior', 'threeinferior',
  'fourinferior', 'fiveinferior', 'sixinferior', 'seveninferior',
  'eightinferior', 'nineinferior', 'centinferior', 'dollarinferior',
  'periodinferior', 'commainferior', 'Agravesmall', 'Aacutesmall',
  'Acircumflexsmall', 'Atildesmall', 'Adieresissmall', 'Aringsmall',
  'AEsmall', 'Ccedillasmall', 'Egravesmall', 'Eacutesmall',
  'Ecircumflexsmall', 'Edieresissmall', 'Igravesmall', 'Iacutesmall',
  'Icircumflexsmall', 'Idieresissmall', 'Ethsmall', 'Ntildesmall',
  'Ogravesmall', 'Oacutesmall', 'Ocircumflexsmall', 'Otildesmall',
  'Odieresissmall', 'OEsmall', 'Oslashsmall', 'Ugravesmall', 'Uacutesmall',
  'Ucircumflexsmall', 'Udieresissmall', 'Yacutesmall', 'Thornsmall',
  'Ydieresissmall'
];

var ExpertSubsetCharset = [
  '.notdef', 'space', 'dollaroldstyle', 'dollarsuperior',
  'parenleftsuperior', 'parenrightsuperior', 'twodotenleader',
  'onedotenleader', 'comma', 'hyphen', 'period', 'fraction',
  'zerooldstyle', 'oneoldstyle', 'twooldstyle', 'threeoldstyle',
  'fouroldstyle', 'fiveoldstyle', 'sixoldstyle', 'sevenoldstyle',
  'eightoldstyle', 'nineoldstyle', 'colon', 'semicolon', 'commasuperior',
  'threequartersemdash', 'periodsuperior', 'asuperior', 'bsuperior',
  'centsuperior', 'dsuperior', 'esuperior', 'isuperior', 'lsuperior',
  'msuperior', 'nsuperior', 'osuperior', 'rsuperior', 'ssuperior',
  'tsuperior', 'ff', 'fi', 'fl', 'ffi', 'ffl', 'parenleftinferior',
  'parenrightinferior', 'hyphensuperior', 'colonmonetary', 'onefitted',
  'rupiah', 'centoldstyle', 'figuredash', 'hypheninferior', 'onequarter',
  'onehalf', 'threequarters', 'oneeighth', 'threeeighths', 'fiveeighths',
  'seveneighths', 'onethird', 'twothirds', 'zerosuperior', 'onesuperior',
  'twosuperior', 'threesuperior', 'foursuperior', 'fivesuperior',
  'sixsuperior', 'sevensuperior', 'eightsuperior', 'ninesuperior',
  'zeroinferior', 'oneinferior', 'twoinferior', 'threeinferior',
  'fourinferior', 'fiveinferior', 'sixinferior', 'seveninferior',
  'eightinferior', 'nineinferior', 'centinferior', 'dollarinferior',
  'periodinferior', 'commainferior'
];



var CIDToUnicodeMaps = {
  'Adobe-Japan1': [[32, 160], {f: 12, c: 33}, [45, 8209], {f: 46, c: 46}, 165,
    {f: 2, c: 93}, [95, 818], [96, 768], {f: 27, c: 97}, 166, 125, [732, 771],
    [700, 8217], 92, [699, 8216], 124, [126, 8764], {f: 3, c: 161}, 8260, 402,
    0, 164, 8220, 171, {f: 2, c: 8249}, {f: 2, c: 64257}, [8210, 8211], 0, 0,
    [183, 8729], 0, 8226, 8218, 8222, 8221, 187, 0, 0, 191, {f: 2, c: 769},
    [175, 772], {f: 3, c: 774}, 778, [184, 807], 779, 808, 780, [822, 8212],
    198, 170, 321, 216, 338, 186, 230, 305, 322, 248, 339, 223, 173, 169, 172,
    174, 0, 0, {f: 2, c: 178}, 181, 185, {f: 3, c: 188}, {f: 6, c: 192},
    {f: 16, c: 199}, 0, {f: 6, c: 217}, {f: 6, c: 224}, {f: 16, c: 231}, 0,
    {f: 7, c: 249}, 352, 376, 381, [773, 8254], 353, 8482, 382, 0, 8194,
    {f: 59, c: 33}, 165, {f: 31, c: 93}, 65512, {f: 2, c: 125}, 0,
    {f: 63, c: 65377}, {s: 243}, [8195, 12288],
    {f: 2, c: 12289}, 65292, 65294, 12539, {f: 2, c: 65306}, 65311, 65281,
    {f: 2, c: 12443}, 180, 65344, 168, 65342, 65507, 65343, {f: 2, c: 12541},
    {f: 2, c: 12445}, 12291, 20189, {f: 3, c: 12293}, 12540, 8213, 8208, 65295,
    65340, [12316, 65374], 8214, 65372, 8230, 8229, {s: 4}, {f: 2, c: 65288},
    {f: 2, c: 12308}, 65339, 65341, 65371, 65373, {f: 10, c: 12296}, 65291,
    [8722, 65293], 177, 215, 247, 65309, 8800, 65308, 65310, {f: 2, c: 8806},
    8734, 8756, 9794, 9792, 176, {f: 2, c: 8242}, 8451, 65509, 65284,
    {f: 2, c: 65504}, 65285, 65283, 65286, 65290, 65312, 167, 9734, 9733, 9675,
    9679, 9678, 9671, 9670, 9633, 9632, 9651, 9650, 9661, 9660, 8251, 12306,
    8594, {f: 2, c: 8592}, 8595, 12307, 8712, 8715, {f: 2, c: 8838},
    {f: 2, c: 8834}, 8746, 8745, {f: 2, c: 8743}, 65506, 8658, 8660, 8704,
    8707, 8736, 8869, 8978, 8706, 8711, 8801, 8786, {f: 2, c: 8810}, 8730,
    8765, 8733, 8757, {f: 2, c: 8747}, 8491, 8240, 9839, 9837, 9834,
    {f: 2, c: 8224}, 182, 9711, {f: 10, c: 65296}, {f: 26, c: 65313},
    {f: 26, c: 65345}, {f: 83, c: 12353}, {f: 86, c: 12449}, {f: 17, c: 913},
    {f: 7, c: 931}, {f: 17, c: 945}, {f: 7, c: 963}, {f: 6, c: 1040}, 1025,
    {f: 32, c: 1046}, 1105, {f: 26, c: 1078}, 20124, 21782, 23043, 38463,
    21696, 24859, 25384, 23030, 36898, 33909, 33564, 31312, 24746, 25569,
    28197, 26093, 33894, 33446, 39925, 26771, 22311, 26017, 25201, 23451,
    22992, 34427, 39156, 32098, 32190, 39822, 25110, 31903, 34999, 23433,
    24245, 25353, 26263, 26696, 38343, 38797, 26447, 20197, 20234, 20301,
    20381, 20553, 22258, 22839, 22996, 23041, 23561, 24799, 24847, 24944,
    26131, 26885, 28858, 30031, 30064, 31227, 32173, 32239, 32963, 33806,
    [12176, 34915], 35586, 36949, 36986, 21307, 20117, 20133, 22495, 32946,
    37057, 30959, [12032, 19968], 22769, 28322, 36920, 31282, 33576, 33419,
    39983, 20801, 21360, 21693, 21729, 22240, 23035, 24341, 39154, 28139,
    32996, 34093, 38498, 38512, 38560, 38907, 21515, 21491, 23431, 28879,
    [12155, 32701], 36802, [12204, 38632], 21359, 40284, 31418, 19985, 30867,
    [12165, 33276], 28198, 22040, 21764, 27421, 34074, 39995, 23013, 21417,
    28006, [12128, 29916], 38287, 22082, 20113, 36939, 38642, 33615, 39180,
    21473, 21942, 23344, 24433, 26144, 26355, 26628, 27704, 27891, 27945,
    29787, 30408, 31310, 38964, 33521, 34907, 35424, 37613, 28082, 30123,
    30410, 39365, 24742, 35585, 36234, 38322, 27022, 21421, 20870, 22290,
    22576, 22852, 23476, 24310, 24616, 25513, 25588, 27839, 28436, 28814,
    28948, 29017, 29141, 29503, 32257, 33398, 33489, 34199, 36960, 37467,
    40219, 22633, 26044, 27738, 29989, 20985, 22830, 22885, 24448, 24540,
    25276, 26106, 27178, 27431, 27572, 29579, 32705, 35158, 40236, 40206,
    [12009, 40644], 23713, 27798, 33659, 20740, 23627, 25014, 33222, 26742,
    29281, [12036, 20057], 20474, 21368, 24681, 28201, 31311, [12211, 38899],
    19979, 21270, 20206, 20309, 20285, 20385, 20339, 21152, 21487, 22025,
    22799, 23233, 23478, 23521, 31185, 26247, 26524, 26550, 27468, 27827,
    [12117, 28779], 29634, 31117, [12146, 31166], 31292, 31623, 33457, 33499,
    33540, 33655, 33775, 33747, 34662, 35506, 22057, 36008, 36838, 36942,
    38686, 34442, 20420, 23784, 25105, [12123, 29273], 30011, 33253, 33469,
    34558, 36032, 38597, 39187, 39381, 20171, 20250, 35299, 22238, 22602,
    22730, 24315, 24555, 24618, 24724, 24674, 25040, 25106, 25296, 25913,
    39745, 26214, 26800, 28023, 28784, 30028, 30342, 32117, 33445, 34809,
    38283, 38542, [12185, 35997], 20977, 21182, 22806, 21683, 23475, 23830,
    24936, 27010, 28079, 30861, 33995, 34903, 35442, 37799, 39608, 28012,
    39336, 34521, 22435, 26623, 34510, 37390, 21123, 22151, 21508, 24275,
    25313, 25785, 26684, 26680, 27579, 29554, 30906, 31339, 35226,
    [12179, 35282], 36203, 36611, 37101, 38307, 38548, [12208, 38761], 23398,
    23731, 27005, {f: 2, c: 38989}, 25499, 31520, 27179, 27263, 26806, 39949,
    28511, 21106, 21917, 24688, 25324, 27963, 28167, 28369, 33883, 35088,
    36676, 19988, 39993, 21494, 26907, 27194, 38788, 26666, 20828, 31427,
    33970, 37340, 37772, 22107, 40232, 26658, 33541, 33841, 31909, 21000,
    33477, [12129, 29926], 20094, 20355, 20896, 23506, 21002, 21208, 21223,
    24059, 21914, 22570, 23014, 23436, 23448, 23515, [12082, 24178], 24185,
    24739, 24863, 24931, 25022, 25563, 25954, 26577, 26707, 26874, 27454,
    27475, 27735, 28450, 28567, 28485, 29872, [12130, 29976], 30435, 30475,
    31487, 31649, 31777, 32233, [12152, 32566], 32752, 32925, 33382, 33694,
    35251, 35532, 36011, 36996, 37969, 38291, 38289, 38306, 38501, 38867,
    39208, 33304, 20024, 21547, 23736, 24012, 29609, 30284, 30524, 23721,
    32747, 36107, 38593, 38929, 38996, 39000, 20225, 20238, 21361, 21916,
    22120, 22522, 22855, 23305, 23492, 23696, 24076, 24190, 24524, 25582,
    26426, 26071, 26082, 26399, 26827, 26820, 27231, 24112, 27589, 27671,
    27773, 30079, 31048, 23395, 31232, 32000, 24509, 35215, 35352, 36020,
    36215, 36556, 36637, 39138, 39438, [12004, 12225, 39740], [12018, 20096],
    20605, 20736, 22931, 23452, 25135, 25216, 25836, 27450, 29344, 30097,
    31047, 32681, 34811, 35516, 35696, 25516, 33738, 38816, 21513, 21507,
    21931, 26708, 27224, 35440, 30759, 26485, [12233, 40653], 21364, 23458,
    33050, 34384, 36870, 19992, 20037, 20167, 20241, 21450, 21560, 23470,
    [12088, 24339], 24613, 25937, 26429, 27714, 27762, 27875, 28792, 29699,
    31350, 31406, 31496, 32026, 31998, 32102, 26087, [12124, 29275], 21435,
    23621, 24040, 25298, 25312, 25369, 28192, 34394, 35377, 36317, 37624,
    28417, 31142, [12226, 39770], 20136, {f: 2, c: 20139}, 20379, 20384, 20689,
    20807, 31478, 20849, 20982, 21332, 21281, 21375, 21483, 21932, 22659,
    23777, 24375, 24394, 24623, 24656, 24685, 25375, 25945, 27211, 27841,
    29378, 29421, 30703, 33016, 33029, 33288, 34126, 37111, 37857, 38911,
    39255, 39514, 20208, 20957, 23597, 26241, 26989, 23616, 26354, 26997,
    [12127, 29577], 26704, 31873, 20677, 21220, 22343, [12081, 24062], 37670,
    [12100, 26020], 27427, 27453, 29748, 31105, 31165, 31563, 32202, 33465,
    33740, 34943, 35167, 35641, 36817, [12198, 37329], 21535, 37504, 20061,
    20534, 21477, 21306, 29399, 29590, 30697, 33510, 36527, 39366, 39368,
    39378, 20855, 24858, 34398, 21936, 31354, 20598, 23507, 36935, 38533,
    20018, 27355, 37351, 23633, 23624, 25496, 31391, 27795, 38772, 36705,
    31402, 29066, 38536, 31874, 26647, 32368, 26705, 37740, 21234, 21531,
    34219, 35347, 32676, 36557, 37089, 21350, 34952, 31041, 20418, 20670,
    21009, 20804, 21843, 22317, 29674, 22411, 22865, 24418, 24452, 24693,
    24950, 24935, 25001, 25522, 25658, 25964, 26223, 26690, 28179, 30054,
    31293, 31995, 32076, 32153, 32331, 32619, 33550, 33610, 34509, 35336,
    35427, 35686, 36605, 38938, 40335, 33464, 36814, 39912, 21127, 25119,
    25731, 28608, 38553, 26689, 20625, [12107, 27424], 27770, 28500,
    [12147, 31348], 32080, [12174, 34880], 35363, [12105, 26376], 20214, 20537,
    20518, 20581, 20860, 21048, 21091, 21927, 22287, 22533, 23244, 24314,
    25010, 25080, 25331, 25458, 26908, 27177, 29309, [12125, 29356], 29486,
    30740, 30831, 32121, 30476, 32937, [12178, 35211], 35609, 36066, 36562,
    36963, 37749, 38522, 38997, 39443, 40568, 20803, 21407, 21427, 24187,
    24358, 28187, 28304, [12126, 29572], 29694, 32067, 33335, [12180, 35328],
    35578, 38480, 20046, 20491, 21476, 21628, 22266, 22993, 23396,
    [12080, 24049], 24235, 24359, [12094, 25144], 25925, 26543, 28246, 29392,
    31946, 34996, 32929, 32993, 33776, [11969, 34382], 35463, 36328, 37431,
    38599, 39015, [12238, 40723], 20116, 20114, 20237, 21320, 21577, 21566,
    23087, 24460, 24481, 24735, 26791, 27278, 29786, 30849, 35486, 35492,
    35703, 37264, 20062, 39881, 20132, 20348, 20399, 20505, 20502, 20809,
    20844, 21151, 21177, 21246, 21402, [12061, 21475], 21521, 21518, 21897,
    22353, 22434, 22909, 23380, 23389, 23439, [12079, 24037], 24039, 24055,
    24184, 24195, 24218, 24247, 24344, 24658, 24908, 25239, 25304, 25511,
    25915, 26114, 26179, 26356, 26477, 26657, 26775, 27083, 27743, 27946,
    28009, 28207, 28317, 30002, 30343, 30828, 31295, 31968, 32005, 32024,
    32094, 32177, 32789, 32771, 32943, 32945, 33108, 33167, 33322, 33618,
    [12175, 34892], 34913, 35611, 36002, 36092, 37066, 37237, 37489, 30783,
    37628, 38308, 38477, 38917, [12217, 39321], [12220, 39640], 40251, 21083,
    21163, 21495, 21512, 22741, 25335, 28640, 35946, 36703, 40633, 20811,
    21051, 21578, 22269, 31296, 37239, 40288, [12234, 40658], 29508, 28425,
    33136, 29969, 24573, 24794, [12219, 39592], 29403, 36796, 27492, 38915,
    20170, 22256, 22372, 22718, 23130, 24680, 25031, 26127, 26118, 26681,
    26801, 28151, 30165, 32058, [12169, 33390], 39746, 20123, 20304, 21449,
    21766, 23919, 24038, 24046, 26619, 27801, 29811, 30722, 35408, 37782,
    35039, 22352, 24231, 25387, 20661, 20652, 20877, 26368, 21705, 22622,
    22971, 23472, 24425, 25165, 25505, 26685, 27507, 28168, 28797, 37319,
    29312, 30741, 30758, 31085, 25998, 32048, 33756, 35009, 36617, 38555,
    21092, 22312, 26448, 32618, 36001, 20916, 22338, 38442, 22586, 27018,
    32948, 21682, 23822, 22524, 30869, 40442, 20316, 21066, 21643, 25662,
    26152, 26388, 26613, 31364, 31574, 32034, 37679, 26716, 39853, 31545,
    21273, 20874, 21047, 23519, 25334, 25774, 25830, 26413, 27578, 34217,
    38609, 30352, 39894, 25420, 37638, 39851, [12139, 30399], 26194, 19977,
    20632, 21442, [12077, 23665], 24808, 25746, 25955, 26719, 29158, 29642,
    29987, 31639, 32386, 34453, 35715, 36059, 37240, 39184, 26028, 26283,
    27531, 20181, 20180, 20282, 20351, 21050, 21496, 21490, 21987, 22235,
    [12064, 22763], 22987, 22985, 23039, [12070, 23376], 23629, 24066, 24107,
    24535, 24605, 25351, [12096, 25903], 23388, 26031, 26045, 26088, 26525,
    [12108, 27490], 27515, [12114, 27663], 29509, 31049, 31169, [12151, 31992],
    32025, 32043, 32930, 33026, [12164, 33267], 35222, 35422, 35433, 35430,
    35468, 35566, 36039, 36060, 38604, 39164, [12013, 27503], 20107, 20284,
    20365, 20816, 23383, 23546, 24904, 25345, 26178, 27425, 28363, 27835,
    29246, 29885, 30164, 30913, [12144, 31034], [12157, 32780], [12159, 32819],
    [12163, 33258], 33940, 36766, 27728, [12229, 40575], 24335, 35672, 40235,
    31482, 36600, 23437, 38635, 19971, 21489, 22519, 22833, 23241, 23460,
    24713, 28287, 28422, 30142, 36074, 23455, 34048, 31712, 20594, 26612,
    33437, 23649, 34122, 32286, 33294, 20889, 23556, 25448, 36198, 26012,
    29038, 31038, 32023, 32773, 35613, [12190, 36554], 36974, 34503, 37034,
    20511, 21242, 23610, 26451, 28796, 29237, 37196, 37320, 37675, 33509,
    23490, 24369, 24825, 20027, 21462, 23432, [12095, 25163], 26417, 27530,
    29417, 29664, 31278, 33131, 36259, 37202, [12216, 39318], 20754, 21463,
    21610, 23551, 25480, 27193, 32172, 38656, 22234, 21454, 21608, 23447,
    23601, 24030, 20462, 24833, 25342, 27954, 31168, 31179, 32066, 32333,
    32722, 33261, [12168, 33311], 33936, 34886, 35186, 35728, 36468, 36655,
    36913, 37195, 37228, 38598, 37276, 20160, 20303, 20805, [12055, 21313],
    24467, 25102, 26580, 27713, 28171, 29539, 32294, 37325, 37507, 21460,
    22809, 23487, 28113, 31069, 32302, 31899, 22654, 29087, 20986, 34899,
    36848, 20426, 23803, 26149, 30636, 31459, 33308, 39423, 20934, 24490,
    26092, 26991, 27529, 28147, 28310, 28516, 30462, 32020, 24033, 36981,
    37255, 38918, 20966, 21021, 25152, 26257, 26329, 28186, 24246, 32210,
    32626, 26360, 34223, 34295, 35576, 21161, 21465, [12069, 22899], 24207,
    24464, 24661, 37604, 38500, 20663, 20767, 21213, 21280, 21319, 21484,
    21736, 21830, 21809, 22039, 22888, 22974, 23100, 23477, 23558,
    [12073, 23567], 23569, 23578, 24196, 24202, 24288, 24432, 25215, 25220,
    25307, 25484, 25463, 26119, 26124, 26157, 26230, 26494, 26786, 27167,
    27189, 27836, 28040, 28169, 28248, 28988, 28966, 29031, 30151, 30465,
    30813, 30977, 31077, 31216, 31456, 31505, 31911, 32057, 32918, 33750,
    33931, 34121, 34909, 35059, 35359, 35388, 35412, 35443, 35937, 36062,
    37284, 37478, 37758, 37912, 38556, 38808, 19978, 19976, 19998, 20055,
    20887, 21104, 22478, 22580, 22732, 23330, 24120, 24773, 25854, 26465,
    26454, 27972, 29366, 30067, 31331, 33976, 35698, 37304, 37664, 22065,
    22516, 39166, 25325, 26893, 27542, 29165, 32340, 32887, [12170, 33394],
    35302, [12215, 39135], 34645, 36785, 23611, 20280, 20449, 20405, 21767,
    23072, 23517, 23529, [12092, 24515], 24910, 25391, 26032, 26187, 26862,
    27035, 28024, 28145, 30003, 30137, 30495, 31070, 31206, 32051,
    [12162, 33251], 33455, 34218, 35242, 35386, [12189, 36523], [12191, 36763],
    36914, 37341, 38663, [12040, 20154], 20161, 20995, 22645, 22764, 23563,
    29978, 23613, 33102, 35338, 36805, 38499, 38765, 31525, 35535, 38920,
    37218, 22259, 21416, 36887, 21561, 22402, 24101, 25512, [12116, 27700],
    28810, 30561, 31883, 32736, 34928, 36930, 37204, 37648, 37656, 38543,
    29790, 39620, 23815, 23913, 25968, 26530, 36264, 38619, 25454, 26441,
    26905, 33733, 38935, 38592, 35070, 28548, 25722, [12072, 23544], 19990,
    28716, 30045, 26159, 20932, 21046, 21218, 22995, 24449, 24615, 25104,
    25919, 25972, 26143, 26228, 26866, 26646, 27491, 28165, 29298,
    [12131, 29983], 30427, 31934, 32854, 22768, 35069, [11972, 35199], 35488,
    35475, 35531, 36893, 37266, [11992, 38738], 38745, [12011, 25993], 31246,
    33030, 38587, 24109, 24796, 25114, 26021, 26132, 26512, [12143, 30707],
    31309, 31821, 32318, 33034, 36012, [12186, 36196], 36321, 36447, 30889,
    20999, 25305, 25509, 25666, 25240, 35373, 31363, 31680, 35500, 38634,
    32118, [12166, 33292], 34633, 20185, 20808, 21315, 21344, 23459, 23554,
    23574, 24029, 25126, 25159, 25776, 26643, 26676, 27849, 27973, 27927,
    26579, 28508, 29006, 29053, 26059, 31359, 31661, 32218, 32330, 32680,
    33146, [12167, 33307], 33337, 34214, 35438, 36046, 36341, 36984, 36983,
    37549, 37521, 38275, 39854, 21069, 21892, 28472, 28982, 20840, 31109,
    32341, 33203, 31950, 22092, 22609, 23720, 25514, 26366, 26365, 26970,
    29401, 30095, 30094, 30990, 31062, 31199, 31895, 32032, 32068, 34311,
    35380, 38459, 36961, [12239, 40736], 20711, 21109, 21452, 21474, 20489,
    21930, 22766, 22863, 29245, 23435, 23652, 21277, 24803, 24819, 25436,
    25475, 25407, 25531, 25805, 26089, 26361, 24035, 27085, 27133, 28437,
    29157, 20105, 30185, 30456, 31379, 31967, 32207, 32156, 32865, 33609,
    33624, 33900, 33980, 34299, 35013, [12187, 36208], 36865, 36973, 37783,
    38684, 39442, 20687, 22679, 24974, 33235, 34101, 36104, 36896, 20419,
    20596, 21063, 21363, 24687, 25417, 26463, 28204, [12188, 36275], 36895,
    20439, 23646, 36042, 26063, 32154, 21330, 34966, 20854, 25539, 23384,
    23403, 23562, 25613, 26449, 36956, 20182, 22810, 22826, 27760, 35409,
    21822, 22549, 22949, 24816, 25171, 26561, 33333, 26965, 38464, 39364,
    39464, 20307, 22534, 23550, 32784, 23729, 24111, 24453, 24608, 24907,
    25140, 26367, 27888, 28382, 32974, 33151, 33492, 34955, 36024, 36864,
    36910, 38538, 40667, 39899, 20195, 21488, [12068, 22823], 31532, 37261,
    38988, 40441, 28381, 28711, 21331, 21828, 23429, 25176, 25246, 25299,
    27810, 28655, 29730, 35351, 37944, 28609, 35582, 33592, 20967, 34552,
    21482, 21481, 20294, 36948, [12192, 36784], 22890, 33073, 24061, 31466,
    36799, 26842, [12181, 35895], 29432, 40008, 27197, 35504, 20025, 21336,
    22022, 22374, 25285, 25506, 26086, 27470, 28129, 28251, 28845, 30701,
    31471, 31658, 32187, 32829, 32966, 34507, 35477, 37723, 22243, 22727,
    24382, 26029, 26262, 27264, 27573, 30007, 35527, 20516, 30693, 22320,
    24347, 24677, 26234, 27744, 30196, 31258, 32622, 33268, 34584, 36933,
    39347, 31689, 30044, [12149, 31481], 31569, 33988, 36880, 31209, 31378,
    33590, 23265, 30528, 20013, 20210, 23449, 24544, 25277, 26172, 26609,
    27880, [12173, 34411], 34935, 35387, 37198, 37619, 39376, 27159, 28710,
    29482, 33511, 33879, 36015, 19969, 20806, 20939, 21899, 23541, 24086,
    24115, 24193, 24340, 24373, 24427, 24500, 25074, 25361, 26274, 26397,
    28526, 29266, 30010, 30522, 32884, 33081, 33144, 34678, 35519, 35548,
    36229, 36339, 37530, [11985, 12199, 38263], 38914, [12227, 40165], 21189,
    25431, 30452, 26389, 27784, 29645, 36035, 37806, 38515, 27941, 22684,
    26894, 27084, 36861, 37786, 30171, 36890, 22618, 26626, 25524, 27131,
    20291, 28460, 26584, 36795, 34086, 32180, 37716, 26943, 28528, 22378,
    22775, 23340, 32044, [12118, 29226], 21514, 37347, 40372, 20141, 20302,
    20572, 20597, 21059, 35998, 21576, 22564, 23450, 24093, 24213, 24237,
    24311, 24351, 24716, 25269, 25402, 25552, 26799, 27712, 30855, 31118,
    31243, 32224, 33351, 35330, 35558, 36420, 36883, 37048, 37165, 37336,
    [12237, 40718], 27877, 25688, 25826, 25973, 28404, 30340, 31515, 36969,
    37841, 28346, 21746, 24505, 25764, 36685, 36845, 37444, 20856, 22635,
    22825, 23637, 24215, 28155, 32399, 29980, 36028, 36578, 39003, 28857,
    20253, 27583, 28593, [12133, 30000], 38651, 20814, 21520, 22581, 22615,
    22956, 23648, 24466, [12099, 26007], 26460, 28193, 30331, 33759, 36077,
    36884, 37117, 37709, 30757, 30778, 21162, 24230, [12063, 22303], 22900,
    24594, 20498, 20826, 20908, 20941, [12049, 20992], 21776, 22612, 22616,
    22871, 23445, 23798, 23947, 24764, 25237, 25645, 26481, 26691, 26812,
    26847, 30423, 28120, 28271, 28059, 28783, 29128, 24403, 30168, 31095,
    31561, 31572, 31570, 31958, 32113, 21040, 33891, 34153, 34276, 35342,
    35588, [12182, 35910], 36367, 36867, 36879, 37913, 38518, 38957, 39472,
    38360, 20685, 21205, 21516, 22530, 23566, 24999, 25758, 27934, 30643,
    31461, 33012, 33796, 36947, 37509, 23776, 40199, 21311, 24471, 24499,
    28060, 29305, 30563, 31167, 31716, 27602, 29420, 35501, 26627, 27233,
    20984, 31361, 26932, 23626, 40182, 33515, 23493, [12195, 37193], 28702,
    22136, 23663, 24775, 25958, 27788, 35930, 36929, 38931, 21585, 26311,
    37389, 22856, 37027, 20869, 20045, 20970, 34201, 35598, 28760, 25466,
    37707, 26978, 39348, 32260, 30071, 21335, 26976, 36575, 38627, 27741,
    [12038, 20108], 23612, 24336, 36841, 21250, 36049, [12161, 32905], 34425,
    24319, [12103, 26085], 20083, [12042, 20837], 22914, 23615, 38894, 20219,
    22922, 24525, 35469, 28641, 31152, 31074, 23527, 33905, 29483, 29105,
    24180, 24565, 25467, 25754, 29123, 31896, 20035, 24316, 20043, 22492,
    22178, 24745, 28611, 32013, 33021, 33075, 33215, 36786, 35223, 34468,
    24052, 25226, 25773, 35207, 26487, 27874, 27966, 29750, 30772, 23110,
    32629, 33453, [12218, 39340], 20467, 24259, 25309, 25490, 25943, 26479,
    30403, 29260, 32972, 32954, 36649, 37197, 20493, 22521, 23186, 26757,
    26995, 29028, 29437, 36023, 22770, 36064, 38506, 36889, 34687, 31204,
    30695, 33833, 20271, 21093, 21338, 25293, 26575, 27850, [12137, 30333],
    31636, 31893, 33334, 34180, 36843, 26333, 28448, 29190, 32283, 33707,
    39361, [12008, 40614], 20989, 31665, 30834, 31672, 32903, 31560, 27368,
    24161, 32908, 30033, 30048, [12043, 20843], 37474, 28300, 30330, 37271,
    39658, 20240, 32624, 25244, 31567, 38309, 40169, 22138, 22617, 34532,
    38588, 20276, 21028, 21322, 21453, 21467, 24070, 25644, 26001, 26495,
    27710, 27726, 29256, 29359, 29677, 30036, 32321, 33324, 34281, 36009,
    31684, [12196, 37318], 29033, 38930, 39151, 25405, 26217, 30058, 30436,
    30928, 34115, 34542, 21290, 21329, 21542, 22915, 24199, 24444, 24754,
    25161, 25209, 25259, 26000, [12112, 27604], 27852, 30130, [12138, 30382],
    30865, 31192, 32203, 32631, 32933, 34987, 35513, 36027, 36991,
    [12206, 38750], [12214, 39131], 27147, 31800, 20633, 23614, 24494, 26503,
    27608, 29749, 30473, 32654, [12240, 40763], 26570, 31255, 21305,
    [12134, 30091], 39661, 24422, 33181, 33777, 32920, 24380, 24517, 30050,
    31558, 36924, 26727, 23019, 23195, 32016, 30334, 35628, 20469, 24426,
    27161, 27703, 28418, 29922, 31080, 34920, 35413, 35961, 24287, 25551,
    30149, 31186, 33495, 37672, 37618, 33948, 34541, 39981, 21697, 24428,
    25996, 27996, 28693, 36007, 36051, 38971, 25935, 29942, 19981, 20184,
    22496, 22827, 23142, 23500, 20904, 24067, 24220, 24598, 25206, 25975,
    26023, 26222, 28014, [12119, 29238], 31526, 33104, 33178, 33433, 35676,
    36000, 36070, 36212, [12201, 38428], 38468, 20398, 25771, 27494, 33310,
    33889, 34154, 37096, 23553, 26963, [12213, 39080], 33914, 34135, 20239,
    21103, 24489, 24133, 26381, 31119, 33145, 35079, 35206, 28149, 24343,
    25173, 27832, 20175, 29289, 39826, 20998, 21563, 22132, 22707, 24996,
    25198, 28954, 22894, 31881, 31966, 32027, 38640, [12098, 25991], 32862,
    19993, 20341, 20853, 22592, 24163, 24179, 24330, 26564, 20006, 34109,
    38281, 38491, [12150, 31859], [12212, 38913], 20731, 22721, 30294, 30887,
    21029, 30629, 34065, 31622, 20559, 22793, [12122, 29255], 31687, 32232,
    36794, 36820, 36941, 20415, 21193, 23081, 24321, 38829, 20445, 33303,
    37610, 22275, 25429, 27497, 29995, 35036, 36628, 31298, 21215, 22675,
    24917, 25098, 26286, [11935, 27597], 31807, 33769, 20515, 20472, 21253,
    21574, 22577, 22857, 23453, 23792, 23791, 23849, 24214, 25265, 25447,
    25918, [12101, 26041], 26379, 27861, 27873, 28921, 30770, 32299, 32990,
    33459, 33804, 34028, 34562, 35090, 35370, 35914, 37030, 37586, 39165,
    40179, 40300, 20047, 20129, 20621, 21078, 22346, 22952, 24125,
    {f: 2, c: 24536}, 25151, 26292, 26395, 26576, 26834, 20882, 32033, 32938,
    33192, 35584, 35980, 36031, 37502, 38450, 21536, 38956, 21271, 20693,
    [12056, 21340], 22696, 25778, 26420, 29287, 30566, 31302, 37350, 21187,
    27809, 27526, 22528, 24140, 22868, 26412, 32763, 20961, 30406, 25705,
    30952, 39764, [12231, 40635], 22475, 22969, 26151, 26522, 27598, 21737,
    27097, 24149, 33180, 26517, 39850, 26622, 40018, 26717, 20134, 20451,
    [12060, 21448], 25273, 26411, 27819, 36804, 20397, 32365, 40639, 19975,
    24930, 28288, 28459, 34067, 21619, 26410, 39749, [11922, 24051], 31637,
    23724, 23494, 34588, 28234, 34001, 31252, 33032, 22937, 31885,
    [11936, 27665], 30496, 21209, 22818, 28961, 29279, [12141, 30683], 38695,
    40289, 26891, 23167, 23064, 20901, 21517, 21629, 26126, 30431, 36855,
    37528, 40180, 23018, 29277, 28357, 20813, 26825, 32191, 32236,
    [12207, 38754], 40634, 25720, 27169, 33538, 22916, 23391, [12113, 27611],
    29467, 30450, 32178, 32791, 33945, 20786, [12106, 26408], 40665,
    [12140, 30446], 26466, 21247, 39173, 23588, 25147, 31870, 36016, 21839,
    24758, 32011, [12200, 38272], 21249, 20063, 20918, 22812, 29242, 32822,
    37326, 24357, [12142, 30690], 21380, 24441, 32004, 34220, 35379, 36493,
    38742, 26611, 34222, 37971, 24841, 24840, 27833, 30290, 35565, 36664,
    21807, 20305, 20778, 21191, 21451, 23461, 24189, 24736, 24962, 25558,
    26377, 26586, 28263, 28044, {f: 2, c: 29494}, 30001, 31056, 35029, 35480,
    36938, [12194, 37009], 37109, 38596, 34701, [12067, 22805], 20104, 20313,
    19982, 35465, 36671, 38928, 20653, 24188, 22934, 23481, 24248, 25562,
    25594, 25793, 26332, 26954, 27096, 27915, 28342, 29076, [12132, 29992],
    31407, [12154, 32650], 32768, 33865, 33993, 35201, 35617, 36362, 36965,
    38525, 39178, 24958, 25233, 27442, 27779, 28020, 32716, 32764, 28096,
    32645, 34746, 35064, 26469, 33713, 38972, 38647, 27931, 32097, 33853,
    37226, 20081, 21365, 23888, 27396, 28651, 34253, 34349, 35239, 21033,
    21519, 23653, 26446, 26792, 29702, 29827, 30178, 35023, 35041,
    [12197, 37324], 38626, 38520, 24459, 29575, [12148, 31435], 33870, 25504,
    30053, 21129, 27969, 28316, 29705, 30041, 30827, 31890, 38534,
    [12015, 31452], [12243, 40845], 20406, 24942, 26053, 34396, 20102, 20142,
    20698, 20001, 20940, 23534, 26009, 26753, 28092, 29471, 30274, 30637,
    31260, 31975, 33391, 35538, 36988, 37327, 38517, 38936, [12050, 21147],
    32209, 20523, 21400, 26519, 28107, 29136, 29747, 33256, 36650, 38563,
    40023, 40607, 29792, 22593, 28057, 32047, 39006, 20196, 20278, 20363,
    20919, 21169, 23994, 24604, 29618, 31036, 33491, 37428, 38583, 38646,
    38666, 40599, 40802, 26278, 27508, 21015, 21155, 28872, 35010, 24265,
    24651, 24976, 28451, 29001, 31806, 32244, 32879, 34030, 36899, 37676,
    21570, 39791, 27347, 28809, 36034, 36335, 38706, 21172, 23105, 24266,
    24324, 26391, 27004, 27028, 28010, 28431, 29282, 29436, 31725,
    [12156, 32769], 32894, 34635, 37070, 20845, 40595, 31108, 32907, 37682,
    35542, 20525, 21644, 35441, 27498, 36036, 33031, 24785, 26528, 40434,
    20121, 20120, 39952, 35435, 34241, 34152, 26880, 28286, 30871, 33109,
    24332, 19984, 19989, 20010, 20017, [12034, 20022], 20028, [12035, 20031],
    20034, 20054, 20056, 20098, [12037, 20101], 35947, 20106, 33298, 24333,
    20110, {f: 2, c: 20126}, [12039, 20128], 20130, 20144, 20147, 20150, 20174,
    20173, 20164, 20166, 20162, 20183, 20190, 20205, 20191, 20215, 20233,
    20314, 20272, 20315, 20317, 20311, 20295, 20342, 20360, 20367, 20376,
    20347, 20329, 20336, 20369, 20335, 20358, 20374, 20760, 20436, 20447,
    20430, 20440, 20443, 20433, 20442, 20432, {f: 2, c: 20452}, 20506, 20520,
    20500, 20522, 20517, 20485, 20252, 20470, 20513, 20521, 20524, 20478,
    20463, 20497, 20486, 20547, 20551, 26371, 20565, 20560, 20552, 20570,
    20566, 20588, 20600, 20608, 20634, 20613, 20660, 20658, {f: 2, c: 20681},
    20659, 20674, 20694, 20702, 20709, 20717, 20707, 20718, 20729, 20725,
    20745, {f: 2, c: 20737}, 20758, 20757, 20756, 20762, 20769, 20794, 20791,
    20796, 20795, [12041, 20799], [11918, 20800], 20818, 20812, 20820, 20834,
    31480, {f: 2, c: 20841}, 20846, 20864, [12044, 20866], 22232, 20876, 20873,
    20879, 20881, 20883, 20885, [12045, 20886], 20900, 20902, 20898,
    {f: 2, c: 20905}, [12046, 20907], 20915, {f: 2, c: 20913}, 20912, 20917,
    20925, 20933, 20937, 20955, [12047, 20960], 34389, 20969, 20973, 20976,
    [12048, 20981], 20990, 20996, 21003, 21012, 21006, 21031, 21034, 21038,
    21043, 21049, 21071, 21060, {f: 2, c: 21067}, 21086, 21076, 21098, 21108,
    21097, 21107, 21119, 21117, 21133, 21140, 21138, 21105, 21128, 21137,
    36776, 36775, {f: 2, c: 21164}, 21180, 21173, 21185, 21197, 21207, 21214,
    21219, 21222, 39149, 21216, 21235, 21237, 21240, [12051, 21241], 21254,
    21256, 30008, 21261, 21264, 21263, [12052, 21269], [12053, 21274], 21283,
    21295, 21297, 21299, [12054, 21304], 21312, 21318, 21317, 19991, 21321,
    21325, 20950, 21342, [12057, 21353], 21358, 22808, 21371, 21367,
    [12058, 21378], 21398, 21408, 21414, 21413, 21422, 21424, [12059, 21430],
    21443, 31762, 38617, 21471, 26364, 29166, 21486, 21480, 21485, 21498,
    21505, 21565, 21568, {f: 2, c: 21548}, 21564, 21550, 21558, 21545, 21533,
    21582, 21647, 21621, 21646, 21599, 21617, 21623, 21616, 21650, 21627,
    21632, 21622, 21636, 21648, 21638, 21703, 21666, 21688, 21669, 21676,
    21700, 21704, 21672, 21675, 21698, 21668, 21694, 21692, 21720,
    {f: 2, c: 21733}, 21775, 21780, 21757, 21742, 21741, 21754, 21730, 21817,
    21824, 21859, 21836, 21806, 21852, 21829, {f: 2, c: 21846}, 21816, 21811,
    21853, 21913, 21888, 21679, 21898, 21919, 21883, 21886, 21912, 21918,
    21934, 21884, 21891, 21929, 21895, 21928, 21978, 21957, 21983, 21956,
    21980, 21988, 21972, 22036, 22007, 22038, 22014, 22013, 22043, 22009,
    22094, 22096, 29151, 22068, 22070, 22066, 22072, 22123, 22116, 22063,
    22124, 22122, 22150, 22144, 22154, 22176, 22164, 22159, 22181, 22190,
    22198, 22196, 22210, 22204, 22209, 22211, 22208, 22216, 22222, 22225,
    22227, [12062, 22231], 22254, 22265, 22272, 22271, 22276, 22281, 22280,
    22283, 22285, 22291, 22296, 22294, 21959, 22300, 22310, {f: 2, c: 22327},
    22350, 22331, 22336, 22351, 22377, 22464, 22408, 22369, 22399, 22409,
    22419, 22432, 22451, 22436, 22442, 22448, 22467, 22470, 22484,
    {f: 2, c: 22482}, 22538, 22486, 22499, 22539, 22553, 22557, 22642, 22561,
    22626, 22603, 22640, 27584, 22610, 22589, 22649, 22661, 22713, 22687,
    22699, 22714, 22750, 22715, 22712, 22702, 22725, 22739, 22737, 22743,
    22745, 22744, 22757, 22748, 22756, 22751, 22767, 22778, 22777,
    {f: 3, c: 22779}, [12065, 22786], [12066, 22794], 22800, 22811, 26790,
    22821, {f: 2, c: 22828}, 22834, 22840, 22846, 31442, 22869, 22864, 22862,
    22874, 22872, 22882, 22880, 22887, 22892, 22889, 22904, 22913, 22941,
    20318, 20395, 22947, 22962, 22982, 23016, 23004, 22925, {f: 2, c: 23001},
    23077, 23071, 23057, 23068, 23049, 23066, 23104, 23148, 23113,
    {f: 2, c: 23093}, 23138, 23146, 23194, 23228, 23230, 23243, 23234, 23229,
    23267, 23255, 23270, 23273, 23254, {f: 2, c: 23290}, 23308, 23307, 23318,
    23346, 23248, 23338, 23350, 23358, 23363, 23365, 23360, 23377, 23381,
    {f: 2, c: 23386}, 23397, 23401, 23408, 23411, 23413, 23416, 25992, 23418,
    [12071, 23424], 23427, 23462, 23480, 23491, 23495, 23497, 23508, 23504,
    23524, 23526, 23522, 23518, 23525, 23531, 23536, 23542, 23539, 23557,
    {f: 2, c: 23559}, 23565, 23571, 23584, [11920, 12074, 23586], 23592,
    [12075, 23608], 23609, 23617, 23622, 23630, 23635, 23632, 23631, 23409,
    23660, [12076, 23662], 20066, 23670, 23673, 23692, 23697, 23700, 22939,
    23723, 23739, 23734, 23740, 23735, 23749, 23742, 23751, 23769, 23785,
    23805, 23802, 23789, 23948, 23786, 23819, 23829, 23831, 23900, 23839,
    23835, 23825, 23828, 23842, 23834, 23833, 23832, 23884, 23890, 23886,
    23883, 23916, 23923, 23926, 23943, 23940, 23938, 23970, 23965, 23980,
    23982, 23997, 23952, 23991, 23996, 24009, 24013, 24019, 24018, 24022,
    [12078, 24027], 24043, 24050, 24053, 24075, 24090, 24089, 24081, 24091,
    {f: 2, c: 24118}, 24132, 24131, 24128, 24142, 24151, 24148, 24159, 24162,
    24164, 24135, {f: 2, c: 24181}, [11923, 12083, 24186], 40636,
    [12084, 24191], 24224, {f: 2, c: 24257}, 24264, 24272, 24271, 24278, 24291,
    24285, {f: 2, c: 24282}, 24290, 24289, {f: 2, c: 24296}, 24300, 24305,
    24307, 24304, [12085, 24308], 24312, [12086, 24318], 24323, 24329, 24413,
    24412, [12087, 24331], 24337, 24342, 24361, 24365, 24376, 24385, 24392,
    24396, 24398, 24367, [11924, 24401], {f: 2, c: 24406}, 24409,
    [12090, 24417], 24429, [12091, 24435], 24439, 24451, 24450, 24447, 24458,
    24456, 24465, 24455, 24478, 24473, 24472, 24480, 24488, 24493, 24508,
    24534, 24571, 24548, 24568, 24561, 24541, 24755, 24575, 24609, 24672,
    24601, 24592, 24617, 24590, 24625, 24603, 24597, 24619, 24614, 24591,
    24634, 24666, 24641, 24682, 24695, 24671, 24650, 24646, 24653, 24675,
    24643, 24676, 24642, 24684, 24683, 24665, 24705, 24717, 24807, 24707,
    24730, 24708, 24731, {f: 2, c: 24726}, 24722, 24743, 24715, 24801, 24760,
    24800, 24787, 24756, 24560, 24765, 24774, 24757, 24792, 24909, 24853,
    24838, {f: 2, c: 24822}, 24832, 24820, 24826, 24835, 24865, 24827, 24817,
    {f: 2, c: 24845}, 24903, 24894, 24872, 24871, 24906, 24895, 24892, 24876,
    24884, 24893, 24898, 24900, 24947, 24951, {f: 3, c: 24920}, 24939, 24948,
    24943, 24933, 24945, 24927, 24925, 24915, 24949, 24985, 24982, 24967,
    25004, 24980, 24986, 24970, 24977, 25003, 25006, 25036, 25034, 25033,
    25079, 25032, 25027, 25030, 25018, 25035, 32633, 25037, 25062, 25059,
    25078, 25082, 25076, 25087, 25085, 25084, 25086, 25088, [12093, 25096],
    25097, 25101, 25100, 25108, 25115, 25118, 25121, 25130, 25134, 25136,
    {f: 2, c: 25138}, 25153, 25166, 25182, 25187, 25179, 25184, 25192, 25212,
    25218, 25225, 25214, {f: 2, c: 25234}, 25238, 25300, 25219, 25236, 25303,
    25297, 25275, 25295, 25343, 25286, 25812, 25288, 25308, 25292, 25290,
    25282, 25287, 25243, 25289, 25356, 25326, 25329, 25383, 25346, 25352,
    25327, 25333, 25424, 25406, 25421, 25628, 25423, 25494, 25486, 25472,
    25515, 25462, 25507, 25487, 25481, 25503, 25525, 25451, 25449, 25534,
    25577, 25536, 25542, 25571, 25545, 25554, 25590, 25540, 25622, 25652,
    25606, 25619, 25638, 25654, 25885, 25623, 25640, 25615, 25703, 25711,
    25718, 25678, 25898, 25749, 25747, 25765, 25769, 25736, 25788, 25818,
    25810, 25797, 25799, 25787, 25816, 25794, 25841, 25831, 33289,
    {f: 2, c: 25824}, 25260, 25827, 25839, 25900, 25846, 25844, 25842, 25850,
    25856, 25853, 25880, 25884, 25861, 25892, 25891, 25899, [12097, 25908],
    [11929, 25909], 25911, 25910, 25912, 30027, 25928, 25942, 25941, 25933,
    25944, 25950, 25949, 25970, 25976, {f: 2, c: 25986}, 35722, 26011, 26015,
    26027, 26039, 26051, 26054, 26049, 26052, 26060, 26066, 26075, 26073,
    [12102, 26080], [11931, 26081], 26097, 26482, 26122, 26115, 26107, 26483,
    {f: 2, c: 26165}, 26164, 26140, 26191, 26180, 26185, 26177, 26206, 26205,
    26212, {f: 2, c: 26215}, 26207, 26210, 26224, 26243, 26248, 26254, 26249,
    26244, 26264, 26269, 26305, 26297, 26313, 26302, 26300, 26308, 26296,
    26326, 26330, 26336, 26175, 26342, 26345, [12104, 26352], 26357, 26359,
    26383, 26390, 26398, {f: 2, c: 26406}, 38712, 26414, 26431, 26422, 26433,
    26424, 26423, 26438, 26462, 26464, 26457, {f: 2, c: 26467}, 26505, 26480,
    26537, 26492, 26474, 26508, 26507, 26534, 26529, 26501, 26551, 26607,
    26548, 26604, 26547, 26601, 26552, 26596, 26590, 26589, 26594, 26606,
    26553, 26574, 26566, 26599, 27292, 26654, 26694, 26665, 26688, 26701,
    26674, 26702, 26803, 26667, 26713, 26723, 26743, 26751, 26783, 26767,
    26797, 26772, 26781, 26779, 26755, 27310, 26809, 26740, 26805, 26784,
    26810, 26895, 26765, 26750, 26881, 26826, 26888, 26840, 26914, 26918,
    26849, 26892, 26829, 26836, 26855, 26837, 26934, 26898, 26884, 26839,
    26851, 26917, 26873, 26848, 26863, 26920, 26922, 26906, 26915, 26913,
    26822, 27001, 26999, 26972, 27000, 26987, 26964, 27006, 26990, 26937,
    26996, 26941, 26969, 26928, 26977, 26974, 26973, 27009, 26986, 27058,
    27054, 27088, 27071, 27073, 27091, 27070, 27086, 23528, 27082, 27101,
    27067, 27075, 27047, 27182, 27025, 27040, 27036, 27029, 27060, 27102,
    27112, 27138, 27163, 27135, 27402, 27129, 27122, 27111, 27141, 27057,
    27166, 27117, 27156, 27115, 27146, 27154, 27329, 27171, 27155, 27204,
    27148, 27250, 27190, 27256, 27207, 27234, 27225, 27238, 27208, 27192,
    27170, 27280, 27277, 27296, 27268, {f: 2, c: 27298}, 27287, 34327, 27323,
    27331, 27330, 27320, 27315, 27308, 27358, 27345, 27359, 27306, 27354,
    27370, 27387, 27397, 34326, 27386, 27410, 27414, 39729, 27423, 27448,
    27447, 30428, 27449, 39150, 27463, 27459, 27465, 27472, 27481, 27476,
    27483, 27487, 27489, 27512, [12109, 27513], {f: 2, c: 27519}, 27524, 27523,
    27533, 27544, 27541, 27550, 27556, {f: 2, c: 27562}, 27567, 27570, 27569,
    [12110, 27571], 27575, 27580, 27590, [12111, 27595], 27603, 27615, 27628,
    27627, 27635, 27631, 40638, 27656, 27667, [12115, 27668], 27675, 27684,
    27683, 27742, 27733, 27746, 27754, 27778, 27789, 27802, 27777, 27803,
    27774, 27752, 27763, 27794, 27792, 27844, 27889, 27859, 27837, 27863,
    27845, 27869, 27822, 27825, 27838, 27834, 27867, 27887, 27865, 27882,
    27935, 34893, 27958, 27947, 27965, 27960, 27929, 27957, 27955, 27922,
    27916, 28003, 28051, 28004, 27994, 28025, 27993, 28046, 28053, 28644,
    28037, 28153, 28181, 28170, 28085, 28103, 28134, 28088, 28102, 28140,
    28126, 28108, 28136, 28114, 28101, 28154, 28121, 28132, 28117, 28138,
    28142, 28205, 28270, 28206, 28185, 28274, 28255, 28222, 28195, 28267,
    28203, 28278, 28237, 28191, 28227, 28218, 28238, 28196, 28415, 28189,
    28216, 28290, 28330, 28312, 28361, 28343, 28371, 28349, 28335, 28356,
    28338, {f: 2, c: 28372}, 28303, 28325, 28354, 28319, 28481, 28433, 28748,
    28396, 28408, 28414, 28479, 28402, 28465, 28399, 28466, 28364, 28478,
    28435, 28407, 28550, 28538, 28536, 28545, 28544, 28527, 28507, 28659,
    28525, 28546, 28540, 28504, 28558, 28561, 28610, 28518, 28595, 28579,
    28577, 28580, 28601, 28614, 28586, 28639, 28629, 28652, 28628, 28632,
    28657, 28654, 28635, 28681, 28683, 28666, 28689, 28673, 28687, 28670,
    28699, 28698, 28532, 28701, 28696, 28703, 28720, 28734, 28722, 28753,
    28771, 28825, 28818, 28847, 28913, 28844, 28856, 28851, 28846, 28895,
    28875, 28893, 28889, 28937, 28925, 28956, 28953, 29029, 29013, 29064,
    29030, 29026, 29004, 29014, 29036, 29071, 29179, 29060, 29077, 29096,
    29100, 29143, 29113, 29118, 29138, 29129, 29140, 29134, 29152, 29164,
    29159, 29173, 29180, 29177, 29183, 29197, 29200, 29211, 29224, 29229,
    29228, 29232, 29234, [12120, 29243], 29244, [12121, 29247], 29248, 29254,
    29259, 29272, 29300, 29310, 29314, 29313, 29319, 29330, 29334, 29346,
    29351, 29369, 29362, 29379, 29382, 29380, 29390, 29394, 29410,
    {f: 2, c: 29408}, 29433, 29431, 20495, 29463, 29450, 29468, 29462, 29469,
    29492, 29487, 29481, 29477, 29502, {f: 2, c: 29518}, 40664, 29527, 29546,
    29544, 29552, 29560, 29557, 29563, 29562, 29640, 29619, 29646, 29627,
    29632, 29669, 29678, 29662, 29858, 29701, 29807, 29733, 29688, 29746,
    29754, 29781, 29759, 29791, 29785, 29761, 29788, 29801, 29808, 29795,
    29802, 29814, 29822, 29835, 29854, 29863, 29898, 29903, 29908, 29681,
    29920, 29923, 29927, 29929, 29934, 29938, {f: 2, c: 29936}, 29944, 29943,
    29956, 29955, 29957, 29964, 29966, 29965, 29973, 29971, 29982, 29990,
    29996, 30012, 30020, 30029, 30026, 30025, 30043, 30022, 30042, 30057,
    30052, 30055, 30059, 30061, 30072, 30070, {f: 2, c: 30086}, 30068, 30090,
    30089, 30082, 30100, 30106, 30109, 30117, 30115, 30146, 30131, 30147,
    30133, 30141, 30136, 30140, 30129, 30157, 30154, 30162, 30169, 30179,
    30174, {f: 2, c: 30206}, 30204, 30209, 30192, 30202, {f: 2, c: 30194},
    30219, 30221, 30217, 30239, 30247, {f: 3, c: 30240}, 30244, 30260, 30256,
    30267, {f: 2, c: 30279}, 30278, 30300, 30296, {f: 2, c: 30305},
    {f: 3, c: 30312}, 30311, 30316, 30320, 30322, [12136, 30326], 30328, 30332,
    30336, 30339, 30344, 30347, 30350, 30358, 30355, {f: 2, c: 30361}, 30384,
    30388, {f: 3, c: 30392}, 30402, 30413, 30422, 30418, 30430, 30433, 30437,
    30439, 30442, 34351, 30459, 30472, 30471, 30468, 30505, 30500, 30494,
    {f: 2, c: 30501}, 30491, {f: 2, c: 30519}, 30535, 30554, 30568, 30571,
    30555, 30565, 30591, 30590, 30585, 30606, 30603, 30609, 30624, 30622,
    30640, 30646, 30649, 30655, {f: 2, c: 30652}, 30651, 30663, 30669, 30679,
    30682, 30684, 30691, 30702, 30716, 30732, 30738, 31014, 30752, 31018,
    30789, 30862, 30836, 30854, 30844, 30874, 30860, 30883, 30901, 30890,
    30895, 30929, 30918, 30923, 30932, 30910, 30908, 30917, 30922, 30956,
    30951, 30938, 30973, 30964, 30983, 30994, 30993, 31001, 31020, 31019,
    31040, 31072, 31063, 31071, 31066, 31061, 31059, 31098, 31103, 31114,
    31133, 31143, 40779, 31146, 31150, 31155, {f: 2, c: 31161}, 31177, 31189,
    31207, 31212, 31201, 31203, 31240, 31245, {f: 2, c: 31256}, 31264, 31263,
    31104, 31281, 31291, 31294, 31287, 31299, 31319, 31305, {f: 2, c: 31329},
    31337, 40861, 31344, 31353, 31357, 31368, 31383, 31381, 31384, 31382,
    31401, 31432, 31408, 31414, 31429, 31428, 31423, 36995, 31431, 31434,
    31437, 31439, 31445, 31443, {f: 2, c: 31449}, 31453, {f: 2, c: 31457},
    31462, 31469, 31472, 31490, 31503, 31498, 31494, 31539, {f: 2, c: 31512},
    31518, 31541, 31528, 31542, 31568, 31610, 31492, 31565, 31499, 31564,
    31557, 31605, 31589, 31604, 31591, {f: 2, c: 31600}, 31596, 31598, 31645,
    31640, 31647, 31629, 31644, 31642, 31627, 31634, 31631, 31581, 31641,
    31691, 31681, 31692, 31695, 31668, 31686, 31709, 31721, 31761, 31764,
    31718, 31717, 31840, 31744, 31751, 31763, 31731, 31735, 31767, 31757,
    31734, 31779, 31783, 31786, 31775, 31799, 31787, 31805, 31820, 31811,
    31828, 31823, 31808, 31824, 31832, 31839, 31844, 31830, 31845, 31852,
    31861, 31875, 31888, 31908, 31917, 31906, 31915, 31905, 31912, 31923,
    31922, 31921, 31918, 31929, 31933, 31936, 31941, 31938, 31960, 31954,
    31964, 31970, 39739, 31983, 31986, 31988, 31990, 31994, 32006, 32002,
    32028, 32021, 32010, 32069, 32075, 32046, 32050, 32063, 32053, 32070,
    32115, 32086, 32078, 32114, 32104, 32110, 32079, 32099, 32147, 32137,
    32091, 32143, 32125, 32155, 32186, 32174, 32163, 32181, 32199, 32189,
    32171, 32317, 32162, 32175, 32220, 32184, 32159, 32176, 32216, 32221,
    32228, 32222, 32251, 32242, 32225, 32261, 32266, 32291, 32289, 32274,
    32305, 32287, 32265, 32267, 32290, 32326, 32358, 32315, 32309, 32313,
    32323, 32311, 32306, 32314, 32359, 32349, 32342, 32350, {f: 2, c: 32345},
    32377, 32362, 32361, 32380, 32379, 32387, 32213, 32381, 36782, 32383,
    {f: 2, c: 32392}, 32396, 32402, 32400, {f: 2, c: 32403}, 32406, 32398,
    {f: 2, c: 32411}, 32568, 32570, 32581, {f: 3, c: 32588}, 32592,
    [12153, 32593], 32597, 32596, 32600, {f: 2, c: 32607}, {f: 2, c: 32616},
    32615, 32632, 32642, 32646, 32643, 32648, 32647, 32652, 32660, 32670,
    32669, 32666, 32675, 32687, 32690, 32697, 32686, 32694, 32696, 35697,
    {f: 2, c: 32709}, 32714, 32725, 32724, 32737, 32742, 32745, 32755, 32761,
    39132, 32774, 32772, 32779, [12158, 32786], {f: 2, c: 32792}, 32796, 32801,
    32808, 32831, 32827, 32842, 32838, 32850, 32856, 32858, 32863, 32866,
    32872, 32883, 32882, 32880, 32886, 32889, 32893, [12160, 32895], 32900,
    32902, 32901, 32923, 32915, 32922, 32941, 20880, 32940, 32987, 32997,
    32985, 32989, 32964, 32986, 32982, 33033, 33007, 33009, 33051, 33065,
    33059, 33071, 33099, 38539, 33094, 33086, 33107, 33105, 33020, 33137,
    33134, {f: 2, c: 33125}, 33140, 33155, 33160, 33162, 33152, 33154, 33184,
    33173, 33188, 33187, 33119, 33171, 33193, 33200, 33205, 33214, 33208,
    33213, 33216, 33218, 33210, 33225, 33229, 33233, 33241, 33240, 33224,
    33242, {f: 2, c: 33247}, 33255, {f: 2, c: 33274}, 33278, {f: 2, c: 33281},
    33285, 33287, 33290, 33293, 33296, 33302, 33321, 33323, 33336, 33331,
    33344, 33369, 33368, 33373, 33370, 33375, 33380, 33378, 33384,
    {f: 2, c: 33386}, 33326, 33393, 33399, [12171, 33400], 33406, 33421, 33426,
    33451, 33439, 33467, 33452, 33505, 33507, 33503, 33490, 33524, 33523,
    33530, 33683, 33539, 33531, 33529, 33502, 33542, 33500, 33545, 33497,
    33589, 33588, 33558, 33586, 33585, 33600, 33593, 33616, 33605, 33583,
    33579, {f: 2, c: 33559}, 33669, 33690, 33706, 33695, 33698, 33686, 33571,
    33678, 33671, 33674, 33660, 33717, 33651, 33653, 33696, 33673, 33704,
    33780, 33811, 33771, 33742, 33789, 33795, 33752, 33803, 33729, 33783,
    33799, 33760, 33778, 33805, 33826, 33824, 33725, 33848, 34054, 33787,
    33901, 33834, 33852, 34138, 33924, 33911, 33899, 33965, 33902, 33922,
    33897, 33862, 33836, 33903, 33913, 33845, 33994, 33890, 33977, 33983,
    33951, 34009, 33997, 33979, 34010, 34000, 33985, 33990, 34006, 33953,
    34081, 34047, 34036, {f: 2, c: 34071}, 34092, 34079, 34069, 34068, 34044,
    34112, 34147, 34136, 34120, 34113, 34306, 34123, 34133, 34176, 34212,
    34184, 34193, 34186, 34216, 34157, 34196, 34203, 34282, 34183, 34204,
    34167, 34174, 34192, 34249, 34234, 34255, 34233, 34256, 34261, 34269,
    34277, 34268, 34297, 34314, 34323, 34315, 34302, 34298, 34310, 34338,
    34330, 34352, 34367, [12172, 34381], 20053, 34388, 34399, 34407, 34417,
    34451, 34467, {f: 2, c: 34473}, {f: 2, c: 34443}, 34486, 34479, 34500,
    34502, 34480, 34505, 34851, 34475, 34516, 34526, 34537, 34540, 34527,
    34523, 34543, 34578, 34566, 34568, 34560, 34563, 34555, 34577, 34569,
    34573, 34553, 34570, 34612, 34623, 34615, 34619, 34597, 34601, 34586,
    34656, 34655, 34680, 34636, 34638, 34676, 34647, 34664, 34670, 34649,
    34643, 34659, 34666, 34821, 34722, 34719, 34690, 34735, 34763, 34749,
    34752, 34768, 38614, 34731, 34756, 34739, 34759, 34758, 34747, 34799,
    34802, 34784, 34831, 34829, 34814, {f: 2, c: 34806}, 34830, 34770, 34833,
    34838, 34837, 34850, 34849, 34865, 34870, 34873, 34855, 34875, 34884,
    34882, 34898, 34905, 34910, 34914, 34923, 34945, 34942, 34974, 34933,
    34941, 34997, 34930, 34946, 34967, 34962, 34990, 34969, 34978, 34957,
    34980, 34992, 35007, 34993, {f: 2, c: 35011}, 35028, {f: 2, c: 35032},
    35037, 35065, 35074, 35068, 35060, 35048, 35058, 35076, 35084, 35082,
    35091, 35139, 35102, 35109, {f: 2, c: 35114}, 35137, 35140, 35131, 35126,
    35128, 35148, 35101, 35168, 35166, 35174, 35172, 35181, 35178, 35183,
    35188, 35191, [12177, 35198], 35203, 35208, 35210, 35219, 35224, 35233,
    35241, 35238, 35244, 35247, 35250, 35258, 35261, {f: 2, c: 35263}, 35290,
    {f: 2, c: 35292}, 35303, 35316, 35320, 35331, 35350, 35344, 35340, 35355,
    35357, 35365, 35382, 35393, 35419, 35410, 35398, 35400, 35452, 35437,
    35436, 35426, 35461, 35458, 35460, 35496, 35489, 35473, {f: 2, c: 35493},
    35482, 35491, 35524, 35533, 35522, 35546, 35563, 35571, 35559, 35556,
    35569, 35604, 35552, 35554, 35575, 35550, 35547, 35596, 35591, 35610,
    35553, 35606, 35600, 35607, 35616, 35635, 38827, 35622, 35627, 35646,
    35624, 35649, 35660, 35663, 35662, 35657, 35670, 35675, 35674, 35691,
    35679, 35692, 35695, 35700, 35709, 35712, 35724, 35726, {f: 2, c: 35730},
    35734, {f: 2, c: 35737}, 35898, 35905, 35903, 35912, 35916, 35918, 35920,
    [12183, 35925], 35938, 35948, [12184, 35960], 35962, 35970, 35977, 35973,
    35978, {f: 2, c: 35981}, 35988, 35964, 35992, 25117, 36013, 36010, 36029,
    {f: 2, c: 36018}, 36014, 36022, 36040, 36033, 36068, 36067, 36058, 36093,
    {f: 2, c: 36090}, {f: 2, c: 36100}, 36106, 36103, 36111, 36109, 36112,
    40782, 36115, 36045, 36116, 36118, 36199, 36205, 36209, 36211, 36225,
    36249, 36290, 36286, 36282, 36303, 36314, 36310, 36300, 36315, 36299,
    {f: 2, c: 36330}, 36319, 36323, 36348, {f: 2, c: 36360}, 36351,
    {f: 2, c: 36381}, 36368, 36383, 36418, 36405, 36400, 36404, 36426, 36423,
    36425, 36428, 36432, 36424, 36441, 36452, 36448, 36394, 36451, 36437,
    36470, 36466, 36476, 36481, 36487, 36485, 36484, 36491, 36490, 36499,
    36497, 36500, 36505, 36522, 36513, 36524, 36528, 36550, 36529, 36542,
    36549, 36552, 36555, 36571, 36579, 36604, 36603, 36587, 36606, 36618,
    36613, 36629, 36626, 36633, 36627, 36636, 36639, 36635, 36620, 36646,
    36659, 36667, 36665, 36677, 36674, 36670, 36684, 36681, 36678, 36686,
    36695, 36700, {f: 3, c: 36706}, 36764, 36767, 36771, 36781, 36783, 36791,
    36826, 36837, 36834, 36842, 36847, 36999, 36852, 36869, {f: 2, c: 36857},
    36881, 36885, 36897, 36877, 36894, 36886, 36875, 36903, 36918, 36917,
    36921, 36856, {f: 4, c: 36943}, 36878, 36937, 36926, 36950, 36952, 36958,
    36968, 36975, 36982, 38568, 36978, 36994, 36989, 36993, 36992, 37002,
    37001, 37007, 37032, 37039, 37041, 37045, 37090, 37092, 25160, 37083,
    37122, 37138, 37145, 37170, 37168, 37194, 37206, 37208, 37219, 37221,
    37225, 37235, 37234, 37259, 37257, 37250, 37282, 37291, 37295, 37290,
    37301, 37300, 37306, {f: 2, c: 37312}, 37321, 37323, 37328, 37334, 37343,
    37345, 37339, 37372, {f: 2, c: 37365}, 37406, 37375, 37396, 37420, 37397,
    37393, 37470, 37463, 37445, 37449, 37476, 37448, 37525, 37439, 37451,
    37456, 37532, 37526, 37523, 37531, 37466, 37583, 37561, 37559, 37609,
    37647, 37626, 37700, 37678, 37657, 37666, 37658, 37667, 37690, 37685,
    37691, 37724, 37728, 37756, 37742, 37718, 37808, {f: 2, c: 37804}, 37780,
    37817, {f: 2, c: 37846}, 37864, 37861, 37848, 37827, 37853, 37840, 37832,
    37860, 37914, 37908, 37907, 37891, 37895, 37904, 37942, 37931, 37941,
    37921, 37946, 37953, 37970, 37956, 37979, 37984, 37986, 37982, 37994,
    37417, 38000, 38005, 38007, 38013, 37978, 38012, 38014, 38017, 38015,
    38274, 38279, 38282, 38292, 38294, {f: 2, c: 38296}, 38304, 38312, 38311,
    38317, 38332, 38331, 38329, 38334, 38346, 28662, 38339, 38349, 38348,
    38357, 38356, 38358, 38364, 38369, 38373, 38370, 38433, 38440,
    {f: 2, c: 38446}, 38466, 38476, 38479, 38475, 38519, 38492, 38494, 38493,
    38495, 38502, 38514, 38508, 38541, 38552, 38549, 38551, 38570, 38567,
    {f: 2, c: 38577}, 38576, 38580, [12202, 38582], 38584, [12203, 38585],
    38606, 38603, 38601, 38605, 35149, 38620, 38669, 38613, 38649, 38660,
    38662, 38664, 38675, 38670, 38673, 38671, 38678, 38681, 38692, 38698,
    38704, 38713, {f: 2, c: 38717}, 38724, 38726, 38728, 38722, 38729, 38748,
    38752, 38756, 38758, 38760, 21202, 38763, 38769, 38777, 38789, 38780,
    38785, 38778, 38790, 38795, {f: 2, c: 38799}, 38812, 38824, 38822, 38819,
    {f: 2, c: 38835}, 38851, 38854, 38856, [12209, 38859], 38876,
    [12210, 38893], 40783, 38898, 31455, 38902, 38901, 38927, 38924, 38968,
    38948, 38945, 38967, 38973, 38982, 38991, 38987, 39019, {f: 3, c: 39023},
    39028, 39027, 39082, 39087, 39089, 39094, 39108, 39107, 39110, 39145,
    39147, 39171, 39177, 39186, 39188, 39192, 39201, {f: 2, c: 39197}, 39204,
    39200, 39212, 39214, {f: 2, c: 39229}, 39234, 39241, 39237, 39248, 39243,
    {f: 2, c: 39249}, 39244, 39253, {f: 2, c: 39319}, 39333, {f: 2, c: 39341},
    39356, 39391, 39387, 39389, 39384, 39377, {f: 2, c: 39405},
    {f: 2, c: 39409}, 39419, 39416, 39425, 39439, 39429, 39394, 39449, 39467,
    39479, 39493, 39490, 39488, 39491, 39486, 39509, 39501, 39515, 39511,
    39519, 39522, 39525, 39524, 39529, 39531, 39530, 39597, 39600, 39612,
    39616, 39631, 39633, {f: 2, c: 39635}, 39646, [12221, 39647],
    {f: 2, c: 39650}, 39654, 39663, 39659, 39662, 39668, 39665, 39671, 39675,
    39686, 39704, 39706, 39711, {f: 2, c: 39714}, [12222, 39717],
    {f: 4, c: 39719}, 39726, [12223, 39727], [12224, 39730], 39748, 39747,
    39759, {f: 2, c: 39757}, 39761, 39768, 39796, 39827, 39811, 39825,
    {f: 2, c: 39830}, {f: 2, c: 39839}, 39848, 39860, 39872, 39882, 39865,
    39878, 39887, {f: 2, c: 39889}, 39907, 39906, 39908, 39892, 39905, 39994,
    39922, 39921, 39920, 39957, 39956, 39945, 39955, 39948, 39942, 39944,
    39954, 39946, 39940, 39982, 39963, 39973, 39972, 39969, 39984, 40007,
    39986, 40006, 39998, 40026, 40032, 40039, 40054, 40056, 40167, 40172,
    40176, 40201, 40200, 40171, 40195, 40198, 40234, 40230, 40367, 40227,
    40223, 40260, 40213, 40210, 40257, 40255, 40254, 40262, 40264,
    {f: 2, c: 40285}, 40292, 40273, 40272, 40281, 40306, 40329, 40327, 40363,
    40303, 40314, 40346, 40356, 40361, 40370, 40388, 40385, 40379, 40376,
    40378, 40390, 40399, 40386, 40409, 40403, 40440, 40422, 40429, 40431,
    40445, {f: 2, c: 40474}, 40478, [12228, 40565], 40569, 40573, 40577, 40584,
    {f: 2, c: 40587}, 40594, 40597, 40593, 40605, [12230, 40613], 40617, 40632,
    40618, 40621, 38753, 40652, {f: 3, c: 40654}, 40660, 40668, 40670, 40669,
    40672, 40677, 40680, 40687, 40692, {f: 2, c: 40694}, [12235, 40697],
    {f: 2, c: 40699}, [12236, 40701], {f: 2, c: 40711}, 30391, 40725, 40737,
    40748, 40766, [12241, 40778], [12242, 40786], 40788, 40803,
    {f: 3, c: 40799}, {f: 2, c: 40806}, 40812, 40810, 40823, 40818, 40822,
    40853, [12244, 40860], [12245, 40864], 22575, 27079, 36953, 29796, 0,
    {f: 76, c: 9472}, {f: 20, c: 9312}, {f: 10, c: 8544}, 13129, 13076, 0,
    13133, 0, 13095, 0, 13110, 13137, 0, 13069, 13094, 0, 13099, 13130, 0,
    {f: 3, c: 13212}, {f: 2, c: 13198}, 13252, 13217, 12317, 12319, 8470,
    13261, 0, {f: 5, c: 12964}, {f: 2, c: 12849}, 12857, 13182, 13181, 13180,
    8750, 8721, {s: 3}, 8735, 8895, 0, 0, 21854, {s: 7}, 167133, 0, 0, 28976,
    0, 40407, {s: 4}, 64054, 0, 0, 22169, 15694, {s: 4}, 20448, 0, 0, 36544, 0,
    194797, {s: 4}, 153716, 32363, 33606, 167670, {s: 3}, 40572, 0, 0, 26171,
    0, 40628, {s: 4}, 26629, {s: 5}, 23650, 0, 194780, 0, 32353, 0, 0, 64070,
    {s: 5}, 34083, 37292, {s: 7}, 34796, {s: 8}, 25620, 0, 0, 39506, {s: 4},
    64074, 0, 194692, {s: 4}, 31774, {s: 6}, 64016, 25681, 0, 0, 63980, 22625,
    39002, 0, 194679, {s: 3}, 31153, 0, 28678, {s: 9}, 22218, {s: 3}, 21085, 0,
    28497, 37297, {s: 10}, 64106, {s: 6}, 38960, 0, 40629, {s: 9}, 33802,
    63939, {f: 2, c: 63890}, 63897, 0, 34847, 194575, 0, 194771, 194584,
    {s: 7}, 137754, 23643, {s: 4}, 25890, 0, 0, 26618, 0, 26766, 0, 148432,
    194848, {s: 21}, 34110, {s: 15}, 30562, {s: 12}, 65075, 0,
    {f: 2, c: 65073}, {s: 4}, 65072, {f: 2, c: 65077}, {f: 2, c: 65081}, 0, 0,
    {f: 2, c: 65079}, {f: 2, c: 65087}, {f: 2, c: 65085}, {f: 4, c: 65089},
    {f: 2, c: 65083}, {s: 41}, {f: 3, c: 12436}, 0, 0, 22099, {s: 41}, 65508,
    65287, 65282, 0, 9665, 9655, 8681, 8679, 8678, 8680, 9634, 9831, 9825,
    9828, 9826, 13216, 13218, {f: 2, c: 13220}, 13207, 8467, 13208, 13235,
    13234, 13233, 13232, {f: 3, c: 13189}, 13259, 13200, 13268, 13206, 13090,
    13078, 13080, 13077, 13059, 13091, 13143, 13122, 13113, 13115, 13056,
    13105, 13127, 13086, 13098, 0, 13183, 8481, 9742, 12342, 12320, {s: 3},
    {f: 9, c: 9352}, {f: 20, c: 9332}, 12881, {f: 10, c: 8560},
    {f: 10, c: 12882}, {f: 26, c: 9372}, 12867, 12861, 12863, 12852, 12856,
    12851, 12860, 12866, 12862, 12854, 12853, 12859, 12864, 12858, 12976,
    12973, 12969, 12975, 12948, 12970, 12952, 12971, 12946, 12945, 12947,
    12972, 12974, 12950, {s: 8}, {f: 3, c: 9131}, 0, {f: 3, c: 9127}, 0, 13260,
    13061, 0, 0, 13215, 13219, 13222, 0, 0, 12958, {f: 2, c: 13192}, 13256,
    8749, 0, 12848, {f: 6, c: 12842}, 12855, 12865, 10145, {s: 3}, 9673, 9824,
    9829, 9827, 9830, {f: 4, c: 9728}, 9758, {f: 2, c: 9756}, 9759, 12953,
    9450, {f: 2, c: 8554}, {s: 3}, {f: 8, c: 9601}, 9615, 9614, 9613, 9612,
    9611, 9610, 9609, {f: 2, c: 9620}, {f: 2, c: 9581}, 9584, 9583, 9552, 9566,
    9578, 9569, {f: 2, c: 9698}, 9701, 9700, 0, 0, {f: 3, c: 9585}, {s: 20},
    20956, 29081, {f: 9, c: 10102}, {s: 3}, {f: 2, c: 8570}, {s: 3}, 8575,
    8458, 8457, 0, 0, 12292, 8646, {f: 2, c: 8644}, 0, {f: 4, c: 12535}, 0, 0,
    12957, {s: 3}, 13179, {s: 3}, 13107, 13134, {s: 30}, 32394, 35100, 37704,
    37512, 34012, 20425, 28859, 26161, 26824, 37625, 26363, 24389,
    [12033, 20008], 20193, 20220, 20224, 20227, 20281, 20310, 20370, 20362,
    20378, 20372, 20429, 20544, 20514, 20479, 20510, 20550, 20592, 20546,
    20628, 20724, 20696, 20810, 20836, 20893, 20926, 20972, 21013, 21148,
    21158, 21184, 21211, 21248, 0, 21284, 21362, 21395, 21426, 21469, 64014,
    21660, 21642, 21673, 21759, 21894, 22361, 22373, 22444, 22472, 22471,
    64015, 0, 22686, 22706, 22795, 22867, 22875, 22877, 22883, 22948, 22970,
    23382, 23488, 29999, 23512, 0, 23582, 23718, 23738, 23797, 23847, 23891, 0,
    23874, 23917, {f: 2, c: 23992}, 24016, 24353, 24372, 24423, 24503, 24542,
    24669, 24709, 24714, 24798, 24789, 24864, 24818, 24849, 24887, 24880,
    24984, 25107, 25254, 25589, 25696, 25757, 25806, 25934, 26112, 26133,
    26121, 26158, 0, 26148, 26213, 26199, 26201, 64018, 26227, 26265, 26272,
    26290, 26303, 26362, 26382, 0, 26470, 26555, 26706, 26560, 0, 26692, 26831,
    64019, 26984, 64020, 27032, 27106, 27184, 27243, 27206, 27251, 27262,
    27362, 27364, 27606, 27711, 27740, 27782, 27759, 27866, 27908, 28039,
    28015, 28054, 28076, 28111, 28152, 28146, 28156, 28217, 28252, 28199,
    28220, 28351, 28552, 28597, 28661, 28677, 28679, 28712, 28805, 28843,
    28943, 28932, 29020, {f: 2, c: 28998}, 0, 29121, 29182, 29361, 29374,
    29476, 64022, 29559, 29629, 29641, 29654, 29667, 29650, 29703, 29685,
    29734, 29738, 29737, 29742, 0, 29833, 29855, 29953, 30063, 30338, 30364,
    30366, 30363, 30374, 64023, 30534, 21167, 30753, 30798, 30820, 30842,
    31024, {f: 3, c: 64024}, 31124, 64027, 31131, 31441, 31463, 64028, 31467,
    31646, 64029, 32072, 0, 32183, 32160, 32214, 32338, 32583, 32673, 64030,
    33537, 33634, 33663, 33735, 33782, 33864, 33972, 34131, 34137, 34155,
    64031, 34224, {f: 2, c: 64032}, 34823, 35061, 35346, 35383, 35449, 35495,
    35518, 35551, 64034, 35574, 35667, 35711, 36080, 36084, 36114, 36214,
    64035, 36559, 0, 64037, 36967, 37086, 64038, 37141, 37159, 37338, 37335,
    37342, {f: 2, c: 37357}, {f: 2, c: 37348}, 37382, 37392, 37386, 37434,
    37440, 37436, 37454, 37465, 37457, 37433, 37479, 37543, {f: 2, c: 37495},
    37607, 37591, 37593, 37584, 64039, 37589, 37600, 37587, 37669, 37665,
    37627, 64040, 37662, 37631, 37661, 37634, 37744, 37719, 37796, 37830,
    37854, 37880, 37937, 37957, 37960, 38290, 0, 64041, 38557, 38575, 38707,
    38715, 38723, 38733, 38735, [12205, 38737], 0, 38999, 39013,
    {f: 2, c: 64042}, 39207, 64044, 39326, 39502, 39641, 39644, 39797, 39794,
    39823, 39857, 39867, 39936, 40304, 40299, 64045, 40473, 40657, 0, 92,
    {s: 634}, 8364, 8486, 0, 0, 64256, {f: 2, c: 64259}, 257, 299, 363, 275,
    333, 256, 298, 362, 274, 332, {f: 4, c: 8539}, {f: 2, c: 8531}, 8304,
    {f: 6, c: 8308}, {f: 10, c: 8320}, 461, 282, 0, 7868, 463, 0, 296, 465, 0,
    467, 366, 360, 462, 283, 0, 7869, 464, 0, 297, 466, 0, 468, 367, 361, 593,
    8049, 8048, 509, 0, 596, 0, 0, 601, 0, 0, 602, 0, 0, 603, 8051, 8050, 0,
    331, 629, 652, 0, 0, 658, 643, 720, {s: 682}, {f: 10, c: 12832}, {s: 108},
    {f: 4, c: 12892}, {f: 15, c: 12977}, {s: 50}, {f: 26, c: 9424},
    {f: 26, c: 9398}, {s: 48}, {f: 47, c: 13008}, 0, {f: 10, c: 12928}, 12944,
    {f: 6, c: 12938}, 0, 12959, {s: 6}, {f: 2, c: 12960}, 12955, 12954, 12963,
    12962, 12951, 0, 12956, 12949, {s: 6}, 9676, {s: 11}, 10111,
    {f: 10, c: 9451}, {s: 510}, 8414, {s: 815}, 13274, {s: 3}, 8448, 13250, 0,
    0, 8453, 0, 13169, 0, 0, 13197, 13211, {s: 3}, {f: 2, c: 13271}, {s: 3},
    {f: 2, c: 13057}, 13060, 13062, 0, 13064, 0, 13063, 13066, 0, 13065, 0,
    13067, 0, 13068, {f: 6, c: 13070}, 0, 13079, 0, 13081, 0, {f: 4, c: 13082},
    {f: 3, c: 13087}, 13092, 0, 13093, 0, 0, {f: 2, c: 13096}, 0, 13101, 0, 0,
    {f: 3, c: 13102}, 13106, 0, 0, {f: 2, c: 13108}, 13116, {s: 3}, 13111, 0,
    13112, 13114, 13117, 13121, {f: 3, c: 13118}, {f: 4, c: 13123}, 13128,
    {f: 2, c: 13131}, {f: 2, c: 13135}, 0, 0, 13138, 13140, 0, 0, 13139,
    {f: 2, c: 13141}, {s: 132}, 8501, 976, 8714, 8463, 0, 981, 987, 977, 0,
    {f: 2, c: 9832}, 9836, {s: 5}, 12347, 0, {f: 3, c: 12339}, 8252, 8265,
    {s: 5}, 8723, 0, 8771, {f: 2, c: 8818}, {s: 6}, {f: 2, c: 12312},
    {f: 2, c: 65375}, {s: 10}, 9115, {f: 2, c: 9117}, 9120, {s: 4}, 9121,
    {f: 2, c: 9123}, 9126, {s: 12}, [9116, 9119, 9122, 9125, 9130], {s: 8},
    9986, 0, 0, 12349, 0, 12447, 0, 0, 8709, 8864, 8854, 8856, 8853, 8855,
    {s: 4}, 9664, 9654, {s: 4}, 8656, 8596, {f: 2, c: 8600}, {f: 2, c: 8598},
    8652, 8651, {s: 10}, 12336, 8967, {s: 8}, 10048, 10047, {s: 7}, 9643, 0,
    9642, 0, 10010, {s: 12}, 9702, {s: 4}, 10070, {s: 379}, {f: 2, c: 65093},
    {s: 679}, 64103, 64098, 32227, [12232, 40643], 28331, 64082, 64061, 64069,
    64062, 27114, 28212, 64096, 64071, 64056, 64066, 64078, 34395, 64105,
    64052, 64099, 25581, 25802, 30799, 64084, 63856, 64077, 64097, 64072,
    64076, {f: 2, c: 64091}, 64081, 64067, 64090, 28041, 29376, 0, 194885,
    64086, 64080, 64049, 64059, 24034, 64063, 64101, 21373, 64055, 64095,
    24501, 64064, 0, 64083, 0, 64085, 64104, 64068, 64089, 26202, 64053, 64075,
    64100, 64065, 64048, 0, 64057, 64051, 27493, 64058, 27599, 64050, 25150,
    64079, 63773, 63964, 63798, 28122, 63952, 26310, 27511, 64087, 37706, 0,
    37636, {s: 120}, 133390, {s: 120}, 35999, 11991, [11965, 158033], {s: 5},
    37555, 38321, 0, 0, 194812, {s: 13}, 194965, {s: 8}, 194794, 0, 26478,
    11974, 0, 194594, {s: 13}, 13314, 0, 0, 26083, {s: 4}, 134071, {s: 10},
    171339, 0, 194611, 24378, {s: 8}, 11945, 0, 20465, {s: 7}, 63753, {s: 7},
    11964, 0, 0, 194732, 26435, {s: 3}, 133732, 35329, 25142, 0, 0, 21555,
    23067, {s: 3}, 25221, 0, 0, 194819, {s: 6}, 21567, {s: 9}, 27506, {s: 4},
    29986, 19256, 0, 0, 24063, {s: 6}, 194827, 29626, 134047, {s: 3}, 194600,
    0, 194849, {s: 5}, 194623, {s: 16}, 194675, {f: 2, c: 11916}, 23577,
    {s: 3}, 131083, 23426, 194642, {s: 5}, 11997, [11999, 39136],
    [11998, 169599], 14221, 0, [11927, 14586], 0, 194887, 0, [11909, 20155],
    131490, {s: 7}, 13599, 0, 194738, 0, 0, [11971, 35200], {s: 4}, 31237,
    {s: 4}, 35498, 0, 32085, 0, 28568, {s: 7}, 25591, 30246, {s: 4},
    [11978, 163767], {s: 5}, 146686, {s: 5}, 13351, 0, 0, 33067, 0, 0, 194842,
    {s: 5}, 11950, {s: 5}, 194714, {s: 3}, 194831, {s: 19}, 22305, 135741,
    194586, 0, 64003, {s: 7}, 21534, 15240, 20839, {s: 4}, 63839, {s: 9},
    20023, {s: 13}, [11946, 150804], 24421, 23020, 194658, 0, 24217, {s: 46},
    13416, {s: 8}, 21200, {s: 9}, 26625, 0, 195024, 195039, {s: 5}, 153215, 0,
    0, 11959, {s: 4}, 36534, 63775, {s: 3}, 63875, {s: 5}, 31867, 63906, 0,
    63898, 0, [11961, 32770], 157360, {s: 4}, [11911, 132648], 0, 0, 131210,
    194604, [11915, 13630], {s: 4}, 21589, 0, 22841, 0, 0, 23414, 194669,
    23572, 14306, 23782, 0, 20040, 0, 0, 194742, {s: 4}, 158105, 25371, 0, 0,
    26211, 0, 194779, 0, 0, 27126, 27014, {s: 3}, 27596, 0, 28183, 0, 0, 27818,
    {s: 3}, [11942, 20012], 0, 0, 29935, 30069, 30188, 30286, 16305, 30570,
    30633, {s: 6}, 31571, 0, 0, 16996, {s: 3}, 194924, 0, 0, 32328, {s: 5},
    11955, {s: 4}, 33089, 17491, 0, [11966, 33401], [11967, 64094],
    [11968, 64093], 0, 20857, 33626, {s: 3}, 17701, 0, 34292, 131248, {s: 4},
    34429, 0, 13358, 35014, {s: 6}, 18406, {s: 8}, 36808, {s: 19}, 166279, 0,
    0, 167447, 0, 0, 38969, {s: 6}, 39432, {s: 4}, 39903, {s: 10}, 148206,
    {s: 5}, 21385, 0, 64017, 194785, 0, 146622, 132625, 0, {f: 2, c: 19972},
    19999, 20011, {f: 2, c: 20015}, {f: 2, c: 20032}, 20036, [11907, 20058],
    20095, 20109, 20118, 20153, 20176, 20192, 20221, 20223, 20235, 20245,
    20320, 20283, 20297, 20308, 20346, {f: 2, c: 20349}, 20375, 20414, 20431,
    20477, {f: 2, c: 20480}, 20496, 20507, 20519, 20526, 20567, 20582, 20586,
    20539, 20623, 20630, 20636, 20684, 20710, 20713, 20719, 20744, 20747,
    20752, 20763, 20766, 20831, 20897, 20924, 0, 20974, 20980, 20993,
    [11913, 20994], 21011, 21065, 21089, 21094, 21139, 21192, 21232,
    {f: 2, c: 21258}, 21310, 21324, 21323, 21345, 21356, 21419, 21466, 21478,
    21493, 21543, 21581, 21606, 21611, 21620, 21645, 21654, 21665, 21677,
    21689, 21695, 21702, 21709, 21774, 21803, 21813, 21834, 21856, 0, 21896,
    21902, 22024, {f: 2, c: 22030}, 22071, 22079, 22089, 22091, 22095, 22118,
    22121, 22127, {f: 2, c: 22129}, 22165, 22170, {f: 2, c: 22188}, 22193,
    22217, 22237, 22244, 22282, 22293, 22307, 22319, {f: 2, c: 22323}, 22348,
    22384, 22412, 22428, 22456, 22502, 22509, {f: 2, c: 22517}, 22527, 22537,
    22560, 22578, 22652, 22656, 22697, 22734, 22736, 22740, 22746, 22761,
    22796, 22820, 22831, 22881, 22893, 22986, 22994, 23005, {f: 2, c: 23011},
    23044, 23052, 23075, 23111, 23125, 23139, 23149, 23166, 23198, 23207,
    23212, 23219, 23264, 23296, 23321, 23333, 23341, 23361, 23420,
    {f: 2, c: 23422}, 23434, [11919, 23587], 23595, 23600, 23651, 23657, 23676,
    23755, 23762, 23796, 23844, 23846, 23875, 23878, 23882, 23954, 23956,
    23961, 23968, 24024, 24032, 24056, 24064, 24082, {f: 2, c: 24084}, 24088,
    24110, 24152, {f: 2, c: 24171}, 24232, 24234, {f: 2, c: 24254}, 0, 24274,
    24327, 24334, {f: 2, c: 24348}, 24354, 24360, 24374, 24379, 24384,
    [12089, 24400], 24408, 24420, 24457, 24476, 24487, 24484, 24495, 24504,
    [11926, 24516], 24521, 24545, 24553, 24557, 24572, 24599, 24602, 24627,
    24673, 24703, 24734, 24740, 24752, 24779, 24795, 24824, {f: 3, c: 24850},
    24860, 24956, 24973, 24991, 25000, 25026, 25055, 25109, 25129, 25155,
    25158, [11928, 25164], 25169, 25174, 25284, 25340, 25354, 25357, 25368,
    25401, {f: 2, c: 25410}, 25445, 25460, 25469, 25476, 25479, 25488, 25502,
    25553, 25564, 25609, 25616, 25634, 25684, 25691, 25709, 25723,
    {f: 2, c: 25790}, 25829, 25847, 25851, 25860, 25878, 25881, 25927, 25959,
    25985, 25989, 26050, 26096, 26098, 26156, 26188, {f: 2, c: 26203}, 26209,
    26219, 0, 26276, 26312, 26348, 26373, 26387, 26419, 26440, 26444, 26486,
    26491, 26544, 26546, 26617, 26583, 26585, 26608, 26668, {f: 2, c: 26672},
    26715, 26738, 26741, 26746, 26756, 26789, 26802, 26832, 26838, 26856,
    26861, {f: 2, c: 26864}, 26876, 26897, 26899, 26933, 26939, 26967, 26979,
    26994, {f: 2, c: 27007}, 27046, 27053, 27063, {f: 2, c: 27094}, 27137,
    27151, 27157, 27176, 27188, 27198, 27205, {f: 2, c: 27216}, 27222, 27227,
    27267, 27273, 27281, {f: 3, c: 27293}, 27356, 27367, 27372, 27422, 27428,
    27445, 27462, 27478, 27488, 27522, 27582, 27617, 27633, 27664, 27699,
    [11937, 27701], 11938, 27737, 27766, 27771, 27781, 27797, 27804, 27856,
    27860, 27862, 27872, {f: 2, c: 27883}, 27886, 27914, 27918, 27921, 27950,
    27991, 27998, 28005, 28034, 28095, 28100, 28106, 28118, 28137, 28194,
    28241, 28359, 28362, 28366, 28413, 28442, 28458, 28463, 28467, 28506,
    28510, 28514, 28541, 28555, 28557, 28562, 28564, 28570, {f: 2, c: 28583},
    28598, 28634, 28638, 0, 28729, 28732, 0, 28756, {f: 2, c: 28765}, 28772,
    [11939, 28780], 28798, 28801, 28821, 28855, {f: 2, c: 28883}, 28888, 28892,
    28935, 28960, 28977, 29002, 29010, 29024, 29049, 29074, 0, 29131, 29139,
    29142, 29184, 29213, 29227, 29240, 29249, 29267, {f: 2, c: 29269}, 29276,
    29325, [11944, 29357], 29364, 29383, 29435, {f: 2, c: 29444}, 29480, 29489,
    29507, 29548, 29564, 29571, {f: 2, c: 29573}, 29589, {f: 3, c: 29598},
    29606, 29611, 29621, 29623, 29628, 29647, 29657, 29673, 29684, 29693,
    29700, 29706, {f: 2, c: 29722}, 29732, 29736, 29740, {f: 3, c: 29743},
    29753, 29764, 29767, 29771, 29773, 29777, 29783, 29798, 29803, 29809,
    29824, {f: 3, c: 29829}, 29840, 29848, 29852, 29856, 29859, 29864, 29867,
    29877, 29887, 29896, 29914, 29918, 30030, 30073, 30081, 30096,
    [12135, 30098], 30099, 30132, 30180, 30201, 30208, 30218, {f: 2, c: 30229},
    30233, 30238, 30253, 30261, 30275, 30283, 30309, 30317, 30319, 30321,
    30324, {f: 2, c: 30372}, 30405, 30412, 30444, 30460, 30516, 30518, 30556,
    {f: 2, c: 30559}, 30578, 30589, 30613, 30634, 30694, 30704, 30708, 30726,
    30754, {f: 2, c: 30765}, 30768, 30773, 30824, 30878, 30920, 30924, 30926,
    30948, {f: 2, c: 30944}, 30962, 30967, 30971, 31025, 0, [11949, 31035],
    31037, 31045, {f: 2, c: 31067}, 31115, 31126, 31128, [12145, 31160], 31163,
    31178, 31194, 31235, 31241, 31249, 31262, 31277, 31289, 31301, 31308,
    31325, 0, 31341, 31352, 31392, 31395, 31411, {f: 2, c: 31419}, 31430,
    31495, 31508, 31527, 31537, 31559, 31566, 31584, 31593, 31597, 31602,
    31633, 31663, 31703, 31705, 31755, 31759, 31776, 31782, 31793, 31798,
    31825, 31833, 31847, 31854, 31856, 31932, 31935, {f: 2, c: 31944}, 31959,
    31961, 31965, 31979, {f: 3, c: 32007}, 32019, 32029, 32035, 32065, 32083,
    32089, 32093, 32122, 32134, {f: 2, c: 32139}, 32204, 32235, 32241, 32249,
    32264, 32273, 32277, 32288, 32327, 32354, 32366, 32371, 32397, 32401,
    32408, 32580, 32591, [11947, 11954, 32594], [11953, 32595], 32609, 32657,
    32703, 32718, 32735, 32741, 32748, {f: 2, c: 32750}, 32762, 32782, 32785,
    32788, 32804, 32806, 32826, 32828, 32864, 32881, 32885, 32926, 32934,
    32939, {f: 2, c: 32983}, 33046, 33048, 33082, 33098, 33100, 33153, 33156,
    33204, 33231, 33273, 33283, 33313, 33330, 33332, 33350, 33355, 33359,
    33422, 33454, 33463, 33470, 33478, 33534, 33603, 33617, 33621, 33670,
    33677, 33682, 33688, 33705, {f: 2, c: 33727}, 33770, 33807, 33809, 33866,
    33910, 33960, 33967, 33984, 33986, 34032, 34045, 34060, 34100, 34142,
    34191, 34231, 34254, 34221, 34322, 34345, 34386, 34403, 34412, 34415,
    34426, 34445, 34449, 34456, {f: 2, c: 34471}, 34554, 34557, 34571, 34579,
    34585, 34590, 34600, 34622, 34673, 34696, 34713, {f: 2, c: 34732}, 34741,
    34774, 34795, 34797, 34817, 0, 34822, 34827, 34836, 34844, 34902, 34911,
    [11970, 34916], 34968, 34986, {f: 2, c: 35005}, 35018, 35026, 35035,
    {f: 2, c: 35056}, 35078, {f: 3, c: 35096}, 35111, 35120, 35134, 35195,
    35284, 35286, 35301, 35313, 35335, 35343, 35349, 35362, 35406, 35455,
    35572, 35615, 35639, {f: 2, c: 35651}, 35668, 35740, 35742, 35911, 35924,
    35955, 36004, 36057, 36065, 36088, 36094, 36123, 36201, 36204, 36228,
    36237, 36245, 36262, 36294, 36302, 36324, 36332, 36384, 36427, 36460,
    36464, 36474, 36498, 36526, 36531, 36561, 36564, 36601, 36631, 36662,
    36774, [12193, 36789], [11981, 36790], 0, 36832, 36836, 36854, 36866,
    36908, 36932, 37000, 37013, 37017, 37019, 37026, 37044, 37079, 37085,
    37108, 37143, 37148, 37169, 37178, 37181, 37192, 37211, 37217, 37220,
    37262, 37278, 37288, {f: 2, c: 37293}, 37298, 37308, 37360, 37367, 37371,
    37383, 37416, 37427, 37432, 37443, 37447, 37455, 37472, 37570,
    {f: 2, c: 37579}, 37599, 37645, 37653, 37663, 37671, 37703, 37714, 0,
    37738, 37741, 37787, 37818, 37801, 37825, 37834, 37858, 37882, 37885,
    37903, 37940, 37951, 37973, 37995, 38002, [11986, 38264], 38310, 38313, 0,
    38324, 38333, 38362, [11983, 11990, 38429], 38465, 38488, 38532, 38564,
    38569, 38610, 195060, 38622, 38633, 38641, 38658, 38665, 38746, 38755,
    38766, 38771, 38810, 38818, {f: 2, c: 38837}, 38873, 38878, 38900, 38922,
    38926, 38942, 38947, 38955, 38974, {f: 2, c: 38994}, 39001, 39020, 39096,
    39098, 39103, 39112, 39141, {f: 2, c: 39218}, 39232, 39245, 39260, 39263,
    39345, {f: 2, c: 39353}, 39369, 39426, 39446, 39460, 39463,
    {f: 2, c: 39469}, 39478, 39480, 39498, 39510, {f: 2, c: 39605}, 39673,
    39683, 39712, {f: 2, c: 39731}, 39795, 39801, 39847, 39873, 39879, 39895,
    39911, 39915, 39927, 39930, 39933, 39947, 39975, 39978, 39990, 40001,
    40019, 40035, 40048, 40055, 40194, 40258, 40263, 40291, 40297, 40316,
    40318, 40333, 40369, 40387, 40391, 40406, 40415, 40427, 40436, 40469,
    40477, 40612, 40616, 40620, 40679, 40686, 40720, 40722, 40727, 40729,
    40751, 40759, 40761, 40769, 40773, 40791, 40808, 40817, 40821, 40848,
    40852, 40866, 0, 13317, 194564, 22048, 24267, 11925, 0, 144954, 0, 28665,
    28390, 29107, [11940, 64073], {s: 4}, [11980, 64102], 0, 23986, 0, 20435,
    20697, 20720, 20931, 22134, 27220, 27905, 28112, 28226, 28377, 29668,
    29729, 30060, 30801, 34805, 144382, 29608, 15091, 13531, 17420, 16010, 0,
    0, 19432, 0, 16090, 15138, 0, 17786, 16531, 0, 18021, 16643, 17043, 18094,
    13448, 140809, {f: 3, c: 63584}, 63610, 63615, {s: 23}, {f: 2, c: 8836},
    {f: 2, c: 8842}, 8713, 0, {f: 2, c: 8965}, {s: 9}, {f: 2, c: 8741},
    {s: 14}, 8802, 0, 8773, 8776, {f: 2, c: 8822}, {s: 4}, 8487, {s: 209},
    {f: 2, c: 8922}, 8533, 8984, {f: 2, c: 7742}, {f: 2, c: 504}, 470, 472,
    474, 476, 260, 728, 317, 346, 350, 356, 377, 379, 261, 731, 318, 347, 711,
    351, 357, 378, 733, 380, 340, 258, 313, 262, 268, 280, 270, 323, 327, 336,
    344, 368, 354, 341, 259, 314, 263, 269, 281, 271, 273, 324, 328, 337, 345,
    369, 355, 729, 264, 284, 292, 308, 348, 364, 265, 285, 293, 309, 349, 365,
    625, 651, 638, 620, 622, 633, 648, 598, 627, 637, 642, 656, 635, 621, 607,
    626, 669, 654, 609, 624, 641, 295, 661, 660, 614, 664, 450, 595, 599, 644,
    608, 403, 616, 649, 600, 604, 606, 592, 623, 650, 612, 594, 653, 613, 674,
    673, 597, 657, 634, 615, 865, 712, 716, 721, 8255, 783, {f: 5, c: 741}, 0,
    0, 805, 812, 825, 796, {f: 2, c: 799}, 829, 809, 815, 734, 804, 816, 828,
    820, {f: 2, c: 797}, {f: 2, c: 792}, 810, {f: 2, c: 826}, 794, {s: 3},
    {f: 2, c: 610}, 618, 628, 630, 632, 640, 655, 665, 668, 671, 688, 690, 695,
    704, {f: 2, c: 736}, {s: 6}, 8862, {s: 287}, 12348, 12543, 0,
    {f: 2, c: 12310}, 9838, 9835, {f: 2, c: 10548}, 10687, 0, 12448, 0,
    {f: 2, c: 10746}, {s: 13}, 962, {f: 10, c: 9461}, {f: 2, c: 9750}, 9649,
    {f: 10, c: 12784}, 0, {f: 6, c: 12794}, {f: 15, c: 9150}, 0, 0, 10003, 0,
    9251, 9166, {f: 4, c: 9680}, {f: 2, c: 8263}, 0, 8273, 8258,
    {f: 16, c: 12688}, {s: 13}, {f: 2, c: 9136}, {f: 12, c: 9842},
    {f: 2, c: 12441}, 8413, {s: 450}, 20296, 20319, 20330, 20332, 20494, 20504,
    20545, 20722, 20688, 20742, 20739, 20789, 20821, 20823, 13493, 20938,
    20962, 21079, 21196, 21206, 21243, 21276, 21347, 21405, 21522, 21631,
    21640, 21840, 21889, 21933, 21966, 22075, 22174, 22185, 22195, 22391,
    22396, 135963, 22479, 22500, 22628, 22665, 136302, 22738, 22752, 34369,
    22923, 22930, 22979, 23059, 23143, 23159, 23172, 23236, 137405, 23421,
    23443, 23570, 64060, 136884, 23674, 23695, 23711, 23715, 23722, 23760,
    138804, 23821, 23879, 23937, 23972, 23975, 24011, 24158, 24313, 24320,
    24322, 24355, 24381, 24404, 24445, 24589, 24596, 24600, 24629, 24647,
    24733, 24788, 24797, 24875, 25020, 25017, 25122, 25178, 25199, 25302,
    25468, 25573, 25721, 25796, 25808, 25897, 26013, 26170, 26146, 26155,
    26160, 26163, 26184, 143812, {f: 2, c: 26231}, 26253, 26299, 26331, 26344,
    26439, 26497, 26515, 26520, 26523, 26620, 26653, 26787, 26890, 26953,
    144836, 26946, 26980, 27045, 27087, 15286, 15299, 27113, 27125, 145215,
    27195, 145251, 27284, 27301, 15375, 27419, 27436, 27495, 27561, 27565,
    27607, 27647, 27653, 27764, 27800, 27899, 27846, 27953, 27961, 27967,
    27992, 28052, 28074, 28123, 28125, 28228, 28254, 28337, 28353, 28432,
    28505, 28513, 28542, 28556, 28576, 28604, 28615, 28618, 28656, 28750,
    28789, 28836, 28900, 28971, 28958, 28974, 29009, 29032, 29061, 29063,
    29114, 29124, 29205, 15935, 29339, 149489, 29479, 29520, 29542, 29602,
    29739, 29766, 29794, 29805, 29862, 29865, 29897, 29951, 29975, 16242,
    30158, 30210, 30216, 30308, 30337, 30365, 30378, 30390, 30414, 30420,
    30438, 30449, 30474, 30489, {f: 2, c: 30541}, 30586, 30592, 30612, 30688,
    152718, 30787, 30830, 30896, 152846, 30893, 30976, 31004, 31022, 31028,
    31046, 31097, 31176, 153457, 31188, 31198, 31211, 31213, 31365, 154052,
    31438, 31485, 31506, 31533, 31547, 31599, 31745, 31795, 155041, 31853,
    31865, 31887, 31892, 31904, 31957, 32049, 32092, 32131, 32166, 32194,
    32296, 32663, 32731, 32821, 32823, 32970, 32992, 33011, 33120,
    {f: 2, c: 33127}, 33133, 33211, 33226, 33239, 17499, 33376, 33396, 158463,
    33441, {f: 2, c: 33443}, 33449, 33471, 33493, 33533, 33536, 33570, 33581,
    33594, 33607, 33661, 33703, 33743, 33745, 33761, 33793, 33798, 33887,
    33904, 33907, 33925, 33950, 33978, 159296, 34098, 34078, 34095, 34148,
    34170, 34188, 34210, 34251, 34285, 34303, {f: 2, c: 34308}, 34320, 159988,
    34328, 34360, 34391, 34402, 17821, 34421, 34488, 34556, 34695, 17898,
    34826, 34832, 35022, 161412, 35122, 35129, 35136, 35220, 35318, 35399,
    35421, 35425, 35445, 35536, 35654, 35673, 35689, 35741, 35913, 35944,
    36271, 36305, 36311, 36387, 36413, 36475, 164471, 18500, 36602, 36638,
    36653, 36692, 164813, 36840, 36846, 36872, 36909, 37015, 37043, 37054,
    {f: 2, c: 37060}, 37063, 37103, 37140, 37142, {f: 2, c: 37154}, 37167,
    37172, 37251, 37361, 37705, {f: 2, c: 37732}, 37795, 37855, 37892, 37939,
    37962, 37987, 38001, 38286, 38303, 38316, 38326, 38347, 38352, 38355,
    18864, 38366, 38565, 38639, 38734, 38805, 38830, 38842, 38849, 38857,
    38875, 38998, 39143, 39256, 39427, 39617, 39619, 39630, 39638, 39682,
    39688, 19479, 39725, 39774, 39782, 39812, 39818, 39838, 39886, 39909,
    39928, 39971, {f: 2, c: 40015}, 40037, {f: 2, c: 40221}, 40259, 40274,
    40330, 40342, 40384, 40364, 40380, 172432, 40423, 40455, 40606, 40623,
    40855, 131209, 19970, 19983, 19986, 20009, 20014, 20039, 131234, 20049,
    13318, 131236, 20073, 20125, 13356, 20156, 20163, 20168, 20203, 20186,
    20209, 20213, 20246, 20324, 20279, 20286, 20312, 131603, {f: 2, c: 20343},
    20354, 20357, 20454, 20402, 20421, 20427, 20434, 13418, 20466, 20499,
    20508, 20558, 20563, 20579, 20643, 20616, {f: 2, c: 20626}, 20629, 20650,
    131883, 20657, {f: 2, c: 20666}, 20676, 20679, 20723, 131969, 20686,
    131953, 20692, 20705, 13458, 132089, 20759, 132170, 20832, 132361, 20851,
    20867, 20875, 13500, 20888, 20899, 20909, 13511, 132566, 20979, 21010,
    21014, 132943, 21077, 21084, 21100, 21111, 21124, 21122, 133127, 21144,
    133178, 21156, {f: 2, c: 21178}, 21194, 21201, 133305, 21239, 21301, 21314,
    133500, 133533, 21351, 21370, 21412, 21428, 133843, 21431, 21440, 133917,
    {f: 2, c: 13661}, 21461, 13667, 21492, 21540, 21544, 13678, 21571, 21602,
    21612, 21653, 21664, 21670, 21678, 21687, 21690, 21699, 134469, 21740,
    21743, 21745, 21747, {f: 2, c: 21760}, 21769, 21820, 21825, 13734, 21831,
    13736, 21860, 134625, 21885, 21890, 21905, 13765, 21970, 134805, 134765,
    21951, 21961, 21964, 21969, 21981, 13786, 21986, 134756, 21993, 22056,
    135007, 22023, 22032, 22064, 13812, 22077, 22080, 22087, 22110, 22112,
    22125, 13829, 22152, 22156, 22173, 22184, 22194, 22213, 22221, 22239,
    22248, {f: 2, c: 22262}, 135681, 135765, 22313, 135803, {f: 2, c: 22341},
    22349, 135796, 22376, 22383, {f: 3, c: 22387}, 22395, 135908, 135895,
    22426, {f: 2, c: 22429}, 22440, 22487, 135933, 22476, 135990, 136004,
    22494, 22512, 13898, 22520, 22523, 22525, 22532, 22558, 22567, 22585,
    136132, 22601, 22604, 22631, {f: 2, c: 22666}, 22669, {f: 2, c: 22671},
    22676, 22685, 22698, 22705, 136301, 22723, 22733, 22754, {f: 2, c: 22771},
    {f: 2, c: 22789}, 22797, 22804, 136663, 13969, 22845, 13977, 22854, 13974,
    158761, 22879, 136775, {f: 2, c: 22901}, 22908, 22943, 22958, 22972, 22984,
    22989, 23006, 23015, 23022, 136966, 137026, 14031, 23053, 23063, 23079,
    23085, 23141, 23162, 23179, 23196, {f: 2, c: 23199}, 23202, 23217, 23221,
    23226, 23231, 23258, 23260, 23269, 23280, 23278, 23285, 23304, 23319,
    23348, 23372, 23378, 23400, 23407, 23425, 23428, 137667, 23446, 23468,
    {f: 2, c: 14177}, 23502, 23510, 14188, 14187, 23537, 23549, 14197, 23555,
    23593, 138326, 23647, {f: 2, c: 23655}, 23664, 138541, 138565, 138616,
    138594, 23688, 23690, 14273, 138657, 138652, 23712, 23714, 23719, 138642,
    23725, 23733, 138679, 23753, 138720, 138803, 23814, 23824, 23851, 23837,
    23840, 23857, 23865, 14312, 23905, 23914, 14324, 23920, 139038, 14333,
    23944, 14336, 23959, 23984, 23988, 139126, 24017, 24023, 139258, 24036,
    24041, 14383, 14390, 14400, 24095, 24126, 24137, 14428, 24150, 14433,
    {f: 2, c: 24173}, 139643, 24229, 24236, 24249, 24262, 24281, 140062, 24317,
    24328, 140205, 24350, 24391, 24419, 24434, 24446, 24463, 24482, 24519,
    24523, {f: 3, c: 24530}, 24546, {f: 2, c: 24558}, 24563, 14615, 24610,
    24612, 14618, 24652, 24725, 24744, 141043, 24753, 24766, 24776, 24793,
    24814, 24821, 24848, 24857, 24862, 24890, 14703, 24897, 24902, 24928,
    141403, {f: 2, c: 24978}, 24983, 24997, 25005, 141483, 25045, 25053, 25077,
    141711, 25123, 25170, 25185, 25188, 25211, 25197, 25203, 25241, 25301,
    142008, 25341, 25347, 25360, {f: 2, c: 142159}, 25394, 25397,
    {f: 2, c: 25403}, 25409, 25412, 25422, 142150, 25433, 142365, 142246,
    25452, 25497, 142372, 25492, 25533, {f: 2, c: 25556}, 25568,
    {f: 2, c: 25579}, 25586, 25630, 25637, 25641, 25647, 25690, 25693, 25715,
    25725, 25735, 25745, 25759, {f: 2, c: 25803}, 25813, 25815, 142817, 25828,
    25855, 14958, 25871, 25876, 14963, 25886, 25906, 25924, 25940, 25963,
    25978, 25988, 25994, 26034, 26037, 26040, 26047, 26057, 26068, 15062,
    26105, 26108, 26116, 26120, 26145, 26154, 26181, 26193, 26190, 15082,
    143811, 143861, 143798, 26218, {f: 2, c: 26220}, 26235, 26240, 26256,
    26258, 15118, 26285, 26289, 26293, 15130, 15132, 15063, 26369, 26386,
    144242, 26393, 144339, 144338, 26445, 26452, 26461, 144336, 144356, 144341,
    26484, 144346, 26514, 144351, 33635, 26640, 26563, 26568, 26578, 26587,
    26615, 144458, 144465, 144459, 26648, 26655, 26669, 144485, 26675, 26683,
    26686, 26693, 26697, 26700, 26709, 26711, 15223, 26731, 26734, 26748,
    26754, 26768, 26774, 15213, {f: 3, c: 26776}, 26780, {f: 2, c: 26794},
    26804, 26811, 26875, 144612, 144730, 26819, 26821, 26828, 26841,
    {f: 2, c: 26852}, 26860, 26871, 26883, 26887, 15239, 144788, 15245, 26950,
    26985, 26988, 27002, 27026, 15268, 27030, 27056, 27066, 27068, 27072,
    27089, 144953, 144967, 144952, 27107, {f: 2, c: 27118}, 27123, 15309,
    27124, 27134, 27153, 27162, 27165, 145180, {f: 2, c: 27186}, 27199, 27209,
    27258, 27214, 27218, 27236, 145164, 27275, 15344, 27297, 145252, 27307,
    27325, 27334, 27348, 27344, 27357, 145407, 145383, {f: 3, c: 27377}, 27389,
    145444, 27403, {f: 3, c: 27407}, 145469, 27415, 15398, 27439, 27466, 27480,
    27500, 27509, [11934, 27514], 27521, 27547, 27566, 146072, 27581,
    {f: 3, c: 27591}, 27610, {f: 2, c: 27622}, 27630, 27650, 27658, 27662,
    27702, 146559, 27725, 27739, 27757, 27780, 27785, 15555, 27796, 27799,
    27821, 27842, 15570, 27868, 27881, 27885, 146688, 27904, 27940,
    {f: 2, c: 27942}, 27751, 27951, 27964, 27995, 28000, 28016,
    {f: 2, c: 28032}, 28042, 28045, 28049, 28056, 146752, 146938, 146937,
    146899, 28075, 28078, 28084, 28098, 27956, 28104, 28110, 28127, 28150,
    28214, 28190, 15633, 28210, {f: 2, c: 28232}, {f: 2, c: 28235}, 28239,
    {f: 2, c: 28243}, 28247, 28259, 15646, 28307, 28327, 28340, 28355, 28469,
    28395, 28409, 28411, 28426, 28428, 28440, 28453, 28470, 28476, 147326,
    28498, 28503, 28512, 28520, 28560, 28566, 28606, 28575, 28581, 28591,
    15716, {f: 2, c: 28616}, 28649, 147606, 28668, 28672, 28682, 28707, 147715,
    28730, 28739, 28743, 28747, 15770, 28773, 28777, 28782, 28790, 28806,
    28823, 147910, 28831, 28849, 147966, 28908, 28874, 28881, 28931, 28934,
    28936, 28940, 15808, 28975, 29008, 29011, 29022, 15828, 29078, 29056,
    29083, 29088, 29090, {f: 2, c: 29102}, 148412, 29145, 29148, 29191, 15877,
    29236, 29241, 29250, 29271, 29283, 149033, {f: 2, c: 29294}, 29304, 29311,
    29326, 149157, 29358, 29360, 29377, 15968, 29388, 15974, 15976, 29427,
    29434, 29447, 29458, {f: 2, c: 29464}, 16003, 29497, 29484, 29491, 29501,
    29522, 16020, 29547, 149654, {f: 2, c: 29550}, 29553, 29569, 29578, 29588,
    29592, 29596, 29605, 29625, 29631, 29637, 29643, 29665, 29671, 29689,
    29715, 29690, 29697, 29779, 29760, 29763, 29778, 29789, 29825, 29832,
    150093, 29842, 29847, 29849, 29857, 29861, 29866, 29881, 29883, 29882,
    29910, 29912, 29931, 150358, 29946, 150383, 29984, 29988, 29994, 16215,
    150550, {f: 2, c: 30013}, 30016, 30024, 30032, 30034, 30066, 30065, 30074,
    {f: 2, c: 30077}, 30092, 16245, 30114, 16247, 30128, 30135,
    {f: 2, c: 30143}, 30150, 30159, 30163, 30173, {f: 2, c: 30175}, 30183,
    30190, 30193, 30211, 30232, 30215, 30223, 16302, 151054, 30227,
    {f: 2, c: 30235}, 151095, 30245, 30248, 30268, 30259, 151146, 16329, 30273,
    151179, 30281, 30293, 16343, 30318, 30357, 30369, 30368, {f: 2, c: 30375},
    30383, 151626, 30409, 151637, 30440, 151842, 30487, 30490, 30509, 30517,
    151977, 16441, 152037, 152013, 30552, 152094, 30588, 152140, 16472, 30618,
    30623, 30626, 30628, {f: 2, c: 30686}, 30692, 30698, 30700, 30715, 152622,
    30725, 30729, 30733, 30745, 30764, 30791, 30826, 152793, 30858, 30868,
    30884, 30877, 30879, 30907, 30933, 30950, {f: 2, c: 30969}, 30974, 152999,
    30992, 31003, 31013, 31050, 31064, 16645, 31079, 31090, 31125, 31137,
    31145, 31156, 31170, 31175, {f: 2, c: 31180}, 31190, 16712, 153513, 153524,
    16719, 31242, 31253, 31259, 16739, 31288, 31303, 31318, 31321, 31324,
    31327, 31335, 31338, 31349, 31362, 31370, 31376, 31404, 154068, 16820,
    31417, 31422, 16831, 31436, 31464, 31476, 154340, 154339, 154353, 31549,
    31530, {f: 2, c: 31534}, 16870, 16883, 31615, 31553, 16878, 31573, 31609,
    31588, 31590, 31603, 154546, 16903, 31632, 31643, 16910, 31669, 31676,
    31685, 31690, 154699, 154724, 31700, 31702, 31706, 31722, 31728, 31747,
    31758, 31813, 31818, 31831, 31838, 31841, 31849, 31855, 155182, 155222,
    155237, 31910, 155234, {f: 2, c: 31926}, 155352, 31940, 155330, 31949,
    155368, 155427, 31974, 155484, 31989, 32003, 17094, 32018, 32030, 155616,
    155604, {f: 2, c: 32061}, 32064, 32071, 155660, 155643, 17110, 32090,
    32106, 32112, 17117, 32127, 155671, 32136, 32151, 155744, 32157, 32167,
    32170, 32182, 32192, 32215, 32217, 32230, 17154, 155885, 64088, 32272,
    32279, 32285, 32295, 32300, 32325, 32373, 32382, {f: 2, c: 32390}, 17195,
    32410, 17219, 32572, 32571, 32574, 32579, 13505, 156272, 156294,
    {f: 2, c: 32611}, 32621, {f: 2, c: 32637}, 32656, 20859, 146702, 32662,
    32668, 32685, 156674, 32707, 32719, 32739, 32754, 32778, 32776, 32790,
    32812, 32816, 32835, 32870, 32891, 32921, 32924, 32932, 32935, 32952,
    157310, 32965, 32981, 32998, 33037, 33013, 33019, 17390, 33077, 33054,
    17392, 33060, 33063, 33068, 157469, 33085, 17416, 33129, 17431, 17436,
    33157, 17442, 33176, 33202, 33217, 33219, 33238, 33243, 157917, 33252,
    157930, 33260, 33277, 33279, 158063, 33284, 158173, 33305, 33314, 158238,
    33340, 33353, 33349, 158296, 17526, 17530, 33367, 158348, 33372, 33379,
    158391, 17553, 33405, 33407, 33411, 33418, 33427, {f: 2, c: 33447}, 33458,
    33460, 33466, 33468, 33506, 33512, 33527, {f: 2, c: 33543}, 33548, 33620,
    33563, 33565, 33584, 33596, 33604, 33623, 17598, 17620, 17587,
    {f: 2, c: 33684}, 33691, 33693, 33737, 33744, 33748, 33757, 33765, 33785,
    33813, 158835, 33815, 33849, 33871, {f: 2, c: 33873}, {f: 2, c: 33881},
    33884, 158941, 33893, 33912, 33916, 33921, 17677, 33943, 33958, 33982,
    17672, {f: 2, c: 33998}, 34003, 159333, 34023, 34026, 34031, 34033, 34042,
    34075, {f: 2, c: 34084}, 34091, 34127, 34159, 17731, 34129,
    {f: 2, c: 34145}, 159636, 34171, 34173, 34175, 34177, 34182, 34195, 34205,
    34207, 159736, {f: 2, c: 159734}, 34236, 34247, 34250, {f: 2, c: 34264},
    34271, 34273, 34278, 34294, 34304, 34321, 34334, 34337, 34340, 34343,
    160013, 34361, 34364, 160057, 34368, 34387, 34390, 34423, 34439, 34441,
    {f: 2, c: 34460}, 34481, 34483, 34497, 34499, 34513, 34517, 34519, 34531,
    34534, 17848, 34565, 34567, 34574, 34576, 34591, 34593, 34595, 34609,
    34618, 34624, 34627, 34641, 34648, {f: 2, c: 34660}, 34674, 34684, 160731,
    160730, 34727, 34697, 34699, 34707, 34720, 160766, 17893, 34750, 160784,
    34753, 34766, 34783, 160841, 34787, {f: 2, c: 34789}, 34794, 34835, 34856,
    34862, 34866, 34876, 17935, 34890, 34904, 161301, 161300, 34921, 161329,
    34927, 34976, 35004, 35008, 161427, 35025, 35027, 17985, 35073, 161550,
    35127, 161571, 35138, 35141, 35145, 161618, 35170, 35209, 35216, 35231,
    35248, 35255, 35288, 35307, 18081, 35315, 35325, 35327, 18095, 35345,
    35348, 162181, 35361, 35381, 35390, 35397, 35405, 35416, 35502, 35472,
    35511, 35543, 35580, 162436, 35594, 35589, 35597, 35612, 35629, 18188,
    35665, 35678, 35702, 35713, 35723, {f: 2, c: 35732}, 35897, 162739, 35901,
    162750, 162759, 35909, 35919, 35927, 35945, 35949, 163000, 35987, 35986,
    35993, 18276, 35995, 36054, 36053, 163232, 36081, 163344, 36105, 36110,
    36296, 36313, 36364, 18429, 36349, 36358, 163978, 36372, 36374,
    {f: 2, c: 36385}, 36391, 164027, 18454, 36406, 36409, 36436, 36450, 36461,
    36463, 36504, 36510, 36533, 36539, 164482, 18510, 164595, 36608, 36616,
    36651, 36672, 36682, 36696, 164876, 36772, 36788, 164949, 36801, 36806,
    64036, 36810, 36813, 36819, 36821, 36849, 36853, 36859, 36876, 36919,
    165227, 36931, 36957, {f: 2, c: 165320}, 36997, 37004, 37008, 37025, 18613,
    37040, 37046, 37059, 37064, 165591, 37084, 37087, 165626, 37110, 37106,
    37120, 37099, {f: 2, c: 37118}, 37124, 37126, 37144, 37150, 37175, 37177,
    {f: 2, c: 37190}, 37207, 37209, 37236, 37241, 37253, 37299, 37302,
    {f: 2, c: 37315}, 166217, 166214, 37356, 37377, {f: 2, c: 37398}, 166251,
    37442, 37450, 37462, 37473, 37477, 37480, 166280, {f: 2, c: 37500}, 37503,
    37513, 37517, 37527, 37529, 37535, 37547, {f: 2, c: 166330}, 37554,
    {f: 2, c: 37567}, 37574, 37582, 37605, 37649, 166430, 166441, 37623, 37673,
    166513, 166467, 37713, 37722, 37739, 37745, 37747, 37793, 166553, 166605,
    37768, 37771, 37775, 37790, 37877, 166628, 166621, 37873, 37831, 37852,
    37863, 37897, {f: 2, c: 37910}, 37883, 37938, 37947, 166849, 166895, 37997,
    37999, 38265, 38278, {f: 2, c: 38284}, 167184, 167281, 38344, 167419,
    167455, 38444, {f: 2, c: 38451}, 167478, 38460, 38497, 167561, 38530,
    167659, 38554, 167730, 18919, 38579, 38586, 38589, 18938, 167928, 38616,
    38618, 38621, 18948, 38676, 38691, 18985, 38710, 38721, 38727, 38743,
    38747, 38762, 168608, 168625, 38806, 38814, {f: 2, c: 38833}, 38846, 38860,
    38865, 38868, 38872, 38881, 38897, 38916, 38925, 38932, 38934, 19132,
    169104, {f: 2, c: 38962}, 38949, 38983, 39014, 39083, 39085, 39088, 169423,
    39095, {f: 2, c: 39099}, 39106, 39111, 39115, 39137, 39139, 39146,
    {f: 2, c: 39152}, 39155, 39176, 19259, 169712, {f: 2, c: 39190}, 169753,
    {f: 3, c: 39194}, 169808, 39217, {f: 3, c: 39226}, 39233, 39238, 39246,
    39264, 39331, 39334, 39357, 39359, 39363, 39380, 39385, 39390, 170182,
    39408, 39417, 39420, 39434, 39441, 39450, 39456, 39473, 39492, 39500,
    39512, 19394, 39599, 19402, 39607, 19410, 39609, 170610, 39622, 39632,
    39634, 39637, 39648, 39653, 39657, 39692, 39696, 39698, 39702, 39708,
    39723, 39741, 19488, 39755, 39779, 39781, {f: 2, c: 39787},
    {f: 2, c: 39798}, 39846, 39852, 171483, 39858, 39864, 39870, 39923, 39896,
    39901, 39914, 39919, 39918, 171541, 171658, 171593, 39958,
    {f: 3, c: 39960}, 39965, 39970, 39977, 171716, 39985, 39991, 40005, 40028,
    171753, {f: 2, c: 40009}, 171739, 40020, 40024, 40027, 40029, 40031,
    {f: 3, c: 40041}, {f: 2, c: 40045}, 40050, 40053, 40058, 40166, 40178,
    40203, [171982, 171991], 40209, {f: 2, c: 40215}, 172079, 19652, 172058,
    40242, 19665, 40266, 40287, 40290, 172281, 172162, 40307, {f: 2, c: 40310},
    40324, 40345, 40353, 40383, 40373, 40377, 40381, 40393, 40410, 40416,
    40419, 19719, 40458, 40450, 40461, 40476, 40571, 139800, 40576, 40581,
    40603, 172940, 40637, 173111, 40671, 40703, 40706, 19831, 40707, 40762,
    40765, 40774, 40787, 40789, 40792, 173553, 40797, 173570, 40809, 40813,
    40816, 173746, 11948, 13844, 14509, 15820, 16348, 17854, 17936, 19326,
    19512, 19681, 19980, {f: 2, c: 20003}, 20089, 20211, 20236, 20249, 20267,
    20270, 20273, 20356, 20382, 20407, 20484, 20492, 20556, 20575, 20578,
    20599, 20622, 20638, 20642, 20675, 20712, 20721, 20734, 20743,
    {f: 3, c: 20748}, 20787, 20792, 20852, 20868, 20920, 20922, 20936, 20943,
    20945, {f: 2, c: 20947}, 20952, 20959, 20997, 21030, 21032, 21035,
    {f: 2, c: 21041}, 21045, 21052, 21082, 21088, 21102, {f: 2, c: 21112},
    21130, 21132, 21217, 21225, 21233, 21251, 21265, 21279, 21293, 21298,
    21309, 21349, 21357, 21369, 21374, 21396, 21401, 21418, 21423, 21434,
    21441, {f: 2, c: 21444}, 21472, 21523, 21546, 21553, {f: 2, c: 21556},
    21580, 21671, 21674, 21681, 21691, 21710, 21738, 21756, 21765, 21768,
    21781, 21799, 21802, 21814, 21841, 21862, 21903, 21906, 21908, 21924,
    21938, 21955, 21958, 21971, 21979, 21996, 21998, 22001, 22006, 22008,
    22021, 22029, {f: 2, c: 22033}, 22060, 22069, 22073, 22093, 22100, 22149,
    22175, 22182, 22199, 22220, 22223, 22233, 22241, 22251, 22253, 22257,
    22279, 22284, {f: 2, c: 22298}, 22301, 22316, 22318, {f: 2, c: 22333},
    22367, 22379, 22381, 22394, 22403, 22423, 22446, 22485, 22503, 22541,
    22566, 22605, 22607, 22623, 22637, 22655, 22657, 22680, 22716, 22815,
    22819, 22873, 22905, 22935, 22959, 22963, 23007, 23025, 23032, 23218,
    23224, 23274, 23286, 23323, 23325, 23329, 23352, 23479, 23511, 23520,
    23583, 23594, 23596, 23606, 23641, 23644, 23661, 23773, 23809, 23860,
    23869, 23897, 23934, 23939, 24007, 24057, 24104, 24114, 24117, 24155,
    24168, 24170, 24183, 24192, 24203, 24243, 24253, 24273, {f: 2, c: 24276},
    24397, 24492, 24554, 24583, 24649, 24660, 24679, 24763, 24772, 24829,
    24842, 24854, 24874, 24886, 24926, 24932, 24955, 24957, 24959, 24989,
    25016, 25052, 25058, 25061, 25064, 25092, 25095, 25137, 25145, 25149,
    25210, 25232, 25256, 25306, 25332, 25366, 25386, 25398, 25414, 25419,
    25427, 25457, 25461, 25471, 25474, 25482, {f: 2, c: 25518}, 25578,
    {f: 2, c: 25592}, 25618, 25624, 25632, 25636, 25642, 25653, 25661, 25663,
    25682, 25695, 25716, 25744, {f: 2, c: 25752}, 25772, 25779, 25837, 25840,
    25883, 25887, 25902, 25929, 25952, 26002, 26005, 26036, 26046, 26056,
    26062, 26064, 26079, 26238, {f: 2, c: 26251}, 26291, 26304, 26319, 26405,
    26421, 26453, 26496, 26511, 26513, 26532, 26545, 26549, 26558, 26664,
    26758, 26859, 26869, 26903, 26931, 26936, 26971, 26981, 27048, 27051,
    27055, 27109, 27121, 27210, 27221, 27239, 27249, 27311, {f: 2, c: 27336},
    27395, 27451, 27455, {f: 2, c: 27517}, 27568, 27639, 27641, 27652, 27657,
    27661, 27692, 27722, 27730, 27732, 27769, 27820, 27828, 27858, 28001,
    28028, 28089, 28144, 28229, 28275, 28283, 28285, 28297, 28348,
    {f: 2, c: 28378}, 28454, 28457, 28464, 28551, 28573, 28590, 28599, 28685,
    28704, 28745, 28824, 28848, {f: 2, c: 28885}, 28997, 29106, 29172, 29207,
    29215, 29251, {f: 2, c: 29263}, 29274, 29280, 29288, 29303, 29316, 29385,
    29413, 29428, 29442, 29451, 29470, 29474, {f: 2, c: 29498}, 29517, 29528,
    29543, 29810, 29871, 29919, 29924, 29940, 29947, 29974, 29985, 30015,
    30046, 30105, 30116, 30145, 30148, 30156, 30167, 30172, 30177, 30191,
    30212, 30220, 30237, 30258, 30264, 30277, 30282, 30303, 30381, 30397,
    30425, 30443, 30448, 30457, 30464, 30478, 30498, 30504, 30511, 30521,
    30526, 30533, 30538, 30543, 30558, 30564, 30567, 30572, 30596,
    {f: 2, c: 30604}, 30614, 30631, 30639, 30647, 30654, 30665, 30673, 30681,
    30705, 30775, 30812, 30846, 30872, 30881, 30897, 30899, 30921, 30931,
    30988, 31007, {f: 2, c: 31015}, 31039, 31042, 31060, 31083, 31100, 31147,
    31172, 31210, 31234, 31244, 31280, 31290, 31300, 31360, 31366, 31380,
    31413, 31421, 31486, 31531, 31607, 31648, 31660, 31664, 31720, 31730,
    31736, 31740, 31742, 31753, 31784, 31791, 31810, {f: 2, c: 31826},
    {f: 3, c: 31835}, 31858, 31869, 31879, 31902, 31930, 31943, 31955, 31962,
    32060, 32077, 32130, 32133, 32141, 32145, 32158, 32179, 32185, 32208,
    32229, {f: 2, c: 32245}, 32303, 32310, 32324, 32367, 32376, 32385, 32573,
    32603, 32605, 32613, 32625, {f: 2, c: 32639}, 32651, 32674,
    {f: 3, c: 32765}, 32775, 32781, 32798, 32825, 32904, 32910, 32975, 32980,
    33005, 33008, 33015, 33018, 33022, 33027, 33047, 33072, 33111, 33135,
    33139, 33163, 33168, 33179, 33182, 33227, 33237, {f: 2, c: 33245}, 33249,
    33263, 33270, 33280, 33291, {f: 2, c: 33299}, 33306, 33338, 33348, 33389,
    33412, 33417, 33425, 33450, 33456, 33488, 33514, 33519, 33526, 33622,
    33656, 33784, 33788, 33880, 33939, 33969, 33981, 34043, 34118, 34134,
    34141, 34181, 34200, 34370, 34374, 34496, 34580, 34594, 34606, 34617,
    34653, 34683, 34700, 34702, {f: 2, c: 34711}, 34718, 34723, 34734, 34751,
    34761, 34778, 34840, 34843, 34861, 34874, 34885, 34891, 34894, 34901,
    34906, 34926, {f: 3, c: 34970}, 35021, 35040, 35055, {f: 2, c: 35086},
    35110, 35125, 35162, 35164, 35179, 35184, 35196, 35237, 35253, 35260,
    35285, 35401, 35415, 35431, 35454, 35462, 35478, 35510, 35529, 35537,
    35549, 35564, 35573, 35590, 35599, 35601, 35653, 35666, 35693, 35704,
    35708, 35710, 35717, 35743, 35915, 35923, 35963, 36026, 36037, 36041,
    36050, 36076, 36085, 36087, 36097, 36099, 36119, 36124, 36206, 36241,
    36255, 36267, 36274, 36309, 36327, {f: 2, c: 36337}, 36340, 36353, 36363,
    36390, 36401, {f: 2, c: 36416}, 36429, 36431, 36444, 36449, 36457, 36465,
    36469, 36471, 36489, 36496, 36501, 36506, 36519, 36521, 36525, 36584,
    36592, 36615, 36632, 36645, 36647, 36652, 36661, 36666, 36675, 36679,
    36689, 36693, {f: 3, c: 36768}, 36773, 36868, 36891, 36911, 36940, 36955,
    36976, 36980, 36985, 37003, 37016, 37024, 37042, 37053, 37065, 37104,
    37125, 37157, 37210, 37223, 37242, 37258, 37265, 37269, 37296, 37307,
    37309, 37314, 37317, 37376, 37385, 37411, 37494, 37518, 37551,
    {f: 2, c: 37563}, 37569, 37571, 37573, 37576, 37652, 37683, 37686, 37720,
    37759, 37762, 37770, 37819, 37836, 37862, 37881, 37890, {f: 2, c: 37901},
    37934, 37964, 38280, 38305, 38335, 38342, 38345, {f: 2, c: 38353}, 38368,
    38372, 38374, 38436, 38449, 38456, 38461, 38484, 38516, 38523, 38527,
    38529, 38531, 38537, 38550, 38574, 38659, 38683, {f: 2, c: 38689}, 38696,
    38705, 38759, 38774, 38781, 38783, 38809, 38815, 38828, 38841, 38861,
    38880, 38895, 38919, 38950, 38958, {f: 2, c: 39010}, 39092, 39109, 39170,
    39185, 39189, 39221, 39240, 39252, 39262, 39393, 39436, 39440, 39459,
    39489, 39505, {f: 2, c: 39613}, 39681, 39689, 39691, {f: 2, c: 39693},
    39705, 39733, 39752, 39765, 39784, 39808, 39814, 39824, 39837, 39856,
    39871, 39880, 39935, 39938, 39964, 39989, 40004, 40022, 40033, 40040,
    40240, 40253, 40298, 40315, 40421, 40425, 40435, 40570, {f: 3, c: 40578},
    40624, 40676, 40688, 40690, 40713, 40719, 40724, 40731, 40738, 40742,
    {f: 2, c: 40746}, 40756, 40794, 40815, 40862, 40869, 131317, 151044,
    151538, 163187, 194581, 194630, 194713, 194726, 194789, 195038, 13790,
    {s: 4}, 172722, 0, 0, 131416, {s: 4}, 132529, 0, 0, 132844, {s: 6}, 134488,
    {s: 21}, 154060, {s: 9}, 14756, 14776, 142914, 0, 0, 14940, 0, 0, 143339,
    0, 0, 162228, 0, 15044, 15051, {s: 5}, 14981, {s: 8}, 15347, 27384, {s: 5},
    15665, {s: 9}, 147531, 0, 15936, 14497, {s: 34}, 158878, {s: 12}, 18207,
    162876, {s: 4}, 18462, {s: 71}, 39709, 39724, 20482, 20958, 21255, 23532,
    63784, 26142, 63785, 28746, 64021, 21857, 27706, 31328, 156492, 34819,
    38315, 38741, 171581, 173594],
  'Adobe-Korea1': [{f: 95, c: 32}, 8361, 8208, 169, 0, 0, [12288, 12644],
    {f: 2, c: 12289}, 12539, 8229, [8230, 8943], 168, 12291, {f: 2, c: 8211},
    8214, 65340, 65374, {f: 2, c: 8216}, {f: 2, c: 8220}, {f: 2, c: 12308},
    {f: 10, c: 12296}, 177, 215, 247, 8800, {f: 2, c: 8804}, 8734, 8756, 176,
    {f: 2, c: 8242}, 8451, 8491, {f: 2, c: 65504}, 65509, 9794, 9792, 8736,
    8869, 8978, 8706, 8711, 8801, 8786, 167, 8251, 9734, 9733, 9675, 9679,
    9678, 9671, 9670, 9633, 9632, 9651, 9650, 9661, 9660, 8594,
    {f: 2, c: 8592}, {f: 2, c: 8595}, 12307, 171, 187, 8730, 8765, 8733, 8757,
    {f: 2, c: 8747}, 8712, 8715, {f: 2, c: 8838}, {f: 2, c: 8834}, 8746, 8745,
    {f: 2, c: 8743}, 65506, 8658, 8660, 8704, 8707, 180, 732, 711, 728, 733,
    730, 729, 184, 731, 161, 191, 8758, 8750, 8721, 8719, 164, 8457, 8240,
    9665, 9664, 9655, 9654, 9828, {f: 2, c: 9824}, 9829, 9831, 9827, 9673,
    9672, 9635, {f: 2, c: 9680}, 9618, {f: 2, c: 9636}, 9640, 9639, 9638, 9641,
    9832, 9743, 9742, 9756, 9758, 182, {f: 2, c: 8224}, 8597, 8599, 8601, 8598,
    8600, 9837, {f: 2, c: 9833}, 9836, 12927, 12828, 8470, 13255, 8482, 13250,
    13272, 8481, {f: 59, c: 65281}, 65510, {f: 33, c: 65341}, 65507,
    {f: 51, c: 12593}, {f: 42, c: 12645}, {f: 10, c: 8560}, {f: 10, c: 8544},
    {f: 17, c: 913}, {f: 7, c: 931}, {f: 17, c: 945}, {f: 7, c: 963}, 9472,
    9474, 9484, 9488, 9496, 9492, 9500, 9516, 9508, 9524, 9532, 9473, 9475,
    9487, 9491, 9499, 9495, 9507, 9523, 9515, 9531, 9547, 9504, 9519, 9512,
    9527, 9535, 9501, 9520, 9509, 9528, 9538, 9490, 9489, 9498, 9497, 9494,
    9493, 9486, 9485, {f: 2, c: 9502}, {f: 2, c: 9505}, {f: 2, c: 9510},
    {f: 2, c: 9513}, {f: 2, c: 9517}, {f: 2, c: 9521}, {f: 2, c: 9525},
    {f: 2, c: 9529}, {f: 2, c: 9533}, {f: 2, c: 9536}, {f: 8, c: 9539},
    {f: 3, c: 13205}, 8467, 13208, 13252, {f: 4, c: 13219}, {f: 10, c: 13209},
    13258, {f: 3, c: 13197}, 13263, {f: 2, c: 13192}, 13256, {f: 2, c: 13223},
    {f: 10, c: 13232}, {f: 5, c: 13184}, {f: 6, c: 13242}, {f: 5, c: 13200},
    8486, {f: 2, c: 13248}, {f: 3, c: 13194}, 13270, 13253, {f: 3, c: 13229},
    13275, {f: 4, c: 13225}, 13277, 13264, 13267, 13251, 13257, 13276, 13254,
    198, 208, 170, 294, 306, 319, 321, 216, 338, 186, 222, 358, 330,
    {f: 28, c: 12896}, {f: 26, c: 9424}, {f: 15, c: 9312}, 189,
    {f: 2, c: 8531}, 188, 190, {f: 4, c: 8539}, 230, 273, 240, 295, 305, 307,
    312, 320, 322, 248, 339, 223, 254, 359, 331, 329, {f: 28, c: 12800},
    {f: 26, c: 9372}, {f: 15, c: 9332}, 185, {f: 2, c: 178}, 8308, 8319,
    {f: 4, c: 8321}, {f: 83, c: 12353}, {f: 86, c: 12449}, {f: 6, c: 1040},
    1025, {f: 32, c: 1046}, 1105, {f: 26, c: 1078}, {f: 2, c: 44032}, 44036,
    {f: 4, c: 44039}, {f: 8, c: 44048}, {f: 5, c: 44057}, 44064, 44068,
    {f: 2, c: 44076}, {f: 3, c: 44079}, {f: 2, c: 44088}, 44092, 44096, 44107,
    44109, 44116, 44120, 44124, {f: 2, c: 44144}, 44148, {f: 2, c: 44151},
    44154, {f: 2, c: 44160}, {f: 4, c: 44163}, {f: 4, c: 44169}, 44176, 44180,
    {f: 2, c: 44188}, {f: 3, c: 44191}, {f: 3, c: 44200}, 44204,
    {f: 2, c: 44207}, {f: 2, c: 44216}, {f: 3, c: 44219}, 44225, 44228, 44232,
    44236, 44245, 44247, {f: 2, c: 44256}, 44260, {f: 2, c: 44263}, 44266,
    44268, {f: 3, c: 44271}, 44275, {f: 2, c: 44277}, {f: 2, c: 44284}, 44288,
    44292, 44294, {f: 2, c: 44300}, 44303, 44305, 44312, 44316, 44320, 44329,
    {f: 2, c: 44332}, {f: 2, c: 44340}, 44344, 44348, {f: 2, c: 44356}, 44359,
    44361, 44368, 44372, 44376, 44385, 44387, {f: 2, c: 44396}, 44400,
    {f: 4, c: 44403}, {f: 3, c: 44411}, 44415, {f: 2, c: 44417},
    {f: 2, c: 44424}, 44428, 44432, {f: 2, c: 44444}, 44452, 44471,
    {f: 2, c: 44480}, 44484, 44488, {f: 2, c: 44496}, 44499, 44508, 44512,
    44516, {f: 2, c: 44536}, 44540, {f: 3, c: 44543}, {f: 2, c: 44552}, 44555,
    44557, 44564, {f: 2, c: 44592}, 44596, {f: 2, c: 44599}, 44602,
    {f: 2, c: 44608}, 44611, {f: 2, c: 44613}, 44618, {f: 3, c: 44620}, 44624,
    44628, 44630, {f: 2, c: 44636}, {f: 3, c: 44639}, 44645, {f: 2, c: 44648},
    44652, 44656, {f: 2, c: 44664}, {f: 3, c: 44667}, {f: 2, c: 44676}, 44684,
    {f: 3, c: 44732}, 44736, 44740, {f: 2, c: 44748}, {f: 3, c: 44751},
    {f: 2, c: 44760}, 44764, 44776, 44779, 44781, 44788, 44792, 44796,
    {f: 2, c: 44807}, 44813, 44816, {f: 2, c: 44844}, 44848, 44850, 44852,
    {f: 2, c: 44860}, 44863, {f: 3, c: 44865}, {f: 2, c: 44872}, 44880,
    {f: 2, c: 44892}, {f: 2, c: 44900}, 44921, 44928, 44932, 44936,
    {f: 2, c: 44944}, 44949, 44956, {f: 2, c: 44984}, 44988, 44992,
    {f: 3, c: 44999}, 45003, {f: 2, c: 45005}, 45012, 45020, {f: 2, c: 45032},
    {f: 2, c: 45040}, 45044, 45048, {f: 2, c: 45056}, 45060, 45068, 45072,
    45076, {f: 2, c: 45084}, 45096, {f: 2, c: 45124}, 45128, 45130, 45132,
    45134, {f: 3, c: 45139}, 45143, 45145, 45149, {f: 2, c: 45180}, 45184,
    45188, {f: 2, c: 45196}, 45199, 45201, {f: 3, c: 45208}, 45212,
    {f: 4, c: 45215}, {f: 2, c: 45224}, {f: 5, c: 45227}, 45233,
    {f: 3, c: 45235}, 45240, 45244, {f: 2, c: 45252}, {f: 3, c: 45255},
    {f: 2, c: 45264}, 45268, 45272, 45280, 45285, {f: 2, c: 45320},
    {f: 2, c: 45323}, 45328, {f: 2, c: 45330}, {f: 2, c: 45336},
    {f: 3, c: 45339}, {f: 3, c: 45347}, 45352, 45356, {f: 2, c: 45364},
    {f: 3, c: 45367}, {f: 2, c: 45376}, 45380, 45384, {f: 2, c: 45392},
    {f: 2, c: 45396}, 45400, 45404, 45408, {f: 2, c: 45432}, 45436, 45440,
    45442, {f: 2, c: 45448}, 45451, 45453, {f: 3, c: 45458}, 45464, 45468,
    45480, 45516, 45520, 45524, {f: 2, c: 45532}, 45535, {f: 2, c: 45544},
    45548, 45552, 45561, 45563, 45565, {f: 2, c: 45572}, 45576,
    {f: 2, c: 45579}, {f: 2, c: 45588}, 45591, 45593, 45600, 45620, 45628,
    45656, 45660, 45664, {f: 2, c: 45672}, {f: 2, c: 45684}, 45692,
    {f: 2, c: 45700}, 45705, {f: 2, c: 45712}, 45716, {f: 3, c: 45720},
    {f: 2, c: 45728}, 45731, {f: 2, c: 45733}, 45738, 45740, 45744, 45748,
    {f: 2, c: 45768}, 45772, 45776, 45778, {f: 2, c: 45784}, 45787, 45789,
    45794, {f: 3, c: 45796}, 45800, {f: 5, c: 45803}, {f: 3, c: 45811},
    {f: 5, c: 45815}, {f: 3, c: 45823}, 45828, 45832, {f: 2, c: 45840},
    {f: 3, c: 45843}, 45852, {f: 3, c: 45908}, 45912, {f: 2, c: 45915},
    {f: 2, c: 45918}, {f: 2, c: 45924}, 45927, 45929, 45931, 45934,
    {f: 2, c: 45936}, 45940, 45944, {f: 2, c: 45952}, {f: 3, c: 45955}, 45964,
    45968, 45972, {f: 2, c: 45984}, 45992, 45996, {f: 2, c: 46020}, 46024,
    {f: 2, c: 46027}, 46030, 46032, {f: 2, c: 46036}, 46039, 46041, 46043,
    46045, 46048, 46052, 46056, 46076, 46096, 46104, 46108, 46112,
    {f: 2, c: 46120}, 46123, 46132, {f: 2, c: 46160}, 46164, 46168,
    {f: 2, c: 46176}, 46179, 46181, 46188, 46208, 46216, 46237, 46244, 46248,
    46252, 46261, 46263, 46265, 46272, 46276, 46280, 46288, 46293,
    {f: 2, c: 46300}, 46304, {f: 2, c: 46307}, 46310, {f: 2, c: 46316}, 46319,
    46321, 46328, {f: 2, c: 46356}, 46360, {f: 2, c: 46363}, {f: 2, c: 46372},
    {f: 4, c: 46375}, {f: 2, c: 46384}, 46388, 46392, {f: 2, c: 46400},
    {f: 3, c: 46403}, {f: 3, c: 46411}, 46416, 46420, {f: 2, c: 46428},
    {f: 3, c: 46431}, {f: 2, c: 46496}, 46500, 46504, {f: 2, c: 46506},
    {f: 2, c: 46512}, {f: 3, c: 46515}, {f: 3, c: 46523}, 46528, 46532,
    {f: 2, c: 46540}, {f: 3, c: 46543}, 46552, 46572, {f: 2, c: 46608}, 46612,
    46616, 46629, 46636, 46644, 46664, 46692, 46696, {f: 2, c: 46748}, 46752,
    46756, {f: 2, c: 46763}, 46769, 46804, 46832, 46836, 46840,
    {f: 2, c: 46848}, 46853, {f: 2, c: 46888}, 46892, {f: 2, c: 46895},
    {f: 2, c: 46904}, 46907, 46916, 46920, 46924, {f: 2, c: 46932}, 46944,
    46948, 46952, {f: 2, c: 46960}, 46963, 46965, {f: 2, c: 46972}, 46976,
    46980, {f: 2, c: 46988}, {f: 4, c: 46991}, {f: 4, c: 46998}, 47004, 47008,
    {f: 2, c: 47016}, {f: 3, c: 47019}, {f: 2, c: 47028}, 47032, 47047, 47049,
    {f: 2, c: 47084}, 47088, 47092, {f: 2, c: 47100}, {f: 3, c: 47103},
    {f: 3, c: 47111}, 47116, 47120, {f: 2, c: 47128}, 47131, 47133,
    {f: 2, c: 47140}, 47144, 47148, {f: 2, c: 47156}, {f: 3, c: 47159}, 47168,
    47172, 47185, 47187, {f: 2, c: 47196}, 47200, 47204, {f: 2, c: 47212},
    47215, 47217, 47224, 47228, 47245, 47272, 47280, 47284, 47288,
    {f: 2, c: 47296}, 47299, 47301, 47308, 47312, 47316, 47325, 47327, 47329,
    {f: 2, c: 47336}, 47340, 47344, {f: 2, c: 47352}, 47355, 47357, 47364,
    47384, 47392, {f: 2, c: 47420}, 47424, 47428, 47436, 47439, 47441,
    {f: 2, c: 47448}, 47452, 47456, {f: 2, c: 47464}, 47467, 47469,
    {f: 2, c: 47476}, 47480, 47484, {f: 2, c: 47492}, 47495, {f: 2, c: 47497},
    {f: 2, c: 47501}, {f: 2, c: 47532}, 47536, 47540, {f: 2, c: 47548}, 47551,
    47553, {f: 2, c: 47560}, 47564, {f: 5, c: 47566}, {f: 2, c: 47576}, 47579,
    {f: 2, c: 47581}, 47585, {f: 3, c: 47587}, 47592, 47596, {f: 2, c: 47604},
    {f: 4, c: 47607}, {f: 2, c: 47616}, 47624, 47637, {f: 2, c: 47672}, 47676,
    47680, 47682, {f: 2, c: 47688}, 47691, {f: 2, c: 47693}, {f: 3, c: 47699},
    47704, 47708, {f: 2, c: 47716}, {f: 3, c: 47719}, {f: 2, c: 47728}, 47732,
    47736, {f: 3, c: 47747}, 47751, 47756, {f: 2, c: 47784}, {f: 2, c: 47787},
    47792, 47794, {f: 2, c: 47800}, 47803, 47805, 47812, 47816,
    {f: 2, c: 47832}, 47868, 47872, 47876, 47885, 47887, 47889, 47896, 47900,
    47904, 47913, 47915, {f: 3, c: 47924}, 47928, {f: 4, c: 47931},
    {f: 2, c: 47940}, 47943, 47945, 47949, {f: 2, c: 47951}, 47956, 47960,
    47969, 47971, 47980, 48008, 48012, 48016, 48036, 48040, 48044, 48052,
    48055, 48064, 48068, 48072, 48080, 48083, {f: 2, c: 48120}, 48124,
    {f: 2, c: 48127}, 48130, {f: 2, c: 48136}, {f: 3, c: 48139}, 48143, 48145,
    {f: 5, c: 48148}, {f: 5, c: 48155}, {f: 2, c: 48164}, 48167, 48169, 48173,
    {f: 2, c: 48176}, 48180, 48184, {f: 2, c: 48192}, {f: 3, c: 48195}, 48201,
    {f: 2, c: 48204}, 48208, 48221, {f: 2, c: 48260}, 48264, {f: 2, c: 48267},
    48270, {f: 2, c: 48276}, 48279, {f: 2, c: 48281}, {f: 2, c: 48288}, 48292,
    {f: 2, c: 48295}, {f: 2, c: 48304}, {f: 3, c: 48307}, {f: 2, c: 48316},
    48320, 48324, 48333, {f: 3, c: 48335}, 48341, 48344, 48348,
    {f: 3, c: 48372}, 48376, 48380, {f: 2, c: 48388}, 48391, 48393, 48400,
    48404, 48420, 48428, 48448, {f: 2, c: 48456}, 48460, 48464,
    {f: 2, c: 48472}, 48484, 48488, {f: 2, c: 48512}, 48516, {f: 4, c: 48519},
    {f: 2, c: 48528}, 48531, 48533, {f: 2, c: 48537}, 48540, 48548, 48560,
    48568, {f: 2, c: 48596}, 48600, 48604, 48617, 48624, 48628, 48632, 48640,
    48643, 48645, {f: 2, c: 48652}, 48656, 48660, {f: 2, c: 48668}, 48671,
    {f: 2, c: 48708}, 48712, 48716, 48718, {f: 2, c: 48724}, 48727,
    {f: 3, c: 48729}, {f: 2, c: 48736}, 48740, 48744, 48746, {f: 2, c: 48752},
    {f: 3, c: 48755}, {f: 3, c: 48763}, 48768, 48772, {f: 2, c: 48780},
    {f: 3, c: 48783}, {f: 2, c: 48792}, 48808, {f: 2, c: 48848}, 48852,
    {f: 2, c: 48855}, 48864, {f: 3, c: 48867}, 48876, 48897, {f: 2, c: 48904},
    {f: 2, c: 48920}, {f: 3, c: 48923}, {f: 2, c: 48960}, 48964, 48968,
    {f: 2, c: 48976}, 48981, 49044, 49072, 49093, {f: 2, c: 49100}, 49104,
    49108, 49116, 49119, 49121, 49212, 49233, 49240, 49244, 49248,
    {f: 2, c: 49256}, {f: 2, c: 49296}, 49300, 49304, {f: 2, c: 49312}, 49315,
    49317, {f: 2, c: 49324}, {f: 2, c: 49327}, {f: 4, c: 49331},
    {f: 2, c: 49340}, {f: 3, c: 49343}, 49349, {f: 2, c: 49352}, 49356, 49360,
    {f: 2, c: 49368}, {f: 3, c: 49371}, {f: 2, c: 49380}, 49384, 49388,
    {f: 2, c: 49396}, 49399, 49401, 49408, 49412, 49416, 49424, 49429,
    {f: 5, c: 49436}, {f: 2, c: 49443}, {f: 2, c: 49446}, {f: 2, c: 49452},
    {f: 3, c: 49455}, 49462, {f: 2, c: 49464}, 49468, 49472, {f: 2, c: 49480},
    {f: 3, c: 49483}, {f: 2, c: 49492}, 49496, 49500, {f: 2, c: 49508},
    {f: 3, c: 49511}, 49520, 49524, 49528, 49541, {f: 3, c: 49548}, 49552,
    49556, 49558, {f: 2, c: 49564}, 49567, 49569, 49573, {f: 2, c: 49576},
    49580, 49584, 49597, 49604, 49608, 49612, 49620, {f: 2, c: 49623}, 49632,
    49636, 49640, {f: 2, c: 49648}, 49651, {f: 2, c: 49660}, 49664, 49668,
    {f: 2, c: 49676}, 49679, 49681, {f: 2, c: 49688}, 49692, {f: 2, c: 49695},
    {f: 2, c: 49704}, 49707, 49709, 49711, {f: 2, c: 49713}, 49716, 49736,
    {f: 2, c: 49744}, 49748, 49752, 49760, 49765, {f: 2, c: 49772}, 49776,
    49780, {f: 2, c: 49788}, 49791, 49793, {f: 2, c: 49800}, 49808, 49816,
    49819, 49821, {f: 2, c: 49828}, 49832, {f: 2, c: 49836}, {f: 2, c: 49844},
    49847, 49849, {f: 2, c: 49884}, 49888, {f: 2, c: 49891}, {f: 3, c: 49899},
    49903, 49905, 49910, {f: 2, c: 49912}, {f: 2, c: 49915}, 49920,
    {f: 2, c: 49928}, {f: 2, c: 49932}, {f: 3, c: 49939}, 49944, 49948,
    {f: 2, c: 49956}, {f: 2, c: 49960}, 49989, {f: 2, c: 50024}, 50028, 50032,
    50034, {f: 2, c: 50040}, {f: 2, c: 50044}, 50052, 50056, 50060, 50112,
    {f: 2, c: 50136}, 50140, {f: 2, c: 50143}, 50146, {f: 2, c: 50152}, 50157,
    {f: 2, c: 50164}, 50168, 50184, 50192, 50212, 50220, 50224, 50228,
    {f: 2, c: 50236}, 50248, {f: 2, c: 50276}, 50280, 50284, {f: 2, c: 50292},
    50297, 50304, 50324, 50332, 50360, 50364, 50409, {f: 2, c: 50416}, 50420,
    50424, 50426, {f: 3, c: 50431}, 50444, 50448, 50452, 50460,
    {f: 2, c: 50472}, 50476, 50480, {f: 2, c: 50488}, 50491, 50493,
    {f: 2, c: 50500}, {f: 3, c: 50504}, {f: 3, c: 50508}, {f: 3, c: 50515},
    {f: 3, c: 50519}, {f: 2, c: 50525}, {f: 2, c: 50528}, 50532, 50536,
    {f: 2, c: 50544}, {f: 3, c: 50547}, {f: 2, c: 50556}, 50560, 50564, 50567,
    {f: 2, c: 50572}, 50575, 50577, 50581, {f: 2, c: 50583}, 50588, 50592,
    50601, {f: 2, c: 50612}, {f: 2, c: 50616}, {f: 4, c: 50619},
    {f: 7, c: 50628}, 50636, 50638, {f: 2, c: 50640}, 50644, 50648,
    {f: 2, c: 50656}, 50659, 50661, {f: 3, c: 50668}, 50672, 50676,
    {f: 2, c: 50678}, {f: 6, c: 50684}, {f: 4, c: 50693}, 50700, 50704,
    {f: 2, c: 50712}, {f: 2, c: 50715}, {f: 2, c: 50724}, 50728,
    {f: 3, c: 50732}, 50736, {f: 3, c: 50739}, 50743, 50745, 50747,
    {f: 2, c: 50752}, 50756, 50760, {f: 2, c: 50768}, {f: 3, c: 50771},
    {f: 2, c: 50780}, 50784, 50796, 50799, 50801, {f: 2, c: 50808}, 50812,
    50816, {f: 2, c: 50824}, 50827, 50829, {f: 2, c: 50836}, 50840, 50844,
    {f: 2, c: 50852}, 50855, 50857, {f: 2, c: 50864}, 50868, {f: 3, c: 50872},
    {f: 2, c: 50880}, 50883, 50885, {f: 2, c: 50892}, 50896, 50900,
    {f: 2, c: 50908}, {f: 2, c: 50912}, {f: 2, c: 50920}, 50924, 50928,
    {f: 2, c: 50936}, 50941, {f: 2, c: 50948}, 50952, 50956, {f: 2, c: 50964},
    50967, 50969, {f: 2, c: 50976}, 50980, 50984, {f: 2, c: 50992}, 50995,
    50997, 50999, {f: 2, c: 51004}, 51008, 51012, 51018, {f: 2, c: 51020},
    51023, {f: 8, c: 51025}, 51036, 51040, 51048, 51051, {f: 2, c: 51060},
    51064, {f: 3, c: 51068}, {f: 3, c: 51075}, {f: 4, c: 51079}, 51086,
    {f: 2, c: 51088}, 51092, {f: 3, c: 51094}, 51098, {f: 2, c: 51104},
    {f: 4, c: 51107}, {f: 2, c: 51116}, 51120, 51124, {f: 2, c: 51132},
    {f: 3, c: 51135}, {f: 2, c: 51144}, 51148, 51150, 51152, 51160, 51165,
    51172, 51176, 51180, {f: 2, c: 51200}, 51204, 51208, 51210,
    {f: 2, c: 51216}, 51219, {f: 2, c: 51221}, {f: 2, c: 51228}, 51232, 51236,
    {f: 2, c: 51244}, 51247, 51249, 51256, 51260, 51264, {f: 2, c: 51272},
    {f: 2, c: 51276}, 51284, {f: 2, c: 51312}, 51316, 51320, 51322,
    {f: 2, c: 51328}, 51331, {f: 3, c: 51333}, {f: 3, c: 51339}, 51348, 51357,
    51359, 51361, 51368, {f: 2, c: 51388}, 51396, 51400, 51404,
    {f: 2, c: 51412}, 51415, 51417, {f: 2, c: 51424}, 51428, 51445,
    {f: 2, c: 51452}, 51456, {f: 3, c: 51460}, {f: 2, c: 51468}, 51471, 51473,
    51480, 51500, 51508, {f: 2, c: 51536}, 51540, 51544, {f: 2, c: 51552},
    51555, 51564, 51568, 51572, 51580, {f: 2, c: 51592}, 51596, 51600,
    {f: 2, c: 51608}, 51611, 51613, {f: 2, c: 51648}, 51652, {f: 2, c: 51655},
    51658, {f: 2, c: 51664}, 51667, {f: 2, c: 51669}, {f: 2, c: 51673},
    {f: 2, c: 51676}, 51680, 51682, 51684, 51687, {f: 2, c: 51692},
    {f: 3, c: 51695}, {f: 2, c: 51704}, 51708, 51712, {f: 2, c: 51720},
    {f: 3, c: 51723}, 51732, 51736, 51753, {f: 2, c: 51788}, 51792, 51796,
    {f: 2, c: 51804}, {f: 3, c: 51807}, 51816, 51837, 51844, 51864,
    {f: 2, c: 51900}, 51904, 51908, {f: 2, c: 51916}, 51919, 51921, 51923,
    {f: 2, c: 51928}, 51936, 51948, 51956, 51976, 51984, 51988, 51992,
    {f: 2, c: 52000}, 52033, {f: 2, c: 52040}, 52044, 52048, {f: 2, c: 52056},
    52061, 52068, {f: 2, c: 52088}, 52124, 52152, 52180, 52196, 52199, 52201,
    {f: 2, c: 52236}, 52240, 52244, {f: 2, c: 52252}, {f: 2, c: 52257},
    {f: 3, c: 52263}, 52268, 52270, 52272, {f: 2, c: 52280}, {f: 4, c: 52283},
    {f: 2, c: 52292}, 52296, 52300, {f: 2, c: 52308}, {f: 3, c: 52311}, 52320,
    52324, 52326, 52328, 52336, 52341, {f: 2, c: 52376}, 52380, 52384,
    {f: 2, c: 52392}, {f: 3, c: 52395}, {f: 2, c: 52404}, 52408, 52412,
    {f: 2, c: 52420}, 52423, 52425, 52432, 52436, 52452, 52460, 52464, 52481,
    {f: 2, c: 52488}, 52492, 52496, {f: 2, c: 52504}, 52507, 52509, 52516,
    52520, 52524, 52537, 52572, 52576, 52580, {f: 2, c: 52588}, 52591, 52593,
    52600, 52616, {f: 2, c: 52628}, 52632, 52636, {f: 2, c: 52644}, 52647,
    52649, 52656, 52676, 52684, 52688, 52712, 52716, 52720, {f: 2, c: 52728},
    52731, 52733, 52740, 52744, 52748, 52756, 52761, {f: 2, c: 52768}, 52772,
    52776, {f: 2, c: 52784}, 52787, 52789, {f: 2, c: 52824}, 52828,
    {f: 3, c: 52831}, {f: 2, c: 52840}, 52843, 52845, {f: 2, c: 52852}, 52856,
    52860, {f: 2, c: 52868}, 52871, 52873, {f: 2, c: 52880}, 52884, 52888,
    {f: 2, c: 52896}, {f: 3, c: 52899}, {f: 2, c: 52908}, 52929,
    {f: 2, c: 52964}, 52968, {f: 2, c: 52971}, {f: 2, c: 52980},
    {f: 3, c: 52983}, {f: 2, c: 52992}, 52996, 53000, {f: 2, c: 53008}, 53011,
    53013, 53020, 53024, 53028, {f: 2, c: 53036}, {f: 3, c: 53039}, 53048,
    {f: 2, c: 53076}, 53080, 53084, {f: 2, c: 53092}, 53095, 53097,
    {f: 2, c: 53104}, 53108, 53112, 53120, 53125, 53132, 53153, 53160, 53168,
    53188, {f: 2, c: 53216}, 53220, 53224, {f: 2, c: 53232}, 53235, 53237,
    53244, 53248, 53252, 53265, 53272, 53293, {f: 2, c: 53300}, 53304, 53308,
    {f: 2, c: 53316}, 53319, 53321, 53328, 53332, 53336, 53344,
    {f: 2, c: 53356}, 53360, 53364, {f: 2, c: 53372}, 53377, {f: 2, c: 53412},
    53416, 53420, {f: 2, c: 53428}, 53431, 53433, {f: 2, c: 53440}, 53444,
    {f: 2, c: 53448}, {f: 2, c: 53456}, {f: 3, c: 53459}, {f: 2, c: 53468},
    53472, 53476, {f: 2, c: 53484}, {f: 3, c: 53487}, 53496, 53517,
    {f: 2, c: 53552}, 53556, 53560, 53562, {f: 2, c: 53568}, {f: 3, c: 53571},
    {f: 2, c: 53580}, 53584, 53588, {f: 2, c: 53596}, 53599, 53601, 53608,
    53612, 53628, 53636, 53640, {f: 2, c: 53664}, 53668, 53672,
    {f: 2, c: 53680}, 53683, 53685, 53690, 53692, 53696, 53720, 53748, 53752,
    53767, 53769, 53776, {f: 2, c: 53804}, 53808, 53812, {f: 2, c: 53820},
    53823, 53825, 53832, 53852, 53860, {f: 2, c: 53888}, 53892, 53896,
    {f: 2, c: 53904}, 53909, 53916, 53920, 53924, 53932, 53937,
    {f: 2, c: 53944}, 53948, {f: 2, c: 53951}, 53954, {f: 2, c: 53960}, 53963,
    53972, 53976, 53980, {f: 2, c: 53988}, {f: 2, c: 54000}, 54004, 54008,
    {f: 2, c: 54016}, 54019, 54021, {f: 3, c: 54028}, 54032, 54036, 54038,
    {f: 2, c: 54044}, {f: 3, c: 54047}, 54053, {f: 2, c: 54056}, 54060, 54064,
    {f: 2, c: 54072}, {f: 3, c: 54075}, {f: 2, c: 54084}, {f: 2, c: 54140},
    54144, 54148, {f: 2, c: 54156}, {f: 3, c: 54159}, {f: 2, c: 54168}, 54172,
    54176, {f: 2, c: 54184}, 54187, 54189, 54196, 54200, 54204,
    {f: 2, c: 54212}, {f: 2, c: 54216}, 54224, 54232, 54241, 54243,
    {f: 2, c: 54252}, 54256, 54260, {f: 2, c: 54268}, 54271, 54273, 54280,
    54301, 54336, 54340, 54364, 54368, 54372, 54381, 54383, {f: 2, c: 54392},
    54396, {f: 2, c: 54399}, 54402, {f: 2, c: 54408}, 54411, 54413, 54420,
    54441, 54476, 54480, 54484, 54492, 54495, 54504, 54508, 54512, 54520,
    54523, 54525, 54532, 54536, 54540, {f: 2, c: 54548}, 54551,
    {f: 2, c: 54588}, 54592, 54596, {f: 2, c: 54604}, 54607, 54609,
    {f: 2, c: 54616}, 54620, 54624, 54629, {f: 2, c: 54632}, 54635, 54637,
    {f: 2, c: 54644}, 54648, 54652, {f: 2, c: 54660}, {f: 3, c: 54663}, 54672,
    54693, {f: 2, c: 54728}, 54732, 54736, 54738, {f: 2, c: 54744}, 54747,
    54749, {f: 2, c: 54756}, 54760, 54764, {f: 2, c: 54772}, 54775, 54777,
    {f: 2, c: 54784}, 54788, 54792, {f: 2, c: 54800}, {f: 3, c: 54803}, 54812,
    54816, 54820, 54829, {f: 2, c: 54840}, 54844, 54848, 54853,
    {f: 2, c: 54856}, 54859, 54861, 54865, {f: 2, c: 54868}, 54872, 54876,
    54887, 54889, {f: 2, c: 54896}, 54900, 54915, 54917, {f: 2, c: 54924},
    54928, 54932, 54941, 54943, 54945, 54952, 54956, 54960, 54969, 54971,
    {f: 2, c: 54980}, 54984, 54988, 54993, 54996, 54999, 55001, 55008, 55012,
    55016, 55024, 55029, {f: 2, c: 55036}, 55040, 55044, 55057,
    {f: 2, c: 55064}, 55068, 55072, {f: 2, c: 55080}, 55083, 55085,
    {f: 2, c: 55092}, 55096, 55100, 55108, 55111, 55113, {f: 2, c: 55120},
    55124, {f: 4, c: 55126}, {f: 2, c: 55136}, 55139, 55141, 55145, 55148,
    55152, 55156, {f: 2, c: 55164}, 55169, {f: 2, c: 55176}, 55180, 55184,
    {f: 2, c: 55192}, 55195, 55197, 20285, 20339, 20551, 20729, 21152, 21487,
    21621, 21733, 22025, 23233, 23478, 26247, {f: 2, c: 26550}, 26607, 27468,
    29634, 30146, 31292, 33499, 33540, 34903, 34952, 35382, [36040, 63747],
    36303, 36603, 36838, 39381, 21051, 21364, 21508, 24682, 24932, 27580,
    29647, 33050, 35258, [12179, 35282], 38307, 20355, 21002, 22718, 22904,
    23014, [12082, 24178], 24185, 25031, 25536, 26438, 26604, 26751, 28567,
    30286, 30475, 30965, 31240, 31487, 31777, 32925, [12169, 33390], 33393,
    35563, 38291, 20075, 21917, 26359, 28212, 30883, 31469, 33883, 35088,
    34638, 38824, 21208, 22350, 22570, 23884, 24863, 25022, 25121, 25954,
    26577, 27204, 28187, [12130, 29976], 30131, 30435, 30640, 32058, 37039,
    {f: 2, c: 37969}, 40853, 21283, 23724, 30002, 32987, 37440, 38296, 21083,
    22536, 23004, 23713, 23831, 24247, 24378, 24394, 24951, 27743, 30074,
    30086, 31968, 32115, 32177, 32652, 33108, 33313, 34193, 35137, 35611,
    37628, [38477, 64009], 40007, 20171, 20215, 20491, 20977, 22607, 24887,
    24894, 24936, 25913, 27114, 28433, 30117, 30342, 30422, 31623, 33445,
    33995, 37799, 38283, 21888, 23458, 22353, 31923, 32697, 37301, 20520,
    21435, 23621, 24040, 25298, 25454, 25818, 25831, 28192, 28844, 31067,
    36317, 36382, 36989, 37445, 37624, 20094, 20214, 20581, [12081, 24062],
    24314, 24838, 26967, 33137, 34388, 36423, 37749, 39467, 20062, 20625,
    26480, 26688, 20745, 21133, 21138, 27298, 30652, 37392, 40660, 21163,
    24623, 36850, 20552, 25001, 25581, 25802, 26684, 27268, 28608, 33160,
    35233, 38548, 22533, 29309, [12125, 29356], 29956, 32121, 32365, 32937,
    [12178, 35211, 64010], 35700, 36963, 40273, 25225, 27770, 28500, 32080,
    32570, 35363, 20860, 24906, 31645, 35609, 37463, 37772, 20140, 20435,
    20510, 20670, 20742, 21185, 21197, 21375, 22384, 22659, 24218, 24465,
    24950, 25004, 25806, 25964, 26223, 26299, [26356, 63745], 26775, 28039,
    28805, 28913, 29855, 29861, 29898, 30169, 30828, 30956, 31455, 31478,
    32069, 32147, 32789, 32831, 33051, 33686, 35686, 36629, 36885, 37857,
    38915, 38968, 39514, 39912, 20418, 21843, 22586, [22865, 63753], 23395,
    23622, 24760, 25106, 26690, 26800, 26856, 28330, 30028, 30328, 30926,
    31293, 31995, 32363, 32380, 35336, 35489, 35903, 38542, 40388, 21476,
    21481, 21578, 21617, 22266, 22993, 23396, 23611, 24235, 25335, 25911,
    25925, 25970, 26272, 26543, 27073, 27837, 30204, 30352, 30590, 31295,
    32660, 32771, 32929, 33167, 33510, 33533, 33776, 34241, 34865, 34996,
    35493, 36764, 37678, 38599, 39015, [12220, 39640], [12238, 40723], 21741,
    26011, 26354, 26767, 31296, [12181, 35895], 40288, 22256, 22372, 23825,
    26118, 26801, 26829, 28414, 29736, 34974, 39908, 27752, [12219, 39592],
    20379, 20844, 20849, 21151, 23380, [12079, 24037], 24656, 24685, 25329,
    25511, 25915, 29657, 31354, 34467, 36002, 38799, [20018, 63749], 23521,
    [12093, 25096], 26524, [12128, 29916], 31185, 33747, 35463, 35506, 36328,
    36942, 37707, 38982, [24275, 64011], 27112, 34303, 37101, 20896, 23448,
    23532, 24931, 26874, 27454, 28748, 29743, 29912, 31649, 32592, 33733,
    35264, 36011, 38364, 39208, 21038, 24669, 25324, 36866, 20362, 20809,
    21281, 22745, 24291, 26336, 27960, 28826, 29378, 29654, 31568, 33009,
    37979, 21350, 25499, 32619, 20054, 20608, 22602, 22750, 24618, 24871,
    25296, 27088, 39745, 23439, 32024, 32945, 36703, 20132, 20689, 21676,
    21932, 23308, 23968, 24039, 25898, 25934, 26657, 27211, 29409, 30350,
    30703, 32094, 32761, 33184, 34126, 34527, 36611, 36686, 37066, 39171,
    39509, 39851, 19992, 20037, 20061, 20167, 20465, 20855, 21246, 21312,
    [12061, 21475], [21477, 63750], 21646, 22036, 22389, 22434, 23495, 23943,
    24272, 25084, 25304, 25937, 26552, 26601, 27083, 27472, 27590, 27628,
    27714, 28317, 28792, 29399, 29590, 29699, 30655, 30697, 31350, 32127,
    32777, [12165, 33276], 33285, 33290, 33503, 34914, 35635, 36092, 36544,
    36881, 37041, 37476, 37558, 39378, 39493, 40169, 40407,
    [12244, 40860, 63751, 63752], 22283, 23616, 33738, 38816, 38827, 40628,
    21531, 31384, 32676, 35033, 36557, 37089, 22528, 23624, 25496, 31391,
    23470, [12088, 24339], 31353, 31406, 33422, 36524, 20518, 21048, 21240,
    21367, 22280, 25331, 25458, 27402, 28099, 30519, 21413, 29527, 34152,
    36470, 38357, 26426, 27331, 28528, 35437, 36556, 39243, 26231, 27512,
    36020, [12225, 39740], 21483, 22317, 22862, 25542, 27131, 29674, 30789,
    31418, 31429, 31998, 33909, 35215, 36211, 36917, 38312, 21243, 22343,
    30023, 31584, 33740, 37406, 27224, 20811, 21067, 21127, 25119, 26840,
    26997, 38553, 20677, 21156, 21220, 25027, [12100, 26020], 26681, 27135,
    29822, 31563, 33465, 33771, 35250, 35641, 36817, 39241, 20170, 22935,
    25810, 26129, 27278, 29748, 31105, 31165, 33449, {f: 2, c: 34942}, 35167,
    37670, 20235, 21450, 24613, 25201, 27762, 32026, 32102, 20120, 20834,
    30684, 32943, 20225, 20238, 20854, 20864, 21980, 22120, 22331, 22522,
    22524, 22804, 22855, 22931, 23492, 23696, 23822, [12080, 24049], 24190,
    24524, 25216, 26071, 26083, {f: 2, c: 26398}, 26462, 26827, 26820, 27231,
    27450, 27683, 27773, 27778, 28103, 29592, 29734, 29738, 29826, 29859,
    30072, 30079, 30849, 30959, 31041, {f: 2, c: 31047}, 31098, 31637, 32000,
    32186, 32648, 32774, 32813, 32908, 35352, 35663, [35912, 63744], 36215,
    37665, 37668, 39138, 39249, {f: 2, c: 39438}, 39525, 40594, 32202, 20342,
    21513, 25326, 26708, [12198, 37329, 63754], 21931, 20794, 23068, 25062,
    [25295, 63835], 25343, 37027, [35582, 63837], 26262, 29014, 38627, 25423,
    25466, 21335, 26511, 26976, 28275, 30007, 32013, 34930, 22218, 23064,
    20035, 20839, [22856, 63756], 26608, 32784, [12069, 22899, 63873],
    [24180, 63886], [25754, 63889], [31178, 63893], [24565, 63907], 24684,
    25288, [25467, 63908], [23527, 63839, 63914], 23511, 21162, 22900, 24361,
    [24594, 63840], 29785, 39377, 28611, 33215, 36786, 24817, 33126,
    [23615, 63933], 23273, 35365, [26491, 63944], [32016, 63951], 33021, 23612,
    [27877, 63971], [21311, 63979], [28346, 63980], 22810, [33590, 63998],
    [20025, 63838], 20150, 20294, 21934, 22296, 22727, 24406, 26039, 26086,
    27264, 27573, 28237, 30701, 31471, 31774, 32222, 34507, 34962, 37170,
    37723, 25787, 28606, 29562, 30136, 36948, 21846, 22349, 25018, 25812,
    26311, 28129, 28251, 28525, 28601, 30192, 32835, 33213, 34113, 35203,
    35527, 35674, 37663, 27795, 30035, 31572, 36367, 36957, 21776, 22530,
    22616, 24162, 25095, 25758, 26848, 30070, [31958, 64003], 34739, 40680,
    20195, 22408, 22382, [12068, 22823], 23565, 23729, 24118, 24453, 25140,
    25825, 29619, 33274, 34955, 36024, 38538, 40667, [23429, 64004], 24503,
    24755, 20498, [12049, 20992], 21040, 22294, 22581, 22615, 23566, 23648,
    23798, 23947, [24230, 64001], 24466, 24764, 25361, 25481, 25623, 26691,
    26873, 27330, 28120, 28193, 28372, 28644, 29182, 30428, 30585, 31153,
    31291, 33796, 35241, 36077, 36339, 36424, 36867, 36884, 36947, 37117,
    37709, 38518, 38876, 27602, 28678, 29272, 29346, 29544, 30563, 31167,
    31716, 32411, [35712, 63834], 22697, 24775, 25958, 26109, 26302, 27788,
    28958, 29129, 35930, 38931, 20077, 31361, 20189, 20908, 20941, 21205,
    21516, 24999, 26481, 26704, 26847, [27934, 64005], 28540, 30140, 30643,
    31461, 33012, 33891, 37509, 20828, [12099, 26007], 26460, 26515, 30168,
    31431, 33651, [12182, 35910], 36887, 38957, 23663, 33216, 33434, 36929,
    36975, 37389, 24471, 23965, 27225, 29128, 30331, 31561, 34276, 35588,
    37159, 39472, [21895, 63755], [25078, 63757], [30313, 63758],
    [32645, 63759], [34367, 63760], [34746, 63761], [35064, 63762],
    [37007, 63763], [27931, 63765], [28889, 63766], [29662, 63767], 32097,
    [33853, 63768], [37226, 63769], [39409, 63770], [20098, 63771],
    [21365, 63772], [27396, 63773], 27410, 28734, [29211, 63774],
    [34349, 63775], [40478, 63776], 21068, 36771, [23888, 63777], 25829, 25900,
    27414, [28651, 63778], 31811, 32412, [34253, 63779], [35172, 63780], 35261,
    [25289, 63781], [33240, 63782], [34847, 63783], [24266, 63784],
    [26391, 63785], [28010, 63786], [29436, 63787], 29701, 29807, 34690,
    [37086, 63788], [20358, 63789], 23821, 24480, 33802, [20919, 63790],
    [25504, 63861], [30053, 63862], [20142, 63863], 20486, [20841, 63864],
    [20937, 63865], [26753, 63866], 27153, 31918, 31921, [31975, 63867],
    [33391, 63868], [35538, 63869], 36635, [37327, 63870], 20406, 20791,
    [21237, 63871], [21570, 63872], [24300, 63874], 24942, 25150,
    [26053, 63875], 27354, [28670, 63876], [31018, 63877], 34268, 34851,
    [38317, 63878], 39522, [39530, 63879], [40599, 63880], [40654, 63881],
    [12050, 21147, 63882], [26310, 63883], [27511, 63884], 28701, 31019,
    [36706, 63885], 38722, [24976, 63887], [25088, 63888], 25891,
    [28451, 63890], [29001, 63891], [29833, 63892], [32244, 63894],
    [32879, 63895], [34030, 63897], [36646, 63896], [36899, 63898],
    [37706, 63899], 20925, [21015, 63900], [21155, 63901], 27916,
    [28872, 63903], [35010, 63904], [24265, 63906], 25986, [27566, 63909],
    28610, [31806, 63910], [29557, 63911], [20196, 63912], 20278,
    [22265, 63913], 23738, [23994, 63915], [24604, 63916], [29618, 63917],
    31533, [32666, 63919], 32718, [32838, 63920], 36894, [37428, 63921],
    [38646, 63922], [38728, 63923], [38936, 63924], 40801, [20363, 63925],
    28583, [31150, 63926], [37300, 63927], [38583, 63928], [21214, 63791],
    25736, [25796, 63792], [27347, 63793], 28510, 28696, [29200, 63794],
    [30439, 63795], [12156, 32769, 63796], [34310, 63797], [34396, 63798],
    [36335, 63799], 36613, [38706, 63800], [39791, 63801], [40442, 63802],
    [12228, 40565], [30860, 63803], [31103, 63804], [32160, 63805],
    [33737, 63806], [37636, 63807], [12229, 40575, 63808], 40595,
    [35542, 63809], [22751, 63810], [24324, 63811], 26407, 28711, 29903,
    [31840, 63812], [32894, 63813], 20769, 28712, [29282, 63814],
    [30922, 63815], [36034, 63816], 36058, 36084, [38647, 63817],
    [20102, 63930], [20698, 63931], [23534, 63932], 24278, [26009, 63934],
    [29134, 63936], [30274, 63937], 30637, 32842, [34044, 63938],
    [36988, 63939], 39719, [12243, 40845, 63940], [22744, 63818], 23105,
    [23650, 63819], [27155, 63820], [28122, 63821], [28431, 63822], 30267,
    [32047, 63823], [32311, 63824], 34078, 35128, 37860, [38475, 63825],
    [21129, 63943], 26066, [26611, 63945], 27060, [27969, 63946],
    [28316, 63947], 28687, [29705, 63948], 29792, [30041, 63949], 30244,
    [30827, 63950], 35628, [39006, 63952], [20845, 63953], [25134, 63954],
    [38520, 63955], 20374, [20523, 63956], [23833, 63957], [28138, 63958],
    32184, [36650, 63959], [24459, 63960], [24900, 63961], [26647, 63962],
    [38534, 63964], [21202, 63826], [32907, 63827], [20956, 63828],
    [20940, 63829], 26974, [31260, 63830], [32190, 63831], [33777, 63832],
    [38517, 63833], 20442, [21033, 63965], 21400, [21519, 63966], 21774,
    [23653, 63967], 24743, [26446, 63969], [26792, 63970], 28012, 29313, 29432,
    [29702, 63972], 29827, [30178, 63973], 31852, [32633, 63974], 32696, 33673,
    [35023, 63975], [35041, 63976], [12197, 37324, 63977], 37328,
    [38626, 63978], 39881, [21533, 63981], 28542, [29136, 63982],
    [29848, 63983], [34298, 63984], 36522, [38563, 63985], [40023, 63986],
    [40607, 63987], [26519, 63988], [28107, 63989], 29747, [33256, 63990],
    38678, 30764, [12148, 31435, 63991], [31520, 63992], [31890, 63993], 25705,
    29802, 30194, 30908, 30952, [12218, 39340], 39764, [12231, 40635], 23518,
    24149, 28448, 33180, 33707, 37000, 19975, 21325, 23081, 24018, 24398,
    24930, 25405, 26217, 26364, 28415, 28459, 28771, 30622, 33836, 34067,
    34875, 36627, 39237, 39995, 21788, 25273, 26411, 27819, 33545, 35178,
    38778, 20129, 22916, {f: 2, c: 24536}, 26395, 32178, 32596, 33426, 33579,
    33725, 36638, 37017, 22475, 22969, 23186, 23504, 26151, 26522, 26757,
    27599, 29028, 32629, 36023, 36067, 36993, 39749, 33032, 35978, 38476,
    39488, [12230, 40613], 23391, 27667, 29467, 30450, 30431, 33804, 20906,
    35219, 20813, 20885, 21193, 26825, 27796, 30468, 30496, 32191, 32236,
    [12207, 38754], 40629, 28357, 34065, 20901, 21517, 21629, 26126, 26269,
    26919, 28319, [12139, 30399], 30609, 33559, 33986, 34719, 37225, 37528,
    40180, 34946, 20398, 20882, 21215, 22982, 24125, 24917, {f: 2, c: 25720},
    26286, 26576, 27169, 27597, [12113, 27611], 29279, 29281, 29761, 30520,
    [12141, 30683], 32791, 33468, 33541, 35584, 35624, 35980, [12106, 26408],
    27792, 29287, [12140, 30446], 30566, 31302, 40361, 27519, 27794, 22818,
    26406, 33945, 21359, 22675, 22937, 24287, 25551, 26164, 26483, 28218,
    29483, 31447, 33495, 37672, 21209, 24043, 25006, 25035, 25098, 25287,
    25771, [12102, 26080], 26969, 27494, [12111, 27595], 28961, 29687, 30045,
    32326, 33310, 33538, 34154, 35491, 36031, 38695, 40289, 22696, 40664,
    20497, 21006, 21563, 21839, [12098, 25991], 27766, {f: 2, c: 32010}, 32862,
    34442, [12200, 38272], 38639, 21247, 27797, 29289, 21619, 23194, 23614,
    23883, 24396, 24494, 26410, 26806, 26979, 28220, 28228, 30473,
    [12150, 31859], 32654, 34183, 35598, 36855, 38753, 40692, 23735, 24758,
    24845, 25003, 25935, {f: 2, c: 26107}, 27665, 27887, 29599, 29641, 32225,
    38292, 23494, 34588, 35600, 21085, 21338, 25293, 25615, 25778, 26420,
    27192, 27850, 29632, 29854, 31636, 31893, 32283, 33162, 33334, 34180,
    36843, 38649, 39361, 20276, 21322, 21453, 21467, 25292, 25644, 25856,
    26001, 27075, 27886, 28504, 29677, 30036, 30242, 30436, 30460, 30928,
    [30971, 63844], 31020, 32070, 33324, 34784, 36820, 38930, 39151, 21187,
    25300, 25765, 28196, 28497, 30332, 36299, 37297, 37474, 39662, 39747,
    20515, 20621, 22346, 22952, 23592, 24135, 24439, 25151, 25918,
    [12101, 26041], 26049, 26121, 26507, 27036, 28354, 30917, 32033, 32938,
    33152, 33323, 33459, 33953, 34444, 35370, 35607, 37030, 38450, 40848,
    20493, 20467, 22521, 24472, 25308, 25490, 26479, 28227, 28953, 30403,
    32972, 32986, {f: 2, c: 35060}, 35097, 36064, 36649, 37197, 38506, 20271,
    20336, 24091, 26575, 26658, [12137, 30333], 30334, 39748, 24161, 27146,
    29033, 29140, 30058, 32321, 34115, 34281, 39132, 20240, 31567, 32624,
    38309, 20961, 24070, 26805, 27710, 27726, 27867, 29359, 31684, 33539,
    27861, 29754, 20731, 21128, 22721, 25816, 27287, 29863, 30294, 30887,
    34327, 38370, 38713, 21342, 24321, 35722, 36776, 36783, 37002, 21029,
    30629, 40009, 40712, 19993, 20482, 20853, 23643, 24183, 26142, 26170,
    26564, 26821, 28851, 29953, 30149, 31177, 31453, 36647, 39200, 39432,
    20445, 22561, 22577, 23542, 26222, 27493, 27921, 28282, 28541, 29668,
    29995, 33769, 35036, 35091, 35676, 36628, 20239, 20693, 21264,
    [12056, 21340], 23443, [24489, 63846], 26381, 31119, 33145, 33583, 34068,
    35079, 35206, 36665, [36667, 64007], 39333, 39954, 26412, 20086, 20472,
    22857, 23553, {f: 2, c: 23791}, 25447, 26834, 28925, 29090, 29739, 32299,
    34028, 34562, 36898, 37586, 40179, [19981, 63847], 20184, 20463, 20613,
    21078, 21103, 21542, 21648, 22496, 22827, 23142, 23386, 23413, 23500,
    24220, 25206, 25975, 26023, 28014, 28325, [12119, 29238], 31526, 31807,
    [12152, 32566], {f: 2, c: 33104}, 33178, 33344, 33433, 33705, 35331, 36000,
    36070, 36091, 36212, 36282, 37096, 37340, [12201, 38428], 38468, 39385,
    40167, [21271, 63843], 20998, 21545, 22132, 22707, 22868, 22894, 24575,
    24996, 25198, 26128, 27774, 28954, 30406, 31881, 31966, 32027, 33452,
    36033, 38640, 20315, 24343, 24447, 25282, 23849, 26379, 26842, 30844,
    32323, 40300, 19989, 20633, [12052, 21269], 21290, 21329, 22915, 23138,
    24199, 24754, 24970, 25161, 25209, 26000, 26503, 27047, [12112, 27604],
    {f: 3, c: 27606}, 27832, 29749, 30202, 30738, 30865, 31189, 31192, 31875,
    32203, 32737, 32933, 33086, 33218, 33778, 34586, 35048, 35513, 35692,
    36027, 37145, [12206, 38750], [12214, 39131], [12240, 40763], 22188, 23338,
    24428, 25996, 27315, 27567, 27996, 28657, 28693, 29277, 29613, 36007,
    36051, 38971, 24977, 27703, 32856, 39425, 20045, 20107, 20123, 20181,
    20282, 20284, 20351, 20447, 20735, 21490, 21496, 21766, 21987, 22235,
    [12064, 22763], 22882, 23057, 23531, 23546, 23556, 24051, 24107, 24473,
    24605, 25448, 26012, 26031, 26614, 26619, 26797, 27515, 27801, 27863,
    28195, 28681, 29509, 30722, 31038, 31040, 31072, 31169, 31721, 32023,
    32114, 32902, 33293, 33678, 34001, 34503, 35039, 35408, 35422, 35613,
    36060, 36198, 36781, 37034, 39164, 39391, 40605, 21066, 26388, 20632,
    21034, [12077, 23665], 25955, 27733, 29642, 29987, 30109, 31639, 33948,
    37240, 38704, 20087, 25746, [27578, 63856], 29022, 34217, 19977, 26441,
    26862, 28183, 33439, 34072, 34923, 25591, 28545, 37394, 39087, 19978,
    20663, 20687, 20767, 21830, 21930, 22039, 23360, 23577, 23776, 24120,
    24202, 24224, 24258, 24819, 26705, 27233, 28248, 29245, 29248,
    [29376, 63994], 30456, 31077, 31665, 32724, 35059, 35316, 35443, 35937,
    36062, 38684, [22622, 63852], 29885, 36093, 21959, 31329, [32034, 63850],
    [12170, 33394], 29298, [12131, 29983], 29989, 31513, 22661, 22779, 23996,
    24207, 24246, 24464, 24661, 25234, 25471, 25933, 26257, 26329, 26360,
    26646, 26866, 29312, 29790, 31598, 32110, 32214, 32626, 32997, 33298,
    34223, 35199, 35475, 36893, 37604, [12233, 40653], [12239, 40736],
    [12067, 22805], 22893, 24109, 24796, 26132, 26227, 26512, 27728, 28101,
    28511, [12143, 30707], 30889, 33990, 37323, 37675, 20185, 20682, 20808,
    21892, 23307, 23459, 25159, 25982, 26059, 28210, 29053, 29697, 29764,
    29831, 29887, 30316, 31146, 32218, 32341, 32680, 33146, 33203, 33337,
    34330, 34796, 35445, 36323, 36984, 37521, 37925, 39245, 39854, 21352,
    23633, 26964, 27844, 27945, 28203, [12166, 33292], 34203, 35131, 35373,
    [35498, 63855, 63905], 38634, 40807, 21089, 26297, 27570, 32406, 34814,
    36109, 38275, 38493, 25885, 28041, 29166, 22478, 22995, 23468, 24615,
    24826, 25104, 26143, 26207, 29481, 29689, 30427, [30465, 63853], 31596,
    32854, 32882, 33125, 35488, 37266, 19990, 21218, 27506, 27927, 31237,
    31545, 32048, 36016, 21484, 22063, 22609, 23477, [12073, 23567], 23569,
    24034, 25152, 25475, 25620, 26157, 26803, 27836, 28040, 28335, 28703,
    28836, 29138, 29990, 30095, 30094, 30233, 31505, 31712, 31787, 32032,
    32057, 34092, 34157, 34311, 35380, 36877, 36961, 37045, 37559, 38902,
    39479, 20439, 23660, 26463, 28049, 31903, 32396, 35606, 36118, 36895,
    23403, 24061, 25613, 33984, 36956, 39137, [29575, 63841, 63963], 23435,
    24730, 26494, 28126, 35359, 35494, 36865, 38924, 21047, 28753, 30862,
    37782, 34928, 37335, 20462, 21463, 22013, 22234, 22402, 22781, 23234,
    23432, 23723, 23744, 24101, 24833, 25101, [12095, 25163], 25480, 25628,
    25910, [25976, 63849], 27193, 27530, [12116, 27700], 27929, 28465, 29159,
    29417, 29560, 29703, 29874, 30246, 30561, 31168, 31319, 31466, 31929,
    32143, 32172, 32353, 32670, 33065, 33585, 33936, 34010, 34282, 34966,
    35504, 35728, 36664, 36930, 36995, 37228, 37526, 37561, 38539,
    {f: 2, c: 38567}, 38614, 38656, 38920, [12216, 39318], 39635, 39706, 21460,
    22654, 22809, 23408, 23487, 28113, 28506, 29087, 29729, 29881, 32901,
    33789, 24033, 24455, 24490, 24642, 26092, 26642, 26991, 27219, 27529,
    27957, 28147, 29667, 30462, 30636, 31565, 32020, 33059, 33308, 33600,
    34036, 34147, 35426, 35524, 37255, 37662, 38918, 39348, 25100, 34899,
    36848, 37477, 23815, 23847, 23913, 29791, 33181, 34664, 28629,
    [25342, 63859], 32722, 35126, 35186, 19998, 20056, 20711, 21213, 21319,
    25215, 26119, 32361, 34821, 38494, 20365, 21273, 22070, 22987, 23204,
    [12075, 23608], 23630, 23629, 24066, 24337, 24643, 26045, 26159, 26178,
    26558, 26612, 29468, [12142, 30690], [12144, 31034], 32709, 33940, 33997,
    35222, 35430, 35433, 35553, [12183, 35925], 35962, 22516, 23508, 24335,
    24687, 25325, 26893, 27542, 28252, 29060, 31698, 34645, [35672, 63996],
    36606, [12215, 39135], 39166, 20280, 20353, 20449, 21627, 23072, 23480,
    24892, 26032, 26216, 29180, 30003, 31070, 32051, 33102, [12162, 33251],
    33688, 34218, 34254, 34563, 35338, [12189, 36523], [12191, 36763], 36805,
    22833, 23460, 23526, 24713, 23529, 23563, [12092, 24515], 27777, 28145,
    28683, 29978, 33455, 35574, [20160, 63997], [12055, 21313], 38617,
    [12114, 27663], 20126, 20420, 20818, 21854, 23077, 23784, 25105,
    [12123, 29273], 33469, 33706, 34558, 34905, 35357, 38463, 38597, 39187,
    40201, 40285, 22538, 23731, 23997, 24132, [24801, 63929], 24853, 25569,
    [27138, 63764, 63836, 63935], 28197, 37122, 37716, 38990, 39952, 40823,
    23433, 23736, 25353, 26191, 26696, 30524, 38593, 38797, 38996, 39839,
    26017, 35585, 36555, 38332, 21813, 23721, 24022, 24245, 26263, 30284,
    33780, 38343, 22739, 25276, 29390, 40232, 20208, 22830, 24591, 26171,
    27523, 31207, 40230, 21395, 21696, 22467, 23830, 24859, 26326, 28079,
    30861, 33406, 38552, 38724, 21380, 25212, 25494, 28082, 32266, 33099,
    38989, 27387, 32588, 40367, 40474, 20063, 20539, 20918, 22812, 24825,
    25590, 26928, 29242, 32822, 37326, 24369, 32004, [33509, 63860], 33903,
    33979, 34277, 36493, 20335, 22756, 23363, 24665, 25562, 25880, 25965,
    26264, 26954, 27171, 27915, 28673, 29036, 30162, 30221, 31155, 31344,
    [12154, 32650], 35140, 35731, 37312, 38525, 39178, 22276, 24481, 26044,
    28417, 30208, 31142, 35486, 39341, [12226, 39770], 40812, 20740, 25014,
    25233, 27277, 33222, 20547, 22576, 24422, 28937, [12180, 35328], 35578,
    23420, 34326, 20474, 20796, 22196, 22852, 25513, 28153, 23978, 26989,
    20870, 20104, 20313, 22914, 27487, 27741, 29877, 30998, 33287, 33349,
    33593, 36671, 36701, 39192, 20134, 22495, 24441, [26131, 63968], 30123,
    32377, 35695, 36870, 39515, 22181, 22567, 23032, 23071, 23476, 24310,
    25424, 25403, 26941, 27783, 27839, 28046, 28051, 28149, 28436, 28895,
    28982, 29017, 29123, 29141, 30799, 30831, 31605, 32227, 32303, 34893,
    36575, 37467, 40182, 24709, 28037, 29105, 38321, 21421, 26579, 28814,
    28976, 29744, 33398, 33490, 38331, 39653, 40573, 26308, 29121,
    [33865, 63854], 22603, 23992, 24433, 26144, 26254, 27001, 27054, 27704,
    27891, 28214, 28481, 28634, 28699, 28719, 29008, 29151, 29552, 29787,
    29908, 30408, 31310, 32403, 33521, 35424, 36814, 37704, 38681, 20034,
    20522, 21000, 21473, 26355, 27757, 28618, 29450, 30591, 31330, 33454,
    34269, 34306, 35028, 35427, 35709, 35947, 37555, 38675, 38928, 20116,
    20237, 20425, 20658, 21320, 21566, 21555, 21978, 22626, 22714, 22887,
    23067, 23524, 24735, 25034, 25942, 26111, 26212, 26791, 27738, 28595,
    28879, 29100, 29522, 31613, 34568, 35492, 39986, 40711, 23627, 27779,
    29508, [12127, 29577], 37434, 28331, 29797, 30239, 31337, 32277, 34314,
    20800, 22725, 25793, 29934, 29973, 30320, 32705, 37013, 38605, 39252,
    28198, [12129, 29926], {f: 2, c: 31401}, 33253, 34521, 34680, 35355, 23113,
    23436, 23451, 26785, 26880, 28003, 29609, 29715, 29740, 30871, 32233,
    32747, 33048, 33109, 33694, 35916, [38446, 63942], 38929, [12104, 26352],
    24448, 26106, 26505, 27754, 29579, 20525, 23043, 27498, 30702, 22806,
    23916, 24013, 29477, 30031, 20709, 20985, 22575, 22829, 22934, 23002,
    23525, 23970, 25303, 25622, 25747, 25854, 26332, 27208, 29183, 29796,
    31368, 31407, 32327, 32350, 32768, 33136, 34799, 35201, 35616, 36953,
    36992, 39250, 24958, 27442, 28020, 32287, 35109, 36785, 20433, 20653,
    20887, 21191, 22471, 22665, 23481, 24248, 24898, 27029, 28044, 28263,
    28342, 29076, 29794, [12132, 29992], 29996, 32883, 33592, 33993, 36362,
    37780, 37854, 20110, 20305, 20598, 20778, [12060, 21448], 21451, 21491,
    23431, 23507, 23588, 24858, 24962, 26100, [12124, 29275], 29591, 29760,
    30402, 31056, 31121, 31161, 32006, [12155, 32701], 33419, 34261, 34398,
    36802, 36935, 37109, 37354, 38533, [12204, 38632], 38633, 21206, 24423,
    26093, 26161, 26671, 29020, 31286, 37057, 38922, 20113, 27218, 27550,
    28560, 29065, 32792, 33464, 34131, 36939, 38549, 38642, 38907, 34074,
    39729, 20112, 29066, 38596, 20803, 21407, 21729, 22291, 22290, 22435,
    23195, 23236, 23491, 24616, 24895, 25588, 27781, 27961, 28274, 28304,
    29232, 29503, 29783, 33489, 34945, 36677, 36960, 38498, 39000, 40219,
    [12105, 26376], 36234, 37470, 20301, 20553, 20702, 21361, 22285, 22996,
    23041, 23561, 24944, 26256, 28205, 29234, 29771, 32239, 32963, 33806,
    33894, 34111, 34655, 34907, 35096, 35586, 36949, [12209, 38859], 39759,
    20083, 20369, 20754, 20842, 21807, 21929, 23418, 23461, {f: 2, c: 24188},
    24254, 24736, 24799, {f: 2, c: 24840}, 25540, 25912, 26377, 26580, 26586,
    {f: 2, c: 26977}, 27833, 27943, 28216, 28641, {f: 2, c: 29494}, 29788,
    30001, 30290, 32173, 33278, 33848, 35029, 35480, 35547, 35565, 36400,
    36418, 36938, 36926, 36986, [12195, 37193], 37321, 37742, 22537, 27603,
    [12161, 32905], 32946, 20801, 22891, 23609, 28516, 29607, 32996, 36103,
    37399, 38287, [12160, 32895], 25102, 28700, 32104, 34701, 22432, 24681,
    24903, 27575, 35518, 37504, 38577, [12036, 20057], 21535, 28139, 34093,
    38512, [12211, 38899], 39150, 25558, 27875, [12194, 37009], 20957, 25033,
    33210, 40441, 20381, 20506, 20736, 23452, 24847, 25087, 25836, 26885,
    27589, 30097, 30691, 32681, 33380, 34191, 34811, [12176, 34915], 35516,
    35696, 37291, [12038, 20108], 20197, 20234, 22839, 23016, 24050, 24347,
    24411, 24609, 29246, 29669, [30064, 63842], 30157, 31227, [12157, 32780],
    [12159, 32819], 32900, 33505, 33617, 36029, 36019, 36999, 39156, 39180,
    28727, 30410, 32714, 32716, 32764, 35610, [12040, 20154], 20161, 20995,
    21360, [21693, 63902], 22240, 23035, 23493, 24341, 24525, 28270, 32106,
    33589, 34451, 35469, 38765, 38775, [12032, 19968], 20314, 20350, 22777,
    [12103, 26085], 28322, 36920, 37808, 39353, 20219, 22764, 22922, 23001,
    24641, 31252, 33615, 36035, [12042, 20837], 21316, 20173, 21097, 23381,
    33471, 20180, [21050, 63999], 21672, 22985, 23039, [12070, 23376], 23383,
    23388, 24675, 24904, 28363, [28825, 63995], 29038, 29574, 29943, 30133,
    30913, 32043, 32773, [12163, 33258], 33576, 34071, 34249, 35566, 36039,
    38604, 20316, 21242, 22204, 26027, 26152, 28796, 28856, 29237, 32189,
    33421, 37196, 38592, 40306, 23409, 26855, 27544, 28538, 30430, 23697,
    26283, 28507, 31668, 31786, 34870, 38620, 19976, 20183, 21280, 22580,
    22715, 22767, 22892, 23559, 24115, 24196, 24373, 25484, 26290, 26454,
    27167, 27299, 27404, 28479, 29254, 29520, 29835, 31456, 31911, 33144,
    33247, 33255, 33674, 33900, 34083, 34196, 34255, 35037, 36115, 37292,
    [12199, 38263], 38556, 20877, 21705, 22312, 23472, 25165, 26448, 26685,
    26771, 28221, 28371, 28797, 32289, 35009, 36001, 36617, 40779, 40782,
    29229, 31631, 35533, 37658, 20295, 20302, 20786, 21632, 22992, 24213,
    25269, 26485, 26990, 27159, 27822, 28186, 29401, 29482, 30141, 31672,
    32053, 33511, 33785, 33879, 34295, 35419, 36015, 36487, 36889, 37048,
    38606, 40799, 21219, 21514, 23265, 23490, 25688, 25973, 28404, 29380,
    30340, 31309, 31515, 31821, 32318, 32735, 33659, 35627, 36042,
    [12186, 36196], 36321, 36447, 36842, 36857, 36969, 37841, 20291, 20346,
    20659, 20840, 20856, 21069, 21098, 22625, 22652, 22880, 23560, 23637,
    24283, 24731, 25136, 26643, 27583, 27656, 28593, 29006, 29728,
    [12133, 30000], 30008, 30033, 30322, 31564, 31627, 31661, 31686, 32399,
    35438, 36670, 36681, 37439, 37523, 37666, 37931, 38651, 39002, 39019,
    39198, [20999, 64000], 25130, 25240, 27993, 30308, 31434, 31680, 32118,
    21344, 23742, 24215, 28472, 28857, 31896, 38673, 39822, 40670, 25509,
    25722, 34678, 19969, 20117, 20141, 20572, 20597, 21576, 22979, 23450,
    24128, 24237, 24311, 24449, 24773, 25402, 25919, 25972, 26060, 26230,
    26232, 26622, 26984, 27273, 27491, 27712, 28096, 28136, 28191, 28254,
    28702, 28833, 29582, 29693, 30010, 30555, 30855, 31118, 31243, 31357,
    31934, 32142, 33351, 35330, 35562, 35998, 37165, 37194, 37336, 37478,
    37580, 37664, 38662, 38742, 38748, 38914, [12237, 40718], 21046, 21137,
    21884, 22564, 24093, 24351, 24716, 25552, 26799, 28639, 31085, 31532,
    33229, 34234, 35069, 35576, 36420, 37261, 38500, 38555, 38717, 38988,
    [12241, 40778], 20430, 20806, 20939, 21161, 22066, 24340, 24427, 25514,
    25805, 26089, 26177, 26362, 26361, 26397, 26781, 26839, 27133, 28437,
    28526, 29031, 29157, [12118, 29226], 29866, 30522, 31062, 31066, 31199,
    31264, 31381, 31895, 31967, 32068, 32368, 32903, 34299, 34468, 35412,
    35519, 36249, 36481, 36896, 36973, 37347, 38459, 38613, [12227, 40165],
    26063, 31751, [12188, 36275], 37827, 23384, 23562, 21330, 25305, 29469,
    20519, 23447, 24478, 24752, 24939, 26837, 28121, 29742, 31278, 32066,
    32156, 32305, 33131, 36394, 36405, 37758, 37912, 20304, 22352, 24038,
    24231, 25387, 32618, 20027, 20303, 20367, 20570, 23005, 32964, 21610,
    21608, 22014, 22863, 23449, 24030, 24282, 26205, 26417, 26609, 26666,
    27880, 27954, 28234, 28557, 28855, 29664, 30087, 31820, 32002, 32044,
    32162, [12168, 33311], 34523, 35387, 35461, [12187, 36208], 36490, 36659,
    36913, 37198, 37202, 37956, 39376, [12149, 31481], 31909, 20426, 20737,
    20934, 22472, 23535, 23803, 26201, 27197, 27994, 28310, 28652, 28940,
    30063, 31459, 34850, 36897, 36981, 38603, 39423, 33537, 20013, 20210,
    34886, 37325, 21373, 27355, 26987, 27713, 33914, 22686, 24974, 26366,
    25327, 28893, 29969, 30151, 32338, 33976, 35657, 36104, 20043, 21482,
    21675, 22320, 22336, 24535, 25345, 25351, 25711, [12096, 25903], 26088,
    26234, 26525, 26547, [12108, 27490], 27744, 27802, 28460, 30693, 30757,
    31049, 31063, 32025, 32930, 33026, [12164, 33267], 33437, 33463, 34584,
    35468, 36100, 36286, 36978, 30452, 31257, 31287, 32340, 32887, 21767,
    21972, 22645, 25391, 25634, 26185, 26187, 26733, 27035, 27524, 27941,
    28337, 29645, 29800, 29857, 30043, 30137, 30433, 30494, 30603, 31206,
    32265, 32285, 33275, 34095, 34967, 35386, 36049, 36587,
    [12192, 36784, 63857], 36914, 37805, 38499, 38515, 38663, 20356, 21489,
    23018, 23241, 24089, 26702, 29894, 30142, 31209, 31378, 33187, 34541,
    36074, 36300, 36845, 26015, 26389, 22519, 28503, 32221, 36655, 37878,
    38598, 24501, 25074, 28548, 19988, 20376, 20511, 21449, 21983, 23919,
    24046, 27425, 27492, 30923, 31642, 36425, [12190, 36554, 63746], 36974,
    25417, 25662, 30528, 31364, 37679, 38015, 40810, 25776, 28591, 29158,
    29864, 29914, 31428, 31762, 32386, 31922, 32408, 35738, 36106, 38013,
    39184, 39244, 21049, 23519, 25830, 26413, 32046, 20717, [21443, 63851],
    22649, {f: 2, c: 24920}, 25082, 26028, 31449, 35730, 35734, 20489, 20513,
    21109, 21809, 23100, 24288, 24432, 24884, 25950, 26124, 26166, 26274,
    27085, 28356, 28466, 29462, 30241, 31379, 33081, 33369, 33750, 33980,
    20661, 22512, 23488, 23528, 24425, 25505, 30758, 32181, 33756, 34081,
    37319, 37365, 20874, 26613, 31574, 36012, 20932, 22971, 24765, 34389,
    20508, 21076, 23610, 24957, 25114, [25299, 64002], 25842, 26021, 28364,
    30240, 33034, 36448, 38495, 38587, 20191, 21315, 21912, 22825, 24029,
    25797, 27849, 28154, 29588, 31359, [12167, 33307], 34214, 36068, 36368,
    36983, 37351, 38369, 38433, 38854, 20984, 21746, 21894, 24505, 25764,
    28552, 32180, 36639, 36685, 37941, 20681, 23574, 27838, 28155, 29979,
    30651, 31805, 31844, 35449, 35522, 22558, 22974, 24086, 25463, 29266,
    30090, 30571, 35548, 36028, 36626, 24307, 26228, 28152, 32893, 33729,
    35531, [12205, 38737], 39894, 21059, 26367, 28053, 28399, 32224, 35558,
    36910, 36958, 39636, 21021, 21119, 21736, 24980, 25220, 25307, 26786,
    26898, 26970, 27189, 28818, 28966, 30813, 30977, 30990, 31186, 31245,
    32918, [12171, 33400], 33493, 33609, 34121, 35970, 36229, 37218, 37259,
    37294, 20419, 22225, 29165, 30679, 34560, 35320, [12072, 23544], 24534,
    26449, 37032, 21474, 22618, 23541, 24740, 24961, 25696, 32317, 32880,
    34085, 37507, 25774, 20652, 23828, 26368, 22684, 25277, 25512, 26894,
    27000, 27166, 28267, 30394, 31179, 33467, 33833, 35535, 36264, 36861,
    37138, 37195, 37276, 37648, 37656, 37786, 38619, 39478, 39949, 19985,
    30044, 31069, 31482, 31569, 31689, 32302, 33988, 36441, 36468, 36600,
    36880, 26149, 26943, 29763, 20986, 26414, 40668, 20805, 24544, 27798,
    34802, 34909, 34935, 24756, 33205, 33795, 36101, 21462, 21561, 22068,
    23094, 23601, 28810, 32736, 32858, 33030, 33261, 36259, 37257, 39519,
    40434, 20596, 20164, 21408, 24827, 28204, 23652, 20360, 20516, 21988,
    23769, 24159, 24677, 26772, 27835, 28100, 29118, 30164, 30196, 30305,
    31258, 31305, 32199, 32251, 32622, 33268, 34473, 36636, 38601, 39347,
    [12242, 40786], 21063, 21189, 39149, 35242, 19971, 26578, 28422, 20405,
    23522, 26517, [27784, 63858], 28024, 29723, 30759, 37341, 37756, 34756,
    31204, 31281, 24555, 20182, 21668, 21822, 22702, 22949, 24816, 25171,
    25302, 26422, 26965, 33333, 38464, 39345, 39389, 20524, 21331, 21828,
    22396, 25176, 25826, 26219, 26589, 28609, 28655, 29730, 29752, 35351,
    37944, 21585, 22022, 22374, 24392, 24986, 27470, 28760, 28845, 32187,
    35477, 22890, 33067, 25506, 30472, 32829, 36010, 22612, 25645, 27067,
    23445, 24081, 28271, 34153, 20812, 21488, 22826, 24608, 24907, 27526,
    27760, 27888, 31518, 32974, 33492, 36294, 37040, 39089, 25799, 28580,
    25745, 25860, 20814, 21520, [12063, 22303], 35342, 24927, 26742, 30171,
    31570, 32113, 36890, 22534, 27084, 33151, 35114, 36864, 38969, 20600,
    22871, 22956, 25237, 36879, 39722, 24925, 29305, 38358, 22369, 23110,
    24052, 25226, 25773, 25850, 26487, 27874, 27966, 29228, 29750, 30772,
    32631, 33453, 36315, 38935, 21028, 22338, 26495, 29256, 29923, 36009,
    36774, 37393, 38442, [12043, 20843], 21485, 25420, 20329, 21764, 24726,
    25943, 27803, 28031, 29260, 29437, 31255, 35207, [12185, 35997], 24429,
    28558, 28921, 33192, 24846, [20415, 63845], 20559, 25153, [12122, 29255],
    31687, 32232, 32745, 36941, 38829, 39449, 36022, 22378, 24179, 26544,
    33805, 35413, 21536, 23318, 24163, 24290, 24330, 25987, 32954, 34109,
    38281, 38491, 20296, 21253, 21261, 21263, 21638, 21754, 22275, 24067,
    24598, 25243, 25265, 25429, 27873, 28006, 30129, 30770, 32990, 33071,
    33502, 33889, 33970, 34957, 35090, 36875, 37610, 39165, 39825, 24133,
    [26292, 64006], 26333, 28689, 29190, 20469, 21117, 24426, 24915, 26451,
    27161, 28418, 29922, 31080, 34920, 35961, 39111, 39108, 39491, 21697,
    31263, 26963, 35575, 35914, [12213, 39080], 39342, 24444, 25259, 30130,
    [12138, 30382], 34987, 36991, 38466, 21305, 24380, 24517, [27852, 63848],
    29644, 30050, [12134, 30091], 31558, 33534, 39325, 20047, 36924, 19979,
    20309, 21414, 22799, 24264, 26160, 27827, 29781, 33655, 34662, 36032,
    36944, 38686, 39957, 22737, 23416, 34384, 35604, 40372, 23506, 24680,
    24717, 26097, 27735, 28450, 28579, 28698, 32597, 32752, {f: 2, c: 38289},
    38480, 38867, 21106, 36676, 20989, 21547, 21688, 21859, 21898, 27323,
    28085, 32216, 33382, 37532, 38519, 40569, 21512, 21704, 30418, 34532,
    38308, 38356, 38492, 20130, 20233, 23022, 23270, 24055, 24658, 25239,
    26477, 26689, 27782, 28207, 32568, 32923, 33322, 38917, 20133, 20565,
    21683, 22419, 22874, 23401, 23475, 25032, 26999, 28023, 28707, 34809,
    35299, 35442, 35559, 36994, 39405, 39608, 21182, 26680, 20502, 24184,
    26447, 33607, [12175, 34892, 64008], 20139, 21521, 22190, 29670, 37141,
    38911, 39177, 39255, [12217, 39321], 22099, 22687, 34395, 35377, 25010,
    27382, 29563, 36562, 27463, 38570, 39511, 22869, 29184, 36203,
    [12208, 38761], 20436, 23796, 24358, 25080, 26203, 27883, 28843,
    [12126, 29572], 29625, 29694, 30505, 30541, 32067, 32098, 32291, 33335,
    34898, 36066, 37449, 39023, 23377, [12147, 31348], [12174, 34880],
    [12212, 38913], 23244, 20448, 21332, 22846, 23805, 25406, 28025, 29433,
    33029, 33031, 33698, 37583, 38960, 20136, 20804, 21009, 22411, 24418,
    27842, 28366, 28677, 28752, 28847, 29074, 29673, [29801, 63918], 33610,
    34722, 34913, 36872, 37026, 37795, 39336, 20846, 24407, 24800, 24935,
    26291, 34137, 36426, 37295, 38795, 20046, 20114, 21628, 22741, 22778,
    22909, 23733, 24359, [12094, 25142], 25160, 26122, 26215, 27627, 28009,
    28111, 28246, 28408, 28564, 28640, 28649, 28765, 29392, 29733, 29786,
    29920, 30355, 31068, 31946, 32286, 32993, 33446, 33899, 33983, 34382,
    34399, 34676, 35703, 35946, 37804, 38912, 39013, 24785, 25110, 37239,
    23130, 26127, 28151, 28222, 29759, 39746, 24573, 24794, 31503, 21700,
    24344, 27742, 27859, 27946, 28888, 32005, 34425, 35340, 40251, 21270,
    21644, 23301, 27194, [12117, 28779], 30069, 31117, [12146, 31166], 33457,
    33775, 35441, 35649, 36008, 38772, 25844, 25899, {f: 2, c: 30906}, 31339,
    20024, 21914, 22864, 23462, 24187, 24739, 25563, 27489, 26213, 26707,
    28185, 29029, 29872, 32008, 36996, 39529, 39973, 27963, [28369, 63748],
    29502, 35905, 38346, 20976, 24140, 24488, 24653, 24822, 24880, 24908,
    {f: 2, c: 26179}, 27045, 27841, 28255, 28361, 28514, 29004, 29852, 30343,
    31681, 31783, 33618, 34647, 36945, 38541, [12232, 40643], 21295, 22238,
    24315, 24458, 24674, 24724, 25079, 26214, 26371, 27292, 28142, 28590,
    28784, 29546, 32362, 33214, 33588, 34516, 35496, 36036, 21123, 29554,
    23446, 27243, 37892, 21742, 22150, 23389, 25928, 25989, 26313, 26783,
    28045, 28102, [12120, 29243], 32948, 37237, 39501, 20399, 20505, 21402,
    21518, 21564, 21897, 21957, 24127, 24460, 26429, 29030, 29661, 36869,
    21211, 21235, 22628, 22734, 28932, 29071, 29179, 34224, 35347,
    [26248, 63941], 34216, 21927, 26244, 29002, 33841, 21321, 21913, 27585,
    24409, 24509, 25582, 26249, 28999, 35569, 36637, 40638, 20241, 25658,
    28875, 30054, 34407, 24676, 35662, 40440, 20807, 20982, 21256, 27958,
    33016, [12234, 40657], 26133, 27427, 28824, 30165, 21507, 23673, 32007,
    35350, [12107, 27424], 27453, 27462, 21560, 24688, 27965, 32725, 33288,
    20694, 20958, 21916, 22123, 22221, 23020, 23305, 24076, 24985, 24984,
    25137, 26206, 26342, 29081, {f: 2, c: 29113}, 29351, 31143, 31232, 32690,
    35440, {s: 163}, {f: 4, c: 12310}, {s: 14}, 8223, 8219, {f: 2, c: 8314},
    {s: 7}, 8316, 0, {f: 2, c: 8317}, {s: 23}, 700, {s: 44}, 8942, 8759,
    {s: 20}, {f: 10, c: 10122}, {s: 36}, {f: 26, c: 9398}, {s: 61},
    {f: 2, c: 8826}, {f: 2, c: 8910}, {f: 2, c: 8832}, {f: 4, c: 8816}, 0,
    8842, 0, 8843, {f: 2, c: 8822}, 8825, {f: 2, c: 8922}, {s: 5}, 8773, 8771,
    8776, 0, 8868, {s: 78}, 8244, {s: 11}, 9839, {s: 4}, 8258, {s: 4}, 10045,
    0, 0, 8226, {s: 4}, {f: 2, c: 8249}, {s: 16}, 10010, 10006, 0, 9711,
    {s: 3}, 10070, 0, 9676, {s: 24}, 9775, {s: 6}, 12320, 0, {f: 10, c: 10102},
    {s: 17}, 12306, 12342, {s: 13}, 8710, 0, 8735, 0, {f: 2, c: 8741}, 0, 8787,
    8785, {f: 2, c: 8806}, 8723, {f: 3, c: 8853}, 0, 8980, 0, 0, 8802, 0, 9649,
    0, 8738, 8784, 0, 0, 8867, 0, 0, {f: 2, c: 8814}, 8837, 8836, 8713, 8716,
    {f: 2, c: 8891}, 8794, 8966, {s: 6}, 12958, 0, 8252, {s: 11}, 9702, {s: 3},
    9663, 9653, 9657, 9667, {s: 4}, 9674, 12849, 12857, 13259, {f: 5, c: 9327},
    {s: 18}, 8656, 8655, 8653, {s: 37}, 8657, 8659, {s: 8}, 8626, 8625, 0,
    8628, 8624, 8627, {s: 14}, 8636, 8640, {s: 10}, {f: 2, c: 8644}, {s: 144},
    {f: 5, c: 9347}, {s: 33}, 12948, {s: 15}, 12965, {s: 93}, 8672, 8674, 8673,
    8675, {s: 4}, 8678, 8680, 8679, 8681, {s: 20}, 9757, 9759, {s: 76}, 12944,
    {f: 6, c: 12938}, {s: 15}, {f: 2, c: 12318}, 8246, 0, 8245, {s: 3}, 12540,
    0, 0, {f: 2, c: 44034}, {f: 2, c: 44037}, {f: 5, c: 44043}, 44056,
    {f: 2, c: 44062}, {f: 3, c: 44065}, {f: 7, c: 44069}, 44078,
    {f: 6, c: 44082}, {f: 2, c: 44090}, {f: 3, c: 44093}, {f: 10, c: 44097},
    44108, {f: 6, c: 44110}, {f: 3, c: 44117}, {f: 3, c: 44121},
    {f: 19, c: 44125}, {f: 2, c: 44146}, {f: 2, c: 44149}, 44153,
    {f: 5, c: 44155}, 44162, {f: 2, c: 44167}, {f: 3, c: 44173},
    {f: 3, c: 44177}, {f: 7, c: 44181}, 44190, {f: 6, c: 44194}, 44203,
    {f: 2, c: 44205}, {f: 7, c: 44209}, 44218, {f: 3, c: 44222},
    {f: 2, c: 44226}, {f: 3, c: 44229}, {f: 3, c: 44233}, {f: 8, c: 44237},
    44246, {f: 8, c: 44248}, {f: 2, c: 44258}, {f: 2, c: 44261}, 44265, 44267,
    {f: 2, c: 44269}, 44274, 44276, {f: 5, c: 44279}, {f: 2, c: 44286},
    {f: 3, c: 44289}, 44293, {f: 5, c: 44295}, 44302, 44304, {f: 6, c: 44306},
    {f: 3, c: 44313}, {f: 3, c: 44317}, {f: 8, c: 44321}, {f: 2, c: 44330},
    {f: 6, c: 44334}, {f: 2, c: 44342}, {f: 3, c: 44345}, {f: 7, c: 44349},
    44358, 44360, {f: 6, c: 44362}, {f: 3, c: 44369}, {f: 3, c: 44373},
    {f: 8, c: 44377}, 44386, {f: 8, c: 44388}, {f: 2, c: 44398},
    {f: 2, c: 44401}, {f: 4, c: 44407}, 44414, 44416, {f: 5, c: 44419},
    {f: 2, c: 44426}, {f: 3, c: 44429}, {f: 11, c: 44433}, {f: 6, c: 44446},
    {f: 18, c: 44453}, {f: 8, c: 44472}, {f: 2, c: 44482}, {f: 3, c: 44485},
    {f: 7, c: 44489}, 44498, {f: 8, c: 44500}, {f: 3, c: 44509},
    {f: 3, c: 44513}, {f: 19, c: 44517}, {f: 2, c: 44538}, {f: 2, c: 44541},
    {f: 6, c: 44546}, 44554, 44556, {f: 6, c: 44558}, {f: 27, c: 44565},
    {f: 2, c: 44594}, {f: 2, c: 44597}, 44601, {f: 5, c: 44603}, 44610, 44612,
    {f: 3, c: 44615}, 44619, 44623, {f: 3, c: 44625}, 44629, {f: 5, c: 44631},
    44638, {f: 3, c: 44642}, {f: 2, c: 44646}, {f: 2, c: 44650},
    {f: 3, c: 44653}, {f: 7, c: 44657}, 44666, {f: 6, c: 44670},
    {f: 6, c: 44678}, {f: 47, c: 44685}, 44735, {f: 3, c: 44737},
    {f: 7, c: 44741}, 44750, {f: 6, c: 44754}, {f: 2, c: 44762},
    {f: 11, c: 44765}, {f: 2, c: 44777}, 44780, {f: 6, c: 44782},
    {f: 3, c: 44789}, {f: 3, c: 44793}, {f: 10, c: 44797}, {f: 4, c: 44809},
    {f: 2, c: 44814}, {f: 27, c: 44817}, {f: 2, c: 44846}, 44849, 44851,
    {f: 7, c: 44853}, 44862, 44864, {f: 4, c: 44868}, {f: 6, c: 44874},
    {f: 11, c: 44881}, {f: 6, c: 44894}, {f: 19, c: 44902}, {f: 6, c: 44922},
    {f: 3, c: 44929}, {f: 3, c: 44933}, {f: 7, c: 44937}, {f: 3, c: 44946},
    {f: 6, c: 44950}, {f: 27, c: 44957}, {f: 2, c: 44986}, {f: 3, c: 44989},
    {f: 6, c: 44993}, 45002, 45004, {f: 5, c: 45007}, {f: 7, c: 45013},
    {f: 11, c: 45021}, {f: 6, c: 45034}, {f: 2, c: 45042}, {f: 3, c: 45045},
    {f: 7, c: 45049}, {f: 2, c: 45058}, {f: 7, c: 45061}, {f: 3, c: 45069},
    {f: 3, c: 45073}, {f: 7, c: 45077}, {f: 10, c: 45086}, {f: 27, c: 45097},
    {f: 2, c: 45126}, 45129, 45131, 45133, {f: 4, c: 45135}, 45142, 45144,
    {f: 3, c: 45146}, {f: 30, c: 45150}, {f: 2, c: 45182}, {f: 3, c: 45185},
    {f: 7, c: 45189}, 45198, 45200, {f: 6, c: 45202}, 45211, {f: 2, c: 45213},
    {f: 5, c: 45219}, 45226, 45232, 45234, {f: 2, c: 45238}, {f: 3, c: 45241},
    {f: 7, c: 45245}, 45254, {f: 6, c: 45258}, {f: 2, c: 45266},
    {f: 3, c: 45269}, {f: 7, c: 45273}, {f: 4, c: 45281}, {f: 34, c: 45286},
    45322, {f: 3, c: 45325}, 45329, {f: 4, c: 45332}, 45338, {f: 5, c: 45342},
    {f: 2, c: 45350}, {f: 3, c: 45353}, {f: 7, c: 45357}, 45366,
    {f: 6, c: 45370}, {f: 2, c: 45378}, {f: 3, c: 45381}, {f: 7, c: 45385},
    {f: 2, c: 45394}, {f: 2, c: 45398}, {f: 3, c: 45401}, {f: 3, c: 45405},
    {f: 23, c: 45409}, {f: 2, c: 45434}, {f: 3, c: 45437}, 45441,
    {f: 5, c: 45443}, 45450, 45452, {f: 4, c: 45454}, {f: 3, c: 45461},
    {f: 3, c: 45465}, {f: 11, c: 45469}, {f: 35, c: 45481}, {f: 3, c: 45517},
    {f: 3, c: 45521}, {f: 7, c: 45525}, 45534, {f: 8, c: 45536},
    {f: 2, c: 45546}, {f: 3, c: 45549}, {f: 8, c: 45553}, 45562, 45564,
    {f: 6, c: 45566}, {f: 2, c: 45574}, {f: 2, c: 45577}, {f: 7, c: 45581},
    45590, 45592, {f: 6, c: 45594}, {f: 19, c: 45601}, {f: 7, c: 45621},
    {f: 27, c: 45629}, {f: 3, c: 45657}, {f: 3, c: 45661}, {f: 7, c: 45665},
    {f: 10, c: 45674}, {f: 6, c: 45686}, {f: 7, c: 45693}, {f: 3, c: 45702},
    {f: 6, c: 45706}, {f: 2, c: 45714}, {f: 3, c: 45717}, {f: 5, c: 45723},
    45730, 45732, {f: 3, c: 45735}, 45739, {f: 3, c: 45741}, {f: 3, c: 45745},
    {f: 19, c: 45749}, {f: 2, c: 45770}, {f: 3, c: 45773}, 45777,
    {f: 5, c: 45779}, 45786, 45788, {f: 4, c: 45790}, 45795, 45799,
    {f: 2, c: 45801}, {f: 3, c: 45808}, 45814, {f: 3, c: 45820},
    {f: 2, c: 45826}, {f: 3, c: 45829}, {f: 7, c: 45833}, 45842,
    {f: 6, c: 45846}, {f: 55, c: 45853}, 45911, {f: 2, c: 45913}, 45917,
    {f: 4, c: 45920}, 45926, 45928, 45930, {f: 2, c: 45932}, 45935,
    {f: 2, c: 45938}, {f: 3, c: 45941}, {f: 7, c: 45945}, 45954,
    {f: 6, c: 45958}, {f: 3, c: 45965}, {f: 3, c: 45969}, {f: 11, c: 45973},
    {f: 6, c: 45986}, {f: 3, c: 45993}, {f: 23, c: 45997}, {f: 2, c: 46022},
    {f: 2, c: 46025}, 46029, 46031, {f: 3, c: 46033}, 46038, 46040, 46042,
    46044, {f: 2, c: 46046}, {f: 3, c: 46049}, {f: 3, c: 46053},
    {f: 19, c: 46057}, {f: 19, c: 46077}, {f: 7, c: 46097}, {f: 3, c: 46105},
    {f: 3, c: 46109}, {f: 7, c: 46113}, 46122, {f: 8, c: 46124},
    {f: 27, c: 46133}, {f: 2, c: 46162}, {f: 3, c: 46165}, {f: 7, c: 46169},
    46178, 46180, {f: 6, c: 46182}, {f: 19, c: 46189}, {f: 7, c: 46209},
    {f: 20, c: 46217}, {f: 6, c: 46238}, {f: 3, c: 46245}, {f: 3, c: 46249},
    {f: 8, c: 46253}, 46262, 46264, {f: 6, c: 46266}, {f: 3, c: 46273},
    {f: 3, c: 46277}, {f: 7, c: 46281}, {f: 4, c: 46289}, {f: 6, c: 46294},
    {f: 2, c: 46302}, {f: 2, c: 46305}, 46309, {f: 5, c: 46311}, 46318, 46320,
    {f: 6, c: 46322}, {f: 27, c: 46329}, {f: 2, c: 46358}, {f: 2, c: 46361},
    {f: 7, c: 46365}, 46374, {f: 5, c: 46379}, {f: 2, c: 46386},
    {f: 3, c: 46389}, {f: 7, c: 46393}, 46402, {f: 5, c: 46406},
    {f: 2, c: 46414}, {f: 3, c: 46417}, {f: 7, c: 46421}, 46430,
    {f: 62, c: 46434}, {f: 2, c: 46498}, {f: 3, c: 46501}, 46505,
    {f: 4, c: 46508}, 46514, {f: 5, c: 46518}, {f: 2, c: 46526},
    {f: 3, c: 46529}, {f: 7, c: 46533}, 46542, {f: 6, c: 46546},
    {f: 19, c: 46553}, {f: 35, c: 46573}, {f: 2, c: 46610}, {f: 3, c: 46613},
    {f: 12, c: 46617}, {f: 6, c: 46630}, {f: 7, c: 46637}, {f: 19, c: 46645},
    {f: 27, c: 46665}, {f: 3, c: 46693}, {f: 51, c: 46697}, {f: 2, c: 46750},
    {f: 3, c: 46753}, {f: 6, c: 46757}, {f: 4, c: 46765}, {f: 34, c: 46770},
    {f: 27, c: 46805}, {f: 3, c: 46833}, {f: 3, c: 46837}, {f: 7, c: 46841},
    {f: 3, c: 46850}, {f: 34, c: 46854}, {f: 2, c: 46890}, {f: 2, c: 46893},
    {f: 7, c: 46897}, 46906, {f: 8, c: 46908}, {f: 3, c: 46917},
    {f: 3, c: 46921}, {f: 7, c: 46925}, {f: 10, c: 46934}, {f: 3, c: 46945},
    {f: 3, c: 46949}, {f: 7, c: 46953}, 46962, 46964, {f: 6, c: 46966},
    {f: 2, c: 46974}, {f: 3, c: 46977}, {f: 7, c: 46981}, 46990,
    {f: 3, c: 46995}, {f: 2, c: 47002}, {f: 3, c: 47005}, {f: 7, c: 47009},
    47018, {f: 6, c: 47022}, {f: 2, c: 47030}, {f: 14, c: 47033}, 47048,
    {f: 34, c: 47050}, {f: 2, c: 47086}, {f: 3, c: 47089}, {f: 7, c: 47093},
    47102, {f: 5, c: 47106}, {f: 2, c: 47114}, {f: 3, c: 47117},
    {f: 7, c: 47121}, 47130, 47132, {f: 6, c: 47134}, {f: 2, c: 47142},
    {f: 3, c: 47145}, {f: 7, c: 47149}, 47158, {f: 6, c: 47162},
    {f: 3, c: 47169}, {f: 12, c: 47173}, 47186, {f: 8, c: 47188},
    {f: 2, c: 47198}, {f: 3, c: 47201}, {f: 7, c: 47205}, 47214, 47216,
    {f: 6, c: 47218}, {f: 3, c: 47225}, {f: 16, c: 47229}, {f: 26, c: 47246},
    {f: 7, c: 47273}, {f: 3, c: 47281}, {f: 3, c: 47285}, {f: 7, c: 47289},
    47298, 47300, {f: 6, c: 47302}, {f: 3, c: 47309}, {f: 3, c: 47313},
    {f: 8, c: 47317}, 47326, 47328, {f: 6, c: 47330}, {f: 2, c: 47338},
    {f: 3, c: 47341}, {f: 7, c: 47345}, 47354, 47356, {f: 6, c: 47358},
    {f: 19, c: 47365}, {f: 7, c: 47385}, {f: 27, c: 47393}, {f: 2, c: 47422},
    {f: 3, c: 47425}, {f: 7, c: 47429}, {f: 2, c: 47437}, 47440,
    {f: 6, c: 47442}, {f: 2, c: 47450}, {f: 3, c: 47453}, {f: 7, c: 47457},
    47466, 47468, {f: 6, c: 47470}, {f: 2, c: 47478}, {f: 3, c: 47481},
    {f: 7, c: 47485}, 47494, 47496, {f: 2, c: 47499}, {f: 29, c: 47503},
    {f: 2, c: 47534}, {f: 3, c: 47537}, {f: 7, c: 47541}, 47550, 47552,
    {f: 6, c: 47554}, {f: 2, c: 47562}, 47565, {f: 5, c: 47571}, 47578, 47580,
    {f: 2, c: 47583}, 47586, {f: 2, c: 47590}, {f: 3, c: 47593},
    {f: 7, c: 47597}, 47606, {f: 5, c: 47611}, {f: 6, c: 47618},
    {f: 12, c: 47625}, {f: 34, c: 47638}, {f: 2, c: 47674}, {f: 3, c: 47677},
    47681, {f: 5, c: 47683}, 47690, 47692, {f: 4, c: 47695}, {f: 2, c: 47702},
    {f: 3, c: 47705}, {f: 7, c: 47709}, 47718, {f: 6, c: 47722},
    {f: 2, c: 47730}, {f: 3, c: 47733}, {f: 10, c: 47737}, 47750,
    {f: 4, c: 47752}, {f: 27, c: 47757}, 47786, {f: 3, c: 47789}, 47793,
    {f: 5, c: 47795}, 47802, 47804, {f: 6, c: 47806}, {f: 3, c: 47813},
    {f: 15, c: 47817}, {f: 34, c: 47834}, {f: 3, c: 47869}, {f: 3, c: 47873},
    {f: 8, c: 47877}, 47886, 47888, {f: 6, c: 47890}, {f: 3, c: 47897},
    {f: 3, c: 47901}, {f: 8, c: 47905}, 47914, {f: 8, c: 47916}, 47927,
    {f: 2, c: 47929}, {f: 5, c: 47935}, 47942, 47944, {f: 3, c: 47946}, 47950,
    {f: 3, c: 47953}, {f: 3, c: 47957}, {f: 8, c: 47961}, 47970,
    {f: 8, c: 47972}, {f: 27, c: 47981}, {f: 3, c: 48009}, {f: 3, c: 48013},
    {f: 19, c: 48017}, {f: 3, c: 48037}, {f: 3, c: 48041}, {f: 7, c: 48045},
    {f: 2, c: 48053}, {f: 8, c: 48056}, {f: 3, c: 48065}, {f: 3, c: 48069},
    {f: 7, c: 48073}, {f: 2, c: 48081}, {f: 36, c: 48084}, {f: 2, c: 48122},
    {f: 2, c: 48125}, 48129, {f: 5, c: 48131}, 48138, 48142, 48144,
    {f: 2, c: 48146}, {f: 2, c: 48153}, {f: 4, c: 48160}, 48166, 48168,
    {f: 3, c: 48170}, {f: 2, c: 48174}, {f: 2, c: 48178}, {f: 3, c: 48181},
    {f: 7, c: 48185}, 48194, {f: 3, c: 48198}, {f: 2, c: 48202},
    {f: 2, c: 48206}, {f: 12, c: 48209}, {f: 38, c: 48222}, {f: 2, c: 48262},
    {f: 2, c: 48265}, 48269, {f: 5, c: 48271}, 48278, 48280, {f: 5, c: 48283},
    {f: 2, c: 48290}, {f: 2, c: 48293}, {f: 7, c: 48297}, 48306,
    {f: 6, c: 48310}, {f: 2, c: 48318}, {f: 3, c: 48321}, {f: 8, c: 48325},
    48334, {f: 3, c: 48338}, {f: 2, c: 48342}, {f: 3, c: 48345},
    {f: 23, c: 48349}, 48375, {f: 3, c: 48377}, {f: 7, c: 48381}, 48390, 48392,
    {f: 6, c: 48394}, {f: 3, c: 48401}, {f: 15, c: 48405}, {f: 7, c: 48421},
    {f: 19, c: 48429}, {f: 7, c: 48449}, {f: 2, c: 48458}, {f: 3, c: 48461},
    {f: 7, c: 48465}, {f: 10, c: 48474}, {f: 3, c: 48485}, {f: 23, c: 48489},
    {f: 2, c: 48514}, {f: 2, c: 48517}, {f: 5, c: 48523}, 48530, 48532,
    {f: 3, c: 48534}, 48539, {f: 7, c: 48541}, {f: 11, c: 48549},
    {f: 7, c: 48561}, {f: 27, c: 48569}, {f: 2, c: 48598}, {f: 3, c: 48601},
    {f: 12, c: 48605}, {f: 6, c: 48618}, {f: 3, c: 48625}, {f: 3, c: 48629},
    {f: 7, c: 48633}, {f: 2, c: 48641}, 48644, {f: 6, c: 48646},
    {f: 2, c: 48654}, {f: 3, c: 48657}, {f: 7, c: 48661}, 48670,
    {f: 36, c: 48672}, {f: 2, c: 48710}, {f: 3, c: 48713}, 48717,
    {f: 5, c: 48719}, 48726, 48728, {f: 4, c: 48732}, {f: 2, c: 48738},
    {f: 3, c: 48741}, 48745, {f: 5, c: 48747}, 48754, {f: 5, c: 48758},
    {f: 2, c: 48766}, {f: 3, c: 48769}, {f: 7, c: 48773}, 48782,
    {f: 6, c: 48786}, {f: 14, c: 48794}, {f: 39, c: 48809}, {f: 2, c: 48850},
    {f: 2, c: 48853}, {f: 7, c: 48857}, {f: 2, c: 48865}, {f: 6, c: 48870},
    {f: 20, c: 48877}, {f: 6, c: 48898}, {f: 14, c: 48906}, 48922,
    {f: 34, c: 48926}, {f: 2, c: 48962}, {f: 3, c: 48965}, {f: 7, c: 48969},
    {f: 3, c: 48978}, {f: 62, c: 48982}, {f: 27, c: 49045}, {f: 20, c: 49073},
    {f: 6, c: 49094}, {f: 2, c: 49102}, {f: 3, c: 49105}, {f: 7, c: 49109},
    {f: 2, c: 49117}, 49120, {f: 90, c: 49122}, {f: 20, c: 49213},
    {f: 6, c: 49234}, {f: 3, c: 49241}, {f: 3, c: 49245}, {f: 7, c: 49249},
    {f: 38, c: 49258}, {f: 2, c: 49298}, {f: 3, c: 49301}, {f: 7, c: 49305},
    49314, 49316, {f: 6, c: 49318}, 49326, {f: 2, c: 49329}, {f: 5, c: 49335},
    49342, {f: 3, c: 49346}, {f: 2, c: 49350}, {f: 2, c: 49354},
    {f: 3, c: 49357}, {f: 7, c: 49361}, 49370, {f: 6, c: 49374},
    {f: 2, c: 49382}, {f: 3, c: 49385}, {f: 7, c: 49389}, 49398, 49400,
    {f: 6, c: 49402}, {f: 3, c: 49409}, {f: 3, c: 49413}, {f: 7, c: 49417},
    {f: 4, c: 49425}, {f: 6, c: 49430}, {f: 2, c: 49441}, 49445,
    {f: 4, c: 49448}, 49454, {f: 4, c: 49458}, 49463, {f: 2, c: 49466},
    {f: 3, c: 49469}, {f: 7, c: 49473}, 49482, {f: 6, c: 49486},
    {f: 2, c: 49494}, {f: 3, c: 49497}, {f: 7, c: 49501}, 49510,
    {f: 6, c: 49514}, {f: 3, c: 49521}, {f: 3, c: 49525}, {f: 12, c: 49529},
    {f: 6, c: 49542}, 49551, {f: 3, c: 49553}, 49557, {f: 5, c: 49559}, 49566,
    49568, {f: 3, c: 49570}, {f: 2, c: 49574}, {f: 2, c: 49578},
    {f: 3, c: 49581}, {f: 12, c: 49585}, {f: 6, c: 49598}, {f: 3, c: 49605},
    {f: 3, c: 49609}, {f: 7, c: 49613}, {f: 2, c: 49621}, {f: 7, c: 49625},
    {f: 3, c: 49633}, {f: 3, c: 49637}, {f: 7, c: 49641}, 49650,
    {f: 8, c: 49652}, {f: 2, c: 49662}, {f: 3, c: 49665}, {f: 7, c: 49669},
    49678, 49680, {f: 6, c: 49682}, {f: 2, c: 49690}, {f: 2, c: 49693},
    {f: 7, c: 49697}, 49706, 49708, 49710, 49712, 49715, {f: 19, c: 49717},
    {f: 7, c: 49737}, {f: 2, c: 49746}, {f: 3, c: 49749}, {f: 7, c: 49753},
    {f: 4, c: 49761}, {f: 6, c: 49766}, {f: 2, c: 49774}, {f: 3, c: 49777},
    {f: 7, c: 49781}, 49790, 49792, {f: 6, c: 49794}, {f: 6, c: 49802},
    {f: 7, c: 49809}, {f: 2, c: 49817}, 49820, {f: 6, c: 49822},
    {f: 2, c: 49830}, {f: 3, c: 49833}, {f: 6, c: 49838}, 49846, 49848,
    {f: 34, c: 49850}, {f: 2, c: 49886}, {f: 2, c: 49889}, {f: 6, c: 49893},
    49902, 49904, {f: 4, c: 49906}, 49911, 49914, {f: 3, c: 49917},
    {f: 7, c: 49921}, {f: 2, c: 49930}, {f: 5, c: 49934}, {f: 2, c: 49942},
    {f: 3, c: 49945}, {f: 7, c: 49949}, {f: 2, c: 49958}, {f: 27, c: 49962},
    {f: 34, c: 49990}, {f: 2, c: 50026}, {f: 3, c: 50029}, 50033,
    {f: 5, c: 50035}, {f: 2, c: 50042}, {f: 6, c: 50046}, {f: 3, c: 50053},
    {f: 3, c: 50057}, {f: 51, c: 50061}, {f: 23, c: 50113}, {f: 2, c: 50138},
    {f: 2, c: 50141}, 50145, {f: 5, c: 50147}, {f: 3, c: 50154},
    {f: 6, c: 50158}, {f: 2, c: 50166}, {f: 15, c: 50169}, {f: 7, c: 50185},
    {f: 19, c: 50193}, {f: 7, c: 50213}, {f: 3, c: 50221}, {f: 3, c: 50225},
    {f: 7, c: 50229}, {f: 10, c: 50238}, {f: 27, c: 50249}, {f: 2, c: 50278},
    {f: 3, c: 50281}, {f: 7, c: 50285}, {f: 3, c: 50294}, {f: 6, c: 50298},
    {f: 19, c: 50305}, {f: 7, c: 50325}, {f: 27, c: 50333}, {f: 3, c: 50361},
    {f: 44, c: 50365}, {f: 6, c: 50410}, {f: 2, c: 50418}, {f: 3, c: 50421},
    50425, {f: 4, c: 50427}, {f: 10, c: 50434}, {f: 3, c: 50445},
    {f: 3, c: 50449}, {f: 7, c: 50453}, {f: 11, c: 50461}, {f: 2, c: 50474},
    {f: 3, c: 50477}, {f: 7, c: 50481}, 50490, 50492, {f: 6, c: 50494},
    {f: 2, c: 50502}, 50507, {f: 4, c: 50511}, 50518, {f: 3, c: 50522}, 50527,
    {f: 2, c: 50530}, {f: 3, c: 50533}, {f: 7, c: 50537}, 50546,
    {f: 6, c: 50550}, {f: 2, c: 50558}, {f: 3, c: 50561}, {f: 2, c: 50565},
    {f: 4, c: 50568}, 50574, 50576, {f: 3, c: 50578}, 50582, {f: 3, c: 50585},
    {f: 3, c: 50589}, {f: 8, c: 50593}, {f: 10, c: 50602}, {f: 2, c: 50614},
    50618, {f: 5, c: 50623}, 50635, 50637, 50639, {f: 2, c: 50642},
    {f: 3, c: 50645}, {f: 7, c: 50649}, 50658, 50660, {f: 6, c: 50662}, 50671,
    {f: 3, c: 50673}, 50677, {f: 4, c: 50680}, {f: 3, c: 50690},
    {f: 3, c: 50697}, {f: 3, c: 50701}, {f: 7, c: 50705}, 50714,
    {f: 7, c: 50717}, {f: 2, c: 50726}, {f: 3, c: 50729}, 50735,
    {f: 2, c: 50737}, 50742, 50744, 50746, {f: 4, c: 50748}, {f: 2, c: 50754},
    {f: 3, c: 50757}, {f: 7, c: 50761}, 50770, {f: 6, c: 50774},
    {f: 2, c: 50782}, {f: 11, c: 50785}, {f: 2, c: 50797}, 50800,
    {f: 6, c: 50802}, {f: 2, c: 50810}, {f: 3, c: 50813}, {f: 7, c: 50817},
    50826, 50828, {f: 6, c: 50830}, {f: 2, c: 50838}, {f: 3, c: 50841},
    {f: 7, c: 50845}, 50854, 50856, {f: 6, c: 50858}, {f: 2, c: 50866},
    {f: 3, c: 50869}, {f: 5, c: 50875}, 50882, 50884, {f: 6, c: 50886},
    {f: 2, c: 50894}, {f: 3, c: 50897}, {f: 7, c: 50901}, {f: 2, c: 50910},
    {f: 6, c: 50914}, {f: 2, c: 50922}, {f: 3, c: 50925}, {f: 7, c: 50929},
    {f: 3, c: 50938}, {f: 6, c: 50942}, {f: 2, c: 50950}, {f: 3, c: 50953},
    {f: 7, c: 50957}, 50966, 50968, {f: 6, c: 50970}, {f: 2, c: 50978},
    {f: 3, c: 50981}, {f: 7, c: 50985}, 50994, 50996, 50998, {f: 4, c: 51000},
    {f: 2, c: 51006}, {f: 3, c: 51009}, {f: 5, c: 51013}, 51019, 51022, 51024,
    {f: 3, c: 51033}, {f: 3, c: 51037}, {f: 7, c: 51041}, {f: 2, c: 51049},
    {f: 8, c: 51052}, {f: 2, c: 51062}, {f: 3, c: 51065}, {f: 4, c: 51071},
    51078, {f: 3, c: 51083}, 51087, {f: 2, c: 51090}, 51093, 51097,
    {f: 5, c: 51099}, 51106, {f: 5, c: 51111}, {f: 2, c: 51118},
    {f: 3, c: 51121}, {f: 7, c: 51125}, 51134, {f: 6, c: 51138},
    {f: 2, c: 51146}, 51149, 51151, {f: 7, c: 51153}, {f: 4, c: 51161},
    {f: 6, c: 51166}, {f: 3, c: 51173}, {f: 3, c: 51177}, {f: 19, c: 51181},
    {f: 2, c: 51202}, {f: 3, c: 51205}, 51209, {f: 5, c: 51211}, 51218, 51220,
    {f: 5, c: 51223}, {f: 2, c: 51230}, {f: 3, c: 51233}, {f: 7, c: 51237},
    51246, 51248, {f: 6, c: 51250}, {f: 3, c: 51257}, {f: 3, c: 51261},
    {f: 7, c: 51265}, {f: 2, c: 51274}, {f: 6, c: 51278}, {f: 27, c: 51285},
    {f: 2, c: 51314}, {f: 3, c: 51317}, 51321, {f: 5, c: 51323}, 51330, 51332,
    {f: 3, c: 51336}, {f: 6, c: 51342}, {f: 8, c: 51349}, 51358, 51360,
    {f: 6, c: 51362}, {f: 19, c: 51369}, {f: 6, c: 51390}, {f: 3, c: 51397},
    {f: 3, c: 51401}, {f: 7, c: 51405}, 51414, 51416, {f: 6, c: 51418},
    {f: 2, c: 51426}, {f: 16, c: 51429}, {f: 6, c: 51446}, {f: 2, c: 51454},
    {f: 3, c: 51457}, {f: 5, c: 51463}, 51470, 51472, {f: 6, c: 51474},
    {f: 19, c: 51481}, {f: 7, c: 51501}, {f: 27, c: 51509}, {f: 2, c: 51538},
    {f: 3, c: 51541}, {f: 7, c: 51545}, 51554, {f: 8, c: 51556},
    {f: 3, c: 51565}, {f: 3, c: 51569}, {f: 7, c: 51573}, {f: 11, c: 51581},
    {f: 2, c: 51594}, {f: 3, c: 51597}, {f: 7, c: 51601}, 51610, 51612,
    {f: 34, c: 51614}, {f: 2, c: 51650}, {f: 2, c: 51653}, 51657,
    {f: 5, c: 51659}, 51666, 51668, {f: 2, c: 51671}, 51675, {f: 2, c: 51678},
    51681, 51683, {f: 2, c: 51685}, {f: 4, c: 51688}, 51694, {f: 6, c: 51698},
    {f: 2, c: 51706}, {f: 3, c: 51709}, {f: 7, c: 51713}, 51722,
    {f: 6, c: 51726}, {f: 3, c: 51733}, {f: 16, c: 51737}, {f: 34, c: 51754},
    {f: 2, c: 51790}, {f: 3, c: 51793}, {f: 7, c: 51797}, 51806,
    {f: 6, c: 51810}, {f: 20, c: 51817}, {f: 6, c: 51838}, {f: 19, c: 51845},
    {f: 35, c: 51865}, {f: 2, c: 51902}, {f: 3, c: 51905}, {f: 7, c: 51909},
    51918, 51920, 51922, {f: 4, c: 51924}, {f: 6, c: 51930}, {f: 11, c: 51937},
    {f: 7, c: 51949}, {f: 19, c: 51957}, {f: 7, c: 51977}, {f: 3, c: 51985},
    {f: 3, c: 51989}, {f: 7, c: 51993}, {f: 31, c: 52002}, {f: 6, c: 52034},
    {f: 2, c: 52042}, {f: 3, c: 52045}, {f: 7, c: 52049}, {f: 3, c: 52058},
    {f: 6, c: 52062}, {f: 19, c: 52069}, {f: 34, c: 52090}, {f: 27, c: 52125},
    {f: 27, c: 52153}, {f: 15, c: 52181}, {f: 2, c: 52197}, 52200,
    {f: 34, c: 52202}, {f: 2, c: 52238}, {f: 3, c: 52241}, {f: 7, c: 52245},
    {f: 3, c: 52254}, {f: 4, c: 52259}, {f: 2, c: 52266}, 52269, 52271,
    {f: 7, c: 52273}, 52282, {f: 5, c: 52287}, {f: 2, c: 52294},
    {f: 3, c: 52297}, {f: 7, c: 52301}, 52310, {f: 6, c: 52314},
    {f: 3, c: 52321}, 52325, 52327, {f: 7, c: 52329}, {f: 4, c: 52337},
    {f: 34, c: 52342}, {f: 2, c: 52378}, {f: 3, c: 52381}, {f: 7, c: 52385},
    52394, {f: 6, c: 52398}, {f: 2, c: 52406}, {f: 3, c: 52409},
    {f: 7, c: 52413}, 52422, 52424, {f: 6, c: 52426}, {f: 3, c: 52433},
    {f: 15, c: 52437}, {f: 7, c: 52453}, {f: 3, c: 52461}, {f: 16, c: 52465},
    {f: 6, c: 52482}, {f: 2, c: 52490}, {f: 3, c: 52493}, {f: 7, c: 52497},
    52506, 52508, {f: 6, c: 52510}, {f: 3, c: 52517}, {f: 3, c: 52521},
    {f: 12, c: 52525}, {f: 34, c: 52538}, {f: 3, c: 52573}, {f: 3, c: 52577},
    {f: 7, c: 52581}, 52590, 52592, {f: 6, c: 52594}, {f: 15, c: 52601},
    {f: 11, c: 52617}, {f: 2, c: 52630}, {f: 3, c: 52633}, {f: 7, c: 52637},
    52646, 52648, {f: 6, c: 52650}, {f: 19, c: 52657}, {f: 7, c: 52677},
    {f: 3, c: 52685}, {f: 23, c: 52689}, {f: 3, c: 52713}, {f: 3, c: 52717},
    {f: 7, c: 52721}, 52730, 52732, {f: 6, c: 52734}, {f: 3, c: 52741},
    {f: 3, c: 52745}, {f: 7, c: 52749}, {f: 4, c: 52757}, {f: 6, c: 52762},
    {f: 2, c: 52770}, {f: 3, c: 52773}, {f: 7, c: 52777}, 52786, 52788,
    {f: 34, c: 52790}, {f: 2, c: 52826}, {f: 2, c: 52829}, {f: 6, c: 52834},
    52842, 52844, {f: 6, c: 52846}, {f: 2, c: 52854}, {f: 3, c: 52857},
    {f: 7, c: 52861}, 52870, 52872, {f: 6, c: 52874}, {f: 2, c: 52882},
    {f: 3, c: 52885}, {f: 7, c: 52889}, 52898, {f: 6, c: 52902},
    {f: 19, c: 52910}, {f: 34, c: 52930}, {f: 2, c: 52966}, {f: 2, c: 52969},
    {f: 7, c: 52973}, 52982, {f: 6, c: 52986}, {f: 2, c: 52994},
    {f: 3, c: 52997}, {f: 7, c: 53001}, 53010, 53012, {f: 6, c: 53014},
    {f: 3, c: 53021}, {f: 3, c: 53025}, {f: 7, c: 53029}, 53038,
    {f: 6, c: 53042}, {f: 27, c: 53049}, {f: 2, c: 53078}, {f: 3, c: 53081},
    {f: 7, c: 53085}, 53094, 53096, {f: 6, c: 53098}, {f: 2, c: 53106},
    {f: 3, c: 53109}, {f: 7, c: 53113}, {f: 4, c: 53121}, {f: 6, c: 53126},
    {f: 20, c: 53133}, {f: 6, c: 53154}, {f: 7, c: 53161}, {f: 19, c: 53169},
    {f: 27, c: 53189}, {f: 2, c: 53218}, {f: 3, c: 53221}, {f: 7, c: 53225},
    53234, 53236, {f: 6, c: 53238}, {f: 3, c: 53245}, {f: 3, c: 53249},
    {f: 12, c: 53253}, {f: 6, c: 53266}, {f: 20, c: 53273}, {f: 6, c: 53294},
    {f: 2, c: 53302}, {f: 3, c: 53305}, {f: 7, c: 53309}, 53318, 53320,
    {f: 6, c: 53322}, {f: 3, c: 53329}, {f: 3, c: 53333}, {f: 7, c: 53337},
    {f: 11, c: 53345}, {f: 2, c: 53358}, {f: 3, c: 53361}, {f: 7, c: 53365},
    {f: 3, c: 53374}, {f: 34, c: 53378}, {f: 2, c: 53414}, {f: 3, c: 53417},
    {f: 7, c: 53421}, 53430, 53432, {f: 6, c: 53434}, {f: 2, c: 53442},
    {f: 3, c: 53445}, {f: 6, c: 53450}, 53458, {f: 6, c: 53462},
    {f: 2, c: 53470}, {f: 3, c: 53473}, {f: 7, c: 53477}, 53486,
    {f: 6, c: 53490}, {f: 20, c: 53497}, {f: 34, c: 53518}, {f: 2, c: 53554},
    {f: 3, c: 53557}, 53561, {f: 5, c: 53563}, 53570, {f: 6, c: 53574},
    {f: 2, c: 53582}, {f: 3, c: 53585}, {f: 7, c: 53589}, 53598, 53600,
    {f: 6, c: 53602}, {f: 3, c: 53609}, {f: 15, c: 53613}, {f: 7, c: 53629},
    {f: 3, c: 53637}, {f: 23, c: 53641}, {f: 2, c: 53666}, {f: 3, c: 53669},
    {f: 7, c: 53673}, 53682, 53684, {f: 4, c: 53686}, 53691, {f: 3, c: 53693},
    {f: 23, c: 53697}, {f: 27, c: 53721}, {f: 3, c: 53749}, {f: 14, c: 53753},
    53768, {f: 6, c: 53770}, {f: 27, c: 53777}, {f: 2, c: 53806},
    {f: 3, c: 53809}, {f: 7, c: 53813}, 53822, 53824, {f: 6, c: 53826},
    {f: 19, c: 53833}, {f: 7, c: 53853}, {f: 27, c: 53861}, {f: 2, c: 53890},
    {f: 3, c: 53893}, {f: 7, c: 53897}, {f: 3, c: 53906}, {f: 6, c: 53910},
    {f: 3, c: 53917}, {f: 3, c: 53921}, {f: 7, c: 53925}, {f: 4, c: 53933},
    {f: 6, c: 53938}, {f: 2, c: 53946}, {f: 2, c: 53949}, 53953,
    {f: 5, c: 53955}, 53962, {f: 8, c: 53964}, {f: 3, c: 53973},
    {f: 3, c: 53977}, {f: 7, c: 53981}, {f: 10, c: 53990}, {f: 2, c: 54002},
    {f: 3, c: 54005}, {f: 7, c: 54009}, 54018, 54020, {f: 6, c: 54022}, 54031,
    {f: 3, c: 54033}, 54037, {f: 5, c: 54039}, 54046, {f: 3, c: 54050},
    {f: 2, c: 54054}, {f: 2, c: 54058}, {f: 3, c: 54061}, {f: 7, c: 54065},
    54074, {f: 6, c: 54078}, {f: 54, c: 54086}, {f: 2, c: 54142},
    {f: 3, c: 54145}, {f: 7, c: 54149}, 54158, {f: 6, c: 54162},
    {f: 2, c: 54170}, {f: 3, c: 54173}, {f: 7, c: 54177}, 54186, 54188,
    {f: 6, c: 54190}, {f: 3, c: 54197}, {f: 3, c: 54201}, {f: 7, c: 54205},
    {f: 2, c: 54214}, {f: 6, c: 54218}, {f: 7, c: 54225}, {f: 8, c: 54233},
    54242, {f: 8, c: 54244}, {f: 2, c: 54254}, {f: 3, c: 54257},
    {f: 7, c: 54261}, 54270, 54272, {f: 6, c: 54274}, {f: 20, c: 54281},
    {f: 34, c: 54302}, {f: 3, c: 54337}, {f: 23, c: 54341}, {f: 3, c: 54365},
    {f: 3, c: 54369}, {f: 8, c: 54373}, 54382, {f: 8, c: 54384},
    {f: 2, c: 54394}, {f: 2, c: 54397}, 54401, {f: 5, c: 54403}, 54410, 54412,
    {f: 6, c: 54414}, {f: 20, c: 54421}, {f: 34, c: 54442}, {f: 3, c: 54477},
    {f: 3, c: 54481}, {f: 7, c: 54485}, {f: 2, c: 54493}, {f: 8, c: 54496},
    {f: 3, c: 54505}, {f: 3, c: 54509}, {f: 7, c: 54513}, {f: 2, c: 54521},
    54524, {f: 6, c: 54526}, {f: 3, c: 54533}, {f: 3, c: 54537},
    {f: 7, c: 54541}, 54550, {f: 36, c: 54552}, {f: 2, c: 54590},
    {f: 3, c: 54593}, {f: 7, c: 54597}, 54606, 54608, {f: 6, c: 54610},
    {f: 2, c: 54618}, {f: 3, c: 54621}, {f: 4, c: 54625}, {f: 2, c: 54630},
    54634, 54636, {f: 6, c: 54638}, {f: 2, c: 54646}, {f: 3, c: 54649},
    {f: 7, c: 54653}, 54662, {f: 6, c: 54666}, {f: 20, c: 54673},
    {f: 34, c: 54694}, {f: 2, c: 54730}, {f: 3, c: 54733}, 54737,
    {f: 5, c: 54739}, 54746, 54748, {f: 6, c: 54750}, {f: 2, c: 54758},
    {f: 3, c: 54761}, {f: 7, c: 54765}, 54774, 54776, {f: 6, c: 54778},
    {f: 2, c: 54786}, {f: 3, c: 54789}, {f: 7, c: 54793}, 54802,
    {f: 6, c: 54806}, {f: 3, c: 54813}, {f: 3, c: 54817}, {f: 8, c: 54821},
    {f: 10, c: 54830}, {f: 2, c: 54842}, {f: 3, c: 54845}, {f: 4, c: 54849},
    {f: 2, c: 54854}, 54858, 54860, {f: 3, c: 54862}, {f: 2, c: 54866},
    {f: 2, c: 54870}, {f: 3, c: 54873}, {f: 10, c: 54877}, 54888,
    {f: 6, c: 54890}, {f: 2, c: 54898}, {f: 14, c: 54901}, 54916,
    {f: 6, c: 54918}, {f: 2, c: 54926}, {f: 3, c: 54929}, {f: 8, c: 54933},
    54942, 54944, {f: 6, c: 54946}, {f: 3, c: 54953}, {f: 3, c: 54957},
    {f: 8, c: 54961}, 54970, {f: 8, c: 54972}, {f: 2, c: 54982},
    {f: 3, c: 54985}, {f: 4, c: 54989}, {f: 2, c: 54994}, {f: 2, c: 54997},
    55000, {f: 6, c: 55002}, {f: 3, c: 55009}, {f: 3, c: 55013},
    {f: 7, c: 55017}, {f: 4, c: 55025}, {f: 6, c: 55030}, {f: 2, c: 55038},
    {f: 3, c: 55041}, {f: 12, c: 55045}, {f: 6, c: 55058}, {f: 2, c: 55066},
    {f: 3, c: 55069}, {f: 7, c: 55073}, 55082, 55084, {f: 6, c: 55086},
    {f: 2, c: 55094}, {f: 3, c: 55097}, {f: 7, c: 55101}, {f: 2, c: 55109},
    55112, {f: 6, c: 55114}, {f: 2, c: 55122}, 55125, {f: 6, c: 55130}, 55138,
    55140, {f: 3, c: 55142}, {f: 2, c: 55146}, {f: 3, c: 55149},
    {f: 3, c: 55153}, {f: 7, c: 55157}, {f: 3, c: 55166}, {f: 6, c: 55170},
    {f: 2, c: 55178}, {f: 3, c: 55181}, {f: 7, c: 55185}, 55194, 55196,
    {f: 6, c: 55198}],
  'Adobe-CNS1': [{f: 95, c: 32}, {s: 3}, 12288, 65292, {f: 2, c: 12289}, 65294,
    8226, 65307, 65306, 65311, 65281, 65072, 8230, 8229, 65104, 65380, 65106,
    183, {f: 4, c: 65108}, 65372, 8211, 65073, 8212, {s: 4}, {f: 2, c: 65288},
    {f: 2, c: 65077}, 65371, 65373, {f: 2, c: 65079}, {f: 2, c: 12308},
    {f: 2, c: 65081}, {f: 2, c: 12304}, {f: 2, c: 65083}, {f: 2, c: 12298},
    {f: 2, c: 65085}, {f: 2, c: 12296}, {f: 2, c: 65087}, {f: 2, c: 12300},
    {f: 2, c: 65089}, {f: 2, c: 12302}, {f: 2, c: 65091}, {f: 6, c: 65113},
    {f: 2, c: 8216}, {f: 2, c: 8220}, {f: 2, c: 12317}, 8245, 8242, 65283,
    65286, 65290, 8251, 167, 12291, 9675, 9679, 9651, 9650, 9678, 9734, 9733,
    9671, 9670, 9633, 9632, 9661, 9660, 12963, 8453, 8254, 0, 65343, 0,
    {f: 2, c: 65097}, {f: 2, c: 65101}, {f: 2, c: 65099}, {f: 3, c: 65119},
    65291, 65293, 215, 247, 177, 8730, 65308, 65310, 65309, {f: 2, c: 8806},
    8800, 8734, 8786, 8801, {f: 5, c: 65122}, 8764, {f: 2, c: 8745}, 8869,
    8736, 8735, 8895, 13266, 13265, 8747, 8750, 8757, 8756, 9792, 9794, 9793,
    9737, 8593, 8595, 8594, 8592, {f: 2, c: 8598}, 8601, 8600, 8741, 8739, 0,
    0, 65295, 65340, 65284, 165, 12306, {f: 2, c: 162}, 65285, 65312, 8451,
    8457, {f: 3, c: 65129}, 13269, {f: 3, c: 13212}, 13262, 13217,
    {f: 2, c: 13198}, 13252, 176, [20825, 58834], [20827, 58835],
    [20830, 58837], [20829, 58836], 20833, 20835, 21991, [29929, 58044],
    [31950, 58191], {f: 8, c: 9601}, 9615, 9614, 9613, 9612, 9611, 9610, 9609,
    9532, 9524, 9516, 9508, 9500, 9620, 9472, 9474, 9621, 9484, 9488, 9492,
    9496, {f: 2, c: 9581}, 9584, 9583, 9552, 9566, 9578, 9569, {f: 2, c: 9698},
    9701, 9700, {f: 3, c: 9585}, {f: 10, c: 65296}, {f: 10, c: 8544},
    {f: 9, c: 12321}, 0, [21316, 57443], 0, {f: 26, c: 65313},
    {f: 26, c: 65345}, {f: 17, c: 913}, {f: 7, c: 931}, {f: 17, c: 945},
    {f: 7, c: 963}, {f: 37, c: 12549}, 729, 714, 711, 715, [9312, 63153],
    [9313, 63154], [9314, 63155], [9315, 63156], [9316, 63157], [9317, 63158],
    [9318, 63159], [9319, 63160], [9320, 63161], [9321, 63162], [9332, 63163],
    [9333, 63164], [9334, 63165], [9335, 63166], [9336, 63167], [9337, 63168],
    [9338, 63169], [9339, 63170], [9340, 63171], [9341, 63172], [8560, 63173],
    [8561, 63174], [8562, 63175], [8563, 63176], [8564, 63177], [8565, 63178],
    [8566, 63179], [8567, 63180], [8568, 63181], [8569, 63182], [12033, 20008],
    [12034, 20022, 63183], [12035, 20031, 63184], [12037, 20101, 63185],
    [12039, 20128, 63186], [12044, 20866, 63187], [12045, 20886, 63188],
    [12046, 20907, 63189], [12051, 21241, 63190], [12054, 21304, 63191],
    [12057, 21353, 63192], [12059, 21430, 63193],
    [12065, 12066, 22786, 22794, 63194], [12071, 23424, 63195],
    [12078, 24027, 63196], [12083, 24186, 63197], [12084, 24191, 63198],
    [12085, 24308], [12089, 24400, 63200], [12090, 24417, 63201],
    [12097, 25908, 63202], [12102, 26080], [12135, 30098, 63204],
    [12136, 30326], [12193, 36789, 63206], [12202, 38582], {f: 32, c: 9216},
    9249, [12032, 19968], [12036, 20057], 19969, 19971, 20035, 20061, 20102,
    [12038, 20108], [12040, 20154], [12041, 20799], [12042, 20837],
    [12043, 20843], [12047, 20960], [12049, 20992], 20993, [12050, 21147],
    [12052, 21269], [12055, 21313], [12056, 21340], [12060, 21448], 19977,
    19979, 19976, 19978, 20011, 20024, 20961, 20037, 20040, 20063, 20062,
    20110, 20129, [20800, 64012], 20995, 21242, 21315, 21449, [12061, 21475],
    [12063, 22303], [12064, 22763], [12067, 22805], [12068, 22823],
    [12069, 22899], [12070, 23376], 23377, 23379, [12072, 23544],
    [12073, 23567], [12074, 23586], [12075, 23608], [12077, 23665], 24029,
    [12079, 24037], [12080, 24049], {f: 2, c: 24050}, [12081, 24062],
    [12082, 24178], [12086, 24318], [12087, 24331], [12088, 24339], 25165,
    19985, 19984, 19981, 20013, 20016, 20025, 20043, 23609, 20104, 20113,
    20117, 20114, 20116, 20130, 20161, 20160, 20163, {f: 2, c: 20166}, 20173,
    {f: 2, c: 20170}, 20164, 20803, 20801, 20839, {f: 2, c: 20845}, 20844,
    20887, 20982, {f: 3, c: 20998}, 21243, {f: 2, c: 21246}, 21270, 21305,
    21320, 21319, 21317, 21342, 21380, 21451, 21450, 21453, 22764, 22825,
    22827, 22826, 22829, 23380, 23569, 23588, 23610, 23663, 24052, 24187,
    24319, {f: 2, c: 24340}, [12092, 24515], [12093, 25096], [12094, 25142],
    [12095, 25163], 25166, [12096, 25903], [12098, 25991], [12099, 26007],
    [12100, 26020], [12101, 26041], [12103, 26085], [12104, 26352],
    [12105, 26376], [12106, 26408], [12107, 27424], [12108, 27490],
    [12109, 27513], [12111, 27595], [12112, 27604], [12113, 27611],
    [12114, 27663], [12116, 27700], [12117, 28779], [12118, 29226],
    [12119, 29238], [12120, 29243], [12122, 29255], [12123, 29273],
    [12124, 29275], [12125, 29356], 29579, 19993, 19990, 19989, 19988, 19992,
    20027, 20045, 20047, 20046, 20197, 20184, {f: 4, c: 20180},
    {f: 2, c: 20195}, 20185, 20190, 20805, 20804, {f: 2, c: 20873}, 20908,
    {f: 2, c: 20985}, 20984, 21002, 21152, 21151, [21253, 57435], 21254, 21271,
    21277, 20191, 21322, 21321, 21345, 21344, 21359, 21358, 21435, 21487,
    21476, 21491, 21484, 21486, 21481, 21480, 21500, 21496, 21493, 21483,
    21478, 21482, 21490, 21489, 21488, 21477, 21485, 21499, 22235, 22234,
    22806, 22830, 22833, 22900, 22902, 23381, 23427, 23612, 24040, 24039,
    24038, {f: 2, c: 24066}, 24179, 24188, 24321, 24344, 24343, 24517, 25098,
    {f: 2, c: 25171}, 25170, 25169, 26021, 26086, 26414, 26412,
    {f: 2, c: 26410}, 26413, 27491, 27597, 27665, 27664, 27704, 27713, 27712,
    27710, 29359, [12126, 29572], [12127, 29577], [12128, 29916],
    [12129, 29926], [12130, 29976], [12131, 29983], [12132, 29992], 29993,
    [12133, 30000], {f: 3, c: 30001}, [12134, 30091], [12137, 30333],
    [12138, 30382], [12139, 30399], [12140, 30446], [12141, 30683],
    [12142, 30690], [12143, 30707], [12144, 31034], [12146, 31166],
    [12147, 31348], [12148, 31435], {f: 2, c: 19998}, {f: 2, c: 20050}, 20073,
    20121, 20132, 20134, 20133, 20223, 20233, 20249, 20234, 20245, 20237,
    {f: 2, c: 20240}, 20239, 20210, 20214, 20219, 20208, 20211, 20221, 20225,
    20235, 20809, 20807, 20806, 20808, 20840, 20849, 20877, 20912, 21015,
    {f: 2, c: 21009}, 21006, 21014, 21155, 21256, 21281, 21280,
    {f: 2, c: 21360}, 21513, 21519, 21516, 21514, 21520, 21505, 21515, 21508,
    21521, 21517, 21512, 21507, 21518, 21510, 21522, 22240, 22238, 22237,
    22323, 22320, 22312, 22317, 22316, 22319, 22313, {f: 2, c: 22809},
    {f: 2, c: 22839}, 22916, 22904, 22915, 22909, 22905, 22914, 22913,
    {f: 2, c: 23383}, {f: 2, c: 23431}, 23429, 23433, 23546, 23574, 23673,
    24030, 24070, 24182, 24180, 24335, 24347, 24537, 24534, 25102,
    {f: 2, c: 25100}, 25104, 25187, 25179, 25176, 25910, 26089, 26088,
    {f: 2, c: 26092}, {f: 2, c: 26354}, 26377, 26429, 26420, 26417, 26421,
    27425, 27492, 27515, 27670, 27741, 27735, 27737, {f: 2, c: 27743}, 27728,
    27733, 27745, 27739, {f: 2, c: 27725}, 28784, 29279, 29277, 30334,
    [12149, 31481], [12150, 31859], [12151, 31992], [12152, 32566],
    [12154, 32650], [12155, 32701], [12156, 32769], 32771, [12157, 32780],
    [12158, 32786], [12159, 32819], [12160, 32895], [12161, 32905],
    {f: 2, c: 32907}, [12162, 33251], [12163, 33258], [12164, 33267],
    [12165, 33276], [12166, 33292], [12167, 33307], [12168, 33311],
    [12169, 33390], [12170, 33394], 33406, [12173, 34411], [12174, 34880],
    [12175, 34892], [12176, 34915], 35199, 38433, 20018, 20136, 20301, 20303,
    20295, 20311, 20318, 20276, 20315, 20309, 20272, {f: 2, c: 20304}, 20285,
    20282, 20280, 20291, 20308, 20284, 20294, 20323, 20316, 20320, 20271,
    20302, 20278, 20313, 20317, 20296, 20314, 20812, 20811, 20813, 20853,
    {f: 2, c: 20918}, 21029, 21028, {f: 2, c: 21033}, 21032, 21163,
    {f: 2, c: 21161}, 21164, 21283, 21363, 21365, 21533, 21549, 21534, 21566,
    21542, 21582, 21543, 21574, 21571, 21555, 21576, 21570, 21531, 21545,
    21578, 21561, 21563, 21560, 21550, {f: 2, c: 21557}, 21536, 21564, 21568,
    21553, 21547, 21535, 21548, 22250, 22256, 22244, 22251, 22346, 22353,
    22336, 22349, 22343, 22350, 22334, 22352, 22351, 22331, 22767, 22846,
    22941, 22930, 22952, 22942, 22947, 22937, 22934, 22925, 22948, 22931,
    22922, 22949, 23389, 23388, {f: 2, c: 23386}, 23436, 23435, 23439, 23596,
    {f: 2, c: 23616}, 23615, 23614, {f: 2, c: 23696}, 23700, 23692, 24043,
    24076, 24207, 24199, 24202, 24311, 24324, 24351, 24420, 24418, 24439,
    24441, 24536, 24524, 24535, 24525, 24561, 24555, 24568, 24554, 25106,
    25105, 25220, 25239, 25238, 25216, 25206, 25225, 25197, 25226, 25212,
    25214, 25209, 25203, 25234, 25199, 25240, 25198, 25237, 25235, 25233,
    25222, 25913, 25915, 25912, 26097, 26356, 26463, {f: 4, c: 26446}, 26460,
    26454, [26462, 57801], 26441, 26438, 26464, 26451, 26455, 27493, 27599,
    27714, 27742, 27801, 27777, {f: 2, c: 27784}, 27781, 27803, 27754, 27770,
    27792, 27760, 27788, 27752, 27798, 27794, 27773, 27779, 27762, 27774,
    27764, 27782, 27766, 27789, 27796, 27800, 27778, 28790, {f: 2, c: 28796},
    28792, 29282, 29281, 29280, 29380, 29378, 29590, 29996, 29995,
    {f: 2, c: 30007}, 30338, 30447, 30691, 31169, 31168, 31167, 31350, 31995,
    32597, 32918, 32915, 32925, 32920, 32923, 32922, 32946, 33391, 33426,
    33419, 33421, [12178, 35211], [12179, 35282], [12180, 35328],
    [12181, 35895], [12182, 35910], [12183, 35925], [12185, 35997],
    [12186, 36196], [12187, 36208], [12188, 36275], [12189, 36523],
    [12190, 36554], [12191, 36763], [12192, 36784], 36802, 36806, 36805, 36804,
    24033, [12194, 37009], 37026, 37034, 37030, 37027, [12195, 37193],
    [12196, 37318], [12197, 37324], 38450, 38446, 38449, 38442, 38444, 20006,
    20054, 20083, 20107, 20123, 20126, {f: 2, c: 20139}, 20335, 20381, 20365,
    20339, 20351, 20332, 20379, 20363, 20358, 20355, 20336, 20341, 20360,
    20329, 20347, 20374, 20350, 20367, 20369, 20346, 20820, 20818, 20821,
    20841, 20855, 20854, 20856, 20925, 20989, 21051, 21048, 21047, 21050,
    21040, 21038, 21046, 21057, 21182, 21179, 21330, 21332, 21331, 21329,
    21350, {f: 3, c: 21367}, 21462, 21460, 21463, 21619, 21621, 21654, 21624,
    21653, 21632, 21627, 21623, 21636, 21650, 21638, 21628, 21648, 21617,
    21622, 21644, 21658, 21602, 21608, 21643, 21629, 21646, 22266, 22403,
    22391, 22378, 22377, 22369, 22374, 22372, 22396, 22812, 22857,
    {f: 2, c: 22855}, 22852, 22868, 22974, 22971, 22996, 22969, 22958, 22993,
    22982, 22992, 22989, 22987, 22995, 22986, 22959, 22963, 22994, 22981,
    23391, 23396, 23395, 23447, 23450, 23448, 23452, 23449, 23451, 23578,
    23624, {f: 2, c: 23621}, 23735, 23713, 23736, 23721, 23723, 23729, 23731,
    24088, 24090, 24086, 24085, 24091, 24081, 24184, 24218, 24215, 24220,
    {f: 2, c: 24213}, 24310, {f: 2, c: 24358}, 24361, {f: 2, c: 24448}, 24447,
    24444, 24541, 24544, 24573, 24565, 24575, 24591, 24596, 24623, 24629,
    24598, 24618, 24597, 24609, 24615, 24617, 24619, 24603, 25110, 25109,
    25151, 25150, 25152, 25215, 25289, 25292, 25284, 25279, 25282, 25273,
    25298, 25307, 25259, {f: 2, c: 25299}, 25291, 25288, 25256, 25277, 25276,
    [25296, 60582], 25305, 25287, 25293, 25269, 25306, 25265, 25304,
    {f: 2, c: 25302}, 25286, 25260, [25294, 61010], 25918, 26023, 26044, 26106,
    26132, 26131, 26124, 26118, 26114, 26126, 26112, 26127, 26133, 26122,
    26119, 26381, 26379, 26477, 26507, 26517, 26481, 26524, 26483, 26487,
    26503, 26525, 26519, {f: 2, c: 26479}, 26495, 26505, 26494, 26512, 26485,
    26522, 26515, 26492, 26474, 26482, 27427, {f: 2, c: 27494}, 27519, 27667,
    27675, 27875, 27880, 27891, 27825, 27852, 27877, 27827, {f: 2, c: 27837},
    27836, 27874, 27819, 27861, 27859, 27832, 27844, 27833, 27841, 27822,
    27863, 27845, 27889, 27839, 27835, 27873, 27867, 27850, 27820, 27887,
    27868, 27862, 27872, 28821, 28814, 28818, 28810, 28825, {f: 2, c: 29228},
    29240, 29256, 29287, 29289, 29376, 29390, 29401, 29399, 29392, 29609,
    29608, 29599, 29611, 29605, 30013, 30109, {f: 2, c: 30105}, 30340, 30402,
    30450, 30452, 30693, 30717, 31038, {f: 2, c: 31040}, 31177, 31176, 31354,
    31353, 31482, 31998, 32596, 32652, 32651, [32773, 58236], 32954, 32933,
    32930, 32945, 32929, 32939, 32937, 32948, 32938, 32943, 33253, 33278,
    33293, 33459, 33437, 33433, 33453, 33469, 33439, 33465, 33457, 33452,
    33445, 33455, 33464, 33443, 33456, 33470, 33463, 34382, 34417, 21021,
    34920, 36555, 36814, 36820, 36817, 37045, 37048, 37041, 37046, 37319,
    [12198, 37329], [12199, 38263], [12200, 38272], [12201, 38428], 38464,
    38463, 38459, 38468, 38466, [12203, 38585], [12204, 38632], 38738,
    [12206, 38750], 20127, {f: 2, c: 20141}, 20449, 20405, 20399, 20415, 20448,
    20433, 20431, 20445, 20419, 20406, 20440, 20447, 20426, 20439, 20398,
    20432, 20420, 20418, 20442, 20430, 20446, 20407, 20823, 20882, 20881,
    20896, 21070, 21059, 21066, 21069, 21068, 21067, 21063, 21191, 21193,
    21187, 21185, 21261, 21335, 21371, 21402, 21467, 21676, 21696, 21672,
    21710, 21705, 21688, 21670, 21683, 21703, 21698, 21693, 21674, 21697,
    21700, 21704, 21679, 21675, 21681, 21691, 21673, 21671, 21695, 22271,
    22402, 22411, 22432, 22435, 22434, 22478, 22446, 22419, 22869, 22865,
    22863, 22862, 22864, 23004, 23000, 23039, 23011, 23016, 23043, 23013,
    23018, 23002, 23014, 23041, 23035, 23401, 23459, 23462, 23460, 23458,
    23461, 23553, {f: 2, c: 23630}, 23629, 23627, 23769, 23762, 24055, 24093,
    24101, 24095, 24189, 24224, 24230, 24314, 24328, 24365, 24421, 24456,
    24453, {f: 2, c: 24458}, 24455, 24460, 24457, 24594, 24605, 24608, 24613,
    24590, 24616, 24653, 24688, 24680, [24674, 60712], 24646, 24643, 24684,
    24683, 24682, 24676, 25153, 25308, 25366, 25353, 25340, 25325, 25345,
    25326, 25341, 25351, 25329, 25335, 25327, 25324, 25342, 25332, 25361,
    25346, 25919, 25925, 26027, 26045, 26082, 26149, 26157, 26144, 26151,
    26159, 26143, 26152, 26161, 26148, 26359, 26623, 26579, 26609, 26580,
    26576, 26604, 26550, 26543, 26613, 26601, 26607, 26564, 26577, 26548,
    26586, 26597, 26552, 26575, 26590, 26611, 26544, 26585, 26594, 26589,
    26578, 27498, 27523, 27526, 27573, 27602, 27607, 27679, 27849, 27915,
    27954, 27946, 27969, 27941, 27916, 27953, 27934, 27927, 27963,
    {f: 2, c: 27965}, 27958, 27931, 27893, 27961, 27943, 27960, 27945, 27950,
    27957, 27918, 27947, 28843, 28858, 28851, 28844, 28847, 28845, 28856,
    28846, 28836, 29232, 29298, 29295, 29300, 29417, {f: 2, c: 29408}, 29623,
    29642, 29627, 29618, 29645, 29632, 29619, 29978, 29997, 30031, 30028,
    30030, 30027, 30123, {f: 2, c: 30116}, {f: 2, c: 30114}, 30328,
    {f: 3, c: 30342}, 30408, 30406, 30403, 30405, 30465, 30457, 30456, 30473,
    30475, 30462, 30460, 30471, 30684, 30722, 30740, {f: 2, c: 30732}, 31046,
    31049, 31048, 31047, {f: 2, c: 31161}, {f: 2, c: 31185}, 31179, 31359,
    31361, 31487, 31485, 31869, 32002, 32005, 32000, 32009, 32007, 32004,
    32006, 32568, 32654, 32703, 32784, 32781, 32785, 32822, 32982, 32997,
    32986, {f: 2, c: 32963}, 32972, 32993, 32987, 32974, 32990, 32996, 32989,
    33268, 33314, 33511, 33539, 33541, 33507, 33499, 33510, 33540, 33509,
    33538, 33545, 33490, 33495, 33521, 33537, 33500, 33492, 33489, 33502,
    33491, 33503, 33519, 33542, 34384, 34425, 34427, 34426, 34893, 34923,
    35201, 35284, 35336, {f: 2, c: 35330}, 35998, 36000, 36212, 36211, 36276,
    36557, 36556, 36848, 36838, 36834, 36842, 36837, 36845, 36843, 36836,
    36840, 37066, 37070, 37057, 37059, 37195, 37194, 37325, 38274, 38480,
    {f: 3, c: 38475}, [12207, 38754], [12208, 38761], [12209, 38859],
    [12210, 38893], [12211, 38899], [12212, 38913], [12213, 39080],
    [12214, 39131], [12215, 39135], [12216, 39318], [12217, 39321], 20056,
    20147, {f: 2, c: 20492}, 20515, 20463, 20518, 20517, 20472, [20521, 57375],
    20502, 20486, 20540, 20511, 20506, 20498, 20497, 20474, 20480, 20500,
    20520, 20465, 20513, 20491, 20505, 20504, 20467, 20462, 20525, 20522,
    20478, 20523, 20489, 20860, {f: 2, c: 20900}, 20898, 20941, 20940, 20934,
    20939, 21078, 21084, 21076, 21083, 21085, 21290, [21375, 57459], 21407,
    21405, 21471, 21736, 21776, 21761, 21815, 21756, 21733, 21746, 21766,
    21754, 21780, 21737, 21741, 21729, 21769, 21742, 21738, 21734, 21799,
    21767, 21757, 21775, {f: 2, c: 22275}, 22466, 22484, 22475, 22467, 22537,
    22799, {f: 2, c: 22871}, 22874, 23057, 23064, 23068, 23071, 23067, 23059,
    23020, 23072, 23075, 23081, 23077, 23052, 23049, 23403, 23640, 23472,
    23475, 23478, 23476, 23470, 23477, 23481, 23480, 23556, 23633, 23637,
    23632, 23789, 23805, 23803, 23786, 23784, 23792, 23798, 23809, 23796,
    24046, 24109, 24107, 24235, 24237, 24231, 24369, 24466, 24465, 24464,
    24665, 24675, 24677, 24656, 24661, 24685, 24681, 24687, 24708, 24735,
    24730, 24717, 24724, 24716, 24709, 24726, 25159, 25331, 25352, 25343,
    25422, 25406, 25391, 25429, 25410, 25414, 25423, 25417, 25402, 25424,
    25405, {f: 2, c: 25386}, 25384, 25421, 25420, {f: 2, c: 25928}, 26009,
    26049, 26053, 26178, 26185, 26191, 26179, 26194, 26188, 26181, 26177,
    26360, {f: 2, c: 26388}, 26391, 26657, 26680, 26696, 26694, 26707, 26681,
    26690, 26708, 26665, 26803, 26647, 26700, 26705, 26685, 26612, 26704,
    26688, 26684, 26691, 26666, 26693, 26643, 26648, 26689, 27530, 27529,
    27575, 27683, {f: 2, c: 27687}, 27686, 27684, 27888, 28010, 28053, 28040,
    28039, 28006, 28024, 28023, 27993, 28051, 28012, 28041, 28014, 27994,
    28020, 28009, 28044, 28042, 28025, 28037, 28005, 28052, 28874, 28888,
    28900, 28889, 28872, 28879, 29241, 29305, 29436, 29433, 29437, 29432,
    29431, 29574, 29677, 29705, 29678, 29664, 29674, 29662, 30036, 30045,
    30044, 30042, 30041, 30142, 30149, 30151, {f: 2, c: 30130}, 30141, 30140,
    30137, 30146, 30136, 30347, 30384, 30410, {f: 2, c: 30413}, 30505,
    {f: 2, c: 30495}, 30504, 30697, 30768, 30759, 30776, 30749, 30772, 30775,
    30757, 30765, 30752, 30751, 30770, 31061, 31056, 31072, 31071, 31062,
    31070, 31069, 31063, 31066, 31204, [31203, 60418], 31207, 31199, 31206,
    31209, 31192, 31364, 31368, 31449, 31494, 31505, 31881, 32033, 32023,
    32011, 32010, 32032, 32034, 32020, 32016, 32021, 32026, 32028, 32013,
    32025, 32027, 32570, 32607, 32660, 32709, 32705, 32774, 32772, 32792,
    32789, 32793, 32791, 32829, 32831, 33009, 33026, 33008, 33029, 33005,
    33012, 33030, 33016, 33011, 33032, 33021, 33034, 33020, 33007, 33261,
    33260, 33280, 33296, {f: 2, c: 33322}, 33320, 33324, 33467, 33579, 33618,
    33620, 33610, 33592, 33616, 33609, 33589, 33588, 33615, 33586, 33593,
    33590, 33559, 33600, 33585, 33576, 33603, 34388, 34442, 34474, 34451,
    34468, 34473, 34444, 34467, 34460, 34928, 34935, {f: 2, c: 34945}, 34941,
    34937, 35352, 35344, 35342, 35340, 35349, 35338, 35351, 35347, 35350,
    35343, 35345, 35912, 35962, 35961, {f: 2, c: 36001}, [36215, 58442], 36524,
    36562, 36564, 36559, 36785, 36865, 36870, 36855, 36864, 36858, 36852,
    36867, 36861, 36869, 36856, 37013, 37089, 37085, 37090, 37202, 37197,
    37196, 37336, 37341, 37335, 37340, 37337, 38275, {f: 2, c: 38498}, 38497,
    38491, 38493, 38500, 38488, 38494, 38587, 39138, [12218, 39340],
    [12219, 39592], [12220, 39640], [12222, 39717], [12224, 39730],
    [12225, 39740], 20094, 20602, [20605, 57382], 20572, 20551, 20547, 20556,
    20570, 20553, 20581, 20598, 20558, 20565, 20597, 20596, 20599, 20559,
    20495, 20591, 20589, 20828, 20885, 20976, 21098, 21103, 21202, 21209,
    21208, 21205, 21264, 21263, 21273, {f: 2, c: 21311}, 21310, 21443, 26364,
    21830, 21866, 21862, 21828, 21854, 21857, 21827, 21834, 21809, 21846,
    21839, 21845, 21807, 21860, 21816, 21806, 21852, 21804, 21859, 21811,
    21825, 21847, 22280, 22283, 22281, 22495, 22533, 22538, 22534, 22496,
    22500, 22522, 22530, 22581, 22519, 22521, 22816, 22882, 23094, 23105,
    23113, 23142, 23146, 23104, 23100, 23138, 23130, 23110, 23114, 23408,
    23495, 23493, 23492, 23490, 23487, 23494, 23561, 23560, 23559, 23648,
    {f: 2, c: 23644}, 23815, 23814, 23822, 23835, 23830, 23842, 23825, 23849,
    23828, 23833, 23844, 23847, 23831, 24034, 24120, 24118, 24115, 24119,
    {f: 2, c: 24247}, 24246, 24245, 24254, 24373, 24375, 24407, 24428, 24425,
    24427, 24471, 24473, 24478, 24472, 24481, 24480, 24476, 24703, 24739,
    24713, 24736, 24744, 24779, 24756, 24806, 24765, 24773, 24763, 24757,
    24796, 24764, 24792, 24789, 24774, 24799, 24760, 24794, 24775,
    {f: 2, c: 25114}, 25160, 25504, 25511, 25458, 25494, 25506, 25509, 25463,
    25447, 25496, 25514, 25457, 25513, 25481, 25475, 25499, 25451, 25512,
    25476, 25480, 25497, 25505, 25516, 25490, 25487, 25472, 25467, 25449,
    25448, 25466, 25949, 25942, 25937, 25945, 25943, 21855, 25935, 25944,
    25941, 25940, 26012, 26011, 26028, 26063, {f: 2, c: 26059}, 26062, 26205,
    26202, 26212, 26216, 26214, 26206, 26361, 21207, 26395, 26753, 26799,
    26786, 26771, 26805, 26751, 26742, 26801, 26791, 26775, 26800, 26755,
    26820, 26797, 26758, 26757, 26772, 26781, 26792, 26783, 26785, 26754,
    27442, 27578, {f: 2, c: 27627}, 27691, 28046, 28092, 28147, 28121, 28082,
    28129, 28108, 28132, 28155, 28154, 28165, 28103, 28107, 28079, 28113,
    28078, 28126, 28153, 28088, 28151, 28149, 28101, 28114, 28186, 28085,
    28122, 28139, 28120, 28138, 28145, 28142, 28136, 28102, 28100, 28074,
    28140, 28095, 28134, 28921, {f: 2, c: 28937}, 28925, 28911, 29245, 29309,
    29313, 29468, 29467, 29462, 29459, 29465, 29575, 29701, 29706, 29699,
    29702, 29694, 29709, 29920, {f: 2, c: 29942}, 29980, 29986,
    {f: 2, c: 30053}, 30050, 30064, 30095, {f: 2, c: 30164}, 30133, 30154,
    30157, 30350, 30420, 30418, 30427, 30519, 30526, 30524, 30518, 30520,
    30522, 30827, 30787, 30798, 31077, 31080, 31085, 31227, 31378, 31381,
    31520, 31528, 31515, 31532, 31526, 31513, 31518, 31534, 31890, 31895,
    31893, 32070, 32067, 32113, 32046, 32057, 32060, 32064, 32048, 32051,
    32068, 32047, 32066, 32050, 32049, 32573, 32670, 32666, 32716, 32718,
    32722, 32796, 32842, 32838, 33071, 33046, 33059, 33067, 33065, 33072,
    33060, 33282, 33333, 33335, 33334, 33337, 33678, 33694, 33688, 33656,
    33698, 33686, 33725, 33707, 33682, 33674, 33683, 33673, 33696, 33655,
    {f: 2, c: 33659}, 33670, 33703, 34389, 24426, 34503, 34496, 34486, 34500,
    34485, 34502, 34507, 34481, 34479, 34505, 34899, 34974, 34952, 34987,
    34962, 34966, 34957, 34955, 35219, 35215, 35370, 35357, 35363, 35365,
    35377, 35373, 35359, 35355, 35362, 35913, 35930, 36009, 36012, 36011,
    36008, 36010, 36007, 36199, 36198, 36286, 36282, 36571, 36575, 36889,
    36877, 36890, 36887, 36899, 36895, 36893, 36880, 36885, 36894, 36896,
    36879, 36898, 36886, 36891, 36884, 37096, 37101, [37117, 58488], 37207,
    37326, 37365, 37350, 37347, 37351, 37357, 37353, 38281, 38506, 38517,
    38515, 38520, 38512, 38516, {f: 2, c: 38518}, 38508, 38592, 38634, 38633,
    31456, 31455, {f: 2, c: 38914}, [12226, 39770], [12227, 40165],
    [12228, 40565], [12229, 40575], [12230, 40613], [12231, 40635], 20642,
    20621, 20613, 20633, 20625, 20608, 20630, 20632, 20634, 26368, 20977,
    21106, {f: 2, c: 21108}, 21097, 21214, 21213, 21211, 21338, 21413, 21883,
    21888, 21927, 21884, 21898, 21917, 21912, 21890, 21916, 21930, 21908,
    21895, 21899, 21891, 21939, 21934, 21919, 21822, 21938, 21914, 21947,
    21932, 21937, 21886, 21897, 21931, 21913, 22285, 22575, 22570, 22580,
    22564, {f: 2, c: 22576}, 22561, 22557, 22560, {f: 2, c: 22777}, 22880,
    [23159, 57587], 23194, 23167, 23186, 23195, 23207, 23411, 23409, 23506,
    23500, 23507, 23504, {f: 2, c: 23562}, 23601, 23884, 23888, 23860, 23879,
    24061, 24133, 24125, 24128, 24131, 24190, 24266, {f: 2, c: 24257}, 24260,
    24380, 24429, {f: 2, c: 24489}, 24488, 24785, 24801, 24754, 24758, 24800,
    24860, 24867, 24826, 24853, 24816, 24827, 24820, 24936, 24817, 24846,
    24822, 24841, 24832, 24850, 25119, 25161, 25507, 25484, 25551, 25536,
    25577, 25545, 25542, 25549, 25554, 25571, 25552, 25569, 25558,
    {f: 2, c: 25581}, 25462, 25588, 25578, 25563, 25682, 25562, 25593, 25950,
    25958, {f: 2, c: 25954}, 26001, 26000, 26031, 26222, 26224, [26228, 57786],
    26230, 26223, 26257, 26234, 26238, 26231, {f: 2, c: 26366}, 26399, 26397,
    26874, 26837, 26848, 26840, 26839, 26885, 26847, 26869, 26862, 26855,
    26873, 26834, 26866, 26851, 26827, 26829, 26893, 26898, 26894, 26825,
    26842, 26990, 26875, 27454, 27450, 27453, 27544, 27542, 27580, 27631,
    {f: 2, c: 27694}, 27692, [28207, 57904], 28216, 28244, 28193, 28210, 28263,
    28234, 28192, 28197, 28195, 28187, 28251, 28248, 28196, 28246, 28270,
    28205, 28198, 28271, 28212, 28237, 28218, 28204, 28227, [28189, 57901],
    28222, 28363, 28297, 28185, 28238, 28259, 28228, 28274, 28265, 28255,
    {f: 2, c: 28953}, 28966, 28976, 28961, 28982, [29038, 57958], 28956, 29260,
    29316, 29312, 29494, 29477, 29492, 29481, 29754, 29738, 29747, 29730,
    29733, {f: 2, c: 29749}, 29748, 29743, 29723, 29734, 29736,
    {f: 2, c: 29989}, 30059, 30058, 30178, 30171, 30179, 30169, 30168, 30174,
    30176, {f: 2, c: 30331}, 30358, 30355, 30388, 30428, 30543, 30701, 30813,
    30828, 30831, 31245, 31240, 31243, 31237, 31232, 31384, 31383, 31382,
    31461, 31459, 31561, 31574, 31558, 31568, 31570, 31572, 31565, 31563,
    31567, [31569, 60510], 31903, 31909, 32094, 32080, 32104, 32085, 32043,
    32110, 32114, 32097, 32102, 32098, 32112, 32115, 21892, {f: 2, c: 32724},
    32779, 32850, 32901, 33109, 33108, 33099, 33105, 33102, 33081, 33094,
    33086, 33100, 33107, 33140, 33298, 33308, 33769, 33795, 33784, 33805,
    33760, 33733, 33803, [33729, 58309], 33775, 33777, 33780, 33879, 33802,
    33776, 33804, 33740, 33789, 33778, 33738, 33848, 33806, 33796, 33756,
    33799, 33748, 33759, 34395, 34527, 34521, 34541, 34516, 34523, 34532,
    34512, 34526, 34903, {f: 2, c: 35009}, 34993, 35203, 35222, 35387, 35424,
    35413, 35422, 35388, 35393, 35412, 35419, 35408, 35398, 35380, 35386,
    35382, 35414, 35937, 35970, 36015, 36028, 36019, 36029, 36033, 36027,
    36032, 36020, 36023, 36022, 36031, 36024, 36234, 36229, 36225, 36302,
    36317, 36299, 36314, 36305, 36300, 36315, 36294, 36603, 36600, 36604,
    36764, 36910, 36917, 36913, 36920, 36914, 36918, 37122, 37109, 37129,
    37118, 37219, 37221, 37327, {f: 2, c: 37396}, 37411, 37385, 37406, 37389,
    37392, 37383, 37393, 38292, 38287, 38283, 38289, 38291, 38290, 38286,
    38538, 38542, 38539, 38525, {f: 2, c: 38533}, 38541, 38514, 38532, 38593,
    38597, 38596, {f: 2, c: 38598}, 38639, 38642, 38860, {f: 2, c: 38917},
    38920, 39143, 39146, 39151, 39145, 39154, 39149, 39342, 39341,
    [12232, 40643], [12233, 40653], [12234, 40657], 20098, 20653, 20661,
    {f: 2, c: 20658}, 20677, 20670, 20652, 20663, 20667, 20655, 20679, 21119,
    21111, 21117, 21215, 21222, 21220, {f: 2, c: 21218}, 21295, 21983, 21992,
    21971, 21990, 21966, 21980, 21959, 21969, {f: 2, c: 21987}, 21999, 21978,
    21985, {f: 2, c: 21957}, 21989, 21961, {f: 2, c: 22290}, 22622, 22609,
    22616, 22615, 22618, 22612, 22635, 22604, 22637, 22602, 22626, 22610,
    22603, 22887, 23233, 23241, 23244, 23230, 23229, 23228, 23219, 23234,
    23218, 23913, 23919, 24140, 24185, 24265, 24264, 24338, 24409, 24492,
    24494, 24858, 24847, 24904, 24863, 24819, 24859, 24825, 24833, 24840,
    24910, 24908, 24900, 24909, 24894, 24884, 24871, 24845, 24838, 24887,
    {f: 2, c: 25121}, 25619, 25662, 25630, 25642, 25645, 25661, 25644, 25615,
    25628, 25620, 25613, 25654, {f: 2, c: 25622}, 25606, 25964, 26015, 26032,
    26263, 26249, {f: 2, c: 26247}, 26262, 26244, 26264, 26253, 26371, 27028,
    26989, 26970, 26999, 26976, 26964, 26997, 26928, 27010, 26954, 26984,
    26987, 26974, 26963, 27001, 27014, 26973, 26979, 26971, 27463, 27506,
    27584, 27583, 27603, 27645, 28322, 28335, 28371, 28342, 28354, 28304,
    28317, 28359, 28357, 28325, 28312, 28348, 28346, 28331, 28369, 28310,
    28316, 28356, 28372, 28330, 28327, 28340, 29006, 29017, 29033, 29028,
    29001, 29031, 29020, 29036, 29030, 29004, 29029, 29022, 28998, 29032,
    29014, 29242, 29266, 29495, 29509, 29503, 29502, 29807, 29786, 29781,
    29791, 29790, 29761, 29759, 29785, 29787, [29788, 58019], 30070, 30072,
    30208, 30192, 30209, 30194, 30193, 30202, 30207, 30196, 30195,
    {f: 2, c: 30430}, 30555, 30571, 30566, 30558, 30563, 30585, 30570, 30572,
    30556, 30565, 30568, 30562, 30702, 30862, 30896, {f: 2, c: 30871}, 30860,
    30857, 30844, 30865, 30867, 30847, 31098, 31103, 31105, 33836, 31165,
    31260, 31258, 31264, 31252, 31263, 31262, {f: 2, c: 31391}, 31607, 31680,
    31584, 31598, 31591, 31921, 31923, 31925, 32147, 32121, 32145, 32129,
    32143, 32091, 32622, {f: 2, c: 32617}, 32626, 32681, 32680, 32676, 32854,
    32856, 32902, 32900, 33137, 33136, 33144, 33125, 33134, 33139, 33131,
    {f: 2, c: 33145}, 33126, 33285, 33351, 33922, 33911, 33853, 33841, 33909,
    33894, 33899, 33865, 33900, 33883, 33852, 33845, 33889, 33891, 33897,
    33901, 33862, 34398, 34396, 34399, 34553, 34579, 34568, 34567, 34560,
    34558, 34555, {f: 2, c: 34562}, 34566, 34570, 34905, 35039, 35028, 35033,
    35036, 35032, 35037, 35041, 35018, 35029, 35026, 35228, 35299, 35435,
    {f: 2, c: 35442}, 35430, 35433, 35440, 35463, 35452, 35427, 35488, 35441,
    35461, 35437, 35426, 35438, 35436, 35449, 35451, 35390, 35432, 35938,
    35978, 35977, 36042, {f: 2, c: 36039}, 36036, 36018, 36035, 36034, 36037,
    36321, 36319, 36328, 36335, 36339, 36346, 36330, 36324, 36326, 36530,
    36611, 36617, 36606, 36618, 36767, 36786, 36939, 36938, 36947, 36930,
    36948, 36924, 36949, 36944, 36935, 36943, 36942, 36941, 36945, 36926,
    36929, 37138, 37143, 37228, 37226, 37225, 37321, 37431, 37463, 37432,
    37437, 37440, 37438, 37467, 37451, 37476, 37457, 37428, 37449, 37453,
    37445, 37433, 37439, 37466, 38296, 38552, {f: 2, c: 38548}, 38605, 38603,
    {f: 2, c: 38601}, 38647, 38651, 38649, 38646, 38742, 38772, 38774,
    {f: 2, c: 38928}, 38931, 38922, 38930, 38924, 39164, 39156,
    {f: 2, c: 39165}, 39347, 39345, 39348, 39649, 40169, 40578, [12237, 40718],
    [12238, 40723], [12239, 40736], 20711, 20718, 20709, 20694, [20717, 60903],
    20698, 20693, 20687, 20689, 20721, 20686, 20713, 20834, 20979, 21123,
    21122, 21297, 21421, 22014, 22016, 22043, 22039, 22013, 22036, 22022,
    22025, {f: 2, c: 22029}, 22007, 22038, 22047, 22024, 22032, 22006, 22296,
    22294, 22645, 22654, 22659, 22675, 22666, 22649, 22661, 22653, 22781,
    22821, 22818, 22820, 22890, 22889, 23265, 23270, 23273, 23255, 23254,
    23256, 23267, 23413, 23518, 23527, 23521, {f: 2, c: 23525}, 23528, 23522,
    23524, 23519, 23565, 23650, 23940, 23943, 24155, 24163, 24149, 24151,
    24148, 24275, 24278, 24330, 24390, 24432, 24505, 24903, 24895, 24907,
    24951, {f: 2, c: 24930}, 24927, 24922, 24920, 24949, 25130, 25735, 25688,
    25684, 25764, 25720, 25695, 25722, 25681, 25703, 25652, 25709, 25723,
    25970, 26017, 26071, 26070, 26274, 26280, 26269, 27036, 27048, 27029,
    27073, 27054, 27091, 27083, 27035, 27063, 27067, 27051, 27060, 27088,
    27085, 27053, 27084, 27046, 27075, 27043, 27465, 27468, 27699, 28467,
    28436, 28414, 28435, 28404, 28457, 28478, 28448, 28460, 28431, 28418,
    28450, 28415, 28399, 28422, 28465, 28472, 28466, 28451, 28437, 28459,
    28463, 28552, 28458, 28396, 28417, 28402, 28364, 28407, 29076, 29081,
    29053, 29066, 29060, 29074, 29246, 29330, 29334, 29508, 29520, 29796,
    29795, 29802, 29808, 29805, 29956, 30097, 30247, 30221, 30219, 30217,
    30227, 30433, 30435, 30596, 30589, 30591, 30561, 30913, 30879, 30887,
    30899, 30889, 30883, {f: 2, c: 31118}, 31117, 31278, 31281, 31402, 31401,
    31469, 31471, 31649, 31637, 31627, 31605, 31639, 31645, 31636, 31631,
    [31672, 58170], 31623, 31620, 31929, {f: 2, c: 31933}, 32187, 32176, 32156,
    {f: 2, c: 32189}, 32160, 32202, 32180, 32178, 32177, 32186, 32162, 32191,
    32181, 32184, 32173, [32210, 58202], 32199, 32172, 32624, {f: 2, c: 32736},
    32735, 32862, 32858, 32903, 33104, 33152, 33167, 33160, 33162, 33151,
    33154, 33255, 33274, 33287, 33300, 33310, 33355, 33993, 33983, 33990,
    33988, 33945, 33950, 33970, 33948, 33995, 33976, 33984, 34003, 33936,
    33980, 34001, 33994, 34623, 34588, 34619, 34594, 34597, 34612, 34584,
    34645, 34615, 34601, 35059, 35074, 35060, 35065, 35064, 35069, 35048,
    35098, 35055, 35494, 35468, 35486, 35491, 35469, 35489, 35475, 35492,
    35498, 35493, 35496, 35480, 35473, 35482, 35495, 35946, 35981, 35980,
    36051, {f: 2, c: 36049}, 36203, 36249, 36245, 36348, 36628, 36626, 36629,
    36627, 36771, 36960, 36952, 36956, 36963, 36953, 36958, 36962, 36957,
    36955, 37145, 37144, 37150, 37237, 37240, 37239, 37236, 37496, 37548,
    37504, 37509, 37528, 37526, 37499, 37523, 37532, 37544, 37500, 37521,
    38305, {f: 2, c: 38312}, 38307, 38309, 38308, 38553, 38556, 38555, 38604,
    38610, 38656, 38780, 38789, 38902, {f: 2, c: 38935}, 39087, 39089, 39171,
    39173, 39180, 39177, 39361, {f: 2, c: 39599}, 39654, {f: 2, c: 39745},
    40180, 40182, 40179, 40636, [12240, 40763], [12241, 40778], 20740, 20736,
    20731, 20725, 20729, 20738, {f: 2, c: 20744}, 20741, 20956,
    {f: 3, c: 21127}, 21133, 21130, 21232, 21426, 22062, 22075, 22073, 22066,
    22079, 22068, 22057, 22099, 22094, 22103, 22132, 22070, {f: 2, c: 22063},
    22656, 22687, 22686, 22707, 22684, 22702, 22697, 22694, 22893, 23305,
    23291, 23307, 23285, 23308, 23304, 23534, 23532, 23529, 23531,
    {f: 2, c: 23652}, 23965, 23956, 24162, 24159, 24161, 24290, 24282, 24287,
    24285, 24291, 24288, 24392, 24433, 24503, 24501, 24950, 24935, 24942,
    24925, 24917, 24962, 24956, 24944, 24939, 24958, 24999, 24976, 25003,
    24974, 25004, 24986, 24996, 24980, 25006, 25134, 25705, 25711, 25721,
    25758, 25778, 25736, [25744, 57745], 25776, 25765, 25747, 25749, 25769,
    25746, 25774, 25773, 25771, 25754, 25772, 25753, 25762, 25779, 25973,
    {f: 2, c: 25975}, 26286, 26283, 26292, 26289, 27171, 27167, 27112, 27137,
    27166, 27161, 27133, 27169, 27155, 27146, 27123, 27138, 27141, 27117,
    27153, 27472, 27470, 27556, {f: 2, c: 27589}, 28479, 28540, 28548, 28497,
    28518, 28500, 28550, 28525, 28507, 28536, 28526, 28558, 28538, 28528,
    28516, 28567, 28504, 28373, 28527, 28512, 28511, 29087, 29100, 29105,
    29096, 29270, 29339, 29518, 29527, 29801, 29835, 29827, 29822, 29824,
    30079, 30240, 30249, 30239, 30244, 30246, {f: 2, c: 30241}, 30362, 30394,
    30436, 30606, 30599, 30604, 30609, 30603, 30923, 30917, 30906, 30922,
    30910, 30933, 30908, 30928, 31295, 31292, 31296, 31293, 31287, 31291,
    31407, 31406, 31661, 31665, 31684, 31668, {f: 2, c: 31686}, 31681, 31648,
    31692, 31946, 32224, 32244, 32239, 32251, 32216, 32236, 32221, 32232,
    32227, 32218, 32222, 32233, 32158, 32217, 32242, 32249, 32629, 32631,
    32687, 32745, 32806, {f: 3, c: 33179}, 33184, 33178, 33176, 34071, 34109,
    34074, 34030, {f: 2, c: 34092}, 34067, 34065, 34083, 34081, 34068, 34028,
    34085, 34047, 34054, 34690, 34676, 34678, 34656, 34662, 34680, 34664,
    34649, 34647, 34636, 34643, 34907, 34909, 35088, 35079, {f: 2, c: 35090},
    35093, 35082, 35516, 35538, 35527, 35524, 35477, 35531, 35576, 35506,
    35529, 35522, 35519, 35504, 35542, 35533, 35510, 35513, 35547, 35916,
    35918, 35948, 36064, 36062, 36070, 36068, {f: 2, c: 36076},
    {f: 2, c: 36066}, 36060, 36074, 36065, 36205, 36255, 36259, 36395, 36368,
    36381, 36386, 36367, 36393, 36383, 36385, 36382, 36538, 36637, 36635,
    36639, 36649, 36646, 36650, 36636, 36638, 36645, 36969, 36974, 36968,
    36973, 36983, 37168, 37165, 37159, 37169, 37255, 37257, 37259, 37251,
    37573, 37563, 37559, 37610, 37604, 37569, 37555, 37564, 37586, 37575,
    37616, 37554, 38317, 38321, 38660, {f: 2, c: 38662}, 38665, 38752, 38797,
    38795, 38799, 38945, 38955, 38940, 39091, 39178, 39187, 39186, 39192,
    39389, 39376, 39391, 39387, 39377, 39381, 39378, 39385, 39607,
    {f: 2, c: 39662}, 39719, 39749, 39748, 39799, 39791, 40198, 40201, 40195,
    40617, 40638, 40654, 22696, [12242, 40786], 20754, 20760, 20756, 20752,
    20757, 20864, 20906, 20957, 21137, 21139, 21235, 22105, 22123, 22137,
    22121, 22116, 22136, 22122, 22120, 22117, 22129, 22127, 22124, 22114,
    22134, 22721, 22718, 22727, 22725, 22894, 23325, 23348, 23416, 23536,
    23566, 24394, 25010, 24977, 25001, 24970, 25037, 25014, 25022, 25034,
    25032, 25136, 25797, 25793, 25803, {f: 2, c: 25787}, 25818, 25796, 25799,
    25794, 25805, 25791, 25810, 25812, 25790, 25972, 26310, 26313, 26297,
    26308, 26311, 26296, 27197, 27192, 27194, 27225, 27243, 27224, 27193,
    27204, 27234, 27233, 27211, 27207, 27189, 27231, 27208, 27481, 27511,
    27653, 28610, 28593, 28577, 28611, 28580, 28609, 28583, 28595, 28608,
    28601, [28598, 60318], 28582, 28576, 28596, 29118, 29129, 29136, 29138,
    29128, 29141, 29113, 29134, 29145, 29148, {f: 2, c: 29123}, 29544, 29852,
    29859, 29848, 29855, 29854, 29922, {f: 2, c: 29964}, 30260, 30264, 30266,
    30439, 30437, 30624, {f: 2, c: 30622}, 30629, 30952, 30938, 30956, 30951,
    31142, {f: 2, c: 31309}, 31302, 31308, 31307, 31418, 31705, 31761, 31689,
    31716, 31707, 31713, 31721, 31718, {f: 2, c: 31957}, 32266, 32273, 32264,
    32283, 32291, 32286, [32285, 58211], 32265, 32272, 32633, 32690,
    {f: 2, c: 32752}, 32750, [32808, 58239], 33203, 33193, 33192, 33275, 33288,
    {f: 2, c: 33368}, 34122, 34137, 34120, {f: 2, c: 34152}, 34115, 34121,
    34157, 34154, 34142, 34691, 34719, 34718, 34722, 34701, 34913, 35114,
    35122, 35109, 35115, 35105, 35242, [35238, 58391], 35558, 35578, 35563,
    35569, 35584, 35548, 35559, 35566, 35582, {f: 2, c: 35585}, 35575, 35565,
    35571, 35574, 35580, 35947, 35949, 35987, 36084, 36420, 36401, 36404,
    36418, 36409, 36405, 36667, 36655, 36664, 36659, 36776, 36774, 36981,
    36980, 36984, 36978, 36988, 36986, 37172, 37266, 37664, 37686, 37624,
    37683, 37679, 37666, 37628, 37675, 37636, 37658, 37648, 37670, 37665,
    37653, 37678, 37657, 38331, {f: 2, c: 38567}, 38570, 38613, 38670, 38673,
    38678, 38669, 38675, 38671, 38747, [38748, 58565], 38758, 38808, 38960,
    38968, 38971, 38967, 38957, 38969, 38948, 39184, 39208, 39198, 39195,
    39201, 39194, 39405, 39394, 39409, 39608, 39612, 39675, 39661, 39720,
    39825, 40213, 40227, 40230, 40232, 40210, 40219, 40664, 40660,
    [12243, 40845], [12244, 40860], 20778, 20767, 20769, 20786, 21237, 22158,
    22144, 22160, 22149, 22151, 22159, 22741, 22739, 22737, 22734, 23344,
    23338, 23332, 23418, 23607, 23656, 23996, 23994, 23997, 23992, 24171,
    24396, 24509, 25033, 25026, 25031, 25062, 25035, 25138, 25140, 25806,
    25802, 25816, 25824, 25840, 25830, 25836, 25841, 25826, 25837,
    {f: 2, c: 25986}, 26329, 26326, 27264, 27284, 27268, 27298, 27292, 27355,
    27299, 27262, 27287, 27280, 27296, 27484, 27566, 27610, 27656, 28632,
    28657, {f: 2, c: 28639}, 28635, 28644, 28651, 28655, 28544, 28652, 28641,
    28649, 28629, 28654, 28656, 29159, [29151, 60361], 29166, 29158, 29157,
    29165, 29164, 29172, 29152, 29237, 29254, 29552, 29554, 29865, 29872,
    29862, 29864, 30278, 30274, 30284, 30442, 30643, 30634, 30640, 30636,
    30631, 30637, 30703, 30967, 30970, 30964, 30959, 30977, 31143, 31146,
    31319, 31423, 31751, 31757, 31742, 31735, 31756, 31712, 31968, 31964,
    31966, 31970, 31967, 31961, 31965, 32302, 32318, 32326, 32311, 32306,
    32323, 32299, 32317, 32305, 32325, 32321, 32308, 32313, 32328, 32309,
    32319, 32303, 32580, 32755, 32764, {f: 2, c: 32881}, 32880, 32879, 32883,
    33222, 33219, 33210, 33218, 33216, 33215, 33213, 33225, 33214, 33256,
    33289, 33393, 34218, 34180, 34174, 34204, 34193, 34196, 34223, 34203,
    34183, 34216, 34186, 34214, 34407, 34752, 34769, 34739, 34770, 34758,
    34731, 34747, 34746, 34760, 34763, 35131, 35126, 35140, 35128, 35133,
    35244, 35598, 35607, 35609, 35611, 35594, 35616, 35613, 35588, 35600,
    35905, 35903, 35955, 36090, 36093, 36092, 36088, 36091, 36264, 36425,
    36427, 36424, 36426, 36676, 36670, 36674, 36677, 36671, 36991, 36989,
    36996, {f: 2, c: 36993}, 36992, 37177, 37283, 37278, 37276, 37709, 37762,
    37672, 37749, 37706, 37733, 37707, 37656, 37758, 37740, 37723, 37744,
    37722, 37716, {f: 3, c: 38346}, 38344, 38342, 38577, 38584, 38614, 38684,
    38686, 38816, 38867, 38982, 39094, 39221, 39425, 39423, 39854, 39851,
    39850, 39853, 40251, 40255, 40587, 40655, 40670, {f: 2, c: 40668}, 40667,
    40766, 40779, 21474, 22165, 22190, 22745, 22744, 23352, 24413, 25059,
    25139, 25844, 25842, 25854, 25862, {f: 2, c: 25850}, 25847, 26039, 26332,
    26406, 27315, 27308, 27331, 27323, 27320, 27330, {f: 2, c: 27310}, 27487,
    27512, 27567, 28681, 28683, 28670, 28678, 28666, 28689, 28687,
    {f: 2, c: 29179}, 29182, 29176, 29559, 29557, 29863, 29887, 29973, 30294,
    30296, 30290, 30653, 30655, {f: 2, c: 30651}, 30990, 31150,
    {f: 2, c: 31329}, 31328, {f: 2, c: 31428}, 31787, 31783, 31786, 31774,
    31779, 31777, 31975, {f: 2, c: 32340}, 32350, 32346, 32353, 32338, 32345,
    32584, 32761, 32763, 32887, 32886, 33229, 33231, 33290, 34255, 34217,
    34253, 34256, 34249, 34224, 34234, 34233, 34799, 34796, 34802, 34784,
    35206, 35250, 35316, 35624, 35641, 35628, 35627, 35920, 36101, 36441,
    36451, 36454, 36452, 36447, 36437, 36544, 36681, 36685, 36999, 36995,
    37000, {f: 2, c: 37291}, 37328, 37780, 37770, 37782, 37794, 37811, 37806,
    37804, 37808, 37784, 37786, 37783, 38356, 38358, 38352, 38357, 38626,
    38620, 38617, 38619, 38622, 38692, 38819, 38822, 38829, 38905, 38989,
    38991, 38988, 38990, 38995, 39098, {f: 2, c: 39230}, 39229, 39214, 39333,
    39438, 39617, 39683, 39686, 39759, 39758, 39757, 39882, 39881, 39933,
    39880, 39872, 40273, 40285, 40288, 40672, 40725, 40748, 20787, 22181,
    22184, {f: 2, c: 22750}, 22754, 23541, 40848, 24300, 25074, 25079, 25078,
    25077, 25856, 25871, 26336, 26333, 27365, 27357, 27354, 27347, 28699,
    28703, 28712, 28698, 28701, 28693, 28696, 29190, 29197, 29272, 29346,
    29560, 29562, 29885, 29898, 29923, 30087, 30086, 30303, 30305, 30663,
    31001, 31153, 31339, 31337, {f: 2, c: 31806}, 31800, 31805, 31799, 31808,
    32363, 32365, 32377, {f: 2, c: 32361}, 32371, 32645, 32694, 32697, 32696,
    33240, 34281, 34269, 34282, 34261, {f: 2, c: 34276}, 34295, 34811, 34821,
    34829, 34809, 34814, 35168, 35167, 35158, 35166, 35649, 35676, 35672,
    35657, 35674, {f: 2, c: 35662}, 35654, 35673, 36104, 36106, 36476, 36466,
    36487, 36470, 36460, 36474, 36468, 36692, 36686, 36781, {f: 2, c: 37002},
    37297, 37294, 37857, 37841, 37855, 37827, 37832, {f: 2, c: 37852}, 37846,
    37858, 37837, 37848, 37860, 37847, 37864, 38364, 38580, 38627, 38698,
    38695, 38753, 38876, 38907, 39006, 39000, 39003, 39100, 39237, 39241,
    39446, 39449, 39693, 39912, 39911, 39894, 39899, 40329, 40289, 40306,
    40298, 40300, 40594, 40599, 40595, 40628, 21240, 22199, 22198, 22196,
    22204, 22756, 23360, 23363, 23421, 23542, 24009, 25080, 25082, 25880,
    25876, 25881, 26342, 26407, 27372, 28734, 28720, 28722, 29200, 29563,
    29903, 30306, 30309, 31014, 31018, 31020, 31019, 31431, 31478, 31820,
    31811, 31821, {f: 2, c: 31983}, 36782, 32381, 32380, 32386, 32588, 32768,
    33242, 33382, 34299, 34297, 34321, 34298, 34310, 34315, 34311, 34314,
    {f: 2, c: 34836}, 35172, 35258, 35320, 35696, 35692, 35686, 35695, 35679,
    35691, 36111, 36109, 36489, 36481, 36485, 36482, 37300, 37323, 37912,
    37891, 37885, 38369, 38704, 39108, 39250, 39249, 39336, 39467, 39472,
    39479, 39477, 39955, 39949, 40569, 40629, 40680, 40751, 40799, 40803,
    40801, {f: 2, c: 20791}, 22209, 22208, 22210, 22804, 23660, 24013, 25084,
    25086, 25885, 25884, 26005, 26345, 27387, 27396, 27386, 27570, 28748,
    29211, 29351, 29910, 29908, 30313, 30675, 31824, 32399, 32396, 32700,
    34327, 34349, 34330, 34851, 34850, 34849, 34847, 35178, 35180, 35261,
    35700, 35703, 35709, 36115, 36490, 36493, 36491, 36703, 36783, 37306,
    37934, 37939, 37941, 37946, 37944, 37938, 37931, 38370, {f: 2, c: 38712},
    38706, [38911, 58586], 39015, 39013, 39255, 39493, 39491, 39488, 39486,
    39631, 39764, 39761, 39981, 39973, 40367, 40372, 40386, 40376, 40605,
    40687, 40729, 40796, {f: 2, c: 40806}, 20796, 20795, 22216, 22218, 22217,
    23423, 24020, 24018, 24398, 25087, 25892, 27402, 27489, 28753, 28760,
    29568, 29924, 30090, 30318, 30316, 31155, 31840, 31839, 32894, 32893,
    33247, 35186, 35183, 35324, 35712, {f: 2, c: 36118}, 36497, 36499, 36705,
    37192, 37956, {f: 2, c: 37969}, {f: 2, c: 38717}, 38851, 38849, 39019,
    39253, 39509, 39501, 39634, 39706, 40009, 39985, 39998, 39995, 40403,
    40407, 40756, 40812, 40810, 40852, 22220, 24022, 25088, 25891, 25899,
    25898, 26348, 27408, 29914, 31434, 31844, 31843, 31845, 32403, 32406,
    32404, 33250, 34360, 34367, 34865, 35722, 37008, 37007, 37987, 37984,
    37988, 38760, 39023, 39260, {f: 2, c: 39514}, 39511, {f: 2, c: 39635},
    39633, 40020, 40023, 40022, 40421, 40607, 40692, 22225, 22761, 25900,
    28766, {f: 2, c: 30321}, [30679, 60226], 32592, 32648, 34870, 34873, 34914,
    35731, 35730, 35734, 33399, 36123, 37312, 37994, 38722, 38728, 38724,
    38854, 39024, 39519, 39714, 39768, 40031, {f: 2, c: 40441},
    {f: 2, c: 40572}, 40711, 40823, 40818, 24307, 27414, 28771, 31852, 31854,
    34875, 35264, 36513, 37313, 38002, 38000, 39025, 39262, 39638, 39715,
    40652, 28772, 30682, 35738, 38007, 38857, 39522, 39525, 32412, 35740,
    36522, 37317, {f: 2, c: 38013}, 38012, {f: 2, c: 40055}, 40695, 35924,
    38015, 40474, 29224, 39530, 39729, 40475, 40478, 31858, 20034, 20060,
    [12048, 20981], [12053, 21274], [12058, 21378], 19975, 19980, 20039, 20109,
    [12062, 22231], [12076, 23662], [12091, 24435], 19983, 20871, 19982, 20014,
    20115, 20162, 20169, 20168, 20888, 21244, 21356, 21433, 22304, 22787,
    22828, [23568, 60417], 24063, 26081, [12110, 27571], 27596, [12115, 27668],
    [12121, 29247], 20017, 20028, 20200, 20188, 20201, 20193, 20189, 20186,
    21004, 21001, 21276, 21324, {f: 2, c: 22306}, 22807, 22831, 23425, 23428,
    23570, 23611, 23668, 23667, 24068, 24192, 24194, 24521, 25097, 25168,
    27669, 27702, 27715, 27711, 27707, 29358, 29360, 29578, [12145, 31160],
    32906, 38430, 20238, 20248, 20268, 20213, 20244, 20209, 20224, 20215,
    20232, 20253, 20226, 20229, 20258, 20243, 20228, 20212, 20242, 20913,
    21011, 21008, 21158, 21282, 21279, 21325, 21386, 21511, 22241, 22239,
    22318, 22314, 22324, 22844, 22912, 22908, 22917, 22907, 22910, 22903,
    22911, 23382, 23573, 23589, 23676, {f: 2, c: 23674}, 23678, 24031,
    [24181, 57646], 24196, 24322, 24346, 24436, 24533, 24532, 24527, 25180,
    25182, 25188, 25185, 25190, 25186, 25177, 25184, 25178, 25189, 25911,
    26095, 26094, 26430, 26425, 26424, 26427, 26426, 26431, 26428, 26419,
    27672, 27718, 27730, 27740, 27727, [27722, 60796], 27732, {f: 2, c: 27723},
    28785, 29278, {f: 2, c: 29364}, 29582, 29994, 30335, 31349, [12153, 32593],
    [12171, 33400], 33404, 33408, 33405, 33407, [12172, 34381], [12177, 35198],
    37017, [37015, 59347], 37016, 37019, 37012, 38434, 38436, 38432, 38435,
    20310, 20283, 20322, 20297, 20307, 20324, 20286, 20327, 20306, 20319,
    20289, 20312, 20269, 20275, 20287, 20321, 20879, 20921, 21020, 21022,
    21025, {f: 2, c: 21165}, 21257, 21347, 21362, {f: 2, c: 21390}, 21552,
    21559, 21546, 21588, 21573, 21529, 21532, 21541, 21528, 21565, 21583,
    21569, 21544, 21540, 21575, 22254, 22247, 22245, 22337, 22341, 22348,
    22345, 22347, 22354, 22790, 22848, 22950, 22936, 22944, 22935, 22926,
    22946, 22928, 22927, 22951, 22945, 23438, 23442, 23592, 23594, 23693,
    23695, 23688, 23691, 23689, 23698, 23690, 23686, 23699, 23701, 24032,
    24074, 24078, 24203, 24201, 24204, 24200, 24205, 24325, 24349, 24440,
    24438, 24530, 24529, 24528, 24557, 24552, 24558, 24563, 24545, 24548,
    24547, 24570, 24559, 24567, 24571, 24576, 24564, 25146, 25219, 25228,
    {f: 2, c: 25230}, 25236, 25223, 25201, 25211, 25210, 25200, 25217, 25224,
    25207, 25213, 25202, 25204, 26096, 26100, 26099, 26098, 26101, 26437,
    26439, 26457, 26453, 26444, 26440, 26461, 26445, 26458, 26443, 27600,
    {f: 2, c: 27673}, 27768, 27751, 27755, 27780, 27787, 27791, 27761, 27759,
    27753, 27802, 27757, 27783, 27797, [27804, 57900], 27750, 27763, 27749,
    27771, 27790, 28788, 28794, 29283, 29375, 29373, 29379, 29382, 29377,
    29370, 29381, 29589, 29591, {f: 2, c: 29587}, 29586, 30010, 30009,
    {f: 2, c: 30100}, 30337, 31037, 32820, 32917, 32921, 32912, 32914, 32924,
    33424, 33423, 33413, 33422, 33425, 33427, 33418, {f: 2, c: 33411},
    [12184, 35960], 36809, 36799, 37023, 37025, 37029, 37022, 37031, 37024,
    38448, 38440, 38447, 38445, 20019, 20376, 20348, 20357, 20349, 20352,
    20359, 20342, 20340, 20361, 20356, 20343, 20300, 20375, 20330, 20378,
    20345, 20353, 20344, 20368, 20380, 20372, 20382, 20370, 20354, 20373,
    20331, 20334, 20894, 20924, 20926, 21045, {f: 2, c: 21042}, 21062, 21041,
    21180, {f: 2, c: 21258}, 21308, 21394, 21396, 21639, 21631, 21633, 21649,
    21634, 21640, 21611, 21626, 21630, 21605, 21612, 21620, 21606, 21645,
    21615, 21601, 21600, 21656, 21603, 21607, 21604, 22263, 22265, 22383,
    22386, 22381, 22379, 22385, 22384, 22390, 22400, 22389, 22395,
    {f: 2, c: 22387}, 22370, 22376, 22397, 22796, 22853, 22965, 22970, 22991,
    22990, 22962, 22988, 22977, 22966, 22972, 22979, 22998, 22961, 22973,
    22976, 22984, 22964, 22983, 23394, 23397, 23443, 23445, 23620, 23623,
    23726, 23716, 23712, 23733, 23727, 23720, 23724, 23711, 23715, 23725,
    23714, 23722, 23719, 23709, 23717, 23734, 23728, 23718, 24087, 24084,
    24089, 24360, {f: 3, c: 24354}, 24404, 24450, 24446, 24445, 24542, 24549,
    24621, 24614, 24601, 24626, 24587, 24628, 24586, 24599, 24627, 24602,
    24606, 24620, 24610, 24589, 24592, 24622, 24595, 24593, 24588, 24585,
    24604, 25108, 25149, 25261, 25268, 25297, 25278, 25258, 25270, 25290,
    25262, 25267, 25263, 25275, 25257, 25264, 25272, 25917, 26024, 26043,
    26121, 26108, 26116, 26130, 26120, 26107, 26115, 26123, 26125, 26117,
    26109, 26129, 26128, 26358, 26378, 26501, 26476, 26510, 26514, 26486,
    26491, 26520, 26502, 26500, 26484, 26509, 26508, 26490, 26527, 26513,
    26521, 26499, 26493, 26497, {f: 2, c: 26488}, 26516, 27429, 27520, 27518,
    27614, 27677, 27795, 27884, 27883, 27886, 27865, 27830, 27860, 27821,
    27879, 27831, 27856, 27842, 27834, 27843, 27846, 27885, 27890, 27858,
    27869, 27828, 27786, 27805, 27776, 27870, 27840, 27952, 27853, 27847,
    27824, 27897, 27855, 27881, 27857, 28820, 28824, 28805, 28819, 28806,
    28804, 28817, 28822, 28802, 28826, 28803, 29290, 29398, 29387, 29400,
    29385, 29404, 29394, 29396, 29402, 29388, 29393, 29604, 29601, 29613,
    29606, 29602, 29600, 29612, 29597, 29917, 29928, {f: 2, c: 30015}, 30014,
    30092, 30104, 30383, 30451, 30449, 30448, 30453, 30712, 30716, 30713,
    30715, 30714, 30711, 31042, 31039, 31173, 31352, 31355, 31483, 31861,
    31997, 32821, 32911, 32942, 32931, 32952, 32949, 32941, 33312, 33440,
    33472, 33451, 33434, 33432, 33435, 33461, 33447, 33454, 33468, 33438,
    33466, 33460, 33448, 33441, 33449, 33474, 33444, 33475, 33462, 33442,
    34416, 34415, {f: 2, c: 34413}, 35926, 36818, 36811, 36819, 36813, 36822,
    36821, 36823, 37042, 37044, 37039, 37043, 37040, 38457, 38461, 38460,
    38458, 38467, 20429, 20421, 20435, 20402, 20425, 20427, 20417, 20436,
    20444, 20441, [20411, 60346], 20403, 20443, 20423, 20438, 20410, 20416,
    20409, 20460, 21060, 21065, 21184, 21186, 21309, 21372, 21399, 21398,
    21401, 21400, 21690, 21665, 21677, 21669, 21711, 21699, 33549, 21687,
    21678, 21718, 21686, {f: 2, c: 21701}, 21664, 21616, 21692, 21666, 21694,
    21618, 21726, 21680, 22453, {f: 2, c: 22430}, 22436, 22412, 22423, 22429,
    22427, 22420, 22424, 22415, 22425, 22437, 22426, 22421, 22772, 22797,
    22867, 23009, 23006, 23022, 23040, 23025, 23005, 23034, 23037, 23036,
    23030, 23012, 23026, 23031, 23003, 23017, 23027, 23029, 23008, 23038,
    23028, 23021, 23464, 23628, 23760, 23768, 23756, 23767, 23755, 23771,
    23774, 23770, 23753, 23751, 23754, 23766, {f: 2, c: 23763}, 23759, 23752,
    23750, 23758, 23775, 23800, 24057, {f: 3, c: 24097}, 24096, 24100, 24240,
    24228, 24226, 24219, 24227, 24229, 24327, 24366, 24406, 24454, 24631,
    24633, 24660, 24690, 24670, 24645, 24659, 24647, 24649, 24667, 24652,
    24640, 24642, 24671, 24612, 24644, 24664, 24678, 24686, {f: 2, c: 25154},
    25295, 25357, 25355, 25333, 25358, 25347, 25323, 25337, 25359, 25356,
    25336, 25334, 25344, {f: 2, c: 25363}, 25338, 25365, 25339, 25328, 25921,
    25923, 26026, 26047, 26166, 26145, 26162, 26165, 26140, 26150, 26146,
    26163, 26155, 26170, 26141, 26164, 26169, 26158, {f: 2, c: 26383}, 26561,
    26610, 26568, 26554, 26588, 26555, 26616, 26584, 26560, 26551, 26565,
    26603, 26596, 26591, 26549, 26573, 26547, 26615, 26614, 26606, 26595,
    26562, 26553, 26574, 26599, 26608, 26546, 26620, 26566, 26605, 26572,
    26542, 26598, 26587, 26618, {f: 2, c: 26569}, 26563, 26602, 26571, 27432,
    27522, 27524, 27574, 27606, 27608, 27616, {f: 2, c: 27680}, 27944, 27956,
    27949, 27935, 27964, 27967, 27922, 27914, 27866, 27955, 27908, 27929,
    27962, 27930, 27921, 27904, 27933, 27970, 27905, 27928, 27959, 27907,
    27919, 27968, 27911, 27936, 27948, 27912, 27938, 27913, 27920, 28855,
    28831, 28862, 28849, 28848, 28833, {f: 2, c: 28852}, 28841, 29249,
    {f: 2, c: 29257}, 29292, 29296, 29299, 29294, 29386, 29412, 29416, 29419,
    29407, 29418, 29414, 29411, 29573, 29644, 29634, 29640, 29637, 29625,
    29622, 29621, 29620, 29675, 29631, 29639, 29630, 29635, 29638, 29624,
    29643, 29932, 29934, 29998, {f: 2, c: 30023}, 30119, 30122, 30329, 30404,
    30472, {f: 3, c: 30467}, 30474, 30455, 30459, 30458, {f: 2, c: 30695},
    30726, {f: 2, c: 30737}, 30725, 30736, 30735, 30734, [30729, 58095], 30723,
    30739, 31050, 31052, 31051, 31045, 31044, 31189, 31181, 31183, 31190,
    31182, 31360, 31358, 31441, {f: 2, c: 31488}, 31866, {f: 2, c: 31864},
    {f: 3, c: 31871}, 32003, 32008, 32001, 32600, 32657, 32653, 32702, 32775,
    {f: 2, c: 32782}, 32788, 32823, 32984, 32967, 32992, 32977, 32968, 32962,
    32976, 32965, 32995, 32985, 32988, 32970, 32981, 32969, 32975, 32983,
    32998, 32973, 33279, 33313, 33428, 33497, 33534, 33529, 33543, 33512,
    33536, 33493, 33594, 33515, 33494, 33524, 33516, 33505, 33522, 33525,
    33548, 33531, 33526, 33520, 33514, 33508, 33504, 33530, 33523, 33517,
    34423, 34420, 34428, 34419, 34881, 34894, 34919, 34922, 34921, 35283,
    35332, 35335, 36210, 36835, 36833, 36846, 36832, 37105, 37053, 37055,
    37077, 37061, 37054, 37063, 37067, 37064, [37332, 60294], 37331, 38484,
    38479, 38481, 38483, 38474, 38478, 20510, 20485, 20487, 20499, 20514,
    20528, 20507, 20469, 20468, 20531, 20535, 20524, {f: 2, c: 20470}, 20503,
    20508, 20512, 20519, 20533, 20527, 20529, 20494, 20826, 20884, 20883,
    20938, {f: 2, c: 20932}, 20936, 20942, 21089, 21082, 21074,
    {f: 2, c: 21086}, 21077, 21090, 21197, 21262, 21406, 21798, 21730, 21783,
    21778, 21735, 21747, 21732, 21786, 21759, 21764, 21768, 21739, 21777,
    21765, 21745, 21770, 21755, {f: 2, c: 21751}, 21728, 21774, 21763, 21771,
    {f: 2, c: 22273}, 22476, 22578, 22485, 22482, 22458, 22470, 22461, 22460,
    22456, 22454, 22463, 22471, 22480, 22457, 22465, 22798, 22858, 23065,
    23062, {f: 2, c: 23085}, 23061, 23055, 23063, 23050, 23070, 23091, 23404,
    23463, 23469, 23468, 23555, 23638, 23636, 23788, 23807, 23790, 23793,
    23799, 23808, 23801, 24105, 24104, 24232, 24238, 24234, 24236, 24371,
    24368, 24423, 24669, 24666, 24679, 24641, 24738, 24712, 24704, 24722,
    24705, 24733, 24707, 24725, 24731, 24727, 24711, 24732, 24718, 25113,
    25158, 25330, 25360, 25430, 25388, {f: 2, c: 25412}, 25398, 25411, 25572,
    25401, 25419, 25418, 25404, 25385, 25409, 25396, 25432, 25428, 25433,
    25389, 25415, 25395, 25434, 25425, 25400, 25431, 25408, 25416, 25930,
    25926, 26054, {f: 2, c: 26051}, 26050, 26186, 26207, 26183, 26193,
    {f: 2, c: 26386}, 26655, 26650, 26697, {f: 2, c: 26674}, 26683, 26699,
    26703, 26646, 26673, 26652, 26677, 26667, 26669, 26671, 26702, 26692,
    26676, 26653, 26642, 26644, 26662, 26664, 26670, 26701, 26682, 26661,
    26656, 27436, 27439, 27437, 27441, 27444, 27501, 32898, 27528, 27622,
    27620, 27624, 27619, 27618, 27623, 27685, 28026, {f: 2, c: 28003}, 28022,
    27917, 28001, 28050, 27992, 28002, 28013, 28015, 28049, 28045, 28143,
    28031, 28038, 27998, [28007, 59078], 28000, 28055, 28016, 28028, 27999,
    28034, 28056, 27951, 28008, 28043, 28030, 28032, 28036, 27926, 28035,
    28027, 28029, 28021, 28048, 28892, 28883, 28881, 28893, 28875, 32569,
    28898, 28887, 28882, 28894, 28896, 28884, 28877, {f: 3, c: 28869}, 28890,
    28878, 28897, 29250, 29304, 29303, 29302, 29440, 29434, 29428, 29438,
    29430, 29427, 29435, 29441, 29651, 29657, 29669, 29654, 29628, 29671,
    29667, 29673, 29660, 29650, 29659, 29652, 29661, 29658, {f: 2, c: 29655},
    29672, {f: 2, c: 29918}, {f: 2, c: 29940}, 29985, 30043, 30047, 30128,
    30145, 30139, 30148, 30144, 30143, 30134, 30138, 30346, 30409, 30493,
    30491, 30480, 30483, 30482, 30499, 30481, 30485, {f: 2, c: 30489}, 30498,
    30503, 30755, 30764, 30754, 30773, 30767, 30760, 30766, 30763, 30753,
    30761, 30771, 30762, 30769, 31060, 31067, 31055, 31068, 31059, 31058,
    31057, {f: 2, c: 31211}, 31200, 31214, 31213, 31210, 31196, 31198, 31197,
    31366, 31369, 31365, {f: 2, c: 31371}, 31370, 31367, 31448, 31504, 31492,
    31507, 31493, 31503, 31496, 31498, 31502, 31497, 31506, 31876, 31889,
    31882, 31884, 31880, 31885, 31877, 32030, 32029, 32017, 32014, 32024,
    32022, 32019, 32031, 32018, 32015, 32012, 32604, 32609, 32606, 32608,
    32605, 32603, 32662, 32658, 32707, 32706, 32704, 32790, 32830, 32825,
    33018, 33010, 33017, 33013, 33025, 33019, 33024, 33281, 33327, 33317,
    33587, 33581, 33604, 33561, 33617, 33573, 33622, 33599, 33601, 33574,
    33564, 33570, 33602, 33614, 33563, 33578, 33544, 33596, 33613, 33558,
    33572, 33568, 33591, 33583, 33577, 33607, 33605, 33612, 33619, 33566,
    33580, 33611, 33575, 33608, 34387, 34386, 34466, 34472, 34454, 34445,
    34449, 34462, 34439, 34455, 34438, 34443, 34458, 34437, 34469, 34457,
    34465, 34471, 34453, 34456, 34446, 34461, 34448, 34452, {f: 2, c: 34883},
    34925, {f: 2, c: 34933}, 34930, 34944, 34929, 34943, 34927, 34947, 34942,
    34932, 34940, 35346, 35911, 35927, 35963, 36004, 36003, 36214, 36216,
    36277, 36279, 36278, 36561, 36563, 36862, 36853, 36866, 36863, 36859,
    36868, 36860, 36854, 37078, 37088, {f: 2, c: 37081}, 37091, 37087, 37093,
    37080, 37083, 37079, 37084, 37092, 37200, {f: 2, c: 37198}, 37333, 37346,
    37338, 38492, 38495, 38588, 39139, [12221, 39647], [12223, 39727], 20095,
    20592, 20586, 20577, 20574, 20576, 20563, 20555, 20573, 20594, 20552,
    20557, 20545, 20571, 20554, 20578, 20501, 20549, 20575, 20585, 20587,
    {f: 2, c: 20579}, 20550, 20544, 20590, 20595, 20567, 20561, 20944, 21099,
    21101, 21100, 21102, 21206, 21203, 21293, 21404, {f: 2, c: 21877}, 21820,
    21837, 21840, 21812, 21802, 21841, 21858, 21814, 21813, 21808, 21842,
    21829, 21772, 21810, 21861, 21838, 21817, 21832, 21805, 21819, 21824,
    21835, 22282, 22279, 22523, 22548, 22498, 22518, 22492, 22516, 22528,
    22509, 22525, 22536, 22520, 22539, 22515, 22479, 22535, 22510, 22499,
    22514, 22501, 22508, 22497, 22542, 22524, 22544, 22503, 22529, 22540,
    22513, 22505, 22512, 22541, 22532, 22876, 23136, 23128, 23125,
    [23143, 60437], 23134, 23096, 23093, 23149, 23120, 23135, 23141, 23148,
    23123, 23140, 23127, 23107, 23133, 23122, 23108, 23131, 23112, 23182,
    23102, 23117, 23097, 23116, 23152, 23145, 23111, 23121, 23126, 23106,
    23132, 23410, 23406, 23489, 23488, 23641, 23838, 23819, 23837, 23834,
    23840, 23820, 23848, 23821, 23846, 23845, 23823, 23856, 23826, 23843,
    23839, 23854, 24126, 24116, 24241, 24244, 24249, {f: 2, c: 24242}, 24374,
    24376, 24475, 24470, 24479, 24714, 24720, 24710, 24766, 24752, 24762,
    {f: 2, c: 24787}, 24783, 24804, 24793, 24797, 24776, 24753, 24795, 24759,
    24778, 24767, 24771, 24781, 24768, 25394, 25445, 25482, 25474, 25469,
    25533, 25502, 25517, 25501, 25495, 25515, 25486, 25455, 25479, 25488,
    25454, 25519, 25461, 25500, 25453, 25518, 25468, 25508, 25403, 25503,
    25464, 25477, 25473, 25489, 25485, 25456, 25939, 26061, 26213, 26209,
    26203, 26201, 26204, 26210, 26392, 26745, 26759, 26768, 26780,
    {f: 2, c: 26733}, 26798, 26795, 26966, 26735, 26787, 26796, 26793, 26741,
    26740, 26802, 26767, 26743, 26770, 26748, 26731, 26738, 26794, 26752,
    26737, 26750, 26779, 26774, 26763, 26784, 26761, 26788, 26744, 26747,
    26769, 26764, 26762, 26749, 27446, 27443, {f: 2, c: 27447}, 27537, 27535,
    {f: 2, c: 27533}, 27532, 27690, 28096, 28075, 28084, 28083, 28276, 28076,
    28137, 28130, 28087, 28150, 28116, 28160, 28104, 28128, 28127, 28118,
    28094, 28133, {f: 2, c: 28124}, 28123, 28148, 28106, 28093, 28141, 28144,
    28090, 28117, 28098, 28111, 28105, 28112, 28146, 28115, 28157, 28119,
    28109, 28131, 28091, 28922, 28941, 28919, 28951, 28916, 28940, 28912,
    28932, 28915, 28944, 28924, 28927, 28934, 28947, 28928, 28920, 28918,
    28939, 28930, 28942, 29310, {f: 2, c: 29307}, 29311, 29469, 29463, 29447,
    29457, 29464, 29450, 29448, 29439, 29455, 29470, 29576, 29686, 29688,
    29685, 29700, 29697, 29693, 29703, 29696, 29690, 29692, 29695, 29708,
    29707, 29684, 29704, 30052, 30051, 30158, 30162, 30159, {f: 2, c: 30155},
    30161, 30160, 30351, 30345, 30419, 30521, 30511, 30509, {f: 2, c: 30513},
    30516, 30515, 30525, 30501, 30523, 30517, 30792, 30802, 30793, 30797,
    30794, 30796, 30758, 30789, 30800, 31076, 31079, {f: 2, c: 31081}, 31075,
    31083, 31073, 31163, 31226, 31224, {f: 2, c: 31222}, 31375, 31380, 31376,
    31541, 31547, 31540, 31525, 31536, 31522, 31524, 31539, 31512, 31530,
    31517, 31537, 31531, 31533, 31535, 31538, 31544, 31514, 31523, 31892,
    31896, 31894, 31907, 32053, 32061, 32056, 32054, 32058, 32069, 32044,
    32041, 32065, 32071, {f: 2, c: 32062}, 32074, 32059, 32040, 32611, 32661,
    {f: 2, c: 32668}, 32667, {f: 2, c: 32714}, 32717, {f: 2, c: 32720}, 32711,
    32719, 32713, 32799, 32798, 32795, 32839, 32835, 32840, 33048, 33061,
    33049, 33051, 33069, 33055, 33068, 33054, 33057, 33045, 33063, 33053,
    33058, 33297, 33336, 33331, 33338, 33332, 33330, 33396, 33680, 33699,
    33704, 33677, 33658, 33651, 33700, 33652, 33679, 33665, 33685, 33689,
    33653, 33684, 33705, 33661, 33667, 33676, 33693, 33691, 33706, 33675,
    33662, 33701, 33711, 33672, 33687, 33712, 33663, 33702, 33671, 33710,
    33654, 34393, 34390, 34495, 34487, 34498, 34497, 34501, 34490, 34480,
    34504, 34489, 34483, 34488, 34508, 34484, {f: 2, c: 34491}, 34499,
    {f: 2, c: 34493}, 34898, 34953, 34965, 34984, 34978, 34986, 34970, 34961,
    34977, 34975, 34968, 34983, 34969, 34971, 34967, 34980, 34988, 34956,
    34963, 34958, 35202, 35286, 35289, 35285, 35376, 35367, 35372, 35358,
    35897, 35899, {f: 2, c: 35932}, 35965, 36005, 36221, 36219, 36217, 36284,
    36290, 36281, 36287, 36289, 36568, 36574, 36573, 36572, 36567,
    {f: 2, c: 36576}, 36900, 36875, 36881, 36892, 36876, 36897, 37103, 37098,
    37104, 37108, {f: 2, c: 37106}, 37076, {f: 2, c: 37099}, 37097, 37206,
    37208, 37210, 37203, 37205, 37356, 37364, 37361, 37363, 37368, 37348,
    37369, {f: 2, c: 37354}, 37367, 37352, 37358, 38266, 38278, 38280, 38524,
    38509, 38507, 38513, 38511, 38591, 38762, 38916, 39141, 39319, 20635,
    20629, 20628, 20638, 20619, 20643, 20611, 20620, 20622, 20637, 20584,
    20636, 20626, 20610, 20615, 20831, 20948, 21266, 21265, 21412, 21415,
    21905, 21928, 21925, 21933, 21879, 22085, 21922, 21907, 21896, 21903,
    21941, 21889, 21923, 21906, 21924, 21885, 21900, 21926, 21887, 21909,
    21921, 21902, 22284, 22569, 22583, 22553, 22558, 22567, 22563, 22568,
    22517, 22600, 22565, 22556, 22555, 22579, 22591, 22582, 22574, 22585,
    22584, 22573, 22572, 22587, 22881, 23215, 23188, 23199, 23162, 23202,
    23198, 23160, 23206, 23164, 23205, 23212, 23189, 23214, 23095, 23172,
    23178, 23191, 23171, 23179, 23209, 23163, 23165, 23180, 23196, 23183,
    23187, 23197, 23530, 23501, 23499, 23508, 23505, 23498, 23502, 23564,
    23600, 23863, 23875, 23915, 23873, 23883, 23871, 23861, 23889, 23886,
    23893, 23859, 23866, 23890, 23869, 23857, 23897, 23874, 23865, 23881,
    23864, 23868, 23858, 23862, 23872, 23877, 24132, 24129, [24408, 57673],
    24486, 24485, 24491, 24777, 24761, 24780, 24802, 24782, 24772, 24852,
    24818, 24842, 24854, 24837, 24821, 24851, 24824, 24828, 24830, 24769,
    24835, 24856, 24861, 24848, 24831, 24836, 24843, 25162, 25492, 25521,
    25520, 25550, 25573, 25576, 25583, 25539, 25757, 25587, 25546, 25568,
    25590, 25557, 25586, 25589, 25697, 25567, 25534, 25565, 25564, 25540,
    25560, 25555, 25538, 25543, 25548, 25547, 25544, 25584, 25559, 25561,
    25906, 25959, 25962, 25956, 25948, 25960, 25957, 25996, {f: 2, c: 26013},
    26030, 26064, 26066, 26236, 26220, 26235, 26240, 26225, 26233, 26218,
    26226, 26369, 26892, 26835, 26884, 26844, 26922, 26860, 26858, 26865,
    26895, 26838, 26871, 26859, 26852, 26870, 26899, 26896, 26867, 26849,
    26887, 26828, 26888, 26992, 26804, 26897, 26863, 26822, 26900, 26872,
    26832, 26877, 26876, 26856, 26891, 26890, 26903, 26830, 26824,
    {f: 2, c: 26845}, 26854, 26868, 26833, 26886, 26836, 26857, 26901, 26917,
    26823, 27449, 27451, 27455, 27452, 27540, 27543, 27545, 27541, 27581,
    27632, {f: 2, c: 27634}, 27696, 28156, {f: 2, c: 28230}, 28191, 28233,
    28296, {f: 2, c: 28220}, 28229, 28258, 28203, 28223, 28225, 28253, 28275,
    28188, 28211, 28235, 28224, 28241, 28219, 28163, 28206, 28254, 28264,
    28252, 28257, 28209, 28200, 28256, 28273, 28267, 28217, 28194, 28208,
    28243, 28261, 28199, 28280, 28260, 28279, 28245, 28281, 28242, 28262,
    {f: 2, c: 28213}, 28250, 28960, 28958, 28975, 28923, 28974, 28977, 28963,
    28965, 28962, 28978, 28959, 28968, 28986, 28955, 29259, 29274,
    {f: 2, c: 29320}, 29318, 29317, 29323, 29458, 29451, 29488, 29474, 29489,
    29491, 29479, 29490, 29485, 29478, 29475, 29493, 29452, 29742, 29740,
    29744, 29739, 29718, 29722, 29729, 29741, 29745, 29732, 29731, 29725,
    29737, 29728, 29746, 29947, 29999, 30063, 30060, 30183, 30170, 30177,
    30182, 30173, 30175, 30180, 30167, 30357, 30354, 30426, {f: 2, c: 30534},
    30532, 30541, 30533, 30538, 30542, {f: 2, c: 30539}, 30686, 30700, 30816,
    {f: 2, c: 30820}, 30812, 30829, 30833, 30826, 30830, 30832, 30825, 30824,
    30814, 30818, 31092, 31091, 31090, 31088, 31234, 31242, 31235, 31244,
    31236, 31385, 31462, 31460, 31562, 31559, 31556, 31560, 31564, 31566,
    31552, 31576, 31557, 31906, 31902, 31912, 31905, 32088, 32111, 32099,
    32083, 32086, 32103, 32106, 32079, 32109, 32092, 32107, 32082, 32084,
    32105, 32081, 32095, 32078, {f: 2, c: 32574}, {f: 2, c: 32613}, 32674,
    {f: 2, c: 32672}, 32727, 32849, {f: 2, c: 32847}, 33022, 32980, 33091,
    33098, 33106, 33103, 33095, 33085, 33101, 33082, 33254, 33262,
    {f: 3, c: 33271}, 33284, {f: 2, c: 33340}, 33343, 33397, 33595,
    [33743, 60382], 33785, 33827, 33728, 33768, 33810, 33767, 33764, 33788,
    33782, 33808, 33734, 33736, 33771, 33763, 33727, 33793, 33757, 33765,
    33752, 33791, 33761, 33739, 33742, 33750, 33781, 33737, 33801,
    [33807, 58332], 33758, 33809, 33798, 33730, 33779, 33749, 33786, 33735,
    33745, 33770, 33811, 33690, 33731, 33772, 33774, 33732, 33787, 33751,
    33762, 33819, 33755, 33790, 34520, 34530, 34534, 34515, 34531, 34522,
    34538, 34525, 34539, 34524, 34540, 34537, 34519, 34536, 34513, 34888,
    34902, 34901, 35002, 35031, 35001, 35000, 35008, 35006, 34998, 35004,
    34999, 35005, 34994, 35073, 35017, 35221, 35224, 35223, 35293,
    {f: 2, c: 35290}, 35406, 35405, 35385, 35417, 35392, {f: 2, c: 35415},
    {f: 2, c: 35396}, 35410, 35400, 35409, 35402, 35404, 35407, 35935, 35969,
    35968, 36026, 36030, 36016, 36025, 36021, 36228, 36224, 36233, 36312,
    36307, 36301, 36295, 36310, 36316, 36303, 36309, 36313, 36296, 36311,
    36293, 36591, 36599, 36602, 36601, 36582, 36590, 36581, 36597,
    {f: 2, c: 36583}, 36598, 36587, 36593, 36588, 36596, 36585, 36909, 36916,
    36911, 37126, 37164, [37124, 60367], 37119, 37116, 37128, 37113, 37115,
    37121, 37120, 37127, 37125, 37123, 37217, 37220, 37215, 37218, 37216,
    37377, 37386, 37413, 37379, 37402, 37414, 37391, 37388, 37376, 37394,
    37375, 37373, 37382, 37380, 37415, 37378, 37404, 37412, 37401, 37399,
    37381, 37398, 38267, 38285, 38284, 38288, 38535, 38526, {f: 2, c: 38536},
    38531, 38528, 38594, 38600, 38595, 38641, 38640, 38764, 38768, 38766,
    38919, 39081, 39147, 40166, [12235, 40697], {f: 2, c: 20099}, 20150, 20669,
    20671, 20678, 20654, 20676, 20682, 20660, 20680, 20674, 20656, 20673,
    20666, 20657, 20683, 20681, 20662, 20664, 20951, 21114, 21112,
    {f: 2, c: 21115}, 21955, 21979, 21964, 21968, 21963, 21962, 21981,
    [21952, 64013], 21972, 21956, 21993, 21951, 21970, 21901, 21967, 21973,
    21986, 21974, 21960, 22002, 21965, 21977, 21954, 22292, 22611, 22632,
    22628, 22607, 22605, 22601, 22639, 22613, 22606, 22621, 22617, 22629,
    22619, 22589, 22627, 22641, 22780, 23239, 23236, 23243, 23226, 23224,
    23217, 23221, 23216, 23231, 23240, 23227, 23238, 23223, 23232, 23242,
    23220, 23222, 23245, 23225, 23184, 23510, {f: 2, c: 23512}, 23583, 23603,
    23921, 23907, 23882, 23909, 23922, 23916, 23902, 23912, 23911, 23906,
    24048, 24143, 24142, 24138, 24141, 24139, 24261, 24268, 24262, 24267,
    24263, 24384, 24495, 24493, 24823, {f: 2, c: 24905}, 24875, 24901, 24886,
    24882, 24878, 24902, 24879, 24911, 24873, 24896, 25120, 37224, 25123,
    25125, 25124, 25541, 25585, 25579, 25616, 25618, 25609, 25632, 25636,
    25651, 25667, 25631, 25621, 25624, 25657, 25655, {f: 2, c: 25634}, 25612,
    25638, 25648, 25640, 25665, 25653, 25647, 25610, 25626, 25664, 25637,
    25639, 25611, 25575, 25627, 25646, 25633, 25614, 25967, 26002, 26067,
    26246, 26252, 26261, 26256, 26251, 26250, 26265, 26260, 26232, 26400,
    26982, 26975, 26936, 26958, 26978, 26993, 26943, 26949, 26986, 26937,
    26946, 26967, 26969, 27002, {f: 2, c: 26952}, 26933, 26988, 26931, 26941,
    26981, 26864, 27000, 26932, 26985, 26944, 26991, 26948, 26998, 26968,
    26945, 26996, 26956, 26939, 26955, 26935, 26972, 26959, 26961, 26930,
    26962, 26927, 27003, 26940, 27462, 27461, 27459, 27458, 27464, 27457,
    27547, {f: 2, c: 27643}, 27641, {f: 2, c: 27639}, 28315, 28374, 28360,
    28303, 28352, 28319, {f: 2, c: 28307}, 28320, 28337, 28345, 28358, 28370,
    28349, 28353, 28318, 28361, 28343, 28336, 28365, 28326, 28367, 28338,
    28350, 28355, 28380, 28376, 28313, 28306, 28302, 28301, 28324, 28321,
    28351, 28339, 28368, 28362, 28311, 28334, 28323, 28999, 29012, 29010,
    29027, 29024, 28993, 29021, [29026, 61080], 29042, 29048, 29034, 29025,
    28994, 29016, 28995, 29003, 29040, 29023, 29008, 29011, 28996, 29005,
    29018, 29263, 29325, 29324, 29329, 29328, 29326, 29500, 29506, 29499,
    29498, 29504, 29514, 29513, 29764, {f: 2, c: 29770}, 29778, 29777, 29783,
    29760, {f: 2, c: 29775}, 29774, 29762, 29766, 29773, 29780, 29921, 29951,
    29950, 29949, 29981, 30073, 30071, 27011, 30191, 30223, 30211, 30199,
    30206, 30204, [30201, 60782], 30200, 30224, 30203, 30198, 30189, 30197,
    30205, 30361, 30389, 30429, 30549, {f: 2, c: 30559}, 30546, 30550, 30554,
    30569, 30567, 30548, 30553, 30573, 30688, 30855, 30874, 30868, 30863,
    30852, 30869, {f: 2, c: 30853}, 30881, 30851, 30841, 30873, 30848, 30870,
    30843, 31100, 31106, 31101, 31097, 31249, {f: 2, c: 31256}, 31250, 31255,
    31253, 31266, 31251, 31259, 31248, 31395, 31394, 31390, 31467, 31590,
    31588, 31597, 31604, 31593, 31602, 31589, 31603, 31601, 31600, 31585,
    31608, 31606, 31587, 31922, 31924, 31919, 32136, 32134, 32128, 32141,
    32127, 32133, 32122, 32142, 32123, 32131, 32124, 32140, 32148, 32132,
    32125, 32146, 32621, 32619, {f: 2, c: 32615}, 32620, 32678, 32677, 32679,
    {f: 2, c: 32731}, 32801, 33124, 33120, 33143, 33116, 33129, 33115, 33122,
    33138, 26401, 33118, 33142, 33127, 33135, 33092, 33121, 33309, 33353,
    33348, 33344, 33346, 33349, 34033, 33855, 33878, 33910, 33913, 33935,
    33933, 33893, 33873, 33856, 33926, 33895, 33840, 33869, 33917, 33882,
    33881, 33908, 33907, 33885, 34055, 33886, 33847, 33850, 33844, 33914,
    33859, 33912, 33842, 33861, 33833, 33753, 33867, 33839, 33858, 33837,
    33887, 33904, 33849, 33870, 33868, 33874, 33903, 33989, 33934, 33851,
    33863, 33846, 33843, 33896, 33918, 33860, 33835, 33888, 33876, 33902,
    33872, 34571, 34564, 34551, 34572, 34554, 34518, 34549, 34637, 34552,
    34574, 34569, 34561, 34550, 34573, 34565, 35030, 35019, {f: 2, c: 35021},
    35038, 35035, 35034, 35020, 35024, 35205, 35227, 35295, 35301, 35300,
    35297, 35296, 35298, 35292, 35302, 35446, 35462, 35455, 35425, 35391,
    35447, 35458, 35460, 35445, 35459, 35457, 35444, 35450, 35900, 35915,
    35914, 35941, 35940, 35942, 35974, {f: 2, c: 35972}, 36044,
    {f: 2, c: 36200}, 36241, 36236, {f: 2, c: 36238}, 36237, {f: 2, c: 36243},
    36240, 36242, 36336, 36320, 36332, 36337, 36334, 36304, 36329, 36323,
    36322, 36327, 36338, 36331, 36340, 36614, 36607, 36609, 36608, 36613,
    {f: 2, c: 36615}, 36610, [36619, 60507], 36946, 36927, 36932, 36937, 36925,
    37136, 37133, 37135, 37137, 37142, 37140, 37131, 37134, {f: 2, c: 37230},
    37448, 37458, 37424, 37434, 37478, 37427, 37477, 37470, 37507, 37422,
    37450, 37446, 37485, 37484, 37455, 37472, 37479, 37487, 37430, 37473,
    37488, 37425, 37460, 37475, 37456, 37490, 37454, 37459, 37452, 37462,
    37426, 38303, 38300, 38302, 38299, {f: 2, c: 38546}, 38545, 38551, 38606,
    38650, 38653, 38648, 38645, 38771, {f: 2, c: 38775}, 38770, 38927,
    {f: 2, c: 38925}, 39084, 39158, 39161, 39343, 39346, 39344, 39349, 39597,
    39595, 39771, 40170, 40173, 40167, 40576, [12236, 40701], 20710, 20692,
    20695, 20712, 20723, 20699, 20714, 20701, 20708, 20691, 20716, 20720,
    20719, 20707, 20704, 20952, {f: 2, c: 21120}, 21225, 21227, 21296, 21420,
    22055, 22037, 22028, 22034, 22012, 22031, 22044, 22017, 22035, 22018,
    22010, 22045, 22020, 22015, 22009, 22665, 22652, 22672, 22680, 22662,
    22657, 22655, 22644, 22667, 22650, 22663, 22673, 22670, 22646, 22658,
    22664, 22651, 22676, 22671, 22782, 22891, 23260, 23278, 23269, 23253,
    23274, 23258, 23277, 23275, 23283, 23266, 23264, 23259, 23276, 23262,
    23261, 23257, 23272, 23263, 23415, 23520, 23523, 23651, 23938, 23936,
    23933, 23942, 23930, 23937, 23927, 23946, 23945, 23944, 23934, 23932,
    23949, 23929, 23935, {f: 2, c: 24152}, 24147, 24280, 24273, 24279, 24270,
    24284, 24277, 24281, 24274, 24276, 24388, 24387, 24431, 24502, 24876,
    24872, 24897, 24926, 24945, 24947, {f: 2, c: 24914}, 24946, 24940, 24960,
    24948, 24916, 24954, 24923, 24933, 24891, 24938, 24929, 24918, 25129,
    25127, 25131, 25643, 25677, 25691, 25693, 25716, 25718, {f: 2, c: 25714},
    25725, 25717, 25702, 25766, 25678, 25730, 25694, 25692, 25675, 25683,
    25696, 25680, 25727, 25663, 25708, 25707, 25689, 25701, 25719, 25971,
    26016, 26273, 26272, 26271, 26373, 26372, 26402, 27057, 27062, 27081,
    27040, 27086, 27030, 27056, 27052, 27068, 27025, 27033, 27022, 27047,
    27021, 27049, 27070, 27055, 27071, 27076, 27069, 27044, 27092, 27065,
    27082, 27034, 27087, 27059, 27027, 27050, 27041, 27038, 27097, 27031,
    27024, 27074, 27061, 27045, 27078, 27466, 27469, 27467, {f: 3, c: 27550},
    {f: 2, c: 27587}, 27646, 28366, 28405, 28401, 28419, 28453, 28408, 28471,
    28411, 28462, 28425, 28494, {f: 2, c: 28441}, 28455, 28440, 28475, 28434,
    28397, 28426, 28470, 28531, 28409, 28398, 28461, 28480, 28464, 28476,
    28469, 28395, 28423, 28430, 28483, 28421, 28413, 28406, 28473, 28444,
    28412, 28474, 28447, 28429, 28446, 28424, 28449, 29063, 29072, 29065,
    29056, 29061, 29058, 29071, 29051, 29062, 29057, 29079, 29252, 29267,
    29335, 29333, 29331, 29507, 29517, 29521, 29516, 29794, 29811, 29809,
    29813, 29810, 29799, 29806, 29952, {f: 2, c: 29954}, 30077, 30096, 30230,
    30216, 30220, 30229, 30225, 30218, 30228, 30392, 30593, 30588, 30597,
    30594, 30574, 30592, 30575, 30590, 30595, 30898, 30890, 30900, 30893,
    30888, 30846, 30891, 30878, 30885, 30880, 30892, 30882, 30884, 31128,
    {f: 2, c: 31114}, 31126, 31125, 31124, 31123, 31127, 31112, 31122, 31120,
    31275, 31306, 31280, 31279, 31272, 31270, 31400, {f: 2, c: 31403}, 31470,
    31624, 31644, 31626, 31633, 31632, 31638, 31629, 31628, 31643, 31630,
    31621, 31640, 21124, 31641, 31652, 31618, 31931, 31935, 31932, 31930,
    32167, 32183, 32194, 32163, 32170, 32193, 32192, 32197, 32157, 32206,
    32196, 32198, {f: 2, c: 32203}, 32175, 32185, 32150, 32188, 32159, 32166,
    32174, 32169, 32161, 32201, 32627, {f: 2, c: 32738}, 32741, 32734, 32804,
    32861, 32860, 33161, 33158, 33155, 33159, 33165, 33164, 33163, 33301,
    33943, 33956, 33953, 33951, 33978, 33998, 33986, 33964, 33966, 33963,
    33977, 33972, 33985, 33997, 33962, 33946, 33969, 34000, 33949, 33959,
    33979, 33954, 33940, 33991, 33996, 33947, 33961, 33967, [33960, 58327],
    34006, 33944, 33974, 33999, 33952, 34007, 34004, 34002, 34011, 33968,
    33937, 34401, 34611, 34595, 34600, 34667, 34624, 34606, 34590, 34593,
    34585, 34587, 34627, 34604, 34625, 34622, 34630, 34592, 34610, 34602,
    34605, 34620, 34578, 34618, 34609, 34613, 34626, {f: 2, c: 34598}, 34616,
    34596, 34586, 34608, 34577, 35063, 35047, {f: 2, c: 35057}, 35066, 35070,
    35054, 35068, 35062, 35067, 35056, 35052, 35051, 35229, 35233, 35231,
    35230, 35305, 35307, 35304, 35499, 35481, 35467, 35474, 35471, 35478,
    35901, {f: 2, c: 35944}, 36053, 36047, 36055, 36246, 36361, 36354, 36351,
    36365, 36349, 36362, 36355, 36359, 36358, 36357, 36350, 36352, 36356,
    {f: 2, c: 36624}, 36622, 36621, 37155, 37148, 37152, 37154, 37151, 37149,
    37146, 37156, 37153, 37147, 37242, 37234, 37241, 37235, 37541, 37540,
    37494, 37531, 37498, 37536, 37524, 37546, 37517, 37542, 37530, 37547,
    37497, 37527, 37503, 37539, 37614, 37518, 37506, 37525, 37538, 37501,
    37512, 37537, 37514, 37510, 37516, 37529, 37543, 37502, 37511, 37545,
    37533, 37515, 37421, 38558, 38561, 38655, 38744, 38781, 38778, 38782,
    38787, 38784, 38786, 38779, 38788, 38785, 38783, 38862, 38861, 38934,
    {f: 2, c: 39085}, 39170, 39168, 39175, 39325, 39324, 39363, 39353, 39355,
    39354, 39362, 39357, 39367, 39601, 39651, 39655, {f: 2, c: 39742},
    {f: 2, c: 39776}, 39775, {f: 2, c: 40177}, 40181, 40615, 20735, 20739,
    20784, 20728, {f: 2, c: 20742}, 20726, 20734, {f: 2, c: 20747}, 20733,
    20746, {f: 2, c: 21131}, 21233, 21231, 22088, 22082, 22092, 22069, 22081,
    22090, 22089, 22086, 22104, 22106, 22080, 22067, 22077, 22060, 22078,
    22072, 22058, 22074, 22298, 22699, 22685, 22705, 22688, 22691, 22703,
    22700, 22693, 22689, 22783, 23295, 23284, 23293, 23287, 23286, 23299,
    23288, 23298, 23289, 23297, 23303, 23301, 23311, 23655, 23961, 23959,
    23967, 23954, 23970, 23955, 23957, 23968, 23964, 23969, 23962, 23966,
    24169, 24157, 24160, 24156, 32243, 24283, 24286, 24289, 24393, 24498,
    24971, 24963, 24953, 25009, 25008, 24994, 24969, 24987, 24979, 25007,
    25005, 24991, 24978, 25002, 24993, 24973, 24934, 25011, 25133, 25710,
    25712, 25750, 25760, 25733, 25751, 25756, 25743, 25739, 25738, 25740,
    25763, 25759, 25704, 25777, 25752, 25974, 25978, 25977, 25979,
    {f: 2, c: 26034}, 26293, 26288, 26281, 26290, 26295, 26282, 26287, 27136,
    27142, 27159, 27109, 27128, 27157, 27121, 27108, 27168, 27135, 27116,
    27106, 27163, 27165, 27134, 27175, 27122, 27118, 27156, 27127, 27111,
    27200, 27144, 27110, 27131, 27149, 27132, 27115, 27145, 27140, 27160,
    27173, 27151, 27126, 27174, 27143, 27124, 27158, 27473, 27557, 27555,
    27554, 27558, 27649, 27648, 27647, 27650, 28481, 28454, 28542, 28551,
    28614, 28562, 28557, 28553, 28556, 28514, 28495, 28549, 28506, 28566,
    28534, 28524, 28546, 28501, 28530, 28498, 28496, 28503, 28564, 28563,
    28509, 28416, 28513, 28523, 28541, 28519, 28560, 28499, 28555, 28521,
    28543, 28565, 28515, 28535, 28522, 28539, 29106, 29103, 29083, 29104,
    29088, 29082, 29097, 29109, 29085, 29093, 29086, 29092, 29089, 29098,
    29084, 29095, 29107, 29336, 29338, 29528, 29522, {f: 3, c: 29534}, 29533,
    29531, 29537, 29530, 29529, 29538, 29831, {f: 2, c: 29833}, 29830, 29825,
    29821, 29829, 29832, 29820, [29817, 58868], 29960, 29959, 30078, 30245,
    30238, 30233, 30237, 30236, 30243, 30234, 30248, 30235, {f: 3, c: 30364},
    30363, 30605, 30607, 30601, 30600, 30925, 30907, 30927, 30924, 30929,
    30926, 30932, 30920, {f: 2, c: 30915}, 30921, 31130, 31137, 31136, 31132,
    31138, [31131, 59175], 27510, 31289, 31410, 31412, 31411, 31671, 31691,
    31678, 31660, 31694, 31663, 31673, 31690, 31669, 31941, 31944, 31948,
    31947, 32247, 32219, 32234, 32231, 32215, 32225, 32259, 32250, 32230,
    32246, 32241, 32240, 32238, 32223, 32630, 32684, 32688, 32685, 32749,
    32747, 32746, 32748, 32742, 32744, 32868, 32871, 33187, 33183, 33182,
    33173, 33186, 33177, 33175, 33302, 33359, 33363, 33362, 33360, 33358,
    33361, 34084, 34107, 34063, 34048, 34089, 34062, 34057, 34061, 34079,
    34058, 34087, 34076, 34043, 34091, 34042, 34056, 34060, 34036, 34090,
    34034, 34069, 34039, 34027, 34035, 34044, 34066, 34026, 34025, 34070,
    34046, 34088, 34077, 34094, 34050, 34045, 34078, 34038, 34097, 34086,
    {f: 2, c: 34023}, 34032, 34031, 34041, 34072, 34080, 34096, 34059, 34073,
    34095, 34402, 34646, {f: 2, c: 34659}, 34679, 34785, 34675, 34648, 34644,
    34651, 34642, 34657, 34650, 34641, 34654, 34669, 34666, 34640, 34638,
    34655, 34653, 34671, 34668, 34682, 34670, 34652, 34661, 34639, 34683,
    34677, 34658, 34663, 34665, 34906, 35077, 35084, 35092, 35083,
    {f: 3, c: 35095}, 35078, 35094, 35089, 35086, 35081, 35234, 35236, 35235,
    35309, 35312, 35308, 35535, 35526, 35512, 35539, 35537, {f: 2, c: 35540},
    35515, 35543, 35518, 35520, 35525, 35544, 35523, 35514, 35517, 35545,
    35902, 35917, 35983, 36069, 36063, 36057, 36072, 36058, 36061, 36071,
    36256, 36252, 36257, 36251, 36384, 36387, 36389, 36388, 36398, 36373,
    36379, 36374, 36369, 36377, {f: 2, c: 36390}, 36372, 36370, 36376, 36371,
    36380, 36375, 36378, 36652, 36644, 36632, 36634, 36640, 36643,
    {f: 2, c: 36630}, 36979, 36976, 36975, 36967, 36971, 37167, 37163,
    {f: 2, c: 37161}, 37170, 37158, 37166, {f: 2, c: 37253}, 37258,
    {f: 2, c: 37249}, 37252, 37248, 37584, {f: 2, c: 37571}, 37568, 37593,
    37558, 37583, 37617, 37599, 37592, 37609, 37591, 37597, 37580, 37615,
    37570, 37608, 37578, 37576, 37582, 37606, 37581, 37589, 37577, 37600,
    37598, 37607, 37585, 37587, 37557, 37601, 37669, 37574, 37556, 38268,
    38316, 38315, 38318, 38320, 38564, 38562, 38611, 38661, 38664, 38658,
    38746, 38794, 38798, 38792, 38864, 38863, 38942, 38941, 38950, 38953,
    38952, 38944, 38939, 38951, 39090, 39176, 39162, 39185, 39188,
    {f: 2, c: 39190}, 39189, 39388, 39373, 39375, {f: 2, c: 39379}, 39374,
    39369, [39382, 60270], 39384, 39371, 39383, 39372, 39603, 39660, 39659,
    39667, 39666, 39665, 39750, 39747, 39783, 39796, 39793, 39782, 39798,
    39797, 39792, 39784, 39780, 39788, 40188, 40186, 40189, 40191, 40183,
    40199, 40192, 40185, 40187, 40200, 40197, 40196, 40579, 40659,
    {f: 2, c: 40719}, 20764, 20755, 20759, 20762, 20753, 20958, 21300, 21473,
    22128, 22112, 22126, 22131, 22118, 22115, 22125, 22130, 22110, 22135,
    22300, 22299, 22728, 22717, 22729, 22719, 22714, 22722, 22716, 22726,
    23319, 23321, 23323, 23329, 23316, 23315, 23312, 23318, [23336, 59539],
    23322, 23328, 23326, 23535, 23980, 23985, 23977, 23975, 23989, 23984,
    23982, 23978, 23976, 23986, 23981, 23983, 23988, {f: 2, c: 24167}, 24166,
    24175, 24297, 24295, 24294, 24296, 24293, 24395, 24508, 24507, 24989,
    25000, 24982, 25029, 25012, 25030, 25025, 25036, 25018, 25023, 25016,
    24972, 25815, 25814, 25808, 25807, 25801, 25789, 25737, 25795, 25819,
    25843, 25817, 25907, 25983, 25980, 26018, 26312, 26302, 26304,
    {f: 2, c: 26314}, 26319, 26301, 26299, 26298, 26316, 26403, 27188, 27238,
    27209, 27239, 27186, 27240, 27198, 27229, 27245, 27254, 27227, 27217,
    27176, 27226, 27195, 27199, 27201, 27242, 27236, 27216, 27215, 27220,
    27247, 27241, 27232, 27196, 27230, 27222, 27221, {f: 2, c: 27213}, 27206,
    27477, 27476, 27478, 27559, {f: 2, c: 27562}, 27592, 27591, 27652, 27651,
    27654, 28589, 28619, 28579, 28615, 28604, 28622, 28616, 28510, 28612,
    28605, 28574, 28618, 28584, 28676, 28581, 28590, 28602, 28588, 28586,
    28623, 28607, 28600, 28578, 28617, 28587, 28621, 28591, 28594, 28592,
    29125, 29122, 29119, 29112, 29142, {f: 2, c: 29120}, 29131, 29140, 29130,
    29127, 29135, 29117, 29144, 29116, 29126, {f: 2, c: 29146},
    {f: 2, c: 29341}, 29545, {f: 2, c: 29542}, 29548, 29541, 29547, 29546,
    29823, 29850, 29856, 29844, 29842, 29845, 29857, 29963, 30080, 30255,
    30253, 30257, 30269, 30259, 30268, 30261, 30258, 30256, 30395, 30438,
    30618, 30621, 30625, 30620, 30619, {f: 2, c: 30626}, 30613, 30617, 30615,
    30941, 30953, 30949, 30954, 30942, 30947, 30939, {f: 2, c: 30945}, 30957,
    {f: 2, c: 30943}, 31140, 31300, 31304, 31303, 31414, 31416, 31413, 31409,
    31415, 31710, 31715, 31719, 31709, 31701, 31717, 31706, 31720, 31737,
    31700, 31722, 31714, 31708, 31723, 31704, 31711, 31954, 31956, 31959,
    {f: 2, c: 31952}, 32274, 32289, 32279, 32268, {f: 2, c: 32287}, 32275,
    32270, 32284, 32277, 32282, 32290, 32267, 32271, 32278, 32269, 32276,
    32293, 32292, 32579, {f: 2, c: 32635}, 32634, 32689, 32751, 32810, 32809,
    32876, 33201, 33190, 33198, 33209, 33205, 33195, 33200, 33196, 33204,
    33202, 33207, 33191, 33266, {f: 3, c: 33365}, 34134, 34117, 34155, 34125,
    34131, 34145, 34136, 34112, 34118, 34148, 34113, 34146, 34116, 34129,
    34119, 34147, 34110, 34139, 34161, 34126, 34158, 34165, 34133, 34151,
    34144, 34188, 34150, 34141, 34132, 34149, 34156, 34403, 34405, 34404,
    34724, 34715, 34703, 34711, 34707, 34706, 34696, 34689, 34710, 34712,
    34681, 34695, 34723, 34693, {f: 2, c: 34704}, 34717, 34692, 34708, 34716,
    34714, 34697, 35102, 35110, 35120, {f: 2, c: 35117}, 35111, 35121, 35106,
    35113, 35107, 35119, 35116, 35103, 35313, 35552, 35554, 35570,
    {f: 2, c: 35572}, 35549, 35604, 35556, 35551, 35568, 35528, 35550, 35553,
    35560, 35583, 35567, 35579, {f: 2, c: 35985}, 35984, 36085, 36078, 36081,
    36080, 36083, 36204, 36206, 36261, 36263, 36403, 36414, 36408, 36416,
    36421, 36406, {f: 2, c: 36412}, 36417, 36400, 36415, 36541, [36662, 60329],
    36654, 36661, 36658, 36665, 36663, 36660, 36982, 36985, 36987, 36998,
    37114, 37171, {f: 2, c: 37173}, 37267, {f: 2, c: 37264}, 37261, 37263,
    37671, 37662, 37640, 37663, 37638, 37647, 37754, 37688, 37692, 37659,
    37667, 37650, 37633, 37702, 37677, 37646, 37645, 37579, 37661, 37626,
    37651, 37625, 37623, 37684, 37634, 37668, 37631, 37673, 37689, 37685,
    37674, 37652, 37644, 37643, 37630, 37641, 37632, 37627, 37654, 38332,
    38349, 38334, {f: 2, c: 38329}, 38326, 38335, 38325, 38333, 38569, 38612,
    38667, 38674, 38672, 38809, 38807, 38804, 38896, 38904, 38965, 38959,
    38962, 39204, 39199, 39207, 39209, 39326, 39406, 39404, 39397, 39396,
    39408, 39395, 39402, 39401, 39399, 39609, 39615, 39604, 39611, 39670,
    39674, 39673, 39671, 39731, 39808, 39813, 39815, 39804, 39806, 39803,
    39810, 39827, 39826, 39824, 39802, 39829, 39805, 39816, 40229, 40215,
    40224, 40222, 40212, 40233, 40221, 40216, 40226, 40208, 40217, 40223,
    40584, {f: 2, c: 40582}, 40622, 40621, {f: 2, c: 40661}, 40698, 40722,
    40765, 20774, 20773, 20770, 20772, 20768, 20777, 21236, 22163,
    {f: 2, c: 22156}, 22150, 22148, 22147, 22142, 22146, 22143, 22145, 22742,
    22740, 22735, 22738, 23341, 23333, 23346, 23331, 23340, 23335, 23334,
    23343, 23342, 23419, {f: 2, c: 23537}, 23991, 24172, 24170, 24510, 25027,
    25013, 25020, 25063, 25056, 25061, 25060, 25064, 25054, 25839, 25833,
    25827, 25835, 25828, 25832, 25985, 25984, 26038, 26074, 26322, 27277,
    27286, 27265, 27301, 27273, 27295, 27291, 27297, 27294, 27271, 27283,
    27278, 27285, 27267, 27304, 27300, 27281, 27263, 27302, 27290, 27269,
    27276, 27282, 27483, 27565, 27657, 28620, 28585, 28660, 28628, 28643,
    28636, 28653, 28647, 28646, 28638, 28658, 28637, 28642, 28648, 29153,
    29169, 29160, 29170, 29156, 29168, 29154, 29555, {f: 2, c: 29550}, 29847,
    29874, 29867, 29840, 29866, 29869, 29873, 29861, 29871, {f: 3, c: 29968},
    29967, 30084, 30275, {f: 2, c: 30280}, 30279, 30372, 30441, 30645, 30635,
    30642, 30647, 30646, 30644, 30641, 30632, 30704, 30963, 30973, 30978,
    {f: 2, c: 30971}, 30975, 30962, 30981, 30969, 30974, 30980, 31147, 31144,
    31324, 31323, 31318, 31320, 31316, 31322, 31422, {f: 2, c: 31424}, 31749,
    31759, 31730, 31744, 31743, 31739, 31758, 31732, 31755, 31731, 31746,
    31753, 31747, 31745, 31736, 31741, [31750, 58176], {f: 2, c: 31728}, 31760,
    31754, 31976, 32301, 32316, 32322, 32307, 38984, 32312, 32298, 32329,
    32320, 32327, 32297, 32332, 32304, 32315, 32310, 32324, 32314, 32581,
    32639, 32638, 32637, 32756, 32754, 32812, 33211, 33220, 33228, 33226,
    33221, 33223, 33212, 33257, 33371, 33370, 33372, 34179, 34176, 34191,
    34215, 34197, 34208, 34187, 34211, 34171, 34212, 34202, 34206, 34167,
    34172, 34185, 34209, 34170, 34168, 34135, 34190, 34198, 34182, 34189,
    34201, 34205, 34177, 34210, 34178, 34184, 34181, 34169, 34166, 34200,
    34192, 34207, 34408, 34750, 34730, 34733, 34757, 34736, 34732, 34745,
    34741, 34748, 34734, 34761, 34755, 34754, 34764, 34743, 34735, 34756,
    34762, 34740, 34742, 34751, 34744, 34749, 34782, 34738, 35125, 35123,
    35132, 35134, 35137, 35154, 35127, 35138, 35245, 35247, 35246,
    {f: 2, c: 35314}, 35614, 35608, 35606, 35601, 35589, 35595, 35618, 35599,
    35602, 35605, 35591, 35597, 35592, 35590, 35612, 35603, 35610, 35919,
    35952, 35954, 35953, 35951, 35989, 35988, 36089, 36207, 36430, 36429,
    36435, 36432, 36428, 36423, 36675, 36672, 36997, 36990, 37176, 37274,
    37282, 37275, 37273, 37279, 37281, 37277, 37280, 37793, 37763, 37807,
    37732, 37718, 37703, 37756, 37720, 37724, 37750, 37705, {f: 2, c: 37712},
    37728, 37741, 37775, 37708, 37738, 37753, 37719, 37717, 37714, 37711,
    37745, 37751, 37755, 37729, 37726, 37731, 37735, 37710, 37721, 38343,
    38336, 38345, 38339, 38341, 38327, 38574, 38576, 38572, 38688, 38687,
    38680, 38685, 38681, 38810, 38817, 38812, 38814, 38813, 38869, 38868,
    38897, 38977, 38980, 38986, 38985, 38981, 38979, 39205, {f: 2, c: 39211},
    39210, 39219, 39218, 39215, 39213, 39217, 39216, 39320, 39331, 39329,
    39426, 39418, 39412, 39415, 39417, 39416, 39414, 39419, {f: 2, c: 39421},
    39420, 39427, 39614, 39678, 39677, 39681, 39676, 39752, 39834, 39848,
    39838, 39835, 39846, 39841, 39845, 39844, 39814, 39842, 39840, 39855,
    40243, 40257, 40295, 40246, {f: 2, c: 40238}, 40241, 40248, 40240, 40261,
    {f: 2, c: 40258}, 40254, 40247, 40256, 40253, 32757, 40237, 40586, 40585,
    40589, 40624, 40648, 40666, 40699, 40703, 40740, 40739, 40738, 40788,
    [12245, 40864], 20785, {f: 2, c: 20781}, 22168, 22172, 22167, 22170, 22173,
    22169, 22896, 23356, {f: 2, c: 23657}, 24000, {f: 2, c: 24173}, 25048,
    25055, {f: 2, c: 25069}, 25073, 25066, 25072, 25067, 25046, 25065, 25855,
    25860, 25853, 25848, 25857, 25859, 25852, 26004, 26075, {f: 2, c: 26330},
    26328, 27333, 27321, 27325, 27361, 27334, 27322, {f: 2, c: 27318}, 27335,
    27316, 27309, 27486, 27593, 27659, 28679, {f: 2, c: 28684}, 28673, 28677,
    28692, 28686, {f: 2, c: 28671}, 28667, 28710, 28668, 28663, 28682,
    [29185, 60224], 29183, 29177, 29187, 29181, 29558, 29880, 29888, 29877,
    29889, 29886, 29878, 29883, 29890, 29972, 29971, 30300, 30308, 30297,
    30288, 30291, 30295, 30298, 30374, 30397, 30444, 30658, 30650, 30988,
    {f: 2, c: 30995}, 30985, 30992, 30994, 30993, 31149, 31148, 31327, 31772,
    31785, 31769, 31776, 31775, 31789, 31773, 31782, 31784, 31778, 31781,
    31792, 32348, 32336, 32342, 32355, 32344, 32354, 32351, 32337, 32352,
    32343, 32339, 32693, 32691, {f: 2, c: 32759}, 32885, {f: 2, c: 33233},
    33232, 33375, 33374, 34228, 34246, 34240, 34243, 34242, 34227, 34229,
    34237, 34247, 34244, 34239, 34251, 34254, 34248, 34245, 34225, 34230,
    34258, 34340, 34232, 34231, 34238, 34409, 34791, 34790, 34786, 34779,
    34795, 34794, 34789, 34783, 34803, 34788, 34772, 34780, 34771, 34797,
    34776, 34787, 34775, 34777, 34817, 34804, 34792, 34781, 35155, 35147,
    35151, 35148, 35142, {f: 2, c: 35152}, 35145, 35626, 35623, 35619, 35635,
    35632, 35637, 35655, 35631, 35644, 35646, 35633, 35621, 35639, 35622,
    35638, 35630, 35620, 35643, 35645, 35642, 35906, 35957, 35993, 35992,
    35991, 36094, 36100, 36098, 36096, 36444, 36450, 36448, 36439, 36438,
    36446, 36453, 36455, 36443, 36442, 36449, 36445, 36457, 36436,
    {f: 3, c: 36678}, 36683, 37160, {f: 2, c: 37178}, 37182, 37288, 37285,
    37287, 37295, 37290, 37813, 37772, 37778, 37815, 37787, 37789, 37769,
    37799, 37774, 37802, 37790, 37798, 37781, 37768, 37785, 37791, 37760,
    37773, 37809, 37777, 37810, 37796, 37800, 37812, 37795, {f: 2, c: 38354},
    38353, 38579, 38615, 38618, 24002, 38623, 38616, 38621, 38691, 38690,
    38693, 38828, 38830, 38824, 38827, 38820, 38826, 38818, 38821, 38871,
    38873, 38870, 38872, 38906, {f: 3, c: 38992}, 39096, 39233, 39228, 39226,
    39439, 39435, 39433, 39437, 39428, 39441, 39434, 39429, 39431, 39430,
    39616, 39644, 39688, {f: 2, c: 39684}, 39721, 39733, 39754, 39756, 39755,
    39879, 39878, 39875, 39871, 39873, 39861, 39864, 39891, 39862, 39876,
    39865, 39869, 40284, 40275, 40271, 40266, 40283, 40267, 40281, 40278,
    40268, 40279, 40274, 40276, 40287, 40280, 40282, 40590, 40588, 40671,
    40705, 40704, [40726, 58693], 40741, 40747, 40746, 40745, 40744, 40780,
    40789, {f: 2, c: 20788}, 21142, 21239, 21428, 22187, 22189,
    {f: 2, c: 22182}, 22186, 22188, 22746, 22749, 22747, 22802,
    {f: 3, c: 23357}, 24003, 24176, 24511, 25083, 25863, 25872, 25869, 25865,
    25868, 25870, 25988, 26078, 26077, 26334, 27367, 27360, 27340, 27345,
    27353, 27339, 27359, 27356, 27344, 27371, 27343, 27341, 27358, 27488,
    27568, 27660, 28697, 28711, 28704, 28694, 28715, {f: 3, c: 28705}, 28713,
    28695, 28708, 28700, 29196, 29194, 29191, 29186, 29189, {f: 2, c: 29349},
    29348, 29347, 29345, 29899, 29893, 29879, 29891, 29974, 30304,
    {f: 2, c: 30665}, 30660, 30705, 31005, 31003, 31009, 31004, 30999, 31006,
    31152, {f: 2, c: 31335}, 31795, 31804, 31801, 31788, 31803, 31980, 31978,
    32374, 32373, 32376, 32368, 32375, 32367, 32378, 32370, 32372, 32360,
    32587, 32586, 32643, 32646, 32695, {f: 2, c: 32765}, 32888, 33239, 33237,
    33291, 33380, 33377, 33379, 34283, 34289, 34285, 34265, 34273, 34280,
    34266, 34263, 34284, 34290, 34296, 34264, 34271, 34275, 34268, 34257,
    34288, 34278, 34287, 34270, 34274, 34816, 34810, 34819, {f: 2, c: 34806},
    34825, 34828, 34827, 34822, 34812, 34824, 34815, 34826, 34818, 35170,
    {f: 2, c: 35162}, 35159, 35169, 35164, 35160, 35165, 35161, 35208, 35255,
    35254, 35318, 35664, 35656, 35658, 35648, 35667, 35670, 35668, 35659,
    35669, 35665, 35650, 35666, 35671, 35907, 35959, 35958, 35994,
    {f: 2, c: 36102}, 36105, 36268, 36266, 36269, 36267, 36461, 36472, 36467,
    36458, 36463, 36475, 36546, 36690, 36689, {f: 2, c: 36687}, 36691, 36788,
    37184, 37183, 37296, 37293, 37854, 37831, 37839, 37826, 37850, 37840,
    37881, 37868, 37836, 37849, 37801, 37862, 37834, 37844, 37870, 37859,
    37845, 37828, 37838, 37824, 37842, 37797, 37863, 38269, {f: 2, c: 38362},
    38625, 38697, {f: 2, c: 38699}, 38696, 38694, 38835, 38839, 38838,
    {f: 3, c: 38877}, 39004, 39001, 39005, 38999, 39103, 39101, 39099, 39102,
    39240, 39239, 39235, {f: 2, c: 39334}, 39450, 39445, 39461, 39453, 39460,
    39451, 39458, 39456, 39463, 39459, 39454, 39452, 39444, 39618, 39691,
    39690, 39694, 39692, 39735, {f: 2, c: 39914}, 39904, 39902, 39908, 39910,
    39906, 39920, 39892, 39895, 39916, 39900, 39897, 39909, 39893, 39905,
    39898, 40311, 40321, 40330, 40324, 40328, 40305, 40320, 40312, 40326,
    {f: 2, c: 40331}, 40317, 40299, {f: 2, c: 40308}, 40304, 40297, 40325,
    40307, 40315, 40322, 40303, 40313, 40319, 40327, 40296, 40596, 40593,
    40640, 40700, 40749, {f: 2, c: 40768}, 40781, {f: 3, c: 40790}, 21303,
    22194, 22197, 22195, 22755, 23365, {f: 2, c: 24006}, {f: 2, c: 24302},
    {f: 2, c: 24512}, 25081, 25879, 25878, 25877, 25875, 26079, 26344,
    {f: 2, c: 26339}, 27379, 27376, 27370, 27368, 27385, 27377,
    {f: 2, c: 27374}, 28732, 28725, 28719, 28727, 28724, 28721, 28738, 28728,
    28735, 28730, 28729, 28714, 28736, 28731, 28723, 28737, {f: 2, c: 29203},
    29352, 29565, 29564, 29882, 30379, 30378, 30398, 30445, 30668,
    {f: 2, c: 30670}, 30669, 30706, 31013, 31011, {f: 2, c: 31015}, 31012,
    31017, 31154, 31342, {f: 2, c: 31340}, 31479, 31817, 31816, 31818, 31815,
    31813, 31982, 32379, 32382, 32385, 32384, 32698, 32767, 32889, 33243,
    33241, {f: 2, c: 33384}, 34338, 34303, 34305, 34302, 34331, 34304, 34294,
    34308, 34313, 34309, 34316, 34301, 34841, {f: 2, c: 34832}, 34839, 34835,
    34838, 35171, 35174, 35257, 35319, 35680, 35690, 35677, 35688, 35683,
    35685, 35687, 35693, 36270, 36486, 36488, 36484, 36697, {f: 2, c: 36694},
    36693, 36696, 36698, 37005, 37187, 37185, 37303, 37301, {f: 2, c: 37298},
    37899, 37907, 37883, 37920, 37903, 37908, 37886, 37909, 37904, 37928,
    37913, 37901, 37877, 37888, 37879, 37895, 37902, 37910, 37906, 37882,
    37897, 37880, 37948, 37898, 37887, 37884, 37900, 37878, 37905, 37894,
    38366, 38368, 38367, {f: 2, c: 38702}, 38841, 38843, {f: 2, c: 38909},
    39008, {f: 2, c: 39010}, 39007, {f: 2, c: 39105}, 39248, 39246, 39257,
    39244, 39243, 39251, 39474, 39476, 39473, 39468, 39466, 39478, 39465,
    39470, 39480, 39469, 39623, 39626, 39622, 39696, 39698, 39697, 39947,
    39944, 39927, 39941, 39954, 39928, 40000, 39943, 39950, 39942, 39959,
    39956, 39945, 40351, 40345, 40356, 40349, 40338, 40344, 40336, 40347,
    40352, 40340, 40348, 40362, 40343, 40353, 40346, 40354, 40360, 40350,
    40355, 40383, 40361, 40342, {f: 2, c: 40358}, 40601, 40603, 40602, 40677,
    40676, 40679, 40678, 40752, 40750, 40795, 40800, 40798, 40797, 40793,
    40849, 20794, 20793, 21144, 21143, 22211, {f: 2, c: 22205}, 23368, 23367,
    24011, 24015, 24305, 25085, 25883, 27394, 27388, 27395, 27384, 27392,
    {f: 2, c: 28739}, 28746, {f: 2, c: 28744}, {f: 2, c: 28741}, 29213, 29210,
    29209, 29566, 29975, 30314, 30672, 31021, 31025, 31023, 31828, 31827,
    31986, 32394, [32391, 60229], 32392, 32395, 32390, 32397, 32589, 32699,
    32816, 33245, 34328, 34346, 34342, 34335, 34339, 34332, 34329, 34343,
    34350, 34337, 34336, 34345, 34334, 34341, 34857, 34845, 34843, 34848,
    34852, 34844, 34859, 34890, 35181, 35177, 35182, 35179, 35322, 35705,
    35704, 35653, {f: 2, c: 35706}, 36112, 36116, 36271, 36494, 36492, 36702,
    36699, 36701, 37190, {f: 2, c: 37188}, 37305, 37951, 37947, 37942, 37929,
    37949, 37936, 37945, 37930, 37943, 37932, 37952, 37937, 38373, 38372,
    38371, 38709, 38714, 38847, 38881, 39012, 39113, 39110, 39104, 39256,
    39254, 39481, 39485, 39494, 39492, 39490, 39489, 39482, 39487, 39629,
    39701, {f: 2, c: 39703}, 39702, 39738, 39762, 39979, 39965, 39964, 39980,
    39971, {f: 2, c: 39976}, 39972, 39969, 40375, 40374, 40380, 40385, 40391,
    40394, 40399, 40382, 40389, 40387, 40379, 40373, 40398, {f: 2, c: 40377},
    40364, 40392, 40369, 40365, 40396, 40371, 40397, 40370, 40570, 40604,
    40683, 40686, 40685, 40731, 40728, 40730, 40753, 40782, 40805, 40804,
    40850, 20153, 22214, 22213, 22219, 22897, {f: 2, c: 23371}, 24021, 24017,
    24306, 25889, 25888, 25894, 25890, 27403, {f: 2, c: 27400}, 27661,
    {f: 3, c: 28757}, 28754, {f: 2, c: 29214}, 29353, 29567, 29912, 29909,
    29913, 29911, 30317, 30381, 31029, 31156, {f: 2, c: 31344}, 31831, 31836,
    31833, 31835, 31834, 31988, 31985, 32401, 32591, 32647, 33246, 33387,
    {f: 2, c: 34356}, 34355, 34348, 34354, 34358, 34860, 34856, 34854, 34858,
    34853, 35185, 35263, 35262, 35323, 35710, 35716, 35714, 35718, 35717,
    35711, 36117, 36501, 36500, 36506, 36498, 36496, {f: 2, c: 36502}, 36704,
    36706, 37191, 37964, 37968, {f: 2, c: 37962}, 37967, 37959, 37957,
    {f: 2, c: 37960}, 37958, 38719, 38883, 39018, 39017, 39115, 39252, 39259,
    39502, {f: 2, c: 39507}, 39500, 39503, 39496, 39498, 39497, 39506, 39504,
    39632, 39705, 39723, 39739, 39766, 39765, 40006, 40008, 39999, 40004,
    39993, 39987, 40001, 39996, 39991, 39988, 39986, 39997, 39990, 40411,
    40402, 40414, 40410, 40395, 40400, 40412, 40401, 40415, 40425, 40409,
    40408, 40406, 40437, 40405, 40413, 40630, 40688, 40757, 40755, 40754,
    40770, 40811, 40853, 40866, 20797, 21145, 22760, 22759, 22898, 23373,
    24024, 34863, 24399, 25089, {f: 2, c: 25091}, 25897, 25893, 26006, 26347,
    {f: 2, c: 27409}, 27407, 27594, 28763, 28762, 29218, 29570, 29569, 29571,
    30320, 30676, 31847, 31846, 32405, 33388, 34362, 34368, 34361, 34364,
    34353, 34363, 34366, 34864, 34866, 34862, 34867, 35190, 35188, 35187,
    35326, 35724, 35726, 35723, 35720, 35909, 36121, 36504, 36708, 36707,
    37308, 37986, 37973, 37981, 37975, 37982, {f: 2, c: 38852}, 38912, 39510,
    39513, {f: 3, c: 39710}, 40018, 40024, 40016, 40010, 40013, 40011, 40021,
    40025, 40012, 40014, 40443, 40439, 40431, 40419, 40427, 40440, 40420,
    40438, 40417, 40430, 40422, 40434, [40432, 60370], 40418, 40428, 40436,
    40435, 40424, 40429, 40642, 40656, {f: 2, c: 40690}, 40710, 40732, 40760,
    40759, 40758, 40771, 40783, 40817, 40816, {f: 2, c: 40814}, 22227, 22221,
    23374, 23661, 25901, {f: 2, c: 26349}, 27411, 28767, 28769, 28765, 28768,
    29219, 29915, 29925, 30677, 31032, 31159, 31158, 31850, 32407, 32649,
    33389, 34371, 34872, 34871, 34869, 34891, {f: 2, c: 35732},
    {f: 3, c: 36510}, 36509, 37310, 37309, 37314, 37995, {f: 2, c: 37992},
    38629, 38726, 38723, 38727, 38855, 38885, 39518, 39637, 39769, 40035,
    40039, 40038, 40034, 40030, 40032, 40450, 40446, 40455, 40451, 40454,
    40453, {f: 2, c: 40448}, 40457, 40447, 40445, 40452, 40608, 40734, 40774,
    {f: 3, c: 40820}, 22228, 25902, 26040, {f: 2, c: 27416}, 27415, 27418,
    28770, 29222, 29354, {f: 2, c: 30680}, 31033, 31849, 31851, 31990, 32410,
    32408, 32411, 32409, {f: 2, c: 33248}, {f: 3, c: 34374}, {f: 2, c: 35193},
    35196, 35195, 35327, {f: 2, c: 35736}, 36517, 36516, 36515, 37998, 37997,
    37999, 38001, 38003, 38729, 39026, 39263, 40040, 40046, 40045, 40459,
    40461, 40464, 40463, 40466, 40465, 40609, 40693, 40713, 40775, 40824,
    40827, 40826, 40825, 22302, 28774, 31855, 34876, 36274, 36518, 37315,
    38004, 38008, 38006, 38005, 39520, [39726, 60830], 40052, 40051, 40049,
    40053, 40468, 40467, 40694, 40714, 40868, 28776, 28773, 31991, 34410,
    34878, 34877, 34879, 35742, 35996, 36521, 36553, 38731, {f: 2, c: 39027},
    39116, 39265, 39339, 39524, {f: 2, c: 39526}, 39716, 40469, 40471, 40776,
    25095, 27422, 29223, 34380, 36520, 38018, {f: 2, c: 38016}, 39529, 39528,
    40473, 34379, 35743, 38019, 40057, 40631, 30325, 39531, 40058, 40477,
    {f: 2, c: 28777}, 29225, 40612, 40830, 40777, 40856, {s: 97}, 65075, 0,
    65076, 65103, [168, 776, 63208], [710, 63209, 65342], [12541, 63210],
    [12542, 63211], [12445, 63212], [12446, 63213], 0, [12293, 63216],
    [12294, 63217], [12295, 63218], [12540, 63219], [63220, 65339],
    [63221, 65341], [10045, 63222], [12353, 63223], [12354, 63224],
    [12355, 63225], [12356, 63226], [12357, 63227], [12358, 63228],
    [12359, 63229], [12360, 63230], [12361, 63231], [12362, 63232],
    [12363, 63233], [12364, 63234], [12365, 63235], [12366, 63236],
    [12367, 63237], [12368, 63238], [12369, 63239], [12370, 63240],
    [12371, 63241], [12372, 63242], [12373, 63243], [12374, 63244],
    [12375, 63245], [12376, 63246], [12377, 63247], [12378, 63248],
    [12379, 63249], [12380, 63250], [12381, 63251], [12382, 63252],
    [12383, 63253], [12384, 63254], [12385, 63255], [12386, 63256],
    [12387, 63257], [12388, 63258], [12389, 63259], [12390, 63260],
    [12391, 63261], [12392, 63262], [12393, 63263], [12394, 63264],
    [12395, 63265], [12396, 63266], [12397, 63267], [12398, 63268],
    [12399, 63269], [12400, 63270], [12401, 63271], [12402, 63272],
    [12403, 63273], [12404, 63274], [12405, 63275], [12406, 63276],
    [12407, 63277], [12408, 63278], [12409, 63279], [12410, 63280],
    [12411, 63281], [12412, 63282], [12413, 63283], [12414, 63284],
    [12415, 63285], [12416, 63286], [12417, 63287], [12418, 63288],
    [12419, 63289], [12420, 63290], [12421, 63291], [12422, 63292],
    [12423, 63293], [12424, 63294], [12425, 63295], [12426, 63296],
    [12427, 63297], [12428, 63298], [12429, 63299], [12430, 63300],
    [12431, 63301], [12432, 63302], [12433, 63303], [12434, 63304],
    [12435, 63305], [12449, 63306], [12450, 63307], [12451, 63308],
    [12452, 63309], [12453, 63310], [12454, 63311], [12455, 63312],
    [12456, 63313], [12457, 63314], [12458, 63315], [12459, 63316],
    [12460, 63317], [12461, 63318], [12462, 63319], [12463, 63320],
    [12464, 63321], [12465, 63322], [12466, 63323], [12467, 63324],
    [12468, 63325], [12469, 63326], [12470, 63327], [12471, 63328],
    [12472, 63329], [12473, 63330], [12474, 63331], [12475, 63332],
    [12476, 63333], [12477, 63334], [12478, 63335], [12479, 63336],
    [12480, 63337], [12481, 63338], [12482, 63339], [12483, 63340],
    [12484, 63341], [12485, 63342], [12486, 63343], [12487, 63344],
    [12488, 63345], [12489, 63346], [12490, 63347], [12491, 63348],
    [12492, 63349], [12493, 63350], [12494, 63351], [12495, 63352],
    [12496, 63353], [12497, 63354], [12498, 63355], [12499, 63356],
    [12500, 63357], [12501, 63358], [12502, 63359], [12503, 63360],
    [12504, 63361], [12505, 63362], [12506, 63363], [12507, 63364],
    [12508, 63365], [12509, 63366], [12510, 63367], [12511, 63368],
    [12512, 63369], [12513, 63370], [12514, 63371], [12515, 63372],
    [12516, 63373], [12517, 63374], [12518, 63375], [12519, 63376],
    [12520, 63377], [12521, 63378], [12522, 63379], [12523, 63380],
    [12524, 63381], [12525, 63382], [12526, 63383], [12527, 63384],
    [12528, 63385], [12529, 63386], [12530, 63387], [12531, 63388],
    [12532, 63389], [12533, 63390], [12534, 63391], [1040, 63392],
    [1041, 63393], [1042, 63394], [1043, 63395], [1044, 63396], [1045, 63397],
    [1025, 63398], [1046, 63399], [1047, 63400], [1048, 63401], [1049, 63402],
    [1050, 63403], [1051, 63404], [1052, 63405], [1053, 63406], [1054, 63407],
    [1055, 63408], [1056, 63409], [1057, 63410], [1058, 63411], [1059, 63412],
    [1060, 63413], [1061, 63414], [1062, 63415], [1063, 63416], [1064, 63417],
    [1065, 63418], [1066, 63419], [1067, 63420], [1068, 63421], [1069, 63422],
    [1070, 63423], [1071, 63424], [1072, 63425], [1073, 63426], [1074, 63427],
    [1075, 63428], [1076, 63429], [1077, 63430], [1105, 63431], [1078, 63432],
    [1079, 63433], [1080, 63434], [1081, 63435], [1082, 63436], [1083, 63437],
    [1084, 63438], [1085, 63439], [1086, 63440], [1087, 63441], [1088, 63442],
    [1089, 63443], [1090, 63444], [1091, 63445], [1092, 63446], [1093, 63447],
    [1094, 63448], [1095, 63449], [1096, 63450], [1097, 63451], [1098, 63452],
    [1099, 63453], [1100, 63454], [1101, 63455], [1102, 63456], [1103, 63457],
    [8679, 63458], [8632, 63459], [8633, 63460], [20033, 63461],
    [63462, 131276], [20058, 63463], [63464, 131210], [20994, 63465],
    [17553, 63466], 63467, [20872, 63468], [13853, 63469], [63470, 161287],
    {s: 40}, [172, 63511, 65506], [63512, 65508], [63513, 65287],
    [63514, 65282], [12849, 63515], [8470, 63516], [8481, 63517], 30849,
    [37561, 58501], 35023, 22715, 24658, 31911, 23290, 9556, 9574, 9559, 9568,
    9580, 9571, 9562, 9577, 9565, 9554, 9572, 9557, {s: 3}, 9560, 9575, 9563,
    9555, 9573, 9558, 9567, 9579, 9570, 9561, 9576, 9564, 9553, {s: 5}, 9619,
    {s: 26}, [58129, 147159], [22462, 58130], [58131, 159443], [28990, 58132],
    [58133, 153568], [27042, 58135], [58136, 166889], [23412, 58137],
    [31305, 58138], [58139, 153825], [58140, 169177], [31333, 58141],
    [31357, 58142], [58143, 154028], [31419, 58144], [31408, 58145],
    [31426, 58146], [31427, 58147], [29137, 58148], [58149, 156813],
    [16842, 58150], [31450, 58151], [31453, 58152], [31466, 58153],
    [16879, 58154], [21682, 58155], [58156, 154625], [31499, 58157],
    [31573, 58158], [31529, 58159], [58160, 152334], [58161, 154878],
    [31650, 58162], [31599, 58163], [33692, 58164], [58165, 154548],
    [58166, 158847], [31696, 58167], [33825, 58168], [31634, 58169], 0,
    [58171, 154912], 0, [33938, 58174], [31738, 58175], 0, [31797, 58177],
    [58178, 154817], [31812, 58179], [31875, 58180], [58181, 149634],
    [31910, 58182], [58184, 148856], [31945, 58185], [31943, 58186],
    [31974, 58187], 0, [31987, 58189], [31989, 58190], [32359, 58192],
    [17693, 58193], [58194, 159300], [32093, 58195], [58196, 159446],
    [32137, 58198], [32171, 58199], [28981, 58200], [32179, 58201], 32214,
    [58203, 147543], [58204, 155689], [32228, 58205], [15635, 58206],
    [32245, 58207], [58208, 137209], [32229, 58209], [58210, 164717], 0,
    [58212, 155937], [58213, 155994], [32366, 58214], 0, [17195, 58216],
    [37996, 58217], [32295, 58218], [32576, 58219], [32577, 58220],
    [32583, 58221], [31030, 58222], [58223, 156368], [39393, 58224],
    [32663, 58225], [58226, 156497], [32675, 58227], [58228, 136801],
    [58229, 131176], [17756, 58230], [58231, 145254], [58233, 164666],
    [32762, 58234], [58235, 156809], 0, [32776, 58237], [32797, 58238], 0,
    [32815, 58240], [58241, 172167], [58242, 158915], [32827, 58243],
    [32828, 58244], [32865, 58245], [58246, 141076], [18825, 58247],
    [58248, 157222], [58249, 146915], [58250, 157416], [26405, 58251],
    [32935, 58252], [58253, 166472], [33031, 58254], [33050, 58255],
    [22704, 58256], [58257, 141046], [27775, 58258], [58259, 156824],
    [25831, 58261], [58262, 136330], [33304, 58263], [58264, 137310],
    [27219, 58265], [58266, 150117], [58267, 150165], [17530, 58268],
    [33321, 58269], [58271, 158290], [58272, 146814], [20473, 58273],
    [58274, 136445], [34018, 58275], [33634, 58276], 0, [58278, 149927],
    [58279, 144688], [58280, 137075], [58281, 146936], [33450, 58282],
    [26907, 58283], [58284, 194964], [16859, 58285], [34123, 58286],
    [33488, 58287], [33562, 58288], [58289, 134678], [58290, 137140],
    [14017, 58291], [58292, 143741], [58293, 144730], [33403, 58294],
    [33506, 58295], [33560, 58296], [58297, 147083], [58298, 159139],
    [58299, 158469], [58300, 158615], [58301, 144846], [15807, 58302],
    [33565, 58303], [21996, 58304], [33669, 58305], [17675, 58306],
    [58307, 159141], [33708, 58308], 0, [33747, 58310], [58312, 159444],
    [27223, 58313], [34138, 58314], [13462, 58315], [58316, 159298],
    [33880, 58318], [58319, 154596], [33905, 58320], [15827, 58321],
    [17636, 58322], [27303, 58323], [33866, 58324], [31064, 58326], 0,
    [58328, 158614], [58329, 159351], [58330, 159299], [34014, 58331], 0,
    [33681, 58333], [17568, 58334], [33939, 58335], [34020, 58336],
    [58337, 154769], [16960, 58338], [58339, 154816], [17731, 58340],
    [34100, 58341], [23282, 58342], 0, [17703, 58344], [34163, 58345],
    [17686, 58346], [26559, 58347], [34326, 58348], [58349, 165413],
    [58350, 165435], [34241, 58351], [58352, 159880], [34306, 58353],
    [58354, 136578], [58355, 159949], [58356, 194994], [17770, 58357],
    [34344, 58358], [13896, 58359], [58360, 137378], [21495, 58361],
    [58362, 160666], [34430, 58363], 0, [58365, 172280], [34798, 58366],
    [58367, 142375], [34737, 58368], [34778, 58369], [34831, 58370, 60990],
    [22113, 58371], [34412, 58372], [26710, 58373], [17935, 58374],
    [34885, 58375], [34886, 58376], [58377, 161248], [58378, 146873],
    [58379, 161252], [34910, 58380], [34972, 58381], [18011, 58382],
    [34996, 58383], [34997, 58384], [35013, 58386], [58388, 161551],
    [35207, 58389], {s: 3}, [35239, 58393], [35260, 58394], [58395, 166437],
    [35303, 58396], [58397, 162084], [58398, 162493], [35484, 58399],
    [30611, 58400], [37374, 58401], [35472, 58402], [58403, 162393],
    [31465, 58404], [58405, 162618], [18195, 58407], [58408, 162616],
    [29052, 58409], [35596, 58410], [35615, 58411], [58412, 152624],
    [58413, 152933], [35647, 58414], 0, [35661, 58416], [35497, 58417],
    [58418, 150138], [35728, 58419], [35739, 58420], [35503, 58421],
    [58422, 136927], [17941, 58423], [34895, 58424], [35995, 58425],
    [58426, 163156], [58427, 163215], [58428, 195028], [14117, 58429],
    [58430, 163155], [36054, 58431], [58432, 163224], [58433, 163261],
    [36114, 58434], [36099, 58435], [58436, 137488], [36059, 58437],
    [28764, 58438], [36113, 58439], [16080, 58441], 0, [36265, 58443],
    [58444, 163842], [58445, 135188], [58446, 149898], [15228, 58447],
    [58448, 164284], [58449, 160012], [31463, 58450], [36525, 58451],
    [36534, 58452], [36547, 58453], [37588, 58454], [36633, 58455],
    [36653, 58456], [58457, 164709], [58458, 164882], [36773, 58459],
    [37635, 58460], [58461, 172703], [58462, 133712], [36787, 58463], 0,
    [58465, 166366], [58466, 165181], [58467, 146875], [24312, 58468],
    [58469, 143970], [36857, 58470], 0, [58474, 140069], [14720, 58475],
    [58476, 159447], [36919, 58477], [58478, 165180], [58479, 162494],
    [36961, 58480], [58481, 165228], [58482, 165387], [37032, 58483],
    [58484, 165651], [37060, 58485], [58486, 165606], [37038, 58487], 0,
    [37223, 58489], [37289, 58491], [37316, 58492], [31916, 58493],
    [58494, 166195], [58495, 138889], [37390, 58496], [27807, 58497],
    [37441, 58498], [37474, 58499], [58500, 153017], [58502, 166598],
    [58503, 146587], [58504, 166668], [58505, 153051], [58506, 134449],
    [37676, 58507], [37739, 58508], [58509, 166625], [58510, 166891],
    [23235, 58512], [58513, 166626], [58514, 166629], [18789, 58515],
    [37444, 58516], [58517, 166892], [58518, 166969], [58519, 166911],
    [37747, 58520], [37979, 58521], [36540, 58522], [38277, 58523],
    [38310, 58524], [37926, 58525], [38304, 58526], [28662, 58527],
    [17081, 58528], [58530, 165592], [58531, 135804], [58532, 146990],
    [18911, 58533], [27676, 58534], [38523, 58535], [38550, 58536],
    [16748, 58537], [38563, 58538], [58539, 159445], [25050, 58540], 58541,
    [30965, 58542], [58543, 166624], [38589, 58544], [21452, 58545],
    [18849, 58546], [58547, 158904], [58548, 131700], [58549, 156688],
    [58550, 168111], [58551, 168165], [58552, 150225], [58553, 137493],
    [58554, 144138], [38705, 58555], [34370, 58556], [38710, 58557],
    [18959, 58558], [17725, 58559], [17797, 58560], [58561, 150249],
    [28789, 58562], [23361, 58563], [38683, 58564], 0, [58566, 168405],
    [38743, 58567], [23370, 58568], [58569, 168427], [38751, 58570],
    [37925, 58571], [20688, 58572], [58573, 143543], [58574, 143548],
    [38793, 58575], [38815, 58576], [38833, 58577], [38846, 58578],
    [38848, 58579], [38866, 58580], [38880, 58581], [58582, 152684],
    [38894, 58583], [29724, 58584], [58585, 169011], 0, [38901, 58587],
    [58588, 168989], [58589, 162170], [19153, 58590], [38964, 58591],
    [38963, 58592], [38987, 58593], [39014, 58594], [15118, 58595],
    [58596, 160117], [15697, 58597], [58598, 132656], [58599, 147804],
    [58600, 153350], [39114, 58601], [39095, 58602], [39112, 58603],
    [39111, 58604], [19199, 58605], [58606, 159015], [58607, 136915],
    [21936, 58608], [39137, 58609], [39142, 58610], [39148, 58611],
    [37752, 58612], [39225, 58613], [58614, 150057], [19314, 58615],
    [58616, 170071], [58617, 170245], [39413, 58618], [39436, 58619],
    [39483, 58620], [39440, 58621], [39512, 58622], [58623, 153381],
    [14020, 58624], [58625, 168113], [58626, 170965], [39648, 58627],
    [39650, 58628], [58629, 170757], [39668, 58630], [19470, 58631],
    [39700, 58632], [39725, 58633], [58634, 165376], [20532, 58635],
    [39732, 58636], [14531, 58638], [58639, 143485], [39760, 58640],
    [39744, 58641], [58642, 171326], [23109, 58643], [58644, 137315],
    [39822, 58645], [39938, 58647], [39935, 58648], [39948, 58649],
    [58650, 171624], [40404, 58651], [58652, 171959], [58653, 172434],
    [58654, 172459], [58655, 172257], [58656, 172323], [58657, 172511],
    [40318, 58658], [40323, 58659], [58660, 172340], [40462, 58661],
    [40388, 58663], [58665, 172435], [58666, 172576], [58667, 137531],
    [58668, 172595], [40249, 58669], [58670, 172217], [58671, 172724],
    [40592, 58672], [40597, 58673], [40606, 58674], [40610, 58675],
    [19764, 58676], [40618, 58677], [40623, 58678], [58679, 148324],
    [40641, 58680], [15200, 58681], [14821, 58682], [15645, 58683],
    [20274, 58684], [14270, 58685], [58686, 166955], [40706, 58687],
    [40712, 58688], [19350, 58689], [37924, 58690], [58691, 159138],
    [40727, 58692, 60836], 0, [40761, 58694], [22175, 58695], [22154, 58696],
    [40773, 58697], [39352, 58698], [58699, 168075], [38898, 58700],
    [33919, 58701], 0, [40809, 58703], [31452, 58704], [40846, 58705],
    [29206, 58706], [19390, 58707], [58708, 149877], [58709, 149947],
    [29047, 58710], [58711, 150008], [58712, 148296], [58713, 150097],
    [29598, 58714], [58715, 166874], [58716, 137466], [31135, 58717],
    [58718, 166270], [58719, 167478], [37737, 58720], [37875, 58721],
    [58722, 166468], [37612, 58723], [37761, 58724], [37835, 58725],
    [58726, 166252], [58727, 148665], [29207, 58728], [16107, 58729],
    [30578, 58730], [31299, 58731], [28880, 58732], [58733, 148595],
    [58734, 148472], [29054, 58735], [58736, 137199], [28835, 58737],
    [58738, 137406], [58739, 144793], [16071, 58740], [58741, 137349],
    [58742, 152623], [58743, 137208], [14114, 58744], [58745, 136955],
    [58746, 137273], [14049, 58747], [58748, 137076], [58749, 137425],
    [58750, 155467], [14115, 58751], [58752, 136896], [22363, 58753],
    [58754, 150053], [58755, 136190], [58756, 135848], [58757, 136134],
    [58758, 136374], [34051, 58759, 58761], [58760, 145062], 0, [33877, 58762],
    [58763, 149908], [58764, 160101], [58765, 146993], [58766, 152924],
    [58767, 147195], [58768, 159826], [17652, 58769], [58770, 145134],
    [58771, 170397], [58772, 159526], [26617, 58773], [14131, 58774],
    [15381, 58775], [15847, 58776], [22636, 58777], [58778, 137506],
    [26640, 58779], [16471, 58780], [58781, 145215], [58782, 147681],
    [58783, 147595], [58784, 147727], [58785, 158753], [21707, 58786],
    [22174, 58787], [58788, 157361], [22162, 58789], [58790, 135135],
    [58791, 134056], [58792, 134669], 0, [58794, 166675], [37788, 58795],
    [20216, 58796], [20779, 58797], [14361, 58798], [58799, 148534],
    [20156, 58800], [58801, 132197], 0, [20299, 58803], [20362, 58804],
    [58805, 153169], [23144, 58806], [58807, 131499], [58808, 132043],
    [14745, 58809], [58810, 131850], [58811, 132116], [13365, 58812],
    [20265, 58813], [58814, 131776], [58815, 167603], [58816, 131701],
    [35546, 58817], [58818, 131596], [20120, 58819], [20685, 58820],
    [20749, 58821], [20386, 58822], [20227, 58823], [58824, 150030],
    [58825, 147082], [20290, 58826], [20526, 58827], [20588, 58828],
    [20609, 58829], [20428, 58830], [20453, 58831], [20568, 58832],
    [20732, 58833], [28278, 58838], [58839, 144789], [58840, 147001],
    [58841, 147135], [28018, 58842], [58843, 137348], [58844, 147081],
    [20904, 58845], [20931, 58846], [58847, 132576], [17629, 58848],
    [58849, 132259], [58850, 132242], [58851, 132241], [36218, 58852],
    [58853, 166556], [58854, 132878], [21081, 58855], [21156, 58856],
    [58857, 133235], [21217, 58858], 0, [18042, 58860], [29068, 58861],
    [58862, 148364], [58863, 134176], [58864, 149932], [58865, 135396],
    [27089, 58866], [58867, 134685], 0, [16094, 58869], [29849, 58870],
    [29716, 58871], [29782, 58872], [29592, 58873], [19342, 58874],
    [58875, 150204], [58876, 147597], [21456, 58877], [13700, 58878],
    [29199, 58879], [58880, 147657], [21940, 58881], [58882, 131909],
    [21709, 58883], [58884, 134086], [22301, 58885], [37469, 58886],
    [38644, 58887], [22493, 58889], [22413, 58890], [22399, 58891],
    [13886, 58892], [22731, 58893], [23193, 58894], [58895, 166470],
    [58896, 136954], [58897, 137071], [58898, 136976], [23084, 58899],
    [22968, 58900], [23166, 58902], [23247, 58903], [23058, 58904],
    [58905, 153926], [58906, 137715], [58907, 137313], [58908, 148117],
    [14069, 58909], [27909, 58910], [29763, 58911], [23073, 58912],
    [58913, 155267], [23169, 58914], [58915, 166871], [58916, 132115],
    [37856, 58917], [29836, 58918], [58919, 135939], [28933, 58920],
    [18802, 58921], [37896, 58922], [58923, 166395], [37821, 58924],
    [14240, 58925], [23582, 58926], [23710, 58927], [24158, 58928],
    [24136, 58929], [58930, 137622], [58931, 137596], [58932, 146158],
    [24269, 58933], [23375, 58934], [58935, 137475], [58936, 137476],
    [14081, 58937], [58938, 137376], [14045, 58939], [58940, 136958],
    [14035, 58941], [33066, 58942], [58943, 166471], [58944, 138682],
    [58945, 144498], [58946, 166312], [24332, 58947, 60916], [24334, 58948],
    [58949, 137511], [58950, 137131], [23147, 58951], [58952, 137019],
    [23364, 58953], [58955, 161277], [34912, 58956], [24702, 58957],
    [58958, 141408], [58959, 140843], [24539, 58960], [16056, 58961],
    [58962, 140719], [58963, 140734], [58964, 168072], [58965, 159603],
    [25024, 58966], [58967, 131134], [58968, 131142], [58969, 140827],
    [24985, 58970], [24984, 58971], [24693, 58972], [58973, 142491],
    [58974, 142599], [58975, 149204], [58976, 168269], [25713, 58977],
    [58978, 149093], [58979, 142186], [14889, 58980], [58981, 142114],
    [58982, 144464], [58983, 170218], [58984, 142968], [25399, 58985],
    [25782, 58987], [25393, 58988], [25553, 58989], [58990, 149987],
    [58991, 142695], [25252, 58992], [58993, 142497], [25659, 58994],
    [25963, 58995], [26994, 58996], [15348, 58997], [58998, 143502],
    [58999, 144045], [59000, 149897], [59001, 144043], [21773, 59002],
    [59003, 144096], [59004, 137433], [59005, 169023], [26318, 59006],
    [59007, 144009], [59008, 143795], [15072, 59009], [59011, 152964],
    [59012, 166690], [59013, 152975], [59014, 136956], [59015, 152923],
    [59016, 152613], [30958, 59017], [59018, 143619], [59019, 137258],
    [59020, 143924], [13412, 59021], [59022, 143887], [59023, 143746],
    [59024, 148169], [26254, 59025], [59026, 159012], [26219, 59027],
    [19347, 59028], [26160, 59029], [59030, 161904], [59031, 138731],
    [26211, 59032], [59033, 144082], [59034, 144097], [26142, 59035],
    [59036, 153714], [14545, 59037], [59038, 145466], [59039, 145340],
    [15257, 59040], [59041, 145314], [59042, 144382], [29904, 59043],
    [15254, 59044], [59046, 149034], [26806, 59047], 0, [15300, 59049],
    [27326, 59050], [59052, 145365], [59053, 148615], [27187, 59054],
    [27218, 59055], [27337, 59056], [27397, 59057], [59058, 137490],
    [25873, 59059], [26776, 59060], [27212, 59061], [15319, 59062],
    [27258, 59063], [27479, 59064], [59065, 147392], [59066, 146586],
    [37792, 59067], [37618, 59068], [59069, 166890], [59070, 166603],
    [37513, 59071], [59072, 163870], [59073, 166364], [37991, 59074],
    [28069, 59075], [28427, 59076], 0, [59079, 147327], [15759, 59080],
    [28164, 59081], [59082, 147516], [23101, 59083], [28170, 59084],
    [22599, 59085], [27940, 59086], [30786, 59087], [28987, 59088],
    [59089, 148250], [59090, 148086], [28913, 59091], [29264, 59092, 61085],
    [29319, 59093], [29332, 59094], [59095, 149391], [59096, 149285],
    [20857, 59097], [59098, 150180], [59099, 132587], [29818, 59100],
    [59101, 147192], [59102, 144991], [59103, 150090], [59104, 149783],
    [59105, 155617], [16134, 59106], [16049, 59107], [59108, 150239],
    [59109, 166947], [59110, 147253], [24743, 59111], [16115, 59112],
    [29900, 59113], [29756, 59114], [37767, 59115], [29751, 59116],
    [17567, 59117], [59118, 159210], [17745, 59119], [30083, 59120],
    [16227, 59121], [59122, 150745], [59123, 150790], [16216, 59124],
    [30037, 59125], [30323, 59126], [59127, 173510], 0, [29800, 59129, 61070],
    [59130, 166604], [59131, 149931], [59132, 149902], [15099, 59133],
    [15821, 59134], [59135, 150094], [16127, 59136], [59137, 149957],
    [59138, 149747], [37370, 59139], [22322, 59140], [37698, 59141],
    [59142, 166627], [59143, 137316], [20703, 59144], [59145, 152097],
    [59146, 152039], [30584, 59147], [59148, 143922], [30478, 59149],
    [30479, 59150], [30587, 59151], [59152, 149143], [59153, 145281],
    [14942, 59154], [59155, 149744], [29752, 59156], [29851, 59157],
    [16063, 59158], [59159, 150202], [59160, 150215], [16584, 59161],
    [59162, 150166], [59163, 156078], [37639, 59164], [59165, 152961],
    [30750, 59166], [30861, 59167], [30856, 59168], [30930, 59169],
    [29648, 59170], [31065, 59171], [59172, 161601], [59173, 153315],
    [16654, 59174], 0, 0, [31141, 59177], [27181, 59178], [59179, 147194],
    [31290, 59180], [31220, 59181], [16750, 59182], [59183, 136934],
    [16690, 59184], [37429, 59185], [31217, 59186], [59187, 134476],
    [59188, 149900], [59189, 131737], [59190, 146874], [59191, 137070],
    [13719, 59192], [21867, 59193], [13680, 59194], [13994, 59195],
    [59196, 131540], [59197, 134157], [31458, 59198], [23129, 59199],
    [59200, 141045], [59201, 154287], [59202, 154268], [23053, 59203],
    [59204, 131675], [30960, 59205], [23082, 59206], [59207, 154566],
    [31486, 59208], [16889, 59209], [31837, 59210], [31853, 59211],
    [16913, 59212], [59213, 154547], [59214, 155324], [59215, 155302],
    [31949, 59216], [59217, 150009], [59218, 137136], [31886, 59219],
    [31868, 59220], [31918, 59221], [27314, 59222], [32220, 59223],
    [32263, 59224], [32211, 59225], [32590, 59226], [59227, 156257],
    [59228, 155996], [59229, 162632], [32151, 59230], [59231, 155266],
    [17002, 59232], [59233, 158581], [59234, 133398], [26582, 59235],
    [59236, 131150], [59237, 144847], [22468, 59238], [59239, 156690],
    [59240, 156664], [32733, 59242], [31527, 59243], [59244, 133164],
    [59245, 154345], [59246, 154947], [31500, 59247], [59248, 155150],
    [39398, 59249], [34373, 59250], [39523, 59251], [27164, 59252],
    [59253, 144447], [59255, 150007], [59256, 157101], [39455, 59257],
    [59258, 157088], 0, [59260, 160039], [59261, 158929], [17642, 59262],
    [33079, 59263], [17410, 59264], [32966, 59265], [33033, 59266],
    [33090, 59267], [59268, 157620], [39107, 59269], [59270, 158274],
    [33378, 59271], [33381, 59272], [59273, 158289], [33875, 59274],
    [59275, 159143], [34320, 59276], [59277, 160283], [23174, 59278],
    [16767, 59279], [59280, 137280], [23339, 59281], [59282, 137377],
    [23268, 59283], [59284, 137432], [34464, 59285], [59286, 195004],
    [59287, 146831], [34861, 59288], [59289, 160802], [23042, 59290],
    [34926, 59291], [20293, 59292], [34951, 59293], [35007, 59294],
    [35046, 59295], [35173, 59296], [35149, 59297], [59298, 153219],
    [35156, 59299], [59300, 161669], [59301, 161668], [59302, 166901],
    [59303, 166873], [59304, 166812], [59305, 166393], [16045, 59306],
    [33955, 59307], [18165, 59308], [18127, 59309], [14322, 59310],
    [35389, 59311], [35356, 59312], [59313, 169032], [24397, 59314],
    [37419, 59315], [59316, 148100], [26068, 59317], [28969, 59318],
    [28868, 59319], [59320, 137285], [40301, 59321], [35999, 59322],
    [36073, 59323], [59324, 163292], [22938, 59325], [30659, 59326],
    [23024, 59327], [14036, 59329], [36394, 59330], [36519, 59331],
    [59332, 150537], [36656, 59333], [36682, 59334], [17140, 59335],
    [27736, 59336], [28603, 59337], [59338, 140065], [18587, 59339],
    [28537, 59340], [28299, 59341], [59342, 137178], [39913, 59343],
    [14005, 59344], [59345, 149807], [37051, 59346], 0, [21873, 59348],
    [18694, 59349], [37307, 59350], [37892, 59351], [59352, 166475],
    [16482, 59353], [59354, 166652], [37927, 59355], [59356, 166941],
    [59357, 166971], [34021, 59358], [35371, 59359], [38297, 59360],
    [38311, 59361], [38295, 59362], [38294, 59363], [59364, 167220],
    [29765, 59365], [16066, 59366], [59367, 149759], [59368, 150082],
    [59369, 148458], [16103, 59370], [59371, 143909], [38543, 59372],
    [59373, 167655], [59374, 167526], [59375, 167525], [16076, 59376],
    [59377, 149997], [59378, 150136], [59379, 147438], [29714, 59380],
    [29803, 59381], [16124, 59382], [38721, 59383], [59384, 168112],
    [26695, 59385], [18973, 59386], [59387, 168083], [59388, 153567], 0,
    [37736, 59390], [59391, 166281], [59392, 166950], [59393, 166703],
    [59394, 156606], [37562, 59395], [23313, 59396], [35689, 59397],
    [18748, 59398], [29689, 59399], [59400, 147995], [38811, 59401], 0,
    [39224, 59403], [59404, 134950], [24001, 59405], [59406, 166853],
    [59407, 150194], [38943, 59408], [59409, 169178], [37622, 59410],
    [59411, 169431], [37349, 59412], [17600, 59413], [59414, 166736],
    [59415, 150119], [59416, 166756], [39132, 59417], [59418, 166469],
    [16128, 59419], [37418, 59420], [18725, 59421], [33812, 59422],
    [39227, 59423], [39245, 59424], [59425, 162566], [15869, 59426], 0,
    [19311, 59428], [39338, 59429], [39516, 59430], [59431, 166757],
    [59432, 153800], [27279, 59433], [39457, 59434], [23294, 59435],
    [39471, 59436], [59437, 170225], [19344, 59438], [59439, 170312],
    [39356, 59440], [19389, 59441], [19351, 59442], [37757, 59443],
    [22642, 59444], [59445, 135938], [22562, 59446], [59447, 149944],
    [59448, 136424], [30788, 59449], [59450, 141087], [59451, 146872],
    [26821, 59452], [15741, 59453], [37976, 59454], [14631, 59455],
    [24912, 59456], [59457, 141185], [59458, 141675], [24839, 59459],
    [40015, 59460], [40019, 59461], [40059, 59462], [39989, 59463],
    [39952, 59464], [39807, 59465], [39887, 59466], [59467, 171565],
    [39839, 59468], [59469, 172533], [59470, 172286], [40225, 59471],
    [19630, 59472], [59473, 147716], [40472, 59474], [19632, 59475],
    [40204, 59476], [59477, 172468], [59478, 172269], [59479, 172275],
    [59480, 170287], [40357, 59481], [33981, 59482], [59483, 159250],
    [59484, 159711], [59485, 158594], [34300, 59486], [17715, 59487],
    [59488, 159140], [59489, 159364], [59490, 159216], [33824, 59491],
    [34286, 59492], [59493, 159232], [59494, 145367], [59495, 155748],
    [31202, 59496], [59497, 144796], [59498, 144960], [59500, 149982],
    [15714, 59501], [37851, 59502], [37566, 59503], [37704, 59504],
    [59505, 131775], [30905, 59506], [37495, 59507], [37965, 59508],
    [20452, 59509], [13376, 59510], [36964, 59511], [59512, 152925],
    [30781, 59513], [30804, 59514], [30902, 59515], [30795, 59516],
    [59517, 137047], [59518, 143817], [59519, 149825], [13978, 59520],
    [20338, 59521], [28634, 59522], [28633, 59523], 0, [28702, 59524, 59525],
    [21524, 59526], [59527, 147893], [22459, 59528], [22771, 59529],
    [22410, 59530], [40214, 59531], [22487, 59532], [28980, 59533],
    [13487, 59534], [59535, 147884], [29163, 59536], [59537, 158784],
    [59538, 151447], 0, [59540, 137141], [59541, 166473], [24844, 59542],
    [23246, 59543], [23051, 59544], [17084, 59545], [59546, 148616],
    [14124, 59547], [19323, 59548], [59549, 166396], [37819, 59550],
    [37816, 59551], [59552, 137430], [59553, 134941], [33906, 59554],
    [59555, 158912], [59556, 136211], [59557, 148218], [59558, 142374],
    [59559, 148417], [22932, 59560], [59561, 146871], [59562, 157505],
    [32168, 59563], [59564, 155995], [59565, 155812], [59566, 149945],
    [59567, 149899], [59568, 166394], [37605, 59569], [29666, 59570],
    [16105, 59571], [29876, 59572], [59573, 166755], [59574, 137375],
    [16097, 59575], [59576, 150195], [27352, 59577], [29683, 59578],
    [29691, 59579], [16086, 59580], [59581, 150078], [59582, 150164],
    [59583, 137177], [59584, 150118], [59585, 132007], [59586, 136228],
    [59587, 149989], [29768, 59588], [59589, 149782], [28837, 59590],
    [59591, 149878], [37508, 59592], [29670, 59593], [37727, 59594],
    [59595, 132350], [37681, 59596], [59597, 166606], [59598, 166422],
    [37766, 59599], [59600, 166887], [59601, 153045], [18741, 59602],
    [59603, 166530], [29035, 59604], [59605, 149827], [59606, 134399],
    [22180, 59607], [59608, 132634], [59609, 134123], [59610, 134328],
    [21762, 59611], [31172, 59612], [59613, 137210], [32254, 59614],
    [59615, 136898], [59616, 150096], [59617, 137298], [17710, 59618],
    [37889, 59619], [14090, 59620], [59621, 166592], [59622, 149933],
    [22960, 59623], [59624, 137407], [59625, 137347], [59626, 160900],
    [23201, 59627], [14050, 59628], [59629, 146779], [14000, 59630],
    [37471, 59631], [23161, 59632], [59633, 166529], [59634, 137314],
    [37748, 59635], [15565, 59636], [59637, 133812], [19094, 59638],
    [14730, 59639], [20724, 59640], [15721, 59641], [15692, 59642],
    [59643, 136092], [29045, 59644], [17147, 59645], [59646, 164376],
    [28175, 59647], [59648, 168164], [17643, 59649], [27991, 59650],
    [59651, 163407], [28775, 59652], [27823, 59653], [15574, 59654],
    [59655, 147437], [59656, 146989], [28162, 59657], [28428, 59658],
    [15727, 59659], [59660, 132085], [30033, 59661], [14012, 59662],
    [13512, 59663], [18048, 59664], [16090, 59665], [18545, 59666],
    [22980, 59667], [37486, 59668], [18750, 59669], [36673, 59670],
    [59671, 166940], [59672, 158656], [22546, 59673], [22472, 59674],
    [14038, 59675], [59676, 136274], [28926, 59677], [59678, 148322],
    [59679, 150129], [59680, 143331], [59681, 135856], [59682, 140221],
    [26809, 59683], [26983, 59684], [59685, 136088], [59686, 144613],
    [59687, 162804], [59688, 145119], [59689, 166531], [59690, 145366],
    [59691, 144378], [59692, 150687], [27162, 59693], [59694, 145069],
    [59695, 158903], [33854, 59696], [17631, 59697], [17614, 59698],
    [59699, 159014], [59700, 159057], [59701, 158850], [59702, 159710], 0, 0,
    [33597, 59705], [59706, 137018], [33773, 59707], [59708, 158848],
    [59709, 159827], [59710, 137179], [22921, 59711], [23170, 59712],
    [59713, 137139], [23137, 59714], [23153, 59715], [59716, 137477],
    [59717, 147964], [14125, 59718], [23023, 59719], [59720, 137020],
    [14023, 59721], [29070, 59722], [37776, 59723], [26266, 59724],
    [59725, 148133], [23150, 59726], [23083, 59727], [59728, 148115],
    [27179, 59729], [59730, 147193], [59731, 161590], [59732, 148571],
    [59733, 148170], [28957, 59734], [59735, 148057], [59736, 166369],
    [20400, 59737], [59738, 159016], [23746, 59739], [59740, 148686],
    [59741, 163405], [59742, 148413], [27148, 59743], [59744, 148054],
    [59745, 135940], 0, [28979, 59747], [59748, 148457], [15781, 59749],
    [27871, 59750], [59751, 194597], [23019, 59754], [24412, 59757],
    [59764, 144128], [31955, 59776], [59783, 162548], [59786, 153334],
    [59790, 162584], [36972, 59791], [33270, 59795], [30476, 59797],
    [27810, 59799], [22269, 59800], [22633, 59828], [26465, 59832],
    [23646, 59838], [22770, 59841], [28857, 59843], [26627, 59853],
    [36795, 59859], [36796, 59861], [20001, 59871], [31545, 59898],
    [15820, 59902], [29482, 57990, 59909], [30048, 59912], [22586, 59920],
    [33446, 59932], [27018, 59940], [24803, 59944], [20206, 59984],
    [39364, 60002], [40639, 60023], [21249, 60025], [26528, 60038],
    [24808, 60046], [20916, 60053], [31363, 60064], [39994, 60075],
    [31432, 60093], [26906, 60098], [22956, 60100], [22592, 60102],
    [21610, 60114], [24807, 60123], [22138, 60125], [26965, 60132],
    [39983, 60133], [34725, 60134], [23584, 60141], [24075, 60143],
    [26398, 60147], [33965, 60157], [35713, 60161], [20088, 60166],
    [25283, 60176], [26709, 60180], 0, [33533, 60190], [35237, 60194],
    [36768, 60196], [38840, 60198], [38983, 60200], [39613, 60201],
    [24497, 60218], [26184, 60219], [26303, 60220], [60221, 162425], 0,
    [60225, 149946], 0, 0, [60230, 131910], [26382, 60232], [26904, 60233],
    [60235, 161367], [60236, 155618], [60239, 161278], [60240, 139418],
    [18640, 60241], [19128, 60242], [60244, 166554], [60247, 147515],
    [60250, 150085], [60251, 132554], [20946, 60252], [60253, 132625],
    [22943, 60254], [60255, 138920], [15294, 60256], [60257, 146687],
    [14747, 60262], [60264, 165352], [60265, 170441], [14178, 60266],
    [60267, 139715], [35678, 60268], [60269, 166734], 0, [29193, 60274],
    [60276, 134264], [60280, 132985], [36570, 60281], [21135, 60283],
    [29041, 60285], [60288, 147274], [60289, 150183], [21948, 60290],
    [60293, 158546], [13427, 60295], [60297, 161330], [18200, 60299],
    [60303, 149823], [20582, 60305], [13563, 60306], [60307, 144332], 0,
    [18300, 60310], [60311, 166216], [60315, 138640], 0, [60320, 162834],
    [36950, 60321], [60323, 151450], [35682, 60324], [23899, 60327],
    [60328, 158711], 0, [60331, 137500], [35562, 60332], [60333, 150006],
    [60335, 147439], [19392, 60337], [60340, 141083], [37989, 60341],
    [60342, 153569], [24981, 60343], [23079, 60344], [60345, 194765], 0,
    [60348, 148769], [20074, 60350], [60351, 149812], [38486, 60352],
    [28047, 60353], [60354, 158909], [35191, 60356], [60359, 156689], 0,
    [31554, 60363], [60364, 168128], [60365, 133649], 0, [31301, 60369],
    [39462, 60372], [13919, 60374], [60375, 156777], [60376, 131105],
    [31107, 60377], [23852, 60380], [60381, 144665], 0, [18128, 60384],
    [30011, 60386], [34917, 60387], [22710, 60389], [14108, 60390],
    [60391, 140685], [15444, 60394], [37505, 60397], [60398, 139642],
    [37680, 60400], [60402, 149968], [27705, 60403], [60406, 134904],
    [34855, 60407], [35061, 60408], [60409, 141606], [60410, 164979],
    [60411, 137137], [28344, 60412], [60413, 150058], [60414, 137248],
    [14756, 60415], 0, 0, [17727, 60419], [26294, 60420], [60421, 171181],
    [60422, 170148], [35139, 60423], [16607, 60427], [60428, 136714],
    [14753, 60429], [60430, 145199], [60431, 164072], [60432, 136133],
    [29101, 60433], [33638, 60434], [60436, 168360], 0, [19639, 60438],
    [60439, 159919], [60440, 166315], [60445, 147834], [31555, 60446],
    [31102, 60447], [28597, 60449], [60450, 172767], [27139, 60451],
    [60452, 164632], [21410, 60453], [60454, 159239], [37823, 60455],
    [26678, 60456], [38749, 59389, 60457], [60458, 164207], [60460, 158133],
    [60461, 136173], [60462, 143919], [23941, 60464], [60465, 166960],
    [22293, 60467], [38947, 60468], [60469, 166217], [23979, 60470],
    [60471, 149896], [26046, 60472], [27093, 60473], [21458, 60474],
    [60475, 150181], [60476, 147329], [15377, 60477], [26422, 60478],
    [60482, 139169], [13770, 60490], [18682, 60493], 0, [30728, 60496],
    [37461, 60497], [17394, 60499], [17375, 60501], [23032, 60505], 0,
    [22155, 60518], [60520, 169449], [36882, 60541], [21953, 60546],
    [17673, 60551], [32383, 60552], [28502, 60553], [27313, 60554],
    [13540, 60556], [60558, 161949], [14138, 60559], 0, [60562, 163876],
    [60565, 162366], [15851, 60567], [60569, 146615], [60574, 156248],
    [22207, 60575], [36366, 60577], [23405, 60578], [25566, 60581], 0,
    [25904, 60585], [22061, 60586], [21530, 60588], [60591, 171416],
    [19581, 60592], [22050, 60593], [22046, 60594], [32585, 60595],
    [22901, 60597], [60598, 146752], [34672, 60599], [33047, 60604],
    [40286, 60605], [36120, 60606], [30267, 60607], [40005, 60608],
    [30286, 60609], [30649, 60610], [37701, 60611], [21554, 60612],
    [33096, 60613], [33527, 60614], [22053, 60615], [33074, 60616],
    [33816, 60617], [32957, 60618], [21994, 60619], [31074, 60620],
    [22083, 60621], [21526, 60622], [60623, 134813], [13774, 60624],
    [22021, 57509, 60625], [22001, 60626], [26353, 60627], [60628, 164578],
    [13869, 60629], [30004, 60630], [22000, 60631], [21946, 60632],
    [21655, 60633], [21874, 60634], [60635, 134209], [60636, 134294],
    [24272, 57652, 60637], [60639, 134774], [60640, 142434], [60641, 134818],
    [40619, 60642], [32090, 60643], 0, [60645, 135285], [25245, 60646],
    [38765, 60647], [21652, 60648], [36045, 60649], [29174, 60650],
    [37238, 60651], [25596, 60652], [25529, 60653], [25598, 60654],
    [21865, 60655], [60656, 142147], [40050, 60657], [60658, 143027],
    [20890, 60659], [13535, 60660], [60661, 134567], [20903, 60662],
    [21581, 60663], [21790, 60664], [21779, 60665], [30310, 60666],
    [36397, 60667], [60668, 157834], [30129, 60669], [32950, 60670],
    [34820, 60671], 0, [35015, 60673], [33206, 60674], [33820, 60675],
    [17644, 60677], [29444, 60678], [33547, 60681], [22139, 60683],
    [37232, 60690], [37384, 60692], [60696, 134905], [29286, 60697],
    [18254, 60699], [60701, 163833], [16634, 60703], [40029, 60704],
    [25887, 60705], [18675, 60707], [60708, 149472], [60709, 171388], 0,
    [60713, 161187], 60715, [60716, 155720], [29091, 60718], [32398, 60719],
    [40272, 60720], [13687, 60723], [27826, 60725], [21351, 60726],
    [14812, 60728], [60731, 149016], [33325, 60734], [21579, 60735], 60739,
    [14930, 60740], [29556, 60742], [60743, 171692], [19721, 60744],
    [39917, 60745], 0, [19547, 60748], [60751, 171998], [33884, 60752],
    [60754, 160434], [25390, 60757], [32037, 60758], [14890, 60761],
    [36872, 60762], [21196, 60763], [15988, 60764], [13946, 60765],
    [17897, 60766], [60767, 132238], [30272, 60768], [23280, 60769],
    [60770, 134838], [30842, 60771], [18358, 60772], [22695, 60773],
    [16575, 60774], [22140, 60775], [39819, 60776], [23924, 60777],
    [30292, 60778], [60779, 173108], [40581, 60780], [19681, 60781], 0,
    [14331, 60783], [24857, 60784], [60786, 148466], 60787, [22109, 60788],
    [60792, 171526], [21044, 60793], [13741, 60795], 0, [40316, 60797],
    [31830, 60798], [39737, 60799], [22494, 60800], [23635, 60802],
    [25811, 60803], [60804, 169168], [60805, 156469], [34477, 60807],
    [60808, 134440], [60811, 134513], 60812, [20990, 60813], [60814, 139023],
    [23950, 60815], [38659, 60816], [60817, 138705], [40577, 60818],
    [36940, 60819], [31519, 60820], [39682, 60821], [23761, 60822],
    [31651, 60823], [25192, 60824], [25397, 60825], [39679, 60826],
    [31695, 60827], [39722, 60828], [31870, 60829], 0, [31810, 60831],
    [31878, 60832], [39957, 60833], [31740, 60834], [39689, 60835], 0, 39982,
    [40794, 60839], [21875, 60840], [23491, 60841], [20477, 60842],
    [40600, 60843], [20466, 60844], [21088, 60845], [21201, 60847],
    [22375, 60848], [20566, 60849], [22967, 60850], [24082, 60851],
    [38856, 60852], [40363, 60853], [36700, 60854], [21609, 60855],
    [38836, 60856], [39232, 60857], [38842, 60858], [21292, 60859],
    [24880, 60860], [26924, 60861], [21466, 60862], [39946, 60863],
    [40194, 60864], [19515, 60865], [38465, 60866], [27008, 60867],
    [20646, 60868], [30022, 60869], [60870, 137069], [39386, 60871],
    [21107, 60872], 60873, [37209, 60874], [38529, 60875], [37212, 60876],
    60877, [37201, 60878], [60879, 167575], [25471, 60880], [27338, 60882],
    [22033, 60883], [37262, 60884], [30074, 60885], [25221, 60886],
    [29519, 60888], [31856, 60889], [60890, 154657], 60892, [30422, 60894],
    [39837, 60895], [20010, 60896], [60897, 134356], [33726, 60898],
    [34882, 60899], 60900, [23626, 60901], [27072, 60902], 0, 0,
    [21023, 60905], [24053, 60906], [20174, 60907], [27697, 60908],
    [60909, 131570], [20281, 60910], [21660, 60911], 0, [21146, 60913],
    [36226, 60914], [13822, 60915], 0, [13811, 60917], 60918, [27474, 60919],
    [37244, 60920], [40869, 60921], [39831, 60922], [38958, 60923],
    [39092, 60924], [39610, 60925], [40616, 60926], [40580, 60927],
    [31508, 60929], 60930, [27642, 60931], [34840, 60932], [32632, 60933],
    60934, [22048, 60935], [60936, 173642], [36471, 60937], [40787, 60938],
    60939, [36308, 60940], [36431, 60941], [40476, 60942], [36353, 60943],
    [25218, 60944], [60945, 164733], [36392, 60946], [36469, 60947],
    [31443, 60948], [31294, 60950], [30936, 60951], [27882, 60952],
    [35431, 60953], [30215, 60954], [40742, 60956], [27854, 60957],
    [34774, 60958], [30147, 60959], [60960, 172722], [30803, 60961],
    [36108, 60963], [29410, 60964], [29553, 60965], [35629, 60966],
    [29442, 60967], [29937, 60968], [36075, 60969], [60970, 150203],
    [34351, 60971], [24506, 60972], [34976, 60973], [17591, 60974], 60975,
    [60977, 159237], 60978, [35454, 60979], [60980, 140571], 60981,
    [24829, 60982], [30311, 60983], [39639, 60984], [40260, 60985],
    [37742, 58859, 60986], [39823, 60987], [34805, 60988], 60989, 0,
    [36087, 60991], [29484, 60992], [38689, 60993], [39856, 60994],
    [13782, 60995], [29362, 60996], [19463, 60997], [31825, 60998],
    [39242, 60999], [24921, 61001], [19460, 61002], [40598, 61003],
    [24957, 61004], 61005, [22367, 61006], [24943, 61007], [25254, 61008],
    [25145, 61009], 0, [14940, 61011], [25058, 61012], [21418, 61013],
    [25444, 61015], [26626, 61016], [13778, 61017], [23895, 61018],
    [36826, 61020], [61021, 167481], 61022, [20697, 61023], [30982, 61025],
    [21298, 61026], [38456, 61027], [61028, 134971], [16485, 61029], 61030,
    [30718, 61031], 61032, [31938, 61033], [61034, 155418], [31962, 61035],
    [31277, 61036], [32870, 61037], [32867, 61038], [32077, 61039],
    [29957, 61040], [29938, 61041], [35220, 61042], [33306, 61043],
    [26380, 61044], [32866, 61045], [61046, 160902], [32859, 61047],
    [29936, 61048], [33027, 61049], [30500, 61050], [35209, 61051],
    [61052, 157644], [30035, 61053], [34729, 61055], [34766, 61056],
    [33224, 61057], [34700, 61058], [35401, 61059], [36013, 61060],
    [35651, 61061], [30507, 61062], [29944, 61063], [34010, 61064],
    [27058, 61066], [36262, 61067], 61068, [35241, 58392, 61069], 0,
    [28089, 61071], [34753, 61072], [61073, 147473], [29927, 61074],
    [15835, 61075], [29046, 61076], [24740, 57702, 61077], [24988, 61078],
    [15569, 61079], 0, [24695, 61081], 61082, [32625, 61083], 0,
    [24809, 61086], [19326, 61087], [57344, 132423], [37595, 57345],
    [57346, 132575], [57347, 147397], [34124, 57348], [17077, 57349],
    [29679, 57350], [20917, 57351], [13897, 57352], [57353, 149826],
    [57354, 166372], [37700, 57355], [57356, 137691], [33518, 57357],
    [57358, 146632], [30780, 57359], [26436, 57360], [25311, 57361],
    [57362, 149811], [57363, 166314], [57364, 131744], [57365, 158643],
    [57366, 135941], [20395, 57367], [57368, 140525], [20488, 57369],
    [57370, 159017], [57371, 162436], [57372, 144896], [57373, 150193],
    [57374, 140563], 0, [57376, 131966], [24484, 57377], [57378, 131968],
    [57379, 131911], [28379, 57380], [57381, 132127], 20702, [20737, 57383],
    [13434, 57384], [20750, 57385], [39020, 57386], [14147, 57387],
    [33814, 57388], [57389, 149924], [57390, 132231], [20832, 57391],
    [57392, 144308], [20842, 57393], [57394, 134143], [57395, 139516],
    [57396, 131813], [57397, 140592], [57398, 132494], [57399, 143923],
    [57400, 137603], [23426, 57401], [34685, 57402], [57403, 132531],
    [57404, 146585], [20914, 57405], [20920, 57406], [40244, 57407],
    [20937, 57408], [20943, 57409], [20945, 57410], [15580, 57411],
    [20947, 57412], [57413, 150182], [20915, 57414], 0, 0, [20973, 57417],
    [33741, 57418], [26942, 57419], [57420, 145197], [24443, 57421],
    [21003, 57422], [21030, 57423], [21052, 57424], [21173, 57425],
    [21079, 57426], [21140, 57427], [21177, 57428], [21189, 57429],
    [31765, 57430], [34114, 57431], [21216, 57432], [34317, 57433],
    [57434, 158483], 0, [57436, 166622], [21833, 57437], [28377, 57438],
    [57439, 147328], [57440, 133460], [57441, 147436], [21299, 57442], 0,
    [57444, 134114], [27851, 57445], [57446, 136998], [26651, 57447],
    [29653, 57448], [24650, 57449], [16042, 57450], [14540, 57451],
    [57452, 136936], [29149, 57453], [17570, 57454], [21357, 57455],
    [21364, 57456], [57457, 165547], [21374, 57458], 0, [57460, 136598],
    [57461, 136723], [30694, 57462], [21395, 57463], [57464, 166555],
    [21408, 57465], [21419, 57466], [21422, 57467], [29607, 57468],
    [57469, 153458], [16217, 57470], [29596, 57471], [21441, 57472],
    [21445, 57473], [27721, 57474], [20041, 57475], [22526, 57476],
    [21465, 57477], [15019, 57478], [57479, 134031], [21472, 57480],
    [57481, 147435], [57482, 142755], [21494, 57483], [57484, 134263],
    [21523, 57485], [28793, 57486], [21803, 57487], [26199, 57488],
    [27995, 57489], [21613, 57490], [57491, 158547], [57492, 134516],
    [21853, 57493], [21647, 57494], [21668, 57495], [18342, 57496],
    [57497, 136973], [57498, 134877], [15796, 57499], [57500, 134477],
    [57501, 166332], [57502, 140952], [21831, 57503], [19693, 57504],
    [21551, 57505], [29719, 57506], [21894, 57507], [21929, 57508], 0,
    [57510, 137431], [57511, 147514], [17746, 57512], [57513, 148533],
    [26291, 57514], [57515, 135348], [22071, 57516], [26317, 57517],
    [57518, 144010], [26276, 57519], 0, [22093, 57521], [22095, 57522],
    [30961, 57523], [22257, 57524], [38791, 57525], [21502, 57526],
    [22272, 57527], [22255, 57528], [22253, 57529], [57530, 166758],
    [13859, 57531], [57532, 135759], [22342, 57533], [57534, 147877],
    [27758, 57535], [28811, 57536], [22338, 57537], [14001, 57538],
    [57539, 158846], [22502, 57540], [57541, 136214], [22531, 57542],
    [57543, 136276], [57544, 148323], [22566, 57545], [57546, 150517], 0,
    [22698, 57548], [13665, 57549], [22752, 57550], [22748, 57551],
    [57552, 135740], [22779, 57553], [23551, 57554], [22339, 57555],
    [57556, 172368], [57557, 148088], [37843, 57558], [13729, 57559],
    [22815, 57560], [26790, 57561], [14019, 57562], [28249, 57563],
    [57564, 136766], [23076, 57565], 0, [57567, 136850], [34053, 57568],
    [22985, 57569], [57570, 134478], [57571, 158849], [57572, 159018],
    [57573, 137180], [23001, 57574], [57575, 137211], [57576, 137138],
    [57577, 159142], [28017, 57578], [57579, 137256], [57580, 136917],
    [23033, 57581], [57582, 159301], [23211, 57583], [23139, 57584],
    [14054, 57585], [57586, 149929], 0, [14088, 57588], [23190, 57589],
    [29797, 57590], [23251, 57591], [57592, 159649], [57593, 140628],
    [57595, 137489], [14130, 57596], [57597, 136888], [24195, 57598],
    [21200, 57599], [23414, 57600], [25992, 57601], [23420, 57602],
    [57603, 162318], [16388, 57604], [18525, 57605], [57606, 131588],
    [23509, 57607], [57609, 137780], [57610, 154060], [57611, 132517],
    [23539, 57612], [23453, 57613], [19728, 57614], [23557, 57615],
    [57616, 138052], [23571, 57617], [29646, 57618], [23572, 57619],
    [57620, 138405], [57621, 158504], [23625, 57622], [18653, 57623],
    [23685, 57624], [23785, 57625], [23791, 57626], [23947, 57627],
    [57628, 138745], [57629, 138807], [23824, 57630], [23832, 57631],
    [23878, 57632], [57633, 138916], [23738, 57634], [24023, 57635],
    [33532, 57636], [14381, 57637], [57638, 149761], [57639, 139337],
    [57640, 139635], [33415, 57641], [14390, 57642], [15298, 57643],
    [24110, 57644], [27274, 57645], 0, 57647, [57648, 148668], [57649, 134355],
    [21414, 57650], [20151, 57651], 0, [21416, 57653], [57654, 137073],
    [24073, 57655], 57656, [57657, 164994], [24313, 57658], [24315, 57659],
    [14496, 57660], [24316, 57661], [26686, 57662], [37915, 57663],
    [24333, 57664], [57665, 131521], [57666, 194708], [15070, 57667],
    [57669, 135994], [24378, 57670], [57671, 157832], [57672, 140240],
    [57674, 140401], [24419, 57675], [57677, 159342], [24434, 57678],
    [37696, 57679], [57680, 166454], [24487, 57681], [23990, 57682],
    [15711, 57683], [57684, 152144], [57685, 139114], [57686, 159992],
    [57687, 140904], [37334, 57688], [57689, 131742], [57690, 166441],
    [24625, 57691], [26245, 57692], [14691, 57694], [15815, 57695],
    [13881, 57696], [22416, 57697], [57698, 141236], [31089, 57699],
    [15936, 57700], [24734, 57701], 0, 0, [57704, 149890], [57705, 149903],
    [57706, 162387], [29860, 57707], [20705, 57708], [23200, 57709],
    [24932, 57710], [24898, 57712], [57713, 194726], [57714, 159442],
    [24961, 57715], [20980, 57716], [57717, 132694], [24967, 57718],
    [23466, 57719], [57720, 147383], [57721, 141407], [25043, 57722],
    [57723, 166813], [57724, 170333], [25040, 57725], [14642, 57726],
    [57727, 141696], [57728, 141505], [24611, 57729], [24924, 57730],
    [25886, 57731], [25483, 57732], [57733, 131352], [25285, 57734],
    [57735, 137072], [25301, 57736], [57737, 142861], [25452, 57738],
    [57739, 149983], [14871, 57740], [25656, 57741], [25592, 57742],
    [57743, 136078], [57744, 137212], [28554, 57746], [57747, 142902], 0,
    [57750, 153373], [25825, 57751], [25829, 57752], [38011, 57753],
    [14950, 57754], [25658, 57755], [14935, 57756], [25933, 57757],
    [28438, 57758], [57759, 150056], [57760, 150051], [25989, 57761],
    [25965, 57762], [25951, 57763], 0, [26037, 57765], [57766, 149824],
    [19255, 57767], [26065, 57768], [16600, 57769], [57770, 137257], 57771,
    [26083, 57772], [24543, 57773], [57774, 144384], [26136, 57775],
    [57776, 143863], [57777, 143864], [26180, 57778], [57779, 143780],
    [57780, 143781], [26187, 57781], [57782, 134773], [26215, 57783],
    [57784, 152038], [26227, 57785], 0, [57788, 143921], [57789, 165364],
    [57790, 143816], [57791, 152339], [30661, 57792], [57793, 141559],
    [39332, 57794], [26370, 57795], [57796, 148380], [57797, 150049],
    [27130, 57799], [57800, 145346], 0, [26471, 57802], [26466, 57803],
    [57804, 147917], [57805, 168173], [26583, 57806], [17641, 57807],
    [26658, 57808], [28240, 57809], [37436, 57810], [26625, 57811],
    [57812, 144358], [57813, 159136], [26717, 57814], [57815, 144495],
    [27105, 57816], [27147, 57817], [57818, 166623], [26995, 57819],
    [26819, 57820], [57821, 144845], [26881, 57822], [26880, 57823],
    [14849, 57825], [57826, 144956], [15232, 57827], [26540, 57828],
    [26977, 57829], [57830, 166474], [17148, 57831], [26934, 57832],
    [27032, 57833], [15265, 57834], [57835, 132041], [33635, 57836],
    [20624, 57837], [27129, 57838], [57839, 144985], [57840, 139562],
    [27205, 57841], [57842, 145155], [27293, 57843], [15347, 57844],
    [26545, 57845], [27336, 57846], [57847, 168348], [15373, 57848],
    [27421, 57849], [57850, 133411], [24798, 57851, 60308], [27445, 57852],
    [27508, 57853], [57854, 141261], [28341, 57855], [57856, 146139], 0,
    [57858, 137560], [14144, 57859], [21537, 57860], [57861, 146266],
    [27617, 57862], [57863, 147196], [27612, 57864], [27703, 57865],
    [57866, 140427], [57867, 149745], [57868, 158545], [27738, 57869],
    [33318, 57870], [27769, 57871], [57872, 146876], [17605, 57873],
    [57874, 146877], [57875, 147876], [57876, 149772], [57877, 149760],
    [57878, 146633], [14053, 57879], [15595, 57880], [57881, 134450],
    [39811, 57882], [57883, 143865], [57884, 140433], [32655, 57885],
    [26679, 57886], [57887, 159013], [57888, 159137], [57889, 159211],
    [28054, 57890], [27996, 57891], [28284, 57892], [28420, 57893],
    [57894, 149887], [57895, 147589], [57896, 159346], [34099, 57897],
    [57898, 159604], [20935, 57899], 0, 0, [33838, 57902], [57903, 166689], 0,
    [57905, 146991], [29779, 57906], [57907, 147330], [31180, 57908],
    [28239, 57909], [23185, 57910], [57911, 143435], [28664, 57912],
    [14093, 57913], [28573, 57914], [57915, 146992], [28410, 57916],
    [57917, 136343], [57918, 147517], [17749, 57919], [37872, 57920],
    [28484, 57921], [28508, 57922], [15694, 57923], [28532, 57924],
    [57925, 168304], [15675, 57926], [28575, 57927], [57928, 147780],
    [28627, 57929], [57930, 147601], [57931, 147797], [57932, 147513],
    [57933, 147440], [57934, 147380], [57935, 147775], [20959, 57936],
    [57937, 147798], [57938, 147799], [57939, 147776], [57940, 156125],
    [28747, 57941], [28798, 57942], [28839, 57943], 0, [28876, 57945],
    [28885, 57946], [28886, 57947], [28895, 57948], [16644, 57949],
    [15848, 57950], [29108, 57951], [29078, 57952], [57953, 148087],
    [28971, 57954], [28997, 57955], [23176, 57956], [29002, 57957], 0,
    [57960, 148325], [29007, 57961], [37730, 57962], [57963, 148161],
    [28972, 57964], [57965, 148570], [57966, 150055], [57967, 150050],
    [29114, 57968], [57969, 166888], [28861, 57970], [29198, 57971],
    [37954, 57972], [29205, 57973], [22801, 57974], [37955, 57975],
    [29220, 57976], [37697, 57977], [57978, 153093], [29230, 57979],
    [29248, 57980], [57981, 149876], [26813, 57982], [29269, 57983],
    [29271, 57984], [15957, 57985], [57986, 143428], [26637, 57987],
    [28477, 57988], [29314, 57989], 0, [29483, 57991], [57992, 149539],
    [57993, 165931], [18669, 57994], [57995, 165892], [29480, 57996],
    [29486, 57997], [29647, 57998], [29610, 57999], [58000, 134202],
    [58001, 158254], [29641, 58002], [29769, 58003], [58004, 147938],
    [58005, 136935], [58006, 150052], [26147, 58007], [14021, 58008],
    [58009, 149943], [58010, 149901], [58011, 150011], [29687, 58012],
    [29717, 58013], [26883, 58014], [58015, 150054], [29753, 58016],
    [16087, 58018], 0, [58020, 141485], [29792, 58021], [58022, 167602],
    [29767, 58023], [29668, 58024], [29814, 58025], [33721, 58026],
    [29804, 58027], [29812, 58029], [37873, 58030], [27180, 58031],
    [29826, 58032], [18771, 58033], [58034, 150156], [58035, 147807],
    [58036, 150137], [58037, 166799], [23366, 58038], [58039, 166915],
    [58040, 137374], [29896, 58041], [58042, 137608], [29966, 58043],
    [29982, 58045], [58046, 167641], [58047, 137803], [23511, 58048],
    [58049, 167596], [37765, 58050], [30029, 58051], [30026, 58052],
    [30055, 58053], [30062, 58054], [58055, 151426], [16132, 58056],
    [58057, 150803], [30094, 58058], [29789, 58059], [30110, 58060],
    [30132, 58061], [30210, 58062], [30252, 58063], [30289, 58064],
    [30287, 58065], [30319, 58066], 58067, [58068, 156661], [30352, 58069],
    [33263, 58070], [14328, 58071], [58072, 157969], [58073, 157966],
    [30369, 58074], [30373, 58075], [30391, 58076], [30412, 58077],
    [58078, 159647], [33890, 58079], [58080, 151709], [58081, 151933],
    [58082, 138780], [30494, 58083], [30502, 58084], [30528, 58085],
    [25775, 58086], [58087, 152096], [30552, 58088], [58089, 144044],
    [30639, 58090], [58091, 166244], [58092, 166248], [58093, 136897],
    [30708, 58094], 0, [26826, 58098], [30895, 58099], [30919, 58100],
    [30931, 58101], [38565, 58102], [31022, 58103], [58104, 153056],
    [30935, 58105], [31028, 58106], [30897, 58107], [58108, 161292],
    [36792, 58109], [34948, 58110], [58113, 140828], [31110, 58114],
    [35072, 58115], [26882, 58116], [31104, 58117], [58118, 153687],
    [31133, 58119], [58120, 162617], [31036, 58121], [31145, 58122],
    [28202, 58123], [58124, 160038], [16040, 58125], [31174, 58126],
    [58127, 168205], [31188, 58128], 0, [21797, 62526], 0, [62528, 134210],
    [62529, 134421], [62530, 151851], [21904, 62531], [62532, 142534],
    [14828, 62533], [62534, 131905], [36422, 62535], [62536, 150968],
    [62537, 169189], 0, [62539, 164030], [30586, 62540], [62541, 142392],
    [14900, 62542], [18389, 62543], [62544, 164189], [62545, 158194],
    [62546, 151018], [25821, 62547], [62548, 134524], [62549, 135092],
    [62550, 134357], 0, [25741, 62552], [36478, 62553], [62554, 134806], 0,
    [62556, 135012], [62557, 142505], [62558, 164438], [62559, 148691], 0,
    [62561, 134470], [62562, 170573], [62563, 164073], [18420, 62564],
    [62565, 151207], [62566, 142530], [39602, 62567], [14951, 62568],
    [62569, 169460], [16365, 62570], [13574, 62571], [62572, 152263],
    [62573, 169940], 0, [62575, 142660], [40302, 62576], [38933, 62577], 0,
    [17369, 62579], 0, [25780, 62581], [21731, 62582], 0, [62584, 142282], 0,
    [14843, 62586], 0, [62588, 157402], [62589, 157462], [62590, 162208],
    [25834, 62591], [62592, 151634], [62593, 134211], [36456, 62594], 0,
    [62596, 166732], [62597, 132913], 0, [18443, 62599], [62600, 131497],
    [16378, 62601], [22643, 62602], [62603, 142733], 0, [62605, 148936],
    [62606, 132348], [62607, 155799], [62608, 134988], 0, [21881, 62610], 0,
    [17338, 62612], 0, [19124, 62614], [62615, 141926], [62616, 135325],
    [33194, 62617], [39157, 62618], [62619, 134556], [25465, 62620],
    [14846, 62621], [62622, 141173], [36288, 62623], [22177, 62624],
    [25724, 62625], [15939, 62626], 0, [62628, 173569], [62629, 134665],
    [62630, 142031], 0, 0, [62633, 135368], [62634, 145858], [14738, 62635],
    [14854, 62636], [62637, 164507], [13688, 62638], [62639, 155209],
    [62640, 139463], 0, 0, [62643, 142514], [62644, 169760], [13500, 62645],
    [27709, 62646], [62647, 151099], 0, 0, [62650, 161140], [62651, 142987],
    [62652, 139784], [62653, 173659], [62654, 167117], [62655, 134778],
    [62656, 134196], [62683, 161337], [62684, 142286], [62687, 142417],
    [14872, 62689], [62691, 135367], [62693, 173618], [62695, 167122],
    [62696, 167321], [62697, 167114], [38314, 62698], 0, [62706, 161630],
    [28992, 62708], 0, [20822, 62385], 0, [20616, 62487], 0, [13459, 62489],
    [20870, 62491], [24130, 63037], [20997, 62495], [21031, 62436],
    [21113, 62497], 0, [13651, 62504], [21442, 62505], [21343, 62715], 0,
    [21823, 62520], 0, [21976, 59986], [13789, 62722], [22049, 63067], 0,
    [22100, 60044], [60148, 135291], 0, [60153, 135379], 0, [61095, 135934], 0,
    0, [14265, 60104], [23745, 61099], [23829, 63066], [23894, 63030],
    [14392, 63036], [20097, 62477], [24253, 63038], [14612, 63042],
    [25017, 63050], [25232, 63054], [25368, 63056], [25690, 63063],
    [25745, 62381], [33133, 62709], [33156, 59922], [33171, 59924],
    [26624, 63080], [15292, 63093], [29327, 60517], [29389, 59781], 0,
    [29497, 59785], [30018, 59811], [30172, 59817], [16320, 59818],
    [60278, 151205], [16343, 59820], 0, 30336, [30348, 59824, 151388],
    [16552, 59845], [30777, 59846], [16643, 59855], [31377, 59863],
    [31771, 59876], [31981, 59884], [32659, 62658], [32686, 59892], 0,
    [33535, 59936], [22623, 59981], [34482, 59960], 0, [34699, 59963],
    [35143, 59969], 0, [35369, 59972], 0, [36465, 59988], [60484, 164233],
    [36528, 59990], 0, [37214, 62443], [37260, 62441], [39182, 60051],
    [39196, 60054], 0, 0, [39809, 60066], [40384, 60080], [40339, 60078],
    [40620, 60085], [19857, 60540], 0, 37818, [40571, 60084], [28809, 63148],
    [29512, 59788], 0, [31129, 59858], [36791, 59997], 0, [39234, 60056],
    {s: 193}, 8364, {s: 4}, [12443, 63518], [12444, 63519], [11904, 63520],
    {f: 5, c: 62211}, [62216, 131340], 62217, [62218, 131281], [62219, 131277],
    {f: 2, c: 62220}, [62222, 131275], [62223, 139240], 62224, [62225, 131274],
    {f: 4, c: 62226}, [62230, 131342], {f: 2, c: 62231}, {f: 2, c: 62776},
    [62778, 138177], [62779, 194680], [12205, 38737, 62780], [62781, 131206],
    [20059, 62782], [20155, 62783], [13630, 62784], [23587, 62785],
    [24401, 62786], [24516, 62787], [14586, 62788], [25164, 62789],
    [25909, 62790], [27514, 62791], [27701, 62792], [27706, 62793],
    [28780, 62794], [29227, 62795], [20012, 62796], [29357, 62797],
    [62798, 149737], [32594, 62799], [31035, 62800], [31993, 62801],
    [32595, 62802], [62803, 156266], [13505, 62804], [62806, 156491],
    [32770, 62807], [32896, 62808], [62809, 157202], [62810, 158033],
    [21341, 62811], [34916, 62812], [35265, 62813], [62814, 161970],
    [35744, 62815], [36125, 62816], [38021, 62817], [38264, 62818],
    [38271, 62819], [38376, 62820], [62821, 167439], [38886, 62822],
    [39029, 62823], [39118, 62824], [39134, 62825], [39267, 62826],
    [62827, 170000], [40060, 62828], [40479, 62829], [40644, 62830],
    [27503, 62831], [62832, 63751], [20023, 62833], [62834, 131207],
    [38429, 62835], [25143, 62836], [38050, 62837], [11908, 63521],
    [11910, 63522], [11911, 63523], [11912, 63524], [11914, 63525],
    [11916, 63526], [11917, 63527], [11925, 63528], [11932, 63529],
    [11941, 63531], [11943, 63532], [11946, 63533], [11948, 63534],
    [11950, 63535], [11958, 63536], [11964, 63537], [11966, 63538],
    [11978, 63540], [11980, 63541], [11981, 63542], [11983, 63543],
    [11990, 63544], [11991, 63545], [11998, 63546], [62368, 172969],
    [62369, 135493], [25866, 62371], [20029, 62374], [28381, 62375],
    [40270, 62376], [37343, 62377], [62380, 161589], [20250, 62382],
    [20264, 62383], [20392, 62384], [20852, 62386], [20892, 62387],
    [20964, 62388], [21153, 62389], [21160, 62390], [21307, 62391],
    [21326, 62392], [21457, 62393], [21464, 62394], [22242, 62395],
    [22768, 62396], [22788, 62397], [22791, 62398], [22834, 62399],
    [22836, 62400], [23398, 62401], [23454, 62402], [23455, 62403],
    [23706, 62404], [24198, 62405], [24635, 62406], [25993, 62407],
    [26622, 62408], [26628, 62409], [26725, 62410], [27982, 62411],
    [28860, 62412], [30005, 62413], [32420, 62414], [32428, 62415],
    [32442, 62416], [32455, 62417], [32463, 62418], [32479, 62419],
    [32518, 62420], [32567, 62421], [33402, 62422], [33487, 62423],
    [33647, 62424], [35270, 62425], [35774, 62426], [35810, 62427],
    [36710, 62428], [36711, 62429], [36718, 62430], [29713, 62431],
    [31996, 62432], [32205, 62433], [26950, 62434], [31433, 62435],
    [30904, 62442], [32956, 62444], [36107, 62446], [33014, 62447],
    [62448, 133607], [32927, 62451], [40647, 62452], [19661, 62453],
    [40393, 62454], [40460, 62455], [19518, 62456], [62457, 171510],
    [62458, 159758], [40458, 62459], [62460, 172339], [13761, 62461],
    [28314, 62463], [33342, 62464], [29977, 62465], [18705, 62467],
    [39532, 62468], [39567, 62469], [40857, 62470], [31111, 62471],
    [62472, 164972], [62473, 138698], [62474, 132560], [62475, 142054],
    [20004, 62476], [20096, 62478], [20103, 62479], [20159, 62480],
    [20203, 62481], [20279, 62482], [13388, 62483], [20413, 62484],
    [15944, 62485], [20483, 62486], [13437, 62488], [13477, 62490],
    [22789, 62492], [20955, 62493], [20988, 62494], [20105, 62496],
    [21136, 62498], [21287, 62499], [13767, 62500], [21417, 62501],
    [13649, 62502], [21424, 62503], [21539, 62506], [13677, 62507],
    [13682, 62508], [13953, 62509], [21651, 62510], [21667, 62511],
    [21684, 62512], [21689, 62513], [21712, 62514], [21743, 62515],
    [21784, 62516], [21795, 62517], [21800, 62518], [13720, 62519],
    [13733, 62521], [13759, 62522], [21975, 62523], [13765, 62524],
    [62525, 163204], [16467, 62538], [62551, 135412], [62555, 134155],
    [62574, 161992], [62580, 155813], [62583, 142668], [62585, 135287],
    [62587, 135279], [62595, 139681], [62609, 134550], [16571, 62611],
    [62631, 142537], [22098, 62641], [62642, 134961], [62657, 157724],
    [62659, 135375], [62660, 141315], [62661, 141625], [13819, 62662],
    [62663, 152035], [62664, 134796], [62665, 135053], [62666, 134826],
    [16275, 62667], [62668, 134960], [62669, 134471], [62670, 135503],
    [62671, 134732], [62673, 134827], [62674, 134057], [62675, 134472],
    [62676, 135360], [62677, 135485], [16377, 62678], [62679, 140950],
    [25650, 62680], [62681, 135085], [62682, 144372], [62685, 134526],
    [62686, 134527], [62688, 142421], [62690, 134808], [62692, 134958],
    [62694, 158544], [21708, 62699], [33476, 62700], [21945, 62701],
    [62703, 171715], [39974, 62704], [39606, 62705], [62707, 142830],
    [33004, 62710], [23580, 62711], [62712, 157042], [33076, 62713],
    [14231, 62714], [62716, 164029], [37302, 62717], [62718, 134906],
    [62719, 134671], [62720, 134775], [62721, 134907], [62723, 151019],
    [13833, 62724], [62725, 134358], [22191, 62726], [62727, 141237],
    [62728, 135369], [62729, 134672], [62730, 134776], [62731, 135288],
    [62732, 135496], [62733, 164359], [62734, 136277], [62735, 134777],
    [62736, 151120], [62737, 142756], [23124, 62738], [62739, 135197],
    [62740, 135198], [62741, 135413], [62742, 135414], [22428, 62743],
    [62744, 134673], [62745, 161428], [62746, 164557], [62747, 135093],
    [62748, 134779], [62749, 151934], [14083, 62750], [62751, 135094],
    [62752, 135552], [62753, 152280], [62754, 172733], [62755, 149978],
    [62756, 137274], [62757, 147831], [62758, 164476], [22681, 62759],
    [21096, 62760], [13850, 62761], [62762, 153405], [31666, 62763],
    [23400, 62764], [18432, 62765], [19244, 62766], [40743, 62767],
    [18919, 62768], [39967, 62769], [39821, 62770], [62771, 154484],
    [62772, 143677], [22011, 62773], [13810, 62774], [22153, 62775],
    [23870, 63028], [23880, 63029], [15868, 63031], [14351, 63032],
    [23972, 63033], [23993, 63034], [14368, 63035], [24357, 63039],
    [24451, 63040], [14600, 63041], [14655, 63043], [14669, 63044],
    [24791, 63045], [24893, 63046], [23781, 63047], [14729, 63048],
    [25015, 63049], [25039, 63051], [14776, 63052], [25132, 63053],
    [25317, 63055], [14840, 63057], [22193, 63058], [14851, 63059],
    [25570, 63060], [25595, 63061], [25607, 63062], [14923, 63064],
    [25792, 63065], [40863, 63068], [14999, 63069], [25990, 63070],
    [15037, 63071], [26111, 63072], [26195, 63073], [15090, 63074],
    [26258, 63075], [15138, 63076], [26390, 63077], [15170, 63078],
    [26532, 63079], [15192, 63081], [26698, 63082], [26756, 63083],
    [15218, 63084], [15217, 63085], [15227, 63086], [26889, 63087],
    [26947, 63088], [29276, 63089], [26980, 63090], [27039, 63091],
    [27013, 63092], [27094, 63094], [15325, 63095], [27237, 63096],
    [27252, 63097], [27249, 63098], [27266, 63099], [15340, 63100],
    [27289, 63101], [15346, 63102], [27307, 63103], [27317, 63104],
    [27348, 63105], [27382, 63106], [27521, 63107], [27585, 63108],
    [27626, 63109], [27765, 63110], [27818, 63111], [15563, 63112],
    [27906, 63113], [27910, 63114], [27942, 63115], [28033, 63116],
    [15599, 63117], [28068, 63118], [28081, 63119], [28181, 63120],
    [28184, 63121], [28201, 63122], [28294, 63123], [63124, 166336],
    [28347, 63125], [28386, 63126], [28378, 63127], [40831, 63128],
    [28392, 63129], [28393, 63130], [28452, 63131], [28468, 63132],
    [15686, 63133], [63134, 147265], [28545, 63135], [28606, 63136],
    [15722, 63137], [15733, 63138], [29111, 63139], [23705, 63140],
    [15754, 63141], [28716, 63142], [15761, 63143], [28752, 63144],
    [28756, 63145], [28783, 63146], [28799, 63147], [63149, 131877],
    [17345, 63150], [13809, 63151], [63152, 134872], [13902, 58134],
    [15789, 58172], [58173, 154725], [26237, 58183], [31860, 58188],
    [29837, 58197], [32402, 58215], [17667, 58232], [58260, 151480],
    [58270, 133901], [58277, 158474], [13438, 58311], [58317, 143087],
    [58325, 146613], [58343, 159385], [34673, 58364], [25537, 58385],
    [30583, 58387], [35210, 58390], [58406, 147343], [35660, 58415],
    [58440, 150729], [18730, 58464], [58471, 172052], [58472, 165564],
    [58473, 165121], [15088, 58490], [28815, 58511], [58529, 140922],
    [58637, 158120], [58646, 148043], [26760, 58662], [58664, 139611],
    [40802, 58702], [37830, 58793], [58802, 131967], [37734, 58888],
    [37519, 58901], [34324, 58954], [58986, 173147], [16784, 59010],
    [26511, 59045], [26654, 59048], [14435, 59051], [59077, 149996],
    [15129, 59128], [33942, 59176], [59241, 149858], [14818, 59254],
    [33920, 59259], [17262, 59328], [38769, 59402], [39323, 59427],
    [18733, 59499], [28439, 59703], [59704, 160009], [28838, 59746],
    [59752, 150095], [32357, 59753], [23855, 59755], [15859, 59756],
    [59758, 150109], [59759, 137183], [32164, 59760], [33830, 59761],
    [21637, 59762], [59763, 146170], [59765, 131604], [22398, 59766],
    [59767, 133333], [59768, 132633], [16357, 59769], [59770, 139166],
    [59771, 172726], [28675, 59772], [59773, 168283], [23920, 59774],
    [29583, 59775], [59777, 166489], [59778, 168992], [20424, 59779],
    [32743, 59780], [29456, 59782], [29496, 59784], [29505, 59787],
    [16041, 59789], [29173, 59792], [59793, 149746], [29665, 59794],
    [16074, 59796], [16081, 59798], [29721, 59801], [29726, 59802],
    [29727, 59803], [16098, 59804], [16112, 59805], [16116, 59806],
    [16122, 59807], [29907, 59808], [16142, 59809], [16211, 59810],
    [30061, 59812], [30066, 59813], [30093, 59814], [16252, 59815],
    [30152, 59816], [30285, 59819], [30324, 59821], [16348, 59822],
    [30330, 59823], [29064, 59825], [22051, 59826], [35200, 59827],
    [16413, 59829], [30531, 59830], [16441, 59831], [16453, 59833],
    [13787, 59834], [30616, 59835], [16490, 59836], [16495, 59837],
    [30654, 59839], [30667, 59840], [30744, 59842], [30748, 59844],
    [30791, 59847], [30801, 59848], [30822, 59849], [33864, 59850],
    [59851, 152885], [31027, 59852], [31026, 59854], [16649, 59856],
    [31121, 59857], [31238, 59860], [16743, 59862], [16818, 59864],
    [31420, 59865], [33401, 59866], [16836, 59867], [31439, 59868],
    [31451, 59869], [16847, 59870], [31586, 59872], [31596, 59873],
    [31611, 59874], [31762, 59875], [16992, 59877], [17018, 59878],
    [31867, 59879], [31900, 59880], [17036, 59881], [31928, 59882],
    [17044, 59883], [36755, 59885], [28864, 59886], [59887, 134351],
    [32207, 59888], [32212, 59889], [32208, 59890], [32253, 59891],
    [32692, 59893], [29343, 59894], [17303, 59895], [32800, 59896],
    [32805, 59897], [32814, 59899], [32817, 59900], [32852, 59901],
    [22452, 59903], [28832, 59904], [32951, 59905], [33001, 59906],
    [17389, 59907], [33036, 59908], [33038, 59910], [33042, 59911],
    [33044, 59913], [17409, 59914], [15161, 59915], [33110, 59916],
    [33113, 59917], [33114, 59918], [17427, 59919], [33148, 59921],
    [17445, 59923], [17453, 59925], [33189, 59926], [22511, 59927],
    [33217, 59928], [33252, 59929], [33364, 59930], [17551, 59931],
    [33398, 59933], [33482, 59934], [33496, 59935], [17584, 59937],
    [33623, 59938], [38505, 59939], [33797, 59941], [28917, 59942],
    [33892, 59943], [33928, 59945], [17668, 59946], [33982, 59947],
    [34017, 59948], [34040, 59949], [34064, 59950], [34104, 59951],
    [34130, 59952], [17723, 59953], [34159, 59954], [34160, 59955],
    [34272, 59956], [17783, 59957], [34418, 59958], [34450, 59959],
    [34543, 59961], [38469, 59962], [17926, 59964], [17943, 59965],
    [34990, 59966], [35071, 59967], [35108, 59968], [35217, 59970],
    [59971, 162151], [35384, 59973], [35476, 59974], [35508, 59975],
    [35921, 59976], [36052, 59977], [36082, 59978], [36124, 59979],
    [18328, 59980], [36291, 59982], [18413, 59983], [36410, 59985],
    [22356, 59987], [22005, 59989], [18487, 59991], [36558, 59992],
    [36578, 59993], [36580, 59994], [36589, 59995], [36594, 59996],
    [36801, 59998], [36810, 59999], [36812, 60000], [36915, 60001],
    [18605, 60003], [39136, 60004], [37395, 60005], [18718, 60006],
    [37416, 60007], [37464, 60008], [37483, 60009], [37553, 60010],
    [37550, 60011], [37567, 60012], [37603, 60013], [37611, 60014],
    [37619, 60015], [37620, 60016], [37629, 60017], [37699, 60018],
    [37764, 60019], [37805, 60020], [18757, 60021], [18769, 60022],
    [37911, 60024], [37917, 60026], [37933, 60027], [37950, 60028],
    [18794, 60029], [37972, 60030], [38009, 60031], [38189, 60032],
    [38306, 60033], [18855, 60034], [38388, 60035], [38451, 60036],
    [18917, 60037], [18980, 60039], [38720, 60040], [18997, 60041],
    [38834, 60042], [38850, 60043], [19172, 60045], [39097, 60047],
    [19225, 60048], [39153, 60049], [22596, 60050], [39193, 60052],
    [39223, 60055], [39261, 60057], [39266, 60058], [19312, 60059],
    [39365, 60060], [19357, 60061], [39484, 60062], [39695, 60063],
    [39785, 60065], [39901, 60067], [39921, 60068], [39924, 60069],
    [19565, 60070], [39968, 60071], [14191, 60072], [60073, 138178],
    [40265, 60074], [40702, 60076], [22096, 60077], [40381, 60079],
    [40444, 60081], [38134, 60082], [36790, 60083], [40625, 60086],
    [40637, 60087], [40646, 60088], [38108, 60089], [40674, 60090],
    [40689, 60091], [40696, 60092], [40772, 60094], [60095, 131220],
    [60096, 131767], [60097, 132000], [38083, 60099], [60101, 132311],
    [38081, 60103], [60105, 132565], [60106, 132629], [60107, 132726],
    [60108, 136890], [22359, 60109], [29043, 60110], [60111, 133826],
    [60112, 133837], [60113, 134079], [60115, 194619], [60116, 134091],
    [21662, 60117], [60118, 134139], [60119, 134203], [60120, 134227],
    [60121, 134245], [60122, 134268], [60124, 134285], [60126, 134325],
    [60127, 134365], [60128, 134381], [60129, 134511], [60130, 134578],
    [60131, 134600], [60135, 134660], [60136, 134670], [60137, 134871],
    [60138, 135056], [60139, 134957], [60140, 134771], [60142, 135100],
    [60144, 135260], [60145, 135247], [60146, 135286], [60149, 135304],
    [60150, 135318], [13895, 60151], [60152, 135359], [60154, 135471],
    [60155, 135483], [21348, 60156], [60158, 135907], [60159, 136053],
    [60160, 135990], [60162, 136567], [60163, 136729], [60164, 137155],
    [60165, 137159], [28859, 60167], [60168, 137261], [60169, 137578],
    [60170, 137773], [60171, 137797], [60172, 138282], [60173, 138352],
    [60174, 138412], [60175, 138952], [60177, 138965], [60178, 139029],
    [29080, 60179], [60181, 139333], [27113, 60182], [14024, 60183],
    [60184, 139900], [60185, 140247], [60186, 140282], [60187, 141098],
    [60188, 141425], [60189, 141647], [60191, 141671], [60192, 141715],
    [60193, 142037], [60195, 142056], [60197, 142094], [60199, 142143],
    [60202, 142412], [60204, 142472], [60205, 142519], [60206, 154600],
    [60207, 142600], [60208, 142610], [60209, 142775], [60210, 142741],
    [60211, 142914], [60212, 143220], [60213, 143308], [60214, 143411],
    [60215, 143462], [60216, 144159], [60217, 144350], [60222, 144743],
    [60223, 144883], [60227, 144922], [60228, 145174], [22709, 60231],
    [60234, 146087], [60237, 146961], [60238, 147129], [60243, 147737],
    [60245, 148206], [60246, 148237], [60248, 148276], [60249, 148374],
    [60258, 148484], [60259, 148694], [22408, 60260], [60261, 149108],
    [60263, 149295], [60271, 149522], [60272, 149755], [60273, 150037],
    [60275, 150208], [22885, 60277], [60279, 151430], [60282, 151596],
    [22335, 60284], [60286, 152217], [60287, 152601], [60291, 152646],
    [60292, 152686], [60296, 152895], [60298, 152926], [60300, 152930],
    [60301, 152934], [60302, 153543], [60304, 153693], [60309, 153859],
    [60312, 154286], [60313, 154505], [60314, 154630], [22433, 60316],
    [29009, 60317], [60319, 155906], [60322, 156082], [60325, 156674],
    [60326, 156746], [60330, 156804], [60334, 156808], [60336, 156946],
    [60338, 157119], [60339, 157365], [22201, 60347], [60349, 157436],
    [13848, 60355], [60357, 157593], [60358, 157806], [60360, 157790],
    [60362, 157895], [60366, 157990], [60368, 158009], [60371, 158202],
    [60373, 158253], [60378, 158260], [60379, 158555], [60383, 158621],
    [60385, 158884], [60388, 159150], [60392, 159819], [60393, 160205],
    [60395, 160384], [60396, 160389], [60399, 160395], [60401, 160486],
    [38047, 60404], [60405, 160848], [14009, 60416], [60424, 161740],
    [60425, 161880], [22230, 60426], [60435, 162269], [60441, 162301],
    [60442, 162314], [60443, 162571], [60444, 163174], [60448, 163849],
    [60459, 163875], [60463, 163912], [60466, 163971], [60479, 163984],
    [60480, 164084], [60481, 164142], [60483, 164175], [60485, 164271],
    [60486, 164378], [60487, 164614], [60488, 164655], [60489, 164746],
    [60491, 164968], [60492, 165546], [25574, 60494], [60495, 166230],
    [60498, 166328], [60500, 166375], [60502, 166376], [60503, 166726],
    [60504, 166868], [60506, 166921], [60508, 167877], [60509, 168172],
    [60511, 168208], [60512, 168252], [15863, 60513], [60514, 168286],
    [60515, 150218], [36816, 60516], [60519, 169191], [60521, 169392],
    [60522, 169400], [60523, 169778], [60524, 170193], [60525, 170313],
    [60526, 170346], [60527, 170435], [60528, 170536], [60529, 170766],
    [60530, 171354], [60531, 171419], [32415, 60532], [60533, 171768],
    [60534, 171811], [19620, 60535], [38215, 60536], [60537, 172691],
    [29090, 60538], [60539, 172799], [60542, 173515], [19868, 60543],
    [60544, 134300], [36798, 60545], [36794, 60547], [60548, 140464],
    [36793, 60549], [60550, 150163], [20202, 60555], [60557, 166700],
    [36480, 60560], [60561, 137205], [60563, 166764], [60564, 166809],
    [60566, 157359], [60568, 161365], [60570, 153141], [60571, 153942],
    [20122, 60572], [60573, 155265], [60576, 134765], [60579, 147080],
    [60580, 150686], [60583, 137206], [60584, 137339], [60587, 154698],
    [60589, 152337], [15814, 60590], [60596, 155352], [19996, 60600],
    [60601, 135146], [60602, 134473], [60603, 145082], [60638, 151880],
    [21982, 60644], [34694, 60672], [60676, 135361], [60679, 149254],
    [23440, 60680], [60682, 157843], [60684, 141044], [60685, 163119],
    [60686, 147875], [60687, 163187], [60688, 159440], [60689, 160438],
    [60691, 135641], [60693, 146684], [60694, 173737], [60695, 134828],
    [60698, 138402], [60700, 151490], [60702, 135147], [60706, 142752],
    [60710, 135148], [60711, 134666], [60714, 135149], [60717, 135559],
    [19994, 60721], [19972, 60722], [23309, 60724], [13996, 60727],
    [21373, 60729], [13989, 60730], [22682, 60732], [60733, 150382],
    [22442, 60736], [60737, 154261], [60738, 133497], [60741, 140389],
    [60746, 146686], [60747, 171824], [60749, 151465], [60750, 169374],
    [60753, 146870], [60755, 157619], [60756, 145184], [60759, 147191],
    [60760, 146988], [60785, 143578], [60789, 135849], [22439, 60790],
    [60791, 149859], [60794, 159918], [60801, 137068], [60806, 160100],
    [60809, 159010], [60810, 150242], [39963, 60837], [60838, 149822],
    [15878, 60846], [60881, 159011], [60887, 132092], [60891, 146685],
    [60893, 149785], [22394, 60904], [21722, 60912], [29050, 60928],
    [60949, 150135], [60955, 166490], [60962, 194624], [60976, 137275],
    [61000, 155993], [61014, 144373], [61019, 166850], [61024, 138566],
    [61054, 159441], [13877, 61065], [61084, 166701], [21024, 61088],
    [15384, 61089], [61090, 146631], [61091, 155351], [61092, 161366],
    [61093, 152881], [61094, 137540], [61096, 170243], [61097, 159196],
    [61098, 159917], [61100, 156077], [61101, 166415], [61102, 145015],
    [61103, 131310], [61104, 157766], [61105, 151310], [17762, 61106],
    [23327, 61107], [61108, 156492], [40784, 61109], [40614, 61110],
    [61111, 156267], [20962, 57415], [21314, 57416], [26285, 57520],
    [22620, 57547], [21843, 57566], [15749, 57594], [24928, 57608],
    [18606, 57668], [38845, 57676], [57693, 137335], [24755, 57703],
    [33828, 57711], [38932, 57748], [57749, 147596], [57764, 143486],
    [57787, 138813], [15147, 57798], [15666, 57824], [57857, 132021],
    [28801, 57944], [23708, 57959], [58017, 132547], [14128, 58028],
    [58096, 136054], [58097, 150034], [58111, 166699], [58112, 155779],
    [256, 62233], [193, 62234], [461, 62235], [192, 62236], [274, 62237],
    [201, 62238], [282, 62239], [200, 62240], [332, 62241], [211, 62242],
    [465, 62243], [210, 62244], 62245, [7870, 62246], 62247, [7872, 62248],
    [202, 62249], [257, 62250], [225, 62251], [462, 62252], [224, 62253],
    [593, 62254], [275, 62255], [233, 62256], [283, 62257], [232, 62258],
    [299, 62259], [237, 62260], [464, 62261], [236, 62262], [333, 62263],
    [243, 62264], [466, 62265], [242, 62266], [363, 62267], [250, 62268],
    [468, 62269], [249, 62270], [470, 62271], [472, 62272], [474, 62273],
    [476, 62274], [252, 62275], 62276, [7871, 62277], 62278, [7873, 62279],
    [234, 62280], [609, 62281], [643, 63551], [592, 63552], [603, 63553],
    [596, 63554], [629, 63555], [339, 63556], [248, 63557], [331, 63558],
    [650, 63559], [618, 63560], {f: 2, c: 62282}, [11933, 63530],
    [11974, 63539], [12003, 63547], 20539, 28158, [62841, 171123], 62842,
    [15817, 62843], 34959, [62845, 147790], 28791, 23797, [19232, 62848],
    [62849, 152013], [13657, 62850], [62851, 154928], 24866, [62853, 166450],
    36775, 37366, 29073, 26393, 29626, [62859, 144001], [62860, 172295],
    [15499, 62861], [62862, 137600], [19216, 62863], 30948, 29698, 20910,
    [62867, 165647], [16393, 62868], 27235, [62870, 172730], [16931, 62871],
    34319, 31274, [62875, 170311], [62876, 166634], 38741, 28749, 21284,
    [62880, 139390], 37876, 30425, [62883, 166371], 62884, 30685, 20131, 20464,
    20668, 20015, 20247, 62891, 21556, 32139, 22674, 22736, [62896, 138678],
    24210, 24217, 24514, [62900, 141074], 25995, [62902, 144377], 26905, 27203,
    [62905, 146531], 27903, 29184, [62909, 148741], 29580, [16091, 62911],
    [62912, 150035], 23317, 29881, 35715, [62916, 154788], [62917, 153237],
    31379, 31724, 31939, 32364, 33528, 34199, 62924, 34960, 62926, 36537,
    62928, 36815, 34143, 39392, 37409, 62933, [62934, 167353], [62935, 136255],
    [16497, 62936], [17058, 62937], 23066, 39016, 26475, [17014, 62944], 22333,
    34262, [62948, 149883], 33471, [62950, 160013], [19585, 62951],
    [62952, 159092], 23931, [62954, 158485], [62955, 159678], {f: 2, c: 62956},
    23446, 62959, 32347],
  'Adobe-GB1': [{f: 95, c: 32}, {f: 3, c: 12288}, [183, 12539], 713, 711, 168,
    12291, 12293, 8212, 65374, 8214, [8230, 8943], {f: 2, c: 8216},
    {f: 2, c: 8220}, {f: 2, c: 12308}, {f: 8, c: 12296}, {f: 2, c: 12310},
    {f: 2, c: 12304}, 177, 215, 247, 8758, {f: 2, c: 8743}, 8721, 8719, 8746,
    8745, 8712, 8759, 8730, 8869, 8741, 8736, 8978, 8857, 8747, 8750, 8801,
    8780, 8776, 8765, 8733, 8800, {f: 2, c: 8814}, {f: 2, c: 8804}, 8734, 8757,
    8756, 9794, 9792, 176, {f: 2, c: 8242}, 8451, 65284, 164, {f: 2, c: 65504},
    8240, 167, 8470, 9734, 9733, 9675, 9679, 9678, 9671, 9670, 9633, 9632,
    9651, 9650, 8251, 8594, {f: 2, c: 8592}, 8595, 12307, {f: 20, c: 9352},
    {f: 20, c: 9332}, {f: 10, c: 9312}, {f: 10, c: 12832}, {f: 12, c: 8544},
    {f: 3, c: 65281}, 65509, {f: 89, c: 65285}, 65507, {f: 83, c: 12353},
    {f: 86, c: 12449}, {f: 17, c: 913}, {f: 7, c: 931}, {f: 17, c: 945},
    {f: 7, c: 963}, {f: 7, c: 59277}, {f: 2, c: 65077}, {f: 2, c: 65081},
    {f: 2, c: 65087}, {f: 2, c: 65085}, {f: 4, c: 65089}, {f: 2, c: 59284},
    {f: 2, c: 65083}, {f: 2, c: 65079}, 65073, 59286, {f: 2, c: 65075},
    {f: 6, c: 1040}, 1025, {f: 32, c: 1046}, 1105, {f: 26, c: 1078}, 257, 225,
    462, 224, 275, 233, 283, 232, 299, 237, 464, 236, 333, 243, 466, 242, 363,
    250, 468, 249, 470, 472, 474, 476, 252, 234, 593, 7743, 324, 328, 505, 609,
    {f: 37, c: 12549}, 0, {f: 76, c: 9472}, {s: 126}, 21834, 38463, 22467,
    25384, 21710, 21769, 21696, 30353, 30284, 34108, 30702, 33406, 30861,
    29233, 38552, 38797, 27688, 23433, 20474, 25353, 26263, 23736, 33018,
    26696, 32942, 26114, 30414, 20985, 25942, 29100, 32753, 34948, 20658,
    22885, 25034, 28595, 33453, 25420, 25170, 21485, 21543, 31494,
    [12043, 20843], 30116, 24052, 25300, 36299, 38774, 25226, 32793, 22365,
    38712, 32610, 29240, [12137, 30333], 26575, 30334, 25670, 20336, 36133,
    25308, 31255, 26001, 29677, 25644, 25203, 33324, 39041, 26495, 29256,
    25198, 25292, 20276, 29923, 21322, 21150, 32458, 37030, 24110, 26758,
    27036, 33152, 32465, 26834, 30917, 34444, 38225, 20621, 35876, 33502,
    32990, 21253, 35090, 21093, 34180, 38649, 20445, 22561, 39281, 23453,
    25265, 25253, 26292, 35961, 40077, 29190, 26479, 30865, 24754, 21329,
    21271, 36744, 32972, 36125, 38049, 20493, 29384, 22791, 24811, 28953,
    34987, 22868, 33519, 26412, 31528, 23849, 32503, 29997, 27893, 36454,
    36856, 36924, [12240, 40763], [12112, 27604], 37145, 31508, 24444, 30887,
    34006, 34109, 27605, 27609, 27606, 24065, 24199, 30201, 38381, 25949,
    24330, 24517, 36767, 22721, 33218, 36991, 38491, 38829, 36793, 32534,
    36140, 25153, 20415, 21464, 21342, {f: 2, c: 36776}, 36779, 36941, 26631,
    24426, 33176, 34920, 40150, 24971, 21035, 30250, 24428, 25996, 28626,
    28392, 23486, 25672, 20853, 20912, 26564, 19993, 31177, 39292, 28851,
    30149, 24182, 29627, 33760, 25773, 25320, 38069, 27874, 21338, 21187,
    25615, 38082, 31636, 20271, 24091, 33334, 33046, 33162, 28196, 27850,
    39539, 25429, [12056, 21340], 21754, 34917, 22496, 19981, 24067, 27493,
    31807, 37096, 24598, 25830, 29468, 35009, 26448, 25165, 36130, 30572,
    36393, 37319, 24425, 33756, 34081, 39184, 21442, 34453, 27531, 24813,
    24808, 28799, 33485, 33329, 20179, 27815, 34255, 25805, 31961, 27133,
    26361, 33609, 21397, 31574, 20391, 20876, 27979, 23618, 36461, 25554,
    21449, 33580, 33590, 26597, 30900, 25661, 23519, 23700, 24046, 35815,
    25286, 26612, 35962, 25600, 25530, 34633, 39307, 35863, 32544, 38130,
    20135, 38416, 39076, 26124, 29462, 22330, 23581, 24120, 38271, 20607,
    32928, [12058, 21378], 25950, 30021, 21809, 20513, 36229, 25220, 38046,
    26397, 22066, 28526, 24034, 21557, 28818, 36710, 25199, 25764, 25507,
    24443, 28552, 37108, [12162, 33251], [12192, 36784], 23576, 26216, 24561,
    27785, 38472, 36225, 34924, 25745, 31216, 22478, 27225, 25104, 21576,
    20056, 31243, 24809, 28548, 35802, 25215, 36894, 39563, 31204, 21507,
    30196, 25345, 21273, 27744, 36831, 24347, 39536, 32827, 40831, 20360,
    23610, [12186, 36196], 32709, 26021, 28861, 20805, 20914, [12173, 34411],
    23815, 23456, 25277, 37228, 30068, 36364, 31264, 24833, 31609, 20167,
    32504, 30597, 19985, 33261, 21021, 20986, 27249, 21416, 36487, 38148,
    38607, 28353, 38500, 26970, 30784, 20648, 30679, 25616, 35302, 22788,
    25571, 24029, 31359, 26941, 20256, 33337, 21912, 20018, 30126, 31383,
    24162, 24202, 38383, 21019, 21561, 28810, 25462, 38180, 22402, 26149,
    26943, 37255, 21767, 28147, 32431, 34850, 25139, 32496, 30133, 33576,
    30913, 38604, 36766, 24904, 29943, 35789, 27492, 21050, 36176, 27425,
    32874, 33905, 22257, 21254, 20174, 19995, 20945, 31895, 37259, 31751,
    20419, 36479, 31713, 31388, 25703, 23828, 20652, 33030, 30209, 31929,
    28140, 32736, 26449, 23384, [12072, 23544], 30923, 25774, 25619, 25514,
    25387, 38169, 25645, 36798, 31572, 30249, 25171, [12068, 22823], 21574,
    [12109, 27513], 20643, 25140, 24102, 27526, 20195, 36151, 34955, 24453,
    36910, 24608, 32829, 25285, 20025, 21333, 37112, 25528, 32966, 26086,
    27694, 20294, 24814, 28129, 35806, 24377, 34507, 24403, 25377, 20826,
    33633, 26723, [12049, 20992], 25443, 36424, 20498, 23707, 31095, 23548,
    21040, 31291, 24764, 36947, 30423, 24503, 24471, 30340, 36460, 28783,
    30331, 31561, 30634, 20979, 37011, 22564, 20302, 28404, 36842, 25932,
    31515, 29380, 28068, 32735, 23265, 25269, 24213, 22320, 33922, 31532,
    24093, 24351, 36882, 32532, 39072, 25474, 28359, 30872, 28857, 20856,
    38747, 22443, 30005, 20291, 30008, 24215, 24806, 22880, 28096, 27583,
    30857, 21500, 38613, 20939, 20993, 25481, 21514, 38035, 35843, 36300,
    29241, 30879, 34678, 36845, 35853, 21472, 19969, 30447, 21486, 38025,
    39030, [12237, 40718], 38189, 23450, 35746, 20002, 19996, 20908, 33891,
    25026, 21160, 26635, 20375, 24683, 20923, 27934, 20828, 25238,
    [12099, 26007], 38497, [12182, 35910], 36887, 30168, 37117, 30563, 27602,
    29322, 29420, 35835, 22581, 30585, 36172, 26460, 38208, 32922, 24230,
    28193, 22930, 31471, 30701, 38203, 27573, 26029, 32526, 22534, 20817,
    38431, 23545, 22697, 21544, 36466, 25958, 39039, 22244, 38045, 30462,
    36929, 25479, 21702, 22810, 22842, 22427, 36530, 26421, 36346, 33333,
    21057, 24816, 22549, 34558, 23784, 40517, 20420, 39069, 35769, 23077,
    24694, 21380, 25212, 36943, 37122, 39295, 24681, [12157, 32780],
    [12041, 20799], [12159, 32819], 23572, 39285, 27953, [12038, 20108], 36144,
    21457, 32602, 31567, 20240, 20047, 38400, 27861, 29648, 34281, 24070,
    30058, 32763, 27146, 30718, 38034, 32321, 20961, 28902, 21453, 36820,
    33539, 36137, 29359, 39277, 27867, 22346, 33459, [12101, 26041], 32938,
    25151, 38450, 22952, 20223, 35775, 32442, 25918, 33778, [12206, 38750],
    21857, 39134, 32933, 21290, 35837, 21536, 32954, 24223, 27832, 36153,
    33452, 37210, 21545, 27675, 20998, 32439, 22367, 28954, 27774, 31881,
    22859, 20221, 24575, 24868, 31914, 20016, 23553, 26539, 34562, 23792,
    38155, 39118, 30127, 28925, 36898, 20911, 32541, 35773, 22857, 20964,
    20315, 21542, 22827, 25975, 32932, 23413, 25206, 25282, 36752, 24133,
    27679, 31526, 20239, 20440, 26381, 28014, 28074, 31119, 34993, 24343,
    29995, 25242, 36741, 20463, 37340, 26023, 33071, 33105, 24220, 33104,
    36212, 21103, 35206, 36171, 22797, 20613, 20184, [12201, 38428],
    [12119, 29238], 33145, 36127, 23500, 35747, 38468, 22919, 32538, 21648,
    22134, 22030, 35813, 25913, 27010, 38041, 30422, 28297, [12082, 24178],
    [12130, 29976], 26438, 26577, 31487, 32925, 36214, 24863, 31174, 25954,
    36195, 20872, 21018, 38050, 32568, 32923, 32434, 23703, 28207, 26464,
    31705, 30347, [12220, 39640], 33167, 32660, 31957, 25630, 38224, 31295,
    21578, 21733, 27468, 25601, [12093, 25096], 40509, 33011, 30105, 21106,
    [12208, 38761], 33883, 26684, 34532, 38401, 38548, 38124, 20010, 21508,
    32473, 26681, 36319, 32789, 26356, 24218, 32697, 22466, 32831, 26775,
    [12079, 24037], 25915, 21151, 24685, 40858, 20379, 36524, 20844, 23467,
    [12088, 24339], 24041, 27742, 25329, 36129, 20849, 38057, 21246, 27807,
    33503, 29399, 22434, 26500, 36141, 22815, 36764, 33735, 21653, 31629,
    20272, 27837, 23396, 22993, [12238, 40723], 21476, 34506, [12219, 39592],
    [12181, 35895], 32929, 25925, 39038, 22266, 38599, 21038, [12128, 29916],
    21072, 23521, 25346, 35074, 20054, 25296, 24618, 26874, 20851, 23448,
    20896, 35266, 31649, 39302, 32592, 24815, 28748, 36143, 20809,
    [12084, 24191], 36891, 29808, 35268, 22317, 30789, 24402, 40863, 38394,
    36712, [12225, 39740], 35809, 30328, 26690, 26588, 36330, 36149, 21053,
    36746, 28378, 26829, 38149, 37101, 22269, 26524, 35065, 36807, 21704,
    39608, 23401, 28023, 27686, 20133, 23475, 39559, 37219, 25000, 37039,
    38889, 21547, 28085, 23506, 20989, 21898, 32597, 32752, 25788, 25421,
    26097, 25022, 24717, 28938, 27735, 27721, 22831, 26477, 33322, 22741,
    22158, 35946, 27627, 37085, 22909, 32791, 21495, 28009, 21621, 21917,
    33655, 33743, 26680, [12146, 31166], 21644, 20309, 21512, 30418, 35977,
    38402, 27827, 28088, 36203, 35088, 40548, 36154, 22079, [12234, 40657],
    30165, 24456, 29408, 24680, 21756, 20136, 27178, 34913, 24658, 36720,
    21700, 28888, 34425, 40511, 27946, 23439, 24344, 32418, 21897, 20399,
    29492, 21564, 21402, 20505, 21518, 21628, 20046, 24573, 29786, 22774,
    33899, 32993, 34676, 29392, 31946, 28246, 24359, 34382, 21804, 25252,
    20114, 27818, 25143, 33457, 21719, 21326, 29502, 28369, 30011, 21010,
    21270, 35805, 27088, 24458, 24576, 28142, 22351, 27426, 29615, 26707,
    36824, 32531, 25442, 24739, 21796, 30186, 35938, 28949, 28067, 23462,
    24187, 33618, 24908, 40644, 30970, 34647, 31783, 30343, 20976, 24822,
    29004, 26179, 24140, 24653, 35854, 28784, 25381, 36745, 24509, 24674,
    34516, 22238, 27585, 24724, 24935, 21321, 24800, 26214, 36159, 31229,
    20250, 28905, 27719, 35763, 35826, 32472, 33636, 26127, 23130, 39746,
    27985, 28151, 35905, 27963, 20249, [12117, 28779], 33719, 25110, 24785,
    38669, 36135, 31096, 20987, 22334, 22522, 26426, 30072, 31293, 31215,
    31637, 32908, 39269, 36857, 28608, 35749, 40481, 23020, 32489, 32521,
    21513, 26497, 26840, 36753, 31821, 38598, 21450, 24613, 30142, 27762,
    21363, 23241, 32423, 25380, [12047, 20960], 33034, [12080, 24049], 34015,
    25216, 20864, 23395, 20238, 31085, 21058, 24760, 27982, 23492, 23490,
    35745, 35760, 26082, 24524, 38469, 22931, 32487, 32426, 22025, 26551,
    22841, 20339, 23478, 21152, 33626, 39050, 36158, 30002, 38078, 20551,
    31292, 20215, 26550, 39550, 23233, 27516, 30417, 22362, 23574, 31546,
    38388, 29006, 20860, 32937, 33392, 22904, 32516, 33575, 26816, 26604,
    30897, 30839, 25315, 25441, 31616, 20461, 21098, 20943, 33616, 27099,
    37492, 36341, 36145, 35265, 38190, 31661, 20214, 20581, 33328, 21073,
    39279, 28176, 28293, 28071, 24314, 20725, 23004, 23558, 27974, 27743,
    30086, 33931, 26728, 22870, 35762, 21280, 37233, 38477, 34121, 26898,
    30977, 28966, 33014, 20132, 37066, 27975, 39556, 23047, 22204, 25605,
    38128, 30699, 20389, 33050, 29409, [12179, 35282], 39290, 32564, 32478,
    21119, 25945, 37237, 36735, 36739, 21483, 31382, 25581, 25509, 30342,
    31224, 34903, 38454, 25130, 21163, 33410, 26708, 26480, 25463, 30571,
    31469, 27905, 32467, 35299, 22992, 25106, 34249, 33445, 30028, 20511,
    20171, 30117, 35819, 23626, [12081, 24062], 31563, [12100, 26020],
    [12198, 37329], 20170, 27941, 35167, 32039, 38182, 20165, 35880, 36827,
    38771, 26187, 31105, 36817, 28908, 28024, 23613, 21170, 33606, 20834,
    33550, 30555, 26230, 40120, 20140, 24778, 31934, 31923, 32463, 20117,
    35686, 26223, 39048, 38745, 22659, 25964, 38236, 24452, 30153, 38742,
    31455, 31454, 20928, 28847, 31384, 25578, 31350, 32416, 29590,
    [12210, 38893], 20037, 28792, 20061, 37202, 21417, 25937, 26087,
    [12165, 33276], 33285, 21646, 23601, 30106, 38816, 25304, 29401, 30141,
    23621, 39545, 33738, 23616, 21632, 30697, 20030, 27822, 32858, 25298,
    25454, 24040, 20855, 36317, 36382, 38191, 20465, 21477, 24807, 28844,
    21095, 25424, 40515, 23071, 20518, 30519, 21367, 32482, 25733, 25899,
    25225, 25496, 20500, 29237, 35273, 20915, 35776, 32477, 22343, 33740,
    38055, 20891, 21531, 23803, 20426, 31459, 27994, 37089, 39567, 21888,
    21654, 21345, 21679, 24320, 25577, 26999, 20975, 24936, 21002, 22570,
    21208, 22350, 30733, 30475, 24247, 24951, 31968, 25179, 25239, 20130,
    28821, 32771, 25335, 28900, 38752, 22391, 33499, 26607, 26869, 30933,
    39063, 31185, 22771, 21683, 21487, 28212, 20811, 21051, 23458, 35838,
    32943, 21827, 22438, 24691, 22353, 21549, 31354, 24656, 23380, 25511,
    25248, [12061, 21475], 25187, 23495, 26543, 21741, 31391, 33510, 37239,
    24211, 35044, 22840, 22446, 25358, 36328, 33007, 22359, 31607, 20393,
    24555, 23485, 27454, 21281, 31568, 29378, 26694, 30719, 30518, 26103,
    20917, 20111, 30420, 23743, 31397, 33909, 22862, 39745, 20608, 39304,
    24871, 28291, 22372, 26118, 25414, 22256, 25324, 25193, 24275, 38420,
    22403, 25289, 21895, 34593, 33098, 36771, 21862, 33713, 26469, 36182,
    34013, 23146, 26639, 25318, 31726, 38417, 20848, 28572, 35888, 25597,
    35272, 25042, 32518, 28866, 28389, 29701, 27028, 29436, 24266, 37070,
    26391, 28010, 25438, 21171, 29282, [12156, 32769], 20332, 23013, 37226,
    28889, 28061, 21202, 20048, 38647, 38253, 34174, 30922, 32047, 20769,
    22418, 25794, 32907, 31867, 27882, 26865, 26974, 20919, 21400, 26792,
    29313, 40654, 31729, 29432, 31163, 28435, 29702, 26446, [12197, 37324],
    40100, 31036, 33673, 33620, 21519, 26647, 20029, 21385, 21169, 30782,
    21382, 21033, 20616, 20363, 20432, 30178, [12148, 31435], 31890, 27813,
    [12202, 38582], [12050, 21147], 29827, 21737, 20457, 32852, 33714, 36830,
    38256, 24265, 24604, 28063, 24088, 25947, 33080, 38142, 24651, 28860,
    32451, 31918, 20937, 26753, 31921, 33391, 20004, 36742, 37327, 26238,
    20142, 35845, 25769, 32842, 20698, 30103, 29134, 23525, 36797, 28518,
    20102, 25730, 38243, 24278, 26009, 21015, 35010, 28872, 21155, 29454,
    29747, 26519, 30967, 38678, 20020, 37051, 40158, 28107, 20955, 36161,
    21533, 25294, 29618, 33777, 38646, 40836, 38083, 20278, 32666, 20940,
    28789, 38517, 23725, 39046, 21478, 20196, 28316, 29705, 27060, 30827,
    39311, 30041, 21016, 30244, 27969, 26611, 20845, 40857, 32843, 21657,
    31548, 31423, 38534, 22404, 25314, 38471, 27004, 23044, 25602, 31699,
    28431, 38475, 33446, 21346, 39045, 24208, 28809, 25523, 21348, 34383,
    40065, 40595, 30860, 38706, 36335, 36162, [12229, 40575], 28510, 31108,
    24405, 38470, 25134, 39540, 21525, 38109, 20387, 26053, 23653, 23649,
    32533, 34385, 27695, 24459, 29575, 28388, 32511, 23782, 25371, 23402,
    28390, 21365, 20081, 25504, 30053, 25249, 36718, 20262, 20177, 27814,
    32438, 35770, 33821, 34746, 32599, 36923, 38179, 31657, 39585, 35064,
    33853, 27931, 39558, 32476, 22920, [12231, 40635], 29595, 30721, 34434,
    39532, 39554, 22043, 21527, 22475, 20080, 40614, 21334, 36808, 33033,
    30610, 39314, 34542, 28385, 34067, 26364, 24930, 28459, 35881, 33426,
    33579, 30450, 27667, 24537, 33725, 29483, 33541, 38170, [12113, 27611],
    [12141, 30683], 38086, 21359, 33538, 20882, 24125, 35980, 36152, 20040,
    29611, 26522, 26757, 37238, 38665, 29028, 27809, 30473, 23186, 38209,
    27599, 32654, 26151, 23504, 22969, 23194, 38376, 38391, 20204, 33804,
    33945, 27308, 30431, 38192, 29467, 26790, 23391, 30511, 37274, 38753,
    31964, 36855, 35868, 24357, [12150, 31859], 31192, 35269, 27852, 34588,
    23494, 24130, 26825, 30496, 32501, 20885, 20813, 21193, 23081, 32517,
    [12207, 38754], 33495, 25551, 30596, 34256, 31186, 28218, 24217, 22937,
    34065, 28781, 27665, 25279, [12139, 30399], 25935, 24751, 38397, 26126,
    34719, 40483, 38125, 21517, 21629, 35884, {f: 2, c: 25720}, 34321, 27169,
    33180, 30952, 25705, 39764, 25273, 26411, 33707, 22696, 40664, 27819,
    28448, 23518, 38476, 35851, 29279, 26576, 25287, 29281, 20137, 22982,
    27597, 22675, 26286, 24149, 21215, 24917, [12106, 26408], [12140, 30446],
    30566, 29287, 31302, 25343, 21738, 21584, 38048, 37027, 23068, 32435,
    27670, 20035, 22902, 32784, 22856, 21335, 30007, 38590, 22218, 25376,
    33041, 24700, 38393, 28118, 21602, 39297, 20869, 23273, 33021, 22958,
    38675, 20522, 27877, 23612, 25311, 20320, 21311, 33147, 36870, 28346,
    34091, 25288, 24180, 30910, 25781, 25467, 24565, 23064, 37247, 40479,
    23615, 25423, 32834, 23421, 21870, 38218, 38221, 28037, 24744, 26592,
    29406, 20957, 23425, 25319, 27870, [12124, 29275], 25197, 38062, 32445,
    33043, 27987, 20892, 24324, 22900, 21162, 24594, [12069, 22899], 26262,
    34384, 30111, 25386, 25062, 31983, 35834, 21734, 27431, 40485, 27572,
    34261, 21589, 20598, 27812, 21866, 36276, 29228, 24085, 24597, 29750,
    25293, 25490, 29260, 24472, 28227, 27966, 25856, 28504, 30424, 30928,
    30460, 30036, 21028, 21467, 20051, 24222, 26049, 32810, 32982, 25243,
    21638, 21032, 28846, 34957, 36305, 27873, 21624, 32986, 22521, 35060,
    36180, 38506, 37197, 20329, 27803, 21943, 30406, 30768, 25256, 28921,
    28558, 24429, 34028, 26842, 30844, 31735, 33192, 26379, 40527, 25447,
    30896, 22383, 30738, 38713, 25209, 25259, 21128, 29749, 27607, 21860,
    33086, 30130, [12138, 30382], 21305, 30174, 20731, 23617, 35692, 31687,
    20559, [12122, 29255], 39575, 39128, 28418, 29922, 31080, 25735, 30629,
    25340, 39057, 36139, 21697, 32856, 20050, 22378, 33529, 33805, 24179,
    20973, 29942, 35780, 23631, 22369, 27900, 39047, 23110, 30772, 39748,
    36843, 31893, 21078, 25169, 38138, 20166, 33670, 33889, 33769, 33970,
    22484, 26420, 22275, 26222, 28006, 35889, 26333, 28689, 26399, 27450,
    26646, 25114, 22971, 19971, 20932, 28422, 26578, 27791, 20854, 26827,
    22855, 27495, 30054, 23822, 33040, 40784, 26071, 31048, 31041, 39569,
    36215, 23682, 20062, 20225, 21551, 22865, 30732, 22120, [12115, 27668],
    36804, 24323, 27773, 27875, 35755, 25488, 24688, 27965, 29301, 25190,
    38030, 38085, 21315, 36801, 31614, 20191, 35878, 20094, 40660, 38065,
    38067, 21069, 28508, 36963, 27973, 35892, 22545, 23884, [12107, 27424],
    27465, 26538, 21595, 33108, 32652, 22681, 34103, 24378, 25250, 27207,
    38201, 25970, 24708, 26725, 30631, 20052, 20392, 24039, 38808, 25772,
    32728, 23789, 20431, 31373, 20999, 33540, 19988, 24623, 31363, 38054,
    20405, 20146, 31206, 29748, 21220, 33465, 25810, 31165, 23517, 27777,
    38738, 36731, 27682, 20542, 21375, 28165, 25806, 26228, 27696, 24773,
    39031, 35831, 24198, 29756, 31351, 31179, 19992, 37041, 29699, 27714,
    22234, 37195, 27845, 36235, 21306, 34502, 26354, 36527, 23624, 39537,
    28192, 21462, 23094, 40843, 36259, 21435, 22280, 39079, 26435, 37275,
    27849, 20840, 30154, 25331, [12125, 29356], 21048, 21149, 32570, 28820,
    30264, 21364, 40522, 27063, 30830, 38592, 35033, 32676, 28982, 29123,
    20873, 26579, 29924, 22756, 25880, 22199, 35753, 39286, 25200, 32469,
    24825, 28909, 22764, 20161, [12040, 20154], 24525, 38887, 20219, 35748,
    20995, 22922, 32427, 25172, 20173, [12103, 26085], 25102, 33592, 33993,
    33635, 34701, 29076, 28342, 23481, 32466, 20887, 25545, 26580,
    [12161, 32905], 33593, 34837, 20754, 23418, 22914, 36785, 20083, 27741,
    [12042, 20837], 35109, 36719, 38446, 34122, 29790, 38160, 38384, 28070,
    33509, 24369, 25746, 27922, 33832, 33134, 40131, 22622, 36187, 19977,
    21441, 20254, 25955, 26705, 21971, 20007, 25620, 39578, 25195, 23234,
    29791, [12170, 33394], 28073, 26862, 20711, 33678, 30722, 26432, 21049,
    27801, 32433, 20667, 21861, 29022, 31579, 26194, 29642, 33515, 26441,
    [12077, 23665], 21024, 29053, 34923, 38378, 38485, 25797, 36193, 33203,
    21892, 27733, 25159, 32558, 22674, 20260, 21830, 36175, 26188, 19978,
    23578, 35059, 26786, 25422, 31245, 28903, 33421, 21242, 38902, 23569,
    21736, 37045, 32461, 22882, 36170, 34503, [12166, 33292], 33293, 36198,
    25668, 23556, 24913, 28041, 31038, 35774, 30775, 30003, 21627, 20280,
    [12189, 36523], 28145, 23072, 32453, 31070, 27784, 23457, 23158, 29978,
    32958, 24910, 28183, 22768, [12131, 29983], 29989, 29298, 21319, 32499,
    30465, 30427, 21097, 32988, 22307, 24072, 22833, 29422, 26045, 28287,
    35799, [12075, 23608], 34417, [12055, 21313], [12143, 30707], 25342, 26102,
    20160, [12215, 39135], 34432, 23454, 35782, 21490, [12142, 30690], 20351,
    23630, 39542, 22987, 24335, [12144, 31034], [12064, 22763], 19990, 26623,
    20107, 25325, 35475, 36893, 21183, 26159, 21980, 22124, 36866, 20181,
    20365, 37322, 39280, [12114, 27663], 24066, 24643, 23460, 35270, 35797,
    25910, [12095, 25163], [12216, 39318], 23432, 23551, 25480, 21806, 21463,
    30246, 20861, 34092, 26530, 26803, 27530, 25234, 36755, 21460, 33298,
    28113, 30095, 20070, 36174, 23408, 29087, 34223, 26257, 26329, 32626,
    34560, [12233, 40653], [12239, 40736], 23646, 26415, 36848, 26641, 26463,
    25101, 31446, 22661, 24246, 25968, 28465, 24661, 21047, 32781, 25684,
    34928, 29993, 24069, 26643, 25332, 38684, 21452, 29245, 35841,
    [12116, 27700], 30561, 31246, 21550, 30636, 39034, 33308, 35828, 30805,
    26388, 28865, 26031, 25749, 22070, 24605, 31169, 21496, 19997, 27515,
    32902, 23546, 21987, 22235, 20282, 20284, 39282, 24051, 26494, 32824,
    24578, 39042, 36865, 23435, 35772, 35829, 25628, 33368, 25822, 22013,
    33487, 37221, 20439, 32032, 36895, 31903, 20723, 22609, 28335, 23487,
    35785, 32899, 37240, 33948, 31639, 34429, 38539, 38543, 32485, 39635,
    30862, 23681, 31319, 36930, 38567, 31071, 23385, 25439, 31499, 34001,
    26797, 21766, 32553, 29712, 32034, 38145, 25152, 22604, 20182, 23427,
    22905, 22612, 29549, 25374, 36427, 36367, 32974, 33492, 25260, 21488,
    27888, 37214, 22826, 24577, 27760, 22349, 25674, 36138, 30251, 28393,
    22363, 27264, 30192, 28525, 35885, 35848, 22374, 27631, 34962, 30899,
    25506, 21497, 28845, 27748, 22616, 25642, 22530, 26848, 33179, 21776,
    31958, 20504, 36538, 28108, 36255, 28907, 25487, 28059, 28372, 32486,
    33796, 26691, 36867, 28120, 38518, 35752, 22871, 29305, 34276, 33150,
    30140, 35466, 26799, 21076, 36386, 38161, 25552, 39064, 36420, 21884,
    20307, 26367, 22159, 24789, 28053, 21059, 23625, 22825, 28155, 22635,
    [12133, 30000], 29980, 24684, 33300, 33094, 25361, 26465, 36834, 30522,
    36339, 36148, 38081, 24086, 21381, 21548, 28867, 27712, 24311, 20572,
    20141, 24237, 25402, 33351, 36890, 26704, 37230, 30643, 21516, 38108,
    24420, 31461, 26742, 25413, 31570, 32479, 30171, 20599, 25237, 22836,
    36879, 20984, 31171, 31361, 22270, 24466, 36884, 28034, 23648,
    [12063, 22303], 21520, 20820, 28237, 22242, 25512, 39059, 33151, 34581,
    35114, 36864, 21534, 23663, 33216, 25302, 25176, 33073, 40501, 38464,
    39534, 39548, 26925, 22949, 25299, 21822, 25366, 21703, 34521, 27964,
    23043, [12129, 29926], 34972, 27498, 22806, 35916, 24367, 28286, 29609,
    39037, 20024, 28919, 23436, 30871, 25405, 26202, 30358, 24779, 23451,
    23113, 19975, 33109, 27754, 29579, 20129, 26505, [12153, 32593], 24448,
    26106, 26395, 24536, 22916, 23041, 24013, 24494, 21361, 38886, 36829,
    26693, 22260, 21807, 24799, 20026, 28493, 32500, 33479, 33806, 22996,
    20255, 20266, 23614, 32428, 26410, 34074, 21619, 30031, 32963, 21890,
    39759, 20301, 28205, 35859, 23561, 24944, 21355, 30239, 28201, 34442,
    [12098, 25991], 38395, 32441, 21563, 31283, 32010, 38382, 21985, 32705,
    29934, 25373, 34583, 28065, 31389, 25105, 26017, 21351, 25569, 27779,
    24043, 21596, 38056, 20044, 27745, 35820, 23627, [12102, 26080], 33436,
    26791, 21566, 21556, [12111, 27595], 27494, 20116, 25410, 21320, 33310,
    20237, 20398, 22366, 25098, 38654, 26212, 29289, 21247, 21153, 24735,
    35823, 26132, 29081, 26512, 35199, 30802, 30717, 26224, 22075, 21560,
    38177, 29306, 31232, 24687, 24076, 24713, 33181, [12067, 22805], 24796,
    29060, 28911, 28330, 27728, 29312, 27268, 34989, 24109, 20064, 23219,
    21916, 38115, 27927, 31995, 38553, 25103, 32454, 30606, 34430, 21283,
    38686, 36758, 26247, 23777, 20384, 29421, 19979, 21414, 22799, 21523,
    25472, 38184, 20808, 20185, 40092, 32420, 21688, 36132, 34900, 33335,
    38386, 28046, 24358, 23244, 26174, 38505, 29616, 29486, 21439, 33146,
    39301, 32673, 23466, 38519, 38480, 32447, 30456, 21410, 38262,
    [12217, 39321], 31665, 35140, 28248, 20065, 32724, 31077, 35814, 24819,
    21709, 20139, 39033, 24055, 27233, 20687, 21521, 35937, 33831, 30813,
    38660, 21066, 21742, 22179, 38144, 28040, 23477, 28102, 26195,
    [12073, 23567], 23389, 26657, 32918, 21880, 31505, 25928, 26964, 20123,
    27463, 34638, 38795, 21327, 25375, 25658, 37034, 26012, 32961, 35856,
    20889, 26800, 21368, 34809, 25032, 27844, 27899, 35874, 23633, 34218,
    33455, 38156, 27427, [12191, 36763], 26032, 24571, [12092, 24515], 20449,
    34885, 26143, 33125, 29481, 24826, 20852, 21009, 22411, 24418, 37026,
    [12175, 34892], 37266, 24184, 26447, 24615, 22995, 20804, 20982, 33016,
    21256, 27769, 38596, 29066, 20241, 20462, 32670, 26429, 21957, 38152,
    31168, 34966, 32483, 22687, 25100, 38656, 34394, 22040, 39035, 24464,
    35768, 33988, 37207, 21465, 26093, 24207, 30044, 24676, 32110, 23167,
    32490, 32493, 36713, 21927, 23459, 24748, 26059, [12126, 29572], 36873,
    30307, 30505, 32474, 38772, 34203, 23398, [12147, 31348], 38634,
    [12174, 34880], 21195, 29071, 24490, 26092, 35810, 23547, 39535, 24033,
    27529, 27739, 35757, 35759, 36874, 36805, 21387, 25276, 40486, 40493,
    21568, 20011, 33469, [12123, 29273], 34460, 23830, 34905, 28079, 38597,
    21713, 20122, 35766, 28937, 21693, 38409, 28895, 28153, 30416, 20005,
    30740, 34578, 23721, 24310, [12180, 35328], 39068, 38414, 28814, 27839,
    22852, 25513, 30524, 34893, 28436, 33395, 22576, 29141, 21388, 30746,
    38593, 21761, 24422, 28976, 23476, 35866, 39564, 27523, 22830, 40495,
    31207, 26472, 25196, 20335, 30113, [12154, 32650], 27915, 38451, 27687,
    20208, 30162, 20859, 26679, 28478, 36992, 33136, 22934, 29814, 25671,
    23591, 36965, 31377, 35875, 23002, 21676, 33280, 33647, 35201, 32768,
    26928, 22094, 32822, 29239, 37326, 20918, 20063, 39029, 25494, 19994,
    21494, 26355, 33099, 22812, 28082, [12032, 19968], 22777, 21307, 25558,
    38129, 20381, 20234, [12176, 34915], 39056, 22839, 36951, 31227, 20202,
    33008, 30097, 27778, 23452, 23016, 24413, 26885, 34433, 20506, 24050,
    [12036, 20057], 30691, 20197, 33402, 25233, 26131, [12194, 37009], 23673,
    20159, 24441, 33222, 36920, 32900, 30123, 20134, 35028, 24847, 27589,
    24518, 20041, 30410, 28322, 35811, 35758, 35850, 35793, 24322, 32764,
    32716, 32462, 33589, 33643, 22240, 27575, [12211, 38899], 38452, 23035,
    21535, 38134, 28139, 23493, 39278, 23609, 24341, 38544, 21360, 33521,
    27185, 23156, 40560, 24212, 32552, 33721, {f: 2, c: 33828}, 33639, 34631,
    36814, 36194, 30408, 24433, 39062, 30828, 26144, 21727, 25317, 20323,
    33219, 30152, 24248, 38605, 36362, 34553, 21647, 27891, 28044, 27704,
    24703, 21191, [12132, 29992], 24189, 20248, 24736, 24551, 23588, 30001,
    37038, 38080, 29369, 27833, 28216, [12195, 37193], 26377, 21451, 21491,
    20305, 37321, 35825, [12060, 21448], 24188, 36802, 28132, 20110, 30402,
    27014, 34398, 24858, 33286, 20313, 20446, 36926, 40060, 24841, 28189,
    28180, 38533, 20104, 23089, [12204, 38632], 19982, 23679, 31161, 23431,
    35821, [12155, 32701], [12127, 29577], 22495, 33419, 37057, 21505, 36935,
    21947, 23786, 24481, 24840, 27442, 29425, 32946, 35465, 28020, 23507,
    35029, 39044, 35947, 39533, 40499, 28170, 20900, 20803, 22435, 34945,
    21407, 25588, 36757, 22253, 21592, 22278, 29503, 28304, 32536, 36828,
    33489, 24895, 24616, 38498, [12104, 26352], 32422, 36234, 36291, 38053,
    23731, 31908, [12105, 26376], 24742, 38405, 32792, 20113, 37095, 21248,
    38504, 20801, 36816, 34164, 37213, 26197, 38901, 23381, 21277, 30776,
    26434, 26685, 21705, 28798, 23472, 36733, 20877, 22312, 21681, 25874,
    26242, 36190, 36163, 33039, 33900, 36973, 31967, 20991, 34299, 26531,
    26089, 28577, 34468, 36481, 22122, 36896, 30338, 28790, 29157, 36131,
    25321, 21017, 27901, 36156, 24590, 22686, 24974, 26366, 36192, 25166,
    21939, 28195, 26413, 36711, 38113, 38392, 30504, 26629, 27048, 21643,
    20045, 28856, 35784, 25688, 25995, 23429, 31364, 20538, 23528, 30651,
    27617, 35449, 31896, 27838, 30415, 26025, 36759, 23853, 23637, 34360,
    26632, 21344, 25112, 31449, 28251, 32509, 27167, 31456, 24432, 28467,
    24352, 25484, 28072, 26454, 19976, 24080, 36134, 20183, 32960, 30260,
    38556, 25307, 26157, 25214, 27836, 36213, 29031, 32617, 20806, 32903,
    21484, 36974, 25240, 21746, 34544, 36761, 32773, 38167, 34071, 36825,
    27993, 29645, 26015, 30495, 29956, 30759, 33275, 36126, 38024, 20390,
    26517, 30137, 35786, 38663, 25391, 38215, 38453, 33976, 25379, 30529,
    24449, 29424, 20105, 24596, 25972, 25327, 27491, 25919, 24103, 30151,
    37073, 35777, 33437, 26525, [12096, 25903], 21553, 34584, 30693, 32930,
    33026, 27713, 20043, 32455, 32844, 30452, 26893, 27542, 25191, 20540,
    20356, 22336, 25351, [12108, 27490], 36286, 21482, 26088, 32440, 24535,
    25370, 25527, [12164, 33267], 33268, 32622, 24092, 23769, 21046, 26234,
    31209, 31258, 36136, 28825, 30164, 28382, 27835, 31378, 20013, 30405,
    24544, 38047, 34935, 32456, 31181, 32959, 37325, 20210, 20247,
    [12168, 33311], 21608, 24030, 27954, 35788, 31909, 36724, 32920, 24090,
    21650, 30385, 23449, 26172, 39588, 29664, 26666, 34523, 26417, 29482,
    35832, 35803, 36880, [12149, 31481], 28891, 29038, 25284, 30633, 22065,
    20027, 33879, 26609, 21161, 34496, 36142, 38136, 31569, 20303, 27880,
    31069, 39547, 25235, [12118, 29226], 25341, 19987, 30742, 36716, 25776,
    36186, 31686, 26729, 24196, 35013, 22918, 25758, 22766, 29366, 26894,
    38181, 36861, 36184, 22368, 32512, 35846, 20934, 25417, 25305, 21331,
    26700, 29730, 33537, 37196, 21828, 30528, 28796, 27978, 20857, 21672,
    36164, 23039, 28363, 28100, 23388, 32043, 20180, 31869, 28371,
    [12070, 23376], [12163, 33258], 28173, 23383, 39683, 26837, 36394, 23447,
    32508, 24635, 32437, 37049, [12187, 36208], 22863, 25549, 31199,
    [12188, 36275], 21330, 26063, 31062, 35781, 38459, 32452, 38075, 32386,
    22068, 37257, 26368, 32618, 23562, 36981, 26152, 24038, 20304, 26590,
    20570, 20316, 22352, 24231, 20109, 19980, 20800, 19984, 24319, 21317,
    19989, 20120, 19998, [12224, 39730], 23404, 22121, [12033, 20008], 31162,
    [12035, 20031], [12052, 21269], 20039, 22829, [12120, 29243], 21358, 27664,
    22239, 32996, 39319, 27603, 30590, 40727, [12034, 20022], 20127, 40720,
    20060, 20073, 20115, 33416, 23387, 21868, 22031, 20164, 21389, 21405,
    21411, 21413, 21422, 38757, 36189, [12053, 21274], 21493, 21286, 21294,
    21310, 36188, 21350, 21347, 20994, 21000, 21006, 21037, 21043,
    {f: 2, c: 21055}, 21068, 21086, 21089, 21084, 33967, 21117, 21122, 21121,
    21136, 21139, [12044, 20866], 32596, 20155, 20163, 20169, 20162, 20200,
    20193, 20203, 20190, 20251, 20211, 20258, 20324, 20213, 20261, 20263,
    20233, 20267, 20318, 20327, 25912, 20314, 20317, 20319, 20311, 20274,
    20285, 20342, 20340, 20369, 20361, 20355, 20367, 20350, 20347, 20394,
    20348, 20396, 20372, 20454, 20456, 20458, 20421, 20442, 20451, 20444,
    20433, 20447, 20472, 20521, 20556, 20467, 20524, 20495, 20526, 20525,
    20478, 20508, 20492, 20517, 20520, 20606, 20547, 20565, 20552, 20558,
    20588, 20603, 20645, 20647, 20649, 20666, 20694, 20742, 20717, 20716,
    20710, 20718, 20743, 20747, 20189, 27709, 20312, 20325, 20430,
    [12245, 40864], 27718, 31860, 20846, 24061, 40649, 39320, 20865, 22804,
    [12051, 21241], 21261, 35335, 21264, 20971, 22809, 20821, [12039, 20128],
    20822, 20147, 34926, 34980, 20149, 33044, 35026, 31104, 23348, 34819,
    32696, [12046, 20907], 20913, 20925, 20924, 20935, [12045, 20886], 20898,
    20901, 35744, {f: 2, c: 35750}, 35754, {f: 2, c: 35764}, 35767,
    {f: 2, c: 35778}, 35787, 35791, 35790, {f: 3, c: 35794}, 35798,
    {f: 2, c: 35800}, 35804, {f: 2, c: 35807}, 35812, {f: 2, c: 35816}, 35822,
    35824, 35827, 35830, 35833, 35836, {f: 2, c: 35839}, 35842, 35844, 35847,
    35852, 35855, {f: 2, c: 35857}, {f: 3, c: 35860}, 35865, 35867, 35864,
    35869, {f: 3, c: 35871}, 35877, 35879, {f: 2, c: 35882}, {f: 2, c: 35886},
    {f: 2, c: 35890}, {f: 2, c: 35893}, [12057, 21353], 21370, 38429, 38434,
    38433, 38449, 38442, 38461, 38460, 38466, 38473, 38484, 38495, 38503,
    38508, 38514, 38516, 38536, 38541, 38551, 38576, 37015, 37019, 37021,
    37017, 37036, 37025, 37044, 37043, 37046, 37050, 37048, 37040, 37071,
    37061, 37054, 37072, 37060, 37063, 37075, 37094, 37090, 37084, 37079,
    37083, 37099, 37103, 37118, 37124, 37154, 37150, 37155, 37169, 37167,
    37177, 37187, 37190, 21005, 22850, 21154, {f: 2, c: 21164}, 21182, 21759,
    21200, 21206, 21232, 21471, 29166, 30669, [12085, 24308], [12048, 20981],
    20988, [12223, 39727], [12059, 21430], 24321, 30042, 24047, 22348, 22441,
    22433, 22654, 22716, 22725, 22737, 22313, 22316, 22314, 22323, 22329,
    {f: 2, c: 22318}, 22364, 22331, 22338, 22377, 22405, 22379, 22406, 22396,
    22395, 22376, 22381, 22390, 22387, 22445, 22436, 22412, 22450, 22479,
    22439, 22452, 22419, 22432, 22485, 22488, 22490, 22489, 22482, 22456,
    22516, 22511, 22520, 22500, 22493, 22539, 22541, 22525, 22509, 22528,
    22558, 22553, 22596, 22560, 22629, 22636, 22657, 22665, 22682, 22656,
    39336, 40729, 25087, 33401, 33405, 33407, 33423, 33418, 33448, 33412,
    33422, 33425, 33431, 33433, 33451, 33464, 33470, 33456, 33480, 33482,
    33507, 33432, 33463, 33454, {f: 2, c: 33483}, 33473, 33449, 33460, 33441,
    33450, 33439, 33476, 33486, 33444, 33505, 33545, 33527, 33508, 33551,
    33543, 33500, 33524, 33490, 33496, 33548, 33531, 33491, 33553, 33562,
    33542, {f: 2, c: 33556}, 33504, 33493, 33564, 33617, {f: 2, c: 33627},
    33544, 33682, 33596, 33588, 33585, 33691, 33630, 33583, 33615, 33607,
    33603, 33631, 33600, 33559, 33632, 33581, 33594, 33587, 33638, 33637,
    33640, 33563, 33641, 33644, 33642, {f: 2, c: 33645}, 33712, 33656,
    {f: 2, c: 33715}, 33696, 33706, 33683, 33692, 33669, 33660, 33718, 33705,
    33661, 33720, 33659, 33688, 33694, 33704, 33722, 33724, 33729, 33793,
    33765, 33752, 22535, 33816, 33803, 33757, 33789, 33750, 33820, 33848,
    33809, 33798, 33748, 33759, 33807, 33795, {f: 2, c: 33784}, 33770, 33733,
    33728, 33830, 33776, 33761, 33884, 33873, 33882, 33881, 33907,
    {f: 2, c: 33927}, 33914, 33929, 33912, 33852, 33862, 33897, 33910, 33932,
    33934, 33841, 33901, 33985, 33997, 34000, 34022, 33981, 34003, 33994,
    33983, 33978, 34016, 33953, 33977, 33972, 33943, 34021, 34019, 34060,
    29965, 34104, 34032, 34105, 34079, 34106, 34134, 34107, 34047, 34044,
    34137, 34120, 34152, 34148, 34142, 34170, 30626, 34115, 34162, 34171,
    34212, 34216, 34183, 34191, 34169, 34222, 34204, 34181, 34233, 34231,
    34224, 34259, 34241, 34268, 34303, 34343, 34309, 34345, 34326, 34364,
    [12086, 24318], 24328, 22844, 22849, 32823, 22869, 22874, 22872, 21263,
    [12074, 23586], 23589, 23596, 23604, 25164, 25194, 25247, 25275, 25290,
    25306, 25303, 25326, 25378, 25334, 25401, 25419, 25411, 25517, 25590,
    25457, 25466, 25486, 25524, 25453, 25516, 25482, 25449, 25518, 25532,
    25586, 25592, 25568, 25599, 25540, 25566, 25550, 25682, 25542, 25534,
    25669, 25665, 25611, 25627, 25632, 25612, 25638, 25633, 25694, 25732,
    25709, 25750, 25722, {f: 2, c: 25783}, 25753, 25786, 25792, 25808, 25815,
    25828, 25826, 25865, 25893, 25902, [12087, 24331], 24530, 29977, 24337,
    21343, 21489, 21501, 21481, 21480, 21499, 21522, 21526, 21510, 21579,
    {f: 3, c: 21586}, 21590, 21571, 21537, 21591, 21593, 21539, 21554, 21634,
    21652, 21623, 21617, 21604, {f: 2, c: 21658}, 21636, 21622, 21606, 21661,
    21712, 21677, 21698, 21684, 21714, 21671, 21670, {f: 2, c: 21715}, 21618,
    21667, 21717, 21691, 21695, 21708, {f: 2, c: 21721}, 21724,
    {f: 2, c: 21673}, 21668, 21725, 21711, 21726, 21787, 21735, 21792, 21757,
    21780, 21747, {f: 2, c: 21794}, 21775, 21777, 21799, 21802, 21863, 21903,
    21941, 21833, 21869, 21825, 21845, 21823, 21840, 21820, 21815, 21846,
    {f: 3, c: 21877}, 21811, 21808, 21852, 21899, 21970, 21891, 21937, 21945,
    21896, 21889, 21919, 21886, 21974, 21905, 21883, 21983, {f: 2, c: 21949},
    21908, 21913, 21994, 22007, 21961, 22047, 21969, {f: 2, c: 21995}, 21972,
    21990, 21981, 21956, 21999, 21989, {f: 2, c: 22002}, {f: 2, c: 21964},
    21992, 22005, 21988, 36756, 22046, 22024, 22028, 22017, 22052, 22051,
    22014, 22016, 22055, 22061, 22104, 22073, 22103, 22060, 22093, 22114,
    22105, 22108, 22092, 22100, 22150, 22116, 22129, 22123, {f: 2, c: 22139},
    22149, 22163, 22191, 22228, [12062, 22231], 22237, 22241, 22261, 22251,
    22265, 22271, 22276, 22282, 22281, 22300, 24079, 24089, 24084, 24081,
    24113, {f: 2, c: 24123}, 24119, 24132, 24148, 24155, 24158, 24161, 23692,
    23674, 23693, 23696, 23702, 23688, {f: 2, c: 23704}, 23697, 23706, 23708,
    23733, 23714, 23741, 23724, 23723, 23729, 23715, 23745, 23735, 23748,
    23762, 23780, 23755, 23781, {f: 2, c: 23810}, 23847, 23846, 23854, 23844,
    23838, 23814, 23835, 23896, 23870, 23860, 23869, 23916, 23899, 23919,
    23901, 23915, 23883, 23882, 23913, 23924, 23938, 23961, 23965, 35955,
    23991, 24005, [12091, 24435], 24439, 24450, 24455, 24457, 24460, 24469,
    24473, 24476, 24488, 24493, 24501, 24508, 34914, [12090, 24417], 29357,
    29360, 29364, {f: 2, c: 29367}, 29379, 29377, 29390, 29389, 29394, 29416,
    29423, 29417, 29426, 29428, 29431, 29441, 29427, 29443, {f: 2, c: 29434},
    29463, 29459, 29473, 29450, 29470, 29469, 29461, 29474, 29497, 29477,
    29484, 29496, 29489, 29520, 29517, 29527, 29536, 29548, 29551, 29566,
    [12167, 33307], 22821, 39143, 22820, [12065, 22786], 39267,
    {f: 6, c: 39271}, 39284, 39287, 39293, 39296, 39300, 39303, 39306, 39309,
    {f: 2, c: 39312}, {f: 3, c: 39315}, 24192, 24209, 24203, 24214, 24229,
    24224, 24249, 24245, 24254, 24243, 36179, 24274, 24273, 24283, 24296,
    24298, 33210, 24516, 24521, 24534, 24527, 24579, 24558, 24580, 24545,
    24548, 24574, {f: 2, c: 24581}, 24554, 24557, 24568, 24601, 24629, 24614,
    24603, 24591, 24589, 24617, 24619, 24586, 24639, 24609, {f: 2, c: 24696},
    24699, 24698, 24642, 24682, 24701, 24726, 24730, 24749, 24733, 24707,
    24722, 24716, 24731, 24812, 24763, 24753, 24797, 24792, 24774, 24794,
    24756, 24864, 24870, 24853, 24867, 24820, 24832, 24846, 24875, 24906,
    24949, 25004, 24980, 24999, 25015, 25044, 25077, 24541, 38579, 38377,
    38379, 38385, 38387, {f: 2, c: 38389}, 38396, 38398, {f: 2, c: 38403},
    38406, 38408, {f: 4, c: 38410}, 38415, 38418, {f: 3, c: 38421},
    {f: 2, c: 38425}, 20012, [12121, 29247], 25109, 27701, 27732, 27740, 27722,
    27811, 27781, 27792, 27796, 27788, {f: 2, c: 27752}, 27764, 27766, 27782,
    27817, 27856, 27860, 27821, {f: 2, c: 27895}, 27889, 27863, 27826, 27872,
    27862, 27898, 27883, 27886, 27825, 27859, 27887, 27902, 27961, 27943,
    27916, 27971, 27976, 27911, 27908, 27929, 27918, 27947, 27981, 27950,
    27957, 27930, 27983, 27986, 27988, 27955, 28049, 28015, 28062, 28064,
    27998, {f: 2, c: 28051}, 27996, 28000, 28028, 28003, 28186, 28103, 28101,
    28126, 28174, 28095, 28128, 28177, 28134, 28125, 28121, 28182, 28075,
    28172, 28078, 28203, 28270, 28238, 28267, 28338, 28255, 28294,
    {f: 2, c: 28243}, 28210, 28197, 28228, 28383, 28337, 28312, 28384, 28461,
    28386, 28325, 28327, 28349, 28347, 28343, 28375, 28340, 28367, 28303,
    28354, 28319, 28514, {f: 2, c: 28486}, 28452, 28437, 28409, 28463, 28470,
    28491, 28532, 28458, 28425, 28457, 28553, 28557, 28556, 28536, 28530,
    28540, 28538, 28625, 28617, 28583, 28601, 28598, 28610, 28641, 28654,
    28638, 28640, 28655, 28698, 28707, 28699, 28729, 28725, 28751, 28766,
    [12071, 23424], 23428, 23445, 23443, 23461, 23480, 29999, 39582, 25652,
    23524, 23534, 35120, 23536, 36423, 35591, 36790, 36819, 36821, 36837,
    36846, 36836, 36841, 36838, 36851, 36840, 36869, 36868, 36875, 36902,
    36881, 36877, 36886, 36897, {f: 2, c: 36917}, 36909, 36911, 36932,
    {f: 2, c: 36945}, 36944, 36968, 36952, 36962, 36955, 26297, 36980, 36989,
    36994, 37000, 36995, 37003, [12089, 24400], 24407, 24406, 24408, 23611,
    21675, 23632, 23641, 23409, 23651, 23654, 32700, 24362, 24361, 24365,
    33396, 24380, 39739, [12076, 23662], 22913, 22915, 22925, {f: 2, c: 22953},
    22947, 22935, 22986, 22955, 22942, 22948, 22994, 22962, 22959, 22999,
    22974, {f: 2, c: 23045}, 23005, 23048, 23011, 23000, 23033, 23052, 23049,
    23090, 23092, 23057, 23075, 23059, 23104, 23143, 23114, 23125, 23100,
    23138, 23157, 33004, 23210, 23195, 23159, 23162, 23230, 23275, 23218,
    23250, 23252, 23224, 23264, 23267, 23281, 23254, 23270, 23256, 23260,
    23305, 23319, 23318, 23346, 23351, 23360, 23573, 23580, 23386, 23397,
    23411, 23377, 23379, 23394, 39541, {f: 2, c: 39543}, 39546, 39551, 39549,
    {f: 2, c: 39552}, 39557, 39560, 39562, 39568, {f: 2, c: 39570}, 39574,
    39576, {f: 3, c: 39579}, {f: 2, c: 39583}, {f: 2, c: 39586}, 39589, 39591,
    32415, 32417, 32419, 32421, {f: 2, c: 32424}, 32429, 32432, 32446,
    {f: 3, c: 32448}, 32457, {f: 2, c: 32459}, 32464, 32468, 32471, 32475,
    {f: 2, c: 32480}, 32488, 32491, {f: 2, c: 32494}, {f: 2, c: 32497}, 32525,
    32502, {f: 2, c: 32506}, 32510, {f: 3, c: 32513}, {f: 2, c: 32519},
    {f: 2, c: 32523}, 32527, {f: 2, c: 32529}, 32535, 32537, 32540, 32539,
    32543, {f: 7, c: 32545}, {f: 4, c: 32554}, {f: 5, c: 32559}, 32565,
    [12083, 24186], 30079, [12078, 24027], 30014, 37013, 29582, 29585, 29614,
    29602, 29599, 29647, 29634, 29649, 29623, 29619, 29632, 29641, 29640,
    29669, 29657, 39036, 29706, 29673, 29671, 29662, 29626, 29682, 29711,
    29738, 29787, 29734, 29733, 29736, 29744, 29742, 29740, 29723, 29722,
    29761, 29788, 29783, 29781, 29785, 29815, 29805, 29822, 29852, 29838,
    {f: 2, c: 29824}, 29831, 29835, 29854, {f: 2, c: 29864}, 29840, 29863,
    29906, 29882, {f: 3, c: 38890}, 26444, 26451, 26462, 26440, 26473, 26533,
    26503, 26474, 26483, 26520, 26535, 26485, 26536, 26526, 26541, 26507,
    26487, 26492, 26608, 26633, 26584, 26634, 26601, 26544, 26636, 26585,
    26549, 26586, 26547, 26589, 26624, 26563, 26552, 26594, 26638, 26561,
    26621, {f: 2, c: 26674}, {f: 2, c: 26720}, 26702, 26722, 26692, 26724,
    26755, 26653, 26709, 26726, 26689, 26727, 26688, 26686, 26698, 26697,
    26665, 26805, 26767, 26740, 26743, 26771, 26731, 26818, 26990, 26876,
    {f: 2, c: 26911}, 26873, 26916, 26864, 26891, 26881, 26967, 26851, 26896,
    26993, 26937, 26976, 26946, 26973, 27012, 26987, 27008, 27032, 27000,
    26932, 27084, {f: 2, c: 27015}, 27086, 27017, 26982, 26979, 27001, 27035,
    27047, 27067, 27051, 27053, 27092, 27057, 27073, 27082, 27103, 27029,
    27104, 27021, 27135, 27183, 27117, {f: 2, c: 27159}, 27237, 27122, 27204,
    27198, 27296, 27216, 27227, 27189, 27278, 27257, 27197, 27176, 27224,
    27260, 27281, 27280, 27305, 27287, 27307, 29495, 29522, {f: 2, c: 27521},
    27527, 27524, {f: 2, c: 27538}, 27533, {f: 2, c: 27546}, 27553, 27562,
    36715, 36717, {f: 3, c: 36721}, {f: 2, c: 36725}, 36728, 36727,
    {f: 2, c: 36729}, 36732, 36734, {f: 2, c: 36737}, 36740, 36743, 36747,
    {f: 3, c: 36749}, 36760, 36762, 36558, 25099, 25111, 25115, 25119, 25122,
    25121, 25125, 25124, 25132, 33255, 29935, 29940, 29951, 29967, 29969,
    29971, [12097, 25908], {f: 3, c: 26094}, 26122, 26137, 26482, 26115, 26133,
    26112, 28805, 26359, 26141, 26164, 26161, 26166, 26165, 32774, 26207,
    26196, 26177, 26191, 26198, 26209, 26199, 26231, 26244, 26252, 26279,
    26269, 26302, {f: 2, c: 26331}, 26342, 26345, {f: 2, c: 36146}, 36150,
    36155, 36157, 36160, {f: 2, c: 36165}, {f: 2, c: 36168}, 36167, 36173,
    36181, 36185, 35271, {f: 3, c: 35274}, {f: 4, c: 35278}, 29294, 29343,
    29277, 29286, 29295, {f: 2, c: 29310}, 29316, 29323, 29325, 29327, 29330,
    25352, 25394, 25520, 25663, 25816, 32772, 27626, 27635, 27645, 27637,
    27641, 27653, 27655, 27654, 27661, 27669, {f: 3, c: 27672}, 27681, 27689,
    27684, 27690, 27698, 25909, 25941, 25963, 29261, 29266, 29270, 29232,
    34402, 21014, 32927, 32924, 32915, 32956, 26378, 32957, 32945, 32939,
    32941, 32948, 32951, {f: 4, c: 32999}, 32987, 32962, 32964, 32985, 32973,
    32983, 26384, 32989, 33003, 33009, 33012, 33005, {f: 2, c: 33037}, 33010,
    33020, 26389, 33042, 35930, 33078, 33054, 33068, 33048, 33074, 33096,
    33100, 33107, 33140, {f: 2, c: 33113}, 33137, 33120, 33129,
    {f: 2, c: 33148}, 33133, 33127, 22605, 23221, 33160, 33154, 33169, 28373,
    33187, 33194, 33228, 26406, 33226, 33211, 33217, 33190, 27428, 27447,
    27449, 27459, 27462, 27481, {f: 3, c: 39121}, 39125, {f: 2, c: 39129},
    [12110, 27571], 24384, 27586, 35315, 26000, 40785, 26003, 26044, 26054,
    26052, 26051, 26060, 26062, 26066, 26070, 28800, 28828, 28822, 28829,
    28859, 28864, 28855, 28843, 28849, 28904, 28874, 28944, 28947, 28950,
    28975, 28977, 29043, 29020, 29032, 28997, 29042, 29002, 29048, 29050,
    29080, 29107, 29109, 29096, 29088, 29152, 29140, 29159, 29177, 29213,
    29224, 28780, 28952, 29030, 29113, 25150, 25149, 25155, {f: 2, c: 25160},
    31035, 31040, 31046, 31049, {f: 2, c: 31067}, 31059, 31066, 31074, 31063,
    31072, 31087, 31079, 31098, 31109, 31114, 31130, 31143, 31155, 24529,
    24528, 24636, 24669, 24666, 24679, 24641, 24665, 24675, 24747, 24838,
    24845, 24925, 25001, 24989, 25035, 25041, 25094, 32896, [12160, 32895],
    27795, 27894, 28156, 30710, 30712, 30720, 30729, {f: 2, c: 30743}, 30737,
    26027, 30765, {f: 2, c: 30748}, {f: 3, c: 30777}, 30751, 30780, 30757,
    30764, 30755, 30761, 30798, 30829, {f: 2, c: 30806}, 30758, 30800, 30791,
    30796, 30826, 30875, 30867, 30874, 30855, 30876, 30881, 30883, 30898,
    30905, 30885, 30932, 30937, 30921, 30956, 30962, 30981, 30964, 30995,
    31012, 31006, 31028, 40859, [12235, 40697], {f: 2, c: 40699}, 30449, 30468,
    30477, 30457, {f: 2, c: 30471}, 30490, 30498, 30489, 30509, 30502, 30517,
    30520, {f: 2, c: 30544}, 30535, 30531, 30554, 30568, 30562, 30565, 30591,
    30605, 30589, 30592, 30604, 30609, {f: 2, c: 30623}, 30640, 30645, 30653,
    30010, 30016, 30030, 30027, 30024, 30043, 30066, 30073, 30083, 32600,
    32609, 32607, 35400, 32616, 32628, 32625, 32633, 32641, 32638, 30413,
    30437, 34866, {f: 3, c: 38021}, 38027, 38026, {f: 2, c: 38028},
    {f: 2, c: 38031}, 38036, 38039, 38037, {f: 3, c: 38042}, {f: 2, c: 38051},
    38059, 38058, 38061, 38060, {f: 2, c: 38063}, 38066, 38068,
    {f: 5, c: 38070}, {f: 2, c: 38076}, 38079, 38084, {f: 7, c: 38088},
    {f: 3, c: 38096}, {f: 3, c: 38101}, 38105, 38104, 38107, {f: 3, c: 38110},
    38114, {f: 2, c: 38116}, {f: 2, c: 38119}, 38122, 38121, 38123,
    {f: 2, c: 38126}, {f: 3, c: 38131}, 38135, 38137, {f: 2, c: 38140}, 38143,
    38147, 38146, {f: 2, c: 38150}, {f: 2, c: 38153}, {f: 3, c: 38157},
    {f: 5, c: 38162}, 38168, 38171, {f: 3, c: 38173}, 38178, {f: 2, c: 38186},
    38185, 38188, {f: 2, c: 38193}, 38196, {f: 3, c: 38198}, 38204,
    {f: 2, c: 38206}, 38210, 38197, {f: 3, c: 38212}, 38217, 38220,
    {f: 2, c: 38222}, {f: 3, c: 38226}, {f: 4, c: 38230}, 38235,
    {f: 2, c: 38238}, 38237, {f: 2, c: 38241}, {f: 9, c: 38244}, 38255,
    {f: 3, c: 38257}, 38202, 30695, 30700, 38601, 31189, 31213, 31203, 31211,
    31238, 23879, 31235, 31234, 31262, 31252, 31289, 31287, 31313, 40655,
    39333, 31344, 30344, 30350, 30355, 30361, 30372, 29918, 29920, 29996,
    40480, 40482, {f: 5, c: 40488}, 40498, 40497, 40502, 40504, 40503,
    {f: 2, c: 40505}, 40510, {f: 2, c: 40513}, 40516, {f: 4, c: 40518},
    {f: 2, c: 40523}, 40526, 40529, 40533, 40535, {f: 3, c: 40538}, 40542,
    40547, {f: 7, c: 40550}, 40561, 40557, 40563, [12135, 30098], 30100, 30102,
    30112, 30109, 30124, 30115, {f: 2, c: 30131}, 30136, 30148, 30129, 30128,
    30147, 30146, 30166, 30157, 30179, 30184, 30182, 30180, 30187, 30183,
    30211, 30193, 30204, 30207, 30224, 30208, 30213, 30220, 30231, 30218,
    30245, 30232, 30229, 30233, 30235, 30268, 30242, 30240, 30272, 30253,
    30256, 30271, 30261, 30275, 30270, 30259, 30285, 30302, 30292, 30300,
    30294, 30315, 30319, 32714, 31462, {f: 2, c: 31352}, 31360, 31366, 31368,
    31381, 31398, 31392, 31404, 31400, 31405, 31411, 34916, 34921, 34930,
    34941, 34943, 34946, 34978, 35014, 34999, 35004, 35017, 35042, 35022,
    35043, 35045, 35057, 35098, 35068, 35048, 35070, 35056, 35105, 35097,
    35091, 35099, 35082, 35124, 35115, 35126, 35137, 35174, 35195,
    [12134, 30091], 32997, 30386, 30388, 30684, [12158, 32786], 32788, 32790,
    32796, 32800, 32802, {f: 3, c: 32805}, 32809, 32808, 32817, 32779, 32821,
    32835, 32838, 32845, 32850, 32873, 32881, 35203, 39032, 39040, 39043,
    39049, {f: 2, c: 39052}, 39055, 39060, {f: 2, c: 39066}, {f: 2, c: 39070},
    {f: 2, c: 39073}, {f: 2, c: 39077}, [12172, 34381], 34388, 34412, 34414,
    34431, 34426, 34428, 34427, 34472, 34445, 34443, 34476, 34461, 34471,
    34467, 34474, 34451, 34473, 34486, 34500, 34485, 34510, 34480, 34490,
    34481, 34479, 34505, 34511, 34484, 34537, {f: 2, c: 34545}, 34541, 34547,
    34512, 34579, 34526, 34548, 34527, 34520, 34513, 34563, 34567, 34552,
    34568, 34570, 34573, 34569, 34595, 34619, 34590, 34597, 34606, 34586,
    34622, 34632, 34612, 34609, 34601, 34615, 34623, 34690, 34594,
    {f: 2, c: 34685}, 34683, 34656, 34672, 34636, 34670, 34699, 34643, 34659,
    34684, 34660, 34649, 34661, 34707, 34735, 34728, 34770, 34758, 34696,
    34693, 34733, 34711, 34691, 34731, 34789, 34732, 34741, 34739, 34763,
    34771, 34749, 34769, 34752, 34762, 34779, 34794, 34784, 34798, 34838,
    34835, 34814, 34826, 34843, 34849, 34873, 34876, [12152, 32566], 32578,
    {f: 2, c: 32580}, 33296, 31482, 31485, 31496, {f: 2, c: 31491}, 31509,
    31498, 31531, 31503, 31559, 31544, 31530, 31513, 31534, 31537, 31520,
    31525, 31524, 31539, 31550, 31518, 31576, 31578, 31557, 31605, 31564,
    31581, 31584, 31598, 31611, 31586, 31602, 31601, 31632, {f: 2, c: 31654},
    31672, 31660, 31645, 31656, 31621, 31658, 31644, 31650, 31659, 31668,
    31697, 31681, 31692, 31709, 31706, {f: 2, c: 31717}, 31722, 31756, 31742,
    31740, 31759, 31766, 31755, 31775, 31786, 31782, 31800, 31809, 31808,
    33278, {f: 2, c: 33281}, 33284, 33260, 34884, {f: 3, c: 33313}, 33325,
    33327, 33320, 33323, 33336, 33339, {f: 2, c: 33331}, 33342, 33348, 33353,
    33355, 33359, 33370, 33375, 33384, 34942, 34949, 34952, 35032, 35039,
    35166, 32669, 32671, 32679, {f: 2, c: 32687}, 32690, 31868, 25929, 31889,
    31901, 31900, 31902, 31906, 31922, {f: 2, c: 31932}, 31937, 31943,
    {f: 2, c: 31948}, 31944, 31941, 31959, 31976, [12169, 33390], 26280, 32703,
    32718, 32725, 32741, 32737, 32742, 32745, 32750, 32755, [12151, 31992],
    32119, 32166, 32174, 32327, 32411, 40632, 40628, 36211, 36228, 36244,
    36241, 36273, 36199, 36205, 35911, 35913, 37194, 37200, {f: 2, c: 37198},
    37220, 37218, 37217, 37232, 37225, 37231, {f: 2, c: 37245}, 37234, 37236,
    37241, 37260, 37253, 37264, 37261, 37265, {f: 2, c: 37282}, 37290,
    {f: 3, c: 37293}, 37301, 37300, 37306, [12183, 35925], 40574, 36280, 36331,
    36357, 36441, 36457, 36277, 36287, 36284, 36282, 36292, {f: 2, c: 36310},
    36314, 36318, {f: 2, c: 36302}, 36315, 36294, 36332, {f: 2, c: 36343},
    36323, 36345, 36347, 36324, 36361, 36349, 36372, 36381, 36383, 36396,
    36398, 36387, 36399, 36410, 36416, 36409, 36405, 36413, 36401, 36425,
    {f: 2, c: 36417}, {f: 2, c: 36433}, 36426, 36464, 36470, 36476, 36463,
    36468, 36485, 36495, 36500, 36496, 36508, 36510, [12184, 35960], 35970,
    35978, 35973, 35992, 35988, 26011, 35286, 35294, 35290, 35292, 35301,
    35307, 35311, 35390, 35622, 38739, 38633, 38643, 38639, 38662, 38657,
    38664, 38671, 38670, 38698, 38701, 38704, 38718, 40832, 40835,
    {f: 6, c: 40837}, 40844, 40702, 40715, 40717, [12203, 38585],
    {f: 2, c: 38588}, 38606, 38610, 30655, 38624, 37518, 37550, 37576, 37694,
    37738, 37834, 37775, 37950, 37995, 40063, 40066, {f: 4, c: 40069}, 31267,
    40075, 40078, {f: 3, c: 40080}, {f: 2, c: 40084}, {f: 2, c: 40090},
    {f: 6, c: 40094}, {f: 5, c: 40101}, 40107, {f: 2, c: 40109},
    {f: 8, c: 40112}, {f: 4, c: 40122}, {f: 4, c: 40132}, {f: 7, c: 40138},
    {f: 3, c: 40147}, {f: 3, c: 40151}, {f: 2, c: 40156}, 40159, 40162, 38780,
    38789, {f: 2, c: 38801}, 38804, 38831, 38827, 38819, 38834, 38836, 39601,
    39600, 39607, 40536, 39606, 39610, 39612, 39617, 39616, 39621, 39618,
    {f: 2, c: 39627}, 39633, 39749, 39747, 39751, 39753, 39752, 39757, 39761,
    39144, 39181, 39214, 39253, 39252, [12221, 39647], 39649, 39654, 39663,
    39659, 39675, 39661, 39673, 39688, 39695, 39699, 39711, 39715,
    {f: 2, c: 40637}, 32315, 40578, {f: 2, c: 40583}, 40587, 40594, 37846,
    40605, 40607, {f: 3, c: 40667}, 40672, 40671, 40674, 40681, 40679, 40677,
    40682, 40687, 40738, 40748, 40751, 40761, 40759, {f: 2, c: 40765}, 40772,
    12295, {s: 13}, 30362, 34297, 31001, 24859, 39599, 35158, 22761, 32631,
    25850, 25943, 38930, 36774, 32070, 24171, 32129, 37770, 35607, 39165,
    23542, 22577, 39825, 36649, [12185, 35997], 37575, 29437, 20633, 24970,
    32179, 31558, 30050, 25987, 24163, 38281, 37002, 32232, 36022, 35722,
    36783, 36782, 27161, 40009, 30303, 28693, 28657, 36051, 25839, 39173,
    25765, 37474, 37457, 39361, 35036, 36001, 21443, 34870, 27544, 24922,
    24920, 29158, 33980, 33369, 20489, 28356, 21408, 20596, 28204, 23652,
    35435, 25881, 25723, 34796, 39262, 35730, 32399, 37855, 29987, 38369,
    39019, 22580, 22039, [12199, 38263], 20767, 33144, 24288, 26274, 37396,
    [12190, 36554], 24505, 22645, 38515, 35183, 31281, 25074, 35488, 39425,
    36978, 39347, [12242, 40786], 29118, 34909, 34802, 23541, 30087, 36490,
    31820, 32162, 37276, 37604, 38619, 30990, 20786, 35320, 34389, 20659,
    30241, 38358, 21109, 37656, 32020, 32189, 36781, 35422, 36060, 32880,
    24478, 21474, 36517, 31428, 37679, 36948, 24118, 36024, 25812, 21934,
    37170, 25763, 33213, 24986, 35477, 24392, 30070, 25803, 40680, 34153,
    27284, 25623, 23798, 31153, 23566, 29128, 37159, 25973, 28364, 36958,
    32224, 39003, 40670, 22666, 38651, 28593, 37347, 35519, 35548, 37336,
    38914, 37664, 35330, 26481, 21205, 26847, 20941, [12222, 39717], 29346,
    29544, 35712, 36077, 37709, 37723, 26039, 32222, 38538, 23565, 22136,
    38931, 37389, 22890, 22702, 40285, 38989, 35355, 24801, 39187, 20818,
    29246, 39180, 36019, 30332, 32624, 38309, 31020, 37353, 29033, 31684,
    36009, 39151, 35370, 32033, [12214, 39131], 35513, 24290, 36027, 32027,
    22707, 22894, 24996, 31966, 35920, 26963, 37586, [12213, 39080], 30219,
    39342, 32299, 35575, 40179, 33178, 36667, 25771, 36628, 36070, 24489,
    36000, 35331, 23142, 32283, 35442, 37411, 33995, 24185, 36245, 36123,
    23713, 21083, 37628, 32177, 23831, 37804, 25841, 40255, 38307, 37499,
    20491, 32102, 40852, 38799, 36002, 37390, 28317, 27083, 36092, 34865,
    39015, 21102, 38364, 35264, 39208, 24931, 36011, 24291, 35215, 27512,
    [12244, 40860], 38312, 36556, 35437, 27331, 36020, 21130, 36645, 37707,
    22283, 36942, 39405, 38867, 28450, 34399, 38305, 40372, 36032, 36703,
    40251, 32005, 22778, 35703, 28396, 22057, 33775, 30059, 21123, 35441,
    25079, 22750, 27489, 29872, 36996, 32233, 35594, 25582, 36637, 36036,
    31330, 26371, 29172, 21295, 35569, 35496, 32362, 33911, 28222, 29554,
    36008, 31117, 25802, 27231, 31309, 39249, 35663, 40388, 32318, 32221,
    26997, 36655, 32026, 25824, 24190, 34186, 21137, 28639, 35336, 35352,
    38555, 32380, 32000, 22846, 33698, 38960, 36040, 37440, 20729, 39381,
    27570, 30435, 22533, 31627, 38291, 33393, 32216, 32365, 27298, 40572,
    25536, 25791, 31777, 20745, 34214, 27323, 37970, 36368, 36068,
    [12178, 35211], 37749, 33382, 21133, 39198, 28472, 28666, 28567, 23559,
    28479, 34083, 27123, 22892, 35611, 37292, 33184, 28550, 39509, 23308,
    25898, 37496, 30703, 20709, 39171, 32371, 32094, 36686, 36611, 38542,
    31680, 28500, 32080, 35489, 32202, 37670, 20677, 35641, 36914, 29180,
    30433, 21185, 33686, 39912, 39514, 32147, 38968, 37857, 24465, 30169,
    31478, 31998, 33290, 39378, 33289, 25818, 37624, 25084, 21127, 40273,
    32121, 35258, 35363, 32118, 37406, 36557, 39423, 38283, 20977, 38982,
    27579, 35506, 22718, 25031, 25715, 24235, 35122, 35463, 22602, 20744,
    23532, 31014, 26336, 34407, 24011, 31418, 39243, 28528, 25844, 38346,
    34847, 33240, 33802, 20358, 36084, 34253, 27396, 25876, 31811, 38348,
    34349, 28734, 35733, 25900, 35261, 25078, 32412, 29211, 28651, 25736,
    21214, 28551, 27138, 37939, 22744, 39006, 31852, 38626, 28757, 35023,
    39881, 31150, 40599, 21426, 21237, 31019, 27511, 28701, 38584, 20486,
    32879, 34030, 36899, 37934, 24976, 28451, 31806, 25986, 33225, 37832,
    25088, 29001, 32244, 31975, 20841, 36635, 35538, 30274, 36988, 37904,
    29557, 33256, 37168, 40023, 36035, 40801, 37428, 38728, 23994, 38936,
    39230, 21129, [12243, 40845], 32894, 22184, 31840, 22751, 25871, 38580,
    27155, 23105, 25695, 31757, 34310, 30439, 39025, 24300, 29200, 25796,
    28407, 34396, 39791, 36034, 37682, 38520, 39522, 37569, 23650, 32311,
    24942, 28670, 32209, 24018, 25891, 23423, 28772, 20098, 25476, 36650,
    20523, 20374, 28138, 32184, 35542, 34367, 32645, 37007, 38012, 31854,
    39486, 39409, 32097, 23229, 29802, 30908, 34718, [12218, 39340], 39393,
    21966, 36023, [12230, 40613], 36067, 36993, 30622, 39237, 34875, 28415,
    35646, 37672, 37466, 36031, 37762, [12200, 38272], 24758, 20497, 37683,
    22818, 35598, 24396, 35219, 32191, 32236, 24287, 28357, 25003, 38313,
    40180, 37528, 35628, 35584, 30045, 37385, 32013, 38627, 25747, 33126,
    24817, 39719, 39186, 25836, 33193, 25862, 37312, [12227, 40165], 32886,
    22169, 38007, 37811, 27320, 29552, 23527, 25840, 28632, 37397, 32016,
    33215, 28611, 36786, 30247, 35582, 27472, 40407, 27590, 22036, 28442,
    30436, 40848, 36064, 22132, 40300, 39449, 39108, 38971, 36007, 34315,
    24977, 35413, 28497, 38935, 25778, 37610, 20693, 27192, 35676, 33229,
    [12241, 40778], 39438, 35912, 21843, 27683, 35350, 29309, 37370, 37467,
    36983, 31805, 35609, 37666, 37463, 28154, 35700, 22649, 27085, 21958,
    22715, 34196, 25654, 37740, 27211, 21932, 20689, 32761, 31429, 31434,
    27453, 35242, 23522, 36629, 27691, 20670, 38915, 35531, 24950, 29898,
    31406, 36264, 21312, 36544, 39493, 40818, 39028, 27402, 21240, 40306,
    30906, 35731, 39250, 25854, 32350, 29105, 38860, 35469, 32009, 27054,
    32104, 36575, 37613, 38287, 28516, 28753, 34217, 39955, 36093, 20632,
    21930, 39479, 25475, 28544, 27578, 32023, 31721, 26348, 38275, 38493,
    36109, 32341, 20663, 36062, 29138, 32057, 36050, 25448, 25885, 25086,
    35373, 32051, 23529, 23352, 33102, 28402, 32882, 32361, 21213, 32854,
    24107, 29509, 28629, 35433, 26178, 34645, 23526, 35672, 39387, 21218,
    36969, 37323, 39166, 35222, 35430, 22781, 29560, 27166, 36664, 26360,
    36118, 23660, 34899, 27193, 31466, 25976, 24101, 38617, 35504, 38918,
    35500, 30889, 29197, 32114, 39164, 39686, 32883, 24939, 38924, 35359,
    35494, 25851, 34311, 35380, 32901, 38614, 38568, 32143, 27506, 23403,
    25613, 32302, 29795, 37782, 29562, 25787, 33274, 24907, 25892, 36010,
    30321, 28760, 22727, 35674, 35527, 22022, 28271, 29145, 28644, 32295,
    35342, 39472, 35588, 37563, 38988, 39636, 26781, 36028, 37941, 24307,
    32893, 28916, 37509, 32113, 38957, 22294, 22615, 22296, 38973, 40213,
    39345, 39389, 27234, 31402, 35178, 24398, 28771, 38929, 33836, 32178,
    [12209, 38859], 36949, 22285, 29234, 28656, 32173, 33894, 20553, 20702,
    32239, 35586, 34907, 32862, 32011, 31337, 21839, 25790, 34680, 28198,
    31401, 21978, 37794, 28879, 35491, 28961, 34154, 22626, 38695, 21209,
    35492, 37675, 29351, 35186, 32722, 37521, 25138, 32048, 34662, 36676,
    23805, 20448, 29433, 22151, 37697, 39854, 32406, 36066, 37532, 38289,
    39023, 38570, 29694, 29563, 32291, 39201, 25010, 32171, 38002, 37129,
    35443, 38911, 38917, 34157, 22210, 37559, 26313, 22063, 21332, 25406,
    33029, 35559, 23531, 28681, 35613, 37573, 37313, 33288, 37561, 32137,
    38920, 35377, 32210, 32396, 36562, 25080, 36984, 30316, 32098, 23416,
    21211, 35426, 23563, 39348, 35347, 35338, 36956, 22739, 40201, 40232,
    21854, 20126, 35357, 38329, 40573, 22196, 38996, 38331, 33399, 21421,
    30831, 35578, 39511, 40230, 26954, 25562, 30221, 38525, 30306, 39178,
    27171, 22575, 35617, 34277, 29242, [12212, 38913], 26989, 33865, 37291,
    37541, 38948, 36986, 20736, 34811, 34269, 20740, 25014, 32681, 35427,
    35696, 35516, 35695, 32377, 34093, 38512, 37504, 39154, 38577, 27387,
    23344, 40441, 25033, 32403, 29801, 34722, 29151, 29074, 34821, 36111,
    31310, 21938, 25793, 20653, 30320, 36404, 20778, 24962, 37109, 37438,
    29494, 35480, 36671, 39192, [12226, 39770], 28417, 33287, 23996, 35486,
    39729, 29508, 35709, 38928, 39341, 40219, 28149, 36677, 22290, 21729,
    22291, 32227, 36960, 39000, 32004, 36493, 38000, 38322, 38642, 37142,
    38549, 36939, 34292, 37270, 26248, 38620, 36617, 25890, 26283, 36106,
    36124, 33247, 38015, 26839, 31432, 36012, 25799, 21063, 28580, 36042,
    36104, 36555, 37720, 38296, 35408, 40779, 20661, 27656, 30430, 26028,
    36670, 23940, 26855, 25136, 32187, 24373, 28466, 24115, 36076, 33081,
    36249, 34756, 36685, 37754, 36889, 35998, 37341, 20597, 35386, 37806,
    38499, 24128, 30309, 37165, 35657, 32340, 32887, 22519, 34937, 32025,
    25711, 25842, 24159, 36074, 28399, 37912, 32066, 31278, 33131, 34886,
    35589, 36600, 30394, 26205, 39519, 35576, 35461, 29165, 30682, 22225,
    36015, 37956, 31689, 39376, 23560, 30938, 36681, 36090, 27137, 33674,
    35037, 22941, 22767, 29376, 37648, 36101, 22684, 32180, 35524, 28310,
    28609, 36039, 28460, 32156, 32317, 32305, 37138, 35419, 32068, 38013,
    21959, 21401, 21428, 38760, 36107, 21293, 21297, 36094, 21060, 21132,
    21108, 20660, 20480, 20630, 20757, 20738, 20756, 20796, 20791, 20712,
    20674, 20795, 20752, 20794, 20681, 31988, 40652, 22213, 40172, 35131,
    33248, 35329, 35344, 35340, 35349, 35635, 35406, 35365, 35393, 35382,
    35398, 35412, 35416, 35410, 35462, 35460, 35455, 35440, 35452, 35445,
    35436, 35438, 35533, 35554, 35425, 35482, 35493, {f: 2, c: 35473}, 35535,
    35537, 35529, 35547, 35543, 35522, 35510, 35574, 35563, 35604, 35585,
    35556, 35565, 35580, 35571, 35558, 35566, 35550, 35624, 35740, 35606,
    35610, 35600, 35627, 35629, 35670, 35673, 35662, 35742, 35691, 35734,
    38488, 37178, 37140, 37172, 37087, 37174, 37126, 37192, 33467, 21233,
    24048, 22538, 22745, 22754, 22752, 22746, 22497, 22607, 22550, 22610,
    22557, 22628, 34188, 34131, 34294, 33703, 33799, 34031, 33511, 34338,
    34086, 22603, 29026, 34136, 34045, 34126, 34184, 34234, 29334, 28366,
    34113, 34254, 34130, 33984, 33874, 33892, 33940, 33845, 34207, 34133,
    40367, 33939, 32264, 34118, 34146, 34078, 39488, 34362, 37795, 34167,
    34334, 34298, 34308, 34282, 34330, 22889, 23607, 25451, 25718, 25759,
    25681, 25692, 25779, 25860, 25878, 25847, 25852, 25883, 22064, 22072,
    22216, 22182, 21764, 21692, 22144, 22109, 22112, 22069, 22006, 22118,
    22130, 22156, 22117, 22044, 22062, 21993, 22038, 22208, 22029, 22195,
    22209, 22127, 36705, 22198, 22165, 22279, 24131, 24172, 24152, 24151,
    23943, 23796, 23888, 23852, 23975, 23968, 23959, 23821, 23992, 23937,
    24020, 24480, 29559, 29505, 29546, 29499, 29547, 29568, 29564, 39136,
    39219, 39145, 39228, {f: 2, c: 39146}, 39149, 39156, 39177, 39185, 39195,
    39223, 39231, 39235, {f: 3, c: 39240}, 39244, 39266, 24289, 36065, 25082,
    25006, 24938, 24894, 24757, 24884, 25036, 24927, 25064, 24827, 24887,
    24818, 24947, 24860, 24978, 38274, 38278, 38344, 38286, 38292, 38284,
    38373, 38317, 38315, 39726, 38316, 38334, 38326, 39721, 38335, 38333,
    38332, 38339, 38347, 38356, 38352, 38357, 38366, 28739, 28505, 28711,
    28696, 28668, 28039, 28025, 28254, 28590, 28687, 28408, 28527, 28150,
    28543, 28678, 28576, 28683, 28775, 28740, 28677, 28535, 28704, 28703,
    28722, 28712, 28765, 39467, 36999, 36885, 37008, 23656, 24371, 23285,
    23255, 23296, 23149, 23304, 23372, 23207, 23291, 23307, 23329, 23338,
    23321, 39380, 39391, 39385, 39478, 39515, 39377, 39384, 39501, 39498,
    39394, 39530, 39439, 39437, 39429, 39490, 39469, 39446, 39489, 39470,
    39480, {f: 2, c: 39491}, 39503, 39525, 39524, 31993, 32006, 32002,
    {f: 2, c: 32007}, 32394, 32028, 32021, 32019, 32058, 32050, 32049, 32272,
    32060, 32064, 32063, 32093, 32078, 32115, 32134, 32131, 32136, 32190,
    32186, 32203, 32212, 32196, 32158, 32172, 32185, 32163, 32176, 32199,
    32217, 32215, 32249, 32242, 32354, 32230, 32246, 32241, 32267, 32225,
    32265, 32285, 32287, 32286, 32301, 32266, 32273, 32381, 32313, 32309,
    32306, 32326, 32325, 32392, 32346, 32338, 32366, 32382, 32368, 32367,
    32408, 29859, 29771, 29903, 38922, 29885, 29759, 29833, 29862, 29908,
    29914, 38873, 38878, 38876, 27050, 27370, 26776, 26838, 27141, 26783,
    27355, 27379, 27368, 27359, 27273, 26895, 27208, 26984, 27071, 27194,
    27292, 27410, 27422, 27357, 27111, 27407, 27414, 27372, 27354, 27384,
    27315, 27367, 27299, 27347, 27358, 27556, 27550, 27566, 27563, 27567,
    36564, 36571, 36594, 36603, 36708, 36601, 36604, 36587, 36580, 36706,
    36602, 36606, 36618, 36615, 36613, 36626, 36646, {f: 2, c: 36638}, 36636,
    36659, 36678, 36692, 25108, 25127, 29964, 26311, 26308, 26249, 26326,
    36033, 36016, 36026, 36029, 36100, 36018, 36037, 36112, 36049, 36058,
    36053, 36075, 36071, 36091, 35224, 35244, 35233, 35263, 35238, 35247,
    35250, 35255, 27647, 27660, 27692, 29272, 26407, 33110, 33242, 33051,
    33214, 33121, 33231, 27487, {f: 2, c: 39086}, 39094, 39100, 39110, 39112,
    36674, 40783, 26005, 29036, 29010, 29079, 29121, 29148, 29182, 31152,
    31118, 31146, 25055, 24932, 25059, 25095, 28585, 30959, 30893, 30824,
    30904, 31018, 31025, 30820, 30973, 30951, 30947, 40853, 30616, 30558,
    30652, 32646, 32648, {f: 3, c: 37330}, 37337, 37335, 37333, 37367, 37351,
    37348, 37702, 37365, 37369, 37384, 37414, 37445, 37393, 37392, 37377,
    37415, 37380, 37413, 37376, 37434, 37478, 37431, 37427, 37461, 37437,
    37432, 37470, {f: 2, c: 37484}, 37439, 37984, 37424, 37449, 37448, 37453,
    37422, 37433, 37944, 37548, 37536, 37498, 37546, 37614, 37583, 37891,
    37603, 37946, 37553, 37542, 37799, 37526, 37580, 37545, 37877, 37523,
    37503, 37801, 37530, 37658, 37547, 37507, 37899, 37544, 37539, 37906,
    37688, 37617, 37847, 37605, 37616, 37615, 37608, 37564, 37597, 37622,
    {f: 2, c: 37926}, 37571, 37599, 37606, 37650, 37638, 37737, 37659, 37696,
    37633, 37653, 37678, 37699, {f: 2, c: 37639}, 37663, 37657, 37733, 37703,
    37750, 37716, 37732, 37802, 37744, 37764, 37860, 37848, 37928, 37767,
    37836, 37784, 37816, 37823, 37798, 37808, 37813, 37964, 37858,
    {f: 2, c: 37852}, 37837, 37854, 37827, 37831, 37841, 37908, 37917, 37879,
    37989, 37907, 37997, 37920, 38009, 37881, 37913, 37962, 37938, 37951,
    37972, 37987, 37758, 31329, 40169, 40182, 40199, 40198, 40227, 40327,
    40469, 40221, 40223, 40421, 40239, 40409, 40240, 40258, 40478, 40275,
    40477, 40288, 40274, 40435, 40284, 40289, 40339, 40298, 40303, 40329,
    40344, 40346, 40384, 40357, 40361, 40386, 40380, 40474, 40403, 40410,
    40431, 40422, 40434, 40440, 40460, 40442, 40475, 30308, 30296, 30311,
    30210, {f: 2, c: 30278}, 30281, 30238, 30267, {f: 2, c: 30317}, 30313,
    30322, 31431, 31414, 35168, 35123, 35165, 35143, 35128, 35172, 30392,
    32814, 32812, 32889, 32885, 38919, {f: 2, c: 38926}, 38945, 38940, 28481,
    38950, 38967, 38990, 38995, 39027, 39010, 39001, 39013, 39020, 39024,
    34787, 34822, 34566, 34851, 34806, 34554, 34799, 34692, 34832, 34760,
    34833, 34747, 34766, 32588, 31716, 31591, 31849, 31731, 31744, 31691,
    31836, 31774, 31787, 31779, 31850, 31839, 33380, 33387, 35018, 32677,
    31986, 31990, 31965, 32310, 40617, 36274, 37317, 37315, 40570, 36489,
    36428, 36498, 36474, 36437, 36506, 36491, 36499, 36497, 36513, 36451,
    36522, 36518, 35316, 35318, 38746, 38722, 38717, 38724, 40788, 40799,
    40793, 40800, 40796, 40806, 40812, 40810, 40823, [12236, 40701], 40703,
    40713, 35726, 38014, 37864, 39799, 39796, 39809, 39811, 39822, 40056,
    31308, 39826, 40031, 39824, 39853, 39834, 39850, 39838, 40045, 39851,
    39837, 40024, 39873, 40058, 39985, 39993, 39971, 39991, 39872, 39882,
    39879, 39933, 39894, {f: 2, c: 39914}, 39905, 39908, 39911, 39901, 39906,
    39920, 39899, 39924, 39892, 40029, 39944, 39952, 39949, 39954, 39945,
    39935, 39968, 39986, 39981, 39976, 39973, 39977, 39987, 39998, 40008,
    39995, 39989, 40005, 40022, 40020, 40018, 40039, 38851, 38845, 38857,
    40379, 39631, 39638, 39637, 39768, 39758, 39255, 39260, 39714, 40695,
    40690, 35180, 38342, 37686, 24390, 34068, 32404, 40803, 22137, 40725,
    22081, 39662, 35079, 31296, 39091, 38308, 39693, 36852, 24409, 31339,
    39138, 20642, 34193, 20760, 25458, 21067, 30543, 32397, 26310, 30637,
    [12228, 40565], 22217, 40692, 28635, 25054, 30663, 28720, 40629, 34890,
    38370, 38854, 31844, 32308, 38822, 40623, 22220, 39089, 27311, 32590,
    31984, 20418, 32363, 40569, 22190, 39706, 33903, 31142, 31858, 39634,
    38587, 32251, 35069, 30787, {f: 10, c: 8560}, {f: 2, c: 714}, 729, 8211,
    8213, 8229, 8245, 8453, 8457, {f: 4, c: 8598}, 8725, 8735, 8739, 8786,
    {f: 2, c: 8806}, 8895, {f: 36, c: 9552}, {f: 15, c: 9601}, {f: 3, c: 9619},
    {f: 2, c: 9660}, {f: 4, c: 9698}, 9737, 8853, 12306, {f: 2, c: 12317},
    {f: 9, c: 12321}, 12963, {f: 2, c: 13198}, {f: 3, c: 13212}, 13217, 13252,
    13262, {f: 2, c: 13265}, 13269, 65072, 65506, 65508, 8481, 12849, 8208,
    12540, {f: 2, c: 12443}, {f: 2, c: 12541}, 12294, {f: 2, c: 12445},
    {f: 10, c: 65097}, {f: 4, c: 65108}, {f: 14, c: 65113}, {f: 4, c: 65128},
    12350, {f: 12, c: 12272}, 19970, {f: 3, c: 19972}, 19983, 19986, 19991,
    {f: 3, c: 19999}, 20003, 20006, 20009, {f: 2, c: 20014}, 20017, 20019,
    20021, 20023, 20028, {f: 3, c: 20032}, 20036, 20038, 20042, 20049, 20053,
    20055, {f: 2, c: 20058}, {f: 4, c: 20066}, {f: 2, c: 20071},
    {f: 6, c: 20074}, 20082, {f: 10, c: 20084}, {f: 3, c: 20095},
    {f: 2, c: 20099}, [12037, 20101], 20103, 20106, 20112, {f: 2, c: 20118},
    20121, {f: 2, c: 20124}, 20131, 20138, {f: 3, c: 20143}, 20148,
    {f: 4, c: 20150}, {f: 3, c: 20156}, 20168, 20172, {f: 2, c: 20175}, 20178,
    {f: 3, c: 20186}, 20192, 20194, {f: 2, c: 20198}, 20201, {f: 3, c: 20205},
    20209, 20212, {f: 3, c: 20216}, 20220, 20222, 20224, {f: 7, c: 20226},
    {f: 2, c: 20235}, {f: 5, c: 20242}, {f: 2, c: 20252}, 20257, 20259,
    {f: 2, c: 20264}, {f: 3, c: 20268}, 20273, 20275, 20277, 20279, 20281,
    20283, {f: 5, c: 20286}, {f: 2, c: 20292}, {f: 6, c: 20295}, 20306, 20308,
    20310, {f: 2, c: 20321}, 20326, 20328, {f: 2, c: 20330}, {f: 2, c: 20333},
    {f: 2, c: 20337}, 20341, {f: 4, c: 20343}, 20349, {f: 3, c: 20352}, 20357,
    20359, 20362, 20364, 20366, 20368, {f: 2, c: 20370}, 20373,
    {f: 3, c: 20376}, 20380, {f: 2, c: 20382}, {f: 2, c: 20385}, 20388, 20395,
    20397, {f: 5, c: 20400}, {f: 9, c: 20406}, {f: 2, c: 20416},
    {f: 4, c: 20422}, {f: 3, c: 20427}, {f: 5, c: 20434}, 20441, 20443, 20450,
    {f: 2, c: 20452}, 20455, {f: 2, c: 20459}, 20464, 20466, {f: 4, c: 20468},
    20473, {f: 3, c: 20475}, 20479, {f: 5, c: 20481}, {f: 2, c: 20487}, 20490,
    20494, 20496, 20499, {f: 3, c: 20501}, 20507, {f: 2, c: 20509}, 20512,
    {f: 3, c: 20514}, 20519, {f: 11, c: 20527}, 20539, 20541, {f: 4, c: 20543},
    {f: 3, c: 20548}, {f: 2, c: 20554}, 20557, {f: 5, c: 20560},
    {f: 4, c: 20566}, 20571, {f: 8, c: 20573}, {f: 6, c: 20582},
    {f: 7, c: 20589}, {f: 3, c: 20600}, {f: 2, c: 20604}, {f: 4, c: 20609},
    {f: 2, c: 20614}, {f: 4, c: 20617}, {f: 8, c: 20622}, 20631,
    {f: 8, c: 20634}, 20644, 20646, {f: 2, c: 20650}, {f: 4, c: 20654}, 20662,
    {f: 2, c: 20664}, {f: 2, c: 20668}, {f: 3, c: 20671}, {f: 2, c: 20675},
    {f: 3, c: 20678}, {f: 5, c: 20682}, 20688, {f: 3, c: 20690},
    {f: 3, c: 20695}, {f: 3, c: 20699}, {f: 6, c: 20703}, {f: 3, c: 20713},
    {f: 4, c: 20719}, 20724, {f: 3, c: 20726}, 20730, {f: 4, c: 20732}, 20737,
    20739, 20741, 20746, {f: 4, c: 20748}, 20753, 20755, {f: 2, c: 20758},
    {f: 6, c: 20761}, 20768, {f: 8, c: 20770}, {f: 7, c: 20779},
    {f: 4, c: 20787}, {f: 2, c: 20792}, {f: 2, c: 20797}, 20802, 20807, 20810,
    20812, {f: 3, c: 20814}, 20819, {f: 3, c: 20823}, 20827, {f: 5, c: 20829},
    {f: 2, c: 20835}, {f: 2, c: 20838}, 20842, 20847, 20850, 20858,
    {f: 2, c: 20862}, {f: 2, c: 20867}, {f: 2, c: 20870}, {f: 2, c: 20874},
    {f: 4, c: 20878}, {f: 2, c: 20883}, 20888, 20890, {f: 3, c: 20893}, 20897,
    20899, {f: 5, c: 20902}, {f: 2, c: 20909}, 20916, {f: 3, c: 20920},
    {f: 2, c: 20926}, {f: 3, c: 20929}, 20933, 20936, 20938, 20942, 20944,
    {f: 9, c: 20946}, 20956, {f: 2, c: 20958}, {f: 2, c: 20962},
    {f: 6, c: 20965}, 20972, 20974, 20978, 20980, 20983, 20990,
    {f: 2, c: 20996}, 21001, {f: 2, c: 21003}, {f: 2, c: 21007},
    {f: 3, c: 21011}, 21020, {f: 2, c: 21022}, {f: 3, c: 21025},
    {f: 3, c: 21029}, 21034, 21036, 21039, {f: 2, c: 21041}, {f: 2, c: 21044},
    21052, 21054, {f: 2, c: 21061}, {f: 2, c: 21064}, {f: 2, c: 21070},
    {f: 2, c: 21074}, 21077, {f: 4, c: 21079}, 21085, {f: 2, c: 21087},
    {f: 3, c: 21090}, 21094, 21096, {f: 3, c: 21099}, {f: 2, c: 21104}, 21107,
    {f: 7, c: 21110}, 21118, 21120, {f: 3, c: 21124}, 21131, {f: 2, c: 21134},
    21138, {f: 7, c: 21140}, 21148, {f: 4, c: 21156}, {f: 3, c: 21166},
    {f: 10, c: 21172}, 21184, 21186, {f: 3, c: 21188}, 21192, 21194,
    {f: 4, c: 21196}, 21201, {f: 2, c: 21203}, 21207, 21210, 21212,
    {f: 2, c: 21216}, 21219, {f: 11, c: 21221}, {f: 3, c: 21234},
    {f: 2, c: 21238}, {f: 3, c: 21243}, {f: 4, c: 21249}, 21255,
    {f: 4, c: 21257}, 21262, {f: 4, c: 21265}, 21272, {f: 2, c: 21275},
    {f: 2, c: 21278}, 21282, {f: 2, c: 21284}, {f: 3, c: 21287},
    {f: 2, c: 21291}, 21296, {f: 6, c: 21298}, [12054, 21304],
    {f: 2, c: 21308}, 21314, 21316, 21318, {f: 3, c: 21323}, 21328,
    {f: 2, c: 21336}, 21339, 21341, 21349, 21352, 21354, {f: 2, c: 21356},
    21362, 21366, 21369, {f: 4, c: 21371}, {f: 2, c: 21376}, 21379,
    {f: 2, c: 21383}, 21386, {f: 7, c: 21390}, {f: 2, c: 21398},
    {f: 2, c: 21403}, 21406, 21409, 21412, 21415, {f: 3, c: 21418},
    {f: 3, c: 21423}, 21427, 21429, {f: 4, c: 21431}, {f: 3, c: 21436}, 21440,
    {f: 4, c: 21444}, {f: 3, c: 21454}, {f: 2, c: 21458}, 21461, 21466,
    {f: 3, c: 21468}, 21473, 21479, 21492, 21498, {f: 3, c: 21502}, 21506,
    21509, 21511, 21515, 21524, {f: 3, c: 21528}, 21532, 21538,
    {f: 2, c: 21540}, 21546, 21552, 21555, {f: 2, c: 21558}, 21562, 21565,
    21567, {f: 2, c: 21569}, {f: 2, c: 21572}, 21575, 21577, {f: 4, c: 21580},
    21585, 21594, {f: 5, c: 21597}, 21603, 21605, 21607, {f: 8, c: 21609},
    21620, {f: 2, c: 21625}, {f: 2, c: 21630}, 21633, 21635, 21637,
    {f: 4, c: 21639}, 21645, 21649, 21651, {f: 2, c: 21655}, 21660,
    {f: 5, c: 21662}, 21669, 21678, 21680, 21682, {f: 3, c: 21685},
    {f: 2, c: 21689}, 21694, 21699, 21701, {f: 2, c: 21706}, 21718, 21720,
    21723, 21728, {f: 3, c: 21730}, {f: 2, c: 21739}, {f: 3, c: 21743},
    {f: 6, c: 21748}, 21755, 21758, 21760, {f: 2, c: 21762}, 21765, 21768,
    {f: 5, c: 21770}, {f: 2, c: 21778}, {f: 6, c: 21781}, {f: 4, c: 21788},
    21793, {f: 2, c: 21797}, {f: 2, c: 21800}, 21803, 21805, 21810,
    {f: 3, c: 21812}, {f: 4, c: 21816}, 21821, 21824, 21826, 21829,
    {f: 2, c: 21831}, {f: 4, c: 21835}, {f: 2, c: 21841}, 21844,
    {f: 5, c: 21847}, 21853, {f: 2, c: 21855}, {f: 2, c: 21858},
    {f: 2, c: 21864}, 21867, {f: 6, c: 21871}, {f: 2, c: 21881}, 21885, 21887,
    {f: 2, c: 21893}, {f: 3, c: 21900}, 21904, {f: 2, c: 21906},
    {f: 3, c: 21909}, {f: 2, c: 21914}, 21918, {f: 7, c: 21920},
    {f: 2, c: 21928}, 21931, 21933, {f: 2, c: 21935}, 21940, 21942, 21944,
    21946, 21948, {f: 5, c: 21951}, 21960, {f: 2, c: 21962}, {f: 2, c: 21967},
    21973, {f: 3, c: 21975}, 21979, 21982, 21984, 21986, 21991,
    {f: 2, c: 21997}, {f: 2, c: 22000}, 22004, {f: 5, c: 22008}, 22015,
    {f: 4, c: 22018}, 22023, {f: 2, c: 22026}, {f: 4, c: 22032}, 22037,
    {f: 2, c: 22041}, 22045, {f: 3, c: 22048}, {f: 2, c: 22053}, 22056,
    {f: 2, c: 22058}, 22067, 22071, 22074, {f: 3, c: 22076}, 22080,
    {f: 10, c: 22082}, {f: 5, c: 22095}, {f: 2, c: 22101}, {f: 2, c: 22106},
    {f: 2, c: 22110}, 22113, 22115, 22119, {f: 2, c: 22125}, 22128, 22131,
    22133, 22135, 22138, {f: 3, c: 22141}, {f: 4, c: 22145}, {f: 4, c: 22152},
    22157, {f: 3, c: 22160}, 22164, {f: 3, c: 22166}, {f: 9, c: 22170},
    {f: 2, c: 22180}, 22183, {f: 5, c: 22185}, {f: 3, c: 22192}, 22197,
    {f: 4, c: 22200}, {f: 3, c: 22205}, {f: 2, c: 22211}, {f: 2, c: 22214},
    22219, {f: 4, c: 22221}, {f: 2, c: 22226}, {f: 2, c: 22229},
    {f: 2, c: 22232}, 22236, 22243, {f: 6, c: 22245}, 22252, {f: 2, c: 22254},
    {f: 2, c: 22258}, {f: 3, c: 22262}, {f: 2, c: 22267}, {f: 3, c: 22272},
    22277, 22284, {f: 4, c: 22286}, {f: 2, c: 22292}, 22295, {f: 3, c: 22297},
    {f: 2, c: 22301}, {f: 3, c: 22304}, {f: 4, c: 22308}, 22315,
    {f: 2, c: 22321}, {f: 5, c: 22324}, {f: 2, c: 22332}, 22335, 22337,
    {f: 4, c: 22339}, {f: 2, c: 22344}, 22347, {f: 5, c: 22354},
    {f: 2, c: 22360}, {f: 2, c: 22370}, 22373, 22375, 22380, 22382,
    {f: 3, c: 22384}, {f: 2, c: 22388}, {f: 3, c: 22392}, {f: 5, c: 22397},
    {f: 4, c: 22407}, {f: 5, c: 22413}, {f: 7, c: 22420}, {f: 4, c: 22428},
    22437, 22440, 22442, 22444, {f: 3, c: 22447}, 22451, {f: 3, c: 22453},
    {f: 9, c: 22457}, {f: 7, c: 22468}, {f: 2, c: 22476}, {f: 2, c: 22480},
    22483, {f: 2, c: 22486}, {f: 2, c: 22491}, 22494, {f: 2, c: 22498},
    {f: 8, c: 22501}, 22510, {f: 4, c: 22512}, {f: 2, c: 22517},
    {f: 2, c: 22523}, {f: 2, c: 22526}, 22529, {f: 2, c: 22531},
    {f: 2, c: 22536}, 22540, {f: 3, c: 22542}, {f: 3, c: 22546},
    {f: 2, c: 22551}, {f: 3, c: 22554}, 22559, {f: 2, c: 22562},
    {f: 5, c: 22565}, {f: 4, c: 22571}, {f: 2, c: 22578}, {f: 14, c: 22582},
    {f: 5, c: 22597}, 22606, 22608, 22611, {f: 2, c: 22613}, {f: 5, c: 22617},
    {f: 3, c: 22623}, 22627, {f: 5, c: 22630}, {f: 8, c: 22637},
    {f: 3, c: 22646}, {f: 4, c: 22650}, 22655, 22658, 22660, {f: 3, c: 22662},
    {f: 7, c: 22667}, {f: 5, c: 22676}, 22683, 22685, {f: 8, c: 22688},
    {f: 4, c: 22698}, {f: 4, c: 22703}, {f: 7, c: 22708}, 22717,
    {f: 2, c: 22719}, {f: 3, c: 22722}, 22726, {f: 9, c: 22728}, 22738, 22740,
    {f: 2, c: 22742}, {f: 3, c: 22747}, 22753, 22755, {f: 4, c: 22757}, 22762,
    22765, {f: 2, c: 22769}, {f: 2, c: 22772}, {f: 2, c: 22775},
    {f: 2, c: 22779}, {f: 4, c: 22782}, 22787, {f: 2, c: 22789},
    {f: 2, c: 22792}, [12066, 22794], {f: 2, c: 22795}, 22798,
    {f: 4, c: 22800}, {f: 2, c: 22807}, 22811, {f: 2, c: 22813},
    {f: 2, c: 22816}, 22819, 22822, 22824, 22828, 22832, {f: 2, c: 22834},
    {f: 2, c: 22837}, 22843, 22845, {f: 2, c: 22847}, 22851, {f: 2, c: 22853},
    22858, {f: 2, c: 22860}, 22864, {f: 2, c: 22866}, 22873, {f: 5, c: 22875},
    22881, {f: 2, c: 22883}, {f: 3, c: 22886}, 22891, 22893, {f: 4, c: 22895},
    22901, 22903, {f: 3, c: 22906}, {f: 3, c: 22910}, 22917, 22921,
    {f: 2, c: 22923}, {f: 4, c: 22926}, {f: 2, c: 22932}, 22936,
    {f: 3, c: 22938}, {f: 4, c: 22943}, {f: 2, c: 22950}, {f: 2, c: 22956},
    {f: 2, c: 22960}, {f: 6, c: 22963}, 22970, {f: 2, c: 22972},
    {f: 7, c: 22975}, {f: 3, c: 22983}, {f: 4, c: 22988}, {f: 2, c: 22997},
    23001, 23003, {f: 5, c: 23006}, 23012, {f: 2, c: 23014}, {f: 3, c: 23017},
    {f: 12, c: 23021}, 23034, {f: 3, c: 23036}, 23040, 23042, {f: 2, c: 23050},
    {f: 4, c: 23053}, 23058, {f: 4, c: 23060}, {f: 3, c: 23065},
    {f: 2, c: 23069}, {f: 2, c: 23073}, 23076, {f: 3, c: 23078},
    {f: 7, c: 23082}, 23091, 23093, {f: 5, c: 23095}, {f: 3, c: 23101},
    {f: 4, c: 23106}, {f: 2, c: 23111}, {f: 10, c: 23115}, {f: 4, c: 23126},
    {f: 7, c: 23131}, {f: 3, c: 23139}, {f: 2, c: 23144}, {f: 2, c: 23147},
    {f: 6, c: 23150}, {f: 2, c: 23160}, {f: 4, c: 23163}, {f: 18, c: 23168},
    {f: 7, c: 23187}, {f: 11, c: 23196}, {f: 2, c: 23208}, {f: 7, c: 23211},
    23220, {f: 2, c: 23222}, {f: 4, c: 23225}, {f: 2, c: 23231},
    {f: 6, c: 23235}, {f: 2, c: 23242}, {f: 5, c: 23245}, 23251, 23253,
    {f: 3, c: 23257}, {f: 3, c: 23261}, 23266, {f: 2, c: 23268},
    {f: 2, c: 23271}, 23274, {f: 5, c: 23276}, {f: 3, c: 23282},
    {f: 5, c: 23286}, {f: 4, c: 23292}, {f: 7, c: 23297}, 23306,
    {f: 9, c: 23309}, 23320, {f: 7, c: 23322}, {f: 8, c: 23330},
    {f: 5, c: 23339}, 23345, 23347, {f: 2, c: 23349}, {f: 7, c: 23353},
    {f: 11, c: 23361}, {f: 3, c: 23373}, 23378, 23382, 23390, {f: 2, c: 23392},
    {f: 2, c: 23399}, {f: 3, c: 23405}, 23410, 23412, {f: 2, c: 23414}, 23417,
    {f: 2, c: 23419}, 23422, 23426, 23430, 23434, {f: 2, c: 23437},
    {f: 3, c: 23440}, 23444, 23446, 23455, {f: 3, c: 23463}, {f: 4, c: 23468},
    {f: 2, c: 23473}, 23479, {f: 3, c: 23482}, {f: 2, c: 23488}, 23491,
    {f: 4, c: 23496}, {f: 3, c: 23501}, 23505, {f: 9, c: 23508}, 23520, 23523,
    23530, 23533, 23535, {f: 4, c: 23537}, 23543, {f: 2, c: 23549}, 23552,
    {f: 2, c: 23554}, 23557, 23564, 23568, {f: 2, c: 23570}, 23575, 23577,
    23579, {f: 4, c: 23582}, 23587, 23590, {f: 4, c: 23592}, {f: 4, c: 23597},
    {f: 2, c: 23602}, {f: 2, c: 23605}, {f: 2, c: 23619}, {f: 2, c: 23622},
    {f: 2, c: 23628}, {f: 3, c: 23634}, {f: 3, c: 23638}, {f: 4, c: 23642},
    23647, 23655, {f: 3, c: 23657}, 23661, 23664, {f: 7, c: 23666},
    {f: 4, c: 23675}, 23680, {f: 5, c: 23683}, {f: 3, c: 23689},
    {f: 2, c: 23694}, {f: 2, c: 23698}, 23701, {f: 4, c: 23709},
    {f: 5, c: 23716}, 23722, {f: 3, c: 23726}, 23730, 23732, 23734,
    {f: 4, c: 23737}, 23742, 23744, {f: 2, c: 23746}, {f: 6, c: 23749},
    {f: 6, c: 23756}, {f: 6, c: 23763}, {f: 7, c: 23770}, {f: 2, c: 23778},
    23783, 23785, {f: 2, c: 23787}, {f: 2, c: 23790}, {f: 3, c: 23793}, 23797,
    {f: 4, c: 23799}, 23804, {f: 4, c: 23806}, {f: 2, c: 23812},
    {f: 5, c: 23816}, {f: 5, c: 23823}, 23829, {f: 3, c: 23832},
    {f: 2, c: 23836}, {f: 5, c: 23839}, 23845, 23848, {f: 2, c: 23850},
    {f: 5, c: 23855}, {f: 8, c: 23861}, {f: 8, c: 23871}, {f: 2, c: 23880},
    {f: 3, c: 23885}, {f: 7, c: 23889}, {f: 2, c: 23897}, 23900,
    {f: 11, c: 23902}, 23914, {f: 2, c: 23917}, {f: 4, c: 23920},
    {f: 12, c: 23925}, 23939, {f: 2, c: 23941}, {f: 15, c: 23944}, 23960,
    {f: 3, c: 23962}, {f: 2, c: 23966}, {f: 6, c: 23969}, {f: 15, c: 23976},
    23993, 23995, {f: 8, c: 23997}, {f: 5, c: 24006}, 24012, {f: 4, c: 24014},
    24019, {f: 6, c: 24021}, 24028, {f: 2, c: 24031}, {f: 2, c: 24035}, 24042,
    {f: 2, c: 24044}, {f: 2, c: 24053}, {f: 5, c: 24056}, {f: 2, c: 24063},
    24068, 24071, {f: 3, c: 24073}, {f: 2, c: 24077}, {f: 2, c: 24082}, 24087,
    {f: 7, c: 24094}, {f: 3, c: 24104}, 24108, {f: 2, c: 24111}, 24114,
    {f: 2, c: 24116}, {f: 2, c: 24121}, {f: 2, c: 24126}, 24129,
    {f: 6, c: 24134}, {f: 7, c: 24141}, 24150, {f: 2, c: 24153},
    {f: 2, c: 24156}, 24160, {f: 7, c: 24164}, {f: 5, c: 24173}, 24181, 24183,
    {f: 3, c: 24193}, 24197, {f: 2, c: 24200}, {f: 3, c: 24204}, 24210, 24216,
    24219, 24221, {f: 4, c: 24225}, {f: 3, c: 24232}, 24236, {f: 5, c: 24238},
    24244, {f: 4, c: 24250}, {f: 10, c: 24255}, {f: 6, c: 24267},
    {f: 2, c: 24276}, {f: 4, c: 24279}, {f: 3, c: 24284}, {f: 4, c: 24292},
    24297, 24299, {f: 6, c: 24301}, 24309, {f: 2, c: 24312}, {f: 3, c: 24315},
    {f: 3, c: 24325}, 24329, {f: 3, c: 24332}, 24336, 24338, 24340, 24342,
    {f: 2, c: 24345}, {f: 3, c: 24348}, {f: 4, c: 24353}, 24360,
    {f: 2, c: 24363}, 24366, 24368, 24370, 24372, {f: 3, c: 24374}, 24379,
    {f: 3, c: 24381}, {f: 5, c: 24385}, 24391, {f: 3, c: 24393}, 24397, 24399,
    24401, 24404, {f: 3, c: 24410}, {f: 3, c: 24414}, 24419, 24421,
    {f: 2, c: 24423}, 24427, {f: 2, c: 24430}, 24434, {f: 3, c: 24436}, 24440,
    24442, {f: 3, c: 24445}, 24451, 24454, {f: 3, c: 24461}, {f: 2, c: 24467},
    24470, {f: 2, c: 24474}, 24477, 24479, {f: 6, c: 24482}, {f: 2, c: 24491},
    {f: 6, c: 24495}, 24502, 24504, {f: 2, c: 24506}, {f: 5, c: 24510},
    {f: 2, c: 24519}, {f: 2, c: 24522}, 24526, {f: 3, c: 24531},
    {f: 3, c: 24538}, {f: 2, c: 24542}, {f: 2, c: 24546}, {f: 2, c: 24549},
    {f: 2, c: 24552}, 24556, {f: 2, c: 24559}, {f: 3, c: 24562},
    {f: 2, c: 24566}, {f: 2, c: 24569}, 24572, {f: 3, c: 24583},
    {f: 2, c: 24587}, {f: 2, c: 24592}, 24595, {f: 2, c: 24599}, 24602,
    {f: 2, c: 24606}, {f: 3, c: 24610}, {f: 3, c: 24620}, {f: 5, c: 24624},
    {f: 5, c: 24630}, {f: 2, c: 24637}, 24640, {f: 7, c: 24644}, 24652,
    {f: 2, c: 24654}, 24657, {f: 2, c: 24659}, {f: 3, c: 24662},
    {f: 2, c: 24667}, {f: 4, c: 24670}, {f: 2, c: 24677}, 24686,
    {f: 2, c: 24689}, {f: 2, c: 24692}, 24695, 24702, {f: 3, c: 24704},
    {f: 4, c: 24709}, {f: 2, c: 24714}, {f: 4, c: 24718}, 24723, 24725,
    {f: 3, c: 24727}, 24732, 24734, {f: 2, c: 24737}, {f: 2, c: 24740}, 24743,
    {f: 2, c: 24745}, 24750, 24752, 24755, 24759, {f: 2, c: 24761},
    {f: 8, c: 24765}, {f: 3, c: 24775}, {f: 5, c: 24780}, {f: 3, c: 24786},
    {f: 2, c: 24790}, 24793, 24795, 24798, {f: 4, c: 24802}, 24810, 24821,
    {f: 2, c: 24823}, {f: 4, c: 24828}, {f: 4, c: 24834}, 24839,
    {f: 3, c: 24842}, {f: 5, c: 24848}, {f: 4, c: 24854}, {f: 2, c: 24861},
    {f: 2, c: 24865}, 24869, {f: 3, c: 24872}, {f: 8, c: 24876},
    {f: 2, c: 24885}, {f: 6, c: 24888}, {f: 8, c: 24896}, 24905, 24909,
    {f: 2, c: 24911}, {f: 3, c: 24914}, {f: 2, c: 24918}, 24921,
    {f: 2, c: 24923}, 24926, {f: 2, c: 24928}, {f: 2, c: 24933}, 24937,
    {f: 2, c: 24940}, 24943, {f: 2, c: 24945}, 24948, {f: 10, c: 24952},
    {f: 7, c: 24963}, {f: 2, c: 24972}, 24975, 24979, {f: 5, c: 24981},
    {f: 2, c: 24987}, {f: 6, c: 24990}, {f: 2, c: 24997}, 25002, 25005,
    {f: 3, c: 25007}, {f: 3, c: 25011}, {f: 6, c: 25016}, {f: 3, c: 25023},
    {f: 4, c: 25027}, {f: 4, c: 25037}, 25043, {f: 9, c: 25045},
    {f: 3, c: 25056}, {f: 2, c: 25060}, 25063, {f: 9, c: 25065},
    {f: 2, c: 25075}, 25081, 25083, 25085, {f: 5, c: 25089}, 25097, 25107,
    25113, {f: 3, c: 25116}, 25120, 25123, 25126, {f: 2, c: 25128}, 25131,
    25133, 25135, 25137, 25141, [12094, 25142], {f: 5, c: 25144}, 25154,
    {f: 3, c: 25156}, 25162, {f: 2, c: 25167}, {f: 3, c: 25173},
    {f: 2, c: 25177}, {f: 7, c: 25180}, {f: 2, c: 25188}, 25192,
    {f: 2, c: 25201}, {f: 2, c: 25204}, {f: 2, c: 25207}, {f: 2, c: 25210},
    25213, {f: 3, c: 25217}, {f: 4, c: 25221}, {f: 6, c: 25227}, 25236, 25241,
    {f: 3, c: 25244}, 25251, {f: 2, c: 25254}, {f: 2, c: 25257},
    {f: 4, c: 25261}, {f: 3, c: 25266}, {f: 3, c: 25270}, 25274, 25278,
    {f: 2, c: 25280}, 25283, 25291, 25295, 25297, 25301, {f: 2, c: 25309},
    {f: 2, c: 25312}, 25316, {f: 2, c: 25322}, 25328, 25330, 25333,
    {f: 4, c: 25336}, 25344, {f: 4, c: 25347}, {f: 4, c: 25354},
    {f: 2, c: 25359}, {f: 4, c: 25362}, {f: 3, c: 25367}, 25372,
    {f: 2, c: 25382}, 25385, {f: 3, c: 25388}, {f: 2, c: 25392},
    {f: 6, c: 25395}, {f: 2, c: 25403}, {f: 3, c: 25407}, 25412,
    {f: 2, c: 25415}, 25418, {f: 4, c: 25425}, {f: 8, c: 25430}, 25440,
    {f: 3, c: 25444}, 25450, 25452, {f: 2, c: 25455}, {f: 3, c: 25459},
    {f: 2, c: 25464}, {f: 4, c: 25468}, 25473, {f: 2, c: 25477}, 25483, 25485,
    25489, {f: 3, c: 25491}, 25495, {f: 7, c: 25497}, 25505, 25508, 25510,
    25515, 25519, {f: 2, c: 25521}, {f: 2, c: 25525}, 25529, 25531, 25533,
    25535, {f: 3, c: 25537}, 25541, {f: 2, c: 25543}, {f: 3, c: 25546}, 25553,
    {f: 3, c: 25555}, {f: 3, c: 25559}, {f: 3, c: 25563}, 25567, 25570,
    {f: 5, c: 25572}, {f: 2, c: 25579}, {f: 3, c: 25583}, 25587, 25589, 25591,
    {f: 4, c: 25593}, 25598, {f: 2, c: 25603}, {f: 5, c: 25606}, 25614,
    {f: 2, c: 25617}, {f: 2, c: 25621}, {f: 3, c: 25624}, 25629, 25631,
    {f: 4, c: 25634}, {f: 3, c: 25639}, 25643, {f: 6, c: 25646}, 25653,
    {f: 3, c: 25655}, {f: 2, c: 25659}, 25662, 25664, {f: 2, c: 25666}, 25673,
    {f: 6, c: 25675}, 25683, {f: 3, c: 25685}, {f: 3, c: 25689}, 25693,
    {f: 7, c: 25696}, 25704, {f: 3, c: 25706}, 25710, {f: 3, c: 25712},
    {f: 2, c: 25716}, 25719, {f: 6, c: 25724}, 25731, 25734, {f: 8, c: 25737},
    25748, {f: 2, c: 25751}, {f: 4, c: 25754}, {f: 3, c: 25760},
    {f: 3, c: 25766}, 25770, 25775, 25777, 25780, 25782, 25785, 25789, 25795,
    25798, {f: 2, c: 25800}, 25804, 25807, 25809, 25811, {f: 2, c: 25813},
    25817, {f: 3, c: 25819}, 25823, 25825, 25827, 25829, {f: 5, c: 25831},
    {f: 2, c: 25837}, 25843, {f: 2, c: 25845}, {f: 2, c: 25848}, 25853, 25855,
    {f: 3, c: 25857}, 25861, {f: 2, c: 25863}, {f: 5, c: 25866},
    {f: 2, c: 25872}, 25875, 25877, 25879, 25882, 25884, {f: 4, c: 25886},
    {f: 4, c: 25894}, 25901, {f: 4, c: 25904}, 25911, 25914, {f: 2, c: 25916},
    {f: 5, c: 25920}, {f: 2, c: 25926}, {f: 2, c: 25930}, {f: 2, c: 25933},
    25936, {f: 3, c: 25938}, 25944, 25946, 25948, {f: 3, c: 25951},
    {f: 2, c: 25956}, {f: 4, c: 25959}, {f: 3, c: 25965}, 25969, 25971, 25974,
    {f: 9, c: 25977}, {f: 3, c: 25988}, {f: 3, c: 25992}, {f: 3, c: 25997},
    26002, 26004, 26006, 26008, 26010, {f: 2, c: 26013}, 26016,
    {f: 2, c: 26018}, 26022, 26024, 26026, 26030, {f: 6, c: 26033}, 26040,
    {f: 2, c: 26042}, {f: 3, c: 26046}, 26050, {f: 4, c: 26055}, 26061,
    {f: 2, c: 26064}, {f: 3, c: 26067}, {f: 8, c: 26072}, 26081,
    {f: 2, c: 26083}, {f: 2, c: 26090}, {f: 4, c: 26098}, {f: 2, c: 26104},
    {f: 5, c: 26107}, 26113, {f: 2, c: 26116}, {f: 3, c: 26119}, 26123, 26125,
    {f: 3, c: 26128}, {f: 3, c: 26134}, {f: 3, c: 26138}, 26142,
    {f: 4, c: 26145}, 26150, {f: 4, c: 26153}, 26158, 26160, {f: 2, c: 26162},
    {f: 5, c: 26167}, 26173, {f: 2, c: 26175}, {f: 7, c: 26180},
    {f: 2, c: 26189}, {f: 2, c: 26192}, {f: 2, c: 26200}, {f: 2, c: 26203},
    26206, 26208, {f: 2, c: 26210}, 26213, 26215, {f: 5, c: 26217},
    {f: 3, c: 26225}, 26229, {f: 2, c: 26232}, {f: 3, c: 26235},
    {f: 3, c: 26239}, 26243, {f: 2, c: 26245}, {f: 2, c: 26250},
    {f: 4, c: 26253}, {f: 4, c: 26258}, {f: 5, c: 26264}, {f: 4, c: 26270},
    {f: 4, c: 26275}, {f: 2, c: 26281}, {f: 2, c: 26284}, {f: 5, c: 26287},
    {f: 4, c: 26293}, {f: 4, c: 26298}, {f: 5, c: 26303}, 26309, 26312,
    {f: 12, c: 26314}, {f: 2, c: 26327}, 26330, {f: 2, c: 26334},
    {f: 5, c: 26337}, {f: 2, c: 26343}, {f: 2, c: 26346}, {f: 3, c: 26349},
    26353, {f: 2, c: 26357}, {f: 2, c: 26362}, 26365, {f: 2, c: 26369},
    {f: 4, c: 26372}, 26380, {f: 2, c: 26382}, {f: 3, c: 26385}, 26390,
    {f: 3, c: 26392}, 26396, 26398, {f: 6, c: 26400}, 26409, 26414, 26416,
    {f: 2, c: 26418}, {f: 4, c: 26422}, {f: 2, c: 26427}, {f: 2, c: 26430},
    26433, {f: 2, c: 26436}, 26439, {f: 2, c: 26442}, 26445, 26450,
    {f: 2, c: 26452}, {f: 5, c: 26455}, 26461, {f: 3, c: 26466},
    {f: 2, c: 26470}, {f: 2, c: 26475}, 26478, 26484, 26486, {f: 4, c: 26488},
    26493, 26496, {f: 2, c: 26498}, {f: 2, c: 26501}, 26504, 26506,
    {f: 4, c: 26508}, {f: 4, c: 26513}, 26518, 26521, 26523, {f: 3, c: 26527},
    26532, 26534, 26537, 26540, 26542, {f: 2, c: 26545}, 26548,
    {f: 8, c: 26553}, 26562, {f: 10, c: 26565}, {f: 3, c: 26581}, 26587, 26591,
    26593, {f: 2, c: 26595}, {f: 3, c: 26598}, {f: 2, c: 26602},
    {f: 2, c: 26605}, 26610, {f: 8, c: 26613}, 26622, {f: 4, c: 26625}, 26630,
    26637, 26640, 26642, {f: 2, c: 26644}, {f: 5, c: 26648}, {f: 3, c: 26654},
    {f: 7, c: 26658}, {f: 7, c: 26667}, {f: 3, c: 26676}, {f: 2, c: 26682},
    26687, 26695, 26699, 26701, 26703, 26706, {f: 10, c: 26710}, 26730,
    {f: 8, c: 26732}, 26741, {f: 9, c: 26744}, 26754, 26756, {f: 8, c: 26759},
    {f: 3, c: 26768}, {f: 3, c: 26772}, {f: 4, c: 26777}, 26782,
    {f: 2, c: 26784}, {f: 3, c: 26787}, {f: 4, c: 26793}, 26798,
    {f: 2, c: 26801}, 26804, {f: 10, c: 26806}, 26817, {f: 6, c: 26819}, 26826,
    26828, {f: 4, c: 26830}, {f: 2, c: 26835}, 26841, {f: 4, c: 26843},
    {f: 2, c: 26849}, {f: 3, c: 26852}, {f: 6, c: 26856}, 26863,
    {f: 3, c: 26866}, {f: 3, c: 26870}, 26875, {f: 4, c: 26877},
    {f: 3, c: 26882}, {f: 5, c: 26886}, 26892, 26897, {f: 12, c: 26899},
    {f: 3, c: 26913}, {f: 8, c: 26917}, {f: 2, c: 26926}, {f: 3, c: 26929},
    {f: 4, c: 26933}, {f: 3, c: 26938}, 26942, {f: 2, c: 26944},
    {f: 7, c: 26947}, {f: 8, c: 26955}, {f: 2, c: 26965}, {f: 2, c: 26968},
    {f: 2, c: 26971}, 26975, {f: 2, c: 26977}, {f: 2, c: 26980}, 26983,
    {f: 2, c: 26985}, 26988, {f: 2, c: 26991}, {f: 3, c: 26994}, 26998,
    {f: 2, c: 27002}, {f: 3, c: 27005}, 27009, 27011, 27013, {f: 3, c: 27018},
    {f: 6, c: 27022}, {f: 2, c: 27030}, {f: 2, c: 27033}, {f: 10, c: 27037},
    27049, 27052, {f: 2, c: 27055}, {f: 2, c: 27058}, {f: 2, c: 27061},
    {f: 3, c: 27064}, {f: 3, c: 27068}, 27072, {f: 8, c: 27074}, 27087,
    {f: 3, c: 27089}, {f: 6, c: 27093}, {f: 3, c: 27100}, {f: 6, c: 27105},
    {f: 5, c: 27112}, {f: 4, c: 27118}, {f: 9, c: 27124}, 27134, 27136,
    {f: 2, c: 27139}, {f: 4, c: 27142}, {f: 8, c: 27147}, {f: 3, c: 27156},
    {f: 4, c: 27162}, 27168, 27170, {f: 4, c: 27172}, 27177, {f: 4, c: 27179},
    27184, {f: 3, c: 27186}, {f: 2, c: 27190}, {f: 2, c: 27195},
    {f: 5, c: 27199}, {f: 2, c: 27205}, {f: 2, c: 27209}, {f: 4, c: 27212},
    {f: 7, c: 27217}, 27226, {f: 3, c: 27228}, 27232, {f: 2, c: 27235},
    {f: 11, c: 27238}, {f: 7, c: 27250}, {f: 2, c: 27258}, {f: 3, c: 27261},
    {f: 3, c: 27265}, {f: 4, c: 27269}, {f: 4, c: 27274}, 27279,
    {f: 2, c: 27282}, {f: 2, c: 27285}, {f: 4, c: 27288}, {f: 3, c: 27293},
    27297, {f: 5, c: 27300}, 27306, {f: 2, c: 27309}, {f: 3, c: 27312},
    {f: 4, c: 27316}, {f: 2, c: 27321}, {f: 7, c: 27324}, {f: 15, c: 27332},
    {f: 6, c: 27348}, 27356, {f: 7, c: 27360}, 27369, 27371, {f: 6, c: 27373},
    {f: 4, c: 27380}, {f: 2, c: 27385}, {f: 8, c: 27388}, {f: 5, c: 27397},
    {f: 4, c: 27403}, {f: 2, c: 27408}, {f: 3, c: 27411}, {f: 7, c: 27415},
    27423, {f: 2, c: 27429}, {f: 10, c: 27432}, {f: 4, c: 27443}, 27448,
    {f: 2, c: 27451}, {f: 4, c: 27455}, {f: 2, c: 27460}, 27464,
    {f: 2, c: 27466}, {f: 3, c: 27469}, {f: 8, c: 27473}, {f: 5, c: 27482},
    27488, {f: 2, c: 27496}, {f: 7, c: 27499}, {f: 4, c: 27507}, 27514,
    {f: 4, c: 27517}, 27525, 27528, 27532, {f: 4, c: 27534}, {f: 2, c: 27540},
    27543, 27545, {f: 2, c: 27548}, {f: 2, c: 27551}, {f: 2, c: 27554},
    {f: 5, c: 27557}, {f: 2, c: 27564}, {f: 2, c: 27568}, 27574,
    {f: 2, c: 27576}, {f: 3, c: 27580}, 27584, {f: 2, c: 27587},
    {f: 4, c: 27591}, 27596, 27598, {f: 2, c: 27600}, 27608, 27610,
    {f: 5, c: 27612}, {f: 8, c: 27618}, {f: 3, c: 27628}, {f: 3, c: 27632},
    27636, {f: 3, c: 27638}, {f: 3, c: 27642}, 27646, {f: 5, c: 27648},
    {f: 3, c: 27657}, 27662, 27666, 27671, {f: 3, c: 27676}, 27680, 27685,
    27693, 27697, 27699, {f: 2, c: 27702}, {f: 4, c: 27705}, {f: 2, c: 27710},
    {f: 3, c: 27715}, 27720, {f: 5, c: 27723}, {f: 3, c: 27729}, 27734,
    {f: 3, c: 27736}, {f: 2, c: 27746}, {f: 3, c: 27749}, {f: 5, c: 27755},
    27761, 27763, 27765, {f: 2, c: 27767}, {f: 3, c: 27770}, {f: 2, c: 27775},
    27780, 27783, {f: 2, c: 27786}, {f: 2, c: 27789}, {f: 2, c: 27793},
    {f: 4, c: 27797}, 27802, {f: 3, c: 27804}, 27808, 27810, 27816, 27820,
    {f: 2, c: 27823}, {f: 4, c: 27828}, 27834, {f: 4, c: 27840},
    {f: 3, c: 27846}, 27851, {f: 3, c: 27853}, {f: 2, c: 27857},
    {f: 3, c: 27864}, {f: 2, c: 27868}, 27871, 27876, {f: 2, c: 27878}, 27881,
    {f: 2, c: 27884}, 27890, 27892, 27897, {f: 2, c: 27903}, {f: 2, c: 27906},
    {f: 2, c: 27909}, {f: 3, c: 27912}, 27917, {f: 3, c: 27919},
    {f: 4, c: 27923}, 27928, {f: 2, c: 27932}, {f: 6, c: 27935}, 27942,
    {f: 2, c: 27944}, {f: 2, c: 27948}, {f: 2, c: 27951}, 27956,
    {f: 3, c: 27958}, 27962, {f: 2, c: 27967}, 27970, 27972, 27977, 27980,
    27984, {f: 4, c: 27989}, 27995, 27997, 27999, {f: 2, c: 28001},
    {f: 2, c: 28004}, {f: 2, c: 28007}, {f: 3, c: 28011}, {f: 4, c: 28016},
    {f: 2, c: 28021}, {f: 2, c: 28026}, {f: 5, c: 28029}, {f: 2, c: 28035},
    28038, {f: 2, c: 28042}, 28045, {f: 2, c: 28047}, 28050, {f: 5, c: 28054},
    28060, 28066, 28069, {f: 2, c: 28076}, {f: 2, c: 28080}, {f: 2, c: 28083},
    {f: 2, c: 28086}, {f: 6, c: 28089}, {f: 3, c: 28097}, {f: 3, c: 28104},
    {f: 4, c: 28109}, {f: 4, c: 28114}, 28119, {f: 3, c: 28122}, 28127,
    {f: 2, c: 28130}, 28133, {f: 3, c: 28135}, 28141, {f: 2, c: 28143}, 28146,
    28148, 28152, {f: 8, c: 28157}, {f: 4, c: 28166}, 28171, 28175,
    {f: 2, c: 28178}, 28181, {f: 2, c: 28184}, {f: 2, c: 28187},
    {f: 2, c: 28190}, 28194, {f: 2, c: 28199}, 28202, 28206, {f: 2, c: 28208},
    28211, {f: 3, c: 28213}, 28217, {f: 3, c: 28219}, {f: 4, c: 28223},
    {f: 8, c: 28229}, {f: 4, c: 28239}, 28245, 28247, {f: 2, c: 28249},
    {f: 2, c: 28252}, {f: 11, c: 28256}, {f: 2, c: 28268}, {f: 14, c: 28272},
    {f: 3, c: 28288}, 28292, {f: 2, c: 28295}, {f: 5, c: 28298},
    {f: 5, c: 28305}, 28311, {f: 3, c: 28313}, 28318, {f: 2, c: 28320},
    {f: 2, c: 28323}, 28326, {f: 2, c: 28328}, {f: 4, c: 28331}, 28336, 28339,
    28341, {f: 2, c: 28344}, 28348, {f: 3, c: 28350}, 28355, 28358,
    {f: 3, c: 28360}, 28365, 28368, 28370, 28374, {f: 2, c: 28376},
    {f: 3, c: 28379}, 28387, 28391, {f: 2, c: 28394}, {f: 2, c: 28397},
    {f: 2, c: 28400}, 28403, {f: 2, c: 28405}, {f: 5, c: 28410}, 28416,
    {f: 3, c: 28419}, {f: 2, c: 28423}, {f: 5, c: 28426}, {f: 3, c: 28432},
    {f: 4, c: 28438}, {f: 5, c: 28443}, 28449, {f: 4, c: 28453}, 28462, 28464,
    {f: 2, c: 28468}, 28471, {f: 5, c: 28473}, 28480, {f: 4, c: 28482},
    {f: 3, c: 28488}, 28492, {f: 3, c: 28494}, {f: 2, c: 28498},
    {f: 3, c: 28501}, {f: 2, c: 28506}, 28509, {f: 3, c: 28511}, 28515, 28517,
    {f: 6, c: 28519}, 28529, 28531, {f: 2, c: 28533}, 28537, 28539,
    {f: 2, c: 28541}, {f: 3, c: 28545}, 28549, {f: 2, c: 28554},
    {f: 8, c: 28559}, {f: 4, c: 28568}, {f: 3, c: 28573}, {f: 2, c: 28578},
    {f: 2, c: 28581}, 28584, {f: 4, c: 28586}, {f: 2, c: 28591}, 28594,
    {f: 2, c: 28596}, {f: 2, c: 28599}, {f: 6, c: 28602}, {f: 5, c: 28612},
    {f: 7, c: 28618}, {f: 2, c: 28627}, {f: 2, c: 28630}, {f: 2, c: 28633},
    {f: 2, c: 28636}, {f: 2, c: 28642}, {f: 6, c: 28645}, {f: 2, c: 28652},
    {f: 8, c: 28658}, 28667, 28669, {f: 6, c: 28671}, {f: 2, c: 28679}, 28682,
    {f: 3, c: 28684}, 28688, {f: 3, c: 28690}, {f: 2, c: 28694}, 28697, 28700,
    28702, {f: 2, c: 28705}, {f: 3, c: 28708}, {f: 7, c: 28713}, 28721,
    {f: 2, c: 28723}, {f: 3, c: 28726}, {f: 4, c: 28730}, {f: 4, c: 28735},
    {f: 7, c: 28741}, {f: 2, c: 28749}, 28752, {f: 3, c: 28754},
    {f: 2, c: 28758}, {f: 4, c: 28761}, {f: 4, c: 28767}, {f: 2, c: 28773},
    {f: 3, c: 28776}, 28782, {f: 4, c: 28785}, 28791, {f: 3, c: 28793}, 28797,
    {f: 4, c: 28801}, {f: 3, c: 28806}, {f: 3, c: 28811}, {f: 3, c: 28815},
    28819, {f: 2, c: 28823}, {f: 2, c: 28826}, {f: 13, c: 28830}, 28848, 28850,
    {f: 3, c: 28852}, 28858, {f: 2, c: 28862}, {f: 4, c: 28868}, 28873,
    {f: 4, c: 28875}, {f: 8, c: 28880}, 28890, {f: 3, c: 28892},
    {f: 4, c: 28896}, 28901, 28906, 28910, {f: 4, c: 28912}, {f: 2, c: 28917},
    28920, {f: 3, c: 28922}, {f: 11, c: 28926}, {f: 5, c: 28939},
    {f: 2, c: 28945}, 28948, 28951, {f: 6, c: 28955}, {f: 4, c: 28962},
    {f: 8, c: 28967}, {f: 4, c: 28978}, {f: 14, c: 28983}, {f: 3, c: 28998},
    29003, 29005, {f: 3, c: 29007}, {f: 9, c: 29011}, 29021, {f: 3, c: 29023},
    29027, 29029, {f: 2, c: 29034}, 29037, {f: 3, c: 29039}, {f: 4, c: 29044},
    29049, {f: 2, c: 29051}, {f: 6, c: 29054}, {f: 5, c: 29061},
    {f: 4, c: 29067}, {f: 2, c: 29072}, 29075, {f: 2, c: 29077},
    {f: 5, c: 29082}, {f: 7, c: 29089}, {f: 3, c: 29097}, {f: 4, c: 29101},
    29106, 29108, {f: 3, c: 29110}, {f: 4, c: 29114}, {f: 2, c: 29119}, 29122,
    {f: 4, c: 29124}, {f: 5, c: 29129}, {f: 3, c: 29135}, 29139,
    {f: 3, c: 29142}, {f: 2, c: 29146}, {f: 2, c: 29149}, {f: 4, c: 29153},
    {f: 5, c: 29160}, {f: 5, c: 29167}, {f: 4, c: 29173}, {f: 2, c: 29178},
    29181, {f: 7, c: 29183}, {f: 6, c: 29191}, {f: 2, c: 29198},
    {f: 10, c: 29201}, 29212, {f: 10, c: 29214}, 29225, 29227,
    {f: 3, c: 29229}, {f: 2, c: 29235}, 29244, {f: 7, c: 29248},
    {f: 3, c: 29257}, {f: 4, c: 29262}, {f: 3, c: 29267}, 29271, 29274, 29276,
    29278, 29280, {f: 3, c: 29283}, 29288, {f: 4, c: 29290}, {f: 2, c: 29296},
    {f: 2, c: 29299}, {f: 3, c: 29302}, {f: 2, c: 29307}, {f: 2, c: 29314},
    {f: 5, c: 29317}, 29324, 29326, {f: 2, c: 29328}, {f: 3, c: 29331},
    {f: 8, c: 29335}, {f: 2, c: 29344}, {f: 4, c: 29347}, {f: 4, c: 29352},
    29358, {f: 3, c: 29361}, 29365, {f: 6, c: 29370}, {f: 3, c: 29381},
    {f: 4, c: 29385}, 29391, 29393, {f: 4, c: 29395}, 29400, {f: 4, c: 29402},
    29407, {f: 6, c: 29410}, {f: 2, c: 29418}, {f: 2, c: 29429},
    {f: 3, c: 29438}, 29442, {f: 6, c: 29444}, {f: 3, c: 29451},
    {f: 4, c: 29455}, 29460, {f: 3, c: 29464}, {f: 2, c: 29471},
    {f: 2, c: 29475}, {f: 3, c: 29478}, 29485, {f: 2, c: 29487},
    {f: 2, c: 29490}, 29493, 29498, {f: 2, c: 29500}, 29504, {f: 2, c: 29506},
    {f: 7, c: 29510}, {f: 2, c: 29518}, 29521, {f: 4, c: 29523},
    {f: 8, c: 29528}, {f: 7, c: 29537}, 29545, 29550, 29553, {f: 2, c: 29555},
    29558, 29561, 29565, 29567, {f: 3, c: 29569}, {f: 2, c: 29573}, 29576,
    29578, {f: 2, c: 29580}, {f: 2, c: 29583}, {f: 4, c: 29586},
    {f: 4, c: 29591}, {f: 3, c: 29596}, {f: 2, c: 29600}, {f: 6, c: 29603},
    29610, {f: 2, c: 29612}, 29617, {f: 3, c: 29620}, {f: 2, c: 29624},
    {f: 4, c: 29628}, 29633, {f: 5, c: 29635}, {f: 2, c: 29643}, 29646,
    {f: 7, c: 29650}, {f: 4, c: 29658}, 29663, {f: 4, c: 29665}, 29670, 29672,
    {f: 3, c: 29674}, {f: 4, c: 29678}, {f: 11, c: 29683}, {f: 4, c: 29695},
    29700, {f: 2, c: 29703}, {f: 4, c: 29707}, {f: 9, c: 29713},
    {f: 6, c: 29724}, {f: 2, c: 29731}, 29735, 29737, 29739, 29741, 29743,
    {f: 2, c: 29745}, {f: 5, c: 29751}, {f: 2, c: 29757}, 29760,
    {f: 9, c: 29762}, {f: 9, c: 29772}, 29782, 29784, 29789, {f: 3, c: 29792},
    {f: 5, c: 29796}, {f: 2, c: 29803}, {f: 2, c: 29806}, {f: 5, c: 29809},
    {f: 6, c: 29816}, 29823, 29826, {f: 3, c: 29828}, 29832, 29834,
    {f: 2, c: 29836}, 29839, {f: 11, c: 29841}, 29853, {f: 4, c: 29855},
    {f: 2, c: 29860}, {f: 6, c: 29866}, {f: 9, c: 29873}, {f: 2, c: 29883},
    {f: 12, c: 29886}, {f: 4, c: 29899}, {f: 2, c: 29904}, 29907,
    {f: 5, c: 29909}, 29915, 29917, 29919, 29921, 29925, {f: 7, c: 29927},
    {f: 4, c: 29936}, 29941, {f: 7, c: 29944}, {f: 4, c: 29952},
    {f: 7, c: 29957}, 29966, 29968, 29970, {f: 4, c: 29972}, 29979,
    {f: 2, c: 29981}, {f: 3, c: 29984}, 29988, {f: 2, c: 29990}, 29994, 29998,
    30004, 30006, 30009, {f: 2, c: 30012}, 30015, {f: 4, c: 30017},
    {f: 2, c: 30022}, {f: 2, c: 30025}, 30029, {f: 4, c: 30032},
    {f: 4, c: 30037}, {f: 4, c: 30046}, {f: 2, c: 30051}, {f: 3, c: 30055},
    {f: 6, c: 30060}, 30067, 30069, 30071, {f: 5, c: 30074}, {f: 3, c: 30080},
    {f: 2, c: 30084}, {f: 3, c: 30088}, {f: 3, c: 30092}, 30096, 30099, 30101,
    30104, {f: 2, c: 30107}, 30110, 30114, {f: 5, c: 30118}, 30125,
    {f: 2, c: 30134}, {f: 2, c: 30138}, {f: 3, c: 30143}, 30150,
    {f: 2, c: 30155}, {f: 4, c: 30158}, 30163, 30167, 30170, {f: 2, c: 30172},
    {f: 3, c: 30175}, 30181, 30185, {f: 4, c: 30188}, {f: 2, c: 30194},
    {f: 4, c: 30197}, {f: 2, c: 30202}, {f: 2, c: 30205}, 30212,
    {f: 4, c: 30214}, {f: 2, c: 30222}, {f: 4, c: 30225}, 30230, 30234,
    {f: 2, c: 30236}, 30243, 30248, 30252, {f: 2, c: 30254}, {f: 2, c: 30257},
    {f: 2, c: 30262}, {f: 2, c: 30265}, 30269, 30273, {f: 2, c: 30276}, 30280,
    {f: 2, c: 30282}, {f: 6, c: 30286}, 30293, 30295, {f: 3, c: 30297}, 30301,
    {f: 2, c: 30304}, 30310, 30312, 30314, {f: 3, c: 30323}, [12136, 30326],
    30327, {f: 2, c: 30329}, {f: 3, c: 30335}, 30339, 30341, {f: 2, c: 30345},
    {f: 2, c: 30348}, {f: 2, c: 30351}, 30354, {f: 2, c: 30356},
    {f: 2, c: 30359}, {f: 9, c: 30363}, {f: 9, c: 30373}, {f: 2, c: 30383},
    30387, {f: 3, c: 30389}, 30393, {f: 4, c: 30395}, {f: 2, c: 30400},
    {f: 2, c: 30403}, 30407, 30409, {f: 2, c: 30411}, 30419, 30421,
    {f: 2, c: 30425}, {f: 2, c: 30428}, 30432, 30434, 30438, {f: 6, c: 30440},
    30448, 30451, {f: 3, c: 30453}, {f: 2, c: 30458}, 30461, {f: 2, c: 30463},
    {f: 2, c: 30466}, {f: 2, c: 30469}, 30474, 30476, {f: 11, c: 30478},
    {f: 4, c: 30491}, 30497, {f: 3, c: 30499}, 30503, {f: 3, c: 30506}, 30510,
    {f: 5, c: 30512}, 30521, 30523, {f: 3, c: 30525}, 30530, {f: 3, c: 30532},
    {f: 7, c: 30536}, {f: 8, c: 30546}, {f: 2, c: 30556}, {f: 2, c: 30559},
    30564, 30567, {f: 2, c: 30569}, {f: 12, c: 30573}, {f: 3, c: 30586},
    {f: 3, c: 30593}, {f: 6, c: 30598}, {f: 2, c: 30607}, {f: 5, c: 30611},
    {f: 5, c: 30617}, 30625, {f: 2, c: 30627}, 30630, 30632, 30635,
    {f: 2, c: 30638}, {f: 2, c: 30641}, 30644, {f: 5, c: 30646}, 30654,
    {f: 7, c: 30656}, {f: 5, c: 30664}, {f: 9, c: 30670}, {f: 2, c: 30680},
    {f: 5, c: 30685}, 30692, 30694, 30696, 30698, {f: 3, c: 30704},
    {f: 2, c: 30708}, 30711, {f: 4, c: 30713}, {f: 6, c: 30723},
    {f: 2, c: 30730}, {f: 3, c: 30734}, 30739, 30741, 30745, 30747, 30750,
    {f: 3, c: 30752}, 30756, 30760, {f: 2, c: 30762}, {f: 2, c: 30766},
    {f: 3, c: 30769}, {f: 2, c: 30773}, 30781, 30783, {f: 2, c: 30785}, 30788,
    30790, {f: 4, c: 30792}, 30797, 30799, 30801, {f: 2, c: 30803},
    {f: 5, c: 30808}, {f: 6, c: 30814}, {f: 3, c: 30821}, 30825,
    {f: 7, c: 30832}, {f: 4, c: 30840}, {f: 10, c: 30845}, 30856,
    {f: 2, c: 30858}, {f: 2, c: 30863}, 30866, {f: 3, c: 30868}, 30873,
    {f: 2, c: 30877}, 30880, 30882, 30884, 30886, 30888, {f: 3, c: 30890},
    {f: 2, c: 30894}, {f: 3, c: 30901}, 30907, 30909, {f: 2, c: 30911},
    {f: 3, c: 30914}, {f: 3, c: 30918}, {f: 4, c: 30924}, {f: 3, c: 30929},
    {f: 3, c: 30934}, {f: 8, c: 30939}, {f: 3, c: 30948}, {f: 3, c: 30953},
    {f: 2, c: 30957}, {f: 2, c: 30960}, 30963, {f: 2, c: 30965},
    {f: 2, c: 30968}, {f: 2, c: 30971}, {f: 3, c: 30974}, {f: 3, c: 30978},
    {f: 8, c: 30982}, {f: 4, c: 30991}, {f: 5, c: 30996}, {f: 4, c: 31002},
    {f: 5, c: 31007}, 31013, {f: 3, c: 31015}, {f: 4, c: 31021},
    {f: 2, c: 31026}, {f: 5, c: 31029}, 31037, 31039, {f: 4, c: 31042}, 31047,
    {f: 9, c: 31050}, {f: 2, c: 31060}, {f: 2, c: 31064}, 31073,
    {f: 2, c: 31075}, 31078, {f: 4, c: 31081}, 31086, {f: 7, c: 31088}, 31097,
    {f: 5, c: 31099}, {f: 2, c: 31106}, {f: 4, c: 31110}, {f: 2, c: 31115},
    {f: 10, c: 31120}, {f: 11, c: 31131}, {f: 2, c: 31144}, {f: 3, c: 31147},
    31151, 31154, {f: 4, c: 31156}, [12145, 31160], 31164, 31167, 31170,
    {f: 2, c: 31172}, {f: 2, c: 31175}, 31178, 31180, {f: 3, c: 31182},
    {f: 2, c: 31187}, {f: 2, c: 31190}, {f: 6, c: 31193}, {f: 3, c: 31200},
    31205, 31208, 31210, 31212, 31214, {f: 7, c: 31217}, {f: 2, c: 31225},
    31228, {f: 2, c: 31230}, 31233, {f: 2, c: 31236}, {f: 4, c: 31239}, 31244,
    {f: 5, c: 31247}, {f: 2, c: 31253}, {f: 2, c: 31256}, {f: 3, c: 31259},
    31263, {f: 2, c: 31265}, {f: 10, c: 31268}, {f: 2, c: 31279}, 31282,
    {f: 3, c: 31284}, 31288, 31290, 31294, {f: 5, c: 31297}, {f: 5, c: 31303},
    {f: 2, c: 31311}, {f: 5, c: 31314}, {f: 9, c: 31320}, {f: 6, c: 31331},
    31338, {f: 4, c: 31340}, {f: 3, c: 31345}, 31349, {f: 4, c: 31355}, 31362,
    31365, 31367, {f: 4, c: 31369}, {f: 3, c: 31374}, {f: 2, c: 31379},
    {f: 3, c: 31385}, 31390, {f: 4, c: 31393}, 31399, 31403, {f: 4, c: 31407},
    {f: 2, c: 31412}, {f: 3, c: 31415}, {f: 4, c: 31419}, {f: 4, c: 31424},
    31430, 31433, {f: 10, c: 31436}, {f: 2, c: 31447}, {f: 4, c: 31450},
    {f: 2, c: 31457}, 31460, {f: 3, c: 31463}, {f: 2, c: 31467}, 31470,
    {f: 6, c: 31472}, {f: 2, c: 31479}, {f: 2, c: 31483}, 31486,
    {f: 3, c: 31488}, 31493, 31495, 31497, {f: 3, c: 31500}, 31504,
    {f: 2, c: 31506}, {f: 3, c: 31510}, 31514, {f: 2, c: 31516}, 31519,
    {f: 3, c: 31521}, 31527, 31529, 31533, {f: 2, c: 31535}, 31538,
    {f: 4, c: 31540}, 31545, 31547, 31549, {f: 6, c: 31551}, 31560, 31562,
    {f: 2, c: 31565}, 31571, 31573, 31575, 31577, 31580, {f: 2, c: 31582},
    31585, {f: 4, c: 31587}, {f: 6, c: 31592}, {f: 2, c: 31599},
    {f: 2, c: 31603}, 31606, 31608, 31610, {f: 2, c: 31612}, 31615,
    {f: 4, c: 31617}, {f: 5, c: 31622}, 31628, {f: 2, c: 31630},
    {f: 3, c: 31633}, 31638, {f: 4, c: 31640}, {f: 3, c: 31646},
    {f: 3, c: 31651}, {f: 3, c: 31662}, {f: 2, c: 31666}, {f: 3, c: 31669},
    {f: 7, c: 31673}, {f: 2, c: 31682}, 31685, 31688, 31690, {f: 4, c: 31693},
    31698, {f: 5, c: 31700}, {f: 2, c: 31707}, {f: 3, c: 31710},
    {f: 2, c: 31714}, {f: 2, c: 31719}, {f: 3, c: 31723}, {f: 2, c: 31727},
    31730, {f: 3, c: 31732}, {f: 4, c: 31736}, 31741, 31743, {f: 6, c: 31745},
    {f: 3, c: 31752}, 31758, {f: 6, c: 31760}, {f: 7, c: 31767}, 31776, 31778,
    {f: 2, c: 31780}, {f: 2, c: 31784}, {f: 12, c: 31788}, {f: 4, c: 31801},
    31810, {f: 8, c: 31812}, {f: 14, c: 31822}, {f: 2, c: 31837},
    {f: 3, c: 31841}, {f: 4, c: 31845}, 31851, 31853, {f: 3, c: 31855},
    {f: 6, c: 31861}, {f: 11, c: 31870}, {f: 7, c: 31882}, {f: 2, c: 31891},
    31894, {f: 3, c: 31897}, {f: 2, c: 31904}, 31907, {f: 4, c: 31910},
    {f: 3, c: 31915}, {f: 2, c: 31919}, {f: 5, c: 31924}, {f: 2, c: 31930},
    {f: 2, c: 31935}, {f: 3, c: 31938}, 31942, 31945, 31947, {f: 7, c: 31950},
    31960, {f: 2, c: 31962}, {f: 6, c: 31969}, {f: 6, c: 31977}, 31985, 31987,
    31989, 31991, 31994, {f: 2, c: 31996}, 31999, 32001, 32003, 32012,
    {f: 2, c: 32014}, {f: 2, c: 32017}, 32022, 32024, {f: 3, c: 32029},
    {f: 4, c: 32035}, {f: 3, c: 32040}, {f: 3, c: 32044}, {f: 5, c: 32052},
    32059, {f: 2, c: 32061}, 32065, 32067, 32069, {f: 7, c: 32071}, 32079,
    {f: 12, c: 32081}, {f: 2, c: 32095}, {f: 3, c: 32099}, 32103,
    {f: 5, c: 32105}, {f: 2, c: 32111}, {f: 2, c: 32116}, 32120,
    {f: 7, c: 32122}, 32130, {f: 2, c: 32132}, 32135, {f: 5, c: 32138},
    {f: 3, c: 32144}, {f: 8, c: 32148}, 32157, {f: 3, c: 32159},
    {f: 2, c: 32164}, {f: 4, c: 32167}, 32175, {f: 3, c: 32181}, 32188,
    {f: 4, c: 32192}, {f: 2, c: 32197}, {f: 2, c: 32200}, {f: 5, c: 32204},
    32211, {f: 2, c: 32213}, {f: 3, c: 32218}, 32223, 32226, {f: 2, c: 32228},
    32231, {f: 2, c: 32234}, {f: 2, c: 32237}, 32240, 32243, 32245,
    {f: 2, c: 32247}, 32250, {f: 12, c: 32252}, {f: 4, c: 32268},
    {f: 9, c: 32274}, 32284, {f: 3, c: 32288}, {f: 3, c: 32292},
    {f: 3, c: 32296}, 32300, {f: 2, c: 32303}, 32307, 32312, 32314, 32316,
    {f: 2, c: 32319}, {f: 3, c: 32322}, {f: 10, c: 32328}, 32339,
    {f: 4, c: 32342}, {f: 3, c: 32347}, {f: 3, c: 32351}, {f: 6, c: 32355},
    32364, {f: 2, c: 32369}, {f: 5, c: 32372}, {f: 2, c: 32378},
    {f: 3, c: 32383}, {f: 5, c: 32387}, 32393, 32395, 32398, {f: 3, c: 32400},
    32405, 32407, {f: 2, c: 32409}, {f: 2, c: 32413}, 32430, 32436,
    {f: 2, c: 32443}, 32470, 32484, 32492, 32505, 32522, 32528, 32542, 32567,
    32569, {f: 7, c: 32571}, 32579, {f: 6, c: 32582}, 32589, 32591,
    {f: 2, c: 32594}, 32598, 32601, {f: 4, c: 32603}, 32608, {f: 5, c: 32611},
    {f: 3, c: 32619}, 32623, 32627, {f: 2, c: 32629}, 32632, {f: 4, c: 32634},
    {f: 2, c: 32639}, {f: 3, c: 32642}, 32647, 32649, 32651, 32653,
    {f: 5, c: 32655}, {f: 5, c: 32661}, {f: 2, c: 32667}, 32672,
    {f: 2, c: 32674}, 32678, 32680, {f: 5, c: 32682}, 32689, {f: 5, c: 32691},
    {f: 2, c: 32698}, 32702, 32704, {f: 3, c: 32706}, {f: 4, c: 32710}, 32715,
    32717, {f: 3, c: 32719}, 32723, {f: 2, c: 32726}, {f: 6, c: 32729},
    {f: 3, c: 32738}, {f: 2, c: 32743}, {f: 4, c: 32746}, 32751, 32754,
    {f: 5, c: 32756}, 32762, {f: 3, c: 32765}, 32770, {f: 4, c: 32775},
    {f: 2, c: 32782}, 32785, 32787, {f: 2, c: 32794}, {f: 3, c: 32797}, 32801,
    {f: 2, c: 32803}, 32811, 32813, {f: 2, c: 32815}, 32818, 32820,
    {f: 2, c: 32825}, 32828, 32830, {f: 2, c: 32832}, {f: 2, c: 32836},
    {f: 3, c: 32839}, {f: 4, c: 32846}, 32851, 32853, 32855, 32857,
    {f: 3, c: 32859}, {f: 10, c: 32863}, {f: 4, c: 32875}, 32884, 32888,
    {f: 3, c: 32890}, {f: 2, c: 32897}, 32904, 32906, {f: 6, c: 32909},
    {f: 2, c: 32916}, 32919, 32921, 32926, 32931, {f: 3, c: 32934}, 32940,
    32944, 32947, {f: 2, c: 32949}, {f: 2, c: 32952}, 32955, 32965,
    {f: 5, c: 32967}, {f: 7, c: 32975}, 32984, {f: 2, c: 32991},
    {f: 2, c: 32994}, 32998, 33006, 33013, 33015, 33017, 33019,
    {f: 4, c: 33022}, {f: 2, c: 33027}, {f: 2, c: 33031}, {f: 2, c: 33035},
    33045, 33047, 33049, {f: 2, c: 33052}, {f: 13, c: 33055}, {f: 2, c: 33069},
    33072, {f: 3, c: 33075}, 33079, {f: 4, c: 33082}, {f: 7, c: 33087}, 33095,
    33097, 33101, 33103, 33106, {f: 2, c: 33111}, {f: 5, c: 33115},
    {f: 3, c: 33122}, 33128, 33130, 33132, 33135, {f: 2, c: 33138},
    {f: 3, c: 33141}, 33153, {f: 5, c: 33155}, 33161, {f: 4, c: 33163}, 33168,
    {f: 6, c: 33170}, 33177, {f: 2, c: 33182}, {f: 2, c: 33185},
    {f: 2, c: 33188}, 33191, {f: 8, c: 33195}, {f: 6, c: 33204}, 33212,
    {f: 2, c: 33220}, {f: 2, c: 33223}, 33227, 33230, {f: 8, c: 33232}, 33241,
    {f: 4, c: 33243}, {f: 2, c: 33249}, {f: 3, c: 33252}, 33257, 33259,
    {f: 5, c: 33262}, {f: 5, c: 33269}, 33277, 33279, 33283, 33291,
    {f: 2, c: 33294}, 33297, 33299, {f: 6, c: 33301}, 33309, 33312,
    {f: 4, c: 33316}, 33321, 33326, 33330, 33338, {f: 2, c: 33340},
    {f: 5, c: 33343}, {f: 2, c: 33349}, 33352, 33354, {f: 3, c: 33356},
    {f: 8, c: 33360}, {f: 4, c: 33371}, {f: 4, c: 33376}, 33381, 33383,
    {f: 2, c: 33385}, {f: 2, c: 33388}, {f: 2, c: 33397}, [12171, 33400],
    {f: 2, c: 33403}, {f: 2, c: 33408}, 33411, {f: 3, c: 33413}, 33417, 33420,
    33424, {f: 4, c: 33427}, {f: 2, c: 33434}, 33438, 33440, {f: 2, c: 33442},
    33447, 33458, {f: 2, c: 33461}, 33466, 33468, {f: 2, c: 33471},
    {f: 2, c: 33474}, {f: 2, c: 33477}, 33481, 33488, 33494, {f: 2, c: 33497},
    33501, 33506, {f: 3, c: 33512}, {f: 3, c: 33516}, 33520, {f: 2, c: 33522},
    {f: 2, c: 33525}, 33528, 33530, {f: 5, c: 33532}, {f: 2, c: 33546}, 33549,
    33552, {f: 2, c: 33554}, 33558, {f: 2, c: 33560}, {f: 10, c: 33565},
    {f: 2, c: 33577}, 33582, 33584, 33586, 33591, 33595, {f: 3, c: 33597},
    {f: 2, c: 33601}, {f: 2, c: 33604}, 33608, {f: 5, c: 33610}, 33619,
    {f: 5, c: 33621}, 33629, 33634, {f: 7, c: 33648}, {f: 2, c: 33657},
    {f: 7, c: 33662}, {f: 2, c: 33671}, {f: 3, c: 33675}, {f: 3, c: 33679},
    {f: 2, c: 33684}, 33687, {f: 2, c: 33689}, 33693, 33695, 33697,
    {f: 4, c: 33699}, {f: 4, c: 33708}, 33717, 33723, {f: 2, c: 33726},
    {f: 3, c: 33730}, 33734, {f: 2, c: 33736}, 33739, {f: 2, c: 33741},
    {f: 4, c: 33744}, 33749, 33751, {f: 3, c: 33753}, 33758, {f: 3, c: 33762},
    {f: 3, c: 33766}, {f: 4, c: 33771}, {f: 5, c: 33779}, {f: 3, c: 33786},
    {f: 3, c: 33790}, 33794, 33797, {f: 2, c: 33800}, 33808, {f: 6, c: 33810},
    {f: 3, c: 33817}, {f: 6, c: 33822}, {f: 3, c: 33833}, {f: 4, c: 33837},
    {f: 3, c: 33842}, {f: 2, c: 33846}, {f: 3, c: 33849}, {f: 8, c: 33854},
    {f: 2, c: 33863}, {f: 7, c: 33866}, {f: 4, c: 33875}, 33880,
    {f: 4, c: 33885}, 33890, 33893, {f: 2, c: 33895}, 33898, 33902, 33904,
    33906, 33908, 33913, {f: 7, c: 33915}, {f: 4, c: 33923}, 33930, 33933,
    {f: 4, c: 33935}, {f: 2, c: 33941}, 33944, {f: 2, c: 33946},
    {f: 4, c: 33949}, {f: 13, c: 33954}, {f: 2, c: 33968}, 33971,
    {f: 3, c: 33973}, 33979, 33982, {f: 2, c: 33986}, {f: 4, c: 33989}, 33996,
    {f: 2, c: 33998}, 34002, {f: 2, c: 34004}, {f: 6, c: 34007}, 34014,
    {f: 2, c: 34017}, 34020, {f: 5, c: 34023}, 34029, {f: 11, c: 34033}, 34046,
    {f: 12, c: 34048}, {f: 4, c: 34061}, 34066, {f: 2, c: 34069},
    {f: 2, c: 34072}, {f: 3, c: 34075}, 34080, 34082, {f: 2, c: 34084},
    {f: 4, c: 34087}, {f: 9, c: 34094}, {f: 3, c: 34110}, 34114,
    {f: 2, c: 34116}, 34119, {f: 3, c: 34123}, {f: 3, c: 34127}, 34132, 34135,
    {f: 4, c: 34138}, {f: 3, c: 34143}, 34147, {f: 3, c: 34149},
    {f: 2, c: 34155}, {f: 4, c: 34158}, 34163, {f: 2, c: 34165}, 34168,
    {f: 2, c: 34172}, {f: 5, c: 34175}, 34182, 34185, 34187, {f: 2, c: 34189},
    34192, {f: 2, c: 34194}, {f: 6, c: 34197}, {f: 2, c: 34205},
    {f: 4, c: 34208}, 34213, 34215, {f: 3, c: 34219}, {f: 6, c: 34225}, 34232,
    {f: 6, c: 34235}, {f: 7, c: 34242}, {f: 3, c: 34250}, {f: 2, c: 34257},
    34260, {f: 6, c: 34262}, {f: 6, c: 34270}, {f: 3, c: 34278},
    {f: 9, c: 34283}, 34293, {f: 2, c: 34295}, {f: 3, c: 34300},
    {f: 4, c: 34304}, {f: 3, c: 34312}, {f: 5, c: 34316}, {f: 4, c: 34322},
    {f: 3, c: 34327}, {f: 3, c: 34331}, {f: 3, c: 34335}, {f: 4, c: 34339},
    34344, {f: 3, c: 34346}, {f: 10, c: 34350}, 34361, 34363, {f: 2, c: 34365},
    {f: 13, c: 34368}, {f: 2, c: 34386}, {f: 4, c: 34390}, 34395, 34397,
    {f: 2, c: 34400}, {f: 4, c: 34403}, {f: 3, c: 34408}, 34413,
    {f: 2, c: 34415}, {f: 7, c: 34418}, {f: 7, c: 34435}, {f: 5, c: 34446},
    34452, {f: 6, c: 34454}, {f: 5, c: 34462}, {f: 2, c: 34469}, 34475,
    {f: 2, c: 34477}, {f: 2, c: 34482}, {f: 3, c: 34487}, {f: 5, c: 34491},
    {f: 3, c: 34497}, 34501, 34504, {f: 2, c: 34508}, {f: 2, c: 34514},
    {f: 3, c: 34517}, 34522, {f: 2, c: 34524}, {f: 4, c: 34528},
    {f: 4, c: 34533}, {f: 3, c: 34538}, 34543, {f: 3, c: 34549},
    {f: 3, c: 34555}, 34559, 34561, {f: 2, c: 34564}, {f: 2, c: 34571},
    {f: 4, c: 34574}, 34580, 34582, 34585, 34587, 34589, {f: 2, c: 34591},
    34596, {f: 3, c: 34598}, {f: 4, c: 34602}, {f: 2, c: 34607},
    {f: 2, c: 34610}, {f: 2, c: 34613}, {f: 3, c: 34616}, {f: 2, c: 34620},
    {f: 7, c: 34624}, {f: 2, c: 34634}, 34637, {f: 4, c: 34639}, 34644, 34646,
    34648, {f: 6, c: 34650}, {f: 2, c: 34657}, {f: 7, c: 34663}, 34671,
    {f: 3, c: 34673}, 34677, 34679, {f: 2, c: 34681}, {f: 3, c: 34687},
    {f: 2, c: 34694}, {f: 2, c: 34697}, 34700, {f: 5, c: 34702},
    {f: 3, c: 34708}, {f: 6, c: 34712}, {f: 2, c: 34720}, {f: 5, c: 34723},
    {f: 2, c: 34729}, 34734, {f: 3, c: 34736}, 34740, {f: 4, c: 34742}, 34748,
    {f: 2, c: 34750}, {f: 3, c: 34753}, 34757, 34759, 34761, {f: 2, c: 34764},
    {f: 2, c: 34767}, {f: 7, c: 34772}, {f: 4, c: 34780}, {f: 2, c: 34785},
    34788, {f: 4, c: 34790}, 34795, 34797, {f: 2, c: 34800}, {f: 3, c: 34803},
    {f: 2, c: 34807}, 34810, {f: 2, c: 34812}, {f: 4, c: 34815}, 34820,
    {f: 3, c: 34823}, {f: 5, c: 34827}, 34834, 34836, {f: 4, c: 34839},
    {f: 3, c: 34844}, 34848, {f: 13, c: 34852}, {f: 3, c: 34867},
    {f: 2, c: 34871}, 34874, {f: 3, c: 34877}, {f: 3, c: 34881},
    {f: 3, c: 34887}, 34891, {f: 5, c: 34894}, {f: 2, c: 34901}, 34904, 34906,
    34908, {f: 3, c: 34910}, {f: 2, c: 34918}, 34922, 34925, 34927, 34929,
    {f: 4, c: 34931}, 34936, {f: 3, c: 34938}, 34944, 34947, {f: 2, c: 34950},
    {f: 2, c: 34953}, 34956, {f: 4, c: 34958}, {f: 3, c: 34963},
    {f: 5, c: 34967}, {f: 5, c: 34973}, 34979, {f: 6, c: 34981}, 34988,
    {f: 3, c: 34990}, {f: 5, c: 34994}, {f: 4, c: 35000}, {f: 4, c: 35005},
    {f: 2, c: 35011}, {f: 2, c: 35015}, {f: 3, c: 35019}, {f: 2, c: 35024},
    35027, {f: 2, c: 35030}, {f: 2, c: 35034}, 35038, {f: 2, c: 35040},
    {f: 2, c: 35046}, {f: 7, c: 35049}, 35058, {f: 3, c: 35061},
    {f: 2, c: 35066}, {f: 3, c: 35071}, {f: 4, c: 35075}, {f: 2, c: 35080},
    {f: 5, c: 35083}, 35089, {f: 5, c: 35092}, {f: 5, c: 35100},
    {f: 3, c: 35106}, {f: 4, c: 35110}, {f: 4, c: 35116}, 35121, 35125, 35127,
    {f: 2, c: 35129}, {f: 5, c: 35132}, {f: 2, c: 35138}, {f: 2, c: 35141},
    {f: 14, c: 35144}, {f: 6, c: 35159}, {f: 3, c: 35169}, 35173,
    {f: 3, c: 35175}, 35179, {f: 2, c: 35181}, {f: 2, c: 35184},
    {f: 8, c: 35187}, {f: 2, c: 35196}, [12177, 35198], 35200, 35202,
    {f: 2, c: 35204}, {f: 4, c: 35207}, {f: 3, c: 35212}, {f: 3, c: 35216},
    {f: 2, c: 35220}, 35223, {f: 8, c: 35225}, {f: 4, c: 35234},
    {f: 3, c: 35239}, 35243, {f: 2, c: 35245}, {f: 2, c: 35248},
    {f: 4, c: 35251}, {f: 2, c: 35256}, {f: 2, c: 35259}, 35262, 35267, 35277,
    {f: 3, c: 35283}, {f: 3, c: 35287}, 35291, 35293, {f: 4, c: 35295}, 35300,
    {f: 4, c: 35303}, {f: 3, c: 35308}, {f: 3, c: 35312}, 35317, 35319,
    {f: 7, c: 35321}, {f: 3, c: 35332}, 35337, 35339, 35341, 35343,
    {f: 2, c: 35345}, 35348, 35351, {f: 2, c: 35353}, 35356, 35358,
    {f: 3, c: 35360}, 35364, {f: 4, c: 35366}, {f: 2, c: 35371},
    {f: 3, c: 35374}, {f: 2, c: 35378}, 35381, {f: 3, c: 35383},
    {f: 3, c: 35387}, {f: 2, c: 35391}, {f: 4, c: 35394}, 35399,
    {f: 5, c: 35401}, 35407, 35409, 35411, {f: 2, c: 35414}, {f: 2, c: 35417},
    {f: 2, c: 35420}, {f: 2, c: 35423}, {f: 2, c: 35428}, {f: 2, c: 35431},
    35434, 35439, 35444, {f: 3, c: 35446}, {f: 2, c: 35450}, {f: 2, c: 35453},
    {f: 4, c: 35456}, 35464, {f: 2, c: 35467}, {f: 3, c: 35470}, 35476,
    {f: 2, c: 35478}, 35481, {f: 3, c: 35483}, 35487, 35490, 35495,
    {f: 3, c: 35497}, {f: 3, c: 35501}, 35505, {f: 3, c: 35507},
    {f: 2, c: 35511}, {f: 2, c: 35514}, {f: 2, c: 35517}, {f: 2, c: 35520},
    35523, {f: 2, c: 35525}, 35528, 35530, 35532, 35534, 35536,
    {f: 3, c: 35539}, {f: 3, c: 35544}, 35549, {f: 3, c: 35551}, 35555, 35557,
    {f: 3, c: 35560}, 35564, {f: 2, c: 35567}, 35570, {f: 2, c: 35572}, 35577,
    35579, 35581, 35583, 35587, 35590, {f: 2, c: 35592}, {f: 3, c: 35595},
    35599, {f: 3, c: 35601}, 35605, 35608, 35612, {f: 3, c: 35614},
    {f: 4, c: 35618}, 35623, {f: 2, c: 35625}, {f: 5, c: 35630},
    {f: 5, c: 35636}, {f: 4, c: 35642}, {f: 10, c: 35647}, {f: 4, c: 35658},
    {f: 6, c: 35664}, 35671, 35675, {f: 9, c: 35677}, {f: 4, c: 35687},
    {f: 2, c: 35693}, {f: 3, c: 35697}, {f: 2, c: 35701}, {f: 5, c: 35704},
    {f: 2, c: 35710}, {f: 9, c: 35713}, {f: 3, c: 35723}, {f: 3, c: 35727},
    35732, {f: 5, c: 35735}, 35741, 35743, 35756, 35761, 35771, 35783, 35792,
    35818, 35849, 35870, {f: 9, c: 35896}, {f: 4, c: 35906}, {f: 2, c: 35914},
    {f: 3, c: 35917}, {f: 4, c: 35921}, {f: 4, c: 35926}, {f: 6, c: 35931},
    {f: 7, c: 35939}, {f: 7, c: 35948}, {f: 4, c: 35956}, {f: 7, c: 35963},
    {f: 2, c: 35971}, {f: 3, c: 35974}, 35979, {f: 7, c: 35981},
    {f: 3, c: 35989}, {f: 4, c: 35993}, 35999, {f: 4, c: 36003},
    {f: 2, c: 36013}, 36017, 36021, 36025, 36030, 36038, 36041,
    {f: 6, c: 36043}, 36052, {f: 4, c: 36054}, 36059, 36061, 36063, 36069,
    {f: 2, c: 36072}, {f: 6, c: 36078}, {f: 5, c: 36085}, {f: 5, c: 36095},
    {f: 2, c: 36102}, 36105, 36108, 36110, {f: 5, c: 36113}, {f: 4, c: 36119},
    36128, {f: 2, c: 36177}, 36183, 36191, 36197, {f: 3, c: 36200}, 36204,
    {f: 2, c: 36206}, {f: 2, c: 36209}, {f: 9, c: 36216}, {f: 2, c: 36226},
    {f: 4, c: 36230}, {f: 5, c: 36236}, {f: 2, c: 36242}, {f: 3, c: 36246},
    {f: 5, c: 36250}, {f: 3, c: 36256}, {f: 4, c: 36260}, {f: 8, c: 36265},
    {f: 2, c: 36278}, 36281, 36283, 36285, {f: 3, c: 36288}, 36293,
    {f: 4, c: 36295}, 36301, 36304, {f: 4, c: 36306}, {f: 2, c: 36312}, 36316,
    {f: 3, c: 36320}, {f: 3, c: 36325}, 36329, {f: 2, c: 36333},
    {f: 3, c: 36336}, 36340, 36342, 36348, {f: 7, c: 36350}, {f: 3, c: 36358},
    36363, {f: 2, c: 36365}, {f: 3, c: 36369}, {f: 8, c: 36373},
    {f: 2, c: 36384}, {f: 5, c: 36388}, 36395, 36397, 36400, {f: 2, c: 36402},
    {f: 3, c: 36406}, {f: 2, c: 36411}, {f: 2, c: 36414}, 36419,
    {f: 2, c: 36421}, {f: 4, c: 36429}, {f: 2, c: 36435}, {f: 3, c: 36438},
    {f: 9, c: 36442}, {f: 2, c: 36452}, {f: 2, c: 36455}, {f: 2, c: 36458},
    36462, 36465, 36467, 36469, {f: 3, c: 36471}, 36475, {f: 2, c: 36477},
    36480, {f: 3, c: 36482}, 36486, 36488, 36492, 36494, {f: 5, c: 36501},
    36507, 36509, {f: 2, c: 36511}, {f: 3, c: 36514}, {f: 3, c: 36519},
    {f: 2, c: 36525}, {f: 2, c: 36528}, {f: 7, c: 36531}, {f: 5, c: 36539},
    {f: 9, c: 36545}, {f: 3, c: 36559}, 36563, {f: 6, c: 36565},
    {f: 3, c: 36572}, {f: 4, c: 36576}, {f: 6, c: 36581}, {f: 6, c: 36588},
    {f: 5, c: 36595}, 36605, {f: 4, c: 36607}, 36612, 36614, 36616,
    {f: 7, c: 36619}, 36627, {f: 5, c: 36630}, {f: 5, c: 36640},
    {f: 2, c: 36647}, {f: 4, c: 36651}, {f: 3, c: 36656}, {f: 4, c: 36660},
    {f: 2, c: 36665}, {f: 2, c: 36668}, {f: 2, c: 36672}, 36675,
    {f: 2, c: 36679}, {f: 3, c: 36682}, {f: 5, c: 36687}, {f: 10, c: 36693},
    36704, 36707, 36709, 36714, 36736, 36748, 36754, 36765, {f: 3, c: 36768},
    {f: 2, c: 36772}, 36775, 36778, 36780, {f: 2, c: 36787}, [12193, 36789],
    {f: 2, c: 36791}, {f: 3, c: 36794}, {f: 2, c: 36799}, 36803, 36806,
    {f: 5, c: 36809}, 36815, 36818, {f: 2, c: 36822}, 36826, {f: 2, c: 36832},
    36835, 36839, 36844, 36847, {f: 2, c: 36849}, {f: 2, c: 36853},
    {f: 3, c: 36858}, {f: 2, c: 36862}, {f: 2, c: 36871}, 36876, 36878, 36883,
    36888, 36892, {f: 2, c: 36900}, {f: 6, c: 36903}, {f: 2, c: 36912},
    {f: 2, c: 36915}, 36919, {f: 2, c: 36921}, 36925, {f: 2, c: 36927}, 36931,
    {f: 2, c: 36933}, {f: 3, c: 36936}, 36940, 36950, {f: 2, c: 36953}, 36957,
    36959, 36961, 36964, {f: 2, c: 36966}, {f: 3, c: 36970}, {f: 3, c: 36975},
    36979, 36982, 36985, 36987, 36990, {f: 2, c: 36997}, 37001,
    {f: 3, c: 37004}, 37010, 37012, 37014, 37016, 37018, 37020,
    {f: 3, c: 37022}, {f: 2, c: 37028}, {f: 3, c: 37031}, 37035, 37037, 37042,
    37047, {f: 2, c: 37052}, {f: 2, c: 37055}, {f: 2, c: 37058}, 37062,
    {f: 2, c: 37064}, {f: 3, c: 37067}, 37074, {f: 3, c: 37076},
    {f: 3, c: 37080}, 37086, 37088, {f: 3, c: 37091}, {f: 2, c: 37097}, 37100,
    37102, {f: 4, c: 37104}, {f: 2, c: 37110}, {f: 4, c: 37113},
    {f: 3, c: 37119}, 37123, 37125, {f: 2, c: 37127}, {f: 8, c: 37130}, 37139,
    37141, {f: 2, c: 37143}, {f: 4, c: 37146}, {f: 3, c: 37151},
    {f: 3, c: 37156}, {f: 5, c: 37160}, 37166, 37171, 37173, {f: 2, c: 37175},
    {f: 8, c: 37179}, {f: 2, c: 37188}, 37191, 37201, {f: 4, c: 37203},
    {f: 2, c: 37208}, {f: 2, c: 37211}, {f: 2, c: 37215}, {f: 3, c: 37222},
    37227, 37229, 37235, {f: 3, c: 37242}, {f: 5, c: 37248}, 37254, 37256,
    37258, {f: 2, c: 37262}, {f: 3, c: 37267}, {f: 3, c: 37271},
    {f: 5, c: 37277}, {f: 6, c: 37284}, {f: 4, c: 37296}, {f: 4, c: 37302},
    {f: 5, c: 37307}, 37314, 37316, [12196, 37318], 37320, 37328, 37334,
    {f: 2, c: 37338}, {f: 5, c: 37342}, {f: 2, c: 37349}, 37352,
    {f: 11, c: 37354}, 37366, 37368, {f: 5, c: 37371}, {f: 2, c: 37378},
    {f: 3, c: 37381}, {f: 3, c: 37386}, 37391, {f: 2, c: 37394},
    {f: 8, c: 37398}, {f: 4, c: 37407}, 37412, {f: 6, c: 37416}, 37423,
    {f: 2, c: 37425}, {f: 2, c: 37429}, {f: 2, c: 37435}, {f: 4, c: 37441},
    {f: 2, c: 37446}, {f: 3, c: 37450}, {f: 3, c: 37454}, {f: 3, c: 37458},
    37462, {f: 2, c: 37464}, {f: 2, c: 37468}, {f: 3, c: 37471},
    {f: 3, c: 37475}, {f: 5, c: 37479}, {f: 6, c: 37486}, {f: 3, c: 37493},
    37497, {f: 3, c: 37500}, {f: 2, c: 37505}, 37508, {f: 8, c: 37510},
    {f: 2, c: 37519}, 37522, {f: 2, c: 37524}, 37527, 37529, 37531,
    {f: 3, c: 37533}, {f: 2, c: 37537}, 37540, 37543, 37549, {f: 2, c: 37551},
    {f: 5, c: 37554}, 37560, 37562, {f: 4, c: 37565}, 37570, 37572, 37574,
    {f: 3, c: 37577}, {f: 2, c: 37581}, {f: 2, c: 37584}, {f: 10, c: 37587},
    37598, {f: 3, c: 37600}, 37607, 37609, {f: 2, c: 37611}, {f: 4, c: 37618},
    37623, {f: 3, c: 37625}, {f: 4, c: 37629}, {f: 4, c: 37634},
    {f: 7, c: 37641}, 37649, {f: 2, c: 37651}, {f: 2, c: 37654},
    {f: 3, c: 37660}, 37665, {f: 3, c: 37667}, 37671, {f: 2, c: 37673},
    {f: 2, c: 37676}, {f: 2, c: 37680}, {f: 2, c: 37684}, 37687,
    {f: 5, c: 37689}, 37695, 37698, {f: 2, c: 37700}, {f: 3, c: 37704}, 37708,
    {f: 6, c: 37710}, {f: 3, c: 37717}, {f: 2, c: 37721}, {f: 8, c: 37724},
    {f: 3, c: 37734}, 37739, {f: 3, c: 37741}, {f: 4, c: 37745},
    {f: 3, c: 37751}, {f: 3, c: 37755}, {f: 3, c: 37759}, 37763,
    {f: 2, c: 37765}, {f: 2, c: 37768}, {f: 4, c: 37771}, {f: 6, c: 37776},
    37783, {f: 9, c: 37785}, {f: 2, c: 37796}, 37800, 37803, 37805, 37807,
    {f: 2, c: 37809}, 37812, {f: 2, c: 37814}, {f: 6, c: 37817},
    {f: 3, c: 37824}, {f: 3, c: 37828}, 37833, 37835, {f: 3, c: 37838},
    {f: 4, c: 37842}, {f: 3, c: 37849}, 37856, 37859, {f: 3, c: 37861},
    {f: 12, c: 37865}, 37878, 37880, {f: 9, c: 37882}, {f: 7, c: 37892},
    {f: 4, c: 37900}, 37905, {f: 3, c: 37909}, {f: 3, c: 37914},
    {f: 2, c: 37918}, {f: 5, c: 37921}, {f: 5, c: 37929}, {f: 3, c: 37935},
    37940, {f: 2, c: 37942}, 37945, {f: 3, c: 37947}, {f: 4, c: 37952},
    {f: 5, c: 37957}, 37963, {f: 5, c: 37965}, 37971, {f: 11, c: 37973},
    {f: 2, c: 37985}, 37988, {f: 5, c: 37990}, 37996, {f: 2, c: 37998}, 38001,
    {f: 4, c: 38003}, 38008, {f: 2, c: 38010}, {f: 5, c: 38016}, 38033, 38038,
    38040, 38087, 38095, {f: 2, c: 38099}, 38106, 38118, 38139, 38172, 38176,
    38183, 38195, 38205, 38211, 38216, 38219, 38229, 38234, 38240, 38254,
    {f: 2, c: 38260}, {f: 7, c: 38264}, 38273, {f: 2, c: 38276},
    {f: 2, c: 38279}, 38282, 38285, 38288, 38290, {f: 3, c: 38293},
    {f: 8, c: 38297}, 38306, {f: 2, c: 38310}, 38314, {f: 4, c: 38318},
    {f: 3, c: 38323}, {f: 2, c: 38327}, 38330, {f: 3, c: 38336},
    {f: 2, c: 38340}, 38343, 38345, {f: 3, c: 38349}, {f: 3, c: 38353},
    {f: 5, c: 38359}, 38365, {f: 2, c: 38367}, {f: 2, c: 38371},
    {f: 2, c: 38374}, 38380, 38399, 38407, 38419, 38424, 38427, 38430, 38432,
    {f: 7, c: 38435}, {f: 3, c: 38443}, {f: 2, c: 38447}, {f: 4, c: 38455},
    38462, 38465, 38467, 38474, {f: 2, c: 38478}, {f: 3, c: 38481},
    {f: 2, c: 38486}, {f: 2, c: 38489}, 38492, 38494, 38496, {f: 2, c: 38501},
    38507, {f: 3, c: 38509}, 38513, {f: 4, c: 38521}, {f: 7, c: 38526}, 38535,
    38537, 38540, {f: 3, c: 38545}, 38550, 38554, {f: 10, c: 38557}, 38569,
    {f: 5, c: 38571}, 38578, 38581, 38583, 38586, 38591, {f: 2, c: 38594},
    38600, {f: 2, c: 38602}, {f: 2, c: 38608}, {f: 2, c: 38611},
    {f: 2, c: 38615}, 38618, {f: 3, c: 38621}, 38625, {f: 4, c: 38628},
    {f: 4, c: 38635}, {f: 2, c: 38640}, {f: 2, c: 38644}, 38648, 38650,
    {f: 2, c: 38652}, 38655, {f: 2, c: 38658}, 38661, {f: 3, c: 38666},
    {f: 3, c: 38672}, {f: 2, c: 38676}, {f: 5, c: 38679}, 38685,
    {f: 8, c: 38687}, {f: 2, c: 38696}, {f: 2, c: 38699}, {f: 2, c: 38702},
    38705, {f: 5, c: 38707}, {f: 3, c: 38714}, {f: 3, c: 38719}, 38723,
    {f: 3, c: 38725}, {f: 8, c: 38729}, [12205, 38737], {f: 2, c: 38740},
    {f: 2, c: 38743}, {f: 2, c: 38748}, 38751, {f: 2, c: 38755},
    {f: 2, c: 38758}, {f: 9, c: 38762}, 38773, {f: 5, c: 38775},
    {f: 8, c: 38781}, {f: 5, c: 38790}, 38796, 38798, 38800, 38803,
    {f: 3, c: 38805}, {f: 7, c: 38809}, {f: 2, c: 38817}, {f: 2, c: 38820},
    {f: 4, c: 38823}, 38828, 38830, {f: 2, c: 38832}, 38835, {f: 8, c: 38837},
    {f: 5, c: 38846}, {f: 2, c: 38852}, {f: 2, c: 38855}, 38858,
    {f: 6, c: 38861}, {f: 5, c: 38868}, {f: 2, c: 38874}, 38877,
    {f: 7, c: 38879}, 38888, {f: 5, c: 38894}, 38900, {f: 8, c: 38903}, 38912,
    38916, 38921, 38923, 38925, {f: 3, c: 38932}, {f: 3, c: 38937},
    {f: 4, c: 38941}, {f: 2, c: 38946}, 38949, {f: 6, c: 38951},
    {f: 2, c: 38958}, {f: 6, c: 38961}, {f: 2, c: 38969}, 38972,
    {f: 8, c: 38974}, {f: 5, c: 38983}, {f: 4, c: 38991}, {f: 3, c: 38997},
    39002, {f: 2, c: 39004}, {f: 3, c: 39007}, {f: 2, c: 39011}, 39014,
    {f: 3, c: 39016}, {f: 2, c: 39021}, 39026, 39051, 39054, 39058, 39061,
    39065, 39075, {f: 5, c: 39081}, 39088, 39090, {f: 2, c: 39092},
    {f: 5, c: 39095}, {f: 7, c: 39101}, 39109, 39111, {f: 5, c: 39113},
    {f: 2, c: 39119}, 39124, {f: 2, c: 39126}, {f: 2, c: 39132}, 39137,
    {f: 4, c: 39139}, 39148, 39150, {f: 2, c: 39152}, 39155, {f: 7, c: 39157},
    {f: 4, c: 39167}, 39172, {f: 3, c: 39174}, 39179, {f: 2, c: 39182},
    {f: 4, c: 39188}, {f: 2, c: 39193}, {f: 2, c: 39196}, {f: 2, c: 39199},
    {f: 6, c: 39202}, {f: 5, c: 39209}, {f: 4, c: 39215}, {f: 3, c: 39220},
    {f: 4, c: 39224}, 39229, {f: 3, c: 39232}, 39236, {f: 2, c: 39238},
    {f: 4, c: 39245}, 39251, 39254, {f: 4, c: 39256}, 39261, {f: 3, c: 39263},
    39268, 39270, 39283, {f: 2, c: 39288}, 39291, 39294, {f: 2, c: 39298},
    39305, 39308, 39310, {f: 11, c: 39322}, {f: 2, c: 39334}, {f: 3, c: 39337},
    {f: 2, c: 39343}, 39346, {f: 12, c: 39349}, {f: 14, c: 39362}, 39379,
    {f: 2, c: 39382}, 39386, 39388, 39390, 39392, {f: 10, c: 39395},
    {f: 3, c: 39406}, {f: 13, c: 39410}, 39424, {f: 3, c: 39426},
    {f: 7, c: 39430}, {f: 6, c: 39440}, {f: 2, c: 39447}, {f: 17, c: 39450},
    39468, 39471, {f: 5, c: 39473}, {f: 5, c: 39481}, 39487, {f: 4, c: 39494},
    {f: 2, c: 39499}, 39502, {f: 5, c: 39504}, 39510, {f: 2, c: 39512},
    {f: 3, c: 39516}, {f: 2, c: 39520}, 39523, {f: 4, c: 39526}, 39531, 39538,
    39555, 39561, {f: 2, c: 39565}, {f: 2, c: 39572}, 39577, 39590,
    {f: 6, c: 39593}, {f: 4, c: 39602}, 39609, 39611, {f: 3, c: 39613},
    {f: 2, c: 39619}, {f: 5, c: 39622}, {f: 2, c: 39629}, 39632, 39639,
    {f: 6, c: 39641}, 39648, {f: 4, c: 39650}, {f: 4, c: 39655}, 39660,
    {f: 9, c: 39664}, 39674, {f: 7, c: 39676}, {f: 2, c: 39684}, 39687,
    {f: 4, c: 39689}, 39694, {f: 3, c: 39696}, {f: 6, c: 39700},
    {f: 4, c: 39707}, {f: 2, c: 39712}, 39716, 39718, 39720, {f: 4, c: 39722},
    39728, {f: 8, c: 39731}, {f: 4, c: 39741}, 39750, {f: 3, c: 39754}, 39760,
    {f: 2, c: 39762}, {f: 3, c: 39765}, 39769, {f: 20, c: 39771},
    {f: 4, c: 39792}, {f: 2, c: 39797}, {f: 9, c: 39800}, 39810,
    {f: 10, c: 39812}, 39823, {f: 7, c: 39827}, {f: 2, c: 39835},
    {f: 11, c: 39839}, 39852, {f: 17, c: 39855}, {f: 5, c: 39874}, 39880,
    {f: 9, c: 39883}, 39893, {f: 4, c: 39895}, 39900, {f: 3, c: 39902}, 39907,
    {f: 2, c: 39909}, 39913, {f: 4, c: 39916}, {f: 3, c: 39921},
    {f: 8, c: 39925}, 39934, {f: 8, c: 39936}, {f: 3, c: 39946},
    {f: 2, c: 39950}, 39953, {f: 12, c: 39956}, {f: 2, c: 39969}, 39972,
    {f: 2, c: 39974}, {f: 3, c: 39978}, {f: 3, c: 39982}, 39988, 39990, 39992,
    39994, {f: 2, c: 39996}, {f: 6, c: 39999}, {f: 2, c: 40006},
    {f: 8, c: 40010}, 40019, 40021, {f: 4, c: 40025}, 40030, {f: 7, c: 40032},
    {f: 5, c: 40040}, {f: 10, c: 40046}, 40057, 40059, {f: 2, c: 40061}, 40064,
    {f: 2, c: 40067}, {f: 2, c: 40073}, 40076, 40079, 40083, {f: 4, c: 40086},
    40093, 40106, 40108, 40111, 40121, {f: 5, c: 40126}, {f: 2, c: 40136},
    {f: 2, c: 40145}, {f: 2, c: 40154}, {f: 2, c: 40160}, {f: 2, c: 40163},
    {f: 3, c: 40166}, {f: 2, c: 40170}, {f: 6, c: 40173}, 40181,
    {f: 15, c: 40183}, 40200, {f: 11, c: 40202}, {f: 5, c: 40214}, 40220,
    40222, {f: 3, c: 40224}, {f: 2, c: 40228}, 40231, {f: 6, c: 40233},
    {f: 10, c: 40241}, {f: 3, c: 40252}, {f: 2, c: 40256}, {f: 14, c: 40259},
    {f: 8, c: 40276}, {f: 2, c: 40286}, {f: 8, c: 40290}, 40299,
    {f: 2, c: 40301}, {f: 2, c: 40304}, {f: 20, c: 40307}, 40328,
    {f: 9, c: 40330}, {f: 4, c: 40340}, 40345, {f: 10, c: 40347},
    {f: 3, c: 40358}, {f: 5, c: 40362}, {f: 4, c: 40368}, {f: 6, c: 40373},
    {f: 3, c: 40381}, 40385, 40387, {f: 14, c: 40389}, {f: 3, c: 40404}, 40408,
    {f: 10, c: 40411}, {f: 8, c: 40423}, {f: 2, c: 40432}, {f: 4, c: 40436},
    {f: 17, c: 40443}, {f: 8, c: 40461}, {f: 4, c: 40470}, 40476, 40484, 40487,
    40494, 40496, 40500, {f: 2, c: 40507}, 40512, 40525, 40528,
    {f: 3, c: 40530}, 40534, 40537, 40541, {f: 4, c: 40543}, 40549,
    {f: 2, c: 40558}, 40562, 40564, {f: 3, c: 40566}, 40571, {f: 2, c: 40576},
    {f: 4, c: 40579}, {f: 2, c: 40585}, {f: 6, c: 40588}, {f: 3, c: 40596},
    {f: 5, c: 40600}, 40606, {f: 5, c: 40608}, {f: 2, c: 40615},
    {f: 5, c: 40618}, {f: 4, c: 40624}, {f: 2, c: 40630}, {f: 2, c: 40633},
    40636, {f: 4, c: 40639}, [12232, 40643], {f: 4, c: 40645},
    {f: 2, c: 40650}, 40656, {f: 2, c: 40658}, {f: 3, c: 40661},
    {f: 2, c: 40665}, 40673, {f: 2, c: 40675}, 40678, {f: 4, c: 40683},
    {f: 2, c: 40688}, 40691, {f: 2, c: 40693}, 40696, 40698, {f: 9, c: 40704},
    40714, 40716, 40719, {f: 2, c: 40721}, 40724, 40726, 40728,
    {f: 6, c: 40730}, 40737, {f: 9, c: 40739}, {f: 2, c: 40749},
    {f: 7, c: 40752}, 40760, 40762, 40764, {f: 5, c: 40767}, {f: 5, c: 40773},
    {f: 3, c: 40780}, 40787, {f: 4, c: 40789}, {f: 2, c: 40794},
    {f: 2, c: 40797}, 40802, {f: 2, c: 40804}, {f: 3, c: 40807}, 40811,
    {f: 5, c: 40813}, {f: 4, c: 40819}, {f: 7, c: 40824}, {f: 2, c: 40833},
    {f: 2, c: 40846}, {f: 3, c: 40849}, {f: 3, c: 40854}, {f: 2, c: 40861},
    {f: 5, c: 40865}, 63788, {f: 3, c: 64013}, 64017, {f: 2, c: 64019}, 64024,
    {f: 3, c: 64031}, {f: 2, c: 64035}, {f: 3, c: 64039}, 11905,
    [59414, 131207], [59415, 131209], [59416, 131276], 11908, 13427, 13383,
    11912, 11915, 59422, 13726, 13850, 13838, 11916, 11927, 14702, 14616,
    59430, 14799, 14815, 14963, 14800, {f: 2, c: 59435}, 15182, 15470, 15584,
    11943, [59441, 136663], 59442, 11946, 16470, 16735, 11950, 17207, 11955,
    {f: 2, c: 11958}, [59451, 141711], 17329, 17324, 11963, 17373, 17622,
    18017, 17996, [59459, 132361], 18211, 18217, 18300, 18317, 11978, 18759,
    18810, 18813, {f: 2, c: 18818}, {f: 2, c: 18821}, 18847, 18843, 18871,
    18870, [59476, 133533], [59477, 147966], 19619, {f: 3, c: 19615}, 19575,
    19618, {f: 7, c: 19731}, 19886, 59492, {s: 226}, 8364, 165, 0, 0, 12351,
    {s: 17}, 12436, {s: 14}, 12535, 12537, 12536, 12538, 0, {f: 3, c: 12339},
    {f: 3, c: 12344}, {f: 3, c: 12586}, {f: 24, c: 12704}, 11904,
    {f: 2, c: 11906}, {f: 3, c: 11909}, {f: 2, c: 11913}, {f: 10, c: 11917},
    {f: 2, c: 11928}, {f: 12, c: 11931}, {f: 2, c: 11944}, {f: 3, c: 11947},
    {f: 4, c: 11951}, {f: 2, c: 11956}, {f: 3, c: 11960}, {f: 14, c: 11964},
    {f: 41, c: 11979}, {f: 71, c: 13312}, {f: 43, c: 13384},
    {f: 298, c: 13428}, {f: 111, c: 13727}, {f: 11, c: 13839},
    {f: 765, c: 13851}, {f: 85, c: 14617}, {f: 96, c: 14703},
    {f: 14, c: 14801}, {f: 147, c: 14816}, {f: 218, c: 14964},
    {f: 287, c: 15183}, {f: 113, c: 15471}, {f: 885, c: 15585},
    {f: 264, c: 16471}, {f: 471, c: 16736}, {f: 116, c: 17208},
    {f: 4, c: 17325}, {f: 43, c: 17330}, {f: 248, c: 17374},
    {f: 373, c: 17623}, {f: 20, c: 17997}, {f: 193, c: 18018},
    {f: 5, c: 18212}, {f: 82, c: 18218}, {f: 16, c: 18301}, {f: 441, c: 18318},
    {f: 50, c: 18760}, {f: 2, c: 18811}, {f: 4, c: 18814}, 18820,
    {f: 20, c: 18823}, {f: 3, c: 18844}, {f: 22, c: 18848}, {f: 703, c: 18872},
    {f: 39, c: 19576}, {f: 111, c: 19620}, {f: 148, c: 19738},
    {f: 7, c: 19887}]
};



var PDFFunction = (function PDFFunctionClosure() {
  var CONSTRUCT_SAMPLED = 0;
  var CONSTRUCT_INTERPOLATED = 2;
  var CONSTRUCT_STICHED = 3;
  var CONSTRUCT_POSTSCRIPT = 4;

  return {
    getSampleArray: function PDFFunction_getSampleArray(size, outputSize, bps,
                                                       str) {
      var i, ii;
      var length = 1;
      for (i = 0, ii = size.length; i < ii; i++) {
        length *= size[i];
      }
      length *= outputSize;

      var array = [];
      var codeSize = 0;
      var codeBuf = 0;
      // 32 is a valid bps so shifting won't work
      var sampleMul = 1.0 / (Math.pow(2.0, bps) - 1);

      var strBytes = str.getBytes((length * bps + 7) / 8);
      var strIdx = 0;
      for (i = 0; i < length; i++) {
        while (codeSize < bps) {
          codeBuf <<= 8;
          codeBuf |= strBytes[strIdx++];
          codeSize += 8;
        }
        codeSize -= bps;
        array.push((codeBuf >> codeSize) * sampleMul);
        codeBuf &= (1 << codeSize) - 1;
      }
      return array;
    },

    getIR: function PDFFunction_getIR(xref, fn) {
      var dict = fn.dict;
      if (!dict) {
        dict = fn;
      }

      var types = [this.constructSampled,
                   null,
                   this.constructInterpolated,
                   this.constructStiched,
                   this.constructPostScript];

      var typeNum = dict.get('FunctionType');
      var typeFn = types[typeNum];
      if (!typeFn) {
        error('Unknown type of function');
      }

      return typeFn.call(this, fn, dict, xref);
    },

    fromIR: function PDFFunction_fromIR(IR) {
      var type = IR[0];
      switch (type) {
        case CONSTRUCT_SAMPLED:
          return this.constructSampledFromIR(IR);
        case CONSTRUCT_INTERPOLATED:
          return this.constructInterpolatedFromIR(IR);
        case CONSTRUCT_STICHED:
          return this.constructStichedFromIR(IR);
        //case CONSTRUCT_POSTSCRIPT:
        default:
          return this.constructPostScriptFromIR(IR);
      }
    },

    parse: function PDFFunction_parse(xref, fn) {
      var IR = this.getIR(xref, fn);
      return this.fromIR(IR);
    },

    constructSampled: function PDFFunction_constructSampled(str, dict) {
      function toMultiArray(arr) {
        var inputLength = arr.length;
        var out = [];
        var index = 0;
        for (var i = 0; i < inputLength; i += 2) {
          out[index] = [arr[i], arr[i + 1]];
          ++index;
        }
        return out;
      }
      var domain = dict.get('Domain');
      var range = dict.get('Range');

      if (!domain || !range) {
        error('No domain or range');
      }

      var inputSize = domain.length / 2;
      var outputSize = range.length / 2;

      domain = toMultiArray(domain);
      range = toMultiArray(range);

      var size = dict.get('Size');
      var bps = dict.get('BitsPerSample');
      var order = dict.get('Order') || 1;
      if (order !== 1) {
        // No description how cubic spline interpolation works in PDF32000:2008
        // As in poppler, ignoring order, linear interpolation may work as good
        info('No support for cubic spline interpolation: ' + order);
      }

      var encode = dict.get('Encode');
      if (!encode) {
        encode = [];
        for (var i = 0; i < inputSize; ++i) {
          encode.push(0);
          encode.push(size[i] - 1);
        }
      }
      encode = toMultiArray(encode);

      var decode = dict.get('Decode');
      if (!decode) {
        decode = range;
      } else {
        decode = toMultiArray(decode);
      }

      var samples = this.getSampleArray(size, outputSize, bps, str);

      return [
        CONSTRUCT_SAMPLED, inputSize, domain, encode, decode, samples, size,
        outputSize, Math.pow(2, bps) - 1, range
      ];
    },

    constructSampledFromIR: function PDFFunction_constructSampledFromIR(IR) {
      // See chapter 3, page 109 of the PDF reference
      function interpolate(x, xmin, xmax, ymin, ymax) {
        return ymin + ((x - xmin) * ((ymax - ymin) / (xmax - xmin)));
      }

      return function constructSampledFromIRResult(args) {
        // See chapter 3, page 110 of the PDF reference.
        var m = IR[1];
        var domain = IR[2];
        var encode = IR[3];
        var decode = IR[4];
        var samples = IR[5];
        var size = IR[6];
        var n = IR[7];
        //var mask = IR[8];
        var range = IR[9];

        if (m != args.length) {
          error('Incorrect number of arguments: ' + m + ' != ' +
                args.length);
        }

        var x = args;

        // Building the cube vertices: its part and sample index
        // http://rjwagner49.com/Mathematics/Interpolation.pdf
        var cubeVertices = 1 << m;
        var cubeN = new Float64Array(cubeVertices);
        var cubeVertex = new Uint32Array(cubeVertices);
        var i, j;
        for (j = 0; j < cubeVertices; j++) {
          cubeN[j] = 1;
        }

        var k = n, pos = 1;
        // Map x_i to y_j for 0 <= i < m using the sampled function.
        for (i = 0; i < m; ++i) {
          // x_i' = min(max(x_i, Domain_2i), Domain_2i+1)
          var domain_2i = domain[i][0];
          var domain_2i_1 = domain[i][1];
          var xi = Math.min(Math.max(x[i], domain_2i), domain_2i_1);

          // e_i = Interpolate(x_i', Domain_2i, Domain_2i+1,
          //                   Encode_2i, Encode_2i+1)
          var e = interpolate(xi, domain_2i, domain_2i_1,
                              encode[i][0], encode[i][1]);

          // e_i' = min(max(e_i, 0), Size_i - 1)
          var size_i = size[i];
          e = Math.min(Math.max(e, 0), size_i - 1);

          // Adjusting the cube: N and vertex sample index
          var e0 = e < size_i - 1 ? Math.floor(e) : e - 1; // e1 = e0 + 1;
          var n0 = e0 + 1 - e; // (e1 - e) / (e1 - e0);
          var n1 = e - e0; // (e - e0) / (e1 - e0);
          var offset0 = e0 * k;
          var offset1 = offset0 + k; // e1 * k
          for (j = 0; j < cubeVertices; j++) {
            if (j & pos) {
              cubeN[j] *= n1;
              cubeVertex[j] += offset1;
            } else {
              cubeN[j] *= n0;
              cubeVertex[j] += offset0;
            }
          }

          k *= size_i;
          pos <<= 1;
        }

        var y = new Float64Array(n);
        for (j = 0; j < n; ++j) {
          // Sum all cube vertices' samples portions
          var rj = 0;
          for (i = 0; i < cubeVertices; i++) {
            rj += samples[cubeVertex[i] + j] * cubeN[i];
          }

          // r_j' = Interpolate(r_j, 0, 2^BitsPerSample - 1,
          //                    Decode_2j, Decode_2j+1)
          rj = interpolate(rj, 0, 1, decode[j][0], decode[j][1]);

          // y_j = min(max(r_j, range_2j), range_2j+1)
          y[j] = Math.min(Math.max(rj, range[j][0]), range[j][1]);
        }

        return y;
      };
    },

    constructInterpolated: function PDFFunction_constructInterpolated(str,
                                                                      dict) {
      var c0 = dict.get('C0') || [0];
      var c1 = dict.get('C1') || [1];
      var n = dict.get('N');

      if (!isArray(c0) || !isArray(c1)) {
        error('Illegal dictionary for interpolated function');
      }

      var length = c0.length;
      var diff = [];
      for (var i = 0; i < length; ++i) {
        diff.push(c1[i] - c0[i]);
      }

      return [CONSTRUCT_INTERPOLATED, c0, diff, n];
    },

    constructInterpolatedFromIR:
      function PDFFunction_constructInterpolatedFromIR(IR) {
      var c0 = IR[1];
      var diff = IR[2];
      var n = IR[3];

      var length = diff.length;

      return function constructInterpolatedFromIRResult(args) {
        var x = n == 1 ? args[0] : Math.pow(args[0], n);

        var out = [];
        for (var j = 0; j < length; ++j) {
          out.push(c0[j] + (x * diff[j]));
        }

        return out;

      };
    },

    constructStiched: function PDFFunction_constructStiched(fn, dict, xref) {
      var domain = dict.get('Domain');

      if (!domain) {
        error('No domain');
      }

      var inputSize = domain.length / 2;
      if (inputSize != 1) {
        error('Bad domain for stiched function');
      }

      var fnRefs = dict.get('Functions');
      var fns = [];
      for (var i = 0, ii = fnRefs.length; i < ii; ++i) {
        fns.push(PDFFunction.getIR(xref, xref.fetchIfRef(fnRefs[i])));
      }

      var bounds = dict.get('Bounds');
      var encode = dict.get('Encode');

      return [CONSTRUCT_STICHED, domain, bounds, encode, fns];
    },

    constructStichedFromIR: function PDFFunction_constructStichedFromIR(IR) {
      var domain = IR[1];
      var bounds = IR[2];
      var encode = IR[3];
      var fnsIR = IR[4];
      var fns = [];

      for (var i = 0, ii = fnsIR.length; i < ii; i++) {
        fns.push(PDFFunction.fromIR(fnsIR[i]));
      }

      return function constructStichedFromIRResult(args) {
        var clip = function constructStichedFromIRClip(v, min, max) {
          if (v > max) {
            v = max;
          } else if (v < min) {
            v = min;
          }
          return v;
        };

        // clip to domain
        var v = clip(args[0], domain[0], domain[1]);
        // calulate which bound the value is in
        for (var i = 0, ii = bounds.length; i < ii; ++i) {
          if (v < bounds[i]) {
            break;
          }
        }

        // encode value into domain of function
        var dmin = domain[0];
        if (i > 0) {
          dmin = bounds[i - 1];
        }
        var dmax = domain[1];
        if (i < bounds.length) {
          dmax = bounds[i];
        }

        var rmin = encode[2 * i];
        var rmax = encode[2 * i + 1];

        var v2 = rmin + (v - dmin) * (rmax - rmin) / (dmax - dmin);

        // call the appropriate function
        return fns[i]([v2]);
      };
    },

    constructPostScript: function PDFFunction_constructPostScript(fn, dict,
                                                                  xref) {
      var domain = dict.get('Domain');
      var range = dict.get('Range');

      if (!domain) {
        error('No domain.');
      }

      if (!range) {
        error('No range.');
      }

      var lexer = new PostScriptLexer(fn);
      var parser = new PostScriptParser(lexer);
      var code = parser.parse();

      return [CONSTRUCT_POSTSCRIPT, domain, range, code];
    },

    constructPostScriptFromIR: function PDFFunction_constructPostScriptFromIR(
                                          IR) {
      var domain = IR[1];
      var range = IR[2];
      var code = IR[3];
      var numOutputs = range.length >> 1;
      var numInputs = domain.length >> 1;
      var evaluator = new PostScriptEvaluator(code);
      // Cache the values for a big speed up, the cache size is limited though
      // since the number of possible values can be huge from a PS function.
      var cache = {};
      // The MAX_CACHE_SIZE is set to ~4x the maximum number of distinct values
      // seen in our tests.
      var MAX_CACHE_SIZE = 2048 * 4;
      var cache_available = MAX_CACHE_SIZE;
      return function constructPostScriptFromIRResult(args) {
        var i, value;
        var key = '';
        var input = new Array(numInputs);
        for (i = 0; i < numInputs; i++) {
          value = args[i];
          input[i] = value;
          key += value + '_';
        }

        var cachedValue = cache[key];
        if (cachedValue !== undefined) {
          return cachedValue;
        }

        var output = new Array(numOutputs);
        var stack = evaluator.execute(input);
        var stackIndex = stack.length - numOutputs;
        for (i = 0; i < numOutputs; i++) {
          value = stack[stackIndex + i];
          var bound = range[i * 2];
          if (value < bound) {
            value = bound;
          } else {
            bound = range[i * 2 +1];
            if (value > bound) {
              value = bound;
            }
          }
          output[i] = value;
        }
        if (cache_available > 0) {
          cache_available--;
          cache[key] = output;
        }
        return output;
      };
    }
  };
})();

function isPDFFunction(v) {
  var fnDict;
  if (typeof v != 'object') {
    return false;
  } else if (isDict(v)) {
    fnDict = v;
  } else if (isStream(v)) {
    fnDict = v.dict;
  } else {
    return false;
  }
  return fnDict.has('FunctionType');
}

var PostScriptStack = (function PostScriptStackClosure() {
  var MAX_STACK_SIZE = 100;
  function PostScriptStack(initialStack) {
    this.stack = initialStack || [];
  }

  PostScriptStack.prototype = {
    push: function PostScriptStack_push(value) {
      if (this.stack.length >= MAX_STACK_SIZE) {
        error('PostScript function stack overflow.');
      }
      this.stack.push(value);
    },
    pop: function PostScriptStack_pop() {
      if (this.stack.length <= 0) {
        error('PostScript function stack underflow.');
      }
      return this.stack.pop();
    },
    copy: function PostScriptStack_copy(n) {
      if (this.stack.length + n >= MAX_STACK_SIZE) {
        error('PostScript function stack overflow.');
      }
      var stack = this.stack;
      for (var i = stack.length - n, j = n - 1; j >= 0; j--, i++) {
        stack.push(stack[i]);
      }
    },
    index: function PostScriptStack_index(n) {
      this.push(this.stack[this.stack.length - n - 1]);
    },
    // rotate the last n stack elements p times
    roll: function PostScriptStack_roll(n, p) {
      var stack = this.stack;
      var l = stack.length - n;
      var r = stack.length - 1, c = l + (p - Math.floor(p / n) * n), i, j, t;
      for (i = l, j = r; i < j; i++, j--) {
        t = stack[i]; stack[i] = stack[j]; stack[j] = t;
      }
      for (i = l, j = c - 1; i < j; i++, j--) {
        t = stack[i]; stack[i] = stack[j]; stack[j] = t;
      }
      for (i = c, j = r; i < j; i++, j--) {
        t = stack[i]; stack[i] = stack[j]; stack[j] = t;
      }
    }
  };
  return PostScriptStack;
})();
var PostScriptEvaluator = (function PostScriptEvaluatorClosure() {
  function PostScriptEvaluator(operators) {
    this.operators = operators;
  }
  PostScriptEvaluator.prototype = {
    execute: function PostScriptEvaluator_execute(initialStack) {
      var stack = new PostScriptStack(initialStack);
      var counter = 0;
      var operators = this.operators;
      var length = operators.length;
      var operator, a, b;
      while (counter < length) {
        operator = operators[counter++];
        if (typeof operator == 'number') {
          // Operator is really an operand and should be pushed to the stack.
          stack.push(operator);
          continue;
        }
        switch (operator) {
          // non standard ps operators
          case 'jz': // jump if false
            b = stack.pop();
            a = stack.pop();
            if (!a) {
              counter = b;
            }
            break;
          case 'j': // jump
            a = stack.pop();
            counter = a;
            break;

          // all ps operators in alphabetical order (excluding if/ifelse)
          case 'abs':
            a = stack.pop();
            stack.push(Math.abs(a));
            break;
          case 'add':
            b = stack.pop();
            a = stack.pop();
            stack.push(a + b);
            break;
          case 'and':
            b = stack.pop();
            a = stack.pop();
            if (isBool(a) && isBool(b)) {
              stack.push(a && b);
            } else {
              stack.push(a & b);
            }
            break;
          case 'atan':
            a = stack.pop();
            stack.push(Math.atan(a));
            break;
          case 'bitshift':
            b = stack.pop();
            a = stack.pop();
            if (a > 0) {
              stack.push(a << b);
            } else {
              stack.push(a >> b);
            }
            break;
          case 'ceiling':
            a = stack.pop();
            stack.push(Math.ceil(a));
            break;
          case 'copy':
            a = stack.pop();
            stack.copy(a);
            break;
          case 'cos':
            a = stack.pop();
            stack.push(Math.cos(a));
            break;
          case 'cvi':
            a = stack.pop() | 0;
            stack.push(a);
            break;
          case 'cvr':
            // noop
            break;
          case 'div':
            b = stack.pop();
            a = stack.pop();
            stack.push(a / b);
            break;
          case 'dup':
            stack.copy(1);
            break;
          case 'eq':
            b = stack.pop();
            a = stack.pop();
            stack.push(a == b);
            break;
          case 'exch':
            stack.roll(2, 1);
            break;
          case 'exp':
            b = stack.pop();
            a = stack.pop();
            stack.push(Math.pow(a, b));
            break;
          case 'false':
            stack.push(false);
            break;
          case 'floor':
            a = stack.pop();
            stack.push(Math.floor(a));
            break;
          case 'ge':
            b = stack.pop();
            a = stack.pop();
            stack.push(a >= b);
            break;
          case 'gt':
            b = stack.pop();
            a = stack.pop();
            stack.push(a > b);
            break;
          case 'idiv':
            b = stack.pop();
            a = stack.pop();
            stack.push((a / b) | 0);
            break;
          case 'index':
            a = stack.pop();
            stack.index(a);
            break;
          case 'le':
            b = stack.pop();
            a = stack.pop();
            stack.push(a <= b);
            break;
          case 'ln':
            a = stack.pop();
            stack.push(Math.log(a));
            break;
          case 'log':
            a = stack.pop();
            stack.push(Math.log(a) / Math.LN10);
            break;
          case 'lt':
            b = stack.pop();
            a = stack.pop();
            stack.push(a < b);
            break;
          case 'mod':
            b = stack.pop();
            a = stack.pop();
            stack.push(a % b);
            break;
          case 'mul':
            b = stack.pop();
            a = stack.pop();
            stack.push(a * b);
            break;
          case 'ne':
            b = stack.pop();
            a = stack.pop();
            stack.push(a != b);
            break;
          case 'neg':
            a = stack.pop();
            stack.push(-a);
            break;
          case 'not':
            a = stack.pop();
            if (isBool(a)) {
              stack.push(!a);
            } else {
              stack.push(~a);
            }
            break;
          case 'or':
            b = stack.pop();
            a = stack.pop();
            if (isBool(a) && isBool(b)) {
              stack.push(a || b);
            } else {
              stack.push(a | b);
            }
            break;
          case 'pop':
            stack.pop();
            break;
          case 'roll':
            b = stack.pop();
            a = stack.pop();
            stack.roll(a, b);
            break;
          case 'round':
            a = stack.pop();
            stack.push(Math.round(a));
            break;
          case 'sin':
            a = stack.pop();
            stack.push(Math.sin(a));
            break;
          case 'sqrt':
            a = stack.pop();
            stack.push(Math.sqrt(a));
            break;
          case 'sub':
            b = stack.pop();
            a = stack.pop();
            stack.push(a - b);
            break;
          case 'true':
            stack.push(true);
            break;
          case 'truncate':
            a = stack.pop();
            a = a < 0 ? Math.ceil(a) : Math.floor(a);
            stack.push(a);
            break;
          case 'xor':
            b = stack.pop();
            a = stack.pop();
            if (isBool(a) && isBool(b)) {
              stack.push(a != b);
            } else {
              stack.push(a ^ b);
            }
            break;
          default:
            error('Unknown operator ' + operator);
            break;
        }
      }
      return stack.stack;
    }
  };
  return PostScriptEvaluator;
})();


var ColorSpace = (function ColorSpaceClosure() {
  // Constructor should define this.numComps, this.defaultColor, this.name
  function ColorSpace() {
    error('should not call ColorSpace constructor');
  }

  ColorSpace.prototype = {
    /**
     * Converts the color value to the RGB color. The color components are
     * located in the src array starting from the srcOffset. Returns the array
     * of the rgb components, each value ranging from [0,255].
     */
    getRgb: function ColorSpace_getRgb(src, srcOffset) {
      var rgb = new Uint8Array(3);
      this.getRgbItem(src, srcOffset, rgb, 0);
      return rgb;
    },
    /**
     * Converts the color value to the RGB color, similar to the getRgb method.
     * The result placed into the dest array starting from the destOffset.
     */
    getRgbItem: function ColorSpace_getRgbItem(src, srcOffset,
                                               dest, destOffset) {
      error('Should not call ColorSpace.getRgbItem');
    },
    /**
     * Converts the specified number of the color values to the RGB colors.
     * The colors are located in the src array starting from the srcOffset.
     * The result is placed into the dest array starting from the destOffset.
     * The src array items shall be in [0,2^bits) range, the dest array items
     * will be in [0,255] range. alpha01 indicates how many alpha components
     * there are in the dest array; it will be either 0 (RGB array) or 1 (RGBA
     * array).
     */
    getRgbBuffer: function ColorSpace_getRgbBuffer(src, srcOffset, count,
                                                   dest, destOffset, bits,
                                                   alpha01) {
      error('Should not call ColorSpace.getRgbBuffer');
    },
    /**
     * Determines the number of bytes required to store the result of the
     * conversion done by the getRgbBuffer method. As in getRgbBuffer,
     * |alpha01| is either 0 (RGB output) or 1 (RGBA output).
     */
    getOutputLength: function ColorSpace_getOutputLength(inputLength,
                                                         alpha01) {
      error('Should not call ColorSpace.getOutputLength');
    },
    /**
     * Returns true if source data will be equal the result/output data.
     */
    isPassthrough: function ColorSpace_isPassthrough(bits) {
      return false;
    },
    /**
     * Fills in the RGB colors in the destination buffer.  alpha01 indicates
     * how many alpha components there are in the dest array; it will be either
     * 0 (RGB array) or 1 (RGBA array).
     */
    fillRgb: function ColorSpace_fillRgb(dest, originalWidth,
                                         originalHeight, width, height,
                                         actualHeight, bpc, comps, alpha01) {
      var count = originalWidth * originalHeight;
      var rgbBuf = null;
      var numComponentColors = 1 << bpc;
      var needsResizing = originalHeight != height || originalWidth != width;
      var i, ii;

      if (this.isPassthrough(bpc)) {
        rgbBuf = comps;
      } else if (this.numComps === 1 && count > numComponentColors &&
          this.name !== 'DeviceGray' && this.name !== 'DeviceRGB') {
        // Optimization: create a color map when there is just one component and
        // we are converting more colors than the size of the color map. We
        // don't build the map if the colorspace is gray or rgb since those
        // methods are faster than building a map. This mainly offers big speed
        // ups for indexed and alternate colorspaces.
        //
        // TODO it may be worth while to cache the color map. While running
        // testing I never hit a cache so I will leave that out for now (perhaps
        // we are reparsing colorspaces too much?).
        var allColors = bpc <= 8 ? new Uint8Array(numComponentColors) :
                                   new Uint16Array(numComponentColors);
        var key;
        for (i = 0; i < numComponentColors; i++) {
          allColors[i] = i;
        }
        var colorMap = new Uint8Array(numComponentColors * 3);
        this.getRgbBuffer(allColors, 0, numComponentColors, colorMap, 0, bpc,
                          /* alpha01 = */ 0);

        var destPos, rgbPos;
        if (!needsResizing) {
          // Fill in the RGB values directly into |dest|.
          destPos = 0;
          for (i = 0; i < count; ++i) {
            key = comps[i] * 3;
            dest[destPos++] = colorMap[key];
            dest[destPos++] = colorMap[key + 1];
            dest[destPos++] = colorMap[key + 2];
            destPos += alpha01;
          }
        } else {
          rgbBuf = new Uint8Array(count * 3);
          rgbPos = 0;
          for (i = 0; i < count; ++i) {
            key = comps[i] * 3;
            rgbBuf[rgbPos++] = colorMap[key];
            rgbBuf[rgbPos++] = colorMap[key + 1];
            rgbBuf[rgbPos++] = colorMap[key + 2];
          }
        }
      } else {
        if (!needsResizing) {
          // Fill in the RGB values directly into |dest|.
          this.getRgbBuffer(comps, 0, width * actualHeight, dest, 0, bpc,
                            alpha01);
        } else {
          rgbBuf = new Uint8Array(count * 3);
          this.getRgbBuffer(comps, 0, count, rgbBuf, 0, bpc,
                            /* alpha01 = */ 0);
        }
      }

      if (rgbBuf) {
        if (needsResizing) {
          PDFImage.resize(rgbBuf, bpc, 3, originalWidth, originalHeight, width,
                          height, dest, alpha01);
        } else {
          rgbPos = 0;
          destPos = 0;
          for (i = 0, ii = width * actualHeight; i < ii; i++) {
            dest[destPos++] = rgbBuf[rgbPos++];
            dest[destPos++] = rgbBuf[rgbPos++];
            dest[destPos++] = rgbBuf[rgbPos++];
            destPos += alpha01;
          }
        }
      }
    },
    /**
     * True if the colorspace has components in the default range of [0, 1].
     * This should be true for all colorspaces except for lab color spaces
     * which are [0,100], [-128, 127], [-128, 127].
     */
    usesZeroToOneRange: true
  };

  ColorSpace.parse = function ColorSpace_parse(cs, xref, res) {
    var IR = ColorSpace.parseToIR(cs, xref, res);
    if (IR instanceof AlternateCS) {
      return IR;
    }
    return ColorSpace.fromIR(IR);
  };

  ColorSpace.fromIR = function ColorSpace_fromIR(IR) {
    var name = isArray(IR) ? IR[0] : IR;
    var whitePoint, blackPoint;

    switch (name) {
      case 'DeviceGrayCS':
        return this.singletons.gray;
      case 'DeviceRgbCS':
        return this.singletons.rgb;
      case 'DeviceCmykCS':
        return this.singletons.cmyk;
      case 'CalGrayCS':
        whitePoint = IR[1].WhitePoint;
        blackPoint = IR[1].BlackPoint;
        var gamma = IR[1].Gamma;
        return new CalGrayCS(whitePoint, blackPoint, gamma);
      case 'PatternCS':
        var basePatternCS = IR[1];
        if (basePatternCS) {
          basePatternCS = ColorSpace.fromIR(basePatternCS);
        }
        return new PatternCS(basePatternCS);
      case 'IndexedCS':
        var baseIndexedCS = IR[1];
        var hiVal = IR[2];
        var lookup = IR[3];
        return new IndexedCS(ColorSpace.fromIR(baseIndexedCS), hiVal, lookup);
      case 'AlternateCS':
        var numComps = IR[1];
        var alt = IR[2];
        var tintFnIR = IR[3];

        return new AlternateCS(numComps, ColorSpace.fromIR(alt),
                                PDFFunction.fromIR(tintFnIR));
      case 'LabCS':
        whitePoint = IR[1].WhitePoint;
        blackPoint = IR[1].BlackPoint;
        var range = IR[1].Range;
        return new LabCS(whitePoint, blackPoint, range);
      default:
        error('Unkown name ' + name);
    }
    return null;
  };

  ColorSpace.parseToIR = function ColorSpace_parseToIR(cs, xref, res) {
    if (isName(cs)) {
      var colorSpaces = res.get('ColorSpace');
      if (isDict(colorSpaces)) {
        var refcs = colorSpaces.get(cs.name);
        if (refcs) {
          cs = refcs;
        }
      }
    }

    cs = xref.fetchIfRef(cs);
    var mode;

    if (isName(cs)) {
      mode = cs.name;
      this.mode = mode;

      switch (mode) {
        case 'DeviceGray':
        case 'G':
          return 'DeviceGrayCS';
        case 'DeviceRGB':
        case 'RGB':
          return 'DeviceRgbCS';
        case 'DeviceCMYK':
        case 'CMYK':
          return 'DeviceCmykCS';
        case 'Pattern':
          return ['PatternCS', null];
        default:
          error('unrecognized colorspace ' + mode);
      }
    } else if (isArray(cs)) {
      mode = cs[0].name;
      this.mode = mode;
      var numComps, params;

      switch (mode) {
        case 'DeviceGray':
        case 'G':
          return 'DeviceGrayCS';
        case 'DeviceRGB':
        case 'RGB':
          return 'DeviceRgbCS';
        case 'DeviceCMYK':
        case 'CMYK':
          return 'DeviceCmykCS';
        case 'CalGray':
          params = cs[1].getAll();
          return ['CalGrayCS', params];
        case 'CalRGB':
          return 'DeviceRgbCS';
        case 'ICCBased':
          var stream = xref.fetchIfRef(cs[1]);
          var dict = stream.dict;
          numComps = dict.get('N');
          if (numComps == 1) {
            return 'DeviceGrayCS';
          } else if (numComps == 3) {
            return 'DeviceRgbCS';
          } else if (numComps == 4) {
            return 'DeviceCmykCS';
          }
          break;
        case 'Pattern':
          var basePatternCS = cs[1];
          if (basePatternCS) {
            basePatternCS = ColorSpace.parseToIR(basePatternCS, xref, res);
          }
          return ['PatternCS', basePatternCS];
        case 'Indexed':
        case 'I':
          var baseIndexedCS = ColorSpace.parseToIR(cs[1], xref, res);
          var hiVal = cs[2] + 1;
          var lookup = xref.fetchIfRef(cs[3]);
          if (isStream(lookup)) {
            lookup = lookup.getBytes();
          }
          return ['IndexedCS', baseIndexedCS, hiVal, lookup];
        case 'Separation':
        case 'DeviceN':
          var name = cs[1];
          numComps = 1;
          if (isName(name)) {
            numComps = 1;
          } else if (isArray(name)) {
            numComps = name.length;
          }
          var alt = ColorSpace.parseToIR(cs[2], xref, res);
          var tintFnIR = PDFFunction.getIR(xref, xref.fetchIfRef(cs[3]));
          return ['AlternateCS', numComps, alt, tintFnIR];
        case 'Lab':
          params = cs[1].getAll();
          return ['LabCS', params];
        default:
          error('unimplemented color space object "' + mode + '"');
      }
    } else {
      error('unrecognized color space object: "' + cs + '"');
    }
    return null;
  };
  /**
   * Checks if a decode map matches the default decode map for a color space.
   * This handles the general decode maps where there are two values per
   * component. e.g. [0, 1, 0, 1, 0, 1] for a RGB color.
   * This does not handle Lab, Indexed, or Pattern decode maps since they are
   * slightly different.
   * @param {Array} decode Decode map (usually from an image).
   * @param {Number} n Number of components the color space has.
   */
  ColorSpace.isDefaultDecode = function ColorSpace_isDefaultDecode(decode, n) {
    if (!decode) {
      return true;
    }

    if (n * 2 !== decode.length) {
      warn('The decode map is not the correct length');
      return true;
    }
    for (var i = 0, ii = decode.length; i < ii; i += 2) {
      if (decode[i] !== 0 || decode[i + 1] != 1) {
        return false;
      }
    }
    return true;
  };

  ColorSpace.singletons = {
    get gray() {
      return shadow(this, 'gray', new DeviceGrayCS());
    },
    get rgb() {
      return shadow(this, 'rgb', new DeviceRgbCS());
    },
    get cmyk() {
      return shadow(this, 'cmyk', new DeviceCmykCS());
    }
  };

  return ColorSpace;
})();

/**
 * Alternate color space handles both Separation and DeviceN color spaces.  A
 * Separation color space is actually just a DeviceN with one color component.
 * Both color spaces use a tinting function to convert colors to a base color
 * space.
 */
var AlternateCS = (function AlternateCSClosure() {
  function AlternateCS(numComps, base, tintFn) {
    this.name = 'Alternate';
    this.numComps = numComps;
    this.defaultColor = new Float32Array(numComps);
    for (var i = 0; i < numComps; ++i) {
      this.defaultColor[i] = 1;
    }
    this.base = base;
    this.tintFn = tintFn;
  }

  AlternateCS.prototype = {
    getRgb: ColorSpace.prototype.getRgb,
    getRgbItem: function AlternateCS_getRgbItem(src, srcOffset,
                                                dest, destOffset) {
      var baseNumComps = this.base.numComps;
      var input = 'subarray' in src ?
        src.subarray(srcOffset, srcOffset + this.numComps) :
        Array.prototype.slice.call(src, srcOffset, srcOffset + this.numComps);
      var tinted = this.tintFn(input);
      this.base.getRgbItem(tinted, 0, dest, destOffset);
    },
    getRgbBuffer: function AlternateCS_getRgbBuffer(src, srcOffset, count,
                                                    dest, destOffset, bits,
                                                    alpha01) {
      var tinted;
      var tintFn = this.tintFn;
      var base = this.base;
      var scale = 1 / ((1 << bits) - 1);
      var baseNumComps = base.numComps;
      var usesZeroToOneRange = base.usesZeroToOneRange;
      var isPassthrough = (base.isPassthrough(8) || !usesZeroToOneRange) &&
                          alpha01 === 0;
      var pos = isPassthrough ? destOffset : 0;
      var baseBuf = isPassthrough ? dest : new Uint8Array(baseNumComps * count);
      var numComps = this.numComps;

      var scaled = new Float32Array(numComps);
      var i, j;
      if (usesZeroToOneRange) {
        for (i = 0; i < count; i++) {
          for (j = 0; j < numComps; j++) {
            scaled[j] = src[srcOffset++] * scale;
          }
          tinted = tintFn(scaled);
          for (j = 0; j < baseNumComps; j++) {
            baseBuf[pos++] = tinted[j] * 255;
          }
        }
      } else {
        for (i = 0; i < count; i++) {
          for (j = 0; j < numComps; j++) {
            scaled[j] = src[srcOffset++] * scale;
          }
          tinted = tintFn(scaled);
          base.getRgbItem(tinted, 0, baseBuf, pos);
          pos += baseNumComps;
        }
      }
      if (!isPassthrough) {
        base.getRgbBuffer(baseBuf, 0, count, dest, destOffset, 8, alpha01);
      }
    },
    getOutputLength: function AlternateCS_getOutputLength(inputLength,
                                                          alpha01) {
      return this.base.getOutputLength(inputLength *
                                       this.base.numComps / this.numComps,
                                       alpha01);
    },
    isPassthrough: ColorSpace.prototype.isPassthrough,
    fillRgb: ColorSpace.prototype.fillRgb,
    isDefaultDecode: function AlternateCS_isDefaultDecode(decodeMap) {
      return ColorSpace.isDefaultDecode(decodeMap, this.numComps);
    },
    usesZeroToOneRange: true
  };

  return AlternateCS;
})();

var PatternCS = (function PatternCSClosure() {
  function PatternCS(baseCS) {
    this.name = 'Pattern';
    this.base = baseCS;
  }
  PatternCS.prototype = {};

  return PatternCS;
})();

var IndexedCS = (function IndexedCSClosure() {
  function IndexedCS(base, highVal, lookup) {
    this.name = 'Indexed';
    this.numComps = 1;
    this.defaultColor = new Uint8Array([0]);
    this.base = base;
    this.highVal = highVal;

    var baseNumComps = base.numComps;
    var length = baseNumComps * highVal;
    var lookupArray;

    if (isStream(lookup)) {
      lookupArray = new Uint8Array(length);
      var bytes = lookup.getBytes(length);
      lookupArray.set(bytes);
    } else if (isString(lookup)) {
      lookupArray = new Uint8Array(length);
      for (var i = 0; i < length; ++i) {
        lookupArray[i] = lookup.charCodeAt(i);
      }
    } else if (lookup instanceof Uint8Array || lookup instanceof Array) {
      lookupArray = lookup;
    } else {
      error('Unrecognized lookup table: ' + lookup);
    }
    this.lookup = lookupArray;
  }

  IndexedCS.prototype = {
    getRgb: ColorSpace.prototype.getRgb,
    getRgbItem: function IndexedCS_getRgbItem(src, srcOffset,
                                              dest, destOffset) {
      var numComps = this.base.numComps;
      var start = src[srcOffset] * numComps;
      this.base.getRgbItem(this.lookup, start, dest, destOffset);
    },
    getRgbBuffer: function IndexedCS_getRgbBuffer(src, srcOffset, count,
                                                  dest, destOffset, bits,
                                                  alpha01) {
      var base = this.base;
      var numComps = base.numComps;
      var outputDelta = base.getOutputLength(numComps, alpha01);
      var lookup = this.lookup;

      for (var i = 0; i < count; ++i) {
        var lookupPos = src[srcOffset++] * numComps;
        base.getRgbBuffer(lookup, lookupPos, 1, dest, destOffset, 8, alpha01);
        destOffset += outputDelta;
      }
    },
    getOutputLength: function IndexedCS_getOutputLength(inputLength, alpha01) {
      return this.base.getOutputLength(inputLength * this.base.numComps,
                                       alpha01);
    },
    isPassthrough: ColorSpace.prototype.isPassthrough,
    fillRgb: ColorSpace.prototype.fillRgb,
    isDefaultDecode: function IndexedCS_isDefaultDecode(decodeMap) {
      // indexed color maps shouldn't be changed
      return true;
    },
    usesZeroToOneRange: true
  };
  return IndexedCS;
})();

var DeviceGrayCS = (function DeviceGrayCSClosure() {
  function DeviceGrayCS() {
    this.name = 'DeviceGray';
    this.numComps = 1;
    this.defaultColor = new Float32Array([0]);
  }

  DeviceGrayCS.prototype = {
    getRgb: ColorSpace.prototype.getRgb,
    getRgbItem: function DeviceGrayCS_getRgbItem(src, srcOffset,
                                                 dest, destOffset) {
      var c = (src[srcOffset] * 255) | 0;
      c = c < 0 ? 0 : c > 255 ? 255 : c;
      dest[destOffset] = dest[destOffset + 1] = dest[destOffset + 2] = c;
    },
    getRgbBuffer: function DeviceGrayCS_getRgbBuffer(src, srcOffset, count,
                                                     dest, destOffset, bits,
                                                     alpha01) {
      var scale = 255 / ((1 << bits) - 1);
      var j = srcOffset, q = destOffset;
      for (var i = 0; i < count; ++i) {
        var c = (scale * src[j++]) | 0;
        dest[q++] = c;
        dest[q++] = c;
        dest[q++] = c;
        q += alpha01;
      }
    },
    getOutputLength: function DeviceGrayCS_getOutputLength(inputLength,
                                                           alpha01) {
      return inputLength * (3 + alpha01);
    },
    isPassthrough: ColorSpace.prototype.isPassthrough,
    fillRgb: ColorSpace.prototype.fillRgb,
    isDefaultDecode: function DeviceGrayCS_isDefaultDecode(decodeMap) {
      return ColorSpace.isDefaultDecode(decodeMap, this.numComps);
    },
    usesZeroToOneRange: true
  };
  return DeviceGrayCS;
})();

var DeviceRgbCS = (function DeviceRgbCSClosure() {
  function DeviceRgbCS() {
    this.name = 'DeviceRGB';
    this.numComps = 3;
    this.defaultColor = new Float32Array([0, 0, 0]);
  }
  DeviceRgbCS.prototype = {
    getRgb: ColorSpace.prototype.getRgb,
    getRgbItem: function DeviceRgbCS_getRgbItem(src, srcOffset,
                                                dest, destOffset) {
      var r = (src[srcOffset] * 255) | 0;
      var g = (src[srcOffset + 1] * 255) | 0;
      var b = (src[srcOffset + 2] * 255) | 0;
      dest[destOffset] = r < 0 ? 0 : r > 255 ? 255 : r;
      dest[destOffset + 1] = g < 0 ? 0 : g > 255 ? 255 : g;
      dest[destOffset + 2] = b < 0 ? 0 : b > 255 ? 255 : b;
    },
    getRgbBuffer: function DeviceRgbCS_getRgbBuffer(src, srcOffset, count,
                                                    dest, destOffset, bits,
                                                    alpha01) {
      if (bits === 8 && alpha01 === 0) {
        dest.set(src.subarray(srcOffset, srcOffset + count * 3), destOffset);
        return;
      }
      var scale = 255 / ((1 << bits) - 1);
      var j = srcOffset, q = destOffset;
      for (var i = 0; i < count; ++i) {
        dest[q++] = (scale * src[j++]) | 0;
        dest[q++] = (scale * src[j++]) | 0;
        dest[q++] = (scale * src[j++]) | 0;
        q += alpha01;
      }
    },
    getOutputLength: function DeviceRgbCS_getOutputLength(inputLength,
                                                          alpha01) {
      return (inputLength * (3 + alpha01) / 3) | 0;
    },
    isPassthrough: function DeviceRgbCS_isPassthrough(bits) {
      return bits == 8;
    },
    fillRgb: ColorSpace.prototype.fillRgb,
    isDefaultDecode: function DeviceRgbCS_isDefaultDecode(decodeMap) {
      return ColorSpace.isDefaultDecode(decodeMap, this.numComps);
    },
    usesZeroToOneRange: true
  };
  return DeviceRgbCS;
})();

var DeviceCmykCS = (function DeviceCmykCSClosure() {
  // The coefficients below was found using numerical analysis: the method of
  // steepest descent for the sum((f_i - color_value_i)^2) for r/g/b colors,
  // where color_value is the tabular value from the table of sampled RGB colors
  // from CMYK US Web Coated (SWOP) colorspace, and f_i is the corresponding
  // CMYK color conversion using the estimation below:
  //   f(A, B,.. N) = Acc+Bcm+Ccy+Dck+c+Fmm+Gmy+Hmk+Im+Jyy+Kyk+Ly+Mkk+Nk+255
  function convertToRgb(src, srcOffset, srcScale, dest, destOffset) {
    var c = src[srcOffset + 0] * srcScale;
    var m = src[srcOffset + 1] * srcScale;
    var y = src[srcOffset + 2] * srcScale;
    var k = src[srcOffset + 3] * srcScale;

    var r =
      (c * (-4.387332384609988 * c + 54.48615194189176 * m +
            18.82290502165302 * y + 212.25662451639585 * k +
            -285.2331026137004) +
       m * (1.7149763477362134 * m - 5.6096736904047315 * y +
            -17.873870861415444 * k - 5.497006427196366) +
       y * (-2.5217340131683033 * y - 21.248923337353073 * k +
            17.5119270841813) +
       k * (-21.86122147463605 * k - 189.48180835922747) + 255) | 0;
    var g =
      (c * (8.841041422036149 * c + 60.118027045597366 * m +
            6.871425592049007 * y + 31.159100130055922 * k +
            -79.2970844816548) +
       m * (-15.310361306967817 * m + 17.575251261109482 * y +
            131.35250912493976 * k - 190.9453302588951) +
       y * (4.444339102852739 * y + 9.8632861493405 * k - 24.86741582555878) +
       k * (-20.737325471181034 * k - 187.80453709719578) + 255) | 0;
    var b =
      (c * (0.8842522430003296 * c + 8.078677503112928 * m +
            30.89978309703729 * y - 0.23883238689178934 * k +
            -14.183576799673286) +
       m * (10.49593273432072 * m + 63.02378494754052 * y +
            50.606957656360734 * k - 112.23884253719248) +
       y * (0.03296041114873217 * y + 115.60384449646641 * k +
            -193.58209356861505) +
       k * (-22.33816807309886 * k - 180.12613974708367) + 255) | 0;

    dest[destOffset] = r > 255 ? 255 : r < 0 ? 0 : r;
    dest[destOffset + 1] = g > 255 ? 255 : g < 0 ? 0 : g;
    dest[destOffset + 2] = b > 255 ? 255 : b < 0 ? 0 : b;
  }

  function DeviceCmykCS() {
    this.name = 'DeviceCMYK';
    this.numComps = 4;
    this.defaultColor = new Float32Array([0, 0, 0, 1]);
  }
  DeviceCmykCS.prototype = {
    getRgb: ColorSpace.prototype.getRgb,
    getRgbItem: function DeviceCmykCS_getRgbItem(src, srcOffset,
                                                 dest, destOffset) {
      convertToRgb(src, srcOffset, 1, dest, destOffset);
    },
    getRgbBuffer: function DeviceCmykCS_getRgbBuffer(src, srcOffset, count,
                                                     dest, destOffset, bits,
                                                     alpha01) {
      var scale = 1 / ((1 << bits) - 1);
      for (var i = 0; i < count; i++) {
        convertToRgb(src, srcOffset, scale, dest, destOffset);
        srcOffset += 4;
        destOffset += 3 + alpha01;
      }
    },
    getOutputLength: function DeviceCmykCS_getOutputLength(inputLength,
                                                           alpha01) {
      return (inputLength / 4 * (3 + alpha01)) | 0;
    },
    isPassthrough: ColorSpace.prototype.isPassthrough,
    fillRgb: ColorSpace.prototype.fillRgb,
    isDefaultDecode: function DeviceCmykCS_isDefaultDecode(decodeMap) {
      return ColorSpace.isDefaultDecode(decodeMap, this.numComps);
    },
    usesZeroToOneRange: true
  };

  return DeviceCmykCS;
})();

//
// CalGrayCS: Based on "PDF Reference, Sixth Ed", p.245
//
var CalGrayCS = (function CalGrayCSClosure() {
  function CalGrayCS(whitePoint, blackPoint, gamma) {
    this.name = 'CalGray';
    this.numComps = 1;
    this.defaultColor = new Float32Array([0]);

    if (!whitePoint) {
      error('WhitePoint missing - required for color space CalGray');
    }
    blackPoint = blackPoint || [0, 0, 0];
    gamma = gamma || 1;

    // Translate arguments to spec variables.
    this.XW = whitePoint[0];
    this.YW = whitePoint[1];
    this.ZW = whitePoint[2];

    this.XB = blackPoint[0];
    this.YB = blackPoint[1];
    this.ZB = blackPoint[2];

    this.G = gamma;

    // Validate variables as per spec.
    if (this.XW < 0 || this.ZW < 0 || this.YW !== 1) {
      error('Invalid WhitePoint components for ' + this.name +
            ', no fallback available');
    }

    if (this.XB < 0 || this.YB < 0 || this.ZB < 0) {
      info('Invalid BlackPoint for ' + this.name + ', falling back to default');
      this.XB = this.YB = this.ZB = 0;
    }

    if (this.XB !== 0 || this.YB !== 0 || this.ZB !== 0) {
      warn(this.name + ', BlackPoint: XB: ' + this.XB + ', YB: ' + this.YB +
           ', ZB: ' + this.ZB + ', only default values are supported.');
    }

    if (this.G < 1) {
      info('Invalid Gamma: ' + this.G + ' for ' + this.name +
           ', falling back to default');
      this.G = 1;
    }
  }

  function convertToRgb(cs, src, srcOffset, dest, destOffset, scale) {
    // A represents a gray component of a calibrated gray space.
    // A <---> AG in the spec
    var A = src[srcOffset] * scale;
    var AG = Math.pow(A, cs.G);

    // Computes L as per spec. ( = cs.YW * AG )
    // Except if other than default BlackPoint values are used.
    var L = cs.YW * AG;
    // http://www.poynton.com/notes/colour_and_gamma/ColorFAQ.html, Ch 4.
    // Convert values to rgb range [0, 255].
    var val = Math.max(295.8 * Math.pow(L, 0.333333333333333333) - 40.8, 0) | 0;
    dest[destOffset] = val;
    dest[destOffset + 1] = val;
    dest[destOffset + 2] = val;
  }

  CalGrayCS.prototype = {
    getRgb: ColorSpace.prototype.getRgb,
    getRgbItem: function CalGrayCS_getRgbItem(src, srcOffset,
                                              dest, destOffset) {
      convertToRgb(this, src, srcOffset, dest, destOffset, 1);
    },
    getRgbBuffer: function CalGrayCS_getRgbBuffer(src, srcOffset, count,
                                                  dest, destOffset, bits,
                                                  alpha01) {
      var scale = 1 / ((1 << bits) - 1);

      for (var i = 0; i < count; ++i) {
        convertToRgb(this, src, srcOffset, dest, destOffset, scale);
        srcOffset += 1;
        destOffset += 3 + alpha01;
      }
    },
    getOutputLength: function CalGrayCS_getOutputLength(inputLength, alpha01) {
      return inputLength * (3 + alpha01);
    },
    isPassthrough: ColorSpace.prototype.isPassthrough,
    fillRgb: ColorSpace.prototype.fillRgb,
    isDefaultDecode: function CalGrayCS_isDefaultDecode(decodeMap) {
      return ColorSpace.isDefaultDecode(decodeMap, this.numComps);
    },
    usesZeroToOneRange: true
  };
  return CalGrayCS;
})();

//
// LabCS: Based on "PDF Reference, Sixth Ed", p.250
//
var LabCS = (function LabCSClosure() {
  function LabCS(whitePoint, blackPoint, range) {
    this.name = 'Lab';
    this.numComps = 3;
    this.defaultColor = new Float32Array([0, 0, 0]);

    if (!whitePoint) {
      error('WhitePoint missing - required for color space Lab');
    }
    blackPoint = blackPoint || [0, 0, 0];
    range = range || [-100, 100, -100, 100];

    // Translate args to spec variables
    this.XW = whitePoint[0];
    this.YW = whitePoint[1];
    this.ZW = whitePoint[2];
    this.amin = range[0];
    this.amax = range[1];
    this.bmin = range[2];
    this.bmax = range[3];

    // These are here just for completeness - the spec doesn't offer any
    // formulas that use BlackPoint in Lab
    this.XB = blackPoint[0];
    this.YB = blackPoint[1];
    this.ZB = blackPoint[2];

    // Validate vars as per spec
    if (this.XW < 0 || this.ZW < 0 || this.YW !== 1) {
      error('Invalid WhitePoint components, no fallback available');
    }

    if (this.XB < 0 || this.YB < 0 || this.ZB < 0) {
      info('Invalid BlackPoint, falling back to default');
      this.XB = this.YB = this.ZB = 0;
    }

    if (this.amin > this.amax || this.bmin > this.bmax) {
      info('Invalid Range, falling back to defaults');
      this.amin = -100;
      this.amax = 100;
      this.bmin = -100;
      this.bmax = 100;
    }
  }

  // Function g(x) from spec
  function fn_g(x) {
    if (x >= 6 / 29) {
      return x * x * x;
    } else {
      return (108 / 841) * (x - 4 / 29);
    }
  }

  function decode(value, high1, low2, high2) {
    return low2 + (value) * (high2 - low2) / (high1);
  }

  // If decoding is needed maxVal should be 2^bits per component - 1.
  function convertToRgb(cs, src, srcOffset, maxVal, dest, destOffset) {
    // XXX: Lab input is in the range of [0, 100], [amin, amax], [bmin, bmax]
    // not the usual [0, 1]. If a command like setFillColor is used the src
    // values will already be within the correct range. However, if we are
    // converting an image we have to map the values to the correct range given
    // above.
    // Ls,as,bs <---> L*,a*,b* in the spec
    var Ls = src[srcOffset];
    var as = src[srcOffset + 1];
    var bs = src[srcOffset + 2];
    if (maxVal !== false) {
      Ls = decode(Ls, maxVal, 0, 100);
      as = decode(as, maxVal, cs.amin, cs.amax);
      bs = decode(bs, maxVal, cs.bmin, cs.bmax);
    }

    // Adjust limits of 'as' and 'bs'
    as = as > cs.amax ? cs.amax : as < cs.amin ? cs.amin : as;
    bs = bs > cs.bmax ? cs.bmax : bs < cs.bmin ? cs.bmin : bs;

    // Computes intermediate variables X,Y,Z as per spec
    var M = (Ls + 16) / 116;
    var L = M + (as / 500);
    var N = M - (bs / 200);

    var X = cs.XW * fn_g(L);
    var Y = cs.YW * fn_g(M);
    var Z = cs.ZW * fn_g(N);

    var r, g, b;
    // Using different conversions for D50 and D65 white points,
    // per http://www.color.org/srgb.pdf
    if (cs.ZW < 1) {
      // Assuming D50 (X=0.9642, Y=1.00, Z=0.8249)
      r = X * 3.1339 + Y * -1.6170 + Z * -0.4906;
      g = X * -0.9785 + Y * 1.9160 + Z * 0.0333;
      b = X * 0.0720 + Y * -0.2290 + Z * 1.4057;
    } else {
      // Assuming D65 (X=0.9505, Y=1.00, Z=1.0888)
      r = X * 3.2406 + Y * -1.5372 + Z * -0.4986;
      g = X * -0.9689 + Y * 1.8758 + Z * 0.0415;
      b = X * 0.0557 + Y * -0.2040 + Z * 1.0570;
    }
    // clamp color values to [0,1] range then convert to [0,255] range.
    dest[destOffset] = r <= 0 ? 0 : r >= 1 ? 255 : Math.sqrt(r) * 255 | 0;
    dest[destOffset + 1] = g <= 0 ? 0 : g >= 1 ? 255 : Math.sqrt(g) * 255 | 0;
    dest[destOffset + 2] = b <= 0 ? 0 : b >= 1 ? 255 : Math.sqrt(b) * 255 | 0;
  }

  LabCS.prototype = {
    getRgb: ColorSpace.prototype.getRgb,
    getRgbItem: function LabCS_getRgbItem(src, srcOffset, dest, destOffset) {
      convertToRgb(this, src, srcOffset, false, dest, destOffset);
    },
    getRgbBuffer: function LabCS_getRgbBuffer(src, srcOffset, count,
                                              dest, destOffset, bits,
                                              alpha01) {
      var maxVal = (1 << bits) - 1;
      for (var i = 0; i < count; i++) {
        convertToRgb(this, src, srcOffset, maxVal, dest, destOffset);
        srcOffset += 3;
        destOffset += 3 + alpha01;
      }
    },
    getOutputLength: function LabCS_getOutputLength(inputLength, alpha01) {
      return (inputLength * (3 + alpha01) / 3) | 0;
    },
    isPassthrough: ColorSpace.prototype.isPassthrough,
    isDefaultDecode: function LabCS_isDefaultDecode(decodeMap) {
      // XXX: Decoding is handled with the lab conversion because of the strange
      // ranges that are used.
      return true;
    },
    usesZeroToOneRange: false
  };
  return LabCS;
})();



var ARCFourCipher = (function ARCFourCipherClosure() {
  function ARCFourCipher(key) {
    this.a = 0;
    this.b = 0;
    var s = new Uint8Array(256);
    var i, j = 0, tmp, keyLength = key.length;
    for (i = 0; i < 256; ++i) {
      s[i] = i;
    }
    for (i = 0; i < 256; ++i) {
      tmp = s[i];
      j = (j + tmp + key[i % keyLength]) & 0xFF;
      s[i] = s[j];
      s[j] = tmp;
    }
    this.s = s;
  }

  ARCFourCipher.prototype = {
    encryptBlock: function ARCFourCipher_encryptBlock(data) {
      var i, n = data.length, tmp, tmp2;
      var a = this.a, b = this.b, s = this.s;
      var output = new Uint8Array(n);
      for (i = 0; i < n; ++i) {
        a = (a + 1) & 0xFF;
        tmp = s[a];
        b = (b + tmp) & 0xFF;
        tmp2 = s[b];
        s[a] = tmp2;
        s[b] = tmp;
        output[i] = data[i] ^ s[(tmp + tmp2) & 0xFF];
      }
      this.a = a;
      this.b = b;
      return output;
    }
  };
  ARCFourCipher.prototype.decryptBlock = ARCFourCipher.prototype.encryptBlock;

  return ARCFourCipher;
})();

var calculateMD5 = (function calculateMD5Closure() {
  var r = new Uint8Array([
    7, 12, 17, 22, 7, 12, 17, 22, 7, 12, 17, 22, 7, 12, 17, 22,
    5, 9, 14, 20, 5, 9, 14, 20, 5, 9, 14, 20, 5, 9, 14, 20,
    4, 11, 16, 23, 4, 11, 16, 23, 4, 11, 16, 23, 4, 11, 16, 23,
    6, 10, 15, 21, 6, 10, 15, 21, 6, 10, 15, 21, 6, 10, 15, 21]);

  var k = new Int32Array([
    -680876936, -389564586, 606105819, -1044525330, -176418897, 1200080426,
    -1473231341, -45705983, 1770035416, -1958414417, -42063, -1990404162,
    1804603682, -40341101, -1502002290, 1236535329, -165796510, -1069501632,
    643717713, -373897302, -701558691, 38016083, -660478335, -405537848,
    568446438, -1019803690, -187363961, 1163531501, -1444681467, -51403784,
    1735328473, -1926607734, -378558, -2022574463, 1839030562, -35309556,
    -1530992060, 1272893353, -155497632, -1094730640, 681279174, -358537222,
    -722521979, 76029189, -640364487, -421815835, 530742520, -995338651,
    -198630844, 1126891415, -1416354905, -57434055, 1700485571, -1894986606,
    -1051523, -2054922799, 1873313359, -30611744, -1560198380, 1309151649,
    -145523070, -1120210379, 718787259, -343485551]);

  function hash(data, offset, length) {
    var h0 = 1732584193, h1 = -271733879, h2 = -1732584194, h3 = 271733878;
    // pre-processing
    var paddedLength = (length + 72) & ~63; // data + 9 extra bytes
    var padded = new Uint8Array(paddedLength);
    var i, j, n;
    for (i = 0; i < length; ++i) {
      padded[i] = data[offset++];
    }
    padded[i++] = 0x80;
    n = paddedLength - 8;
    while (i < n) {
      padded[i++] = 0;
    }
    padded[i++] = (length << 3) & 0xFF;
    padded[i++] = (length >> 5) & 0xFF;
    padded[i++] = (length >> 13) & 0xFF;
    padded[i++] = (length >> 21) & 0xFF;
    padded[i++] = (length >>> 29) & 0xFF;
    padded[i++] = 0;
    padded[i++] = 0;
    padded[i++] = 0;
    // chunking
    // TODO ArrayBuffer ?
    var w = new Int32Array(16);
    for (i = 0; i < paddedLength;) {
      for (j = 0; j < 16; ++j, i += 4) {
        w[j] = (padded[i] | (padded[i + 1] << 8) |
                (padded[i + 2] << 16) | (padded[i + 3] << 24));
      }
      var a = h0, b = h1, c = h2, d = h3, f, g;
      for (j = 0; j < 64; ++j) {
        if (j < 16) {
          f = (b & c) | ((~b) & d);
          g = j;
        } else if (j < 32) {
          f = (d & b) | ((~d) & c);
          g = (5 * j + 1) & 15;
        } else if (j < 48) {
          f = b ^ c ^ d;
          g = (3 * j + 5) & 15;
        } else {
          f = c ^ (b | (~d));
          g = (7 * j) & 15;
        }
        var tmp = d, rotateArg = (a + f + k[j] + w[g]) | 0, rotate = r[j];
        d = c;
        c = b;
        b = (b + ((rotateArg << rotate) | (rotateArg >>> (32 - rotate)))) | 0;
        a = tmp;
      }
      h0 = (h0 + a) | 0;
      h1 = (h1 + b) | 0;
      h2 = (h2 + c) | 0;
      h3 = (h3 + d) | 0;
    }
    return new Uint8Array([
      h0 & 0xFF, (h0 >> 8) & 0xFF, (h0 >> 16) & 0xFF, (h0 >>> 24) & 0xFF,
      h1 & 0xFF, (h1 >> 8) & 0xFF, (h1 >> 16) & 0xFF, (h1 >>> 24) & 0xFF,
      h2 & 0xFF, (h2 >> 8) & 0xFF, (h2 >> 16) & 0xFF, (h2 >>> 24) & 0xFF,
      h3 & 0xFF, (h3 >> 8) & 0xFF, (h3 >> 16) & 0xFF, (h3 >>> 24) & 0xFF
    ]);
  }
  return hash;
})();

var NullCipher = (function NullCipherClosure() {
  function NullCipher() {}

  NullCipher.prototype = {
    decryptBlock: function NullCipher_decryptBlock(data) {
      return data;
    }
  };

  return NullCipher;
})();

var AES128Cipher = (function AES128CipherClosure() {
  var rcon = new Uint8Array([
    0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c,
    0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a,
    0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd,
    0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a,
    0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80,
    0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6,
    0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72,
    0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc,
    0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02, 0x04, 0x08, 0x10,
    0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e,
    0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5,
    0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94,
    0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb, 0x8d, 0x01, 0x02,
    0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c, 0xd8, 0xab, 0x4d,
    0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a, 0xd4, 0xb3, 0x7d,
    0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd, 0x61, 0xc2, 0x9f,
    0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a, 0x74, 0xe8, 0xcb,
    0x8d, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x1b, 0x36, 0x6c,
    0xd8, 0xab, 0x4d, 0x9a, 0x2f, 0x5e, 0xbc, 0x63, 0xc6, 0x97, 0x35, 0x6a,
    0xd4, 0xb3, 0x7d, 0xfa, 0xef, 0xc5, 0x91, 0x39, 0x72, 0xe4, 0xd3, 0xbd,
    0x61, 0xc2, 0x9f, 0x25, 0x4a, 0x94, 0x33, 0x66, 0xcc, 0x83, 0x1d, 0x3a,
    0x74, 0xe8, 0xcb, 0x8d]);

  var s = new Uint8Array([
    0x63, 0x7c, 0x77, 0x7b, 0xf2, 0x6b, 0x6f, 0xc5, 0x30, 0x01, 0x67, 0x2b,
    0xfe, 0xd7, 0xab, 0x76, 0xca, 0x82, 0xc9, 0x7d, 0xfa, 0x59, 0x47, 0xf0,
    0xad, 0xd4, 0xa2, 0xaf, 0x9c, 0xa4, 0x72, 0xc0, 0xb7, 0xfd, 0x93, 0x26,
    0x36, 0x3f, 0xf7, 0xcc, 0x34, 0xa5, 0xe5, 0xf1, 0x71, 0xd8, 0x31, 0x15,
    0x04, 0xc7, 0x23, 0xc3, 0x18, 0x96, 0x05, 0x9a, 0x07, 0x12, 0x80, 0xe2,
    0xeb, 0x27, 0xb2, 0x75, 0x09, 0x83, 0x2c, 0x1a, 0x1b, 0x6e, 0x5a, 0xa0,
    0x52, 0x3b, 0xd6, 0xb3, 0x29, 0xe3, 0x2f, 0x84, 0x53, 0xd1, 0x00, 0xed,
    0x20, 0xfc, 0xb1, 0x5b, 0x6a, 0xcb, 0xbe, 0x39, 0x4a, 0x4c, 0x58, 0xcf,
    0xd0, 0xef, 0xaa, 0xfb, 0x43, 0x4d, 0x33, 0x85, 0x45, 0xf9, 0x02, 0x7f,
    0x50, 0x3c, 0x9f, 0xa8, 0x51, 0xa3, 0x40, 0x8f, 0x92, 0x9d, 0x38, 0xf5,
    0xbc, 0xb6, 0xda, 0x21, 0x10, 0xff, 0xf3, 0xd2, 0xcd, 0x0c, 0x13, 0xec,
    0x5f, 0x97, 0x44, 0x17, 0xc4, 0xa7, 0x7e, 0x3d, 0x64, 0x5d, 0x19, 0x73,
    0x60, 0x81, 0x4f, 0xdc, 0x22, 0x2a, 0x90, 0x88, 0x46, 0xee, 0xb8, 0x14,
    0xde, 0x5e, 0x0b, 0xdb, 0xe0, 0x32, 0x3a, 0x0a, 0x49, 0x06, 0x24, 0x5c,
    0xc2, 0xd3, 0xac, 0x62, 0x91, 0x95, 0xe4, 0x79, 0xe7, 0xc8, 0x37, 0x6d,
    0x8d, 0xd5, 0x4e, 0xa9, 0x6c, 0x56, 0xf4, 0xea, 0x65, 0x7a, 0xae, 0x08,
    0xba, 0x78, 0x25, 0x2e, 0x1c, 0xa6, 0xb4, 0xc6, 0xe8, 0xdd, 0x74, 0x1f,
    0x4b, 0xbd, 0x8b, 0x8a, 0x70, 0x3e, 0xb5, 0x66, 0x48, 0x03, 0xf6, 0x0e,
    0x61, 0x35, 0x57, 0xb9, 0x86, 0xc1, 0x1d, 0x9e, 0xe1, 0xf8, 0x98, 0x11,
    0x69, 0xd9, 0x8e, 0x94, 0x9b, 0x1e, 0x87, 0xe9, 0xce, 0x55, 0x28, 0xdf,
    0x8c, 0xa1, 0x89, 0x0d, 0xbf, 0xe6, 0x42, 0x68, 0x41, 0x99, 0x2d, 0x0f,
    0xb0, 0x54, 0xbb, 0x16]);

  var inv_s = new Uint8Array([
    0x52, 0x09, 0x6a, 0xd5, 0x30, 0x36, 0xa5, 0x38, 0xbf, 0x40, 0xa3, 0x9e,
    0x81, 0xf3, 0xd7, 0xfb, 0x7c, 0xe3, 0x39, 0x82, 0x9b, 0x2f, 0xff, 0x87,
    0x34, 0x8e, 0x43, 0x44, 0xc4, 0xde, 0xe9, 0xcb, 0x54, 0x7b, 0x94, 0x32,
    0xa6, 0xc2, 0x23, 0x3d, 0xee, 0x4c, 0x95, 0x0b, 0x42, 0xfa, 0xc3, 0x4e,
    0x08, 0x2e, 0xa1, 0x66, 0x28, 0xd9, 0x24, 0xb2, 0x76, 0x5b, 0xa2, 0x49,
    0x6d, 0x8b, 0xd1, 0x25, 0x72, 0xf8, 0xf6, 0x64, 0x86, 0x68, 0x98, 0x16,
    0xd4, 0xa4, 0x5c, 0xcc, 0x5d, 0x65, 0xb6, 0x92, 0x6c, 0x70, 0x48, 0x50,
    0xfd, 0xed, 0xb9, 0xda, 0x5e, 0x15, 0x46, 0x57, 0xa7, 0x8d, 0x9d, 0x84,
    0x90, 0xd8, 0xab, 0x00, 0x8c, 0xbc, 0xd3, 0x0a, 0xf7, 0xe4, 0x58, 0x05,
    0xb8, 0xb3, 0x45, 0x06, 0xd0, 0x2c, 0x1e, 0x8f, 0xca, 0x3f, 0x0f, 0x02,
    0xc1, 0xaf, 0xbd, 0x03, 0x01, 0x13, 0x8a, 0x6b, 0x3a, 0x91, 0x11, 0x41,
    0x4f, 0x67, 0xdc, 0xea, 0x97, 0xf2, 0xcf, 0xce, 0xf0, 0xb4, 0xe6, 0x73,
    0x96, 0xac, 0x74, 0x22, 0xe7, 0xad, 0x35, 0x85, 0xe2, 0xf9, 0x37, 0xe8,
    0x1c, 0x75, 0xdf, 0x6e, 0x47, 0xf1, 0x1a, 0x71, 0x1d, 0x29, 0xc5, 0x89,
    0x6f, 0xb7, 0x62, 0x0e, 0xaa, 0x18, 0xbe, 0x1b, 0xfc, 0x56, 0x3e, 0x4b,
    0xc6, 0xd2, 0x79, 0x20, 0x9a, 0xdb, 0xc0, 0xfe, 0x78, 0xcd, 0x5a, 0xf4,
    0x1f, 0xdd, 0xa8, 0x33, 0x88, 0x07, 0xc7, 0x31, 0xb1, 0x12, 0x10, 0x59,
    0x27, 0x80, 0xec, 0x5f, 0x60, 0x51, 0x7f, 0xa9, 0x19, 0xb5, 0x4a, 0x0d,
    0x2d, 0xe5, 0x7a, 0x9f, 0x93, 0xc9, 0x9c, 0xef, 0xa0, 0xe0, 0x3b, 0x4d,
    0xae, 0x2a, 0xf5, 0xb0, 0xc8, 0xeb, 0xbb, 0x3c, 0x83, 0x53, 0x99, 0x61,
    0x17, 0x2b, 0x04, 0x7e, 0xba, 0x77, 0xd6, 0x26, 0xe1, 0x69, 0x14, 0x63,
    0x55, 0x21, 0x0c, 0x7d]);

  var mix = new Uint32Array([
    0x00000000, 0x0e090d0b, 0x1c121a16, 0x121b171d, 0x3824342c, 0x362d3927,
    0x24362e3a, 0x2a3f2331, 0x70486858, 0x7e416553, 0x6c5a724e, 0x62537f45,
    0x486c5c74, 0x4665517f, 0x547e4662, 0x5a774b69, 0xe090d0b0, 0xee99ddbb,
    0xfc82caa6, 0xf28bc7ad, 0xd8b4e49c, 0xd6bde997, 0xc4a6fe8a, 0xcaaff381,
    0x90d8b8e8, 0x9ed1b5e3, 0x8ccaa2fe, 0x82c3aff5, 0xa8fc8cc4, 0xa6f581cf,
    0xb4ee96d2, 0xbae79bd9, 0xdb3bbb7b, 0xd532b670, 0xc729a16d, 0xc920ac66,
    0xe31f8f57, 0xed16825c, 0xff0d9541, 0xf104984a, 0xab73d323, 0xa57ade28,
    0xb761c935, 0xb968c43e, 0x9357e70f, 0x9d5eea04, 0x8f45fd19, 0x814cf012,
    0x3bab6bcb, 0x35a266c0, 0x27b971dd, 0x29b07cd6, 0x038f5fe7, 0x0d8652ec,
    0x1f9d45f1, 0x119448fa, 0x4be30393, 0x45ea0e98, 0x57f11985, 0x59f8148e,
    0x73c737bf, 0x7dce3ab4, 0x6fd52da9, 0x61dc20a2, 0xad766df6, 0xa37f60fd,
    0xb16477e0, 0xbf6d7aeb, 0x955259da, 0x9b5b54d1, 0x894043cc, 0x87494ec7,
    0xdd3e05ae, 0xd33708a5, 0xc12c1fb8, 0xcf2512b3, 0xe51a3182, 0xeb133c89,
    0xf9082b94, 0xf701269f, 0x4de6bd46, 0x43efb04d, 0x51f4a750, 0x5ffdaa5b,
    0x75c2896a, 0x7bcb8461, 0x69d0937c, 0x67d99e77, 0x3daed51e, 0x33a7d815,
    0x21bccf08, 0x2fb5c203, 0x058ae132, 0x0b83ec39, 0x1998fb24, 0x1791f62f,
    0x764dd68d, 0x7844db86, 0x6a5fcc9b, 0x6456c190, 0x4e69e2a1, 0x4060efaa,
    0x527bf8b7, 0x5c72f5bc, 0x0605bed5, 0x080cb3de, 0x1a17a4c3, 0x141ea9c8,
    0x3e218af9, 0x302887f2, 0x223390ef, 0x2c3a9de4, 0x96dd063d, 0x98d40b36,
    0x8acf1c2b, 0x84c61120, 0xaef93211, 0xa0f03f1a, 0xb2eb2807, 0xbce2250c,
    0xe6956e65, 0xe89c636e, 0xfa877473, 0xf48e7978, 0xdeb15a49, 0xd0b85742,
    0xc2a3405f, 0xccaa4d54, 0x41ecdaf7, 0x4fe5d7fc, 0x5dfec0e1, 0x53f7cdea,
    0x79c8eedb, 0x77c1e3d0, 0x65daf4cd, 0x6bd3f9c6, 0x31a4b2af, 0x3fadbfa4,
    0x2db6a8b9, 0x23bfa5b2, 0x09808683, 0x07898b88, 0x15929c95, 0x1b9b919e,
    0xa17c0a47, 0xaf75074c, 0xbd6e1051, 0xb3671d5a, 0x99583e6b, 0x97513360,
    0x854a247d, 0x8b432976, 0xd134621f, 0xdf3d6f14, 0xcd267809, 0xc32f7502,
    0xe9105633, 0xe7195b38, 0xf5024c25, 0xfb0b412e, 0x9ad7618c, 0x94de6c87,
    0x86c57b9a, 0x88cc7691, 0xa2f355a0, 0xacfa58ab, 0xbee14fb6, 0xb0e842bd,
    0xea9f09d4, 0xe49604df, 0xf68d13c2, 0xf8841ec9, 0xd2bb3df8, 0xdcb230f3,
    0xcea927ee, 0xc0a02ae5, 0x7a47b13c, 0x744ebc37, 0x6655ab2a, 0x685ca621,
    0x42638510, 0x4c6a881b, 0x5e719f06, 0x5078920d, 0x0a0fd964, 0x0406d46f,
    0x161dc372, 0x1814ce79, 0x322bed48, 0x3c22e043, 0x2e39f75e, 0x2030fa55,
    0xec9ab701, 0xe293ba0a, 0xf088ad17, 0xfe81a01c, 0xd4be832d, 0xdab78e26,
    0xc8ac993b, 0xc6a59430, 0x9cd2df59, 0x92dbd252, 0x80c0c54f, 0x8ec9c844,
    0xa4f6eb75, 0xaaffe67e, 0xb8e4f163, 0xb6edfc68, 0x0c0a67b1, 0x02036aba,
    0x10187da7, 0x1e1170ac, 0x342e539d, 0x3a275e96, 0x283c498b, 0x26354480,
    0x7c420fe9, 0x724b02e2, 0x605015ff, 0x6e5918f4, 0x44663bc5, 0x4a6f36ce,
    0x587421d3, 0x567d2cd8, 0x37a10c7a, 0x39a80171, 0x2bb3166c, 0x25ba1b67,
    0x0f853856, 0x018c355d, 0x13972240, 0x1d9e2f4b, 0x47e96422, 0x49e06929,
    0x5bfb7e34, 0x55f2733f, 0x7fcd500e, 0x71c45d05, 0x63df4a18, 0x6dd64713,
    0xd731dcca, 0xd938d1c1, 0xcb23c6dc, 0xc52acbd7, 0xef15e8e6, 0xe11ce5ed,
    0xf307f2f0, 0xfd0efffb, 0xa779b492, 0xa970b999, 0xbb6bae84, 0xb562a38f,
    0x9f5d80be, 0x91548db5, 0x834f9aa8, 0x8d4697a3]);

  function expandKey128(cipherKey) {
    var b = 176, result = new Uint8Array(b);
    result.set(cipherKey);
    for (var j = 16, i = 1; j < b; ++i) {
      // RotWord
      var t1 = result[j - 3], t2 = result[j - 2],
          t3 = result[j - 1], t4 = result[j - 4];
      // SubWord
      t1 = s[t1]; t2 = s[t2]; t3 = s[t3]; t4 = s[t4];
      // Rcon
      t1 = t1 ^ rcon[i];
      for (var n = 0; n < 4; ++n) {
        result[j] = (t1 ^= result[j - 16]); j++;
        result[j] = (t2 ^= result[j - 16]); j++;
        result[j] = (t3 ^= result[j - 16]); j++;
        result[j] = (t4 ^= result[j - 16]); j++;
      }
    }
    return result;
  }

  function decrypt128(input, key) {
    var state = new Uint8Array(16);
    state.set(input);
    var i, j, k;
    var t, u, v;
    // AddRoundKey
    for (j = 0, k = 160; j < 16; ++j, ++k) {
      state[j] ^= key[k];
    }
    for (i = 9; i >= 1; --i) {
      // InvShiftRows
      t = state[13]; state[13] = state[9]; state[9] = state[5];
      state[5] = state[1]; state[1] = t;
      t = state[14]; u = state[10]; state[14] = state[6];
      state[10] = state[2]; state[6] = t; state[2] = u;
      t = state[15]; u = state[11]; v = state[7]; state[15] = state[3];
      state[11] = t; state[7] = u; state[3] = v;
      // InvSubBytes
      for (j = 0; j < 16; ++j) {
        state[j] = inv_s[state[j]];
      }
      // AddRoundKey
      for (j = 0, k = i * 16; j < 16; ++j, ++k) {
        state[j] ^= key[k];
      }
      // InvMixColumns
      for (j = 0; j < 16; j += 4) {
        var s0 = mix[state[j]], s1 = mix[state[j + 1]],
            s2 = mix[state[j + 2]], s3 = mix[state[j + 3]];
        t = (s0 ^ (s1 >>> 8) ^ (s1 << 24) ^ (s2 >>> 16) ^ (s2 << 16) ^
            (s3 >>> 24) ^ (s3 << 8));
        state[j] = (t >>> 24) & 0xFF;
        state[j + 1] = (t >> 16) & 0xFF;
        state[j + 2] = (t >> 8) & 0xFF;
        state[j + 3] = t & 0xFF;
      }
    }
    // InvShiftRows
    t = state[13]; state[13] = state[9]; state[9] = state[5];
    state[5] = state[1]; state[1] = t;
    t = state[14]; u = state[10]; state[14] = state[6];
    state[10] = state[2]; state[6] = t; state[2] = u;
    t = state[15]; u = state[11]; v = state[7]; state[15] = state[3];
    state[11] = t; state[7] = u; state[3] = v;
    for (j = 0; j < 16; ++j) {
      // InvSubBytes
      state[j] = inv_s[state[j]];
      // AddRoundKey
      state[j] ^= key[j];
    }
    return state;
  }

  function AES128Cipher(key) {
    this.key = expandKey128(key);
    this.buffer = new Uint8Array(16);
    this.bufferPosition = 0;
  }

  function decryptBlock2(data, finalize) {
    var i, j, ii, sourceLength = data.length,
        buffer = this.buffer, bufferLength = this.bufferPosition,
        result = [], iv = this.iv;
    for (i = 0; i < sourceLength; ++i) {
      buffer[bufferLength] = data[i];
      ++bufferLength;
      if (bufferLength < 16) {
        continue;
      }
      // buffer is full, decrypting
      var plain = decrypt128(buffer, this.key);
      // xor-ing the IV vector to get plain text
      for (j = 0; j < 16; ++j) {
        plain[j] ^= iv[j];
      }
      iv = buffer;
      result.push(plain);
      buffer = new Uint8Array(16);
      bufferLength = 0;
    }
    // saving incomplete buffer
    this.buffer = buffer;
    this.bufferLength = bufferLength;
    this.iv = iv;
    if (result.length === 0) {
      return new Uint8Array([]);
    }
    // combining plain text blocks into one
    var outputLength = 16 * result.length;
    if (finalize) {
      // undo a padding that is described in RFC 2898
      var lastBlock = result[result.length - 1];
      outputLength -= lastBlock[15];
      result[result.length - 1] = lastBlock.subarray(0, 16 - lastBlock[15]);
    }
    var output = new Uint8Array(outputLength);
    for (i = 0, j = 0, ii = result.length; i < ii; ++i, j += 16) {
      output.set(result[i], j);
    }
    return output;
  }

  AES128Cipher.prototype = {
    decryptBlock: function AES128Cipher_decryptBlock(data, finalize) {
      var i, sourceLength = data.length;
      var buffer = this.buffer, bufferLength = this.bufferPosition;
      // waiting for IV values -- they are at the start of the stream
      for (i = 0; bufferLength < 16 && i < sourceLength; ++i, ++bufferLength) {
        buffer[bufferLength] = data[i];
      }
      if (bufferLength < 16) {
        // need more data
        this.bufferLength = bufferLength;
        return new Uint8Array([]);
      }
      this.iv = buffer;
      this.buffer = new Uint8Array(16);
      this.bufferLength = 0;
      // starting decryption
      this.decryptBlock = decryptBlock2;
      return this.decryptBlock(data.subarray(16), finalize);
    }
  };

  return AES128Cipher;
})();

var CipherTransform = (function CipherTransformClosure() {
  function CipherTransform(stringCipherConstructor, streamCipherConstructor) {
    this.stringCipherConstructor = stringCipherConstructor;
    this.streamCipherConstructor = streamCipherConstructor;
  }
  CipherTransform.prototype = {
    createStream: function CipherTransform_createStream(stream, length) {
      var cipher = new this.streamCipherConstructor();
      return new DecryptStream(stream, length,
        function cipherTransformDecryptStream(data, finalize) {
          return cipher.decryptBlock(data, finalize);
        }
      );
    },
    decryptString: function CipherTransform_decryptString(s) {
      var cipher = new this.stringCipherConstructor();
      var data = stringToBytes(s);
      data = cipher.decryptBlock(data, true);
      return bytesToString(data);
    }
  };
  return CipherTransform;
})();

var CipherTransformFactory = (function CipherTransformFactoryClosure() {
  var defaultPasswordBytes = new Uint8Array([
    0x28, 0xBF, 0x4E, 0x5E, 0x4E, 0x75, 0x8A, 0x41,
    0x64, 0x00, 0x4E, 0x56, 0xFF, 0xFA, 0x01, 0x08,
    0x2E, 0x2E, 0x00, 0xB6, 0xD0, 0x68, 0x3E, 0x80,
    0x2F, 0x0C, 0xA9, 0xFE, 0x64, 0x53, 0x69, 0x7A]);

  function prepareKeyData(fileId, password, ownerPassword, userPassword,
                          flags, revision, keyLength, encryptMetadata) {
    var hashDataSize = 40 + ownerPassword.length + fileId.length;
    var hashData = new Uint8Array(hashDataSize), i = 0, j, n;
    if (password) {
      n = Math.min(32, password.length);
      for (; i < n; ++i) {
        hashData[i] = password[i];
      }
    }
    j = 0;
    while (i < 32) {
      hashData[i++] = defaultPasswordBytes[j++];
    }
    // as now the padded password in the hashData[0..i]
    for (j = 0, n = ownerPassword.length; j < n; ++j) {
      hashData[i++] = ownerPassword[j];
    }
    hashData[i++] = flags & 0xFF;
    hashData[i++] = (flags >> 8) & 0xFF;
    hashData[i++] = (flags >> 16) & 0xFF;
    hashData[i++] = (flags >>> 24) & 0xFF;
    for (j = 0, n = fileId.length; j < n; ++j) {
      hashData[i++] = fileId[j];
    }
    if (revision >= 4 && !encryptMetadata) {
      hashData[i++] = 0xFF;
      hashData[i++] = 0xFF;
      hashData[i++] = 0xFF;
      hashData[i++] = 0xFF;
    }
    var hash = calculateMD5(hashData, 0, i);
    var keyLengthInBytes = keyLength >> 3;
    if (revision >= 3) {
      for (j = 0; j < 50; ++j) {
        hash = calculateMD5(hash, 0, keyLengthInBytes);
      }
    }
    var encryptionKey = hash.subarray(0, keyLengthInBytes);
    var cipher, checkData;

    if (revision >= 3) {
      for (i = 0; i < 32; ++i) {
        hashData[i] = defaultPasswordBytes[i];
      }
      for (j = 0, n = fileId.length; j < n; ++j) {
        hashData[i++] = fileId[j];
      }
      cipher = new ARCFourCipher(encryptionKey);
      checkData = cipher.encryptBlock(calculateMD5(hashData, 0, i));
      n = encryptionKey.length;
      var derivedKey = new Uint8Array(n), k;
      for (j = 1; j <= 19; ++j) {
        for (k = 0; k < n; ++k) {
          derivedKey[k] = encryptionKey[k] ^ j;
        }
        cipher = new ARCFourCipher(derivedKey);
        checkData = cipher.encryptBlock(checkData);
      }
      for (j = 0, n = checkData.length; j < n; ++j) {
        if (userPassword[j] != checkData[j]) {
          return null;
        }
      }
    } else {
      cipher = new ARCFourCipher(encryptionKey);
      checkData = cipher.encryptBlock(defaultPasswordBytes);
      for (j = 0, n = checkData.length; j < n; ++j) {
        if (userPassword[j] != checkData[j]) {
          return null;
        }
      }
    }
    return encryptionKey;
  }

  function decodeUserPassword(password, ownerPassword, revision, keyLength) {
    var hashData = new Uint8Array(32), i = 0, j, n;
    n = Math.min(32, password.length);
    for (; i < n; ++i) {
      hashData[i] = password[i];
    }
    j = 0;
    while (i < 32) {
      hashData[i++] = defaultPasswordBytes[j++];
    }
    var hash = calculateMD5(hashData, 0, i);
    var keyLengthInBytes = keyLength >> 3;
    if (revision >= 3) {
      for (j = 0; j < 50; ++j) {
        hash = calculateMD5(hash, 0, hash.length);
      }
    }

    var cipher, userPassword;
    if (revision >= 3) {
      userPassword = ownerPassword;
      var derivedKey = new Uint8Array(keyLengthInBytes), k;
      for (j = 19; j >= 0; j--) {
        for (k = 0; k < keyLengthInBytes; ++k) {
          derivedKey[k] = hash[k] ^ j;
        }
        cipher = new ARCFourCipher(derivedKey);
        userPassword = cipher.encryptBlock(userPassword);
      }
    } else {
      cipher = new ARCFourCipher(hash.subarray(0, keyLengthInBytes));
      userPassword = cipher.encryptBlock(ownerPassword);
    }
    return userPassword;
  }

  var identityName = Name.get('Identity');

  function CipherTransformFactory(dict, fileId, password) {
    var filter = dict.get('Filter');
    if (!isName(filter) || filter.name != 'Standard') {
      error('unknown encryption method');
    }
    this.dict = dict;
    var algorithm = dict.get('V');
    if (!isInt(algorithm) ||
        (algorithm != 1 && algorithm != 2 && algorithm != 4)) {
      error('unsupported encryption algorithm');
    }
    this.algorithm = algorithm;
    var keyLength = dict.get('Length') || 40;
    if (!isInt(keyLength) ||
        keyLength < 40 || (keyLength % 8) !== 0) {
      error('invalid key length');
    }

    // prepare keys
    var ownerPassword = stringToBytes(dict.get('O')).subarray(0, 32);
    var userPassword = stringToBytes(dict.get('U')).subarray(0, 32);
    var flags = dict.get('P');
    var revision = dict.get('R');
    var encryptMetadata = (algorithm == 4 &&  // meaningful when V is 4
      dict.get('EncryptMetadata') !== false); // makes true as default value
    this.encryptMetadata = encryptMetadata;

    var fileIdBytes = stringToBytes(fileId);
    var passwordBytes;
    if (password) {
      passwordBytes = stringToBytes(password);
    }

    var encryptionKey = prepareKeyData(fileIdBytes, passwordBytes,
                                       ownerPassword, userPassword, flags,
                                       revision, keyLength, encryptMetadata);
    if (!encryptionKey && !password) {
      throw new PasswordException('No password given',
                                  PasswordResponses.NEED_PASSWORD);
    } else if (!encryptionKey && password) {
      // Attempting use the password as an owner password
      var decodedPassword = decodeUserPassword(passwordBytes, ownerPassword,
                                               revision, keyLength);
      encryptionKey = prepareKeyData(fileIdBytes, decodedPassword,
                                     ownerPassword, userPassword, flags,
                                     revision, keyLength, encryptMetadata);
    }

    if (!encryptionKey) {
      throw new PasswordException('Incorrect Password',
                                  PasswordResponses.INCORRECT_PASSWORD);
    }

    this.encryptionKey = encryptionKey;

    if (algorithm == 4) {
      this.cf = dict.get('CF');
      this.stmf = dict.get('StmF') || identityName;
      this.strf = dict.get('StrF') || identityName;
      this.eff = dict.get('EFF') || this.strf;
    }
  }

  function buildObjectKey(num, gen, encryptionKey, isAes) {
    var key = new Uint8Array(encryptionKey.length + 9), i, n;
    for (i = 0, n = encryptionKey.length; i < n; ++i) {
      key[i] = encryptionKey[i];
    }
    key[i++] = num & 0xFF;
    key[i++] = (num >> 8) & 0xFF;
    key[i++] = (num >> 16) & 0xFF;
    key[i++] = gen & 0xFF;
    key[i++] = (gen >> 8) & 0xFF;
    if (isAes) {
      key[i++] = 0x73;
      key[i++] = 0x41;
      key[i++] = 0x6C;
      key[i++] = 0x54;
    }
    var hash = calculateMD5(key, 0, i);
    return hash.subarray(0, Math.min(encryptionKey.length + 5, 16));
  }

  function buildCipherConstructor(cf, name, num, gen, key) {
    var cryptFilter = cf.get(name.name);
    var cfm;
    if (cryptFilter !== null && cryptFilter !== undefined) {
      cfm = cryptFilter.get('CFM');
    }
    if (!cfm || cfm.name == 'None') {
      return function cipherTransformFactoryBuildCipherConstructorNone() {
        return new NullCipher();
      };
    }
    if ('V2' == cfm.name) {
      return function cipherTransformFactoryBuildCipherConstructorV2() {
        return new ARCFourCipher(buildObjectKey(num, gen, key, false));
      };
    }
    if ('AESV2' == cfm.name) {
      return function cipherTransformFactoryBuildCipherConstructorAESV2() {
        return new AES128Cipher(buildObjectKey(num, gen, key, true));
      };
    }
    error('Unknown crypto method');
  }

  CipherTransformFactory.prototype = {
    createCipherTransform:
      function CipherTransformFactory_createCipherTransform(num, gen) {
      if (this.algorithm == 4) {
        return new CipherTransform(
          buildCipherConstructor(this.cf, this.stmf,
                                 num, gen, this.encryptionKey),
          buildCipherConstructor(this.cf, this.strf,
                                 num, gen, this.encryptionKey));
      }
      // algorithms 1 and 2
      var key = buildObjectKey(num, gen, this.encryptionKey, false);
      var cipherConstructor = function buildCipherCipherConstructor() {
        return new ARCFourCipher(key);
      };
      return new CipherTransform(cipherConstructor, cipherConstructor);
    }
  };

  return CipherTransformFactory;
})();



var PatternType = {
  FUNCTION_BASED: 1,
  AXIAL: 2,
  RADIAL: 3,
  FREE_FORM_MESH: 4,
  LATTICE_FORM_MESH: 5,
  COONS_PATCH_MESH: 6,
  TENSOR_PATCH_MESH: 7
};

var Pattern = (function PatternClosure() {
  // Constructor should define this.getPattern
  function Pattern() {
    error('should not call Pattern constructor');
  }

  Pattern.prototype = {
    // Input: current Canvas context
    // Output: the appropriate fillStyle or strokeStyle
    getPattern: function Pattern_getPattern(ctx) {
      error('Should not call Pattern.getStyle: ' + ctx);
    }
  };

  Pattern.parseShading = function Pattern_parseShading(shading, matrix, xref,
                                                       res) {

    var dict = isStream(shading) ? shading.dict : shading;
    var type = dict.get('ShadingType');

    switch (type) {
      case PatternType.AXIAL:
      case PatternType.RADIAL:
        // Both radial and axial shadings are handled by RadialAxial shading.
        return new Shadings.RadialAxial(dict, matrix, xref, res);
      case PatternType.FREE_FORM_MESH:
      case PatternType.LATTICE_FORM_MESH:
      case PatternType.COONS_PATCH_MESH:
      case PatternType.TENSOR_PATCH_MESH:
        return new Shadings.Mesh(shading, matrix, xref, res);
      default:
        UnsupportedManager.notify(UNSUPPORTED_FEATURES.shadingPattern);
        return new Shadings.Dummy();
    }
  };
  return Pattern;
})();

var Shadings = {};

// A small number to offset the first/last color stops so we can insert ones to
// support extend.  Number.MIN_VALUE appears to be too small and breaks the
// extend. 1e-7 works in FF but chrome seems to use an even smaller sized number
// internally so we have to go bigger.
Shadings.SMALL_NUMBER = 1e-2;

// Radial and axial shading have very similar implementations
// If needed, the implementations can be broken into two classes
Shadings.RadialAxial = (function RadialAxialClosure() {
  function RadialAxial(dict, matrix, xref, res) {
    this.matrix = matrix;
    this.coordsArr = dict.get('Coords');
    this.shadingType = dict.get('ShadingType');
    this.type = 'Pattern';
    var cs = dict.get('ColorSpace', 'CS');
    cs = ColorSpace.parse(cs, xref, res);
    this.cs = cs;

    var t0 = 0.0, t1 = 1.0;
    if (dict.has('Domain')) {
      var domainArr = dict.get('Domain');
      t0 = domainArr[0];
      t1 = domainArr[1];
    }

    var extendStart = false, extendEnd = false;
    if (dict.has('Extend')) {
      var extendArr = dict.get('Extend');
      extendStart = extendArr[0];
      extendEnd = extendArr[1];
    }

    if (this.shadingType === PatternType.RADIAL &&
       (!extendStart || !extendEnd)) {
      // Radial gradient only currently works if either circle is fully within
      // the other circle.
      var x1 = this.coordsArr[0];
      var y1 = this.coordsArr[1];
      var r1 = this.coordsArr[2];
      var x2 = this.coordsArr[3];
      var y2 = this.coordsArr[4];
      var r2 = this.coordsArr[5];
      var distance = Math.sqrt((x1 - x2) * (x1 - x2) + (y1 - y2) * (y1 - y2));
      if (r1 <= r2 + distance &&
          r2 <= r1 + distance) {
        warn('Unsupported radial gradient.');
      }
    }

    this.extendStart = extendStart;
    this.extendEnd = extendEnd;

    var fnObj = dict.get('Function');
    var fn;
    if (isArray(fnObj)) {
      var fnArray = [];
      for (var j = 0, jj = fnObj.length; j < jj; j++) {
        var obj = xref.fetchIfRef(fnObj[j]);
        if (!isPDFFunction(obj)) {
          error('Invalid function');
        }
        fnArray.push(PDFFunction.parse(xref, obj));
      }
      fn = function radialAxialColorFunction(arg) {
        var out = [];
        for (var i = 0, ii = fnArray.length; i < ii; i++) {
          out.push(fnArray[i](arg)[0]);
        }
        return out;
      };
    } else {
      if (!isPDFFunction(fnObj)) {
        error('Invalid function');
      }
      fn = PDFFunction.parse(xref, fnObj);
    }

    // 10 samples seems good enough for now, but probably won't work
    // if there are sharp color changes. Ideally, we would implement
    // the spec faithfully and add lossless optimizations.
    var diff = t1 - t0;
    var step = diff / 10;

    var colorStops = this.colorStops = [];

    // Protect against bad domains so we don't end up in an infinte loop below.
    if (t0 >= t1 || step <= 0) {
      // Acrobat doesn't seem to handle these cases so we'll ignore for
      // now.
      info('Bad shading domain.');
      return;
    }

    var rgbColor;
    for (var i = t0; i <= t1; i += step) {
      rgbColor = cs.getRgb(fn([i]), 0);
      var cssColor = Util.makeCssRgb(rgbColor);
      colorStops.push([(i - t0) / diff, cssColor]);
    }

    var background = 'transparent';
    if (dict.has('Background')) {
      rgbColor = cs.getRgb(dict.get('Background'), 0);
      background = Util.makeCssRgb(rgbColor);
    }

    if (!extendStart) {
      // Insert a color stop at the front and offset the first real color stop
      // so it doesn't conflict with the one we insert.
      colorStops.unshift([0, background]);
      colorStops[1][0] += Shadings.SMALL_NUMBER;
    }
    if (!extendEnd) {
      // Same idea as above in extendStart but for the end.
      colorStops[colorStops.length - 1][0] -= Shadings.SMALL_NUMBER;
      colorStops.push([1, background]);
    }

    this.colorStops = colorStops;
  }

  RadialAxial.prototype = {
    getIR: function RadialAxial_getIR() {
      var coordsArr = this.coordsArr;
      var shadingType = this.shadingType;
      var type, p0, p1, r0, r1;
      if (shadingType == PatternType.AXIAL) {
        p0 = [coordsArr[0], coordsArr[1]];
        p1 = [coordsArr[2], coordsArr[3]];
        r0 = null;
        r1 = null;
        type = 'axial';
      } else if (shadingType == PatternType.RADIAL) {
        p0 = [coordsArr[0], coordsArr[1]];
        p1 = [coordsArr[3], coordsArr[4]];
        r0 = coordsArr[2];
        r1 = coordsArr[5];
        type = 'radial';
      } else {
        error('getPattern type unknown: ' + shadingType);
      }

      var matrix = this.matrix;
      if (matrix) {
        p0 = Util.applyTransform(p0, matrix);
        p1 = Util.applyTransform(p1, matrix);
      }

      return ['RadialAxial', type, this.colorStops, p0, p1, r0, r1];
    }
  };

  return RadialAxial;
})();

// All mesh shading. For now, they will be presented as set of the triangles
// to be drawn on the canvas and rgb color for each vertex.
Shadings.Mesh = (function MeshClosure() {
  function MeshStreamReader(stream, context) {
    this.stream = stream;
    this.context = context;
    this.buffer = 0;
    this.bufferLength = 0;
  }
  MeshStreamReader.prototype = {
    get hasData() {
      if (this.stream.end) {
        return this.stream.pos < this.stream.end;
      }
      if (this.bufferLength > 0) {
        return true;
      }
      var nextByte = this.stream.getByte();
      if (nextByte < 0) {
        return false;
      }
      this.buffer = nextByte;
      this.bufferLength = 8;
      return true;
    },
    readBits: function MeshStreamReader_readBits(n) {
      var buffer = this.buffer;
      var bufferLength = this.bufferLength;
      if (n === 32) {
        if (bufferLength === 0) {
          return ((this.stream.getByte() << 24) |
            (this.stream.getByte() << 16) | (this.stream.getByte() << 8) |
            this.stream.getByte()) >>> 0;
        }
        buffer = (buffer << 24) | (this.stream.getByte() << 16) |
          (this.stream.getByte() << 8) | this.stream.getByte();
        var nextByte = this.stream.getByte();
        this.buffer = nextByte & ((1 << bufferLength) - 1);
        return ((buffer << (8 - bufferLength)) |
          ((nextByte & 0xFF) >> bufferLength)) >>> 0;
      }
      if (n === 8 && bufferLength === 0) {
        return this.stream.getByte();
      }
      while (bufferLength < n) {
        buffer = (buffer << 8) | this.stream.getByte();
        bufferLength += 8;
      }
      bufferLength -= n;
      this.bufferLength = bufferLength;
      this.buffer = buffer & ((1 << bufferLength) - 1);
      return buffer >> bufferLength;
    },
    align: function MeshStreamReader_align() {
      this.buffer = 0;
      this.bufferLength = 0;
    },
    readFlag: function MeshStreamReader_readFlag() {
      return this.readBits(this.context.bitsPerFlag);
    },
    readCoordinate: function MeshStreamReader_readCoordinate() {
      var bitsPerCoordinate = this.context.bitsPerCoordinate;
      var xi = this.readBits(bitsPerCoordinate);
      var yi = this.readBits(bitsPerCoordinate);
      var decode = this.context.decode;
      var scale = bitsPerCoordinate < 32 ? 1 / ((1 << bitsPerCoordinate) - 1) :
        2.3283064365386963e-10; // 2 ^ -32
      return [
        xi * scale * (decode[1] - decode[0]) + decode[0],
        yi * scale * (decode[3] - decode[2]) + decode[2]
      ];
    },
    readComponents: function MeshStreamReader_readComponents() {
      var numComps = this.context.numComps;
      var bitsPerComponent = this.context.bitsPerComponent;
      var scale = bitsPerComponent < 32 ? 1 / ((1 << bitsPerComponent) - 1) :
        2.3283064365386963e-10; // 2 ^ -32
      var decode = this.context.decode;
      var components = [];
      for (var i = 0, j = 4; i < numComps; i++, j += 2) {
        var ci = this.readBits(bitsPerComponent);
        components.push(ci * scale * (decode[j + 1] - decode[j]) + decode[j]);
      }
      if (this.context.colorFn) {
        components = this.context.colorFn(components);
      }
      return this.context.colorSpace.getRgb(components, 0);
    }
  };

  function decodeType4Shading(mesh, reader) {
    var coords = mesh.coords;
    var colors = mesh.colors;
    var operators = [];
    var ps = []; // not maintaining cs since that will match ps
    var verticesLeft = 0; // assuming we have all data to start a new triangle
    while (reader.hasData) {
      var f = reader.readFlag();
      var coord = reader.readCoordinate();
      var color = reader.readComponents();
      if (verticesLeft === 0) { // ignoring flags if we started a triangle
        assert(0 <= f && f <= 2, 'Unknown type4 flag');
        switch (f) {
          case 0:
            verticesLeft = 3;
            break;
          case 1:
            ps.push(ps[ps.length - 2], ps[ps.length - 1]);
            verticesLeft = 1;
            break;
          case 2:
            ps.push(ps[ps.length - 3], ps[ps.length - 1]);
            verticesLeft = 1;
            break;
        }
        operators.push(f);
      }
      ps.push(coords.length);
      coords.push(coord);
      colors.push(color);
      verticesLeft--;

      reader.align();
    }

    var psPacked = new Int32Array(ps);

    mesh.figures.push({
      type: 'triangles',
      coords: psPacked,
      colors: psPacked
    });
  }

  function decodeType5Shading(mesh, reader, verticesPerRow) {
    var coords = mesh.coords;
    var colors = mesh.colors;
    var ps = []; // not maintaining cs since that will match ps
    while (reader.hasData) {
      var coord = reader.readCoordinate();
      var color = reader.readComponents();
      ps.push(coords.length);
      coords.push(coord);
      colors.push(color);
    }

    var psPacked = new Int32Array(ps);

    mesh.figures.push({
      type: 'lattice',
      coords: psPacked,
      colors: psPacked,
      verticesPerRow: verticesPerRow
    });
  }

  var MIN_SPLIT_PATCH_CHUNKS_AMOUNT = 3;
  var MAX_SPLIT_PATCH_CHUNKS_AMOUNT = 20;

  var TRIANGLE_DENSITY = 20; // count of triangles per entire mesh bounds

  var getB = (function getBClosure() {
    function buildB(count) {
      var lut = [];
      for (var i = 0; i <= count; i++) {
        var t = i / count, t_ = 1 - t;
        lut.push(new Float32Array([t_ * t_ * t_, 3 * t * t_ * t_,
          3 * t * t * t_, t * t * t]));
      }
      return lut;
    }
    var cache = [];
    return function getB(count) {
      if (!cache[count]) {
        cache[count] = buildB(count);
      }
      return cache[count];
    };
  })();

  function buildFigureFromPatch(mesh, index) {
    var figure = mesh.figures[index];
    assert(figure.type === 'patch', 'Unexpected patch mesh figure');

    var coords = mesh.coords, colors = mesh.colors;
    var pi = figure.coords;
    var ci = figure.colors;

    var figureMinX = Math.min(coords[pi[0]][0], coords[pi[3]][0],
                              coords[pi[12]][0], coords[pi[15]][0]);
    var figureMinY = Math.min(coords[pi[0]][1], coords[pi[3]][1],
                              coords[pi[12]][1], coords[pi[15]][1]);
    var figureMaxX = Math.max(coords[pi[0]][0], coords[pi[3]][0],
                              coords[pi[12]][0], coords[pi[15]][0]);
    var figureMaxY = Math.max(coords[pi[0]][1], coords[pi[3]][1],
                              coords[pi[12]][1], coords[pi[15]][1]);
    var splitXBy = Math.ceil((figureMaxX - figureMinX) * TRIANGLE_DENSITY /
                             (mesh.bounds[2] - mesh.bounds[0]));
    splitXBy = Math.max(MIN_SPLIT_PATCH_CHUNKS_AMOUNT,
               Math.min(MAX_SPLIT_PATCH_CHUNKS_AMOUNT, splitXBy));
    var splitYBy = Math.ceil((figureMaxY - figureMinY) * TRIANGLE_DENSITY /
                             (mesh.bounds[3] - mesh.bounds[1]));
    splitYBy = Math.max(MIN_SPLIT_PATCH_CHUNKS_AMOUNT,
               Math.min(MAX_SPLIT_PATCH_CHUNKS_AMOUNT, splitYBy));

    var verticesPerRow = splitXBy + 1;
    var figureCoords = new Int32Array((splitYBy + 1) * verticesPerRow);
    var figureColors = new Int32Array((splitYBy + 1) * verticesPerRow);
    var k = 0;
    var cl = new Uint8Array(3), cr = new Uint8Array(3);
    var c0 = colors[ci[0]], c1 = colors[ci[1]],
      c2 = colors[ci[2]], c3 = colors[ci[3]];
    var bRow = getB(splitYBy), bCol = getB(splitXBy);
    for (var row = 0; row <= splitYBy; row++) {
      cl[0] = ((c0[0] * (splitYBy - row) + c2[0] * row) / splitYBy) | 0;
      cl[1] = ((c0[1] * (splitYBy - row) + c2[1] * row) / splitYBy) | 0;
      cl[2] = ((c0[2] * (splitYBy - row) + c2[2] * row) / splitYBy) | 0;

      cr[0] = ((c1[0] * (splitYBy - row) + c3[0] * row) / splitYBy) | 0;
      cr[1] = ((c1[1] * (splitYBy - row) + c3[1] * row) / splitYBy) | 0;
      cr[2] = ((c1[2] * (splitYBy - row) + c3[2] * row) / splitYBy) | 0;

      for (var col = 0; col <= splitXBy; col++, k++) {
        if ((row === 0 || row === splitYBy) &&
            (col === 0 || col === splitXBy)) {
          continue;
        }
        var x = 0, y = 0;
        var q = 0;
        for (var i = 0; i <= 3; i++) {
          for (var j = 0; j <= 3; j++, q++) {
            var m = bRow[row][i] * bCol[col][j];
            x += coords[pi[q]][0] * m;
            y += coords[pi[q]][1] * m;
          }
        }
        figureCoords[k] = coords.length;
        coords.push([x, y]);
        figureColors[k] = colors.length;
        var newColor = new Uint8Array(3);
        newColor[0] = ((cl[0] * (splitXBy - col) + cr[0] * col) / splitXBy) | 0;
        newColor[1] = ((cl[1] * (splitXBy - col) + cr[1] * col) / splitXBy) | 0;
        newColor[2] = ((cl[2] * (splitXBy - col) + cr[2] * col) / splitXBy) | 0;
        colors.push(newColor);
      }
    }
    figureCoords[0] = pi[0];
    figureColors[0] = ci[0];
    figureCoords[splitXBy] = pi[3];
    figureColors[splitXBy] = ci[1];
    figureCoords[verticesPerRow * splitYBy] = pi[12];
    figureColors[verticesPerRow * splitYBy] = ci[2];
    figureCoords[verticesPerRow * splitYBy + splitXBy] = pi[15];
    figureColors[verticesPerRow * splitYBy + splitXBy] = ci[3];

    mesh.figures[index] = {
      type: 'lattice',
      coords: figureCoords,
      colors: figureColors,
      verticesPerRow: verticesPerRow
    };
  }

  function decodeType6Shading(mesh, reader) {
    // A special case of Type 7. The p11, p12, p21, p22 automatically filled
    var coords = mesh.coords;
    var colors = mesh.colors;
    var ps = new Int32Array(16); // p00, p10, ..., p30, p01, ..., p33
    var cs = new Int32Array(4); // c00, c30, c03, c33
    while (reader.hasData) {
      var f = reader.readFlag();
      assert(0 <= f && f <= 3, 'Unknown type6 flag');
      var i, ii;
      var pi = coords.length;
      for (i = 0, ii = (f !== 0 ? 8 : 12); i < ii; i++) {
        coords.push(reader.readCoordinate());
      }
      var ci = colors.length;
      for (i = 0, ii = (f !== 0 ? 2 : 4); i < ii; i++) {
        colors.push(reader.readComponents());
      }
      var tmp1, tmp2, tmp3, tmp4;
      switch (f) {
        case 0:
          ps[12] = pi + 3; ps[13] = pi + 4;  ps[14] = pi + 5;  ps[15] = pi + 6;
          ps[ 8] = pi + 2; /* values for 5, 6, 9, 10 are    */ ps[11] = pi + 7;
          ps[ 4] = pi + 1; /* calculated below              */ ps[ 7] = pi + 8;
          ps[ 0] = pi;     ps[ 1] = pi + 11; ps[ 2] = pi + 10; ps[ 3] = pi + 9;
          cs[2] = ci + 1; cs[3] = ci + 2;
          cs[0] = ci;     cs[1] = ci + 3;
          break;
        case 1:
          tmp1 = ps[12]; tmp2 = ps[13]; tmp3 = ps[14]; tmp4 = ps[15];
          ps[12] = pi + 5; ps[13] = pi + 4;  ps[14] = pi + 3;  ps[15] = pi + 2;
          ps[ 8] = pi + 6; /* values for 5, 6, 9, 10 are    */ ps[11] = pi + 1;
          ps[ 4] = pi + 7; /* calculated below              */ ps[ 7] = pi;
          ps[ 0] = tmp1;   ps[ 1] = tmp2;    ps[ 2] = tmp3;    ps[ 3] = tmp4;
          tmp1 = cs[2]; tmp2 = cs[3];
          cs[2] = ci + 1; cs[3] = ci;
          cs[0] = tmp1;   cs[1] = tmp2;
          break;
        case 2:
          ps[12] = ps[15]; ps[13] = pi + 7; ps[14] = pi + 6;   ps[15] = pi + 5;
          ps[ 8] = ps[11]; /* values for 5, 6, 9, 10 are    */ ps[11] = pi + 4;
          ps[ 4] = ps[7];  /* calculated below              */ ps[ 7] = pi + 3;
          ps[ 0] = ps[3];  ps[ 1] = pi;     ps[ 2] = pi + 1;   ps[ 3] = pi + 2;
          cs[2] = cs[3]; cs[3] = ci + 1;
          cs[0] = cs[1]; cs[1] = ci;
          break;
        case 3:
          ps[12] = ps[0];  ps[13] = ps[1];   ps[14] = ps[2];   ps[15] = ps[3];
          ps[ 8] = pi;     /* values for 5, 6, 9, 10 are    */ ps[11] = pi + 7;
          ps[ 4] = pi + 1; /* calculated below              */ ps[ 7] = pi + 6;
          ps[ 0] = pi + 2; ps[ 1] = pi + 3;  ps[ 2] = pi + 4;  ps[ 3] = pi + 5;
          cs[2] = cs[0]; cs[3] = cs[1];
          cs[0] = ci;    cs[1] = ci + 1;
          break;
      }
      // set p11, p12, p21, p22
      ps[5] = coords.length;
      coords.push([
        (-4 * coords[ps[0]][0] - coords[ps[15]][0] +
          6 * (coords[ps[4]][0] + coords[ps[1]][0]) -
          2 * (coords[ps[12]][0] + coords[ps[3]][0]) +
          3 * (coords[ps[13]][0] + coords[ps[7]][0])) / 9,
        (-4 * coords[ps[0]][1] - coords[ps[15]][1] +
          6 * (coords[ps[4]][1] + coords[ps[1]][1]) -
          2 * (coords[ps[12]][1] + coords[ps[3]][1]) +
          3 * (coords[ps[13]][1] + coords[ps[7]][1])) / 9
      ]);
      ps[6] = coords.length;
      coords.push([
        (-4 * coords[ps[3]][0] - coords[ps[12]][0] +
          6 * (coords[ps[2]][0] + coords[ps[7]][0]) -
          2 * (coords[ps[0]][0] + coords[ps[15]][0]) +
          3 * (coords[ps[4]][0] + coords[ps[14]][0])) / 9,
        (-4 * coords[ps[3]][1] - coords[ps[12]][1] +
          6 * (coords[ps[2]][1] + coords[ps[7]][1]) -
          2 * (coords[ps[0]][1] + coords[ps[15]][1]) +
          3 * (coords[ps[4]][1] + coords[ps[14]][1])) / 9
      ]);
      ps[9] = coords.length;
      coords.push([
        (-4 * coords[ps[12]][0] - coords[ps[3]][0] +
          6 * (coords[ps[8]][0] + coords[ps[13]][0]) -
          2 * (coords[ps[0]][0] + coords[ps[15]][0]) +
          3 * (coords[ps[11]][0] + coords[ps[1]][0])) / 9,
        (-4 * coords[ps[12]][1] - coords[ps[3]][1] +
          6 * (coords[ps[8]][1] + coords[ps[13]][1]) -
          2 * (coords[ps[0]][1] + coords[ps[15]][1]) +
          3 * (coords[ps[11]][1] + coords[ps[1]][1])) / 9
      ]);
      ps[10] = coords.length;
      coords.push([
        (-4 * coords[ps[15]][0] - coords[ps[0]][0] +
          6 * (coords[ps[11]][0] + coords[ps[14]][0]) -
          2 * (coords[ps[12]][0] + coords[ps[3]][0]) +
          3 * (coords[ps[2]][0] + coords[ps[8]][0])) / 9,
        (-4 * coords[ps[15]][1] - coords[ps[0]][1] +
          6 * (coords[ps[11]][1] + coords[ps[14]][1]) -
          2 * (coords[ps[12]][1] + coords[ps[3]][1]) +
          3 * (coords[ps[2]][1] + coords[ps[8]][1])) / 9
      ]);
      mesh.figures.push({
        type: 'patch',
        coords: new Int32Array(ps), // making copies of ps and cs
        colors: new Int32Array(cs)
      });
    }
  }

  function decodeType7Shading(mesh, reader) {
    var coords = mesh.coords;
    var colors = mesh.colors;
    var ps = new Int32Array(16); // p00, p10, ..., p30, p01, ..., p33
    var cs = new Int32Array(4); // c00, c30, c03, c33
    while (reader.hasData) {
      var f = reader.readFlag();
      assert(0 <= f && f <= 3, 'Unknown type7 flag');
      var i, ii;
      var pi = coords.length;
      for (i = 0, ii = (f !== 0 ? 12 : 16); i < ii; i++) {
        coords.push(reader.readCoordinate());
      }
      var ci = colors.length;
      for (i = 0, ii = (f !== 0 ? 2 : 4); i < ii; i++) {
        colors.push(reader.readComponents());
      }
      var tmp1, tmp2, tmp3, tmp4;
      switch (f) {
        case 0:
          ps[12] = pi + 3; ps[13] = pi + 4;  ps[14] = pi + 5;  ps[15] = pi + 6;
          ps[ 8] = pi + 2; ps[ 9] = pi + 13; ps[10] = pi + 14; ps[11] = pi + 7;
          ps[ 4] = pi + 1; ps[ 5] = pi + 12; ps[ 6] = pi + 15; ps[ 7] = pi + 8;
          ps[ 0] = pi;     ps[ 1] = pi + 11; ps[ 2] = pi + 10; ps[ 3] = pi + 9;
          cs[2] = ci + 1; cs[3] = ci + 2;
          cs[0] = ci;     cs[1] = ci + 3;
          break;
        case 1:
          tmp1 = ps[12]; tmp2 = ps[13]; tmp3 = ps[14]; tmp4 = ps[15];
          ps[12] = pi + 5; ps[13] = pi + 4;  ps[14] = pi + 3;  ps[15] = pi + 2;
          ps[ 8] = pi + 6; ps[ 9] = pi + 11; ps[10] = pi + 10; ps[11] = pi + 1;
          ps[ 4] = pi + 7; ps[ 5] = pi + 8;  ps[ 6] = pi + 9;  ps[ 7] = pi;
          ps[ 0] = tmp1;   ps[ 1] = tmp2;    ps[ 2] = tmp3;    ps[ 3] = tmp4;
          tmp1 = cs[2]; tmp2 = cs[3];
          cs[2] = ci + 1; cs[3] = ci;
          cs[0] = tmp1;   cs[1] = tmp2;
          break;
        case 2:
          ps[12] = ps[15]; ps[13] = pi + 7; ps[14] = pi + 6;  ps[15] = pi + 5;
          ps[ 8] = ps[11]; ps[ 9] = pi + 8; ps[10] = pi + 11; ps[11] = pi + 4;
          ps[ 4] = ps[7];  ps[ 5] = pi + 9; ps[ 6] = pi + 10; ps[ 7] = pi + 3;
          ps[ 0] = ps[3];  ps[ 1] = pi;     ps[ 2] = pi + 1;  ps[ 3] = pi + 2;
          cs[2] = cs[3]; cs[3] = ci + 1;
          cs[0] = cs[1]; cs[1] = ci;
          break;
        case 3:
          ps[12] = ps[0];  ps[13] = ps[1];   ps[14] = ps[2];   ps[15] = ps[3];
          ps[ 8] = pi;     ps[ 9] = pi + 9;  ps[10] = pi + 8;  ps[11] = pi + 7;
          ps[ 4] = pi + 1; ps[ 5] = pi + 10; ps[ 6] = pi + 11; ps[ 7] = pi + 6;
          ps[ 0] = pi + 2; ps[ 1] = pi + 3;  ps[ 2] = pi + 4;  ps[ 3] = pi + 5;
          cs[2] = cs[0]; cs[3] = cs[1];
          cs[0] = ci;    cs[1] = ci + 1;
          break;
      }
      mesh.figures.push({
        type: 'patch',
        coords: new Int32Array(ps), // making copies of ps and cs
        colors: new Int32Array(cs)
      });
    }
  }

  function updateBounds(mesh) {
    var minX = mesh.coords[0][0], minY = mesh.coords[0][1],
      maxX = minX, maxY = minY;
    for (var i = 1, ii = mesh.coords.length; i < ii; i++) {
      var x = mesh.coords[i][0], y = mesh.coords[i][1];
      minX = minX > x ? x : minX;
      minY = minY > y ? y : minY;
      maxX = maxX < x ? x : maxX;
      maxY = maxY < y ? y : maxY;
    }
    mesh.bounds = [minX, minY, maxX, maxY];
  }

  function packData(mesh) {
    var i, ii, j, jj;

    var coords = mesh.coords;
    var coordsPacked = new Float32Array(coords.length * 2);
    for (i = 0, j = 0, ii = coords.length; i < ii; i++) {
      var xy = coords[i];
      coordsPacked[j++] = xy[0];
      coordsPacked[j++] = xy[1];
    }
    mesh.coords = coordsPacked;

    var colors = mesh.colors;
    var colorsPacked = new Uint8Array(colors.length * 3);
    for (i = 0, j = 0, ii = colors.length; i < ii; i++) {
      var c = colors[i];
      colorsPacked[j++] = c[0];
      colorsPacked[j++] = c[1];
      colorsPacked[j++] = c[2];
    }
    mesh.colors = colorsPacked;

    var figures = mesh.figures;
    for (i = 0, ii = figures.length; i < ii; i++) {
      var figure = figures[i], ps = figure.coords, cs = figure.colors;
      for (j = 0, jj = ps.length; j < jj; j++) {
        ps[j] *= 2;
        cs[j] *= 3;
      }
    }
  }

  function Mesh(stream, matrix, xref, res) {
    assert(isStream(stream), 'Mesh data is not a stream');
    var dict = stream.dict;
    this.matrix = matrix;
    this.shadingType = dict.get('ShadingType');
    this.type = 'Pattern';
    this.bbox = dict.get('BBox');
    var cs = dict.get('ColorSpace', 'CS');
    cs = ColorSpace.parse(cs, xref, res);
    this.cs = cs;
    this.background = dict.has('Background') ?
      cs.getRgb(dict.get('Background'), 0) : null;

    var fnObj = dict.get('Function');
    var fn;
    if (!fnObj) {
      fn = null;
    } else if (isArray(fnObj)) {
      var fnArray = [];
      for (var j = 0, jj = fnObj.length; j < jj; j++) {
        var obj = xref.fetchIfRef(fnObj[j]);
        if (!isPDFFunction(obj)) {
          error('Invalid function');
        }
        fnArray.push(PDFFunction.parse(xref, obj));
      }
      fn = function radialAxialColorFunction(arg) {
        var out = [];
        for (var i = 0, ii = fnArray.length; i < ii; i++) {
          out.push(fnArray[i](arg)[0]);
        }
        return out;
      };
    } else {
      if (!isPDFFunction(fnObj)) {
        error('Invalid function');
      }
      fn = PDFFunction.parse(xref, fnObj);
    }

    this.coords = [];
    this.colors = [];
    this.figures = [];

    var decodeContext = {
      bitsPerCoordinate: dict.get('BitsPerCoordinate'),
      bitsPerComponent: dict.get('BitsPerComponent'),
      bitsPerFlag: dict.get('BitsPerFlag'),
      decode: dict.get('Decode'),
      colorFn: fn,
      colorSpace: cs,
      numComps: fn ? 1 : cs.numComps
    };
    var reader = new MeshStreamReader(stream, decodeContext);

    var patchMesh = false;
    switch (this.shadingType) {
      case PatternType.FREE_FORM_MESH:
        decodeType4Shading(this, reader);
        break;
      case PatternType.LATTICE_FORM_MESH:
        var verticesPerRow = dict.get('VerticesPerRow') | 0;
        assert(verticesPerRow >= 2, 'Invalid VerticesPerRow');
        decodeType5Shading(this, reader, verticesPerRow);
        break;
      case PatternType.COONS_PATCH_MESH:
        decodeType6Shading(this, reader);
        patchMesh = true;
        break;
      case PatternType.TENSOR_PATCH_MESH:
        decodeType7Shading(this, reader);
        patchMesh = true;
        break;
      default:
        error('Unsupported mesh type.');
        break;
    }

    if (patchMesh) {
      // dirty bounds calculation for determining, how dense shall be triangles
      updateBounds(this);
      for (var i = 0, ii = this.figures.length; i < ii; i++) {
        buildFigureFromPatch(this, i);
      }
    }
    // calculate bounds
    updateBounds(this);

    packData(this);
  }

  Mesh.prototype = {
    getIR: function Mesh_getIR() {
      return ['Mesh', this.shadingType, this.coords, this.colors, this.figures,
        this.bounds, this.matrix, this.bbox, this.background];
    }
  };

  return Mesh;
})();

Shadings.Dummy = (function DummyClosure() {
  function Dummy() {
    this.type = 'Pattern';
  }

  Dummy.prototype = {
    getIR: function Dummy_getIR() {
      return ['Dummy'];
    }
  };
  return Dummy;
})();

function getTilingPatternIR(operatorList, dict, args) {
  var matrix = dict.get('Matrix');
  var bbox = dict.get('BBox');
  var xstep = dict.get('XStep');
  var ystep = dict.get('YStep');
  var paintType = dict.get('PaintType');
  var tilingType = dict.get('TilingType');

  return [
    'TilingPattern', args, operatorList, matrix, bbox, xstep, ystep,
    paintType, tilingType
  ];
}


var PartialEvaluator = (function PartialEvaluatorClosure() {
  function PartialEvaluator(pdfManager, xref, handler, pageIndex,
                            uniquePrefix, idCounters, fontCache) {
    this.pdfManager = pdfManager;
    this.xref = xref;
    this.handler = handler;
    this.pageIndex = pageIndex;
    this.uniquePrefix = uniquePrefix;
    this.idCounters = idCounters;
    this.fontCache = fontCache;
  }

  // Trying to minimize Date.now() usage and check every 100 time
  var TIME_SLOT_DURATION_MS = 20;
  var CHECK_TIME_EVERY = 100;
  function TimeSlotManager() {
    this.reset();
  }
  TimeSlotManager.prototype = {
    check: function TimeSlotManager_check() {
      if (++this.checked < CHECK_TIME_EVERY) {
        return false;
      }
      this.checked = 0;
      return this.endTime <= Date.now();
    },
    reset: function TimeSlotManager_reset() {
      this.endTime = Date.now() + TIME_SLOT_DURATION_MS;
      this.checked = 0;
    }
  };

  var deferred = Promise.resolve();

  var TILING_PATTERN = 1, SHADING_PATTERN = 2;

  PartialEvaluator.prototype = {
    hasBlendModes: function PartialEvaluator_hasBlendModes(resources) {
      if (!isDict(resources)) {
        return false;
      }

      var processed = Object.create(null);
      if (resources.objId) {
        processed[resources.objId] = true;
      }

      var nodes = [resources];
      while (nodes.length) {
        var key;
        var node = nodes.shift();
        // First check the current resources for blend modes.
        var graphicStates = node.get('ExtGState');
        if (isDict(graphicStates)) {
          graphicStates = graphicStates.getAll();
          for (key in graphicStates) {
            var graphicState = graphicStates[key];
            var bm = graphicState['BM'];
            if (isName(bm) && bm.name !== 'Normal') {
              return true;
            }
          }
        }
        // Descend into the XObjects to look for more resources and blend modes.
        var xObjects = node.get('XObject');
        if (!isDict(xObjects)) {
          continue;
        }
        xObjects = xObjects.getAll();
        for (key in xObjects) {
          var xObject = xObjects[key];
          if (!isStream(xObject)) {
            continue;
          }
          var xResources = xObject.dict.get('Resources');
          // Checking objId to detect an infinite loop.
          if (isDict(xResources) &&
              (!xResources.objId || !processed[xResources.objId])) {
            nodes.push(xResources);
            if (xResources.objId) {
              processed[xResources.objId] = true;
            }
          }
        }
      }
      return false;
    },

    buildFormXObject: function PartialEvaluator_buildFormXObject(resources,
                                                                 xobj, smask,
                                                                 operatorList,
                                                                 initialState) {
      var matrix = xobj.dict.get('Matrix');
      var bbox = xobj.dict.get('BBox');
      var group = xobj.dict.get('Group');
      if (group) {
        var groupOptions = {
          matrix: matrix,
          bbox: bbox,
          smask: smask,
          isolated: false,
          knockout: false
        };

        var groupSubtype = group.get('S');
        var colorSpace;
        if (isName(groupSubtype) && groupSubtype.name === 'Transparency') {
          groupOptions.isolated = (group.get('I') || false);
          groupOptions.knockout = (group.get('K') || false);
          colorSpace = (group.has('CS') ?
            ColorSpace.parse(group.get('CS'), this.xref, resources) : null);
        }

        if (smask && smask.backdrop) {
          colorSpace = colorSpace || ColorSpace.singletons.rgb;
          smask.backdrop = colorSpace.getRgb(smask.backdrop, 0);
        }

        operatorList.addOp(OPS.beginGroup, [groupOptions]);
      }

      operatorList.addOp(OPS.paintFormXObjectBegin, [matrix, bbox]);

      return this.getOperatorList(xobj,
        (xobj.dict.get('Resources') || resources), operatorList, initialState).
        then(function () {
          operatorList.addOp(OPS.paintFormXObjectEnd, []);

          if (group) {
            operatorList.addOp(OPS.endGroup, [groupOptions]);
          }
        });
    },

    buildPaintImageXObject:
        function PartialEvaluator_buildPaintImageXObject(resources, image,
                                                         inline, operatorList,
                                                         cacheKey, cache) {
      var self = this;
      var dict = image.dict;
      var w = dict.get('Width', 'W');
      var h = dict.get('Height', 'H');

      if (!(w && isNum(w)) || !(h && isNum(h))) {
        warn('Image dimensions are missing, or not numbers.');
        return;
      }
      if (PDFJS.maxImageSize !== -1 && w * h > PDFJS.maxImageSize) {
        warn('Image exceeded maximum allowed size and was removed.');
        return;
      }

      var imageMask = (dict.get('ImageMask', 'IM') || false);
      var imgData, args;
      if (imageMask) {
        // This depends on a tmpCanvas being filled with the
        // current fillStyle, such that processing the pixel
        // data can't be done here. Instead of creating a
        // complete PDFImage, only read the information needed
        // for later.

        var width = dict.get('Width', 'W');
        var height = dict.get('Height', 'H');
        var bitStrideLength = (width + 7) >> 3;
        var imgArray = image.getBytes(bitStrideLength * height);
        var decode = dict.get('Decode', 'D');
        var canTransfer = image instanceof DecodeStream;
        var inverseDecode = (!!decode && decode[0] > 0);

        imgData = PDFImage.createMask(imgArray, width, height,
                                      canTransfer, inverseDecode);
        imgData.cached = true;
        args = [imgData];
        operatorList.addOp(OPS.paintImageMaskXObject, args);
        if (cacheKey) {
          cache.key = cacheKey;
          cache.fn = OPS.paintImageMaskXObject;
          cache.args = args;
        }
        return;
      }

      var softMask = (dict.get('SMask', 'SM') || false);
      var mask = (dict.get('Mask') || false);

      var SMALL_IMAGE_DIMENSIONS = 200;
      // Inlining small images into the queue as RGB data
      if (inline && !softMask && !mask && !(image instanceof JpegStream) &&
          (w + h) < SMALL_IMAGE_DIMENSIONS) {
        var imageObj = new PDFImage(this.xref, resources, image,
                                    inline, null, null);
        // We force the use of RGBA_32BPP images here, because we can't handle
        // any other kind.
        imgData = imageObj.createImageData(/* forceRGBA = */ true);
        operatorList.addOp(OPS.paintInlineImageXObject, [imgData]);
        return;
      }

      // If there is no imageMask, create the PDFImage and a lot
      // of image processing can be done here.
      var uniquePrefix = (this.uniquePrefix || '');
      var objId = 'img_' + uniquePrefix + (++this.idCounters.obj);
      operatorList.addDependency(objId);
      args = [objId, w, h];

      if (!softMask && !mask && image instanceof JpegStream &&
          image.isNativelySupported(this.xref, resources)) {
        // These JPEGs don't need any more processing so we can just send it.
        operatorList.addOp(OPS.paintJpegXObject, args);
        this.handler.send('obj',
          [objId, this.pageIndex, 'JpegStream', image.getIR()]);
        return;
      }

      PDFImage.buildImage(self.handler, self.xref, resources, image, inline).
        then(function(imageObj) {
          var imgData = imageObj.createImageData(/* forceRGBA = */ false);
          self.handler.send('obj', [objId, self.pageIndex, 'Image', imgData],
            [imgData.data.buffer]);
        }).then(null, function (reason) {
          warn('Unable to decode image: ' + reason);
          self.handler.send('obj', [objId, self.pageIndex, 'Image', null]);
        });

      operatorList.addOp(OPS.paintImageXObject, args);
      if (cacheKey) {
        cache.key = cacheKey;
        cache.fn = OPS.paintImageXObject;
        cache.args = args;
      }
    },

    handleSMask: function PartialEvaluator_handleSmask(smask, resources,
                                                       operatorList,
                                                       stateManager) {
      var smaskContent = smask.get('G');
      var smaskOptions = {
        subtype: smask.get('S').name,
        backdrop: smask.get('BC')
      };
      return this.buildFormXObject(resources, smaskContent, smaskOptions,
                            operatorList, stateManager.state.clone());
    },

    handleTilingType:
        function PartialEvaluator_handleTilingType(fn, args, resources,
                                                   pattern, patternDict,
                                                   operatorList) {
      // Create an IR of the pattern code.
      var tilingOpList = new OperatorList();
      return this.getOperatorList(pattern,
        (patternDict.get('Resources') || resources), tilingOpList).
        then(function () {
          // Add the dependencies to the parent operator list so they are
          // resolved before sub operator list is executed synchronously.
          operatorList.addDependencies(tilingOpList.dependencies);
          operatorList.addOp(fn, getTilingPatternIR({
            fnArray: tilingOpList.fnArray,
            argsArray: tilingOpList.argsArray
          }, patternDict, args));
        });
    },

    handleSetFont:
        function PartialEvaluator_handleSetFont(resources, fontArgs, fontRef,
                                                operatorList, state) {
      // TODO(mack): Not needed?
      var fontName;
      if (fontArgs) {
        fontArgs = fontArgs.slice();
        fontName = fontArgs[0].name;
      }

      var self = this;
      return this.loadFont(fontName, fontRef, this.xref, resources).then(
          function (translated) {
        if (!translated.font.isType3Font) {
          return translated;
        }
        return translated.loadType3Data(self, resources, operatorList).then(
            function () {
          return translated;
        });
      }).then(function (translated) {
        state.font = translated.font;
        translated.send(self.handler);
        return translated.loadedName;
      });
    },

    handleText: function PartialEvaluator_handleText(chars, state) {
      var font = state.font;
      var glyphs = font.charsToGlyphs(chars);
      var isAddToPathSet = !!(state.textRenderingMode &
                              TextRenderingMode.ADD_TO_PATH_FLAG);
      if (font.data && (isAddToPathSet || PDFJS.disableFontFace)) {
        var buildPath = function (fontChar) {
          if (!font.renderer.hasBuiltPath(fontChar)) {
            var path = font.renderer.getPathJs(fontChar);
            this.handler.send('commonobj', [
              font.loadedName + '_path_' + fontChar,
              'FontPath',
              path
            ]);
          }
        }.bind(this);

        for (var i = 0, ii = glyphs.length; i < ii; i++) {
          var glyph = glyphs[i];
          if (glyph === null) {
            continue;
          }
          buildPath(glyph.fontChar);

          // If the glyph has an accent we need to build a path for its
          // fontChar too, otherwise CanvasGraphics_paintChar will fail.
          var accent = glyph.accent;
          if (accent && accent.fontChar) {
            buildPath(accent.fontChar);
          }
        }
      }

      return glyphs;
    },

    setGState: function PartialEvaluator_setGState(resources, gState,
                                                   operatorList, xref,
                                                   stateManager) {

      // TODO(mack): This should be rewritten so that this function returns
      // what should be added to the queue during each iteration
      function setGStateForKey(gStateObj, key, value) {
        switch (key) {
          case 'Type':
            break;
          case 'LW':
          case 'LC':
          case 'LJ':
          case 'ML':
          case 'D':
          case 'RI':
          case 'FL':
          case 'CA':
          case 'ca':
            gStateObj.push([key, value]);
            break;
          case 'Font':
            promise = promise.then(function () {
              return self.handleSetFont(resources, null, value[0],
                                        operatorList, stateManager.state).
                then(function (loadedName) {
                  operatorList.addDependency(loadedName);
                  gStateObj.push([key, [loadedName, value[1]]]);
                });
            });
            break;
          case 'BM':
            gStateObj.push([key, value]);
            break;
          case 'SMask':
            if (isName(value) && value.name === 'None') {
              gStateObj.push([key, false]);
              break;
            }
            var dict = xref.fetchIfRef(value);
            if (isDict(dict)) {
              promise = promise.then(function () {
                return self.handleSMask(dict, resources, operatorList,
                                        stateManager);
              });
              gStateObj.push([key, true]);
            } else {
              warn('Unsupported SMask type');
            }

            break;
          // Only generate info log messages for the following since
          // they are unlikey to have a big impact on the rendering.
          case 'OP':
          case 'op':
          case 'OPM':
          case 'BG':
          case 'BG2':
          case 'UCR':
          case 'UCR2':
          case 'TR':
          case 'TR2':
          case 'HT':
          case 'SM':
          case 'SA':
          case 'AIS':
          case 'TK':
            // TODO implement these operators.
            info('graphic state operator ' + key);
            break;
          default:
            info('Unknown graphic state operator ' + key);
            break;
        }
      }

      // This array holds the converted/processed state data.
      var gStateObj = [];
      var gStateMap = gState.map;
      var self = this;
      var promise = Promise.resolve();
      for (var key in gStateMap) {
        var value = gStateMap[key];
        setGStateForKey(gStateObj, key, value);
      }
      return promise.then(function () {
        operatorList.addOp(OPS.setGState, [gStateObj]);
      });
    },

    loadFont: function PartialEvaluator_loadFont(fontName, font, xref,
                                                 resources) {

      function errorFont() {
        return Promise.resolve(new TranslatedFont('g_font_error',
          new ErrorFont('Font ' + fontName + ' is not available'), font));
      }
      var fontRef;
      if (font) { // Loading by ref.
        assert(isRef(font));
        fontRef = font;
      } else { // Loading by name.
        var fontRes = resources.get('Font');
        if (fontRes) {
          fontRef = fontRes.getRaw(fontName);
        } else {
          warn('fontRes not available');
          return errorFont();
        }
      }
      if (this.fontCache.has(fontRef)) {
        return this.fontCache.get(fontRef);
      }

      font = xref.fetchIfRef(fontRef);
      if (!isDict(font)) {
        return errorFont();
      }

      // We are holding font.translated references just for fontRef that are not
      // dictionaries (Dict). See explanation below.
      if (font.translated) {
        return font.translated;
      }

      var fontCapability = createPromiseCapability();

      var preEvaluatedFont = this.preEvaluateFont(font, xref);
      var descriptor = preEvaluatedFont.descriptor;
      var fontID = fontRef.num + '_' + fontRef.gen;
      if (isDict(descriptor)) {
        if (!descriptor.fontAliases) {
          descriptor.fontAliases = Object.create(null);
        }

        var fontAliases = descriptor.fontAliases;
        var hash = preEvaluatedFont.hash;
        if (fontAliases[hash]) {
          var aliasFontRef = fontAliases[hash].aliasRef;
          if (aliasFontRef && this.fontCache.has(aliasFontRef)) {
            this.fontCache.putAlias(fontRef, aliasFontRef);
            return this.fontCache.get(fontRef);
          }
        }

        if (!fontAliases[hash]) {
          fontAliases[hash] = {
            fontID: Font.getFontID()
          };
        }

        fontAliases[hash].aliasRef = fontRef;
        fontID = fontAliases[hash].fontID;
      }

      // Workaround for bad PDF generators that don't reference fonts
      // properly, i.e. by not using an object identifier.
      // Check if the fontRef is a Dict (as opposed to a standard object),
      // in which case we don't cache the font and instead reference it by
      // fontName in font.loadedName below.
      var fontRefIsDict = isDict(fontRef);
      if (!fontRefIsDict) {
        this.fontCache.put(fontRef, fontCapability.promise);
      }

      // Keep track of each font we translated so the caller can
      // load them asynchronously before calling display on a page.
      font.loadedName = 'g_font_' + (fontRefIsDict ?
        fontName.replace(/\W/g, '') : fontID);

      font.translated = fontCapability.promise;

      // TODO move promises into translate font
      var translatedPromise;
      try {
        translatedPromise = Promise.resolve(
          this.translateFont(preEvaluatedFont, xref));
      } catch (e) {
        translatedPromise = Promise.reject(e);
      }

      translatedPromise.then(function (translatedFont) {
        fontCapability.resolve(new TranslatedFont(font.loadedName,
          translatedFont, font));
      }, function (reason) {
        // TODO fontCapability.reject?
        UnsupportedManager.notify(UNSUPPORTED_FEATURES.font);
        fontCapability.resolve(new TranslatedFont(font.loadedName,
          new ErrorFont(reason instanceof Error ? reason.message : reason),
          font));
      });
      return fontCapability.promise;
    },

    buildPath: function PartialEvaluator_buildPath(operatorList, fn, args) {
      var lastIndex = operatorList.length - 1;
      if (lastIndex < 0 ||
          operatorList.fnArray[lastIndex] !== OPS.constructPath) {
        operatorList.addOp(OPS.constructPath, [[fn], args]);
      } else {
        var opArgs = operatorList.argsArray[lastIndex];
        opArgs[0].push(fn);
        Array.prototype.push.apply(opArgs[1], args);
      }
    },

    handleColorN: function PartialEvaluator_handleColorN(operatorList, fn, args,
          cs, patterns, resources, xref) {
      // compile tiling patterns
      var patternName = args[args.length - 1];
      // SCN/scn applies patterns along with normal colors
      var pattern;
      if (isName(patternName) &&
          (pattern = patterns.get(patternName.name))) {
        var dict = (isStream(pattern) ? pattern.dict : pattern);
        var typeNum = dict.get('PatternType');

        if (typeNum == TILING_PATTERN) {
          var color = cs.base ? cs.base.getRgb(args, 0) : null;
          return this.handleTilingType(fn, color, resources, pattern,
                                       dict, operatorList);
        } else if (typeNum == SHADING_PATTERN) {
          var shading = dict.get('Shading');
          var matrix = dict.get('Matrix');
          pattern = Pattern.parseShading(shading, matrix, xref, resources);
          operatorList.addOp(fn, pattern.getIR());
          return Promise.resolve();
        } else {
          return Promise.reject('Unknown PatternType: ' + typeNum);
        }
      }
      // TODO shall we fail here?
      operatorList.addOp(fn, args);
      return Promise.resolve();
    },

    getOperatorList: function PartialEvaluator_getOperatorList(stream,
                                                               resources,
                                                               operatorList,
                                                               initialState) {

      var self = this;
      var xref = this.xref;
      var imageCache = {};

      assert(operatorList);

      resources = (resources || Dict.empty);
      var xobjs = (resources.get('XObject') || Dict.empty);
      var patterns = (resources.get('Pattern') || Dict.empty);
      var stateManager = new StateManager(initialState || new EvalState());
      var preprocessor = new EvaluatorPreprocessor(stream, xref, stateManager);
      var timeSlotManager = new TimeSlotManager();

      return new Promise(function next(resolve, reject) {
        timeSlotManager.reset();
        var stop, operation, i, ii, cs;
        while (!(stop = timeSlotManager.check()) &&
               (operation = preprocessor.read())) {
          var args = operation.args;
          var fn = operation.fn;

          switch (fn | 0) {
            case OPS.paintXObject:
              if (args[0].code) {
                break;
              }
              // eagerly compile XForm objects
              var name = args[0].name;
              if (imageCache.key === name) {
                operatorList.addOp(imageCache.fn, imageCache.args);
                args = [];
                continue;
              }

              var xobj = xobjs.get(name);
              if (xobj) {
                assert(isStream(xobj), 'XObject should be a stream');

                var type = xobj.dict.get('Subtype');
                assert(isName(type),
                  'XObject should have a Name subtype');

                if (type.name === 'Form') {
                  stateManager.save();
                  return self.buildFormXObject(resources, xobj, null,
                                               operatorList,
                                               stateManager.state.clone()).
                    then(function () {
                      stateManager.restore();
                      next(resolve, reject);
                    }, reject);
                } else if (type.name === 'Image') {
                  self.buildPaintImageXObject(resources, xobj, false,
                    operatorList, name, imageCache);
                  args = [];
                  continue;
                } else if (type.name === 'PS') {
                  // PostScript XObjects are unused when viewing documents.
                  // See section 4.7.1 of Adobe's PDF reference.
                  info('Ignored XObject subtype PS');
                  continue;
                } else {
                  error('Unhandled XObject subtype ' + type.name);
                }
              }
              break;
            case OPS.setFont:
              var fontSize = args[1];
              // eagerly collect all fonts
              return self.handleSetFont(resources, args, null,
                                        operatorList, stateManager.state).
                then(function (loadedName) {
                  operatorList.addDependency(loadedName);
                  operatorList.addOp(OPS.setFont, [loadedName, fontSize]);
                  next(resolve, reject);
                }, reject);
            case OPS.endInlineImage:
              var cacheKey = args[0].cacheKey;
              if (cacheKey && imageCache.key === cacheKey) {
                operatorList.addOp(imageCache.fn, imageCache.args);
                args = [];
                continue;
              }
              self.buildPaintImageXObject(resources, args[0], true,
                operatorList, cacheKey, imageCache);
              args = [];
              continue;
            case OPS.showText:
              args[0] = self.handleText(args[0], stateManager.state);
              break;
            case OPS.showSpacedText:
              var arr = args[0];
              var combinedGlyphs = [];
              var arrLength = arr.length;
              for (i = 0; i < arrLength; ++i) {
                var arrItem = arr[i];
                if (isString(arrItem)) {
                  Array.prototype.push.apply(combinedGlyphs,
                    self.handleText(arrItem, stateManager.state));
                } else if (isNum(arrItem)) {
                  combinedGlyphs.push(arrItem);
                }
              }
              args[0] = combinedGlyphs;
              fn = OPS.showText;
              break;
            case OPS.nextLineShowText:
              operatorList.addOp(OPS.nextLine);
              args[0] = self.handleText(args[0], stateManager.state);
              fn = OPS.showText;
              break;
            case OPS.nextLineSetSpacingShowText:
              operatorList.addOp(OPS.nextLine);
              operatorList.addOp(OPS.setWordSpacing, [args.shift()]);
              operatorList.addOp(OPS.setCharSpacing, [args.shift()]);
              args[0] = self.handleText(args[0], stateManager.state);
              fn = OPS.showText;
              break;
            case OPS.setTextRenderingMode:
              stateManager.state.textRenderingMode = args[0];
              break;

            case OPS.setFillColorSpace:
              stateManager.state.fillColorSpace =
                ColorSpace.parse(args[0], xref, resources);
              continue;
            case OPS.setStrokeColorSpace:
              stateManager.state.strokeColorSpace =
                ColorSpace.parse(args[0], xref, resources);
              continue;
            case OPS.setFillColor:
              cs = stateManager.state.fillColorSpace;
              args = cs.getRgb(args, 0);
              fn = OPS.setFillRGBColor;
              break;
            case OPS.setStrokeColor:
              cs = stateManager.state.strokeColorSpace;
              args = cs.getRgb(args, 0);
              fn = OPS.setStrokeRGBColor;
              break;
            case OPS.setFillGray:
              stateManager.state.fillColorSpace = ColorSpace.singletons.gray;
              args = ColorSpace.singletons.gray.getRgb(args, 0);
              fn = OPS.setFillRGBColor;
              break;
            case OPS.setStrokeGray:
              stateManager.state.strokeColorSpace = ColorSpace.singletons.gray;
              args = ColorSpace.singletons.gray.getRgb(args, 0);
              fn = OPS.setStrokeRGBColor;
              break;
            case OPS.setFillCMYKColor:
              stateManager.state.fillColorSpace = ColorSpace.singletons.cmyk;
              args = ColorSpace.singletons.cmyk.getRgb(args, 0);
              fn = OPS.setFillRGBColor;
              break;
            case OPS.setStrokeCMYKColor:
              stateManager.state.strokeColorSpace = ColorSpace.singletons.cmyk;
              args = ColorSpace.singletons.cmyk.getRgb(args, 0);
              fn = OPS.setStrokeRGBColor;
              break;
            case OPS.setFillRGBColor:
              stateManager.state.fillColorSpace = ColorSpace.singletons.rgb;
              args = ColorSpace.singletons.rgb.getRgb(args, 0);
              break;
            case OPS.setStrokeRGBColor:
              stateManager.state.strokeColorSpace = ColorSpace.singletons.rgb;
              args = ColorSpace.singletons.rgb.getRgb(args, 0);
              break;
            case OPS.setFillColorN:
              cs = stateManager.state.fillColorSpace;
              if (cs.name === 'Pattern') {
                return self.handleColorN(operatorList, OPS.setFillColorN,
                  args, cs, patterns, resources, xref).then(function() {
                    next(resolve, reject);
                  }, reject);
              }
              args = cs.getRgb(args, 0);
              fn = OPS.setFillRGBColor;
              break;
            case OPS.setStrokeColorN:
              cs = stateManager.state.strokeColorSpace;
              if (cs.name === 'Pattern') {
                return self.handleColorN(operatorList, OPS.setStrokeColorN,
                  args, cs, patterns, resources, xref).then(function() {
                    next(resolve, reject);
                  }, reject);
              }
              args = cs.getRgb(args, 0);
              fn = OPS.setStrokeRGBColor;
              break;

            case OPS.shadingFill:
              var shadingRes = resources.get('Shading');
              if (!shadingRes) {
                error('No shading resource found');
              }

              var shading = shadingRes.get(args[0].name);
              if (!shading) {
                error('No shading object found');
              }

              var shadingFill = Pattern.parseShading(shading, null, xref,
                resources);
              var patternIR = shadingFill.getIR();
              args = [patternIR];
              fn = OPS.shadingFill;
              break;
            case OPS.setGState:
              var dictName = args[0];
              var extGState = resources.get('ExtGState');

              if (!isDict(extGState) || !extGState.has(dictName.name)) {
                break;
              }

              var gState = extGState.get(dictName.name);
              return self.setGState(resources, gState, operatorList, xref,
                stateManager).then(function() {
                  next(resolve, reject);
                }, reject);
            case OPS.moveTo:
            case OPS.lineTo:
            case OPS.curveTo:
            case OPS.curveTo2:
            case OPS.curveTo3:
            case OPS.closePath:
              self.buildPath(operatorList, fn, args);
              continue;
          }
          operatorList.addOp(fn, args);
        }
        if (stop) {
          deferred.then(function () {
            next(resolve, reject);
          });
          return;
        }
        // Some PDFs don't close all restores inside object/form.
        // Closing those for them.
        for (i = 0, ii = preprocessor.savedStatesDepth; i < ii; i++) {
          operatorList.addOp(OPS.restore, []);
        }
        resolve();
      });
    },

    getTextContent: function PartialEvaluator_getTextContent(stream, resources,
                                                             stateManager) {

      stateManager = (stateManager || new StateManager(new TextState()));

      var textContent = {
        items: [],
        styles: Object.create(null)
      };
      var bidiTexts = textContent.items;
      var SPACE_FACTOR = 0.35;
      var MULTI_SPACE_FACTOR = 1.5;

      var self = this;
      var xref = this.xref;

      resources = (xref.fetchIfRef(resources) || Dict.empty);

      // The xobj is parsed iff it's needed, e.g. if there is a `DO` cmd.
      var xobjs = null;
      var xobjsCache = {};

      var preprocessor = new EvaluatorPreprocessor(stream, xref, stateManager);

      var operation;
      var textState;

      function newTextChunk() {
        var font = textState.font;
        if (!(font.loadedName in textContent.styles)) {
          textContent.styles[font.loadedName] = {
            fontFamily: font.fallbackName,
            ascent: font.ascent,
            descent: font.descent,
            vertical: font.vertical
          };
        }
        return {
          str: '',
          dir: null,
          width: 0,
          height: 0,
          transform: null,
          fontName: font.loadedName
        };
      }

      function runBidi(textChunk) {
        var bidiResult = PDFJS.bidi(textChunk.str, -1, textState.font.vertical);
        textChunk.str = bidiResult.str;
        textChunk.dir = bidiResult.dir;
        return textChunk;
      }

      function handleSetFont(fontName, fontRef) {
        return self.loadFont(fontName, fontRef, xref, resources).
          then(function (translated) {
            textState.font = translated.font;
            textState.fontMatrix = translated.font.fontMatrix ||
              FONT_IDENTITY_MATRIX;
          });
      }

      function buildTextGeometry(chars, textChunk) {
        var font = textState.font;
        textChunk = textChunk || newTextChunk();
        if (!textChunk.transform) {
          // 9.4.4 Text Space Details
          var tsm = [textState.fontSize * textState.textHScale, 0,
                     0, textState.fontSize,
                     0, textState.textRise];
          var trm = textChunk.transform = Util.transform(textState.ctm,
                                    Util.transform(textState.textMatrix, tsm));
          if (!font.vertical) {
            textChunk.height = Math.sqrt(trm[2] * trm[2] + trm[3] * trm[3]);
          } else {
            textChunk.width = Math.sqrt(trm[0] * trm[0] + trm[1] * trm[1]);
          }
        }
        var width = 0;
        var height = 0;
        var glyphs = font.charsToGlyphs(chars);
        var defaultVMetrics = font.defaultVMetrics;
        for (var i = 0; i < glyphs.length; i++) {
          var glyph = glyphs[i];
          if (!glyph) { // Previous glyph was a space.
            width += textState.wordSpacing * textState.textHScale;
            continue;
          }
          var vMetricX = null;
          var vMetricY = null;
          var glyphWidth = null;
          if (font.vertical) {
            if (glyph.vmetric) {
              glyphWidth = glyph.vmetric[0];
              vMetricX = glyph.vmetric[1];
              vMetricY = glyph.vmetric[2];
            } else {
              glyphWidth = glyph.width;
              vMetricX = glyph.width * 0.5;
              vMetricY = defaultVMetrics[2];
            }
          } else {
            glyphWidth = glyph.width;
          }

          var glyphUnicode = glyph.unicode;
          if (glyphUnicode in NormalizedUnicodes) {
            glyphUnicode = NormalizedUnicodes[glyphUnicode];
          }
          glyphUnicode = reverseIfRtl(glyphUnicode);

          // The following will calculate the x and y of the individual glyphs.
          // if (font.vertical) {
          //   tsm[4] -= vMetricX * Math.abs(textState.fontSize) *
          //             textState.fontMatrix[0];
          //   tsm[5] -= vMetricY * textState.fontSize *
          //             textState.fontMatrix[0];
          // }
          // var trm = Util.transform(textState.textMatrix, tsm);
          // var pt = Util.applyTransform([trm[4], trm[5]], textState.ctm);
          // var x = pt[0];
          // var y = pt[1];

          var tx = 0;
          var ty = 0;
          if (!font.vertical) {
            var w0 = glyphWidth * textState.fontMatrix[0];
            tx = (w0 * textState.fontSize + textState.charSpacing) *
                 textState.textHScale;
            width += tx;
          } else {
            var w1 = glyphWidth * textState.fontMatrix[0];
            ty = w1 * textState.fontSize + textState.charSpacing;
            height += ty;
          }
          textState.translateTextMatrix(tx, ty);

          textChunk.str += glyphUnicode;
        }

        var a = textState.textLineMatrix[0];
        var b = textState.textLineMatrix[1];
        var scaleLineX = Math.sqrt(a * a + b * b);
        a = textState.ctm[0];
        b = textState.ctm[1];
        var scaleCtmX = Math.sqrt(a * a + b * b);
        if (!font.vertical) {
          textChunk.width += width * scaleCtmX * scaleLineX;
        } else {
          textChunk.height += Math.abs(height * scaleCtmX * scaleLineX);
        }
        return textChunk;
      }

      var timeSlotManager = new TimeSlotManager();

      return new Promise(function next(resolve, reject) {
        timeSlotManager.reset();
        var stop;
        while (!(stop = timeSlotManager.check()) &&
               (operation = preprocessor.read())) {
          textState = stateManager.state;
          var fn = operation.fn;
          var args = operation.args;
          switch (fn | 0) {
            case OPS.setFont:
              textState.fontSize = args[1];
              return handleSetFont(args[0].name).then(function() {
                next(resolve, reject);
              }, reject);
            case OPS.setTextRise:
              textState.textRise = args[0];
              break;
            case OPS.setHScale:
              textState.textHScale = args[0] / 100;
              break;
            case OPS.setLeading:
              textState.leading = args[0];
              break;
            case OPS.moveText:
              textState.translateTextLineMatrix(args[0], args[1]);
              textState.textMatrix = textState.textLineMatrix.slice();
              break;
            case OPS.setLeadingMoveText:
              textState.leading = -args[1];
              textState.translateTextLineMatrix(args[0], args[1]);
              textState.textMatrix = textState.textLineMatrix.slice();
              break;
            case OPS.nextLine:
              textState.carriageReturn();
              break;
            case OPS.setTextMatrix:
              textState.setTextMatrix(args[0], args[1], args[2], args[3],
                args[4], args[5]);
              textState.setTextLineMatrix(args[0], args[1], args[2], args[3],
                args[4], args[5]);
              break;
            case OPS.setCharSpacing:
              textState.charSpacing = args[0];
              break;
            case OPS.setWordSpacing:
              textState.wordSpacing = args[0];
              break;
            case OPS.beginText:
              textState.textMatrix = IDENTITY_MATRIX.slice();
              textState.textLineMatrix = IDENTITY_MATRIX.slice();
              break;
            case OPS.showSpacedText:
              var items = args[0];
              var textChunk = newTextChunk();
              var offset;
              for (var j = 0, jj = items.length; j < jj; j++) {
                if (typeof items[j] === 'string') {
                  buildTextGeometry(items[j], textChunk);
                } else {
                  var val = items[j] / 1000;
                  if (!textState.font.vertical) {
                    offset = -val * textState.fontSize * textState.textHScale *
                      textState.textMatrix[0];
                    textState.translateTextMatrix(offset, 0);
                    textChunk.width += offset;
                  } else {
                    offset = -val * textState.fontSize *
                      textState.textMatrix[3];
                    textState.translateTextMatrix(0, offset);
                    textChunk.height += offset;
                  }
                  if (items[j] < 0 && textState.font.spaceWidth > 0) {
                    var fakeSpaces = -items[j] / textState.font.spaceWidth;
                    if (fakeSpaces > MULTI_SPACE_FACTOR) {
                      fakeSpaces = Math.round(fakeSpaces);
                      while (fakeSpaces--) {
                        textChunk.str += ' ';
                      }
                    } else if (fakeSpaces > SPACE_FACTOR) {
                      textChunk.str += ' ';
                    }
                  }
                }
              }
              bidiTexts.push(runBidi(textChunk));
              break;
            case OPS.showText:
              bidiTexts.push(runBidi(buildTextGeometry(args[0])));
              break;
            case OPS.nextLineShowText:
              textState.carriageReturn();
              bidiTexts.push(runBidi(buildTextGeometry(args[0])));
              break;
            case OPS.nextLineSetSpacingShowText:
              textState.wordSpacing = args[0];
              textState.charSpacing = args[1];
              textState.carriageReturn();
              bidiTexts.push(runBidi(buildTextGeometry(args[2])));
              break;
            case OPS.paintXObject:
              if (args[0].code) {
                break;
              }

              if (!xobjs) {
                xobjs = (resources.get('XObject') || Dict.empty);
              }

              var name = args[0].name;
              if (xobjsCache.key === name) {
                if (xobjsCache.texts) {
                  Util.concatenateToArray(bidiTexts, xobjsCache.texts.items);
                  Util.extendObj(textContent.styles, xobjsCache.texts.styles);
                }
                break;
              }

              var xobj = xobjs.get(name);
              if (!xobj) {
                break;
              }
              assert(isStream(xobj), 'XObject should be a stream');

              var type = xobj.dict.get('Subtype');
              assert(isName(type),
                'XObject should have a Name subtype');

              if ('Form' !== type.name) {
                xobjsCache.key = name;
                xobjsCache.texts = null;
                break;
              }

              stateManager.save();
              var matrix = xobj.dict.get('Matrix');
              if (isArray(matrix) && matrix.length === 6) {
                stateManager.transform(matrix);
              }

              return self.getTextContent(xobj,
                xobj.dict.get('Resources') || resources, stateManager).
                then(function (formTextContent) {
                  Util.concatenateToArray(bidiTexts, formTextContent.items);
                  Util.extendObj(textContent.styles, formTextContent.styles);
                  stateManager.restore();

                  xobjsCache.key = name;
                  xobjsCache.texts = formTextContent;

                  next(resolve, reject);
                }, reject);
            case OPS.setGState:
              var dictName = args[0];
              var extGState = resources.get('ExtGState');

              if (!isDict(extGState) || !extGState.has(dictName.name)) {
                break;
              }

              var gsStateMap = extGState.get(dictName.name);
              var gsStateFont = null;
              for (var key in gsStateMap) {
                if (key === 'Font') {
                  assert(!gsStateFont);
                  gsStateFont = gsStateMap[key];
                }
              }
              if (gsStateFont) {
                textState.fontSize = gsStateFont[1];
                return handleSetFont(gsStateFont[0]).then(function() {
                  next(resolve, reject);
                }, reject);
              }
              break;
          } // switch
        } // while
        if (stop) {
          deferred.then(function () {
            next(resolve, reject);
          });
          return;
        }
        resolve(textContent);
      });
    },

    extractDataStructures: function
      partialEvaluatorExtractDataStructures(dict, baseDict,
                                            xref, properties) {
      // 9.10.2
      var toUnicode = (dict.get('ToUnicode') || baseDict.get('ToUnicode'));
      if (toUnicode) {
        properties.toUnicode = this.readToUnicode(toUnicode, xref, properties);
      }
      if (properties.composite) {
        // CIDSystemInfo helps to match CID to glyphs
        var cidSystemInfo = dict.get('CIDSystemInfo');
        if (isDict(cidSystemInfo)) {
          properties.cidSystemInfo = {
            registry: cidSystemInfo.get('Registry'),
            ordering: cidSystemInfo.get('Ordering'),
            supplement: cidSystemInfo.get('Supplement')
          };
        }

        var cidToGidMap = dict.get('CIDToGIDMap');
        if (isStream(cidToGidMap)) {
          properties.cidToGidMap = this.readCidToGidMap(cidToGidMap);
        }
      }

      // Based on 9.6.6 of the spec the encoding can come from multiple places
      // and depends on the font type. The base encoding and differences are
      // read here, but the encoding that is actually used is chosen during
      // glyph mapping in the font.
      // TODO: Loading the built in encoding in the font would allow the
      // differences to be merged in here not require us to hold on to it.
      var differences = [];
      var baseEncodingName = null;
      var encoding;
      if (dict.has('Encoding')) {
        encoding = dict.get('Encoding');
        if (isDict(encoding)) {
          baseEncodingName = encoding.get('BaseEncoding');
          baseEncodingName = (isName(baseEncodingName) ?
                              baseEncodingName.name : null);
          // Load the differences between the base and original
          if (encoding.has('Differences')) {
            var diffEncoding = encoding.get('Differences');
            var index = 0;
            for (var j = 0, jj = diffEncoding.length; j < jj; j++) {
              var data = diffEncoding[j];
              if (isNum(data)) {
                index = data;
              } else {
                differences[index++] = data.name;
              }
            }
          }
        } else if (isName(encoding)) {
          baseEncodingName = encoding.name;
        } else {
          error('Encoding is not a Name nor a Dict');
        }
        // According to table 114 if the encoding is a named encoding it must be
        // one of these predefined encodings.
        if ((baseEncodingName !== 'MacRomanEncoding' &&
             baseEncodingName !== 'MacExpertEncoding' &&
             baseEncodingName !== 'WinAnsiEncoding')) {
          baseEncodingName = null;
        }
      }

      if (baseEncodingName) {
        properties.defaultEncoding = Encodings[baseEncodingName].slice();
      } else {
        encoding = (properties.type === 'TrueType' ?
                    Encodings.WinAnsiEncoding : Encodings.StandardEncoding);
        // The Symbolic attribute can be misused for regular fonts
        // Heuristic: we have to check if the font is a standard one also
        if (!!(properties.flags & FontFlags.Symbolic)) {
          encoding = (!properties.file && /Symbol/i.test(properties.name) ?
                      Encodings.SymbolSetEncoding :
                      Encodings.MacRomanEncoding);
        }
        properties.defaultEncoding = encoding;
      }

      properties.differences = differences;
      properties.baseEncodingName = baseEncodingName;
      properties.dict = dict;
    },

    readToUnicode: function PartialEvaluator_readToUnicode(toUnicode) {
      var cmapObj = toUnicode;
      if (isName(cmapObj)) {
        return CMapFactory.create(cmapObj).map;
      } else if (isStream(cmapObj)) {
        var cmap = CMapFactory.create(cmapObj).map;
        // Convert UTF-16BE
        // NOTE: cmap can be a sparse array, so use forEach instead of for(;;)
        // to iterate over all keys.
        cmap.forEach(function(token, i) {
          var str = [];
          for (var k = 0; k < token.length; k += 2) {
            var w1 = (token.charCodeAt(k) << 8) | token.charCodeAt(k + 1);
            if ((w1 & 0xF800) !== 0xD800) { // w1 < 0xD800 || w1 > 0xDFFF
              str.push(w1);
              continue;
            }
            k += 2;
            var w2 = (token.charCodeAt(k) << 8) | token.charCodeAt(k + 1);
            str.push(((w1 & 0x3ff) << 10) + (w2 & 0x3ff) + 0x10000);
          }
          cmap[i] = String.fromCharCode.apply(String, str);
        });
        return cmap;
      }
      return null;
    },

    readCidToGidMap: function PartialEvaluator_readCidToGidMap(cidToGidStream) {
      // Extract the encoding from the CIDToGIDMap
      var glyphsData = cidToGidStream.getBytes();

      // Set encoding 0 to later verify the font has an encoding
      var result = [];
      for (var j = 0, jj = glyphsData.length; j < jj; j++) {
        var glyphID = (glyphsData[j++] << 8) | glyphsData[j];
        if (glyphID === 0) {
          continue;
        }
        var code = j >> 1;
        result[code] = glyphID;
      }
      return result;
    },

    extractWidths: function PartialEvaluator_extractWidths(dict, xref,
                                                           descriptor,
                                                           properties) {
      var glyphsWidths = [];
      var defaultWidth = 0;
      var glyphsVMetrics = [];
      var defaultVMetrics;
      var i, ii, j, jj, start, code, widths;
      if (properties.composite) {
        defaultWidth = dict.get('DW') || 1000;

        widths = dict.get('W');
        if (widths) {
          for (i = 0, ii = widths.length; i < ii; i++) {
            start = widths[i++];
            code = xref.fetchIfRef(widths[i]);
            if (isArray(code)) {
              for (j = 0, jj = code.length; j < jj; j++) {
                glyphsWidths[start++] = code[j];
              }
            } else {
              var width = widths[++i];
              for (j = start; j <= code; j++) {
                glyphsWidths[j] = width;
              }
            }
          }
        }

        if (properties.vertical) {
          var vmetrics = (dict.get('DW2') || [880, -1000]);
          defaultVMetrics = [vmetrics[1], defaultWidth * 0.5, vmetrics[0]];
          vmetrics = dict.get('W2');
          if (vmetrics) {
            for (i = 0, ii = vmetrics.length; i < ii; i++) {
              start = vmetrics[i++];
              code = xref.fetchIfRef(vmetrics[i]);
              if (isArray(code)) {
                for (j = 0, jj = code.length; j < jj; j++) {
                  glyphsVMetrics[start++] = [code[j++], code[j++], code[j]];
                }
              } else {
                var vmetric = [vmetrics[++i], vmetrics[++i], vmetrics[++i]];
                for (j = start; j <= code; j++) {
                  glyphsVMetrics[j] = vmetric;
                }
              }
            }
          }
        }
      } else {
        var firstChar = properties.firstChar;
        widths = dict.get('Widths');
        if (widths) {
          j = firstChar;
          for (i = 0, ii = widths.length; i < ii; i++) {
            glyphsWidths[j++] = widths[i];
          }
          defaultWidth = (parseFloat(descriptor.get('MissingWidth')) || 0);
        } else {
          // Trying get the BaseFont metrics (see comment above).
          var baseFontName = dict.get('BaseFont');
          if (isName(baseFontName)) {
            var metrics = this.getBaseFontMetrics(baseFontName.name);

            glyphsWidths = this.buildCharCodeToWidth(metrics.widths,
                                                     properties);
            defaultWidth = metrics.defaultWidth;
          }
        }
      }

      // Heuristic: detection of monospace font by checking all non-zero widths
      var isMonospace = true;
      var firstWidth = defaultWidth;
      for (var glyph in glyphsWidths) {
        var glyphWidth = glyphsWidths[glyph];
        if (!glyphWidth) {
          continue;
        }
        if (!firstWidth) {
          firstWidth = glyphWidth;
          continue;
        }
        if (firstWidth != glyphWidth) {
          isMonospace = false;
          break;
        }
      }
      if (isMonospace) {
        properties.flags |= FontFlags.FixedPitch;
      }

      properties.defaultWidth = defaultWidth;
      properties.widths = glyphsWidths;
      properties.defaultVMetrics = defaultVMetrics;
      properties.vmetrics = glyphsVMetrics;
    },

    isSerifFont: function PartialEvaluator_isSerifFont(baseFontName) {
      // Simulating descriptor flags attribute
      var fontNameWoStyle = baseFontName.split('-')[0];
      return (fontNameWoStyle in serifFonts) ||
              (fontNameWoStyle.search(/serif/gi) !== -1);
    },

    getBaseFontMetrics: function PartialEvaluator_getBaseFontMetrics(name) {
      var defaultWidth = 0;
      var widths = [];
      var monospace = false;
      var lookupName = (stdFontMap[name] || name);

      if (!(lookupName in Metrics)) {
        // Use default fonts for looking up font metrics if the passed
        // font is not a base font
        if (this.isSerifFont(name)) {
          lookupName = 'Times-Roman';
        } else {
          lookupName = 'Helvetica';
        }
      }
      var glyphWidths = Metrics[lookupName];

      if (isNum(glyphWidths)) {
        defaultWidth = glyphWidths;
        monospace = true;
      } else {
        widths = glyphWidths;
      }

      return {
        defaultWidth: defaultWidth,
        monospace: monospace,
        widths: widths
      };
    },

    buildCharCodeToWidth:
        function PartialEvaluator_bulildCharCodeToWidth(widthsByGlyphName,
                                                        properties) {
      var widths = Object.create(null);
      var differences = properties.differences;
      var encoding = properties.defaultEncoding;
      for (var charCode = 0; charCode < 256; charCode++) {
        if (charCode in differences &&
            widthsByGlyphName[differences[charCode]]) {
          widths[charCode] = widthsByGlyphName[differences[charCode]];
          continue;
        }
        if (charCode in encoding && widthsByGlyphName[encoding[charCode]]) {
          widths[charCode] = widthsByGlyphName[encoding[charCode]];
          continue;
        }
      }
      return widths;
    },

    preEvaluateFont: function PartialEvaluator_preEvaluateFont(dict, xref) {
      var baseDict = dict;
      var type = dict.get('Subtype');
      assert(isName(type), 'invalid font Subtype');

      var composite = false;
      var uint8array;
      if (type.name == 'Type0') {
        // If font is a composite
        //  - get the descendant font
        //  - set the type according to the descendant font
        //  - get the FontDescriptor from the descendant font
        var df = dict.get('DescendantFonts');
        if (!df) {
          error('Descendant fonts are not specified');
        }
        dict = (isArray(df) ? xref.fetchIfRef(df[0]) : df);

        type = dict.get('Subtype');
        assert(isName(type), 'invalid font Subtype');
        composite = true;
      }

      var descriptor = dict.get('FontDescriptor');
      if (descriptor) {
        var hash = new MurmurHash3_64();
        var encoding = baseDict.getRaw('Encoding');
        if (isName(encoding)) {
          hash.update(encoding.name);
        } else if (isRef(encoding)) {
          hash.update(encoding.num + '_' + encoding.gen);
        }

        var toUnicode = dict.get('ToUnicode') || baseDict.get('ToUnicode');
        if (isStream(toUnicode)) {
          var stream = toUnicode.str || toUnicode;
          uint8array = stream.buffer ?
            new Uint8Array(stream.buffer.buffer, 0, stream.bufferLength) :
            new Uint8Array(stream.bytes.buffer,
                           stream.start, stream.end - stream.start);
          hash.update(uint8array);

        } else if (isName(toUnicode)) {
          hash.update(toUnicode.name);
        }

        var widths = dict.get('Widths') || baseDict.get('Widths');
        if (widths) {
          uint8array = new Uint8Array(new Uint32Array(widths).buffer);
          hash.update(uint8array);
        }
      }

      return {
        descriptor: descriptor,
        dict: dict,
        baseDict: baseDict,
        composite: composite,
        hash: hash ? hash.hexdigest() : ''
      };
    },

    translateFont: function PartialEvaluator_translateFont(preEvaluatedFont,
                                                           xref) {
      var baseDict = preEvaluatedFont.baseDict;
      var dict = preEvaluatedFont.dict;
      var composite = preEvaluatedFont.composite;
      var descriptor = preEvaluatedFont.descriptor;
      var type = dict.get('Subtype');
      var maxCharIndex = (composite ? 0xFFFF : 0xFF);
      var properties;

      if (!descriptor) {
        if (type.name == 'Type3') {
          // FontDescriptor is only required for Type3 fonts when the document
          // is a tagged pdf. Create a barbebones one to get by.
          descriptor = new Dict(null);
          descriptor.set('FontName', Name.get(type.name));
        } else {
          // Before PDF 1.5 if the font was one of the base 14 fonts, having a
          // FontDescriptor was not required.
          // This case is here for compatibility.
          var baseFontName = dict.get('BaseFont');
          if (!isName(baseFontName)) {
            error('Base font is not specified');
          }

          // Using base font name as a font name.
          baseFontName = baseFontName.name.replace(/[,_]/g, '-');
          var metrics = this.getBaseFontMetrics(baseFontName);

          // Simulating descriptor flags attribute
          var fontNameWoStyle = baseFontName.split('-')[0];
          var flags =
            (this.isSerifFont(fontNameWoStyle) ? FontFlags.Serif : 0) |
            (metrics.monospace ? FontFlags.FixedPitch : 0) |
            (symbolsFonts[fontNameWoStyle] ? FontFlags.Symbolic :
                                             FontFlags.Nonsymbolic);

          properties = {
            type: type.name,
            name: baseFontName,
            widths: metrics.widths,
            defaultWidth: metrics.defaultWidth,
            flags: flags,
            firstChar: 0,
            lastChar: maxCharIndex
          };
          this.extractDataStructures(dict, dict, xref, properties);
          properties.widths = this.buildCharCodeToWidth(metrics.widths,
                                                        properties);
          return new Font(baseFontName, null, properties);
        }
      }

      // According to the spec if 'FontDescriptor' is declared, 'FirstChar',
      // 'LastChar' and 'Widths' should exist too, but some PDF encoders seem
      // to ignore this rule when a variant of a standart font is used.
      // TODO Fill the width array depending on which of the base font this is
      // a variant.
      var firstChar = (dict.get('FirstChar') || 0);
      var lastChar = (dict.get('LastChar') || maxCharIndex);

      var fontName = descriptor.get('FontName');
      var baseFont = dict.get('BaseFont');
      // Some bad PDFs have a string as the font name.
      if (isString(fontName)) {
        fontName = Name.get(fontName);
      }
      if (isString(baseFont)) {
        baseFont = Name.get(baseFont);
      }

      if (type.name !== 'Type3') {
        var fontNameStr = fontName && fontName.name;
        var baseFontStr = baseFont && baseFont.name;
        if (fontNameStr !== baseFontStr) {
          info('The FontDescriptor\'s FontName is "' + fontNameStr +
               '" but should be the same as the Font\'s BaseFont "' +
               baseFontStr + '"');
          // Workaround for cases where e.g. fontNameStr = 'Arial' and
          // baseFontStr = 'Arial,Bold' (needed when no font file is embedded).
          if (fontNameStr && baseFontStr &&
              baseFontStr.search(fontNameStr) === 0) {
            fontName = baseFont;
          }
        }
      }
      fontName = (fontName || baseFont);

      assert(isName(fontName), 'invalid font name');

      var fontFile = descriptor.get('FontFile', 'FontFile2', 'FontFile3');
      if (fontFile) {
        if (fontFile.dict) {
          var subtype = fontFile.dict.get('Subtype');
          if (subtype) {
            subtype = subtype.name;
          }
          var length1 = fontFile.dict.get('Length1');
          var length2 = fontFile.dict.get('Length2');
        }
      }

      properties = {
        type: type.name,
        name: fontName.name,
        subtype: subtype,
        file: fontFile,
        length1: length1,
        length2: length2,
        loadedName: baseDict.loadedName,
        composite: composite,
        wideChars: composite,
        fixedPitch: false,
        fontMatrix: (dict.get('FontMatrix') || FONT_IDENTITY_MATRIX),
        firstChar: firstChar || 0,
        lastChar: (lastChar || maxCharIndex),
        bbox: descriptor.get('FontBBox'),
        ascent: descriptor.get('Ascent'),
        descent: descriptor.get('Descent'),
        xHeight: descriptor.get('XHeight'),
        capHeight: descriptor.get('CapHeight'),
        flags: descriptor.get('Flags'),
        italicAngle: descriptor.get('ItalicAngle'),
        coded: false
      };

      if (composite) {
        var cidEncoding = baseDict.get('Encoding');
        if (isName(cidEncoding)) {
          properties.cidEncoding = cidEncoding.name;
        }
        properties.cMap = CMapFactory.create(cidEncoding,
          { url: PDFJS.cMapUrl, packed: PDFJS.cMapPacked }, null);
        properties.vertical = properties.cMap.vertical;
      }
      this.extractDataStructures(dict, baseDict, xref, properties);
      this.extractWidths(dict, xref, descriptor, properties);

      if (type.name === 'Type3') {
        properties.isType3Font = true;
      }

      return new Font(fontName.name, fontFile, properties);
    }
  };

  return PartialEvaluator;
})();

var TranslatedFont = (function TranslatedFontClosure() {
  function TranslatedFont(loadedName, font, dict) {
    this.loadedName = loadedName;
    this.font = font;
    this.dict = dict;
    this.type3Loaded = null;
    this.sent = false;
  }
  TranslatedFont.prototype = {
    send: function (handler) {
      if (this.sent) {
        return;
      }
      var fontData = this.font.exportData();
      handler.send('commonobj', [
        this.loadedName,
        'Font',
        fontData
      ]);
      this.sent = true;
    },
    loadType3Data: function (evaluator, resources, parentOperatorList) {
      assert(this.font.isType3Font);

      if (this.type3Loaded) {
        return this.type3Loaded;
      }

      var translatedFont = this.font;
      var loadCharProcsPromise = Promise.resolve();
      var charProcs = this.dict.get('CharProcs').getAll();
      var fontResources = this.dict.get('Resources') || resources;
      var charProcKeys = Object.keys(charProcs);
      var charProcOperatorList = {};
      for (var i = 0, n = charProcKeys.length; i < n; ++i) {
        loadCharProcsPromise = loadCharProcsPromise.then(function (key) {
          var glyphStream = charProcs[key];
          var operatorList = new OperatorList();
          return evaluator.getOperatorList(glyphStream, fontResources,
            operatorList).
            then(function () {
              charProcOperatorList[key] = operatorList.getIR();

              // Add the dependencies to the parent operator list so they are
              // resolved before sub operator list is executed synchronously.
              parentOperatorList.addDependencies(operatorList.dependencies);
            });
        }.bind(this, charProcKeys[i]));
      }
      this.type3Loaded = loadCharProcsPromise.then(function () {
        translatedFont.charProcOperatorList = charProcOperatorList;
      });
      return this.type3Loaded;
    }
  };
  return TranslatedFont;
})();

var OperatorList = (function OperatorListClosure() {
  var CHUNK_SIZE = 1000;
  var CHUNK_SIZE_ABOUT = CHUNK_SIZE - 5; // close to chunk size

  function getTransfers(queue) {
    var transfers = [];
    var fnArray = queue.fnArray, argsArray = queue.argsArray;
    for (var i = 0, ii = queue.length; i < ii; i++) {
      switch (fnArray[i]) {
        case OPS.paintInlineImageXObject:
        case OPS.paintInlineImageXObjectGroup:
        case OPS.paintImageMaskXObject:
          var arg = argsArray[i][0]; // first param in imgData
          if (!arg.cached) {
            transfers.push(arg.data.buffer);
          }
          break;
      }
    }
    return transfers;
  }

  function OperatorList(intent, messageHandler, pageIndex) {
    this.messageHandler = messageHandler;
    this.fnArray = [];
    this.argsArray = [];
    this.dependencies = {};
    this.pageIndex = pageIndex;
    this.intent = intent;
  }

  OperatorList.prototype = {
    get length() {
      return this.argsArray.length;
    },

    addOp: function(fn, args) {
      this.fnArray.push(fn);
      this.argsArray.push(args);
      if (this.messageHandler) {
        if (this.fnArray.length >= CHUNK_SIZE) {
          this.flush();
        } else if (this.fnArray.length >= CHUNK_SIZE_ABOUT &&
                   (fn === OPS.restore || fn === OPS.endText)) {
          // heuristic to flush on boundary of restore or endText
          this.flush();
        }
      }
    },

    addDependency: function(dependency) {
      if (dependency in this.dependencies) {
        return;
      }
      this.dependencies[dependency] = true;
      this.addOp(OPS.dependency, [dependency]);
    },

    addDependencies: function(dependencies) {
      for (var key in dependencies) {
        this.addDependency(key);
      }
    },

    addOpList: function(opList) {
      Util.extendObj(this.dependencies, opList.dependencies);
      for (var i = 0, ii = opList.length; i < ii; i++) {
        this.addOp(opList.fnArray[i], opList.argsArray[i]);
      }
    },

    getIR: function() {
      return {
        fnArray: this.fnArray,
        argsArray: this.argsArray,
        length: this.length
      };
    },

    flush: function(lastChunk) {
      new QueueOptimizer().optimize(this);
      var transfers = getTransfers(this);
      this.messageHandler.send('RenderPageChunk', {
        operatorList: {
          fnArray: this.fnArray,
          argsArray: this.argsArray,
          lastChunk: lastChunk,
          length: this.length
        },
        pageIndex: this.pageIndex,
        intent: this.intent
      }, transfers);
      this.dependencies = {};
      this.fnArray.length = 0;
      this.argsArray.length = 0;
    }
  };

  return OperatorList;
})();

var StateManager = (function StateManagerClosure() {
  function StateManager(initialState) {
    this.state = initialState;
    this.stateStack = [];
  }
  StateManager.prototype = {
    save: function () {
      var old = this.state;
      this.stateStack.push(this.state);
      this.state = old.clone();
    },
    restore: function () {
      var prev = this.stateStack.pop();
      if (prev) {
        this.state = prev;
      }
    },
    transform: function (args) {
      this.state.ctm = Util.transform(this.state.ctm, args);
    }
  };
  return StateManager;
})();

var TextState = (function TextStateClosure() {
  function TextState() {
    this.ctm = new Float32Array(IDENTITY_MATRIX);
    this.fontSize = 0;
    this.font = null;
    this.fontMatrix = FONT_IDENTITY_MATRIX;
    this.textMatrix = IDENTITY_MATRIX.slice();
    this.textLineMatrix = IDENTITY_MATRIX.slice();
    this.charSpacing = 0;
    this.wordSpacing = 0;
    this.leading = 0;
    this.textHScale = 1;
    this.textRise = 0;
  }

  TextState.prototype = {
    setTextMatrix: function TextState_setTextMatrix(a, b, c, d, e, f) {
      var m = this.textMatrix;
      m[0] = a; m[1] = b; m[2] = c; m[3] = d; m[4] = e; m[5] = f;
    },
    setTextLineMatrix: function TextState_setTextMatrix(a, b, c, d, e, f) {
      var m = this.textLineMatrix;
      m[0] = a; m[1] = b; m[2] = c; m[3] = d; m[4] = e; m[5] = f;
    },
    translateTextMatrix: function TextState_translateTextMatrix(x, y) {
      var m = this.textMatrix;
      m[4] = m[0] * x + m[2] * y + m[4];
      m[5] = m[1] * x + m[3] * y + m[5];
    },
    translateTextLineMatrix: function TextState_translateTextMatrix(x, y) {
      var m = this.textLineMatrix;
      m[4] = m[0] * x + m[2] * y + m[4];
      m[5] = m[1] * x + m[3] * y + m[5];
    },
    calcRenderMatrix: function TextState_calcRendeMatrix(ctm) {
      // 9.4.4 Text Space Details
      var tsm = [this.fontSize * this.textHScale, 0,
                0, this.fontSize,
                0, this.textRise];
      return Util.transform(ctm, Util.transform(this.textMatrix, tsm));
    },
    carriageReturn: function TextState_carriageReturn() {
      this.translateTextLineMatrix(0, -this.leading);
      this.textMatrix = this.textLineMatrix.slice();
    },
    clone: function TextState_clone() {
      var clone = Object.create(this);
      clone.textMatrix = this.textMatrix.slice();
      clone.textLineMatrix = this.textLineMatrix.slice();
      clone.fontMatrix = this.fontMatrix.slice();
      return clone;
    }
  };
  return TextState;
})();

var EvalState = (function EvalStateClosure() {
  function EvalState() {
    this.ctm = new Float32Array(IDENTITY_MATRIX);
    this.font = null;
    this.textRenderingMode = TextRenderingMode.FILL;
    this.fillColorSpace = ColorSpace.singletons.gray;
    this.strokeColorSpace = ColorSpace.singletons.gray;
  }
  EvalState.prototype = {
    clone: function CanvasExtraState_clone() {
      return Object.create(this);
    },
  };
  return EvalState;
})();

var EvaluatorPreprocessor = (function EvaluatorPreprocessorClosure() {
  // Specifies properties for each command
  //
  // If variableArgs === true: [0, `numArgs`] expected
  // If variableArgs === false: exactly `numArgs` expected
  var OP_MAP = {
    // Graphic state
    w: { id: OPS.setLineWidth, numArgs: 1, variableArgs: false },
    J: { id: OPS.setLineCap, numArgs: 1, variableArgs: false },
    j: { id: OPS.setLineJoin, numArgs: 1, variableArgs: false },
    M: { id: OPS.setMiterLimit, numArgs: 1, variableArgs: false },
    d: { id: OPS.setDash, numArgs: 2, variableArgs: false },
    ri: { id: OPS.setRenderingIntent, numArgs: 1, variableArgs: false },
    i: { id: OPS.setFlatness, numArgs: 1, variableArgs: false },
    gs: { id: OPS.setGState, numArgs: 1, variableArgs: false },
    q: { id: OPS.save, numArgs: 0, variableArgs: false },
    Q: { id: OPS.restore, numArgs: 0, variableArgs: false },
    cm: { id: OPS.transform, numArgs: 6, variableArgs: false },

    // Path
    m: { id: OPS.moveTo, numArgs: 2, variableArgs: false },
    l: { id: OPS.lineTo, numArgs: 2, variableArgs: false },
    c: { id: OPS.curveTo, numArgs: 6, variableArgs: false },
    v: { id: OPS.curveTo2, numArgs: 4, variableArgs: false },
    y: { id: OPS.curveTo3, numArgs: 4, variableArgs: false },
    h: { id: OPS.closePath, numArgs: 0, variableArgs: false },
    re: { id: OPS.rectangle, numArgs: 4, variableArgs: false },
    S: { id: OPS.stroke, numArgs: 0, variableArgs: false },
    s: { id: OPS.closeStroke, numArgs: 0, variableArgs: false },
    f: { id: OPS.fill, numArgs: 0, variableArgs: false },
    F: { id: OPS.fill, numArgs: 0, variableArgs: false },
    'f*': { id: OPS.eoFill, numArgs: 0, variableArgs: false },
    B: { id: OPS.fillStroke, numArgs: 0, variableArgs: false },
    'B*': { id: OPS.eoFillStroke, numArgs: 0, variableArgs: false },
    b: { id: OPS.closeFillStroke, numArgs: 0, variableArgs: false },
    'b*': { id: OPS.closeEOFillStroke, numArgs: 0, variableArgs: false },
    n: { id: OPS.endPath, numArgs: 0, variableArgs: false },

    // Clipping
    W: { id: OPS.clip, numArgs: 0, variableArgs: false },
    'W*': { id: OPS.eoClip, numArgs: 0, variableArgs: false },

    // Text
    BT: { id: OPS.beginText, numArgs: 0, variableArgs: false },
    ET: { id: OPS.endText, numArgs: 0, variableArgs: false },
    Tc: { id: OPS.setCharSpacing, numArgs: 1, variableArgs: false },
    Tw: { id: OPS.setWordSpacing, numArgs: 1, variableArgs: false },
    Tz: { id: OPS.setHScale, numArgs: 1, variableArgs: false },
    TL: { id: OPS.setLeading, numArgs: 1, variableArgs: false },
    Tf: { id: OPS.setFont, numArgs: 2, variableArgs: false },
    Tr: { id: OPS.setTextRenderingMode, numArgs: 1, variableArgs: false },
    Ts: { id: OPS.setTextRise, numArgs: 1, variableArgs: false },
    Td: { id: OPS.moveText, numArgs: 2, variableArgs: false },
    TD: { id: OPS.setLeadingMoveText, numArgs: 2, variableArgs: false },
    Tm: { id: OPS.setTextMatrix, numArgs: 6, variableArgs: false },
    'T*': { id: OPS.nextLine, numArgs: 0, variableArgs: false },
    Tj: { id: OPS.showText, numArgs: 1, variableArgs: false },
    TJ: { id: OPS.showSpacedText, numArgs: 1, variableArgs: false },
    '\'': { id: OPS.nextLineShowText, numArgs: 1, variableArgs: false },
    '"': { id: OPS.nextLineSetSpacingShowText, numArgs: 3,
           variableArgs: false },

    // Type3 fonts
    d0: { id: OPS.setCharWidth, numArgs: 2, variableArgs: false },
    d1: { id: OPS.setCharWidthAndBounds, numArgs: 6, variableArgs: false },

    // Color
    CS: { id: OPS.setStrokeColorSpace, numArgs: 1, variableArgs: false },
    cs: { id: OPS.setFillColorSpace, numArgs: 1, variableArgs: false },
    SC: { id: OPS.setStrokeColor, numArgs: 4, variableArgs: true },
    SCN: { id: OPS.setStrokeColorN, numArgs: 33, variableArgs: true },
    sc: { id: OPS.setFillColor, numArgs: 4, variableArgs: true },
    scn: { id: OPS.setFillColorN, numArgs: 33, variableArgs: true },
    G: { id: OPS.setStrokeGray, numArgs: 1, variableArgs: false },
    g: { id: OPS.setFillGray, numArgs: 1, variableArgs: false },
    RG: { id: OPS.setStrokeRGBColor, numArgs: 3, variableArgs: false },
    rg: { id: OPS.setFillRGBColor, numArgs: 3, variableArgs: false },
    K: { id: OPS.setStrokeCMYKColor, numArgs: 4, variableArgs: false },
    k: { id: OPS.setFillCMYKColor, numArgs: 4, variableArgs: false },

    // Shading
    sh: { id: OPS.shadingFill, numArgs: 1, variableArgs: false },

    // Images
    BI: { id: OPS.beginInlineImage, numArgs: 0, variableArgs: false },
    ID: { id: OPS.beginImageData, numArgs: 0, variableArgs: false },
    EI: { id: OPS.endInlineImage, numArgs: 1, variableArgs: false },

    // XObjects
    Do: { id: OPS.paintXObject, numArgs: 1, variableArgs: false },
    MP: { id: OPS.markPoint, numArgs: 1, variableArgs: false },
    DP: { id: OPS.markPointProps, numArgs: 2, variableArgs: false },
    BMC: { id: OPS.beginMarkedContent, numArgs: 1, variableArgs: false },
    BDC: { id: OPS.beginMarkedContentProps, numArgs: 2,
           variableArgs: false },
    EMC: { id: OPS.endMarkedContent, numArgs: 0, variableArgs: false },

    // Compatibility
    BX: { id: OPS.beginCompat, numArgs: 0, variableArgs: false },
    EX: { id: OPS.endCompat, numArgs: 0, variableArgs: false },

    // (reserved partial commands for the lexer)
    BM: null,
    BD: null,
    'true': null,
    fa: null,
    fal: null,
    fals: null,
    'false': null,
    nu: null,
    nul: null,
    'null': null
  };

  function EvaluatorPreprocessor(stream, xref, stateManager) {
    // TODO(mduan): pass array of knownCommands rather than OP_MAP
    // dictionary
    this.parser = new Parser(new Lexer(stream, OP_MAP), false, xref);
    this.stateManager = stateManager;
    this.nonProcessedArgs = [];
  }

  EvaluatorPreprocessor.prototype = {
    get savedStatesDepth() {
      return this.stateManager.stateStack.length;
    },

    read: function EvaluatorPreprocessor_read() {
      var args = [];
      while (true) {
        var obj = this.parser.getObj();
        if (isEOF(obj)) {
          return null; // no more commands
        }
        if (!isCmd(obj)) {
          // argument
          if (obj !== null && obj !== undefined) {
            args.push((obj instanceof Dict ? obj.getAll() : obj));
            assert(args.length <= 33, 'Too many arguments');
          }
          continue;
        }

        var cmd = obj.cmd;
        // Check that the command is valid
        var opSpec = OP_MAP[cmd];
        if (!opSpec) {
          warn('Unknown command "' + cmd + '"');
          continue;
        }

        var fn = opSpec.id;

        // Some post script commands can be nested, e.g. /F2 /GS2 gs 5.711 Tf
        if (!opSpec.variableArgs && args.length !== opSpec.numArgs) {
          while (args.length > opSpec.numArgs) {
            this.nonProcessedArgs.push(args.shift());
          }

          while (args.length < opSpec.numArgs && this.nonProcessedArgs.length) {
            args.unshift(this.nonProcessedArgs.pop());
          }
        }

        // Validate the number of arguments for the command
        if (opSpec.variableArgs) {
          if (args.length > opSpec.numArgs) {
            info('Command ' + fn + ': expected [0,' + opSpec.numArgs +
                 '] args, but received ' + args.length + ' args');
          }
        } else {
          if (args.length < opSpec.numArgs) {
            // If we receive too few args, it's not possible to possible
            // to execute the command, so skip the command
            info('Command ' + fn + ': because expected ' +
                 opSpec.numArgs + ' args, but received ' + args.length +
                 ' args; skipping');
            args = [];
            continue;
          } else if (args.length > opSpec.numArgs) {
            info('Command ' + fn + ': expected ' + opSpec.numArgs +
                 ' args, but received ' + args.length + ' args');
          }
        }

        // TODO figure out how to type-check vararg functions
        this.preprocessCommand(fn, args);

        return { fn: fn, args: args };
      }
    },

    preprocessCommand:
        function EvaluatorPreprocessor_preprocessCommand(fn, args) {
      switch (fn | 0) {
        case OPS.save:
          this.stateManager.save();
          break;
        case OPS.restore:
          this.stateManager.restore();
          break;
        case OPS.transform:
          this.stateManager.transform(args);
          break;
      }
    }
  };
  return EvaluatorPreprocessor;
})();

var QueueOptimizer = (function QueueOptimizerClosure() {
  function addState(parentState, pattern, fn) {
    var state = parentState;
    for (var i = 0, ii = pattern.length - 1; i < ii; i++) {
      var item = pattern[i];
      state = (state[item] || (state[item] = []));
    }
    state[pattern[pattern.length - 1]] = fn;
  }

  function handlePaintSolidColorImageMask(index, count, fnArray, argsArray) {
    // Handles special case of mainly LaTeX documents which
    // use image masks to draw lines with the current fill style.
    // 'count' groups of (save, transform, paintImageMaskXObject, restore)+
    // have been found at index.
    for (var i = 0; i < count; i++) {
      var arg = argsArray[index + 4 * i + 2];
      var imageMask = arg.length == 1 && arg[0];
      if (imageMask && imageMask.width == 1 && imageMask.height == 1 &&
          (!imageMask.data.length || (imageMask.data.length == 1 &&
                                      imageMask.data[0] === 0))) {
        fnArray[index + 4 * i + 2] = OPS.paintSolidColorImageMask;
        continue;
      }
      break;
    }
    return count - i;
  }

  var InitialState = [];

  addState(InitialState,
    [OPS.save, OPS.transform, OPS.paintInlineImageXObject, OPS.restore],
    function foundInlineImageGroup(context) {
      // grouping paintInlineImageXObject's into paintInlineImageXObjectGroup
      // searching for (save, transform, paintInlineImageXObject, restore)+
      var MIN_IMAGES_IN_INLINE_IMAGES_BLOCK = 10;
      var MAX_IMAGES_IN_INLINE_IMAGES_BLOCK = 200;
      var MAX_WIDTH = 1000;
      var IMAGE_PADDING = 1;

      var fnArray = context.fnArray, argsArray = context.argsArray;
      var j = context.currentOperation - 3, i = j + 4;
      var ii = fnArray.length;

      for (; i < ii && fnArray[i - 4] === fnArray[i]; i++) {}
      var count = Math.min((i - j) >> 2, MAX_IMAGES_IN_INLINE_IMAGES_BLOCK);
      if (count < MIN_IMAGES_IN_INLINE_IMAGES_BLOCK) {
        context.currentOperation = i - 1;
        return;
      }
      // assuming that heights of those image is too small (~1 pixel)
      // packing as much as possible by lines
      var maxX = 0;
      var map = [], maxLineHeight = 0;
      var currentX = IMAGE_PADDING, currentY = IMAGE_PADDING;
      var q;
      for (q = 0; q < count; q++) {
        var transform = argsArray[j + (q << 2) + 1];
        var img = argsArray[j + (q << 2) + 2][0];
        if (currentX + img.width > MAX_WIDTH) {
          // starting new line
          maxX = Math.max(maxX, currentX);
          currentY += maxLineHeight + 2 * IMAGE_PADDING;
          currentX = 0;
          maxLineHeight = 0;
        }
        map.push({
          transform: transform,
          x: currentX, y: currentY,
          w: img.width, h: img.height
        });
        currentX += img.width + 2 * IMAGE_PADDING;
        maxLineHeight = Math.max(maxLineHeight, img.height);
      }
      var imgWidth = Math.max(maxX, currentX) + IMAGE_PADDING;
      var imgHeight = currentY + maxLineHeight + IMAGE_PADDING;
      var imgData = new Uint8Array(imgWidth * imgHeight * 4);
      var imgRowSize = imgWidth << 2;
      for (q = 0; q < count; q++) {
        var data = argsArray[j + (q << 2) + 2][0].data;
        // copy image by lines and extends pixels into padding
        var rowSize = map[q].w << 2;
        var dataOffset = 0;
        var offset = (map[q].x + map[q].y * imgWidth) << 2;
        imgData.set(data.subarray(0, rowSize), offset - imgRowSize);
        for (var k = 0, kk = map[q].h; k < kk; k++) {
          imgData.set(data.subarray(dataOffset, dataOffset + rowSize), offset);
          dataOffset += rowSize;
          offset += imgRowSize;
        }
        imgData.set(data.subarray(dataOffset - rowSize, dataOffset), offset);
        while (offset >= 0) {
          data[offset - 4] = data[offset];
          data[offset - 3] = data[offset + 1];
          data[offset - 2] = data[offset + 2];
          data[offset - 1] = data[offset + 3];
          data[offset + rowSize] = data[offset + rowSize - 4];
          data[offset + rowSize + 1] = data[offset + rowSize - 3];
          data[offset + rowSize + 2] = data[offset + rowSize - 2];
          data[offset + rowSize + 3] = data[offset + rowSize - 1];
          offset -= imgRowSize;
        }
      }
      // replacing queue items
      fnArray.splice(j, count * 4, OPS.paintInlineImageXObjectGroup);
      argsArray.splice(j, count * 4,
        [{ width: imgWidth, height: imgHeight, kind: ImageKind.RGBA_32BPP,
           data: imgData }, map]);
      context.currentOperation = j;
    });

  addState(InitialState,
    [OPS.save, OPS.transform, OPS.paintImageMaskXObject, OPS.restore],
    function foundImageMaskGroup(context) {
      // grouping paintImageMaskXObject's into paintImageMaskXObjectGroup
      // searching for (save, transform, paintImageMaskXObject, restore)+
      var MIN_IMAGES_IN_MASKS_BLOCK = 10;
      var MAX_IMAGES_IN_MASKS_BLOCK = 100;
      var MAX_SAME_IMAGES_IN_MASKS_BLOCK = 1000;

      var fnArray = context.fnArray, argsArray = context.argsArray;
      var j = context.currentOperation - 3, i = j + 4;
      var ii = fnArray.length, q;

      for (; i < ii && fnArray[i - 4] === fnArray[i]; i++) {}
      var count = (i - j) >> 2;
      count = handlePaintSolidColorImageMask(j, count, fnArray, argsArray);
      if (count < MIN_IMAGES_IN_MASKS_BLOCK) {
        context.currentOperation = i - 1;
        return;
      }

      var isSameImage = false;
      var transformArgs;
      if (argsArray[j + 1][1] === 0 && argsArray[j + 1][2] === 0) {
        i = j + 4;
        isSameImage = true;
        for (q = 1; q < count; q++, i += 4) {
          var prevTransformArgs = argsArray[i - 3];
          transformArgs = argsArray[i + 1];
          if (argsArray[i - 2][0] !== argsArray[i + 2][0] ||
              prevTransformArgs[0] !== transformArgs[0] ||
              prevTransformArgs[1] !== transformArgs[1] ||
              prevTransformArgs[2] !== transformArgs[2] ||
              prevTransformArgs[3] !== transformArgs[3]) {
            if (q < MIN_IMAGES_IN_MASKS_BLOCK) {
              isSameImage = false;
            } else {
              count = q;
            }
            break; // different image or transform
          }
        }
      }

      if (isSameImage) {
        count = Math.min(count, MAX_SAME_IMAGES_IN_MASKS_BLOCK);
        var positions = new Float32Array(count * 2);
        i = j + 1;
        for (q = 0; q < count; q++) {
          transformArgs = argsArray[i];
          positions[(q << 1)] = transformArgs[4];
          positions[(q << 1) + 1] = transformArgs[5];
          i += 4;
        }

        // replacing queue items
        fnArray.splice(j, count * 4, OPS.paintImageMaskXObjectRepeat);
        argsArray.splice(j, count * 4, [argsArray[j + 2][0],
                         argsArray[j + 1][0], argsArray[j + 1][3], positions]);

        context.currentOperation = j;
      } else {
        count = Math.min(count, MAX_IMAGES_IN_MASKS_BLOCK);
        var images = [];
        for (q = 0; q < count; q++) {
          transformArgs = argsArray[j + (q << 2) + 1];
          var maskParams = argsArray[j + (q << 2) + 2][0];
          images.push({ data: maskParams.data, width: maskParams.width,
                        height: maskParams.height,
                        transform: transformArgs });
        }

        // replacing queue items
        fnArray.splice(j, count * 4, OPS.paintImageMaskXObjectGroup);
        argsArray.splice(j, count * 4, [images]);

        context.currentOperation = j;
      }
    });

  addState(InitialState,
    [OPS.save, OPS.transform, OPS.paintImageXObject, OPS.restore],
    function (context) {
      var MIN_IMAGES_IN_BLOCK = 3;
      var MAX_IMAGES_IN_BLOCK = 1000;

      var fnArray = context.fnArray, argsArray = context.argsArray;
      var j = context.currentOperation - 3, i = j + 4;
      if (argsArray[j + 1][1] !== 0 || argsArray[j + 1][2] !== 0) {
        return;
      }
      var ii = fnArray.length;
      var transformArgs;
      for (; i + 3 < ii && fnArray[i - 4] === fnArray[i]; i += 4) {
        if (fnArray[i - 3] !== fnArray[i + 1] ||
            fnArray[i - 2] !== fnArray[i + 2] ||
            fnArray[i - 1] !== fnArray[i + 3]) {
          break;
        }
        if (argsArray[i - 2][0] !== argsArray[i + 2][0]) {
          break; // different image
        }
        var prevTransformArgs = argsArray[i - 3];
        transformArgs = argsArray[i + 1];
        if (prevTransformArgs[0] !== transformArgs[0] ||
            prevTransformArgs[1] !== transformArgs[1] ||
            prevTransformArgs[2] !== transformArgs[2] ||
            prevTransformArgs[3] !== transformArgs[3]) {
          break; // different transform
        }
      }
      var count = Math.min((i - j) >> 2, MAX_IMAGES_IN_BLOCK);
      if (count < MIN_IMAGES_IN_BLOCK) {
        context.currentOperation = i - 1;
        return;
      }

      var positions = new Float32Array(count * 2);
      i = j + 1;
      for (var q = 0; q < count; q++) {
        transformArgs = argsArray[i];
        positions[(q << 1)] = transformArgs[4];
        positions[(q << 1) + 1] = transformArgs[5];
        i += 4;
      }
      var args = [argsArray[j + 2][0], argsArray[j + 1][0],
                  argsArray[j + 1][3], positions];
      // replacing queue items
      fnArray.splice(j, count * 4, OPS.paintImageXObjectRepeat);
      argsArray.splice(j, count * 4, args);

      context.currentOperation = j;
    });

  addState(InitialState,
    [OPS.beginText, OPS.setFont, OPS.setTextMatrix, OPS.showText, OPS.endText],
    function (context) {
      // moving single chars with same font into beginText/endText groups
      // searching for (beginText, setFont, setTextMatrix, showText, endText)+
      var MIN_CHARS_IN_BLOCK = 3;
      var MAX_CHARS_IN_BLOCK = 1000;

      var fnArray = context.fnArray, argsArray = context.argsArray;
      var j = context.currentOperation - 4, i = j + 5;
      var ii = fnArray.length;

      for (; i < ii && fnArray[i - 5] === fnArray[i]; i++) {
        if (fnArray[i] === OPS.setFont) {
          if (argsArray[i - 5][0] !== argsArray[i][0] ||
              argsArray[i - 5][1] !== argsArray[i][1]) {
            break;
          }
        }
      }
      var count = Math.min(((i - j) / 5) | 0, MAX_CHARS_IN_BLOCK);
      if (count < MIN_CHARS_IN_BLOCK) {
        context.currentOperation = i - 1;
        return;
      }
      if (j >= 4 && fnArray[j - 4] === fnArray[j + 1] &&
          fnArray[j - 3] === fnArray[j + 2] &&
          fnArray[j - 2] === fnArray[j + 3] &&
          fnArray[j - 1] === fnArray[j + 4] &&
          argsArray[j - 4][0] === argsArray[j + 1][0] &&
          argsArray[j - 4][1] === argsArray[j + 1][1]) {
        // extending one block ahead (very first block might have 'dependency')
        count++;
        j -= 5;
      }
      var k = j + 7;
      i = j + 4;
      for (var q = 1; q < count; q++) {
        fnArray[i] = fnArray[k];
        argsArray[i] = argsArray[k];
        fnArray[i + 1] = fnArray[k + 1];
        argsArray[i + 1] = argsArray[k + 1];
        i += 2;
        k += 5;
      }
      var removed = (count - 1) * 3;
      fnArray.splice(i, removed);
      argsArray.splice(i, removed);

      context.currentOperation = i;
    });

  function QueueOptimizer() {}

  QueueOptimizer.prototype = {
    optimize: function QueueOptimizer_optimize(queue) {
      var fnArray = queue.fnArray, argsArray = queue.argsArray;
      var context = {
        currentOperation: 0,
        fnArray: fnArray,
        argsArray: argsArray
      };
      var i, ii = argsArray.length;
      var state;
      for (i = 0; i < ii; i++) {
        state = (state || InitialState)[fnArray[i]];
        if (typeof state === 'function') { // we found some handler
          context.currentOperation = i;
          state = state(context);
          i = context.currentOperation;
          ii = context.fnArray.length;
        }
      }
    }
  };
  return QueueOptimizer;
})();


var BUILT_IN_CMAPS = [
// << Start unicode maps.
'Adobe-GB1-UCS2',
'Adobe-CNS1-UCS2',
'Adobe-Japan1-UCS2',
'Adobe-Korea1-UCS2',
// >> End unicode maps.
'78-EUC-H',
'78-EUC-V',
'78-H',
'78-RKSJ-H',
'78-RKSJ-V',
'78-V',
'78ms-RKSJ-H',
'78ms-RKSJ-V',
'83pv-RKSJ-H',
'90ms-RKSJ-H',
'90ms-RKSJ-V',
'90msp-RKSJ-H',
'90msp-RKSJ-V',
'90pv-RKSJ-H',
'90pv-RKSJ-V',
'Add-H',
'Add-RKSJ-H',
'Add-RKSJ-V',
'Add-V',
'Adobe-CNS1-0',
'Adobe-CNS1-1',
'Adobe-CNS1-2',
'Adobe-CNS1-3',
'Adobe-CNS1-4',
'Adobe-CNS1-5',
'Adobe-CNS1-6',
'Adobe-GB1-0',
'Adobe-GB1-1',
'Adobe-GB1-2',
'Adobe-GB1-3',
'Adobe-GB1-4',
'Adobe-GB1-5',
'Adobe-Japan1-0',
'Adobe-Japan1-1',
'Adobe-Japan1-2',
'Adobe-Japan1-3',
'Adobe-Japan1-4',
'Adobe-Japan1-5',
'Adobe-Japan1-6',
'Adobe-Korea1-0',
'Adobe-Korea1-1',
'Adobe-Korea1-2',
'B5-H',
'B5-V',
'B5pc-H',
'B5pc-V',
'CNS-EUC-H',
'CNS-EUC-V',
'CNS1-H',
'CNS1-V',
'CNS2-H',
'CNS2-V',
'ETHK-B5-H',
'ETHK-B5-V',
'ETen-B5-H',
'ETen-B5-V',
'ETenms-B5-H',
'ETenms-B5-V',
'EUC-H',
'EUC-V',
'Ext-H',
'Ext-RKSJ-H',
'Ext-RKSJ-V',
'Ext-V',
'GB-EUC-H',
'GB-EUC-V',
'GB-H',
'GB-V',
'GBK-EUC-H',
'GBK-EUC-V',
'GBK2K-H',
'GBK2K-V',
'GBKp-EUC-H',
'GBKp-EUC-V',
'GBT-EUC-H',
'GBT-EUC-V',
'GBT-H',
'GBT-V',
'GBTpc-EUC-H',
'GBTpc-EUC-V',
'GBpc-EUC-H',
'GBpc-EUC-V',
'H',
'HKdla-B5-H',
'HKdla-B5-V',
'HKdlb-B5-H',
'HKdlb-B5-V',
'HKgccs-B5-H',
'HKgccs-B5-V',
'HKm314-B5-H',
'HKm314-B5-V',
'HKm471-B5-H',
'HKm471-B5-V',
'HKscs-B5-H',
'HKscs-B5-V',
'Hankaku',
'Hiragana',
'KSC-EUC-H',
'KSC-EUC-V',
'KSC-H',
'KSC-Johab-H',
'KSC-Johab-V',
'KSC-V',
'KSCms-UHC-H',
'KSCms-UHC-HW-H',
'KSCms-UHC-HW-V',
'KSCms-UHC-V',
'KSCpc-EUC-H',
'KSCpc-EUC-V',
'Katakana',
'NWP-H',
'NWP-V',
'RKSJ-H',
'RKSJ-V',
'Roman',
'UniCNS-UCS2-H',
'UniCNS-UCS2-V',
'UniCNS-UTF16-H',
'UniCNS-UTF16-V',
'UniCNS-UTF32-H',
'UniCNS-UTF32-V',
'UniCNS-UTF8-H',
'UniCNS-UTF8-V',
'UniGB-UCS2-H',
'UniGB-UCS2-V',
'UniGB-UTF16-H',
'UniGB-UTF16-V',
'UniGB-UTF32-H',
'UniGB-UTF32-V',
'UniGB-UTF8-H',
'UniGB-UTF8-V',
'UniJIS-UCS2-H',
'UniJIS-UCS2-HW-H',
'UniJIS-UCS2-HW-V',
'UniJIS-UCS2-V',
'UniJIS-UTF16-H',
'UniJIS-UTF16-V',
'UniJIS-UTF32-H',
'UniJIS-UTF32-V',
'UniJIS-UTF8-H',
'UniJIS-UTF8-V',
'UniJIS2004-UTF16-H',
'UniJIS2004-UTF16-V',
'UniJIS2004-UTF32-H',
'UniJIS2004-UTF32-V',
'UniJIS2004-UTF8-H',
'UniJIS2004-UTF8-V',
'UniJISPro-UCS2-HW-V',
'UniJISPro-UCS2-V',
'UniJISPro-UTF8-V',
'UniJISX0213-UTF32-H',
'UniJISX0213-UTF32-V',
'UniJISX02132004-UTF32-H',
'UniJISX02132004-UTF32-V',
'UniKS-UCS2-H',
'UniKS-UCS2-V',
'UniKS-UTF16-H',
'UniKS-UTF16-V',
'UniKS-UTF32-H',
'UniKS-UTF32-V',
'UniKS-UTF8-H',
'UniKS-UTF8-V',
'V',
'WP-Symbol'];

// CMap, not to be confused with TrueType's cmap.
var CMap = (function CMapClosure() {
  function CMap(builtInCMap) {
    // Codespace ranges are stored as follows:
    // [[1BytePairs], [2BytePairs], [3BytePairs], [4BytePairs]]
    // where nBytePairs are ranges e.g. [low1, high1, low2, high2, ...]
    this.codespaceRanges = [[], [], [], []];
    this.numCodespaceRanges = 0;
    this.map = [];
    this.vertical = false;
    this.useCMap = null;
    this.builtInCMap = builtInCMap;
  }
  CMap.prototype = {
    addCodespaceRange: function(n, low, high) {
      this.codespaceRanges[n - 1].push(low, high);
      this.numCodespaceRanges++;
    },

    mapRange: function(low, high, dstLow) {
      var lastByte = dstLow.length - 1;
      while (low <= high) {
        this.map[low] = dstLow;
        // Only the last byte has to be incremented.
        dstLow = dstLow.substr(0, lastByte) +
                 String.fromCharCode(dstLow.charCodeAt(lastByte) + 1);
        ++low;
      }
    },

    mapRangeToArray: function(low, high, array) {
      var i = 0;
      while (low <= high) {
        this.map[low] = array[i++];
        ++low;
      }
    },

    mapOne: function(src, dst) {
      this.map[src] = dst;
    },

    lookup: function(code) {
      return this.map[code];
    },

    readCharCode: function(str, offset) {
      var c = 0;
      var codespaceRanges = this.codespaceRanges;
      var codespaceRangesLen = this.codespaceRanges.length;
      // 9.7.6.2 CMap Mapping
      // The code length is at most 4.
      for (var n = 0; n < codespaceRangesLen; n++) {
        c = ((c << 8) | str.charCodeAt(offset + n)) >>> 0;
        // Check each codespace range to see if it falls within.
        var codespaceRange = codespaceRanges[n];
        for (var k = 0, kk = codespaceRange.length; k < kk;) {
          var low = codespaceRange[k++];
          var high = codespaceRange[k++];
          if (c >= low && c <= high) {
            return [c, n + 1];
          }
        }
      }

      return [0, 1];
    }

  };
  return CMap;
})();

var IdentityCMap = (function IdentityCMapClosure() {
  function IdentityCMap(vertical, n) {
    CMap.call(this);
    this.vertical = vertical;
    this.addCodespaceRange(n, 0, 0xffff);
    this.mapRange(0, 0xffff, '\u0000');
  }
  Util.inherit(IdentityCMap, CMap, {});

  return IdentityCMap;
})();

var BinaryCMapReader = (function BinaryCMapReaderClosure() {
  function fetchBinaryData(url) {
    var nonBinaryRequest = PDFJS.disableWorker;
    var request = new XMLHttpRequest();
    request.open('GET', url, false);
    if (!nonBinaryRequest) {
      try {
        request.responseType = 'arraybuffer';
        nonBinaryRequest = request.responseType !== 'arraybuffer';
      } catch (e) {
        nonBinaryRequest = true;
      }
    }
    if (nonBinaryRequest && request.overrideMimeType) {
      request.overrideMimeType('text/plain; charset=x-user-defined');
    }
    request.send(null);
    if (request.status === 0 && /^https?:/i.test(url)) {
      error('Unable to get binary cMap at: ' + url);
    }
    if (nonBinaryRequest) {
      var data = Array.prototype.map.call(request.responseText, function (ch) {
        return ch.charCodeAt(0) & 255;
      });
      return new Uint8Array(data);
    }
    return new Uint8Array(request.response);
  }

  function hexToInt(a, size) {
    var n = 0;
    for (var i = 0; i <= size; i++) {
      n = (n << 8) | a[i];
    }
    return n >>> 0;
  }

  function hexToStr(a, size) {
    return String.fromCharCode.apply(null, a.subarray(0, size + 1));
  }

  function addHex(a, b, size) {
    var c = 0;
    for (var i = size; i >= 0; i--) {
      c += a[i] + b[i];
      a[i] = c & 255;
      c >>= 8;
    }
  }

  function incHex(a, size) {
    var c = 1;
    for (var i = size; i >= 0 && c > 0; i--) {
      c += a[i];
      a[i] = c & 255;
      c >>= 8;
    }
  }

  var MAX_NUM_SIZE = 16;
  var MAX_ENCODED_NUM_SIZE = 19; // ceil(MAX_NUM_SIZE * 7 / 8)

  function BinaryCMapStream(data) {
    this.buffer = data;
    this.pos = 0;
    this.end = data.length;
    this.tmpBuf = new Uint8Array(MAX_ENCODED_NUM_SIZE);
  }

  BinaryCMapStream.prototype = {
    readByte: function () {
      if (this.pos >= this.end) {
        return -1;
      }
      return this.buffer[this.pos++];
    },
    readNumber: function () {
      var n = 0;
      var last;
      do {
        var b = this.readByte();
        if (b < 0) {
          error('unexpected EOF in bcmap');
        }
        last = !(b & 0x80);
        n = (n << 7) | (b & 0x7F);
      } while (!last);
      return n;
    },
    readSigned: function () {
      var n = this.readNumber();
      return (n & 1) ? ~(n >>> 1) : n >>> 1;
    },
    readHex: function (num, size) {
      num.set(this.buffer.subarray(this.pos,
        this.pos + size + 1));
      this.pos += size + 1;
    },
    readHexNumber: function (num, size) {
      var last;
      var stack = this.tmpBuf, sp = 0;
      do {
        var b = this.readByte();
        if (b < 0) {
          error('unexpected EOF in bcmap');
        }
        last = !(b & 0x80);
        stack[sp++] = b & 0x7F;
      } while (!last);
      var i = size, buffer = 0, bufferSize = 0;
      while (i >= 0) {
        while (bufferSize < 8 && stack.length > 0) {
          buffer = (stack[--sp] << bufferSize) | buffer;
          bufferSize += 7;
        }
        num[i] = buffer & 255;
        i--;
        buffer >>= 8;
        bufferSize -= 8;
      }
    },
    readHexSigned: function (num, size) {
      this.readHexNumber(num, size);
      var sign = num[size] & 1 ? 255 : 0;
      var c = 0;
      for (var i = 0; i <= size; i++) {
        c = ((c & 1) << 8) | num[i];
        num[i] = (c >> 1) ^ sign;
      }
    },
    readString: function () {
      var len = this.readNumber();
      var s = '';
      for (var i = 0; i < len; i++) {
        s += String.fromCharCode(this.readNumber());
      }
      return s;
    }
  };

  function processBinaryCMap(url, cMap, extend) {
    var data = fetchBinaryData(url);
    var stream = new BinaryCMapStream(data);

    var header = stream.readByte();
    cMap.vertical = !!(header & 1);

    var useCMap = null;
    var start = new Uint8Array(MAX_NUM_SIZE);
    var end = new Uint8Array(MAX_NUM_SIZE);
    var char = new Uint8Array(MAX_NUM_SIZE);
    var charCode = new Uint8Array(MAX_NUM_SIZE);
    var tmp = new Uint8Array(MAX_NUM_SIZE);
    var code;

    var b;
    while ((b = stream.readByte()) >= 0) {
      var type = b >> 5;
      if (type === 7) { // metadata, e.g. comment or usecmap
        switch (b & 0x1F) {
          case 0:
            stream.readString(); // skipping comment
            break;
          case 1:
            useCMap = stream.readString();
            break;
        }
        continue;
      }
      var sequence = !!(b & 0x10);
      var dataSize = b & 15;

      assert(dataSize + 1 <= MAX_NUM_SIZE);

      var ucs2DataSize = 1;
      var subitemsCount = stream.readNumber();
      var i;
      switch (type) {
        case 0: // codespacerange
          stream.readHex(start, dataSize);
          stream.readHexNumber(end, dataSize);
          addHex(end, start, dataSize);
          cMap.addCodespaceRange(dataSize + 1, hexToInt(start, dataSize),
                                 hexToInt(end, dataSize));
          for (i = 1; i < subitemsCount; i++) {
            incHex(end, dataSize);
            stream.readHexNumber(start, dataSize);
            addHex(start, end, dataSize);
            stream.readHexNumber(end, dataSize);
            addHex(end, start, dataSize);
            cMap.addCodespaceRange(dataSize + 1, hexToInt(start, dataSize),
                                   hexToInt(end, dataSize));
          }
          break;
        case 1: // notdefrange
          stream.readHex(start, dataSize);
          stream.readHexNumber(end, dataSize);
          addHex(end, start, dataSize);
          code = stream.readNumber();
          // undefined range, skipping
          for (i = 1; i < subitemsCount; i++) {
            incHex(end, dataSize);
            stream.readHexNumber(start, dataSize);
            addHex(start, end, dataSize);
            stream.readHexNumber(end, dataSize);
            addHex(end, start, dataSize);
            code = stream.readNumber();
            // nop
          }
          break;
        case 2: // cidchar
          stream.readHex(char, dataSize);
          code = stream.readNumber();
          cMap.mapOne(hexToInt(char, dataSize), String.fromCharCode(code));
          for (i = 1; i < subitemsCount; i++) {
            incHex(char, dataSize);
            if (!sequence) {
              stream.readHexNumber(tmp, dataSize);
              addHex(char, tmp, dataSize);
            }
            code = stream.readSigned() + (code + 1);
            cMap.mapOne(hexToInt(char, dataSize), String.fromCharCode(code));
          }
          break;
        case 3: // cidrange
          stream.readHex(start, dataSize);
          stream.readHexNumber(end, dataSize);
          addHex(end, start, dataSize);
          code = stream.readNumber();
          cMap.mapRange(hexToInt(start, dataSize), hexToInt(end, dataSize),
                        String.fromCharCode(code));
          for (i = 1; i < subitemsCount; i++) {
            incHex(end, dataSize);
            if (!sequence) {
              stream.readHexNumber(start, dataSize);
              addHex(start, end, dataSize);
            } else {
              start.set(end);
            }
            stream.readHexNumber(end, dataSize);
            addHex(end, start, dataSize);
            code = stream.readNumber();
            cMap.mapRange(hexToInt(start, dataSize), hexToInt(end, dataSize),
                          String.fromCharCode(code));
          }
          break;
        case 4: // bfchar
          stream.readHex(char, ucs2DataSize);
          stream.readHex(charCode, dataSize);
          cMap.mapOne(hexToInt(char, ucs2DataSize),
                      hexToStr(charCode, dataSize));
          for (i = 1; i < subitemsCount; i++) {
            incHex(char, ucs2DataSize);
            if (!sequence) {
              stream.readHexNumber(tmp, ucs2DataSize);
              addHex(char, tmp, ucs2DataSize);
            }
            incHex(charCode, dataSize);
            stream.readHexSigned(tmp, dataSize);
            addHex(charCode, tmp, dataSize);
            cMap.mapOne(hexToInt(char, ucs2DataSize),
                        hexToStr(charCode, dataSize));
          }
          break;
        case 5: // bfrange
          stream.readHex(start, ucs2DataSize);
          stream.readHexNumber(end, ucs2DataSize);
          addHex(end, start, ucs2DataSize);
          stream.readHex(charCode, dataSize);
          cMap.mapRange(hexToInt(start, ucs2DataSize),
                        hexToInt(end, ucs2DataSize),
                        hexToStr(charCode, dataSize));
          for (i = 1; i < subitemsCount; i++) {
            incHex(end, ucs2DataSize);
            if (!sequence) {
              stream.readHexNumber(start, ucs2DataSize);
              addHex(start, end, ucs2DataSize);
            } else {
              start.set(end);
            }
            stream.readHexNumber(end, ucs2DataSize);
            addHex(end, start, ucs2DataSize);
            stream.readHex(charCode, dataSize);
            cMap.mapRange(hexToInt(start, ucs2DataSize),
                          hexToInt(end, ucs2DataSize),
                          hexToStr(charCode, dataSize));
          }
          break;
        default:
          error('Unknown type: ' + type);
          break;
      }
    }

    if (useCMap) {
      extend(useCMap);
    }
    return cMap;
  }

  function BinaryCMapReader() {}

  BinaryCMapReader.prototype = {
    read: processBinaryCMap
  };

  return BinaryCMapReader;
})();

var CMapFactory = (function CMapFactoryClosure() {
  function strToInt(str) {
    var a = 0;
    for (var i = 0; i < str.length; i++) {
      a = (a << 8) | str.charCodeAt(i);
    }
    return a >>> 0;
  }

  function expectString(obj) {
    if (!isString(obj)) {
      error('Malformed CMap: expected string.');
    }
  }

  function expectInt(obj) {
    if (!isInt(obj)) {
      error('Malformed CMap: expected int.');
    }
  }

  function parseBfChar(cMap, lexer) {
    while (true) {
      var obj = lexer.getObj();
      if (isEOF(obj)) {
        break;
      }
      if (isCmd(obj, 'endbfchar')) {
        return;
      }
      expectString(obj);
      var src = strToInt(obj);
      obj = lexer.getObj();
      // TODO are /dstName used?
      expectString(obj);
      var dst = obj;
      cMap.mapOne(src, dst);
    }
  }

  function parseBfRange(cMap, lexer) {
    while (true) {
      var obj = lexer.getObj();
      if (isEOF(obj)) {
        break;
      }
      if (isCmd(obj, 'endbfrange')) {
        return;
      }
      expectString(obj);
      var low = strToInt(obj);
      obj = lexer.getObj();
      expectString(obj);
      var high = strToInt(obj);
      obj = lexer.getObj();
      if (isInt(obj) || isString(obj)) {
        var dstLow = isInt(obj) ? String.fromCharCode(obj) : obj;
        cMap.mapRange(low, high, dstLow);
      } else if (isCmd(obj, '[')) {
        obj = lexer.getObj();
        var array = [];
        while (!isCmd(obj, ']') && !isEOF(obj)) {
          array.push(obj);
          obj = lexer.getObj();
        }
        cMap.mapRangeToArray(low, high, array);
      } else {
        break;
      }
    }
    error('Invalid bf range.');
  }

  function parseCidChar(cMap, lexer) {
    while (true) {
      var obj = lexer.getObj();
      if (isEOF(obj)) {
        break;
      }
      if (isCmd(obj, 'endcidchar')) {
        return;
      }
      expectString(obj);
      var src = strToInt(obj);
      obj = lexer.getObj();
      expectInt(obj);
      var dst = String.fromCharCode(obj);
      cMap.mapOne(src, dst);
    }
  }

  function parseCidRange(cMap, lexer) {
    while (true) {
      var obj = lexer.getObj();
      if (isEOF(obj)) {
        break;
      }
      if (isCmd(obj, 'endcidrange')) {
        return;
      }
      expectString(obj);
      var low = strToInt(obj);
      obj = lexer.getObj();
      expectString(obj);
      var high = strToInt(obj);
      obj = lexer.getObj();
      expectInt(obj);
      var dstLow = String.fromCharCode(obj);
      cMap.mapRange(low, high, dstLow);
    }
  }

  function parseCodespaceRange(cMap, lexer) {
    while (true) {
      var obj = lexer.getObj();
      if (isEOF(obj)) {
        break;
      }
      if (isCmd(obj, 'endcodespacerange')) {
        return;
      }
      if (!isString(obj)) {
        break;
      }
      var low = strToInt(obj);
      obj = lexer.getObj();
      if (!isString(obj)) {
        break;
      }
      var high = strToInt(obj);
      cMap.addCodespaceRange(obj.length, low, high);
    }
    error('Invalid codespace range.');
  }

  function parseWMode(cMap, lexer) {
    var obj = lexer.getObj();
    if (isInt(obj)) {
      cMap.vertical = !!obj;
    }
  }

  function parseCMap(cMap, lexer, builtInCMapParams, useCMap) {
    var previous;
    var embededUseCMap;
    objLoop: while (true) {
      var obj = lexer.getObj();
      if (isEOF(obj)) {
        break;
      } else if (isName(obj)) {
        if (obj.name === 'WMode') {
          parseWMode(cMap, lexer);
        }
        previous = obj;
      } else if (isCmd(obj)) {
        switch (obj.cmd) {
          case 'endcmap':
            break objLoop;
          case 'usecmap':
            if (isName(previous)) {
              embededUseCMap = previous.name;
            }
            break;
          case 'begincodespacerange':
            parseCodespaceRange(cMap, lexer);
            break;
          case 'beginbfchar':
            parseBfChar(cMap, lexer);
            break;
          case 'begincidchar':
            parseCidChar(cMap, lexer);
            break;
          case 'beginbfrange':
            parseBfRange(cMap, lexer);
            break;
          case 'begincidrange':
            parseCidRange(cMap, lexer);
            break;
        }
      }
    }

    if (!useCMap && embededUseCMap) {
      // Load the usecmap definition from the file only if there wasn't one
      // specified.
      useCMap = embededUseCMap;
    }
    if (useCMap) {
      extendCMap(cMap, builtInCMapParams, useCMap);
    }
  }

  function extendCMap(cMap, builtInCMapParams, useCMap) {
    cMap.useCMap = createBuiltInCMap(useCMap, builtInCMapParams);
    // If there aren't any code space ranges defined clone all the parent ones
    // into this cMap.
    if (cMap.numCodespaceRanges === 0) {
      var useCodespaceRanges = cMap.useCMap.codespaceRanges;
      for (var i = 0; i < useCodespaceRanges.length; i++) {
        cMap.codespaceRanges[i] = useCodespaceRanges[i].slice();
      }
      cMap.numCodespaceRanges = cMap.useCMap.numCodespaceRanges;
    }
    // Merge the map into the current one, making sure not to override
    // any previously defined entries.
    for (var key in cMap.useCMap.map) {
      if (key in cMap.map) {
        continue;
      }
      cMap.map[key] = cMap.useCMap.map[key];
    }
  }

  function parseBinaryCMap(name, builtInCMapParams) {
    var url = builtInCMapParams.url + name + '.bcmap';
    var cMap = new CMap(true);
    new BinaryCMapReader().read(url, cMap, function (useCMap) {
      extendCMap(cMap, builtInCMapParams, useCMap);
    });
    return cMap;
  }

  function createBuiltInCMap(name, builtInCMapParams) {
    if (name === 'Identity-H') {
      return new IdentityCMap(false, 2);
    } else if (name === 'Identity-V') {
      return new IdentityCMap(true, 2);
    }
    if (BUILT_IN_CMAPS.indexOf(name) === -1) {
      error('Unknown cMap name: ' + name);
    }
    assert (builtInCMapParams, 'buildin cmap parameters are not provided');

    if (builtInCMapParams.packed) {
      return parseBinaryCMap(name, builtInCMapParams);
    }

    var request = new XMLHttpRequest();
    var url = builtInCMapParams.url + name;
    request.open('GET', url, false);
    request.send(null);
    if (request.status === 0 && /^https?:/i.test(url)) {
      error('Unable to get cMap at: ' + url);
    }
    var cMap = new CMap(true);
    var lexer = new Lexer(new StringStream(request.responseText));
    parseCMap(cMap, lexer, builtInCMapParams, null);
    return cMap;
  }

  return {
    create: function (encoding, builtInCMapParams, useCMap) {
      if (isName(encoding)) {
        return createBuiltInCMap(encoding.name, builtInCMapParams);
      } else if (isStream(encoding)) {
        var cMap = new CMap();
        var lexer = new Lexer(encoding);
        try {
          parseCMap(cMap, lexer, builtInCMapParams, useCMap);
        } catch (e) {
          warn('Invalid CMap data. ' + e);
        }
        return cMap;
      }
      error('Encoding required.');
    }
  };
})();


// Unicode Private Use Area
var PRIVATE_USE_OFFSET_START = 0xE000;
var PRIVATE_USE_OFFSET_END = 0xF8FF;
var SKIP_PRIVATE_USE_RANGE_F000_TO_F01F = false;

// PDF Glyph Space Units are one Thousandth of a TextSpace Unit
// except for Type 3 fonts
var PDF_GLYPH_SPACE_UNITS = 1000;

// Hinting is currently disabled due to unknown problems on windows
// in tracemonkey and various other pdfs with type1 fonts.
var HINTING_ENABLED = false;

// Accented charactars are not displayed properly on windows, using this flag
// to control analysis of seac charstrings.
var SEAC_ANALYSIS_ENABLED = false;

var FontFlags = {
  FixedPitch: 1,
  Serif: 2,
  Symbolic: 4,
  Script: 8,
  Nonsymbolic: 32,
  Italic: 64,
  AllCap: 65536,
  SmallCap: 131072,
  ForceBold: 262144
};

var Encodings = {
  ExpertEncoding: ['', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    'space', 'exclamsmall', 'Hungarumlautsmall', '', 'dollaroldstyle',
    'dollarsuperior', 'ampersandsmall', 'Acutesmall', 'parenleftsuperior',
    'parenrightsuperior', 'twodotenleader', 'onedotenleader', 'comma',
    'hyphen', 'period', 'fraction', 'zerooldstyle', 'oneoldstyle',
    'twooldstyle', 'threeoldstyle', 'fouroldstyle', 'fiveoldstyle',
    'sixoldstyle', 'sevenoldstyle', 'eightoldstyle', 'nineoldstyle', 'colon',
    'semicolon', 'commasuperior', 'threequartersemdash', 'periodsuperior',
    'questionsmall', '', 'asuperior', 'bsuperior', 'centsuperior', 'dsuperior',
    'esuperior', '', '', 'isuperior', '', '', 'lsuperior', 'msuperior',
    'nsuperior', 'osuperior', '', '', 'rsuperior', 'ssuperior', 'tsuperior',
    '', 'ff', 'fi', 'fl', 'ffi', 'ffl', 'parenleftinferior', '',
    'parenrightinferior', 'Circumflexsmall', 'hyphensuperior', 'Gravesmall',
    'Asmall', 'Bsmall', 'Csmall', 'Dsmall', 'Esmall', 'Fsmall', 'Gsmall',
    'Hsmall', 'Ismall', 'Jsmall', 'Ksmall', 'Lsmall', 'Msmall', 'Nsmall',
    'Osmall', 'Psmall', 'Qsmall', 'Rsmall', 'Ssmall', 'Tsmall', 'Usmall',
    'Vsmall', 'Wsmall', 'Xsmall', 'Ysmall', 'Zsmall', 'colonmonetary',
    'onefitted', 'rupiah', 'Tildesmall', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', 'exclamdownsmall', 'centoldstyle', 'Lslashsmall',
    '', '', 'Scaronsmall', 'Zcaronsmall', 'Dieresissmall', 'Brevesmall',
    'Caronsmall', '', 'Dotaccentsmall', '', '', 'Macronsmall', '', '',
    'figuredash', 'hypheninferior', '', '', 'Ogoneksmall', 'Ringsmall',
    'Cedillasmall', '', '', '', 'onequarter', 'onehalf', 'threequarters',
    'questiondownsmall', 'oneeighth', 'threeeighths', 'fiveeighths',
    'seveneighths', 'onethird', 'twothirds', '', '', 'zerosuperior',
    'onesuperior', 'twosuperior', 'threesuperior', 'foursuperior',
    'fivesuperior', 'sixsuperior', 'sevensuperior', 'eightsuperior',
    'ninesuperior', 'zeroinferior', 'oneinferior', 'twoinferior',
    'threeinferior', 'fourinferior', 'fiveinferior', 'sixinferior',
    'seveninferior', 'eightinferior', 'nineinferior', 'centinferior',
    'dollarinferior', 'periodinferior', 'commainferior', 'Agravesmall',
    'Aacutesmall', 'Acircumflexsmall', 'Atildesmall', 'Adieresissmall',
    'Aringsmall', 'AEsmall', 'Ccedillasmall', 'Egravesmall', 'Eacutesmall',
    'Ecircumflexsmall', 'Edieresissmall', 'Igravesmall', 'Iacutesmall',
    'Icircumflexsmall', 'Idieresissmall', 'Ethsmall', 'Ntildesmall',
    'Ogravesmall', 'Oacutesmall', 'Ocircumflexsmall', 'Otildesmall',
    'Odieresissmall', 'OEsmall', 'Oslashsmall', 'Ugravesmall', 'Uacutesmall',
    'Ucircumflexsmall', 'Udieresissmall', 'Yacutesmall', 'Thornsmall',
    'Ydieresissmall'],
  MacExpertEncoding: ['', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    'space', 'exclamsmall', 'Hungarumlautsmall', 'centoldstyle',
    'dollaroldstyle', 'dollarsuperior', 'ampersandsmall', 'Acutesmall',
    'parenleftsuperior', 'parenrightsuperior', 'twodotenleader',
    'onedotenleader', 'comma', 'hyphen', 'period', 'fraction', 'zerooldstyle',
    'oneoldstyle', 'twooldstyle', 'threeoldstyle', 'fouroldstyle',
    'fiveoldstyle', 'sixoldstyle', 'sevenoldstyle', 'eightoldstyle',
    'nineoldstyle', 'colon', 'semicolon', '', 'threequartersemdash', '',
    'questionsmall', '', '', '', '', 'Ethsmall', '', '', 'onequarter',
    'onehalf', 'threequarters', 'oneeighth', 'threeeighths', 'fiveeighths',
    'seveneighths', 'onethird', 'twothirds', '', '', '', '', '', '', 'ff',
    'fi', 'fl', 'ffi', 'ffl', 'parenleftinferior', '', 'parenrightinferior',
    'Circumflexsmall', 'hypheninferior', 'Gravesmall', 'Asmall', 'Bsmall',
    'Csmall', 'Dsmall', 'Esmall', 'Fsmall', 'Gsmall', 'Hsmall', 'Ismall',
    'Jsmall', 'Ksmall', 'Lsmall', 'Msmall', 'Nsmall', 'Osmall', 'Psmall',
    'Qsmall', 'Rsmall', 'Ssmall', 'Tsmall', 'Usmall', 'Vsmall', 'Wsmall',
    'Xsmall', 'Ysmall', 'Zsmall', 'colonmonetary', 'onefitted', 'rupiah',
    'Tildesmall', '', '', 'asuperior', 'centsuperior', '', '', '', '',
    'Aacutesmall', 'Agravesmall', 'Acircumflexsmall', 'Adieresissmall',
    'Atildesmall', 'Aringsmall', 'Ccedillasmall', 'Eacutesmall', 'Egravesmall',
    'Ecircumflexsmall', 'Edieresissmall', 'Iacutesmall', 'Igravesmall',
    'Icircumflexsmall', 'Idieresissmall', 'Ntildesmall', 'Oacutesmall',
    'Ogravesmall', 'Ocircumflexsmall', 'Odieresissmall', 'Otildesmall',
    'Uacutesmall', 'Ugravesmall', 'Ucircumflexsmall', 'Udieresissmall', '',
    'eightsuperior', 'fourinferior', 'threeinferior', 'sixinferior',
    'eightinferior', 'seveninferior', 'Scaronsmall', '', 'centinferior',
    'twoinferior', '', 'Dieresissmall', '', 'Caronsmall', 'osuperior',
    'fiveinferior', '', 'commainferior', 'periodinferior', 'Yacutesmall', '',
    'dollarinferior', '', 'Thornsmall', '', 'nineinferior', 'zeroinferior',
    'Zcaronsmall', 'AEsmall', 'Oslashsmall', 'questiondownsmall',
    'oneinferior', 'Lslashsmall', '', '', '', '', '', '', 'Cedillasmall', '',
    '', '', '', '', 'OEsmall', 'figuredash', 'hyphensuperior', '', '', '', '',
    'exclamdownsmall', '', 'Ydieresissmall', '', 'onesuperior', 'twosuperior',
    'threesuperior', 'foursuperior', 'fivesuperior', 'sixsuperior',
    'sevensuperior', 'ninesuperior', 'zerosuperior', '', 'esuperior',
    'rsuperior', 'tsuperior', '', '', 'isuperior', 'ssuperior', 'dsuperior',
    '', '', '', '', '', 'lsuperior', 'Ogoneksmall', 'Brevesmall',
    'Macronsmall', 'bsuperior', 'nsuperior', 'msuperior', 'commasuperior',
    'periodsuperior', 'Dotaccentsmall', 'Ringsmall'],
  MacRomanEncoding: ['', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    'space', 'exclam', 'quotedbl', 'numbersign', 'dollar', 'percent',
    'ampersand', 'quotesingle', 'parenleft', 'parenright', 'asterisk', 'plus',
    'comma', 'hyphen', 'period', 'slash', 'zero', 'one', 'two', 'three',
    'four', 'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon',
    'less', 'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F',
    'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U',
    'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright',
    'asciicircum', 'underscore', 'grave', 'a', 'b', 'c', 'd', 'e', 'f', 'g',
    'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
    'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright', 'asciitilde', '',
    'Adieresis', 'Aring', 'Ccedilla', 'Eacute', 'Ntilde', 'Odieresis',
    'Udieresis', 'aacute', 'agrave', 'acircumflex', 'adieresis', 'atilde',
    'aring', 'ccedilla', 'eacute', 'egrave', 'ecircumflex', 'edieresis',
    'iacute', 'igrave', 'icircumflex', 'idieresis', 'ntilde', 'oacute',
    'ograve', 'ocircumflex', 'odieresis', 'otilde', 'uacute', 'ugrave',
    'ucircumflex', 'udieresis', 'dagger', 'degree', 'cent', 'sterling',
    'section', 'bullet', 'paragraph', 'germandbls', 'registered', 'copyright',
    'trademark', 'acute', 'dieresis', 'notequal', 'AE', 'Oslash', 'infinity',
    'plusminus', 'lessequal', 'greaterequal', 'yen', 'mu', 'partialdiff',
    'summation', 'product', 'pi', 'integral', 'ordfeminine', 'ordmasculine',
    'Omega', 'ae', 'oslash', 'questiondown', 'exclamdown', 'logicalnot',
    'radical', 'florin', 'approxequal', 'Delta', 'guillemotleft',
    'guillemotright', 'ellipsis', 'space', 'Agrave', 'Atilde', 'Otilde', 'OE',
    'oe', 'endash', 'emdash', 'quotedblleft', 'quotedblright', 'quoteleft',
    'quoteright', 'divide', 'lozenge', 'ydieresis', 'Ydieresis', 'fraction',
    'currency', 'guilsinglleft', 'guilsinglright', 'fi', 'fl', 'daggerdbl',
    'periodcentered', 'quotesinglbase', 'quotedblbase', 'perthousand',
    'Acircumflex', 'Ecircumflex', 'Aacute', 'Edieresis', 'Egrave', 'Iacute',
    'Icircumflex', 'Idieresis', 'Igrave', 'Oacute', 'Ocircumflex', 'apple',
    'Ograve', 'Uacute', 'Ucircumflex', 'Ugrave', 'dotlessi', 'circumflex',
    'tilde', 'macron', 'breve', 'dotaccent', 'ring', 'cedilla', 'hungarumlaut',
    'ogonek', 'caron'],
  StandardEncoding: ['', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    'space', 'exclam', 'quotedbl', 'numbersign', 'dollar', 'percent',
    'ampersand', 'quoteright', 'parenleft', 'parenright', 'asterisk', 'plus',
    'comma', 'hyphen', 'period', 'slash', 'zero', 'one', 'two', 'three',
    'four', 'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon',
    'less', 'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F',
    'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U',
    'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright',
    'asciicircum', 'underscore', 'quoteleft', 'a', 'b', 'c', 'd', 'e', 'f',
    'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u',
    'v', 'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright', 'asciitilde',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', 'exclamdown',
    'cent', 'sterling', 'fraction', 'yen', 'florin', 'section', 'currency',
    'quotesingle', 'quotedblleft', 'guillemotleft', 'guilsinglleft',
    'guilsinglright', 'fi', 'fl', '', 'endash', 'dagger', 'daggerdbl',
    'periodcentered', '', 'paragraph', 'bullet', 'quotesinglbase',
    'quotedblbase', 'quotedblright', 'guillemotright', 'ellipsis',
    'perthousand', '', 'questiondown', '', 'grave', 'acute', 'circumflex',
    'tilde', 'macron', 'breve', 'dotaccent', 'dieresis', '', 'ring', 'cedilla',
    '', 'hungarumlaut', 'ogonek', 'caron', 'emdash', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', 'AE', '', 'ordfeminine', '', '',
    '', '', 'Lslash', 'Oslash', 'OE', 'ordmasculine', '', '', '', '', '', 'ae',
    '', '', '', 'dotlessi', '', '', 'lslash', 'oslash', 'oe', 'germandbls'],
  WinAnsiEncoding: ['', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    'space', 'exclam', 'quotedbl', 'numbersign', 'dollar', 'percent',
    'ampersand', 'quotesingle', 'parenleft', 'parenright', 'asterisk', 'plus',
    'comma', 'hyphen', 'period', 'slash', 'zero', 'one', 'two', 'three',
    'four', 'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon',
    'less', 'equal', 'greater', 'question', 'at', 'A', 'B', 'C', 'D', 'E', 'F',
    'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U',
    'V', 'W', 'X', 'Y', 'Z', 'bracketleft', 'backslash', 'bracketright',
    'asciicircum', 'underscore', 'grave', 'a', 'b', 'c', 'd', 'e', 'f', 'g',
    'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
    'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright', 'asciitilde',
    'bullet', 'Euro', 'bullet', 'quotesinglbase', 'florin', 'quotedblbase',
    'ellipsis', 'dagger', 'daggerdbl', 'circumflex', 'perthousand', 'Scaron',
    'guilsinglleft', 'OE', 'bullet', 'Zcaron', 'bullet', 'bullet', 'quoteleft',
    'quoteright', 'quotedblleft', 'quotedblright', 'bullet', 'endash',
    'emdash', 'tilde', 'trademark', 'scaron', 'guilsinglright', 'oe', 'bullet',
    'zcaron', 'Ydieresis', 'space', 'exclamdown', 'cent', 'sterling',
    'currency', 'yen', 'brokenbar', 'section', 'dieresis', 'copyright',
    'ordfeminine', 'guillemotleft', 'logicalnot', 'hyphen', 'registered',
    'macron', 'degree', 'plusminus', 'twosuperior', 'threesuperior', 'acute',
    'mu', 'paragraph', 'periodcentered', 'cedilla', 'onesuperior',
    'ordmasculine', 'guillemotright', 'onequarter', 'onehalf', 'threequarters',
    'questiondown', 'Agrave', 'Aacute', 'Acircumflex', 'Atilde', 'Adieresis',
    'Aring', 'AE', 'Ccedilla', 'Egrave', 'Eacute', 'Ecircumflex', 'Edieresis',
    'Igrave', 'Iacute', 'Icircumflex', 'Idieresis', 'Eth', 'Ntilde', 'Ograve',
    'Oacute', 'Ocircumflex', 'Otilde', 'Odieresis', 'multiply', 'Oslash',
    'Ugrave', 'Uacute', 'Ucircumflex', 'Udieresis', 'Yacute', 'Thorn',
    'germandbls', 'agrave', 'aacute', 'acircumflex', 'atilde', 'adieresis',
    'aring', 'ae', 'ccedilla', 'egrave', 'eacute', 'ecircumflex', 'edieresis',
    'igrave', 'iacute', 'icircumflex', 'idieresis', 'eth', 'ntilde', 'ograve',
    'oacute', 'ocircumflex', 'otilde', 'odieresis', 'divide', 'oslash',
    'ugrave', 'uacute', 'ucircumflex', 'udieresis', 'yacute', 'thorn',
    'ydieresis'],
  SymbolSetEncoding: ['', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    'space', 'exclam', 'universal', 'numbersign', 'existential', 'percent',
    'ampersand', 'suchthat', 'parenleft', 'parenright', 'asteriskmath', 'plus',
    'comma', 'minus', 'period', 'slash', 'zero', 'one', 'two', 'three', 'four',
    'five', 'six', 'seven', 'eight', 'nine', 'colon', 'semicolon', 'less',
    'equal', 'greater', 'question', 'congruent', 'Alpha', 'Beta', 'Chi',
    'Delta', 'Epsilon', 'Phi', 'Gamma', 'Eta', 'Iota', 'theta1', 'Kappa',
    'Lambda', 'Mu', 'Nu', 'Omicron', 'Pi', 'Theta', 'Rho', 'Sigma', 'Tau',
    'Upsilon', 'sigma1', 'Omega', 'Xi', 'Psi', 'Zeta', 'bracketleft',
    'therefore', 'bracketright', 'perpendicular', 'underscore', 'radicalex',
    'alpha', 'beta', 'chi', 'delta', 'epsilon', 'phi', 'gamma', 'eta', 'iota',
    'phi1', 'kappa', 'lambda', 'mu', 'nu', 'omicron', 'pi', 'theta', 'rho',
    'sigma', 'tau', 'upsilon', 'omega1', 'omega', 'xi', 'psi', 'zeta',
    'braceleft', 'bar', 'braceright', 'similar', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', 'Euro', 'Upsilon1', 'minute', 'lessequal',
    'fraction', 'infinity', 'florin', 'club', 'diamond', 'heart', 'spade',
    'arrowboth', 'arrowleft', 'arrowup', 'arrowright', 'arrowdown', 'degree',
    'plusminus', 'second', 'greaterequal', 'multiply', 'proportional',
    'partialdiff', 'bullet', 'divide', 'notequal', 'equivalence',
    'approxequal', 'ellipsis', 'arrowvertex', 'arrowhorizex', 'carriagereturn',
    'aleph', 'Ifraktur', 'Rfraktur', 'weierstrass', 'circlemultiply',
    'circleplus', 'emptyset', 'intersection', 'union', 'propersuperset',
    'reflexsuperset', 'notsubset', 'propersubset', 'reflexsubset', 'element',
    'notelement', 'angle', 'gradient', 'registerserif', 'copyrightserif',
    'trademarkserif', 'product', 'radical', 'dotmath', 'logicalnot',
    'logicaland', 'logicalor', 'arrowdblboth', 'arrowdblleft', 'arrowdblup',
    'arrowdblright', 'arrowdbldown', 'lozenge', 'angleleft', 'registersans',
    'copyrightsans', 'trademarksans', 'summation', 'parenlefttp',
    'parenleftex', 'parenleftbt', 'bracketlefttp', 'bracketleftex',
    'bracketleftbt', 'bracelefttp', 'braceleftmid', 'braceleftbt', 'braceex',
    '', 'angleright', 'integral', 'integraltp', 'integralex', 'integralbt',
    'parenrighttp', 'parenrightex', 'parenrightbt', 'bracketrighttp',
    'bracketrightex', 'bracketrightbt', 'bracerighttp', 'bracerightmid',
    'bracerightbt'],
  zapfDingbatsEncoding: ['', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    'space', 'a1', 'a2', 'a202', 'a3', 'a4', 'a5', 'a119', 'a118', 'a117',
    'a11', 'a12', 'a13', 'a14', 'a15', 'a16', 'a105', 'a17', 'a18', 'a19',
    'a20', 'a21', 'a22', 'a23', 'a24', 'a25', 'a26', 'a27', 'a28', 'a6', 'a7',
    'a8', 'a9', 'a10', 'a29', 'a30', 'a31', 'a32', 'a33', 'a34', 'a35', 'a36',
    'a37', 'a38', 'a39', 'a40', 'a41', 'a42', 'a43', 'a44', 'a45', 'a46',
    'a47', 'a48', 'a49', 'a50', 'a51', 'a52', 'a53', 'a54', 'a55', 'a56',
    'a57', 'a58', 'a59', 'a60', 'a61', 'a62', 'a63', 'a64', 'a65', 'a66',
    'a67', 'a68', 'a69', 'a70', 'a71', 'a72', 'a73', 'a74', 'a203', 'a75',
    'a204', 'a76', 'a77', 'a78', 'a79', 'a81', 'a82', 'a83', 'a84', 'a97',
    'a98', 'a99', 'a100', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '', '',
    '', '', 'a101', 'a102', 'a103', 'a104', 'a106', 'a107', 'a108', 'a112',
    'a111', 'a110', 'a109', 'a120', 'a121', 'a122', 'a123', 'a124', 'a125',
    'a126', 'a127', 'a128', 'a129', 'a130', 'a131', 'a132', 'a133', 'a134',
    'a135', 'a136', 'a137', 'a138', 'a139', 'a140', 'a141', 'a142', 'a143',
    'a144', 'a145', 'a146', 'a147', 'a148', 'a149', 'a150', 'a151', 'a152',
    'a153', 'a154', 'a155', 'a156', 'a157', 'a158', 'a159', 'a160', 'a161',
    'a163', 'a164', 'a196', 'a165', 'a192', 'a166', 'a167', 'a168', 'a169',
    'a170', 'a171', 'a172', 'a173', 'a162', 'a174', 'a175', 'a176', 'a177',
    'a178', 'a179', 'a193', 'a180', 'a199', 'a181', 'a200', 'a182', '', 'a201',
    'a183', 'a184', 'a197', 'a185', 'a194', 'a198', 'a186', 'a195', 'a187',
    'a188', 'a189', 'a190', 'a191']
};

/**
 * Hold a map of decoded fonts and of the standard fourteen Type1
 * fonts and their acronyms.
 */
var stdFontMap = {
  'ArialNarrow': 'Helvetica',
  'ArialNarrow-Bold': 'Helvetica-Bold',
  'ArialNarrow-BoldItalic': 'Helvetica-BoldOblique',
  'ArialNarrow-Italic': 'Helvetica-Oblique',
  'ArialBlack': 'Helvetica',
  'ArialBlack-Bold': 'Helvetica-Bold',
  'ArialBlack-BoldItalic': 'Helvetica-BoldOblique',
  'ArialBlack-Italic': 'Helvetica-Oblique',
  'Arial': 'Helvetica',
  'Arial-Bold': 'Helvetica-Bold',
  'Arial-BoldItalic': 'Helvetica-BoldOblique',
  'Arial-Italic': 'Helvetica-Oblique',
  'Arial-BoldItalicMT': 'Helvetica-BoldOblique',
  'Arial-BoldMT': 'Helvetica-Bold',
  'Arial-ItalicMT': 'Helvetica-Oblique',
  'ArialMT': 'Helvetica',
  'Courier-Bold': 'Courier-Bold',
  'Courier-BoldItalic': 'Courier-BoldOblique',
  'Courier-Italic': 'Courier-Oblique',
  'CourierNew': 'Courier',
  'CourierNew-Bold': 'Courier-Bold',
  'CourierNew-BoldItalic': 'Courier-BoldOblique',
  'CourierNew-Italic': 'Courier-Oblique',
  'CourierNewPS-BoldItalicMT': 'Courier-BoldOblique',
  'CourierNewPS-BoldMT': 'Courier-Bold',
  'CourierNewPS-ItalicMT': 'Courier-Oblique',
  'CourierNewPSMT': 'Courier',
  'Helvetica-Bold': 'Helvetica-Bold',
  'Helvetica-BoldItalic': 'Helvetica-BoldOblique',
  'Helvetica-Italic': 'Helvetica-Oblique',
  'Symbol-Bold': 'Symbol',
  'Symbol-BoldItalic': 'Symbol',
  'Symbol-Italic': 'Symbol',
  'TimesNewRoman': 'Times-Roman',
  'TimesNewRoman-Bold': 'Times-Bold',
  'TimesNewRoman-BoldItalic': 'Times-BoldItalic',
  'TimesNewRoman-Italic': 'Times-Italic',
  'TimesNewRomanPS': 'Times-Roman',
  'TimesNewRomanPS-Bold': 'Times-Bold',
  'TimesNewRomanPS-BoldItalic': 'Times-BoldItalic',
  'TimesNewRomanPS-BoldItalicMT': 'Times-BoldItalic',
  'TimesNewRomanPS-BoldMT': 'Times-Bold',
  'TimesNewRomanPS-Italic': 'Times-Italic',
  'TimesNewRomanPS-ItalicMT': 'Times-Italic',
  'TimesNewRomanPSMT': 'Times-Roman',
  'TimesNewRomanPSMT-Bold': 'Times-Bold',
  'TimesNewRomanPSMT-BoldItalic': 'Times-BoldItalic',
  'TimesNewRomanPSMT-Italic': 'Times-Italic'
};

/**
 * Holds the map of the non-standard fonts that might be included as a standard
 * fonts without glyph data.
 */
var nonStdFontMap = {
  'ComicSansMS': 'Comic Sans MS',
  'ComicSansMS-Bold': 'Comic Sans MS-Bold',
  'ComicSansMS-BoldItalic': 'Comic Sans MS-BoldItalic',
  'ComicSansMS-Italic': 'Comic Sans MS-Italic',
  'LucidaConsole': 'Courier',
  'LucidaConsole-Bold': 'Courier-Bold',
  'LucidaConsole-BoldItalic': 'Courier-BoldOblique',
  'LucidaConsole-Italic': 'Courier-Oblique',
  'MS-Gothic': 'MS Gothic',
  'MS-Gothic-Bold': 'MS Gothic-Bold',
  'MS-Gothic-BoldItalic': 'MS Gothic-BoldItalic',
  'MS-Gothic-Italic': 'MS Gothic-Italic',
  'MS-Mincho': 'MS Mincho',
  'MS-Mincho-Bold': 'MS Mincho-Bold',
  'MS-Mincho-BoldItalic': 'MS Mincho-BoldItalic',
  'MS-Mincho-Italic': 'MS Mincho-Italic',
  'MS-PGothic': 'MS PGothic',
  'MS-PGothic-Bold': 'MS PGothic-Bold',
  'MS-PGothic-BoldItalic': 'MS PGothic-BoldItalic',
  'MS-PGothic-Italic': 'MS PGothic-Italic',
  'MS-PMincho': 'MS PMincho',
  'MS-PMincho-Bold': 'MS PMincho-Bold',
  'MS-PMincho-BoldItalic': 'MS PMincho-BoldItalic',
  'MS-PMincho-Italic': 'MS PMincho-Italic'
};

var serifFonts = {
  'Adobe Jenson': true, 'Adobe Text': true, 'Albertus': true,
  'Aldus': true, 'Alexandria': true, 'Algerian': true,
  'American Typewriter': true, 'Antiqua': true, 'Apex': true,
  'Arno': true, 'Aster': true, 'Aurora': true,
  'Baskerville': true, 'Bell': true, 'Bembo': true,
  'Bembo Schoolbook': true, 'Benguiat': true, 'Berkeley Old Style': true,
  'Bernhard Modern': true, 'Berthold City': true, 'Bodoni': true,
  'Bauer Bodoni': true, 'Book Antiqua': true, 'Bookman': true,
  'Bordeaux Roman': true, 'Californian FB': true, 'Calisto': true,
  'Calvert': true, 'Capitals': true, 'Cambria': true,
  'Cartier': true, 'Caslon': true, 'Catull': true,
  'Centaur': true, 'Century Old Style': true, 'Century Schoolbook': true,
  'Chaparral': true, 'Charis SIL': true, 'Cheltenham': true,
  'Cholla Slab': true, 'Clarendon': true, 'Clearface': true,
  'Cochin': true, 'Colonna': true, 'Computer Modern': true,
  'Concrete Roman': true, 'Constantia': true, 'Cooper Black': true,
  'Corona': true, 'Ecotype': true, 'Egyptienne': true,
  'Elephant': true, 'Excelsior': true, 'Fairfield': true,
  'FF Scala': true, 'Folkard': true, 'Footlight': true,
  'FreeSerif': true, 'Friz Quadrata': true, 'Garamond': true,
  'Gentium': true, 'Georgia': true, 'Gloucester': true,
  'Goudy Old Style': true, 'Goudy Schoolbook': true, 'Goudy Pro Font': true,
  'Granjon': true, 'Guardian Egyptian': true, 'Heather': true,
  'Hercules': true, 'High Tower Text': true, 'Hiroshige': true,
  'Hoefler Text': true, 'Humana Serif': true, 'Imprint': true,
  'Ionic No. 5': true, 'Janson': true, 'Joanna': true,
  'Korinna': true, 'Lexicon': true, 'Liberation Serif': true,
  'Linux Libertine': true, 'Literaturnaya': true, 'Lucida': true,
  'Lucida Bright': true, 'Melior': true, 'Memphis': true,
  'Miller': true, 'Minion': true, 'Modern': true,
  'Mona Lisa': true, 'Mrs Eaves': true, 'MS Serif': true,
  'Museo Slab': true, 'New York': true, 'Nimbus Roman': true,
  'NPS Rawlinson Roadway': true, 'Palatino': true, 'Perpetua': true,
  'Plantin': true, 'Plantin Schoolbook': true, 'Playbill': true,
  'Poor Richard': true, 'Rawlinson Roadway': true, 'Renault': true,
  'Requiem': true, 'Rockwell': true, 'Roman': true,
  'Rotis Serif': true, 'Sabon': true, 'Scala': true,
  'Seagull': true, 'Sistina': true, 'Souvenir': true,
  'STIX': true, 'Stone Informal': true, 'Stone Serif': true,
  'Sylfaen': true, 'Times': true, 'Trajan': true,
  'Trinité': true, 'Trump Mediaeval': true, 'Utopia': true,
  'Vale Type': true, 'Bitstream Vera': true, 'Vera Serif': true,
  'Versailles': true, 'Wanted': true, 'Weiss': true,
  'Wide Latin': true, 'Windsor': true, 'XITS': true
};

var symbolsFonts = {
  'Dingbats': true, 'Symbol': true, 'ZapfDingbats': true
};

// Glyph map for well-known standard fonts. Sometimes Ghostscript uses CID fonts
// but does not embed the CID to GID mapping. The mapping is incomplete for all
// glyphs, but common for some set of the standard fonts.
var GlyphMapForStandardFonts = {
  '2': 10, '3': 32, '4': 33, '5': 34, '6': 35, '7': 36, '8': 37, '9': 38,
  '10': 39, '11': 40, '12': 41, '13': 42, '14': 43, '15': 44, '16': 45,
  '17': 46, '18': 47, '19': 48, '20': 49, '21': 50, '22': 51, '23': 52,
  '24': 53, '25': 54, '26': 55, '27': 56, '28': 57, '29': 58, '30': 894,
  '31': 60, '32': 61, '33': 62, '34': 63, '35': 64, '36': 65, '37': 66,
  '38': 67, '39': 68, '40': 69, '41': 70, '42': 71, '43': 72, '44': 73,
  '45': 74, '46': 75, '47': 76, '48': 77, '49': 78, '50': 79, '51': 80,
  '52': 81, '53': 82, '54': 83, '55': 84, '56': 85, '57': 86, '58': 87,
  '59': 88, '60': 89, '61': 90, '62': 91, '63': 92, '64': 93, '65': 94,
  '66': 95, '67': 96, '68': 97, '69': 98, '70': 99, '71': 100, '72': 101,
  '73': 102, '74': 103, '75': 104, '76': 105, '77': 106, '78': 107, '79': 108,
  '80': 109, '81': 110, '82': 111, '83': 112, '84': 113, '85': 114, '86': 115,
  '87': 116, '88': 117, '89': 118, '90': 119, '91': 120, '92': 121, '93': 122,
  '94': 123, '95': 124, '96': 125, '97': 126, '98': 196, '99': 197, '100': 199,
  '101': 201, '102': 209, '103': 214, '104': 220, '105': 225, '106': 224,
  '107': 226, '108': 228, '109': 227, '110': 229, '111': 231, '112': 233,
  '113': 232, '114': 234, '115': 235, '116': 237, '117': 236, '118': 238,
  '119': 239, '120': 241, '121': 243, '122': 242, '123': 244, '124': 246,
  '125': 245, '126': 250, '127': 249, '128': 251, '129': 252, '130': 8224,
  '131': 176, '132': 162, '133': 163, '134': 167, '135': 8226, '136': 182,
  '137': 223, '138': 174, '139': 169, '140': 8482, '141': 180, '142': 168,
  '143': 8800, '144': 198, '145': 216, '146': 8734, '147': 177, '148': 8804,
  '149': 8805, '150': 165, '151': 181, '152': 8706, '153': 8721, '154': 8719,
  '156': 8747, '157': 170, '158': 186, '159': 8486, '160': 230, '161': 248,
  '162': 191, '163': 161, '164': 172, '165': 8730, '166': 402, '167': 8776,
  '168': 8710, '169': 171, '170': 187, '171': 8230, '210': 218, '223': 711,
  '227': 353, '229': 382, '234': 253, '253': 268, '254': 269, '258': 258,
  '268': 283, '269': 313, '278': 328, '284': 345, '292': 367, '305': 963,
  '306': 964, '307': 966, '308': 8215, '309': 8252, '310': 8319, '311': 8359,
  '312': 8592, '313': 8593, '337': 9552, '493': 1039, '494': 1040, '705': 1524,
  '706': 8362, '710': 64288, '711': 64298, '759': 1617, '761': 1776,
  '763': 1778, '775': 1652, '777': 1764, '778': 1780, '779': 1781, '780': 1782,
  '782': 771, '783': 64726, '786': 8363, '788': 8532, '790': 768, '791': 769,
  '792': 768, '795': 803, '797': 64336, '798': 64337, '799': 64342,
  '800': 64343, '801': 64344, '802': 64345, '803': 64362, '804': 64363,
  '805': 64364, '2424': 7821, '2425': 7822, '2426': 7823, '2427': 7824,
  '2428': 7825, '2429': 7826, '2430': 7827, '2433': 7682, '2678': 8045,
  '2679': 8046, '2830': 1552, '2838': 686, '2840': 751, '2842': 753,
  '2843': 754, '2844': 755, '2846': 757, '2856': 767, '2857': 848, '2858': 849,
  '2862': 853, '2863': 854, '2864': 855, '2865': 861, '2866': 862, '2906': 7460,
  '2908': 7462, '2909': 7463, '2910': 7464, '2912': 7466, '2913': 7467,
  '2914': 7468, '2916': 7470, '2917': 7471, '2918': 7472, '2920': 7474,
  '2921': 7475, '2922': 7476, '2924': 7478, '2925': 7479, '2926': 7480,
  '2928': 7482, '2929': 7483, '2930': 7484, '2932': 7486, '2933': 7487,
  '2934': 7488, '2936': 7490, '2937': 7491, '2938': 7492, '2940': 7494,
  '2941': 7495, '2942': 7496, '2944': 7498, '2946': 7500, '2948': 7502,
  '2950': 7504, '2951': 7505, '2952': 7506, '2954': 7508, '2955': 7509,
  '2956': 7510, '2958': 7512, '2959': 7513, '2960': 7514, '2962': 7516,
  '2963': 7517, '2964': 7518, '2966': 7520, '2967': 7521, '2968': 7522,
  '2970': 7524, '2971': 7525, '2972': 7526, '2974': 7528, '2975': 7529,
  '2976': 7530, '2978': 1537, '2979': 1538, '2980': 1539, '2982': 1549,
  '2983': 1551, '2984': 1552, '2986': 1554, '2987': 1555, '2988': 1556,
  '2990': 1623, '2991': 1624, '2995': 1775, '2999': 1791, '3002': 64290,
  '3003': 64291, '3004': 64292, '3006': 64294, '3007': 64295, '3008': 64296,
  '3011': 1900, '3014': 8223, '3015': 8244, '3017': 7532, '3018': 7533,
  '3019': 7534, '3075': 7590, '3076': 7591, '3079': 7594, '3080': 7595,
  '3083': 7598, '3084': 7599, '3087': 7602, '3088': 7603, '3091': 7606,
  '3092': 7607, '3095': 7610, '3096': 7611, '3099': 7614, '3100': 7615,
  '3103': 7618, '3104': 7619, '3107': 8337, '3108': 8338, '3116': 1884,
  '3119': 1885, '3120': 1885, '3123': 1886, '3124': 1886, '3127': 1887,
  '3128': 1887, '3131': 1888, '3132': 1888, '3135': 1889, '3136': 1889,
  '3139': 1890, '3140': 1890, '3143': 1891, '3144': 1891, '3147': 1892,
  '3148': 1892, '3153': 580, '3154': 581, '3157': 584, '3158': 585, '3161': 588,
  '3162': 589, '3165': 891, '3166': 892, '3169': 1274, '3170': 1275,
  '3173': 1278, '3174': 1279, '3181': 7622, '3182': 7623, '3282': 11799,
  '3316': 578, '3379': 42785, '3393': 1159, '3416': 8377
};

// Some characters, e.g. copyrightserif, mapped to the private use area and
// might not be displayed using standard fonts. Mapping/hacking well-known chars
// to the similar equivalents in the normal characters range.
function mapSpecialUnicodeValues(code) {
  if (code >= 0xFFF0 && code <= 0xFFFF) { // Specials unicode block.
    return 0;
  }
  switch (code) {
    case 0xF8E9: // copyrightsans
    case 0xF6D9: // copyrightserif
      return 0x00A9; // copyright

    case 0xF8E8: // registersans
    case 0xF6DA: // registerserif
      return 0x00AE; // registered

    case 0xF8EA: // trademarksans
    case 0xF6DB: // trademarkserif
      return 0x2122; // trademark

    default:
      return code;
  }
}

var UnicodeRanges = [
  { 'begin': 0x0000, 'end': 0x007F }, // Basic Latin
  { 'begin': 0x0080, 'end': 0x00FF }, // Latin-1 Supplement
  { 'begin': 0x0100, 'end': 0x017F }, // Latin Extended-A
  { 'begin': 0x0180, 'end': 0x024F }, // Latin Extended-B
  { 'begin': 0x0250, 'end': 0x02AF }, // IPA Extensions
  { 'begin': 0x02B0, 'end': 0x02FF }, // Spacing Modifier Letters
  { 'begin': 0x0300, 'end': 0x036F }, // Combining Diacritical Marks
  { 'begin': 0x0370, 'end': 0x03FF }, // Greek and Coptic
  { 'begin': 0x2C80, 'end': 0x2CFF }, // Coptic
  { 'begin': 0x0400, 'end': 0x04FF }, // Cyrillic
  { 'begin': 0x0530, 'end': 0x058F }, // Armenian
  { 'begin': 0x0590, 'end': 0x05FF }, // Hebrew
  { 'begin': 0xA500, 'end': 0xA63F }, // Vai
  { 'begin': 0x0600, 'end': 0x06FF }, // Arabic
  { 'begin': 0x07C0, 'end': 0x07FF }, // NKo
  { 'begin': 0x0900, 'end': 0x097F }, // Devanagari
  { 'begin': 0x0980, 'end': 0x09FF }, // Bengali
  { 'begin': 0x0A00, 'end': 0x0A7F }, // Gurmukhi
  { 'begin': 0x0A80, 'end': 0x0AFF }, // Gujarati
  { 'begin': 0x0B00, 'end': 0x0B7F }, // Oriya
  { 'begin': 0x0B80, 'end': 0x0BFF }, // Tamil
  { 'begin': 0x0C00, 'end': 0x0C7F }, // Telugu
  { 'begin': 0x0C80, 'end': 0x0CFF }, // Kannada
  { 'begin': 0x0D00, 'end': 0x0D7F }, // Malayalam
  { 'begin': 0x0E00, 'end': 0x0E7F }, // Thai
  { 'begin': 0x0E80, 'end': 0x0EFF }, // Lao
  { 'begin': 0x10A0, 'end': 0x10FF }, // Georgian
  { 'begin': 0x1B00, 'end': 0x1B7F }, // Balinese
  { 'begin': 0x1100, 'end': 0x11FF }, // Hangul Jamo
  { 'begin': 0x1E00, 'end': 0x1EFF }, // Latin Extended Additional
  { 'begin': 0x1F00, 'end': 0x1FFF }, // Greek Extended
  { 'begin': 0x2000, 'end': 0x206F }, // General Punctuation
  { 'begin': 0x2070, 'end': 0x209F }, // Superscripts And Subscripts
  { 'begin': 0x20A0, 'end': 0x20CF }, // Currency Symbol
  { 'begin': 0x20D0, 'end': 0x20FF }, // Combining Diacritical Marks For Symbols
  { 'begin': 0x2100, 'end': 0x214F }, // Letterlike Symbols
  { 'begin': 0x2150, 'end': 0x218F }, // Number Forms
  { 'begin': 0x2190, 'end': 0x21FF }, // Arrows
  { 'begin': 0x2200, 'end': 0x22FF }, // Mathematical Operators
  { 'begin': 0x2300, 'end': 0x23FF }, // Miscellaneous Technical
  { 'begin': 0x2400, 'end': 0x243F }, // Control Pictures
  { 'begin': 0x2440, 'end': 0x245F }, // Optical Character Recognition
  { 'begin': 0x2460, 'end': 0x24FF }, // Enclosed Alphanumerics
  { 'begin': 0x2500, 'end': 0x257F }, // Box Drawing
  { 'begin': 0x2580, 'end': 0x259F }, // Block Elements
  { 'begin': 0x25A0, 'end': 0x25FF }, // Geometric Shapes
  { 'begin': 0x2600, 'end': 0x26FF }, // Miscellaneous Symbols
  { 'begin': 0x2700, 'end': 0x27BF }, // Dingbats
  { 'begin': 0x3000, 'end': 0x303F }, // CJK Symbols And Punctuation
  { 'begin': 0x3040, 'end': 0x309F }, // Hiragana
  { 'begin': 0x30A0, 'end': 0x30FF }, // Katakana
  { 'begin': 0x3100, 'end': 0x312F }, // Bopomofo
  { 'begin': 0x3130, 'end': 0x318F }, // Hangul Compatibility Jamo
  { 'begin': 0xA840, 'end': 0xA87F }, // Phags-pa
  { 'begin': 0x3200, 'end': 0x32FF }, // Enclosed CJK Letters And Months
  { 'begin': 0x3300, 'end': 0x33FF }, // CJK Compatibility
  { 'begin': 0xAC00, 'end': 0xD7AF }, // Hangul Syllables
  { 'begin': 0xD800, 'end': 0xDFFF }, // Non-Plane 0 *
  { 'begin': 0x10900, 'end': 0x1091F }, // Phoenicia
  { 'begin': 0x4E00, 'end': 0x9FFF }, // CJK Unified Ideographs
  { 'begin': 0xE000, 'end': 0xF8FF }, // Private Use Area (plane 0)
  { 'begin': 0x31C0, 'end': 0x31EF }, // CJK Strokes
  { 'begin': 0xFB00, 'end': 0xFB4F }, // Alphabetic Presentation Forms
  { 'begin': 0xFB50, 'end': 0xFDFF }, // Arabic Presentation Forms-A
  { 'begin': 0xFE20, 'end': 0xFE2F }, // Combining Half Marks
  { 'begin': 0xFE10, 'end': 0xFE1F }, // Vertical Forms
  { 'begin': 0xFE50, 'end': 0xFE6F }, // Small Form Variants
  { 'begin': 0xFE70, 'end': 0xFEFF }, // Arabic Presentation Forms-B
  { 'begin': 0xFF00, 'end': 0xFFEF }, // Halfwidth And Fullwidth Forms
  { 'begin': 0xFFF0, 'end': 0xFFFF }, // Specials
  { 'begin': 0x0F00, 'end': 0x0FFF }, // Tibetan
  { 'begin': 0x0700, 'end': 0x074F }, // Syriac
  { 'begin': 0x0780, 'end': 0x07BF }, // Thaana
  { 'begin': 0x0D80, 'end': 0x0DFF }, // Sinhala
  { 'begin': 0x1000, 'end': 0x109F }, // Myanmar
  { 'begin': 0x1200, 'end': 0x137F }, // Ethiopic
  { 'begin': 0x13A0, 'end': 0x13FF }, // Cherokee
  { 'begin': 0x1400, 'end': 0x167F }, // Unified Canadian Aboriginal Syllabics
  { 'begin': 0x1680, 'end': 0x169F }, // Ogham
  { 'begin': 0x16A0, 'end': 0x16FF }, // Runic
  { 'begin': 0x1780, 'end': 0x17FF }, // Khmer
  { 'begin': 0x1800, 'end': 0x18AF }, // Mongolian
  { 'begin': 0x2800, 'end': 0x28FF }, // Braille Patterns
  { 'begin': 0xA000, 'end': 0xA48F }, // Yi Syllables
  { 'begin': 0x1700, 'end': 0x171F }, // Tagalog
  { 'begin': 0x10300, 'end': 0x1032F }, // Old Italic
  { 'begin': 0x10330, 'end': 0x1034F }, // Gothic
  { 'begin': 0x10400, 'end': 0x1044F }, // Deseret
  { 'begin': 0x1D000, 'end': 0x1D0FF }, // Byzantine Musical Symbols
  { 'begin': 0x1D400, 'end': 0x1D7FF }, // Mathematical Alphanumeric Symbols
  { 'begin': 0xFF000, 'end': 0xFFFFD }, // Private Use (plane 15)
  { 'begin': 0xFE00, 'end': 0xFE0F }, // Variation Selectors
  { 'begin': 0xE0000, 'end': 0xE007F }, // Tags
  { 'begin': 0x1900, 'end': 0x194F }, // Limbu
  { 'begin': 0x1950, 'end': 0x197F }, // Tai Le
  { 'begin': 0x1980, 'end': 0x19DF }, // New Tai Lue
  { 'begin': 0x1A00, 'end': 0x1A1F }, // Buginese
  { 'begin': 0x2C00, 'end': 0x2C5F }, // Glagolitic
  { 'begin': 0x2D30, 'end': 0x2D7F }, // Tifinagh
  { 'begin': 0x4DC0, 'end': 0x4DFF }, // Yijing Hexagram Symbols
  { 'begin': 0xA800, 'end': 0xA82F }, // Syloti Nagri
  { 'begin': 0x10000, 'end': 0x1007F }, // Linear B Syllabary
  { 'begin': 0x10140, 'end': 0x1018F }, // Ancient Greek Numbers
  { 'begin': 0x10380, 'end': 0x1039F }, // Ugaritic
  { 'begin': 0x103A0, 'end': 0x103DF }, // Old Persian
  { 'begin': 0x10450, 'end': 0x1047F }, // Shavian
  { 'begin': 0x10480, 'end': 0x104AF }, // Osmanya
  { 'begin': 0x10800, 'end': 0x1083F }, // Cypriot Syllabary
  { 'begin': 0x10A00, 'end': 0x10A5F }, // Kharoshthi
  { 'begin': 0x1D300, 'end': 0x1D35F }, // Tai Xuan Jing Symbols
  { 'begin': 0x12000, 'end': 0x123FF }, // Cuneiform
  { 'begin': 0x1D360, 'end': 0x1D37F }, // Counting Rod Numerals
  { 'begin': 0x1B80, 'end': 0x1BBF }, // Sundanese
  { 'begin': 0x1C00, 'end': 0x1C4F }, // Lepcha
  { 'begin': 0x1C50, 'end': 0x1C7F }, // Ol Chiki
  { 'begin': 0xA880, 'end': 0xA8DF }, // Saurashtra
  { 'begin': 0xA900, 'end': 0xA92F }, // Kayah Li
  { 'begin': 0xA930, 'end': 0xA95F }, // Rejang
  { 'begin': 0xAA00, 'end': 0xAA5F }, // Cham
  { 'begin': 0x10190, 'end': 0x101CF }, // Ancient Symbols
  { 'begin': 0x101D0, 'end': 0x101FF }, // Phaistos Disc
  { 'begin': 0x102A0, 'end': 0x102DF }, // Carian
  { 'begin': 0x1F030, 'end': 0x1F09F }  // Domino Tiles
];

var MacStandardGlyphOrdering = [
  '.notdef', '.null', 'nonmarkingreturn', 'space', 'exclam', 'quotedbl',
  'numbersign', 'dollar', 'percent', 'ampersand', 'quotesingle', 'parenleft',
  'parenright', 'asterisk', 'plus', 'comma', 'hyphen', 'period', 'slash',
  'zero', 'one', 'two', 'three', 'four', 'five', 'six', 'seven', 'eight',
  'nine', 'colon', 'semicolon', 'less', 'equal', 'greater', 'question', 'at',
  'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 'L', 'M', 'N', 'O',
  'P', 'Q', 'R', 'S', 'T', 'U', 'V', 'W', 'X', 'Y', 'Z', 'bracketleft',
  'backslash', 'bracketright', 'asciicircum', 'underscore', 'grave', 'a', 'b',
  'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k', 'l', 'm', 'n', 'o', 'p', 'q',
  'r', 's', 't', 'u', 'v', 'w', 'x', 'y', 'z', 'braceleft', 'bar', 'braceright',
  'asciitilde', 'Adieresis', 'Aring', 'Ccedilla', 'Eacute', 'Ntilde',
  'Odieresis', 'Udieresis', 'aacute', 'agrave', 'acircumflex', 'adieresis',
  'atilde', 'aring', 'ccedilla', 'eacute', 'egrave', 'ecircumflex', 'edieresis',
  'iacute', 'igrave', 'icircumflex', 'idieresis', 'ntilde', 'oacute', 'ograve',
  'ocircumflex', 'odieresis', 'otilde', 'uacute', 'ugrave', 'ucircumflex',
  'udieresis', 'dagger', 'degree', 'cent', 'sterling', 'section', 'bullet',
  'paragraph', 'germandbls', 'registered', 'copyright', 'trademark', 'acute',
  'dieresis', 'notequal', 'AE', 'Oslash', 'infinity', 'plusminus', 'lessequal',
  'greaterequal', 'yen', 'mu', 'partialdiff', 'summation', 'product', 'pi',
  'integral', 'ordfeminine', 'ordmasculine', 'Omega', 'ae', 'oslash',
  'questiondown', 'exclamdown', 'logicalnot', 'radical', 'florin',
  'approxequal', 'Delta', 'guillemotleft', 'guillemotright', 'ellipsis',
  'nonbreakingspace', 'Agrave', 'Atilde', 'Otilde', 'OE', 'oe', 'endash',
  'emdash', 'quotedblleft', 'quotedblright', 'quoteleft', 'quoteright',
  'divide', 'lozenge', 'ydieresis', 'Ydieresis', 'fraction', 'currency',
  'guilsinglleft', 'guilsinglright', 'fi', 'fl', 'daggerdbl', 'periodcentered',
  'quotesinglbase', 'quotedblbase', 'perthousand', 'Acircumflex',
  'Ecircumflex', 'Aacute', 'Edieresis', 'Egrave', 'Iacute', 'Icircumflex',
  'Idieresis', 'Igrave', 'Oacute', 'Ocircumflex', 'apple', 'Ograve', 'Uacute',
  'Ucircumflex', 'Ugrave', 'dotlessi', 'circumflex', 'tilde', 'macron',
  'breve', 'dotaccent', 'ring', 'cedilla', 'hungarumlaut', 'ogonek', 'caron',
  'Lslash', 'lslash', 'Scaron', 'scaron', 'Zcaron', 'zcaron', 'brokenbar',
  'Eth', 'eth', 'Yacute', 'yacute', 'Thorn', 'thorn', 'minus', 'multiply',
  'onesuperior', 'twosuperior', 'threesuperior', 'onehalf', 'onequarter',
  'threequarters', 'franc', 'Gbreve', 'gbreve', 'Idotaccent', 'Scedilla',
  'scedilla', 'Cacute', 'cacute', 'Ccaron', 'ccaron', 'dcroat'];

function getUnicodeRangeFor(value) {
  for (var i = 0, ii = UnicodeRanges.length; i < ii; i++) {
    var range = UnicodeRanges[i];
    if (value >= range.begin && value < range.end) {
      return i;
    }
  }
  return -1;
}

function isRTLRangeFor(value) {
  var range = UnicodeRanges[13];
  if (value >= range.begin && value < range.end) {
    return true;
  }
  range = UnicodeRanges[11];
  if (value >= range.begin && value < range.end) {
    return true;
  }
  return false;
}

// The normalization table is obtained by filtering the Unicode characters
// database with <compat> entries.
var NormalizedUnicodes = {
  '\u00A8': '\u0020\u0308',
  '\u00AF': '\u0020\u0304',
  '\u00B4': '\u0020\u0301',
  '\u00B5': '\u03BC',
  '\u00B8': '\u0020\u0327',
  '\u0132': '\u0049\u004A',
  '\u0133': '\u0069\u006A',
  '\u013F': '\u004C\u00B7',
  '\u0140': '\u006C\u00B7',
  '\u0149': '\u02BC\u006E',
  '\u017F': '\u0073',
  '\u01C4': '\u0044\u017D',
  '\u01C5': '\u0044\u017E',
  '\u01C6': '\u0064\u017E',
  '\u01C7': '\u004C\u004A',
  '\u01C8': '\u004C\u006A',
  '\u01C9': '\u006C\u006A',
  '\u01CA': '\u004E\u004A',
  '\u01CB': '\u004E\u006A',
  '\u01CC': '\u006E\u006A',
  '\u01F1': '\u0044\u005A',
  '\u01F2': '\u0044\u007A',
  '\u01F3': '\u0064\u007A',
  '\u02D8': '\u0020\u0306',
  '\u02D9': '\u0020\u0307',
  '\u02DA': '\u0020\u030A',
  '\u02DB': '\u0020\u0328',
  '\u02DC': '\u0020\u0303',
  '\u02DD': '\u0020\u030B',
  '\u037A': '\u0020\u0345',
  '\u0384': '\u0020\u0301',
  '\u03D0': '\u03B2',
  '\u03D1': '\u03B8',
  '\u03D2': '\u03A5',
  '\u03D5': '\u03C6',
  '\u03D6': '\u03C0',
  '\u03F0': '\u03BA',
  '\u03F1': '\u03C1',
  '\u03F2': '\u03C2',
  '\u03F4': '\u0398',
  '\u03F5': '\u03B5',
  '\u03F9': '\u03A3',
  '\u0587': '\u0565\u0582',
  '\u0675': '\u0627\u0674',
  '\u0676': '\u0648\u0674',
  '\u0677': '\u06C7\u0674',
  '\u0678': '\u064A\u0674',
  '\u0E33': '\u0E4D\u0E32',
  '\u0EB3': '\u0ECD\u0EB2',
  '\u0EDC': '\u0EAB\u0E99',
  '\u0EDD': '\u0EAB\u0EA1',
  '\u0F77': '\u0FB2\u0F81',
  '\u0F79': '\u0FB3\u0F81',
  '\u1E9A': '\u0061\u02BE',
  '\u1FBD': '\u0020\u0313',
  '\u1FBF': '\u0020\u0313',
  '\u1FC0': '\u0020\u0342',
  '\u1FFE': '\u0020\u0314',
  '\u2002': '\u0020',
  '\u2003': '\u0020',
  '\u2004': '\u0020',
  '\u2005': '\u0020',
  '\u2006': '\u0020',
  '\u2008': '\u0020',
  '\u2009': '\u0020',
  '\u200A': '\u0020',
  '\u2017': '\u0020\u0333',
  '\u2024': '\u002E',
  '\u2025': '\u002E\u002E',
  '\u2026': '\u002E\u002E\u002E',
  '\u2033': '\u2032\u2032',
  '\u2034': '\u2032\u2032\u2032',
  '\u2036': '\u2035\u2035',
  '\u2037': '\u2035\u2035\u2035',
  '\u203C': '\u0021\u0021',
  '\u203E': '\u0020\u0305',
  '\u2047': '\u003F\u003F',
  '\u2048': '\u003F\u0021',
  '\u2049': '\u0021\u003F',
  '\u2057': '\u2032\u2032\u2032\u2032',
  '\u205F': '\u0020',
  '\u20A8': '\u0052\u0073',
  '\u2100': '\u0061\u002F\u0063',
  '\u2101': '\u0061\u002F\u0073',
  '\u2103': '\u00B0\u0043',
  '\u2105': '\u0063\u002F\u006F',
  '\u2106': '\u0063\u002F\u0075',
  '\u2107': '\u0190',
  '\u2109': '\u00B0\u0046',
  '\u2116': '\u004E\u006F',
  '\u2121': '\u0054\u0045\u004C',
  '\u2135': '\u05D0',
  '\u2136': '\u05D1',
  '\u2137': '\u05D2',
  '\u2138': '\u05D3',
  '\u213B': '\u0046\u0041\u0058',
  '\u2160': '\u0049',
  '\u2161': '\u0049\u0049',
  '\u2162': '\u0049\u0049\u0049',
  '\u2163': '\u0049\u0056',
  '\u2164': '\u0056',
  '\u2165': '\u0056\u0049',
  '\u2166': '\u0056\u0049\u0049',
  '\u2167': '\u0056\u0049\u0049\u0049',
  '\u2168': '\u0049\u0058',
  '\u2169': '\u0058',
  '\u216A': '\u0058\u0049',
  '\u216B': '\u0058\u0049\u0049',
  '\u216C': '\u004C',
  '\u216D': '\u0043',
  '\u216E': '\u0044',
  '\u216F': '\u004D',
  '\u2170': '\u0069',
  '\u2171': '\u0069\u0069',
  '\u2172': '\u0069\u0069\u0069',
  '\u2173': '\u0069\u0076',
  '\u2174': '\u0076',
  '\u2175': '\u0076\u0069',
  '\u2176': '\u0076\u0069\u0069',
  '\u2177': '\u0076\u0069\u0069\u0069',
  '\u2178': '\u0069\u0078',
  '\u2179': '\u0078',
  '\u217A': '\u0078\u0069',
  '\u217B': '\u0078\u0069\u0069',
  '\u217C': '\u006C',
  '\u217D': '\u0063',
  '\u217E': '\u0064',
  '\u217F': '\u006D',
  '\u222C': '\u222B\u222B',
  '\u222D': '\u222B\u222B\u222B',
  '\u222F': '\u222E\u222E',
  '\u2230': '\u222E\u222E\u222E',
  '\u2474': '\u0028\u0031\u0029',
  '\u2475': '\u0028\u0032\u0029',
  '\u2476': '\u0028\u0033\u0029',
  '\u2477': '\u0028\u0034\u0029',
  '\u2478': '\u0028\u0035\u0029',
  '\u2479': '\u0028\u0036\u0029',
  '\u247A': '\u0028\u0037\u0029',
  '\u247B': '\u0028\u0038\u0029',
  '\u247C': '\u0028\u0039\u0029',
  '\u247D': '\u0028\u0031\u0030\u0029',
  '\u247E': '\u0028\u0031\u0031\u0029',
  '\u247F': '\u0028\u0031\u0032\u0029',
  '\u2480': '\u0028\u0031\u0033\u0029',
  '\u2481': '\u0028\u0031\u0034\u0029',
  '\u2482': '\u0028\u0031\u0035\u0029',
  '\u2483': '\u0028\u0031\u0036\u0029',
  '\u2484': '\u0028\u0031\u0037\u0029',
  '\u2485': '\u0028\u0031\u0038\u0029',
  '\u2486': '\u0028\u0031\u0039\u0029',
  '\u2487': '\u0028\u0032\u0030\u0029',
  '\u2488': '\u0031\u002E',
  '\u2489': '\u0032\u002E',
  '\u248A': '\u0033\u002E',
  '\u248B': '\u0034\u002E',
  '\u248C': '\u0035\u002E',
  '\u248D': '\u0036\u002E',
  '\u248E': '\u0037\u002E',
  '\u248F': '\u0038\u002E',
  '\u2490': '\u0039\u002E',
  '\u2491': '\u0031\u0030\u002E',
  '\u2492': '\u0031\u0031\u002E',
  '\u2493': '\u0031\u0032\u002E',
  '\u2494': '\u0031\u0033\u002E',
  '\u2495': '\u0031\u0034\u002E',
  '\u2496': '\u0031\u0035\u002E',
  '\u2497': '\u0031\u0036\u002E',
  '\u2498': '\u0031\u0037\u002E',
  '\u2499': '\u0031\u0038\u002E',
  '\u249A': '\u0031\u0039\u002E',
  '\u249B': '\u0032\u0030\u002E',
  '\u249C': '\u0028\u0061\u0029',
  '\u249D': '\u0028\u0062\u0029',
  '\u249E': '\u0028\u0063\u0029',
  '\u249F': '\u0028\u0064\u0029',
  '\u24A0': '\u0028\u0065\u0029',
  '\u24A1': '\u0028\u0066\u0029',
  '\u24A2': '\u0028\u0067\u0029',
  '\u24A3': '\u0028\u0068\u0029',
  '\u24A4': '\u0028\u0069\u0029',
  '\u24A5': '\u0028\u006A\u0029',
  '\u24A6': '\u0028\u006B\u0029',
  '\u24A7': '\u0028\u006C\u0029',
  '\u24A8': '\u0028\u006D\u0029',
  '\u24A9': '\u0028\u006E\u0029',
  '\u24AA': '\u0028\u006F\u0029',
  '\u24AB': '\u0028\u0070\u0029',
  '\u24AC': '\u0028\u0071\u0029',
  '\u24AD': '\u0028\u0072\u0029',
  '\u24AE': '\u0028\u0073\u0029',
  '\u24AF': '\u0028\u0074\u0029',
  '\u24B0': '\u0028\u0075\u0029',
  '\u24B1': '\u0028\u0076\u0029',
  '\u24B2': '\u0028\u0077\u0029',
  '\u24B3': '\u0028\u0078\u0029',
  '\u24B4': '\u0028\u0079\u0029',
  '\u24B5': '\u0028\u007A\u0029',
  '\u2A0C': '\u222B\u222B\u222B\u222B',
  '\u2A74': '\u003A\u003A\u003D',
  '\u2A75': '\u003D\u003D',
  '\u2A76': '\u003D\u003D\u003D',
  '\u2E9F': '\u6BCD',
  '\u2EF3': '\u9F9F',
  '\u2F00': '\u4E00',
  '\u2F01': '\u4E28',
  '\u2F02': '\u4E36',
  '\u2F03': '\u4E3F',
  '\u2F04': '\u4E59',
  '\u2F05': '\u4E85',
  '\u2F06': '\u4E8C',
  '\u2F07': '\u4EA0',
  '\u2F08': '\u4EBA',
  '\u2F09': '\u513F',
  '\u2F0A': '\u5165',
  '\u2F0B': '\u516B',
  '\u2F0C': '\u5182',
  '\u2F0D': '\u5196',
  '\u2F0E': '\u51AB',
  '\u2F0F': '\u51E0',
  '\u2F10': '\u51F5',
  '\u2F11': '\u5200',
  '\u2F12': '\u529B',
  '\u2F13': '\u52F9',
  '\u2F14': '\u5315',
  '\u2F15': '\u531A',
  '\u2F16': '\u5338',
  '\u2F17': '\u5341',
  '\u2F18': '\u535C',
  '\u2F19': '\u5369',
  '\u2F1A': '\u5382',
  '\u2F1B': '\u53B6',
  '\u2F1C': '\u53C8',
  '\u2F1D': '\u53E3',
  '\u2F1E': '\u56D7',
  '\u2F1F': '\u571F',
  '\u2F20': '\u58EB',
  '\u2F21': '\u5902',
  '\u2F22': '\u590A',
  '\u2F23': '\u5915',
  '\u2F24': '\u5927',
  '\u2F25': '\u5973',
  '\u2F26': '\u5B50',
  '\u2F27': '\u5B80',
  '\u2F28': '\u5BF8',
  '\u2F29': '\u5C0F',
  '\u2F2A': '\u5C22',
  '\u2F2B': '\u5C38',
  '\u2F2C': '\u5C6E',
  '\u2F2D': '\u5C71',
  '\u2F2E': '\u5DDB',
  '\u2F2F': '\u5DE5',
  '\u2F30': '\u5DF1',
  '\u2F31': '\u5DFE',
  '\u2F32': '\u5E72',
  '\u2F33': '\u5E7A',
  '\u2F34': '\u5E7F',
  '\u2F35': '\u5EF4',
  '\u2F36': '\u5EFE',
  '\u2F37': '\u5F0B',
  '\u2F38': '\u5F13',
  '\u2F39': '\u5F50',
  '\u2F3A': '\u5F61',
  '\u2F3B': '\u5F73',
  '\u2F3C': '\u5FC3',
  '\u2F3D': '\u6208',
  '\u2F3E': '\u6236',
  '\u2F3F': '\u624B',
  '\u2F40': '\u652F',
  '\u2F41': '\u6534',
  '\u2F42': '\u6587',
  '\u2F43': '\u6597',
  '\u2F44': '\u65A4',
  '\u2F45': '\u65B9',
  '\u2F46': '\u65E0',
  '\u2F47': '\u65E5',
  '\u2F48': '\u66F0',
  '\u2F49': '\u6708',
  '\u2F4A': '\u6728',
  '\u2F4B': '\u6B20',
  '\u2F4C': '\u6B62',
  '\u2F4D': '\u6B79',
  '\u2F4E': '\u6BB3',
  '\u2F4F': '\u6BCB',
  '\u2F50': '\u6BD4',
  '\u2F51': '\u6BDB',
  '\u2F52': '\u6C0F',
  '\u2F53': '\u6C14',
  '\u2F54': '\u6C34',
  '\u2F55': '\u706B',
  '\u2F56': '\u722A',
  '\u2F57': '\u7236',
  '\u2F58': '\u723B',
  '\u2F59': '\u723F',
  '\u2F5A': '\u7247',
  '\u2F5B': '\u7259',
  '\u2F5C': '\u725B',
  '\u2F5D': '\u72AC',
  '\u2F5E': '\u7384',
  '\u2F5F': '\u7389',
  '\u2F60': '\u74DC',
  '\u2F61': '\u74E6',
  '\u2F62': '\u7518',
  '\u2F63': '\u751F',
  '\u2F64': '\u7528',
  '\u2F65': '\u7530',
  '\u2F66': '\u758B',
  '\u2F67': '\u7592',
  '\u2F68': '\u7676',
  '\u2F69': '\u767D',
  '\u2F6A': '\u76AE',
  '\u2F6B': '\u76BF',
  '\u2F6C': '\u76EE',
  '\u2F6D': '\u77DB',
  '\u2F6E': '\u77E2',
  '\u2F6F': '\u77F3',
  '\u2F70': '\u793A',
  '\u2F71': '\u79B8',
  '\u2F72': '\u79BE',
  '\u2F73': '\u7A74',
  '\u2F74': '\u7ACB',
  '\u2F75': '\u7AF9',
  '\u2F76': '\u7C73',
  '\u2F77': '\u7CF8',
  '\u2F78': '\u7F36',
  '\u2F79': '\u7F51',
  '\u2F7A': '\u7F8A',
  '\u2F7B': '\u7FBD',
  '\u2F7C': '\u8001',
  '\u2F7D': '\u800C',
  '\u2F7E': '\u8012',
  '\u2F7F': '\u8033',
  '\u2F80': '\u807F',
  '\u2F81': '\u8089',
  '\u2F82': '\u81E3',
  '\u2F83': '\u81EA',
  '\u2F84': '\u81F3',
  '\u2F85': '\u81FC',
  '\u2F86': '\u820C',
  '\u2F87': '\u821B',
  '\u2F88': '\u821F',
  '\u2F89': '\u826E',
  '\u2F8A': '\u8272',
  '\u2F8B': '\u8278',
  '\u2F8C': '\u864D',
  '\u2F8D': '\u866B',
  '\u2F8E': '\u8840',
  '\u2F8F': '\u884C',
  '\u2F90': '\u8863',
  '\u2F91': '\u897E',
  '\u2F92': '\u898B',
  '\u2F93': '\u89D2',
  '\u2F94': '\u8A00',
  '\u2F95': '\u8C37',
  '\u2F96': '\u8C46',
  '\u2F97': '\u8C55',
  '\u2F98': '\u8C78',
  '\u2F99': '\u8C9D',
  '\u2F9A': '\u8D64',
  '\u2F9B': '\u8D70',
  '\u2F9C': '\u8DB3',
  '\u2F9D': '\u8EAB',
  '\u2F9E': '\u8ECA',
  '\u2F9F': '\u8F9B',
  '\u2FA0': '\u8FB0',
  '\u2FA1': '\u8FB5',
  '\u2FA2': '\u9091',
  '\u2FA3': '\u9149',
  '\u2FA4': '\u91C6',
  '\u2FA5': '\u91CC',
  '\u2FA6': '\u91D1',
  '\u2FA7': '\u9577',
  '\u2FA8': '\u9580',
  '\u2FA9': '\u961C',
  '\u2FAA': '\u96B6',
  '\u2FAB': '\u96B9',
  '\u2FAC': '\u96E8',
  '\u2FAD': '\u9751',
  '\u2FAE': '\u975E',
  '\u2FAF': '\u9762',
  '\u2FB0': '\u9769',
  '\u2FB1': '\u97CB',
  '\u2FB2': '\u97ED',
  '\u2FB3': '\u97F3',
  '\u2FB4': '\u9801',
  '\u2FB5': '\u98A8',
  '\u2FB6': '\u98DB',
  '\u2FB7': '\u98DF',
  '\u2FB8': '\u9996',
  '\u2FB9': '\u9999',
  '\u2FBA': '\u99AC',
  '\u2FBB': '\u9AA8',
  '\u2FBC': '\u9AD8',
  '\u2FBD': '\u9ADF',
  '\u2FBE': '\u9B25',
  '\u2FBF': '\u9B2F',
  '\u2FC0': '\u9B32',
  '\u2FC1': '\u9B3C',
  '\u2FC2': '\u9B5A',
  '\u2FC3': '\u9CE5',
  '\u2FC4': '\u9E75',
  '\u2FC5': '\u9E7F',
  '\u2FC6': '\u9EA5',
  '\u2FC7': '\u9EBB',
  '\u2FC8': '\u9EC3',
  '\u2FC9': '\u9ECD',
  '\u2FCA': '\u9ED1',
  '\u2FCB': '\u9EF9',
  '\u2FCC': '\u9EFD',
  '\u2FCD': '\u9F0E',
  '\u2FCE': '\u9F13',
  '\u2FCF': '\u9F20',
  '\u2FD0': '\u9F3B',
  '\u2FD1': '\u9F4A',
  '\u2FD2': '\u9F52',
  '\u2FD3': '\u9F8D',
  '\u2FD4': '\u9F9C',
  '\u2FD5': '\u9FA0',
  '\u3036': '\u3012',
  '\u3038': '\u5341',
  '\u3039': '\u5344',
  '\u303A': '\u5345',
  '\u309B': '\u0020\u3099',
  '\u309C': '\u0020\u309A',
  '\u3131': '\u1100',
  '\u3132': '\u1101',
  '\u3133': '\u11AA',
  '\u3134': '\u1102',
  '\u3135': '\u11AC',
  '\u3136': '\u11AD',
  '\u3137': '\u1103',
  '\u3138': '\u1104',
  '\u3139': '\u1105',
  '\u313A': '\u11B0',
  '\u313B': '\u11B1',
  '\u313C': '\u11B2',
  '\u313D': '\u11B3',
  '\u313E': '\u11B4',
  '\u313F': '\u11B5',
  '\u3140': '\u111A',
  '\u3141': '\u1106',
  '\u3142': '\u1107',
  '\u3143': '\u1108',
  '\u3144': '\u1121',
  '\u3145': '\u1109',
  '\u3146': '\u110A',
  '\u3147': '\u110B',
  '\u3148': '\u110C',
  '\u3149': '\u110D',
  '\u314A': '\u110E',
  '\u314B': '\u110F',
  '\u314C': '\u1110',
  '\u314D': '\u1111',
  '\u314E': '\u1112',
  '\u314F': '\u1161',
  '\u3150': '\u1162',
  '\u3151': '\u1163',
  '\u3152': '\u1164',
  '\u3153': '\u1165',
  '\u3154': '\u1166',
  '\u3155': '\u1167',
  '\u3156': '\u1168',
  '\u3157': '\u1169',
  '\u3158': '\u116A',
  '\u3159': '\u116B',
  '\u315A': '\u116C',
  '\u315B': '\u116D',
  '\u315C': '\u116E',
  '\u315D': '\u116F',
  '\u315E': '\u1170',
  '\u315F': '\u1171',
  '\u3160': '\u1172',
  '\u3161': '\u1173',
  '\u3162': '\u1174',
  '\u3163': '\u1175',
  '\u3164': '\u1160',
  '\u3165': '\u1114',
  '\u3166': '\u1115',
  '\u3167': '\u11C7',
  '\u3168': '\u11C8',
  '\u3169': '\u11CC',
  '\u316A': '\u11CE',
  '\u316B': '\u11D3',
  '\u316C': '\u11D7',
  '\u316D': '\u11D9',
  '\u316E': '\u111C',
  '\u316F': '\u11DD',
  '\u3170': '\u11DF',
  '\u3171': '\u111D',
  '\u3172': '\u111E',
  '\u3173': '\u1120',
  '\u3174': '\u1122',
  '\u3175': '\u1123',
  '\u3176': '\u1127',
  '\u3177': '\u1129',
  '\u3178': '\u112B',
  '\u3179': '\u112C',
  '\u317A': '\u112D',
  '\u317B': '\u112E',
  '\u317C': '\u112F',
  '\u317D': '\u1132',
  '\u317E': '\u1136',
  '\u317F': '\u1140',
  '\u3180': '\u1147',
  '\u3181': '\u114C',
  '\u3182': '\u11F1',
  '\u3183': '\u11F2',
  '\u3184': '\u1157',
  '\u3185': '\u1158',
  '\u3186': '\u1159',
  '\u3187': '\u1184',
  '\u3188': '\u1185',
  '\u3189': '\u1188',
  '\u318A': '\u1191',
  '\u318B': '\u1192',
  '\u318C': '\u1194',
  '\u318D': '\u119E',
  '\u318E': '\u11A1',
  '\u3200': '\u0028\u1100\u0029',
  '\u3201': '\u0028\u1102\u0029',
  '\u3202': '\u0028\u1103\u0029',
  '\u3203': '\u0028\u1105\u0029',
  '\u3204': '\u0028\u1106\u0029',
  '\u3205': '\u0028\u1107\u0029',
  '\u3206': '\u0028\u1109\u0029',
  '\u3207': '\u0028\u110B\u0029',
  '\u3208': '\u0028\u110C\u0029',
  '\u3209': '\u0028\u110E\u0029',
  '\u320A': '\u0028\u110F\u0029',
  '\u320B': '\u0028\u1110\u0029',
  '\u320C': '\u0028\u1111\u0029',
  '\u320D': '\u0028\u1112\u0029',
  '\u320E': '\u0028\u1100\u1161\u0029',
  '\u320F': '\u0028\u1102\u1161\u0029',
  '\u3210': '\u0028\u1103\u1161\u0029',
  '\u3211': '\u0028\u1105\u1161\u0029',
  '\u3212': '\u0028\u1106\u1161\u0029',
  '\u3213': '\u0028\u1107\u1161\u0029',
  '\u3214': '\u0028\u1109\u1161\u0029',
  '\u3215': '\u0028\u110B\u1161\u0029',
  '\u3216': '\u0028\u110C\u1161\u0029',
  '\u3217': '\u0028\u110E\u1161\u0029',
  '\u3218': '\u0028\u110F\u1161\u0029',
  '\u3219': '\u0028\u1110\u1161\u0029',
  '\u321A': '\u0028\u1111\u1161\u0029',
  '\u321B': '\u0028\u1112\u1161\u0029',
  '\u321C': '\u0028\u110C\u116E\u0029',
  '\u321D': '\u0028\u110B\u1169\u110C\u1165\u11AB\u0029',
  '\u321E': '\u0028\u110B\u1169\u1112\u116E\u0029',
  '\u3220': '\u0028\u4E00\u0029',
  '\u3221': '\u0028\u4E8C\u0029',
  '\u3222': '\u0028\u4E09\u0029',
  '\u3223': '\u0028\u56DB\u0029',
  '\u3224': '\u0028\u4E94\u0029',
  '\u3225': '\u0028\u516D\u0029',
  '\u3226': '\u0028\u4E03\u0029',
  '\u3227': '\u0028\u516B\u0029',
  '\u3228': '\u0028\u4E5D\u0029',
  '\u3229': '\u0028\u5341\u0029',
  '\u322A': '\u0028\u6708\u0029',
  '\u322B': '\u0028\u706B\u0029',
  '\u322C': '\u0028\u6C34\u0029',
  '\u322D': '\u0028\u6728\u0029',
  '\u322E': '\u0028\u91D1\u0029',
  '\u322F': '\u0028\u571F\u0029',
  '\u3230': '\u0028\u65E5\u0029',
  '\u3231': '\u0028\u682A\u0029',
  '\u3232': '\u0028\u6709\u0029',
  '\u3233': '\u0028\u793E\u0029',
  '\u3234': '\u0028\u540D\u0029',
  '\u3235': '\u0028\u7279\u0029',
  '\u3236': '\u0028\u8CA1\u0029',
  '\u3237': '\u0028\u795D\u0029',
  '\u3238': '\u0028\u52B4\u0029',
  '\u3239': '\u0028\u4EE3\u0029',
  '\u323A': '\u0028\u547C\u0029',
  '\u323B': '\u0028\u5B66\u0029',
  '\u323C': '\u0028\u76E3\u0029',
  '\u323D': '\u0028\u4F01\u0029',
  '\u323E': '\u0028\u8CC7\u0029',
  '\u323F': '\u0028\u5354\u0029',
  '\u3240': '\u0028\u796D\u0029',
  '\u3241': '\u0028\u4F11\u0029',
  '\u3242': '\u0028\u81EA\u0029',
  '\u3243': '\u0028\u81F3\u0029',
  '\u32C0': '\u0031\u6708',
  '\u32C1': '\u0032\u6708',
  '\u32C2': '\u0033\u6708',
  '\u32C3': '\u0034\u6708',
  '\u32C4': '\u0035\u6708',
  '\u32C5': '\u0036\u6708',
  '\u32C6': '\u0037\u6708',
  '\u32C7': '\u0038\u6708',
  '\u32C8': '\u0039\u6708',
  '\u32C9': '\u0031\u0030\u6708',
  '\u32CA': '\u0031\u0031\u6708',
  '\u32CB': '\u0031\u0032\u6708',
  '\u3358': '\u0030\u70B9',
  '\u3359': '\u0031\u70B9',
  '\u335A': '\u0032\u70B9',
  '\u335B': '\u0033\u70B9',
  '\u335C': '\u0034\u70B9',
  '\u335D': '\u0035\u70B9',
  '\u335E': '\u0036\u70B9',
  '\u335F': '\u0037\u70B9',
  '\u3360': '\u0038\u70B9',
  '\u3361': '\u0039\u70B9',
  '\u3362': '\u0031\u0030\u70B9',
  '\u3363': '\u0031\u0031\u70B9',
  '\u3364': '\u0031\u0032\u70B9',
  '\u3365': '\u0031\u0033\u70B9',
  '\u3366': '\u0031\u0034\u70B9',
  '\u3367': '\u0031\u0035\u70B9',
  '\u3368': '\u0031\u0036\u70B9',
  '\u3369': '\u0031\u0037\u70B9',
  '\u336A': '\u0031\u0038\u70B9',
  '\u336B': '\u0031\u0039\u70B9',
  '\u336C': '\u0032\u0030\u70B9',
  '\u336D': '\u0032\u0031\u70B9',
  '\u336E': '\u0032\u0032\u70B9',
  '\u336F': '\u0032\u0033\u70B9',
  '\u3370': '\u0032\u0034\u70B9',
  '\u33E0': '\u0031\u65E5',
  '\u33E1': '\u0032\u65E5',
  '\u33E2': '\u0033\u65E5',
  '\u33E3': '\u0034\u65E5',
  '\u33E4': '\u0035\u65E5',
  '\u33E5': '\u0036\u65E5',
  '\u33E6': '\u0037\u65E5',
  '\u33E7': '\u0038\u65E5',
  '\u33E8': '\u0039\u65E5',
  '\u33E9': '\u0031\u0030\u65E5',
  '\u33EA': '\u0031\u0031\u65E5',
  '\u33EB': '\u0031\u0032\u65E5',
  '\u33EC': '\u0031\u0033\u65E5',
  '\u33ED': '\u0031\u0034\u65E5',
  '\u33EE': '\u0031\u0035\u65E5',
  '\u33EF': '\u0031\u0036\u65E5',
  '\u33F0': '\u0031\u0037\u65E5',
  '\u33F1': '\u0031\u0038\u65E5',
  '\u33F2': '\u0031\u0039\u65E5',
  '\u33F3': '\u0032\u0030\u65E5',
  '\u33F4': '\u0032\u0031\u65E5',
  '\u33F5': '\u0032\u0032\u65E5',
  '\u33F6': '\u0032\u0033\u65E5',
  '\u33F7': '\u0032\u0034\u65E5',
  '\u33F8': '\u0032\u0035\u65E5',
  '\u33F9': '\u0032\u0036\u65E5',
  '\u33FA': '\u0032\u0037\u65E5',
  '\u33FB': '\u0032\u0038\u65E5',
  '\u33FC': '\u0032\u0039\u65E5',
  '\u33FD': '\u0033\u0030\u65E5',
  '\u33FE': '\u0033\u0031\u65E5',
  '\uFB00': '\u0066\u0066',
  '\uFB01': '\u0066\u0069',
  '\uFB02': '\u0066\u006C',
  '\uFB03': '\u0066\u0066\u0069',
  '\uFB04': '\u0066\u0066\u006C',
  '\uFB05': '\u017F\u0074',
  '\uFB06': '\u0073\u0074',
  '\uFB13': '\u0574\u0576',
  '\uFB14': '\u0574\u0565',
  '\uFB15': '\u0574\u056B',
  '\uFB16': '\u057E\u0576',
  '\uFB17': '\u0574\u056D',
  '\uFB4F': '\u05D0\u05DC',
  '\uFB50': '\u0671',
  '\uFB51': '\u0671',
  '\uFB52': '\u067B',
  '\uFB53': '\u067B',
  '\uFB54': '\u067B',
  '\uFB55': '\u067B',
  '\uFB56': '\u067E',
  '\uFB57': '\u067E',
  '\uFB58': '\u067E',
  '\uFB59': '\u067E',
  '\uFB5A': '\u0680',
  '\uFB5B': '\u0680',
  '\uFB5C': '\u0680',
  '\uFB5D': '\u0680',
  '\uFB5E': '\u067A',
  '\uFB5F': '\u067A',
  '\uFB60': '\u067A',
  '\uFB61': '\u067A',
  '\uFB62': '\u067F',
  '\uFB63': '\u067F',
  '\uFB64': '\u067F',
  '\uFB65': '\u067F',
  '\uFB66': '\u0679',
  '\uFB67': '\u0679',
  '\uFB68': '\u0679',
  '\uFB69': '\u0679',
  '\uFB6A': '\u06A4',
  '\uFB6B': '\u06A4',
  '\uFB6C': '\u06A4',
  '\uFB6D': '\u06A4',
  '\uFB6E': '\u06A6',
  '\uFB6F': '\u06A6',
  '\uFB70': '\u06A6',
  '\uFB71': '\u06A6',
  '\uFB72': '\u0684',
  '\uFB73': '\u0684',
  '\uFB74': '\u0684',
  '\uFB75': '\u0684',
  '\uFB76': '\u0683',
  '\uFB77': '\u0683',
  '\uFB78': '\u0683',
  '\uFB79': '\u0683',
  '\uFB7A': '\u0686',
  '\uFB7B': '\u0686',
  '\uFB7C': '\u0686',
  '\uFB7D': '\u0686',
  '\uFB7E': '\u0687',
  '\uFB7F': '\u0687',
  '\uFB80': '\u0687',
  '\uFB81': '\u0687',
  '\uFB82': '\u068D',
  '\uFB83': '\u068D',
  '\uFB84': '\u068C',
  '\uFB85': '\u068C',
  '\uFB86': '\u068E',
  '\uFB87': '\u068E',
  '\uFB88': '\u0688',
  '\uFB89': '\u0688',
  '\uFB8A': '\u0698',
  '\uFB8B': '\u0698',
  '\uFB8C': '\u0691',
  '\uFB8D': '\u0691',
  '\uFB8E': '\u06A9',
  '\uFB8F': '\u06A9',
  '\uFB90': '\u06A9',
  '\uFB91': '\u06A9',
  '\uFB92': '\u06AF',
  '\uFB93': '\u06AF',
  '\uFB94': '\u06AF',
  '\uFB95': '\u06AF',
  '\uFB96': '\u06B3',
  '\uFB97': '\u06B3',
  '\uFB98': '\u06B3',
  '\uFB99': '\u06B3',
  '\uFB9A': '\u06B1',
  '\uFB9B': '\u06B1',
  '\uFB9C': '\u06B1',
  '\uFB9D': '\u06B1',
  '\uFB9E': '\u06BA',
  '\uFB9F': '\u06BA',
  '\uFBA0': '\u06BB',
  '\uFBA1': '\u06BB',
  '\uFBA2': '\u06BB',
  '\uFBA3': '\u06BB',
  '\uFBA4': '\u06C0',
  '\uFBA5': '\u06C0',
  '\uFBA6': '\u06C1',
  '\uFBA7': '\u06C1',
  '\uFBA8': '\u06C1',
  '\uFBA9': '\u06C1',
  '\uFBAA': '\u06BE',
  '\uFBAB': '\u06BE',
  '\uFBAC': '\u06BE',
  '\uFBAD': '\u06BE',
  '\uFBAE': '\u06D2',
  '\uFBAF': '\u06D2',
  '\uFBB0': '\u06D3',
  '\uFBB1': '\u06D3',
  '\uFBD3': '\u06AD',
  '\uFBD4': '\u06AD',
  '\uFBD5': '\u06AD',
  '\uFBD6': '\u06AD',
  '\uFBD7': '\u06C7',
  '\uFBD8': '\u06C7',
  '\uFBD9': '\u06C6',
  '\uFBDA': '\u06C6',
  '\uFBDB': '\u06C8',
  '\uFBDC': '\u06C8',
  '\uFBDD': '\u0677',
  '\uFBDE': '\u06CB',
  '\uFBDF': '\u06CB',
  '\uFBE0': '\u06C5',
  '\uFBE1': '\u06C5',
  '\uFBE2': '\u06C9',
  '\uFBE3': '\u06C9',
  '\uFBE4': '\u06D0',
  '\uFBE5': '\u06D0',
  '\uFBE6': '\u06D0',
  '\uFBE7': '\u06D0',
  '\uFBE8': '\u0649',
  '\uFBE9': '\u0649',
  '\uFBEA': '\u0626\u0627',
  '\uFBEB': '\u0626\u0627',
  '\uFBEC': '\u0626\u06D5',
  '\uFBED': '\u0626\u06D5',
  '\uFBEE': '\u0626\u0648',
  '\uFBEF': '\u0626\u0648',
  '\uFBF0': '\u0626\u06C7',
  '\uFBF1': '\u0626\u06C7',
  '\uFBF2': '\u0626\u06C6',
  '\uFBF3': '\u0626\u06C6',
  '\uFBF4': '\u0626\u06C8',
  '\uFBF5': '\u0626\u06C8',
  '\uFBF6': '\u0626\u06D0',
  '\uFBF7': '\u0626\u06D0',
  '\uFBF8': '\u0626\u06D0',
  '\uFBF9': '\u0626\u0649',
  '\uFBFA': '\u0626\u0649',
  '\uFBFB': '\u0626\u0649',
  '\uFBFC': '\u06CC',
  '\uFBFD': '\u06CC',
  '\uFBFE': '\u06CC',
  '\uFBFF': '\u06CC',
  '\uFC00': '\u0626\u062C',
  '\uFC01': '\u0626\u062D',
  '\uFC02': '\u0626\u0645',
  '\uFC03': '\u0626\u0649',
  '\uFC04': '\u0626\u064A',
  '\uFC05': '\u0628\u062C',
  '\uFC06': '\u0628\u062D',
  '\uFC07': '\u0628\u062E',
  '\uFC08': '\u0628\u0645',
  '\uFC09': '\u0628\u0649',
  '\uFC0A': '\u0628\u064A',
  '\uFC0B': '\u062A\u062C',
  '\uFC0C': '\u062A\u062D',
  '\uFC0D': '\u062A\u062E',
  '\uFC0E': '\u062A\u0645',
  '\uFC0F': '\u062A\u0649',
  '\uFC10': '\u062A\u064A',
  '\uFC11': '\u062B\u062C',
  '\uFC12': '\u062B\u0645',
  '\uFC13': '\u062B\u0649',
  '\uFC14': '\u062B\u064A',
  '\uFC15': '\u062C\u062D',
  '\uFC16': '\u062C\u0645',
  '\uFC17': '\u062D\u062C',
  '\uFC18': '\u062D\u0645',
  '\uFC19': '\u062E\u062C',
  '\uFC1A': '\u062E\u062D',
  '\uFC1B': '\u062E\u0645',
  '\uFC1C': '\u0633\u062C',
  '\uFC1D': '\u0633\u062D',
  '\uFC1E': '\u0633\u062E',
  '\uFC1F': '\u0633\u0645',
  '\uFC20': '\u0635\u062D',
  '\uFC21': '\u0635\u0645',
  '\uFC22': '\u0636\u062C',
  '\uFC23': '\u0636\u062D',
  '\uFC24': '\u0636\u062E',
  '\uFC25': '\u0636\u0645',
  '\uFC26': '\u0637\u062D',
  '\uFC27': '\u0637\u0645',
  '\uFC28': '\u0638\u0645',
  '\uFC29': '\u0639\u062C',
  '\uFC2A': '\u0639\u0645',
  '\uFC2B': '\u063A\u062C',
  '\uFC2C': '\u063A\u0645',
  '\uFC2D': '\u0641\u062C',
  '\uFC2E': '\u0641\u062D',
  '\uFC2F': '\u0641\u062E',
  '\uFC30': '\u0641\u0645',
  '\uFC31': '\u0641\u0649',
  '\uFC32': '\u0641\u064A',
  '\uFC33': '\u0642\u062D',
  '\uFC34': '\u0642\u0645',
  '\uFC35': '\u0642\u0649',
  '\uFC36': '\u0642\u064A',
  '\uFC37': '\u0643\u0627',
  '\uFC38': '\u0643\u062C',
  '\uFC39': '\u0643\u062D',
  '\uFC3A': '\u0643\u062E',
  '\uFC3B': '\u0643\u0644',
  '\uFC3C': '\u0643\u0645',
  '\uFC3D': '\u0643\u0649',
  '\uFC3E': '\u0643\u064A',
  '\uFC3F': '\u0644\u062C',
  '\uFC40': '\u0644\u062D',
  '\uFC41': '\u0644\u062E',
  '\uFC42': '\u0644\u0645',
  '\uFC43': '\u0644\u0649',
  '\uFC44': '\u0644\u064A',
  '\uFC45': '\u0645\u062C',
  '\uFC46': '\u0645\u062D',
  '\uFC47': '\u0645\u062E',
  '\uFC48': '\u0645\u0645',
  '\uFC49': '\u0645\u0649',
  '\uFC4A': '\u0645\u064A',
  '\uFC4B': '\u0646\u062C',
  '\uFC4C': '\u0646\u062D',
  '\uFC4D': '\u0646\u062E',
  '\uFC4E': '\u0646\u0645',
  '\uFC4F': '\u0646\u0649',
  '\uFC50': '\u0646\u064A',
  '\uFC51': '\u0647\u062C',
  '\uFC52': '\u0647\u0645',
  '\uFC53': '\u0647\u0649',
  '\uFC54': '\u0647\u064A',
  '\uFC55': '\u064A\u062C',
  '\uFC56': '\u064A\u062D',
  '\uFC57': '\u064A\u062E',
  '\uFC58': '\u064A\u0645',
  '\uFC59': '\u064A\u0649',
  '\uFC5A': '\u064A\u064A',
  '\uFC5B': '\u0630\u0670',
  '\uFC5C': '\u0631\u0670',
  '\uFC5D': '\u0649\u0670',
  '\uFC5E': '\u0020\u064C\u0651',
  '\uFC5F': '\u0020\u064D\u0651',
  '\uFC60': '\u0020\u064E\u0651',
  '\uFC61': '\u0020\u064F\u0651',
  '\uFC62': '\u0020\u0650\u0651',
  '\uFC63': '\u0020\u0651\u0670',
  '\uFC64': '\u0626\u0631',
  '\uFC65': '\u0626\u0632',
  '\uFC66': '\u0626\u0645',
  '\uFC67': '\u0626\u0646',
  '\uFC68': '\u0626\u0649',
  '\uFC69': '\u0626\u064A',
  '\uFC6A': '\u0628\u0631',
  '\uFC6B': '\u0628\u0632',
  '\uFC6C': '\u0628\u0645',
  '\uFC6D': '\u0628\u0646',
  '\uFC6E': '\u0628\u0649',
  '\uFC6F': '\u0628\u064A',
  '\uFC70': '\u062A\u0631',
  '\uFC71': '\u062A\u0632',
  '\uFC72': '\u062A\u0645',
  '\uFC73': '\u062A\u0646',
  '\uFC74': '\u062A\u0649',
  '\uFC75': '\u062A\u064A',
  '\uFC76': '\u062B\u0631',
  '\uFC77': '\u062B\u0632',
  '\uFC78': '\u062B\u0645',
  '\uFC79': '\u062B\u0646',
  '\uFC7A': '\u062B\u0649',
  '\uFC7B': '\u062B\u064A',
  '\uFC7C': '\u0641\u0649',
  '\uFC7D': '\u0641\u064A',
  '\uFC7E': '\u0642\u0649',
  '\uFC7F': '\u0642\u064A',
  '\uFC80': '\u0643\u0627',
  '\uFC81': '\u0643\u0644',
  '\uFC82': '\u0643\u0645',
  '\uFC83': '\u0643\u0649',
  '\uFC84': '\u0643\u064A',
  '\uFC85': '\u0644\u0645',
  '\uFC86': '\u0644\u0649',
  '\uFC87': '\u0644\u064A',
  '\uFC88': '\u0645\u0627',
  '\uFC89': '\u0645\u0645',
  '\uFC8A': '\u0646\u0631',
  '\uFC8B': '\u0646\u0632',
  '\uFC8C': '\u0646\u0645',
  '\uFC8D': '\u0646\u0646',
  '\uFC8E': '\u0646\u0649',
  '\uFC8F': '\u0646\u064A',
  '\uFC90': '\u0649\u0670',
  '\uFC91': '\u064A\u0631',
  '\uFC92': '\u064A\u0632',
  '\uFC93': '\u064A\u0645',
  '\uFC94': '\u064A\u0646',
  '\uFC95': '\u064A\u0649',
  '\uFC96': '\u064A\u064A',
  '\uFC97': '\u0626\u062C',
  '\uFC98': '\u0626\u062D',
  '\uFC99': '\u0626\u062E',
  '\uFC9A': '\u0626\u0645',
  '\uFC9B': '\u0626\u0647',
  '\uFC9C': '\u0628\u062C',
  '\uFC9D': '\u0628\u062D',
  '\uFC9E': '\u0628\u062E',
  '\uFC9F': '\u0628\u0645',
  '\uFCA0': '\u0628\u0647',
  '\uFCA1': '\u062A\u062C',
  '\uFCA2': '\u062A\u062D',
  '\uFCA3': '\u062A\u062E',
  '\uFCA4': '\u062A\u0645',
  '\uFCA5': '\u062A\u0647',
  '\uFCA6': '\u062B\u0645',
  '\uFCA7': '\u062C\u062D',
  '\uFCA8': '\u062C\u0645',
  '\uFCA9': '\u062D\u062C',
  '\uFCAA': '\u062D\u0645',
  '\uFCAB': '\u062E\u062C',
  '\uFCAC': '\u062E\u0645',
  '\uFCAD': '\u0633\u062C',
  '\uFCAE': '\u0633\u062D',
  '\uFCAF': '\u0633\u062E',
  '\uFCB0': '\u0633\u0645',
  '\uFCB1': '\u0635\u062D',
  '\uFCB2': '\u0635\u062E',
  '\uFCB3': '\u0635\u0645',
  '\uFCB4': '\u0636\u062C',
  '\uFCB5': '\u0636\u062D',
  '\uFCB6': '\u0636\u062E',
  '\uFCB7': '\u0636\u0645',
  '\uFCB8': '\u0637\u062D',
  '\uFCB9': '\u0638\u0645',
  '\uFCBA': '\u0639\u062C',
  '\uFCBB': '\u0639\u0645',
  '\uFCBC': '\u063A\u062C',
  '\uFCBD': '\u063A\u0645',
  '\uFCBE': '\u0641\u062C',
  '\uFCBF': '\u0641\u062D',
  '\uFCC0': '\u0641\u062E',
  '\uFCC1': '\u0641\u0645',
  '\uFCC2': '\u0642\u062D',
  '\uFCC3': '\u0642\u0645',
  '\uFCC4': '\u0643\u062C',
  '\uFCC5': '\u0643\u062D',
  '\uFCC6': '\u0643\u062E',
  '\uFCC7': '\u0643\u0644',
  '\uFCC8': '\u0643\u0645',
  '\uFCC9': '\u0644\u062C',
  '\uFCCA': '\u0644\u062D',
  '\uFCCB': '\u0644\u062E',
  '\uFCCC': '\u0644\u0645',
  '\uFCCD': '\u0644\u0647',
  '\uFCCE': '\u0645\u062C',
  '\uFCCF': '\u0645\u062D',
  '\uFCD0': '\u0645\u062E',
  '\uFCD1': '\u0645\u0645',
  '\uFCD2': '\u0646\u062C',
  '\uFCD3': '\u0646\u062D',
  '\uFCD4': '\u0646\u062E',
  '\uFCD5': '\u0646\u0645',
  '\uFCD6': '\u0646\u0647',
  '\uFCD7': '\u0647\u062C',
  '\uFCD8': '\u0647\u0645',
  '\uFCD9': '\u0647\u0670',
  '\uFCDA': '\u064A\u062C',
  '\uFCDB': '\u064A\u062D',
  '\uFCDC': '\u064A\u062E',
  '\uFCDD': '\u064A\u0645',
  '\uFCDE': '\u064A\u0647',
  '\uFCDF': '\u0626\u0645',
  '\uFCE0': '\u0626\u0647',
  '\uFCE1': '\u0628\u0645',
  '\uFCE2': '\u0628\u0647',
  '\uFCE3': '\u062A\u0645',
  '\uFCE4': '\u062A\u0647',
  '\uFCE5': '\u062B\u0645',
  '\uFCE6': '\u062B\u0647',
  '\uFCE7': '\u0633\u0645',
  '\uFCE8': '\u0633\u0647',
  '\uFCE9': '\u0634\u0645',
  '\uFCEA': '\u0634\u0647',
  '\uFCEB': '\u0643\u0644',
  '\uFCEC': '\u0643\u0645',
  '\uFCED': '\u0644\u0645',
  '\uFCEE': '\u0646\u0645',
  '\uFCEF': '\u0646\u0647',
  '\uFCF0': '\u064A\u0645',
  '\uFCF1': '\u064A\u0647',
  '\uFCF2': '\u0640\u064E\u0651',
  '\uFCF3': '\u0640\u064F\u0651',
  '\uFCF4': '\u0640\u0650\u0651',
  '\uFCF5': '\u0637\u0649',
  '\uFCF6': '\u0637\u064A',
  '\uFCF7': '\u0639\u0649',
  '\uFCF8': '\u0639\u064A',
  '\uFCF9': '\u063A\u0649',
  '\uFCFA': '\u063A\u064A',
  '\uFCFB': '\u0633\u0649',
  '\uFCFC': '\u0633\u064A',
  '\uFCFD': '\u0634\u0649',
  '\uFCFE': '\u0634\u064A',
  '\uFCFF': '\u062D\u0649',
  '\uFD00': '\u062D\u064A',
  '\uFD01': '\u062C\u0649',
  '\uFD02': '\u062C\u064A',
  '\uFD03': '\u062E\u0649',
  '\uFD04': '\u062E\u064A',
  '\uFD05': '\u0635\u0649',
  '\uFD06': '\u0635\u064A',
  '\uFD07': '\u0636\u0649',
  '\uFD08': '\u0636\u064A',
  '\uFD09': '\u0634\u062C',
  '\uFD0A': '\u0634\u062D',
  '\uFD0B': '\u0634\u062E',
  '\uFD0C': '\u0634\u0645',
  '\uFD0D': '\u0634\u0631',
  '\uFD0E': '\u0633\u0631',
  '\uFD0F': '\u0635\u0631',
  '\uFD10': '\u0636\u0631',
  '\uFD11': '\u0637\u0649',
  '\uFD12': '\u0637\u064A',
  '\uFD13': '\u0639\u0649',
  '\uFD14': '\u0639\u064A',
  '\uFD15': '\u063A\u0649',
  '\uFD16': '\u063A\u064A',
  '\uFD17': '\u0633\u0649',
  '\uFD18': '\u0633\u064A',
  '\uFD19': '\u0634\u0649',
  '\uFD1A': '\u0634\u064A',
  '\uFD1B': '\u062D\u0649',
  '\uFD1C': '\u062D\u064A',
  '\uFD1D': '\u062C\u0649',
  '\uFD1E': '\u062C\u064A',
  '\uFD1F': '\u062E\u0649',
  '\uFD20': '\u062E\u064A',
  '\uFD21': '\u0635\u0649',
  '\uFD22': '\u0635\u064A',
  '\uFD23': '\u0636\u0649',
  '\uFD24': '\u0636\u064A',
  '\uFD25': '\u0634\u062C',
  '\uFD26': '\u0634\u062D',
  '\uFD27': '\u0634\u062E',
  '\uFD28': '\u0634\u0645',
  '\uFD29': '\u0634\u0631',
  '\uFD2A': '\u0633\u0631',
  '\uFD2B': '\u0635\u0631',
  '\uFD2C': '\u0636\u0631',
  '\uFD2D': '\u0634\u062C',
  '\uFD2E': '\u0634\u062D',
  '\uFD2F': '\u0634\u062E',
  '\uFD30': '\u0634\u0645',
  '\uFD31': '\u0633\u0647',
  '\uFD32': '\u0634\u0647',
  '\uFD33': '\u0637\u0645',
  '\uFD34': '\u0633\u062C',
  '\uFD35': '\u0633\u062D',
  '\uFD36': '\u0633\u062E',
  '\uFD37': '\u0634\u062C',
  '\uFD38': '\u0634\u062D',
  '\uFD39': '\u0634\u062E',
  '\uFD3A': '\u0637\u0645',
  '\uFD3B': '\u0638\u0645',
  '\uFD3C': '\u0627\u064B',
  '\uFD3D': '\u0627\u064B',
  '\uFD50': '\u062A\u062C\u0645',
  '\uFD51': '\u062A\u062D\u062C',
  '\uFD52': '\u062A\u062D\u062C',
  '\uFD53': '\u062A\u062D\u0645',
  '\uFD54': '\u062A\u062E\u0645',
  '\uFD55': '\u062A\u0645\u062C',
  '\uFD56': '\u062A\u0645\u062D',
  '\uFD57': '\u062A\u0645\u062E',
  '\uFD58': '\u062C\u0645\u062D',
  '\uFD59': '\u062C\u0645\u062D',
  '\uFD5A': '\u062D\u0645\u064A',
  '\uFD5B': '\u062D\u0645\u0649',
  '\uFD5C': '\u0633\u062D\u062C',
  '\uFD5D': '\u0633\u062C\u062D',
  '\uFD5E': '\u0633\u062C\u0649',
  '\uFD5F': '\u0633\u0645\u062D',
  '\uFD60': '\u0633\u0645\u062D',
  '\uFD61': '\u0633\u0645\u062C',
  '\uFD62': '\u0633\u0645\u0645',
  '\uFD63': '\u0633\u0645\u0645',
  '\uFD64': '\u0635\u062D\u062D',
  '\uFD65': '\u0635\u062D\u062D',
  '\uFD66': '\u0635\u0645\u0645',
  '\uFD67': '\u0634\u062D\u0645',
  '\uFD68': '\u0634\u062D\u0645',
  '\uFD69': '\u0634\u062C\u064A',
  '\uFD6A': '\u0634\u0645\u062E',
  '\uFD6B': '\u0634\u0645\u062E',
  '\uFD6C': '\u0634\u0645\u0645',
  '\uFD6D': '\u0634\u0645\u0645',
  '\uFD6E': '\u0636\u062D\u0649',
  '\uFD6F': '\u0636\u062E\u0645',
  '\uFD70': '\u0636\u062E\u0645',
  '\uFD71': '\u0637\u0645\u062D',
  '\uFD72': '\u0637\u0645\u062D',
  '\uFD73': '\u0637\u0645\u0645',
  '\uFD74': '\u0637\u0645\u064A',
  '\uFD75': '\u0639\u062C\u0645',
  '\uFD76': '\u0639\u0645\u0645',
  '\uFD77': '\u0639\u0645\u0645',
  '\uFD78': '\u0639\u0645\u0649',
  '\uFD79': '\u063A\u0645\u0645',
  '\uFD7A': '\u063A\u0645\u064A',
  '\uFD7B': '\u063A\u0645\u0649',
  '\uFD7C': '\u0641\u062E\u0645',
  '\uFD7D': '\u0641\u062E\u0645',
  '\uFD7E': '\u0642\u0645\u062D',
  '\uFD7F': '\u0642\u0645\u0645',
  '\uFD80': '\u0644\u062D\u0645',
  '\uFD81': '\u0644\u062D\u064A',
  '\uFD82': '\u0644\u062D\u0649',
  '\uFD83': '\u0644\u062C\u062C',
  '\uFD84': '\u0644\u062C\u062C',
  '\uFD85': '\u0644\u062E\u0645',
  '\uFD86': '\u0644\u062E\u0645',
  '\uFD87': '\u0644\u0645\u062D',
  '\uFD88': '\u0644\u0645\u062D',
  '\uFD89': '\u0645\u062D\u062C',
  '\uFD8A': '\u0645\u062D\u0645',
  '\uFD8B': '\u0645\u062D\u064A',
  '\uFD8C': '\u0645\u062C\u062D',
  '\uFD8D': '\u0645\u062C\u0645',
  '\uFD8E': '\u0645\u062E\u062C',
  '\uFD8F': '\u0645\u062E\u0645',
  '\uFD92': '\u0645\u062C\u062E',
  '\uFD93': '\u0647\u0645\u062C',
  '\uFD94': '\u0647\u0645\u0645',
  '\uFD95': '\u0646\u062D\u0645',
  '\uFD96': '\u0646\u062D\u0649',
  '\uFD97': '\u0646\u062C\u0645',
  '\uFD98': '\u0646\u062C\u0645',
  '\uFD99': '\u0646\u062C\u0649',
  '\uFD9A': '\u0646\u0645\u064A',
  '\uFD9B': '\u0646\u0645\u0649',
  '\uFD9C': '\u064A\u0645\u0645',
  '\uFD9D': '\u064A\u0645\u0645',
  '\uFD9E': '\u0628\u062E\u064A',
  '\uFD9F': '\u062A\u062C\u064A',
  '\uFDA0': '\u062A\u062C\u0649',
  '\uFDA1': '\u062A\u062E\u064A',
  '\uFDA2': '\u062A\u062E\u0649',
  '\uFDA3': '\u062A\u0645\u064A',
  '\uFDA4': '\u062A\u0645\u0649',
  '\uFDA5': '\u062C\u0645\u064A',
  '\uFDA6': '\u062C\u062D\u0649',
  '\uFDA7': '\u062C\u0645\u0649',
  '\uFDA8': '\u0633\u062E\u0649',
  '\uFDA9': '\u0635\u062D\u064A',
  '\uFDAA': '\u0634\u062D\u064A',
  '\uFDAB': '\u0636\u062D\u064A',
  '\uFDAC': '\u0644\u062C\u064A',
  '\uFDAD': '\u0644\u0645\u064A',
  '\uFDAE': '\u064A\u062D\u064A',
  '\uFDAF': '\u064A\u062C\u064A',
  '\uFDB0': '\u064A\u0645\u064A',
  '\uFDB1': '\u0645\u0645\u064A',
  '\uFDB2': '\u0642\u0645\u064A',
  '\uFDB3': '\u0646\u062D\u064A',
  '\uFDB4': '\u0642\u0645\u062D',
  '\uFDB5': '\u0644\u062D\u0645',
  '\uFDB6': '\u0639\u0645\u064A',
  '\uFDB7': '\u0643\u0645\u064A',
  '\uFDB8': '\u0646\u062C\u062D',
  '\uFDB9': '\u0645\u062E\u064A',
  '\uFDBA': '\u0644\u062C\u0645',
  '\uFDBB': '\u0643\u0645\u0645',
  '\uFDBC': '\u0644\u062C\u0645',
  '\uFDBD': '\u0646\u062C\u062D',
  '\uFDBE': '\u062C\u062D\u064A',
  '\uFDBF': '\u062D\u062C\u064A',
  '\uFDC0': '\u0645\u062C\u064A',
  '\uFDC1': '\u0641\u0645\u064A',
  '\uFDC2': '\u0628\u062D\u064A',
  '\uFDC3': '\u0643\u0645\u0645',
  '\uFDC4': '\u0639\u062C\u0645',
  '\uFDC5': '\u0635\u0645\u0645',
  '\uFDC6': '\u0633\u062E\u064A',
  '\uFDC7': '\u0646\u062C\u064A',
  '\uFE49': '\u203E',
  '\uFE4A': '\u203E',
  '\uFE4B': '\u203E',
  '\uFE4C': '\u203E',
  '\uFE4D': '\u005F',
  '\uFE4E': '\u005F',
  '\uFE4F': '\u005F',
  '\uFE80': '\u0621',
  '\uFE81': '\u0622',
  '\uFE82': '\u0622',
  '\uFE83': '\u0623',
  '\uFE84': '\u0623',
  '\uFE85': '\u0624',
  '\uFE86': '\u0624',
  '\uFE87': '\u0625',
  '\uFE88': '\u0625',
  '\uFE89': '\u0626',
  '\uFE8A': '\u0626',
  '\uFE8B': '\u0626',
  '\uFE8C': '\u0626',
  '\uFE8D': '\u0627',
  '\uFE8E': '\u0627',
  '\uFE8F': '\u0628',
  '\uFE90': '\u0628',
  '\uFE91': '\u0628',
  '\uFE92': '\u0628',
  '\uFE93': '\u0629',
  '\uFE94': '\u0629',
  '\uFE95': '\u062A',
  '\uFE96': '\u062A',
  '\uFE97': '\u062A',
  '\uFE98': '\u062A',
  '\uFE99': '\u062B',
  '\uFE9A': '\u062B',
  '\uFE9B': '\u062B',
  '\uFE9C': '\u062B',
  '\uFE9D': '\u062C',
  '\uFE9E': '\u062C',
  '\uFE9F': '\u062C',
  '\uFEA0': '\u062C',
  '\uFEA1': '\u062D',
  '\uFEA2': '\u062D',
  '\uFEA3': '\u062D',
  '\uFEA4': '\u062D',
  '\uFEA5': '\u062E',
  '\uFEA6': '\u062E',
  '\uFEA7': '\u062E',
  '\uFEA8': '\u062E',
  '\uFEA9': '\u062F',
  '\uFEAA': '\u062F',
  '\uFEAB': '\u0630',
  '\uFEAC': '\u0630',
  '\uFEAD': '\u0631',
  '\uFEAE': '\u0631',
  '\uFEAF': '\u0632',
  '\uFEB0': '\u0632',
  '\uFEB1': '\u0633',
  '\uFEB2': '\u0633',
  '\uFEB3': '\u0633',
  '\uFEB4': '\u0633',
  '\uFEB5': '\u0634',
  '\uFEB6': '\u0634',
  '\uFEB7': '\u0634',
  '\uFEB8': '\u0634',
  '\uFEB9': '\u0635',
  '\uFEBA': '\u0635',
  '\uFEBB': '\u0635',
  '\uFEBC': '\u0635',
  '\uFEBD': '\u0636',
  '\uFEBE': '\u0636',
  '\uFEBF': '\u0636',
  '\uFEC0': '\u0636',
  '\uFEC1': '\u0637',
  '\uFEC2': '\u0637',
  '\uFEC3': '\u0637',
  '\uFEC4': '\u0637',
  '\uFEC5': '\u0638',
  '\uFEC6': '\u0638',
  '\uFEC7': '\u0638',
  '\uFEC8': '\u0638',
  '\uFEC9': '\u0639',
  '\uFECA': '\u0639',
  '\uFECB': '\u0639',
  '\uFECC': '\u0639',
  '\uFECD': '\u063A',
  '\uFECE': '\u063A',
  '\uFECF': '\u063A',
  '\uFED0': '\u063A',
  '\uFED1': '\u0641',
  '\uFED2': '\u0641',
  '\uFED3': '\u0641',
  '\uFED4': '\u0641',
  '\uFED5': '\u0642',
  '\uFED6': '\u0642',
  '\uFED7': '\u0642',
  '\uFED8': '\u0642',
  '\uFED9': '\u0643',
  '\uFEDA': '\u0643',
  '\uFEDB': '\u0643',
  '\uFEDC': '\u0643',
  '\uFEDD': '\u0644',
  '\uFEDE': '\u0644',
  '\uFEDF': '\u0644',
  '\uFEE0': '\u0644',
  '\uFEE1': '\u0645',
  '\uFEE2': '\u0645',
  '\uFEE3': '\u0645',
  '\uFEE4': '\u0645',
  '\uFEE5': '\u0646',
  '\uFEE6': '\u0646',
  '\uFEE7': '\u0646',
  '\uFEE8': '\u0646',
  '\uFEE9': '\u0647',
  '\uFEEA': '\u0647',
  '\uFEEB': '\u0647',
  '\uFEEC': '\u0647',
  '\uFEED': '\u0648',
  '\uFEEE': '\u0648',
  '\uFEEF': '\u0649',
  '\uFEF0': '\u0649',
  '\uFEF1': '\u064A',
  '\uFEF2': '\u064A',
  '\uFEF3': '\u064A',
  '\uFEF4': '\u064A',
  '\uFEF5': '\u0644\u0622',
  '\uFEF6': '\u0644\u0622',
  '\uFEF7': '\u0644\u0623',
  '\uFEF8': '\u0644\u0623',
  '\uFEF9': '\u0644\u0625',
  '\uFEFA': '\u0644\u0625',
  '\uFEFB': '\u0644\u0627',
  '\uFEFC': '\u0644\u0627'
};

function reverseIfRtl(chars) {
  var charsLength = chars.length;
  //reverse an arabic ligature
  if (charsLength <= 1 || !isRTLRangeFor(chars.charCodeAt(0))) {
    return chars;
  }
  var s = '';
  for (var ii = charsLength - 1; ii >= 0; ii--) {
    s += chars[ii];
  }
  return s;
}

function adjustWidths(properties) {
  if (properties.fontMatrix[0] === FONT_IDENTITY_MATRIX[0]) {
    return;
  }
  // adjusting width to fontMatrix scale
  var scale = 0.001 / properties.fontMatrix[0];
  var glyphsWidths = properties.widths;
  for (var glyph in glyphsWidths) {
    glyphsWidths[glyph] *= scale;
  }
  properties.defaultWidth *= scale;
}

var Glyph = (function GlyphClosure() {
  function Glyph(fontChar, unicode, accent, width, vmetric, operatorListId) {
    this.fontChar = fontChar;
    this.unicode = unicode;
    this.accent = accent;
    this.width = width;
    this.vmetric = vmetric;
    this.operatorListId = operatorListId;
  }

  Glyph.prototype.matchesForCache =
      function(fontChar, unicode, accent, width, vmetric, operatorListId) {
    return this.fontChar === fontChar &&
           this.unicode === unicode &&
           this.accent === accent &&
           this.width === width &&
           this.vmetric === vmetric &&
           this.operatorListId === operatorListId;
  };

  return Glyph;
})();

/**
 * 'Font' is the class the outside world should use, it encapsulate all the font
 * decoding logics whatever type it is (assuming the font type is supported).
 *
 * For example to read a Type1 font and to attach it to the document:
 *   var type1Font = new Font("MyFontName", binaryFile, propertiesObject);
 *   type1Font.bind();
 */
var Font = (function FontClosure() {
  function Font(name, file, properties) {
    var charCode;

    this.name = name;
    this.loadedName = properties.loadedName;
    this.isType3Font = properties.isType3Font;
    this.sizes = [];

    this.glyphCache = {};

    var names = name.split('+');
    names = names.length > 1 ? names[1] : names[0];
    names = names.split(/[-,_]/g)[0];
    this.isSerifFont = !!(properties.flags & FontFlags.Serif);
    this.isSymbolicFont = !!(properties.flags & FontFlags.Symbolic);
    this.isMonospace = !!(properties.flags & FontFlags.FixedPitch);

    var type = properties.type;
    this.type = type;

    this.fallbackName = (this.isMonospace ? 'monospace' :
                         (this.isSerifFont ? 'serif' : 'sans-serif'));

    this.differences = properties.differences;
    this.widths = properties.widths;
    this.defaultWidth = properties.defaultWidth;
    this.composite = properties.composite;
    this.wideChars = properties.wideChars;
    this.cMap = properties.cMap;
    this.ascent = properties.ascent / PDF_GLYPH_SPACE_UNITS;
    this.descent = properties.descent / PDF_GLYPH_SPACE_UNITS;
    this.fontMatrix = properties.fontMatrix;

    var unicode = this.buildToUnicode(properties);
    this.toUnicode = properties.toUnicode = unicode.toUnicode;
    this.isIdentityUnicode = properties.isIdentityUnicode = unicode.isIdentity;

    this.toFontChar = [];

    if (properties.type == 'Type3') {
      for (charCode = 0; charCode < 256; charCode++) {
        this.toFontChar[charCode] = (this.differences[charCode] ||
                                     properties.defaultEncoding[charCode]);
      }
      return;
    }

    this.cidEncoding = properties.cidEncoding;
    this.vertical = properties.vertical;
    if (this.vertical) {
      this.vmetrics = properties.vmetrics;
      this.defaultVMetrics = properties.defaultVMetrics;
    }

    if (!file || file.isEmpty) {
      if (file) {
        // Some bad PDF generators will include empty font files,
        // attempting to recover by assuming that no file exists.
        warn('Font file is empty in "' + name + '" (' + this.loadedName + ')');
      }

      this.missingFile = true;
      // The file data is not specified. Trying to fix the font name
      // to be used with the canvas.font.
      var fontName = name.replace(/[,_]/g, '-');
      var isStandardFont = fontName in stdFontMap;
      fontName = stdFontMap[fontName] || nonStdFontMap[fontName] || fontName;

      this.bold = (fontName.search(/bold/gi) != -1);
      this.italic = ((fontName.search(/oblique/gi) != -1) ||
                     (fontName.search(/italic/gi) != -1));

      // Use 'name' instead of 'fontName' here because the original
      // name ArialBlack for example will be replaced by Helvetica.
      this.black = (name.search(/Black/g) != -1);

      // if at least one width is present, remeasure all chars when exists
      this.remeasure = Object.keys(this.widths).length > 0;
      if (isStandardFont && type === 'CIDFontType2' &&
          properties.cidEncoding.indexOf('Identity-') === 0) {
        // Standard fonts might be embedded as CID font without glyph mapping.
        // Building one based on GlyphMapForStandardFonts.
        var map = [];
        for (var code in GlyphMapForStandardFonts) {
          map[+code] = GlyphMapForStandardFonts[code];
        }
        this.toFontChar = map;
        this.toUnicode = map;
      } else if (/Symbol/i.test(fontName)) {
        var symbols = Encodings.SymbolSetEncoding;
        for (charCode in symbols) {
          var fontChar = GlyphsUnicode[symbols[charCode]];
          if (!fontChar) {
            continue;
          }
          this.toFontChar[charCode] = fontChar;
        }
      } else if (isStandardFont) {
        this.toFontChar = [];
        for (charCode in properties.defaultEncoding) {
          var glyphName = properties.differences[charCode] ||
                          properties.defaultEncoding[charCode];
          this.toFontChar[charCode] = GlyphsUnicode[glyphName];
        }
      } else {
        for (charCode in this.toUnicode) {
          this.toFontChar[charCode] = this.toUnicode[charCode].charCodeAt(0);
        }
      }
      this.loadedName = fontName.split('-')[0];
      this.loading = false;
      return;
    }

    // Some fonts might use wrong font types for Type1C or CIDFontType0C
    var subtype = properties.subtype;
    if (subtype == 'Type1C' && (type != 'Type1' && type != 'MMType1')) {
      // Some TrueType fonts by mistake claim Type1C
      if (isTrueTypeFile(file)) {
        subtype = 'TrueType';
      } else {
        type = 'Type1';
      }
    }
    if (subtype == 'CIDFontType0C' && type != 'CIDFontType0') {
      type = 'CIDFontType0';
    }
    // XXX: Temporarily change the type for open type so we trigger a warning.
    // This should be removed when we add support for open type.
    if (subtype === 'OpenType') {
      type = 'OpenType';
    }

    var data;
    switch (type) {
      case 'Type1':
      case 'CIDFontType0':
        this.mimetype = 'font/opentype';

        var cff = (subtype == 'Type1C' || subtype == 'CIDFontType0C') ?
          new CFFFont(file, properties) : new Type1Font(name, file, properties);

        adjustWidths(properties);

        // Wrap the CFF data inside an OTF font file
        data = this.convert(name, cff, properties);
        break;

      case 'OpenType':
      case 'TrueType':
      case 'CIDFontType2':
        this.mimetype = 'font/opentype';

        // Repair the TrueType file. It is can be damaged in the point of
        // view of the sanitizer
        data = this.checkAndRepair(name, file, properties);
        break;

      default:
        error('Font ' + type + ' is not supported');
        break;
    }

    this.data = data;

    // Transfer some properties again that could change during font conversion
    this.fontMatrix = properties.fontMatrix;
    this.widths = properties.widths;
    this.defaultWidth = properties.defaultWidth;
    this.encoding = properties.baseEncoding;
    this.seacMap = properties.seacMap;

    this.loading = true;
  }

  Font.getFontID = (function () {
    var ID = 1;
    return function Font_getFontID() {
      return String(ID++);
    };
  })();

  function int16(b0, b1) {
    return (b0 << 8) + b1;
  }

  function int32(b0, b1, b2, b3) {
    return (b0 << 24) + (b1 << 16) + (b2 << 8) + b3;
  }

  function getMaxPower2(number) {
    var maxPower = 0;
    var value = number;
    while (value >= 2) {
      value /= 2;
      maxPower++;
    }

    value = 2;
    for (var i = 1; i < maxPower; i++) {
      value *= 2;
    }
    return value;
  }

  function string16(value) {
    return String.fromCharCode((value >> 8) & 0xff, value & 0xff);
  }

  function safeString16(value) {
    // clamp value to the 16-bit int range
    value = (value > 0x7FFF ? 0x7FFF : (value < -0x8000 ? -0x8000 : value));
    return String.fromCharCode((value >> 8) & 0xff, value & 0xff);
  }

  function createOpenTypeHeader(sfnt, file, numTables) {
    // Windows hates the Mac TrueType sfnt version number
    if (sfnt == 'true') {
      sfnt = string32(0x00010000);
    }

    // sfnt version (4 bytes)
    var header = sfnt;

    // numTables (2 bytes)
    header += string16(numTables);

    // searchRange (2 bytes)
    var tablesMaxPower2 = getMaxPower2(numTables);
    var searchRange = tablesMaxPower2 * 16;
    header += string16(searchRange);

    // entrySelector (2 bytes)
    header += string16(Math.log(tablesMaxPower2) / Math.log(2));

    // rangeShift (2 bytes)
    header += string16(numTables * 16 - searchRange);

    file.file += header;
    file.virtualOffset += header.length;
  }

  function createTableEntry(file, tag, data) {
    // offset
    var offset = file.virtualOffset;

    // length
    var length = data.length;

    // Per spec tables must be 4-bytes align so add padding as needed
    while (data.length & 3) {
      data.push(0x00);
    }
    while (file.virtualOffset & 3) {
      file.virtualOffset++;
    }

    // checksum
    var checksum = 0, n = data.length;
    for (var i = 0; i < n; i += 4) {
      checksum = (checksum + int32(data[i], data[i + 1], data[i + 2],
                                   data[i + 3])) | 0;
    }

    var tableEntry = (tag + string32(checksum) +
                      string32(offset) + string32(length));
    file.file += tableEntry;
    file.virtualOffset += data.length;
  }

  function isTrueTypeFile(file) {
    var header = file.peekBytes(4);
    return readUint32(header, 0) === 0x00010000;
  }

  /**
   * Rebuilds the char code to glyph ID map by trying to replace the char codes
   * with their unicode value. It also moves char codes that are in known
   * problematic locations.
   * @return {Object} Two properties:
   * 'toFontChar' - maps original char codes(the value that will be read
   * from commands such as show text) to the char codes that will be used in the
   * font that we build
   * 'charCodeToGlyphId' - maps the new font char codes to glyph ids
   */
  function adjustMapping(charCodeToGlyphId, properties) {
    var toUnicode = properties.toUnicode;
    var isSymbolic = !!(properties.flags & FontFlags.Symbolic);
    var isIdentityUnicode = properties.isIdentityUnicode;
    var newMap = Object.create(null);
    var toFontChar = [];
    var usedFontCharCodes = [];
    var nextAvailableFontCharCode = PRIVATE_USE_OFFSET_START;
    for (var originalCharCode in charCodeToGlyphId) {
      originalCharCode |= 0;
      var glyphId = charCodeToGlyphId[originalCharCode];
      var fontCharCode = originalCharCode;
      // First try to map the value to a unicode position if a non identity map
      // was created.
      if (!isIdentityUnicode && originalCharCode in toUnicode) {
        var unicode = toUnicode[fontCharCode];
        // TODO: Try to map ligatures to the correct spot.
        if (unicode.length === 1) {
          fontCharCode = unicode.charCodeAt(0);
        }
      }
      // Try to move control characters, special characters and already mapped
      // characters to the private use area since they will not be drawn by
      // canvas if left in their current position. Also, move characters if the
      // font was symbolic and there is only an identity unicode map since the
      // characters probably aren't in the correct position (fixes an issue
      // with firefox and thuluthfont).
      if ((fontCharCode in usedFontCharCodes ||
           fontCharCode <= 0x1f || // Control chars
           fontCharCode === 0x7F || // Control char
           fontCharCode === 0xAD || // Soft hyphen
           (fontCharCode >= 0x80 && fontCharCode <= 0x9F) || // Control chars
           // Prevent drawing characters in the specials unicode block.
           (fontCharCode >= 0xFFF0 && fontCharCode <= 0xFFFF) ||
           (isSymbolic && isIdentityUnicode)) &&
          nextAvailableFontCharCode <= PRIVATE_USE_OFFSET_END) { // Room left.
        // Loop to try and find a free spot in the private use area.
        do {
          fontCharCode = nextAvailableFontCharCode++;

          if (SKIP_PRIVATE_USE_RANGE_F000_TO_F01F && fontCharCode === 0xF000) {
            fontCharCode = 0xF020;
            nextAvailableFontCharCode = fontCharCode + 1;
          }

        } while (fontCharCode in usedFontCharCodes &&
                 nextAvailableFontCharCode <= PRIVATE_USE_OFFSET_END);
      }

      newMap[fontCharCode] = glyphId;
      toFontChar[originalCharCode] = fontCharCode;
      usedFontCharCodes[fontCharCode] = true;
    }
    return {
      toFontChar: toFontChar,
      charCodeToGlyphId: newMap,
      nextAvailableFontCharCode: nextAvailableFontCharCode
    };
  }

  function getRanges(glyphs) {
    // Array.sort() sorts by characters, not numerically, so convert to an
    // array of characters.
    var codes = [];
    for (var charCode in glyphs) {
      codes.push({ fontCharCode: charCode | 0, glyphId: glyphs[charCode] });
    }
    codes.sort(function fontGetRangesSort(a, b) {
      return a.fontCharCode - b.fontCharCode;
    });

    // Split the sorted codes into ranges.
    var ranges = [];
    var length = codes.length;
    for (var n = 0; n < length; ) {
      var start = codes[n].fontCharCode;
      var codeIndices = [codes[n].glyphId];
      ++n;
      var end = start;
      while (n < length && end + 1 == codes[n].fontCharCode) {
        codeIndices.push(codes[n].glyphId);
        ++end;
        ++n;
        if (end === 0xFFFF) {
          break;
        }
      }
      ranges.push([start, end, codeIndices]);
    }

    return ranges;
  }

  function createCmapTable(glyphs) {
    var ranges = getRanges(glyphs);
    var numTables = ranges[ranges.length - 1][1] > 0xFFFF ? 2 : 1;
    var cmap = '\x00\x00' + // version
               string16(numTables) +  // numTables
               '\x00\x03' + // platformID
               '\x00\x01' + // encodingID
               string32(4 + numTables * 8); // start of the table record

    var i, ii, j, jj;
    for (i = ranges.length - 1; i >= 0; --i) {
      if (ranges[i][0] <= 0xFFFF) { break; }
    }
    var bmpLength = i + 1;

    if (ranges[i][0] < 0xFFFF && ranges[i][1] === 0xFFFF) {
      ranges[i][1] = 0xFFFE;
    }
    var trailingRangesCount = ranges[i][1] < 0xFFFF ? 1 : 0;
    var segCount = bmpLength + trailingRangesCount;
    var segCount2 = segCount * 2;
    var searchRange = getMaxPower2(segCount) * 2;
    var searchEntry = Math.log(segCount) / Math.log(2);
    var rangeShift = 2 * segCount - searchRange;

    // Fill up the 4 parallel arrays describing the segments.
    var startCount = '';
    var endCount = '';
    var idDeltas = '';
    var idRangeOffsets = '';
    var glyphsIds = '';
    var bias = 0;

    var range, start, end, codes;
    for (i = 0, ii = bmpLength; i < ii; i++) {
      range = ranges[i];
      start = range[0];
      end = range[1];
      startCount += string16(start);
      endCount += string16(end);
      codes = range[2];
      var contiguous = true;
      for (j = 1, jj = codes.length; j < jj; ++j) {
        if (codes[j] !== codes[j - 1] + 1) {
          contiguous = false;
          break;
        }
      }
      if (!contiguous) {
        var offset = (segCount - i) * 2 + bias * 2;
        bias += (end - start + 1);

        idDeltas += string16(0);
        idRangeOffsets += string16(offset);

        for (j = 0, jj = codes.length; j < jj; ++j) {
          glyphsIds += string16(codes[j]);
        }
      } else {
        var startCode = codes[0];

        idDeltas += string16((startCode - start) & 0xFFFF);
        idRangeOffsets += string16(0);
      }
    }

    if (trailingRangesCount > 0) {
      endCount += '\xFF\xFF';
      startCount += '\xFF\xFF';
      idDeltas += '\x00\x01';
      idRangeOffsets += '\x00\x00';
    }

    var format314 = '\x00\x00' + // language
                    string16(segCount2) +
                    string16(searchRange) +
                    string16(searchEntry) +
                    string16(rangeShift) +
                    endCount + '\x00\x00' + startCount +
                    idDeltas + idRangeOffsets + glyphsIds;

    var format31012 = '';
    var header31012 = '';
    if (numTables > 1) {
      cmap += '\x00\x03' + // platformID
              '\x00\x0A' + // encodingID
              string32(4 + numTables * 8 +
                       4 + format314.length); // start of the table record
      format31012 = '';
      for (i = 0, ii = ranges.length; i < ii; i++) {
        range = ranges[i];
        start = range[0];
        codes = range[2];
        var code = codes[0];
        for (j = 1, jj = codes.length; j < jj; ++j) {
          if (codes[j] !== codes[j - 1] + 1) {
            end = range[0] + j - 1;
            format31012 += string32(start) + // startCharCode
                           string32(end) + // endCharCode
                           string32(code); // startGlyphID
            start = end + 1;
            code = codes[j];
          }
        }
        format31012 += string32(start) + // startCharCode
                       string32(range[1]) + // endCharCode
                       string32(code); // startGlyphID
      }
      header31012 = '\x00\x0C' + // format
                    '\x00\x00' + // reserved
                    string32(format31012.length + 16) + // length
                    '\x00\x00\x00\x00' + // language
                    string32(format31012.length / 12); // nGroups
    }

    return stringToArray(cmap +
                         '\x00\x04' + // format
                         string16(format314.length + 4) + // length
                         format314 + header31012 + format31012);
  }

  function validateOS2Table(os2) {
    var stream = new Stream(os2.data);
    var version = stream.getUint16();
    // TODO verify all OS/2 tables fields, but currently we validate only those
    // that give us issues
    stream.getBytes(60); // skipping type, misc sizes, panose, unicode ranges
    var selection = stream.getUint16();
    if (version < 4 && (selection & 0x0300)) {
      return false;
    }
    var firstChar = stream.getUint16();
    var lastChar = stream.getUint16();
    if (firstChar > lastChar) {
      return false;
    }
    stream.getBytes(6); // skipping sTypoAscender/Descender/LineGap
    var usWinAscent = stream.getUint16();
    if (usWinAscent === 0) { // makes font unreadable by windows
      return false;
    }

    // OS/2 appears to be valid, resetting some fields
    os2.data[8] = os2.data[9] = 0; // IE rejects fonts if fsType != 0
    return true;
  }

  function createOS2Table(properties, charstrings, override) {
    override = override || {
      unitsPerEm: 0,
      yMax: 0,
      yMin: 0,
      ascent: 0,
      descent: 0
    };

    var ulUnicodeRange1 = 0;
    var ulUnicodeRange2 = 0;
    var ulUnicodeRange3 = 0;
    var ulUnicodeRange4 = 0;

    var firstCharIndex = null;
    var lastCharIndex = 0;

    if (charstrings) {
      for (var code in charstrings) {
        code |= 0;
        if (firstCharIndex > code || !firstCharIndex) {
          firstCharIndex = code;
        }
        if (lastCharIndex < code) {
          lastCharIndex = code;
        }

        var position = getUnicodeRangeFor(code);
        if (position < 32) {
          ulUnicodeRange1 |= 1 << position;
        } else if (position < 64) {
          ulUnicodeRange2 |= 1 << position - 32;
        } else if (position < 96) {
          ulUnicodeRange3 |= 1 << position - 64;
        } else if (position < 123) {
          ulUnicodeRange4 |= 1 << position - 96;
        } else {
          error('Unicode ranges Bits > 123 are reserved for internal usage');
        }
      }
    } else {
      // TODO
      firstCharIndex = 0;
      lastCharIndex = 255;
    }

    var bbox = properties.bbox || [0, 0, 0, 0];
    var unitsPerEm = (override.unitsPerEm ||
                      1 / (properties.fontMatrix || FONT_IDENTITY_MATRIX)[0]);

    // if the font units differ to the PDF glyph space units
    // then scale up the values
    var scale = (properties.ascentScaled ? 1.0 :
                 unitsPerEm / PDF_GLYPH_SPACE_UNITS);

    var typoAscent = (override.ascent ||
                      Math.round(scale * (properties.ascent || bbox[3])));
    var typoDescent = (override.descent ||
                       Math.round(scale * (properties.descent || bbox[1])));
    if (typoDescent > 0 && properties.descent > 0 && bbox[1] < 0) {
      typoDescent = -typoDescent; // fixing incorrect descent
    }
    var winAscent = override.yMax || typoAscent;
    var winDescent = -override.yMin || -typoDescent;

    return '\x00\x03' + // version
           '\x02\x24' + // xAvgCharWidth
           '\x01\xF4' + // usWeightClass
           '\x00\x05' + // usWidthClass
           '\x00\x00' + // fstype (0 to let the font loads via font-face on IE)
           '\x02\x8A' + // ySubscriptXSize
           '\x02\xBB' + // ySubscriptYSize
           '\x00\x00' + // ySubscriptXOffset
           '\x00\x8C' + // ySubscriptYOffset
           '\x02\x8A' + // ySuperScriptXSize
           '\x02\xBB' + // ySuperScriptYSize
           '\x00\x00' + // ySuperScriptXOffset
           '\x01\xDF' + // ySuperScriptYOffset
           '\x00\x31' + // yStrikeOutSize
           '\x01\x02' + // yStrikeOutPosition
           '\x00\x00' + // sFamilyClass
           '\x00\x00\x06' +
           String.fromCharCode(properties.fixedPitch ? 0x09 : 0x00) +
           '\x00\x00\x00\x00\x00\x00' + // Panose
           string32(ulUnicodeRange1) + // ulUnicodeRange1 (Bits 0-31)
           string32(ulUnicodeRange2) + // ulUnicodeRange2 (Bits 32-63)
           string32(ulUnicodeRange3) + // ulUnicodeRange3 (Bits 64-95)
           string32(ulUnicodeRange4) + // ulUnicodeRange4 (Bits 96-127)
           '\x2A\x32\x31\x2A' + // achVendID
           string16(properties.italicAngle ? 1 : 0) + // fsSelection
           string16(firstCharIndex ||
                    properties.firstChar) + // usFirstCharIndex
           string16(lastCharIndex || properties.lastChar) +  // usLastCharIndex
           string16(typoAscent) + // sTypoAscender
           string16(typoDescent) + // sTypoDescender
           '\x00\x64' + // sTypoLineGap (7%-10% of the unitsPerEM value)
           string16(winAscent) + // usWinAscent
           string16(winDescent) + // usWinDescent
           '\x00\x00\x00\x00' + // ulCodePageRange1 (Bits 0-31)
           '\x00\x00\x00\x00' + // ulCodePageRange2 (Bits 32-63)
           string16(properties.xHeight) + // sxHeight
           string16(properties.capHeight) + // sCapHeight
           string16(0) + // usDefaultChar
           string16(firstCharIndex || properties.firstChar) + // usBreakChar
           '\x00\x03';  // usMaxContext
  }

  function createPostTable(properties) {
    var angle = Math.floor(properties.italicAngle * (Math.pow(2, 16)));
    return ('\x00\x03\x00\x00' + // Version number
            string32(angle) + // italicAngle
            '\x00\x00' + // underlinePosition
            '\x00\x00' + // underlineThickness
            string32(properties.fixedPitch) + // isFixedPitch
            '\x00\x00\x00\x00' + // minMemType42
            '\x00\x00\x00\x00' + // maxMemType42
            '\x00\x00\x00\x00' + // minMemType1
            '\x00\x00\x00\x00');  // maxMemType1
  }

  function createNameTable(name, proto) {
    if (!proto) {
      proto = [[], []]; // no strings and unicode strings
    }

    var strings = [
      proto[0][0] || 'Original licence',  // 0.Copyright
      proto[0][1] || name,                // 1.Font family
      proto[0][2] || 'Unknown',           // 2.Font subfamily (font weight)
      proto[0][3] || 'uniqueID',          // 3.Unique ID
      proto[0][4] || name,                // 4.Full font name
      proto[0][5] || 'Version 0.11',      // 5.Version
      proto[0][6] || '',                  // 6.Postscript name
      proto[0][7] || 'Unknown',           // 7.Trademark
      proto[0][8] || 'Unknown',           // 8.Manufacturer
      proto[0][9] || 'Unknown'            // 9.Designer
    ];

    // Mac want 1-byte per character strings while Windows want
    // 2-bytes per character, so duplicate the names table
    var stringsUnicode = [];
    var i, ii, j, jj, str;
    for (i = 0, ii = strings.length; i < ii; i++) {
      str = proto[1][i] || strings[i];

      var strBufUnicode = [];
      for (j = 0, jj = str.length; j < jj; j++) {
        strBufUnicode.push(string16(str.charCodeAt(j)));
      }
      stringsUnicode.push(strBufUnicode.join(''));
    }

    var names = [strings, stringsUnicode];
    var platforms = ['\x00\x01', '\x00\x03'];
    var encodings = ['\x00\x00', '\x00\x01'];
    var languages = ['\x00\x00', '\x04\x09'];

    var namesRecordCount = strings.length * platforms.length;
    var nameTable =
      '\x00\x00' +                           // format
      string16(namesRecordCount) +           // Number of names Record
      string16(namesRecordCount * 12 + 6);   // Storage

    // Build the name records field
    var strOffset = 0;
    for (i = 0, ii = platforms.length; i < ii; i++) {
      var strs = names[i];
      for (j = 0, jj = strs.length; j < jj; j++) {
        str = strs[j];
        var nameRecord =
          platforms[i] + // platform ID
          encodings[i] + // encoding ID
          languages[i] + // language ID
          string16(j) + // name ID
          string16(str.length) +
          string16(strOffset);
        nameTable += nameRecord;
        strOffset += str.length;
      }
    }

    nameTable += strings.join('') + stringsUnicode.join('');
    return nameTable;
  }

  Font.prototype = {
    name: null,
    font: null,
    mimetype: null,
    encoding: null,
    get renderer() {
      var renderer = FontRendererFactory.create(this);
      return shadow(this, 'renderer', renderer);
    },

    exportData: function Font_exportData() {
      var data = {};
      for (var i in this) {
        if (this.hasOwnProperty(i)) {
          data[i] = this[i];
        }
      }
      return data;
    },

    checkAndRepair: function Font_checkAndRepair(name, font, properties) {
      function readTableEntry(file) {
        var tag = bytesToString(file.getBytes(4));

        var checksum = file.getInt32();
        var offset = file.getInt32() >>> 0;
        var length = file.getInt32() >>> 0;

        // Read the table associated data
        var previousPosition = file.pos;
        file.pos = file.start ? file