/*
 * Copyright © 2009 Benny Bottema (benny@bennybottema.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simplejavamail.api.email;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.jetbrains.annotations.NotNull;

import java.util.Arrays;

/**
 * Possible encoders for email content (text/html/iCalendar). Default is {@link #QUOTED_PRINTABLE}. This list reflects Jakarta Mail's supported encoders as found at
 * {@code StreamProvider.EncoderTypes}
 *
 * @see "StreamProvider.EncoderTypes"
 */
@RequiredArgsConstructor
@Getter
public enum ContentTransferEncoding {

	BASE_64("base64"),
	B("b"),
	Q("q"),
	BINARY("binary"),
	BIT7("7bit"),
	BIT8("8bit"),
	QUOTED_PRINTABLE("quoted-printable"),
	UU("uuencode"),
	X_UU("x-uuencode"),
	X_UUE("x-uue");

	private final String encoder;

	public static ContentTransferEncoding byEncoder(@NotNull final String encoder) {
		return Arrays.stream(values())
				.filter(c -> c.encoder.equalsIgnoreCase(encoder))
				.findFirst()
				.orElseThrow(() -> new IllegalArgumentException("unknown content transfer encoder: " + encoder));
	}

	@Override
	public String toString() {
		return encoder;
	}
}
