/* *##%
 * Copyright (c) 2009 morin. All rights reserved.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/
package org.sharengo.wikitty.jdbc;

import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_BOOLEAN_VALUE;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_DATE_VALUE;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_DELETION_DATE;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_EXTENSION;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_FIELDNAME;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_ID;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_NUMBER_VALUE;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_TEXT_VALUE;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.COL_VERSION;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.JDBC_DRIVER;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_CLEAR_WIKITTY;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_CREATION_WIKITTY_ADMIN;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_CREATION_WIKITTY_DATA;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_DELETE_WIKITTY_ADMIN;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_DELETE_WIKITTY_DATA;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_INSERT_WIKITTY_ADMIN;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_INSERT_WIKITTY_DATA;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_SELECT_NOTDELETED;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_SELECT_WHERE;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_SELECT_WHERE_NOTDELETED;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.QUERY_UPDATE_WIKITTY_ADMIN;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.TABLE_WIKITTY_ADMIN;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.TABLE_WIKITTY_DATA;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.commitJDBCConnection;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.doQuery;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.getConnection;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.getJDBCConnection;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.loadProperties;
import static org.sharengo.wikitty.jdbc.WikittyJDBCUtil.rollbackJDBCConnection;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.FieldType;
import org.sharengo.wikitty.UpdateResponse;
import org.sharengo.wikitty.Wikitty;
import org.sharengo.wikitty.WikittyException;
import org.sharengo.wikitty.WikittyExtension;
import org.sharengo.wikitty.WikittyExtensionStorage;
import org.sharengo.wikitty.WikittyStorage;
import org.sharengo.wikitty.WikittyTransaction;
import org.sharengo.wikitty.WikittyUtil;

/**
 *
 * @author morin
 * @version $Revision: 25 $
 *
 * Last update: $Date: 2010-04-26 15:17:55 +0200 (lun., 26 avril 2010) $
 * by : $Author: echatellier $
 */
public class WikittyStorageJDBC implements WikittyStorage {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static protected Log log = LogFactory.getLog(WikittyStorageJDBC.class);

    /** Properties file */
    protected final Properties conf;

    /** used to parse list field from hbase data. ex: extension.fieldname[11/15] */
    static final private Pattern listFieldPattern =
            Pattern.compile("(.*)\\[(\\d+)/(\\d+)\\]");
    
    protected WikittyExtensionStorage extensionStorage;
    
    public WikittyStorageJDBC(WikittyExtensionStorage extensionStorage) {
        this(extensionStorage, null);
    }
    
    public WikittyStorageJDBC(WikittyExtensionStorage extensionStorage, Properties properties) {
        conf = loadProperties(properties);
        
        this.extensionStorage = extensionStorage;
        //check if JDBC_DRIVER ca be loaded
        try {
            Class.forName(conf.getProperty(JDBC_DRIVER));
        } catch (ClassNotFoundException eee) {
            if (log.isFatalEnabled()) {
                log.fatal("Couldn't find the driver!");
                eee.printStackTrace();
            }
            throw new WikittyException(eee);
        }

        Connection connection = getJDBCConnection(conf);
        try {
            Statement statement = connection.createStatement();
            statement.execute(conf.getProperty(QUERY_CREATION_WIKITTY_ADMIN));
            statement.execute(conf.getProperty(QUERY_CREATION_WIKITTY_DATA));
            commitJDBCConnection(connection);
        } catch(SQLException eee) {
            rollbackJDBCConnection(connection);
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    protected String getColName(FieldType.TYPE type) {
        String result;
        switch(type) {
            case BOOLEAN:
                result = COL_BOOLEAN_VALUE;
                break;
            case DATE:
                result = COL_DATE_VALUE;
                break;
            case NUMERIC:
                result = COL_NUMBER_VALUE;
                break;
            case STRING:
            case WIKITTY:
            default:
                result = COL_TEXT_VALUE;
                break;
        }
        return result;
    }

    @Override
    public UpdateResponse store(WikittyTransaction transaction,
            Collection<Wikitty> wikitties,
            boolean disableAutoVersionIncrement) throws WikittyException {
        Connection connection = getConnection(conf);
        try {
            UpdateResponse result = new UpdateResponse();
            for (Wikitty wikitty : wikitties) {
                String query = String.format(conf.getProperty(QUERY_SELECT_WHERE),
                        COL_VERSION, TABLE_WIKITTY_ADMIN, COL_ID);
                PreparedStatement statement = connection.prepareStatement(query);
                statement.setString(1, wikitty.getId());
                ResultSet versionResultSet = statement.executeQuery();

                String extensionList = "";
                boolean wikittyAlreadyExists = versionResultSet.first();
                String actualVersion = null;
                String requestedVersion = wikitty.getVersion();

                if (wikittyAlreadyExists) {
                    actualVersion = versionResultSet.getString(COL_VERSION);
                }
                
                // compute new version, but not change wikitty
                String newVersion = null;
                if (disableAutoVersionIncrement) {
                    if (actualVersion == null) { //no version in place
                        if (requestedVersion == null) { //no version requested
                            newVersion = WikittyUtil.DEFAULT_VERSION;
                        } else { //version requested
                            newVersion = requestedVersion;
                        }
                    } else { //version in place is not null
                        if (requestedVersion == null) { //no version requested
                            newVersion = WikittyUtil.incrementMajorRevision(actualVersion);
                        } else if (WikittyUtil.versionEquals(actualVersion, requestedVersion)) { //same version
                            // wikitty is not modified, do nothing
                            continue;
                        } else if (WikittyUtil.versionGreaterThan(requestedVersion, actualVersion)) { //requested version is newer
                            newVersion = requestedVersion;
                        } else { //requested version is obsolete
                            throw new WikittyException(String.format(
                                    "Your wikitty '%s' is obsolete (saving: '%s'; existing: '%s')", wikitty.getId(), requestedVersion, actualVersion));
                        }
                    }
                } else {
                    if (WikittyUtil.versionEquals(actualVersion, requestedVersion)) {
                        // wikitty is not modified, do nothing
                        continue;
                    } else if (WikittyUtil.versionGreaterThan(actualVersion, requestedVersion)) {
                        throw new WikittyException(String.format(
                                "Your wikitty '%s' is obsolete", wikitty.getId()));
                    } else {
                        newVersion = WikittyUtil.incrementMajorRevision(actualVersion);
                    }
                }

                if (!wikittyAlreadyExists) {
                    // insert wikitty object
                    String q = conf.getProperty(QUERY_INSERT_WIKITTY_ADMIN);
                    doQuery(connection, q, wikitty.getId(), newVersion, "");
                }
                result.addVersionUpdate(wikitty.getId(), newVersion);

                doQuery(connection, conf.getProperty(QUERY_DELETE_WIKITTY_DATA), wikitty.getId());

                for (WikittyExtension ext : wikitty.getExtensions()) {
                    extensionList += "," + ext.getId();
                    for (String fieldName : ext.getFieldNames()) {
                        FieldType type = ext.getFieldType(fieldName);
                        if (type.isCollection()) {
                            List<Object> list = wikitty.getFieldAsList(ext.getName(),
                                    fieldName, Object.class);
                            if (list != null) {
                                for (int i = 0; i < list.size(); i++) {
                                    Object value = list.get(i);
                                    String colName = getColName(type.getType());
                                    String q = String.format(conf.getProperty(QUERY_INSERT_WIKITTY_DATA), colName);
                                    doQuery(connection, q,
                                            wikitty.getId(),
                                            ext.getName() + "." + fieldName + "[" + i + "/" + list.size() + "]",
                                            value);
                                }
                            } else {
                                if (type.isNotNull()) {
                                    throw new WikittyException(String.format(
                                            "Field %s in extension %s can't be null",
                                            fieldName, ext.getName()));
                                }
                            }
                        } else {
                            Object value = wikitty.getFieldAsObject(ext.getName(), fieldName);
                            if (value != null) {
                                String colName = getColName(type.getType());
                                String q = String.format(conf.getProperty(QUERY_INSERT_WIKITTY_DATA), colName);
                                doQuery(connection, q,
                                        wikitty.getId(),
                                        ext.getName() + "." + fieldName,
                                        value);
                            } else {
                                if (type.isNotNull()) {
                                    throw new WikittyException(String.format("Field %s in extension %s can't be null",
                                            fieldName, ext.getName()));
                                }
                            }
                        }
                    }
                }

                if (extensionList.length() > 0) {
                    extensionList = extensionList.substring(1);
                    
                    String extensionList2 = "";
                    String[] splits = extensionList.split(",");

                    // update extensions to the lastest versions
                    for (String split : splits) {
                        String extName = WikittyExtension.computeName(split);
                        String lastVersion = extensionStorage.getLastVersion(transaction, extName);
                        extensionList2 += "," + WikittyExtension.computeId(extName, lastVersion);
                    }
                    
                    // delete first ','
                    extensionList = extensionList2.substring(1);
                }
                
                // update extensions in wikitty object
                String q = conf.getProperty(QUERY_UPDATE_WIKITTY_ADMIN);
                doQuery(connection, q, newVersion, extensionList, wikitty.getId());

                wikitty.setVersion(newVersion);
                wikitty.clearDirty();
            }

            return result;
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    @Override
    public boolean exists(WikittyTransaction transaction, String id) {
        Connection connection = getConnection(conf);
        try {
            //select the data with the id "id" in the admin table
            String q = String.format(conf.getProperty(QUERY_SELECT_WHERE), COL_ID,
                    TABLE_WIKITTY_ADMIN, COL_ID);
            PreparedStatement statement = connection.prepareStatement(q);
            statement.setString(1, id);
            ResultSet resultSet = statement.executeQuery();
            // return true if the query has a result
            boolean result = resultSet.first();
            return result;
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    @Override
    public boolean isDeleted(WikittyTransaction transaction, String id) {
        Connection connection = getConnection(conf);
        try {
            //select the data with the id "id" in the admin table
            String q = String.format(conf.getProperty(QUERY_SELECT_WHERE),
                    COL_DELETION_DATE, TABLE_WIKITTY_ADMIN, COL_ID);
            PreparedStatement statement = connection.prepareStatement(q);
            statement.setString(1, id);
            ResultSet resultSet = statement.executeQuery();
            boolean result;
            if(resultSet.first()) {
                result = resultSet.getDate(COL_DELETION_DATE) != null;
                return result;
            } else {
                throw new WikittyException(String.format(
                            "Wikitty with id '%s' doesn't exists", id));
            }
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    @Override
    public Wikitty restore(WikittyTransaction transaction,
            String id, String... fqFieldName) throws WikittyException {
        Connection connection = getConnection(conf);
        try {
            //select the data with the id "id" in the admin table
            String q = String.format(conf.getProperty(
                    QUERY_SELECT_WHERE_NOTDELETED), "*", TABLE_WIKITTY_ADMIN, COL_ID);
            PreparedStatement statement = connection.prepareStatement(q);
            statement.setString(1, id);
            ResultSet adminResultSet = statement.executeQuery();

            if (adminResultSet.first()) {
                String version = adminResultSet.getString(COL_VERSION);
                String extensionList = adminResultSet.getString(COL_EXTENSION);
                //select the data with the id "id" in the data table
                String qdata = String.format(conf.getProperty(QUERY_SELECT_WHERE),
                        "*", TABLE_WIKITTY_DATA, COL_ID);
                PreparedStatement sta = connection.prepareStatement(qdata);
                sta.setString(1, id);
                ResultSet dataResultSet = sta.executeQuery();

                Wikitty result = constructWikitty(transaction, id, version, extensionList,
                        dataResultSet, fqFieldName);
                return result;
            } else {
                throw new WikittyException(String.format(
                        "Can't restore wikitty '%s'", id));
            }
        } catch (SQLException eee) {
            throw new WikittyException(String.format(
                    "Can't restore wikitty '%s'", id), eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    @Override
    public UpdateResponse delete(WikittyTransaction transaction, Collection<String> ids) throws WikittyException {
        Connection connection = getConnection(conf);
        try {
            UpdateResponse result = new UpdateResponse();
            Date now = new Date();

            for (String id : ids) {
                // FIXME poussin 20100114 perhaps, it's not necessary to throw exception for that
                // if no exception needed, suppresse problem of tx in tx
                if (!exists(transaction, id)) {
                    throw new WikittyException(String.format(
                            "Wikitty with id '%s' doesn't exists", id));
                } else if (isDeleted(transaction, id)) {
                    throw new WikittyException(String.format(
                            "Wikitty with id '%s' is already deleted", id));
                }
                // addVersionUpdate delete date field
                doQuery(connection, conf.getProperty(QUERY_DELETE_WIKITTY_ADMIN), id);
                result.addDeletionDateUpdate(id, now);

            }

            return result;
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    @Override
    public void scanWikitties(WikittyTransaction transaction, Scanner scanner) {
        Connection connection = getConnection(conf);
        try {
            Statement statement = connection.createStatement();
            
            // get all wikitties
            // fails with QUERY_SELECT
            ResultSet resultSet = statement.executeQuery(
                    String.format(conf.getProperty(QUERY_SELECT_NOTDELETED),
                    COL_ID, TABLE_WIKITTY_ADMIN));
    
            resultSet.beforeFirst();
            while (resultSet.next()) {
                String id = resultSet.getString(COL_ID);
                Wikitty wikitty = restore(transaction, id);
                scanner.scan(wikitty);
            }
        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }

    /**
     * Create Wikitty from h2 tables
     * @param id the id of the wikitty to restore
     * @param version the version of the wikitty to restore
     * @param extensionList the list of the extensions of the wikitty to restore
     * @param resulSet the ResultSet as the result of the selection of the in the data table
     * @param fqFieldName minimum field to restore
     * @return
     */
    protected Wikitty constructWikitty(WikittyTransaction transaction, String id, String version, String extensionList,
            ResultSet resultSet, String... fqFieldName) throws SQLException {
        Set<String> acceptedField = new HashSet<String>(Arrays.asList(fqFieldName));
        Wikitty result = new Wikitty(id);
        result.setVersion(version);
        if (extensionList != null && !"".equals(extensionList)) {
            for (String ext : extensionList.split(",")) {
                String extName = WikittyExtension.computeName(ext);
                String extVersion = WikittyExtension.computeVersion(ext);

                WikittyExtension extension = extensionStorage.restore(transaction, extName, extVersion);
                result.addExtension(extension);
            }
        }

        // load field
        Map<String, Object[]> listFieldMap = new HashMap<String, Object[]>();
        resultSet.beforeFirst();
        while (resultSet.next()) {
            // fqfieldName fully qualified fieldName (extention.fieldname)
            String fqfieldName = resultSet.getString(COL_FIELDNAME);
            if (isAcceptedField(acceptedField, fqfieldName)) {
                FieldType type = result.getFieldType(fqfieldName);
                Object value = null;
                switch (type.getType()) {
                    case BOOLEAN:
                        value = resultSet.getBoolean(COL_BOOLEAN_VALUE);
                        break;
                    case DATE:
                        java.sql.Timestamp timestamp = resultSet.getTimestamp(COL_DATE_VALUE);
                        if(timestamp != null) {
                            value = new java.util.Date(timestamp.getTime());
                        }
                        break;
                    case NUMERIC:
                        value = resultSet.getBigDecimal(COL_NUMBER_VALUE);
                        break;
                    case STRING:
                        value = resultSet.getString(COL_TEXT_VALUE);
                        break;
                    case WIKITTY:
                        value = resultSet.getString(COL_TEXT_VALUE);
                        break;
                    default:
                        value = resultSet.getString(COL_TEXT_VALUE);
                        break;
                }

                if (type.isCollection()) {
                    // for list just stock array of element in map
                    Matcher match = listFieldPattern.matcher(fqfieldName);
                    if (match.find()) {
                        fqfieldName = match.group(1);
                        int index = Integer.parseInt(match.group(2));
                        Object[] array = listFieldMap.get(fqfieldName);
                        if (array == null) {
                            int size = Integer.parseInt(match.group(3));
                            array = new Object[size];
                            listFieldMap.put(fqfieldName, array);
                        }
                        array[index] = value;
                    } else {
                        if(log.isErrorEnabled()) {
                            log.error(String.format(
                                    "Can't read list field correctly '%s'", fqfieldName));
                        }
                    }
                } else {
                    result.setFqField(fqfieldName, value);
                }
            }
        }

        // add fieldList in wikitty
        for (String fieldName : listFieldMap.keySet()) {
            Object[] array = listFieldMap.get(fieldName);
            List list = new ArrayList(Arrays.asList(array));
            result.setFqField(fieldName, list );
        }

        return result;
    }

    /**
     * Test if fqfieldName is in acceptedField
     * @param acceptedField list of all accepted field
     * @param fqfieldName fully qualified field name with potential [n/m] at end
     * @return if fqfieldName without potential [n/m] is in acceptedField or if acceptedField is empty
     */
    protected boolean isAcceptedField(Set<String> acceptedField, String fqfieldName) {
        boolean result = acceptedField.isEmpty();
        if (!result) {
            int crochet = fqfieldName.indexOf("[");
            if (crochet != -1) {
                fqfieldName = fqfieldName.substring(0, crochet);
            }

            result = acceptedField.contains(fqfieldName);
        }
        return result;
    }

    @Override
    public void clear(WikittyTransaction transaction) {
        Connection connection = getConnection(conf);
        try {
            doQuery(connection, conf.getProperty(QUERY_CLEAR_WIKITTY));

        } catch (SQLException eee) {
            throw new WikittyException(eee);
        } finally {
            WikittyJDBCUtil.closeQuietly(connection);
        }
    }
}
