/* *##%
 * Copyright (c) 2009 Sharengo, Guillaume Dufrene, Benjamin POUSSIN.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;


import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 *
 * @author poussin
 * @version $Revision: 1 $
 *
 * Last update: $Date: 2010-04-16 10:29:38 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class PagedResult<T> implements Serializable {

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(PagedResult.class);

    private static final long serialVersionUID = 1L;

    /** indice element in global search result */
    protected int firstIndice;
    /** total number of result if we call the query for all possible result */
    protected int numFound;
    /** query really executed */
    protected String queryString;
    /** list of result in the wanted interval */
    protected List<T> results;
    /** facet asked or null if no facet */
    protected Map<String, List<FacetTopic>> facets;

    public PagedResult(int FirstIndice, int numFound, String queryString,
            Map<String, List<FacetTopic>> facets, List<T> results) {
        this.firstIndice = FirstIndice;
        this.numFound = numFound;
        this.queryString = queryString;
        this.facets = facets;
        this.results = Collections.unmodifiableList(results);
    }

    /**
     * Converte all result to the wanted type and return new PagedResult with
     * this new result list. If some result don't have the right extension (clazz)
     * this extension is automatically added if autoconvert is true. Else
     * an exception is throwed when result without extension is found.
     *
     * When you used autoconvert = false, you have a potentialy probleme when
     * you have modified a BusinessEntity to have new extension and all your
     * wikitty object are not uptodate in database.
     *
     * @param clazz
     * @param autoconvert
     * @return
     */
    public <E extends BusinessEntityWikitty> PagedResult<E> cast(
            WikittyService wikittyService, Class<E> clazz, boolean autoconvert) {
        List<E> castedResult = new ArrayList<E>(results.size());
        E sample = WikittyUtil.newInstance(clazz);
        Collection<WikittyExtension> wantedExtension = sample.getStaticExtensions();
        for (T t : results) {
            if (t == null) {
                castedResult.add(null);
            } else {
                Wikitty w = null;
                if (t instanceof Wikitty) {
                    w = (Wikitty)t;
                } else if (t instanceof BusinessEntityWikitty) {
                    w = ((BusinessEntityWikitty) t).getWikitty();
                } else {
                    throw new WikittyException(String.format(
                            "Illegal object result class '%s' can't convert it to wikitty",
                            t.getClass().getName(), clazz.getName()));
                }
                
                Collection<WikittyExtension> wikittyExtension = w.getExtensions();
                if (autoconvert || wikittyExtension.containsAll(wantedExtension)) {
                    E e = WikittyUtil.newInstance(wikittyService, clazz, (Wikitty) t);
                    castedResult.add(e);
                } else {
                    throw new WikittyException(String.format(
                            "Illegal object result class '%s' can't convert it to '%s'" +
                            "there is no same extension %s != %s",
                            t.getClass().getName(), clazz.getName(),
                            wikittyExtension, wantedExtension));
                }
            }
        }
        PagedResult<E> result = new PagedResult<E>(
                firstIndice, numFound, queryString, facets, castedResult);
        return result;
    }

    public int getFirstIndice() {
        return firstIndice;
    }

    public int getNumFound() {
        return numFound;
    }

    public String getQueryString() {
        return queryString;
    }

    /**
     * Return name of all facet used in query
     * @return
     */
    public Collection<String> getFacetNames() {
        Collection<String> result = facets.keySet();
        return result;
    }

    /**
     * Return all topic for the specified facet
     * @param facetName name of the wanted facet
     * @return
     */
    public List<FacetTopic> getTopic(String facetName) {
        List<FacetTopic> result = facets.get(facetName);
        return result;
    }

    /**
     * Get map represent facets
     * @return
     */
    public Map<String, List<FacetTopic>> getFacets() {
        return facets;
    }

    /**
     * Return the firstIndice element in result or throw an exception if no element
     * available
     * @return
     */
    public T getFirst() {
        T result = get(0);
        return result;
    }

    public T get(int i) {
        T result = results.get(i);
        return result;
    }

    /**
     * Return unmodifiable list of all result
     * @return
     */
    public List<T> getAll() {
        return results;
    }

    /**
     * Return the number of result in this object
     * @return
     */
    public int size() {
        int result = results.size();
        return result;
    }

}
