/* *##%
 * Copyright (c) 2009 Sharengo, Guillaume Dufrene, Benjamin POUSSIN.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *##%*/

package org.sharengo.wikitty;

import java.beans.XMLDecoder;
import java.beans.XMLEncoder;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sharengo.wikitty.search.Restriction;

/**
 * Criteria represent the search on the wikitty with restriction on field and 
 * informations on facet, paggination and sortting.
 *<p>
 * For search multiple extension use ALL_EXTENSIONS as extension name and
 * specify field type.
 * <p>
 * Example : Criteria.ALL_EXTENSIONS + Criteria.SEPARATOR + <fieldName> + Criteria.SEPARATOR + FieldType.TYPE.<fieldType>
 * 
 * @author poussin
 * @version $Revision: 5 $
 *
 * Last update: $Date: 2010-04-16 11:59:15 +0200 (ven., 16 avril 2010) $
 * by : $Author: echatellier $
 */
public class Criteria implements Serializable {

    /** serialVersionUID. */
    private static final long serialVersionUID = 2590223960861630283L;

    /** to use log facility, just put in your code: log.info(\"...\"); */
    static private Log log = LogFactory.getLog(Criteria.class);

    /** Use to not limit result */
    static public int ALL_ELEMENTS = -1;

    /** Search on all extension */
    static public String ALL_EXTENSIONS = "*";

    /** Search on all values */
    static public String ALL_VALUES = "*";

    /** Separator between extension name, field name and type */
    static public String SEPARATOR = ".";

    /**
     * Criteria can have name this help for presentation and is very
     * useful for facet created by criteria
     */
    protected String name;

    /** All restriction on fields */
    private Restriction restriction;

    /** First index to get result */
    private int firstIndex = 0;
    /** Last index to get result */
    private int endIndex = ALL_ELEMENTS;

    /** Facet on criteria */
    private List<Criteria> facetCriteria;
    /** Facet on field */
    private List<String> facetField;

    /** Sort ascending on fields */
    private List<String> sortAscending;
    /** Sort descending on fields */
    private List<String> sortDescending;

    /**
     * Default constuctor
     */
    public Criteria() {
    }

    /**
     * Create named criteria
     * @param name
     */
    public Criteria(String name) {
        this.name = name;
    }

    /**
     * Deserialize xml to criteria
     * @param xml
     * @return criteria
     */
    public static Criteria fromXML(String xml) {
        InputStream inputStream = new ByteArrayInputStream(xml.getBytes());
        XMLDecoder decoder = new XMLDecoder(inputStream);
        Criteria criteria = (Criteria) decoder.readObject();
        return criteria;
    }

    /**
     * Serialize criteria to xml
     * @param criteria
     * @return xml string
     */
    public static String toXML(Criteria criteria) {
        OutputStream outputStream = new ByteArrayOutputStream();
        XMLEncoder encoder = new XMLEncoder(outputStream);
        encoder.writeObject(criteria);
        encoder.close();
        String result = outputStream.toString();
        return result;
    }

    public Restriction getRestriction() {
        return restriction;
    }

    public void setRestriction(Restriction restriction) {
        this.restriction = restriction;
    }
    
    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public int getFirstIndex() {
        return firstIndex;
    }

    public Criteria setFirstIndex(int firstIndex) {
        this.firstIndex = firstIndex;
        return this;
    }
    
    public int getEndIndex() {
        return endIndex;
    }
    
    public Criteria setEndIndex(int endIndex) {
        this.endIndex = endIndex;
        return this;
    }

    public List<Criteria> getFacetCriteria() {
        return facetCriteria;
    }
    
    public Criteria addFacetCriteria(Criteria criteria) {
        if (facetCriteria == null) {
            facetCriteria = new ArrayList<Criteria>();
        }
        facetCriteria.add(criteria);
        return this;
    }
    
    public Criteria setFacetCriteria(Criteria[] facetCriteria) {
        this.facetCriteria = Arrays.asList(facetCriteria);
        return this;
    }

    public List<String> getFacetField() {
        return facetField;
    }
    
    public Criteria addFacetField(String field) {
        if (facetField == null) {
            facetField = new ArrayList<String>();
        }
        facetField.add(field);
        return this;
    }

    public Criteria setFacetField(String[] facetField) {
        this.facetField = Arrays.asList(facetField);
        return this;
    }

    public List<String> getSortAscending() {
        return sortAscending;
    }

    public Criteria addSortAscending(String field) {
        if(sortAscending == null) {
            sortAscending = new ArrayList<String>();
        }
        sortAscending.add(field);
        return this;
    }

    public Criteria setSortAscending(String[] sortAscending) {
        this.sortAscending = Arrays.asList(sortAscending);
        return this;
    }

    public List<String> getSortDescending() {
        return sortDescending;
    }

    public Criteria addSortDescending(String field) {
        if(sortDescending == null) {
            sortDescending = new ArrayList<String>();
        }
        sortDescending.add(field);
        return this;
    }

    public Criteria setSortDescending(String[] sortDescending) {
        this.sortDescending = Arrays.asList(sortDescending);
        return this;
    }

}
