package org.nuiton.eugengo.generator;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.io.Writer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.nuiton.eugene.models.object.ObjectModelClass;
import org.nuiton.eugene.models.object.ObjectModelClassifier;
import org.nuiton.eugene.models.object.ObjectModelDependency;
import org.nuiton.eugene.models.object.ObjectModelOperation;
import org.nuiton.eugengo.EugengoUtils;

public class ServiceAbstractGenerator extends WikengoCommonGenerator {

    private final static Log log = LogFactory.getLog(ServiceAbstractGenerator.class);

    @Override
    public String getFilenameForClassifier(ObjectModelClassifier clazz) {
        String fqn = getQualifiedName(clazz);
        return fqn.replace('.', File.separatorChar) + ".java";
    }

    public static String getQualifiedName(ObjectModelClassifier clazz) {
        return getPackageName(clazz) + "." + getClassName(clazz);
    }

    private static String getClassName(ObjectModelClassifier clazz) {
        return "Abstract" + clazz.getName();
    }

    private static String getPackageName(ObjectModelClassifier clazz) {
        return clazz.getPackageName() + ".impl";
    }

    public void generateFromClassifier(Writer output, ObjectModelClassifier clazz)
            throws IOException {
        if (!EugengoUtils.isService(clazz)) {
            return;
        }

        //Crud specific (look for the crud dto and entity)
        boolean isCrud = EugengoUtils.isCrudService(clazz);
        String crudDtoFqn = null;
        String crudEntityFqn = null;
        if (isCrud) {

            ObjectModelClassifier dto = null;
            ObjectModelClassifier entity = null;
            for (ObjectModelDependency dependency : clazz.getDependencies()) {
                ObjectModelClassifier supplier = dependency.getSupplier();
                if (supplier != null && EugengoUtils.isDto(supplier)) {
                    dto = supplier;
                }
                if (supplier != null && EugengoUtils.isEntity(supplier)) {
                    entity = supplier;
                }
            }
            if (dto == null) {
                log.error("missing dto dependency: " + clazz.getQualifiedName());
                return;
            }
            if (entity == null) {
                log.error("missing entity dependency: " + clazz.getQualifiedName());
                return;
            }

            crudDtoFqn = dto.getQualifiedName();
            crudEntityFqn = entity.getQualifiedName();
        }

        generateCopyright(output);

        String packageName = getPackageName(clazz);
        String name = getClassName(clazz);
        String parentInterface = clazz.getQualifiedName();
output.write("package "+packageName+";\n");
output.write("\n");
output.write("");

        String technicalExceptionClass = "org.sharengo.exceptions.TechnicalException";

        clearImports();
        addImport(technicalExceptionClass);
        if (isCrud) {
            addImport("org.sharengo.orm.DAO");
            addImport("org.sharengo.utils.utils.exceptions.EmptyParameterException");
            addImport("org.sharengo.utils.utils.exceptions.NullParameterException");
            addImport("org.sharengo.utils.utils.services.impl.AbstractCrudSrv");
            addImport(crudDtoFqn);
            addImport(crudEntityFqn);
            addImport(Serializable.class);
            addImport(Log.class);
            addImport(LogFactory.class);
        }
        lookForIocImports(clazz);
        if (clazz instanceof ObjectModelClass) {
            lookForAttributeImports((ObjectModelClass)clazz);
        }
        lookForOperationImports(clazz);
        addImport(parentInterface); //Import the corresponding interface
        generateImports(output, packageName);

        String extension = "implements " + getType(parentInterface);
        if (isCrud) {
            extension = "extends AbstractCrudSrv<" + getType(crudDtoFqn) + ", " + getType(crudEntityFqn) + "> " + extension;
        }
        generateClazzDocumentation(output, clazz);
output.write("public abstract class "+name+" "+extension+" {\n");
output.write("\n");
output.write("");
        if (isCrud) {
output.write("    private final static Log log = LogFactory.getLog("+name+".class);\n");
output.write("\n");
output.write("");
        }

        for (ObjectModelDependency dep : clazz.getDependencies()) {
            ObjectModelClassifier supplier = dep.getSupplier();
            if (supplier != null && EugengoUtils.isService(supplier)) {
                generateIocDependency(output, dep);
            }
        }

        if (clazz instanceof ObjectModelClass) {
            generateAttributesDeclaration(output, ((ObjectModelClass)clazz));
        }

        if (isCrud) {
            String dtoType = getType(crudDtoFqn);
            String entityType = getType(crudEntityFqn);
            String predictedDaoName = EugengoUtils.toLowerCaseFirstLetter(entityType) + "Dao";

output.write("    protected "+dtoType+" convertToDto("+entityType+" entity) {\n");
output.write("        return null;\n");
output.write("    }\n");
output.write("\n");
output.write("    protected void convertToEntity("+dtoType+" dto,\n");
output.write("            "+entityType+" entity) {\n");
output.write("    }\n");
output.write("\n");
output.write("    protected "+entityType+" createEntity() {\n");
output.write("        //return new "+entityType+"();\n");
output.write("        return null;\n");
output.write("    }\n");
output.write("\n");
output.write("    protected DAO<"+entityType+"> getDao() {\n");
output.write("        //return "+predictedDaoName+";\n");
output.write("        return null;\n");
output.write("    }\n");
output.write("\n");
output.write("    protected Serializable getIdFromDto("+dtoType+" dto) {\n");
output.write("        //return dto.getId();\n");
output.write("        return null;\n");
output.write("    }\n");
output.write("\n");
output.write("    protected Log getLog() {\n");
output.write("        return log;\n");
output.write("    }\n");
output.write("\n");
output.write("    protected void validateDto("+dtoType+" dto)\n");
output.write("            throws EmptyParameterException, NullParameterException {\n");
output.write("    }\n");
output.write("\n");
output.write("");

        }

        for (ObjectModelOperation op : clazz.getOperations()) {
            generateOperationHeader(output, op, false, technicalExceptionClass);
            String opName = op.getName();
output.write("        throw new UnsupportedOperationException(\""+name+"#"+opName+"\");\n");
output.write("    }\n");
output.write("\n");
output.write("");
        }
output.write("} //"+name+"\n");
output.write("");

    }

}
