/*
 * Decompiled with CFR 0.152.
 */
package org.openqa.selenium.grid.sessionmap.local;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import java.net.URI;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.logging.Logger;
import org.openqa.selenium.NoSuchSessionException;
import org.openqa.selenium.events.EventBus;
import org.openqa.selenium.grid.config.Config;
import org.openqa.selenium.grid.data.NodeRemovedEvent;
import org.openqa.selenium.grid.data.NodeRestartedEvent;
import org.openqa.selenium.grid.data.Session;
import org.openqa.selenium.grid.data.SessionClosedEvent;
import org.openqa.selenium.grid.data.SessionClosedReason;
import org.openqa.selenium.grid.data.SessionRemovalInfo;
import org.openqa.selenium.grid.log.LoggingOptions;
import org.openqa.selenium.grid.server.EventBusOptions;
import org.openqa.selenium.grid.sessionmap.SessionMap;
import org.openqa.selenium.internal.Require;
import org.openqa.selenium.remote.RemoteTags;
import org.openqa.selenium.remote.SessionId;
import org.openqa.selenium.remote.tracing.AttributeKey;
import org.openqa.selenium.remote.tracing.AttributeMap;
import org.openqa.selenium.remote.tracing.Span;
import org.openqa.selenium.remote.tracing.Tracer;

public class LocalSessionMap
extends SessionMap {
    private static final Logger LOG = Logger.getLogger(LocalSessionMap.class.getName());
    private final EventBus bus;
    private final IndexedSessionMap knownSessions = new IndexedSessionMap();
    private final Cache<SessionId, SessionRemovalInfo> recentlyRemovedSessions = Caffeine.newBuilder().expireAfterWrite(Duration.ofMinutes(60L)).build();

    public LocalSessionMap(Tracer tracer, EventBus bus) {
        super(tracer);
        this.bus = (EventBus)Require.nonNull((String)"Event bus", (Object)bus);
        bus.addListener(SessionClosedEvent.listener(data -> this.removeWithReason(data.getSessionId(), data.getReason())));
        bus.addListener(NodeRemovedEvent.listener(nodeStatus -> this.batchRemoveByUri(nodeStatus.getExternalUri(), SessionClosedReason.NODE_REMOVED)));
        bus.addListener(NodeRestartedEvent.listener(previousNodeStatus -> this.batchRemoveByUri(previousNodeStatus.getExternalUri(), SessionClosedReason.NODE_RESTARTED)));
    }

    public static SessionMap create(Config config) {
        Tracer tracer = new LoggingOptions(config).getTracer();
        EventBus bus = new EventBusOptions(config).getEventBus();
        return new LocalSessionMap(tracer, bus);
    }

    @Override
    public boolean isReady() {
        return this.bus.isReady();
    }

    @Override
    public boolean add(Session session) {
        Require.nonNull((String)"Session", (Object)session);
        SessionId id = session.getId();
        this.knownSessions.put(id, session);
        try (Span span = this.tracer.getCurrentContext().createSpan("local_sessionmap.add");){
            AttributeMap attributeMap = this.tracer.createAttributeMap();
            attributeMap.put(AttributeKey.LOGGER_CLASS.getKey(), this.getClass().getName());
            RemoteTags.SESSION_ID.accept(span, id);
            RemoteTags.SESSION_ID_EVENT.accept(attributeMap, id);
            String sessionAddedMessage = String.format("Added session to local Session Map, Id: %s, Node: %s", id, session.getUri());
            span.addEvent(sessionAddedMessage, attributeMap);
            LOG.info(sessionAddedMessage);
        }
        return true;
    }

    @Override
    public Session get(SessionId id) {
        Require.nonNull((String)"Session ID", (Object)id);
        Session session = this.knownSessions.get(id);
        if (session == null) {
            SessionRemovalInfo removalInfo = (SessionRemovalInfo)this.recentlyRemovedSessions.getIfPresent((Object)id);
            if (removalInfo != null) {
                throw new NoSuchSessionException(String.format("Unable to find session with ID: %s. Session was %s", id, removalInfo));
            }
            throw new NoSuchSessionException("Unable to find session with ID: " + String.valueOf(id));
        }
        return session;
    }

    @Override
    public void remove(SessionId id) {
        this.removeWithReason(id, SessionClosedReason.QUIT_COMMAND);
    }

    private void removeWithReason(SessionId id, SessionClosedReason reason) {
        Require.nonNull((String)"Session ID", (Object)id);
        Require.nonNull((String)"Reason", (Object)((Object)reason));
        Session removedSession = this.knownSessions.remove(id);
        String reasonText = reason.getReasonText();
        if (removedSession != null) {
            this.recentlyRemovedSessions.put((Object)id, (Object)new SessionRemovalInfo(reasonText, removedSession.getUri()));
            LOG.fine(String.format("Tracked removal for session %s with reason: %s", id, reasonText));
        }
        try (Span span = this.tracer.getCurrentContext().createSpan("local_sessionmap.remove");){
            AttributeMap attributeMap = this.tracer.createAttributeMap();
            attributeMap.put(AttributeKey.LOGGER_CLASS.getKey(), this.getClass().getName());
            RemoteTags.SESSION_ID.accept(span, id);
            RemoteTags.SESSION_ID_EVENT.accept(attributeMap, id);
            String sessionDeletedMessage = String.format("Deleted session from local Session Map, Id: %s, Node: %s, Reason: %s", id, removedSession != null ? String.valueOf(removedSession.getUri()) : "unidentified", reasonText);
            span.addEvent(sessionDeletedMessage, attributeMap);
            LOG.info(sessionDeletedMessage);
        }
    }

    private void batchRemoveByUri(URI externalUri, SessionClosedReason closeReason) {
        Set<SessionId> sessionsToRemove = this.knownSessions.getSessionsByUri(externalUri);
        if (sessionsToRemove.isEmpty()) {
            return;
        }
        this.knownSessions.batchRemove(sessionsToRemove);
        for (SessionId sessionId : sessionsToRemove) {
            this.recentlyRemovedSessions.put((Object)sessionId, (Object)new SessionRemovalInfo(closeReason.getReasonText(), externalUri));
        }
        try (Span span = this.tracer.getCurrentContext().createSpan("local_sessionmap.batch_remove");){
            AttributeMap attributeMap = this.tracer.createAttributeMap();
            attributeMap.put(AttributeKey.LOGGER_CLASS.getKey(), this.getClass().getName());
            attributeMap.put("removal.reason", closeReason.getReasonText());
            attributeMap.put("node.uri", externalUri.toString());
            attributeMap.put("sessions.count", (long)sessionsToRemove.size());
            String batchRemoveMessage = String.format("Batch removed %d sessions from local Session Map for Node %s (reason: %s)", new Object[]{sessionsToRemove.size(), externalUri, closeReason});
            span.addEvent(batchRemoveMessage, attributeMap);
            LOG.info(batchRemoveMessage);
        }
    }

    private static class IndexedSessionMap {
        private final ConcurrentMap<SessionId, Session> sessions = new ConcurrentHashMap<SessionId, Session>();
        private final ConcurrentMap<URI, Set<SessionId>> sessionsByUri = new ConcurrentHashMap<URI, Set<SessionId>>();
        private final Object coordinationLock = new Object();

        private IndexedSessionMap() {
        }

        public Session get(SessionId id) {
            return (Session)this.sessions.get(id);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void put(SessionId id, Session session) {
            Object object = this.coordinationLock;
            synchronized (object) {
                URI sessionUri;
                Session previous = this.sessions.put(id, session);
                if (previous != null && previous.getUri() != null) {
                    this.cleanupUriIndex(previous.getUri(), id);
                }
                if ((sessionUri = session.getUri()) != null) {
                    this.sessionsByUri.computeIfAbsent(sessionUri, k -> ConcurrentHashMap.newKeySet()).add(id);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public Session remove(SessionId id) {
            Object object = this.coordinationLock;
            synchronized (object) {
                Session removed = (Session)this.sessions.remove(id);
                if (removed != null && removed.getUri() != null) {
                    this.cleanupUriIndex(removed.getUri(), id);
                }
                return removed;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void batchRemove(Set<SessionId> sessionIds) {
            Object object = this.coordinationLock;
            synchronized (object) {
                HashMap<URI, Set> uriToSessionIds = new HashMap<URI, Set>();
                for (SessionId sessionId : sessionIds) {
                    Session session = (Session)this.sessions.remove(sessionId);
                    if (session == null || session.getUri() == null) continue;
                    uriToSessionIds.computeIfAbsent(session.getUri(), k -> new HashSet()).add(sessionId);
                }
                for (Map.Entry entry : uriToSessionIds.entrySet()) {
                    this.cleanupUriIndex((URI)entry.getKey(), (Set)entry.getValue());
                }
            }
        }

        private void cleanupUriIndex(URI uri, SessionId sessionId) {
            this.sessionsByUri.computeIfPresent(uri, (key, sessionIds) -> {
                sessionIds.remove(sessionId);
                return sessionIds.isEmpty() ? null : sessionIds;
            });
        }

        private void cleanupUriIndex(URI uri, Set<SessionId> sessionIdsToRemove) {
            this.sessionsByUri.computeIfPresent(uri, (key, sessionIds) -> {
                sessionIds.removeAll(sessionIdsToRemove);
                return sessionIds.isEmpty() ? null : sessionIds;
            });
        }

        public Set<SessionId> getSessionsByUri(URI uri) {
            Set result = (Set)this.sessionsByUri.get(uri);
            return result != null && !result.isEmpty() ? Set.copyOf(result) : Set.of();
        }

        public Set<Map.Entry<SessionId, Session>> entrySet() {
            return Collections.unmodifiableSet(this.sessions.entrySet());
        }

        public Collection<Session> values() {
            return Collections.unmodifiableCollection(this.sessions.values());
        }

        public int size() {
            return this.sessions.size();
        }

        public boolean isEmpty() {
            return this.sessions.isEmpty();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void clear() {
            Object object = this.coordinationLock;
            synchronized (object) {
                this.sessions.clear();
                this.sessionsByUri.clear();
            }
        }
    }
}

