package org.planx.xpath.expr;

import java.util.ArrayList;
import java.util.List;
import org.planx.xpath.Context;
import org.planx.xpath.Environment;
import org.planx.xpath.Navigator;
import org.planx.xpath.XPathException;
import org.planx.xpath.object.*;

/**
 * A list of predicates, including the ability to filter a
 * node set through all predicates.
 **/
public class PredicateList {
    private List predicates;

    /**
     * Construct an empty list of predicates.
     **/
    public PredicateList() {
        predicates = new ArrayList();
    }

    public void addPredicate(Expression expr) {
        predicates.add(0, expr);
    }

    /**
     * Evaluates all predicates for each node in the node set
     * and returns the matching nodes. The predicates will be
     * evaluated by setting the context size to the size of the
     * node set and context position to the index of a node in
     * the node set.
     **/
    public XNodeSet evaluate(XNodeSet ctxtSet, Environment env, Navigator nav)
                                                        throws XPathException {

        // If there are no predicates all nodes match.
        // To speed up things return immediately.

        if (predicates.isEmpty()) return ctxtSet;

        // Initialise

        int ctxtSize = ctxtSet.size();
        boolean[] filter = new boolean[ctxtSize];
        for (int i = 0; i < ctxtSize; i++) filter[i] = true;

        Context ctxt = new Context();
        ctxt.setSize(ctxtSize);

        // Loop through predicates

        for (int i = 0, max = predicates.size(); i < max; i++) {

            // Evaluate predicate expression for each node

            Expression expr = (Expression) predicates.get(i);

            for (int j = 0; j < ctxtSize; j++) {
                ctxt.setNode(ctxtSet.get(j));
                ctxt.setPosition(j + 1);
                XObject value = expr.evaluate(ctxt, env, nav);

                // Convert results to boolean and update filter array

                if (value instanceof XNumber) {
                    // Check if number = context position
                    XNumber num = (XNumber) value;
                    filter[j] = filter[j] && (num.intValue() == (j+1));
                } else {
                    // Convert to boolean
                    filter[j] = filter[j] && value.booleanValue();
                }
            }
        }

        // Only retain nodes with all predicates true

        XNodeSet matching = new XNodeSet();
        for (int i = 0; i < ctxtSize; i++) {
            if (filter[i]) {
                matching.add(ctxtSet.get(i));
            }
        }
        return matching;
    }

    public String toString() {
        return predicates.isEmpty() ? "" : predicates.toString();
    }
}
