package org.planx.xpath.expr;

import java.util.ArrayList;
import java.util.Set;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import org.planx.xpath.Context;
import org.planx.xpath.Environment;
import org.planx.xpath.Navigator;
import org.planx.xpath.XPathException;
import org.planx.xpath.expr.axis.Axis;
import org.planx.xpath.expr.axis.AxisIterator;
import org.planx.xpath.object.*;

/**
 * A LocationPath consists of a list of {@link Step} objects
 * and an indication of whether it is an absolute location
 * path or not.
 **/
public class LocationPath extends Expression {
    protected List steps;
    protected boolean isAbsolute;

    public LocationPath() {
        steps = new ArrayList();
    }

    public void appendStep(Step step) {
        steps.add(step);
    }

    public void prependStep(Step step) {
        steps.add(0, step);
    }

    public boolean isAbsolute() {
        return isAbsolute;
    }

    public void setAbsolute(boolean isAbsolute) {
        this.isAbsolute = isAbsolute;
    }

    public XObject evaluate(Context ctxt, Environment env, Navigator nav)
                                                   throws XPathException {
        XNodeSet atomSet = new XNodeSet(1);
        atomSet.add(ctxt.getNode());
        return evaluate(atomSet, env, nav);
    }

    /**
     * Evaluate this location path using each of the nodes in the
     * specified set as context node. The result is the union of
     * all the sets found this way. The results are returned in
     * document order.
     **/
    public XObject evaluate(XNodeSet nodes, Environment env, Navigator nav)
                                                     throws XPathException {
        // If absolute: Make root only node in set

        if (isAbsolute) {
            Object root = nav.getRoot(nodes.get(0));
            nodes = new XNodeSet(1);
            nodes.add(root);
        }

        // Evaluate steps of location path

        Axis axis = null;

        for (int i = 0, numOfSteps = steps.size(); i < numOfSteps; i++) {
            Step step = (Step) steps.get(i);
            axis = step.getAxis();
            NodeTest nodeTest = step.getNodeTest();
            PredicateList predicates = step.getPredicates();

            XNodeSet axisNodes = new XNodeSet();
            Set unique = new HashSet();

            // Find union of nodes matching node test before
            // filtering through predicates

            for (int j = 0, size = nodes.size(); j < size; j++) {
                Object ctxtNode = nodes.get(j);

                // Step through each node on the axis
                // and filter through node test

                int principalNodeType = axis.getPrincipalNodeType();
                AxisIterator axisIt = axis.iterator(ctxtNode, nav);
                while (axisIt.hasNext()) {
                    Object node = axisIt.next();

                    if (nodeTest.matches(node, principalNodeType, nav)
                                                && !unique.contains(node)) {
                        axisNodes.add(node);
                        unique.add(node);
                    }
                }
            }

            // Filter node set through predicates

            nodes = predicates.evaluate(axisNodes, env, nav);

            // Ensure that nodes are in document order

            if (axis.isReverse()) {
                int size = nodes.size();
                XNodeSet reverse = new XNodeSet(size);
                for (int j = 0; j < size; j++) {
                    reverse.add(nodes.get(size-j-1));
                }
                nodes = reverse;
            }
        }
        return nodes;
    }

    public String toString() {
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < steps.size(); i++) {
            if ((i > 0) || isAbsolute) sb.append("/");
            sb.append(steps.get(i).toString());
        }
        return sb.toString();
    }
}
