package org.planx.xmlstore.stores;

import java.io.*;
import java.net.*;
import java.util.List;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.*;
import org.planx.xmlstore.nodes.*;
import org.planx.xmlstore.references.*;
import static org.planx.xmlstore.stores.NetworkProtocol.*;

/**
 * @author Henning Niss
 * @author Thomas Ambus
 */
public class NetworkProxy extends AbstractXMLStore {
    private static Streamer<Node> inlineNodeStreamer = new InlineNodeStreamer();
    private Streamer<Reference> vrefStreamer;
    private Socket socket;
    private DataInputStream in;
    private DataOutputStream out;
    private InetAddress addr;
    private int port;

    public NetworkProxy(String host, int port) throws IOException,
                                             UnknownHostException {
        this(InetAddress.getByName(host),port);
    }

    public NetworkProxy(InetSocketAddress sockaddr) throws IOException {
        this(sockaddr.getAddress(), sockaddr.getPort());
    }

    public NetworkProxy(InetAddress addr, int port) throws IOException {
        this.addr = addr;
        this.port = port;
        socket = new Socket(addr, port);
        in = new DataInputStream(new BufferedInputStream(socket.getInputStream()));
        out = new DataOutputStream(new BufferedOutputStream(socket.getOutputStream()));
        vrefStreamer = Streamers.getPolymorphicStreamer(Reference.class);

        // Protocol hallo handshake: get version

        out.writeByte(REQUEST_VERSION);
        out.flush();
        byte version = in.readByte();
        if (version != PROTOCOL_VERSION)
            throw new NetworkProtocolException("Unknown protocol version");
    }

    protected Node resolvedLoad(Reference vref) throws IOException,
                                         UnknownReferenceException {
        checkClosed();
        out.writeByte(REQUEST_LOAD);
        vrefStreamer.toStream(out, vref);
        out.flush();

        byte status = in.readByte();
        checkErrors(status);
        Node node = inlineNodeStreamer.fromStream(in);
        return node;
    }

    /**
     * The subtree rooted at the specified <code>Node</code> is sent to the
     * remote <code>XMLStore</code> and saved there. A <code>Reference</code>
     * to the root is returned.
     */
    public Reference save(Node node) throws IOException {
        try {
            checkClosed();
            out.writeByte(REQUEST_SAVE);
            inlineNodeStreamer.toStream(out, node);
            out.flush();

            byte status = in.readByte();
            checkErrors(status);
            Reference vref = vrefStreamer.fromStream(in);
            return vref;
        } catch (UnknownReferenceException e) {
            throw new NetworkProtocolException
                ("Save should not cause UnknownReferenceException");
        }
    }

    public void close() throws IOException {
        in.close();
        in = null;
        out.close();
        out = null;
        socket.close();
        socket = null;
    }

    public String toString() {
        return addr.toString()+":"+port;
    }

    protected void checkClosed() {
        if (socket == null) throw new IllegalStateException("XMLStore is closed");
    }

    private void checkErrors(byte status) throws IOException,
                              UnknownReferenceException {
        switch (status) {
        case RESPONSE_OK:
            return;
        case RESPONSE_IO_EXCEPTION:
            throw new IOException("Remote exception");
        case RESPONSE_UNKNOWN_VREF:
            throw new UnknownReferenceException("Remote exception");
        default:
            throw new NetworkProtocolException("Unknown response code "+status);
        }
    }
}
