package org.planx.xmlstore.references;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import org.planx.msd.Discriminator;
import org.planx.msd.Discriminators;
import org.planx.msd.DiscriminatorFactory;
import org.planx.msd.Extractor;
import org.planx.msd.Memory;
import org.planx.msd.array.IntegerArrayDiscriminator;
import org.planx.msd.util.AbstractDiscriminator;
import org.planx.util.Converter;
import org.planx.xmlstore.Attribute;
import org.planx.xmlstore.Node;
import org.planx.xmlstore.Reference;
import org.planx.xmlstore.io.Streamer;
import org.planx.xmlstore.io.Streamers;

/**
 * A <code>RelativeReference</code> consists of a <code>Reference</code>
 * and a path. The path is an integer array specifying, in order from the
 * origin, the indexes of children to visit in order to reach the <code>Node</code>
 * identified by this <code>RelativeReference</code>.
 *
 * @author Thomas Ambus
 */
public class RelativeReference implements Reference, Serializable {
    public static final byte CLASS_ID = 0x03;
    private Reference root;
    private RelativeReference parent;  // if parent != null then path must be set
    private int[] path;                // only used if parent == null
    private int childIndex;            // only used if parent != null

    public RelativeReference(Reference root) {
        this.root = root;
        this.parent = null;
        this.path = new int[0];
    }

    public RelativeReference(Reference root, int[] path) {
        this.root = root;
        this.parent = null;
        this.path = path;
    }

    public RelativeReference(Reference root, int childIndex) {
        this((root instanceof RelativeReference) ?
             (RelativeReference) root : new RelativeReference(root), childIndex);
    }

    public RelativeReference(RelativeReference parent, int childIndex) {
        this.root = parent.root;
        this.parent = parent;
        this.path = null;
        this.childIndex = childIndex;
    }

    public Reference getRoot() {
        return root;
    }

    private int[] internalPath() {
        if (parent == null) return path;
        else return getPath();
    }

    public int[] getPath() {
        return gather(0);
    }

    private int[] gather(int pos) {
        int[] p;
        if (parent != null) {
            p = parent.gather(pos + 1);
            p[p.length - pos - 1] = childIndex;
        } else {
            p = new int[pos + path.length];
            System.arraycopy(path, 0, p, 0, path.length);
        }
        return p;
    }

    /**
     * Returns <code>true</code> if and only if the specified object is a
     * <code>RelativeReference</code> consisting of a root equal to the
     * root of this reference, and a path equal to the path of this reference.
     */
    public boolean equals(Object o) {
        if (o == this) return true;
        if (!(o instanceof RelativeReference)) return false;
        RelativeReference vref = (RelativeReference) o;
        return vref.getRoot().equals(root) &&
               Arrays.equals(internalPath(), vref.getPath());
    }

    public int hashCode() {
        return root.hashCode()^Arrays.hashCode(internalPath());
    }

    public String toString() {
        return root+":"+Arrays.toString(internalPath());
    }

    private void writeObject(java.io.ObjectOutputStream out) throws IOException {
        out.writeObject(root);
        out.writeObject(getPath());
    }

    private void readObject(java.io.ObjectInputStream in) throws IOException,
                                                      ClassNotFoundException {
        root = (Reference) in.readObject();
        path = (int[]) in.readObject();
        parent = null;
    }

    public static Streamer<RelativeReference> getStreamer() {
        final Streamer<Reference> vrefStreamer =
            Streamers.getPolymorphicStreamer(Reference.class);

        return new Streamer<RelativeReference>() {
            public void toStream(DataOutput out, RelativeReference vref)
                        throws IOException {
                vrefStreamer.toStream(out, vref.root);
                Streamers.writePosArray(out, vref.internalPath());
            }
            public RelativeReference fromStream(DataInput in)
                        throws IOException {
                Reference root = vrefStreamer.fromStream(in);
                int[] path = Streamers.readPosArray(in);
                return new RelativeReference(root, path);
            }
        };
    }

    public static Discriminator<RelativeReference> getDiscriminator() {
        return getDiscriminator(DiscriminatorFactory.instance().getMemory());
    }

    public static Discriminator<RelativeReference> getDiscriminator(
                                            Memory memory) {
        return new RelRefDiscriminator(memory);
    }

    private static class RelRefDiscriminator extends
                    AbstractDiscriminator<RelativeReference> {
        private Discriminator<Reference> rootDisc;
        private Discriminator<int[]> pathDisc;

        public RelRefDiscriminator(Memory memory) {
            rootDisc = DiscriminatorFactory.instance().
                        getPolymorphicDiscriminator(Reference.class);
            pathDisc = new IntegerArrayDiscriminator(memory);
        }

        public <U,S> Collection<List<S>> discriminate(
                  List<? extends U> values,
                  final Extractor<U,? extends RelativeReference,S> e) {

            Extractor<U,Reference,U> eRoot =
            new Extractor<U,Reference,U>() {
                public Reference getLabel(U elm) {
                    return e.getLabel(elm).root;
                }
                public U getValue(U elm) {
                    return elm;
                }
            };

            Extractor<U,int[],S> ePath = new Extractor<U,int[],S>() {
                public int[] getLabel(U elm) {
                    RelativeReference vref = e.getLabel(elm);
                    return vref.internalPath();
                }
                public S getValue(U elm) {
                    return e.getValue(elm);
                }
            };
            return Discriminators.discriminate(values, rootDisc,
                                        eRoot, pathDisc, ePath);
        }
    }
}
