package org.planx.xmlstore.nameserver;

import java.io.IOException;
import org.planx.xmlstore.*;
import org.planx.xmlstore.io.PersistentMap;
import org.planx.xmlstore.io.Streamer;
import org.planx.xmlstore.io.Streamers;
import org.planx.xmlstore.references.AbstractReferenceListener;
import org.planx.xmlstore.references.ReferenceListener;
import org.planx.xmlstore.references.Locator;
import org.planx.xmlstore.stores.LocalXMLStore;

/**
 * A <code>NameServer</code> for local use only. A <code>LocalNameServer</code> is
 * associated with a specific <code>XMLStore</code> and will ensure that the
 * <code>XMLStore</code> retains <code>Reference</code>s bound and releases
 * old rebound <code>Reference</code>s.
 *
 * @author Henning Niss
 * @author Thomas Ambus
 */
public class LocalNameServer implements NameServer {
    private PersistentMap<String,Reference> map;
    private LocalXMLStore xmlstore;

    public LocalNameServer(LocalXMLStore xmlstore) throws IOException {
        this.xmlstore = xmlstore;
        String name = xmlstore.toString();
        Streamer<String> s1 = Streamers.stringStreamer();
        Streamer<Reference> s2 = Streamers.getPolymorphicStreamer(Reference.class);
        map = new PersistentMap<String,Reference>(name + "_nameserver.hash", s1, s2);
        xmlstore.addReferenceListener(new LocalNameListener());
    }

    public Reference lookup(String docName) {
        return map.get(docName);
    }

    public void bind(String name, Reference ref) throws IOException,
                                          NameAlreadyBoundException {
        try {
            Reference tmp = lookup(name);
            if (tmp != null && tmp.equals(ref)) return;
            if (tmp != null) throw new NameAlreadyBoundException(name, ref);
            map.put(name, ref);
            xmlstore.retain(ref);
        } catch (UnknownReferenceException e) {
            throw new IOException(e.toString());
        }
    }

    public void rebind(String name, Reference oldr, Reference newr)
                       throws IOException, StaleReferenceException {
        try {
            Reference tmp = lookup(name);
            if (tmp == null || tmp.equals(oldr)) {
                map.put(name, newr);
                xmlstore.retain(newr);
                xmlstore.release(oldr); // old Reference no longer needed
            } else {
                throw new StaleReferenceException(name,oldr,tmp);
            }
        } catch (UnknownReferenceException e) {
            throw new IOException(e.toString());
        }
    }

    /**
     * When a reference is updated, replaces all occurrences of the old reference
     * with the new in map.
     */
    private class LocalNameListener extends AbstractReferenceListener {
        public void referenceMoved(Locator oldLoc, Locator newLoc) {
            try {
                map.replace(oldLoc, newLoc);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        }
    }
}
