package org.planx.xmlstore.io;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.util.Collection;
import java.util.List;
import org.planx.msd.Discriminator;
import org.planx.msd.Discriminators;
import org.planx.msd.DiscriminatorFactory;
import org.planx.msd.Extractor;
import org.planx.msd.Memory;
import org.planx.msd.array.ByteArrayDiscriminator;
import org.planx.msd.lang.PolymorphicDiscriminator;
import org.planx.msd.number.IntegerDiscriminator;
import org.planx.msd.util.AbstractDiscriminator;
import org.planx.xmlstore.Reference;
import org.planx.xmlstore.references.Locator;

/**
 * A <code>LocalLocator</code> consists of a file offset, a length, and a
 * <code>FileSystemIdentifier</code>. The file offset and the
 * <code>FileSystemIdentifier</code> uniquely identifies a
 * <code>LocalLocator</code>. The length is merely used for
 * efficient de-allocation of the space occupied in the <code>FileSystem</code>.
 * <p>
 * TODO: Only path compression is currently performed. Add union-by-rank.
 *
 * @author Thomas Ambus
 */
public class LocalLocator implements Locator, Comparable<LocalLocator> {
    public static final byte CLASS_ID = 0x01;
    private static Streamer<LocalLocator> compactStreamer;
    private static Streamer<LocalLocator> locatorStreamer;
    protected int off;
    protected int len;
    protected FileSystemIdentifier fsi;
    protected LocalLocator eqref = null;

    static {
        compactStreamer = new LocatorStreamer(true);
        locatorStreamer = new LocatorStreamer(false);

        PolymorphicStreamer<Reference> ps =
            Streamers.getPolymorphicStreamer(Reference.class);
        ps.addStreamer(LocalLocator.CLASS_ID, LocalLocator.class,
            locatorStreamer);

        DiscriminatorFactory factory = DiscriminatorFactory.instance();
        PolymorphicDiscriminator<Reference> pd =
            factory.getPolymorphicDiscriminator(Reference.class);
        pd.addDiscriminator(LocalLocator.class,
            LocalLocator.getDiscriminator(factory.getMemory()));
    }

    public LocalLocator(int off, int len, FileSystemIdentifier fsi) {
        if (off < 0) throw new IllegalArgumentException("Negative off");
        this.off = off;
        this.len = len;
        this.fsi = fsi;
    }

    private LocalLocator() {}

    /**
     * Careful with this! It breaks union-find for in-memory locators.
     */
    public LocalLocator clone() {
        LocalLocator l = new LocalLocator();
        l.off = getOff();
        l.len = getLen();
        l.fsi = getFileSystemId();
        return l;
    }

    protected LocalLocator find() {
        if (eqref == null) return this;
        eqref = eqref.find();
        return eqref;
    }

    public FileSystemIdentifier getFileSystemId() {
        return find().fsi;
    }

    public void setFileSystemId(FileSystemIdentifier fsi) {
        find().fsi = fsi;
    }

    public void locatorMoved(LocalLocator loc) {
        if (loc == this) return;
        eqref = loc;
        fsi = null;
    }

    public int getOff() {
        return find().off;
    }

    // Only allowed in this package
    void setOff(int off) {
        find().off = off;
    }

    public int getLen() {
        return find().len;
    }

    // Only allowed in this package
    void setLen(int len) {
        find().len = len;
    }

    public boolean isContained(LocalLocator l) {
        if (l == null) return false;
        int off1 = getOff();
        int off2 = l.getOff();
        return off1 <= off2 && off2+l.getLen() <= off1+getLen()
               && getFileSystemId().equals(l.getFileSystemId());
    }

    public boolean equals(Object o) {
        if (o == this) return true;
        if (!(o instanceof LocalLocator)) return false;
        LocalLocator loc = (LocalLocator) o;
        return getOff() == loc.getOff() && getFileSystemId().
                                equals(loc.getFileSystemId());
    }

    public int hashCode() {
        return getOff() ^ getFileSystemId().hashCode();
    }

    public int compareTo(LocalLocator loc) {
        return getOff() - loc.getOff();
    }

    public String toString() {
        FileSystemIdentifier fsi = getFileSystemId();
        return ""+((fsi==null)?"null":fsi)+":"+getOff()+","+getLen();
    }

    public static Streamer<LocalLocator> getStreamer(boolean isCompact) {
        return (isCompact) ? compactStreamer : locatorStreamer;
    }

    private static class LocatorStreamer implements Streamer<LocalLocator> {
        private static Streamer<FileSystemIdentifier> fsiStreamer =
            FileSystemIdentifier.getStreamer();
        private boolean isCompact = false;

        LocatorStreamer(boolean isCompact) {
            this.isCompact = isCompact;
        }

        public void toStream(DataOutput out, LocalLocator l)
                                        throws IOException {
            Streamers.writeShortInt(out, l.getOff());
            if (!isCompact) {
                Streamers.writeShortInt(out, l.getLen());
                fsiStreamer.toStream(out, l.getFileSystemId());
            }
        }

        public LocalLocator fromStream(DataInput in) throws IOException {
            int off = Streamers.readShortInt(in);
            int len = (isCompact) ? 0 : Streamers.readShortInt(in);
            FileSystemIdentifier locFsi = (isCompact) ? null :
                                fsiStreamer.fromStream(in);
            return new LocalLocator(off, len, locFsi);
        }
    }

    public static Discriminator<LocalLocator> getDiscriminator(Memory memory) {
        return new LocatorDiscriminator(memory);
    }

    private static class LocatorDiscriminator extends
                        AbstractDiscriminator<LocalLocator> {
        private Discriminator<Integer> offDisc;
        private Discriminator<byte[]> fsiDisc;

        public LocatorDiscriminator(Memory memory) {
            offDisc = new IntegerDiscriminator(memory);
            fsiDisc = new ByteArrayDiscriminator(memory);
        }

        public <U,S> Collection<List<S>> discriminate(
                          List<? extends U> values,
                          final Extractor<U,? extends LocalLocator,S> e) {
            Extractor<U,Integer,U> e1 = new Extractor<U,Integer,U>() {
                public Integer getLabel(U elm) {
                    return e.getLabel(elm).getOff();
                }
                public U getValue(U elm) {
                    return elm;
                }
            };
            Extractor<U,byte[],S> e2 = new Extractor<U,byte[],S>() {
                public byte[] getLabel(U elm) {
                    return e.getLabel(elm).getFileSystemId().internal();
                }
                public S getValue(U elm) {
                    return e.getValue(elm);
                }
            };
            return Discriminators.discriminate(
                values, offDisc, e1, fsiDisc, e2);
        }
    }
}
