package org.planx.util;

import java.lang.ref.WeakReference;
import java.lang.ref.ReferenceQueue;
import java.util.*;

/**
 * @author Thomas Ambus
 */
public class WeakCollection<E> extends AbstractCollection<E> {
    private ReferenceQueue<E> queue;
    private Entry<E> head;
    private int size = 0;

    public WeakCollection() {
        queue = new ReferenceQueue<E>();
        head = new Entry<E>(null, null, null, queue);
        head.next = head.prev = head;
    }

    public WeakCollection(Collection<? extends E> c) {
        this();
        addAll(c);
    }

    public boolean add(E elm) {
        expungeStaleEntries();
        Entry<E> e = new Entry<E>(elm, head, head.next, queue);
        e.prev.next = e;
        e.next.prev = e;
        size++;
        return true;
    }

    private void remove(Entry<E> e) {
        expungeStaleEntries();
        if (e.prev != null) e.prev.next = e.next;
        if (e.next != null) e.next.prev = e.prev;
        e.prev = null;
        e.next = null;
        size--;
    }

    public int size() {
        expungeStaleEntries();
        return size;
    }

    private void expungeStaleEntries() {
        Entry<E> e;
        while ((e = (Entry<E>) queue.poll()) != null) {
            if (e.prev != null) e.prev.next = e.next;
            if (e.next != null) e.next.prev = e.prev;
            e.prev = null;
            e.next = null;
            size--;
        }
    }

    private static class Entry<E> extends WeakReference<E> {
        Entry<E> prev;
        Entry<E> next;

        Entry(E elm, Entry<E> prev, Entry<E> next, ReferenceQueue<E> queue) {
            super(elm, queue);
            this.prev = prev;
            this.next = next;
        }
    }

    public Iterator<E> iterator() {
        expungeStaleEntries();
        return new WeakIterator();
    }

    private class WeakIterator implements Iterator<E> {
        private Entry<E> last = null;
        private Entry<E> next = head;
        private E nextElm = null; // strong reference to prevent disappearence
        private E lastElm = null; // strong reference to prevent disappearence

        public boolean hasNext() {
            while (nextElm == null) {
                next = next.next;
                if (next == head) return false;
                nextElm = next.get();
            }
            return true;
        }

        public E next() {
            if (!hasNext()) throw new NoSuchElementException();
            lastElm = nextElm;
            nextElm = null;
            last = next;
            return lastElm;
        }

        public void remove() {
            if (last == null) throw new IllegalStateException();
            hasNext();
            WeakCollection.this.remove(last);
            last = null;
            lastElm = null;
        }
    }
}
