package org.planx.msd.util;

import java.util.Arrays;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import org.planx.msd.*;
import org.planx.util.Association;
import org.planx.util.Pair;

/**
 * A utility class that provides various methods for weak-sorting of lists.
 * Weak-sorting means sorting a multiset of lists according to any total order.
 * The lists are, of course, sorted according to the same order.
 *
 * @author Thomas Ambus
 */
public final class WeakSorter {
    private WeakSorter() {}

    /**
     * Sorts a multiset of lists and returns a list of pairs, where the first
     * component is a sorted list, and the second component is the corresponding
     * original list. If <code>doRemoveDuplicates</code> is <code>true</code>
     * duplicates are removed from the sorted lists.
     * The argument discriminator must be capable of discriminating the elements
     * contained in the lists.
     */
    public static <U,E,S> List<Pair<List<E>,S>> sortLists(Discriminator<E> d,
                                                    List<? extends U> values,
                                          Extractor<U,? extends List<E>,S> e,
                                                  boolean doRemoveDuplicates) {
        // Create list of all contained elements and make total order

        int insize = values.size();
        List<IWrap<E>> in = new ArrayList<IWrap<E>>();
        Pair[] sorted = new Pair[insize];

        for (int i=0; i<insize; i++) {
            // Create input
            U elm = values.get(i);
            List<E> list = e.getLabel(elm);
            for (E t : list) {
                in.add(new IWrap<E>(t,i));
            }
            // Prepare result
            List<E> newList = new ArrayList<E>(list.size());
            S value = e.getValue(elm);
            sorted[i] = new Association<List<E>,S>(newList,value);
        }

        Extractor<IWrap<E>,E,IWrap<E>> ext =
                new Extractor<IWrap<E>,E,IWrap<E>>() {
            public E getLabel(IWrap<E> elm) {
                return elm.obj;
            }
            public IWrap<E> getValue(IWrap<E> elm) {
                return elm;
            }
        };
        Collection<List<IWrap<E>>> order = d.discriminate(in, ext);

        // Loop through contained elements in order and re-assign them to
        // lists in this order

        for (List<IWrap<E>> group : order) {
            int prevIdx = -1;
            for (IWrap<E> wrap : group) {
                int idx = wrap.i;
                if (!doRemoveDuplicates || (doRemoveDuplicates &&
                                            idx != prevIdx)) {
                    Pair<List<E>,S> p = sorted[idx];
                    List<E> list = p.getFirst();
                    list.add(wrap.obj);
                }
                prevIdx = idx;
            }
        }
        List result = Arrays.asList(sorted);
        return (List<Pair<List<E>,S>>) result;
    }

    /**
     * Sorts the specified lists in-place. The provided discriminator
     * must be capable of discriminating the elements in the lists.
     */
    public static <E> void sort(Discriminator<E> d,
                        List<? extends List<E>> values) {
        Extractor<List<E>,List<E>,List<E>> e =
            Discriminators.identityExtractor();
        sort(d, values, e);
    }

    /**
     * Sorts a multiset of lists in-place. The provided discriminator
     * must be capable of discriminating the elements in the lists.
     */
    public static <U,E> void sort(Discriminator<E> d, List<? extends U> values,
                                            Extractor<U,? extends List<E>,?> e) {
        List<IWrap<E>> in = new ArrayList<IWrap<E>>();

        int insize = values.size();
        for (int i=0; i<insize; i++) {
            List<E> list = e.getLabel(values.get(i));
            for (E elm : list) {
                in.add(new IWrap<E>(elm,i));
            }
        }

        Extractor<IWrap<E>,E,IWrap<E>> ext =
                new Extractor<IWrap<E>,E,IWrap<E>>() {
            public E getLabel(IWrap<E> elm) {
                return elm.obj;
            }
            public IWrap<E> getValue(IWrap<E> elm) {
                return elm;
            }
        };
        Collection<List<IWrap<E>>> order = d.discriminate(in, ext);

        int[] cursors = new int[insize];
        for (List<IWrap<E>> group : order) {
            for (IWrap<E> wrap : group) {
                int idx = wrap.i;
                List<E> list = e.getLabel(values.get(idx));
                int pos = cursors[idx]++;
                list.set(pos, wrap.obj);
            }
        }
    }

    private static class IWrap<E> {
        E obj;
        int i;
        public IWrap(E obj, int i) {
            this.obj=obj;
            this.i=i;
        }
    }
}
