package org.planx.msd.util;

import java.util.AbstractList;
import java.util.Collection;
import java.util.List;
import org.planx.msd.*;

/**
 * This class provides a skeletal implementation of a
 * <code>Discriminator</code> that adapts its input to suit that of
 * another <code>Discriminator</code>.
 * The type arguments <code>T</code> and <code>E</code> corresponds to
 * the type adapted from and the type adapted to, respectively. The
 * implementation need only provide a {@link #transform} method.
 *
 * @author Thomas Ambus
 */
public abstract class DiscriminatorAdapter<T,E> extends
                            AbstractDiscriminator<T> {
    private final Discriminator<E> disc;

    /**
     * Constructs a new <code>DiscriminatorAdapter</code> which adapts
     * its input to the specified <code>Discriminator</code>.
     */
    public DiscriminatorAdapter(Discriminator<E> d) {
        disc = d;
    }

    public <U,S> Collection<List<S>> discriminate(List<? extends U> values,
                                        final Extractor<U,? extends T,S> e) {
        Extractor<U,E,S> ext = new Extractor<U,E,S>() {
            public E getLabel(U elm) {
                return transform(e.getLabel(elm));
            }
            public S getValue(U elm) {
                return e.getValue(elm);
            }
        };
        return disc.discriminate(values, ext);
    }

    /**
     * Transform a value of the input type <code>T</code> to the type
     * <code>E</code> expected as input to the wrapped
     * <code>Discriminator</code>.
     */
    protected abstract E transform(T value);
}
